; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
; ---------------------------------------------------------------------
;  If !SERVER is being invoked by the user locally, we will create the 
;  following directory structure for them to simulate an ESE environment.
;  
;  Directory structure of ESE:
;  -> ese_top (or <directoryroot> if specified)  
;     -> docroot
;        -> ese
;           -> data
;           -> jobs
;              -> local (job sandbox)
;  -> tasks
; ---------------------------------------------------------------------


; ---------------------------------------------------------------------
FUNCTION !SERVER::Init, obj, $
  DIRECTORYROOT=directoryroot, $ ;used for specifying top level folder when running in local mode
  HOSTNAME=hostname,           $
  DOCROOT=docroot,             $
  ESEROOT=eseroot,             $
  DATADIR=datadir,             $
  DATAURL=dataurl,             $
  JOBSDIR=jobsdir,             $
  JOBNUMBER=jobnumber,         $
  LOCALDIR=localdir,           $
  LOCALURL=localurl,           $
  NOSERVER=noserver,           $
  TASKDIR=taskdir,             $
  PROTOCOL=protocol,           $
  ROOTURL=rooturl,             $
  ESEURL=eseurl,               $
  SERVER=server,               $
  _EXTRA=ex
  COMPILE_OPT idl2, hidden
  ON_ERROR,2
  HEAP_NOSAVE,self

  IF (ISA(obj) && ISA(obj, 'OBJREF')) THEN BEGIN
    self.__obj__ = obj
  ENDIF

  ; - Default top level of ESE directory structure if running in local mode -
  tmprootdir = (ISA(directoryroot) ? directoryroot : FILEPATH('',/tmp)) + PATH_SEP() + 'ese_top'

  ; - DOCROOT -
  IF ~ISA(docroot) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      docroot = tmprootdir + PATH_SEP() + 'docroot'
      FILE_MKDIR, docroot
    ENDIF ELSE BEGIN
      docroot = ''
    ENDELSE
  ENDIF


  ; - ESEROOT -
  IF ~ISA(eseroot) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      eseroot = docroot + PATH_SEP() + 'ese'
      FILE_MKDIR, eseroot
    ENDIF ELSE BEGIN
      eseroot = ''
    ENDELSE
  ENDIF

  ; - Task -
  IF ~ISA(taskdir) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      taskdir = tmprootdir + PATH_SEP() + 'tasks'
      FILE_MKDIR, taskdir
    ENDIF ELSE BEGIN
      taskdir = ''
    ENDELSE
  ENDIF

  ; - Data Directory -
  IF ~ISA(datadir) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      datadir = eseroot + PATH_SEP() + 'data'
      FILE_MKDIR, datadir
    ENDIF ELSE BEGIN
      datadir = ''
    ENDELSE
  END

  ; - Jobs Directory -
  IF ~ISA(jobsdir) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      jobsdir = eseroot + PATH_SEP() + 'jobs'
      FILE_MKDIR, jobsdir
    ENDIF ELSE BEGIN
      jobsdir = ''
    ENDELSE
  ENDIF

  ; - Local Directory (job sandbox) -
  IF ~ISA(localdir) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      IF ~ISA(jobnumber) THEN BEGIN
        CD, jobsdir, CURRENT=cur
        dir = FILE_SEARCH(/TEST_DIRECTORY)
        jobnumber = STRTRIM(STRING(dir[-1] + 1),2)
        CD, cur
      ENDIF

      localdir = jobsdir + PATH_SEP() + jobnumber
      FILE_MKDIR, localdir
    ENDIF ELSE IF KEYWORD_SET(server) THEN BEGIN
      CD, C=localdir
      IF STRPOS(localdir,PATH_SEP(),/REVERSE_SEARCH)  EQ (STRLEN(localdir)-1) THEN BEGIN
        localdir = STRMID(path,0,STRLEN(path)-1)
      ENDIF
    ENDIF ELSE BEGIN
      localdir = ''
    ENDELSE
  ENDIF

  ; - Protocol -
  IF ~ISA(protocol) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      IF !version.os EQ 'Win32' THEN BEGIN
        protocol = 'file:///'
      ENDIF ELSE BEGIN
        protocol = 'file://'
      ENDELSE
    ENDIF ELSE BEGIN
      protocol = ''
    ENDELSE
  ENDIF

  ; - Hostname -
  IF ~ISA(hostname) THEN BEGIN
    hostname = tmprootdir
  ENDIF

  ; - Data URL -
  IF ~ISA(dataurl) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      dataurl = protocol + datadir
    ENDIF ELSE BEGIN
      dataurl = ''
    ENDELSE
  ENDIF

  ; - Local URL -
  IF ~ISA(localurl) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      localurl = protocol + localdir
    ENDIF ELSE BEGIN
      localurl = ''
    ENDELSE
  ENDIF

  ; - Docroot URL -
  IF ~ISA(rooturl) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      rooturl = protocol + docroot
    ENDIF ELSE BEGIN
      rooturl = ''
    ENDELSE
  ENDIF

  ; - ESEROOT URL -
  IF ~ISA(eseurl) THEN BEGIN
    IF KEYWORD_SET(noserver) THEN BEGIN
      eseurl = protocol + eseroot
    ENDIF ELSE BEGIN
      eseurl = ''
    ENDELSE
  ENDIF

  self._localdir  = localdir
  self._datadir   = datadir
  self._docroot   = docroot
  self._eseroot   = eseroot
  self._taskdir   = taskdir
  self._dataurl   = dataurl
  self._localurl  = localurl
  self._rooturl   = rooturl
  self._eseurl    = eseurl
  self._hostname  = hostname
  self._protocol  = protocol

  ; Initialize our url handlers
  self.__urlHandlers = UrlHandlers()
  self.__urlDirectives = LIST()
  
  self._cleanupProperties,LOCAL=noserver

  ; Initialize Error Messages
  self.__msg_invalid_file = "Invalid input file."
  self.__msg_invalid_type = "Invalid input type."
  self.__msg_invalid_url  = "Invalid input url."
  self.__msg_invalid_hash = "Invalid input hash. Hash must contain 'url' key."
  self.__msg_specify_url  = "Please specify an input url."
  self.__msg_specify_file = "Please specify an input file."
  self.__msg_suggest_str  = "Invalid input url. Try using /STRING."



  RETURN, 1
END

;+
; :Description:
;    Given the name of a file in the ESE config folder, return the full
;    path filename.
;
; :Params:
;    filename
;
;
;-
FUNCTION !server::ese_config_folder, filename
  COMPILE_OPT idl2

  path_name = self._eseroot
  file_name = filename
  base_name = FILE_BASENAME(file_name)
  IF (!VERSION.os_family EQ 'Windows') THEN BEGIN
    path_name = path_name.substring(0,path_name.lastindexof('\')-1)
    path_name = path_name.substring(0,path_name.lastindexof('\')-1)
  ENDIF ELSE BEGIN
    path_name = path_name.substring(0,path_name.lastindexof('/')-1)
    path_name = path_name.substring(0,path_name.lastindexof('/')-1)
  ENDELSE

  RETURN, path_name + PATH_SEP() + 'bin' + PATH_SEP() + base_name

END

;-------------------------------------------------------------------------------
PRO !SERVER::GetProperty, DATADIR=datadir,     $
  DOCROOT=docroot,     $
  ESEROOT=eseroot,     $
  DATAURL=dataurl,     $
  LOCALURL=localurl,   $
  LOCALDIR=localdir,   $
  ROOTURL=rooturl,     $
  ESEURL=eseurl,       $
  HOSTNAME=hostname,   $
  PROTOCOL=protocol,   $
  TASKDIR=taskdir
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  IF ARG_PRESENT(datadir) THEN $
    datadir=self._datadir

  IF ARG_PRESENT(localdir) THEN $
    localdir=self._localdir

  IF ARG_PRESENT(docroot) THEN $
    docroot=self._docroot

  IF ARG_PRESENT(eseroot) THEN $
    eseroot=self._eseroot

  IF ARG_PRESENT(dataurl) THEN $
    dataurl=self._dataurl

  IF ARG_PRESENT(localurl) THEN $
    localurl=self._localurl

  IF ARG_PRESENT(rooturl) THEN $
    rooturl=self._rooturl

  IF ARG_PRESENT(eseurl) THEN $
    eseurl=self._eseurl

  IF ARG_PRESENT(hostname) THEN $
    hostname=self._hostname

  IF ARG_PRESENT(protocol) THEN $
    protocol=self._protocol

  IF ARG_PRESENT(taskdir) THEN $
    taskdir=self._taskdir

END

;-------------------------------------------------------------------------------
PRO !SERVER::SetProperty,     $
  DOCROOT=docroot,     $
  ESEROOT=eseroot,     $
  DATADIR=datadir,     $
  DATAURL=dataurl,     $
  LOCALDIR=localdir,   $
  LOCALURL=localurl,   $
  ROOTURL=rooturl,     $
  ESEURL=eseurl,       $
  HOSTNAME=hostname,   $
  PROTOCOL=protocol,   $
  TASKDIR=taskdir
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  IF ISA(datadir) THEN $
    self._datadir=datadir

  IF ISA(localdir) THEN $
    self._localdir = localdir

  IF ISA(docroot) THEN $
    self._docroot = docroot

  IF ISA(eseroot) THEN $
    self._eseroot = eseroot

  IF ISA(dataurl) THEN $
    self._dataurl = dataurl

  IF ISA(localurl) THEN $
    self._localurl = localurl

  IF ISA(rooturl) THEN $
    self._rooturl = rooturl

  IF ISA(eseurl) THEN $
    self._eseurl = eseurl

  IF ISA(hostname) THEN $
    self._hostname = hostname

  IF ISA(protocol) THEN $
    self._protocol = protocol

  IF ISA(taskdir) THEN $
    self._taskdir = taskdir

END

;-------------------------------------------------------------------------------
FUNCTION !SERVER::_overloadPrint
  COMPILE_OPT idl2,hidden
  ON_ERROR,2

  ; TODO: Make this pretty

  props = ['DATADIR','LOCALDIR','TASKDIR','DOCROOT','ESEROOT', $
    'DATAURL','LOCALURL','ROOTURL', 'ESEURL', $
    'HOSTNAME','PROTOCOL']
  result = OBJ_CLASS(self) + ' <' + STRTRIM(OBJ_VALID(self,/GET_HEAP_ID),2) + '>'
  result = [result, STRARR(N_ELEMENTS(props))]
  i=1
  FOREACH p, props DO BEGIN
    CASE (p) OF
      'DATADIR': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._datadir,2)
      END
      'LOCALDIR': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._localdir,2)
      END
      'DOCROOT': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._docroot,2)
      END
      'ESEROOT': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._eseroot,2)
      END
      'DATAURL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._dataurl,2)
      END
      'LOCALURL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._localurl,2)
      END
      'ROOTURL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._rooturl,2)
      END
      'ESEURL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._eseurl,2)
      END
      'HOSTNAME': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._hostname,2)
      END
      'PROTOCOL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._protocol,2)
      END
      'TASKDIR': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._taskdir,2)
      END
      'TASKURL': BEGIN
        result[i] = STRING(p, FORMAT='("  ", A-25, " = ")') + STRTRIM(self._taskurl,2)
      END
      ELSE: BEGIN
        result[i] = STRING('UNKNOWN PROPERTY ', FORMAT='("  ", A-25, " : ")') + STRTRIM(p,2)
      END
    ENDCASE

    i++
  ENDFOREACH

  result = REFORM(result, 1, N_ELEMENTS(result), /OVERWRITE)

  RETURN, result
END

;-------------------------------------------------------------------------------
PRO !SERVER::_cleanupProperties, LOCAL=local
  uh = self.__urlHandlers
  self._localdir  = uh.makepathpretty(self._localdir)
  self._datadir   = uh.makepathpretty(self._datadir)
  self._docroot   = uh.makepathpretty(self._docroot)
  self._eseroot   = uh.makepathpretty(self._eseroot)
  self._taskdir   = uh.makepathpretty(self._taskdir)
  self._dataurl   = uh.makeurlpretty(self._dataurl)
  self._localurl  = uh.makeurlpretty(self._localurl)
  self._rooturl   = uh.makeurlpretty(self._rooturl)
  self._eseurl    = uh.makeurlpretty(self._eseurl)

  IF KEYWORD_SET(local) THEN BEGIN
    self._hostname = uh.makepathpretty(self._hostname)
  ENDIF ELSE BEGIN
    self._hostname = uh.makeurlpretty(self._hostname)
  ENDELSE
END



;-------------------------------------------------------------------------------
FUNCTION !SERVER::_isMutuallyExclusive, data, local, docroot, task, message=message
  COMPILE_OPT idl2,hidden
  ON_ERROR,2

  ret = []
  IF ISA(data)    THEN ret = [ret,'DATA']
  IF ISA(local)   THEN ret = [ret,'LOCAL']
  IF ISA(docroot) THEN ret = [ret,'DOCROOT']
  IF ISA(eseroot) THEN ret = [ret,'ESEROOT']
  IF ISA(task)    THEN ret = [ret,'TASK']

  IF N_ELEMENTS(ret) GT 1 THEN BEGIN
    IF ARG_PRESENT(message) THEN message = 'Keywords ' + STRJOIN(ret,', ') + $
      ' are mutually exclusive.'
    RETURN, 0
  ENDIF

  RETURN, 1
END

;-------------------------------------------------------------------------------
FUNCTION !SERVER::FILEPATH, file, DATA=data,       $
  LOCAL=local,     $
  DOCROOT=docroot, $
  TASK=task

  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  IF ~ISA(file) THEN MESSAGE,self.__msg_specify_file
  IF SIZE(file,/type) NE 7 THEN MESSAGE,self.__msg_invalid_type
  IF ~self._isMutuallyExclusive(data,local,docroot,task,message=msg) THEN MESSAGE,msg

  path=''
  IF KEYWORD_SET(data) THEN BEGIN
    path = self._datadir
  ENDIF ELSE IF KEYWORD_SET(docroot) THEN BEGIN
    path = self._docroot
  ENDIF ELSE IF KEYWORD_SET(eseroot) THEN BEGIN
    path = self._eseroot
  ENDIF ELSE IF KEYWORD_SET(task) THEN BEGIN
    path = self._taskdir
  ENDIF ELSE BEGIN
    path = self._localdir
  ENDELSE

  root = []
  FOREACH f, file DO BEGIN
    IF STRPOS(f,'/') NE 0 && STRPOS(f,'\') NE 0 THEN BEGIN
      root = [root, path + PATH_SEP()]
    ENDIF
  ENDFOREACH

  IF N_ELEMENTS(root) EQ 0 THEN root = path

  RETURN, (self.__urlHandlers).makepathpretty(root + STRTRIM(file))
END


;+
; :Description:
;    Given a url to a server resource, remove the resource from the
;    system.  This is currently not removing the file from
;    the file system.  Instead, it is just removing it from the
;    an OGC service.
;
; :Params:
;    urlin
;
;
;-
FUNCTION !SERVER::UrlToDelete, urlin
  COMPILE_OPT idl2
  ON_ERROR,2

  IF ~ISA(urlin) THEN MESSAGE,self.__msg_specify_url
  IF ~(ISA(urlin,'HASH') || ISA(urlin,'LIST') || SIZE(urlin,/type) EQ 7) THEN $
    MESSAGE,self.__msg_invalid_type

  IF SIZE(urlin,/type) EQ 7 THEN BEGIN
    IF N_ELEMENTS(urlin) GT 1 THEN BEGIN
      string=1
    ENDIF ELSE IF ~(STRPOS(urlin,'[') EQ 0 || STRPOS(urlin,'{') EQ 0) THEN BEGIN
      string=1
    ENDIF
  ENDIF
  
  ret = self.__urlHandlers.deleteUrl(urlin)
  
  return, ret

END



;+
; :Description:
;    Given a url, do the proper conversion to a local file path if appropriate,
;    e.g. for file scheme or http scheme when it is on our ESE domain,
;    otherwise, leave it alone, e.g. jpip and http uri's on other domains.
;    
;    This method provides a huge performace gain within ESE.  If a url
;    can be transformed into a local filename, ENVI can eliminate the 
;    overhead of transferring the data across a network connection and
;    just open the file directly.
; :Params:
;    urlin : this may be:
;              a list of url strings
;              a list of url hashes e.g. {'url','http://...'}
;              a hash e.g. {'url','http://...'}
;              a string  e.g. 'http://...',  or a json formatted string
;-
FUNCTION !SERVER::UrlToFile, urlin
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  IF ~ISA(urlin) THEN MESSAGE,self.__msg_specify_url
  IF ~(ISA(urlin,'HASH') || ISA(urlin,'LIST') || ISA(urlin,'STRING')) THEN $
    MESSAGE,self.__msg_invalid_type

  IF ISA(urlin,'STRING') THEN BEGIN
    IF N_ELEMENTS(urlin) GT 1 THEN BEGIN
      string=1
    ENDIF ELSE IF ~(STRPOS(urlin,'[') EQ 0 || STRPOS(urlin,'{') EQ 0) THEN BEGIN
      string=1
    ENDIF
  ENDIF

  url=[]
  IF ~KEYWORD_SET(string) THEN BEGIN
    IF ~(ISA(urlin,'HASH') || ISA(urlin,'LIST')) THEN BEGIN
      json = json_parse(urlin)
    ENDIF ELSE BEGIN
      json = urlin
    ENDELSE
    IF ISA(json,'LIST') THEN BEGIN
      FOREACH j,json DO BEGIN
        if (ISA(j,'string')) then begin
          url = [url,j]
        endif else begin
          if (ISA(j,'hash')) then begin
            if ~j.haskey('url') then message, self.__msg_invalid_hash
              url=[url,j['url']]
            endif else begin
              message, self.__msg_invalid_url
            endelse 
        endelse       
      ENDFOREACH
    ENDIF ELSE BEGIN
      IF ~json.haskey('url') THEN MESSAGE,self.__msg_invalid_hash
      url = json['url']
    ENDELSE
  ENDIF ELSE BEGIN
    url = urlin
  ENDELSE

  ; Initialize our return list
  ret=list()
  
  FOR i=0, N_ELEMENTS(url)-1 DO BEGIN

    ; Let our configured url handlers try to derive a filename from the url.
    ; For example, an OGC service might be able to derive a filename from the url.
    handler_ret = self.__urlHandlers.UrlToFile( url[i] )

    IF (handler_ret NE !NULL && N_ELEMENTS(handler_ret) GT 0) THEN BEGIN
      ret.add,  handler_ret[0]
    ENDIF ELSE BEGIN
      ; A local filename could not be derived from the url.
      ; So, leave the url as is.
      ; ENVI may pull the data from the server hosting the url.
      ret.add, url[i]
    ENDELSE

  ENDFOR

  IF N_ELEMENTS(ret) EQ 0 THEN ret=url
  IF N_ELEMENTS(ret) EQ 1 THEN ret=ret[0]

  RETURN, ret

END


;+
; :Description:
;    Add a string directive to our list of url handler directives.
;    Adding the proper string to this last can effect the
;    behavior of the registered url handlers.
;    
;    For example, if you add the directive 'SkipGeoServer', 
;    the geoserver_handler will not attempt to register the
;    asset with GeoServer.
;
; :Params:
;    stringDirective 
;-
PRO !SERVER::AddUrlHandlerDirective, stringDirective
   COMPILE_OPT idl2
   
   if ( ~isa(stringDirective,/string,/scalar) && ~isa(stringDirective,/string,/array)) then begin
    self.error, '!SERVER::AddUrlHandlerDirective  - Directive must be a string scalar or string array.'
   endif
   
   self.__urlDirectives.add, stringDirective
   
END

;+
; :Description:
;    Return true of false.
;    URL handler functions can obtain this list of strings
;    to obtain additional directives influencing the
;    behavior of a specific handler.  
;    
;  :Returns: 
;    List of strings, which are the directives.
;-
FUNCTION !SERVER::UrlHandlerDirectiveExists, strDirective
  COMPILE_OPT idl2
  
   uhd_array = self.__urlDirectives.toArray()  
   if (uhd_array eq !NULL) THEN BEGIN 
     uhd_array = STRARR(1)
   ENDIF    
   exists = WHERE(STRMATCH(uhd_array, strDirective, /FOLD_CASE) )
   IF ( total(exists) ne -1 ) THEN BEGIN
     RETURN, !TRUE
   ENDIF
   
   RETURN, !FALSE
   
END

;-------------------------------------------------------------------------------
FUNCTION !SERVER::FileToUrl, filein, DATA=data,       $
  LOCAL=local,     $
  DOCROOT=docroot
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  IF ~ISA(filein) THEN MESSAGE,self.__msg_specify_url
  IF SIZE(filein,/type) NE 7 THEN MESSAGE,self.__msg_invalid_type
  IF ~self._isMutuallyExclusive(data,local,docroot,task,message=msg) THEN MESSAGE,msg


  IF KEYWORD_SET(data) THEN BEGIN
    prefix = self._datadir
    url_prefix = self._dataurl
  ENDIF ELSE IF KEYWORD_SET(docroot) THEN BEGIN
    prefix = self._docroot
    url_prefix = self._rooturl
  ENDIF ELSE BEGIN
    prefix = self._localdir
    url_prefix = self._localurl
  ENDELSE
  
  
  result = list()
  FOREACH file,filein,idx DO BEGIN
    file=STRTRIM(file)
    l = self.__urlHandlers.fileToUrl(PREFIX_FILE=prefix,FILE=file,PREFIX_URL=url_prefix)

    IF (l NE !NULL && N_ELEMENTS(l) GT 0) THEN BEGIN
      result.add, l[0]
    ENDIF ELSE BEGIN
      result.add, ''
    ENDELSE

  ENDFOREACH

  ; Make the correct return value
  IF (result.Count() EQ 0) THEN RETURN, filein
  IF (result.Count() EQ 1) THEN RETURN, result[0]
  RETURN,result

END

; Use instead of ese_notify_progress
;-------------------------------------------------------------------------------
PRO !SERVER::notify, progressPct, progressText
  COMPILE_OPT idl2,hidden
  ON_ERROR,2

  IF ~ISA(progressPct) THEN progressPct = ''
  IF ~ISA(progressText) THEN progressText = ''

  ;; CONSTRUCT PROGRESS DATA
  progressData     = (progressPct EQ '' && progressText EQ '') ? $
    '' : STRING(progressPct) + '::' + progressText
  PRINT, progressData
  !null = IDLNotify('eseNotification','progress',progressData)

END


; Send a custom error message message back to the client
; and fail the task immediately
;-------------------------------------------------------------------------------
PRO !SERVER::error, errorMessage
  COMPILE_OPT idl2,hidden
  ON_ERROR,2

  IF ~ISA(errorMessage) THEN errorMessage = ''

  errorString = STRTRIM(STRING(errorMessage, /PRINT),2)

  PRINT, '!SERVER.error called with message:'
  PRINT, errorString
  PRINT

  !null = IDLNotify('eseNotification','error',errorString)

  ; return to $MAIN$
  PRINT, 'Ending task and returning to $MAIN$'
    RETALL

END


FUNCTION !SERVER::_overloadImpliedPrint, varname
  return, self._overloadPrint()
END


; Notes:
;   Consider adding a temp directory which will be cleaned up after the task is
;   run
;-------------------------------------------------------------------------------
PRO !SERVER__define
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  struct = {!SERVER, INHERITS IDL_OBJECT, $
    _datadir: '',                 $
    _localdir: '',                $
    _taskdir: '',                 $
    _docroot: '',                 $
    _eseroot: '',                 $
    _dataurl: '',                 $
    _localurl: '',                $
    _rooturl: '',                 $
    _eseurl: '',                  $
    _hostname: '',                $
    _protocol: '',                $
    __urlHandlers: OBJ_NEW(),     $
    __urlDirectives: OBJ_NEW(),      $
    __msg_invalid_file: '',       $
    __msg_invalid_type: '',       $
    __msg_invalid_url: '',        $
    __msg_invalid_hash: '',       $
    __msg_specify_url: '',        $
    __msg_specify_file: '',       $
    __msg_suggest_str: ''         $
  }

END
