;----------------------------------------------------------------------------
;
;  Copyright (c) NV5 Geospatial Solutions, Inc. All
;  rights reserved. This software includes information which is
;  proprietary to and a trade secret of NV5 Geospatial Solutions, Inc.
;  It is not to be disclosed to anyone outside of this organization.
;  Reproduction by any means whatsoever is prohibited without express
;  written permission.
;
;----------------------------------------------------------------------------


; Name:
;   Python_utils
;
; Purpose:
;   Routines for enabling embedded Python installation, and adding/removing Python packages



;-------------------------------------------------------------------------
;+
; :Description:
;   Checking which PYTHON bridge DLM is currently loaded
;
; :Returns: string
;
; :Arguments:
;   info: out, optional, string
; :Keywords:
;   None
;-
function PyUtils::GetCurrentPython, info
  compile_opt idl2, hidden, static

  python_version_loaded = ''
  full_version = ''
  help, /DLM, NAMES="PYTHON*", output=installedDLMstext

  for index = 0L, n_elements(installedDLMstext)-1,3 do begin
    line = installedDLMstext[index]
    line_full_version = installedDLMstext[index+1]
    line_python_version = ((line.split(' '))[1]).replace("PYTHON","")
    if line.indexOf('(loaded)') ne -1 then begin
      python_version_loaded = line_python_version
      version_index = where((line_full_version.split(' ')).startswith('Version') eq 1)
      full_version = (((line_full_version.split(' '))[version_index+2]).replace(",",""))[0]
    endif
  endfor

  if ARG_PRESENT(info) then begin
    info = dictionary()
    if python_version_loaded ne '' then begin
      PyUTILS.FindPython, version=python_version_loaded, output=out_pythons
      if N_ELEMENTS(out_pythons) eq 1 then begin
        info = (out_pythons[(out_pythons.keys())[0]])[0]
      endif
    endif
  endif

  return, python_version_loaded
end


;-------------------------------------------------------------------------
;+
; :Description:
;   Add a path folder to PYTHONPATH environment variable if not already added.
;   The existence of the folder is not checked.
;
; :Arguments:
;   path: in, required, string
;
;-
pro PyUtils::_AddPathToPythonPath, path
  compile_opt idl2, hidden, static

  python_path_env = getenv("PYTHONPATH")
  if ~python_path_env.contains(path) then begin
    sep = path_sep(/SEARCH_PATH)
    python_path_env = python_path_env eq '' ? path : path + sep + python_path_env
    setenv, "PYTHONPATH=" + python_path_env
  endif

end


;-------------------------------------------------------------------------
;+
; :Description:
;   Finds the embedded python version by looking at
;   folder idl-pythonXYZ in <IDL> bin foider
;   
;   :Outputs
;   - the Python embedded version as a string with no dots
;
; :Returns: any
;
;-
function PyUtils::GetEmbeddedPythonVersion
  compile_opt idl2, hidden, static

  python_path = !DLM_PATH + path_sep() + "idl-python*"
  found = file_search(python_path, count=count)
  if count eq 1 then begin
    return, (found.replace(python_path.substring(0,-2),''))[0]
  endif
  return, ''

end


;-------------------------------------------------------------------------
;+
; :Description:
;   Finds all the installed python versions
;
; :Keywords:
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;   skip_embedded: bidirectional, optional, any
;     Placeholder docs for argument, keyword, or property
;   version: bidirectional, optional, any
;     Placeholder docs for argument, keyword, or property
;-
pro PyUtils::FindPython, version=version, OUTPUT=output, SKIP_EMBEDDED=skip_embedded
  compile_opt idl2, static

  on_error, 2

  output = ''

  embedded_path = PyUtils._GetEmbeddedPythonFolder()

  case (!version.OS_FAMILY.tolower()) of
    "windows": begin
      locate_python = "where"
      python_exe = 'python.exe'
      spawn, [locate_python,python_exe], paths, error, /NOSHELL, EXIT_STATUS=exitstatus
      if (exitStatus gt 0 && error[0] ne '') then begin
        message, error
      endif
    end
    "unix": begin
      ; Search for python3.XY in all the PATH
      python_exe = 'python3'
      is_darwin = !version.OS eq 'darwin'
      raw_paths = []
      paths = []
      system_paths = (getenv("PATH")).split(path_sep(/SEARCH_PATH))
      foreach s_path, system_paths do begin
        ; skip symlink folder like /bin
        if file_test(s_path, /SYMLINK) then continue
        raw_paths = file_search(s_path+path_sep()+python_exe+".*", /TEST_EXECUTABLE, /TEST_REGULAR, count=count)
        if count gt 0 then begin
          foreach rp, raw_paths do begin
            is_symlink = file_test(rp, /SYMLINK)

            ; skip python3.XY-intel64
            if is_darwin then begin
              if rp.indexOf("-intel64") eq -1 && $
              rp.indexOf("-config") eq -1 then begin
                if paths eq !NULL || ~total(paths.contains(rp)) then paths = [paths,rp]
              endif
            endif else begin
              ; skip python3.XY-config
              if rp.indexOf("-config") eq -1 then begin
                 if paths eq !NULL || ~total(paths.contains(rp)) then paths = [paths,rp]
              endif
            endelse
          endforeach
        endif
      endforeach
      ; here we need to add embedded version
    end
    else: begin
    end
  endcase

  do_skip_embedded = KEYWORD_SET(SKIP_EMBEDDED)

  output = Hash()

  embedded_path = PyUtils._GetEmbeddedPythonFolder()

  ; reverse the loop so embedded version is last in the loop
  if do_skip_embedded then paths = reverse(paths)

  foreach python_path, paths do begin
    ; embedded version
    if python_path.contains(embedded_path) && do_skip_embedded then continue
    is_embedded = python_path.contains(!DLM_PATH) ? !true : !false
    spawn, [python_path, '--version'], output_version, error_python, /NOSHELL, EXIT_STATUS=exitstatus
    ; skip version like default Microsoft AppData\Local\Microsoft\WindowsApps\python.exe
    if (exitStatus gt 0 && error_python[0] ne '') then begin
      continue
    endif
    this_version = (output_version.replace('python','',/FOLD_CASE)).trim()
    short_version_str = (this_version.split('\.'))[0:1].join('')
    ; search only for version input
    if KEYWORD_SET(version) && version ne short_version_str then continue
    dot_short_version = (this_version.split('\.'))[0:1].join('.')
    if ~output.HasKey('PYTHON'+short_version_str) then begin
      output['PYTHON'+short_version_str] = List(DICTIONARY( 'PATH' , python_path, $
        'VERSION' , this_version[0], $
        'SHORT_VERSION', dot_short_version, $
        'EMBEDDED', is_embedded ))
      if KEYWORD_SET(version) && version eq short_version_str then break
    endif else begin
      output['PYTHON'+short_version_str].add, DICTIONARY( 'PATH' , python_path, $
        'VERSION' , this_version[0], $
        'SHORT_VERSION', dot_short_version, $
        'EMBEDDED', is_embedded )
    endelse
  endforeach

  if ~ARG_PRESENT(output) then begin
    print, JSON_Serialize(output, /PRETTY)
  endif

end


;-------------------------------------------------------------------------
;+
; :Description:
;   Checks if PATH variable contains the paths to embedded Python
;
; :Returns: any
;
; :Outputs:
;   True / False
;
;-
function PyUtils::_checkPythonPaths
  compile_opt idl2, hidden, static

  paths = (getenv("PATH")).split(path_sep(/SEARCH_PATH))
  python_paths = paths[where(paths.contains("python", /FOLD_CASE))]
  embedded_python_path = PyUtils._GetEmbeddedPythonFolder()
  ok = total(python_paths.contains(embedded_python_path)) gt 0
  return, ok

end


;-------------------------------------------------------------------------
;+
; :Description:
;   This method retrieves the license information for a Python package or for all.
;   Uses the python_package_license.py located in <IDL>\lib\bridges
;   This procedure only works with the IDL embedded Python.
;   Calling this while a different Python is loaded will result in an error.
;
; :Arguments:
;   package: in, optional, any
;     A string giving the name of the package.
;
; :Keywords:
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;-
pro PyUtils::PipGetLicense, package, OUTPUT=output
  compile_opt idl2, static, hidden

  on_error, 2

  ; We only want to retrieve licenses for our embedded Python.
  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    PyUtils.Load
  endif
  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    message, 'Unable to load embedded Python. Cannot retrieve licenses.'
  endif
  defsysv, '!IDLPYEMBEDDED', exists=isEmbedded
  if (~isEmbedded) then begin
    message, 'You can only retrieve licenses for embedded Python packages.'
  endif

  output = ''

  >>>package = None
  if package ne !NULL then begin
    Python.package = package
  endif

  >>>import python_package_license
  >>>from python_package_license import getPackagesLicenses
  >>>package_licenses = getPackagesLicenses(package)

  package_licenses = Python.package_licenses

  if package_licenses.Count() eq 0 && package ne !NULL then begin
    output = "Package " + package + " not found."
  endif else begin
    output = package_licenses
  endelse
  if ~ARG_PRESENT(output) then begin
    print, JSON_SERIALIZE(output, /PRETTY)
  endif
end


;-------------------------------------------------------------------------
;+
; :Description:
;   Add a path folder to PATH environment variable but check if not already added.
;   The existence of the folder is not checked.
;
; :Arguments:
;   path: in, required, String
;     A string giving the path to add.
;
;-
pro PyUtils::_AddPathToSystemPath, path
  compile_opt idl2, static

  vpath = getenv('PATH')
  envpaths = vpath.split(path_sep(/SEARCH_PATH))
  foreach pypath, envpaths do begin
    if (pypath eq path) then begin
      return
    endif
  endforeach

  ; No path has been found in PATH
  ; add paths at the beginning
  envpaths = [path, envpaths]

  setenv, "PATH=" + envpaths.join(path_sep(/search_path))  ; ";" or ":"
end


;-------------------------------------------------------------------------
;+
; :Description:
;   returns the user's .idl python folder
;
; :Returns: A string with the path to the folder
;
;-
function PyUtils::_GetUserBase
  compile_opt idl2, hidden, static

  ; This seems to be the simplest way to get the user's .idl folder.
  result = !package_path
  ; Strip off the idl/packages subfolders
  if (result.endsWith('packages')) then result = file_dirname(result)
  if (result.endsWith(path_sep() + 'idl')) then result = file_dirname(result)
  ; Add our idl-python + python version number
  result += path_sep() + PyUtils._GetEmbeddedPythonName()

  if ~file_test(result, /DIRECTORY) then begin
    FILE_MKDIR, result
  endif
  return, result

end


;---------------------------------------------------------------------------
;+
; :Returns: String
;   A string representing the current Python version, for example "3.13"
;
; :Keywords:
;   python_exe: input, optional, String
;     A string containing the full path to the Python executable.
;
;-
function PyUtils::_GetSysPythonRawVersion, python_exe=python_exe
  compile_opt idl2, static

  sysVersionCmd = "import sys; print('.'.join([str(sys.version_info.major),str(sys.version_info.minor)]))"
  if (!VERSION.OS_FAMILY eq 'Windows') then begin
    sysVersionCmd = '"' + sysVersionCmd + '"'
  endif
  ; use the python_exe given as a keyword
  if ~keyword_set(python_exe) then python_exe = PyUTILS._GetPythonExe()
  ; idl-disable-next-line unused-var
  SPAWN, [python_exe, '-c', sysVersionCmd], raw_version, stderr, /noshell, EXIT_STATUS=exitStatus
  if (exitStatus gt 0) then begin
    MESSAGE, 'Failed to get python sys.version_info.major/minor'
  endif
  return, raw_version
end


;-------------------------------------------------------------------------
;+
;-
pro PyUtils::_AddUserSitePackageToPythonPath
  compile_opt idl2, hidden, static

  ; IDLPYEMBEDDED is not defined at this stage
  ; if ~PyUtils._EmbeddedPythonIsLoaded() then return 

  user_base = PyUtils._GetUserBase()
  setenv, 'PYTHONUSERBASE=' + user_base

  case (!version.OS_FAMILY.tolower()) of
    "windows": begin
      ; Python path must add the pythonXY
      version = PyUtils.GetEmbeddedPythonVersion()
      user_site_package = ([user_base,'Python' + version,'site-packages']).join(path_sep())
    end
    "unix": begin
      raw_version = PyUtils._GetSysPythonRawVersion()
      user_site_package = ([user_base  ,'lib','python' + raw_version,'site-packages']).join(path_sep())
    end
    else: begin
    end
  endcase

  ; Setting sys.path permanently requires to insert user_site_package in %PYTHONPATH%
  ; environment variable. We cannot use sitecustomize.py which is read-only
  ; We need to make sure that the user site package folder exists, so we create it.
  ; otherwise after installing a package, for some reason it's not found by Python.Import.
  if ~file_test(user_site_package, /DIRECTORY) then begin
    file_mkdir, user_site_package 
  endif
  PyUtils._AddPathToPythonPath, user_site_package
end


;-------------------------------------------------------------------------
;+
; :Description:
;   Defines the system path for the session that enables
;   the embedded Python installation.
;   The embedded installation path is IDLXX\bin\bin.x86_64\idl-pythonXY
;
; :Keywords:
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;-
pro PyUtils::_SetSystemPath, output=output
  compile_opt idl2, hidden, static

  on_error, 2
  output = ''
  ; If we already have Python loaded, just quietly return.
  newVersionDef = !false
  currentDlmVersion = PyUtils.GetCurrentPython()
  version = PyUtils.GetEmbeddedPythonVersion()

  if N_ELEMENTS(version) eq 0 then begin
    output = 'Python embedded installation not found in '+!DLM_PATH+'.'
    if ~arg_present(output) then begin
      message, output
    endif
    return
  endif
  if currentDlmVersion ne '' &&  currentdlmversion ne version then begin
    output = ["Python embedded version "+version+" cannot be activated when Python"+currentDlmVersion+" DLM is loaded.", $
      "Please restart the application."]
    if ~arg_present(output) then begin
      message, output.join('')
    endif
    return
  endif else if currentDlmVersion ne '' &&  ~PyUTILS._checkPythonPaths() then begin
    output = ["PYTHON"+currentDlmVersion+" DLM is loaded, but PATH environment variable does not contain the embedded Python version.",$
      "Please restart the application."]
    if ~arg_present(output) then begin
      message, output.join('')
    endif
    return
  endif else if currentdlmversion ne version && ~PyUTILS._checkPythonPaths() then begin
    newVersionDef = !true
  endif

  defsysv, '!IDLPYEMBEDDED',exists=exists

  if exists then return

  python_path = PyUtils._GetEmbeddedPythonFolder()
  if file_test(python_path, /DIRECTORY) eq 0 then begin
    output = python_path+" does not exist, reverting to system Python."
    if ~arg_present(output) then begin
      message, output.join(''), /NONAME, /INFORMATIONAL
    endif
    return
  endif

  PyUtils._AddPathToSystemPath, python_path
  user_base = PyUtils._GetUserBase()
  setenv, 'PYTHONUSERBASE=' + user_base
  python_exe = ''

  case (!version.OS_FAMILY.tolower()) of
    "windows": begin
      ; Python path must add the pythonXY
      python_exe = 'python.exe'
      embedded_site_package = ([!DLM_PATH,"idl-python"+version,'Lib','site-packages']).join(path_sep())
      PyUtils._AddPathToPythonPath, embedded_site_package
      user_bin_folder = ([user_base,'Python' + version,'Scripts']).join(path_sep())
    end
    "unix": begin
      python_exe = 'python'
      lib_dir = !DLM_PATH+path_sep()+"idl-python"+version+path_sep()+'lib'
      ld_paths = getenv("LD_LIBRARY_PATH")
      if ld_paths eq '' || ~ld_paths.contains(lib_dir) then begin
        setenv, "LD_LIBRARY_PATH="+lib_dir+(ld_paths ne '' ? path_sep(/SEARCH_PATH)+ld_paths : '')
      endif
      if !version.OS eq 'darwin' then begin
        ld_paths = getenv("DYLD_LIBRARY_PATH")
        if ld_paths eq '' || ~ld_paths.contains(lib_dir) then begin
          setenv, "DYLD_LIBRARY_PATH="+lib_dir+(ld_paths ne '' ? path_sep(/SEARCH_PATH)+ld_paths : '')
        endif
      endif
      user_bin_folder = ([user_base  ,'bin']).join(path_sep())
    end
    else: begin
    end
  endcase

  PyUtils._AddPathToSystemPath, user_bin_folder

  PyUtils._AddUserSitePackageToPythonPath

  ; for Python to IDL bridge we add '<IDL_DIR>/lib/bridges' to PYTHONPATH
  idl_python_bridges_path = !DIR+path_sep()+'lib'+path_sep()+'bridges'
  PyUtils._AddPathToPythonPath, idl_python_bridges_path

  ; This is harmless if this system var already exists
  defsysv, '!IDLPYEMBEDDED', 1

  output = `Python is embedded version ${version / 100.0, '(f4.2)'}.`
  if newVersionDef && ~ARG_PRESENT(output) then begin
    message, output, /info, /noname
  endif

end


;-------------------------------------------------------------------------
;+
; :Returns: String
;
;-
function PyUtils::_GetEmbeddedPythonName
  compile_opt idl2, static

  version = PyUtils.GetEmbeddedPythonVersion()
  return, "idl-python"+version
end


;-------------------------------------------------------------------------
;+
; :Returns: String
;
;-
function PyUtils::_GetEmbeddedPythonFolder
  compile_opt idl2, static

  python_path = !DLM_PATH + path_sep() + PyUtils._GetEmbeddedPythonName()
  if (!version.OS_FAMILY eq 'unix') then begin
    python_path += path_sep() + 'bin'
  endif
  return, python_path
end


;-------------------------------------------------------------------------
;+
; :Returns: any
;
; :Arguments:
;   cmd: bidirectional, required, any
;     Placeholder docs for argument, keyword, or property
;   output: bidirectional, required, any
;     Placeholder docs for argument, keyword, or property
;   error: bidirectional, required, any
;     Placeholder docs for argument, keyword, or property
;
; :Keywords:
;   stdout: bidirectional, optional, any
;     Placeholder docs for argument, keyword, or property
;-
function PyUtils::_PySpawn, cmd, output, error, STDOUT=stdout
  compile_opt idl2, static, hidden

;  python_path = PyUtils._GetEmbeddedPythonFolder()
;  cd, python_path, current=curdir
  if (keyword_set(stdout)) then begin
    spawn, cmd, EXIT_STATUS=exitstatus, /NOSHELL, unit = stdout
    ; file_poll_input can throw an error if the spawn finishes and
    ; stdout goes away before the timeout. So just quietly ignore.
    catch, err
    if (err eq 0) then begin
      while (file_poll_input(stdout, timeout = 15)) do begin
        output = ''
        readf, stdout, output
        message, output, /noname, /info
      endwhile
    endif else begin
      catch, /cancel
      message, /reset
    endelse
    free_lun, stdout
  endif else begin
    spawn, cmd, output, error, /NOSHELL, EXIT_STATUS=exitstatus
  endelse
;  cd, curdir
  return, exitstatus
end


;-------------------------------------------------------------------------
;+
; :Description:
;   Use FindPython to return the correct python executable
;   corresponding to the loaded Python DLM
;
; :Returns: any
;
; :Outputs:
;   The full Python executable path
;
;-
function PyUtils::_GetPythonExe
  compile_opt idl2, static

  PyUTILS.FindPython, output=output
  version = PyUTILS.GetCurrentPython()
  if version eq '' then begin
    version = PyUTILS.GetEmbeddedPythonVersion()
    ; If our embedded Python is missing, just return the first Python in the list.
    if (version eq '') then begin
      foreach key, output do begin
        if (n_elements(key) gt 0) then begin
          return, key[0, 'PATH']
        endif
      endforeach
    endif
  endif

  if output.haskey("PYTHON"+version) then begin
    ; Since the embedded version is at the head of the list, we always return the first one.
    return, ((output["PYTHON"+version])[0]).path
  endif

  python_exe = 'python'
  case (!version.OS_FAMILY.tolower()) of
    "windows": begin
      python_exe = !dlm_path+path_sep()+'idl-python'+version+path_sep()+'python.exe'
    end
    "unix": begin
      python_exe = !dlm_path+path_sep()+'idl-python'+version+path_sep()+'bin'+path_sep()+'python'
    end
    else: begin
    end
  endcase
  return, python_exe
end


;-------------------------------------------------------------------------
;+
; :Returns: A boolean indicating whether the embedded Python is loaded.
;
;-
function PyUtils::_EmbeddedPythonIsLoaded
  compile_opt idl2, static
  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then return, !false
  defsysv, '!IDLPYEMBEDDED', exists=isEmbedded
  return, isEmbedded
end


;---------------------------------------------------------------------------
;+
; Add the supplied path to either the PYTHONPATH (Windows)
; or the (DY)LD_LIBRARY_PATH (Unix)
; 
; :Arguments:
;   sysPrefix: input, required, String
;
;-
pro PyUtils::_AddSystemPaths, sysPrefix
  compile_opt idl2, hidden, static

  case (!version.OS_FAMILY.tolower()) of
    "windows": begin
      system_lib = ([sysPrefix,'Lib']).join(path_sep())
      PyUtils._AddPathToPythonPath, system_lib
      system_site_package = ([sysPrefix,'Lib','site-packages']).join(path_sep())
      PyUtils._AddPathToPythonPath, system_site_package
      system_dll_path = ([sysPrefix,'DLLs']).join(path_sep())
      PyUtils._AddPathToPythonPath, system_dll_path
    end
    "unix": begin
      lib_dir = sysPrefix+path_sep()+'lib'
      ld_paths = getenv("LD_LIBRARY_PATH")
      if ld_paths eq '' || ~ld_paths.contains(lib_dir) then begin
        setenv, "LD_LIBRARY_PATH="+lib_dir+(ld_paths ne '' ? path_sep(/SEARCH_PATH)+ld_paths : '')
      endif
      if !version.OS eq 'darwin' then begin
        ld_paths = getenv("DYLD_LIBRARY_PATH")
        if ld_paths eq '' || ~ld_paths.contains(lib_dir) then begin
          setenv, "DYLD_LIBRARY_PATH="+lib_dir+(ld_paths ne '' ? path_sep(/SEARCH_PATH)+ld_paths : '')
        endif
      endif
    end
    else: begin
    end
  endcase

  ; Make sure the sys_prefix is on PATH
  PyUTILS._addPathToSystemPath, sysPrefix
end


;-------------------------------------------------------------------------
;+
; :Description:
;   This method retrieves the list of installed Python packages.
;   This procedure only works with the IDL embedded Python.
;   Calling this while a different Python is loaded will result in an error.
;
; :Keywords:
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;   system: in, optional, Boolean
;     Set this keyword to only return the system packages.
;   user: in, optional, Boolean
;     Set this keyword to only return the user packages.
;-
pro PyUtils::PipList, system=system, user=user, output=output
  compile_opt idl2, static
  on_error, 2

  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    PyUtils.Load
  endif

  if ~PyUtils._EmbeddedPythonIsLoaded() then begin
    message, "Loaded Python DLM is not the embedded version."
  endif
  if (keyword_set(system) && keyword_set(user)) then begin
    message, 'Keyword conflict: SYSTEM and USER.'
  endif

  ; Make sure we have our user's Python package path (in .idl) on our path.
  PyUtils._AddUserSitePackageToPythonPath

  python_exe = PyUtils._GetPythonExe()
  cmd = [python_exe,'-m','pip','list']
  if KEYWORD_SET(user) then begin
    user_option = "--user"
    cmd = [cmd, user_option]
  endif

  exitStatus = PyUtils._PySpawn(cmd, output, error)

  if (exitStatus gt 0) then begin
    message, error
  endif

  if (keyword_set(system)) then begin
    ; if system required, we skip the user
    PyUTILS.PipList, /USER, output=userPackages
    userPackages = userPACKAGES.compress()
    foreach user_p, userPackages do begin
      if (user_p.startsWith('Package', /FOLD_CASE)) then continue
      output = output[where(output.compress() ne user_p, /null)]
    endforeach
  endif

  if ~ARG_PRESENT(output) then begin
    foreach out, output do begin
      message, out, /NONAME, /INFORMATIONAL, /NOPREFIX
    endforeach
  endif

end


;-------------------------------------------------------------------------
;+
; :Description:
;   This method installs a package in the user's .idl python folder
;   This procedure only works with the IDL embedded Python.
;   Calling this while a different Python is loaded will result in an error.
;
; :Arguments:
;   package: in, optional, any
;     A string giving the name of the package.
;
; :Keywords:
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;   upgrade: in, optional, Boolean
;     Set this keyword to upgrade the package if it is already installed.
;     If the package is not already installed, then this keyword is ignored
;     and the latest version is installed.   
;   version: in, optional, String
;     Set this keyword to a string giving the version to install.
;     By default, this method installs the latest version.
;-
pro PyUtils::PipInstall, package, OUTPUT=output, UPGRADE=upgrade, VERSION=version
  compile_opt idl2, static
  on_error, 2

  output = ''
  error = ''

  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    PyUtils.Load
  endif

  if ~PyUtils._EmbeddedPythonIsLoaded() then begin
    message, "Loaded Python DLM is not the embedded version."
  endif
  if package eq !NULL then begin
    message, 'Package is required.'
  endif

  if (PyUtils._EmbeddedPythonIsLoaded()) then begin
    PyUtils._SetSystemPath, OUTPUT=output
  endif

  if output[0] ne '' then begin
    foreach out, output do begin
      message, out, /NONAME, /INFORMATIONAL
    endforeach
  endif

  user_base = PyUtils._GetUserBase()
  setenv, 'PYTHONUSERBASE='+user_base

  python_exe = PyUTILS._GetPythonExe()
  if (!version.os_family eq 'Windows') then begin
    python_exe = '"' + python_exe + '"'
  endif
  versionStr = isa(version) ? "==" + version : ""
  cmd = [python_exe, '-m', 'pip', 'install', package + versionStr, $
    '--user', '--index=https://pypi.python.org/simple']

  if KEYWORD_SET(upgrade) then begin
    upgrade_opt = "--upgrade"
    cmd = [cmd,upgrade_opt]
  endif

  exitStatus = PyUtils._PySpawn(cmd, output, error, stdout = ~arg_present(output))

  if (exitStatus gt 0 && error[0] ne '') then begin
    message, error
  endif

  if (error[0] ne '') then begin
    output = [output, error]
  endif else begin
    success_mesg = 'Package '+package+' successfully installed in '+user_base+'.'
    if total(output.contains("Requirement already satisfied: "+package)) eq 1 then begin
      success_mesg = 'Package '+package+' already installed in '+user_base+'.'
    endif
    output = [output, success_mesg]
  endelse
end


;-------------------------------------------------------------------------
;+
; :Description:
;   This method uninstalls a Python package installed by the user.
;   This procedure only works with the IDL embedded Python.
;   Calling this while a different Python is loaded will result in an error.
;
; :Arguments:
;   package: in, optional, String
;     A string giving the name of the package.
;
; :Keywords:
;   all: in, optional, Boolean
;     If this keyword is set then all user packages will be removed.
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;-
pro PyUtils::PipUninstall, packageIn, ALL=all, OUTPUT=output
  compile_opt idl2, static
  on_error, 2

  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    PyUtils.Load
  endif

  if ~PyUtils._EmbeddedPythonIsLoaded() then begin
    message, "Loaded Python DLM is not the embedded version."
  endif

  if (n_elements(packageIn) eq 0 && ~keyword_set(all)) then begin
    message, 'No package was specified.'
  endif

  PyUtils._SetSystemPath, OUTPUT=output
  user_base = PyUtils._GetUserBase()
  setenv, 'PYTHONUSERBASE=' + user_base

  if KEYWORD_SET(all) then begin
    cmd = [PyUTILS._GetPythonExe(),'-m','pip','freeze','--user']
    exitStatus = PyUtils._PySpawn(cmd, user_packages, error)

    if (exitStatus gt 0 && error[0] ne '') then begin
      message, error
    endif

    if user_packages[0] eq '' then begin
      output = 'No user packages found.'
      if ~ARG_PRESENT(output) then begin
        message, output, /noname, /informational
      endif
      return
    endif
    output = []
    foreach package_and_version, user_packages do begin
      endsub = package_and_version.indexOf("==")-1
      package = package_and_version.substring(0, endsub)
      if (arg_present(output)) then begin
        PyUtils.PipUninstall, package, output=output_single
        if ((strjoin(output_single)).contains('error', /fold_case)) then begin
          message, 'Error uninstalling ' + package + ', skipping', /info
        endif else begin
          output = [output, output_single]
        endelse
      endif else begin
        PyUtils.PipUninstall, package
      endelse
    endforeach
    return
  endif

  package = packageIn

  ; Packages installed from a local file/url (rather than a standard repo)
  ; may have an @ in the piplist name. Strip this out to get the package name.
  if (package.contains(' @')) then begin
    package = (package.split('@'))[0]
  endif

  output = []
  if package ne !NULL then begin
    ; Do not allow user to uninstall IDL system packages like numpy
    PyUtils.PipList, /system, OUTPUT=syspackages
    sysPackages = strlowcase(sysPACKAGES)
    sysPackages = (sysPackages.extract('.* ')).trim()
    if (max(sysPackages eq strlowcase(package)) eq 1) then begin
      message, 'Package ' + package + ' is a system package. Cannot uninstall.'
    endif
  endif

  ; Remove package's imported modules
  ; TODO : find a simple way to avoid using python_define routines

  if (PyUtils._EmbeddedPythonIsLoaded()) then begin
    Python.package_name = package
    >>>import sys
    >>>loaded_package_modules = [key for key, value in sys.modules.items() if package_name in str(value)]
    foreach pkey, Python.loaded_package_modules do begin
      Python.key = pkey
      >>>a = sys.modules.pop(key)
    endforeach
  endif

  ; change folder to python
  python_exe = PyUtils._GetPythonExe()
  cmd = [python_exe, '-m', 'pip', 'uninstall', '-y', package]
  exitStatus = PyUtils._PySpawn(cmd, output, error, stdout = ~arg_present(output))

  if (exitStatus gt 0) then begin
    if ~arg_present(output) then begin
      message, output + (isa(error) ? error : '')
    endif
  endif


  if (isa(error) && error[0] ne '') then begin
    output = [output, error]
  endif else begin
    mesg = 'Package '+package+' successfully removed from '+user_base+'.'
    output = [output, mesg]
  endelse

end


;-------------------------------------------------------------------------
;+
; :Description:
;   Retrieve and display information about an installed package.
;   This procedure only works with the IDL embedded Python.
;   Calling this while a different Python is loaded will result in an error.
;
; :Arguments:
;   package: in, required, String
;     A string giving the name of the package.
;
; :Keywords:
;   files: in, optional, Boolean
;     Set this keyword to output a list of the files within the package.
;   output: out, optional, String
;     By default, this method prints out the information
;     to the IDL console. If the OUTPUT keyword is set to a named variable
;     then the output will be returned in that variable instead.
;   verbose: in, optional, Boolean
;     Set this keyword to produce additional output.
;-
pro PyUtils::PipShow, package, OUTPUT=output, FILES=files, VERBOSE=verbose
  compile_opt idl2, static
  on_error, 2

  defsysv, '!IDLPYLOADED', exists=isLoaded
  if (~isLoaded) then begin
    PyUtils.Load
  endif

  if ~PyUtils._EmbeddedPythonIsLoaded() then begin
    message, "Loaded Python DLM is not the embedded version."
  endif

  ; Make sure we have our user's Python package path (in .idl) on our path.
  PyUtils._AddUserSitePackageToPythonPath

  python_exe = PyUtils._GetPythonExe()

  cmd = [python_exe, '-m', 'pip', 'show', package]
  if keyword_set(files) then begin
    file_option = '--files'
    cmd = [cmd, file_option]
  endif

  if keyword_set(verbose) then begin
    verbose_option = '--verbose'
    cmd = [cmd, verbose_option]
  endif

  exitStatus = PyUtils._PySpawn(cmd, output, error)

  if (exitStatus gt 0) then begin
    message, error
  endif

  if ~ARG_PRESENT(output) then begin
    foreach out, output do begin
      message, out, /NONAME, /INFORMATIONAL, /NOPREFIX
    endforeach
  endif

end


;---------------------------------------------------------------------------
;+
; The PyUtils::Load procedure is used to manully load the IDL->Python bridge.
; By default the embedded Python is loaded. If a specific version is required,
; the version can be passed as an argument.
; 
; Tip: If you want to use the embedded Python, you do not need to call this
; procedure. The embedded Python will automatically be loaded when the
; first Python command is used.
;
; :Arguments:
;   version: input, optional, string
;     Set this argument to a scalar string giving the name of the Python version.
;     This string should have the format "pythonXY" where X and Y are the major
;     and minor version numbers of the Python installation. For example, to load
;     Python 3.13, set version to "python313".
;
;-
pro PyUtils::Load, version

  compile_opt idl2, hidden, static
  ON_ERROR, 2

  defsysv, '!IDLPYLOADED', exists=exists
  if ~exists then defsysv, '!IDLPYLOADED', ''
  if !IDLPYLOADED then begin
    if (ISA(version)) then begin
      if version.startswith('python', /FOLD_CASE) then begin
        short_version = version.replace('python','', /FOLD_CASE)
      endif
      if !IDLPYLOADED ne short_version then begin
        message, 'Python version ' + !IDLPYLOADED + $
          ' already loaded. Cannot load version ' + short_version+'.'
      endif
      return
    endif
    return
  endif

  ;  if (ISA(version)) then begin
  ;    DLM_LOAD, version
  ;    !IDLPYLOADED = version.replace('python', '', /FOLD_CASE)
  ;  endif

  ; define paths for embedded Python
  defsysv, '!IDLPYEMBEDDED', exists=exists
  if ~exists && ~ISA(version) then begin
    PyUtils._SetSystemPath, output = output
    if output[0] ne '' then begin
      message, output, /NONAME, /INFORMATIONAL
    endif
  endif else if ISA(version) then begin
    DLM_LOAD, version
    !IDLPYLOADED = version.replace('python', '', /FOLD_CASE)
  endif


  python_exe = PyUTILS._GetPythonExe()

  ; Determine which Python version to load.
  SPAWN, [python_exe, '--version'], stdout, stderr, /noshell, EXIT_STATUS=exitStatus

  if exitStatus ne 0 then begin
    message, python_exe+' --version failed'+stderr
  endif

  ; stdout or stderr should have the form "Python X.Y.Z ..."
  ; We need to convert this to "PythonXY"
  the_version = stdout.StartsWith('Python') ? stdout : stderr
  the_version = (the_version.Extract('[0-9]\.[0-9]+')).Replace('.','')

  if (the_version eq '') then begin
    MESSAGE, 'Unable to determine Python version: Make sure Python is on your system path.'
  endif

  ; if dlm_load has been called
  if (!IDLPYLOADED ne '' && !IDLPYLOADED ne the_version) then begin
    ; find all python executables in PATH
    where_cmd = 'where'
    if !version.OS_FAMILY ne 'Windows' then begin
      where_cmd = 'which'
    endif
    spawn, [where_cmd, python_exe], python_exe_list, stderr, /noshell
    right_version_path = !NULL
    index = 0
    while right_version_path eq !NULL && index lt n_elements(python_exe_list) do begin
      python_exe = python_exe_list[index]
      SPAWN, [python_exe, '--version'], a_version, stderr, /noshell, exit_status=exitst
      if exitst eq 0 && stderr[0] eq '' then begin
        a_version = a_version.StartsWith('Python') ? a_version : stderr
        a_version = (a_version.Extract('[0-9]\.[0-9]+')).Replace('.','')
        if a_version eq !IDLPYLOADED then begin
          ; push the right Python path ahead of PATH
          paths = (getenv("PATH")).split(path_sep(/SEARCH_PATH))
          paths_list = List(paths, /EXTRACT)
          index_right_version = where(paths_list eq FILE_DIRNAME(python_exe_list[index]))
          paths_list.move, index_right_version, 0
          new_paths_array = paths_list.ToArray()
          setenv, "PATH="+new_paths_array.join(path_sep(/SEARCH_PATH))
          the_version = !IDLPYLOADED
          right_version_path = python_exe_list[index]
        endif
      endif
      index++
    endwhile
    if right_version_path eq !NULL then begin
      message, 'Could not find python executable for loaded DLM version '+!IDLPYLOADED+'.'
    endif
  endif

  HELP, 'Python' + the_version, /DLM, OUTPUT=dlmOutput

  ;Set up Windows/Unix specific python code. Windows should be wrapped in
  ; quotes because it is executed as a dos command even though we set /noshell
  numpyCmd = 'import numpy;print(numpy.version.version)'
  ;  sysExeCmd = 'import sys; print(sys.executable)'
  sysPrefixCmd = 'import sys; print(sys.prefix)'
  basePrefixCmd = 'import sys; print(sys.base_prefix)'

  if (!VERSION.OS_FAMILY eq 'Windows') then begin
    numpyCmd = '"' + numpyCmd + '"'
    ;    sysExeCmd = '"' + sysExeCmd + '"'
    sysPrefixCmd = '"' + sysPrefixCmd + '"'
    basePrefixCmd = '"' + basePrefixCmd + '"'
  endif

  ;  Get the sys.executable so we can print what python we are using
  ;  sys.executable can be wrong sometimes, nothing to do with python
  ;  SPAWN, [python_exe, '-c', sysExeCmd], sysExe, stderr, /noshell, EXIT_STATUS=exitStatus
  ;  if (exitStatus gt 0) then begin
  ;    MESSAGE, 'Failed to get python sys.executable'
  ;  endif

  ; Get the sys.prefix so we can match the python home
  ; Python paths MUST be defined before trying to check numpy!
  SPAWN, [python_exe, '-c', sysPrefixCmd], sysPrefix, stderr, /noshell, EXIT_STATUS=exitStatus
  PyUTILS._AddSystemPaths, sysPrefix

  ; If we are in a virtual environment, the environment won't have all of the
  ; needed libraries. In this case, add the base (parent) folders to our path.
  SPAWN, [python_exe, '-c', basePrefixCmd], basePrefix, stderr, /noshell, EXIT_STATUS=exitStatus
  if (basePrefix ne sysPrefix) then begin
    PyUTILS._AddSystemPaths, basePrefix
  endif

  ;  if (dlmOutput[0].Contains('Unknown')) then begin
  ;    MESSAGE, sysExe + ' is not a supported python version. '
  ;  endif

  ; Verify numpy is installed.  IDL will crash with no error if it is not!
  SPAWN,[python_exe, '-c', numpyCmd] , stdout, numpyStderr, /NOSHELL, EXIT_STATUS=exitStatus
  np = (the_version ge 311) ? '2.' : '1.'
  if (exitStatus ne 0 || ~stdout.startsWith(np)) then begin
    msg = (exitStatus ne 0) ? 'Python numpy is not installed' : $
      `Incompatible numpy ${stdout[0]}: must be version ${np}*`
      MESSAGE, msg + '. numpy can be installed by running ' + $
      `"python -m pip install numpy==${np}*" or "conda install numpy==${np}*".`
  endif


  ; For Python to IDL bridge add <!DIR>\lib\bridges to PYTHONPATH
  idl_python_bridges_path = !DIR+path_sep()+'lib'+path_sep()+'bridges'
  PyUtils._AddPathToPythonPath, idl_python_bridges_path

  DLM_LOAD, 'Python' + the_version

  ; Our embedded python initializes using Py_Initialize.
  ; Before this call though it calls Py_SetProgramName and Py_SetPythonHome
  ; so the correct python distribution is used.  We use PYTHON_INIT
  ; to pass in the program name and python home strings.
  ; Changed to python_exe because sys.executable can be wrong sometimes why?

  PYTHON_INIT, python_exe, sysPrefix

  !IDLPYLOADED = the_version
end


;-------------------------------------------------------------------------
;+
; :PyUtils:
;
;-
pro PyUtils__Define
  compile_opt idl2, hidden
  _pyutils = {PyUtils, inherits IDL_Object}
  STRUCT_HIDE, _pyutils
end
