; $Id$
;###############################################################################
;
; NAME:
;   wd_SPINSPSDmaskPixels
;
; PURPOSE:
;
; CATEGORY:
;   SPINS Data Reduction
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   May, 2002
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################



;===============================================================================
; wd_SPINSPSDmaskPixels_updMskList
;
; PURPOSE:
;   Maintains a list of detectors that are to be masked from the final
;   reduced data.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   detnos [in] - detector number whose state has changed masked<->unmasked
;
; KEYWORDS:
;
pro wd_SPINSPSDmaskPixels_updMskList, sPtr, detnos
compile_opt idl2

; grap the current mask list
if  ptr_valid((*sPtr).mskListPtr) then list = (*(*sPtr).mskListPtr)
ptr_free,(*sPtr).mskListPtr     ; clean (empty) the current list

lsize = n_elements(list)
if (lsize gt 0) then begin      ; list exists
    result = where(list eq detnos)
    if (result[0] ge 0) then begin ; detnos is in the list already - remove it
        lsize = lsize - 1
        if (result[0] le lsize-1) then $
          for i = result[0], lsize-1 do list[i] = list[i+1]
    endif else begin            ; detnos not in list - add it
        lsize = lsize + 1
        list = [list,detnos]
    endelse
endif else begin                ; no list yet - create one
    lsize = 1
    list = [detnos]
endelse 

if (lsize ge (*sPtr).ndets) then begin ; removing all detectors!!!
    message = ['Attempting to mask all detectors.', $
               'No output can be generated!']
    void = dialog_message(message,/information,dialog_parent=(*sPtr).wTLB)
endif


; update state variable
lstring = ''

if (lsize gt 0) then begin
    newlist = list[0:lsize-1]
    newlist = newlist[sort(newlist)] ; sort in ascending order
    (*sPtr).mskListPtr = ptr_new(newlist)
    if (lsize eq 1) then begin
      lstring =   strtrim(string(newlist[0]),2)
    endif else begin
        beg = newlist[0]
        range = 0
        for i = 1,lsize-1 do begin
            if (newlist[i] eq newlist[i-1]+1) then begin
                range=1
                continue
            endif
            if (range) then begin
                lstring = lstring+strtrim(string(beg),2)+'-'+strtrim(string(newlist[i-1]),2)+' '
                beg = newlist[i]
                range = 0
            endif else begin
                lstring = lstring+strtrim(string(newlist[i-1]),2)+' '
                beg = newlist[i]
            endelse
        endfor
        if (range) then begin
            lstring = lstring+strtrim(string(beg),2)+'-'+strtrim(string(newlist[i-1]),2)+' '
        endif else begin
            lstring = lstring+strtrim(string(newlist[i-1]),2)+' '
        endelse
    endelse  
endif 

; replace all ' ' with ','
lstring = strtrim(lstring,2)
while ((pos = stregex(lstring,' ')) ne -1) do strput, lstring,',',pos

; update list in mask widget
widget_control, (*sPtr).wMskList, set_value=lstring

end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_SPINSPSDmaskPixels_plotMskList
;
; PURPOSE:
;   Mark all masked detectors with grey bars in the display window.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;
pro wd_SPINSPSDmaskPixels_plotMskList, sPtr
compile_opt idl2

if (~ptr_valid((*sPtr).mskListPtr)) then return

list = (*(*sPtr).mskListPtr)

lsize = n_elements(list)
ymin = !y.crange[0]  &  ymax = !y.crange[1]
yvals = [ymin,ymin,ymax,ymax]
for i = 0, lsize-1 do begin
    xmin = list[i]-0.50  &  xmax=list[i]+0.50
    xvals = [xmin,xmax,xmax,xmin]
    polyfill,xvals,yvals
endfor


end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_SPINSPSDmaskPixels_plotClick
;
; PURPOSE:
;   Event handler for button events dispatched from the plot
;   window. When detector display mode is on (ie samp, bkgd views)
;   then deal with left-mouse clicks to mask/unmask dets else
;   zoom/reset display. Mark all masked detectors with grey bars in
;   the display window.
;
; PARAMETERS:
;   event [in] - event structure
;
pro wd_SPINSPSDmaskPixels_plotClick, event
compile_opt idl2

widget_control, event.top, get_uvalue=sPtr

if (event.press eq 1) then begin ; left mb press
    (*sPtr).zp1 = [event.x,event.y] ; record mouse location
    (*sPtr).mouseType = event.press ; need to know afterwards that a left mb was pressed
    
    ;; draw a vertical line at the mouse location
    ymin = !y.crange[0]  &  ymax = !y.crange[1]
    xydat = convert_coord(event.x,event.y,/device,/to_data)
    plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
    (*sPtr).zp2 = [-1000.0,-1.0]   ; used to determine if motion event occurs b/c zp2 will be different
endif 

;; if motion event and left mb was just pressed
if ((event.type eq 2) and ((*sPtr).mouseType eq 1) ) then begin
    (*sPtr).zp2 = [event.x, event.y]
    zp1 = (*sPtr).zp1
    zp2 = (*sPtr).zp2
    ;; copy winPix to winVis and draw vertical lines at current mouse 
    ;; location and where left mb was initially pressed 
    device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix]
    ymin = !y.crange[0]  &  ymax = !y.crange[1]
    xydat = convert_coord(zp1[0],zp1[1],/device,/to_data)
    plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
    xydat = convert_coord(zp2[0],zp2[1],/device,/to_data)
    plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
endif

if (event.release eq 1) then begin ; left mb released

    lolim = (*sPtR).lolim
    hilim = (*sPtr).hilim
    if ( ((*sPtr).zp2)[0] ne -1000.0) then begin
        zp1 = (*sPtr).zp1   ; initial xvalue for left mouse press
        zp2 = (*sPtr).zp2   ; final mouse xvalue after motion event
        x1 = min([zp1[0],zp2[0]],max=x2)
        y = zp1[1]
        d1 = 0
        d2 = 0
        ;; if the x coord of mouse occurs outside the detector range, ignore it
        xydat = convert_coord(x1,y,/device,/to_data)
        d1 = round(xydat[0])
        xydat = convert_coord(x2,y,/device,/to_data)
        d2 = round(xydat[0])
        for detnos = d1,d2 do begin
            if ( (detnos lt lolim) or (detnos gt hilim) ) then continue
             
            ;; add/remove det from mask list if it isn't/is there followed by an
            ;; update to the mask widget
            wd_SPINSPSDmaskPixels_updMskList, sPtr,detnos
        endfor 
    endif else begin
        zp1 = (*sPtr).zp1   ; initial xvalue for left mouse press
        x1 = zp1[0]
        y = zp1[1]
        xydat = convert_coord(x1,y,/device,/to_data)
        detnos = round(xydat[0])
        ;; add/remove det from mask list if it isn't/is there followed by an
        ;; update to the mask widget
        if ( (detnos ge lolim) and (detnos le hilim) ) then wd_SPINSPSDmaskPixels_updMskList, sPtr,detnos
    endelse 
    ;; refresh plot window 
    wd_SPINSPSDmaskPixels_plotData, sPtr
    (*sPtr).zp2 = [0.0,0.0]
    (*sPtr).mouseType = 0   ; reset mouseType as prev press been dealt with
 endif 
    

end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_SPINSPSDmaskPixels_setSliderValue
;
; PURPOSE:
;   Slider values are continuous. However, the energy values that the
;   slider represents only have certain values given by the energy
;   array. This routine ensures that the slider only sets/displays
;   valid energy values that are present in the energy array. This
;   routine must be called whenever the data being displayed changes.
;
; PARAMETERS:
;   wid [in] - widget id of slider
;
;   energy [in] - energy transfer array
;
;   dvalue [in] - the energy value initially selected using the
;                 slider. This is used to locate the closest valid
;                 energy.
;
; KEYWORDS:
;
;pro wd_SPINSPSDmaskPixels_setSliderValue, wid, energy, dvalue
;compile_opt idl2
;
;; if input parameter dvalue missing, use current slider value instead
;if (n_elements(dvalue) le 0) then $
;  widget_control, wid, get_value=dvalue ;dvalue = fslider_get_value(wid) 
;
;; now locate the closest entry to dvalue in the energy array
;index = locateIndex(energy,dvalue)
;
;; update slider value with this 'proper' energy 
;widget_control, wid, set_value=energy[index]
;
;; save this index in the uvalue of the widget
;widget_control, wid, set_uvalue=index
;
;end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_SPINSPSDmaskPixels_syncSlider
;
; PURPOSE:
;   Ensure that the energy range displayed in the slider matches the
;   currently selected dataset.
;
; PARAMETERS:
;   wid [in] - widget id of slider
;
;   energy [in] - energy transfer array
;
;   reset [in] - flag. If set, the slider value is set to its minimum.
;
; KEYWORDS:
;
;pro wd_SPINSPSDmaskPixels_syncSlider, wid, energy, reset
;compile_opt idl2
;
;dmin = min(energy, max=dmax)
;if (keyword_set(reset)) then begin
;    ;; set slider range to be same as energy array. Initialise it to its min value
;    dvalue = dmin
;endif else begin
;    widget_control, wid, get_value=dvalue 
;    if (dvalue lt dmin) then dvalue = dmin
;    if (dvalue gt dmax) then dvalue = dmax
;endelse
;
;range = [dvalue,dmin,dmax]
;widget_control, wid, set_value = range
;wd_SPINSPSDmaskPixels_setSliderValue, wid, energy, davlue
;
;end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_sliderEvent
;
; PURPOSE:
;   Event handler: for energy cut slider widget.
;
; PARAMETERS:
;   event [in] - event structure.
;
;pro wd_SPINSPSDmaskPixels_sliderEvent, event
;compile_opt idl2
;
;widget_control,event.top,get_uvalue=sPtr
;
;energy = (*sPtr).energy
;
;; now alter the slider value so it is a valid value within the energy array.
;wd_SPINSPSDmaskPixels_setSliderValue,(*sPtr).wSlider, energy
;
;; finally, update the plot.
;wd_SPINSPSDmaskPixels_plotdata, sPtr
;
;end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_SPINSPSDmaskPixels_plotData
;
; PURPOSE:
;   Plot data in the draw widget.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;
; KEYWORDS:
;
pro wd_SPINSPSDmaskPixels_plotData, sPtr
compile_opt idl2

oData = (*sPtr).oData
detNos = (*sPtr).detNos
nDets = (*sPtr).nDets
xtitle = 'PSD Channel Nos'
if ((*sPtr).sourceIndex eq 0) then begin
  ytitle = 'Integrated Counts'
  detCnts = (*sPtr).counts
  sigma = (*sPtr).error
endif else begin
  ytitle = 'Channel Efficiencies'
  detCnts = (*sPtr).chEff
  sigma = (*sPtr).chEffError 
endelse

;; get index of energy value set by slider
;widget_control, (*sPtr).wSlider, get_uvalue=eIndex
  
;; set winPix as destination window initially
wset, (*sPtr).winPix
  
;; plot detector counts vs detector number
plot,detNos,detCnts, $
     xstyle=1,/ynozero,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
     xtitle=xtitle,subtitle='',ytitle=ytitle,title='', $
     xrange=[0,ndets+2]
;; plot errors
;sigma = detCnts
;index = where(sigma le 0.0, zeroCnts)
;if (zeroCnts gt 0) then sigma[index] = 1.0
;sigma = sqrt(sigma)
errplot, detNos, detCnts-sigma, detCnts+sigma ; plot errors

;; display masked detectors
wd_SPINSPSDmaskPixels_plotMskList, sPtr

wset, (*sPtr).winVis    ; set winVis as destination window
device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis


end


;===============================================================================
; fans_Help
;
; PURPOSE:
;   Event handler. Display FANS data reduction manual.
;
; PARAMETERS:
;   event [in] - event structure.
;
pro wd_SPINSPSDmaskPixels_Help,event
compile_opt idl2

widget_control,event.top,get_uvalue = pState
void = launch_help(!DAVE_PDFHELP_DIR+'spinspsdmaskPixels_manual.pdf',tlb = event.top)
return
end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_resize
;
; PURPOSE:
;   Event handler. Resize the tlb widget.
;
; PARAMETERS:
;   event [in] - event structure
;
pro wd_SPINSPSDmaskPixels_resize, event
compile_opt idl2

if dave_set_focus(event) then return

widget_control, event.top, get_uvalue=sPtr

; new plot window width is maximum of the two r-values
(*sPtr).winWidth = (event.x - (*sPtr).restOfWidth) > (*sPtr).minXSize

; new plot window height is maximum of two r-values
(*sPtr).winHeight = (event.y - (*sPtr).restOfHeight) > (*sPtr).minYSize


; Delete and recreate pixmap window using new dimensions
wdelete, (*sPtr).winPix
window,/free,/pixmap,xsize=(*sPtr).winWidth,ysize=(*sPtr).winHeight
(*sPtr).winPix = !d.window
; Also resize visible window using new dimensions 
widget_control, (*sPtr).winVisID, xsize=(*sPtr).winWidth, ysize=(*sPtr).winHeight

widget_control, (*sPtr).wTLB, /realize

wd_SPINSPSDmaskPixels_plotData, sPtr

;wd_SPINSPSDmaskPixels_syncSlider,(*sPtr).wSlider,(*sPtr).energy, 0


end



;===============================================================================
; wd_SPINSPSDmaskPixels_cleanup
;
; PURPOSE:
;   Cleanup routine called when the main widget is destroyed
;
; PARAMETERS:
;   wTLB [in] - the widget id of the top-level-base that is being
;                killed.
;
pro wd_SPINSPSDmaskPixels_cleanup, wTLB
compile_opt idl2

widget_control, wTLB, get_uvalue=sPtr

device, decomposed = (*sPtr).olddc
tvlct,(*sPtr).oldrgb.r,(*sPtr).oldrgb.g,(*sPtr).oldrgb.b
wdelete, (*sPtr).winPix         ; delete pixmap window


;
;; deal with the remainder of the state pointer
ptr_free, (*sPtr).mskListPtr, sPtr

end
;-------------------------------------------------------------------------------


;===============================================================================
; wd_SPINSPSDmaskPixels_event
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
pro wd_SPINSPSDmaskPixels_event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_SPINSPSDmaskPixels_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control, event.top, get_uvalue=sPtr
uname = widget_info(event.id,/uname)
sname = tag_names(event,/structure_name)

;; resize event
if (strcmp(sname,'WIDGET_BASE')) then begin

   return
endif

;;
case strupcase(uname) of
   'DISCARD': begin
      widget_control, event.top, /destroy
   end
   
   'ACCEPT': begin
      widget_control, (*sPtr).wMskList, get_value=maskedDets
      print,'maskeddets:',maskedDets

      ;; this is a USERDEF property type hence need to set both the
      ;; property value and the userdef attribute (the userdef attribute
      ;; determines what gets displayed in propertysheet!)
      (*sPtr).tool->SetProperty, maskedDets=maskedDets
      (*sPtr).tool->SetPropertyAttribute, 'MASKEDDETS', userdef=maskedDets

      widget_control, event.top, /destroy
   end
   
   'DRAW': begin
      wd_SPINSPSDmaskPixels_plotClick, event
   end
   
   'INTENSITYSOURCE': begin
      (*sPtr).sourceIndex = event.index
      wd_BT7PSDmaskchannels_plotData, sPtr
   end
 
;   'ESLIDER': begin
;      wd_SPINSPSDmaskPixels_sliderEvent, event
;   end
   
   'HELP': wd_SPINSPSDmaskPixels_Help, event
   

   else:
endcase

end


;===============================================================================
;+
; wd_SPINSPSDmaskPixels
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
;   group_leader [in] - the ID of the parent widget from which this
;                       dialog was launched.
;
;   dataDir [in] - The default location for raw data files
;
;   workDir [in] - The default location for saving reduced files
;
;   DAVETool [in] - Reference to the DAVE tool object
;-
pro wd_SPINSPSDmaskPixels, oUI, oData, calibParamsIsSet, chEff, _EXTRA=etc
compile_opt idl2

if (~obj_valid(oUI)) then return
if (~obj_valid(oData)) then return

if (n_elements(calibParamsIsSet) eq 0) then calibParamsIsSet = 0

tvlct,rorig,gorig,borig,/get
oldrgb = {r:rorig,g:gorig,b:borig}
device, get_decomposed = olddc
device,decomposed = 0
colors = hfbs_GetColor(/Load, Start=1) ; custom (16) color table

oUI->GetProperty, group_leader = group_leader
oTool = oUI->GetTool()
oTool->GetProperty, maskedDets=maskedDetsString
if (~strcmp(maskedDetsString,'')) then $
   void = compactstring2intvector(maskedDetsString, maskedList)

; define container base widgets; make it resizeable
wTLB = widget_base(group_leader=group_leader,/col,title='SPINS PSD Data Reduction - Mask Detector Pixels' $
                  ,uname='SPINSMASKDETECTOR',/tlb_size_events,/modal)

wCB = widget_base(wTLB,/col,frame=1);,/align_center)
wCBa = widget_base(wCB,/row,/align_center)
wCBe = widget_base(wCB,/row,/align_center)
wCBb = widget_base(wCB,/row,/align_center)
wCBc = widget_base(wCB,/col,/align_center,/frame)
wCBd = widget_base(wCB,/row,/align_center,/frame)

xSize = 800
ySize = 350
wVis = widget_draw(wCBb,xsize=xsize,ysize=ysize,/button_events,/motion_events,uname='Draw')
;wSlider = cw_fslider(wCBc,title='Energy Transfer',xsize=xSize,uname='eslider', $
;                       min=0.0,max=220.0,format='(F6.2)',uvalue=0)
;widget_control, wSlider, sensitive=0
void = cw_coloredlabel(wCBa,value='Pixels currently masked:',foreground_color=[255,0,0])
;voID = widget_label(wCBa,value='Masked detector list:',/dynamic_resize)
vInfo = widget_info(voID,/geometry)
wMskList = widget_text(wCBa,value=maskedDetsString,scr_xsize=0.98*(xSize-vInfo.xsize))

void = cw_coloredlabel(wCBe,value='Get Pixel intensities from: ',foreground_color=[255,0,0])
values = ['First loaded sample dataset','Pixel efficiencies from calibration data']
wSource = widget_droplist(wCBe,uname='IntensitySource',value=values,scr_xsize=0.98*(xSize-vInfo.xsize) $
                          ,sensitive=calibParamsIsSet)
sourceIndex = 0  ; ie 'First loaded sample dataset'

text = strarr(2)
text[0] = 'Vertical axis shows PSD Pixel number. On plot, use left mouse to select (click/drag) Pixels to mask.'
text[1] = 'Masked Pixels are grayed in plot and shown in masked list. When satisfied accept or discard changes.'
;for i=0,1 do void = widget_label(wCBc,value=text[i])
for i=0,1 do void = cw_coloredlabel(wCBc,value=text[i],foreground_color=[0,0,255])

; 'Help' button
;void = widget_button(wCBd,value='Manual',uname='man',sensitive=0)
; 'File' button
void = widget_button(wCBd,value='Accept Changes',uname='Accept',xoffset=100)
void = widget_button(wCBd,value='Discard Changes',uname='Discard',xoffset=100)


; realise widgets
widget_control, wTLB, /realize

tlbInfo = widget_info(wTLB,/geometry)
restOfHeight = tlbInfo.scr_ysize - ySize
restOfWidth = tlbInfo.scr_xsize - xSize

; create a pixmap window with same size as visible window
; and note its number
window,/free,/pixmap,xsize=xsize,ysize=ysize
winPix = !d.window 

; get the visible window nos and set it as default plot window;
widget_control, wVis, get_value=winVis;,sensitive=1
wset, winVis

if (~oData->GetMetaData('OriginalCounts',counts)) then begin
   oZ = oData->GetByName('Counts')
   status = oZ->GetMetaData('OriginalCounts',counts)
endif

;status = oZ->GetData(counts)
dims = size(Counts,/dimensions)
nDets = dims[0]
detNos = indgen(nDets) + 1
lolim = detNos[0] - 0.5
hilim = detNos[nDets-1] + 0.49
if (size(Counts,/n_dimensions) eq 2) then $
  counts = total(temporary(counts),2) ; sum all data points for each Pixel
error = sqrt(counts)
if (~calibParamsIsSet) then chEff = fltarr(nDets) + 1.0
chEffError = 0.05*chEff

; define state structure
;energy = (*(*dataPtr).energyPtr)
mskListPtr = (n_elements(maskedList) gt 0)? ptr_new(maskedList) : ptr_new()
state = {wTLB:wTLB $
        ,wMskList:wMskList $ ; Masked detector list widget ID
        ,mskListPtr:mskListPtr $ ; ptr to array of detectors to be masked
        ,wVis:wVis $   ; visible plot window widget ID
        ,winVis:winVis $       ; visible window number
        ,winPix:winPix $       ; pixmap window number
        ,winWidth:xsize $      ; visible/pixmap window width
        ,winHeight:ysize $     ; visible/pixmap window height
        ,restOfHeight:restOfHeight $ ; height of tlb - plot window height
        ,restofWidth:restofWidth $   ; width of tlb - plot window width
        ,minXSize:0.5*xSize $      ; minimum width for plot window: 0.5 of initial value 
        ,minYSize:0.5*ySize $      ; minimum plotwindow height: 0.5 of initial value
;        ,wSlider:wSlider $ ; Energy transfer cut widget ID
        ,wSource:wSource $
        ,sourceIndex:sourceIndex $
        ,oData:oData $     ; ptr to dataset
        ,counts:counts $
        ,error:error $
        ,chEff:chEff $
        ,cheffError:chEffError $
        ,detNos:detNos $
        ,nDets:nDets $
        ,lolim:lolim $
        ,hilim:hilim $
;        ,energy:energy $       ; energy array of dataset
        ,zp1:[0.0,0.0] $       ; zoom selection - first point
        ,zp2:[0.0,0.0] $       ; zoom selection - second point
        ,mouseType:0 $         ; mouse, 1==left, 2==middle, 4==right
        ,accept:0 $
        ,tool:oTool $
        ,oldrgb:oldrgb $       ; rbg values outside fans_datreduc
        ,olddc:olddc   $       ; previous device decompose setting
        ,colors:colors $       ; colors within fans_datreduc from a custom 16-color table 
;        ,mev2wnos:8.065541 $    ; meV to wavenumber conversion factor
;        ,DAVETool:oDAVETool $  ; object reference of DAVE2 Main Tool 
        }
sPtr = ptr_new(state,/no_copy)
widget_control, wTLB, set_uvalue=sPtr

; Plot the data
wd_SPINSPSDmaskPixels_plotData, sPtr
;wd_SPINSPSDmaskPixels_syncSlider, (*sPtr).wSlider, (*sPtr).energy 

widget_control, wTLB, kill_notify='wd_SPINSPSDmaskPixels_cleanup'
xmanager, 'wd_SPINSPSDmaskPixels',wTLB, /no_block

;help, maskList

end
