; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataTranspose
;
; PURPOSE:
;   A simple operation that transposes 2D plottable data
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   RedoOperation
;   UndoOperation
;   Transpose
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Jan 2007
;-
;###############################################################################


;===============================================================================
; DAVEopDataTranspose::Transpose
; 
; PURPOSE:
;   Transpose the dataset contained within the parameterset object
;
; PARAMETERS:
;   oTarget [in]  - The parameterset object containing the data to be transposed
;
;   oCmd [in|out] - A command object to store change details
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
;function DAVEopDataTranspose::Transpose, oTarget, oCmd
;compile_opt idl2
;
;oTool = Self->GetTool()
;
;
;
;
;oParent->GetProperty, nDimensions=nDim, axis1Ref=oInd1, axis2Ref=oInd2 $
;         ,dataRef=oDep, errorRef=oErr, trmtRef=oTrmt
;
;if (nDim ne 2) then begin
;    oTool->StatusMessage,'Can only transpose datasets with 2 independent variables!'
;    return, 0
;endif
;
;if (~obj_valid(oDep) || ~obj_valid(oInd1) || ~obj_valid(oInd2)) then begin
;   oTool->StatusMessage,'Missing data. Cannot proceed with transpose!'
;   return, 0
;endif
;
;if (~oDep->GetData(zData)) then return, 0
;
;;; Swap axis types for the independent axes. The axis type property
;;; is used to automatically match data and visualisation parameters
;oInd1->GetProperty, axistype=xType   ; expect axisType to be 0 for first independent axis
;oInd2->GetProperty, axistype=yType   ; axpest axisType to be 1 for second ind. axis
;oInd1->SetProperty, axisType=yType
;oInd2->SetProperty, axisType=xType
;
;;; Swap the parameter names for the independent axes data
;oParmSet->remove, [oInd1,oInd2]
;oParmSet->Add, [oInd1,oInd2], parameter_name=strtrim(string(fix([yType,xType])),2)
;
;;; transpose the data and error
;void = oDep->SetData(transpose(zData),/no_copy,/no_notify)
;if (obj_valid(oErr)) then begin
;    errPresent = oErr->GetData(eData)
;    if (errPresent) then begin
;      void = oErr->SetData(transpose(eData),/no_copy,/no_notify)
;
;    endif
;endif
;
;;; If there are visualizations using this dataset, the parameter data
;;; of the visualization need to be swapped to reflect the transposed dataset.
;oVis = oDep->getdataobservers(/all,ISA='IDLitVisualization',count=visCnt)
;if (visCnt gt 0) then begin
;    for i=0,visCnt-1 do begin
;        if (obj_valid(oVis[i]) && obj_isa(oVis[i],'IDLitVisualization')) then begin
;            oX = oVis[i]->GetParameter('X',count=xCnt) ; Note: oX should be equiv to oInd1, etc
;            oY = oVis[i]->GetParameter('Y',count=yCnt)
;            if ((xCnt eq 1) && (yCnt eq 1)) then begin
;                void = oVis[i]->setData(oX,parameter_name='Y',/no_update)
;                void = oVis[i]->setData(oY,parameter_name='X',/no_update)
;            endif
;        endif
;    endfor
;endif
;
;;; notify observers about data change
;oDep->NotifyDataChange
;if (errPresent) then oErr->NotifyDataChange
;oInd1->NotifyDataChange
;oInd2->NotifyDataChange
;
;oDep->NotifyDataComplete
;if (errPresent) then oErr->NotifyDataComplete
;oInd1->NotifyDataComplete
;oInd2->NotifyDataComplete
;
;return, 1
;
;end


;===============================================================================
; DAVEopDataTranspose::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj created by DoAction()
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataTranspose::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Loop through rest of commands and undo the changes that were made
; to the targets
for i=0,nCmds-1 do begin
    oCmds[i]->GetProperty, target_identifier=idParent
    oParent = oTool->GetByIdentifier(idParent)
    if (~obj_valid(oParent)) then begin
        oTool->StatusMessage,'Missing dataset! Transpose undo could not be completed'
        continue
    endif

    ;; Basically, transpose data again to undo previous transpose!
   if (~oParent->Transpose()) then continue
    

   ; Restore the treatment history
   void = oCmds[i]->GetItem('ID_TREATMENT',idTrmt)
   oTrmt = oTool->GetByIdentifier(idTrmt)
   if (obj_valid(oTrmt)) then begin
      void = oCmds[i]->GetItem('OLD_TREATMENT',trmt)
      void = oTrmt->SetData(trmt,/no_copy)
   endif
    
endfor

return, 1

end


;===============================================================================
; DAVEopDataTranspose::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj created by DoAction()
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataTranspose::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0


; Loop through command objects and redo the transpose
for i=0,nCmds-1 do begin
    oCmds[i]->GetProperty, target_identifier=idParent
    oParent = oTool->GetByIdentifier(idParent)
    if (~obj_valid(oParent)) then begin
        oTool->StatusMessage,'Missing dataset! Transpose redo could not be completed'
        continue
    endif

    ;; Basically, transpose data again to redo previous transpose!
   if (~oParent->Transpose()) then continue
    

   ; Restore the treatment history
   void = oCmds[i]->GetItem('ID_TREATMENT',idTrmt)
   oTrmt = oTool->GetByIdentifier(idTrmt)
   if (obj_valid(oTrmt)) then begin
      void = oCmds[i]->GetItem('NEW_TREATMENT',trmt)
      void = oTrmt->SetData(trmt,/no_copy)
   endif
endfor

return, 1

end


;===============================================================================
; DAVEopDataTranspose::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's scalefactor to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataTranspose::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be transposed
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP']
nValid = n_elements(validTypes)
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
   oTarget = (n_elements(oTarget) gt 0)? [oTarget,oRes] : oRes
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
if (n_elements(oTarget) eq 0) then begin
    oTool->StatusMessage, 'Invalid data selection(s) - no data that can be transposed!'
    return, obj_new()
endif


; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Transpose data in selected targets
; NB: no properties are being modified for the target objects hence
; there are no initial/final values to be recorded
for i = 0,n_elements(oTarget)-1 do begin
   ; Retrieve the parent dataset object since this is more useful
   oParent = getmaindataobjectparent(oTarget[i])
   if (~obj_valid(oParent)) then return, 0
   
   ; Don't know how to transpose data objects of classes that have not
   ; implemented the transpose() method
   if (~obj_hasmethod(oParent,'Transpose')) then begin
      oSel[i]->GetProperty, name=name
      oTool->StatusMessage, "Don't know how to transpose "+name
      continue
   endif
   
   ; Do the actual data transpose and loop immediately, if unsuccessful
   if (~oParent->Transpose()) then continue

   ; Create a command object to record useful info
   oCmd = obj_new('IDLitCommand',target_identifier=oParent->getfullidentifier())

   ; modifye treatment history and record in command object
   oParent->GetProperty, trmtRef=oTrmt
   if (obj_valid(oTrmt)) then begin
      void = oTrmt->GetData(trmt)
      void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
      void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
      ;; modify treatment info accordingly
      line = '____________________________________________________'
      trmt = [trmt, $
              line, $
              'Timestamp: '+systime(), $
              'Perform transpose of the data']
      void = oCmd->AddItem('NEW_TREATMENT',trmt) ; save modified treatment info
      void = oTrmt->SetData(trmt,/no_copy)
   endif

   oCmdSet->Add, oCmd
endfor

; If no commands were created, ==> no transpose was successful so return a null commandset
void = oCmdSet->Get(/all,count=nCmds)
return, (nCmds eq 0)? obj_new() : oCmdSet

end


;===============================================================================
; DAVEopDataTranspose::Cleanup
; 
; PURPOSE:
;   DAVEopDataTranspose class cleanup
;
pro DAVEopDataTranspose::Cleanup

compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataTranspose::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataTranspose::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
if (~self->IDLitOperation::Init(NAME='Data Transpose' $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP']


; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=1

; This operation is reversible
; This operation is not expensive
self->SetProperty, reversible_operation=1, expensive_operation=0

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataTranspose__define
; 
; PURPOSE:
;   DAVEopDataTranspose class structure definition
;
pro DAVEopDataTranspose__define

compile_opt idl2

struct = {DAVEopDataTranspose $
          ,inherits IDLitOperation $
         }

end
