
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   Daveopdatasum
;
; PURPOSE:
;   A simple operation for summing two or more datasets together. The
;   datasets must of the same size (dimensions and elements in each dimension).
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   ResetOpAttributes
;   SetProperty
;   Sum
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; Daveopdatasum::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset*   - Eg dataset1, dataset2, ...
;                These are the names of the datasets that the user has selected to be
;                included in the sum. Because the number of datasets are not known a priori,
;                it would be very cumbersome to itemise these properties as keywords to this 
;                procedure. Instead they are received via the _REF_EXTRA keyword and 
;                additional processing is applied with the use of the Scope_Varfetch() to
;                determine when these properties are requested. The actual values for the
;                properties are maintained in the class variables Self.propertyNames and
;                Self.datasetsIndex
;
;   ndata      - The number of datasets to be summed
;   
;   resultname - The name of the resultant dataset. The user has the option to specify
;                this although a standard default is always set otherwise.
;
;   algorithm  - The algorithm to be used: simple sum or weighted average; default is 
;                simple sum
;
; RETURN VALUE:
;
pro Daveopdatasum::GetProperty ,ndata=ndata,resultname=resultname,algorithm=algorithm $
                  ,_REF_EXTRA=etc
compile_opt idl2

;ndata
if (arg_present(ndata)) then ndata=self.ndata
;name of dataset to store result
if (arg_present(resultName)) then resultName=self.resultName
; Algorithm
if (arg_present(algorithm)) then algorithm=self.algorithm

; dataset* (dataset1, dataset2, etc). These are the names of the datasets that are to 
; be included in the sum
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   for i=0, nEtc-1 do begin
      index = where(etc[i] eq strupcase(propNames), found)
      if (found) then (Scope_Varfetch(etc[i],/REF_EXTRA)) = Self.datasetsIndex[index]
   endfor
endif

; also call base class accessor
if(nEtc gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; Daveopdatasum::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset*   - Eg dataset1, dataset2, ...
;                These are the names of the datasets that the user has selected to be
;                included in the sum. Because the number of datasets are not known a priori,
;                it would be very cumbersome to itemise these properties as keywords to this 
;                procedure. Instead they are received via the _EXTRA keyword and 
;                additional processing is applied with the use of the Scope_Varfetch() to
;                determine when these properties are being set. The actual values for the
;                properties are maintained in the class variables Self.propertyNames and
;                Self.datasetsIndex
;
;   ndata      - The number of datasets to be summed
;
;   nmax       - Maximum nos of datasets available (ie. that is present in data browser)
;
;   resultname - The name of the resultant dataset. The user has the option to specify
;                this although a standard default is always set otherwise.
;
;   algorithm  - The algorithm to be used: simple sum or weighted average; default is 
;                simple sum
;
; RETURN VALUE:
;
pro Daveopdatasum::SetProperty,ndata=ndata,nmax=nmax,resultname=resultname,algorithm=algorithm $
                      ,_EXTRA=etc
compile_opt idl2

; Nmax
if (n_elements(nmax)) then self.nmax = nmax > 2 ; must be >=2

; Ndata
if (n_elements(ndata)) then begin
    self.ndata= ndata
    for i=0,self.nmax-1 do begin
        hide = (i ge self.ndata)
        propName = Self.propertyNames[i]
        self->SetPropertyAttribute, propName, hide=hide
    endfor
endif

; Name of dataset to store results
if (n_elements(resultName)) then begin
    self.resultName=resultName
    self.namechanged = 1
endif

; Algorithm
if (n_elements(algorithm)) then self.algorithm=algorithm

; dataset* properties
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   etcTags = tag_names(etc)
   for i=0, nEtc-1 do begin
      index = where(etcTags[i] eq strupcase(propNames), found)
      if (found) then  Self.datasetsIndex[index] = etc.(i) ;Scope_Varfetch(etc.(i))
   endfor
endif

; Also Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; Daveopdatasum::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function Daveopdatasum::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName, algorithm=algorithm
void = oCmd->AddItem('OLD_RESULTNAME',resultName)
void = oCmd->AddItem('OLD_ALGORITHM',algorithm)

; Add the command to command set
oCmdSet->Add, oCmd


return, 1

end


;===============================================================================
; Daveopdatasum::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function Daveopdatasum::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName,algorithm=algorithm
void = oCmd->AddItem('NEW_RESULTNAME',resultName)
void = oCmd->AddItem('NEW_ALGORITHM',algorithm)

return, 1

end


;===============================================================================
; Daveopdatasum::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function Daveopdatasum::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's properties.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; Daveopdatasum::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function Daveopdatasum::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 2) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Restore operation attributes
void = oCmds[0]->GetItem('OLD_RESULTNAME',resultName)
void = oCmds[0]->GetItem('OLD_ALGORITHM',algorithm)
self->SetProperty, resultName=resultName, algorithm=algorithm

; Undo previous sum by deleting the previously generated output from the Data Manager
void = oCmds[1]->GetItem('FULLOUTPUTID',idDataset)
oDataset = oTool->GetByIdentifier(idDataset)
oDM = oTool->GetService("DAVEDATA_MANAGER")	
status = oDM->DeleteData(oDataset->GetFullIdentifier())

; Indicate that the tool has been modified and force it to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; Daveopdatasum::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function Daveopdatasum::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 2) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Restore operation attributes
void = oCmds[0]->GetItem('NEW_RESULTNAME',resultName)
void = oCmds[0]->GetItem('NEW_ALGORITHM',algorithm)
self->SetProperty, algorithm=algorithm, resultName=resultName

; Restrieve useful attributes 
void = oCmds[1]->GetItem('ALGORITHMNAME',algoName)
void = oCmds[1]->GetItem('IDDATASETS',idDatasets)
void = oCmds[1]->GetItem('DATASETNAMES',names)
void = oCmds[1]->GetItem('OUTPUTNAME',resName)
void = oCmds[1]->GetItem('OUTPUTID',resID)
nData = n_elements(idDatasets)

newTrmt = '______________________________'
newTrmt = [newTrmt,'Sum operation performed using "'+algoName+'" algorithm']
for i=0,nData-1 do begin
   idDataset = idDatasets[i]
   oDataset = oTool->GetByIdentifier(idDataset)
   if (~obj_valid(oDataset)) then continue

   oDataset->GetProperty, dataRef=oDat, errorRef=oErr  ; get the dependent and error in dependent data
   if (~obj_valid(oDat) || ~obj_valid(oErr)) then continue

   if (~oDat->GetData(dat) || ~oErr->GetData(err)) then begin
       msg = 'Sum: Error retrieving data from dataset!'
       oTool->StatusMessage,msg
       obj_destroy, oCmdSet
       return, 0
   endif
   if (i eq 0) then begin
       dS1 = size(dat)
       dS2 = size(err)
       ;; dims and nos of elements should be the same for data and error
       passed = ((dS1[0] eq dS2[0]) && (dS1[1] eq dS2[1]) && (dS1[2] eq dS2[2]))
       if (~passed) then begin
           msg = 'Sum: Data and error have inconsistent dimensions and/or nos of elements!'
           oTool->StatusMessage,msg
           obj_destroy, oCmdSet
           return, 0
       endif
       firstDataset = oDataset
   endif
   s1 = size(dat)
   s2 = size(err)
   ;; dims and nos of elements should be the same for data and error
   passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2]))
   if (~passed) then begin
       msg = 'Sum: Data and error have inconsistent dimensions and/or nos of elements!'
       oTool->StatusMessage,msg
       obj_destroy, oCmdSet
       return, 0
   endif
   ;; dims and nos of elements should be the same for old and cur data
   passed = ((dS1[0] eq s1[0]) && (dS1[1] eq s1[1]) && (dS1[2] eq s1[2]))
   if (~passed) then begin
       msg = 'Sum: Dims and/or elements for '+names[i]+' are different from other datasets!'
       oTool->StatusMessage,msg
       obj_destroy, oCmdSet
       return, 0
   endif

   ;; perform sum
   case strupcase(algoName) of
       'SIMPLE SUM': begin     ;simple sum
           data = (i eq 0)? dat : temporary(data) + dat
           error = (i eq 0)? err^2 : temporary(error) + err^2
       end

       'WEIGHTED AVERAGE': begin
           err = (temporary(err))^2
           error = (i eq 0)? 1.0/err : temporary(error) + 1.0/err
           data = (i eq 0)? dat/err : temporary(data) + dat/err
       end

       else: begin
           oTool->StatusMessage, 'Could not redo sum - unknown algorithm!'
           obj_destroy, oCmdSet
           return, 0
       end
   endcase

   newTrmt = [newTrmt,'Include "'+names[i]+'" in sum']

endfor
case strupcase(algoName) of
    'SIMPLE SUM': begin         ;simple sum
        error = sqrt(temporary(error))
    end
    
    'WEIGHTED AVERAGE': begin
        data = temporary(data)/error
        error = sqrt(1/temporary(error))
    end

    else: begin
        oTool->StatusMessage, 'Sum could not be performed - unknown algorithm!'
        obj_destroy, oCmdSet
        return, 0
    end
endcase

if (obj_hasmethod(firstDataset,'Clone')) then begin
   ; use clone method to replicate first dataset
   oRes = firstDataset->Clone()
   oRes->SetProperty, name=resName, identifier=resID
   oTool->AddDatasetToDataManager, oRes
endif else begin
   ; use DM to replicate first dataset
   oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
   if (~oDM->CopyData(firstDataset->GetFullIdentifier(), name=resName, copyid=resID)) then return, 0
   oRes = oTool->GetByIdentifier(resID)
endelse

; Get the treatment info and update it
trmtExist = 0
oRes->GetProperty, trmtRef=oTrmt, dataRef=oDat, errorRef=oErr
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then $
  void = oTrmt->SetData([trmt,newTrmt],/no_copy,/no_notify) ; modify treatment

void = oErr->SetData(error,/no_copy,/no_notify)
void = oDat->SetData(data,/no_copy,/no_notify)

; Update 'OUTPUTREF' entry
void = oCmds[1]->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier(),/overwrite)

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; Daveopdatasum::ResetOpAttributes
; 
; PURPOSE:
;   Reset attributes for this operation to reflect the currently
;   selected datasets
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function Daveopdatasum::ResetOpAttributes, selNames
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the all datasets in Data Manager
oDatasets = oTool->GetDAVEDataManagerContents(count=nMax)
if (nMax lt 2) then return, 0

nSel = n_elements(selNames)
selIndex = []
theRestIndex = []

; Retrieve ids and names of datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValidTypes = n_elements(validTypes)
nValid = 0
For i=0,nmax-1 do begin
    if (~obj_valid(oDatasets[i])) then continue

    ;; Does dataset contain valid target data type?
    j = 0
    repeat begin
        oRes = oDatasets[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValidTypes)

    if (~obj_valid(oRes)) then continue

    oDatasets[i]->GetProperty, name=name
    names = (n_elements(names) gt 0)? [names,name] : name
    validDatasets = (n_elements(validDatasets) gt 0)? [validDatasets,oDatasets[i]] : oDatasets[i]
    
    nValid++
    
    ; if current dataset is selected, record it in 'selIndex', otherwise record it in 'theRestIndex'
    index = where(strcmp(selNames,name),cnt)
    if (cnt eq 1) then selIndex = [selIndex,i] $
      else theRestIndex = [theRestIndex,i] 
endfor

if (nValid lt 2) then return, 0

; number of valid datasets should not exceed hardnmax
if (nValid gt Self.hardNmax) then begin
   msg = "Maximum datasets that can be summed is "+string(Self.hardNmax,format='(I-3)')
   msg = [msg,'Consider deleting a few datasets fron the Data Browser']
   msg = [msg,'until the totat number is less than '+string(Self.hardNmax,format='(I-3)')]
   Self->ErrorMessage, msg, severity=2
   return, 0
endif

; rearrange so that selected datasets are at the beginning
if (n_elements(selIndex) gt 0 && n_elements(theRestIndex) gt 0) then begin
  index = [selIndex,theRestIndex]
  validDatasets = validDatasets[index]
  names = names[index]
endif

self->SetProperty, nmax = nValid ; the nos of datasets with valid targets
if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
self.validDatasetsPtr = ptr_new(validDatasets) ; the object refs of valid datasets found in DM


; Set -
; ndata to number of selected dataset
; range of ndata to between 2 and nos of valid datasets
; a default name to be used to store the result
self->SetPropertyattribute, 'ndata', valid_range=[2,self.nmax]
self->SetProperty, ndata=self.selCount

guess1 = self.resultName
guess2 = IDLitGetUniqueName(names, guess1) 
if (guess2 ne guess1) then begin
    pos = strpos(guess2,'_',/reverse_search)
    if (pos ne -1) then begin
        guess1  = strmid(guess2,0,pos) + ' '
        len = strlen(guess2) - (pos+1)
        guess1 += strmid(guess2,pos+1,len)
    endif
endif

self.namechanged = 0            ; indicate using default result name
self.resultName = guess2        ; may contain an underscore
self.resultID = guess1          ; the underscore replaced by space

if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr
self.namesPtr = ptr_new(names)  ; the string names associated with the valid datasets found in the DM

; Assign the valid datasets (upto nmax) to the dataset* (registered) properties of the operation
for i = 0,self.nmax-1 do begin
    propName = Self.propertyNames[i]
    self->SetPropertyAttribute, propName, enumlist=(names)
    index = i
    if (i lt self.selCount) then begin
        ;; assign selected datasets as defaults
        index = where(names eq selNames[i], cnt)
        index = (cnt eq 1)? index[0] : i
    endif
    self.datasetsIndex[i] = index
endfor

return, 1

end


;===============================================================================
; Daveopdatasum::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Sum the specified datasets 
;   and store the results in a new dataset stored in the Data Browser.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function Daveopdatasum::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt lt 2) then begin
    oTool->StatusMessage, 'Please, select at least two top-level datasets from the Data Browser tree.'
    return, obj_new()
endif

; Locate datasets containing experiment data
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValid = n_elements(validTypes)
validName = 'DAVE Data Manager'
for i = 0,n_elements(oSelections)-1 do begin
    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oRes = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)


    if (~obj_valid(oRes)) then continue
    oSelections[i]->GetProperty, name=selName
    selNames = (n_elements(selNames) gt 0)? [selNames,selName] : selName
endfor

if (n_elements(selNames) lt 2) then begin
    oTool->StatusMessage, 'Sum cannot be performed using selected data!'
    return, obj_new()
endif

self.selCount = n_elements(selNames)


; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

;; Set operation properties based on the selected data
if (~self->ResetOpAttributes(selNames)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
;hasPropSet = 0b
if (doUI) then begin
    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
endif

;; Perform sum and record changes in undo/redo buffer
if (~self->Sum(oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; Daveopdatasum::Sum
; 
; PURPOSE:
;   Perform difference b/n two datasets and record changes in the
;   operation's undo/redo buffer.
;
; PARAMETERS:
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0
;
function Daveopdatasum::Sum, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; create a command object to store useful info. The target for the
; command is the operation itself. For the sum operation, only
; one sum will be performed per command set object created! If
; this were not the case, then it would be better to make a dataset
; object as the target.
oCmd = obj_new('IDLitCommand',target_identifier=self->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0
oCmdSet->Add, oCmd              ; Add the command to command set

; Operation settings
;resName = IDLitGetUniqueName((*self.namesPtr),self.resultName)
resName = self.resultName
nData = self.nData
algoIndex = self.algorithm
algoName = (self.algorithms)[algoIndex]
;targetIDs = (*self.targetIDsPtr)[self.datasetsIndex[0:nData-1]]
;datasetIDs = (*self.datasetIDsPtr)[self.datasetsIndex[0:nData-1]]
oDatasets = (*self.validDatasetsPtr)[self.datasetsIndex[0:nData-1]]
names = (*self.namesPtr)[self.datasetsIndex[0:nData-1]]

newTrmt = ['==================================================' $
            ,'Timestamp: '+systime()]
newTrmt = [newTrmt,'Sum operation performed using "'+algoName+'" algorithm']
idDatasets = strarr(nData)
for i=0,nData-1 do begin
   oDataset = oDatasets[i]
   idDatasets[i] = oDataset->GetFullIdentifier()
   if (~obj_valid(oDataset)) then continue

   oDataset->GetProperty, dataRef=oDat, errorRef=oErr  ; get the dependent and error in dependent data
   if (~obj_valid(oDat) || ~obj_valid(oErr)) then continue

;   oDats = (n_elements(oDats) le 0)? oDat : [oDats,oDat]
;   oErrs = (n_elements(oErrs) le 0)? oErr : [oErrs,oErr]
;   validIDs = (n_elements(validIDs) le 0)? targetIDs[i] : [validIDs,targetIDs[i]]

   if (~oDat->GetData(dat) || ~oErr->GetData(err)) then begin
       msg = 'Sum: Error retrieving data from dataset!'
       oTool->StatusMessage,msg
       return, 0
   endif
   if (i eq 0) then begin
       dS1 = size(dat)
       dS2 = size(err)
       ;; dims and nos of elements should be the same for data and error
       passed = ((dS1[0] eq dS2[0]) && (dS1[1] eq dS2[1]) && (dS1[2] eq dS2[2]))
       if (~passed) then begin
           msg = 'Sum: Data and error have inconsistent dimensions and/or nos of elements!'
           oTool->StatusMessage,msg
           return, 0
       endif
       firstDataset = oDataset
   endif
   s1 = size(dat)
   s2 = size(err)
   ;; dims and nos of elements should be the same for data and error
   passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2]))
   if (~passed) then begin
       msg = 'Sum: Data and error have inconsistent dimensions and/or nos of elements!'
       oTool->StatusMessage,msg
       return, 0
   endif
   ;; dims and nos of elements should be the same for old and cur data
   passed = ((dS1[0] eq s1[0]) && (dS1[1] eq s1[1]) && (dS1[2] eq s1[2]))
   if (~passed) then begin
       msg = 'Sum: Dims and/or elements for '+names[i]+' are different from other datasets!'
       oTool->StatusMessage,msg
       return, 0
   endif

   ;; perform sum
   case strupcase(algoName) of
       'SIMPLE SUM': begin     ;simple sum
           data = (i eq 0)? dat : temporary(data) + dat
           error = (i eq 0)? err^2 : temporary(error) + err^2
       end

       'WEIGHTED AVERAGE': begin
           err = (temporary(err))^2
           error = (i eq 0)? 1.0/err : temporary(error) + 1.0/err
           data = (i eq 0)? dat/err : temporary(data) + dat/err
       end

       else: begin
           oTool->StatusMessage, 'Sum could not be performed - unknown algorithm!'
           return, 0
       end
   endcase

   newTrmt = [newTrmt,'Include "'+names[i]+'" in sum']

endfor
case strupcase(algoName) of
   'SIMPLE SUM': begin         ;simple sum
       error = sqrt(temporary(error))
   end
   
   'WEIGHTED AVERAGE': begin
       data = temporary(data)/error
       error = sqrt(1/temporary(error))
   end

   else: begin
       oTool->StatusMessage, 'Sum could not be performed - unknown algorithm!'
       return, 0
   end
endcase

; Store useful details
void = oCmd->AddItem('IDDATASETS',idDatasets)
void = oCmd->AddItem('DATASETNAMES',names)
void = oCmd->AddItem('ALGORITHMNAME',algoName)

; copy of the first dataset into the results dataset
; NB this creates an entry in the data browser
resID = (self.namechanged)? resName : self.resultID
void = oCmd->AddItem('OUTPUTNAME',resName)
void = oCmd->AddItem('OUTPUTID',resID)
if (obj_hasmethod(firstDataset,'Clone')) then begin
   ; use clone method to replicate first dataset
   oRes = firstDataset->Clone()
   oRes->SetProperty, name=resName, identifier=resID
   oTool->AddDatasetToDataManager, oRes
endif else begin
   ; use DM to replicate first dataset
   oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
   if (~oDM->CopyData(firstDataset->GetFullIdentifier(), name=resName, copyid=resID)) then return, 0
   oRes = oTool->GetByIdentifier(resID)
endelse

; Get the treatment info and update it
trmtExist = 0
oRes->GetProperty, trmtRef=oTrmt, dataRef=oDat, errorRef=oErr
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then $
  void = oTrmt->SetData([trmt,newTrmt],/no_copy,/no_notify) ; modify treatment

void = oErr->SetData(error,/no_copy,/no_notify)
void = oDat->SetData(data,/no_copy,/no_notify)

; Save useful info for redo/undo
;void = oCmd->AddItem('FIRSTDATASETID',firstDatasetID)
void = oCmd->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier())

; No need to notify observers because there should be none!
;oDat->notifyDataChange
;oDat->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; Daveopdatasum::Cleanup
; 
; PURPOSE:
;   Daveopdatasum class cleanup
;
pro Daveopdatasum::Cleanup
compile_opt idl2

if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; Daveopdatasum::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function Daveopdatasum::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is reversible
; This operation is not expensive
if (~self->IDLitOperation::Init(NAME='Data Sum' $
                                ,reversible_operation=1,expensive_computation=0 $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register properties for this operation
self.algorithms = ['SIMPLE SUM','WEIGHTED AVERAGE']
self.algorithm = 0
self->RegisterProperty, 'algorithm', description='The algorithm to be used for sum' $
  ,name='Algorithm to use',sensitive=1,enumlist=self.algorithms
self->RegisterProperty, 'resultName',/string, description='Name of dataset to store result' $
  ,name='Output dataset name',sensitive=1
self->RegisterProperty, 'ndata', /integer, description='Number of datasets to be summed' $
  ,name='Nos of datasets to sum',sensitive=1,valid_range=[2,2]

Self.hardNmax = 99
for i = 0,Self.hardNmax-1 do begin
    self.datasetsIndex[i] = i
    NicePropName = 'Dataset '+strtrim(string(i+1),2)
    prop = 'dataset'+strtrim(string(i+1, format='(I02)'),2)
    self->RegisterProperty, prop, enumlist=['empty'], description=NicePropName $
      ,name=NicePropName, sensitive=1,hide=1
    Self.propertyNames[i] = prop
endfor

self.resultName = 'SUMRESULT'
; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; Daveopdatasum__define
; 
; PURPOSE:
;   Daveopdatasum class structure definition
;
pro Daveopdatasum__define

compile_opt idl2

struc = {Daveopdatasum $
         ,inherits IDLitOperation $
         ,hardNmax:99 $
         ,nmax:0B $             ; maximum nos of datasets available (present in data browser)
         ,datasetsIndex:bindgen(99) $ ; indices (of valid datasets) to be included in sum. 
                                             ;An absolute limit of 99 is imposed! Therefore nmax must be <= 99
         ,propertyNames:strarr(99) $   ; property names created dynamically to represent datasets
         ,namechanged:0B $       ; use to indicate when resultName field has been modified
         ,resultName:'' $       ; name of dataset to store the sum
         ,resultID:'' $         ; base id of dataset to store the sum
         ,algorithms:['',''] $  ; available algorithms to perform sum
         ,algorithm:0B  $       ; the one used
         ,ndata:0B $            ; nos of datasets to be included in sum
         ,selCount:0B $         ; nos of datasets currently selected
         ,validDatasetsPtr:ptr_new() $ ; ptr to ids of valid datasets
         ,namesPtr:ptr_new() $  ; ptr to names of valid datasets
        }

end
