; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   daveopdataReplaceValue
;
; PURPOSE:
;   A simple operation that adds an additive offset to the data of an
;   IDLitData object.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Larry R. Kneller
; NIST Center for Neutron Research
; kneller@nist.gov; (301) 975-8839
;
;-
;###############################################################################


;===============================================================================
; daveopdataReplaceValue::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro daveopdataReplaceValue::GetProperty, replacement=replacement,valuetoreplace=valuetoreplace, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(replacement)) then $
  replacement =  self._replacement

if (arg_present(valuetoreplace)) then $
  valuetoreplace =  self._valuetoreplace

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; daveopdataReplaceValue::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro daveopdataReplaceValue::SetProperty, replacement=replacement,valuetoreplace=valuetoreplace, _EXTRA=etc
compile_opt idl2

; The UI dialog should be resposible for checking the integrity of the
; value 
if (n_elements(replacement) gt 0) then $
  self._replacement = replacement
if (n_elements(valuetoreplace) gt 0) then $
  self._valuetoreplace = valuetoreplace

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; daveopdataReplaceValue::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function daveopdataReplaceValue::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, replacement=replacement,valuetoreplace=valuetoreplace
void = oCmd->AddItem('OLD_REPLACEMENT',replacement)
void = oCmd->AddItem('OLD_VALUETOREPLACE',valuetoreplace)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; daveopdataReplaceValue::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function daveopdataReplaceValue::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, replacement=replacement,valuetoreplace=valuetoreplace
void = oCmd->AddItem('NEW_REPLACEMENT',replacement)
void = oCmd->AddItem('NEW_VALUETOREPLACE',valuetoreplace)

return, 1

end


;===============================================================================
; daveopdataReplaceValue::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function daveopdataReplaceValue::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; daveopdataReplaceValue::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function daveopdataReplaceValue::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'old' offset value and undo the change on the target
if (oCmds[0]->GetItem('OLD_REPLACEMENT',replacement)) then begin
    oTarget->SetProperty, replacement=replacement
endif
if (oCmds[0]->GetItem('OLD_VALUETOREPLACE',valuetoreplace)) then begin
    oTarget->SetProperty, valuetoreplace=valuetoreplace
endif

; Loop through rest of commands and undo the changes that were made
; to the targets
if (nCmds gt 1) then begin
    void = oCmds[0]->GetItem('NEW_REPLACEMENT',replacement) ; retrieve offset that was applied to data
    void = oCmds[0]->GetItem('NEW_VALUETOREPLACE',valuetoreplace) ; retrieve offset that was applied to data
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Undo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin

            ;REPLACE THE SELECTED VALUES
            if n_elements(data) gt 0 then begin
              whreplace = where(abs((data - valuetoreplace)/valuetoreplace) lt 0.000001,replaceCount)
              if replaceCount gt 0 then begin
          print,replacecount

                data[whreplace] = replacement
              endif
            endif


            ;; modify the data
            void = oTarget->SetData(data) ; undo 
            ;; revert to previous treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('OLD_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; daveopdataReplaceValue::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function daveopdataReplaceValue::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'new' values and redo the change on the target
; (self)
if (oCmds[0]->GetItem('NEW_REPLACEMENT',replacement)) then begin
    oTarget->SetProperty, replacement=replacement
endif

if (oCmds[0]->GetItem('NEW_VALUETOREPLACE',valuetoreplace)) then begin
    oTarget->SetProperty, valuetoreplace=valuetoreplace
endif

; Loop through rest of commands and redo the changes that were made
; to the targets (datasets)
if (nCmds gt 1) then begin
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Redo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin

            ;REPLACE THE SELECTED VALUES
            if n_elements(data) gt 0 then begin
              whreplace = where(abs((data - valuetoreplace)/valuetoreplace) lt 0.000001,replaceCount)
              if replaceCount gt 0 then begin
          print,replacecount

                data[whreplace] = replacement
              endif
            endif

            ;; modify the data
            void = oTarget->SetData(data) ; 
            ;; re-apply the new treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('NEW_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; daveopdataReplaceValue::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function daveopdataReplaceValue::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oTarget = oTool->GetSelectedData()
void = where(obj_valid(oTarget),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate all those that are IDLitData objects, and not
; containers.
for i=0,n_elements(oTarget)-1 do begin
    if (obj_valid(oTarget[i]) && $
        obj_isa(oTarget[i],'IDLitData') && $
      ~obj_isa(oTarget[i],'IDL_Container') ) then $
        index = (n_elements(index) gt 0)? [index,i] : i
endfor
if (n_elements(index) eq 0) then begin
    oTool->StatusMessage, 'Invalid selection(s). Select data items not folders!'
    return, obj_new()
endif

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Record initial values for the properties of this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final values for the properties of this operation.
    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

; Apply the changes to the data
; NB: no properties are being modified for the target objects hence
; there are no initial/final values to be recorded
for i = 0,n_elements(index)-1 do begin
   if (oTarget[i]->GetData(data)) then begin
   
      valuetoreplace = self._valuetoreplace
      replacement = self._replacement
      ;REPLACE THE SELECTED VALUES
      if n_elements(data) gt 0 then begin
        whreplace = where(abs((data - valuetoreplace)/valuetoreplace) lt 0.000001,replaceCount)
        
        if replaceCount gt 0 then begin
          print,replacecount
          data[whreplace] = replacement
        endif
      endif
      ;; modify the data
      void = oTarget[i]->SetData(data)
      ;; create a command object to record this change.
      oCmd = obj_new('IDLitCommand',target_identifier=oTarget[i]->getfullidentifier())

      ;; Handle treatment history
      oTrmt = LocateTreatmentObject(oTarget[i])
      if (obj_valid(oTrmt)) then begin
         oTarget[i]->GetProperty, name=name
         void = oTrmt->GetData(trmt)         ; retrieve existing trmt details
         
         ; create new treatment info
         line = '____________________________________________________'
         newtrmt = [trmt, $
                   line, $
                   'Timestamp: '+systime(), $
                   'Apply a substitution of '+strtrim(string(self._replacement),2)+'For' $
                                             +strtrim(string(self._valuetoreplace),2)+" to '"+name+"'"]

         void = oTrmt->SetData(newtrmt)   ; modify the tratment details
         
         ; record info to enable undo/redo functionality
         void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier()) ; store old treatment object reference
         void = oCmd->AddItem('OLD_TREATMENT',trmt) ; store old treatment info
         void = oCmd->AddItem('NEW_TREATMENT',newtrmt) ; and new treatment info
      endif

      oCmdSet->Add, oCmd
   endif
endfor

; return the command set obj
return, oCmdSet

end


;===============================================================================
; daveopdataReplaceValue::Cleanup
; 
; PURPOSE:
;   daveopdataReplaceValue class cleanup
;
pro daveopdataReplaceValue::Cleanup

compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; daveopdataReplaceValue::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function daveopdataReplaceValue::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
if (~self->IDLitOperation::Init(NAME='Data Replacement' $
                                ,_EXTRA=etc)) then return, 0
;types=['IDLVECTOR','IDLARRAY2D','IDLARRAY3D','IDLIMAGE'] 

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; This operation is reversible
; This operation is not expensive
self->SetProperty, reversible_operation=1, expensive_operation=0

; Register an ValueToReplace and Replacement properties for this operation
self->RegisterProperty, 'ValueToReplace', /float, description='Value to be substituted for in data' $
  ,name='ValueToReplace',sensitive=1
self->RegisterProperty, 'Replacement', /float, description='Substitution Value to Apply in data' $
  ,name='Replacement',sensitive=1

; init values to 0.0
self._replacement = 0.0
self._valuetoreplace = 0.0

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; daveopdataReplaceValue__define
; 
; PURPOSE:
;   daveopdataReplaceValue class structure definition
;
pro daveopdataReplaceValue__define

compile_opt idl2

struc = {daveopdataReplaceValue $
         ,inherits IDLitOperation $
         ,_replacement:0.0 $         ; value to be substituted in
         ,_valuetoreplace:0.0 $      ;value to be replaced by subsitute
        }

end
