
;===============================================================================
;+
; NAME:
;   Nstarks
;
; PURPOSE:
;   Defines the integrand
;     [n*(s) exp(i*k*s)]
;   Note that k and the parameters a2, a4 and a6 are specified using the common block
;
; PARAMETERS
;   sVal (in) - the input value of s.
;
; CATEGORY:
;   Quantum Fluids Analysis
;-
;===============================================================================
function Nstarks, s, k=k, muPtr=muPtr, _Extra=etc
  compile_opt idl2

  ;common nstars_common, k, muPtr
  if (n_elements(k) eq 0 || n_elements(muPtr) eq 0) then begin
    print, 'missing keyword parameter(s)'
    return, 0.0
  endif

  d2 = 0.50                     ; 1/2.0
  d4 = 0.041666666666666664D    ; 1/24.0
  d6 = 0.0013888888888888889D   ; 1/720.0
  a2 = (*muPtr).a2
  a4 = (*muPtr).a4
  a6 = (*muPtr).a6

  return, s^2 * exp(-a2*s^2*d2 + a4*s^4*d4 - a6*s^6*d6) * Sinc(k*s)

end


;==============================================================================
; Jqys - the full JQY(s) assuming a model momentum distribution in superfluid 4He
; alpha*[n0*beta + n0*beta*f + A*gama]
; In the IA, beta is 1 and the n0*beta term is not valid!
;==============================================================================
function Jqys, s, n0=n0, A=A, muPtr=muPtr, _Extra=etc
  compile_opt idl2

  if ((*muPtr).b6 eq 0.0) then begin
    ;; evaluate the IA only because FSE does not converge and hence the FT will be unpredictable
    ;; In this case, beta = 1 and the alpha*n0*beta turns to n0 and is left out here
    ;; to be added later after the FT is done (ie is added to J(Q,y) after the FT is evaluated!)
    return, Jqyalpha(s,muPtr=muPtr,_Extra=etc)*(n0*FOFS(s,muPtr=muPtr,_Extra=etc) $
                                                   + A*JqyGamma(s,muPtr=muPtr,_Extra=etc))
  endif else begin
    return, JqyAlpha(s,muPtr=muPtr,_Extra=etc)*( n0*JqyBeta(s,muPtr=muPtr,_Extra=etc)* $
      (1 + FOFS(s,muPtr=muPtr,_Extra=etc)) + A*JqyGamma(s,muPtr=muPtr,_Extra=etc) )
  endelse

;  Return, Jqyalpha(s,muPtr=muPtr,_Extra=etc)*( n0*Jqybeta(s,muPtr=muPtr,_Extra=etc)* $
;    (1 + Fofs(s,muPtr=muPtr,_Extra=etc)) + A*Jqygamma(s,muPtr=muPtr,_Extra=etc) )

end


;==============================================================================
; Jqys - the full Jqy(s) in the concolution approach
; alpha*[n0*beta + n0*beta*f + A*gama]
; In the IA, beta is 1 and the n0*beta term is not valid!
;==============================================================================
function CAJqys, s, y=y, muPtr=muPtr, _Extra=etc
compile_opt idl2
    
d2 = 0.50                     ; 1/2.0
d3 = 0.16666666666666666D     ; 1/6.0
d4 = 0.041666666666666664D    ; 1/24.0
d5 = 0.0083333333333333332D   ; 1/720.0
d6 = 0.0013888888888888889D   ; 1/720.0

alpha = Cos(  Y*s $
            + (*muPtr).u3*s^3*d3 $
            - (*muPtr).u5*s^5*d5  )

gamma = Exp(- (*muPtr).u2*s^2*d2 $
            + (*muPtr).u4*s^4*d4 $
            - (*muPtr).u6*s^6*d6  )
 
return, alpha*gamma
end


;==============================================================================
; The even function component of the
; complex Exponential part of Jqy(s)
;==============================================================================
function JqyAlpha, s, y=y, muPtr=muPtr, _Extra=etc
  compile_opt idl2

  d3 = 0.16666666666666666D     ; 1/6.0
  d5 = 0.0083333333333333332D   ; 1/720.0
  return, Cos(  Y*s $
              + (*muPtr).b3*s^3*d3 $
              - (*muPtr).b5*s^5*d5)

end


;==============================================================================
; The even function component of the
; non-complex Exponential part of Rqy(s)
;==============================================================================
function JqyBeta, s, muPtr=muPtr, _Extra=etc
  compile_opt idl2

  d4 = 0.041666666666666664D    ; 1/24.0
  d6 = 0.0013888888888888889D   ; 1/720.0
  return, Exp( (*muPtr).b4*s^4*d4 $
              -(*muPtr).b6*s^6*d6)
end


;==============================================================================
; The even function component of the
; non-complex Exponential part of Jqy(s)
;==============================================================================
function JqyGamma, s, muPtr=muPtr, _Extra=etc
  compile_opt idl2

  d2 = 0.50                     ; 1/2.0
  d4 = 0.041666666666666664D    ; 1/24.0
  d6 = 0.0013888888888888889D   ; 1/720.0

  return, exp( - (*muPtr).a2*s^2*d2 $
               + ((*muPtr).a4+(*muPtr).b4)*s^4*d4 $
               - ((*muPtr).a6+(*muPtr).b6)*s^6*d6 )

end


;==============================================================================
; s-space representation of the Final State function
;==============================================================================
function rqs, s, muPtr
  compile_opt idl2

  if (n_params() ne 2) then return, 0

  d3 = 0.16666666666666666D     ; 1/6.0
  d5 = 0.0083333333333333332D   ; 1/720.0
  d6 = 0.0013888888888888889D   ; 1/720.0

  u3 = (*muPtr).b3
  u5 = (*muPtr).b5
  u6 = (*muPtr).b6
  calc = 0.0d
  arg = -u6*s^6*d6
  calc = EXP(arg)*COS( u3*s^3*d3 - u5*s^5*d5 )

  return, calc
end
