; $Id$
;###############################################################################
;
; NAME:
;  TASRED_DATA__DEFINE
;
; PURPOSE:
;  Simple TAS data reader for PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN, curve fitting
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
; TASRED_DATA__DEFINE
;
; Simple and stupid object class to read TAS data and allow the user
; access to various parameters.
;
; Written by R.M.Dimeo (04/18/03)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro tasred_data::cleanup
ptr_free,self.hdef_ptr,self.text_ptr,self.hnum_ptr,self.prp_ptr
ptr_free,self.data_ptr,self.header_ptr
ptr_free,self.x_ptr,self.y_ptr,self.yerr_ptr
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tas_get_date,filename
dummy = ''
openr,lun,filename,/get_lun
readf,lun,dummy
print,dummy
free_lun,lun,/force
if dummy eq '' then begin
	date_string = ''
endif else begin
	source = dummy
	pos1 = strpos(source,"'")
	pos2 = strpos(source,"'",pos1+1)
	pos3 = strpos(source,"'",pos2+1)
	pos4 = strpos(source,"'",pos3+1)
	length = pos4-pos3-1
	date_string = strmid(source,pos3+1,length)
endelse
return,date_string
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function convert_tasmonth,month
months =	['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct', $
			 'Nov','Dec']
month_num = 1+(where(strupcase(months) eq strupcase(month)))[0]
return,month_num
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function convert_tasdate,date_string
; This function converts the date string into a Julian date
; Decompose the date string into individual elements based on
; the separator (blank or hyphen).
; Is the primary delimiter a hyphen or a space?
hyphen = strpos(date_string,'-')
if hyphen[0] eq -1 then begin	; blank is the delimiter
	separator = ' '
	result = strsplit(date_string,separator,/extract)
	month_string = result[0]
	month_num = convert_tasmonth(month_string)
	day = fix(result[1]) & year = fix(result[2]) & time = result[3]
	separator = ':'
	result = strsplit(time,separator,/extract)
	hour = fix(result[0]) & minute = fix(result[1])
endif else begin				; hyphen is the delimiter
	separator = '-'
	result = strsplit(date_string,separator,/extract)
	day = fix(result[0])
	month_num = convert_tasmonth(result[1])
	year_and_time = result[2]
	result = strsplit(year_and_time,' ',/extract)
	year = fix(result[0]) & time = result[1]
	result = strsplit(time,':',/extract)
	hour = fix(result[0]) & minute = fix(result[1])
endelse
julian_date = JULDAY(month_num, day, year, hour, minute)
return,julian_date
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::set,		wid = wid,							$
								indep_var_index = indep_var_index
if n_elements(indep_var_index) ne 0 then begin
	data = *self.data_ptr
	dsize = size(data)
	if dsize[1] lt indep_var_index then return,0
	*self.x_ptr = reform(data[indep_var_index,*])
	self.indep_var_index = indep_var_index
	self.xlabel = (*self.prp_ptr)[indep_var_index]
endif
if n_elements(wid) ne 0 then self.wid = wid
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::get,		display_name = display_name,		$
								filename = filename,				$
								path = path,						$
								xvals = xvals,						$
								xlabel = xlabel,					$
								ylabel = ylabel,					$
								qty = qty,							$
								header = header,					$
								prp = prp,							$
								hnum = hnum,						$
								hdef = hdef,						$
								text = text,						$
								date_string = date_string,			$
								julian_date = julian_date,			$
								dqty = dqty,						$
								wid = wid,							$
								indep_var_index = indep_var_index


if arg_present(indep_var_index) then begin
	indep_var_index = self.indep_var_index
endif
if arg_present(wid) then wid = self.wid
if arg_present(prp) then prp = *self.prp_ptr
if arg_present(hnum) then hnum = *self.hnum_ptr
if arg_present(hdef) then hdef = *self.hdef_ptr
if arg_present(text) then text = *self.text_ptr
if arg_present(header) then header = *self.header_ptr
if arg_present(xvals) then xvals = *self.x_ptr
if arg_present(qty) then qty = *self.y_ptr
if arg_present(dqty) then dqty = *self.yerr_ptr
if arg_present(display_name) then display_name = self.display_name
if arg_present(filename) then filename = self.filename
if arg_present(path) then path = self.path
if arg_present(xlabel) then xlabel = self.xlabel
if arg_present(ylabel) then ylabel = self.ylabel
if arg_present(date_string) or arg_present(julian_date) then begin
	source = (*self.header_ptr)[0]
	target = "'"+self.display_name+"'"
	length = strlen(target)
	new_string = strmid(source,length+2)
	quote_pos = strpos(new_string,"'")
	date_string = strmid(new_string,0,quote_pos)
	julian_date = convert_tasdate(date_string)
endif
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::plot
x = *self.x_ptr
y = *self.y_ptr
yerr = *self.yerr_ptr
plot,x,y,psym = 4,xtitle = self.xlabel,ytitle = self.ylabel, $
	title = self.display_name
errplot,x,y-yerr,y+yerr,width = 0.0
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::read_in_data
tas_ptr = ptr_new(/allocate_heap)
*tas_ptr = opan_tasrd(self.filename)
self.points = (*tas_ptr).points
self.heads = (*tas_ptr).heads
self.pars = (*tas_ptr).pars
*self.header_ptr = (*tas_ptr).header
*self.text_ptr = (*tas_ptr).text
*self.hnum_ptr = (*tas_ptr).hnum
*self.prp_ptr = (*tas_ptr).prp
*self.data_ptr = (*tas_ptr).data


whereI = where((*tas_ptr).prp eq 'I',count)
*self.y_ptr = (*tas_ptr).data[whereI[0],*]
*self.yerr_ptr = sqrt(*self.y_ptr)
; Now cycle through the fields from 0 to whereI[0]-1 and determine
; which parameter is varying...this is the independent variable
index = -1
for j = whereI[0]-1,0,-1 do begin
	d = (*tas_ptr).data[j,*]
	if n_elements(d) eq 1 then begin
		index = 0
	endif else begin
		if (*tas_ptr).data[j,1] ne (*tas_ptr).data[j,0] then begin
			index = j
		endif
	endelse
endfor
self.indep_var_index = index
*self.x_ptr = (*tas_ptr).data[index,*]
self.xlabel = (*tas_ptr).prp[index]
self.ylabel = 'Intensity'
ptr_free,tas_ptr
self.wid = 0L

return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::get_filedates
; Here we must open up the files, read in the first lines, get out the dates,
; then close them.
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function tasred_data::init,		filename = filename,		$
								path = path,				$
								errmsg = errmsg

catch,the_error
if the_error ne 0 then begin
	catch,/cancel
	;void = dialog_message(!error_state.msg)
	errmsg = !error_state.msg
	if ptr_valid(tas_ptr) then ptr_free,tas_ptr
	if ptr_valid(self.hdef_ptr) then ptr_free,self.hdef_ptr
	if ptr_valid(self.hnum_ptr) then ptr_free,self.hnum_ptr
	if ptr_valid(self.prp_ptr) then ptr_free,self.prp_ptr
	if ptr_valid(self.data_ptr) then ptr_free,self.data_ptr
	if ptr_valid(self.header_ptr) then ptr_free,self.header_ptr
	if ptr_valid(self.x_ptr) then ptr_free,self.x_ptr
	if ptr_valid(self.y_ptr) then ptr_free,self.y_ptr
	if ptr_valid(self.yerr_ptr) then ptr_free,self.yerr_ptr
	return,0
endif
errmsg = 'No errors generated initializing TAS DATA object'
if n_elements(filename) eq 0 then begin
	errmsg = 'A filename must be provided'
	return,0
endif
if n_elements(path) eq 0 then begin
	errmsg = 'A path must be provided'
	return,0
endif

self.display_name = strmid(filename,strlen(path))
self.filename = filename
self.hdef_ptr = ptr_new(/allocate_heap)
self.text_ptr = ptr_new(/allocate_heap)
self.hnum_ptr = ptr_new(/allocate_heap)
self.prp_ptr = ptr_new(/allocate_heap)
self.data_ptr = ptr_new(/allocate_heap)
self.header_ptr = ptr_new(/allocate_heap)
self.x_ptr = ptr_new(/allocate_heap)
self.y_ptr = ptr_new(/allocate_heap)
self.yerr_ptr = ptr_new(/allocate_heap)
ret = self->get_filedates()
ret = self->read_in_data()
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro tasred_data__define

define =	{	tasred_data,				$
				filename:'',				$	; path-qualified
				path:'',					$
				display_name:'',			$
				points:0L,					$
				heads:0L,					$
				pars:0L,					$
				text_ptr:ptr_new(),			$
				hdef_ptr:ptr_new(),			$
				hnum_ptr:ptr_new(),			$
				prp_ptr:ptr_new(),			$
				data_ptr:ptr_new(),			$
				x_ptr:ptr_new(),			$
				y_ptr:ptr_new(),			$
				yerr_ptr:ptr_new(),			$
				xlabel:'',					$
				ylabel:'',					$
				wid:0L,						$
				indep_var_index:0,			$
				header_ptr:ptr_new()		}

end
