; $Id$
;###############################################################################
;
; NAME:
;  Func_2D__define
;
; PURPOSE:
;  Class definition for a fit function component in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN, curve fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::Cleanup
  Ptr_free,self.extra
  Ptr_free,self.xptr
  ;ptr_free,self.qPtr
  Ptr_free,self.yptr
  Ptr_free,self.stepptr
  Ptr_free,self.ymaxptr,self.xmodeptr
  Ptr_free,self.lowptr,self.lowvalptr
  Ptr_free,self.highptr,self.highvalptr
  Ptr_free,self.fixptr,self.fixedvalptr
  Ptr_free,self.parmerrorptr
  Ptr_free,self.tiedptr
  Ptr_free,self.parmptr,self.resptr
end;Func_2D::cleanup


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function Func_2D::getDrawMessage
  y = Call_function(self.name,*self.xptr,*self.parmptr, $
    expr = (*Self.expr), qVals=(*Self.qvals), $
    extra = *self.extra,drawMessage = drawMessage, $
    fit_fun_filename = self.fit_fun_filename)
  Return,drawMessage
end;Func_2D::getDrawMessage


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::changefirst,x,y,xrange
  ; Called when the user is initializing the function graphically.

  if  (Self.candraw eq 0) then Return ; no need to proceed

  parms = *self.parmptr

  if N_elements(*self.resptr) eq 0 then begin
    calc = Call_function(self.name,*self.xptr,parms, $
      expr = (*Self.expr), extra = *self.extra, $
      func_dataHash = Self.func_data, qVals=(*Self.qvals), $
      qgroup=self.qgroup,groupNumber=self.groupnumber, $
      /changefirst, xMouseClick=x, yMouseClick=y, xrange=xrange)
  endif else begin
    calc = Call_function(self.name,*self.xptr,parms, $
      expr = (*Self.expr), resPtr = self.resptr,extra = *self.extra, $
      func_dataHash = Self.func_data, qVals=(*Self.qvals), $
      qgroup=self.qgroup,groupNumber=self.groupnumber, $
      /changefirst, xMouseClick=x, yMouseClick=y, xrange=xrange)
  endelse

  *Self.parmptr = parms
  *self.yptr = calc

  Return

end;Func_2D::changefirst


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::changesecond,x,y,xrange
  ; Called when the user is initializing the function graphically.


  if  (Self.candraw eq 0) then Return ; no need to proceed

  parms = *self.parmptr

  if N_elements(*self.resptr) eq 0 then begin
    calc = Call_function(self.name,*self.xptr,parms, $
      expr = (*Self.expr), extra = *self.extra, $
      func_dataHash = Self.func_data, qVals=(*Self.qvals), $
      qgroup=self.qgroup,groupNumber=self.groupnumber, $
      /changesecond, xMouseClick=x, yMouseClick=y, xrange=xrange)
  endif else begin
    calc = Call_function(self.name,*self.xptr,parms, $
      expr = (*Self.expr), resPtr = self.resptr,extra = *self.extra, $
      func_dataHash = Self.func_data, qVals=(*Self.qvals), $
      qgroup=self.qgroup,groupNumber=self.groupnumber, $
      /changesecond, xMouseClick=x, yMouseClick=y, xrange=xrange)
  endelse

  *Self.parmptr = parms
  *self.yptr = calc

  Return
end;Func_2D::changesecond


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::draw,overplot = overplot
  if N_elements(overplot) eq 0 then overplot = 0
  yVals = *self.yptr
  y = (yVals.ndim eq 2)? yVals[*self.groupnumber -1] : yVals
  if overplot eq 0 then begin
    Plot,*self.xptr,y,psym = 0
  endif else begin
    Oplot,*self.xptr,y,psym = 0
  endelse
end;Func_2D::draw


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::setproperty, $
  name = name,$
  xvalues = xvalues, $
  xunits = xunits, $
  qgroup=qgroup,$
  groupNumber=groupNumber,$
  func_dataHash=func_dataHash, $
  step = step, $
  parms = parms, $
  fixed = fixed, $
  fixvalues = fixvalues, $
  low = low, $
  lovalues = lovalues, $
  high = high, $
  hivalues = hivalues, $
  parmError = parmError, $
  canDraw = canDraw, $
  active = active, $
  tied = tied, $
  expr = expr, $
  rlimit = rlimit,  $
  resptr = resptr,  $
  resolutionRequiredFlag=resolutionRequiredFlag, $ ; is instrumental resolution part of the function definition?
  extConvolFlag=extConvolFlag, $                   ; external numerical convolution with resolution necessary
  qVals=qVals, twoDimFlag=twoDimFlag, $
  groupsToBeFitted=groupsToBeFitted, $
  calculate = calculate ; set this keyword to populate self.yPtr with
  ; the updated values

  !except = 0
  if N_elements(rlimit) ne 0 then self.rlimit = rlimit
  if N_elements(resptr) ne 0 then begin
    if Ptr_valid(resptr) ne 0 then *self.resptr = *resptr
  endif

  if N_elements(xunits) ne 0 then self.xunits = xunits
  if N_elements(resolutionRequiredFlag) ne 0 then Self.resolutionRequiredFlag = resolutionRequiredFlag
  if N_elements(extConvolFlag) ne 0 then self.extconvolflag = extConvolFlag
  if N_elements(expr) ne 0 then (*self.expr) = expr
  if N_elements(canDraw) ne 0 then self.candraw = canDraw
  if N_elements(active) ne 0 then Self.active = active
  if N_elements(tied) ne 0 then *self.tiedptr = tied
  if N_elements(step) ne 0 then *self.stepptr = step
  if N_elements(name) ne 0 then self.name = name
  if N_elements(xvalues) ne 0 then *self.xptr = xvalues
  if n_elements(xunits) ne 0 then self.xunits = xunits
  if N_elements(qgroup) ne 0 then self.qgroup=qgroup;*self.qPtr = qgroup
  if N_elements(groupNumber) ne 0 then self.groupnumber=groupNumber
  if N_elements(qVals) ne 0 then (*self.qvals)=qVals
  if N_elements(twoDimFlag) ne 0 then self.twodimflag=twoDimFlag
  if Isa(func_dataHash,'hash') then Self.func_data = func_dataHash
  if N_elements(parms) ne 0 then *self.parmptr = parms
  if N_elements(parmError) ne 0 then *self.parmerrorptr = parmError
  if N_elements(fixed) ne 0 then *self.fixptr = fixed
  if N_elements(fixvalues) ne 0 then *self.fixedvalptr = fixvalues
  if N_elements(low) ne 0 then *self.lowptr = low
  if N_elements(lovalues) ne 0 then *self.lowvalptr = lovalues
  if N_elements(high) ne 0 then *self.highptr = high
  if N_elements(hivalues) ne 0 then *self.highvalptr = hivalues
  ; Now that we have all of the information, calculate the function if
  ; the user so desires.
  if Keyword_set(calculate) then begin
    if N_elements(*self.xptr) ne 0 then begin
      if (N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0) then begin
        *self.yptr = Call_function(self.name,*self.xptr,*self.parmptr, $
          expr = (*Self.expr), extra = *self.extra, $
          func_dataHash = Self.func_data, qVals=(*Self.qvals), groupsToBeFitted=groupsToBeFitted, $
          qgroup=self.qgroup,groupNumber=self.groupnumber)  ; fit_fun_filename = self.fit_fun_filename,
      endif else begin
        *self.yptr = Call_function(self.name,*self.xptr,*self.parmptr, $
          expr = (*Self.expr), resPtr = self.resptr,extra = *self.extra, $
          func_dataHash = Self.func_data, qVals=(*Self.qvals), groupsToBeFitted=groupsToBeFitted, $
          qgroup=self.qgroup,groupNumber=self.groupnumber)  ; fit_fun_filename = self.fit_fun_filename,
      endelse
    endif else begin
      Print,'Error...no valid x-values'
    endelse
  endif
end;Func_2D::setProperty


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D::getproperty, $
  name = name,  $
  xvalues = xvalues, $
  qgroup=qgroup,$
  groupNumber=groupNumber,$
  qVals=qVals, twoDimFlag=twoDimFlag, $
  func_dataHash=func_dataHash, $
  yvalues = yvalues,  $
  step = step, $
  parms = parms, $
  fixed = fixed, $
  fixvalues = fixvalues, $
  low = low, $
  lovalues = lovalues, $
  high = high, $
  hivalues = hivalues,$
  parmnames = parmnames,$
  parmError = parmError, $
  tied = tied, $
  canDraw = canDraw, $
  active = active, $
  fit_fun_filename = fit_fun_filename,  $
  single_parmnames = single_parmnames,   $
  multi_parmnames = multi_parmnames,   $
  xunits=xunits, $
  resolutionRequiredFlag = resolutionRequiredFlag, $    ; is instrumental resolution part of the function definition?
  extConvolFlag=extConvolFlag, $                   ; external numerical convolution with resolution necessary
  expr = expr
  ;if arg_present(parmnames) then begin
  ;  z = Call_function(self.name,parmnames = parmnames,expr=(*Self.expr))
  ;  ;qgroup = self.qgroup;*self.qPtr
  ;  ;;SO DO I ENTER THE QVALUES IN THE FOLLOWING MANNER OR DO I HAVE TO ENTER THEM VIA *self.extra?????????
  ;  ;;THIS IS WHERE ALL THE ACTUAL EVALUATION HAPPENS, AND THUS WHERE THE Qgroup VALUE NEEDS TO BE ENTERED.
  ;  ;if self.expr eq '' then begin
  ;  ;  z = call_function(self.name,parmnames = parmnames, $
  ;  ;    extra = *self.extra,fit_fun_filename = self.fit_fun_filename,qgroup=qgroup)
  ;  ;endif else begin
  ;  ;  z = call_function(self.name,*self.xPtr,*self.parmPtr, $
  ;  ;                    parmnames = parmnames, $
  ;  ;                    expr = self.expr, extra = *self.extra, $
  ;  ;                    fit_fun_filename = self.fit_fun_filename,qgroup=qgroup)
  ;  ;endelse
  ;endif

  if arg_present(xunits) then xunits = Self.xunits
  if Arg_present(single_parmnames) then single_parmnames = *self.single_parmnamesptr
  if Arg_present(multi_parmnames) then multi_parmnames = *self.multi_parmnamesptr
  if Arg_present(resolutionRequiredFlag) then resolutionRequiredFlag = self.resolutionRequiredFlag
  if Arg_present(extConvolFlag) then extConvolFlag = self.extconvolflag
  if Arg_present(fit_fun_filename) then fit_fun_filename = self.fit_fun_filename
  if Arg_present(expr) then expr = (*self.expr)
  if Arg_present(canDraw) then canDraw = self.candraw
  if Arg_present(active) then active = Self.active
  if Arg_present(tied) then tied = *self.tiedptr
  if Arg_present(step) then step = *self.stepptr
  if Arg_present(name) then name = self.name
  if Arg_present(xvalues) then xvalues = *self.xptr
  if arg_present(xunits) then xunits = self.xunits
  if Arg_present(qgroup) then qgroup = self.qgroup;*self.qPtr
  if Arg_present(groupNumber) then groupNumber = self.groupnumber
  if Arg_present(twoDimFlag) then twoDimFlag = self.twodimflag
  if Arg_present(qVals) then qVals = *self.qvals
  if Arg_present(yvalues) then yvalues = *self.yptr
  if Arg_present(parmNames) then parmNames = *self.parmnamesptr
  if Arg_present(parms) then parms = *self.parmptr
  if Arg_present(parmError) then parmError = *self.parmerrorptr
  if Arg_present(fixed) then fixed = *self.fixptr
  if Arg_present(fixvalues) then fixvalues = *self.fixedvalptr
  if Arg_present(low) then low = *self.lowptr
  if Arg_present(lovalues) then lovalues = *self.lowvalptr
  if Arg_present(high) then high = *self.highptr
  if Arg_present(hivalues) then hivalues = *self.highvalptr
  if Arg_present(func_dataHash) then func_dataHash=Self.func_data
end;Func_2D::getProperty


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function Func_2D::init, $
  name = name, $
  xvalues = xvalues,$
  datSize = datSize,$         ; dimensions of the expr't data to be fitted
  qgroup = qgroup,$           ;THIS MAKES func AWARE OF THE Q-VALUE FOR ITS GROUP
  groupNumber = groupNumber,$ ;THIS MAKES func AWARE OF THE NUMBER OF THE GROUP IT BELONGS TO
  workDir = workDir, $
  wTLB=wTLB, $
  func_dataHash=func_dataHash, $
  qVals=qVals, twoDimFlag=twoDimFlag, $
  fit_fun_filename = fit_fun_filename, $
  step = step, $
  parms = parms, $
  fixed = fixed, $
  fixvalues = fixvalues, $
  low = low, $
  lovalues = lovalues, $
  high = high, $
  hivalues = hivalues, $
  tied = tied, $
  resPtr = resPtr, $
  rlimit = rlimit, $
  expr = expr, $
  xunits = xunits,      $
  resolutionRequiredFlag=resolutionRequiredFlag, $   ; is instrumental resolution part of the function definition?
  extConvolFlag=extConvolFlag, $                   ; external numerical convolution with resolution necessary
  active = active, $
  _Extra = extra

  ; Name must be present!
  self.extra = Ptr_new(extra)
  if N_elements(name) eq 0 then Return,-1
  self.parmerrorptr = Ptr_new(/allocate_heap)
  self.name = name
  self.stepptr = Ptr_new(/allocate_heap)
  self.xptr = Ptr_new(/allocate_heap)
  self.yptr = Ptr_new(/allocate_heap)
  if N_elements(xvalues) eq 0 then begin
    xlo = -10.0 & xhi = 10.0 & nx = 100
    dx = (xhi-xlo)/(nx-1.0) & x = xlo+dx*Findgen(nx)
    *self.xptr = x
  endif else begin
    *self.xptr = xvalues
  endelse
  if (N_elements(qVals) eq 0) then qVals = 0.0
  Self.qvals = Ptr_new(qVals)
  Self.twodimflag = (N_elements(twoDimFlag) eq 0)? 0 : twoDimFlag
  if N_elements(qgroup) eq 0 then qgroup = 0.0
  self.qgroup = qgroup
  if N_elements(groupNumber) eq 0 then groupNumber=1
  self.groupnumber = groupNumber
  if N_elements(resolutionRequiredFlag) eq 0 then resolutionRequiredFlag=0
  self.resolutionRequiredFlag = resolutionRequiredFlag
  if N_elements(extConvolFlag) eq 0 then extConvolFlag=1
  self.extconvolflag = extConvolFlag
  if n_elements(xunits) eq 0 then xunits = ''
  Self.xunits = xunits

  ; Let's fill up the function with some dummy data if none is given
  ; when instantiating the class....
  self.resptr = Ptr_new(/allocate_heap)
  self.parmptr = Ptr_new(/allocate_heap)
  self.parmnamesptr = Ptr_new(/allocate_heap)
  self.single_parmnamesptr = Ptr_new(/allocate_heap)
  self.multi_parmnamesptr = Ptr_new(/allocate_heap)
  self.lowptr = Ptr_new(/allocate_heap)
  self.lowvalptr = Ptr_new(/allocate_heap)
  self.highptr = Ptr_new(/allocate_heap)
  self.highvalptr = Ptr_new(/allocate_heap)
  self.fixptr = Ptr_new(/allocate_heap)
  self.tiedptr = Ptr_new(/allocate_heap)
  self.fixedvalptr = Ptr_new(/allocate_heap)
  self.ymaxptr = Ptr_new(/allocate_heap)
  self.xmodeptr = Ptr_new(/allocate_heap)
  if (Isa(func_dataHash,'hash')) then begin
    ; this is executed when making a copy of an existing function
    Self.func_data = func_dataHash
  endif else begin
    ; this is executed when making a brand new object
    if (Float((strtok(!version.release,/extract))[0]) ge 8.4) then begin
      func_dataHash = Hash(/fold_case) ; use fold_case keyword to make keys case insensitive for the hash
    endif else begin
      func_dataHash = Hash()           ; fold_case keyword only for IDL 8.4 and newer
    endelse
    func_dataHash['initialized'] = 0
    Self.func_data = func_dataHash
  endelse

  if N_elements(xunits) ne 0 then self.xunits = xunits
  self.fit_fun_filename = ""
  if (N_elements(expr) ne 0) then begin
    Self.expr = Ptr_new(expr)
  endif else if N_elements(fit_fun_filename) ne 0 then begin
    self.fit_fun_filename = fit_fun_filename
    nlines = File_lines(fit_fun_filename)
    expr = Strarr(nlines)
    Openr,lun,fit_fun_filename,/get_lun
    Readf, lun, expr
    Free_lun,lun,/force
    Self.expr = Ptr_new(expr)
  endif else begin
    expr = []
    Self.expr = Ptr_new('')
  endelse


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Call function without x parameter to determine parameter names, inittial parameter guesses, etc
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  if ((N_elements(expr) gt 0) && expr[0].Strlen() ne 0) then begin
    ; Test if we're defining a user function.  We are explicitly
    ; assuming here that the xvalues, expression, parms, and
    ; parmnames are being supplied here through keywords!!!!

    *self.parmptr = parms
    dummy = Call_function(self.name, $   ;(self.name,xvalues,parms, $
      Qgroup=Qgroup,xunits=xunits,qVals=qVals,xVals=xValues, $
      func_dataHash=func_dataHash, $
      groupNumber=groupNumber,$
      parmnames = parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames, $
      twoDimFlag=twoDimFlag, initParms = parms, $
      expr = (*Self.expr), $
      fit_fun_filename = self.fit_fun_filename, $
      canDraw = canDraw, $
      _Extra=extra)
    nparms = N_elements(parmnames)
  endif else begin
    ; Now take care of all of the library of fit functions
    ; Call the function to get out the parameter names and initial guess parameters
    status = Call_function(self.name,parmnames = parmnames, twoDimFlag=twoDimFlag, initParms = parms $
      ,single_parmNames=single_parmNames, multi_parmNames=multi_parmNames,xunits=xunits, Qgroup=Qgroup $
      ,groupNumber=groupNumber,func_dataHash=func_dataHash, datSize=datSize,qVals=qVals,xVals=xValues $
      ,wTLB=wTLB, workDir=workDir,_Extra=extra ) ;extra = *self.extra)
    nparms = N_elements(parmnames)
    if (status eq 0) then Return, 0
  endelse

  if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then *self.resptr = *resPtr
  if N_elements(rlimit) ne 0 then self.rlimit = rlimit
  unity = 1.0+0.0*Findgen(nparms)
  zerof = 0.0*unity
  zeroi = 0*Indgen(nparms)
  *self.tiedptr = N_elements(tied) eq 0 ? Replicate(0,nparms) : tied
  *self.stepptr = N_elements(step) eq 0 ? Replicate(0D,nparms) : step
  *self.parmerrorptr = Replicate(0D,nparms)
  *self.parmptr = N_elements(parms) eq 0 ? unity : parms
  if (N_elements(parmnames) gt 0) then (*Self.parmnamesptr) = parmnames
  if (N_elements(single_parmnames) gt 0) then (*Self.single_parmnamesptr) = single_parmnames
  if (N_elements(multi_parmnames) gt 0) then (*Self.multi_parmnamesptr) = multi_parmnames

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Call function a second time to evaluate it and save the calculation
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  canDraw = 0
  if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
    *self.yptr = Call_function(self.name,*self.xptr,*self.parmptr, $
      Qgroup=Qgroup,xunits=xunits,$
      func_dataHash=func_dataHash, $
      groupNumber=groupNumber,qVals=qVals,$
      expr = (*Self.expr), canDraw = canDraw, $
      fit_fun_filename = self.fit_fun_filename,  $
      resPtr = self.resptr, $
      rlimit = self.rlimit, $
      _Extra=extra)
  endif else if (Strupcase(self.name) ne 'PAN_QENS') then begin
    *self.yptr = Call_function(self.name,*self.xptr,*self.parmptr, $, $
      Qgroup=Qgroup,xunits=xunits,qVals=qVals,$
      groupNumber=groupNumber,$
      func_dataHash=func_dataHash, $
      fit_fun_filename = self.fit_fun_filename,  $
      expr = (*Self.expr), canDraw = canDraw, $
      _Extra=extra)
  endif

  self.candraw = canDraw
  if (N_elements(active) eq 0) then active = 1
  Self.active = active

  *self.fixptr = N_elements(fixed) eq 0 ? zeroi : fixed
  *self.fixedvalptr = N_elements(fixvalues) eq 0 ? *self.parmptr : fixvalues
  *self.lowptr = N_elements(low) eq 0 ? zeroi : low
  *self.lowvalptr = N_elements(lovalues) eq 0 ? zerof : lovalues
  *self.highptr = N_elements(high) eq 0 ? zeroi : high
  *self.highvalptr = N_elements(hivalues) eq 0 ? zerof : hivalues
  Return,1

end;Func_2D::init


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro Func_2D__define
  ;
  define = {Func_2D, $
    name:"", $
    expr:Ptr_new(/allocate_heap), $
    fit_fun_filename:"",  $
    candraw:0, $
    active:0, $                 ; flag used to determine if function is currently active (1) or not (0). Can be used to temporarily switch a function off
    func_data:Obj_new(),  $     ; a hash object that can be used to store misc function data
    extra:Ptr_new(/allocate_heap), $
    ymaxptr:Ptr_new(/allocate_heap), $
    xmodeptr:Ptr_new(/allocate_heap), $
    parmptr:Ptr_new(/allocate_heap), $
    parmnamesptr:Ptr_new(/allocate_heap), $
    parmerrorptr:Ptr_new(/allocate_heap), $
    single_parmnamesptr:Ptr_new(),        $
    multi_parmnamesptr:Ptr_new(),         $
    ;NEW FOR QGroup VALUE
    qgroup:0.0,$;ptr_new(/allocate_heap),$
    groupnumber:0,$
    xunits:'', $
    xptr:Ptr_new(/allocate_heap),  $
    yptr:Ptr_new(/allocate_heap),  $
    qvals:Ptr_new(/allocate_heap),  $     ; ptr to group values
    twodimflag:0, $                       ; flag indicating whether this is a 2D==1 or 1D==0 (default) function
    stepptr:Ptr_new(/allocate_heap), $
    fixptr:Ptr_new(/allocate_heap),  $
    fixedvalptr:Ptr_new(/allocate_heap),  $
    lowptr:Ptr_new(/allocate_heap),  $
    lowvalptr:Ptr_new(/allocate_heap),  $
    highptr:Ptr_new(/allocate_heap),  $
    tiedptr:Ptr_new(/allocate_heap), $
    resptr:Ptr_new(/allocate_heap), $
    resolutionRequiredFlag:0, $          ; is instrumental resolution part of the function definition?
    extconvolflag:0, $                  ; external numerical convolution with resolution necessary
    rlimit:[0.0,0.0], $
    highvalptr:Ptr_new(/allocate_heap) $
  }
end;Func_2D__define