; $Id$
;###############################################################################
;
; NAME:
;  DAVE_READ_ASCII
;
; PURPOSE:
;  Function that reads in various formats of ascii data.
;
; CATEGORY:
;  DAVE, ASCII data reader.
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
; +
; NAME:
;
;  DAVE_READ_ASCII
;
; PURPOSE:
;
;  This function reads in an ASCII file that belongs to one of
;  a number of different classes of ASCII files produced on various
;  neutron spectrometers.  If it cannot identify the format then
;  it returns 0.  Otherwise it returns a structured variable
;  containing the data.
;
; AUTHOR:
;
;  Robert Dimeo
;  National Institute of Standards and Technology
;  Center for Neutron Research
;  100 Bureau Drive, Mail Stop 8562
;  Gaithersburg, MD 20899
;  Tel: (301) 975-8135
;  Email: robert.dimeo@nist.gov
;
; CATEGORY:
;
;  DAVE application
; -
; ********************************* ;
function dave_read_ascii_read_neat,filename,err = err,emsg = emsg
err = 0
emsg = 'No errors generated'
catch,the_error
if the_error ne 0 then begin
   if n_elements(lun) ne 0 then free_lun,lun,/force
   err = 1
   emsg = 'Problem reading in file'
   return,0
endif
dum = ''
openr,lun,filename,/get_lun

readf,lun,dum
reads,strtrim(dum,2),v1,v2,v3,format = '(f10.5,e10.4,i1)'

nx = fix(v1) & ny = fix(v2)
dat = fltarr(3,nx)
data = fltarr(3,nx,ny)
dummy = ''
y = fltarr(ny)
for i = 0,ny-1 do begin
   readf,lun,dummy
   this_string = strsplit(dummy,/extract)
   y[i] = float(this_string[0])
   readf,lun,dat
   data[0:2,0:nx-1,i] = dat[0:2,0:nx-1]
endfor
free_lun,lun,/force
index = 1
energy = data[0,*,0]
esort = sort(energy)
energy = energy[esort]

nx = n_elements(esort)
x = reform(energy)
qty = reform(data[1,esort,0:ny-1],nx,ny)
dqty = reform(data[2,esort,0:ny-1],nx,ny)
ny = n_elements(y)

zeros = where(qty eq 0.0,count_zeros)
if count_zeros gt 0 then qty[zeros] = 1.0

d = {x:x,y:y,qty:qty,dqty:dqty,xunits:'E:mev',yunits:'ANGLE:degrees'}
return,d
end
; ****************************************** ;
function dave_read_iris_fast,filename,err = err,emsg = emsg
err = 0B & emsg = ''
dstr = 0
catch,the_error
if the_error ne 0 then begin
   catch,/cancel
   emsg = !error_state.msg
   err = 1B
   message,emsg,/informational
   if n_elements(lun) ne 0 then free_lun,lun,/force
   return,0
endif
dummy = ''
openr,lun,filename,/get_lun
; Read in the first few lines
readf,lun,dummy
readf,lun,dummy
readf,lun,dummy   ; the 3rd line
this_string = strsplit(dummy,/extract)
ny = fix(this_string[1])
nx = fix(this_string[2])
; Initialize some arrays
angles = fltarr(ny)
datStr = {instring:'',data:fltarr(3,nx)}
str_arr = replicate(datStr,ny)
; Read in one giant structure array containing all of the
; data all at one time.
readf,lun,str_arr
free_lun,lun,/force

x = reform(str_arr[0].data[0,*])
data = reform(str_arr.data[1,*])
error = reform(str_arr.data[2,*])
for i = 0,ny-1 do angles[i] = float((strsplit(str_arr[i].instring,/extract))[0])
ef = float((strsplit(str_arr[0].instring,/extract))[3])
catch,/cancel

dstr = { x:x,y:angles,              $
         qty:data,                  $
         dqty:error,                $
         xunits:'E:mev',            $
         yunits:'ANGLE:degrees'     }
return,dstr
end
; ********************************* ;
function dave_read_ascii_read_iris,filename,err = err,emsg = emsg
err = 0
emsg = 'No errors generated'
catch,the_error
if the_error ne 0 then begin
   if n_elements(lun) ne 0 then free_lun,lun,/force
   err = 1
   emsg = 'Problem reading in file'
   return,0
endif
count = 0L
dummy = ''
openr,lun,filename,/get_lun
readf,lun,dummy
readf,lun,dummy
readf,lun,dummy   ; the 3rd line
this_string = strsplit(dummy,/extract)
ny = fix(this_string[1])
nx = fix(this_string[2])
angle = fltarr(ny)
dat = fltarr(3,nx)
data = fltarr(3,nx,ny)
for i = 0,ny-1 do begin
   ; read in the header line
   readf,lun,dummy
   angle[i] = float((strsplit(dummy,/extract))[0])
   readf,lun,dat
   data[0:2,*,i] = dat[0:2,*]
endfor
free_lun,lun,/force

y = angle
energy = data[0,*,0]
esort = sort(energy)
energy = energy[esort]

nx = n_elements(esort)
x = reform(energy)
qty = reform(data[1,esort,0:ny-1],nx,ny)
dqty = reform(data[2,esort,0:ny-1],nx,ny)

zeros = where(qty eq 0.0,count_zeros)
if count_zeros gt 0 then dqty[zeros] = 1.0
d = {x:x,y:y,qty:qty,dqty:dqty,xunits:'E:mev',yunits:'ANGLE:degrees'}
return,d
end
; ********************************* ;
function dave_read_ascii_read_qens,filename,err = err,emsg = emsg
err = 0
emsg = 'No errors generated'
catch,the_error
if the_error ne 0 then begin
   if n_elements(lun) ne 0 then free_lun,lun,/force
   err = 1
   emsg = 'Problem reading in file'
   return,0
endif
count = 0L
dum = ''
openr,lun,filename,/get_lun

while not eof(lun) do begin
   readf,lun,dum
   if count eq 0L then contents = dum else $
      contents = [contents,dum]
   count = count+1L
endwhile
free_lun,lun,/force

; Now that the data file has been successfully read in, let's
; parse out the necessary information

; How many commas are there in the first row?  This is the number
; of columns...
ncommas = n_elements(strsplit(contents[0],','))
nx = count-1
ny = ncommas  ; number of groups
array = fltarr(nx,ny)
for i = 0L,nx-1 do begin
   txt_array = strsplit(contents[i+1],',',/extract)
   array[i,0:ny-1] = float(txt_array)
endfor
energy = array[0:nx-1,0]
qty = float(array[0:nx-1,1:ny-2:2])
dqty = float(array[0:nx-1,2:ny-1:2])

labels = strsplit(contents[0],',',/extract)
ylabels = labels[1:ny-1]
ytitles = strarr(ny-1)
y = fltarr(ny-1)
for i = 0,ny-2 do begin
   dot_pos = strpos(ylabels[i],'.')
   substring = strmid(ylabels[i],dot_pos-1,4)
   y[i] = float(substring)
endfor
x = energy
y = y[0:ny-2:2]
where_zero = where(qty eq 0.0,count_zero)
if count_zero gt 0 then dqty[where_zero] = 1.0
d = {x:x,y:y,qty:qty,dqty:dqty,xunits:'E (meV)',yunits:'wavevector:A-1'}
return,d
end
; ********************************* ;
function dave_read_ascii_read_prova,filename,err = err,emsg = emsg

;LRK 062207
;ADDING THIS TO HELP JOHN AND DANIELLA WITH A PROJECT.
err = 0
emsg = 'No errors generated'
catch,the_error
if the_error ne 0 then begin

	print,'TESTING PROVA'
   if n_elements(lun) ne 0 then free_lun,lun,/force
   err = 1
   emsg = 'Problem reading in file'
   return,0
endif
count = 0L
dum = ''
openr,lun,filename,/get_lun

while not eof(lun) do begin
   readf,lun,dum
   if count eq 0L then contents = dum else $
      contents = [contents,dum]
   count = count+1L
endwhile
free_lun,lun,/force

; Now that the data file has been successfully read in, let's
; parse out the necessary information

; How many commas are there in the first row?  This is the number
; of columns...
ncommas = n_elements(strsplit(contents[0],','))
nx = count-1
ny = ncommas  ; number of groups
array = fltarr(nx,ny)
for i = 0L,nx-1 do begin
   txt_array = strsplit(contents[i+1],',',/extract)
   array[i,0:ny-1] = float(txt_array)
endfor
energy = array[0:nx-1,0]

qty = float(array[0:nx-1,1:ny-1])

;NO ERROR BARS IN THIS FILE, SO FILL IN dqty WITH 1.0's
dqty = 0.0*float(array[0:nx-1,1:ny-1]) + 1.0

labels = strsplit(contents[0],',',/extract)
;;;;;ylabels = labels[1:ny-1]
;;;;;ytitles = strarr(ny-1)
;;;;;y = fltarr(ny-1)
;;;;;for i = 0,ny-2 do begin
;;;;;   dot_pos = strpos(ylabels[i],'.')
;;;;;   substring = strmid(ylabels[i],dot_pos-1,4)
;;;;;   y[i] = float(substring)
;;;;;endfor
x = energy
;;;;;y = y[0:ny-2:2]
where_zero = where(qty eq 0.0,count_zero)
if count_zero gt 0 then dqty[where_zero] = 1.0
y = findgen(ny-1)

help,x
help,y
help,qty
help,dqty
d = {x:x,y:y,qty:qty,dqty:dqty,xunits:'E (meV)',yunits:'Group #'}
return,d
end
; ********************************* ;
function dave_read_ascii_read_spe,filename,err = err,emsg = emsg
err = 0
emsg = 'No errors generated'
openr,lun,filename,/get_lun
dm_load_spe,lun,zdat=qty,err=dqty,xdat=y,ydat=x
qty = transpose(qty) & dqty = transpose(dqty)
d = {x:x,y:y,qty:qty,dqty:dqty,xunits:'E (meV)',yunits:'ANGLE:degrees'}
return,d
end
; ********************************* ;
function dave_read_ascii_check_instrument,filename,err = err,emsg = emsg
; If the format can be identified then read it in.
format_identified = 0B
err = 0
emsg = 'No errors generated'

; Which instrument did this file come from?
; Read in the first 5 lines
dum = ''
dat_string = strarr(5)
openr,lun,filename,/get_lun
for i = 0,4 do begin
   readf,lun,dum
   dat_string[i] = dum
endfor
free_lun,lun,/force

; Check if it is a .SPE file
txt = '.SPE'
if (strpos(strupcase(filename),txt))[0] ne -1 then begin
   format_identified = 1B
   d = dave_read_ascii_read_spe(filename,err = err,emsg = emsg)
   print,'SPE file'
   return,d
endif

; Check if it is an IRIS data file
this_string = strsplit(dat_string[1],/extract)
if n_elements(this_string) gt 1 then begin
   iris_pos = strpos(this_string[0],'IRS')
   if iris_pos[0] ne -1 then begin
      format_identified = 1B
      ; Read it in
      d = dave_read_iris_fast(filename,err = err,emsg = emsg)
;      d = dave_read_ascii_read_iris(filename,err = err,emsg = emsg)
      print,'Iris data set'
      return,d
   endif
endif
; Check if it is a NEAT data file
if n_elements(this_string) eq 3 then begin
   format_identified = 1B
   ; Read it in
   d = dave_read_ascii_read_neat(filename,err = err,emsg = emsg)
   if err eq 0 then begin
      print,'NEAT data set'
      return,d
   endif
endif
; Check if it is a QENS data file
if n_elements(strsplit(dat_string[0],',',/extract)) gt 5 then begin
   this_string = strsplit(dat_string[0],',',/extract)
   if strtrim(this_string[0],2) eq 'w (meV)' then begin
      format_identified = 1B
      ; Read it in
      d = dave_read_ascii_read_qens(filename)
      print,'QENS data set'
      return,d
   endif
endif
; Check if it is a 'PROVA' data file (FOR LACK OF A BETTER NAME RIGHT NOW)	-- LRK 062207
if n_elements(strsplit(dat_string[0],',',/extract)) gt 5 then begin
   this_string = strsplit(dat_string[0],',',/extract)
   if strtrim(this_string[0],2) eq 'A' then begin
      format_identified = 1B
      ; Read it in
      d = dave_read_ascii_read_prova(filename)
      print,'PROVA data set'
      return,d
   endif
endif
if ~format_identified then begin
   d = 0
   err = 1
   emsg = 'File format could not be identified'
endif
return,d
end
; ********************************* ;
function dave_read_ascii_save_dave,filename,d
; Save as a formatted group dave file
dot_pos = strpos(filename,'.')
basename = strmid(filename,0,dot_pos)


output_name = basename+'.dave'
ret = create_dave_pointer(                               $
               daveptr,                          $
               instrument = 'Ascii file',            $
               qty = d.qty,                  $
               qtlabel = 'Intensity (arb units)',  $
               err = d.dqty,                   $
               xvals = d.x,                    $
               xtype = 'POINT',                $
               xunits = 'ENERGY:mev',            $
               xlabel = 'E (meV)',               $
               yvals = d.y,                    $
               ytype = 'POINT',                $
               yunits = 'ANGLE:degrees',         $
               ylabel = 'phi (o)',               $
               specificstr = {x:0.0},             $
               treatment = 'No treatment');,     $
;              dname = dname,        $
;              dunits = dunits,      $
;              dlegend = dlegend,      $
;              dqty = dqty,       $
;              derr = derr,       $
;              ermsg = errmsg

save,daveptr,filename = output_name
heap_free,daveptr
retval = 1
return,retval
end
; ********************************* ;
function dave_read_ascii,filename,err = err, emsg = emsg
err = 0
emsg = 'No errors generated'

; Is the file valid?
if ~file_test(filename) then begin
   err = 1
   emsg = 'File not found'
   return,0
endif

; Which instrument did it come from?
d = dave_read_ascii_check_instrument(filename,err = err,emsg = emsg)
retval = d
return,retval
end
; ********************************* ;
pro test_ascii_read,filetype = filetype
if n_elements(filetype) eq 0 then filetype = 0
case filetype of
0: $
   begin
      dir = 'c:\bhd\dave_development\dave\IRIS_DATA\'
      filename = dir+'HELIUM.TXT'
   end
1: $
   begin
      dir = 'c:\bhd\dave_development\dave\NEAT_DATA\'
      filename = dir+'10701neat000.ino'
   end
2: $
   begin
      dir = 'c:\bhd\dave_development\dave\QENS_DATA\'
      filename = dir+'QENS_Sqw_7908-n7904-.dat'
   end
3: begin
      filename = dialog_pickfile(/read)
   end
else: return
endcase

d = dave_read_ascii(filename,err = err,emsg = emsg)
if err ne 0 then print,emsg
ret = dave_read_ascii_save_dave(filename,d)
end