; $Id$
;###############################################################################
;
; NAME:
;  OPAN_PLOTPARMS
;
; PURPOSE:
;  This widget program allows the user to plot the group-dependence of the
;  fit parameters.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN, curve fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
pro plotParmsCleanup,tlb
widget_control,tlb,get_uvalue = pState
if ((*pState).notifyIds)[0] ne (-1L) then begin
  s = size((*pState).notifyIDs)
  if s[0] eq 1 then count = 0 else count = s[2]-1
  for j = 0,count do begin
    plotParmsInfo = {plotParmsEvent,$
                             ID:(*pState).notifyIDs[0,j],$
                             Top:(*pState).notifyIDs[1,j],$
                             Handler:0l}
    if widget_info((*pState).notifyIDs[0,j],/valid_id) then begin $
      widget_control,(*pState).notifyIDs[0,j],send_event = plotParmsInfo
    endif
  endfor
endif

; Now clean up all of the pointers and delete the pixmaps used in this module
ptr_free,(*pState).yPtr,(*pState).pPtr, (*pState).perrPtr, (*pState).ycleanPtr
wdelete,(*pState).winPix
ptr_free,pState
return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plotParmsSave,event

if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'opan_plotparms in procedure plotParmsSave: Error encountered'
      eMsg = 'An error or unusual condition was encountered in plotParmsSave!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif


; Save the parameters in 3-column ascii format
widget_control,event.top,get_uvalue = pState
widget_control,event.id,get_uvalue = choice

filename = (*pState).filename
if (Strlen(filename) gt 0) then begin
  ; locate ext portion of filename
  i = Strpos(filename,'.',/reverse_search)
  ext = (i lt 0)? '' : Strmid(filename,i)
  ; strip dir path and ext from filename
  filename = File_basename(filename,ext,/fold_case)
endif

if (choice eq 'ALL') then begin
  ; retrieve all the fitted parameters
  qvals = (*pState).x0
  qLabel = (*pState).xtitle
  parmNames = (*pstate).truncatedParmNames
  nParms = n_elements((*pState).p)
  nGroups = n_elements((*pState).p[0])
  p = dblarr(nParms,ngroups)
  perr = p
  for i=0,nparms-1 do begin
    p[i,*] = (*pState).p[i]
    perr[i,*] = (*pState).perr[i]
  endfor
  
  for i=0,nparms-1 do parmNames[i] = Strjoin(Strsplit(parmNames[i],/extract))  ; remove any white space in parameter names
  headers = Strarr(2*nparms)
  headers[Indgen(nparms)*2] = parmNames
  headers[Indgen(nparms)*2 + 1] = 'Error'
  headers = ['GroupValue',headers]

  filename = filename+'_fitParameters.txt'

  workDir = (*pState).workdir
  filename = Dialog_pickfile(dialog_parent = Event.top,/write,path = workDir, $
    title = 'Ascii file to store fit parameters',filter = '*.txt',$
    default_extension='txt',/overwrite_prompt,file=filename)
  if filename eq '' then Return
  Openw,lun,filename,/get_lun

  nstr = strtrim(string(nparms*2 +1),2)
  aformat = '('+nstr+'A20)'
  nformat = '('+nstr+'G20.7)'
  ;printf, lun, title
  printf, lun ,headers,format=aformat
  line = dblarr(2*nparms+1)
  for i=0,ngroups-1 do begin
    line[0] = qvals[i]
    line[indgen(nparms)*2+1] = p[*,i]
    line[Indgen(nparms)*2+2] = perr[*,i]
    printf, lun, line, format=nformat
  endfor
  Free_lun,lun
  msg = 'Output saved in: '+filename
  ok = Dialog_message(/info,msg,dialog_parent=Event.top)

endif else begin
  if N_elements(*(*pState).pPtr) eq 0 then Return

  parmIndex = (*pstate).selectedIndex
  filename = filename+'_fitParameter_p'+strtrim(string(parmIndex),2)+'.txt'

  workDir = (*pState).workdir
  filename = Dialog_pickfile(dialog_parent = Event.top,/write,path = workDir, $
    title = 'Ascii file to store fit parameters',filter = '*.txt',$
    default_extension='txt',/overwrite_prompt,file=filename)
  if filename eq '' then Return
  Openw,lun,filename,/get_lun

  x = *(*pState).yPtr
  y = *(*pState).pPtr
  yerr = *(*pState).perrPtr
  nx = N_elements(x)
  parmName = (*pstate).truncatedParmNames[parmIndex]

  strout = ['#',(*pState).xtitle,parmName,'Error']
  Printf,lun,strout,format='(A1,A19, 2A20)'
  for i = 0,nx-1 do Printf,lun,x[i],y[i],yerr[i],format='(3G20.7)'
  Free_lun,lun
  msg = 'Output saved in: '+filename
  ok = Dialog_message(/info,msg,dialog_parent=Event.top)

endelse

return
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plotParmsQuit,event
widget_control,event.top,/destroy
return
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro opan_drawParms,event

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'PAN Plot Parameters: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState


if ptr_valid((*pState).yPtr) and ptr_valid((*pState).pPtr) and ptr_valid((*pState).perrPtr) then begin

  if (n_elements(*(*pState).yPtr) gt 0) and $
        (n_elements(*(*pState).pPtr) gt 0) and $
          (n_elements(*(*pState).perrPtr) gt 0) then begin 
                
    x = *(*pState).yPtr
    y = *(*pState).pPtr
    yerr = *(*pState).perrPtr
    
    if (*pState).autoscale eq 1 then begin
      dx = 0.2*(max(x)-min(x))
      (*pState).xrange = [min(x)-dx,max(x)+dx]
      dy = 0.5*(max(y)+max(yerr)-min(y)+max(yerr))
      (*pState).yrange = [min(y)-dy-max(yerr),max(y)+dy+max(yerr)]
    endif
    device,get_decomposed=dc
    device,decomposed=1                  
    plot,x,y,psym = 4,xrange = (*pState).xrange,yrange = (*pState).yrange, $
         xstyle = 1,ystyle = 1,xtitle = (*pState).xtitle, $
         ytitle = (*pState).ytitle,title = (*pState).title,$
         color=0L,background=(2L)^24 - 1
    errplot,x,y-yerr,y+yerr,width = 0.0,color=0L
    device,decomposed=dc
  endif;n_elements
endif;ptr_valid
return
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plotParmsAccept,event

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      eTitle = 'PAN Plot Parameters: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
widget_control,(*pState).parm2Plot,get_uvalue = parmIndex

(*pState).ytitle = (*pState).yTitlValues[parmIndex]
*(*pState).yptr = (*pState).x0
*(*pState).pptr = reform((*pState).p[parmIndex])
*(*pState).perrptr = reform((*pState).perr[parmIndex])
(*pState).title = (*pState).titlValues[parmIndex]

widget_control, (*pstate).xminID, set_value=min(*(*pState).yptr)
Widget_control, (*pstate).xmaxid, set_value=max(*(*pState).yptr)
Widget_control, (*pstate).yminid, set_value=Min(*(*pState).pptr)
Widget_control, (*pstate).ymaxid, set_value=Max(*(*pState).pptr)

Wset,(*pState).winpix
Opan_drawparms,Event
Wset,(*pState).winvis
Device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winpix]

return
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Export fit Parameter details to DAVE Data Manager
pro exportToDM,event, use_defined_param=use_defined_param

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      eTitle = 'Export to Data Manager: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
if (~obj_valid((*pstate).daveTool)) then return

if (keyword_set(use_defined_param)) then begin
  fitObject = (*pState).fitObject
  oc = fitObject[0]

  *(*pstate).yPtr = (*pstate).x
  *(*pstate).pPtr = (*pstate).y
  *(*pstate).perrPtr = (*pstate).yerr

  (*pstate).xtitle = tex2idl((*pstate).xLabel)
  (*pstate).ytitle = tex2idl((*pstate).yLabel)
  (*pstate).title = 'Derived Parameter'

  filename = (*pState).filename
  if (strlen(filename) gt 0) then begin
    ; locate ext portion of filename
    i = strpos(filename,'.',/reverse_search)
    ext = (i lt 0)? '' : strmid(filename,i)
    ; strip dir path and ext from filename
    filename = file_basename(filename,ext,/fold_case)
  endif

  nameTag = filename+'_fitParameter'
  nameTag = strtrim(nameTag,2)

  n = n_elements((*pstate).y)
  data = fltarr(3,n)
  header = strarr(3)
  data[0,*] = (*pstate).x
  data[1,*] = (*pstate).y
  data[2,*] = (*pstate).yerr
  header[0] = (*pstate).xtitle
  header[1] = (*pstate).ytitle
  header[2] = 'Error'

  oData = obj_new('ASCIIColumnDataset',filename=nameTag,data=data,header=header)
  if (~obj_valid(oData)) then return

  trmt = 'This dataset results from a model function fit to '+(*pState).filename
  trmt = [trmt,'Number of functions in model was ',strtrim(string(oc->count()),2)]
  trmt = [trmt,'Functions in model:: '+strjoin(oc->getnames(),', ')]
  trmt = [trmt,'This dataset contains a derived fitted parameter']
  trmt = [trmt,'with parameter name: '+(*pstate).ytitle]

  oData->SetProperty, treatment=trmt
  (*pstate).daveTool->AddDatasetToDataManager, oData
  
  return
endif

Widget_control,Event.top,get_uvalue = pState
Widget_control,(*pState).parm2plot,get_uvalue = parmIndex

y = (*pState).x0
p = (*pState).p[parmIndex]
perr = (*pState).perr[parmIndex]

(*pState).ytitle = (*pState).ytitlvalues[parmIndex]
*(*pState).yptr = y
*(*pState).pptr = p
*(*pState).perrptr = perr
(*pState).title = (*pState).titlvalues[parmIndex]
fitObject = (*pState).fitObject
okindex = (*pState).okindex
oc = fitObject[okindex[0]]

filename = (*pState).filename
if (Strlen(filename) gt 0) then begin
  ; locate ext portion of filename
  i = Strpos(filename,'.',/reverse_search)
  ext = (i lt 0)? '' : Strmid(filename,i)
  ; strip dir path and ext from filename
  filename = File_basename(filename,ext,/fold_case)
endif

nameTag = filename+'_fitParameter'+Strtrim(String(parmIndex),2)
nameTag = Strtrim(nameTag,2)

n = N_elements(y)
data = Fltarr(3,n)
header = Strarr(3)
data[0,*] = y
data[1,*] = p
data[2,*] = perr
header[0] = (*pstate).xtitle
header[1] = (*pState).ytitle
header[2] = 'Error'

oData = Obj_new('ASCIIColumnDataset',filename=nameTag,data=data,header=header)
if (~Obj_valid(oData)) then Return

trmt = 'This dataset results from a model function fit to '+(*pState).filename
trmt = [trmt,'Number of functions in model was ',Strtrim(String(oc->Count()),2)]
trmt = [trmt,'Functions in model:: '+Strjoin(oc->Getnames(),', ')]
trmt = [trmt,'This dataset contains fitted '+(*pState).parmNames[parmIndex]]
;trmt = [trmt,'with parameter name: '+(*pState).ytitle]

oData->Setproperty, treatment=trmt
(*pstate).davetool->Adddatasettodatamanager, oData

return
end




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Export fit Parameter data to a new PAN session for fitting
pro exportToPAN,event, use_defined_param=use_defined_param

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'Export to Data Manager: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
if (~obj_valid((*pstate).daveTool)) then return

if (keyword_set(use_defined_param)) then begin
  fitObject = (*pState).fitObject
  oc = fitObject[0]

  *(*pstate).yPtr = (*pstate).x
  *(*pstate).pPtr = (*pstate).y
  *(*pstate).perrPtr = (*pstate).yerr

  (*pstate).xtitle = tex2idl((*pstate).xLabel)
  (*pstate).ytitle = tex2idl((*pstate).yLabel)
  (*pstate).title = 'Derived Parameter'

  filename = (*pState).filename
  if (strlen(filename) gt 0) then begin
    ; locate ext portion of filename
    i = strpos(filename,'.',/reverse_search)
    ext = (i lt 0)? '' : strmid(filename,i)
    ; strip dir path and ext from filename
    filename = file_basename(filename,ext,/fold_case)
  endif

  trmt = 'This dataset results from a model function fit to '+(*pState).filename
  trmt = [trmt,'Number of functions in model was ',strtrim(string(oc->count()),2)]
  trmt = [trmt,'Functions in model:: '+strjoin(oc->getnames(),', ')]
  trmt = [trmt,'This dataset contains a derived fitted parameter']
  trmt = [trmt,'with parameter name: '+(*pstate).ytitle]

  retVal = create_dave_pointer(davePtr $
    ,qty = (*pstate).y $
    ,qtlabel = (*pstate).ytitle $
    ,err = (*pstate).yerr $
    ,xvals = (*pstate).x $
    ,xtype = 'POINTS' $
    ,xlabel = (*pstate).xtitle $
    ,treatment = trmt $
    )

  if (ptr_valid(davePtr)) then  begin
    oVoid = obj_new('OPAN',notifyID=[(*pState).tlb,(*pState).tlb],group_leader=(*pState).tlb $
      ,davePtr=davePtr,workDir=(*pState).workDir, daveTool=(*pState).daveTool)

    heap_free, davePtr
  endif
  
  return
endif


Widget_control,Event.top,get_uvalue = pState
Widget_control,(*pState).parm2plot,get_uvalue = parmIndex

y = (*pState).x0
p = (*pState).p[parmIndex]
perr = (*pState).perr[parmIndex]

(*pState).ytitle = (*pState).ytitlvalues[parmIndex]
*(*pState).yptr = y
*(*pState).pptr = p
*(*pState).perrptr = perr
(*pState).title = (*pState).titlvalues[parmIndex]
fitObject = (*pState).fitobject
okindex = (*pState).okindex
oc = fitObject[okindex[0]]

filename = (*pState).filename
if (Strlen(filename) gt 0) then begin
  ; locate ext portion of filename
  i = Strpos(filename,'.',/reverse_search)
  ext = (i lt 0)? '' : Strmid(filename,i)
  ; strip dir path and ext from filename
  filename = File_basename(filename,ext,/fold_case)
endif

nameTag = filename+'_fitParameter'+Strtrim(String(parmIndex),2)
nameTag = Strtrim(nameTag,2)

trmt = 'This dataset results from a model function fit to '+(*pState).filename
trmt = [trmt,'Number of functions in model was ',Strtrim(String(oc->Count()),2)]
trmt = [trmt,'Functions in model:: '+Strjoin(oc->Getnames(),', ')]
trmt = [trmt,'This dataset contains fitted '+(*pState).parmnames[parmIndex]]

retVal = Create_dave_pointer(davePtr $
  ,qty = p $
  ,qtlabel = (*pState).ytitle $
  ,qtUnits = '' $
  ,err = perr $
  ,xvals = y $
  ,xtype = 'POINTS' $
  ,xlabel = (*pstate).xtitle $
  ,xUnits = '' $
  ,treatment = trmt $
  )

if (Ptr_valid(davePtr)) then  begin
  oVoid = Obj_new('OPAN',notifyID=[(*pState).tlb,(*pState).tlb],group_leader=(*pState).tlb $
    ,davePtr=davePtr,workDir=(*pState).workdir, daveTool=(*pState).davetool)

  Heap_free, davePtr
endif

return
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro opanParmsDraw,event


if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'opan_plotparms in procedure plotParmsSave: Error encountered'
      eMsg = 'An error or unusual condition was encountered in opanParmsDraw!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

device,get_decomposed=dc
device,decomposed=1

widget_control,event.top,get_uvalue = pState
case event.type of
0: begin    ; button press
     (*pState).mouse = event.press
     if (*pState).mouse eq 4 then begin   
       ; right mouse button - signals return to autoscale of plot
       (*pState).autoscale = 1
       widget_control, (*pState).autoscaleID, set_value=1
       Widget_control, (*pState).xyrangebaseid, sensitive = 0

       wset,(*pState).winPix
       opan_drawParms,event
       wset,(*pState).winVis
       device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winPix]
       return
     endif
     if (*pState).mouse eq 1 then begin   
       ; left mouse button - signal start of rubber-band zoom and no autoscale of plot
       (*pState).xbox[0] = event.x
       (*pState).ybox[0] = event.y
       wset,(*pState).winVis
       device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winPix]
       empty
       (*pState).autoscale = 0
       Widget_control, (*pState).autoscaleID, set_value=0
       Widget_control, (*pState).xyrangebaseid, sensitive = 1

       widget_control,(*pState).win,/draw_motion_events
     endif
   end
1: begin ; button release
    if (*pState).mouse eq 1 then begin
     xll = (*pState).xbox[0] < (*pState).xbox[1]
     yll = (*pState).ybox[0] < (*pState).ybox[1]
     w = abs((*pState).xbox[1] - (*pState).xbox[0])
     h = abs((*pState).ybox[1] - (*pState).ybox[0])
     xur = xll + w
     yur = yll + h
     ll = convert_coord(xll,yll,/device,/to_data)
     ur = convert_coord(xur,yur,/device,/to_data)
     (*pState).xrange = [ll[0],ur[0]]
     (*pState).yrange = [ll[1],ur[1]]

     ; synchronize with the widget field controls for specifying ranges manually
     Widget_control, (*pState).xminid, set_value=(*pState).xrange[0]
     Widget_control, (*pState).xmaxid, set_value=(*pState).xrange[1]
     Widget_control, (*pState).yminid, set_value=(*pState).yrange[0]
     Widget_control, (*pState).ymaxid, set_value=(*pState).yrange[1]


     wset,(*pState).winPix
     opan_drawParms,event
     wset,(*pState).winVis
     device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winPix]
     (*pState).mouse = 0B
     widget_control,(*pState).win,draw_motion_events = 0
    endif
   end
2: begin ; mouse motion
     if (*pState).mouse eq 1 then begin
        (*pState).xbox[1] = event.x
        (*pState).ybox[1] = event.y
        xc = [(*pState).xbox[0],event.x,event.x,$
              (*pState).xbox[0],$
              (*pState).xbox[0]]
        yc = [(*pState).ybox[0],(*pState).ybox[0],$
              event.y,event.y,$
              (*pState).ybox[0]]
       wset,(*pState).winPix
       opan_drawParms,event
       wset,(*pState).winVis
       device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winPix]
        plots,xc,yc,/device,color=40L+240L*256L+40L*256L^2

        empty
     endif
   end
else:
endcase

device,decomposed=dc

return
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plotParmsPrint,event

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'PAN Plot Parameters: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
workDir = (*!dave_defaults).workDir

filename = (*pState).filename
if (Strlen(filename) gt 0) then begin
  ; locate ext portion of filename
  i = Strpos(filename,'.',/reverse_search)
  ext = (i lt 0)? '' : Strmid(filename,i)
  ; strip dir path and ext from filename
  filename = File_basename(filename,ext,/fold_case)
endif

filename = filename+'.ps'

workDir = (*pState).workdir

keywords = PSConfig(Cancel=cancelled,group_leader = event.top,$
           filename = filename,color = 0,directory = workDir)
IF cancelled THEN RETURN
thisDevice = !D.Name
Set_Plot, 'PS'
Device, _Extra=keywords
opan_drawParms,event
Device, /Close_File
Set_Plot, thisDevice
return
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plotParmsJPEG,event

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'PAN Plot Parameters: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
thisDevice = !D.Name
xsize = !d.x_size & ysize = !d.y_size
window,/free,/pixmap,xsize = xsize,ysize = ysize
winPix = !d.window
opan_drawParms,event
device,get_visual_depth = thisDepth
if thisDepth eq 8 then begin
  tvlct,r,g,b,/get
  image2d = tvrd()
  s = size(image2d,/dimensions)
  image24 = bytarr(3,s[0],s[1])
  tvlct,r,g,b,/get
  image24[0,*,*] = r[image2d]
  image24[1,*,*] = g[image2d]
  image24[2,*,*] = b[image2d]
endif
if thisDepth gt 8 then begin
  device,decomposed = 1
  image24=tvrd(true = 1)
endif
wdelete,winPix

filename = (*pState).filename
if (Strlen(filename) gt 0) then begin
  ; locate ext portion of filename
  i = Strpos(filename,'.',/reverse_search)
  ext = (i lt 0)? '' : Strmid(filename,i)
  ; strip dir path and ext from filename
  filename = File_basename(filename,ext,/fold_case)
endif

filename = filename+'.jpg'

workDir = (*pState).workdir
filename = Dialog_pickfile(dialog_parent = Event.top,/write,path = workDir, $
  title = 'Choose file to save parameter plot image',filter = ['*.jpg','*.JPG'],$
  default_extension='jpg',/overwrite_prompt,file=filename)
if filename eq '' then Return

s = Size(image24, /Dimensions)
newx = Round(300.0 * s[1] / 72)
newy = Round(300.0 * s[2] / 72)
highResImage = Congrid(image24, 3, newx, newy, /Interp)
write_jpeg,filename,255-highResImage,true = 1,quality = 100
Set_Plot, thisDevice
return
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro logParms,event

; RTA - Insert Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'PAN Plot Parameters: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control,event.top,get_uvalue = pState
if n_elements(*(*pState).stringPtr) eq 0 then return
thisDevice = !D.Name
xsize = !d.x_size & ysize = !d.y_size
window,/free,/pixmap,xsize = xsize,ysize = ysize
winPix = !d.window
opan_drawParms,event
device,get_visual_depth = thisDepth
if thisDepth eq 8 then begin
  tvlct,r,g,b,/get
  image2d = tvrd()
  s = size(image2d,/dimensions)
  image24 = bytarr(3,s[0],s[1])
  tvlct,r,g,b,/get
  image24[0,*,*] = r[image2d]
  image24[1,*,*] = g[image2d]
  image24[2,*,*] = b[image2d]
endif
if thisDepth gt 8 then begin
  device,decomposed = 1
  image24=tvrd(true = 1)
endif
wdelete,winPix

; Create the filename
counter = 0
res = 1
while res eq 1 do begin
  filename = 'panplot_'+strtrim(string(counter),2)+'.jpg'
  ;testFile = filepath(filename, ROOT_DIR=(*pState).workDir,subdir=[(*pState).logDirectory])
  testFile = filepath(filename, ROOT_DIR=(*pState).logDirectory)
  res = file_test(testFile)
  counter = counter + 1
endwhile

s = Size(image24, /Dimensions)
newx = Round(150.0 * s[1] / 72)
newy = Round(150.0 * s[2] / 72)
highResImage = Congrid(image24, 3, newx, newy, /Interp)

imgFile = filename

write_jpeg,testFile,255-highResImage,true = 1,quality = 100
Set_Plot, thisDevice

; Ok, the file has been written.  Now we must add the plot to the HTML file.
strout = *(*pState).stringPtr
end_of_html = where(strout eq '</body>')
index = end_of_html[0] - 1
nstr = n_elements(strout)

read_jpeg,testFile,img
imgSize = size(img)
sx = imgSize[2] & sy = imgSize[3]
aspect2 = 1.0*sy/(1.0*sx)
outSx = 400 & outSy = fix(1.0*outSx*aspect2)
imgLine = '<img src="'+imgFile+'" width="'+ $
          strtrim(string(outSx),2)+ $
          '" height="'+strtrim(string(outSy),2)+'"><br>'

strout1 = [strout[0:index],'<p>',imgLine,strout[index:nstr-1],'<p>']
*(*pState).stringPtr = strout1
return
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro opan_plotParms_event,event
if dave_set_focus(event) then return

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
   catch, catchError
   if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'opan_plotParms_event: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
      catch, /cancel
      return
   endif
endif

widget_control, event.top, get_uvalue=sPtr

uname = widget_info(event.id,/uname)

rangeHasChanged = uname.Matches('xmin',/fold) || $
                  uname.Matches('xmax',/fold) || $
                  uname.Matches('ymin',/fold) || $
                  uname.Matches('ymax',/fold)

if (rangeHasChanged) then begin
  widget_control, (*sPtr).xminID, get_value=xmin
  Widget_control, (*sPtr).xmaxID, get_value=xmax
  Widget_control, (*sPtr).yminID, get_value=ymin
  Widget_control, (*sPtr).ymaxID, get_value=ymax
  (*sPtr).xrange = [xmin,xmax]
  (*sPtr).yrange = [ymin,ymax]
  
  Wset,(*sPtr).winpix
  Opan_drawparms,Event
  Wset,(*sPtr).winvis
  Device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winpix]

endif

case strupcase(uname) of
  'AUTOSCALE': begin
    (*sPtr).autoscale = event.select
    widget_control, (*sPtr).xyRangeBaseID, sensitive = ~event.select

    if ((*sPtr).autoscale eq 0) then begin
      ; if autoscale is off then read existing x,y-ranges and use those for plotting
      Widget_control, (*sPtr).xminid, get_value=xmin
      Widget_control, (*sPtr).xmaxid, get_value=xmax
      Widget_control, (*sPtr).yminid, get_value=ymin
      Widget_control, (*sPtr).ymaxid, get_value=ymax
      (*sPtr).xrange = [xmin,xmax]
      (*sPtr).yrange = [ymin,ymax]
    endif

    Wset,(*sPtr).winpix
    Opan_drawparms,Event
    Wset,(*sPtr).winvis
    Device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winpix]
  end
  
  'DEFINE_VARS': void = plotParms_DefineVars(event)

  'DEFINE_VARS_FLAG': begin
    widget_control, event.id, get_uvalue=wIDs
    n = n_elements(wIDs)
    for i=0,n-1 do widget_control, wIDs[i], sensitive=event.select    
  end
  
  'PLOT_VARS': begin    
;TODO - fix this section
;    x = (*sPtr).x
;    y = (*sPtr).y
;    yErr = (*sPtr).yerr
;    (*sPtr).xtitle = 'Q'
;    (*sPtr).ytitle = 'Fits'
;    (*sPtr).title = 'Derived Parameter'

;    if (*sPtr).autoscale eq 1 then begin
;      dx = 0.2*(max(x)-min(x))
;      (*sPtr).xrange = [min(x)-dx,max(x)+dx]
;      dy = 0.5*(max(y)+max(yerr)-min(y)+max(yerr))
;      (*sPtr).yrange = [min(y)-dy-max(yerr),max(y)+dy+max(yerr)]
;    endif


    *(*sPtr).yPtr = (*sPtr).x
    *(*sPtr).pPtr = (*sPtr).y
    *(*sPtr).perrPtr = (*sPtr).yerr

    (*sPtr).xtitle = tex2idl((*sPtr).xLabel)
    (*sPtr).ytitle = tex2idl((*sPtr).yLabel)
    (*sPtr).title = 'Derived Parameter'

    wset,(*sPtr).winPix
    opan_drawParms,event
    wset,(*sPtr).winVis
    device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winPix]



;    wset,(*sPtr).winPix
;
;    device,get_decomposed=dc
;    device,decomposed=1
;    plot,x,y,psym = 4,xrange = (*sPtr).xrange,yrange = (*sPtr).yrange, $
;      xstyle = 1,ystyle = 1,xtitle = (*sPtr).xtitle, $
;      ytitle = (*sPtr).ytitle,title = (*sPtr).title,$
;      color=0L,background=(2L)^24 - 1
;    errplot,x,y-yerr,y+yerr,width = 0.0,color=0L
;    device,decomposed=dc
;
;    wset,(*sPtr).winVis
;    device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winPix]    
    
  end
  
  'FIT_VARS': begin
    exportToPAN, event, /use_defined_param
  end
  
  'EXPORT_VARS': begin
    exportToDM, event, /use_defined_param
  end
  
  else:
endcase

; Handle the resize events
if tag_names(event,/structure_name) eq 'WIDGET_BASE' then begin
  base2geom = widget_info((*sPtr).base2,/geometry)
  base3geom = widget_info((*sPtr).base3,/geometry)
  xsize = event.x
  ysize = event.y

  ; New data window dimensions
  newxsize = xsize
  newysize = ysize - base2geom.ysize - base3geom.ysize

  widget_control,(*sPtr).win,draw_xsize = newxsize, $
                 draw_ysize = newysize
  wdelete,(*sPtr).winPix
  window,/free,/pixmap,xsize = newxsize,ysize = newysize
  (*sPtr).winPix = !d.window

  if n_elements(*(*sPtr).pPtr) eq 0 then return

  wset,(*sPtr).winPix
  opan_drawParms,event
  wset,(*sPtr).winVis
  device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winPix]

  return
endif

if event.id eq (*sPtr).parm2plot then begin
  widget_control, event.id, set_uvalue=event.index
  (*sPtr).selectedIndex = event.index
  plotParmsAccept,event
endif

return
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro opan_plotParms,  fitObject, y, $
           group_leader = group_leader, $
           daveTool = daveTool, $
           filename = filename, $
           notifyIds = notifyIds, $
           xtitle = xtitle, $
           ytitle = ytitle, $
           title = title, $
           workDir = workDir, $
           logDirectory = logDirectory, $
           groupsIncludedInFit = groupsIncludedInFit, $
           stringPtr = stringPtr

twoDimFlag = (fitObject[0])->Get2DFlag()
if (twoDimFlag) then begin
  ; the model contains 2D functions
  
  ; only the first model is relevant for 2D
  ngroups = n_elements(fitObject)
  okIndex = indgen(ngroups)       ; TODO: assumes all groups are included in fit, but what if this is not the case
  oc = fitObject[0]
  ncurves = oc->count()
  chisq = oc->getchisq()
  if ((ncurves eq 0) or (chisq eq -1.0)) then Return    ; no functions in model or fit not yet performed

  parmNames = oc->Getparmnames()
  nparmsTotal = n_elements(parmNames)
  nSingParms = n_elements(oc->GetSingleParmnames())
  nMultiParms = n_elements(oc->GetMultiParmnames())
  parms = oc->GetParms()
  parmErr = oc->GetParmError()
  
  ; try to retrieve the multi-parameters - these are parameters that have a group-dependence
  ;counter = 0
  nValidParms = 0
  pData = dblarr(nSingParms+nMultiParms,ngroups)    ;dblarr(fix(nparmsTotal/ngroups),ngroups)
  peData = pData
  yTitlValues = strarr(nSingParms+nMultiParms)      ;strarr(fix(nparmsTotal/ngroups))
  titlValues = yTitlValues
  parmNames = yTitlValues
  truncatedParmNames = yTitlValues
  index = Indgen(ngroups)
  sindex = 0
  for i=0,ncurves-1 do begin
    func = oc->Get(position=i)
    func->getproperty, single_parmnames=singleParms, multi_parmnames=multiParms, name=funcName
    funcName = Strupcase(strmid(funcName,4))        ; strip off the pan_ prefix

    ; process single parameters: ie group-independent ones
    nSingle = (singleParms[0].Strlen() eq 0)? 0 : n_elements(singleParms)
;    if (nSingle gt 0) then begin
      for j=0,nSingle-1 do begin
        pData[nValidParms,*] = replicate(parms[sindex],ngroups)
        peData[nValidParms,*] = replicate(parmErr[sindex],ngroups)
        yTitlValues[nValidParms] = funcName+': '+singleParms[j]
        titlValues[nValidParms] = 'Parameter #'+Strtrim(String(sindex),2)+' in fit model'
        parmNames[nValidParms] = ' Parameter #'+Strtrim(String(sindex),2)+': '+Strupcase(funcName)+': '+singleParms[j]+'  '
        truncatedParmNames[nValidParms] = singleParms[j]
        nValidParms++
        index = index + 1
        sindex++
      endfor
;    endif

    ; process multi parameters: ie group-dependent ones
    nMulti = (multiParms[0].Strlen() eq 0)? 0 : n_elements(multiParms)
;    if (nMulti gt 0) then begin
      for j=0,nMulti-1 do begin
        pData[nValidParms,*] = parms[index]
        peData[nValidParms,*] = parmErr[index]
        yTitlValues[nValidParms] = funcName+': '+multiParms[j]
        titlValues[nValidParms] = 'Parameter #'+strtrim(string(index[0]),2)+' to #'+strtrim(string(index[ngroups-1]),2)+' in fit model'
        parmNames[nValidParms] = ' Parameter #'+strtrim(string(index[0]),2)+' to #'+strtrim(string(index[ngroups-1]),2)+': '+Strupcase(funcName)+': '+multiParms[j]+'  '
        truncatedParmNames[nValidParms] = multiParms[j]
        nValidParms++
        index = index + ngroups
        sindex = sindex + ngroups
      endfor
;    endif
  endfor
  
  nParms = nValidParms
  if (nParms lt 1) then Return     ; no fitted parameters present
  
endif else begin
  ; the model contains 1D functions

  ; Eliminate groups for which the model contains no define functions
  ; Or for which the data is not yet fitted
  ngroups = n_elements(fitObject)
  totalFits = 0
  okIndex = []
  for i = 0,ngroups-1 do begin
    oc = fitObject[i]
    ncurves = oc->count()
    ;nparms = N_elements(oc->Getparms())
    chisq = oc->getchisq()
    if ((ncurves gt 0) and (chisq ne -1.00)) then begin
      totalFits = totalFits + 1
      okIndex = [okIndex,i]
    endif
  endfor
  if (totalFits eq 0) then Return                   ; no fitted data present
  
  oc = fitobject[okindex[0]]
  ngroups = totalFits
  ncurves = oc->count()
  parmNames = oc->Getparmnames()
  nparms = n_elements(parmNames)
  yTitlValues = parmNames
  titlValues = parmNames
  truncatedParmNames = parmNames
  for j=0, nparms-1 do begin
    funcName = oc->GetCurveName(j)                  ; retrieve the actual function that this parameter belongs to
    funcName = Strupcase(Strmid(funcName,4))        ; strip off the 'pan_' prefix
    titlValues[j] = 'Parameter #'+strtrim(string(j),2)+' in fit model'
    yTitlValues[j] = funcName+': '+parmNames[j]
    parmNames[j] = ' Parameter #'+strtrim(string(j),2)+': '+Strupcase(funcName)+': '+parmNames[j]+'  '
  endfor
  pData = dblarr(nparms,ngroups)
  peData = dblarr(nparms,ngroups)
  for i = 0,ngroups-1 do begin
    oc = fitObject[okIndex[i]]
    pData[*,i] = oc->getparms()
    peData[*,i] = oc->getparmError()
  endfor
endelse

x0 = Reform(y[0,okIndex])

; copy the q-dependent fitted parameter/error details to 2 hash variables
p = obj_new('Hash')
perr = obj_new('Hash')
for i=0,nParms-1 do begin
  p[i] = reform(pData[i,*])
  pErr[i] = reform(peData[i,*])
endfor

; Widget definition section
xsize = 5
if n_elements(workDir) eq 0 then workDir = ''
if n_elements(notifyIds) eq 0 then notifyIds = (-1L)
if n_elements(group_leader) eq 0 then begin
  tlb = widget_base(/col,title = 'Parameter plotting utility', $
        /base_align_center,/tlb_size_events)
endif else begin
  tlb = widget_base(group_leader = group_leader, /row, $
        title = 'Parameter plotting utility', $
        /base_align_center,/tlb_size_events)
endelse

base1 = widget_base(tlb,/col)
base2 = Widget_base(base1,/col,/align_center)

base3 = widget_base(base1,/col,/frame)
base3_1 = widget_base(base3,/row,/grid)
base3_2 = widget_base(base3,/row,/grid)
tt = 'Check this button to define and use secondary variables based on the already fitted parameters'
font2 = (!version.os_family eq 'Windows')? 'Lucida Sans Typewriter*14' : $
  'lucidasanstypewriter-10'
;base3_1_1 = widget_base(base3_1,/row,/nonexclusive)
;wDefine = widget_button(base3_1_1,value='',uname='DEFINE_VARS_FLAG',tooltip=tt)
;txt = 'Define or use variables based on the existing fitted parameters'
;widget_control, wDefine, set_button=0
;void = CW_ColoredLabel(base3_1,value=txt,forground_color=[128,0,128])
txt = ['Use parameter that is derived from the existing fitted parameters']
wDefine = cw_bgroup(base3_1,txt,/row,/nonexclusive,font=font2,uname='DEFINE_VARS_FLAG')

tt = 'Click button to define a new parameter based on the existing fitted parameters'
tt0 = 'Plot the derived parameter in the display area below'
tt1 = 'Fit the derived parameter in a new PAN window/session'
tt2 = 'Export the derived parameter to the DAVE Data Browser for further manipulation'
wIDs = lonarr(4)
wIDs[0] = widget_button(base3_2,value = 'Derived Parameter Defn',uname='DEFINE_VARS',sensitive=0,tooltip=tt) ; ,event_func = 'plotParms_DefineVars'
wIDs[1] = widget_button(base3_2,value = 'Plot Derived Parameter',uname='PLOT_VARS',sensitive=0,tooltip=tt0)
wIDs[2] = widget_button(base3_2,value = 'Fit it in PAN',uname='FIT_VARS',sensitive=0,tooltip=tt1)
wIDs[3] = widget_button(base3_2,value = 'Export to Data Browser',uname='EXPORT_VARS',sensitive=0,tooltip=tt2)
widget_control, wDefine, set_uvalue=wIDs 

base4 = widget_base(base1,/col,/frame)
base4_1 = Widget_base(base4,/col)
base4_2 = Widget_base(base4,/row,/align_center)

;geom = widget_info(base3,/geometry)
;wxsize = geom.scr_xsize & wysize = 400
;win = widget_draw(base4_1,xsize = wxsize,ysize = wysize,/button_events, $
;      event_pro = 'opanParmsDraw',frame=2)
      
autoscaleID = cw_bgroup(base4_2,'Autoscale',/row,/nonexclusive,uname='autoscale',set_value=1)
xyRangeBaseID = Widget_base(base4_2,/row,/grid,/align_center,sensitive=0)
IDLge91 = (Float(!version.release) ge 9.1)
if (IDLge91) then begin
  text_bg_color = [230,230,255]
endif
xminID = Dave_cw_field(xyRangeBaseID,title = 'Min x',value = 0.0,/float,/return_events,/focus_events,tab_mode=1,uname='xmin',xsize=10,text_bg_color=text_bg_color)
xmaxID = Dave_cw_field(xyRangeBaseID,title = 'Max x',value = 0.0,/float,/return_events,/focus_events,tab_mode=1,uname='xmin',xsize=10,text_bg_color=text_bg_color)
yminID = Dave_cw_field(xyRangeBaseID,title = 'Min y',value = 0.0,/float,/return_events,/focus_events,tab_mode=1,uname='xmin',xsize=10,text_bg_color=text_bg_color)
ymaxID = Dave_cw_field(xyRangeBaseID,title = 'Max y',value = 0.0,/float,/return_events,/focus_events,tab_mode=1,uname='xmin',xsize=10,text_bg_color=text_bg_color)

tt = 'Save the active (displayed) parameter to an ascii text file in column format'
tt0 = 'Save all fitted parameters to an ascii text file in column format'
tt1 = 'Print the display window'
tt2 = 'Save the display window as a JPEG image'
tt3 = 'Close this Plot Fit Parameter window'
base5 = widget_base(base1,/row,/grid,/frame)
void = widget_button(base5,value = 'Save Parameter',event_pro = 'plotParmsSave',tooltip=tt,uvalue='')
void = widget_button(base5,value = 'Save All Parameters',event_pro = 'plotParmsSave',tooltip=tt0,uvalue='ALL')
void = widget_button(base5,value = 'Print Plot',event_pro = 'plotParmsPrint',tooltip=tt1)
void = widget_button(base5,value = 'Plot to JPEG',event_pro = 'plotParmsJPEG',tooltip=tt2)
if n_elements(*stringPtr) gt 0 then sensitive = 1 else sensitive = 0
void = widget_button(base5,value = 'Add to log file', $
       event_pro = 'logParms',sensitive = sensitive)
void = widget_button(base5,value = 'Dismiss',event_pro = 'plotParmsQuit',tooltip=tt3)

geom = Widget_info(base5,/geometry)
wxsize = geom.scr_xsize & wysize = 400
win = Widget_draw(base4_1,xsize = wxsize,ysize = wysize,/button_events, $
  event_pro = 'opanParmsDraw',frame=2)


base2A = Widget_base(base2,/row,/align_center)
base2B = Widget_base(base2,/row,/grid)
;fsize = 8
void = Widget_label(base2A,value='Selected Parameter:')
parm2plot = Widget_combobox(base2A,/dynamic_resize,/list_events,/flat,value=parmNames,uname='selectParameter',uvalue=0)
;parm2plot = cw_field(base2A,value = 1,/string, /return_events, title = 'Parameter #',xsize = fsize)
tt0 = 'Plot the chosen parameter in the display area below'
tt1 = 'Fit the chosen parameter in a new PAN window/session'
tt2 = 'Export the chosen parameter to the DAVE Data Browser for further manipulation'
plotID = Widget_button(base2B,value = 'Plot Parameter',event_pro = 'plotParmsAccept',tooltip=tt0)
void   = Widget_button(base2B,value = 'Fit in PAN',event_pro = 'exportToPAN',tooltip=tt1)
void   = Widget_button(base2B,value = 'Export To Data Browser',event_pro = 'exportToDM',tooltip=tt2)


widget_control,tlb,/realize
widget_control,win,get_value = winVis
window,/free,/pixmap,xsize = wxsize,ysize = wysize
winPix = !d.window

state = {tlb:tlb,$
         parm2plot:parm2plot,$
;         info:info, $
         fitObject:fitObject, $
         okindex:okindex, $
         selectedIndex:0, $
         ycleanPtr:ptr_new(reform(y[0,*]),/no_copy), $
         yPtr:ptr_new(reform(y[0,*]),/no_copy), $
         pPtr:ptr_new(/allocate_heap), $
         perrPtr:ptr_new(/allocate_heap), $
         p:         p, $
         pErr:      pErr, $
         x0:        x0, $
         x:         x0, $
         y:         x0, $
         yErr:      x0, $
         xLabel:    '', $
         yLabel:    '', $
         varDefFlag:   0, $
         varDefStatus: 0, $
         parmNames: parmNames, $
         truncatedParmNames:truncatedParmNames, $
         nCurves:   nCurves, $
         daveTool:daveTool, $
         filename:filename, $
         workDir:workDir, $
         logDirectory:logDirectory, $
         stringPtr:stringPtr, $
         base2:base2, $
         base3:base3, $
         base5:base5, $
         win:win, $
         winVis:winVis, $
         winPix:winPix, $
         mouse:0B, $
         xbox:[0.0,1.0], $
         ybox:[0.0,1.0], $
         autoscale:1, $
         xrange:[0.0,1.0], $
         yrange:[0.0,1.0], $
         autoscaleID:autoscaleID, $
         xyrangebaseid:xyRangeBaseID, $
         xminID:xminID, $
         xmaxID:xmaxID, $
         yminID:yminID, $
         ymaxID:ymaxID, $
         xtitle:xtitle, $
         ytitle:ytitle, $
         title:title, $
         ytitlValues:yTitlValues, $
         titlValues:titlValues, $
         notifyIds:notifyIds}

pState = ptr_new(state,/no_copy)
widget_control,tlb,set_uvalue = pState
widget_control, wIDs[0], set_uvalue=pState
ret = dave_set_focus(tlb)
xmanager,'opan_plotparms',tlb,cleanup = 'plotParmsCleanup',/no_block

; make up a plot parameter button event call and call the event handler to process it
; thus plotting the default first fitted parameter in the list
event = {widget_button}
event.id = plotID
event.handler = plotID
event.top = tlb
event.select = 1
plotParmsAccept, event

return
end