;==============================================================================
; Cleanup pro
pro ng5aPlusData::cleanup
   compile_opt idl2
   
   ptr_free, self.HPtr,self.KPtr,self.LPtr,self.tempPtr,self.hfieldPtr,self.timePtr $
      ,self.EPtr,self.A1Ptr,self.A2Ptr,self.A3Ptr,self.A4Ptr,self.A5Ptr,self.A6Ptr $
      ,self.cntsPtr,self.EpPtr,self.QpPtr,self.QhpPtr,self.QkpPtr,self.QlpPtr,self.dataPtr
      
end

;==============================================================================
; Read calibration data file
function ng5aPlusData::readFile
   compile_opt idl2
   
   ;begin error handler------------------------------------------------------------
   defsysv,'!debug',exist=debugExist
   if (debugExist && (!debug eq 0)) then begin
      catch, catchError
      if (catchError ne 0) then begin
      ;;print, 'Error handled!'
         eTitle = 'Error encountered in ng5aPlusData::isNG5Calib()'
         eMsg = 'An error or unusual condition was encountered!'
         eMsg = [eMsg,'Please, report the following to the DAVE team:']
         eMsg = [eMsg,!error_state.msg]
         void = dialog_message(/error,eMsg,title=eTitle)
         free_lun,lun,/force ; close file, if open
         catch, /cancel
         return, 0
      endif
   endif
   ;end error handler-------------------------------------------------------------
   
   nlines = file_lines(self.filename)
   filename = self.filename
   openr, lun,filename,/get_lun
   
   ; read 1st line and process it
   buffer = ''
   readf, lun, buffer
   
   self.date = strmid(buffer,16,17)        ; experiment date
   self.scanType = strmid(buffer,36,1)     ; scan type 'Q' | 'I'
   toks = long(strsplit(strmid(buffer,39,18),/extract))
   self.mon = toks[0]*toks[1]        ; mon counts
   npts = fix(strmid(buffer,64,6)) ; suggested nos of data points in file
   
   ; Read and process remaninder of header block
   len = (strcmp(self.scanType,'Q'))? 11 : 12
   buffer = strarr(len)
   readf, lun, buffer
   
   self.scanTitle = buffer[1]
   
   if (strcmp(self.scanType,'Q')) then begin
      toks = float(strsplit(buffer[2],/extract,count=ntoks))
      if (ntoks eq 14) then begin
         self.collimation = toks[0:3]
         self.mosaic = toks[4:6]
         self.xtalOrient = toks[7:13]
      endif
      
      toks = float(strsplit(buffer[4],/extract,count=ntoks))
      if (ntoks eq 6) then self.lattParam = toks
      
      toks = float(strsplit(buffer[6],/extract,count=ntoks))
      if (ntoks eq 7) then begin
         self.efixValue = toks[2]
         self.monoDspace = toks[3]
         self.anaDspace = toks[4]
         temp = toks[5]
      endif
      
      self.efixFlag = (strcmp(strmid(buffer[7],27,2),'EA'))? 1 : 0
      
      
      toks = float(strsplit(buffer[8],/extract,count=ntoks))
      if (ntoks eq 7) then hField = toks[6]
      
      toks = strsplit(buffer[10],':',/extract,count=ntoks)
      if (ntoks eq 2) then begin
         nPix = fix(toks[1])
         self.nPix = nPix
      endif
      
      buffer = ''
      readf, lun, buffer
      toks = strsplit(buffer,/extract,count=nCols)
      buffer = fltarr(nCols)
      asdBuffer = lonarr(nPix)
      asd = lonarr(npts,nPix)
      h = fltarr(npts)
      k = fltarr(npts)
      l = fltarr(npts)
      e = fltarr(npts)
      time = fltarr(npts)
      counts = lonarr(npts)
      if (nCols eq 7) then temp = fltarr(npts)
      i = 0
      while (~eof(lun)) do begin
         readf,lun,buffer,asdBuffer
         asd[i,*] = asdBuffer
         h[i] = buffer[0]
         k[i] = buffer[1]
         l[i] = buffer[2]
         e[i] = buffer[3]
         case nCols of
            6: begin
               time[i] = buffer[4]
               counts[i] = buffer[5]
            end
            
            7: begin
               temp[i] = buffer[4]
               time[i] = buffer[5]
               counts[i] = buffer[6]
            end
            
            else:
         endcase
         i++
      endwhile
      
      self.nDat = i
      
      self.dataPtr = ptr_new(asd[0:i-1,*],/no_copy)
      self.hoPtr = ptr_new(h[0:i-1],/no_copy)
      self.koPtr = ptr_new(k[0:i-1],/no_copy)
      self.loPtr = ptr_new(l[0:i-1],/no_copy)
      self.eoPtr = ptr_new(e[0:i-1],/no_copy)
      self.timePtr = ptr_new(time[0:i-1],/no_copy)
      self.tempPtr = ptr_new(temp[0:i-1],/no_copy)
      self.cntsPtr = ptr_new(counts[0:i-1],/no_copy)
      
      
   endif else begin
      toks = float(strsplit(buffer[2],/extract,count=ntoks))
      if (ntoks eq 12) then begin
         self.collimation = toks[0:3]
         self.mosaic = toks[4:6]
         self.wavelength = toks[7]
         hfield = toks[10]
      endif
      
      toks = strsplit(buffer[11],':',/extract,count=ntoks)
      if (ntoks eq 2) then begin
         nASD = fix(toks[1])
      endif
   endelse
   
   free_lun,lun,/force
   
   return, 1
end

;==============================================================================
; Determine if this is an NG5 (SPINS) data file with PSD used in a+ mode
; where pixels are integrated in the vertical direction.
function ng5aPlusData::isNG5aPlusFile
   compile_opt idl2
   
   ;begin error handler------------------------------------------------------------
   defsysv,'!debug',exist=debugExist
   if (debugExist && (!debug eq 0)) then begin
      catch, catchError
      if (catchError ne 0) then begin
      ;;print, 'Error handled!'
         eTitle = 'Error encountered in ng5aPlusData::isNG5aPlusFile()'
         eMsg = 'An error or unusual condition was encountered!'
         eMsg = [eMsg,'Please, report the following to the DAVE team:']
         eMsg = [eMsg,!error_state.msg]
         void = dialog_message(/error,eMsg,title=eTitle)
         catch, /cancel
         return, 0
      endif
   endif
   ;end error handler-------------------------------------------------------------
   
   filename = self.filename
   openr, lun,filename,/get_lun
   
   buffer = ''
   readf, lun, buffer
   ; It must be an ICP Q-buffer file: line 1, col 37 should be letter Q
   char = strmid(buffer,36,1)
   if (~strcmp(char,'Q') && ~strcmp(char,'I')) then return, 0
   
   len = (strcmp(char,'Q'))? 11 : 12
   buffer = strarr(len)
   readf, lun, buffer
   free_lun,lun,/force
   
   ; Is there a '#ASD' entry on line twelve? If so, this is a+ mode SPINS data file
   if (~strcmp(strmid(strtrim(buffer[len-1],2),0,4),'#ASD')) then return, 0
   
   ;; Measurement must have been performed in fixed-final energy mode
   ;if (~strcmp(strmid(buffer[8],27,8),'EA fixed')) then return, 0
   
   return, 1
end

;==============================================================================
pro ng5aPlusData::getProperty,filename=filename,date=date,scanType=scanType $
      ,scanTitle=scanTitle,collimation=collimation,mosaic=mosaic,xtalOrient=xtalOrient $
      ,lattParam=lattParam,monoDspace=monoDspace,anaDspace=anaDspace $
      ,EfixFlag=EfixFlag,EfixValue=EfixValue,wavelength=wavelength $
      ,mon=mon,nDat=nDat,nPix=nPix,A1=A1,A2=A2,A3=A3,A4=A4,A5=A5,A6=A6 $
      ,cnts=cnts,data=data,temp=temp,time=time,hfield=hfield $
      ,ho=ho,ko=ko,lo=lo,Eo=Eo,Ep=Ep,hp=hp,kp=kp,lp=lp,Qp=Qp
      
   compile_opt idl2

   if (arg_present(EfixFlag)) then begin
      EfixFlag = self.EfixFlag
   endif
   if (arg_present(EfixValue)) then begin
      EfixValue = self.EfixValue
   endif
   if (arg_present(wavelength)) then begin
      wavelength = self.wavelength
   endif
   if (arg_present(mon)) then begin
      mon = self.mon
   endif
   if (arg_present(nDat)) then begin
      nDat = self.nDat
   endif
   if (arg_present(nPix)) then begin
      nPix = self.nPix
   endif
   if (arg_present(mosaic)) then begin
      mosaic = self.mosaic
   endif
   if (arg_present(xtalOrient)) then begin
      xtalOrient = self.xtalOrient
   endif
   if (arg_present(lattParam)) then begin
      lattParam = self.lattParam
   endif
   if (arg_present(monoDspace)) then begin
      monoDspace = self.monoDspace
   endif
   if (arg_present(anaDspace)) then begin
      anaDspace = self.anaDspace
   endif
   if (arg_present(filename)) then begin
      filename = self.filename
   endif
   if (arg_present(date)) then begin
      date = self.date
   endif
   if (arg_present(scanType)) then begin
      scanType = self.scanType
   endif
   if (arg_present(scanTitle)) then begin
      scanTitle = self.scanTitle
   endif
   if (arg_present(collimation)) then begin
      collimation = self.collimation
   endif

   if (arg_present(cnts)) then $
      if (ptr_valid(self.cntsPtr)) then  cnts = (*self.cntsPtr)

   if (arg_present(data)) then $
      if (ptr_valid(self.dataPtr)) then  data = (*self.dataPtr)

   if (arg_present(temp)) then $
      if (ptr_valid(self.tempPtr)) then  temp = (*self.tempPtr)
   if (arg_present(hfield)) then $
      if (ptr_valid(self.hfieldPtr)) then  temp = (*self.hfieldPtr)
   if (arg_present(time)) then $
      if (ptr_valid(self.timePtr)) then  time = (*self.timePtr)

   if (arg_present(A1)) then $
      if (ptr_valid(self.A1Ptr)) then  A1 = (*self.A1Ptr)
   if (arg_present(A2)) then $
      if (ptr_valid(self.A2Ptr)) then  A2 = (*self.A2Ptr)
   if (arg_present(A3)) then $
      if (ptr_valid(self.A3Ptr)) then  A3 = (*self.A3Ptr)
   if (arg_present(A4)) then $
      if (ptr_valid(self.A4Ptr)) then  A4 = (*self.A4Ptr)
   if (arg_present(A5)) then $
      if (ptr_valid(self.A5Ptr)) then  A5 = (*self.A5Ptr)
   if (arg_present(A6)) then $
      if (ptr_valid(self.A6Ptr)) then  A6 = (*self.A6Ptr)

   if (arg_present(ho)) then $
      if (ptr_valid(self.hoPtr)) then  ho = (*self.hoPtr)
   if (arg_present(ko)) then $
      if (ptr_valid(self.koPtr)) then  ko = (*self.koPtr)
   if (arg_present(lo)) then $
      if (ptr_valid(self.loPtr)) then  lo = (*self.loPtr)
   if (arg_present(Eo)) then $
      if (ptr_valid(self.EoPtr)) then  Eo = (*self.EoPtr)
   if (arg_present(Ep)) then $
      if (ptr_valid(self.EpPtr)) then  Ep = (*self.EpPtr)
   if (arg_present(Qp)) then $
      if (ptr_valid(self.QpPtr)) then  Qp = (*self.QpPtr)
   if (arg_present(hp)) then $
      if (ptr_valid(self.hpPtr)) then  hp = (*self.hpPtr)
   if (arg_present(kp)) then $
      if (ptr_valid(self.kpPtr)) then  kp = (*self.kpPtr)
   if (arg_present(lp)) then $
      if (ptr_valid(self.lpPtr)) then  lp = (*self.lpPtr)   
end

;==============================================================================
pro ng5aPlusData::setProperty, mon=mon,nDat=nDat,nPix=nPix,xtalOrient=xtalOrient $
      ,cnts=cnts,data=data,temp=temp,time=time,hfield=hfield,lattParam=lattParam $
      ,ho=ho,ko=ko,lo=lo,Eo=Eo,Ep=Ep,hp=hp,kp=kp,lp=lp,Qp=Qp

   compile_opt idl2
   
   if (n_elements(mon) eq 1) then begin
      self.mon = mon
   endif
   
   if (n_elements(nDat) eq 1) then begin
      self.nDat = nDat
   endif
   
   if (n_elements(nPix) eq 1) then begin
      self.nPix = nPix
   endif
   
   if (n_elements(xtalOrient) eq 1) then begin
      self.xtalOrient = xtalOrient
   endif
   
   if (n_elements(lattParam) eq 1) then begin
      self.xtalOrient = lattParam
   endif
   
   if (n_elements(cnts) gt 0) then begin
      if (ptr_valid(self.cnts)) then $
         (*self.cnts) = cnts $
      else $
         self.cnts = ptr_new(cnts)
   endif
   
   if (n_elements(data) gt 0) then begin
      if (ptr_valid(self.data)) then $
         (*self.data) = data $
      else $
         self.data = ptr_new(data)
   endif
   
   if (n_elements(temp) gt 0) then begin
      if (ptr_valid(self.temp)) then $
         (*self.temp) = temp $
      else $
         self.temp = ptr_new(temp)
   endif
   
   if (n_elements(time) gt 0) then begin
      if (ptr_valid(self.time)) then $
         (*self.temp) = time $
      else $
         self.time = ptr_new(time)
   endif
   
   if (n_elements(hfield) gt 0) then begin
      if (ptr_valid(self.hfield)) then $
         (*self.hfield) = hfield $
      else $
         self.hfield = ptr_new(hfield)
   endif
   
   if (n_elements(Ho) gt 0) then begin
      if (ptr_valid(self.HoPtr)) then $
         (*self.HoPtr) = Ho $
      else $
         self.HoPtr = ptr_new(Ho)
   endif
   
   if (n_elements(Ko) gt 0) then begin
      if (ptr_valid(self.KoPtr)) then $
         (*self.KoPtr) = Ko $
      else $
         self.KoPtr = ptr_new(Ko)
   endif
   
   if (n_elements(lo) gt 0) then begin
      if (ptr_valid(self.loPtr)) then $
         (*self.loPtr) = lo $
      else $
         self.loPtr = ptr_new(Lo)
   endif
   
   if (n_elements(Eo) gt 0) then begin
      if (ptr_valid(self.EoPtr)) then $
         (*self.EoPtr) = Eo $
      else $
         self.EoPtr = ptr_new(Eo)
   endif
   
   if (n_elements(Ep) gt 0) then begin
      if (ptr_valid(self.EpPtr)) then $
         (*self.EpPtr) = Ep $
      else $
         self.EpPtr = ptr_new(Ep)
   endif
   
   if (n_elements(Qp) gt 0) then begin
      if (ptr_valid(self.QpPtr)) then $
         (*self.QpPtr) = Qp $
      else $
         self.QpPtr = ptr_new(Qp)
   endif
   
   if (n_elements(Hp) gt 0) then begin
      if (ptr_valid(self.HpPtr)) then $
         (*self.HpPtr) = Hp $
      else $
         self.HpPtr = ptr_new(Hp)
   endif
   
   if (n_elements(Kp) gt 0) then begin
      if (ptr_valid(self.KpPtr)) then $
         (*self.KpPtr) = Kp $
      else $
         self.KpPtr = ptr_new(Kp)
   endif
   
   if (n_elements(lp) gt 0) then begin
      if (ptr_valid(self.lpPtr)) then $
         (*self.lpPtr) = lp $
      else $
         self.lpPtr = ptr_new(Lp)
   endif
   
end

;==============================================================================
function ng5aPlusData::init, filename
   compile_opt idl2
   
   ; filename parameter must be present
   if (n_params() lt 1) then begin
      print, 'A calibration data filename is required'
      return, 0
   endif
   
   ; check filename exist
   finfo = file_info(filename)
   if (finfo.read ne 1) then begin
      print,filename,' is not found or cannot be read'
      return, 0
   endif
   
   self.filename = filename
   
   ; Is this a proper ng5 calibration file?
   if (~self->isNg5aPlusFile()) then begin
      print,filename,' is not an NG5 PSD data file in a+ mode'
      return, 0
   endif
   
   ; Read the data and init data structure
   void = self->readFile()
   
   return, 1
end

;==============================================================================
pro ng5aPlusData__Define
   compile_opt idl2
   
   struct = {ng5aPlusData $
      ,filename:'' $
      ,date:'' $            ; Experiment date: '16-JUN-2001 20:54'
      ,scanType:'' $        ; 'I' | 'Q'
      ,scanTitle:'' $       ; scan title
      
      ,collimation:[0,0,0,0] $ ;
      ,mosaic:[0,0,0] $     ; mosaic spread
      ,xtalOrient:[0.,0.,0.,0.,0.,0.,0.] $  ; sample orientation
      ,lattParam:[0.,0.,0.,0.,0.,0.]  $     ;sample lattice parameters in real space
      
      ,monoDspace:0.0 $     ; monochromator d-spacing
      ,anaDspace:0.0 $      ; analyzer d-spacing
      ,EfixFlag:0 $         ; 0 => Fixed Ei, 1 => Fixed Ef (expected to be 1 for PSD data!!!)
      ,EfixValue:0.0 $        ; norminal Fixed energy value
      ,wavelength:0.0 $     ; wavelength
      
      ,mon:0L $             ; monitor count (value * prefactor)
      ,nDat:0 $             ; nos of data points
      ,nPix:0 $             ; nos of pixels
      
      ,hoPtr:ptr_new() $     ; norminal H in rlu - array of size nDat from raw data
      ,koPtr:ptr_new() $     ; norminal K in rlu - array of size nDat from raw data
      ,loPtr:ptr_new() $     ; norminal L in rlu - array of size nDat from raw data
      ,EoPtr:ptr_new() $     ; norminal Energy transfer - array of size nDat from raw data
      ,tempPtr:ptr_new() $  ; Temperature - array of size nDat from raw data
      ,hfieldPtr:ptr_new() $     ; Magnetic field - array of size nDat from raw data
      ,timePtr:ptr_new() $  ; Temperature - array of size nDat from raw data
      ,A1Ptr:ptr_new() $    ; angle A1 values
      ,A2Ptr:ptr_new() $    ; angle A2 values
      ,A3Ptr:ptr_new() $    ; angle A3 values
      ,A4Ptr:ptr_new() $    ; angle A4 values
      ,A5Ptr:ptr_new() $    ; angle A5 values
      ,A6Ptr:ptr_new() $    ; angle A6 values
      ,cntsPtr:ptr_new() $  ; integrated detector counts
      
      ,EpPtr:ptr_new() $    ; Energy transfer for data pt, pixel - array of size (nDat,nPix)
      ,QpPtr:ptr_new() $    ; Average Q for data pt, pixel - array of size (nDat,nPix)
      ,QhpPtr:ptr_new() $   ; Q along H for data pt, pixel - array of size (nDat,nPix)
      ,QkpPtr:ptr_new() $   ; Q along K for data pt, pixel - array of size (nDat,nPix)
      ,QlpPtr:ptr_new() $   ; Q along L for data pt, pixel - array of size (nDat,nPix)
      ,dataPtr:ptr_new() $  ; Detector counts for data pt, pixel - array of size (nDat,nPix)
      }
end