; Convert hkl values to basis vectors given two orientation vectors
; orient1 = (x1,y1,z1)
; orient2 = (x2,y2,z2)
; ==> orient3 = orient1 x orient2 = (x3,y3,z3)
;
; hkl = (h,k,l)
;
; Then basis (=(b1,b2,b3)) is defined as
;   b1 = determ(mat2)/determ(mat1)
;   b2 = determ(mat3)/determ(mat1)
;   b3 = determ(mat4)/determ(mat1)
; where
;        x1 x2 x3
; mat1 = y1 y2 y3
;        z1 z2 z3
;
;        k  x2 x3
; mat2 = k  y2 y3
;        l  z2 z3
;
;        x1 h  x3
; mat3 = y1 k  y3
;        z1 l  z3
;
;        x1 x2 h
; mat4 = y1 y2 k
;        z1 z2 l
;
function hklTobasis, orient1, orient2, hkl
compile_opt idl2

basis = fltarr(3)

catch, theError
if (theError ne 0) then begin
   catch, /cancel
   return, basis
endif

; orient1, orient2 and hkl should be vectors of length 3
if (n_elements(orient1) ne 3 || n_elements(orient2) ne 3 || n_elements(hkl) ne 3) then return, basis

orient3 = crossp(orient1, orient2)

mat1 = transpose([[orient1],[orient2],[orient3]])
mat2 = transpose([[hkl],[orient2],[orient3]])
mat3 = transpose([[orient1],[hkl],[orient3]])
mat4 = transpose([[orient1],[orient2],[hkl]])

D = determ(mat1,/check)
if (D eq 0.0) then return, 0

basis[0] = determ(mat2,/check) / D
basis[1] = determ(mat3,/check) / D
basis[2] = determ(mat4,/check) / D

return, basis
end