; $Id$
;----------------------------------------------------------------------------------------------------------
; Procedure to subtract background data point by point. This is used in conjunction with the 
; routine extract_data which reads data files and populates the data arrays used here.
;
; Input: 
;     data    : Signal Data
;     bg_data : Background Data 
;     columns : The detectors, monitor, and devices in the data and bg_data arrays
; Output :
;     data    : Background subtracted Signal data
; Optional inputs :
;     devs    : devices that are compared between Signal and Background data for subtraction
;     binsize : tolerances for the values of the devices being compared
;          
; The columns in the data arrays are passed through the vector columns (specified in extract_data)
;
; Columns : [Monitor, Spec, Diff, ErrSpec, ErrDiff, Device Columns (specified in extract_data)]

; If devs not specified, then default devs are a2, a3 and kidney.
; If tolerances not specified they are calculated from the Signal data using the calc_binsize
; 
; The function concat_data used to concatenate the background data in case the files are 
; repeated or the points in the files are to be averaged. It uses the devices passsed through devs.
; 
; Written by: Vivek Thampy and Collin Broholm (01/05/2010)
;----------------------------------------------------------------------------------------------------------
pro MACS_bkg_sub, data, bg_data, columns, devs=devs, binsize=binsize, white=white, _extra=_extra

   if n_elements(white) ne 1 then white = -123.79
   
   n_data = n_elements(data[0,*])
   n_dets = n_elements(where((strmid(columns,0,4) eq 'SPEC') or (strmid(columns,0,4) eq 'DIFF')))
   
   if n_elements(devs) eq 0  then devs = ['A2', 'A3', 'KIDNEY'] 
   n_devs  = n_elements(devs)
   dev_ran = fltarr(n_devs,2)
   dev_col = intarr(n_devs)
   bg_data = MACS_concat_data(bg_data, devs, columns)
   
   ;-------------------------------------------------------------------------------------------------------
   ; Calculate device tolerances if not specified. Calculate range of device values fom data and bin sizes.
   ; Exclude background data points that are beyond that range  
   ;-------------------------------------------------------------------------------------------------------
   if n_elements(binsize) ne n_devs then binsize = fltarr(n_devs)
   for i = 0, n_devs-1 do begin
      dev_col[i] = where(columns eq strupcase(devs[i]))
      if (binsize[i] eq 0) then binsize[i] = MACS_calc_binsize(data[dev_col[i],*], inp_bg_data=bg_data[dev_col[i],*])
      dev_ran[i,0] = [[min(data[dev_col[i],*]) - binsize[i]/2], [max(data[dev_col[i],*]) + binsize[i]/2]]  
      lis = where((bg_data[dev_col[i],*] ge dev_ran[i,0]) and (bg_data[dev_col[i],*] le dev_ran[i,1]),nlis)
      if nlis gt 0 then bg_data = bg_data[*,lis] else begin
         data[0:n_dets,*] = white & return
      endelse
   end 
   
   ;-------------------------------------------------------------------------------------------------------
   ; Index the background data on the devices (bg_inds).
   ; The n-dimensional indices (for n devices) are converted into 1-dimensional indices (inds)
   ; Create a background grid with background data indexed by inds. This grid contains only the Monitor and
   ; detector (both SPEC and DIFF) counts.  
   ;-------------------------------------------------------------------------------------------------------
   n_bg_data = n_elements(bg_data[0,*])
   bg_inds   = long(fix((bg_data[dev_col,*] - rebin(dev_ran[*,0], n_devs, n_bg_data))/rebin(reform(binsize), n_devs, n_bg_data)))
   dev_bins  = round((dev_ran[*,1] - dev_ran[*,0])/binsize)
   
   inds = bg_inds[0,*]
   for j = 1, n_devs-1 do inds += bg_inds[j,*] * MACS_cmproduct(dev_bins[0:j-1])
   
   bg_grid = fltarr(n_dets+1, MACS_cmproduct(dev_bins))
   bg_grid[0:n_dets, inds] = bg_data[0:n_dets, *]
   
   ;-------------------------------------------------------------------------------------------------------
   ; Index the Signal data on the devices (sig_inds).
   ; The n-dimensional indices (for n devices) are converted into 1-dimensional indices (inds)
   ; Go through the signal data point by point, compare indices and subtract
   ;-------------------------------------------------------------------------------------------------------
   sig_inds = long(fix((data[dev_col,*] - rebin(dev_ran[*,0], n_devs, n_data))/rebin(reform(binsize), n_devs, n_data)))
   inds = sig_inds[0,*]
   for j=1, n_devs-1 do inds += sig_inds[j,*] * MACS_cmproduct(dev_bins[0:j-1])

   data[01:n_dets/2, *] -= bg_grid[01:n_dets/2, inds]
   data[n_dets/2+1:n_dets, *]  = sqrt(data[n_dets/2+1:n_dets, inds]^2 + bg_grid[n_dets/2+1:n_dets, inds]^2)
   
   zero = where(bg_grid[0, inds] eq 0, n_zero)
   if n_zero gt 0 then data[0, zero] = white
   
   no_white = where(data[0,*] ne white, nno_white)
   if nno_white gt 0 then data = data[*, no_white]
   
end;MACS_bkg_sub
