; $Id$
;###############################################################################
;
; NAME:
;   fans_datreduc
;
; PURPOSE:
;   FANS data redunction module. It performs the following tasks
;   ** Load one or more sample files
;   ** Load one or more background files
;   ** Load one or more Fast bkgd files
;   ** Allows user to mask (remove) detectors after visual inspection
;      of data
;   ** Correct data for fast bkgd (using a polynomial fit to it)
;   ** Perform bkgd subtraction if required.
;
; CATEGORY:
;   FANS Data Reduction
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   May, 2002
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################


;===============================================================================
; fans_maskDets
;
; PURPOSE:
;   Function removes masked detectors and returns the good ones in an
;   array
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; RETURN VALUES:
;   Array of 'good' detectors to be used in subsequent data
;   reduction.
;
function fans_maskDets, sPtr

if ( (*sPtr).sampCount gt 0) then begin
    sampDataPtr = (*(*sPtr).sDataPtr)[(*sPtr).sampIndex]
    ndets = (*sampDataPtr).ndets
endif else ndets = 50
goodDets = indgen(ndets-2)+3    ; leave out monitors (dets 1 and 2)

widget_control, (*sPtr).mskOnID, get_uvalue=mskon
if (mskon and ptr_valid((*sPtr).mskListPtr)) then begin ; masked dets to be removed
    badDets = (*(*sPtr).mskListPtr)
    ;; remove bad dets from good dets
    nbad = n_elements(badDets)
    if (nbad ge ndets-2) then begin ; removing all detectors!!!
        widget_control, (*sPtr).oViewID, get_uvalue=on
        if (on) then erase      ; clear plot window
        return, -1
    endif
    for i = 0,nbad-1 do $
      goodDets = goodDets[where(goodDets ne badDets[i])]
endif

return, goodDets

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_fastPar
;
; PURPOSE:
;   Event handler. Fit fast background data by manually changing
;   function parameters.
;
; PARAMETERS:
;   event [in] - event structure to be handled.
;
pro fans_fastPar, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_fastPar: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

widget_control, event.top, get_uvalue=sPtr

; retrieve parameters from the text widget
widget_control, event.id, get_value=parstr
if (n_elements(parstr) le 0) then return
fpar = [0.0,0.0,0.0,0.0,0.0,0.0]
tokens = strsplit(parstr[0],' ,',/extract) ; tokenize the string
for i=0,n_elements(tokens)-1 do begin ; convert each token into a float
    reads, tokens[i], af
    fpar[i] = af
endfor 
if ( (fpar[0] eq 0.0) and (fpar[1] eq 0.0) and (fpar[2] eq 0.0) ) then $ ; no fit parameters yet
  return

; generate curve from input poly parameters - first get energy array
index = (*sPtr).fastIndex
fdStructPtr = (*(*sPtr).fDataPtr)[index] ; get a ptr to the approp. fans data struct
omega = (*(*fdStructPtr).energyPtr) ; energy tranfer

; if detectors are to be masked, this has to be taken into account
; when fitting the fast bkgd data
reset = 0
widget_control, (*sPtr).mskOnID, get_uvalue=mskon
if (mskon) then begin 
    goodDets = -1
    goodDets = fans_maskDets(sPtr) ; get a list of good dets
    if ((ptr_valid((*sPtr).mskListPtr) eq 1) or goodDets[0] ne -1) then begin
        reset = 1               ; to make sure data reverts to orig cnts afterwards
        (*sPtr).updFast = 0     ; records that fast fit is in sync with mask dets
        nspec = (*fastDataPtr).nspec
        origTotCnts = (*(*fastDataPtr).totCntsPtr)   
        for i = 0,nspec-1 do begin  ; re-calculate total cnts using only good dets
            (*(*fastDataPtr).totCntsPtr)[i] = total((*(*fastDataPtr).detCntsPtr)[goodDets-1,i])
            (*(*fastDataPtr).totErrsPtr)[i] = sqrt(total(((*(*fastDataPtr).detErrsPtr)[goodDets-1,i])^2))
        endfor
    endif
endif

fans_plotData, sPtr, 'fast', err=err,emsg=emsg ; plot fast bkgd

; reset output calculation flag
(*sPtr).doOutpCalc = 1

; make 'sub FFit' widget active
widget_control, (*sPtr).subFastID, sensitive=1

; reset original total cnts if necessary
if (reset) then (*(*fastDataPtr).totCntsPtr) =  origTotCnts

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_fitFuncSel
;
; PURPOSE:
;   Event handler; Selecting a new fast bkgd fit function.
;
; PARAMETERS:
;   event [in] - event structure to be handled
;
pro fans_fitFuncSel, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_fitFuncSel: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control, event.top, get_uvalue=sPtr

(*sPtr).fitFunc = event.index
(*sPtr).fitPresent = 0          ; require refitting of fast bkgd
(*sPtr).doOutpCalc = 1          ; force re-evaluation of output data

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_Fastfit
;
; PURPOSE:
;   Event handler; Initiate a new fast bkgd fit.
;
; PARAMETERS:
;   event [in] - event structure to be handled
;
; KEYWORDS:
;   sPtr [in] - pointer to state structure
;
pro fans_Fastfit, event, sPtr=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_fastfit: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

if (n_elements(sPtr) eq 0) then $
  widget_control, event.top, get_uvalue=sPtr

index = (*sPtr).fastIndex
fastDataPtr = (*(*sPtr).fDataPtr)[index] ; get a ptr to fast data
omega = (*(*fastDataPtr).energyPtr) ; energy tranfers
dspace = (*fastDataPtr).monDSpace ; get the monochromator d-space value

; if detectors are to be masked, this has to be taken into account
; when fitting the fast bkgd data
reset = 0
widget_control, (*sPtr).mskOnID, get_uvalue=mskon
if (mskon) then begin 
    goodDets = -1
    goodDets = fans_maskDets(sPtr) ; get a list of good dets
    if ((ptr_valid((*sPtr).mskListPtr) eq 1) or goodDets[0] ne -1) then begin
        reset = 1               ; to make sure data reverts to orig cnts afterwards
        (*sPtr).updFast = 0     ; records that fast fit is in sync with mask dets
        nspec = (*fastDataPtr).nspec
        origTotCnts = (*(*fastDataPtr).totCntsPtr)
        origErrCnts = (*(*fastDataPtr).totErrsPtr)
        for i = 0,nspec-1 do begin ; re-calculate total cnts using only good dets
            (*(*fastDataPtr).totCntsPtr)[i] = total((*(*fastDataPtr).detCntsPtr)[goodDets-1,i])
            (*(*fastDataPtr).totErrsPtr)[i] = sqrt(total(((*(*fastDataPtr).detErrsPtr)[goodDets-1,i])^2))
        endfor
    endif
endif
totCnts = (*(*fastDataPtr).totCntsPtr)
totErrs = (*(*fastDataPtr).totErrsPtr)

case (*sPtr).fitFunc of
    0: begin
        funcName = 'fans_Arcsin'        
        par = [1.0,1.0,1.0,0.0,0.0,0.0]
        par[4] = dspace
        par[5] = ((*(*sPtr).mfoPtr)[3] eq 0)? 1.0 : (*sPtr).mev2wnos
        fixpar=[1,1,1,1,0,0]    ; <== fifth and sixth are fixed during fitting
    end
    1: begin
        par = [1.0,1.0]
        fixpar=[1,1]
        funcName = 'fans_Linear'        
    end
    2: begin
        par = [1.0,1.0,1.0]
        fixpar=[1,1,1]
        funcName = 'fans_Quad'        
    end
    3: begin
        par = [1.0,1.0,1.0,1.0]
        fixpar=[1,1,1,1]
        funcName = 'fans_Cubic'        
    end
endcase

; do a non-linear lsqfit using a Levenberg-Marquardt algorithm
yfit = lmfit(omega,totCnts,par,fita=fixpar,weights=1/(totErrs^2),function_name=funcName,/double)

; display fit results in fitted par widget
;mval = max(ABS(par))
;fstr = '(G12.4)'
;parstr = ''
;for i = 0,4 do begin
;    buf = strtrim(string(format=fstr,par[i]),2)
;    parstr = parstr+buf+' '
;endfor
;widget_control, (*sPtr).parID, set_value=strtrim(parstr,2)
(*(*sPtr).fitParsPtr) = par 
(*sPtr).fitPresent = 1
fans_plotData, sPtr, 'fast', err=err,emsg=emsg ; plot fast bkgd

; make 'sub FFit' widget active
widget_control, (*sPtr).subFastID, sensitive=1

; switch on output calculation flag
(*sPtr).doOutpCalc = 1

; reset original total cnts if necessary
if (reset) then begin
    (*(*fastDataPtr).totCntsPtr) = origTotCnts
    (*(*fastDataPtr).totErrsPtr) = origErrCnts
endif

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_changeViewData
;
; PURPOSE:
;   Event handler; Change type (sample, fast bkgd, samp bkgd) of data
;   to be displayed/plotted.
;
; PARAMETERS:
;   event [in] - event structure to be handled
;
pro fans_changeViewData, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_changeviewdata: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

widget_control, event.top, get_uvalue=sPtr

dataType = widget_info(event.id,/uname)

; radio buttons fire an event when they are selected/deselected and
; both will result in a call to fans_changeDispType. When a button is
; deslected/selected, its uvalue has to be changed to 0/1 respectively.
switch event.select of
    0: begin                    ; (exclusive) radio button deselected
        switch dataType of

            'samp': begin
                widget_control, (*sPtr).sViewID, set_uvalue=0
;                widget_control, (*sPtr).plotWinID, sensitive=0
                break
            end

            'bkgd': begin
                widget_control, (*sPtr).bViewID, set_uvalue=0
;                widget_control, (*sPtr).plotWinID, sensitive=0
                break
            end

            'fast': begin
                ;; switch off some functionality
                widget_control, (*sPtr).fViewID, set_uvalue=0
                widget_control, (*sPtr).fitBkgdID, sensitive=0
                ;widget_control, (*sPtr).fitLabID, sensitive=0
                ;widget_control, (*sPtr).parID, sensitive=0
                break
            end

            'outp': begin
                widget_control, (*sPtr).oViewID, set_uvalue=0
                break
            end

        endswitch

        break
    end
    
    1: begin                    ; (exclusive) radio button selected
        switch dataType of
            
            'samp': begin
                
                widget_control, (*sPtr).sViewID, set_uvalue=1
;                widget_control, (*sPtr).plotWinID, sensitive=1
                widget_control, (*sPtr).eSliderID, sensitive=1
                ;; Update current data
                index = (*sPtr).sampIndex
                dataPtr = (*(*sPtr).sDataPtr)[index]
                
                ;; Update plot and slider using selected data
                updatePlotSlider, sPtr,dataType,dataPtr,err=err,emsg=emsg
                break
            end

            'bkgd': begin
                widget_control, (*sPtr).bViewID, set_uvalue=1
;                widget_control, (*sPtr).plotWinID, sensitive=1
                widget_control, (*sPtr).eSliderID, sensitive=1
                ;; Update current data
                index = (*sPtr).bkgdIndex
                dataPtr = (*(*sPtr).bDataPtr)[index]

                ;; Update plot and slider using selected data
                updatePlotSlider, sPtr,dataType,dataPtr,err=err,emsg=emsg

                break
            end

            'fast': begin
                ;; switch on some functionality specific to Fast Bkgd
                widget_control, (*sPtr).fViewID, set_uvalue=1
                widget_control, (*sPtr).fitBkgdID, sensitive=1
                ;widget_control, (*sPtr).fitLabID, sensitive=1
                ;widget_control, (*sPtr).parID, sensitive=1
                widget_control, (*sPtr).eSliderID, sensitive=0

                ;; update plot
                fans_plotData, sPtr, 'fast', err=err,emsg=emsg

                break
            end

            'outp': begin
                widget_control, (*sPtr).oViewID, set_uvalue=1
                widget_control, (*sPtr).eSliderID, sensitive=0
                ;; Calculate output data
                fans_outpCalc, sPtr, err=err, emsg=emsg
                if (err ne 0) then begin
                    printerror,err,emsg
                    return
                endif

                ;; update plot
                fans_plotData, sPtr, 'outp', err=err,emsg=emsg

                break
            end

        endswitch


        
        break
    end

endswitch                       ; switch event.select
    
end



;===============================================================================
; updatePlotSlider
;
; PURPOSE:
;    Updates the plot and slider widgets with the current data array
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in] - data type being displayed: 'samp'|'bkgd'|'fast'
;
;   dataPtr [in] - pointer to data to be plotted.
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro updatePlotSlider, sPtr,dataType,dataPtr,err=err,emsg=emsg
; update plot
fans_plotData,sPtr,dataType

; update energy slider
fans_syncSlider,(*sPtr).eSliderID,(*dataPtr).energyPtr,0,err=err,emsg=emsg
if (err ne 0) then begin
    if (!debug) then printerror,err,emsg
    return
endif

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_dropList
;
; PURPOSE:
;    Event handler. Selection has changed in one of the 3 droplist widgets.
;
; PARAMETERS:
;   event [in] - event dtructure
;
pro fans_dropList, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_droplist: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

; Even when there is only one element in a droplist widget, on windows
; (but not linux), an event is still generated. This is a problem when
; the list _is_'empty' which in this case will contain an item 
; 'no files yet'. Deal with this by ignoring events when the number of
; items < 1.
nosItems = widget_info(event.id,/droplist_number)
if (nosItems le 1) then return

widget_control, event.top, get_uvalue=sPtr

dataType = widget_info(event.id,/uname)

switch dataType of
    
    'samp': begin

        ;; save selected index - useful when not in 'separate' mode
        (*sPtr).sampSepIndex = event.index

        ;; nothing to do if files are not analysed seperately
        if ((*(*sPtr).mfoPtr)[0] ne 2) then break

        ;; make selected data the current data
        (*sPtr).sampIndex = event.index
        
        ;; reset output calculation flag
        (*sPtr).doOutpCalc = 1

        ;; if the display option is 'Sample', then update plot and slider
        widget_control, (*sPtr).sViewID, get_uvalue=on
        if (on) then $
          updatePlotSlider, sPtr,dataType,(*(*sPtr).sDataPtr)[event.index],err=err,emsg=emsg

        ;; if the display option is 'Output', then update output plot
        widget_control, (*sPtr).oViewID, get_uvalue=on
        if (on) then begin    
            ;; Calculate output data
            fans_outpCalc, sPtr, err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
            fans_plotData, sPtr, 'outp', err=err,emsg=emsg
        endif

        break
    end


    'bkgd': begin
        ;; save selected index - useful when not in 'separate' mode
        (*sPtr).bkgdSepIndex = event.index

        ;; nothing to do if files are not analysed seperately
        if ((*(*sPtr).mfoPtr)[1] ne 2) then break

        ;; make selected data the current data
        (*sPtr).bkgdIndex = event.index

        ;; reset output calculation flag
        (*sPtr).doOutpCalc = 1

        ;; if the display option is 'Sample', then update plot
        widget_control, (*sPtr).bViewID, get_uvalue=on
        if (on) then $
          updatePlotSlider, sPtr,dataType,(*(*sPtr).bDataPtr)[event.index],err=err,emsg=emsg

        ;; if the display option is 'Output', then update output plot
        widget_control, (*sPtr).oViewID, get_uvalue=on
        if (on) then begin
            ;; Calculate output data
            fans_outpCalc, sPtr, err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
            fans_plotData, sPtr, 'outp', err=err,emsg=emsg
        endif

        break
    end

    'fast': begin

        ;; save selected index - useful when not in 'separate' mode
        (*sPtr).fastSepIndex = event.index

        ;; nothing to do if files are not analysed seperately
        if ((*(*sPtr).mfoPtr)[2] ne 2) then break

        ;; make selected data the current data
        (*sPtr).fastIndex = event.index

        ;; reset output calculation flag
        ;(*sPtr).doOutpCalc = 1

        ;; if the display option is 'Sample', then update plot
        widget_control, (*sPtr).fViewID, get_uvalue=on
        if (on) then $
          fans_plotData, sPtr, 'fast', err=err,emsg=emsg
        ;;updatePlotSlider, sPtr,dataType,(*(*sPtr).fDataPtr)[event.index],err=err,emsg=emsg

        break
    end


endswitch


end



;===============================================================================
; getFileList
;
; PURPOSE:
;   Get a list of files to be reduced from the user.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   title [in] - to be used for file chooser dialog
;
;   lFileList [out] - fully qualified chosen files.
;
;   fileList [out] - basename of selected files.
;
;   count [out] - number of files chosen.
;
pro getFileList, sPtr, title,lFileList, fileList, count

datDir = (*sPtr).datDir         ; get current data directory

cd, datDir[0], cur=curDir       ; cd to fans data dir
lFileList = dialog_pickfile(title=title,/multiple_files,get_path=newDir, $
                            filter='*.bt4', /must_exist,dialog_parent=(*sPtr).tlbID)
cd, curDir                      ; go back to previous dir

count = (strtrim(lFilelist[0],2) eq '') ? 0 : n_elements(lFileList)

if (count le 0) then return     ; no need to carry on!

(*sPtr).datDir = newDir[0]      ; make newDir the current data dirctory
;; strip dir name from lFileList and store as fileList
fileList = strarr(count)
for i = 0,count-1 do begin
    pos = strpos(lFileList[i],!delimiter,/reverse_search) + 1
    length = strlen(lFileList[i]) - pos
    fileList[i] = strmid(lFileList[i],pos,length)
endfor


end
;-------------------------------------------------------------------------------



;===============================================================================
; getFileList
;
; PURPOSE:
;   Event handler. From the file top-level menu
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_fileMenu, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_filemenu: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control,event.top,get_uvalue=sPtr

dataType = widget_info(event.id,/uname)

switch dataType of
    
    ;; change directory
    'dir': begin
        oldDir = (*sPtr).datDir ; get current setting
        
        newDir = dialog_pickfile(dialog_parent=event.top,/directory,path=oldDir[0]) ; set new dir
        
        if (strlen(newDir) le 0) then newDir = oldDir[0] ; use oldDir if nothing selected
        
        (*sPtr).datDir = newDir 
        
        break
    end

    ;; Choose sample file(s) to reduce
    'samp': begin
        
        ;; get list of files to be load
        getFileList, sPtr,'Select Sample Files',lFileList,fileList,count
        if (count le 0) then break ; bail out

        ;; read the data from the files
        fans_loadFiles, dataType,sPtr,lFileList,fileList,count,err=err,emsg=emsg 
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        ;; make samp and output views sensitive
        widget_control, (*sPtr).sViewID, sensitive=1
        widget_control, (*sPtr).oViewID, sensitive=1

        ;; reset status of multiple sample files processing
        (*sPtr).sampStatus = 0

        ;; if there are multiple files, sum or merge them if required
        if ((count gt 1) and ((*(*sPtr).mfoPtr)[0] ne 2)) then begin
            case (*(*sPtr).mfoPtr)[0] of
                0: fans_sumData, sPtr, dataType, err=err, emsg=emsg
                1: fans_apdData, sPtr, dataType, err=err, emsg=emsg
            endcase 
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
        endif 


        ;; if the display option is 'Sample', then display the current
        ;; dataset in the sample list
        widget_control, (*sPtr).sViewID, get_uvalue=on
        if (on) then begin 

            ;; display current data in list
            fans_plotdata,sPtr,dataType

            ;; Reset the Energy Slider - it should be in sync with the
            ;; current data (ie being plotted). 
            index = (*sPtr).sampIndex
            fdStructPtr = (*(*sPtr).sDataPtr)[index]
            fans_syncSlider,(*sPtr).eSliderID,(*fdStructPtr).energyPtr, 1,err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
            
            ;; make plot window and eslider sensitive 
;            widget_control, (*sPtr).winVisID, sensitive=1
            widget_control, (*sPtr).eSliderID, sensitive=1

        endif

        ;; switch on output calculation flag
        (*sPtr).doOutpCalc = 1

        ;; Reset the mask detector list and widget display
        ;if ptr_valid((*sPtr).mskListPtr) then ptr_free,(*sPtr).mskListPtr
        ;widget_control, (*sPtr).mskListID, set_value=''

        ;; if the display option is 'Output', then update output plot
        widget_control, (*sPtr).oViewID, get_uvalue=on
        if (on) then begin
            ;; Calculate output data
            fans_outpCalc, sPtr, err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
            fans_plotData, sPtr, 'outp', err=err,emsg=emsg
        endif
        
        break
    end

   
    ;; choose bkgd file(s)
    'bkgd': begin

        getFileList, sPtr,'Select Background Files',lFileList,fileList,count
        
        if (count le 0) then break ; bail out
        
        fans_loadFiles, dataType,sPtr,lFileList,fileList,count,err=err,emsg=emsg 
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        ;; make bkgd view and subBkgd sensitive
        widget_control, (*sPtr).bViewID, sensitive=1
        widget_control, (*sPtr).subBkgdID, sensitive=1

        ;; reset status of multiple bkgd files processing
        (*sPtr).bkgdStatus = 0

        ;; if there are multiple files, sum or merge them if required
        if ((count gt 1) and ((*(*sPtr).mfoPtr)[1] ne 2)) then begin
            case (*(*sPtr).mfoPtr)[1] of
                0: fans_sumData, sPtr, dataType, err=err, emsg=emsg
                1: fans_apdData, sPtr, dataType, err=err, emsg=emsg
            endcase 
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
        endif 

        ;; if the display option is 'Bkgd', then display the current
        ;; dataset in the bkgd list
        widget_control, (*sPtr).bViewID, get_uvalue=on
        if (on) then begin 

            ;; display current data
            fans_plotdata,sPtr,dataType

            ;; Reset the Energy Slider - it should be in sync with the
            ;; current data (ie being plotted).
            index = (*sPtr).bkgdIndex 
            fdStructPtr = (*(*sPtr).bDataPtr)[index]
            fans_syncSlider,(*sPtr).eSliderID,(*fdStructPtr).energyPtr, err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif

            ;; make plot window and eslider sensitive 
;            widget_control, (*sPtr).winVisID, sensitive=1
            widget_control, (*sPtr).eSliderID, sensitive=1

        endif

        ;; switch on output calculation flag
        (*sPtr).doOutpCalc = 1

        ;; if the display option is 'Output', then update output plot
        widget_control, (*sPtr).oViewID, get_uvalue=on
        if (on) then begin
            ;; Calculate output data
            fans_outpCalc, sPtr, err=err, emsg=emsg
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
            fans_plotData, sPtr, 'outp', err=err,emsg=emsg
        endif
        
        break
    end
    
    ;; choose Fast bkdg file(s)
    'fast': begin
        
        getFileList, sPtr,'Select Fast Background Files',lFileList,fileList,count
        
        if (count le 0) then break   ; bail out
        
        fans_loadFiles, dataType,sPtr,lFileList,fileList,count,err=err,emsg=emsg 
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif
        
        ;; make fast view sensitive
        widget_control, (*sPtr).fViewID, sensitive=1

        ;; reset status of multiple fast bkgd files processing
        (*sPtr).fastStatus = 0
        widget_control, (*sPtr).subFastID, sensitive=0

        ;; reset fast bkgd fit parameters to 0.0, 0.0, 0.0
        ;parstr = '0.00  0.00  0.00'
        ;widget_control, (*sPtr).parID, set_value=strtrim(parstr,2)

        ;; if there are multiple files, sum or merge them if required
        if ((count gt 1) and ((*(*sPtr).mfoPtr)[2] ne 2)) then begin
            case (*(*sPtr).mfoPtr)[2] of
                0: fans_sumData, sPtr, dataType, err=err, emsg=emsg
                1: fans_apdData, sPtr, dataType, err=err, emsg=emsg
            endcase 
            if (err ne 0) then begin
                printerror,err,emsg
                return
            endif
        endif 


        ;; if the display option is 'Fast', then display the
        ;; data. Note that, unlike for samp and bkgd, only one fast
        ;; data is possible. Multiple fast files will have been merged
        ;; above if they cover different energy ranges.
        widget_control, (*sPtr).fViewID, get_uvalue=on
        if (on) then $
          fans_plotdata,sPtr,dataType
        
        ;; fast bkgd fit not available
        (*sPtr).fitPresent = 0

        ;; switch on output calculation flag
        (*sPtr).doOutpCalc = 1

        break
    end

    ;; save output file using DAVE file format
    'sdave': begin

        fans_2Dave, sPtr,err=err,emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        break
    end

    ;; save output file using DAVE file format
    'sdavetool': begin

        fans_2DaveTool, sPtr,err=err,emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        break
    end

    ;; save output file using 3 Col ascii file format
    'ascii': begin

        fans_2Ascii, sPtr,err=err,emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        break
    end

    ;; save 2D (only individual detector cnts) output file using DAVE file format
    's2dave': begin

        fans_2DDave, sPtr,err=err,emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif

        break
    end

    ;; exit program
    'exit': begin

        widget_control, event.top, /destroy

        break
    end
        
        
endswitch 

end



;===============================================================================
; printerror
;
; PURPOSE:
;   Debug tool: print current error and error message on the command line
;
; PARAMETERS:
;   err [in] - the error number to be displayed
;
;   emsg [in] - the error message to be displayed
;
pro printerror, err, emsg

if (!debug) then begin
    if (n_params() ne 2) then begin
        print,'In "printerror" - Invalid number of parameters in call to "printerror"'
        return
    endif
    
    print,'ERROR encountered... '
    print,'error number: ',err
    print,'error message: ',emsg
endif

end
;-------------------------------------------------------------------------------



;===============================================================================
; cleanDataPtr
;
; PURPOSE:
;   Free or cleanup internal data structure containing FANS data
;   (should now use the IDL procedure heap_free).
;
; PARAMETERS:
;   dataPtr [in] - data structure to be freed
;
pro cleanDataPtr, dataPtr

; obviously, dataPtr has to be valid
if ( ptr_valid(dataPtr) eq 0) then return

; get the data structure and cleanup all its pointers. The FANS data
; structure is defined in fans_initDataPtr
ds = *dataPtr
if ptr_valid(ds.qxPtr) then ptr_free, ds.qxPtr
if ptr_valid(ds.qyPtr) then ptr_free, ds.qyPtr
if ptr_valid(ds.qzPtr) then ptr_free, ds.qzPtr
if ptr_valid(ds.tempPtr) then ptr_free, ds.tempPtr
if ptr_valid(ds.timePtr) then ptr_free, ds.timePtr
if ptr_valid(ds.energyPtr) then ptr_free, ds.energyPtr
if ptr_valid(ds.detCntsPtr) then ptr_free, ds.detCntsPtr
if ptr_valid(ds.detErrsPtr) then ptr_free, ds.detErrsPtr
if ptr_valid(ds.totCntsPtr) then ptr_free, ds.totCntsPtr
if ptr_valid(ds.totErrsPtr) then ptr_free, ds.totErrsPtr
if ptr_valid(ds.historyPtr) then ptr_free, ds.historyPtr
if ptr_valid(ds.monPtr) then ptr_free, ds.monPtr
; finally, clean dataPtr
ptr_free, dataPtr

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_cleanDataPtr
;
; PURPOSE:
;   Wrapper procedure to cleanDataPtr enabling clean up of all
;   datasets for a particular data type (samp, bkgd or fast) by making
;   a single call 
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in] - data type string: 'samp'|'bkgd'|'fast'
;
pro fans_cleanDataPtrs, sPtr, dataType

; can't have this
if ((ptr_valid(sPtr) eq 0) or (n_elements(dataType) le 0)) then begin
    if (!debug) then $
      print,'ERROR: In "fans_cleanDataPtrs" - invalid input parameter(s)'
    return
endif

switch dataType of

    'samp': begin
        i = 0
        while (i lt n_elements((*(*sPtr).sDataPtr)) ) do begin
            cleanDataPtr, (*(*sPtr).sDataPtr)[i]
            i = i+1
        endwhile
        ptr_free, (*sPtr).sDataPtr
        (*sPtr).sampCount = 0

        break
    end

    'bkgd': begin
        i = 0
        while (i lt  n_elements((*(*sPtr).bDataPtr)) ) do begin
            cleanDataPtr, (*(*sPtr).bDataPtr)[i]
            i = i+1
        endwhile
        ptr_free, (*sPtr).bDataPtr
        (*sPtr).bkgdCount = 0

        break
    end

    'fast': begin
        i = 0
        while (i lt  n_elements((*(*sPtr).fDataPtr)) ) do begin
            cleanDataPtr, (*(*sPtr).fDataPtr)[i]
            i = i+1
        endwhile
        ptr_free, (*sPtr).fDataPtr
        (*sPtr).fastCount = 0

        break
    end

    'outp': begin
        i = 0
        while (i lt  n_elements((*(*sPtr).oDataPtr)) ) do begin
            cleanDataPtr, (*(*sPtr).oDataPtr)[i]
            i = i+1
        endwhile
        ptr_free, (*sPtr).oDataPtr
        (*sPtr).outpCount = 0

        break
    end

endswitch



end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_initDataPtr
;
; PURPOSE:
;   Define a suitable data structure that can hold contents of FANS
;   data file.
;
; PARAMETERS:
;   rvaluePtr [in] - Pointer to an existed data structure. If
;                    specified, then initialise the new data structure
;                    with it's contents.
;
; RETURN VALUE:
;   a valid data structure for storing FANS data.
;
function fans_initDataPtr, rvaluePtr

; If there is an rvaluePtr, 
fdStr = {filename:'', $
         date:'', $
         scan:'T', $
         monitor:0L, $   
         prefactor:0L, $
         base:'TIME', $
         type:'RAW', $
         title:'', $
         collimation:'00 00 00 00', $
         mosaic:'0  0  0', $
         orientation:'0 0 0 0.000 0 0 0', $
         latticeA:0.0, $
         latticeB:0.0, $
         latticeC:0.0, $
         latticeAl:0.0, $
         latticeBe:0.0, $
         latticeGa:0.0, $
         eStart:0.0, $
         eFinal:0.0, $
         eDelta:0.0, $
         monDSpace:0.0, $
         anaDSpace:0.0, $
         tStart:0.0, $
         tInc:0.0, $
         qhCenter:0.0, $
         qkCenter:0.0, $
         qlCenter:0.0, $
         qhDelta:0.0, $
         qkDelta:0.0, $
         qlDelta:0.0, $
         hfield:0.0, $          ; header info ends here
         tflag:0B, $
         ndets:128, $           ; however, it seems only 50 are currently active - see below
         nspec:0, $
         monPtr:ptr_new(), $ ;monitor array
         qxPtr:ptr_new(), $
         qyPtr:ptr_new(), $
         qzPtr:ptr_new(), $
         tempPtr:ptr_new(), $
         timePtr:ptr_new(), $
         energyPtr:ptr_new(), $
         detCntsPtr:ptr_new(), $
         totCntsPtr:ptr_new(), $
         totErrsPtr:ptr_new(), $ ; error in totCnts
         detErrsPtr:ptr_new(), $ ; error in detCnts
         historyPtr:ptr_new() $ ; ptr to a string array containing history of file 
        }


if ( ptr_valid(rvaluePtr) gt 0) then begin ; copy values from rvaluePtr
    
    rvalue = *rvaluePtr

    for i = 0,33 do $           ; copy the ordinary structure members
      fdStr.(i) = rvalue.(i)

    for i = 34,n_tags(rvalue)-1 do $            ; copy the pointer structure members
      fdStr.(i) = ptr_new( *(rvalue.(i)) ) ; b/c ptrs in fdStr are initia. to NULL
endif

lvaluePtr = ptr_new(fdStr)

return, lvaluePtr               

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_readData
;
; PURPOSE:
;   Read the contents of a .bt4 file
;
; PARAMETERS:
;   file [in] - filename to be read.
;
;   dataPtr [in|out] - data structure to read file contents into
;
;   nosActDets [in]  - number of active detectors in this file
;
;   conv_factor [in] - energy conversion factor meV <-> wavenumbers
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_readData, file, dataPtr, nosActDets,conv_factor,err=err, emsg=emsg
; Read data from the specified FANS file. The data is then loaded into a
; FANS data structure stored in IDL heap memory. A pointer to the heap
; is then returned.

; reset !error_state
message,/reset_error_state

; check parameter count
if (n_params() ne 4) then begin
    err = 1
    emsg = '"In fans_readData" - Invalid number of procedure parameters'
    return
endif

; check file parameter is defined
if (n_elements(file) le 0 or n_elements(nosActDets) le 0) then begin
    err = 2
    emsg = 'In "fans_readData" - Input parameter(s) undefined'
    return
endif

; open input file
openr, lun, file, /get_lun, error=err

if (err ne 0) then begin        ; if error, get the error msg and exit
    emsg = 'In "fans_readData" - '+!error_state.msg    
    return
endif


; initialised dataPtr to point to a FANS data structure if necessary
if not ptr_valid(dataPtr) then dataPtr = fans_initDataPtr()

; get file header and sort out the useful info from it
line = ''
header = strarr(5)
readf, lun, header
(*dataPtr).filename = strmid(header[0],1,12)
(*dataPtr).date = strmid(header[0],16,17)
(*dataPtr).scan = strmid(header[0],36,1)
(*dataPtr).monitor = long(strmid(header[0],39,12))
(*dataPtr).prefactor = long(strmid(header[0],52,4))
(*dataPtr).base = strmid(header[0],58,4)
(*dataPtr).type = strmid(header[0],71,3)
(*dataPtr).title = strtrim(header[2],2)
(*dataPtr).collimation = strmid(header[3],1,11)
(*dataPtr).mosaic = strmid(header[3],18,8)
(*dataPtr).orientation = strmid(header[3],36,23)
buf = fltarr(6) & readf,lun,buf
(*dataPtr).latticeA = buf[0]
(*dataPtr).latticeB = buf[1]
(*dataPtr).latticeC = buf[2]
(*dataPtr).latticeAl = buf[3]
(*dataPtr).latticeBe = buf[4]
(*dataPtr).latticeGa = buf[5]
readf, lun,line & buf = fltarr(7) & readf, lun,buf
(*dataPtr).eStart = buf[0]
(*dataPtr).eDelta = buf[1]
(*dataPtr).eFinal = buf[2]
(*dataPtr).monDSpace = buf[3]
(*dataPtr).anaDSpace = buf[4] 
(*dataPtr).tStart = buf[5]
(*dataPtr).tInc = buf[6]
readf, lun,line & readf, lun,buf
(*dataPtr).qhCenter = buf[0]
(*dataPtr).qkCenter = buf[1]
(*dataPtr).qlCenter = buf[2]
(*dataPtr).qhDelta = buf[3]
(*dataPtr).qkDelta = buf[4]
(*dataPtr).qlDelta = buf[5]
(*dataPtr).hfield = buf[6]
readf, lun,line,line
if (strmid(line,37,1) eq 'T') then (*dataPtr).tflag = 1

; now in the data section.
hbuf = ((*dataPtr).tflag)? fltarr(7) : fltarr(6)
dbuf = lonarr((*dataPtr).ndets)
nspec = 2000 & i = 0 
time = (temp = (totErrs = (totCnts =  (omega = (qx = (qy = (qz = fltarr(nspec))))))))
;nosActDets = (*sPtr).nosActDets 
detCnts = fltarr(nosActDets,nspec)
detErrs = fltarr(nosActDets,nspec)
 
while (not eof(lun)) do begin
    readf, lun, hbuf,dbuf       ; read one data block (per energy transfer)
    qx[i] = hbuf[0]
    qy[i] = hbuf[1]
    qz[i] = hbuf[2]
    omega[i] = hbuf[3]
    if ( (*dataPtr).tflag) then begin
        temp[i] = hbuf[4]
        time[i] = hbuf[5]
        totCnts[i] = hbuf[6]
    endif else begin
        temp[i] = (*dataPtr).tStart
        time[i] = hbuf[4]
        totCnts[i] = hbuf[5]
    endelse
    totErrs[i] = (totCnts[i] le 0)? 1 : sqrt(totCnts[i]) ; assign 1 to erros if cnts <= 0
    detCnts[*,i] = dbuf[0:nosActDets-1] ; trim to active detectors only
    res = where(dbuf le 0)      ; locate indices where detcts <= 0
    if (res[0] ne -1) then begin ; and assign an error value of 1 to
        dbuf[res] = 1
    endif
    detErrs[*,i] = sqrt(dbuf[0:nosActDets-1])
    i = i + 1
endwhile

free_lun, lun

; move data into the heap and save pointers in dataPtr; trim spectra
; to number of points read from file
(*dataPtr).nspec = i 
(*dataPtr).ndets = nosActDets
(*dataPtr).qxPtr = ptr_new(qx[0:i-1],/no_copy)
(*dataPtr).qyPtr = ptr_new(qy[0:i-1],/no_copy)
(*dataPtr).qzPtr = ptr_new(qz[0:i-1],/no_copy)
(*dataPtr).energyPtr = ptr_new(omega[0:i-1]*conv_factor,/no_copy)
(*dataPtr).tempPtr = ptr_new(temp[0:i-1],/no_copy)
(*dataPtr).timePtr = ptr_new(time[0:i-1],/no_copy)
(*dataPtr).totCntsPtr = ptr_new(totCnts[0:i-1],/no_copy)
(*dataPtr).totErrsPtr = ptr_new(totErrs[0:i-1],/no_copy)
(*dataPtr).detCntsPtr = ptr_new(detCnts[*,0:i-1],/no_copy)
(*dataPtr).detErrsPtr = ptr_new(detErrs[*,0:i-1],/no_copy)
(*dataPtr).monPtr = ptr_new(fltarr(i) + (*dataPtr).monitor * (*dataPtr).prefactor)

; finally check for untrapped errors
err = 0
emsg = ''
if (!error_state.code ne 0) then begin
    err = !error_state.code
    emsg = 'In "fans_loadFiles" - ' + !error_state.msg
    fans_cleanDataPtrs, dataPtr
    message,/reset_error_state
endif



end
;-------------------------------------------------------------------------------



;===============================================================================
; defaultOutpFiles
;
; PURPOSE:
;   Determine an output filename(s) for storing the reduced data from the
;   input sample filename.
;
; PARAMETERS:
;   sPtr [in] - Pointer to state structure.
;
;   ofiles [out] - The output filename
;
pro defaultOutpFiles, sPtr, ofiles

for i = 0,n_elements(ofiles)-1 do begin
    ofile = (*(*sPtr).sampListPtr)[i]
    ;; get base name for output file [replace the current file extension with '.dave' and add the]
    ofiles[i] = strmid(ofiles[i],0,strpos(ofiles[i],'.',/reverse_search)) ;+'.dave'
endfor 

(*(*sPtr).outpListPtr) = ofiles

end
;-------------------------------------------------------------------------------



;===============================================================================
; getFileList
;
; PURPOSE:
;   Get a list of files to be reduced from the user.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in|out] - data type string: 'samp'|'bkgd'|'fast'
;
;   lFileList [in] - fully qualified chosen files.
;
;   fileList [in] - basename of selected files.
;
;   count [in] - number of files chosen.
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_loadFiles, dataType, sPtr,lFileList,fileList,count,err=err, emsg=emsg

err=0  &  emsg=''

; check parameter count
if (n_params() ne 5) then begin
    err = 1
    emsg = 'In "fans_loadFiles" - Invalid number of procedure parameters'
    return
endif

; check input parameters are defined
if ( (n_elements(dataType) le 0) or (ptr_valid(sPtr) eq 0) or $
     (n_elements(lFileList) le 0) or (n_elements(fileList) le 0) or $
     (n_elements(count) le 0) ) then begin
    err = 2
    emsg = 'In "fans_loadFiles" - Input parameter(s) undefined'
    return
endif

; which energy units is selected - meV or wavenumbers? Set the proper
;                                  conversion factor
conv_factor = ((*(*sPtr).mfoPtr)[3] eq 0)? 1.0 : (*sPtr).mev2wnos
switch dataType of
    ;; load sample files
    'samp': begin
        ;; clean up any previous sample data
        if ( ptr_valid((*sPtr).sDataPtr) ) then $
          fans_cleanDataPtrs, sPtr, 'samp'

        ;; load new sample file list
        widget_control, (*sPtr).sampID, set_value=fileList
        (*(*sPtr).sampListPtr) = lFileList
        (*sPtr).sampCount = count
        (*sPtr).sampIndex = 0

        ;; use sample filenames to create default output filename(s)
        ;; and store in sPtr
        defaultOutpFiles, sPtr, fileList

        ;; loop through files and read data
        tmparr=ptrarr(count)    ; create a pointer array - initialised to NULL!
        tmpPtr = 0 & i=0        ; tmpPtr will point to the FANS data later

        while (i lt count) do begin
            fans_readData, lFileList[i],tmpPtr,(*sPtr).nosActDets,conv_factor,err=err,emsg=emsg ; read the data file
            
            if (err ne 0) then return ; unsuccessful so no need to continue

            ;; fill the historyPtr
            history = ['Sample file(s): ', lFileList[i]]
            (*tmpPtr).historyPtr = ptr_new(history)

            tmparr[i] = tmpPtr  ; copy ptr to an array
            tmpPtr = 0          ; make tmpPtr an invalid ptr
            i = i+1
        end
        
        ;; move arr of ptrs to the heap 
        (*sPtr).sDataPtr = ptr_new(tmparr,/no_copy)
        
        break
    end

    ;; load bkgd files
    'bkgd': begin
        ;; clean up any previous bkgd data
        if ( ptr_valid((*sPtr).bDataPtr) ) then $
          fans_cleanDataPtrs, sPtr, 'bkgd'

        ;; load new sample file list
        widget_control, (*sPtr).bkgdID, set_value=fileList
        (*(*sPtr).bkgdListPtr) = lFileList
        (*sPtr).bkgdCount = count
        (*sPtr).bkgdIndex = 0

        ;; loop through files and read data
        tmparr=ptrarr(count)    ; create a pointer array - initialised to NULL!
        tmpPtr = 0 & i=0        ; tmpPtr will point to the FANS data later

        while (i lt count) do begin
            fans_readData, lFileList[i],tmpPtr,(*sPtr).nosActDets,conv_factor,err=err,emsg=emsg ; read the data file
            
            if (err ne 0) then return ; unsuccessful so no need to continue

            ;; fill the historyPtr
            history = ['Background file(s): ', lFileList[i]]
            (*tmpPtr).historyPtr = ptr_new(history)

            tmparr[i] = tmpPtr  ; copy ptr to an array
            tmpPtr = 0          ; make tmpPtr an invalid ptr
            i = i+1
        end
        
        ;; move arr of ptrs to the heap 
        (*sPtr).bDataPtr = ptr_new(tmparr,/no_copy)
  
        break
    end

    ;; load fast bkgd files
    'fast': begin
        ;; clean up any previous fast bkgd data
        if ( ptr_valid((*sPtr).fDataPtr) ) then $
          fans_cleanDataPtrs, sPtr, 'fast'

        ;; load new sample file list
        widget_control, (*sPtr).fastID, set_value=fileList
        (*(*sPtr).fastListPtr) = lFileList
        (*sPtr).fastCount = count
        (*sPtr).fastIndex = 0

        ;; loop through files and read data
        tmparr=ptrarr(count)    ; create a pointer array - initialised to NULL!
        tmpPtr = 0 & i=0        ; tmpPtr will point to the FANS data later

        while (i lt count) do begin
            fans_readData, lFileList[i],tmpPtr,(*sPtr).nosActDets,conv_factor,err=err,emsg=emsg ; read the data file
            
            if (err ne 0) then return ; unsuccessful so no need to continue

            ;; fill the historyPtr
            history = ['Fast Background file(s): ', lFileList[i]]
            (*tmpPtr).historyPtr = ptr_new(history)

            tmparr[i] = tmpPtr  ; copy ptr in an array
            tmpPtr = 0          ; make tmpPtr an invalid ptr
            i = i+1
        end
        
        ;; move arr of ptrs to the heap 
        (*sPtr).fDataPtr = ptr_new(tmparr,/no_copy)

        break
    end
endswitch

end



;===============================================================================
; fans_apdData
;
; PURPOSE:
;   Append sample, bkgd or fast bkgd datasets subject to the
;   requirement that the energy range of the data DO NOT COMPLETELY
;   OVERLAP. Any overlapping points in the higher energy-range data is
;   ignored.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in] - data type string: 'samp'|'bkgd'|'fast'
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_apdData, sPtr, dataType, err=err, emsg=emsg
err=0  & emsg=''

; get appropriate data; samp, bkgd or fast bkgd
switch dataType of

    'samp': begin
        nd = (*sPtr).sampCount
        if (nd le 1) then return
        dataPtr = (*sPtr).sDataPtr
        typeName = 'Sample'
        
        break
    end

    'bkgd': begin
        nd = (*sPtr).bkgdCount
        if (nd le 1) then return
        dataPtr = (*sPtr).bDataPtr
        typeName = 'Background'

        break
    end

    'fast': begin
        nd = (*sPtr).fastCount
        if (nd le 1) then return
        dataPtr = (*sPtr).fDataPtr
        typeName = 'Fast bkgd'

        break
    end

endswitch

; Get the min and max energy values of the dataset
minima = fltarr(nd) - 1
maxima = minima
for i = 0,nd-1 do begin
    mine = min( (*(*((*dataPtr)[i])).energyPtr), max=maxe)
    minima[i] = mine
    maxima[i] = maxe
endfor 
ord = sort(minima)              ; find order of increasing min energy


; make a copy of the dataset in the list with the smallest min
; energy. 
mergePtr = fans_initDataPtr((*dataPtr)[ord[0]])
e1Ptr = (*mergePtr).energyPtr  
mon1 = float((*mergePtr).monitor * (*mergePtr).prefactor)
;oldHist = (*(*mergePtr).historyPtr)
(*(*mergePtr).historyPtr)[0] = typeName+' data is a merger of the following file(s):'

; loop through and merge the datasets.
min1 = minima[ord[0]]  &  max1 = maxima[ord[0]]
for i = 1,nd-1 do begin
    j = ord[i]

    ;; ignore cases of complete overlap
    min2 = minima[j]  &  max2 = maxima[j]
    if ( (min1 le min2)  and (max1 ge max2) ) then continue ; total overlap - goto next data
    
    ;; identify partially overlapping regions. Only the upper bounds
    ;; of the overlap is important since, for FANS, it was decided
    ;; that the overlapping data for the higher energy scan should
    ;; simply be discarded.
    curPtr = (*dataPtr)[j]
    e2Ptr = (*curPtr).energyPtr
    ii = 0
    res = where( *e2Ptr gt max1)
    if (res[0] ge 0) then ii = res[0] ; ii is the *e2Ptr index of upper bound of overlap
    
    ;; now copy non-overlapping data from the higher energy scan to
    ;; the merged data
    ;; NOTE: the second data is scale to the monitor counts of the first
    ndets = (*mergePtr).ndets
    nspec1 = (*mergePtr).nspec
    nspec2 = (*curPtr).nspec
    mon2 = float((*mergePtr).monitor * (*mergePtr).prefactor)
    scale = mon1/mon2
    (*mergePtr).eFinal = max2
    (*mergePtr).nspec = (*mergePtr).nspec + nspec2 - ii
    nspec = (*mergePtr).nspec
    (*(*mergePtr).qxPtr) = [(*(*mergePtr).qxPtr),(*(*curPtr).qxPtr)[ii:nspec2-1]]
    (*(*mergePtr).qyPtr) = [(*(*mergePtr).qyPtr),(*(*curPtr).qyPtr)[ii:nspec2-1]]
    (*(*mergePtr).qzPtr) = [(*(*mergePtr).qzPtr),(*(*curPtr).qzPtr)[ii:nspec2-1]]
    (*(*mergePtr).tempPtr) = [(*(*mergePtr).tempPtr),(*(*curPtr).tempPtr)[ii:nspec2-1]]
    (*(*mergePtr).energyPtr) = [(*(*mergePtr).energyPtr),(*(*curPtr).energyPtr)[ii:nspec2-1]]
    (*(*mergePtr).timePtr) = [(*(*mergePtr).timePtr),(*(*curPtr).timePtr)[ii:nspec2-1] * scale]
    (*(*mergePtr).totCntsPtr) = [(*(*mergePtr).totCntsPtr),(*(*curPtr).totCntsPtr)[ii:nspec2-1] * scale]
    (*(*mergePtr).totErrsPtr) = [(*(*mergePtr).totErrsPtr),(*(*curPtr).totErrsPtr)[ii:nspec2-1] * scale]
    detCnts = fltarr(ndets,nspec)
    detErrs = fltarr(ndets,nspec)
    detCnts[0:ndets-1,0:nspec1-1] = (*(*mergePtr).detCntsPtr)
    detErrs[0:ndets-1,0:nspec1-1] = (*(*mergePtr).detErrsPtr)
    detCnts[0:ndets-1,nspec1:nspec-1] = (*(*curPtr).detCntsPtr)[*,ii:nspec2-1] * scale
    detErrs[0:ndets-1,nspec1:nspec-1] = (*(*curPtr).detErrsPtr)[*,ii:nspec2-1] * scale
    (*(*mergePtr).detCntsPtr) = detCnts
    (*(*mergePtr).detErrsPtr) = detErrs
    ;;(*(*mergePtr).detCntsPtr) = [(*(*mergePtr).detCntsPtr),(*(*curPtr).detCntsPtr)[*,ii:nspec2-1]]
    max1 = max2

    ;; update history pointer with merged data filename (index 1 of
    ;; the dataPtr.historyPtr)
    (*(*mergePtr).historyPtr) = [(*(*mergePtr).historyPtr),(*(*curPtr).historyPtr)[1]]
endfor

; Add merged data to list of data. Update info for appropriate data type
if (ptr_valid(mergePtr) eq 0) then return     

*dataPtr = [*dataPtr,mergePtr]  
switch dataType of 

    'samp': begin
        (*sPtr).sampIndex = nd
;        (*sPtr).sampCount = nd + 1
        (*sPtr).sampStatus = 2
        break
    end

    'bkgd': begin
        (*sPtr).bkgdIndex = nd
;        (*sPtr).bkgdCount = nd + 1
        (*sPtr).bkgdStatus = 2
        break
    end 

    'fast': begin
        (*sPtr).fastIndex = nd
;        (*sPtr).fastCount = nd + 1
        (*sPtr).fastStatus = 2
        break
    end 

endswitch 

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_sumData
;
; PURPOSE:
;   Calculates the sum or weighted average of all the current sample,
;   bkgd or fast bkgd data.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in] - data type string: 'samp'|'bkgd'|'fast'
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_sumData, sPtr, dataType, err=err, emsg=emsg
err=0  & emsg=''

; get appropriate data; samp or bkgd
switch dataType of

    'samp': begin
        nd = (*sPtr).sampCount
        if (nd le 1) then return
        dataPtr = (*sPtr).sDataPtr
        fnames = (*(*sPtr).sampListPtr)
        typeName = 'Sample '
        id = 0
        break
    end

    'bkgd': begin
        nd = (*sPtr).bkgdCount
        if (nd le 1) then return
        dataPtr = (*sPtr).bDataPtr
        fnames = (*(*sPtr).bkgdListPtr)
        typeName = 'Background '
        id = 1
        break
    end

    'fast': begin
        nd = (*sPtr).fastCount
        if (nd le 1) then return
        dataPtr = (*sPtr).fDataPtr
        fnames = (*(*sPtr).fastListPtr)
        typeName = 'Fast bkgd '
        id = 2
        break
    end

endswitch

; Get the min and max energy values of the dataset
minima = fltarr(nd) - 1
maxima = minima
for i = 0,nd-1 do begin
    mine = min( (*(*((*dataPtr)[i])).energyPtr), max=maxe)
    minima[i] = mine
    maxima[i] = maxe
endfor 
ord = sort(minima)              ; find order of increasing min energy

; make a copy of the first dataset in the list. Note that dataPtr
; points to an array of ptrs (each pointing to a dataset).
sumPtr = fans_initDataPtr((*dataPtr)[ord[0]])
monConstant = (*sumPtr).monitor * (*sumPtr).prefactor ; record that for first dataset
omega1 = (*(*sumPtr).energyPtr)
temp1 =  (*(*sumPtr).tempPtr)
(*(*sumPtr).historyPtr)[0] = typeName+'data is a sum of the following file(s):'

; Monitor requires special attention! The monitor count is made of a
; product of the mon cnt and the prefactor and it is not possible to
; keep track of these separately when files are averaged. So set monitor
; count to mon*pref and pref to 1. Subsequent data are scaled to the
; mon*pref value of the first dataset.
;(*sumPtr).monitor = (*sumPtr).monitor * (*sumPtr).prefactor
;(*sumPtr).prefactor = 1

; loop through and sum all the datasets
; only the time, energy, detCnts, totCnts and totErrs components need
; to be averaged! - this is done in the inner loop
min1 = minima[ord[0]]  &  max1 = maxima[ord[0]]
for i = 1,nd-1 do begin         
    j = ord[i]

    ;; if dataset barely overlap or not at all then skip the current
    ;; dataset and advise user to select the merge option instead
    min2 = minima[j]  &  max2 = maxima[j]
    ;; get the current dataset to be dealt with
    curPtr = (*dataPtr)[j]
    omega2 = (*(*curPtr).energyPtr)
    temp2 = (*(*curPtr).tempPtr)
    size1 = n_elements(omega1)
    size2 = n_elements(omega2)
;    mon1 = float((*sumPtr).monitor * (*sumPtr).prefactor)
;    mon2 = float((*curPtr).monitor * (*curPtr).prefactor)
;    mon_scale = mon1/mon2
;    mon_scale_sqrt = sqrt(mon_scale)
    monitor1 = (*(*sumPtr).monPtr)
    monitor2 = (*(*curPtr).monPtr)
    
    ;; Examine various exclusive scenarios

    ;---------------------------------------------------------------
    ;; Scenario 1 - min2 > min1 ==> there is a 'no overlap' region in
    ;; dataset1. Have to copy this data into a buffer
    ibeg1 = 0 & iend1=0 & ibeg2=0 & iend2=0
    if (min1 ne min2) then begin
        res = where(omega1 ge min2) ; ge is deliberate
        iend1 = (res[0] le 0)? size1-1 : res[0] - 1
        if (iend1 ge 0) then begin
            omega = omega1[ibeg1:iend1]
            times = (*(*sumPtr).timePtr)[ibeg1:iend1]
            temperature = (*(*sumPtr).tempPtr)[ibeg1:iend1]
            totCnts = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
;            totErrs = (*(*sumPtr).totErrsPtr)[ibeg1:iend1]
            detCnts = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
;            detErrs = (*(*sumPtr).detErrsPtr)[*,ibeg1:iend1]
            monitor = monitor1[ibeg1:iend1]
            ibeg1=iend1+1
            iend1 = iend1 + 1
        endif
    endif
    
    ;----------------------------------------------------------------------
    ;; Scenario 2 - the overlap region;
    if (max1 lt min2) then goto, scenario3 ; inadequate overlap
    res = where(omega1 gt max2) ; the 'gt' is deliberate!
    iend1 = (res[0] lt 0)? size1-1 : res[0]-1 ; upper bound of overlap for ds1
    if (iend1 lt ibeg1) then iend1 = ibeg1
    ibeg2 = 0                   ; lower bound of overlap for ds2
    res = where(omega2 gt max1) 
    iend2 = (res[0] lt 0)? size2-1 : res[0]-1 ; upper bound of overlap for ds2
    if (iend2 lt ibeg2) then iend2 = ibeg2
    if ((where(omega1[ibeg1:iend1] ne omega2[ibeg2:iend2]))[0] ge 0) then begin 
        ;; ==> in overlap region, omega1 != omega2
        ;; hence need to merge (interleave) the data

        ;; determine which of dataset has a smaller min omega and use
        ;; that ds as base for inserting or appending the other ds 
        if (omega1[ibeg1] le omega2[ibeg2]) then begin
            om = omega1[ibeg1:iend1]
            mn = monitor1[ibeg1:iend1]
            temp = temp1[ibeg1:iend1]
            t = (*(*sumPtr).timePtr)[ibeg1:iend1]
            x = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
;            w = (*(*sumPtr).totErrsPtr)[ibeg1:iend1]
            xd = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
;            wd = (*(*sumPtr).detErrsPtr)[*,ibeg1:iend1]
            om2 = omega2[ibeg2:iend2]
            mn2 = monitor2[ibeg2:iend2]
            tmp2 = temp2[ibeg2:iend2]
            t2 = (*(*curPtr).timePtr)[ibeg2:iend2] ;* mon_scale
            x2 = (*(*curPtr).totCntsPtr)[ibeg2:iend2] ;* mon_scale
;            w2 = (*(*curPtr).totErrsPtr)[ibeg2:iend2] * mon_scale ;mon_scale_sqrt
            xd2 = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2] ;* mon_scale
;            wd2 = (*(*curPtr).detErrsPtr)[*,ibeg2:iend2] * mon_scale ;mon_scale_sqrt
            ibeg = 0
            iend = iend2 - ibeg2
            len = n_elements(om) - 1
            first_dset = 1 
        endif else begin
            om = omega2[ibeg2:iend2]
            mn = monitor2[ibeg2:iend2]
            temp = temp2[ibeg2:iend2]
            t = (*(*curPtr).timePtr)[ibeg2:iend2] ;* mon_scale
            x = (*(*curPtr).totCntsPtr)[ibeg2:iend2] ;* mon_scale
;            w = (*(*curPtr).totErrsPtr)[ibeg2:iend2] * mon_scale ;mon_scale_sqrt
            xd = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2] ;* mon_scale
;            wd = (*(*curPtr).detErrsPtr)[*,ibeg2:iend2] * mon_scale ;mon_scale_sqrt
            om2 = omega1[ibeg1:iend1]
            mn2 = monitor1[ibeg1:iend1]
            tmp2 = temp1[ibeg1:iend1]
            t2 = (*(*sumPtr).timePtr)[ibeg1:iend1]
            x2 = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
;            w2 = (*(*sumPtr).totErrsPtr)[ibeg1:iend1]
            xd2 = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
;            wd2 = (*(*sumPtr).detErrsPtr)[*,ibeg1:iend1]
            ibeg = 0
            iend = iend1 - ibeg1
            len = n_elements(om) - 1 ;iend2 - ibeg2
            first_dset = 2
        endelse
        for k=ibeg,iend do begin
            ;; loop thro and locate pos of om2[k] in om

            res = where(om ge om2[k])
            if ( res[0] ge 0 ) then begin 
                if (om2[k] eq om[res[0]]) then begin
                    ;; ==> energy value of om2[k] matches with one in om
                    ;; therefore perform an average.
                    kk = res[0]
                    x3 = x[kk]
                    x4 = x2[k]
                    t3 = t[kk]
                    t4 = t2[k]
                    xd3 = xd[*,kk]
                    xd4 = xd2[*,k]

                    ;; 1) Average total counts
                    x[kk] = x[kk] + x2[k]
                    mn[kk] = mn[kk] + mn2[k]
                    ;w[kk] = 0.5 * sqrt(sum)
                    ;; 2) Average counting times
                    t[kk] = t[kk] + t2[k]
                    ;; 3) Average detector counts
                    xd[*,kk] = xd[*,kk] + xd2[*,k]
                    ;wd[*,kk] = 0.5 * sqrt(sum)

;                    ;; 1) Average total counts
;                    w3sq = (w[kk])^2
;                    w4sq = (w2[k])^2
;                    invwsq = (1.0/w3sq) + (1.0/w4sq)
;                    x[kk] = ((x3/w3sq) + (x4/w4sq)) / invwsq
;                    w[kk] = sqrt(1/invwsq)
;                    ;; 2) Average counting times
;                    t[kk]  = ((t3/w3sq) + (t4/w4sq)) / invwsq
;                    ;; 3) Average detector counts
;                    wd3sq = (wd[*,kk])^2
;                    wd4sq = (wd2[*,k])^2
;                    invwdsq = (1.0/wd3sq) + (1.0/wd4sq) 
;                    xd[*,kk] = ((xd3/wd3sq) + (xd4/wd4sq)) / invwdsq
;                    wd[*,kk] = sqrt(1/invwdsq) 


                    continue    ; move on to next for loop iteration
                endif
            endif  
            sz = size(xd)
            nw = sz[1]  
            ns = (sz[0] eq 1)? 1 : sz[2]
            tmpxd = fltarr(nw,ns+1)
            tmpwd = fltarr(nw,ns+1)
            if (res[0] lt 0) then begin
                ;; ==> need to _append_ ds2 values at end
                om = [om,om2[k]]
                temp = [temp,tmp2[k]]
                t = [t,t2[k]]
                x = [x,x2[k]]
                mn = [mn,mn2[k]]
;                w = [w,w2[k]]
                tmpxd[*,0:ns-1] = xd
;                tmpwd[*,0:ns-1] = wd
                tmpxd[*,ns] = xd2[*,k]
;                tmpwd[*,ns] = wd2[*,k]
                xd = tmpxd
                wd = tmpwd
            endif else begin
                ;; ==> need to _insert_ ds2 vals before res[0] index
                ;; in ds1
                kk = res[0]
                om = [om[0:kk-1],om2[k],om[kk:len]]
                temp = [temp[0:kk-1],tmp2[k],temp[kk:len]]
                t = [t[0:kk-1],t2[k],t[kk:len]]
                x = [x[0:kk-1],x2[k],x[kk:len]]
                mn = [mn[0:kk-1],mn2[k],mn[kk:len]]
;                w = [w[0:kk-1],w2[k],w[kk:len]]
                tmpxd[*,0:kk-1] = xd[*,0:kk-1]
;                tmpwd[*,0:kk-1] = wd[*,0:kk-1]
                tmpxd[*,kk] = xd2[*,k]
;                tmpwd[*,kk] = wd2[*,k]
                tmpxd[*,kk+1:ns] = xd[*,kk:len]
;                tmpwd[*,kk+1:ns] = wd[*,kk:len]
                xd = tmpxd
;                wd = tmpwd
            endelse
            len = n_elements(om) - 1 ; update len to reflect expanded buffers
        endfor
        iend1 = iend1 + 1
        iend2 = iend2 + 1
    endif else begin 
        ;; ==> in overlap region, omega1 == omega2
        ;; hence do the averaging in the overlap region
        om = omega1[ibeg1:iend1]
        mn = monitor1[ibeg1:iend1]
        mn2 = monitor2[ibeg2:iend2]
        temp = temp1[ibeg1:iend1]

        ;; 1) Average total counts
        buf1 = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
        buf2 = (*(*curPtr).totCntsPtr)[ibeg2:iend2]
        x =  buf1 + buf2
        mn = mn + mn2
        
        ;; 2) Average counting times
        buf1 = (*(*sumPtr).timePtr)[ibeg1:iend1]
        buf2 = (*(*curPtr).timePtr)[ibeg2:iend2]; * mon_scale
        t = buf1 + buf2

        ;; 3) Average detector counts
        buf1 = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
        buf2 = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2]; * mon_scale
        xd = buf1 + buf2
        ;wd = 0.5 * sqrt(sum)

;        ;; 1) Average total counts
;        x1 = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
;        x2 = (*(*curPtr).totCntsPtr)[ibeg2:iend2] * mon_scale
;        w1sq = ((*(*sumPtr).totErrsPtr)[ibeg1:iend1])^2
;        w2sq = ((*(*curPtr).totErrsPtr)[ibeg2:iend2])^2 * (mon_scale)^2
;        invwsq = (1.0/w1sq) + (1.0/w2sq)
;        x = ((x1/w1sq) + (x2/w2sq)) / invwsq
;        w = sqrt(1/invwsq)
;        ;; 2) Average counting times
;        t1 = (*(*sumPtr).timePtr)[ibeg1:iend1]
;        t2 = (*(*curPtr).timePtr)[ibeg2:iend2] * mon_scale
;        t  = ((t1/w1sq) + (t2/w2sq)) / invwsq
;        ;; 3) Average detector counts
;        xd1 = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
;        xd2 = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2] * mon_scale
;        wd1sq = ((*(*sumPtr).detErrsPtr)[*,ibeg1:iend1])^2
;        wd2sq = ((*(*curPtr).detErrsPtr)[*,ibeg2:iend2])^2 * (mon_scale)^2
;        invwdsq = (1.0/wd1sq) + (1.0/wd2sq) 
;        xd = ((xd1/wd1sq) + (xd2/wd2sq)) / invwdsq
;        wd = sqrt(1/invwdsq)

        iend2 = iend2 + 1
        iend1 = iend1 + 1
    endelse

    ;; include the om,t,x,w,xd,wd calculated using either interleaving or
    ;; averaging into the sum buffers
    if (ibeg1 eq 0) then begin
        omega = om
        monitor = mn
        temperature = temp
        times = t
        totCnts = x
;        totErrs = w
        detCnts = xd
;        detErrs = wd
    endif else begin
        ;; sum buffers already contain data from Scenario 1
        omega = [omega,om]
        monitor = [monitor,mn]
        temperature = [temperature,temp]
        times = [times,t]
        totCnts = [totCnts,x]
;        totErrs = [totErrs,w]
        if ((iend1 eq ibeg1) and (n_elements(om) eq 1)) then begin  
            ;; ie only a one data point overlap ==> xd,wd are 1D not 2D!
            sz = size(detCnts)
            nw = sz[1]
            ns = (sz[0] eq 1)? 1 : sz[2]
            tmpdetCnts = fltarr(nw,ns+1)
            tmpdetErrs = fltarr(nw,ns+1)
            tmpdetCnts[*,0:ns-1] = detCnts
;            tmpdetErrs[*,0:ns-1] = detErrs
            tmpdetCnts[*,ns] = xd
;            tmpdetErrs[*,ns] = wd
            detCnts = tmpdetCnts
;            detErrs = tmpdetErrs
        endif else begin
            sz1 = size(detCnts)
            sz2 = size(xd)
            nw = sz1[1]
            ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
            ns = (sz2[0] eq 1)? 1 : sz2[2]
            ns = ns + ns1
            tmpdetCnts = fltarr(nw,ns)
;            tmpdetErrs = fltarr(nw,ns)
            tmpdetCnts[*,0:ns1-1] = detCnts
;            tmpdetErrs[*,0:ns1-1] = detErrs
            tmpdetCnts[*,ns1:ns-1] = xd
;            tmpdetErrs[*,ns1:ns-1] = wd
            detCnts = tmpdetCnts
;            detErrs = tmpdetErrs
        endelse
    endelse
 

    ;----------------------------------------------------------------------
    ;; Scenario 3 - beyond the overlap region - include any remaining data
    ;; Only ds1 OR ds2 could still contain unaccounted data; NOT
    ;; BOTH. However, both could be all used up already
    scenario3:
;    if (((iend1 eq 0) and (iend1 lt size1-1)) or (iend1 lt size1-1))then begin ; pick up the remaining data in ds1
    if ((iend1 eq 0) or (iend1 le size1-1)) then begin ; pick up the remaining data in ds1
        res = where(omega1[iend1] le omega, count)
        ibeg1 = (count lt 1)? iend1 : iend1 + 1
        iend1 = size1 - 1
        omega = [omega,omega1[ibeg1:iend1]]
        monitor = [monitor,monitor1[ibeg1:iend1]]
        temperature = [temperature,temp1[ibeg1:iend1]]
        times = [times,(*(*sumPtr).timePtr)[ibeg1:iend1]]
        totCnts = [totCnts,(*(*sumPtr).totCntsPtr)[ibeg1:iend1]]
;        totErrs = [totErrs,(*(*sumPtr).totErrsPtr)[ibeg1:iend1]]
        sz1 = size(detCnts)
        sz2 = size((*(*sumPtr).detCntsPtr)[*,ibeg1:iend1])
        nw = sz1[1]
        ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
        ns = (sz2[0] eq 1)? 1 : sz2[2]
        ns = ns + ns1
        tmpdetCnts = fltarr(nw,ns)
;        tmpdetErrs = fltarr(nw,ns)
        tmpdetCnts[*,0:ns1-1] = detCnts
;        tmpdetErrs[*,0:ns1-1] = detErrs
        tmpdetCnts[*,ns1:ns-1] = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1] 
;        tmpdetErrs[*,ns1:ns-1] = (*(*sumPtr).detErrsPtr)[*,ibeg1:iend1]
        detCnts = tmpdetCnts
;        detErrs = tmpdetErrs
        iend1 = iend1 + 1
    endif else if ((iend2 eq 0) or (iend2 le size2-1)) then begin ; pick up the remaining data in ds2
        res = where(omega2[iend2] le omega, count)
        ibeg2 = (count lt 1)? iend2 : iend2 + 1
        iend2 = size2 - 1
        omega = [omega,omega2[ibeg2:iend2]]
        monitor = [monitor,monitor2[ibeg2:iend2]]
        temperature = [temperature,temp2[ibeg2:iend2]]
        times = [times,(*(*curPtr).timePtr)[ibeg2:iend2]] 
        totCnts = [totCnts,(*(*curPtr).totCntsPtr)[ibeg2:iend2]]
;        totErrs = [totErrs,(*(*curPtr).totErrsPtr)[ibeg2:iend2]*mon_scale]
        sz1 = size(detCnts)
        sz2 = size((*(*curPtr).detCntsPtr)[*,ibeg2:iend2])
        nw = sz1[1]
        ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
        ns = (sz2[0] eq 1)? 1 : sz2[2]
        ns = ns + ns1
        tmpdetCnts = fltarr(nw,ns)
;        tmpdetErrs = fltarr(nw,ns)
        tmpdetCnts[*,0:ns1-1] = detCnts
;        tmpdetErrs[*,0:ns1-1] = detErrs
        tmpdetCnts[*,ns1:ns-1] = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2]
;        tmpdetErrs[*,ns1:ns-1] = (*(*curPtr).detErrsPtr)[*,ibeg2:iend2]*mon_scale
        detCnts = tmpdetCnts
;        detErrs = tmpdetErrs
        iend2 = iend2 + 1
    endif 

    ;----------------------------------------------------------
    ;; Phew! Now, should have reached the end of both datasets
    ;; If not, then there is something wrong so inform user but do nothing
    if ( (iend1 le size1-1) or (iend2 le size2-1) ) then begin
        message = ['Processing data from: ', $
                   fnames[j], $
                   'It apeears not all the data was accounted for!']
        void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)
    endif


    ;;------------------------------------------------------------
    ;; replace sum data with calculations
    (*(*sumPtr).energyPtr) = omega
    (*(*sumPtr).timePtr) = times
    (*(*sumPtr).tempPtr) = temperature
    (*(*sumPtr).totCntsPtr) = totCnts
;    (*(*sumPtr).totErrsPtr) = totErrs
    (*(*sumPtr).detCntsPtr) = detCnts
;    (*(*sumPtr).detErrsPtr) = detErrs
    (*(*sumPtr).monPtr) = monitor
    (*sumPtr).nspec = n_elements(omega)
  
    ;; update historyPtr
    (*(*sumPtr).historyPtr) = [(*(*sumPtr).historyPtr),(*(*curPtr).historyPtr)[1]]

    ;; update certain quantities before the next loop
    omega1 = (*(*sumPtr).energyPtr)
    temp1 = (*(*sumPtr).tempPtr)
    min1 = min(omega1, max=max1)
endfor

; normalize spectra to a constant monitor
ndet = (size((*(*sumPtr).detCntsPtr)))[1]
factor = monConstant / (*(*sumPtr).monPtr)
(*(*sumPtr).totErrsPtr) = sqrt((*(*sumPtr).totCntsPtr)) * factor
(*(*sumPtr).totCntsPtr) = (*(*sumPtr).totCntsPtr) * factor
(*(*sumPtr).detErrsPtr) = sqrt((*(*sumPtr).detCntsPtr)) * ((fltarr(ndet)+1)#factor)
(*(*sumPtr).detCntsPtr) = (*(*sumPtr).detCntsPtr) * ((fltarr(ndet)+1)#factor)
(*(*sumPtr).timePtr) = (*(*sumPtr).timePtr) * factor

; Assign sum to the appropriate destination
*dataPtr = [*dataPtr,sumPtr]  
switch dataType of 
    'samp': begin               ; add the sumPtr to sample dataset
        (*sPtr).sampIndex = nd
        (*sPtr).sampStatus = 1
        break
    end
    'bkgd': begin               ; add the sumPtr to bkgd dataset
        (*sPtr).bkgdIndex = nd
        (*sPtr).bkgdStatus = 1
        break
    end 
    'fast': begin               ; add the sumPtr to fast bkgd dataset
        (*sPtr).fastIndex = nd
        (*sPtr).fastStatus = 1
        break
    end 
endswitch 

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_ApplySettings
;
; PURPOSE:
;   Whenever settings for treatment of multiple files are modified
;   from the settings dialog, this procedure is called to apply them.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   old_eunit [in] - Flag indicating the existing energy units
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_ApplySettings, sPtr, old_eunit, err=err, emsg=emsg
err=0  & emsg=''

; if energy units have changed then apply it
new_eunit = (*(*sPtr).mfoPtr)[3]
if (new_eunit ne old_eunit) then begin
    ;; if new_eunit==0 then we are converting wavenumber-->meV
    ;; else converting meV-->wavenumber
    conv_factor = (new_eunit eq 0)? 1.0/((*sPtr).mev2wnos) : (*sPtr).mev2wnos 
    
    ;; convert energy units of sample data
    if ((*sPtr).sampCount gt 0) then begin
        for i=0, n_elements((*(*sPtr).sDataPtr))-1 do begin
            (*(*(*(*sPtr).sDataPtr)[i]).energyPtr) = (*(*(*(*sPtr).sDataPtr)[i]).energyPtr)*conv_factor
        endfor
    endif

    ;; convert energy units of bkgd data
    if ((*sPtr).bkgdCount gt 0) then begin
        for i=0, n_elements((*(*sPtr).bDataPtr))-1 do begin
            (*(*(*(*sPtr).bDataPtr)[i]).energyPtr) = (*(*(*(*sPtr).bDataPtr)[i]).energyPtr)*conv_factor
        endfor
    endif

    ;; convert energy units of fast bkgd data
    if ((*sPtr).fastCount gt 0) then begin
        for i=0, n_elements((*(*sPtr).fDataPtr))-1 do begin
            (*(*(*(*sPtr).fDataPtr)[i]).energyPtr) = (*(*(*(*sPtr).fDataPtr)[i]).energyPtr)*conv_factor
        endfor
        ;; force a fast bkgd fit to be performed
        ;(*sPtr).fitPresent = 0
        fans_Fastfit, sPtr=sPtr
    endif

endif

; Apply other required setting changes

; Deal with Sample data
if ((*sPtr).sampCount gt 1) then begin

    switch (*(*sPtr).mfoPtr)[0] of
        
        ;; data to be summed
        0: begin
            if ((*sPtr).sampStatus eq 1) then begin ; summed data exists
                (*sPtr).sampIndex = (*sPtr).sampCount ; - just point to it
                break
            endif 
            fans_sumData, sPtr, 'samp', err=err, emsg=emsg
            break
        end
        
        ;; data to be merged
        1: begin
            if ((*sPtr).sampStatus eq 2) then begin ; merged data exists
                (*sPtr).sampIndex = (*sPtr).sampCount ; - just point to it
                break
            endif 
            fans_apdData, sPtr, 'samp', err=err, emsg=emsg
            break
        end
        
        ;; data to be treated separately
        2: begin
            (*sPtr).sampIndex = (*sPtr).sampSepIndex
            break
        end
        
    endswitch 
    
endif 

; Deal with Bkgd data
if ((*sPtr).bkgdCount gt 1) then begin

    switch (*(*sPtr).mfoPtr)[1] of
        
        ;; data to be summed
        0: begin
            if ((*sPtr).bkgdStatus eq 1) then begin ; summed data exists
                (*sPtr).bkgdIndex = (*sPtr).bkgdCount ; - just point to it
                break
            endif 
            fans_sumData, sPtr, 'bkgd', err=err, emsg=emsg
            break
        end
        
        ;; data to be merged
        1: begin
            if ((*sPtr).bkgdStatus eq 2) then begin ; merged data exists
                (*sPtr).bkgdIndex = (*sPtr).bkgdCount ; - just point to it
                break
            endif 
            fans_apdData, sPtr, 'bkgd', err=err, emsg=emsg
            break
        end
        
        ;; data to be treated separately
        2: begin
            (*sPtr).bkgdIndex = (*sPtr).bkgdSepIndex
            break
        end
        
    endswitch 
    
endif 

; Deal with fast bkgd data
if ((*sPtr).fastCount gt 1) then begin

    switch (*(*sPtr).mfoPtr)[2] of
        
        ;; data to be summed
        0: begin
            if ((*sPtr).fastStatus eq 1) then begin ; merged data exists
                (*sPtr).fastIndex = (*sPtr).fastCount ; - just point to it
                break
            endif 
            fans_sumData, sPtr, 'fast', err=err, emsg=emsg
            break
        end
        
        ;; data to be merged
        1: begin
            if ((*sPtr).fastStatus eq 2) then begin ; merged data exists
                (*sPtr).fastIndex = (*sPtr).fastCount ; - just point to it
                break
            endif 
            fans_apdData, sPtr, 'fast', err=err, emsg=emsg
            break
        end
        
        ;; data to be treated separately
        2: begin
            (*sPtr).fastIndex = (*sPtr).fastSepIndex
            break
        end
        
    endswitch 
    
endif 


;; reset output calculation flag
(*sPtr).doOutpCalc = 1

;; Update plot and/or slider using appropriate data
widget_control, (*sPtr).sViewID, get_uvalue=sViewOn
widget_control, (*sPtr).bViewID, get_uvalue=bViewOn
widget_control, (*sPtr).fViewID, get_uvalue=fViewOn
widget_control, (*sPtr).oViewID, get_uvalue=oViewOn
case 1 of
    sViewOn: begin
        if ((*sPtr).sampCount lt 1) then return
        index = (*sPtr).sampIndex
        dataPtr = (*(*sPtr).sDataPtr)[index]
        updatePlotSlider, sPtr,'samp',dataPtr,err=err,emsg=emsg
    end

    bViewOn: begin
        if ((*sPtr).bkgdCount lt 1)  then return
        index = (*sPtr).bkgdIndex
        dataPtr = (*(*sPtr).bDataPtr)[index]
        updatePlotSlider, sPtr,'bkgd',dataPtr,err=err,emsg=emsg
    end

    fViewOn: begin
        if ((*sPtr).fastCount lt 1)  then return
        fans_plotData, sPtr,'fast',err=err,emsg=emsg ; can't view det cuts
    end 

    oViewOn: begin
        if ((*sPtr).sampCount lt 1)  then return
        fans_outpCalc, sPtr, err=err, emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif
        fans_plotData, sPtr,'outp',err=err,emsg=emsg ; can't view det cuts
    end 

endcase
        
end 
;-------------------------------------------------------------------------------



;===============================================================================
; fans_ApplySettings
;
; PURPOSE:
;   Event handler. Deal with changes in ouput options (mask, bkgd sub
;   and fast bkgd sub settings).
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_outpOpts, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_outpOpts: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

widget_control, event.top, get_uvalue=sPtr

; store the state of the non-exclusive button in its uvalue
widget_control, event.id, set_uvalue=event.select

; reset status of output calculation flag
(*sPtr).doOutpCalc = 1

;; if the display option is 'Output', then update output plot
widget_control, (*sPtr).oViewID, get_uvalue=on
if (on) then begin
    ;; Calculate output data
    fans_outpCalc, sPtr, err=err, emsg=emsg
    if (err ne 0) then begin
        printerror,err,emsg
        return
    endif
    fans_plotData, sPtr, 'outp', err=err,emsg=emsg
endif

end 
;-------------------------------------------------------------------------------



;===============================================================================
; fans_OutpCalc
;
; PURPOSE:
;   Calculate the final reduced data according to the current dialog
;   settings.
;   Does the following:
;   1) Remove masked detectors if requested
;   2) Remove fast background (using fit parameters)
;   3) Remove sample independent background (empty can)
;   4) For fast bkgd subtraction, it is first normalized to counts
;      per minute and then scaled to the sample (or bkgd) scan.
;   5) When subtracting empty can bkgd, time or monitor normalization
;      can be performed. Again the bkgd is _scaled_to_ the sample.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_OutpCalc, sPtr, err=err, emsg=emsg
err=0  &  emsg=''


; Only proceed if the doOutpCalc flag is set and 
; sample data exist
if ( (*sPtr).doOutpCalc eq 0 or (*sPtr).sampCount lt 1) then begin
    if ((*sPtr).outpCount eq 0) then begin
        err=1
        emsg='Output Calculation not performed! - No output data available'
    endif
    return
endif

; delete any existing output data
if (ptr_valid((*sPtr).oDataPtr)) then $
  fans_cleanDataPtrs, sPtr, 'outp'

; get ptr to sample data
sampDataPtr = (*(*sPtr).sDataPtr)[(*sPtr).sampIndex] 

; are masked detectors to be removed
widget_control, (*sPtr).mskOnID, get_uvalue=mskon
if (mskon) then begin 
    goodDets = fans_maskDets(sPtr)
    if ( (ptr_valid((*sPtr).mskListPtr) eq 0) or (goodDets[0] eq -1)) then mskon=0
endif

; should a fast bkgd be subtracted
widget_control, (*sPtr).subFastID, get_uvalue=subfast
if (subfast) then begin
    ;; if masked dets have changed since last fast fit then warn user
    ;; and exit
    if (mskon and (*sPtr).updFast) then begin
        message = ['Masked detectors have altered since last fit to fast bkgd!', $
                   'Please redo fast fit before displaying output']
        void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)
        err=1
        emsg = 'In "fans_OutpCalc" - fast fit is not in sync with mask dets'
        return 
    endif      
    ;; make sure fit parameters exist
    if (not (*sPtr).fitPresent) then begin 
        message = ['Fast bkgd fit has not been performed!', $
                   'Please do that before proceeding.']
        void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/error)
        err = 2
        emsg = 'In "fans_OutpCalc" - No fast fit parameters present'
        return 
    endif 
endif


; should bkgd be subtracted
nspec1 = (*sampDataPtr).nspec
widget_control, (*sPtr).subBkgdID, get_uvalue=subbkgd
if (subbkgd and (*sPtr).bkgdCount ge 1) then begin
    if (subbkgd) then $
      bkgdDataPtr = (*(*sPtr).bDataPtr)[(*sPtr).bkgdIndex] ; get ptr bkgd data
    nspec2 = (*bkgdDataPtr).nspec
    if (nspec1 ne nspec2) then begin
        message = ['Sample and bkgd files are dissimilar!', $
                   'They have different energy scans (range)', $
                   'Please examine your current file selections.']
        void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/error)
        err = 3
        emsg = 'In "fans_OutpCalc" - Samp & bkgd have different energies!' 
        return
    endif
endif


; now calculate output
;
; 1) make copies of samp and bkgd data
outpPtr = fans_initDataPtr(sampDataPtr)
(*(*outpPtr).historyPtr) = ['DAVE - FANS: Session log. Initialised on '+systime(), $
                            '----------------------------------------------------------', $
                         ' ', $
                         (*(*outpPtr).historyPtr)]
if (subbkgd) then begin 
    bkgdPtr = fans_initDataPtr(bkgdDataPtr)
    (*(*outpPtr).historyPtr) = [(*(*outpPtr).historyPtr), $
                                ' ', $
                                '____________________________________________________', $
                                'Sample independent background subtraction made', $
                                (*(*bkgdPtr).historyPtr)] 
endif 

; 2) apply mask to samp and bkgd by recalculating counts at each energy
if (mskon) then begin
    for i = 0,nspec1-1 do begin
        ;; in exp below, gooddets refer to the dets so need to sub 1
        ;; to get indices
        (*(*outpPtr).totCntsPtr)[i] = total((*(*outpPtr).detCntsPtr)[goodDets-1,i])
        (*(*outpPtr).totErrsPtr)[i] = sqrt(total(((*(*outpPtr).detErrsPtr)[goodDets-1,i])^2))
        if (subbkgd) then begin
            (*(*bkgdPtr).totCntsPtr)[i] = total((*(*bkgdPtr).detCntsPtr)[goodDets-1,i])
            (*(*outpPtr).totErrsPtr)[i] = sqrt(total(((*(*bkgdPtr).detErrsPtr)[goodDets-1,i])^2))
        endif
    endfor
endif

; 3) subtract fast bkgd from samp and bkgd - fast bkgd is scaled up before subtraction
if (subfast) then begin
    omega = (*(*outpPtr).energyPtr)
    ;fpar[5] = ((*(*sPtr).mfoPtr)[3] eq 0)? 1.0 : (*sPtr).mev2wnos ; convert factor b/n meV <--> wavenumbers 
    ;fastfit = (fans_fitfunc(omega,fpar))[*,0] ;; b/c fuction returns values + partial derivatives wrt fpar
    fpar = (*(*sPtr).fitParsPtr)
    parstr = strtrim(string(fpar,format='(G10.4)'),2)
    case (*sPtr).fitFunc of
        0: begin
            funcName = 'Arcsin'
            fastfit = (fans_Arcsin(omega,fpar))[*,0]
        end
        1: begin
            funcName = 'Linear'
            fastfit = (fans_Linear(omega,fpar))[*,0]
        end
        2: begin
            funcName = 'Quadratic'
            fastfit = (fans_Quad(omega,fpar))[*,0]
        end
        3: begin
            funcName = 'Cubic'
            fastfit = (fans_Cubic(omega,fpar))[*,0]
        end
    endcase
    if ((*sPtr).fastCount ge 1) then begin
        ;; get fast bkgd counting time - usually fast bkgd is
        ;; collected in fixed time mode so just use the fist value
        fastPtr = (*(*sPtr).fDataPtr)[(*sPtr).fastIndex]
        ftime = (*(*fastPtr).timePtr)[0]
        ;; update historyPtr
        (*(*outpPtr).historyPtr) = [(*(*outpPtr).historyPtr), $
                                    ' ', $
                                    '____________________________________________________', $
                                    (*(*fastPtr).historyPtr), $
                                    'Fast bkgd correction performed ... ', $
                                    'Model function: '+funcName, $
                                    '--- fit parameters used: '+parstr] 
    endif else begin
        ftime = 1.0             ; no fast data so assume fit is in per 1 minute counts
        ;fpar[4] = 3.35
        (*(*outpPtr).historyPtr) = [(*(*outpPtr).historyPtr), $
                                    ' ', $
                                    '____________________________________________________', $
                                    'Fast bkgd correction performed ... ', $
                                    'Model function: '+funcName, $
                                    '--- fit parameters used: '+parstr] 
    endelse 
    (*(*outpPtr).totCntsPtr) = (*(*outpPtr).totCntsPtr) - fastfit/ftime * (*(*outpPtr).timePtr) 
    if (subbkgd) then $
      (*(*bkgdPtr).totCntsPtr) = (*(*bkgdPtr).totCntsPtr) - fastfit/ftime * (*(*bkgdPtr).timePtr)
endif

; 4) Subtract bkgd if necessary - bkgd is (monitor) scaled to samp before subtraction
if (subbkgd) then begin
    smon = (*outpPtr).monitor * (*outpPtr).prefactor
    bmon = (*bkgdPtr).monitor * (*bkgdPtr).prefactor
    scalefactor = float(smon) / float(bmon)
    bkgdCnts = (*(*bkgdPtr).totCntsPtr) * scalefactor
    bkgdCErr = (*(*bkgdPtr).totErrsPtr) * scalefactor
    (*(*outpPtr).totCntsPtr) = (*(*outpPtr).totCntsPtr) - bkgdCnts
    (*(*outpPtr).totErrsPtr) = sqrt((*(*outpPtr).totErrsPtr)^2 + bkgdCerr^2)
endif

; 5) Move results to sPtr
tmparr = ptrarr(1)
tmparr[0] = outpPtr
(*sPtr).outpCount = 1
(*sPtr).outpIndex = 0
(*sPtr).oDataPtr = ptr_new(tmparr,/no_copy)
(*sPtr).doOutpCalc = 0

; remove any temperal heap data that was created
cleanDataPtr, bkgdPtr

;; update historyPtr with detector info
mTxt = strtrim((*outpPtr).monitor * (*outpPtr).prefactor,2)
text = ['Monitor counts = '+mTxt, $
        'Number of active detectors = '+strtrim((*outpPtr).ndets,2), $
        '--- the first two are monitors']
if (mskon) then begin
    widget_control, (*sPtr).mskListID, get_value=lstring
    text = [text, $
            'Number of masked detectors = '+strtrim(n_elements(*(*sPtr).mskListPtr),2), $
            'Masked detector list = '+lstring]
endif 
(*(*outpPtr).historyPtr) = [(*(*outpPtr).historyPtr), $
                         ' ', $
                         text]

; no errors
err=0  &  emsg=''

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_2Dave
;
; PURPOSE:
;   Save reduced data (integrated counts vs energy transfer) to a DAVE format file
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_2Dave, sPtr, err=err, emsg=emsg, no_output=no_output, nameTag=nameTag
err=0  &  emsg=''

; nothing to do if no sample data present
if ((*sPtr).sampCount lt 1) then begin
    err=1
    emsg='No sample data available - cannot determine output!'
    return
endif 

; Is output data up to date? - if not, evaluate it
if ( (*sPtr).doOutpCalc) then begin
    fans_OutpCalc, sPtr, err=err, emsg=emsg
    if (err ne 0) then return
endif

; copy data into the davePtr
; NOTE: the davePtr is passed down from DAVE.pro! This is not a good
; thing b/c there is no way of knowing what type of data it contains
; sice each instrument has a different specificPtr. Problems arise
; when the specificPtr contains ptrs, it will not be possible to
; clean the ptrs before rewriting the specificPtr. Thus there are
; memory leaks! 
;
; ########### Need a better solution!!!!!!!!!!!
;
davePtr = (*sPtr).davePtr
outpPtr = (*(*sPtr).oDataPtr)[(*sPtr).outpIndex]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 1) deal with the davePtr.commonStr

; first, free-up any pointers within commonStr
ptr_free,(*(*davePtr).dataStrPtr).commonStr.histPtr
ptr_free,(*(*davePtr).dataStrPtr).commonStr.treatmentPtr


qty = (*(*outpPtr).totCntsPtr)
qerr = (*(*outpPtr).totErrsPtr)
;Data is already normalized to monitor of first dataset 
;If using a custom monitor to normalize then need to rescale
monConstant = (*outpPtr).monitor * (*outpPtr).prefactor
wID = widget_info((*sPtr).tlbID,find_by_uname='USEMON')
if (widget_info(wID,/valid)) then begin
   widget_control, wID, get_uvalue=isSet
   if (isSet) then begin
      wID = widget_info((*sPtr).tlbID,find_by_uname='MONVAL')
      if (widget_info(wID,/valid)) then begin
         widget_control, wID, get_value=monValue
         if (monValue gt 0.0) then begin
             normFac = float(monValue)/monConstant
             qerr = qerr * normFac
             qty = qty * normFac
             monConstant=monValue
         endif
      endif
   endif
endif

;; title
switch (*outpPtr).base of
    'TIME': begin
        ftime = fix((*(*outpPtr).timePtr)[0] * 60)
        histlabel = 'Counts per '+strtrim(string(ftime),2)+' secs'
        break
    end
    
    'NEUT':
    else: begin
        monCnts = (*outpPtr).monitor * (*outpPtr).prefactor
        histlabel = 'Counts per '+strtrim(string(monConstant),2)+' mon'
        break
    end
endswitch
        
; fill it up
;dummy=strarr(1)

(*(*davePtr).dataStrPtr).commonStr.instrument = 'FANS'
(*(*davePtr).dataStrPtr).commonStr.histPtr = $
  ptr_new({qty:qty, $ ; *** may need to redefine as 2D
           err:qerr, $ ; *** may need to redefine as 2D
           x:(*(*outpPtr).energyPtr), $
           y:0.0 $              ; *** not clear what to do here - FANS data is 1D!
          })
(*(*davePtr).dataStrPtr).commonStr.xlabel = 'Energy Transfer'
(*(*davePtr).dataStrPtr).commonStr.xunits = ((*(*sPtr).mfoPtr)[3] eq 0)? 'meV' : '!ncm!u-1!n'
(*(*davePtr).dataStrPtr).commonStr.xtype = 'points'
(*(*davePtr).dataStrPtr).commonStr.ylabel = ''
(*(*davePtr).dataStrPtr).commonStr.yunits = ''
(*(*davePtr).dataStrPtr).commonStr.ytype = 'points'
monCnts = (*outpPtr).monitor * (*outpPtr).prefactor

(*(*davePtr).dataStrPtr).commonStr.histlabel = histlabel
(*(*davePtr).dataStrPtr).commonStr.histunits = ''
(*(*davePtr).dataStrPtr).commonStr.treatmentPtr = ptr_new((*(*outpPtr).historyPtr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 2) deal with davePtr.specificPtr - instrument specific attributes

; brutally clean specificPtr - have to use heap_free in order to
;                              recursively deal with inner ptrs
heap_free,(*(*davePtr).dataStrPtr).specificPtr

specificStr = { $
            filename:(*outpPtr).filename, $
            date:(*outpPtr).date, $
            scan:(*outpPtr).scan, $
            monitor:(*outpPtr).monitor, $
            prefactor:(*outpPtr).prefactor, $
            base:(*outpPtr).base, $
            type:(*outpPtr).type, $
            title:(*outpPtr).title, $
            collimation:(*outpPtr).collimation, $
            mosaic:(*outpPtr).mosaic, $
            orientation:(*outpPtr).orientation, $
            latticeA:(*outpPtr).latticeA, $
            latticeB:(*outpPtr).latticeB, $
            latticeC:(*outpPtr).latticeC, $
            latticeAl:(*outpPtr).latticeAl, $
            latticeBe:(*outpPtr).latticeBe, $
            latticeGa:(*outpPtr).latticeGa, $
            eStart:(*outpPtr).eStart, $
            eFinal:(*outpPtr).eFinal, $
            eDelta:(*outpPtr).eDelta, $
            monDSpace:(*outpPtr).monDSpace, $
            anaDSpace:(*outpPtr).anaDSpace, $
            tStart:(*outpPtr).tStart, $
            tInc:(*outpPtr).tInc, $
            qhCenter:(*outpPtr).qhCenter, $
            qkCenter:(*outpPtr).qkCenter, $
            qlCenter:(*outpPtr).qlCenter, $
            qhDelta:(*outpPtr).qhDelta, $
            qkDelta:(*outpPtr).qkDelta, $
            qlDelta:(*outpPtr).qlDelta, $
            hfield:(*outpPtr).hfield, $
            tflag:(*outpPtr).tflag, $
            ndets:(*outpPtr).ndets, $
            nspec:(*outpPtr).nspec $
           }

(*(*davePtr).dataStrPtr).specificPtr = ptr_new(specificStr)

; determine a sensible name tag
index = ((*sPtr).sampIndex eq (*sPtr).sampCount)? 0 : (*sPtr).sampIndex
nameTag = (*(*sPtr).outpListPtr)[index]

; exit now if simply updating the davePtr
if (keyword_set(no_output)) then return

; now save to disk
;index = (*sPtr).sampIndex
;if ( index eq (*sPtr).sampCount) then $
;  index = 0
;
def_ofile = nameTag+'.dave'

repeat begin
    okay = 1
    ofile = dialog_pickfile(dialog_parent=(*sPtr).tlbID, $
                            title="Select output filename", $
                            path=(*sPtr).workDir, $
                            file=def_ofile, $
                            get_path=newPath, $
                            filter='*.dave' $
                           )

    if (strtrim(ofile,2) eq '') then return ; dialog_pickfile was cancelled so do nothing further

    if ( strmid(ofile,strlen(ofile)-1,1) eq !delimiter) then $ ; ie only a directory was return...
      ofile = ofile+def_ofile   ; then append def_ofile to it
    
    ;; if file exists then make sure user wishes to overide it
    void = findfile(ofile,count=cnt)
    if (cnt) then begin
        message = [ofile+' exists','Do you want to overwrite it?']
        ans = dialog_message(message,dialog_parent=(*sPtr).tlbid,/question)
        okay = (strtrim(ans,2) eq 'Yes')? 1 : 0 
    endif
endrep until okay

if (strlen(newPath) gt 1) then $
  (*sPtr).workDir = newPath     ; update current working directory

; write to disk and inform user of ouput destination
save, davePtr, filename=ofile
message=['Output saved in:',ofile]
void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_2DaveTool
;
; PURPOSE:
;   Send reduced data to the data manager folder of the DAVE Main Tool
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_2DaveTool, sPtr, err=err, emsg=emsg
compile_opt idl2
err=0  &  emsg=''

; Make sure output buffer is updated
fans_2Dave, sPtr, err=err, emsg=emsg, nameTag=nameTag, /no_output
if (err ne 0) then return

if ((n_elements(nameTag) le 0) || (strtrim(nameTag,2) eq '')) then nameTag='Untitled'

; Send data to DAVE Data Manager Folder
((*sPtr).DAVETool)->AddDavePtrToDataManager, (*sPtr).davePtr, nameTag


end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_2DDave
;
; PURPOSE:
;   Save reduced data (individual detector counts) to a DAVE
;   format file. The save file is a 2D file
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_2DDave, sPtr, err=err, emsg=emsg
err=0  &  emsg=''

; nothing to do if no sample data present
if ((*sPtr).sampCount lt 1) then begin
    err=1
    emsg='No sample data available - cannot determine output!'
    return
endif 

; Is output data up to date? - if not, evaluate it
if ( (*sPtr).doOutpCalc) then begin
    fans_OutpCalc, sPtr, err=err, emsg=emsg
    if (err ne 0) then return
endif

; copy data into the davePtr
; NOTE: the davePtr is passed down from DAVE.pro! This is not a good
; thing b/c there is no way of knowing what type of data it contains
; sice each instrument has a different specificPtr. Problems arise
; when the specificPtr contains ptrs, it will not be possible to
; clean the ptrs before rewriting the specificPtr. Thus there are
; memory leaks! 
;
; ########### Need a better solution!!!!!!!!!!!
;
davePtr = (*sPtr).davePtr
outpPtr = (*(*sPtr).oDataPtr)[(*sPtr).outpIndex]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 1) deal with the davePtr.commonStr

; first, free-up any pointers within commonStr
ptr_free,(*(*davePtr).dataStrPtr).commonStr.histPtr
ptr_free,(*(*davePtr).dataStrPtr).commonStr.treatmentPtr

;; title
switch (*outpPtr).base of
    'TIME': begin
        ftime = fix((*(*outpPtr).timePtr)[0] * 60)
        histlabel = 'Detector Cnts per '+strtrim(string(ftime),2)+' secs'
        break
    end
    
    'NEUT':
    else: begin
        monCnts = (*outpPtr).monitor * (*outpPtr).prefactor
        histlabel = 'Detector Cnts per '+strtrim(string(monCnts),2)+' mon'
        break
    end
endswitch     

; fill it up
ndets = (*outpPtr).ndets
(*(*davePtr).dataStrPtr).commonStr.instrument = 'FANS'
(*(*davePtr).dataStrPtr).commonStr.histPtr = $
  ptr_new({qty:transpose((*(*outpPtr).detCntsPtr)[2:ndets-1,*]), $ 
           err:transpose((*(*outpPtr).detErrsPtr)[2:ndets-1,*]), $ 
           x:(*(*outpPtr).energyPtr), $
           y:indgen(ndets-2)+3 $ 
          })
(*(*davePtr).dataStrPtr).commonStr.xlabel = 'Energy Transfer'
(*(*davePtr).dataStrPtr).commonStr.xunits = ((*(*sPtr).mfoPtr)[3] eq 0)? 'meV' : '!ncm!u-1!n'
(*(*davePtr).dataStrPtr).commonStr.xtype = 'points'
(*(*davePtr).dataStrPtr).commonStr.ylabel = 'Detector Nos'
(*(*davePtr).dataStrPtr).commonStr.yunits = ''
(*(*davePtr).dataStrPtr).commonStr.ytype = 'points'
monCnts = (*outpPtr).monitor * (*outpPtr).prefactor
(*(*davePtr).dataStrPtr).commonStr.histlabel = histlabel
(*(*davePtr).dataStrPtr).commonStr.histunits = ''
(*(*davePtr).dataStrPtr).commonStr.treatmentPtr = ptr_new((*(*outpPtr).historyPtr))
(*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr) = $
  [(*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr), $
   '=================================================', $
   'The data in the 2D FANS files have not been ', $
   'corrected for instrument fast background. ', $
   'The data are essentially the raw detector ', $
   'counts for the sample file(s) listed above. ', $
   'However, the following are performed: ', $
   '    * if present, subtraction of container scatt.', $
   '    * proper monitor norm. if more than one file.' ]  

;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 2) deal with davePtr.specificPtr - instrument specific attributes

; brutally clean specificPtr - have to use heap_free in order to
;                              recursively deal with inner ptrs
heap_free,(*(*davePtr).dataStrPtr).specificPtr

specificStr = { $
            filename:(*outpPtr).filename, $
            date:(*outpPtr).date, $
            scan:(*outpPtr).scan, $
            monitor:(*outpPtr).monitor, $
            prefactor:(*outpPtr).prefactor, $
            base:(*outpPtr).base, $
            type:(*outpPtr).type, $
            title:(*outpPtr).title, $
            collimation:(*outpPtr).collimation, $
            mosaic:(*outpPtr).mosaic, $
            orientation:(*outpPtr).orientation, $
            latticeA:(*outpPtr).latticeA, $
            latticeB:(*outpPtr).latticeB, $
            latticeC:(*outpPtr).latticeC, $
            latticeAl:(*outpPtr).latticeAl, $
            latticeBe:(*outpPtr).latticeBe, $
            latticeGa:(*outpPtr).latticeGa, $
            eStart:(*outpPtr).eStart, $
            eFinal:(*outpPtr).eFinal, $
            eDelta:(*outpPtr).eDelta, $
            monDSpace:(*outpPtr).monDSpace, $
            anaDSpace:(*outpPtr).anaDSpace, $
            tStart:(*outpPtr).tStart, $
            tInc:(*outpPtr).tInc, $
            qhCenter:(*outpPtr).qhCenter, $
            qkCenter:(*outpPtr).qkCenter, $
            qlCenter:(*outpPtr).qlCenter, $
            qhDelta:(*outpPtr).qhDelta, $
            qkDelta:(*outpPtr).qkDelta, $
            qlDelta:(*outpPtr).qlDelta, $
            hfield:(*outpPtr).hfield, $
            tflag:(*outpPtr).tflag, $
            ndets:(*outpPtr).ndets, $
            nspec:(*outpPtr).nspec $
           }

(*(*davePtr).dataStrPtr).specificPtr = ptr_new(specificStr)


; now save to disk
index = (*sPtr).sampIndex
if ( index eq (*sPtr).sampCount) then $
  index = 0

def_ofile = (*(*sPtr).outpListPtr)[index]+'_2D.dave'

repeat begin
    okay = 1
    ofile = dialog_pickfile(dialog_parent=(*sPtr).tlbID, $
                            title="Select output filename", $
                            path=(*sPtr).workDir, $
                            file=def_ofile, $
                            get_path=newPath, $
                            filter='*.dave' $
                           )

    if (strtrim(ofile,2) eq '') then return ; dialog_pickfile was cancelled so do nothing further

    if ( strmid(ofile,strlen(ofile)-1,1) eq !delimiter) then $ ; ie only a directory was return...
      ofile = ofile+def_ofile   ; then append def_ofile to it

    ;; if file exists then make sure user wishes to overide it
    void = findfile(ofile,count=cnt)
    if (cnt) then begin
        message = [ofile+' exists','Do you want to overwrite it?']
        ans = dialog_message(message,dialog_parent=(*sPtr).tlbid,/question)
        okay = (strtrim(ans,2) eq 'Yes')? 1 : 0 
    endif
endrep until okay

if (strlen(newPath) gt 1) then $
  (*sPtr).workDir = newPath     ; update current working directory

; write to disk and inform user of ouput destination
save, davePtr, filename=ofile
message=['Output saved in:',ofile]
void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_2Ascii
;
; PURPOSE:
;   Save reduced data to an ASCII file
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_2Ascii, sPtr, err=err, emsg=emsg
err=0  &  emsg=''

; nothing to do if no sample data present
if ((*sPtr).sampCount lt 1) then begin
    err=1
    emsg='No sample data available - cannot determine output!'
    return
endif 

; Is output data up to date? - if not, evaluate it
if ( (*sPtr).doOutpCalc) then begin
    fans_OutpCalc, sPtr, err=err, emsg=emsg
    if (err ne 0) then return
endif

; get the data for output
outpPtr = (*(*sPtr).oDataPtr)[(*sPtr).outpIndex]

; now save to disk
index = (*sPtr).sampIndex
if ( index eq (*sPtr).sampCount) then $
  index = 0

def_ofile = (*(*sPtr).outpListPtr)[index]+'.dat'

repeat begin
    okay = 1
    ofile = dialog_pickfile(dialog_parent=(*sPtr).tlbID, $
                            title="Select output filename", $
                            path=(*sPtr).workDir, $
                            file=def_ofile, $
                            get_path=newPath, $
                            filter='*.dat' $
                           )
    if (strtrim(ofile,2) eq '') then return ; dialog_pickfile was cancelled so do nothing further
    
    if ( strmid(ofile,strlen(ofile)-1,1) eq !delimiter) then $ ; ie only a directory was returned...
      ofile = ofile+def_ofile   ; then append def_ofile to it

    ;; if file exists then make sure user wishes to overide it
    void = findfile(ofile,count=cnt)
    if (cnt) then begin
        message = [ofile+' exists','Do you want to overwrite it?']
        ans = dialog_message(message,dialog_parent=(*sPtr).tlbid,/question)
        okay = (strtrim(ans,2) eq 'Yes')? 1 : 0 
    endif
endrep until okay

if (strlen(newPath) gt 1) then $
  (*sPtr).workDir = newPath     ; update current working directory
;(*sPtr).workDir = newPath

; open input file
openw, lun, ofile, /get_lun, error=err
if (err ne 0) then begin        ; if error, get the error msg and exit
    emsg = 'In "fans_readData" - '+!error_state.msg    
    return
endif

; write to disk and inform user of ouput destination
omega = (*(*outpPtr).energyPtr)
cnts = (*(*outpPtr).totCntsPtr)
cnts_err = (*(*outpPtr).totErrsPtr)
time = (*(*outpPtr).timePtr)
temp = (*(*outpPtr).tempPtr)

;Data is already normalized to monitor of first dataset 
;If using a custom monitor to normalize then need to rescale
monConstant = (*outpPtr).monitor * (*outpPtr).prefactor
wID = widget_info((*sPtr).tlbID,find_by_uname='USEMON')
if (widget_info(wID,/valid)) then begin
   widget_control, wID, get_uvalue=isSet
   if (isSet) then begin
      wID = widget_info((*sPtr).tlbID,find_by_uname='MONVAL')
      if (widget_info(wID,/valid)) then begin
         widget_control, wID, get_value=monValue
         if (monValue gt 0.0) then begin
             normFac = float(monValue)/monConstant
             cnts_Err = cnts_Err * normFac
             Cnts = Cnts * normFac
             monConstant=monValue
         endif
      endif
   endif
endif

nd = n_elements(omega)
for i = 0,nd-1 do printf, lun, format='(5(F11.4,2X))',omega[i],cnts[i],cnts_err[i],time[i],temp[i]

free_lun, lun, /force

message=['Output saved in:',ofile]
void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)

end


;===============================================================================
; fans_updMskList
;
; PURPOSE:
;   Maintains a list of detectors that are to be masked from the final
;   reduced data.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   detnos [in] - detector number whose state has changed masked<->unmasked
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_updMskList, sPtr, detnos, err=err,emsg=emsg
err=0 & emsg=''

; one input parameters necessary
if (n_params() ne 2) then begin
    err = 1
    emsg = 'In "fans_updMskList" - Invalid number of parameters'
    return
endif

; check input parameters are defined
if ( (n_elements(detnos) le 0)  or (ptr_valid(sPtr) eq 0) ) then begin
    err = 2
    emsg = 'In "fans_updMskList" - Input parameter(s) undefined'
    return
endif

; grap the current mask list
if  ptr_valid((*sPtr).mskListPtr) then list = (*(*sPtr).mskListPtr)
ptr_free,(*sPtr).mskListPtr     ; clean (empty) the current list

lsize = n_elements(list)
if (lsize gt 0) then begin      ; list exists
    result = where(list eq detnos)
    if (result[0] ge 0) then begin ; detnos is in the list already - remove it
        lsize = lsize - 1
        if (result[0] le lsize-1) then $
          for i = result[0], lsize-1 do list[i] = list[i+1]
    endif else begin            ; detnos not in list - add it
        lsize = lsize + 1
        list = [list,detnos]
    endelse
endif else begin                ; no list yet - create one
    lsize = 1
    list = [detnos]
endelse 

if (lsize ge (*(*(*sPtr).sDataPtr)[(*sPtr).sampIndex]).ndets-2) then begin ; removing all detectors!!!
    message = ['Attempting to mask all detectors.', $
               'No output can be generated!']
    void = dialog_message(message,/information,dialog_parent=(*sPtr).tlbID)
endif


; update state variable
lstring = ''
; mskOnID should be desensitized and off (ie deselected) by default unless there
; are actually detectors to be masked.
widget_control, (*sPtr).mskOnID, get_uvalue=mskon ; get current msk status (ie whether on/off)
widget_control, (*sPtr).mskOnID, sensitive=0 ; should be desensitized
if (mskon) then $               ; if on, switch it off! but remember it was on
  widget_control, (*sPtr).mskOnID, set_button=0, set_uvalue=0

if (lsize gt 0) then begin
    widget_control, (*sPtr).mskOnID, sensitive=1
    if (mskon) then $           ; if it was on previously, switch it back on
      widget_control, (*sPtr).mskOnID, set_button=1, set_uvalue=mskon
    newlist = list[0:lsize-1]
    newlist = newlist[sort(newlist)] ; sort in ascending order
    (*sPtr).mskListPtr = ptr_new(newlist)
    if (lsize eq 1) then begin
      lstring =   strtrim(string(newlist[0]),2)
    endif else begin
        beg = newlist[0]
        range = 0
        for i = 1,lsize-1 do begin
            if (newlist[i] eq newlist[i-1]+1) then begin
                range=1
                continue
            endif
            if (range) then begin
                lstring = lstring+strtrim(string(beg),2)+'-'+strtrim(string(newlist[i-1]),2)+' '
                beg = newlist[i]
                range = 0
            endif else begin
                lstring = lstring+strtrim(string(newlist[i-1]),2)+' '
                beg = newlist[i]
            endelse
        endfor
        if (range) then begin
            lstring = lstring+strtrim(string(beg),2)+'-'+strtrim(string(newlist[i-1]),2)+' '
        endif else begin
            lstring = lstring+strtrim(string(newlist[i-1]),2)+' '
        endelse
    endelse  
;    strarr = strtrim(string(newlist),1) 
;    for i = 0,lsize-1 do $
;      lstring = lstring + strarr[i] + ' '
endif 

; switch on output calculation flag
(*sPtr).doOutpCalc = 1

; indicates Fast bkgd fit has to be recalculated
(*sPtr).updFast = 1


; update list in mask widget
widget_control, (*sPtr).mskListID, set_value=lstring

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_plotMskList
;
; PURPOSE:
;   Mark all masked detectors with grey bars in the display window.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_plotMskList, sPtr, err=err, emsg=emsg
err=0 & emsg=''

; one input parameters necessary
if (n_params() ne 1) then begin
    err = 1
    emsg = 'In "fans_plotMskList" - Invalid number of parameters'
    return
endif

; check input parameters are defined
if (ptr_valid(sPtr) eq 0) then begin
    err = 2
    emsg = 'In "fans_plotMskList" - Input parameter undefined'
    return
endif

if ( ptr_valid((*sPtr).mskListPtr) eq 0) then return

list = (*(*sPtr).mskListPtr)

lsize = n_elements(list)
ymin = !y.crange[0]  &  ymax = !y.crange[1]
yvals = [ymin,ymin,ymax,ymax]
for i = 0, lsize-1 do begin
    xmin = list[i]-0.50  &  xmax=list[i]+0.50
    xvals = [xmin,xmax,xmax,xmin]
    polyfill,xvals,yvals
endfor


end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_plotClick
;
; PURPOSE:
;   Event handler for button events dispatched from the plot
;   window. When detector display mode is on (ie samp, bkgd views)
;   then deal with left-mouse clicks to mask/unmask dets else
;   zoom/reset display. Mark all masked detectors with grey bars in
;   the display window.
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_plotClick, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_plotClick: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control, event.top, get_uvalue=sPtr

; determine current view mode
widget_control, (*sPtr).sViewID, get_uvalue=svOn
widget_control, (*sPtr).bViewID, get_uvalue=bvOn
widget_control, (*sPtr).fViewID, get_uvalue=fvOn
widget_control, (*sPtr).oViewID, get_uvalue=ovOn

if svOn then dataType = 'samp'
if bvOn then dataType = 'bkgd'
if fvOn then dataType = 'fast'
if ovOn then dataType = 'outp'

if ( (dataType eq 'samp') or (dataType eq 'bkgd') ) then begin ; mask/unmask dets

    
    if (event.press eq 1) then begin ; left mb press
        (*sPtr).zp1 = [event.x,event.y] ; record mouse location
        (*sPtr).mouseType = event.press ; need to know afterwards that a left mb was pressed
        
        ;; draw a vertical line at the mouse location
        ymin = !y.crange[0]  &  ymax = !y.crange[1]
        xydat = convert_coord(event.x,event.y,/device,/to_data)
        plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
        (*sPtr).zp2 = [-1000.0,-1.0]   ; used to determine if motion event occurs b/c zp2 will be different
    endif 

    ;; if motion event and left mb was just pressed
    if ((event.type eq 2) and ((*sPtr).mouseType eq 1) ) then begin
        (*sPtr).zp2 = [event.x, event.y]
        zp1 = (*sPtr).zp1
        zp2 = (*sPtr).zp2
        ;; copy winPix to winVis and draw vertical lines at current mouse 
        ;; location and where left mb was initially pressed 
        device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix]
        ymin = !y.crange[0]  &  ymax = !y.crange[1]
        xydat = convert_coord(zp1[0],zp1[1],/device,/to_data)
        plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
        xydat = convert_coord(zp2[0],zp2[1],/device,/to_data)
        plots, [xydat[0],xydat[0]], [ymin,ymax], color=(*sPtr).colors.red, /data
    endif

    if (event.release eq 1) then begin ; left mb released

        lolim=2.5
        hilim=50.49
        if (ptr_valid((*sPtr).sDataPtr)) then begin
            hilim = (*(*(*sPtr).sDataPtr)[0]).ndets + 0.49
        endif

        if ( ((*sPtr).zp2)[0] ne -1000.0) then begin
            zp1 = (*sPtr).zp1   ; initial xvalue for left mouse press
            zp2 = (*sPtr).zp2   ; final mouse xvalue after motion event
            x1 = min([zp1[0],zp2[0]],max=x2)
            y = zp1[1]
            d1 = 0
            d2 = 0
            ;; if the x coord of mouse occurs outside the detector range, ignore it
            xydat = convert_coord(x1,y,/device,/to_data)
            d1 = round(xydat[0])
            xydat = convert_coord(x2,y,/device,/to_data)
            d2 = round(xydat[0])
            for detnos = d1,d2 do begin
                if ( (detnos lt lolim) or (detnos gt hilim) ) then continue
                
                ;; add/remove det from mask list if it isn't/is there followed by an
                ;; update to the mask widget
                fans_updMskList, sPtr,detnos
            endfor 
        endif else begin
            zp1 = (*sPtr).zp1   ; initial xvalue for left mouse press
            x1 = zp1[0]
            y = zp1[1]
            xydat = convert_coord(x1,y,/device,/to_data)
            detnos = round(xydat[0])
            ;; add/remove det from mask list if it isn't/is there followed by an
            ;; update to the mask widget
            if ( (detnos ge lolim) and (detnos le hilim) ) then fans_updMskList, sPtr,detnos
        endelse 
        ;; refresh plot window 
        fans_plotData, sPtr, dataType, err=err,emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif
        (*sPtr).zp2 = [0.0,0.0]
        (*sPtr).mouseType = 0   ; reset mouseType as prev press been dealt with
    endif 
    
endif else begin                ; zoom/reset plot

    case event.type of
        0: begin                ; button press
            (*sPtr).mouseType = event.press
            if ((*sPtr).mouseType eq 4) then begin ; right-hand mb so reset plot to full range
                fans_plotData, sPtr, dataType, err=err, emsg=emsg
                if (err ne 0) then begin
                    printerror,err,emsg
                    return
                endif
                (*sPtr).mouseType = 0 ; set mouse type to undefined value
                return
            endif
            if ((*sPtr).mouseType eq 1) then begin ; left-hand mb so record mouse location
                (*sPtr).zp1 = [event.x,event.y]
            endif
        end
        1: begin                ; button release
            if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
                zp1 = (*sPtr).zp1
                zp2 = (*sPtr).zp2
                x = min([zp1[0], zp2[0]])
                y = min([zp1[1], zp2[1]])
                w = abs(zp1[0] - zp2[0])
                h = abs(zp1[1] - zp2[1])
                x2 = x+w
                y2 = y+h
                lc = convert_coord(x, y, /device, /to_data)
                uc = convert_coord(x2, y2, /device, /to_data)
                fans_plotData, sPtr, dataType,xrange=[lc[0],uc[0]], yrange=[lc[1],uc[1]],err=err,emsg=emsg
                if (err ne 0) then begin
                    printerror,err,emsg
                    return
                endif
                if (err ne 0) then begin
                    printerror,err,emsg
                    return
                endif
                (*sPtr).zp2 = [0.0,0.0] ; reset to zero
            endif
            (*sPtr).mouseType = 0
        end
        2: begin                ; mouse motion
            if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
                (*sPtr).zp2 = [event.x, event.y]
                zp1 = (*sPtr).zp1
                zp2 = (*sPtr).zp2
                xc = [zp1[0], event.x, event.x, zp1[0], zp1[0]]
                yc = [zp1[1], zp1[1], event.y, event.y, zp1[1]]
                device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
                plots, xc, yc, color=(*sPtr).colors.red, /device
            endif
        end
        else:
    endcase    
endelse                         ; zoom/reset



end
;-------------------------------------------------------------------------------


;===============================================================================
; fans_setSliderValue
;
; PURPOSE:
;   Slider values are continuous. However, the energy values that the
;   slider represents only have certain values given by the energy
;   array. This routine ensures that the slider only sets/displays
;   valid energy values that are present in the energy array. This
;   routine must be called whenever the data being displayed changes.
;
; PARAMETERS:
;   wid [in] - widget id of slider
;
;   energyPtr [in] - pointer to energy transfer array
;
;   dvalue [in] - the energy value initially selected using the
;                 slider. This is used to locate the closest valid
;                 energy.
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_setSliderValue, wid, energyPtr, dvalue, err=err, emsg=emsg

err=0  &  emsg = ''

; one input parameters necessary
if (n_params() lt 2) then begin
    err = 1
    emsg = 'In "fans_setSliderValue" - Invalid number of parameters'
    return
endif

; check input parameters are defined
if ( (n_elements(wid) le 0)  or (ptr_valid(energyPtr) eq 0) ) then begin
    err = 2
    emsg = 'In "fans_setSliderValue" - Input parameter(s) undefined'
    return
endif

; if input parameter dvalue missing, use current slider value instead
if (n_elements(dvalue) le 0) then $
  widget_control, wid, get_value=dvalue ;dvalue = fslider_get_value(wid) 

; now locate the closest entry to dvalue in the energy array
index = locateIndex((*energyPtr),dvalue)

; update slider value with this 'proper' energy 
widget_control, wid, set_value=(*energyPtr)[index]

; save this index in the uvalue of the widget
widget_control, wid, set_uvalue=index

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_syncSlider
;
; PURPOSE:
;   Ensure that the energy range displayed in the slider matches the
;   currently selected dataset.
;
; PARAMETERS:
;   wid [in] - widget id of slider
;
;   energyPtr [in] - pointer to energy transfer array
;
;   reset [in] - flag. If set, the slider value is set to its minimum.
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;
pro fans_syncSlider, wid, energyPtr, reset, err=err, emsg=emsg

err=0  &  emsg = ''
; two input parameters necessary
if (n_params() ne 3) then begin
    err = 1
    emsg = 'In "syncSlider" - Invalid number of procedure parameters'
    return
endif

; check input parameters are defined
if ((n_elements(wid) le 0)or(ptr_valid(energyPtr) eq 0)or(n_elements(reset) eq 0)) then begin
    err = 2
    emsg = 'In "syncSlider" - Input parameter(s) undefined'
    return
endif

info = size(*energyPtr)
if (info[0] ne 1) then begin
    err = 3
    emsg = 'In "syncSlider" - Input parameter has wrong dimension'
    return
endif

dmin = min(*energyPtr, max=dmax)
if (reset gt 0) then begin
    ;; set slider range to be same as energy array. Initialise it to its
    ;; min value
    range = [dmin,dmin,dmax]
    widget_control, wid, set_value = range
    fans_setSliderValue, wid, energyPtr, dmin, err=err, emsg=emsg
endif else begin
    widget_control, wid, get_value=dvalue ;dvalue = fslider_get_value(wid)
    if (dvalue lt dmin) then dvalue = dmin
    if (dvalue gt dmax) then dvalue = dmax
    range = [dvalue,dmin,dmax]
    widget_control, wid, set_value = range
    fans_setSliderValue, wid, energyPtr, davlue, err=err,emsg=emsg
endelse

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_sliderEvent
;
; PURPOSE:
;   Event handler: for energy cut slider widget.
;
; PARAMETERS:
;   event [in] - event structure.
;
pro fans_sliderEvent, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_sliderEvent: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control,event.top,get_uvalue=sPtr

; determine which type of data is being displayed. Note that it only
; make sense to view energy slices for sample and bkgd data. Cuts for
; Fast bkgd and output data are not implemented

widget_control, (*sPtr).sViewID, get_uvalue=svOn
widget_control, (*sPtr).bViewID, get_uvalue=bvOn
;widget_control, (*sPtr).oViewID, get_uvalue=ovOn

if svOn then dataType = 'samp'
if bvOn then dataType = 'bkgd'
;if ovOn then dataType = 'outp'

if ( n_elements(dataType) le 0) then begin
    if (!debug) then print,'Improper behaviour within "fans_sliderEvent"'
    return
endif

; get the energy array of current data
switch dataType of

    'samp': begin
        index = (*sPtr).sampIndex
        fdStructPtr = (*(*sPtr).sDataPtr)[index] ; get a ptr to the approp. fans data struct
        break
    end

    'bkgd': begin
        index = (*sPtr).bkgdIndex
        fdStructPtr = (*(*sPtr).bDataPtr)[index] ; get a ptr to the approp. fans data struct
        break
    end

;    'outp': begin
;        print,'To be implemented...'
;        return
;        index = (*sPtr).outpIndex
;        fdStructPtr = (*(*sPtr).oDataPtr)[index] ; get a ptr to the approp. fans data struct
;        break
;    end


endswitch

energyPtr = (*fdStructPtr).energyPtr

; now alter the slider value so it is a valid value within the energy array.
fans_setSliderValue,(*sPtr).eSliderID,energyPtr,err=err,emsg=emsg
if (err ne 0) then begin
    printerror,err,emsg
    return
endif

; finally, update the plot.
fans_plotdata,sPtr,dataType

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_sumData
;
; PURPOSE:
;   Plot data in the draw widget.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;   dataType [in] - data type string: 'samp'|'bkgd'|'fast'|'outp'
;
;   xrange [in] - range of x values to be displayed
;
;   yrange [in] - range of y values to be displayed
;
; KEYWORDS:
;   err [out] - err number, if an error occured
;
;   emsg [out] - explanatory error message, if an error occured
;



;===============================================================================
pro fans_plotData, sPtr, dataType, xrange=xrge, yrange=yrge, err=err,emsg=emsg

err = 0  &  emsg = ''

; two input parameters necessary
if (n_params() ne 2) then begin
    err = 1
    emsg = '"In fans_plotData" - Invalid number of procedure parameters'
    return
endif;===============================================================================


; check input parameters are defined
if ( (n_elements(dataType) le 0)  or (ptr_valid(sPtr) eq 0) ) then begin
    err = 2
    emsg = 'In "fans_plotData" - Input parameter(s) undefined'
    return
endif


switch dataType of

    'samp': begin
        index = (*sPtr).sampIndex
        fdStructPtr = (*(*sPtr).sDataPtr)[index] ; get a ptr to the approp. fans data struct
        date = (*fdStructPtr).date
        title = (*fdStructPtr).title
        subtitle = 'Expt date: '+date
        xtitle = 'Detector number'
        ;; determine appropriate label based on scan type.
        switch (*fdStructPtr).base of
            'TIME': begin
                ftime = fix((*(*fdStructPtr).timePtr)[0] * 60)
                ytitle = 'Counts per '+strtrim(string(ftime),2)+' secs'
                break
            end
            
            'NEUT':
            else: begin
                monCnts = (*fdStructPtr).monitor * (*fdStructPtr).prefactor
                ytitle = 'Counts per '+strtrim(string(monCnts),2)+' mon'
                break
            end
        endswitch        
        ;; get index of energy value set by slider
        widget_control, (*sPtr).eSliderID, get_uvalue=eIndex
        
        ;; get detector counts for this energy
        ;; NB dets 1 and 2 are monitors!
        ndets = (*fdStructPtr).ndets
        nspec = (*fdStructPtr).nspec
        if ( (eIndex lt 0) or (eIndex ge nspec) ) then eIndex=0 
        detCnts = (*(*fdStructPtr).detCntsPtr)[2:ndets-1,eIndex] ; ignore monitors
        detNos = indgen(ndets-2)+3

        ;; set winPix as destination window initially
        wset, (*sPtr).winPix
        
        ;; plot detector counts vs detector number
        plot,detNos,detCnts, $
             xstyle=1,/ynozero,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
             xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title, $
             xrange=[0,ndets+2]
        ;; plot errors
            sigma = (*(*fdStructPtr).detErrsPtr)[2:ndets-1,eIndex] 
            errplot, detNos, detCnts-sigma, detCnts+sigma ; plot errors

        ;; display masked detectors
        fans_plotMskList, sPtr

        wset, (*sPtr).winVis    ; set winVis as destination window
        device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis

        break
    end

    'bkgd': begin
        index = (*sPtr).bkgdIndex
        fdStructPtr = (*(*sPtr).bDataPtr)[index] ; get a ptr to the approp. fans data struct
        date = (*fdStructPtr).date
        title = (*fdStructPtr).title
        subtitle = 'Expt date: '+date
        xtitle = 'Detector number'
        ;; determine appropriate label based on scan type.
        switch (*fdStructPtr).base of
            'TIME': begin
                ftime = fix((*(*fdStructPtr).timePtr)[0] * 60)
                ytitle = 'Counts per '+strtrim(string(ftime),2)+' secs'
                break
            end
            
            'NEUT':
            else: begin
                monCnts = (*fdStructPtr).monitor * (*fdStructPtr).prefactor
                ytitle = 'Counts per '+strtrim(string(monCnts),2)+' mon'
                break
            end
        endswitch        

        ;; get index of energy value set by slider
        widget_control, (*sPtr).eSliderID, get_uvalue=eIndex
        
        ;; get detector counts for this energy
        ;; NB dets 1 and 2 are monitors!
        ndets = (*fdStructPtr).ndets
        nspec = (*fdStructPtr).nspec
        if ( (eIndex lt 0) or (eIndex ge nspec) ) then eIndex=0 
        detCnts = (*(*fdStructPtr).detCntsPtr)[2:ndets-1,eIndex] ; ignore monitors
        detNos = indgen(ndets-2)+3
        
        ;; set winPix as destination window initially
        wset, (*sPtr).winPix

        ;; plot detector counts vs detector number
        plot,detNos,detCnts, $
             xstyle=1,/ynozero,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
             xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title, $
             xrange=[0,ndets+2]
        ;; plot errors
            sigma = (*(*fdStructPtr).detErrsPtr)[2:ndets-1,eIndex] ;0.5*sqrt(detCnts)
            errplot, detNos, detCnts-sigma, detCnts+sigma ; plot errors

        ;; display masked detectors
        fans_plotMskList, sPtr

        wset, (*sPtr).winVis    ; set winVis as destination window
        device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
        
        break
    end

    'fast': begin
        index = (*sPtr).fastIndex
        fdStructPtr = (*(*sPtr).fDataPtr)[index] ; get a ptr to the approp. fans data struct
        omega = (*(*fdStructPtr).energyPtr) ; energy tranfers
        ;dspace = (*fdStructPtr).monDSpace ; get the monochromator d-space value
        widget_control, (*sPtr).mskOnID, get_uvalue=mskon
        if (mskon) then begin 
            goodDets = -1
            goodDets = fans_maskDets(sPtr) ; get a list of good dets
            if ((ptr_valid((*sPtr).mskListPtr) eq 1) or goodDets[0] ne -1) then begin
                nspec = (*fdStructPtr).nspec
                totCnts = (sigma = fltarr(nspec))
                ;; in exps below, gooddets refer to the dets so need to sub 1
                ;; to get correct indices
                for i = 0,nspec-1 do begin ; re-calculate total cnts using only good dets
                    totCnts[i] = total((*(*fdStructPtr).detCntsPtr)[goodDets-1,i])
                    sigma[i] = sqrt(total(((*(*fdStructPtr).detErrsPtr)[goodDets-1,i])^2))
                endfor
            endif
        endif else begin
            totCnts = (*(*fdStructPtr).totCntsPtr) ; total counts
            sigma = (*(*fdStructPtr).totErrsPtr)
        endelse
        date = (*fdStructPtr).date
        title = (*fdStructPtr).title
        subtitle = 'Expt date: '+date
        unit_str = ((*(*sPtr).mfoPtr)[3] eq 0)? '(meV)' : '(!ncm!u-1!n)'
        xtitle = 'Energy Transfer '+unit_str
        ;; determine appropriate label based on scan type.
        switch (*fdStructPtr).base of
            'TIME': begin
                ftime = fix((*(*fdStructPtr).timePtr)[0] * 60)
                ytitle = 'Counts per '+strtrim(string(ftime),2)+' secs'
                break
            end
            
            'NEUT':
            else: begin
                monCnts = (*fdStructPtr).monitor * (*fdStructPtr).prefactor
                ytitle = 'Counts per '+strtrim(string(monCnts),2)+' mon'
                break
            end
        endswitch        
        
        ;; set winPix as destination window initially
        wset, (*sPtr).winPix

        if ( (n_elements(xrge) eq 0) or (n_elements(yrge) eq 0) ) then begin
            plot,omega,totCnts, $
              xstyle=1,ystyle=1,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
              xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title
        endif else begin 
            plot,omega,totCnts, $
              xrange=xrge,yrange=yrge, $
              xstyle=1,ystyle=1,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
              xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title
        endelse 
        ;; plot errors
        errplot, omega, totCnts-sigma, totCnts+sigma ; plot errors

        ;; plot fit, if available
        ;; retrieve parameters from the text widget
        ;widget_control, (*sPtr).parID, get_value=parstr
        ;if (n_elements(parstr) le 0) then goto, fitdisp_bypassed
        ;fpar = [1.0,1.0,1.0,0.0,0.0,0.0]
        ;tokens = strsplit(parstr[0],' ,',/extract) ; tokenize the string
        ;for i=0,n_elements(tokens)-1 do begin ; convert each token into a float
        ;    reads, tokens[i], af
        ;    fpar[i] = af
        ;endfor 
        ;if ( (fpar[0] eq 0.0) and (fpar[1] eq 0.0) and (fpar[2] eq 0.0) ) then $ ; no fit parameters yet
        ;  goto, fitdisp_bypassed
        ;fpar[4] = dspace
        ;fpar[5] = ((*(*sPtr).mfoPtr)[3] eq 0)? 1.0 : (*sPtr).mev2wnos
        if ((*sPtr).fitPresent) then begin
            fpar = (*(*sPtr).fitParsPtr)
            case (*sPtr).fitFunc of
                0: yfit=(fans_Arcsin(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar
                1: yfit=(fans_Linear(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
                2: yfit=(fans_Quad(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
                3: yfit=(fans_Cubic(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
            endcase
            oplot, omega,yfit,linestyle=2,thick=4,color=(*sPtr).colors.magenta ; overplot fit 
        endif
        fitdisp_bypassed:
        wset, (*sPtr).winVis    ; set winVis as destination window
        device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
        break
    end

    'outp': begin
        index = (*sPtr).outpIndex
        fdStructPtr = (*(*sPtr).oDataPtr)[index] ; get a ptr to the approp. fans data struct
        omega = (*(*fdStructPtr).energyPtr) ; energy tranfers
        totCnts = (*(*fdStructPtr).totCntsPtr) ; total counts
        totErr = (*(*fdStructPtr).totErrsPtr) ; error in counts
        ;monitor = (*(*fdStructPtr).monPtr)
        date = (*fdStructPtr).date
        title = (*fdStructPtr).title
        subtitle = 'Expt date: '+date
        unit_str = ((*(*sPtr).mfoPtr)[3] eq 0)? '(meV)' : '(!ncm!u-1!n)'
        xtitle = 'Energy Transfer '+unit_str

        ;Data is already normalized to monitor of first dataset 
        ;If using a custom monitor to normalize then need to rescale
        monConstant = (*fdStructPtr).monitor * (*fdStructPtr).prefactor
        wID = widget_info((*sPtr).tlbID,find_by_uname='USEMON')
        if (widget_info(wID,/valid)) then begin
           widget_control, wID, get_uvalue=isSet
           if (isSet) then begin
              wID = widget_info((*sPtr).tlbID,find_by_uname='MONVAL')
              if (widget_info(wID,/valid)) then begin
                 widget_control, wID, get_value=monValue
                 if (monValue gt 0.0) then begin
                     normFac = float(monValue)/monConstant
                     totErr = totErr * normFac
                     totCnts = totCnts * normFac
                     monConstant=monValue
                 endif
              endif
           endif
        endif
        
        ;; determine appropriate label based on scan type.
        switch (*fdStructPtr).base of
            'TIME': begin
                ftime = fix((*(*fdStructPtr).timePtr)[0] * 60)
                ytitle = 'Counts per '+strtrim(string(ftime),2)+' secs'
                break
            end
            
            'NEUT':
            else: begin
                ;monCnts = (*fdStructPtr).monitor * (*fdStructPtr).prefactor
                ytitle = 'Counts per '+strtrim(string(monConstant),2)+' mon'
                break
            end
        endswitch        
        ;; set winPix as destination window initially
        wset, (*sPtr).winPix

        if ( (n_elements(xrge) eq 0) or (n_elements(yrge) eq 0) ) then begin
            plot,omega,totCnts, $
              xstyle=1,ystyle=1,psym=-4,symsize=1.0,linestyle=0,charsize=1.4, $
              xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title
            ;; plot errors
            ;sigma = (*(*fdStructPtr).totErrsPtr)
            errplot, omega, totCnts-totErr, totCnts+totErr ; plot errors            
        endif else begin
            plot,omega,totCnts, $
              xrange=xrge,yrange=yrge, $
              xstyle=1,ystyle=1,psym=-4,symsize=1.0,linestyle=0,charsize=1.4, $
              xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title
            ;; plot errors
            ;sigma = (*(*fdStructPtr).totErrsPtr)
            errplot, omega, totCnts-totErr, totCnts+totErr ; plot errors
        endelse 
        wset, (*sPtr).winVis    ; set winVis as destination window
        device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
        break
    end
    
endswitch

end


;===============================================================================
; fans_Help
;
; PURPOSE:
;   Event handler. Display FANS data reduction manual.
;
; PARAMETERS:
;   event [in] - event structure.
;
pro fans_Help,event
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_Help: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

widget_control,event.top,get_uvalue = pState
void = launch_help(!DAVE_PDFHELP_DIR+'fans_dave_manual.pdf',tlb = event.top)
return
end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_resize
;
; PURPOSE:
;   Event handler. Resize the tlb widget.
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_resize, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_resize: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


if dave_set_focus(event) then return

widget_control, event.top, get_uvalue=sPtr

; new plot window width is maximum of the two r-values
(*sPtr).winWidth = (event.x - (*sPtr).restOfWidth) > (*sPtr).minXSize

; new plot window height is maximum of two r-values
(*sPtr).winHeight = (event.y - (*sPtr).restOfHeight) > (*sPtr).minYSize


; Delete and recreate pixmap window using new dimensions
wdelete, (*sPtr).winPix
window,/free,/pixmap,xsize=(*sPtr).winWidth,ysize=(*sPtr).winHeight
(*sPtr).winPix = !d.window
; Also resize visible window using new dimensions 
widget_control, (*sPtr).winVisID, xsize=(*sPtr).winWidth, ysize=(*sPtr).winHeight

widget_control, (*sPtr).tlbID, /realize

; Finally, update the display
widget_control, (*sPtr).sViewID, get_uvalue=sViewOn
widget_control, (*sPtr).bViewID, get_uvalue=bViewOn
widget_control, (*sPtr).fViewID, get_uvalue=fViewOn
widget_control, (*sPtr).oViewID, get_uvalue=oViewOn
case 1 of
    sViewOn: begin
        if ((*sPtr).sampCount lt 1) then return
        index = (*sPtr).sampIndex
        dataPtr = (*(*sPtr).sDataPtr)[index]
        updatePlotSlider, sPtr,'samp',dataPtr,err=err,emsg=emsg
    end

    bViewOn: begin
        if ((*sPtr).bkgdCount lt 1)  then return
        index = (*sPtr).bkgdIndex
        dataPtr = (*(*sPtr).bDataPtr)[index]
        updatePlotSlider, sPtr,'bkgd',dataPtr,err=err,emsg=emsg
    end

    fViewOn: begin
        if ((*sPtr).fastCount lt 1)  then return
        fans_plotData, sPtr,'fast',err=err,emsg=emsg ; can't view det cuts
    end 

    oViewOn: begin
        if ((*sPtr).sampCount lt 1)  then return
        fans_outpCalc, sPtr, err=err, emsg=emsg
        if (err ne 0) then begin
            printerror,err,emsg
            return
        endif
        fans_plotData, sPtr,'outp',err=err,emsg=emsg ; can't view det cuts
    end 

endcase


end



;===============================================================================
; fans_settings_event
;
; PURPOSE:
;   Event handler. Main handler for settings dialog.
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_settings_event, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_settings_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;***********************
; 1 meV = 8.065541 cm-1
;***********************
widget_control, event.top, get_uvalue=sPtr

sname = tag_names(event,/structure_name)
if (sname eq 'WIDGET_DROPLIST') then begin
    (*sPtr).index = event.index
    return
endif

ename = widget_info(event.id,/uname)

if (event.select eq 0) then begin ; ==> a radio button is being released
    case ename of               ; need to know when switching from 'separate'
        'ssep': (*sPtr).stype = 1
        'bsep': (*sPtr).stype = 2
        'fsep': (*sPtr).stype = 3
        else: return 
    endcase 
    return
endif

; If reach here ==> radio button is being set
case ename of

    'ssum': (*(*sPtr).mfoPtr)[0] = 0
    'sapd': (*(*sPtr).mfoPtr)[0] = 1
    'ssep': (*(*sPtr).mfoPtr)[0] = 2
    
    'bsum': (*(*sPtr).mfoPtr)[1] = 0
    'bapd': (*(*sPtr).mfoPtr)[1] = 1
    'bsep': (*(*sPtr).mfoPtr)[1] = 2

    'fsum': (*(*sPtr).mfoPtr)[2] = 0
    'fapd': (*(*sPtr).mfoPtr)[2] = 1
    'fsep': (*(*sPtr).mfoPtr)[2] = 2

    'mev': (*(*sPtr).mfoPtr)[3] = 0
    'wnos': (*(*sPtr).mfoPtr)[3] = 1

    'cancel': widget_control, event.top, /destroy

    'ok': begin
        (*sPtr).exitOK = 1
        widget_control, event.top, /destroy
    end
 
endcase

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_settings_event
;
; PURPOSE:
;   Event handler. For event generated when 'Change Options' button is
;   clicked. Build modal dialog that lets user set preferences on how
;   multiple files are to be handled.
;
; PARAMETERS:
;   event [in] - event structure
;
pro fans_settings, event

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fans_settings: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control, event.top, get_uvalue = sPtr
mfoPtr=(*sPtr).mfoPtr
;mfoPtr = ptr_new([1,1,1,0,0])

; build the settings dialog
tlbID = widget_base(/col,title='CONFIGURE OPTIONS',/modal,group_leader=event.top)

cb0ID = widget_base(tlbID,/col,frame=1,/align_center,/grid)

void = widget_label(cb0ID,value='Multiple input files settings')

cb1ID = widget_base(cb0ID,/row)
cb1aID = widget_base(cb1ID,/row,/align_center)
cb1bID = widget_base(cb1ID,/row,/exclusive)

cb2ID = widget_base(cb0ID,/row)
cb2aID = widget_base(cb2ID,/row,/align_center)
cb2bID = widget_base(cb2ID,/row,/exclusive)

cb3ID = widget_base(cb0ID,/row)
cb3aID = widget_base(cb3ID,/row,/align_center)
cb3bID = widget_base(cb3ID,/row,/exclusive)

cb4ID = widget_base(tlbID,/row,frame=1,/align_center)
void = widget_label(cb4ID,value='Energy Units:')
;elist = ['mev','wavenumbers']
cb4aID = widget_base(cb4ID,/row,/exclusive,/align_center)
;cb4aID = widget_droplist(cb4ID,value=elist,2),uname='eunit',uvalue=elist)
;eunit='mev'

cb5ID = widget_base(tlbID,/row,/align_center,frame=1)

cb6ID = widget_base(tlbID,/row,/align_center)

void = widget_label(cb1aID,value='    Sample:')
sSumID = widget_button(cb1bID,value='sum',uname='ssum')
sapdID = widget_button(cb1bID,value='merge',uname='sapd',sensitive=0)
ssepID = widget_button(cb1bID,value='separate',uname='ssep')
case (*mfoPtr)[0] of 
    0: widget_control, sSumID, /set_button
    1: widget_control, sapdID, /set_button
    2: widget_control, ssepID, /set_button
endcase

void = widget_label(cb2aID,value='Background:')
bSumID = widget_button(cb2bID,value='sum',uname='bsum')
bapdID = widget_button(cb2bID,value='merge',uname='bapd',sensitive=0)
bsepID = widget_button(cb2bID,value='separate',uname='bsep')
case (*mfoPtr)[1] of 
    0: widget_control, bSumID, /set_button
    1: widget_control, bapdID, /set_button
    2: widget_control, bsepID, /set_button
endcase

void = widget_label(cb3aID,value=' Fast Bkgd:')
fSumID = widget_button(cb3bID,value='sum',uname='fsum')
fapdID = widget_button(cb3bID,value='merge',uname='fapd',sensitive=0)
fsepID = widget_button(cb3bID,value='separate',uname='fsep')
case (*mfoPtr)[2] of 
    0: widget_control, fSumID, /set_button
    1: widget_control, fapdID, /set_button
    2: widget_control, fsepID, /set_button
endcase

mevID = widget_button(cb4aID,value='meV',uname='mev')
wnosID = widget_button(cb4aID,value='wavenumbers',uname='wnos')
case (*mfoPtr)[3] of
    0: widget_control, mevID, /set_button
    1: widget_control, wnosID, /set_button
endcase 

void = widget_label(cb5ID,value='Number of active detectors:')
dlist = indgen(89)+40           ; can choose from 10 to 128
result = where(dlist eq (*sPtr).nosActDets)
nActDetID = widget_droplist(cb5ID,value=strtrim(string(dlist),2),uname='adet',uvalue=dlist)
widget_control, nActDetID, set_droplist_select=result[0] ; default is 50 detectors

okID = widget_button(cb6ID,value='OK',uname='ok')
cancelID = widget_button(cb6ID,value='CANCEL',uname='cancel')

widget_control, tlbID, /realize

tmpPtr = ptr_new({mfoPtr:ptr_new(*mfoPtr), $       ;ptr_new([*mfoPtr,0,0,50]), $
                  stype:0, $
                  exitOK:0, $
                  index:result[0] $
                 })
widget_control, tlbID, set_uvalue=tmpPtr

;store current mfoPtr settings
old_mfo = *mfoPtr

; lanch dialog and let user choose options
xmanager, 'fans_settings',tlbID

if ( (*tmpPtr).exitOK ) then begin ; ==> dialog exited with 'ok' so apply changes
    ;; note the current nos of active detectors
    (*sPtr).nosActDets = dlist[(*tmpPtr).index]

    ;; if switching from 'separate' analysis, record current indices
    if ((*tmpPtr).stype gt 0) then begin
        case (*tmpPtr).stype of
            1: (*sPtr).sampSepIndex = (*sPtr).sampIndex
            2: (*sPtr).bkgdSepIndex = (*sPtr).bkgdIndex
            3: (*sPtr).fastSepIndex = (*sPtr).fastIndex
        endcase 
    endif
    
    ;; make necessary updates to data
    (*mfoPtr) = (*(*tmpPtr).mfoPtr)
    if ( n_elements(where(*mfoPtr eq old_mfo)) ne 4 ) then begin ; ie if settings have altered
        fans_ApplySettings, sPtr, old_mfo[3], err=err, emsg=emsg
        if (err ne 0) then begin
            printerror, err, emsg
            return
        endif
    endif 
endif

ptr_free,(*tmpPtr).mfoPtr,tmpPtr                 ; no longer required

end




;===============================================================================
; dummy
;
; PURPOSE:
;  dummy event handler
;
; PARAMETERS:
;   event [in] - event structure to be handled.
;
pro dummy, event
if (!debug) then begin
    print,'Dummy handler called...'

    help, event,/structure

    widget_control, event.top, get_uvalue=sPtr
    
    widget_control, event.id, get_value=value
    widget_control, event.id, get_uvalue=uvalue
    uname = widget_info(event.id,/uname)
    print,'DUMMY: Value = ',value,'  **Uname = ',uname ;,'  **Uvalue = ',uvalue
endif

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_datReduc_cleanup
;
; PURPOSE:
;   Cleanup routine called when the main widget is destroyed
;
; PARAMETERS:
;   tlbID [in] - the widget id of the top-level-base that is being
;                killed.
;
pro fans_datReduc_cleanup, tlbID

;print, 'Cleaning up...'
widget_control, tlbID, get_uvalue=sPtr

if (!debug) then print,'Validity of sPtr is ',ptr_valid(sPtr)

;---- BEGIN DAVE specific cleanup section
;fansInfo = {FANSDATREDUCEVENT,$
;          id:(*sPtr).notifyIDs[0],$
;          top:(*sPtr).notifyIDs[1],$
;          handler:0L}
;if widget_info((*sPtr).notifyIDs[0],/valid_id) then begin
;  widget_control,(*sPtr).notifyIDs[0],send_event = fansInfo
;endif
;---- END DAVE specific section

tvlct,(*sPtr).oldrgb.r,(*sPtr).oldrgb.g,(*sPtr).oldrgb.b
wdelete, (*sPtr).winPix         ; delete pixmap window

;; delete all data currently in memory
if (ptr_valid((*sPtr).sDataPtr)) then $
  fans_cleanDataPtrs, sPtr,'samp' ; sample data
if (ptr_valid((*sPtr).bDataPtr)) then $
  fans_cleanDataPtrs, sPtr,'bkgd' ; background data
if (ptr_valid((*sPtr).fDataPtr)) then $
  fans_cleanDataPtrs, sPtr,'fast' ; fast background data
if (ptr_valid((*sPtr).oDataPtr)) then $
  fans_cleanDataPtrs, sPtr,'outp' ; output data


;; delete davePtr
;heap_free, (*sPtr).davePtr
;
;; deal with the remainder of the state pointer
ptr_free, (*sPtr).mfoPtr, $
          (*sPtr).sampListPtr, $
          (*sPtr).bkgdListPtr, $
          (*sPtr).fastListPtr, $
          (*sPtr).outpListPtr, $
          (*sPtr).mskListPtr, $
          (*sPtr).fitParsPtr, $
          sPtr

if (!debug) then print,'Validity of sPtr is ',ptr_valid(sPtr)

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_dataReduc
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
;   group_leader [in] - the ID of the parent widget from which this
;                       dialog was launched.
;
;   dataDir [in] - The default location for raw data files
;
;   workDir [in] - The default location for saving reduced files
;
;   DAVETool [in] - Reference to the DAVE tool object
;
pro fans_datReduc, group_leader=group_leader, workDir=workDir, dataDir=dataDir, DAVETool=oDAVETool, _EXTRA=etc
;pro fans_datReduc, davePtr,notifyIds=notifyIds,group_leader=group_leader
compile_opt idl2

tvlct,rorig,gorig,borig,/get
oldrgb = {r:rorig,g:gorig,b:borig}
device,decomposed = 0
colors = hfbs_GetColor(/Load, Start=1) ; custom (16) color table

; define container base widgets; make it resizeable
tlbID = widget_base(group_leader=group_leader,/col,title='FANS DATA REDUCTION',mbar=mbar,/tlb_size_events)

cb2ID = widget_base(tlbID,/row,frame=1,/align_center)

cb3ID = widget_base(tlbID,/col,frame=1,/align_center)
cb3aID = widget_base(cb3ID,/row)
cb3bID = widget_base(cb3aID,/row)
cb3cID = widget_base(cb3aID,/row,/exclusive,event_pro='fans_changeviewdata')
cb3dID = widget_base(cb3aID,/row,frame=1)

cb4ID = widget_base(tlbID,/col,frame=1);,/align_center)
cb4aID = widget_base(cb4ID,/row,event_pro='fans_sliderEvent',/align_center)
cb4bID = widget_base(cb4ID,/row,event_pro='fans_sliderEvent',/align_center)
cb4cID = widget_base(cb4ID,/row,event_pro='fans_sliderEvent',/align_center)

cb5ID = widget_base(tlbID,/row,frame=1,/align_center)
cb5aID = widget_base(cb5ID,/row)
cb5bID = widget_base(cb5ID,/row,/nonexclusive,/grid_layout)
cb5cID = widget_base(cb5ID,/row)
;void = widget_base(cb5ID,/row)
;cb5cID = widget_base(void,/row,/nonexclusive,/grid_layout)
;cb5dID = widget_base(void,/row)

; bulid the menus. There are three drop-down menus: file, settings and help
fileMenu = widget_button(mbar, value='File',/menu, $
                         event_pro='fans_filemenu')
settingsMenu = widget_button(mbar, value='Settings',/menu, $
                         event_pro='fans_settings')
helpMenu = widget_button(mbar, value='Help',/menu, $
                         event_pro='fans_Help')

; fill the 'File' drop-down menu
dirMID = widget_button(fileMenu,value='Change Data Directory',uname='dir')
sampMID = widget_button(fileMenu,value='Select Sample File(s)',uname='samp')
bkgdMID = widget_button(fileMenu,value='Select Bkgd File(s)',uname='bkgd')
fbkgdMID = widget_button(fileMenu,value='Select Fast Bkgd File(s)',uname='fast')
save1MID = widget_button(fileMenu,value='Save (DAVE) File',uname='sdave',/separator)
save2MID = widget_button(fileMenu,value='Save (ASCII) File',uname='ascii')
save3MID = widget_button(fileMenu,value='Save (2D DAVE) File',uname='s2dave')
save4MID = widget_button(fileMenu,value='Export to DAVE Data Manager',uname='sdavetool')
exitMID = widget_button(fileMenu,value='Quit',uname='exit',/separator)

; 'Settings' menu
optsID = widget_button(settingsMenu,value='Change Options',uname='opts')

; fill the 'Help' drop-down menu
;aboutMID = widget_button(helpMenu,value='About FANS_DATREDUC',uname='about')
manMID = widget_button(helpMenu,value='Manual',uname='man',sensitive=1)

empty='  No files yet  '
voID = widget_label(cb2ID,value='Samples:')
sampID = widget_droplist(cb2ID,value=empty,uname='samp',uvalue=0,event_pro='fans_dropList')
voID = widget_label(cb2ID,value='Bkgd:')
bkgdID = widget_droplist(cb2ID,value=empty,uname='bkgd',event_pro='fans_dropList')
voID = widget_label(cb2ID,value='Fast Bkgd:',xoffset=50)
fastID = widget_droplist(cb2ID,value=empty,uname='fast',event_pro='fans_dropList')

voID = widget_label(cb3bID,value='Display:',/dynamic_resize)
sViewID = widget_button(cb3cID,value='Sample',uname='samp',uvalue=1,sensitive=0)
bViewID = widget_button(cb3cID,value='Bkgd',uname='bkgd',uvalue=0,sensitive=0)
fViewID = widget_button(cb3cID,value='Fast',uname='fast',uvalue=0,sensitive=0)
oViewID = widget_button(cb3cID,value='Output',uname='outp',uvalue=0,sensitive=0)
widget_control, sViewID,/set_button
fitBkgdID = widget_button(cb3dID,value='Fit Fast Bkgd',sensitive=0,event_pro='fans_Fastfit')
; dropdown for available fit functions
funcs = ['Arcsin','Linear','Quadratic','Cubic']
fnameID = widget_combobox(cb3dID,value=funcs,event_pro='fans_fitFuncSel',xsize=85)
void =  widget_label(cb3dID,value='<--Select model function',/dynamic_resize)

cbInfo = widget_info(cb3ID,/geometry)
xSize = fix(cbInfo.scr_xsize - 2*cbInfo.xpad)
ySize = fix(0.6*xSize)
winVisID = widget_draw(cb4bID,xsize=xsize,ysize=ysize,/button_events,/motion_events,event_pro='fans_plotClick')
eSliderID = cw_fslider(cb4cID,title='Energy Transfer Cut',xsize=xSize,uname='eslider', $
                       min=0.0,max=220.0,format='(F6.2)',uvalue=0)
widget_control, eSliderID, sensitive=0

voID = widget_label(cb4aID,value='Masked detector list:',/dynamic_resize)
vInfo = widget_info(voID,/geometry)
mskListID = widget_text(cb4aID,value='',scr_xsize=0.98*(xSize-vInfo.xsize))

voID = widget_label(cb5aID,value='Output Settings:')
;geom = widget_info(void,/geometry)
mskOnID = widget_button(cb5bID,value='Apply Mask',uvalue=0,sensitive=0,uname='mskon', $
                        event_pro='fans_outpOpts')
subBkgdID = widget_button(cb5bID,value='Sub Bkgd',uvalue=0,sensitive=0,uname='sbkgd', $
                          event_pro='fans_outpOpts')
subFastID = widget_button(cb5bID,value='Sub Fast Fit',uvalue=0,sensitive=0,uname='sfast', $
                          event_pro='fans_outpOpts')
setMonID = widget_button(cb5bID,value='Use this mon:',uvalue=0,uname='USEMON', $
                          event_pro='fans_outpOpts',xoffset=100)
monValID = cw_field(cb5cID,title='',value=1,/long,xsize=10,uname='MONVAL')
; realise widgets
widget_control, tlbID, /realize

tlbInfo = widget_info(tlbID,/geometry)
restOfHeight = tlbInfo.scr_ysize - ySize
restOfWidth = tlbInfo.scr_xsize - xSize

; create a pixmap window with same size as visible window
; and note its number
window,/free,/pixmap,xsize=xsize,ysize=ysize
winPix = !d.window 

; get the visible window nos and set it as default plot window;
widget_control, winVisID, get_value=winVis,sensitive=1
wset, winVis

; define a template davePtr to be used for storing output data
void = create_dave_pointer(davePtr, qty=fltarr(2))

; define and set uvalue for the top level base widget
workDir = (n_elements(workDir) le 0)? '' : workDir
dataDir = (n_elements(dataDir) le 0)? '' : dataDir
; define and set uvalue for the top level base widget
state = {tlbID:tlbID, $
         mfoPtr:ptr_new([0,2,0,0]), $ ; options for multiple files - user configurable 
         nosActDets:50 , $      ; nos of active detectors - user configurable!
         datDir:dataDir, $      ; the data directory
         workDir:workDir, $     ; the work directory
         sampID:sampID, $       ; Sample droplist widget ID
         sampListPtr:ptr_new(/allocate_heap), $ ; ptr to an array of sample files
         sampCount:0, $         ; nos of Sample files
         sampIndex:0, $         ;
         sampSepIndex:0, $
         sampStatus:0, $
         bkgdID:bkgdID, $       ; Bkgd widget ID
         bkgdListPtr:ptr_new(/allocate_heap), $ ; ptr to an array of Bkgd files
         bkgdCount:0, $         ; nos of Bkgd files
         bkgdIndex:0, $         ;
         bkgdSepIndex:0, $
         bkgdStatus:0, $
         fastID:fastID, $       ; Fast bkgd widget ID
         fastListPtr:ptr_new(/allocate_heap), $ ; ptr to a array of Fast bkgd files
         fastCount:0, $         ; nos of Fast bkgd files
         fastIndex:0, $         ;
         fastSepIndex:0, $
         fastStatus:0, $
         outpListPtr:ptr_new(/allocate_heap), $ ; ptr to output files
         outpCount:0, $         ; nos of Fast bkgd files
         outpIndex:0, $         ; 
         doOutpCalc:1, $        ; flag to indicate whether output should be calc.
         sViewID:sViewID, $     ; Disolay Sample widget ID
         bViewID:bViewID, $     ; Display bkgd widget ID
         fViewID:fViewID, $     ; Display Fast bkgd widget ID
         oViewID:oViewID, $     ; Display Output widget ID
         fitBkgdID:fitBkgdID, $ ; fIt bkgd widget ID
         fitFunc:0, $           ; the currently selected fit function index
         fitParsPtr:ptr_new(/allocate_heap), $ ; list of fit parameters
         fitPresent:0, $        ; is fast bkgd fitted
         ;parID:parID, $         ; fitted Fast bkgd parameters widget ID
         mskListID:mskListID, $ ; Masked detector list widget ID
         mskListPtr:ptr_new(), $ ; ptr to array of detectors to be masked
         updFast:0, $           ; = 1 whenever masked det list changes
         winVisID:winVisID, $   ; visible plot window widget ID
         winVis:winVis, $       ; visible window number
         winPix:winPix, $       ; pixmap window number
         winWidth:xsize, $      ; visible/pixmap window width
         winHeight:ysize, $     ; visible/pixmap window height
         restOfHeight:restOfHeight, $ ; height of tlb - plot window height
         restofWidth:restofWidth, $   ; width of tlb - plot window width
         minXSize:xSize, $      ; minimum width for plot window, set by initial value 
         minYSize:ySize, $      ; minimum plotwindow height, set by initial value
         eSliderID:eSliderID, $ ; Energy transfer cut widget ID
         mskOnID:mskOnID, $     ; Apply Mask widget ID
         subBkgdID:subBkgdID, $ ; sub Bkgd widget ID
         subFastID:subFastID, $ ; sub Fast Bkgd fit widget ID
         sDataPtr:ptr_new(), $  ; ptr to an array of sample data ptrs
         bDataPtr:ptr_new(), $  ; ptr to an array of bkgd data ptrs
         fDataPtr:ptr_new(), $  ; ptr to an array of Fast bkgd data ptrs
         oDataPtr:ptr_new(), $  ; ptr to an array of output data ptrs
         davePtr:davePtr, $     ; davePtr obtained from dave
         zp1:[0.0,0.0], $       ; zoom selection - first point
         zp2:[0.0,0.0], $       ; zoom selection - second point
         mouseType:0, $         ; mouse, 1==left, 2==middle, 4==right
         oldrgb:oldrgb, $       ; rbg values outside fans_datreduc
         colors:colors, $       ; colors within fans_datreduc from a custom 16-color table 
         mev2wnos:8.065541, $    ; meV to wavenumber conversion factor
         DAVETool:oDAVETool $  ; object reference of DAVE2 Main Tool 
;         notifyIds:notifyIds $
        }
sPtr = ptr_new(state,/no_copy)
widget_control, tlbID, set_uvalue=sPtr

xmanager, 'fans_datreduc',tlbID,event_handler='fans_resize',cleanup='fans_datreduc_cleanup' ;,/no_block


end
