; $Id$
; Written by J.R.D. Copley.
;************************************************************************************************
;###############################################################################
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;###############################################################################
pro DCSToolsPerformance_cleanup,tlb
;************************************************************************************************
;
compile_opt strictarr
widget_control,tlb,get_uvalue=state,/no_copy
ptr_free,(state).npsptr
obj_destroy,[state.fluxplot,state.fluxplot2,state.resnplot]
;
end


;************************************************************************************************
function DCSToolsPerformance_resn,wl,rmode=rmode
;************************************************************************************************
mspeed = 20000
if n_elements(rmode) eq 0 then rmode=1
;measured elastic energy resolution values
if rmode eq 1 then $        ;low resl
   mresn = [[1.5,4100.],[1.8, 2290],[2.3,1153.5],[2.8,620.],[3.5, 328],[4.1,199.4],$
            [4.5,175.1],[4.8,130.6],[5.0, 109.5],[5.9,69.9],[6.0,68.7],[6.8, 46.3],$
            [7.2, 38.4],[7.5, 34.5],[8.0, 29.15],[9.2,20.4],[ 10,  15],[ 12,  8.3],$
            [ 15,  4.6]] $
else if rmode eq 2 then $   ;medium resl
   mresn = [[1.8, 990.],[2.3,  470],[  3,   211],[  4,  93],[4.5,  74],[  5,   51],$
            [  6, 31.5],[  7,   21],[  8, 14.45],[ 10,8.35],[ 12, 5.6],[ 14,  3.5],$
            [ 16,  2.7],[ 18,    2],[ 20,   1.5]]
if n_elements(wl) eq 0 then return, mresn
case rmode of
    1:      resn = interpol(mresn[1,*],mresn[0,*],wl)
    2:      resn = interpol(mresn[1,*],mresn[0,*],wl)
    3:      resn = dcs_jcc_resn(wl,wl,rmode,mspeed)
    else:   resn = fltarr(n_elements(wl))
endcase
return, resn
end

;************************************************************************************************
pro DCSToolsPerformance_handler,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'DCSToolsPerformance_handler: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


IF DAVE_SET_FOCUS(EVENT) THEN RETURN
;
widget_control,event.top,get_uvalue=state
;handle done event
done=state.done
if (event.id eq done) then begin
	widget_control,event.top,/destroy
	return
endif
;handle mouse motion event in the plots
if tag_names(event,/structure_name) eq 'DM_PLOT_MOTION' then begin
   messagetext=state.messagetext
   npsptr=state.npsptr
   value0=""
   value1=""
   value2=""
   wl=event.xval
   if finite(wl) then begin
      flux1=dcs_jcc_flux(wl,1,npsptr)
	  flux2=dcs_jcc_flux(wl,2,npsptr)
	  resn1=DCSToolsPerformance_resn(wl,rmode=1)
	  resn2=DCSToolsPerformance_resn(wl,rmode=2)
	  resn1a=13220/wl^2.947
	  resn2a=3948/wl^2.656
	  min2th=5.02       ;minimum detector angle
	  max2th=139.91     ;maximum detector angle
	  qmin=2*!pi/wl*sin(min2th/2.*!dtor)*2
	  qmax=2*!pi/wl*sin(max2th/2.*!dtor)*2
	  E0=81.8042/wl/wl
	  value0=[string(wl,format='("Wavelength:",(f8.3)," '+string('c5'XB)+'")'),$
	    string(E0,format='("    E0:",(f8.3)," meV")'),"",$
		"Neutrons at sample (n/s):","",$
		"    Low resolution (mode 1):"+string(flux1,format='(e10.2)'),"",$
		"    Medium resolution (mode 2):"+string(flux2,format='(e10.2)'),"",$
		"Elastic scattering Q range:","",$
		"    Q_min = "+string(qmin,format='(f8.3)')+string('c5'XB)+'^-1',$
		"    Q_max = "+string(qmax,format='(f8.3)')+string('c5'XB)+'^-1']
	  value1=value0
	  value2=[string(wl,format='("Wavelength:",(f8.3)," '+string('c5'XB)+'")'),$
	  	string(E0,format='("    E0:",(f8.3)," meV")'),"",$
		"Elastic resolution (ueV)","  (model: fit to measurement):","",$
		"    Low resolution (mode 1):"+string(resn1,format='(f8.1)')+':'+string(resn1a,format='(f8.1)'),"",$
		"    Medium resolution (mode 2):"+string(resn2,format='(f8.1)')+':'+string(resn2a,format='(f8.1)')]
   endif
   widget_control,messagetext[0],set_value=value0
   widget_control,messagetext[1],set_value=value1
   widget_control,messagetext[2],set_value=value2
endif
end


;************************************************************************************************
pro DCSToolsPerformance,$
	group_leader = group_leader, workDir=workDir, dataDir=dataDir, $
;	notify_ids = notify_ids,$
	register_name = register_name, $
	_EXTRA=etc

;************************************************************************************************
;
compile_opt strictarr
;
;
;;;;;;;added following 4 lines for non-modal widget action
if n_elements(group_leader) eq 0 then group_leader = 0L
;if n_elements(notify_ids) eq 0 then notify_ids = [0L,0L]
if n_elements(register_name) eq 0 then register_name = 'DCSToolsPerformance'
if xregistered(register_name) then return
if n_elements(workDir) eq 0 then workDir = ''
if n_elements(dataDir) eq 0 then dataDir = ''


;
; Lines commented out.
;widget_control,event.top,get_uvalue=statePtr
;davePtr=(*statePtr).davePtr
;dcs_sysvars,davePtr=davePtr
;tvlct,r,g,b,/get
;rgbptr=ptr_new({r:r,g:g,b:b})
;
screensize=!screensize
;
xsz=screensize[0]
ysz=screensize[1]
messagetext=lonarr(3)
if !version.os_family eq 'Windows' then $
   messagexsize=36 $
else $
   messagexsize=46
;
tlb=widget_base(title='Flux and Resolution at 20000 rpm',/row,group_leader=group_leader,map=0)
	plots=widget_base(tlb,/col)
		fluxbase=widget_base(plots,/row)
			fluxplot=obj_new('dm_plot',/compound,widgetbase=fluxbase,group_leader=tlb,$
			    xsize=0.5*xsz,ysize=0.32*ysz,/draw_motion_events,/ylog,/hidelegend)
			messagetext[0]=widget_text(fluxbase,ysize=14,xsize=messagexsize,value="")
		fluxbase2=widget_base(plots,/row)
			fluxplot2=obj_new('dm_plot',/compound,widgetbase=fluxbase2,group_leader=tlb,$
			    xsize=0.5*xsz,ysize=0.32*ysz,/draw_motion_events,/ylog,/hidelegend)
			messagetext[1]=widget_text(fluxbase2,ysize=14,xsize=messagexsize,value="")
		resnbase=widget_base(plots,/row)
			resnplot=obj_new('dm_plot',/compound,widgetbase=resnbase,group_leader=tlb,$
			    xsize=0.5*xsz,ysize=0.20*ysz,/draw_motion_events,/ylog,/hidelegend)
			messagetext[2]=widget_text(resnbase,ysize=10,xsize=messagexsize,value="")
; Realize the widget
done=widget_button(plots,value='Done')
;
centertlb,tlb
widget_control,tlb,/realize,/map
;
dcs_getnps,npsptr
;
; Plot the top frame.
ylim=[0.9e2,1.2e6]
for rmode=1,2 do begin
    wl=((*npsptr).(rmode-1))[0,*]
    flux=((*npsptr).(rmode-1))[1,*]
	fluxplot->add_plot,wl,flux,color=(['red','green'])[rmode-1],thick=2.0,$
	    legend=(['low','medium'])[rmode-1],psym='circle'
endfor
fluxplot->setproperty,xrange=[0,12],yrange=ylim,$
    xtitle='Wavelength ('+string('c5'XB)+')',ytitle='Neutrons at sample (n/s)'
;
; Plot the middle frame.
;
for rmode=1,2 do begin
	wl=((*npsptr).(rmode-1))[0,*]
    flux=((*npsptr).(rmode-1))[1,*]
	fluxplot2->add_plot,wl,flux,color=(['red','green'])[rmode-1],thick=2.0,$
	    legend=(['low','medium'])[rmode-1],psym='circle'
endfor
fluxplot2->setproperty,xrange=[1,3],yrange=ylim,$
    xtitle='Wavelength ('+string('c5'XB)+')',ytitle='Neutrons at sample (n/s)'
;
;
mspeed=20000
ylim=[2,9e3]
;
; Plot the bottom frame.
;
for rmode=1,2 do begin
    mresn=DCSToolsPerformance_resn(rmode=rmode)
    wl=mresn[0,*] & resn=mresn[1,*]
	resnplot->add_plot,wl,resn,color=(['red','green'])[rmode-1],thick=2.0,$
	    legend=(['low','medium'])[rmode-1],layer=10+rmode,psym='circle'
	resna=([13220,3948])[rmode-1]/wl^(([2.947,2.656])[rmode-1])
	resnplot->add_plot,wl,resna,color='black',thick=2,linestyle=3,$
	    legend=(['low','medium'])[rmode-1]+' model'
endfor
resnplot->setproperty,xrange=[1,12],yrange=ylim,$
    xtitle='Wavelength ('+string('c5'XB)+')',ytitle='Resolution FWHM (ueV)'
;
state={done:done,$
	messagetext:messagetext,$
	npsptr:npsptr,$
;	notify_ids:notify_ids,  $              ; for non-modal widget
	fluxplot:fluxplot, $
	fluxplot2:fluxplot2, $
	resnplot:resnplot}
widget_control,tlb,set_uvalue=state
;
RET = DAVE_SET_FOCUS(TLB)
;
; added register_name variable to the call to xmanager.
xmanager,register_name,tlb,/no_block,cleanup='DCSToolsPerformance_cleanup',$
	event_handler="DCSToolsPerformance_handler"
;
; Lines commented out.
;dave_makeDavePtr,davePtr=davePtr
;(*statePtr).davePtr=davePtr
;tvlct,(*rgbptr).r,(*rgbptr).g,(*rgbptr).b
;ptr_free,rgbptr
end
