; $Id:  $
;#######################################################################
;
; NAME:
;  dm_polanalysis
;
; PURPOSE:
;  This program reduces raw data into mslice readable neutron polarization 
;  data file for DCS and MACS. The calcuation is based on the equations provided
;  by Wangchun Chen (12/6/2021).
;     
; CATEGORY:
;  file tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;clear all objects and widget hierarchy
pro dm_polanalysis_Exit,tlb
    widget_control,tlb,get_uvalue=state,/no_copy
    dm_polanalysis_saveparm,state,/quiet
    obj_destroy,[state.open_filesel,state.save_filesel]
    ptr_free,state.pol_parm,state.ana_parm,state.tau_parm,state.file_hist,state.nsffilePtr,state.sffilePtr,state.nsfdataPtr,state.sfdataPtr,state.nsfPtr,state.sfPtr
    widget_control,tlb,/destroy
end

;given timestring, calculate the estern timezone, return 'EST', or 'EDT'
function dm_polanalysis_timezone,timestring
    timezone = strarr(n_elements(timestring))
    cdf_epoch,epoch_0,1970,1,1,0,0,0,/compute_epoch
    for i=0,n_elements(timestring)-1 do begin
        tmpnumber = dm_to_number(timestring[i],/epoch)
        tmp_date  = systime(elapsed=tmpnumber,/utc)
        dow  = (where(['Mon','Tue','Wed','Thu','Fri','Sat','Sun'] eq strmid(tmp_date,0,3)))[0]
        cdf_epoch,tmpnumber*(1000d)+epoch_0,year,month,day,hour,minute,second,/breakdown_epoch
        dow  = ((35+dow+1-day) mod 7) ;dow for day=1
        hour = hour+minute/60.+second/3600.
        dst  = (year gt 2023) or ((year eq 2023) and (month gt 3)) or ((month gt 3) and (month lt 11)) or ((month eq 3) and ((day+dow gt 14) or ((day+dow eq 14) and (hour ge 2)))) or ((month eq 11) and ((day+dow lt 7) or ((day+dow eq 7) and (hour lt 2))))
        timezone[i] = (['EST','EDT'])[dst]
    endfor
    if n_elements(timestring) eq 1 then return,timezone[0] else return,timezone
end

function dm_polanalysis_filename,infile,dcs=dcs,macs=macs,f_ind=f_ind
    if n_elements(infile) eq 0 then return,''
    if n_elements(f_ind) eq 0 then f_ind = 0
    file = infile[0]
    if keyword_set(dcs) then begin
       ind  = strpos(file,'_',/reverse_search)
       tmp  = strsplit(strmid(file,ind),'_.',/extract)
       tmp1 = dm_to_string(f_ind) 
       if strlen(tmp1) lt strlen(tmp[0]) then tmp1 = strjoin(replicate('0',strlen(tmp[0])-strlen(tmp1)))+tmp1
       file = strmid(file,0,ind)+'_'+tmp1+'.'+strjoin(tmp[1:*],'.')
    endif 
    return,file
end

;load parameter file
pro dm_polanalysis_loadparm,state,parmfile,dataDir=dataDir,workDir=workDir,init=init
    if n_elements(parmfile) eq 0 then parmfile = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'polanal.prm'
    WIDGET_CONTROL,/HOURGLASS
    openr,unit,parmfile,/get_lun,error=err
    if err ne 0 then return
    clear_history = 1b
    tmp  = ''
    readf,unit,tmp
    point_lun,unit,0  ;reset the reading point to the beginning
    if stregex(tmp,'(dm_pol|filetyp)',/boolean,/fold_case) then begin  ;this is the .prm file
       change_ftype = 1b
       while(~ eof(unit)) do begin
          readf,unit,tmp
          tmp  = strtrim(tmp,2)
          head = strmid(tmp,0,8)
          tmp  = strmid(tmp,8,strlen(tmp)-8)
          case strlowcase(head) of
               'filetyp:':state.ftype = tmp
               'ncnrftp:':state.ncnrftp = dm_to_number(tmp,/int)
               'datadir:':dataDir  = tmp
               'workdir:':workDir  = tmp
               'polparm:':begin
                          ptr_free,state.pol_parm
                          tmpvalue = strsplit(tmp,'&',/extract,/preserve)
                          n_val    = n_elements(tmpvalue)<n_elements(state.pol_parm)
                          for i=0,n_val-1 do state.pol_parm[i] = ptr_new(abs(dm_to_number(strsplit(tmpvalue[i],', ',/extract),/double)))
                          end
               'anaparm:':begin
                          ptr_free,state.ana_parm
                          tmpvalue = strsplit(tmp,'&',/extract,/preserve)
                          n_val    = n_elements(tmpvalue)<n_elements(state.ana_parm)
                          for i=0,n_val-1 do state.ana_parm[i] = ptr_new(abs(dm_to_number(strsplit(tmpvalue[i],', ',/extract),/double)))
                          end
               'samptyp:':state.samptype  = dm_to_number(tmp)
               'monitor:':state.monchoice = dm_to_number(tmp)
               'history:':begin
                          if clear_history then begin
                             clear_history   = 0b
                             state.n_history = 0
                             totnum          = n_elements(state.file_hist)  ;make sure the capacity is not exceeded
                             ptr_free,state.file_hist
                          endif
                          state.file_hist[state.n_history mod totnum] = ptr_new(strsplit(tmp,string('A4'xb),/extract,/preserve))
                          state.n_history = state.n_history+1
                          end
               else:
          endcase
       endwhile
    endif else begin  ;wangchun's 3he cell parameter file
       mesg = ['','','','']
       while(~ eof(unit)) do begin
          readf,unit,tmp
          tmp = strtrim(tmp,2)
          if strlen(tmp) gt 0 then begin
             tmp1 = strsplit(tmp,' '+string(9b),/extract)
             if strmatch(tmp1[0],'#cellname',/fold_case) then begin
                ind_PorA  = (where(strmatch(tmp1,'pora',/fold_case)))[0]
                ind_iDate = (where(strmatch(tmp1,'idate',/fold_case)))[0]
                ind_iTime = (where(strmatch(tmp1,'itime',/fold_case)))[0]
                ind_tzone = (where(strmatch(tmp1,'timezone',/fold_case)))[0]
                ind_iunix = (where(strmatch(tmp1,'iunixtime',/fold_case)))[0]
                ind_ipol  = (where(strmatch(tmp1,'ipol',/fold_case)))[0]
                ind_lifet = (where(strmatch(tmp1,'t(hr)',/fold_case)))[0]
                ind_durat = (where(strmatch(tmp1,'telapsed',/fold_case)))[0] ;p_f = p_i*exp[-telasped/t(hr)]
                ind_nsl   = (where(strmatch(tmp1,'nsl',/fold_case)))[0]      ;opacity = nsl/lambda
                ind_wavel = (where(strmatch(tmp1,'lambda',/fold_case)))[0]
                ind_te    = (where(strmatch(tmp1,'tempty',/fold_case)))[0]
             endif else begin
                if n_elements(ind_PorA) eq 0 then continue
                p_or_a    = (where(['p','a'] eq strlowcase(strmid(tmp1[ind_PorA],0,1))))[0]
                iHe       = dm_to_number(tmp1[ind_ipol],/double)
                duration  = dm_to_number(tmp1[ind_durat],/double)
                itime     = dm_to_number(tmp1[ind_iunix],/double)
                tmp_date  = strsplit(tmp1[ind_iDate],'/',/extract)
                if strlen(tmp_date[2]) eq 2 then tmp_date[2] = '20'+tmp_date[2]
                tmp_time  = strsplit(tmp1[ind_iTime],':',/extract,count=n_time)
                if n_time eq 2 then tmp_time = [tmp_time,'00']
                if ind_tzone ge 0 then tmp_tzone = tmp1[ind_tzone] else tmp_tzone = dm_polanalysis_timezone(strjoin(tmp_date,'/')+' '+strjoin(tmp_time,':'))
                itime1    = dm_to_number(strjoin(tmp_date,'/')+' '+strjoin(tmp_time,':')+' '+tmp_tzone,/epoch,/double)
                if itime ne itime1 then mesg = [[mesg],[tmp1[0],tmp1[ind_iunix],tmp1[ind_iDate]+' '+tmp1[ind_iTime],dm_to_string(itime1,/int)]]
                Te        = dm_to_number(tmp1[ind_te],/double)
                opacity   = dm_to_number(tmp1[ind_nsl],/double)/dm_to_number(tmp1[ind_wavel],/double) 
                fHe       = iHe*exp(-duration/dm_to_number(tmp1[ind_lifet],/double))
                if p_or_a eq 0 then begin
                   if n_elements(p_iHe) eq 0 then begin
                      p_iHe      = iHe
                      p_fHe      = fHe
                      p_itime    = itime
                      p_duration = duration
                      p_opacity  = opacity
                      p_Te       = Te
                   endif else begin
                      p_iHe      = [p_iHe,iHe]
                      p_fHe      = [p_fHe,fHe]
                      p_itime    = [p_itime,itime]
                      p_duration = [p_duration,duration]
                      p_opacity  = [p_opacity,opacity]
                      p_Te       = [p_Te,Te]
                   endelse
                endif else begin
                   if n_elements(a_iHe) eq 0 then begin
                      a_iHe      = iHe
                      a_fHe      = fHe
                      a_itime    = itime
                      a_duration = duration
                      a_opacity  = opacity
                      a_Te       = Te
                   endif else begin
                      a_iHe      = [a_iHe,iHe]
                      a_fHe      = [a_fHe,fHe]
                      a_itime    = [a_itime,itime]
                      a_duration = [a_duration,duration]
                      a_opacity  = [a_opacity,opacity]
                      a_Te       = [a_Te,Te]
                   endelse
                endelse
             endelse
          endif
       endwhile
       if n_elements(p_iHe) ne 0 then begin
          ptr_free,state.pol_parm
          state.pol_parm[0] = ptr_new(p_iHe,/no_copy)
          state.pol_parm[1] = ptr_new(p_fHe,/no_copy)
          state.pol_parm[2] = ptr_new(p_itime,/no_copy)
          state.pol_parm[3] = ptr_new(p_duration,/no_copy)
          state.pol_parm[4] = ptr_new(p_opacity,/no_copy)
          state.pol_parm[5] = ptr_new(p_Te,/no_copy)
       endif
       if n_elements(a_iHe) ne 0 then begin
          ptr_free,state.ana_parm
          state.ana_parm[0] = ptr_new(a_iHe,/no_copy)
          state.ana_parm[1] = ptr_new(a_fHe,/no_copy)
          state.ana_parm[2] = ptr_new(a_itime,/no_copy)
          state.ana_parm[3] = ptr_new(a_duration,/no_copy)
          state.ana_parm[4] = ptr_new(a_opacity,/no_copy)
          state.ana_parm[5] = ptr_new(a_Te,/no_copy)
       endif
       if ~keyword_set(init) and (n_elements(mesg) gt 4) then begin
          info = ['The initial unix timestamp of '+strjoin(reform(mesg[0,1:*]),' and ')+([' is ',' are '])[n_elements(mesg) gt 8]+'inconsistent with the date and time string. '+$
                  'Please check the parameter file.',' ']
          for i=1,n_elements(mesg[0,*])-1 do info = [info,mesg[0,i]+':'+strjoin(replicate(' ',1>(9-strlen(mesg[0,i]))))+string(9b)+mesg[1,i]+', '+mesg[2,i]+' ('+mesg[3,i]+')']
          ok = dialog_message(info,dialog_parent=state.tlb,/center)
       endif
    endelse
    free_lun,unit
    if keyword_set(init) then return
    ;set file type and menu buttons
    if keyword_set(change_ftype) then dm_polanalysis_changeftype,state,workDir=workDir
    ;polarizer and analyzer info
    wids = [state.pol_input,state.ana_input] & vals = [state.pol_parm,state.ana_parm]
    for i=0,n_elements(wids)-1 do begin
        if ptr_valid(vals[i]) then val = *vals[i] else val = !values.f_nan
        widget_control,wids[i],set_value=dm_to_string(val,int=(((wids[i] eq state.pol_input[2]) or (wids[i] eq state.ana_input[2])) and total(finite(val,/nan) eq 0)),sep=', ')
    endfor
end

;parameters:
;   mon_counts:  [ntchan, nfile] or [ntchan] array
;keyword:
;   intint:     if set, calculate integrated intensity, otherwise sum
function dm_polanalysis_monitor,mon_counts,intint=intint   
    mon_sum = total(mon_counts,1)
    if keyword_set(intint) then begin
       for i=0,n_elements(mon_sum)-1 do begin
           dm_gaussfit,mon_counts[*,i],params=params,fitnotgood=fitnotgood
           if (~ fitnotgood) then mon_sum[i] = abs(params[0]*params[2])
       endfor
    endif
    if n_elements(mon_sum) eq 1 then return, mon_sum[0] else return, mon_sum
end

;load file name, and A2 (DCS) only
pro dm_polanalysis_loadfile,state,nsf=nsf,sf=sf
    state.open_filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,dFiles=dfiles,ftpserver=ftpserver,ftpbufferdir=ftpbufferdir
    n_file = n_elements(open_file)
    if n_file eq 0 then return
    is_ftp = obj_valid(ftpobj)
    while(strmid(open_path,0,1,/reverse_offset) eq  ([state.pathsep,'/'])[is_ftp]) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    
    if state.ftype eq 'DCS' then begin   ;DCS 
       davePtr = dm_define_pointer()
       nofirstline = 0 & omithisto=2               ;dcs_read_binaryOctavefile keyword
       for i=0,n_file-1 do begin
           file = open_path+([state.pathsep,'/'])[is_ftp]+open_file[i]
           if is_ftp then begin
              ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[0])
              file = ftpbufferdir+state.pathsep+'ftptmp'+open_file[0]
           endif
           dcs_read_binaryOctavefile,file,nofirstline,omithisto,var_read,unknown,readerror,davePtr=davePtr
           if is_ftp then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
           if strlen(readerror) ne 0 then begin
              heap_free,davePtr & tmp=check_math(mask=32)
              ok = dialog_message([open_file[i],readerror],/error,dialog_parent=state.tlb,/center)
              return
           end
           this = dm_locate_datastrptr(davePtr)
           if n_elements(a2) eq 0 then a2 = (*(*this).specificPtr).motor_pos[1] $
           else a2 = [a2,(*(*this).specificPtr).motor_pos[1]]
           ch_wl      = float((*(*this).specificPtr).ch_wl)                         ;wavelength
           ch_ms      = float((*(*this).specificPtr).ch_ms)                         ;chopper master speed
           ch_srdenom = (*(*this).specificPtr).ch_srdenom                           ;chopper speed ratio denomenator
           ch_srmode  = (*(*this).specificPtr).ch_srmode                            ;chopper speed ratio mode
           ch_res     = (*(*this).specificPtr).ch_res                               ;resolution mode
           tsdmin     = float((*(*this).specificPtr).tsdmin)                        ;tsdmin
           info       = [ch_wl,ch_res,ch_ms,ch_srmode,ch_srdenom,tsdmin]
           if n_elements(info0) eq 0 then begin
              info0 = info
              file0 = state.open_filesel->filebasename(open_file[i])
           endif else if total(abs(info-info0)) ne 0 then begin
              heap_free,davePtr & tmp=check_math(mask=32)
              file1 = state.open_filesel->filebasename(open_file[i])
              maxln = max(state.open_filesel->wavlstrlen([info0[0],ch_wl],/nozeroend))
              tmp   = [state.open_filesel->formatlinestring(file0+':  ',info0,/wavelength,maxlength=maxln),state.open_filesel->formatlinestring(file1+':  ',info,/wavelength,maxlength=maxln)]
              ok    = dialog_message(['Selected data files are of different wavelength settings.',tmp],/error,dialog_parent=state.tlb,/center)
              return
           endif
       endfor
       heap_free,davePtr
       if is_ftp then open_path = ftpserver+open_path
       if keyword_set(nsf) then begin
          ptr_free,state.nsffilePtr
          state.nsffilePtr = ptr_new({filedir:open_path,filename:open_file,a2:a2,info:info})
          widget_control,state.tickNSFBut,/set_button
       endif else if keyword_set(sf)  then begin
          ptr_free,state.sffilePtr
          state.sffilePtr = ptr_new({filedir:open_path,filename:open_file,a2:a2,info:info})
          widget_control,state.tickSFBut,/set_button
       endif
    endif else begin  ;MACS
       check_sf = 1b  ;flag for checking spin flip information
       for i=0,n_file-1 do begin
           file = open_path+([state.pathsep,'/'])[is_ftp]+open_file[i]
           if is_ftp then begin
              ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[0])
              file = ftpbufferdir+state.pathsep+'ftptmp'+open_file[0]
           endif
           
           if is_ftp then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
           
       endfor    
      
    endelse
    if state.idl_version ge 5.6 then $
       widget_control,([state.loadNSFBut,state.loadSFBut])[keyword_set(sf)],tooltip=(['NSF','SF'])[keyword_set(sf)]+' file'+(['','s'])[n_file gt 1]+$
          ' loaded ('+file0+(['',' -> '+state.open_filesel->filebasename(open_file[n_file-1])])[n_file gt 1]+')'
end

pro dm_polanalysis_readdcs,state,open_path,open_file,nsf=nsf,sf=sf
    if (n_elements(open_path) eq 0) or (n_elements(open_file) eq 0) then return
    state.open_filesel->getproperty,ftpobj=ftpobj,ftpserver=ftpserver,ftpbufferdir=ftpbufferdir
    
    is_ftp = stregex(open_path,'(https|ftp)://',/boolean,/fold_case)
    if is_ftp  then begin
       if ~obj_valid(ftpobj) then begin ;make sure ftpobj is valid
          state.open_filesel->set_path,open_path
          state.open_filesel->getproperty,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir,ftpserver=ftpserver
       endif
       open_path = strmid(open_path,strlen(ftpserver))
    endif
    
    davePtr = dm_define_pointer()
    nofirstline = 0 & omithisto=0               ;dcs_read_binaryOctavefile keyword
    for i=0,n_elements(open_file)-1 do begin
        file = open_path+([state.pathsep,'/'])[is_ftp]+open_file[i]
        if is_ftp then begin
           ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[0])
           file = ftpbufferdir+state.pathsep+'ftptmp'+open_file[0]
        endif
        dcs_read_binaryOctavefile,file,nofirstline,omithisto,var_read,unknown,readerror,davePtr=davePtr,/histodata_floating
        if is_ftp then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
        if strlen(readerror) ne 0 then begin
           heap_free,davePtr & tmp=check_math(mask=32)
           ok = dialog_message([open_file[i],readerror],/error,dialog_parent=state.tlb,/center)
           return
        end
        this = dm_locate_datastrptr(davePtr)
        thisqty  = (*(*this).commonStr.histPtr).qty
        thishigh = (*(*this).specificPtr).histohigh
        if i eq 0 then begin  ;figure out time channel number
           ch_wl      = float((*(*this).specificPtr).ch_wl);wavelength
           ch_ms      = float((*(*this).specificPtr).ch_ms);chopper master speed
           ch_srdenom = (*(*this).specificPtr).ch_srdenom  ;chopper speed ratio denomenator
           ntchan     = (*(*this).specificPtr).ntchan      ;number of time channels
           speriod    = 6.0d7/ch_ms*ch_srdenom             ;time between pulses at sample in micro second
           ; Define the number of good time channels for the He3 detectors, ngtchan0,
           ; and the number of good time channels for the high end inputs,  ngtchan1,
           twid       = (*(*this).specificPtr).tchanlook*0.05
           times0     = [0,reform(total(twid[0,*],/cumulative))]
           times1     = [0,reform(total(twid[1,*],/cumulative))]
           tstart     = times0[0:ntchan-1]
           tstop      = times0[1:ntchan] & times0 = 0
           chan       = where(tstart le speriod-0.05 and tstop gt speriod-0.05) + 1
           ngtchan0   = chan[0]
           if ngtchan0 eq 0 then ngtchan0 = 1000L
           tstart     = times1[0:ntchan-1]
           tstop      = times1[1:ntchan] & times1 = 0
           chan       = where(tstart le speriod-0.05 and tstop gt speriod-0.05) + 1
           ngtchan1   = chan[0]
           if ngtchan1 eq 0 then ngtchan1 = 1000L
           conversion = !dcs_hsq2mn/(!dcs_hom/!dcs_dsd)^2
           tmean      = ((*(*(*this).specificPtr).time_propsPtr)[0:ngtchan0-1]+(*(*(*this).specificPtr).time_propsPtr)[1:ngtchan0])*0.5
           emean      = float(!dcs_hsq2mn/ch_wl^2-conversion/tmean^2)
           Ei         = !dcs_hsq2mn/(ch_wl)^2
           Ef         = Ei-emean
        endif
        ;check jumpy file
        tmp_sum = total(thisqty,1)
        tmp_max = max(tmp_sum) & tmp_mean = mean(tmp_sum)
        tmp_ind = where(tmp_sum ge 10.0*tmp_mean,tmp_count)
        if tmp_count ne 0 then tmp_sum[tmp_ind] = tmp_max  ;enlarge the jumpy noise signal
        tmp = A_CORRELATE(tmp_sum,[31,248])                ;31 is the period, 248=31*8 is to avoid mistake
        if min(tmp) gt 0.3 then begin                      ;0.3 is the preset criterion
           if ((*(*this).specificPtr).ncycles eq 1) or (dm_to_number((*(*this).specificPtr).start_date,/epoch) lt dm_to_number('Jan 1 00:00:00 2015',/epoch)) then tmp_mesg = '' $
           else tmp_mesg = 'try patching this file using Mslice File->File Tools->Patch DCS Files or '  ;patching is only possible if ncycles> 1 and for files with each cycles saved
           ok = dialog_message(open_file[i]+' might be a "jumpy" data file. If this is the case, please '+tmp_mesg+'discard this file.',dialog_parent=state.tlb,title='Warning: '+open_file[i],/center)
        endif
        tmp_sum = 0
        ;check and correct bitflip errors silently
        nhedet   = (*(*this).specificPtr).nhedet               ;number of He-3 detectors
        tdh_dsum = [long64(total(thisqty,1,/double)),long64(total(thishigh,1,/double))]
        tdh_tsum = long64(total(thisqty,2,/double))+long64(total(thishigh,2,/double))
        bad_t = where((tdh_tsum-(*(*this).specificPtr).timsum) ne 0,tcount)    ;disagreement in time sums over detector
        bad_d = where((tdh_dsum-(*(*this).specificPtr).detsum) ne 0,dcount)    ;disagreement in detector sums over time
        if tcount lt 50 and dcount lt 50 then begin
           ;the following code can only correct isolated bitflip errors
           for ii=0,tcount-1 do begin
               for jj=0,dcount-1 do begin
                   tmp1 = tdh_tsum[bad_t[ii]]-((*(*this).specificPtr).timsum)[bad_t[ii]]
                   tmp2 = tdh_dsum[bad_d[jj]]-((*(*this).specificPtr).detsum)[bad_d[jj]]
                   if (tmp1 eq tmp2) and (bad_t[ii] lt ngtchan0) and (bad_d[jj] lt nhedet) then begin
                      thisqty[bad_t[ii],bad_d[jj]] = thisqty[bad_t[ii],bad_d[jj]]-tmp1
                      tdh_dsum = [long64(total(thisqty,1,/double)),long64(total(thishigh,1,/double))]
                      tdh_tsum = long64(total(thisqty,2,/double))+long64(total(thishigh,2,/double))
                   endif
            endfor
          endfor
        endif
        if n_elements(duration) eq 0 then begin 
           duration  = (*(*this).specificPtr).duration
           n_cycle   = (*(*this).specificPtr).ncycles
           startdate = (*(*this).specificPtr).start_date+' '+dm_polanalysis_timezone((*(*this).specificPtr).start_date)
           stopdate  = (*(*this).specificPtr).stop_date+' '+dm_polanalysis_timezone((*(*this).specificPtr).stop_date)
           a2        = (*(*this).specificPtr).motor_pos[1]
           histohigh = temporary(thishigh)
           qty       = temporary(thisqty)
           n_det     = n_elements(qty[0,*])
        endif else begin
           duration  = [duration,(*(*this).specificPtr).duration]
           n_cycle   = [n_cycle,(*(*this).specificPtr).ncycles]
           startdate = [startdate,(*(*this).specificPtr).start_date+' '+dm_polanalysis_timezone((*(*this).specificPtr).start_date)]
           stopdate  = [stopdate,(*(*this).specificPtr).stop_date+' '+dm_polanalysis_timezone((*(*this).specificPtr).stop_date)]
           a2        = [a2,(*(*this).specificPtr).motor_pos[1]]
           histohigh = [[[temporary(histohigh)]],[[temporary(thishigh)]]]
           qty       = [[[temporary(qty)]],[[temporary(thisqty)]]]
        endelse
    endfor
    heap_free,davePtr
    if is_ftp then open_path = ftpserver+open_path
    if keyword_set(nsf) then begin
       ptr_free,state.nsfdataPtr
       state.nsfdataPtr = ptr_new({duration:temporary(duration),n_cycle:n_cycle,startdate:temporary(startdate),stopdate:temporary(stopdate),a2:temporary(a2),$
                                   histohigh:temporary(histohigh),qty:temporary(qty),ngtchan:[ngtchan0,ngtchan1],Ei:Ei,Ef:Ef,n_det:n_det,filedir:open_path,$
                                   filename:open_file})
    endif else if keyword_set(sf)  then begin
       ptr_free,state.sfdataPtr
       state.sfdataPtr = ptr_new({duration:temporary(duration),n_cycle:n_cycle,startdate:temporary(startdate),stopdate:temporary(stopdate),a2:temporary(a2),$
                                  histohigh:temporary(histohigh),qty:temporary(qty),ngtchan:[ngtchan0,ngtchan1],Ei:Ei,Ef:Ef,n_det:n_det,filedir:open_path,$
                                  filename:open_file})
    endif
end

;This proceudre calculates the matrix elements T00,T01,T10,T11
;parameters: 
;   p_parms: polarizer parameters pointer array [p_iHe, p_fHe, p_itime, p_duration, p_opacity, p_Te]
;   a_parms: analyzer  parameters pointer array [a_iHe, a_fHe, a_itime, a_duration, a_opacity, a_Te]    
;   Ei:      incident energy in meV, scalar or [n_E] array
;   Ef:      final energy in meV, [n_E] array or scalar, Ei and Ef should be one scalar, and one array
;   ts_NSF:  unix time stamp of NSF data, [n_file] or scalar
;   ct_NSF:  counting time of NSF data, in sec, [n_file] or scalar]
;   ts_SF:   unix time stamp of SF data, [n_file] or scalar
;   ct_SF:   counting time of SF data, in sec, [n_file] or scalar
;keywords:
;   n_det:   number of detector, if specified, T00,T01,T10,T11 will be [n_E,n_det] array
;   mon_NSF: monitor count, used for weighted mean for multiple NSF files
;   mon_SF:  monitor count, used for weighted mean for multiple SF files
;   T00,T01,T10,T11: return the matix elements [n_E] array or [n_E,n_det] array if n_det keyword is specified
pro dm_polanalysis_matrix_cal,p_parms,a_parms,Ei,Ef,ts_NSF,ct_NSF,ts_SF,ct_SF,n_det=n_det,mon_NSF=mon_NSF,mon_SF=mon_SF,T00=T00,T01=T01,T10=T10,T11=T11    
    n_E            = (n_elements(Ei))>(n_elements(Ef))             
    n_NSF          = n_elements(ts_NSF)
    n_SF           = n_elements(ts_SF)
 
    T00            = fltarr(n_E)
    T01            = fltarr(n_E)
    T10            = fltarr(n_E)
    T11            = fltarr(n_E)
    
    P_iHe_NSF      = fltarr(n_NSF)
    P_fHe_NSF      = fltarr(n_NSF)
    P_itime_NSF    = fltarr(n_NSF)
    P_duration_NSF = fltarr(n_NSF)
    P_opacity_NSF  = fltarr(n_NSF)
    P_Te_NSF       = fltarr(n_NSF)
    P_iHe_SF       = fltarr(n_SF)
    P_fHe_SF       = fltarr(n_SF)
    P_itime_SF     = fltarr(n_SF)
    P_duration_SF  = fltarr(n_SF)
    P_opacity_SF   = fltarr(n_SF)
    P_Te_SF        = fltarr(n_SF)
    A_iHe_NSF      = fltarr(n_NSF)
    A_fHe_NSF      = fltarr(n_NSF)
    A_itime_NSF    = fltarr(n_NSF)
    A_duration_NSF = fltarr(n_NSF)
    A_opacity_NSF  = fltarr(n_NSF)
    A_Te_NSF       = fltarr(n_NSF)
    A_iHe_SF       = fltarr(n_SF)
    A_fHe_SF       = fltarr(n_SF)
    A_itime_SF     = fltarr(n_SF)
    A_duration_SF  = fltarr(n_SF)
    A_opacity_SF   = fltarr(n_SF)
    A_Te_SF        = fltarr(n_SF)
    P_itime        = *p_parms[2]
    ind_Psort      = sort(P_itime)
    A_itime        = *a_parms[2]
    ind_Asort      = sort(A_itime)
    for i=0,n_NSF-1 do begin
        ind_PNSF          = where(P_itime[ind_Psort] le ts_NSF[i],cnt_PNSF)
        ind_ANSF          = where(A_itime[ind_Asort] le ts_NSF[i],cnt_ANSF)
        P_iHe_NSF[i]      = (*p_parms[0])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        P_fHe_NSF[i]      = (*p_parms[1])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        P_itime_NSF[i]    = (*p_parms[2])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        P_duration_NSF[i] = (*p_parms[3])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        P_opacity_NSF[i]  = (*p_parms[4])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        P_Te_NSF[i]       = (*p_parms[5])[ind_Psort[ind_PNSF[cnt_PNSF-1]]]
        A_iHe_NSF[i]      = (*a_parms[0])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
        A_fHe_NSF[i]      = (*a_parms[1])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
        A_itime_NSF[i]    = (*a_parms[2])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
        A_duration_NSF[i] = (*a_parms[3])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
        A_opacity_NSF[i]  = (*a_parms[4])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
        A_Te_NSF[i]       = (*a_parms[5])[ind_Asort[ind_ANSF[cnt_ANSF-1]]]
    endfor
    for i=0,n_SF-1 do begin
        ind_PSF           = where(P_itime[ind_Psort] le ts_SF[i],cnt_PSF)
        ind_ASF           = where(A_itime[ind_Asort] le ts_SF[i],cnt_ASF)
        P_iHe_SF[i]       = (*p_parms[0])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        P_fHe_SF[i]       = (*p_parms[1])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        P_itime_SF[i]     = (*p_parms[2])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        P_duration_SF[i]  = (*p_parms[3])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        P_opacity_SF[i]   = (*p_parms[4])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        P_Te_SF[i]        = (*p_parms[5])[ind_Psort[ind_PSF[cnt_PSF-1]]]
        A_iHe_SF[i]       = (*a_parms[0])[ind_Asort[ind_ASF[cnt_ASF-1]]]
        A_fHe_SF[i]       = (*a_parms[1])[ind_Asort[ind_ASF[cnt_ASF-1]]]
        A_itime_SF[i]     = (*a_parms[2])[ind_Asort[ind_ASF[cnt_ASF-1]]]
        A_duration_SF[i]  = (*a_parms[3])[ind_Asort[ind_ASF[cnt_ASF-1]]]
        A_opacity_SF[i]   = (*a_parms[4])[ind_Asort[ind_ASF[cnt_ASF-1]]]
        A_Te_SF[i]        = (*a_parms[5])[ind_Asort[ind_ASF[cnt_ASF-1]]]
    endfor
    
    if n_elements(mon_NSF) ne n_NSF then mon_NSF = replicate(1.0,n_NSF)
    if n_elements(mon_SF)  ne n_SF  then mon_SF  = replicate(1.0,n_SF)
    Lambda_i     = dm_e2lambda(Ei)
    Lambda_f     = dm_e2lambda(Ef)
    if n_elements(Lambda_i) ne n_E then Lambda_i = replicate(Lambda_i,n_E)
    if n_elements(Lambda_f) ne n_E then Lambda_f = replicate(Lambda_f,n_E)

    ;calculate the elapsed time (hours) for both polarizer and analyzer at both NSF and SF measurements
    eltime_P_NSF = ((ts_NSF - P_itime_NSF) + ct_NSF/2.)/3600.
    eltime_A_NSF = ((ts_NSF - A_itime_NSF) + ct_NSF/2.)/3600.
    eltime_P_SF  = ((ts_SF  - P_itime_SF)  + ct_SF/2.)/3600.
    eltime_A_SF  = ((ts_SF  - A_itime_SF)  + ct_SF/2.)/3600.

    ;calculate 3He polarizations for both polarizer and analyzer at both NSF and SF measurements
    P_He_NSF     = dm_polanalysis_HePolCal(P_iHe_NSF, P_fHe_NSF, P_duration_NSF, eltime_P_NSF)
    P_He_SF      = dm_polanalysis_HePolCal(P_iHe_SF,  P_fHe_SF,  P_duration_SF,  eltime_P_SF)
    A_He_NSF     = dm_polanalysis_HePolCal(A_iHe_NSF, A_fHe_NSF, A_duration_NSF, eltime_A_NSF)
    A_He_SF      = dm_polanalysis_HePolCal(A_iHe_SF,  A_fHe_SF,  A_duration_SF,  eltime_A_SF)

    ;calculate transmissions of both majority and minority spin states for both polarizer and analyzer at both NSF and SF measurements
    ;polarizer
    Tplus_P_NSF  = dm_polanalysis_Tplus(P_Te_NSF,  P_He_NSF, Lambda_i#P_opacity_NSF)
    Tplus_P_SF   = dm_polanalysis_Tplus(P_Te_SF,   P_He_SF,  Lambda_i#P_opacity_SF)
    Tminus_P_NSF = dm_polanalysis_Tminus(P_Te_NSF, P_He_NSF, Lambda_i#P_opacity_NSF)
    Tminus_P_SF  = dm_polanalysis_Tminus(P_Te_SF,  P_He_SF,  Lambda_i#P_opacity_SF)
    ;analyzer: arrays of n_Ef wavelengths
    Tplus_A_NSF  = dm_polanalysis_Tplus(A_Te_NSF,  A_He_NSF, Lambda_f#A_opacity_NSF)
    Tplus_A_SF   = dm_polanalysis_Tplus(A_Te_SF,   A_He_SF,  Lambda_f#A_opacity_SF)
    Tminus_A_NSF = dm_polanalysis_Tminus(A_Te_NSF, A_He_NSF, Lambda_f#A_opacity_NSF)
    Tminus_A_SF  = dm_polanalysis_Tminus(A_Te_SF,  A_He_SF,  Lambda_f#A_opacity_SF)

    ;calculate transmission matrix arrays
    T00 = reform(matrix_multiply(mon_NSF, Tplus_P_NSF*Tplus_A_NSF  + Tminus_P_NSF*Tminus_A_NSF,/btran)/total(mon_NSF))
    T01 = reform(matrix_multiply(mon_NSF, Tminus_P_NSF*Tplus_A_NSF + Tplus_P_NSF*Tminus_A_NSF,/btran)/total(mon_NSF))
    T10 = reform(matrix_multiply(mon_SF,  Tminus_P_SF*Tplus_A_SF   + Tplus_P_SF*Tminus_A_SF,/btran)/total(mon_SF))
    T11 = reform(matrix_multiply(mon_SF,  Tplus_P_SF*Tplus_A_SF    + Tminus_P_SF*Tminus_A_SF,/btran)/total(mon_SF))
    if n_elements(n_det) ne 0 then begin
       arr1 = replicate(1.0,n_det)
       T00  = matrix_multiply(T00,arr1,/btranspose)
       T01  = matrix_multiply(T01,arr1,/btranspose)
       T10  = matrix_multiply(T10,arr1,/btranspose)
       T11  = matrix_multiply(T11,arr1,/btranspose)
    endif
End

;calculate the 3He polarization
;t in hours is the time elapsed from when a cell is placed on the beam to when the measurement is taken.
function dm_polanalysis_HePolCal,iHe,fHe,duration,t
    return, iHe*exp(-t/duration*alog(iHe/fHe))
end

;calculate the transmission of the minority spin state for a single or multiple wavelength
;Te:      [n_file] array
;pHe:     [n_file] array
;opacity: [n_E,n_file] array
;return   [n_E,n_file] array
function dm_polanalysis_Tminus,Te,pHe,opacity
    n_E     = n_elements(opacity[*,0])
    tmp_Te  = replicate(1.0,n_E)#Te
    tmp_pHe = replicate(1.0,n_E)#pHe
    return, reform(tmp_Te*exp(-opacity*(1.0+tmp_pHe)))
end

;calculate the transmission of the majority spin state for a single or multiple wavelength
;Te:      [n_file] array
;pHe:     [n_file] array
;opacity: [n_E,n_file] array
;return   [n_E,n_file] array
function dm_polanalysis_Tplus,Te,pHe,opacity
    n_E     = n_elements(opacity[*,0])
    tmp_Te  = replicate(1.0,n_E)#Te
    tmp_pHe = replicate(1.0,n_E)#pHe
    return, reform(tmp_Te*exp(-opacity*(1.0-tmp_pHe)))
end

;core reduction 
pro dm_polanalysis_reduction,state
    ptr_free,state.nsfPtr,state.sfPtr
    
    case state.ftype of
         'DCS': begin
                i_mon    = 3 ;BM1
                n_gtc    = (*state.nsfdataPtr).ngtchan
                mon_NSF  = dm_polanalysis_monitor((*state.nsfdataPtr).histohigh[0:(n_gtc[1]-1),i_mon,*],intint=state.monchoice)
                mon_SF   = dm_polanalysis_monitor((*state.sfdataPtr).histohigh[0:(n_gtc[1]-1),i_mon,*],intint=state.monchoice)
                ts_NSF   = dm_to_number((*state.nsfdataPtr).startdate,/epoch)
                ct_NSF   = (*state.nsfdataPtr).duration
                ts_SF    = dm_to_number((*state.sfdataPtr).startdate,/epoch)
                ct_SF    = (*state.sfdataPtr).duration
                n_nsf    = n_elements(ct_NSF)
                n_sf     = n_elements(ct_SF)
                
                dm_polanalysis_matrix_cal,state.pol_parm,state.ana_parm,(*state.nsfdataPtr).Ei,(*state.nsfdataPtr).Ef,ts_NSF,ct_NSF,ts_SF,ct_SF,$
                          n_det=(*state.nsfdataPtr).n_det,mon_NSF=mon_NSF,mon_SF=mon_SF,T00=T00,T01=T01,T10=T10,T11=T11
                factor   = total(mon_NSF)/total(mon_SF)                        
                raw_NSF  = (*state.nsfdataPtr).qty[0:(n_gtc[0]-1),*,*]
                raw_SF   = (*state.sfdataPtr).qty[0:(n_gtc[0]-1),*,*]
                if n_nsf gt 1 then raw_NSF = total(temporary(raw_NSF),3)
                if n_sf  gt 1 then raw_SF  = total(temporary(raw_SF),3)
                tmp1     = (T00*T11-T01*T10)
                raw_SF   = temporary(raw_SF)*factor                   ;intensity normalized to mon_NSF
                cor_NSF  = ( T11*raw_NSF - T01*raw_SF)/tmp1
                cor_SF   = (-T10*raw_NSF + T00*raw_SF)/tmp1/factor    ;monitor count switch back to mon_SF
                raw_SF   = temporary(raw_SF)*factor                   ;error bar^2
                tmp2     = mean(tmp1)
                err0_NSF = sqrt(mean(T11)^2+mean(T01)^2)/tmp2         ;zero intensity error bar
                err0_SF  = sqrt(mean(T10)^2+mean(T00)^2)/tmp2/factor  
                err_NSF  = sqrt(temporary(T11)^2*raw_NSF+temporary(T01)^2*raw_SF)/tmp1
                err_SF   = sqrt(temporary(T10)^2*temporary(raw_NSF)+temporary(T00)^2*temporary(raw_SF))/temporary(tmp1)/factor
                histohigh_NSF = (*state.nsfdataPtr).histohigh[*,*,*]
                histohigh_SF  = (*state.sfdataPtr).histohigh[*,*,*]
                if n_nsf gt 1 then histohigh_NSF = total(temporary(histohigh_NSF),3)
                if n_sf  gt 1 then histohigh_SF  = total(temporary(histohigh_SF),3)
                state.nsfPtr = ptr_new({qty:temporary(cor_NSF),err:temporary(err_NSF),histohigh:temporary(histohigh_NSF),a2:(*state.nsfdataPtr).a2[0],duration:total(ct_NSF),$
                                        n_cycle:total((*state.nsfdataPtr).n_cycle),startdate:(*state.nsfdataPtr).startdate,stopdate:(*state.nsfdataPtr).stopdate,$
                                        filename:(*state.nsfdataPtr).filename,ngtchan:n_gtc,zeroerr:err0_NSF})
                state.sfPtr  = ptr_new({qty:temporary(cor_SF),err:temporary(err_SF),histohigh:temporary(histohigh_SF),a2:(*state.sfdataPtr).a2[0],duration:total(ct_SF),$
                                        n_cycle:total((*state.sfdataPtr).n_cycle),startdate:(*state.sfdataPtr).startdate,stopdate:(*state.sfdataPtr).stopdate,$
                                        filename:(*state.sfdataPtr).filename,ngtchan:n_gtc,zeroerr:err0_SF})
                end
         'MACS':
         else:
    endcase
end 

;save parameter file
pro dm_polanalysis_saveparm,state,parmfile,quiet=quiet
    if n_elements(parmfile) eq 0 then parmfile = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'polanal.prm'
    openw,unit,parmfile,/get_lun,error=openerr
    if openerr ne 0 then begin
       if ~keyword_set(quiet) then ok = dialog_message("Can't write in "+parmfile+'.',/error,dialog_parent=state.tlb,/center)
       return
    endif
    state.open_filesel->getproperty,path=open_path,ftpobj=ftpobj,ftpserver=ftpserver
    state.save_filesel->getproperty,path=save_path
    if obj_valid(ftpobj) then open_path=ftpserver+open_path
    printf,unit,'#dm_polanalysis '+state.mod_date+' IDL '+!version.release+' '+!version.os_name
    printf,unit,'filetyp:'+state.ftype
    printf,unit,'ncnrftp:'+dm_to_string(state.ncnrftp,/int)
    printf,unit,'datadir:'+open_path
    printf,unit,'workdir:'+save_path
    tmp = strarr(n_elements(state.pol_parm))
    for i=0,n_elements(tmp)-1 do if ptr_valid(state.pol_parm[i]) then tmp[i] = dm_to_string(*state.pol_parm[i],int=(i eq 2),sep=',')
    printf,unit,'polparm:'+strjoin(tmp,'&')
    tmp = strarr(n_elements(state.ana_parm))
    for i=0,n_elements(tmp)-1 do if ptr_valid(state.ana_parm[i]) then tmp[i] = dm_to_string(*state.ana_parm[i],int=(i eq 2),sep=',')
    printf,unit,'anaparm:'+strjoin(tmp,'&')
    printf,unit,'samptyp:'+dm_to_string(state.samptype,/int)
    printf,unit,'monitor:'+dm_to_string(state.monchoice,/int)
    for i=0,state.n_history-1 do begin
        file_hist = (*(state.file_hist[i]))[0]
        for j=1,n_elements(*(state.file_hist[i]))-1 do file_hist=file_hist+string('A4'xb)+(*(state.file_hist[i]))[j]
        printf,unit,'history:'+file_hist
    endfor
    free_lun,unit
end

pro dm_polanalysis_writedcs,state,infile,out_nsf,out_sf
    if (~ptr_valid(state.nsfPtr)) or (~ptr_valid(state.sfPtr)) then return
    
    ;check if the file is compressed
    compress = (strlowcase(strmid(infile,2,3,/reverse_offset)) eq '.gz')
    ;Read header to determine byte ordering ("endianness")
    openr,unit,infile,compress=compress,/get_lun
    header = bytarr(11)
    readu,unit,header
    free_lun,unit
    case string(header[9]) of
         "B": begin
              openr,unitr,infile,/swap_if_little_endian,compress=compress,/get_lun
              openw,unitnsf,out_nsf,/swap_if_little_endian,compress=compress,/get_lun
              openw,unitsf,out_sf,/swap_if_little_endian,compress=compress,/get_lun
              end
         "L": begin
              openr,unitr,infile,/swap_if_big_endian,compress=compress,/get_lun
              openw,unitnsf,out_nsf,/swap_if_big_endian,compress=compress,/get_lun
              openw,unitsf,out_sf,/swap_if_big_endian,compress=compress,/get_lun
              end
         else:begin
              ok = dialog_message('Cannot determine byte ordering of file '+(*state.nsfPtr).filename[0]+'. No data files are saved.',/center,dialog_parent=state.tlb)
              abort = 1b
              end
    endcase
    if keyword_set(abort) then return
    
    n_gtc     = (*state.nsfPtr).ngtchan
    ;get the start date and stop date from the first and last measured file
    tmp       = min(dm_to_number((*state.nsfPtr).startdate,/epoch),nsf_start,subscript_max=nsf_stop)
    tmp       = min(dm_to_number((*state.sfPtr).startdate,/epoch),sf_start,subscript_max=sf_stop)
    start_nsf = ((*state.nsfPtr).startdate)[nsf_start]
    stop_nsf  = ((*state.nsfPtr).stopdate)[nsf_stop]
    start_sf  = ((*state.sfPtr).startdate)[sf_start]
    stop_sf   = ((*state.sfPtr).stopdate)[sf_stop]
    
    readu,unitr,header
    writeu,unitnsf,header
    writeu,unitsf,header
    ; Define the types of some variables
    nlgth   = 0l
    dldocgf = bytarr(5)
    dt      = 0b
    padding = 0b
    scalar  = 0.0d
    n1      = 0l
    n2      = 0l
    nstrgs  = 0l
    nchars  = 0l
    while (~eof(unitr)) do begin
          readu,unitr,nlgth              ; read name_length
          name = bytarr(nlgth)
          readu,unitr,name               ; read name
          readu,unitr,dldocgf            ; read 5 bytes of doc_length, doc, and global flag
          readu,unitr,dt                 ; read data type
          writeu,unitnsf,nlgth
          writeu,unitnsf,name
          writeu,unitnsf,dldocgf
          writeu,unitnsf,dt
          writeu,unitsf,nlgth
          writeu,unitsf,name
          writeu,unitsf,dldocgf
          writeu,unitsf,dt
          name = strlowcase(string(name))
          case dt of
               1: begin                  ; read scalar
                  readu,unitr,padding
                  writeu,unitnsf,padding
                  writeu,unitsf,padding
                  readu,unitr,scalar
                  ; Patchable scalars
                  case name of
                       'duration':   scalar = double((*state.nsfPtr).duration)
                       'ncycles':    scalar = double((*state.nsfPtr).n_cycle)
                       else:
                  endcase
                  writeu,unitnsf,scalar
                  case name of
                       'duration':   scalar = double((*state.sfPtr).duration)
                       'ncycles':    scalar = double((*state.sfPtr).n_cycle)
                    else:
                  endcase
                  writeu,unitsf,scalar
                  if name eq 'shutter_stat' then begin  ;add polanal field
                     name = byte('polanal')
                     nlgth = n_elements(name)
                     writeu,unitnsf,nlgth
                     writeu,unitnsf,name
                     writeu,unitnsf,dldocgf
                     writeu,unitnsf,dt
                     writeu,unitsf,nlgth
                     writeu,unitsf,name
                     writeu,unitsf,dldocgf
                     writeu,unitsf,dt
                     writeu,unitnsf,padding
                     writeu,unitsf,padding
                     writeu,unitnsf,1d
                     writeu,unitsf,1d
                  endif
                  end
               2: begin                  ; read matrix
                  readu,unitr,n1,n2
                  nhi = (n1>n2)
                  nlo = (n1<n2)
                  writeu,unitnsf,n1,n2
                  writeu,unitsf,n1,n2
                  readu,unitr,padding
                  writeu,unitnsf,padding
                  writeu,unitsf,padding
                  if (nlo eq 1) then data=dblarr(nhi) else data=dblarr(n1,n2)
                  readu,unitr,data
                  case name of
                       'histodata':  begin
                                     data_nsf = data
                                     data_sf  = temporary(data)
                                     data_nsf[0:(n_gtc[0]-1),*] = (*state.nsfPtr).qty
                                     data_sf[0:(n_gtc[0]-1),*]  = (*state.sfPtr).qty
                                     writeu,unitnsf,data_nsf
                                     writeu,unitsf,data_sf
                                     end
                       'histohigh':  begin
                                     writeu,unitnsf,double((*state.nsfPtr).histohigh)
                                     writeu,unitsf,double((*state.sfPtr).histohigh)
                                     end
                       'motor_pos':  begin
                                     data[1] = (*state.nsfPtr).a2
                                     writeu,unitnsf,data
                                     data[1] = (*state.sfPtr).a2
                                     writeu,unitsf,data
                                     end              
                       else:         begin
                                     writeu,unitnsf,data
                                     writeu,unitsf,data 
                                     end
                  endcase
                  if name eq 'histodata' then begin ; add error bar
                     name = byte('histoerror')
                     nlgth = n_elements(name)
                     writeu,unitnsf,nlgth
                     writeu,unitnsf,name
                     writeu,unitnsf,dldocgf
                     writeu,unitnsf,dt
                     writeu,unitsf,nlgth
                     writeu,unitsf,name
                     writeu,unitsf,dldocgf
                     writeu,unitsf,dt
                     writeu,unitnsf,n1,n2
                     writeu,unitsf,n1,n2
                     writeu,unitnsf,padding
                     writeu,unitsf,padding
                     data_nsf[0:(n_gtc[0]-1),*] = (*state.nsfPtr).err
                     data_sf[0:(n_gtc[0]-1),*]  = (*state.sfPtr).err
                     data_nsf[-1]               = (*state.nsfPtr).zeroerr
                     data_sf[-1]                = (*state.sfPtr).zeroerr
                     writeu,unitnsf,temporary(data_nsf)  ;error bar data
                     writeu,unitsf,temporary(data_sf)    ;error bar data
                  endif
                  end
               5: begin                  ; read string
                  readu,unitr,nchars     ; read number of characters in string element
                  if nchars gt 0 then begin
                     str = bytarr(nchars)
                     readu,unitr,str     ; read string element
                  endif
                  case name of
                       'start_date': begin
                                     str_nsf    = byte(start_nsf)
                                     str_sf     = byte(start_sf)    
                                     nchars_nsf = long(n_elements(str_nsf))
                                     nchars_sf  = long(n_elements(str_sf))
                                     end
                       'stop_date':  begin
                                     str_nsf    = byte(stop_nsf)
                                     str_sf     = byte(stop_sf)    
                                     nchars_nsf = long(n_elements(str_nsf))
                                     nchars_sf  = long(n_elements(str_sf))
                                     end
                       'comments':   begin ;add 'NSF' and 'SF' to the comment
                                     if nchars eq 0 then tmpstr = '' else tmpstr = string(str)
                                     ;check if NSF is already in the comment, remove it if yes
                                     pos = stregex(tmpstr,'(^nsf[ ,]*|^ *sf[ ,]+|[ ,]*sf$|[ ,]*nsf|[ ,]*non[- ]*sf|[ ,]*non[- ]*spin[- ]*flip|[ ,]+sf[ ,]+|[ ,]*spin[- ]*flip)',/fold_case,length=len)
                                     if pos[0] eq -1 then tmpstr = strmid(tmpstr,0,pos[0])+strmid(tmpstr,pos[0]+len[0])
                                     if strlen(tmpstr) gt 0 then tmpstr = ' '+tmpstr
                                     str_nsf    = byte('NSF'+tmpstr)
                                     str_sf     = byte('SF'+tmpstr)
                                     nchars_nsf = long(n_elements(str_nsf))
                                     nchars_sf  = long(n_elements(str_sf))
                                     end
                       else:
                  endcase
                  if (name eq 'comments') or (name eq 'start_date') or (name eq 'stop_date') then begin
                     writeu,unitnsf,nchars_nsf
                     writeu,unitsf,nchars_sf
                     writeu,unitnsf,str_nsf
                     writeu,unitsf,str_sf
                  endif else begin
                     writeu,unitnsf,nchars
                     writeu,unitsf,nchars
                     if nchars gt 0 then begin
                        writeu,unitnsf,str
                        writeu,unitsf,str
                     endif
                  endelse
                  end
               7: begin                  ; read string array
                  readu,unitr,nstrgs     ; read number of strings in array
                  writeu,unitnsf,nstrgs
                  writeu,unitsf,nstrgs
                  for j=0L,nstrgs-1 do begin
                      readu,unitr,nchars ; read number of characters in string element
                      writeu,unitnsf,nchars
                      writeu,unitsf,nchars
                      str = bytarr(nchars)
                      readu,unitr,str    ; read string element
                      writeu,unitnsf,str
                      writeu,unitsf,str
                  endfor
                  end
               else:
          endcase
    endwhile
    free_lun,unitr
    free_lun,unitnsf
    free_lun,unitsf
end

;change file type
pro dm_polanalysis_changeftype,state,ftype=ftype,workDir=workDir
    widget_control,state.tlb,update=0
    if n_elements(ftype) ne 0 then state.ftype = ftype 
    widget_control,state.openlabel,set_value=state.ftype+' Data File Directory'
    is_dcs  = widget_info(state.dcsmonsum,/valid_id)
    is_macs = widget_info(state.macsmoncount,/valid_id)
    for i=0,3 do widget_control,state.tau_lines[i],map=(state.ftype eq 'MACS')
    case state.ftype of
         'DCS': begin
                if ~is_dcs then begin
                   if widget_info(state.macsmontime,/valid_id) then widget_control,state.macsmontime,/destroy
                   if widget_info(state.macsmoncount,/valid_id) then widget_control,state.macsmoncount,/destroy
                   state.dcsmonsum = dm_widget_button(state.monmenu,value='Sum',set_button=(state.monchoice eq 0))
                   state.dcsmonint = dm_widget_button(state.monmenu,value='Integrated Intensity',set_button=(state.monchoice eq 1))
                endif
                dm_toggle_menubut,check=([state.dcsmonsum,state.dcsmonint])[state.monchoice],uncheck=([state.dcsmonint,state.dcsmonsum])[state.monchoice]
                fext = '.DCS'
                end
         'MACS':begin
                if ~is_macs then begin
                   if widget_info(state.dcsmonint,/valid_id) then widget_control,state.dcsmonint,/destroy
                   if widget_info(state.dcsmonsum,/valid_id) then widget_control,state.dcsmonsum,/destroy
                   state.macsmoncount = dm_widget_button(state.monmenu,value='Monitor Count',set_button=(state.monchoice eq 0))
                   state.macsmontime  = dm_widget_button(state.monmenu,value='Time',set_button=(state.monchoice eq 1))
                endif
                dm_toggle_menubut,check=([state.macsmoncount,state.macsmontime])[state.monchoice],uncheck=([state.macsmontime,state.macsmoncount])[state.monchoice]
                fext = '.NG0,.BT9'
                end
         else:  
    endcase
    dm_toggle_menubut,check=([state.dcsBut,state.macsBut])[state.ftype eq 'MACS'],uncheck=([state.dcsBut,state.macsBut])[state.ftype eq 'DCS']
    dm_toggle_menubut,check=([state.powder,state.crystal])[state.samptype],uncheck=([state.crystal,state.powder])[state.samptype]
    widget_control,state.tlb,tlb_set_title= state.ftype+' Polarization Analysis'
    state.open_filesel->set_filter,fext,path=state.dataDir
    state.open_fileSel->set_ftpbuffer,(['',dm_define_pointer(/gettempdir)])[state.ncnrftp]
    state.open_fileSel->getproperty,ncnrftp=ncnrftp
    state.ncnrftp = ncnrftp
    dm_set_button,state.ftpBut,state.ncnrftp
    state.save_filesel->set_filter,fext,path=workDir
    widget_control,state.loadNSFBut,set_value='Load NSF '+(['Powder','Single Crystal'])[state.samptype]+' Files'
    widget_control,state.loadSFBut,set_value='Load SF '+(['Powder','Single Crystal'])[state.samptype]+' Files'
    if state.idl_version ge 5.6 then begin
       widget_control,state.loadNSFBut,tooltip="Select non-spin-flip files in the "+state.ftype+" data file directory and click this button to load files."
       widget_control,state.loadSFBut,tooltip="Select spin-flip files in the "+state.ftype+" data file directory and click this button to load files."
       widget_control,state.convertBut,tooltip="Load NSF and SF data files, enter polarizer and analyzer parameters, then click this button to reduce files."
    endif
    ptr_free,state.nsffilePtr,state.sffilePtr,state.nsfdataPtr,state.sfdataPtr,state.nsfPtr,state.sfPtr
    widget_control,state.tickNSFBut,set_button=0
    widget_control,state.tickSFBut,set_button=0
    widget_control,state.cmdCol,/update
    widget_control,state.tlb,/map,/update
end

pro dm_polanalysis_setfilehistory,state,dir,files,choose=choose,ftpobj=ftpobj,ftpserver=ftpserver,info=info
    if n_elements(choose) eq 1 then begin
       ftype = strmid((*(state.file_hist[choose]))[0],6)
       if ftype ne state.ftype then begin
          wid = ([state.dcsBut,state.macsBut])[ftype eq 'MACS']
          dm_polanalysis_changeftype,{WIDGET_BUTTON,ID:wid,TOP:state.tlb,HANDLER:state.tlb,SELECT:1}
       endif
       state.open_filesel->set_path,(*(state.file_hist[choose]))[2]
       state.open_filesel->getproperty,dFiles=dFiles
       for i=3,n_elements(*(state.file_hist[choose]))-1 do begin
           ind = where(dFiles eq (*(state.file_hist[choose]))[i],count)
           if count ne 0 then begin
              if n_elements(selected) eq 0 then selected=ind $
              else selected=[selected,ind]
           endif
       endfor
       if n_elements(selected) ne 0 then state.open_filesel->set_file,selected,/highlight
       return
    endif
    num = n_elements(state.file_hist)    ;capacity
    nfiles = n_elements(files)
    if (n_elements(dir) ne 0) and (nfiles ne 0) then begin
       if obj_valid(ftpobj) then dir1 = ftpserver+dir else dir1 = dir
       if n_elements(info) eq 0 then info=''
       info = info+files[0] & if nfiles gt 1 then info = info+' to '+files[nfiles-1]
       for i=0,state.n_history-1 do begin  ;check if this entry already exists, if existed move to the top
           if dir1 ne (*(state.file_hist[i]))[2] then continue
           if n_elements(files) ne n_elements(*(state.file_hist[i]))-3 then continue
           same = 1b
           for j=0, n_elements(files)-1 do begin
               if files[j] ne (*(state.file_hist[i]))[3+j] then same=0b
           endfor
           if same then begin ;entry exists, move that entry to the top
              (*(state.file_hist[i]))[1] = info   ;update the info anyway
              if i ne 0 then state.file_hist[0:i] = shift(state.file_hist[0:i],1)
              for j=0,i do widget_control,state.fhistBut[j],set_value=dm_to_string(j,/int)+' '+(*(state.file_hist[j]))[1]
              return
           endif
       endfor
       if state.n_history lt num then begin ;empty room available
          if state.n_history ne 0 then $
          state.file_hist[1:state.n_history] = state.file_hist[0:state.n_history-1]
          state.file_hist[0] = ptr_new(['ftype_'+state.ftype,info,dir1,files])
          state.n_history = state.n_history+1
       endif else begin                    ;full
          ptr_free,state.file_hist[num-1]
          state.file_hist[1:num-1] = state.file_hist[0:num-2]
          state.file_hist[0] = ptr_new(['ftype_'+state.ftype,info,dir1,files])
       endelse
    endif
    widget_control,state.fhistmenu,sensitive=(state.n_history gt 0)
    widget_control,state.clearhist,sensitive=(state.n_history gt 0)
    for i=0,num-1 do begin
        if widget_info(state.fhistBut[i],/valid_id) then widget_control,state.fhistBut[i],/destroy
    endfor
    for i=0,state.n_history-1 do begin
        state.fhistBut[i] = widget_button(state.fhistmenu,value=dm_to_string(i,/int)+' '+(*(state.file_hist[i]))[1])
    endfor
end

;event handler
pro dm_polanalysis_event,event
    compile_opt IDL2
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=state,/no_copy
    state.open_filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,dFiles=dfiles,ftpserver=ftpserver,ftpbufferdir=ftpbufferdir
    state.save_filesel->getproperty,path=save_path

    while(strmid(open_path,0,1,/reverse_offset) eq  ([state.pathsep,'/'])[obj_valid(ftpobj)]) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    while(strmid(save_path,0,1,/reverse_offset) eq  state.pathsep) do  save_path = strmid(save_path,0,strlen(save_path)-1)
    ind = where([state.loadNSFBut,state.loadSFBut] eq event.id,cnt)
    if cnt eq 1 then begin
       if n_elements(open_file) eq 0 then begin
          ok = dialog_message('Please select '+(['NSF','SF'])[ind]+' data files on the left first.',/error,/center,dialog_parent=event.handler)
          widget_control,event.handler,set_uvalue=state,/no_copy
          return
       endif
       dm_polanalysis_setfilehistory,state,open_path,open_file,ftpobj=ftpobj,ftpserver=ftpserver,info='['+(['NSF','SF'])[ind]+']: '
    endif
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=event.handler,!error_state.msg,/error,/center)
       catch,/cancel
       if widget_info(state.cmdCol,/map) eq 0 then widget_control,state.cmdCol,/map
       if widget_info(state.tlb,/update) eq 0 then widget_control,state.tlb,/update
       widget_control,event.handler,set_uvalue=state,/no_copy
       if obj_valid(mesgobj) then obj_destroy,mesgobj
       return
    end

    if strlowcase(tag_names(event,/structure)) eq 'dm_filesel_select' then event.id = -999
    case event.id of
         state.tlb:     widget_control,event.id,scr_xsize=state.geom[0],scr_ysize=state.geom[1]
         state.doneBut: begin
                        widget_control,event.handler,set_uvalue=state,/no_copy
                        widget_control,event.handler,/destroy
                        return
                        end        
         state.clearhist:begin
                        ptr_free,state.file_hist
                        state.n_history = 0
                        dm_polanalysis_setfilehistory,state
                        end
         state.dcsBut:  if state.ftype ne 'DCS'  then begin
                        state.samptype = 0
                        dm_polanalysis_changeftype,state,ftype='DCS'
                        endif
         state.macsBut: if state.ftype ne 'MACS' then begin
                        state.samptype = 1
                        dm_polanalysis_changeftype,state,ftype='MACS'
                        endif
         state.loadpmBut:begin
                        path = state.parmdir
                        parmfile = dm_choose_file('prm,txt',dialog_parent=state.tlb,/read,path=path)
                        if strlen(parmfile) ne 0 then begin
                           dm_polanalysis_loadparm,state,parmfile
                           state.parmdir = path
                        endif
                        end
         state.savepmBut:begin
                        path = state.parmdir
                        parmfile = dm_choose_file('prm',dialog_parent=state.tlb,/write,path=path)
                        if strlen(parmfile) ne 0 then begin
                           dm_polanalysis_saveparm,state,parmfile
                           state.parmdir = path
                        endif
                        end
         state.powder:  if state.samptype ne 0 then begin
                        state.samptype = 0
                        widget_control,event.id,/set_button
                        widget_control,state.crystal,set_button=0
                        widget_control,state.loadNSFBut,set_value='Load NSF Powder Files'
                        widget_control,state.loadSFBut,set_value='Load SF Powder Files'
                        end
         state.crystal: if state.samptype ne 1 then begin
                        state.samptype = 1
                        widget_control,event.id,/set_button
                        widget_control,state.powder,set_button=0
                        widget_control,state.loadNSFBut,set_value='Load NSF Single Crystal Files'
                        widget_control,state.loadSFBut,set_value='Load SF Single Crystal Files'
                        end
         state.dcsmonsum:if state.monchoice ne 0 then begin
                        state.monchoice = 0
                        widget_control,event.id,/set_button
                        widget_control,state.dcsmonint,set_button=0
                        end
         state.dcsmonint:if state.monchoice ne 1 then begin
                        state.monchoice = 1
                        widget_control,event.id,/set_button
                        widget_control,state.dcsmonsum,set_button=0
                        end
         state.macsmoncount:if state.monchoice ne 0 then begin
                        state.monchoice = 0
                        widget_control,event.id,/set_button
                        widget_control,state.macsmontime,set_button=0
                        end
         state.macsmontime:if state.monchoice ne 1 then begin
                        state.monchoice = 1
                        widget_control,event.id,/set_button
                        widget_control,state.macsmoncount,set_button=0
                        end
         state.ftpBut:  begin
                        state.open_fileSel->set_ftpbuffer,(['',dm_define_pointer(/gettempdir)])[1-state.ncnrftp]                        
                        state.open_fileSel->getproperty,ncnrftp=ncnrftp
                        if state.ncnrftp eq ncnrftp then ok = dialog_message('Cannot establish connection to NCNR Data Repository.',/error,/center,dialog_parent=event.handler) $
                        else begin
                           state.ncnrftp = ncnrftp
                           dm_set_button,event.id,state.ncnrftp
                           if state.ncnrftp then state.open_filesel->set_path,/ftp
                        endelse                      
                        end               
         state.helpBut: begin
                        info = ['To reduce the data, first enter the 3He polarizer and analyzer cell information, '+(['neutron depolarization information, ',''])[state.ftype eq 'DCS']+$
                                'and load the NSF and SF files by selecting them from the '+state.ftype+' data file directory and press the corresponding load button. '+(['',$
                                'Only the file names and A2 angles are stored while loading the data. '])[state.ftype eq 'DCS']+'Press the Convert button to '+(['','actually read the whole '+$
                                'data file, '])[state.ftype eq 'DCS']+'reduce the data and save them in the reduced file directory.','','Use comma (,) to separate multiple cell parameters'+$
                                ([' and neutron depolarization values. The ','. The cell '])[state.ftype eq 'DCS']+'parameters can also be read from a parameter file through the File menu. '+$
                                'The unix timestamp is the number of seconds that have elapsed since 00:00:00 UTC on January 1st, 1970. The cell opacity is the value at neutron wavelength'+$
                                ' of 1 '+string('c5'XB)+', and can be calculated from the opacity at wavelength lambda by Opacity(lambda)/lambda.']
                        if state.ftype eq 'DCS' then $
                           info = [info,'','DCS sample can be powder or single crystal type. In powder type, all files will be combined. In single crystal type, only files with the same '+$
                                   'A2 angle will be combined. The current sample type is '+(['powder','single crystal'])[state.samptype]+$
                                   '. DCS monitor data are Gaussian distributed over time. The monitor count can be either sum or integrated intensity. The current choice is '+$
                                   (['sum','integrated intensity'])[state.monchoice]+'.'] $
                        else $
                           info = [info,'','MACS sample can be powder or single crystal type. In powder type, A3 will not be checked. In single crystal type, A3 will be checked and matched. '+$
                                   'The current sample type is '+(['powder','single crystal'])[state.samptype]+'. Data normalization can use either neutron monitor count or time. '+$
                                   'The current choice is '+(['monitor count','time'])[state.monchoice]+'.']
                        info[-1] = info[-1]+' They can be changed in the Options menu. Temperature and other sample enviroment information are not checked for matching NSF and SF files.'
                        info = [info,'','Yiming Qiu (yiming.qiu@nist.gov)   '+state.mod_date]
                        ok = dialog_message(info,/center,/info,dialog_parent=event.handler,title='About '+state.ftype+' Polarization Analysis')
                        end
         state.loadNSFBut: dm_polanalysis_loadfile,state,/nsf
         state.loadSFBut:  dm_polanalysis_loadfile,state,/sf
         state.convertBut:begin
                        if state.ftype ne 'DCS' then break
                        mesg = '' & tmp = ''
                        if ~ptr_valid(state.nsffilePtr) then mesg = [mesg,'NSF']
                        if ~ptr_valid(state.sffilePtr)  then mesg = [mesg,'SF']
                        if n_elements(mesg) gt 1 then mesg = ['','load '+strjoin(mesg[1:*],' and ')+' data']
                        if total(ptr_valid(state.pol_parm)) ne n_elements(state.pol_parm) then missing_pol = 1b $
                        else begin
                           n_tmp = fltarr(n_elements(state.pol_parm))
                           for i=0,n_elements(state.pol_parm)-1 do n_tmp[i]= total(finite(*state.pol_parm[i]))
                           if max(n_tmp,min=min_tmp) ne min_tmp then missing_pol = 1b
                        endelse
                        if total(ptr_valid(state.ana_parm)) ne n_elements(state.ana_parm) then missing_ana = 1b $
                        else begin
                           n_tmp = fltarr(n_elements(state.ana_parm))
                           for i=0,n_elements(state.ana_parm)-1 do n_tmp[i]= total(finite(*state.ana_parm[i]))
                           if max(n_tmp,min=min_tmp) ne min_tmp then missing_ana = 1b
                        endelse
                        if keyword_set(missing_pol) then tmp = [tmp,'polarizer']
                        if keyword_set(missing_ana) then tmp = [tmp,'analyzer']
                        if n_elements(tmp) gt 1 then mesg = [mesg,'enter all '+strjoin(tmp[1:*],' and ')+' parameters']
                        if n_elements(mesg) gt 1 then mesg = ['','Please '+strjoin(mesg[1:*],', and ')+'.']
                        if ptr_valid(state.nsffilePtr) and ptr_valid(state.sffilePtr) then begin
                           if total(abs((*state.nsffilePtr).info-(*state.sffilePtr).info)) ne 0 then begin
                              maxl = max(state.open_filesel->wavlstrlen([(*state.nsffilePtr).info[0],(*state.sffilePtr).info[0]],/nozeroend))
                              sep  = ([' ',string(9b)])[!version.os_family eq 'Windows']
                              tmp  = [state.open_filesel->formatlinestring('  NSF:'+sep,(*state.nsffilePtr).info,/wavelength,maxlength=maxl),$
                                      state.open_filesel->formatlinestring('    SF:'+sep,(*state.sffilePtr).info,/wavelength,maxlength=maxl)]
                              mesg = [mesg,'','Selected NSF and SF files are of different wavelength settings.',tmp]
                           endif
                        endif
                        if n_elements(mesg) gt 1 then begin
                           ok = dialog_message(mesg[1:*],/error,/center,dialog_parent=event.handler)
                           break
                        endif

                        a2_NSF = (*state.nsffilePtr).a2
                        a2_SF  = (*state.sffilePtr).a2
                        if state.samptype eq 0 then begin
                           a2_NSF[*] = 0.0
                           a2_SF[*]  = 0.0
                        endif else begin
                           a2_NSF = round(a2_NSF*1000.0)/1000.
                           a2_SF  = round(a2_SF*1000.0)/1000.
                        endelse
                        a2_uniq = a2_NSF[uniq(a2_NSF,sort(a2_NSF))]        
                        
                        ;read the nsf file and save the reduced file simultaneously
                        open_path = (*state.nsffilePtr).filedir
                        is_ftp    = stregex(open_path,'(https|ftp)://',/boolean,/fold_case)
                        if is_ftp then begin
                           if ~obj_valid(ftpobj) then begin ;make sure ftpobj is valid
                              state.open_filesel->set_path,(*state.nsfdataPtr).filedir
                              state.open_filesel->getproperty,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir,ftpserver=ftpserver
                           endif
                           open_path = strmid(open_path,strlen(ftpserver))
                        endif
                                         
                        mesgobj =  obj_new('dm_progress',title='Wait ...',message='Please wait',group_leader=state.tlb) 
                                                
                        sf_nomatch = [0.]
                        
                        for i=0L,n_elements(a2_uniq)-1 do begin
                            ind_nsf = where(a2_NSF eq a2_uniq[i],n_nsf)
                            tmp_sf  = min(abs(a2_SF-a2_uniq[i]),i_tmp)                ;find the closet sf a2 angle
                            ind_sf  = where(a2_SF eq a2_SF[i_tmp],n_sf)
                            if tmp_sf ne 0 then sf_nomatch = [sf_nomatch,a2_uniq[i]]  ;save no match info
                            
                            if n_elements(a2_uniq) gt 1 then num_str = dm_to_string(i+1)+'/'+dm_to_string(n_elements(a2_uniq))+' ' else num_str = ''
                            
                            ;read NSF and SF files
                            mesgobj->update,message=num_str+'loading raw nsf files'
                            dm_polanalysis_readdcs,state,(*state.nsffilePtr).filedir,(*state.nsffilePtr).filename[ind_nsf],/nsf
                            mesgobj->update,message=num_str+'loading raw sf files'
                            dm_polanalysis_readdcs,state,(*state.sffilePtr).filedir,(*state.sffilePtr).filename[ind_sf],/sf
                            
                            ;polarization reduction
                            mesgobj->update,message=num_str+'polarization calculation'
                            dm_polanalysis_reduction,state
                            
                            infile = open_path+([state.pathsep,'/'])[is_ftp]+(*state.nsfPtr).filename[0]  ;read the first NSF file
                            if is_ftp then begin
                               ok = ftpobj->GetFileContent(infile,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+(*state.nsfPtr).filename[0])
                               infile = ftpbufferdir+state.pathsep+'ftptmp'+(*state.nsfPtr).filename[0]
                            endif
                            
                            ;generate save file names
                            out_nsf = save_path+state.pathsep+'NSF_'+dm_polanalysis_filename((*state.nsfPtr).filename[0],/dcs,f_ind=i)
                            out_sf  = save_path+state.pathsep+'SF_'+dm_polanalysis_filename((*state.sfPtr).filename[0],/dcs,f_ind=i)

                            ;save files
                            mesgobj->update,message=num_str+'saving reduced files'
                            dm_polanalysis_writedcs,state,infile,out_nsf,out_sf
                            state.save_filesel->set_path      ;update the save directory
                            
                            if is_ftp then file_delete,infile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET    ;delete the temporary ftp file
                            if (~ obj_valid(mesgobj)) then break  ;stopped from progress bar
                        endfor
                        if obj_valid(mesgobj) then begin
                           if n_elements(sf_nomatch) gt 1 then $
                              ok = dialog_message('No exactly matching SF files are found for A2='+dm_to_string(sf_nomatch[1:*],sep=', ')+'. Files with the closest A2 are used.',/center,dialog_parent=state.tlb)
                           obj_destroy,mesgobj
                        endif
                        end
         else:          begin
                        ind = where(state.pol_input eq event.id,cnt)  ;polarizer parameter event
                        if cnt eq 1 then begin
                           ptr_free,state.pol_parm[ind[0]]
                           widget_control,event.id,get_value=tmp
                           if (ind[0] eq 2) and ~stregex(tmp,'^[ \t]*([0-9]*[ \t,]*)*$',/boolean) then state.pol_parm[ind[0]] = ptr_new(dm_to_number(tmp,/epoch,/double)) $
                           else state.pol_parm[ind[0]] = ptr_new(abs(dm_to_number(strsplit(tmp,', ',/extract),/double)))
                           break
                        endif
                        ind = where(state.ana_input eq event.id,cnt)  ;polarizer parameter event
                        if cnt eq 1 then begin
                           ptr_free,state.ana_parm[ind[0]]
                           widget_control,event.id,get_value=tmp
                           if (ind[0] eq 2) and ~stregex(tmp,'^[ \t]*([0-9]*[ \t,]*)*$',/boolean) then state.ana_parm[ind[0]] = ptr_new(dm_to_number(tmp,/epoch,/double)) $
                           else state.ana_parm[ind[0]] = ptr_new(abs(dm_to_number(strsplit(tmp,', ',/extract),/double)))
                           break
                        endif
                        ind = where(state.fhistBut eq event.id,cnt)   ;file history event
                        if cnt eq 1 then begin
                           dm_polanalysis_setfilehistory,state,choose=ind[0]       
                           break
                        endif
                        end
    endcase
    widget_control,event.handler,set_uvalue=state,/no_copy
end

;main program
pro dm_polanalysis,event,dataDir=dataDir,workDir=workDir,macs=macs,dcs=dcs,ncnrftp=ncnrftp
;    if keyword_set(macs) then message,'Polarizaton analysis for MACS file is still under development.'
    state = {group_leader:       0L, $   ;group leader
             tlb:                0L, $   ;top level base
             idl_version:        0e, $   ;save idl version number
             ftype:              '', $   ;file type
             mod_date:    '6/2025', $   ;latest update date
             dcsBut:             0L, $   ;DCS type
             macsBut:            0L, $   ;MACS type
             loadpmBut:          0L, $   ;load parameter button
             savepmBut:          0L, $   ;save parameter button
             fhistmenu:          0L, $   ;file history menu
             optnmenu:           0L, $   ;option menu
             monmenu:            0L, $   ;monitor type menu
             dcsmonsum:          0L, $   ;sum for monitor
             dcsmonint:          0L, $   ;integrated intensity for monitor
             macsmoncount:       0L, $   ;neutron count for monitor
             macsmontime:        0L, $   ;time for monitor
             monchoice:          0b, $   ;DCS: 0-sum 1-integrated intensity  MACS: 0-monitor count 1-time
             powder:             0L, $   ;powder button
             crystal:            0L, $   ;single crystal button
             samptype:           0b, $   ;0-powder 1-single crystal (dcs default 0, macs default 1)
             ftpBut:             0L, $
             ncnrftp:            0b, $   ;flag for allowing access to NCNR ftp server
             file_hist:  ptrarr(10), $   ;save the latest 10 file selections
             fhistBut:   lonarr(10), $   ;file history buttons
             n_history:          0s, $   ;number of histories
             clearhist:          0L, $   ;clear history button
             cmdCol:             0L, $   ;center column base
             openlabel:          0L, $   ;open file selector label
             loadSFBut:          0L, $   ;load spinflip data button
             tickSFBut:          0L, $   ;tick mark for load sf
             loadNSFBut:         0L, $   ;load non-spinflip data button
             tickNSFBut:         0L, $   ;tick mark for load nsf
             convertBut:         0L, $   ;convert button
             doneBut:            0L, $   ;done button
             helpBut:            0L, $   ;Help button
             open_filesel:obj_new(), $   ;open file selector
             save_filesel:obj_new(), $   ;save file selector
             dataDir:            '', $   ;save initial data directory
             pathsep:            '', $   ;path seperator
             parmdir:            '', $   ;save parameter file directory
             pol_parm:    ptrarr(6), $   ;polarizer parameters
             pol_input:   lonarr(6), $   ;polarizer parameter input text boxes
             ana_parm:    ptrarr(6), $   ;analyzer parameters
             ana_input:   lonarr(6), $   ;analyzerzer parameter input text boxes
             tau_parm:    ptrarr(2), $   ;tau value and tau time stamp
             tau_input:   lonarr(2), $   ;tau parameter input text boxes
             tau_lines:   lonarr(4), $   ;lines for tau_input
             nsffilePtr:  ptr_new(), $   ;a pointer to non spin-flip file info
             sffilePtr:   ptr_new(), $   ;a pointer to spin-flip file info
             nsfdataPtr:  ptr_new(), $   ;a pointer for non spin-flip raw data
             sfdataPtr:   ptr_new(), $   ;a pointer for spin-flip raw data
             nsfPtr:      ptr_new(), $   ;a pointer for the reduced nsf data
             sfPtr:       ptr_new(), $   ;a pointer for the reduced sf data
             geom:    [0L,0L,0L,0L]  $   ;xsize,ysize,xoffset,yoffset
    }
    state.ftype = 'DCS' 
    state.samptype = keyword_set(macs)
    dm_polanalysis_loadparm,state,dataDir=dataDir,workDir=workDir,/init
    if n_elements(workDir) eq 0 then cd,current=workDir
    if n_elements(dataDir) ne 0 then state.dataDir = dataDir
    if n_elements(ncnrftp) ne 0 then state.ncnrftp = keyword_set(ncnrftp)
    state.parmdir = workDir
    if keyword_set(macs) then state.ftype = 'MACS' $
    else if keyword_set(dcs)  then state.ftype = 'DCS'
    registerName = 'dm_polanalysis'
    if xregistered(registerName) then begin   ;only allow one copy to be running at one time
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0,get_uvalue=state1
       if state1.ftype ne state.ftype then begin  ;switch type
          dm_polanalysis_changeftype,state1,ftype=state.ftype
          widget_control,id,set_uvalue=state1
       endif
       return   
    endif
    
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists ;should exist if starts from dave
    if (~ exists) then begin
       file = file_which('dcs_DetPos2.dat',/INCLUDE_CURRENT_DIR)
       if float(!version.release) ge 6.0 then path = file_dirname(file,/mark_directory) $
       else begin
          id   = strpos(file,pathsep,/reverse_search)
          path = strmid(file,0,id+1)
       endelse
       defsysv,'!DAVE_AUXILIARY_DIR',path
    endif
    
    if n_elements(event) ne 0 then begin
       state.group_leader = event.top
       state.tlb = widget_base(title=title,/col,kill_notify='dm_polanalysis_Exit',group_leader=event.top,/tlb_size_event,mbar=mbar,xpad=0,ypad=1,map=0)
    endif else $
       state.tlb = widget_base(title=title,/col,kill_notify='dm_polanalysis_Exit',/tlb_size_event,mbar=mbar,xpad=0,ypad=1,map=0)
    
    state.idl_version = dm_to_number(!version.release)
    iconfile = !DAVE_AUXILIARY_DIR+['mslice_open.bmp','mslice_save.bmp']

    ;menu bar
    filemenu        = widget_button(mbar,value='File',/menu)
    state.optnmenu  = widget_button(mbar,value='Options',/menu)
    helpmenu        = widget_button(mbar,value='Help',/menu)
    ftypmenu        = widget_button(filemenu,value='File Type',/menu)
    state.dcsBut    = dm_widget_button(ftypmenu,value='DCS')
    state.macsBut   = dm_widget_button(ftypmenu,value='MACS')
    state.loadpmBut = dm_widget_button(filemenu,value='Load Parameters...',/separator,accelerator='Ctrl+L',imagefile=iconfile[0],/notchecked)
    state.savepmBut = dm_widget_button(filemenu,value='Save Parameters...',accelerator='Ctrl+S',imagefile=iconfile[1],/notchecked)   
    state.fhistmenu = widget_button(filemenu,value='Recent Files',/menu,sensitive=(state.n_history gt 0),/separator)
    for i=0,state.n_history-1 do begin
        state.fhistBut[i] = widget_button(state.fhistmenu,value=dm_to_string(i,/int)+' '+(*(state.file_hist[i]))[1])
    endfor
    state.clearhist = widget_button(filemenu,value='Clear File History',uname='clearHistory',sensitive=(state.n_history gt 0))
    state.doneBut   = widget_button(filemenu,value='Exit',/separator)
    sampmenu        = widget_button(state.optnmenu,value='Sample Type',/menu)
    state.powder    = dm_widget_button(sampmenu,value='Powder',set_button=(state.samptype eq 0))
    state.crystal   = dm_widget_button(sampmenu,value='Single Crystal',set_button=(state.samptype eq 1))
    state.monmenu   = widget_button(state.optnmenu,value='Monitor Count',/menu)
    state.ftpBut    = dm_widget_button(state.optnmenu,value='Allow NCNR Data Repository',set_button=state.ncnrftp,/separator)
    state.helpBut   = widget_button(helpmenu,value='Help',_extra=(state.idl_version ge 6.1)?{accelerator:'F1'}:{no_copy:1})
    
    ;menu bar separator for windows system
    if !version.os_family eq 'Windows' then begin
       mbar_sep     = widget_label(state.tlb,sensitive=0,/dynamic_resize,value=' ',scr_ysize=5)
       tmp          = 0
    endif else tmp  = 4
    
    row             = widget_base(state.tlb,/row,ypad=0)
    openCol         = widget_base(row,/col,ypad=0)
    state.cmdCol    = widget_base(row,/col,/align_center,ypad=0,xpad=0)
    saveCol         = widget_base(row,/col,ypad=0)
    state.openlabel = widget_label(openCol,value=state.ftype+' Data File Directory')
    label           = widget_label(saveCol,value='Reduced File Directory')
    state.open_filesel = obj_new('dm_filesel',parent=openCol,ysize=27+tmp,/frame,filter=fext,path=dataDir,group_leader=state.tlb,ncnrftp=state.ncnrftp)
    state.save_filesel = obj_new('dm_filesel',parent=saveCol,ysize=27+tmp,/frame,filter=fext,path=workDir)
    state.pathsep = dm_define_pointer(/getpathsep)
    line  = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0)
    label = widget_label(line,value='Cell Window Transmission',/align_center,sensitive=0)
    geom  = widget_info(label,/geometry)
    xsize = geom.scr_xsize/2+35 & ysize = geom.scr_ysize
    widget_control,label,scr_xsize=xsize,set_value='P o l a r i z e r'
    label = widget_label(line,value='A n a l y z e r',scr_xsize = xsize,/align_center,sensitive=0)
    label = widget_label(state.cmdCol,value=' ',scr_ysize=ysize/3)
    names = ['Initial Polarization','Final Polarization','Initial Unix Timestamp','Duration (hr)','Cell Opacity (@ 1 '+string('c5'XB)+')','Cell Window Transmission']
    for i=0,n_elements(state.pol_input)-1 do begin
        label = widget_label(state.cmdCol,value=names[i],/align_center)
        line  = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/tab_mode)
        state.pol_input[i] = widget_text(line,value=ptr_valid(state.pol_parm[i])?dm_to_string(*state.pol_parm[i],int=(i eq 2),sep=', '):'',scr_xsize=xsize,/editable,/all_events)
        label = widget_label(line,value=' ')
        state.ana_input[i] = widget_text(line,value=ptr_valid(state.ana_parm[i])?dm_to_string(*state.ana_parm[i],int=(i eq 2),sep=', '):'',scr_xsize=xsize,/editable,/all_events)
    endfor
    geom  = widget_info(label,/geometry)
    xsize1= xsize*2+geom.scr_xsize-1
    label = widget_label(state.cmdCol,value=' ',scr_ysize=ysize/2)
    for i=0,1 do begin 
        state.tau_lines[2*i] = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/tab_mode,/align_center)
        label = widget_label(state.tau_lines[2*i],value='Depolarization'+(['',' Timestamp'])[i],/align_center)
        state.tau_lines[2*i+1] = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/tab_mode,/align_center)
        state.tau_input[i] = widget_text(state.tau_lines[2*i+1],value=ptr_valid(state.tau_parm[i])?dm_to_string(*state.tau_parm[i],sep=', '):'',scr_xsize=xsize1,/editable,/all_events) 
    endfor
    label = widget_label(state.cmdCol,value=' ',scr_ysize=ysize*3/4)
    row   = widget_base(state.cmdCol,/row,xpad=0,ypad=0,/align_left,space=0,/align_bottom)
    exclbas = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,/align_left)
    state.tickNSFBut = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',/align_left,sensitive=0)
    geom  = widget_info(state.tickNSFBut,/geometry)
    xsize1= xsize*2-geom.scr_xsize
    state.loadNSFBut = widget_button(row,value='Load NSF Files',scr_xsize=xsize1)
    row   = widget_base(state.cmdCol,/row,xpad=0,ypad=0,/align_left,space=0,/align_bottom)
    exclbas = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,/align_left)
    state.tickSFBut = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',/align_left,sensitive=0)
    state.loadSFBut = widget_button(row,value='Load SF Files',scr_xsize=xsize1)
    label = widget_label(state.cmdCol,value=' ',scr_ysize=ysize*3/4)
    state.convertBut = widget_button(state.cmdCol,value='Convert')
    state.open_filesel->getproperty,ncnrftp=ncnrftp
    state.ncnrftp = ncnrftp
    dm_polanalysis_changeftype,state
    dm_center_kid, state.tlb,state.group_leader,/side   ;centering the top level base
    widget_control,state.cmdCol,/map
    widget_control,state.tlb,/update,/realize,/map
    geom = widget_info(state.tlb,/geometry)
    state.geom = [geom.scr_xsize,geom.scr_ysize,geom.xoffset,geom.yoffset]
    widget_control,state.tlb,set_uvalue=state
    xmanager,registerName,state.tlb,cleanup='dm_polanalysis_Exit',event_handler='dm_polanalysis_event',/no_block
end