; $Id: $
;#######################################################################
;
; NAME:
;  dm_filetools
;
; PURPOSE:
;  handle file type menu and file tool menu events and all data file loading related events
;
; CATEGORY:
;  dcs_mslice
;
; HISTORY:
;  02/2010: added NXSPE support - Andrei Savici (saviciat@ornl.gov)   
; 
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameter:
;   eventname: event name = widget_info(event.id,/uname)
; keyword:
;   event:     passed event structure

;files need to be included
@dm_load_dcsdata
@dm_load_macs
@dm_load_spe
@dm_load_nxspe
@dm_load_phx
@dm_load_inx
@dm_load_wand
@dm_patch
@dm_polanalysis

;combine data with same rotation angles
;parameters:
;   ang:         [nang] rotation angle
;   qty,dqty:    [*,nang] or [*,*,nang] data
;   weight:      [nang] weight of the data
;keywords:
;   extraavg:    [nang,*], optional extra data to be combined, the operation is averaging
;   extrasum:    [nang,*], optional extra data to be combined, the operation is summing
;   checkfinite: flag for checking finite values of qty
;   tolerance:   tolerance value for comparing angles
pro dm_combine_psi,ang,qty,dqty,weight,extraavg=extraavg,extrasum=extrasum,checkfinite=checkfinite,tolerance=tolerance
    WIDGET_CONTROL,/HOURGLASS
    if n_elements(ang) lt 2 then return   ;no action required
    if n_elements(tolerance) eq 0 then tolerance = 0.0 else tolerance = abs(tolerance[0])
    type = size(ang,/type)
    if (type ne size('a',/type)) and (tolerance ne 0) then newang = round(ang/tolerance+(0.0001d),/L64)*tolerance else newang = ang
    ind  = sort(newang) & nang = n_elements(newang) 
    uind = uniq(newang[ind]) & nun = n_elements(uind)
    if n_elements(extraavg) ne 0 then check_extraavg = (n_elements(extraavg[*,0]) eq nang) else check_extraavg = 0b
    if n_elements(extrasum) ne 0 then check_extrasum = (n_elements(extrasum[*,0]) eq nang) else check_extrasum = 0b
    if nun ne nang then begin
       ndim   = size(qty,/n_dimensions)
       newang = newang[ind]
       weight = weight[ind,*]
       if check_extraavg  then extraavg = extraavg[ind,*]
       if check_extrasum  then extrasum = extrasum[ind,*]
       if ndim eq 2 then begin      ;single E
          qty  = qty[*,ind]
          dqty = dqty[*,ind]
       endif else begin             ;multiple E
          qty  = qty[*,*,ind]
          dqty = dqty[*,*,ind]
       endelse
       dqty = dqty^2                ;error bar
       j = 0L
       for i=0L,nun-1 do begin
           qtmp = (0d)
           etmp = (0d)
           wgt  = (0d)
           if check_extraavg then extavg = 0.0
           if check_extrasum then extsum = 0.0
           while(newang[j] eq newang[uind[i]]) do begin
              if ndim eq 2 then begin     ;single E
                 if keyword_set(checkfinite) then begin
                    mask = where(finite(qty[*,j],/nan),mskcount)
                    ind  = where(finite(qtmp[*],/nan),count)
                    if count gt 0 then begin
                       qtmp[ind] = qty[ind,j]*wgt
                       etmp[ind] = dqty[ind,j]*((wgt+weight[j])^2-weight[j]^2)
                    endif
                    if (mskcount gt 0) and (wgt ne 0) then begin
                       qty[mask,j]  = qtmp[mask]/wgt
                       dqty[mask,j] = etmp[mask]*((wgt+weight[j])^2-wgt^2)/(wgt^2*weight[j]^2) 
                    endif
                 endif
                 qtmp = temporary(qtmp)+qty[*,j]*weight[j]
                 etmp = temporary(etmp)+dqty[*,j]*(weight[j]^2)
              endif else begin            ;multiple E
                 if keyword_set(checkfinite) then begin
                    ii = 0L
                    while(total(finite(qty[ii,*,j])) eq 0) do ii=ii+1   ;in case the first time channel is masked
                    mask = where(finite(qty[ii,*,j],/nan),mskcount)
                    ii = 0L
                    while(total(finite(qtmp[ii,*])) eq 0) do ii=ii+1    ;in case the first time channel is masked
                    ind = where(finite(qtmp[ii,*],/nan),count) 
                    if count gt 0 then begin
                       qtmp[*,ind] = qty[*,ind,j]*wgt
                       etmp[*,ind] = dqty[*,ind,j]*((wgt+weight[j])^2-weight[j]^2)
                    endif
                    if (mskcount gt 0) and (wgt ne 0) then begin
                       qty[*,mask,j]  = qtmp[*,mask]/wgt
                       dqty[*,mask,j] = etmp[*,mask]*((wgt+weight[j])^2-wgt^2)/(wgt^2*weight[j]^2) 
                    endif
                 endif 
                 qtmp = temporary(qtmp)+qty[*,*,j]*weight[j]
                 etmp = temporary(etmp)+dqty[*,*,j]*(weight[j]^2)
              endelse
              wgt = temporary(wgt)+weight[j]
              if check_extraavg then extavg = extavg+extraavg[j,*]*weight[j]
              if check_extrasum then extsum = extsum+extrasum[j,*]
              j   = j+1L
              if j eq nang then break
           endwhile
           if wgt eq 0 then wgt=1.0
           weight[i] = wgt
           if check_extraavg then extraavg[i,*] = extavg/wgt
           if check_extrasum then extrasum[i,*] = extsum
           etmp = sqrt(temporary(etmp))
           if ndim eq 2 then begin    ;single E
              qty[*,i]  = qtmp/wgt
              dqty[*,i] = etmp/wgt
           endif else begin           ;multiple E
              qty[*,*,i]  = qtmp/wgt
              dqty[*,*,i] = etmp/wgt
           endelse 
       endfor
       qtmp = 0 & etmp = 0  ;free up the memory
       if ndim eq 2 then begin        ;single E
          qty  = qty[*,0:nun-1]
          dqty = dqty[*,0:nun-1]
       endif else begin               ;multiple E
          qty  = qty[*,*,0:nun-1]
          dqty = dqty[*,*,0:nun-1]
       endelse
       newang = newang[uind]
       weight = weight[0:nun-1]
       if check_extraavg then extraavg = extraavg[0:nun-1,*]
       if check_extrasum then extrasum = extrasum[0:nun-1,*]
    endif
    if type ne size('a',/type) then ang = fix(newang,type=type) $
    else ang = newang
end

;combine macs data with same A3, Kidney, and energy, data have not been monitor normalized, histo data are histo_weight normalized if histo_weight is present
;parameters:
;   qty,dqty:     [2,ndet,ndat] or [2*ndet,ntchan,ndat] for histo data,  
;   weight:       [ndat,2] or [ndat,3 ] for histo data, weight of the data
;   cfx:          [ndat] flag of cfx status
;   A3,kidney,Ei,Ef: [ndat] 
;keywords:
;   montype:      monitor type, either 0-time or 1-monitor count
;   atolerance:   tolerance value for comparing A3 angle, default 0.01
;   ktolerance:   tolerance value for comparing kidney angle, default 0.1
;   etolerance:   tolerance value for comparing energy, default 0.01
;   viewt:        if set, temperature is regarded as a one of the viewing axis, default is not
;   ptai:         [ndat]
;   temperature:  [ndat,3]
;   hfield:       [ndat], magnetic field, if present, it is a criterion for combination
;   specify:      [ndat], specified quantity, if present, it is a criterion for combination
;   histo_weight: [ntchan,ndat], for histogram data
;   rawint:       if set, data are raw histogram data
pro dm_combine_macs,qty,dqty,weight,cfx,A3,kidney,Ei,Ef,montype=montype,ptai=ptai,temperature=temperature,hfield=hfield,specify=specify,histo_weight=histo_weight,etolerance=etolerance,$
    atolerance=atolerance,ktolerance=ktolerance,viewt=viewt,checkfinite=checkfinite,rawint=rawint
    WIDGET_CONTROL,/HOURGLASS
    ndat = n_elements(kidney)
    is_histo = (n_elements(histo_weight) gt ndat) ;qty is histo_weight normalized unless rawint is set
    if n_elements(weight[0,*]) eq 1 then weight = (is_histo)?[[weight],[weight],[weight]]:[[weight],[weight]]
    if n_elements(montype) eq 0 then montype = 1 else montype = (montype)<1
    if (ndat lt 1) or (n_elements(cfx) ne ndat) or (n_elements(A3) ne ndat) or (n_elements(Ei) ne ndat) or (n_elements(Ef) ne ndat) then return  
    if n_elements(atolerance) eq 0 then atolerance = 0.01 else atolerance = abs(atolerance)
    if n_elements(ktolerance) eq 0 then ktolerance = 0.1  else ktolerance = abs(ktolerance)
    if n_elements(etolerance) eq 0 then etolerance = 0.01 else etolerance = abs(etolerance)
    tmp = strtrim(string(cfx),2)+'_'+((etolerance gt 0)?(strtrim(string(round(Ei/etolerance+(0.1d),/L64)),2)+'_'+strtrim(string(round(Ef/etolerance+(0.1d),/L64)),2)):$
          (strtrim(string(Ei),2)+'_'+strtrim(string(Ef),2)))+'_'+((atolerance gt 0)?strtrim(string(round(A3/atolerance+(0.1d),/L64)),2):strtrim(string(A3),2))+'_'+$
          ((ktolerance gt 0)?strtrim(string(round(kidney/ktolerance+(0.1d),/L64)),2):strtrim(string(kidney),2))
    pyn = (n_elements(ptai) eq ndat)
    if pyn then tmp = tmp+'_'+strtrim(string(ptai),2)
    tyn = (n_elements(temperature) eq ndat*3)
    if keyword_set(viewt) and tyn then begin
       for i=0L,ndat-1 do tmp[i] = tmp[i]+'_'+dm_to_string(temperature[i,*],resolution=3,separator='_')
    endif
    if n_elements(hfield) eq ndat then begin
       tmp = tmp+'_'+strtrim(string(hfield),2)
       hyn = 1b
    endif else hyn = 0b
    if n_elements(specify) eq ndat then begin
       tmp = tmp+'_'+strtrim(string(specify),2)
       syn = 1b
    endif else syn = 0b
    ind    = sort(tmp)
    uind   = uniq(tmp[ind]) 
    nun    = n_elements(uind)
    if nun eq ndat then return
    dims   = size(qty,/dim)
    qtytyp = size(qty,/type)
    tmp    = tmp[ind]
    qty    = qty[*,*,ind] 
    dqty   = dqty[*,*,ind]
    weight = weight[ind,*]
    A3     = A3[ind]
    kidney = kidney[ind]
    Ei     = Ei[ind]
    Ef     = Ef[ind]
    cfx    = cfx[ind[uind]]
    if tyn then temperature = temperature[ind,*]
    if pyn then ptai = ptai[ind[uind]]
    if hyn then hfield = hfield[ind[uind]]
    if syn then specify = specify[ind[uind]]
    if keyword_set(checkfinite) then begin
       ind_nan = where(finite(qty,/nan),count)
       if count gt 0 then begin
          qty[ind_nan] = 0.0
          dqty[temporary(ind_nan)] = 0.0
       endif
    endif
    if is_histo then begin  ;histo data
       histo_weight = histo_weight[*,ind]
       if ~keyword_set(rawint) then begin ;revert to raw intensity
          for i=0,dims[0]-1 do begin
              qty[i,*,*]  = qty[i,*,*]*histo_weight
              dqty[i,*,*] = dqty[i,*,*]*histo_weight
          endfor
       endif
    endif
    dqty = dqty^2                ;error bar
    j = 0LL
    for i=0LL,nun-1 do begin
        qtytmp = 0.0
        errtmp = 0.0
        a3tmp  = 0.0
        kidtmp = 0.0
        eitmp  = 0.0
        eftmp  = 0.0
        wgt    = fltarr(3)
        if tyn then ttmp = dblarr(3)
        if is_histo then histmp = fltarr(dims[1])
        while(tmp[j] eq tmp[uind[i]]) do begin
           if is_histo then histmp = temporary(histmp)+weight[j,montype]*histo_weight[*,j]
           qtytmp = temporary(qtytmp)+qty[*,*,j]
           errtmp = temporary(errtmp)+dqty[*,*,j]
           a3tmp  = temporary(a3tmp)+A3[j]*weight[j,montype]
           kidtmp = temporary(kidtmp)+kidney[j]*weight[j,montype]
           eitmp  = temporary(eitmp)+Ei[j]*weight[j,montype]
           eftmp  = temporary(eftmp)+Ef[j]*weight[j,montype]
           if tyn then ttmp = temporary(ttmp)+temperature[j,*]*weight[j,montype]
           wgt    = temporary(wgt)+weight[j,*]
           j      = j+1LL
           if j eq ndat then break
        endwhile
        wgt0 = wgt[montype]
        if wgt0 eq 0 then wgt0 = 1.0
        weight[i,*] = wgt
        if is_histo then begin
           histmp = temporary(histmp)/wgt0
           histo_weight[*,i] = histmp
           histmp = replicate(1.0,dims[0])#temporary(histmp)
           tmp_ind1 = where(histmp eq 0,tmp_cnt1)
           if keyword_set(rawint) then begin
              tmp_dat = temporary(qtytmp)
              tmp_err = sqrt(temporary(errtmp))
           endif else begin
              if tmp_cnt1 ne 0 then histmp[tmp_ind1] = 1.0
              tmp_dat = temporary(qtytmp)/histmp
              tmp_err = sqrt(temporary(errtmp))/temporary(histmp)
           endelse
           if tmp_cnt1 ne 0 then begin
              tmp_dat[tmp_ind1] = !values.d_nan
              tmp_err[tmp_ind1] = !values.d_nan
           endif
           qty[*,*,i]  = temporary(tmp_dat)
           dqty[*,*,i] = temporary(tmp_err)
        endif else begin
           qty[*,*,i]  = temporary(qtytmp)
           dqty[*,*,i] = sqrt(temporary(errtmp)) 
        endelse
        A3[i]     = temporary(a3tmp)/wgt0
        kidney[i] = temporary(kidtmp)/wgt0
        Ei[i]     = temporary(eitmp)/wgt0
        Ef[i]     = temporary(eftmp)/wgt0
        if tyn then temperature[i,*] = temporary(ttmp)/wgt0
    endfor
    nun  = nun-1
    qty  = qty[*,*,0:nun]
    dqty = dqty[*,*,0:nun]
    A3   = A3[0:nun]
    Ei   = Ei[0:nun]
    Ef   = Ef[0:nun]
    kidney = kidney[0:nun]
    weight = weight[0:nun,*]
    if tyn then temperature = temperature[0:nun,*]
    if is_histo then histo_weight = histo_weight[*,0:nun]
end

;construct legend from comment string
function dm_getlegend,comment,legend
    if n_elements(legend) eq 0 then legend = ''
    if n_elements(comment) eq 0 then return,legend
    tmpcomment = ' '+comment+' '
    units = '(mk)?(k)?(tesla)?(t)?(gauss)?(kpa)?(mpa)?(gpa)?(pa)?(mtorr)?(torr)?(mbar)?(kbar)?(bar)?(psi)?(ksi)?(atm)?(mev)?(ev)?(v)?(kv)?(amp)?' ;unit string
    head  = 'tbhpeix'
    sep   = ', ;/'+string(9b)
    repeat begin 
       tmp = stregex(tmpcomment,'['+sep+']+['+head+']? *[=~<>]? *[0-9p.-]+ *'+units+'['+sep+']+',length=len,/subexpr,/fold_case)
       ind = where(len gt 0,count)
       if count le 1 then begin
          tmp = stregex(tmpcomment,'['+sep+']+['+head+']? *[=~<>] *[^ ]+ *'+units+'['+sep+']+',length=len,/fold_case)
          ind = where(len gt 0,count)
          if count eq 1 then begin
             legend = [legend,dm_strtrim(strmid(tmpcomment,tmp[0],len[0]),2,trim=sep)]
             tmpcomment = ' '+strmid(tmpcomment,0,tmp[0])+' '+strmid(tmpcomment,tmp[0]+len[0])+' '
          endif else begin
             tmp = stregex(tmpcomment,'base *t *$',length=len,/fold_case)
             ind = where(len gt 0,count)
             if count eq 1 then begin
                legend = [legend,dm_strtrim(strmid(tmpcomment,tmp[0],len[0]),2,trim=sep)]
                tmpcomment = ' '+strmid(tmpcomment,0,tmp[0])+' '+strmid(tmpcomment,tmp[0]+len[0])+' '
             endif else begin
                tmp = stregex(tmpcomment,'[=~<>]?[0-9p.-]+ *'+units+'['+sep+']+',length=len,/subexpr,/fold_case)
                ind = where(len gt 0,count) 
                if count eq 1 then tmpcomment = ' '+strmid(tmpcomment,0,tmp[0])+' '+strmid(tmpcomment,tmp[0]+len[0])+' '
             endelse
          endelse
       endif
       if count gt 1 then begin
          for i=1,count-1 do begin
              tmp1 = strtrim(strmid(tmpcomment,0,tmp[ind[i]]),2)
              if strlowcase(strmid(tmp1,0,1,/reverse)) ne 'p' then begin ;if p is part of a number, like 1p8, it can't be alone
                 legend = [legend,dm_strtrim(strmid(tmpcomment,tmp[0],len[0]),2,trim=sep)]
                 break
              endif
          endfor
          tmpcomment = ' '+strmid(tmpcomment,0,tmp[0])+' '+strmid(tmpcomment,tmp[0]+len[0])+' '
       endif
    endrep until count lt 1
    if n_elements(legend) gt 1 then begin  ;compile legend string according to their positions
       ind = where(strlen(legend) gt 0,count)
       if count gt 0 then tmplegend = legend[ind] else tmplegend = legend[0]
       count = n_elements(tmplegend)
       pos = lonarr(count)
       for i=0,count-1 do pos[i] = strpos(comment,tmplegend[i])
       ind = sort(pos) & legend = ''
       for i=0,count-1 do legend = legend+((strlen(legend) gt 0)?', ':'')+tmplegend[ind[i]]
       legend = strtrim(legend,2)
    endif
    if strlen(legend) eq 0 then begin      ;use the last word as the legend if no legend found based on preset criterion
       tmp = strsplit(comment,', ;'+string(9b),/extract)
       legend = tmp[(0>(n_elements(tmp)-1))]
    endif
    return,legend
end

;my version of strtrim, trim all characters specified in trim keyword
function dm_strtrim,instring,flag,trim=trim
    if n_elements(trim) eq 0 then return,strtrim(instring,flag)
    if n_elements(flag) eq 0 then flag = 0
    outstring = instring 
    case flag[0] of 
       0:    while stregex(outstring,'['+trim+']+$',/boolean) do outstring = strmid(outstring,0,strlen(outstring)-1)
       1:    while stregex(outstring,'^['+trim+']+',/boolean) do outstring = strmid(outstring,1)
       2:    begin
             while stregex(outstring,'['+trim+']+$',/boolean) do outstring = strmid(outstring,0,strlen(outstring)-1)
             while stregex(outstring,'^['+trim+']+',/boolean) do outstring = strmid(outstring,1)
             end
       else: return,strtrim(instring,flag)
    endcase
    return,outstring
end

;load an angle file for script operation
function dcs_mslice::dm_load_angle,angle,directory=directory
    if size(angle[0],/type) eq 7 then begin
       if n_elements(directory) ne 0 then angle = directory+self.pathsep+angle
       openr,unit,angle,/get_lun
       tmp = '' & ang = ''
       while(~ eof(unit)) do begin
         readf,unit,tmp
         tmp = strtrim(tmp,2)
         if strlen(tmp) ne 0 then begin
            tmp1 = strsplit(tmp,' ,;'+string(9b),/extract,count=count1)
            if count1 ne 0 then ang = [ang,tmp1]
         endif
       endwhile
       free_lun,unit
       if n_elements(ang) gt 1 then ang = dm_to_number(ang[1:*]) else ang=0
    endif else ang = angle
    return,ang
end

function dcs_mslice::dm_filetools,eventname,angle=angle,directory=directory,eief=eief,event=event,filename=filename,init=init,$
    keep_extravaxis=keep_extravaxis,pdt_reset=pdt_reset,phxfile=phxfile,scriptmode=scriptmode,emptset=emptset
    
    handled = 1b    ;event handled flag
    if keyword_set(scriptmode) then event = {WIDGET_BUTTON, ID:0L, TOP:self.tlb, HANDLER:self.tlb,SELECT:1}
    if strmid(eventname,0,6) eq 'ftype_' then begin ;file type selection event
       type = dm_to_number(strmid(eventname,6,strlen(eventname)-6),/int)
       eventname = 'ftype_'
    endif
    
    is_dcs  = (self.instrname eq 'dcs')
    is_macs = (self.instrname eq 'macs')
    is_wand = (self.instrname eq 'wand')
    
    ;the following events need to have files selected or specified in script mode
    tmp_events = ['emptbut','bgflbut','bgtcbut','bgdkbut','vandbut','vandescanbut','loadbut','addbut']
    tmp_fnames = ['empty can','background','background','dark count','vanadium','two vanadium','data','data']
    ind = where(tmp_events eq strlowcase(eventname),count)
    if count ne 0 then begin
       self.filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,ftpserver=ftpserver
       if obj_valid(ftpobj) then pathsep = '/' else pathsep = self.pathsep
       if keyword_set(scriptmode) and n_elements(filename) ne 0  then open_file=filename
       if keyword_set(scriptmode) and n_elements(directory) ne 0 then open_path=directory
       if keyword_set(scriptmode) and n_elements(phxfile) ne 0 then phxfile=open_path+self.pathsep+phxfile
       n_file = n_elements(open_file)
       if n_file eq 0 then begin
          mesg = 'Please select '+tmp_fnames[ind[0]]+' files first.'
       endif else if (tmp_events[ind[0]] eq 'vandescanbut') and (n_file ne 2) then begin
          mesg = 'Please select '+(['only two vanadium files','two vanadium files first'])[n_file eq 1]+', each measured at a differnt kidney angle.'
       endif
       if n_elements(mesg) ne 0 then begin
          ok = dialog_message(dialog_parent=self.tlb,mesg,/error,/center)
          if obj_valid(ftpboj) then ftpobj->CloseConnections
          return,handled
       endif
       bkgrmenu = widget_info(self.tlb,find_by_uname='bkgrMenu')
    endif else n_file = 0
    
    WIDGET_CONTROL,/HOURGLASS
    
    case strlowcase(eventname) of
        'emptbut':begin  ;load empty can file(s)
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs_emptycan,open_path,open_file,parent=event.handler,error=error,info=info,$
                        comment=comment,qty=qty,dqty=dqty,ang=ang,weight=weight,adddata=ptr_valid(self.dataStrPtr),ftpobj=ftpobj
                    end
              'macs':begin
                    self->dm_load_macs,open_path,open_file,a3=a3,deteff=deteff,ei=ei,ef=ef,error=error,info=info,cfx=cfx,comment=comment,kidney=kidney,parent=event.handler,$
                        qty=qty,dqty=dqty,title=title,temperature=temperature,weight=weight,histo_weight=histo_weight,montype=montype,histo_nchan=histo_nchan,$
                        histo_width=histo_width,histo_t0=histo_t0,ftpobj=ftpobj
                    end
              'spe':begin     ;spe file
                    e_range = self.e_range
                    if n_elements(angle) ne 0 then ang=self->dm_load_angle(angle,directory=directory)
                    self->dm_load_spe,open_path,open_file,parent=event.handler,error=error,weight=weight,$
                        qty=qty,dqty=dqty,ang=ang,t_chan=self.e_bin[self.samp_typ],e_range=e_range,$
                        eint_yn=self.eint_yn,avgsum=self.bin_avgsum,diffuse=(self.samp_typ ge 2),$
                        title='loading empty can file...',diffetyp=(self.samp_typ eq 3),$
                        adddata=ptr_valid(self.dataStrPtr),scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) and (~ptr_valid(self.dataStrPtr)) then self.e_range=e_range
                    end
              'nxspe':begin   ;nxspe file
                    e_range = self.e_range
                    self->dm_load_nxspe,open_path,open_file,parent=event.handler,error=error,weight=weight,$
                        qty=qty,dqty=dqty,ang=ang,t_chan=self.e_bin[self.samp_typ],e_range=e_range,$
                        eint_yn=self.eint_yn,avgsum=self.bin_avgsum,diffuse=(self.samp_typ ge 2),$
                        title='loading empty can file...',diffetyp=(self.samp_typ eq 3),$
                        adddata=ptr_valid(self.dataStrPtr),scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) and (~ptr_valid(self.dataStrPtr)) then self.e_range=e_range
                    end
              'inx':begin     ;inx file
                    e_range = self.e_range
                    self->dm_load_inx,open_path,open_file,parent=event.handler,error=error,weight=weight,$
                        qty=qty,dqty=dqty,ang=ang,t_chan=self.e_bin[self.samp_typ],e_range=e_range,$
                        eint_yn=self.eint_yn,avgsum=self.bin_avgsum,diffuse=(self.samp_typ ge 2),$
                        title='loading empty can file...',diffetyp=(self.samp_typ eq 3),comment=comment,$
                        adddata=ptr_valid(self.dataStrPtr),scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) and (~ptr_valid(self.dataStrPtr)) then self.e_range=e_range
                    end
              'wand':begin    ;wand file
                    self->dm_load_wand,open_path,open_file,ang=ang,error=error,parent=event.handler,qty=qty,dqty=dqty,temperature=temperature,$
                        weight=weight,scriptmode=scriptmode
                    end   
                 else: error = 1b ;do nothing
            endcase
            if keyword_set(error) or (n_elements(qty) eq 0) then break
            if (~keyword_set(scriptmode)) and (~self.ftoolflag[5]) and ((n_elements(a3) gt 1) or (n_elements(ang) gt 1)) then begin  ;check a3 tolerance
               if n_elements(a3) gt 1 then tmp_a3 = a3[sort(a3)] else tmp_a3 = ang[sort(ang)]
               tmp_a3 = round((tmp_a3[1:*]-tmp_a3[0:(n_elements(tmp_a3)-1)])*1e2)/1e2
               tmp_ind = where(tmp_a3 gt 0,tmp_cnt)
               if tmp_cnt gt 0 then begin
                  tmp_min = min(tmp_a3[tmp_ind])
                  if ([self.psi_tolerance[0],self.macs_tolerance[2,0]])[is_macs] gt tmp_min then begin
                     ok = dialog_message(dialog_parent=self.tlb,'The minimum '+(['psi','A3'])[is_macs]+' step is '+dm_to_string(tmp_min)+$
                          ', smaller than the empty can subtraction tolerance value of '+dm_to_string(([self.psi_tolerance[0],self.macs_tolerance[2,0]])[is_macs])+$
                          '. You can change the tolerance value at Parameter->Change Empty Can Subtraction Tolerance Value menu.',/info,/center)
                     self.ftoolflag[5] = 1b
                  endif
               endif
            endif
            if n_elements(emptset) eq 0 then emptset=0 else emptset=emptset[0]
            ptr_free,self.bgdata[emptset]
            if is_macs then begin
               if n_elements(histo_weight) eq 0 then histo_weight = 1.0
               if n_file gt 1 then dm_combine_macs,qty,dqty,weight,cfx,A3,kidney,Ei,Ef,ptai=ptai,histo_weight=histo_weight,etolerance=self.macs_tolerance[0,1],$
                  ktolerance=self.macs_tolerance[1,1],atolerance=self.macs_tolerance[2,1],checkfinite=self.ftypecheckmask[0],montype=self.mon_typ[1],$
                  rawint=(keyword_set(self.macshistomode) and self.macs_rawintn)
               if self.macshistomode eq 2 then self.macshistot0[1] = histo_t0
               self.bgdata[emptset] = ptr_new({qty:temporary(qty),err:temporary(dqty),rotation:temporary(A3),kidney:temporary(kidney),weight:temporary(weight),$
                  histo_weight:temporary(histo_weight),cfx:temporary(cfx),Ei:temporary(Ei),Ef:temporary(Ef)},/no_copy)                  
            endif else begin
               if n_elements(ang) eq 0 then ang=0
               if (self.samp_typ eq 3) and (self.e_bin[4] gt 1) then begin
                  tmp = size(qty,/dimension)
                  dm_grid_ebin,self.e_bin[4],qty,dqty,fltarr(tmp[0]),avgsum=self.bin_avgsum
               endif
               if n_elements(weight) ne n_elements(ang) then weight = fltarr(n_elements(ang))+1.0
               ;combine data with the same angles
               dm_combine_psi,ang,qty,dqty,weight,checkfinite=self.ftypecheckmask[1],tolerance=self.psi_tolerance[1]
               self.bgdata[emptset] = ptr_new({qty:temporary(qty),err:temporary(dqty),rotation:temporary(ang),weight:temporary(weight)},/no_copy)
            endelse
            if ptr_valid(self.bgdata[0]) and ptr_valid(self.bgdata[1]) then begin
               n_emp = total(ptr_valid(self.bgdata[1:*]))
               if ptr_valid(self.ssfacPtr) then ssfac_str = 'read from '+file_basename((*self.ssfacPtr).file) else ssfac_str = 'ssfac = '+dm_to_string(self.ssfac)
               repeat begin
                  tmp = dm_dialog_input('empty can '+dm_to_string(1+indgen(n_emp+1))+' factor:',/float,default=self.empcanfactor[0:n_emp],$
                        info=[(['','I=I_samp-ssfac*('+strjoin('factor'+dm_to_string(indgen(1+n_emp)+1)+'*I_can'+dm_to_string(indgen(1+n_emp)+1),' + ')+'),'])[n_emp gt 0],$
                        'ssfac: self-shielding factor, currently '+ssfac_str+'.'],dialog_parent=self.tlb)
                  if total(finite(tmp)) eq 1+n_emp then begin
                     self.empcanfactor[0:n_emp] = tmp
                     ok2 = 1b
                  endif
               endrep until keyword_set(ok2)
            endif
            if n_elements(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' empty can data '+open_file[0]+((n_file gt 1)?'...':'')
            self->my_widget_control,'bgclBut',bkgrmenu,/sensitive
            dm_set_button,widget_info(bkgrmenu,find_by_uname='emptBut'),ptr_valid(self.bgdata[0])
            if self.extraempcan_yn then begin
               self->my_widget_control,'emfcBut',widget_info(self.tlb,find_by_uname='parmMenu'),sensitive=ptr_valid(self.bgdata[1])
               for i=2,n_elements(self.bgdata) do dm_set_button,widget_info(bkgrmenu,find_by_uname='emptBut'+dm_to_string(i)),ptr_valid(self.bgdata[i-1]),sensitive=ptr_valid(self.bgdata[i-2])
            endif
            if self.idl_version ge 5.6 then self->my_widget_control,'clearDat',tooltip='Background data will not be cleared.'
          end
        'bgflbut':begin  ;load detector dependent background file(s)
            ;quest for the energy range
            default = dm_to_string(self.bgeran)
            repeat begin
                new = dm_dialog_input(['from:','to:'],title='Energy range',xsize=100,default=default,/float,dialog_parent=self.tlb,info='In units of meV')
                if total(finite(new)) eq 2 then begin
                   if new[1] gt new[0] then begin
                      self.bgeran = new
                      ok2 = 1b
                   endif
                endif
            endrep until keyword_set(ok2)
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs_detbackground,open_path,open_file,parent=event.handler,error=error,comment=comment,qty=qty,dqty=dqty,ftpobj=ftpobj
                    end
              'spe':begin     ;spe file
                    self->dm_load_spe_detbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty
                    end
              'nxspe':begin   ;nxspe file
                    self->dm_load_nxspe_detbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty
                    end   
              'inx':begin     ;inx file
                    self->dm_load_inx_detbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty,comment=comment
                    end
              else: error = 1b ;do nothing
            endcase
            if keyword_set(error) then break
            if n_elements(qty) ne 0 then begin ;background file
               ptr_free,self.bgdetsPtr
               self.bgdetsPtr = ptr_new(qty)
               if n_elements(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' det. background data'
               self->my_widget_control,'bgclBut',bkgrmenu,/sensitive
               dm_set_button,widget_info(bkgrmenu,find_by_uname='bgflBut'),1
               if self.idl_version ge 5.6 then self->my_widget_control,'clearDat',tooltip='Background data will not be cleared.'
            endif
          end
        'bgtcbut':begin  ;load time channel background file(s)
            default = self.dcor_eqran & ok = 1b
            eqran   = dm_dialog_input(['E_start (meV):','E_end (meV):','2theta_start:','2theta_end:'],$
                        title='Specify E, 2theta range:',xsize=80,default=default,$
                        info=['The averaged intensity in the specified two theta range',$
                              'will be used as the background for the time channels in',$
                              'the specified E range.'],dialog_parent=self.tlb)
            message = 'Invalid input.'
            if total(finite(eqran)) ne 4 then $
               ok = 0b $
            else if (eqran[0] ge eqran[1]) or (eqran[2] ge eqran[3]) then $
               ok = 0b $
            else if ptr_valid(self.detPosPtr) then begin
               det_tt = (*self.detPosPtr).two_theta
               index  = where((abs(det_tt) ge eqran[2]) and (abs(det_tt) le eqran[3]),count)
               if count eq 0 then begin
                  ok = 0b
                  message = 'Two theta range is incompatible with the detector angles.'
               endif
            endif
            if (~ ok) then begin
               message = [message,'The background file'+((n_file gt 1)?'s are':' is')+' not loaded.']
               ok = dialog_message(message,dialog_parent=self.tlb,/center)
               break
            endif
            self.dcor_eqran = eqran
            widget_control,/hourglass
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs_tchanbackground,open_path,open_file,parent=event.handler,error=error,comment=comment,qty=qty,dqty=dqty,emean=emean,ftpobj=ftpobj
                    end
              'spe':begin     ;spe file
                    self->dm_load_spe_tchanbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty,emean=emean
                    end
              'nxspe':begin   ;nxspe file
                    self->dm_load_nxspe_tchanbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty,emean=emean
                    end   
              'inx':begin     ;inx file
                    self->dm_load_inx_tchanbackground,open_path,open_file,parent=event.handler,error=error,qty=qty,dqty=dqty,comment=comment,emean=emean
                    end
              else: error = 1b ;do nothing
            endcase
            if keyword_set(error) then break
            if (~ ptr_valid(self.detPosPtr)) then begin
               ok = dialog_message('The detector information is missing. Background data is discarded.',dialog_parent=self.tlb,/center)
               break
            endif
            if n_elements(qty) ne 0 then begin ;background file
               if (self.samp_typ eq 3) and (self.e_bin[4] gt 1) then begin
                  dm_grid_ebin,self.e_bin[4],qty,dqty,emean,avgsum=self.bin_avgsum
               endif
               det_tt = (*self.detPosPtr).two_theta
               ;for SPE and NXSPE file type, check masks
               if self.ftypecheckmask[1] then begin
                  i = 0L
                  while(total(finite(qty[i,*])) eq 0) do i=i+1    ;in case the first time channel is masked
                  index1 = where(finite(qty[i,*]),count1)
                  if count1 ne n_elements(det_tt) then begin
                     qty = qty[*,index1]
                     det_tt = det_tt[index1]
                  endif
                  index1 = 0
               endif
               index = where((abs(det_tt) ge eqran[2]) and (abs(det_tt) le eqran[3]),count)
               if count eq 0 then begin
                  ok = 0b
                  ok = dialog_message('Two theta range is incompatible with the detector angles.',dialog_parent=self.tlb,/center)
                  break
               endif
               qty = qty[*,index] 
               if count gt 1 then qty = total(qty,2)/count
               ptr_free,self.bgtchnPtr
               self.bgtchnPtr = ptr_new({emean:emean,qty:qty})
               if n_elements(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' tchan. background data'
               self->my_widget_control,'bgclBut',bkgrmenu,/sensitive
               dm_set_button,widget_info(bkgrmenu,find_by_uname='bgtcBut'),1
               if self.idl_version ge 5.6 then self->my_widget_control,'clearDat',tooltip='Background data will not be cleared.'
            endif
          end
        'bgdkbut':begin  ;shutter closed dark count file(s)
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs_darkcount,open_path,open_file,parent=event.handler,error=error,comment=comment,bgrate=bgrate,dbgrate=dbgrate,ftpobj=ftpobj
                    end
              'spe':begin     ;spe file
                    ok = dialog_message('Not implemented.',dialog_parent=self.tlb,/center)
                    error = 1b
                    end
              'nxspe':begin   ;nxspe file
                    ok = dialog_message('Not implemented.',dialog_parent=self.tlb,/center)
                    error = 1b
                    end   
              'inx':begin     ;inx file
                    ok = dialog_message('Not implemented.',dialog_parent=self.tlb,/center)
                    error = 1b
                    end
              else: error = 1b ;do nothing
            endcase
            if keyword_set(error) then break
            if n_elements(bgrate) ne 0 then begin ;background file eg. empty can files
               ptr_free,self.bgratePtr
               self.bgratePtr = ptr_new([[bgrate],[dbgrate]])
               if n_elements(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' dark count file'
               self->my_widget_control,'bgclBut',bkgrmenu,/sensitive
               dm_set_button,widget_info(bkgrmenu,find_by_uname='bgdkBut'),1
               dm_set_button,widget_info(bkgrmenu,find_by_uname='bgovBut'),0
               if self.idl_version ge 5.6 then self->my_widget_control,'clearDat',tooltip='Background data will not be cleared.'
            endif
          end
        'vandbut':begin  ;load vanadium file(s)
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs_vanadium,open_path,open_file,parent=event.handler,error=error,comment=comment,qty=qty,ei=eief,info=info,ftpobj=ftpobj
                    end
              'macs':begin    ;macs file
                    self->dm_load_macs,open_path,open_file,a3=a3,deteff=deteff,ei=ei,ef=ef,error=error,info=info,cfx=cfx,comment=comment,kidney=kidney,parent=event.handler,$
                        qty=qty,dqty=dqty,temperature=temperature,weight=weight,histo_weight=histo_weight,ftpobj=ftpobj,baddet_id=baddet_id,/skipextra
                    if keyword_set(error) then break
                    weight = weight[*,self.mon_typ[1]]
                    tmpind = where(weight eq 0,count)
                    if count ne 0 then weight[tmpind] = 1.0
                    if n_elements(temperature) ne 0 then info = {temperature:round(mean(temperature[*,0]))}
                    ;ask whether to bypass the bad detector
                    nbd = n_elements(baddet_id)
                    if nbd ne 0 then begin
                       ok = dialog_message(dialog_parent=self.tlb,['The anaylzer angle A5 in detector '+dm_to_string(baddet_id,sep=', ')+([' are',' is'])[nbd eq 1]+' more than 0.5 degree off.',$
                          'Do you want to bypass '+(['them','it'])[nbd eq 1]+' in the vanadium calculation?','','(Detector number starts from 0.)'],/question,/center,title='bad detector channel')
                       if ok eq 'No' then tmp = temporary(baddet_id)
                       widget_control,/hourglass
                    endif
                    end      
              'spe':begin     ;spe file
                    self->dm_load_spe,open_path,open_file,qty=qty,parent=event.handler,title='loading vanadium file...',error=error,eief=eief,phxfile=phxfile,info=info
                    end
              'nxspe':begin   ;nxspe file
                    self->dm_load_nxspe,open_path,open_file,qty=qty,parent=event.handler,title='loading vanadium file...',error=error,eief=eief,info=info
                    end   
              'inx':begin     ;inx file
                    self->dm_load_inx,open_path,open_file,qty=qty,parent=event.handler,title='loading vanadium file...',error=error,eief=eief,comment=comment,info=info
                    end
              'wand':begin    ;wand file
                    self->dm_load_wand,open_path,open_file,error=error,parent=event.handler,temperature=temperature,eief=eief,qty=qty,dqty=dqty,/sum
                    if ~keyword_set(error) then temperature = mean(temperature)
                    end
              else: error = 1b ;do nothing
            endcase
            if keyword_set(error) or (n_elements(qty) eq 0) then break
            if n_elements(temperature) eq 0 then temperature=300.0
            if n_elements(info)    eq 0 then info={temperature:mean(temperature)}
            if n_elements(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' vanadium data'
            if n_elements(phxfile) ne 0 then self.phxfile=phxfile
            
            ;for MACS, first monitor normalization
            if is_macs then begin
               if self.macs_lamb2 then weight = dm_macs_lambda2correction(ei,weight,cfx)
               monfac = weight[0]/weight
               for i=0L,n_elements(monfac)-1 do qty[*,*,i] = qty[*,*,i]*monfac[i]
               tth = kidney#(fltarr(20)+1.0)
               for i=0,19 do tth[*,i] = tth[*,i]+(*self.detPosPtr).two_theta[i]-76.
               if self.instrgeom eq 0 then eief = mean(ei) else eief = mean(ef)
            endif else begin
               tth = (*self.detPosPtr).two_theta
               ;for SPE and NXSPE and WAND file type, check masks
               if self.ftypecheckmask[1] then begin
                  i = 0L
                  while(total(finite(qty[i,*])) eq 0) do i=i+1    ;in case the first time channel is masked
                  maskdet = where(finite(qty[i,*],/nan),count)
                  if count eq 0 then tmp=temporary(maskdet)
               endif
            endelse
            
            ;check for empty can subtraction 
            if (self.samp_typ lt 2) and ptr_valid(self.bgdata[0]) then begin  ;empty can subtraction
               size1 = size(qty,/dimensions)
               if is_macs then subtract = 1b $
               else begin
                  size2 = size((*self.bgdata[0]).qty,/dimensions)
                  subtract = (n_elements(size1) eq n_elements(size2) and (total(abs(size1-size2)) eq 0))
               endelse
               if subtract then begin
                  ans = dialog_message('Do you want to subtract the empty can data from the vanadium data?',/question,dialog_parent=self.tlb,/center)
                  if strupcase(ans) eq 'YES' then begin
                     ssfac = 1.0     ;self-shielding factor
                     if ptr_valid(self.ssfacPtr) then begin
                        if (*self.ssfacPtr).det_or_en eq 0 then begin ;detector dependent ssfac
                           if size1[1] eq n_elements((*self.ssfacPtr).ssfac ) then ssfac = make_array(size1[0],value=1d)#(*self.ssfacPtr).ssfac else begin
                              ok = dialog_message('Incompatible detector dependent self-shielding factor array size ('+dm_to_string(n_elements((*self.ssfacPtr).ssfac))+') and detector number ('+$
                                   dm_to_string(size1[1])+'). ssfac=1.0 is used instead.',dialog_parent=self.tlb,/center)
                              widget_control,/hourglass
                           endelse
                        endif else begin  ;energy dependent ssfac
                           if size1[0] eq n_elements((*self.ssfacPtr).ssfac ) then ssfac = (*self.ssfacPtr).ssfac#make_array(size1[1],value=1d) else begin
                              ok = dialog_message('Incompatible energy dependent self-shielding factor array size ('+dm_to_string(n_elements((*self.ssfacPtr).ssfac))+') and time channel number ('+$
                                   dm_to_string(size1[0])+'). ssfac=1.0 is used instead.',dialog_parent=self.tlb,/center)
                              widget_control,/hourglass
                           endelse
                        endelse
                     endif else if finite(self.ssfac) then ssfac = self.ssfac
                     if is_macs then begin
                        en = ei-ef & nen = n_elements(en)
                        monitor = weight[0]
                     endif else a3 = 0
                     for i=0,n_elements(self.bgdata)-1 do begin
                         if ptr_valid(self.bgdata[i]) then self->dm_subbg,qty,dqty,n_file=1,nen=nen,angle=a3,ssfac=ssfac,en=en,Ei=ei,Ef=ef,kidney=kidney,monitor=monitor,mesg=mesg,$
                            empty_count=empty_count,bgset=i,tchan_count=0
                     endfor
                     if n_elements(empty_count) ne 0 then begin
                        if empty_count gt 30 then mesg = [mesg,'...',dm_to_string(empty_count-30)+' more mismatch'+(['es are',' is'])[empty_count eq 31]+' not listed.']
                        if empty_count gt 0  then ok = dialog_message([mesg,'Try increasing the empty can subtraction tolerance values.','','This message is only a reminder.'],dialog_parent=self.tlb,/center)
                        widget_control,/hourglass
                     endif
                  endif
               endif
            endif
           
            ;Debye Waller factor correction
            if n_tags(info) ne 0 then begin
               tnames = tag_names(info)
               index  = where(tnames eq 'TEMPERATURE',count)
               if count ne 0 then $
                  temperature = info.temperature $
               else $
                  temperature = 300.0
            endif
            temperature = dm_dialog_input('Vanadium Temperature(K):',default=dm_to_string(temperature),title='Debye-Waller Factor',$
               info=['For the Debye-Waller factor of vanadium.','Press Skip to skip the Debye-Waller factor calculation.'],dialog_parent=self.tlb,cancel=cancel,cbstring='Skip')
            dw = 1.0
            if ~keyword_set(cancel) then begin
               if strlen(temperature) ne 0 then begin
                  temperature = dm_to_number(temperature,/float)
                  if (~ finite(temperature)) then begin
                     ok  = dialog_message('Invalid inputs. Debye-Waller factor for vanadium is not calculated.',dialog_parent=self.tlb,/center)
                     widget_control,/hourglass
                  endif else begin
                     dw  = dm_calc_dwfactor(temperature,tth,eief,maskdet=maskdet)
                  endelse
               endif
            endif
               
            ;add absorption correction
            mesg = [ 'Press Skip to skip the absorption correction for vanadium file,',$
                     'or press OK without filling in all the information.',$
                     '',$
                     'L0 is the energy independent mean free path;',$
                     'L1 is the energy dependent mean free path for',$
                     '2200m/s or 25.3 meV neutron.',$
                     'The effective mean free path is:',$
                     'L=1/{1/L0+1/(L1*sqrt[E(meV)/25.3])}' ]
            default = ['2.8','2.7','','']
            parm = dm_dialog_input(['L0(cm):','L1(cm):','Rout(cm):','Rin(cm):'],default=default,info=mesg,title='Absorption correction for vanadium',$
                   xsize=140,dialog_parent=self.tlb,cancel=cancel,cbstring='Skip')
            abs_factor = 1.0
            if ~keyword_set(cancel) then begin     
               if min(strlen(parm)) gt 0 then begin
                  parm = dm_to_number(parm,/float)
                  if total(finite(parm)) eq 4 then begin
                     tth_dim = size(tth,/dimension)
                     abs_factor = dm_calc_shielding([1,parm],eief,0,reform(tth,n_elements(tth)),group_leader=self.tlb)
                     abs_factor = reform(abs_factor,tth_dim,/overwrite)
                  endif else begin
                     ok = dialog_message('Invalid inputs. Absorption correction for vanadium sample is disabled.',dialog_parent=self.tlb,/center)
                  endelse
               endif
            endif
           
            if is_macs then begin
               ;ask for the a4 mask range
               effmask = dm_dialog_input(['ignore A4 from:','to:','calculation method:','symmetrize:'],default=[self.macs_effmask,0,0],xsize=165,dialog_parent=self.tlb,$
                           droplist_content=[ptr_new(['average of the intensity ratios','ratio of the averaged intensity']),ptr_new(['Yes','No'])],$
                           is_droplist=[0,0,1,1],/return_number,info=['Data with A4 within the ignored range will not be used in the efficiency calculation.',$
                           'Symmetrization will make the counts of detectors with same |A4| on both sides equal to each other.'],cancel=cancel)
               if keyword_set(cancel) then break
               if total(finite(effmask,/nan)) gt 0 then begin
                  ok = dialog_message('Invalid range. New detector efficiency is not calculated.',dialog_parent=self.tlb,/center)       
                  break
               endif   
               effmask[0:1] = [min(effmask[0:1]),max(effmask[0:1])]
               self.macs_effmask = effmask[0:1]
               if keyword_set(self.macshistomode) then begin
                  qty = total(qty,2)
                  tmpsize = size(qty,/dim)
                  qty = reform(qty,[2,20,tmpsize[1]])
               endif
               tmpqty = transpose(qty,[0,2,1])
               for i=0,1 do tmpqty[i,*,*] = reform(tmpqty[i,*,*])/dw*abs_factor
               tmpqty = transpose(tmpqty,[0,2,1])
               eff = dm_calc_deteff(tmpqty,kidney=kidney,maskdet=self.macs_effmask,intavg=round(effmask[2]),macssym=round(1-effmask[3]),baddet_id=baddet_id)
               ind = where(finite(eff,/nan),count)
               if count ne 0 then begin
                  eff[ind] = deteff[ind]
                  for i=0,1 do eff[i,*] = eff[i,*]*20./total(eff[i,*])
               endif 
               ;now plot the data
               plottitle = ['original','new']+' det. efficiency'
               ploteffs  = reform([[[deteff[*,*]]],[[eff[*,*]]]])
               plotobjs  = objarr(2)
               self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep
               for j=0,1 do begin
                   all = self.gr_list->get(/all,count=count)
                   for i=0,count-1 do begin
                       all[i]->getproperty,title=title
                       if title eq plottitle[j] then begin
                          self.current = all[i]
                          break
                       endif
                   endfor
                   if ~obj_valid(self.current) then begin
                      self.current = obj_new('dm_plot',parentobj=self,background=(['black','white'])[self.pbgcol_bw],/isolatin1,xsize=(*self.pdim_list)[0,self.pdim_choice],$
                           ysize=(*self.pdim_list)[1,self.pdim_choice],notooltip=self.pnotooltip,legdcolumns=2,/legdshowoutline,render_method=self.ptrm_choice,tickdir=self.tdir_choice)
                      self->dm_add_plotwindow,type=-1,title=plottitle[j]
                   endif else self.current->erase,/nodraw
                   plotobjs[j] = self.current    
                   xmin  = min(tth,max=xmax)
                   xran  = [xmin,xmax]
                   colrs = self.current->getcolor(/list)
                   colrs = colrs[0:n_elements(colrs)-2] ;remove the user define color
                   for i=0,19 do begin
                       self.current->add_plot,tth[*,i],qty[0,i,*]*ploteffs[0,i,j],color=colrs[i mod n_elements(colrs)],psym='circle',linestyle='no line',legend=dm_to_string(i+1)
                       self.current->add_plot,tth[*,i],qty[1,i,*]*ploteffs[1,i,j],color=colrs[i mod n_elements(colrs)],psym='circle',linestyle='no line'
                   endfor
                   ind  = where(abs(transpose(tth)) ge 25)
                   yran = [min((reform(qty[0,*,*]))[ind]),max((reform(qty[1,*,*]))[ind])] 
                   if self.mon_typ[1] eq 0 then mon_lab = dm_to_string(weight[0])+' sec' $
                   else if self.mon_typ[1] eq 1 then mon_lab ='monitor='+dm_to_string(weight[0],/exp) $
                   else mon_lab = dm_to_string(weight[0])+' pulse'+(['','s'])[weight[0] gt 1]
                   self.current->setproperty,xtit='A4 (\deg)',ytit='Intensity (counts/'+mon_lab+')',xran=xran+[-5,5],yran=yran+[-0.1,0.1]*(yran[1]-yran[0]),$
                       title=plottitle[j],/showygrid,/keep,/hidelegend
               endfor
               self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep,/reset
               ptr_free,self.eff1
               self.eff1 = ptr_new(deteff,/no_copy)
               self->dm_arrange_plot,plotobjs
               info = '    SPEC       DIFF'
               for i=0,19 do info = [info,dm_to_string(eff[*,i],strlength=12,sep=',')]
               info = [info,'ignore A4 range: ['+dm_to_string(effmask[0:1],sep=', ')+']']
               ok = dialog_message(info,dialog_parent=self.tlb,/center,/info,title='det efficiency factor')
            endif else begin
               if size(qty,/n_dimensions) eq 1 then qty = reform(qty,1,n_elements(qty))
               eff = dm_calc_deteff(qty,/fit,maskdet=maskdet)/dw*abs_factor & maskdet=0
            endelse
            ptr_free,self.eff
            self.eff = ptr_new(eff,/no_copy)
            self->my_widget_control,'vdclBut',bkgrmenu,/sensitive
            dm_set_button,widget_info(bkgrmenu,find_by_uname='vandBut'),1
            if is_macs then begin
               dm_set_button,widget_info(bkgrmenu,find_by_uname='vandEscanBut'),0
               self->my_widget_control,'macsdeteffBut',/sensitive
            endif
          end
        'vandescanbut':begin ;only for MACS
            effmask = dm_dialog_input(['ignore A4 from:','to:'],default=self.macs_effmask,dialog_parent=self.tlb,cancel=cancel,$
                           info='Data with A4 within the ignored range will not be used in the efficiency calculation.')
            if keyword_set(cancel) then break
            if total(finite(effmask,/nan)) gt 0 then begin
               ok = dialog_message('Invalid range. New detector efficiency is not calculated.',dialog_parent=self.tlb,/center)       
               break
            endif   
            effmask = [min(effmask),max(effmask)]
            self.macs_effmask = effmask
            
            dep  = 'spec'+string(indgen(20)+1,format='(i02)')
            ytit = 'Spec (cts)'
            ei   = fltarr(2)
            intint = fltarr(20,2)
            plotobjs = objarr(2)
            self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep
            for i=0,1 do begin
                file = open_path+pathsep+open_file[i]
                if obj_valid(ftpobj) then begin
                   ok = ftpobj->GetFileContent(file,localfilename=self.dirs[2]+self.pathsep+'ftptmp'+open_file[i])
                   file = self.dirs[2]+self.pathsep+'ftptmp'+open_file[i]
                endif
                dm_load_macs,file,data,colinfo,maspacing=maspacing,deteff=deteff,group_leader=self.tlb,error=error,header=header
                if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                tmpind = where(stregex(header, '#scan +[0-9]+ +',/boolean,/fold_case),count)
                if count eq 1 then begin
                   tmp = strsplit(header[tmpind],/extract)
                   tmp = strlowcase(tmp[n_elements(tmp)-1])
                endif else tmp = 'a5'
                tmpind = where(['a1','a2','a5','a6'] eq tmp,count)
                if count eq 1 then begin
                   if tmp eq 'a5' then ind  = 'analyzertheta'+string(indgen(20)+1,format='(i02)') $
                   else ind  = tmp
                   xtit = tmp+' (\deg)'
                endif else begin
                   ok = dialog_message('Invalid energy scan files. New detector efficiency is not calculated.',dialog_parent=self.tlb,/center)
                   return,handled       
                endelse
                ind_a2 = (where(colinfo eq 'a2'))[0]
                ind_kn = (where(colinfo eq 'kidney'))[0]
                ind_wt = (where(colinfo eq 'monitor'))[0]
                ind_t  = (where(colinfo eq 'temp',cnt_t))[0]
                
                ;Debye Waller factor correction
                if i eq 0 then begin
                   if cnt_t ne 0 then temperature = mean(data[*,ind_t]) else temperature = 300.0
                   temperature = dm_dialog_input('Vanadium Temperature(K):',default=dm_to_string(temperature),title='Debye-Waller Factor',$
                      info=['For the Debye-Waller factor of vanadium.','Press Skip to skip the Debye-Waller factor calculation.'],dialog_parent=self.tlb,cancel=cancel,cbstring='Skip')
                   if ~keyword_set(cancel) then begin
                      if strlen(temperature) ne 0 then begin
                         temperature = dm_to_number(temperature,/float)
                         if (~ finite(temperature)) then ok = dialog_message('Invalid inputs. Debye-Waller factor for vanadium is not calculated.',dialog_parent=self.tlb,/center)
                      endif 
                   endif else temperature = !values.f_nan
                endif
                
                ;self.current = obj_new()
                all = self.gr_list->get(/all,count=count)
                for j=0,count-1 do begin
                    all[j]->getproperty,title=title
                    if title eq open_file[i]  then begin
                       self.current = all[j]
                       break
                    endif
                endfor
                if ~obj_valid(self.current) then begin
                   self.current = obj_new('dm_plot',parentobj=self,background=(['black','white'])[self.pbgcol_bw],/isolatin1,xsize=(*self.pdim_list)[0,self.pdim_choice],$
                           ysize=(*self.pdim_list)[1,self.pdim_choice],notooltip=self.pnotooltip,title=open_file[i],legdcolumns=2,render_method=self.ptrm_choice,tickdir=self.tdir_choice)
                   self->dm_add_plotwindow,type=-1,title=open_file[i] 
                endif else self.current->erase,/nodraw
                
                plotobjs[i] = self.current    
                pcol = self.current->getcolor(/list)
                pcol = pcol[0:n_elements(pcol)-2] ;remove the user define color
                psym = self.current->getpsym(/list)
                psym = psym[0:4] & psym = [psym,'filled '+psym]
                
                ei[i]  = mean(81.8042/((2*maspacing[0]*sin(!dtor*data[*,ind_a2]/2))^2)) ;A2->Ei
                kidney = mean(data[*,ind_kn]) 
                a4     = kidney-76.0+findgen(20)*8.0
                ignore = where((a4 ge effmask[0]) and (a4 le effmask[1]))
                if i eq 0 then moncnt0 = data[0,ind_wt]
                monfac = moncnt0/(1>(data[*,ind_wt]))
                if finite(temperature) then dw = dm_calc_dwfactor(temperature,a4,mean(ei)) else dw = fltarr(20)+1.0 ;calculate Deby Waller factor
                for j=0,n_elements(dep)-1 do begin
                    xind = where(colinfo eq ind[j<(n_elements(ind)-1)],count) & if count eq 0 then continue
                    yind = where(colinfo eq dep[j],count) & if count eq 0 then continue
                    x = data[*,xind[0]] & y = reform(data[*,yind[0]]) & yerr = sqrt(y)*monfac/dw[j] & y = y*monfac/dw[j]
                    ind_0 = where(yerr le 0,cnt_0)
                    if cnt_0 ne 0 then yerr[ind_0] = 1
                    dm_gaussfit,y,x=x,params=params,measure_errors=yerr
                    intint[j,i] = sqrt(2*!PI)*params[0]*params[2]
                    xmin = min(x,max=xmax)
                    xx   = xmin+findgen(101)/100*(xmax-xmin)
                    yy   = params[0]*exp(-(xx-params[1])^2/(2*params[2]^2))
                    ymin = min([y-yerr])
                    ymax = max([y+yerr])
                    if j eq 0 then begin
                       xran = [xmin,xmax]
                       yran = [ymin,ymax]
                    endif else begin
                       xran[0] = (xran[0])<(xmin)
                       xran[1] = (xran[1])>(xmax)
                       yran[0] = (yran[0])<(ymin)
                       yran[1] = (yran[1])>(ymax)
                    endelse
                    tmp = where(ignore eq j,cnt)
                    if cnt eq 1 then intint[j,i] = !values.f_nan
                    self.current->add_plot,x,y,yerr=yerr,color=pcol[j mod n_elements(pcol)],psym=psym[j mod n_elements(psym)],linestyle='no line',legend=dm_to_string(j+1)+(cnt eq 1?' ignored':'')
                    self.current->add_plot,xx,yy,color=pcol[j mod n_elements(pcol)],psym='no symbol',linestyle='solid'
                endfor
                
                self.current->setproperty,xran=xran+[-0.08,0.08]*(xran[1]-xran[0]),yran=yran+[-0.08,0.08]*(yran[1]-yran[0]),/legdshowoutline,/keep,xtit=xtit,ytit=ytit
            endfor
            
            self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep,/reset
            eff = deteff
            ptr_free,self.eff1
            self.eff1 = ptr_new(deteff,/no_copy)
            self->dm_arrange_plot,plotobjs        
            if abs(ei[0]-ei[1]) gt 0.1 then ok = dialog_message('The two files are measured at different Ei.',dialog_parent=self.tlb,/center)
            weights = total(intint,2,/nan)/(1>total(finite(intint),2)) ;average of two file
            ind = where(weights gt 0,cnt)
            weights = 1./weights[ind]
            eff[0,ind] = weights*cnt/total(weights) 
            if cnt ne 20 then eff[0,*] = eff[0,*]*20./total(eff[0,*])
            ptr_free,self.eff
            self.eff = ptr_new(eff)
            self->my_widget_control,'vdclBut',bkgrmenu,/sensitive
            dm_set_button,widget_info(bkgrmenu,find_by_uname='vandBut'),0
            dm_set_button,widget_info(bkgrmenu,find_by_uname='vandEscanBut'),1
            self->my_widget_control,'macsdeteffBut',/sensitive
            info = '    SPEC       DIFF'
            for i=0,19 do info = [info,dm_to_string(eff[*,i],strlength=12,sep=',')]
            info = [info,'ignore A4 range: ['+dm_to_string(effmask,sep=', ')+']']
            ok = dialog_message(info,dialog_parent=self.tlb,/center,/info,title='det efficiency factor')
            comment = 'MACS A5 scan vanadium file'
          end
        'loadbut':begin  ;load data file(s)
            if ~obj_valid(ftpobj) then self->setproperty,dataDir=open_path else self->setproperty,dataDir=ftpserver+open_path     
            ;clear the data and projection pointer
            ptr_free,self.kfactor,self.dataStrPtr,self.projStrPtr
            self->my_widget_control,['addBut','clearDat','mkplBut'],sensitive=0  
            self->set_display_button
            
            case self.instrname of
              'dcs':begin     ;dcs data
                    self->dm_load_dcs,open_path,open_file,parent=event.handler,error=error,temperature=temperature,hfield=hfield,$
                        comment=comment,qty=qty,dqty=dqty,weight=weight,ang=ang,ei=eief,info=info,$
                        ewid=ewid,emean=emean,monrate=monrate,kfactor=kfactor,tmean=tmean,adddata=ptr_valid(self.bgdata[0]),ftpobj=ftpobj,zero_error=zero_error
                    end
              'macs':begin
                    self->dm_load_macs,open_path,open_file,a3=a3,deteff=deteff,ei=ei,ef=ef,error=error,info=info,cfx=cfx,ptai=ptai,comment=comment,kidney=kidney,$
                        parent=event.handler,qty=qty,dqty=dqty,temperature=temperature,hfield=hfield,specify=specify,montype=montype,weight=weight,histo_weight=histo_weight,$
                        histo_nchan=histo_nchan,histo_width=histo_width,histo_t0=histo_t0,ftpobj=ftpobj,zero_error=zero_error
                    end      
              'spe':begin     ;spe file
                    e_range = self.e_range
                    if n_elements(angle) ne 0 then ang = self->dm_load_angle(angle,directory=directory)
                    self->dm_load_spe,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),$
                        phxfile=phxfile,avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,$
                        qty=qty,dqty=dqty,eief=eief,ang=ang,emean=emean,info=info,ewid=ewid,weight=weight,$
                        eint_yn=self.eint_yn,kfactor=kfactor,diffetyp=(self.samp_typ eq 3),adddata=ptr_valid(self.bgdata[0]),$
                        scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) then self.e_range=e_range
                    end
              'nxspe':begin   ;nxspe file
                    e_range = self.e_range
                    self->dm_load_nxspe,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),$
                        avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,$
                        qty=qty,dqty=dqty,eief=eief,ang=ang,emean=emean,info=info,lamcorr=lamcorr,ewid=ewid,weight=weight,$
                        eint_yn=self.eint_yn,kfactor=kfactor,diffetyp=(self.samp_typ eq 3),adddata=ptr_valid(self.bgdata[0]),$
                        scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) then self.e_range=e_range
                    if ~keyword_set(error) then self.lamcorrnxspe = lamcorr
                    end   
              'inx':begin     ;inx file
                    e_range = self.e_range
                    self->dm_load_inx,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),$
                        avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,$
                        qty=qty,dqty=dqty,eief=eief,ang=ang,emean=emean,info=info,ewid=ewid,eint_yn=self.eint_yn,weight=weight,$
                        kfactor=kfactor,diffetyp=(self.samp_typ eq 3),comment=comment,adddata=ptr_valid(self.bgdata[0]),$
                        scriptmode=scriptmode
                    if (self.samp_typ ge 2) and (n_elements(e_range) ne 0) then self.e_range=e_range
                    end
              'wand':begin    ;wand file
                    self->dm_load_wand,open_path,open_file,ang=ang,error=error,parent=event.handler,qty=qty,dqty=dqty,eief=eief,emean=emean,$
                        temperature=temperature,weight=weight,scriptmode=scriptmode,zero_error=zero_error
                    if ~keyword_set(error) and ~self.extravaxis_yn[0] then legend = 'T='+dm_to_string(mean(temperature),resolution=2)+' K'
                    end
              else: error = 1b ;do nothing
            endcase
            n_data = n_elements(qty)
            if keyword_set(error) or (n_data eq 0) then break
            if (n_file gt 1) and (~keyword_set(scriptmode)) and (~self.ftoolflag[6]) and ((n_elements(a3) gt 1) or (n_elements(ang) gt 1)) then begin  ;check a3 tolerance
               if n_elements(a3) gt 1 then tmp_a3 = a3[sort(a3)] else tmp_a3 = ang[sort(ang)]
               tmp_a3 = round((tmp_a3[1:*]-tmp_a3[0:(n_elements(tmp_a3)-1)])*1e2)/1e2
               tmp_ind = where(tmp_a3 gt 0,tmp_cnt)
               if tmp_cnt gt 0 then begin
                  tmp_min = min(tmp_a3[tmp_ind])
                  if ([self.psi_tolerance[1],self.macs_tolerance[2,1]])[is_macs] gt tmp_min then begin
                     ok = dialog_message(dialog_parent=self.tlb,'The minimum '+(['psi','A3'])[is_macs]+' step is '+dm_to_string(tmp_min)+$
                          ', smaller than the file combination tolerance value of '+dm_to_string(([self.psi_tolerance[1],self.macs_tolerance[2,1]])[is_macs])+$
                          '. You can change the tolerance value at Parameter->Change File Combination Tolerance Value menu.',/info,/center)
                     self.ftoolflag[6] = 1b
                  endif
               endif
            endif   
            if n_elements(temperature) eq 0 then temperature = 300.0
            if size(temperature,/n_dim) eq 2 then t_mean =  mean(temperature[-1,*]) else t_mean = mean(temperature)
            if n_elements(hfield)  eq 0 then hfield = 0.0
            if n_elements(specify) eq 0 then specify = 0.0
            if n_elements(monrate) eq 0 then monrate = 1.0
            if n_elements(kfactor) eq 0 then kfactor = 1.0
            if n_elements(info)    eq 0 then info = {temperature:t_mean}
            if n_elements(tmean)   eq 0 then tmean = 0.0
            if n_elements(phxfile) ne 0 then self.phxfile = phxfile
            if n_elements(zero_error) ne 0 then self.zero_error[1] = zero_error
            if ptr_valid(self.file_recent) then begin
               tmp = where(*self.file_recent eq open_file[0],count)
               if count eq 0 then self.userfwarning = 1b  ;reset the warning flag
               ptr_free,self.file_recent
            endif else self.userfwarning = 1b  ;reset the warning flag
            self.file_recent = ptr_new([open_path,open_file])
            self.ftoolflag[4] = 0
            self.monrate = monrate
            self.binthreshold = 0.0 ;reset the binning threshold
            if is_macs then begin
               if n_elements(histo_weight) eq 0 then histo_weight = 1.0
               self.psi[3] = A3[0] ;save the first A3 angle
               self.macs_mon[0,0:(1+keyword_set(self.macshistomode))] = weight[0,*]
               if n_elements(montype) ne 0 then begin
                  if (montype ne self.mon_typ[1]) and (self.mon_typ[1] ne 2) then begin
                     ans = dialog_message('Do you want to switch the monitor choice '+(['from time to monitor counts?','from monitor counts to time?','from pulse number to '+(['time?','monitor counts?'])[montype]])[self.mon_typ[1]],dialog_parent=self.tlb,/question,/center)
                     if strupcase(ans) eq 'YES' then self.mon_typ[1] = montype
                     self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[1]
                     self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[1]]+')'
                  endif
               endif
               self->reset_intnlab
               if n_file gt 1 then dm_combine_macs,qty,dqty,weight,cfx,A3,kidney,Ei,Ef,ptai=ptai,temperature=temperature,hfield=hfield,specify=specify,histo_weight=histo_weight,$
                  viewt=self.extravaxis_yn[1],etolerance=self.macs_tolerance[0,1],ktolerance=self.macs_tolerance[1,1],atolerance=self.macs_tolerance[2,1],checkfinite=self.ftypecheckmask[0],$
                  montype=self.mon_typ[1],rawint=(keyword_set(self.macshistomode) and self.macs_rawintn)
               if self.macshistomode eq 2 then begin
                  self.macshistonchan = histo_nchan
                  self.macshistowidth = histo_width
                  self.macshistot0[0] = histo_t0
               endif
               kfactor = sqrt(Ei/Ef)
               ;set projection parameters
               if ~self.locklattuv then begin
                  unames = ['latt'+['A','B','C','AA','BB','CC'],'orie'+['U'+['h','k','l'],'V'+['h','k','l']]]
                  values = [info.latticeparm,info.latticeori]
                  self->script,'set_parameter',name=unames,value=values
                  ;check viewing axes to see if they are on the scattering plane, if not, replace them
                  if ~(self->viewonscattplane()) then self->script,'set_parameter',name='viewU'+['1'+['h','k','l'],'2'+['h','k','l']],value=info.latticeori
               endif
               ;check instrument geometry
               minEi = min(Ei,max=maxEi) & minEf = min(Ef,max=maxEf)
               if (self.instrgeom eq 1) and (abs(minEf-maxEf) gt 0.002) then switchgeom = 1b  ;inverse geometry
               if (self.instrgeom eq 0) and (abs(minEi-maxEi) gt 0.002) then switchgeom = 1b  ;direct geometry  
               if keyword_set(switchgeom) then begin
                  ans = dialog_message(dialog_parent=self.tlb,['The instrument might be in the '+(['inverse','direct'])[self.instrgeom]+' geometry mode.',$
                         'Do you want to switch to the '+(['inverse','direct'])[self.instrgeom]+' geometry?'],/question,/center)
                  if strupcase(ans) eq 'YES' then self->setproperty,instrgeom=(1-self.instrgeom)
               endif
               self.eief         = ([(minEi+maxEi)/2.0,(minEf+maxEf)/2.0])[self.instrgeom]
               self.estartend    = dm_startend(Ei-Ef,resolution=3)
               self.kidney[0:2]  = dm_to_string(dm_startend(kidney,tolerance=0.1))
               self.psi[0:2]     = dm_startend(A3,resolution=2)
               self.ftoolflag[4] = (n_elements(qty) ne n_data)
               self.dataStrPtr   = ptr_new({qty:temporary(qty),err:temporary(dqty),rotation:temporary(A3),kidney:temporary(kidney),weight:temporary(weight),$
                    histo_weight:temporary(histo_weight),cfx:temporary(cfx),ptai:temporary(ptai),Ei:temporary(Ei),Ef:temporary(Ef),temperature:temporary(temperature),$
                    hfield:temporary(hfield),specify:temporary(specify),info:info})
               if n_elements(deteff) eq 2* n_elements((*self.detPosPtr).two_theta) then begin
                  if ~ptr_valid(self.eff1) then begin
                     if self.ftoolflag[1] then begin
                        ans = dialog_message(dialog_parent=self.tlb,'Do you want to overwrite the detector efficiency by the efficiency info stored in the data file?',/question,/center,/DEFAULT_NO)
                        if strupcase(ans) eq 'YES' then self.ftoolflag[1] = 0b
                     endif
                     if ~self.ftoolflag[1] then begin
                        ptr_free,self.eff
                        self.eff = ptr_new(deteff)   
                     endif
                  endif else begin  ;update the backup efficiency pointer
                     ptr_free,self.eff1
                     self.eff1 = ptr_new(deteff)   
                  endelse
               endif
               self->my_widget_control,'macsdeteffBut',sensitive=ptr_valid(self.eff)
               if ptr_valid(self.mask) and (self.masktype eq 1) then self->clear_mask,/keep_ar,/message
            endif else begin
               self.eief = abs(eief)
               ;readjust psi value inteligently
               if (n_elements(ang) ne 0) and (self.samp_typ ge 1) then begin
                  self->my_widget_control,'oriepsiLab',self.projbas,set_value='   <- '+((self.samp_typ eq 1)?'':'first ')+'angle of ki^u ('+dm_to_string(ang[0],resolution=2)+' read)'
                  if finite(self.orie_offset) then begin
                     oldpsi = self.orie_psi
                     self->script,'set_parameter',name='oriePsi',value=ang[0]+self.orie_offset,resolution=2,resetpsioffset=0
                     ;readjust flat plate absorption angle intelligently
                     if ptr_valid(self.absp_info) and finite(oldpsi) and (~keyword_set(scriptmode)) then begin
                        if ((*(self.absp_info))[0] eq 3) and ((self.orie_psi-oldpsi) ne 0) then begin  ;flat plate
                           (*(self.absp_info))[n_elements(*(self.absp_info))-1] = (*(self.absp_info))[n_elements(*(self.absp_info))-1]+self.orie_psi-oldpsi
                           ok = dialog_message(dialog_parent=self.tlb,'Please check the absorption correction flat plate angle.',/info,/center)
                        endif
                     endif
                  endif
               endif
               ;check instrument geometry to see if it is consistent with the energy
               if ((self.instrgeom eq 0) and (max(emean) ge eief)) or ((self.instrgeom eq 1) and (min(emean) le -eief)) then begin
                  ans = dialog_message(dialog_parent=self.tlb,['The instrument might be '+(['an inverse','a direct'])[self.instrgeom]+' geometry one.',$
                         'Do you want to switch to the '+(['inverse','direct'])[self.instrgeom]+' geometry?'],/question,/center)
                  if strupcase(ans) eq 'YES' then self->setproperty,instrgeom=(1-self.instrgeom)
               endif
            
               if self.samp_typ ge 2 then begin
                  nang = n_elements(ang)
                  if n_elements(weight) eq 0 then weight=fltarr(nang)+1.0
                  if (self.samp_typ eq 3) and (self.e_bin[4] gt 1) then begin
                     dm_grid_ebin,self.e_bin[4],qty,dqty,emean,avgsum=self.bin_avgsum
                     if n_elements(kfactor) gt 1 then $
                        dm_grid_ebin,self.e_bin[4],kfactor $
                     else begin
                        if self.instrgeom eq 0 then $
                           kfactor = sqrt(eief/(eief-emean)) $
                        else $
                           kfactor = sqrt((eief+emean)/eief)
                     endelse
                     if n_elements(ewid) ne 0 then dm_grid_ebin,self.e_bin[4],ewid
                  endif
                  if n_elements(ewid) eq 0 then ewid=1.0
                  self.psi[3]   = ang[0]  ;save the angle of first file
                  self.psi[0:2] = dm_startend(ang,resolution=2)
                  ;combine data with the same psi angles
                  if is_dcs and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
                     ndat = n_elements(weight)
                     tmp = dm_to_string(ang,resolution=2)
                     if self.extravaxis_yn[1] then for i=0L,ndat-1 do tmp[i] = tmp[i]+'T'+dm_to_string(temperature[*,i],resolution=3,separator='_')
                     if self.extravaxis_yn[2] then tmp = tmp+'H'+dm_to_string(hfield,resolution=5)
                     dm_combine_psi,tmp,qty,dqty,weight,checkfinite=self.ftypecheckmask[1]
                     ndat1 = n_elements(weight)
                     if ndat1 ne ndat then begin
                        ang = fltarr(ndat1)
                        if self.extravaxis_yn[1] then temperature = fltarr(3,ndat1)
                        if self.extravaxis_yn[2] then hfield = fltarr(ndat1)
                        for i=0L,ndat1-1 do begin
                            tmp1 = stregex(tmp[i],'([^TH]*)T*([^H]*)H*([^T]*)',/extract,/subexpr)
                            ang[i] = dm_to_number(tmp1[1])
                            if self.extravaxis_yn[1] then temperature[*,i] = dm_to_number(strsplit(tmp1[2],'_',/extract))
                            if self.extravaxis_yn[2] then hfield[i] = dm_to_number(tmp1[3])
                        endfor
                     endif 
                  endif else if is_wand and self.extravaxis_yn[0] then begin ;temperature is a viewing axis
                     ndat = n_elements(weight)
                     tmp = dm_to_string(ang,resolution=2)+'_'+dm_to_string(temperature,resolution=3)
                     dm_combine_psi,tmp,qty,dqty,weight,checkfinite=self.ftypecheckmask[1]
                     ndat1 = n_elements(weight) 
                     if ndat1 ne ndat then begin
                        ang = fltarr(ndat1) & temperature = fltarr(ndat1)
                        for i=0L,ndat1-1 do begin
                            tmp1 = dm_to_number(strsplit(tmp[i],'_',/extract,count=count))
                            if count eq 2 then begin
                               ang[i] = tmp1[0]
                               temperature[i] = tmp1[1]
                            endif
                        endfor
                     endif
                  endif else begin
                     dm_combine_psi,ang,qty,dqty,weight,checkfinite=self.ftypecheckmask[1],extraavg=temperature,tolerance=self.psi_tolerance[1]
                  endelse
               endif else begin
                  if n_elements(weight) eq 0 then weight=1.0
                  if n_elements(ewid) eq 0   then ewid=fltarr(n_elements(emean))+1.0
                  if n_elements(ang) eq 0    then ang=!values.f_nan
               endelse
               self.ftoolflag[4] = (n_elements(qty) ne n_data)
               self.dataStrPtr = ptr_new({qty:temporary(qty),err:temporary(dqty),weight:temporary(weight),temperature:temporary(temperature),hfield:temporary(hfield),$
                    specify:temporary(specify),rotation:temporary(ang),info:info,eief:temporary(eief),ewid:temporary(ewid),e:temporary(emean),$
                    time:tmean,filename:open_file,filepath:open_path},/no_copy)
               if finite(self.orie_psi,/nan) and (self.samp_typ ge 2) then begin
                  self->script,'set_parameter',name='oriePsi',value=(*self.dataStrPtr).rotation[0],resolution=2
               endif
            endelse   
            self.kfactor = ptr_new(kfactor,/no_copy)
            self->my_widget_control,['addBut','clearDat','mkplBut'],/sensitive    ;allow adding data & clear data & plot mask
            self->set_display_button
            if n_elements(legend) eq 0 then legend = ''
            if n_elements(comment) eq 0 then comment = ''
            set_legend = 1b & set_title = 1b
          end
        'addbut':begin   ;add dcs data file to current data pointer
            case self.instrname of
              'dcs':begin     ;dcs data
                    info = (*self.dataStrPtr).info
                    self->dm_load_dcs,open_path,open_file,parent=event.handler,error=error,comment=comment,qty=qty,dqty=dqty,weight=weight,ang=ang,$
                        info=info,title='adding data file...',/adddata,ftpobj=ftpobj,temperature=temperature,hfield=hfield,zero_error=zero_error
                    end
              'macs':begin    ;macs data
                    info = (*self.dataStrPtr).info
                    self->dm_load_macs,open_path,open_file,a3=a3,ei=ei,ef=ef,error=error,info=info,cfx=cfx,ptai=ptai,comment=comment,kidney=kidney,parent=event.handler,$
                        qty=qty,dqty=dqty,temperature=temperature,hfield=hfield,specify=specify,weight=weight,histo_weight=histo_weight,montype=montype,ftpobj=ftpobj,$
                        zero_error=zero_error
                    end
              'spe':begin     ;spe file
                    info = (*self.dataStrPtr).info
                    e_range = self.e_range
                    self->dm_load_spe,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),weight=weight,$
                        avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,qty=qty,dqty=dqty,$
                        eief=eief,ang=ang,emean=emean,info=info,eint_yn=self.eint_yn,diffetyp=(self.samp_typ eq 3),/adddata
                    end
              'nxspe':begin   ;nxspe file
                    info = (*self.dataStrPtr).info
                    e_range = self.e_range
                    self->dm_load_nxspe,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),weight=weight,$
                        avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,qty=qty,dqty=dqty,$
                        eief=eief,ang=ang,emean=emean,info=info,eint_yn=self.eint_yn,diffetyp=(self.samp_typ eq 3),/adddata
                    end
              'inx':begin     ;inx file
                    info = (*self.dataStrPtr).info
                    e_range = self.e_range
                    self->dm_load_inx,open_path,open_file,parent=event.handler,diffuse=(self.samp_typ ge 2),weight=weight,$
                        avgsum=self.bin_avgsum,e_range=e_range,t_chan=self.e_bin[self.samp_typ],error=error,qty=qty,dqty=dqty,$
                        eief=eief,ang=ang,emean=emean,info=info,eint_yn=self.eint_yn,diffetyp=(self.samp_typ eq 3),/adddata
                    end
              'wand':begin    ;wand file
                    self->dm_load_wand,open_path,open_file,ang=ang,error=error,parent=event.handler,qty=qty,dqty=dqty,temperature=temperature,$
                        eief=eief,emean=emean,weight=weight,zero_error=zero_error
                    end                    
              else: error = 1b ;do nothing
            endcase
            if keyword_set(error) or (n_elements(qty) eq 0) then break
            if (~keyword_set(scriptmode)) and (~self.ftoolflag[6]) and ((n_elements(a3) gt 1) or (n_elements(ang) gt 1)) then begin  ;check a3 tolerance
               if n_elements(a3) gt 1 then tmp_a3 = a3[sort(a3)] else tmp_a3 = ang[sort(ang)]
               tmp_a3 = round((tmp_a3[1:*]-tmp_a3[0:(n_elements(tmp_a3)-1)])*1e2)/1e2
               tmp_ind = where(tmp_a3 gt 0,tmp_cnt)
               if tmp_cnt gt 0 then begin
                  tmp_min = min(tmp_a3[tmp_ind])
                  if ([self.psi_tolerance[1],self.macs_tolerance[2,1]])[is_macs] gt tmp_min then begin
                     ok = dialog_message(dialog_parent=self.tlb,'The minimum '+(['psi','A3'])[is_macs]+' step is '+dm_to_string(tmp_min)+$
                          ', smaller than the file combination tolerance value of '+dm_to_string(([self.psi_tolerance[1],self.macs_tolerance[2,1]])[is_macs])+$
                          '. You can change the tolerance value at Parameter->Change File Combination Tolerance Value menu.',/info,/center)
                     self.ftoolflag[6] = 1b
                  endif
               endif
            endif
            if n_elements(comment) eq 0 then comment = ''
            if ptr_valid(self.file_recent) then (*self.file_recent) = [(*self.file_recent),open_file] else self.file_recent = ptr_new([open_path,open_file])
            if n_elements(zero_error) ne 0 then self.zero_error[1] = (self.zero_error[1])<(zero_error)
            if is_macs then begin
               info   = (*self.dataStrPtr).info
               qty    = [[[(*self.dataStrPtr).qty]],[[temporary(qty)]]]
               n_data = n_elements(qty)
               err    = [[[(*self.dataStrPtr).err]],[[temporary(dqty)]]]
               weight = [(*self.dataStrPtr).weight,temporary(weight)]
               cfx    = [(*self.dataStrPtr).cfx,temporary(cfx)]
               ptai   = [(*self.dataStrPtr).ptai,temporary(ptai)]
               A3     = [(*self.dataStrPtr).rotation,temporary(A3)]
               kidney = [(*self.dataStrPtr).kidney,temporary(kidney)]
               Ei     = [(*self.dataStrPtr).Ei,temporary(Ei)]
               Ef     = [(*self.dataStrPtr).Ef,temporary(Ef)]
               temperature = [(*self.dataStrPtr).temperature,temporary(temperature)]
               if n_elements(histo_weight) eq 0 then histo_weight = 1.0 else histo_weight = [[(*self.dataStrPtr).histo_weight],[temporary(histo_weight)]]
               if self.extravaxis_yn[2] then hfield = [(*self.dataStrPtr).hfield,temporary(hfield)] else hfield = 0
               if self.extravaxis_yn[0] then specify = [(*self.dataStrPtr).specify,temporary(specify)] else specify = 0
               ;combine data
               dm_combine_macs,qty,err,weight,cfx,A3,kidney,Ei,Ef,ptai=ptai,temperature=temperature,hfield=hfield,specify=specify,histo_weight=histo_weight,$
                  viewt=self.extravaxis_yn[1],etolerance=self.macs_tolerance[0,1],ktolerance=self.macs_tolerance[1,1],atolerance=self.macs_tolerance[2,1],checkfinite=self.ftypecheckmask[0],$
                  montype=self.mon_typ[1],rawint=(keyword_set(self.macshistomode) and self.macs_rawintn)
               ptr_free,self.kfactor,self.dataStrPtr
               self.ftoolflag[4] = (self.ftoolflag[4] or (n_elements(qty) ne n_data))
               self.kfactor     = ptr_new(sqrt(Ei/Ef))
               self.estartend   = dm_startend(Ei-Ef,resolution=3)
               self.kidney[0:2] = dm_to_string(dm_startend(kidney,tolerance=0.1))
               self.psi[0:2]    = dm_startend(A3,resolution=2)
               self.dataStrPtr  = ptr_new({qty:temporary(qty),err:temporary(err),rotation:temporary(A3),kidney:temporary(kidney),weight:temporary(weight),$
                                    histo_weight:temporary(histo_weight),cfx:temporary(cfx),ptai:temporary(ptai),Ei:temporary(Ei),Ef:temporary(Ef),$
                                    temperature:temporary(temperature),hfield:temporary(hfield),specify:temporary(specify),info:info})
               if ptr_valid(self.mask) and (self.masktype eq 1) then self->clear_mask,/keep_ar,/message                     
            endif else begin
               if self.samp_typ ge 2 then begin
                  if n_elements(weight) eq 0 then weight = fltarr(n_elements(ang))+1.0
                  if self.samp_typ eq 3 then begin
                     if self.e_bin[4] gt 1 then begin
                        tmp = size(qty,/dimension)
                        dm_grid_ebin,self.e_bin[4],qty,dqty,fltarr(tmp[0]),avgsum=self.bin_avgsum
                     endif
                     qty = [[[(*self.dataStrPtr).qty]],[[temporary(qty)]]]
                     err = [[[(*self.dataStrPtr).err]],[[temporary(dqty)]]]
                  endif else begin
                     qty = [[(*self.dataStrPtr).qty],[temporary(qty)]]
                     err = [[(*self.dataStrPtr).err],[temporary(dqty)]]
                  endelse
                  n_data = n_elements(qty)
                  weight = [(*self.dataStrPtr).weight,temporary(weight)]
                  ang    = [(*self.dataStrPtr).rotation,temporary(ang)]
                  filename = [(*self.dataStrPtr).filename,open_file]
                  e      = (*self.dataStrPtr).e
                  eief   = (*self.dataStrPtr).eief
                  info   = (*self.dataStrPtr).info
                  ewid   = (*self.dataStrPtr).ewid
                  path   = (*self.dataStrPtr).filepath
                  time   = (*self.dataStrPtr).time
                  
                  if is_dcs and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then temperature = [[(*self.dataStrPtr).temperature],[temperature]] $
                  else if is_wand then temperature = [(*self.dataStrPtr).temperature,temperature] $
                  else temperature = (*self.dataStrPtr).temperature
                  if is_dcs and (self.extravaxis_yn[2]) then hfield= [(*self.dataStrPtr).hfield,hfield] $
                  else hfield = (*self.dataStrPtr).hfield
                  specify = 0.0
                  ;combine data with the same psi angles
                  if is_dcs and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
                     ndat = n_elements(weight)
                     tmp = dm_to_string(ang,resolution=2)
                     if self.extravaxis_yn[1] then for i=0L,ndat-1 do tmp[i] = tmp[i]+'T'+dm_to_string(temperature[*,i],resolution=3,separator='_')
                     if self.extravaxis_yn[2] then tmp = tmp+'H'+dm_to_string(hfield,resolution=5)
                     dm_combine_psi,tmp,qty,dqty,weight,checkfinite=self.ftypecheckmask[1]
                     ndat1 = n_elements(weight)
                     if ndat1 ne ndat then begin
                        ang = fltarr(ndat1)
                        if self.extravaxis_yn[1] then temperature = fltarr(3,ndat1)
                        if self.extravaxis_yn[2] then hfield = fltarr(ndat1)
                        for i=0L,ndat1-1 do begin
                            tmp1 = stregex(tmp[i],'([^TH]*)T*([^H]*)H*([^T]*)',/extract,/subexpr)
                            ang[i] = dm_to_number(tmp1[1])
                            if self.extravaxis_yn[1] then temperature[*,i] = dm_to_number(strsplit(tmp1[2],'_',/extract))
                            if self.extravaxis_yn[2] then hfield[i] = dm_to_number(tmp1[3])
                        endfor
                     endif
                  endif else if is_wand and self.extravaxis_yn[0] then begin ;temperature is a viewing axis
                     ndat = n_elements(weight)
                     tmp = dm_to_string(ang,resolution=2)+'_'+dm_to_string(temperature,resolution=3)
                     dm_combine_psi,tmp,qty,err,weight,checkfinite=self.ftypecheckmask[1]
                     ndat1 = n_elements(weight) 
                     if ndat1 ne ndat then begin
                        ang = fltarr(ndat1) & temperature = fltarr(ndat1)
                        for i=0L,ndat1-1 do begin
                            tmp1 = dm_to_number(strsplit(tmp[i],'_',/extract,count=count))
                            if count eq 2 then begin
                               ang[i] = tmp1[0]
                               temperature[i] = tmp1[1]
                            endif
                        endfor
                     endif
                  endif else $
                     dm_combine_psi,ang,qty,err,weight,checkfinite=self.ftypecheckmask[1],extraavg=temperature,tolerance=self.psi_tolerance[1]
                  self.psi[0:2] = dm_startend(ang,resolution=2)
                  self.ftoolflag[4] = (self.ftoolflag[4] or (n_elements(qty) ne n_data))
                  ptr_free,self.dataStrPtr
                  self.dataStrPtr = ptr_new({qty:temporary(qty),err:temporary(err),weight:temporary(weight),temperature:temporary(temperature),$
                      hfield:temporary(hfield),specify:temporary(specify),rotation:temporary(ang),info:info,eief:temporary(eief),ewid:temporary(ewid),$
                      e:temporary(e),time:time,filename:temporary(filename),filepath:path},/no_copy)
               endif else begin
                  if is_dcs and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
                     qty = [[[(*self.dataStrPtr).qty]],[[temporary(qty)]]]
                     err = [[[(*self.dataStrPtr).err]],[[temporary(dqty)]]]
                     n_data = n_elements(qty)
                     weight = [(*self.dataStrPtr).weight,temporary(weight)]
                     ang    = (*self.dataStrPtr).rotation
                     filename = [(*self.dataStrPtr).filename,open_file]
                     e      = (*self.dataStrPtr).e
                     eief   = (*self.dataStrPtr).eief
                     info   = (*self.dataStrPtr).info
                     ewid   = (*self.dataStrPtr).ewid
                     path   = (*self.dataStrPtr).filepath
                     time   = (*self.dataStrPtr).time
                     specify = (*self.dataStrPtr).specify
                     temperature = [[(*self.dataStrPtr).temperature],[temperature]] 
                     if (self.extravaxis_yn[2]) then hfield= [(*self.dataStrPtr).hfield,hfield] else hfield = (*self.dataStrPtr).hfield
                     ptr_free,self.dataStrPtr
                     self.ftoolflag[4] = (self.ftoolflag[4] or (n_elements(qty) ne n_data))
                     self.dataStrPtr = ptr_new({qty:temporary(qty),err:temporary(err),weight:temporary(weight),temperature:temporary(temperature),$
                         hfield:temporary(hfield),specify:temporary(specify),rotation:temporary(ang),info:info,eief:temporary(eief),ewid:temporary(ewid),$
                         e:temporary(e),time:time,filename:temporary(filename),filepath:path},/no_copy)
                  endif else begin
                     if n_elements(weight) eq 0 then weight = 1.0
                     (*self.dataStrPtr).qty = ((*self.dataStrPtr).qty)*((*self.dataStrPtr).weight)+qty*weight
                     (*self.dataStrPtr).err = sqrt((((*self.dataStrPtr).err)*((*self.dataStrPtr).weight))^2+(dqty*weight)^2)
                     (*self.dataStrPtr).weight = ((*self.dataStrPtr).weight)+weight
                     (*self.dataStrPtr).qty = ((*self.dataStrPtr).qty)/((*self.dataStrPtr).weight)
                     (*self.dataStrPtr).err = ((*self.dataStrPtr).err)/((*self.dataStrPtr).weight)
                  endelse
               endelse
            endelse
          end
        'ftype_':   begin
            if self.ftype eq type then break   ;no change
            widget_control,self.tlb,update=0
            case self.instrname of    ;giving warning of losing the detector offset value
                 'macs':  warning = (*self.detPosPtr).two_theta[0] ne 0
                 'wand':  warning = (*self.detPosPtr).two_theta[0] ne -11.236
                 else:
            endcase
            if keyword_set(warning) then ok = dialog_message(dialog_parent=self.tlb,'The detector offset angle info will be lost.',/info,/center)
            self.ftype = type
            ;remove all relevant data
            if (~ keyword_set(init)) then begin
               ptr_free,self.detPosPtr,self.dataStrPtr,self.projStrPtr
               self->set_display_button
               ;clear all possible background data and mask
               self.monrate = 1.0
               ptr_free,self.eff,self.eff1,self.bgtchnPtr,self.bgratePtr,self.bgdetsPtr,self.bgdata,self.mask,self.mask_ar,self.a3mask,self.bga3mask
               self.maskstr = ''
               self.dcsdetmask[*] = 0
               self.macs_ptaionly = 0
            endif
            self->dm_filemenu,keep_extravaxis=keep_extravaxis       ;reset file menu and other menus
            ;change the about ... information in the help menu
            (*self.info)[0] = '--Mslice for '+(*self.ftypename)[type]+' --'
            widget_control,self.tlb,tlb_set_title=(*self.info)[0]
            self->my_widget_control,'selfBut',set_value='About Mslice for '+(*self.ftypename)[type]
            if obj_valid(self.fileSel) then self.fileSel->set_filter,'.'+(*self.ftypeext)[type]          ;change file filter
            widget_control,self.tlb,/map,/update
          end
        'dm_filesel_newfilter': begin
            ind = where('.'+strlowcase(*self.ftypeext) eq strlowcase(event.filter),count)
            if count eq 1 then ok = self->dm_filetools('ftype_'+dm_to_string(ind[0]))
          end

        ;DCS tools
        'dcs_cspebut':   dm_dcs2spe,event,workdir=self.dirs[1],datadir=self.dirs[0],ncnrftp=self.ncnrftp  ;convert dcs files to spe files
        'cryst_algnbut': dcs_cryst_align,event,[self.latt_parm,self.orie_u,self.orie_v],workdir=self.dirs[1],datadir=self.dirs[0],bgcolor=(['black','white'])[self.pbgcol_bw],$
                            macs=is_macs,wand=is_wand,render_method=self.ptrm_choice,tickdir=self.tdir_choice,ncnrftp=self.ncnrftp   ;also available as macs and wand file tool
        'dcs_convbut':   begin
                         dm_ereslconv,event,workdir=self.dirs[1],bgcolor=(['black','white'])[self.pbgcol_bw],noinfo=self.ftoolflag[0],notooltip=self.pnotooltip,resltype=2,lattparm=self.latt_parm 
                         self.ftoolflag[0] = 1b  ;no dialog message next time
                         end
        'dcs_patchbut':  dm_patch,event,dataDir=self.dirs[0],workDir=self.dirs[1],ncnrftp=self.ncnrftp
        'dcs_wbeambut':  dcs_whitebeam,group_leader=self.tlb,workDir=self.dirs[1],dataDir=self.dirs[0],DAVETool=self.DAVETool
        'panalbut':      dm_polanalysis,event,dataDir=self.dirs[0],workDir=self.dirs[1],ncnrftp=self.ncnrftp,dcs=is_dcs,macs=is_macs
        'dm_copyftpbut': dm_copyftp,event,dataDir=self.dirs[0],workDir=self.dirs[1],filter='.'+(*self.ftypeext)[self.ftype]   ;also available as macs file tool
        ;MACS tools
        'macs_seqbut' :  dm_macs_sequencer,event,workdir=self.dirs[1],textboxsize=self.macs_seq_geom[0:1],lefthanded=self.macs_seq_geom[2]
        'macs_patchbut': dm_patch,event,dataDir=self.dirs[0],workDir=self.dirs[1],/macs
        ;NXSPE tools
        'nxspe_savephxbut': self->save_phx
        ;put the events of your own instrument specific tools below

        else: begin
            if stregex(eventname,'emptbut[0-9]+',/boolean,/fold_case) then $  ;extra empty can sets
               handled = self->dm_filetools('emptbut',event=event,emptset=dm_to_number(strmid(eventname,7))-1,angle=angle,scriptmode=scriptmode,filename=filename,directory=directory) $ 
            else handled = 0b      ;event not handled
        endelse
    endcase

    ;set file history for file loading events
    if (n_file ne 0) and (n_elements(open_path) ne 0) and (n_elements(comment) ne 0) and ~keyword_set(error) then begin
       comment = comment[0]
       if (n_elements(info) ne 0) and stregex(comment,'T ?= *$',/boolean) then begin  ;patch the comment ending with "T=" with the temperature set point
          field = tag_names(info)
          loc  = where(strupcase(field) eq 'TEMPERATURE',count)
          if count gt 0 then comment = comment+dm_to_string(info.temperature)+' K'
       endif
       if (n_elements(info) ne 0) and stregex(comment,'B ?= *$',/boolean) then begin  ;patch the comment ending with "B=" with the magnetic field value
          field = tag_names(info)
          loc  = where(strupcase(field) eq 'MAGFIELD',count)
          if count gt 0 then comment = comment+dm_to_string(info.magfield,res=4)+' T'
       endif
       if keyword_set(set_legend) and (~self.lock_title) then begin
          legend = dm_getlegend(comment,legend)
          self->script,'set_parameter',name='legdLab',value=legend
       endif
       if keyword_set(set_title) and (~self.lock_title) then self->script,'set_parameter',name='titlLab',value=comment
       if strlen(comment) eq 0 then comment=(*self.ftypename)[self.ftype]+' data '+open_file[0]+((n_file gt 1)?'...':'')
       self->set_filehistory,comment,open_path,open_file,ftpobj=ftpobj,eventname=eventname,emptset=emptset
       self->dm_save_parameter,self.parmname,/quiet
    endif

    if obj_valid(ftpboj) then ftpobj->CloseConnections

    return,handled
end

