; $Id: dm_ereslconv.pro,v 1.13 2017/05/10 14:53:38 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_ereslconv
;
; PURPOSE:
;  This program convolutes data with DCS energy or Q resolution
;
; CATEGORY:
;  DCS data tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2018
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;x,y,fwhm are 1D arrays with the same size
function dm_1dres_convol,x,y,fwhm,error=error
    error  = 1b 
    nx     = n_elements(x)
    dx     = x[1:nx-1] - x[0:nx-2]
    dx     = abs([dx[0],dx])
    if nx eq 0 then return,0
    if (nx ne n_elements(y)) or (nx ne n_elements(fwhm)) then return,0
    ; Inflate the x,dx,y vector to an array that is NX by NX.
    dxmat  = rebin(dx,nx,nx,/sample)
    xmat   = rebin(x,nx,nx,/sample)
    ymat   = rebin(y,nx,nx,/sample)
    sigma  = fwhm/sqrt(8*alog(2d))
    ; Inflate the width from an NX element vector to an array
    ; that is NX by NX.
    sigmat = rebin(sigma,nx,nx,/sample)
    ; Calculate the difference matrix for the argument in the Gaussian.
    diff   = transpose(xmat)-xmat
    ; Resolution matrix
    rmat   = 1/sqrt(2.0*!dpi*sigmat^2)*exp(-0.5*(diff/sigmat)^2)
    ; Clear error flag
    error  = 0b 
    ; Integral
    return,total(ymat*rmat*dxmat,1,/nan)
end

;use 2D separable convolution
function dm_2dres_convol,x,y,int,xfwhm,yfwhm,error=error,message=message,dialog_parent=dialog_parent
    error = 1b
    nx  = n_elements(x)
    ny  = n_elements(y)
    ni  = n_elements(int)
    nxw = n_elements(xfwhm)
    nyw = n_elements(yfwhm)
    if (nx eq 0) or (ny eq 0) or (ni eq 0) or (nxw eq 0) or (nyw eq 0) then return,0
    if ni ne nx*ny then return,0
    
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,/center)
       if obj_valid(mesg) then obj_destroy,mesg
       return,0
    end
    
    if n_elements(message) ne 1 then message = 'Calculate 2D resolution convolution. Please wait ...'
    mesg = obj_new('dm_progress',title='Calculate ...',message=message,group_leader=dialog_parent,/nostop) 
    
    out = int
    
    ;x resolution
    if nxw eq nx then begin
       for i=0L,ny-1 do begin
           out[*,i] = dm_1dres_convol(x,out[*,i],xfwhm)
           mesg->update,message=message+'Q resl: '+dm_to_string(i+1)+'/'+dm_to_string(ny)
       endfor
    endif else begin
       for i=0L,ny-1 do begin
           out[*,i] = dm_1dres_convol(x,out[*,i],xfwhm[*,i])
           mesg->update,message=message+'Q resl: '+dm_to_string(i+1)+'/'+dm_to_string(ny)
       endfor
    endelse
    
    ;y resolution
    out = transpose(out)
    if nyw eq ny then begin
       for i=0L,nx-1 do begin
           out[*,i] = dm_1dres_convol(y,out[*,i],yfwhm)
           mesg->update,message=message+'E resl: '+dm_to_string(i+1)+'/'+dm_to_string(nx)
       endfor
    endif else begin
       yfwhm = transpose(yfwhm)
       for i=0L,nx-1 do begin
           out[*,i] = dm_1dres_convol(y,out[*,i],yfwhm[*,i])
           mesg->update,message=message+'E resl: '+dm_to_string(i+1)+'/'+dm_to_string(nx)
       endfor
       yfwhm = transpose(yfwhm)
    endelse
    out = transpose(out)
    
    ;reset NANs
    ind = where(finite(int,/nan),count) 
    if count ne 0 then out[ind] = !values.f_nan

    error = 0b  ;clear error flag
    obj_destroy,mesg
    return,out
end

pro dm_ereslconv_readfile,file,xdat=xdat,ydat=ydat,idat=idat,ierr=ierr,error=error,dialog_parent=dialog_parent,type=type,qtype=qtype,qaxis=qaxis,qoffset=qoffset,lattparm=lattparm
    ;determine the q-type 
    if type gt 0 then begin
       qtype = dm_dialog_input('Q type:',default=0,dialog_parent=dialog_parent,/align_center,cancel=cancel,droplist_content=ptr_new(['Powder','Single Crystal']),is_droplist=1,xsize=90,/return_number)
       if keyword_set(cancel) then error = 1 $
       else if qtype eq 1 then begin
          info = ['Lattice parameters are a, b, c, alpha, beta, and gamma.','Q-axis and Q-offset are in reciprocal space hkl values.','For example, if the Q scan is along (h,h,0.5),',$
                  'then Q-axis will be (1,1,0), and Q-offset will be (0,0,0.5).']
          name = ['Lattice Parameters:','Q-axis:','Q-offset:']
          notok = 1b & splitstr = '[, ]'
          ans = [(total(finite(lattparm,/nan)) eq 0)?dm_to_string(lattparm,sep=', '):'',(total(finite(qaxis,/nan)) eq 0)?dm_to_string(qaxis,sep=', '):'',(total(finite(qoffset,/nan)) eq 0)?dm_to_string(qoffset,sep=', '):'']
          while notok do begin
             ans = dm_dialog_input(name,default=ans,dialog_parent=dialog_parent,/align_center,info=info,cancel=cancel,xsize=220)
             if keyword_set(cancel) then begin
                error = 1 & break
             endif
             lattparm = dm_to_number(strsplit(ans[0],splitstr,/extract),/float)
             qaxis    = dm_to_number(strsplit(ans[1],splitstr,/extract),/float)
             qoffset  = dm_to_number(strsplit(ans[2],splitstr,/extract),/float)
             if n_elements(qoffset) eq 1 then begin
                if (qoffset[0] eq 0) or finite(qoffset,/nan) then qoffset = [0.,0,0]
             endif
             notok = ((total(finite(lattparm)) ne 6) or (total(finite(qaxis)) ne 3) or (total(finite(qoffset)) ne 3))
          endwhile
       endif
       if keyword_set(error) then return
    endif 
    error = 0
    
    mesg = obj_new('dm_progress',title='Loading data ...',message='Loading '+file[0]+'. Please wait ...',group_leader=dialog_parent,/nostop) 
    
    catch, myerror
    if myerror ne 0 then begin
       error = 1
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,/center)
       if n_elements(unit) ne 0 then free_lun,unit
       if obj_valid(mesg) then obj_destroy,mesg
       return
    end
    
    if type lt 2 then begin 
       openr,unit,file[0],/get_lun,error=error
       if error ne 0 then begin
          ok = dialog_message("Can't read file "+file[0]+'.',dialog_parent=dialog_parent,/center,/error)
          obj_destroy,mesg
          return
       endif
       tmp = '' & line = 0
       while (~ eof(unit)) do begin
          readf,unit,tmp & tmp = strtrim(tmp,2)
          if strlen(tmp) eq 0 then continue
          tmp1 = strsplit(tmp,' ,'+string(9b),/extract)
          tmp1 = dm_to_number(tmp1,/float,count=count)
          if total(finite(tmp1)) eq count then begin
             if n_elements(xdat) eq 0 then begin
                xdat = tmp1[0]
                idat = tmp1[1]
                if n_elements(tmp1) gt 2 then ierr = tmp1[2]
             endif else begin
                xdat = [xdat,tmp1[0]]
                idat = [idat,tmp1[1]]
                if n_elements(tmp1) gt 2 then ierr = [ierr,tmp1[2]]
             endelse
          endif
       endwhile
       free_lun,unit
    endif else begin
       dm_load_spe,file[0],zdat=idat,xdat=xdat,ydat=ydat,error=error,group_leader=dialog_parent
       if keyword_set(error) then begin
          if obj_valid(mesg) then obj_destroy,mesg
          return
       endif
       ind = where(idat le -1e20,count)
       if count gt 0 then idat[ind] = !values.f_nan
       ny = n_elements(ydat) & nz = size(idat,/dimensions)
       if ny eq nz[1]+1 then ydat = (ydat[0:(ny-2)]+ydat[1:(ny-1)])/2.0
    endelse
    
    obj_destroy,mesg
end

pro dm_ereslconv_writefile,file=file,xdat=xdat,ydat=ydat,idat=idat,path=path,resulttype=resulttype,dialog_parent=dialog_parent,xtit=xtit
    file = dm_choose_file('',/write,dialog_parent=dialog_parent,title='Save '+(['convolution','E resolution','Q resolution'])[resulttype]+$
           ' result as an ASCII'+(['',' SPE'])[n_elements(ydat) gt 0]+' file.',path=path,file=file)
    if strlen(file) eq 0 then return
    openw,unit,file,/get_lun,error=openerr
    if openerr ne 0 then begin
       ok = dialog_message("Can't write in "+file[0],/error,/center,dialog_parent=dialog_parent)
    endif else begin
       widget_control,/hourglass  ;busy signal
       nxdat = n_elements(xdat) & nydat = n_elements(ydat) & nidat = n_elements(idat)
       if n_elements(ydat) eq 0 then begin  ;1D
          for i=0L,nxdat-1 do printf,unit,xdat[i],' ',idat[i]
          free_lun,unit
       endif else begin         ;2D
          free_lun,unit
          ind  = where(finite(idat,/nan),count)
          if count ne 0 then begin
             idat[ind] = -1e20
          endif
          ierr = fltarr(size(idat,/dimension))
          if n_elements(xtit) eq 0 then xtit = '|Q| (AA^-1)'
          dm_write_spe,file,xdat,ydat,idat,ierr,xtit=xtit,ytit='E (meV)',ztit=(['Intensity','E resl (meV)','Q resl (AA^-1)'])[resulttype],group_leader=dialog_parent
       endelse             
    endelse
end

pro dm_ereslconv_resetbutton,pState,clear=clear
    if keyword_set(clear) then begin
       ptr_free,(*pState).xPtr,(*pState).yPtr,(*pState).iPtr,(*pState).convPtr
       (*pState).plotWin->erase
       (*pState).colind = 0
       (*pState).iran[*] = !values.f_nan
    endif
    widget_control,(*pState).calcBut, sensitive=ptr_valid((*pState).xPtr)
    widget_control,(*pState).saveBut, sensitive=ptr_valid((*pState).convPtr)
    widget_control,(*pState).clearBut,sensitive=ptr_valid((*pState).xPtr)
    widget_control,(*pState).printBut,sensitive=ptr_valid((*pState).xPtr)
end

pro dm_ereslconv_event,event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=pState

    if (*pState).enotebook then $
       IF DAVE_SET_FOCUS(EVENT) THEN RETURN

    widget_type = widget_info(event.id,/type)
    case widget_type of
       3:    begin                                               ;text box
             widget_control,event.id,get_value=tmp_value
             tmp_value = strtrim(tmp_value,2)
             end
       8:    tmp_value = widget_info(event.id,/droplist_select)  ;droplist
       12:   tmp_value = event.index                             ;combobox
       else: tmp_value = 0
    endcase

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=(*pState).tlb,!error_state.msg,/error,/center)
       catch,/cancel
       return
    end
 
    case (event.id) of
       (*pState).tlb:    widget_control,event.id,scr_xsize=(*pState).geom[0],scr_ysize=(*pState).geom[1]
       (*pState).ereslBut: if (*pState).resltype ne 0 then begin
            (*pState).resltype = 0
            dm_toggle_menubut,check=(*pState).ereslBut,uncheck=[(*pState).qreslBut,(*pState).qereslBut]
            widget_control,(*pState).ereslrBut,sensitive=((*pState).resltype ne 1)
            widget_control,(*pState).qreslrBut,sensitive=((*pState).resltype ne 0)
            if (*pState).resulttype eq 2 then (*pState).resulttype = 0
            dm_toggle_menubut,check=([(*pState).convrBut,(*pState).ereslrBut,(*pState).qreslrBut])[(*pState).resulttype],$
               uncheck=([(*pState).convrBut,(*pState).ereslrBut,(*pState).qreslrBut])[((*pState).resulttype+[1,2]) mod 3]
            if float(!version.release) ge 5.6 then begin  ;tips for some widget buttons
               widget_control,(*pState).loadBut,tooltip='Select the file from the left, then load the '+(['(E, I)','(Q, I)'])[(*pState).resltype]+' 2-column ascii data file.'
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['convolution of the ',''])[(*pState).resulttype]+'DCS energy resolution '+(['with','of'])[(*pState).resulttype]+' the loaded data.'
            endif
            widget_control,(*pState).tlb,tlb_set_title='DCS Energy Resolution Convolution Utility'
            if widget_info((*pState).enLab,/valid_id) then widget_control,(*pState).enLab,/destroy
            if widget_info((*pState).enTxt,/valid_id) then widget_control,(*pState).enTxt,/destroy
            dm_ereslconv_resetbutton,pState,/clear
            (*pState).plotWin->setproperty,xtit=(*pState).xtit[(*pState).resltype],ytit='Intensity (arb. unit)',title='',/nodraw
          end  
       (*pState).qreslBut: if (*pState).resltype ne 1 then begin
            (*pState).resltype = 1
            dm_toggle_menubut,check=(*pState).qreslBut,uncheck=[(*pState).ereslBut,(*pState).qereslBut]
            widget_control,(*pState).ereslrBut,sensitive=((*pState).resltype ne 1)
            widget_control,(*pState).qreslrBut,sensitive=((*pState).resltype ne 0)
            if (*pState).resulttype eq 1 then (*pState).resulttype = 0
            dm_toggle_menubut,check=([(*pState).convrBut,(*pState).ereslrBut,(*pState).qreslrBut])[(*pState).resulttype],$
               uncheck=([(*pState).convrBut,(*pState).ereslrBut,(*pState).qreslrBut])[((*pState).resulttype+[1,2]) mod 3]
            if float(!version.release) ge 5.6 then begin  ;tips for some widget buttons
               widget_control,(*pState).loadBut,tooltip='Select the file from the left, then load the '+(['(E, I)','(Q, I)'])[(*pState).resltype]+' 2-column ascii data file.'
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['convolution of the ','',''])[(*pState).resulttype]+'DCS Q resolution '+(['with','of','of'])[(*pState).resulttype]+' the loaded data.'
            endif
            widget_control,(*pState).tlb,tlb_set_title='DCS Q Resolution Convolution Utility'
            (*pState).enLab = widget_label((*pState).ccol,value='E transfer (meV)')
            (*pState).enTxt = widget_text((*pState).ccol,value=dm_to_string((*pState).en),/editable,/all_events)
            geom  = widget_info((*pState).exitBut,/geometry)
            widget_control,(*pState).enTxt,scr_xsize=geom.scr_xsize
            dm_ereslconv_resetbutton,pState,/clear
            (*pState).plotWin->setproperty,xtit=(*pState).xtit[(*pState).resltype],ytit='Intensity (arb. unit)',title='',/nodraw
          end  
       (*pState).qereslBut: if (*pState).resltype ne 2 then begin
            (*pState).resltype = 2
            dm_toggle_menubut,check=(*pState).qereslBut,uncheck=[(*pState).ereslBut,(*pState).qreslBut]
            widget_control,(*pState).ereslrBut,/sensitive
            widget_control,(*pState).qreslrBut,/sensitive
            if float(!version.release) ge 5.6 then begin  ;tips for some widget buttons
               widget_control,(*pState).loadBut,tooltip='Select the file from the left, then load the (Q, E, Intensity) SPE file.'
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['2D convolution of the ','',''])[(*pState).resulttype]+'DCS '+$
                  (['Q and E','E','Q'])[(*pState).resulttype]+' resolution '+(['with','of','of'])[(*pState).resulttype]+' the loaded data.'
            endif
            widget_control,(*pState).tlb,tlb_set_title='DCS Q&E 2D Resolution Convolution Utility'
            if widget_info((*pState).enLab,/valid_id) then widget_control,(*pState).enLab,/destroy
            if widget_info((*pState).enTxt,/valid_id) then widget_control,(*pState).enTxt,/destroy
            dm_ereslconv_resetbutton,pState,/clear
            (*pState).plotWin->setproperty,xtit=(*pState).xtit[(*pState).resltype],ytit='E (meV)',itit='Intensity (arb. unit)',/nodraw
          end  
       (*pstate).convrBut: if (*pState).resulttype ne 0 then begin   
            (*pState).resulttype = 0
            dm_toggle_menubut,check=(*pState).convrBut,uncheck=[(*pState).ereslrBut,(*pState).qreslrBut]
            if float(!version.release) ge 5.6 then begin
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['convolution of the ','',''])[(*pState).resulttype]+'DCS '+$
                    (['E','Q',(['Q and E','E','Q'])[(*pState).resulttype]])[(*pState).resltype]+' resolution '+(['with','of','of'])[(*pState).resulttype]+' the loaded data.'
               widget_control,(*pState).saveBut,tooltip='Save the last '+(['convolution','E resolution','Q resolution'])[(*pState).resulttype]+' result in an ascii '+$
                    (['','','SPE '])[(*pState).resltype]+'file.'      
            endif
            ptr_free,(*pState).convPtr
            dm_ereslconv_resetbutton,pState
          end
       (*pstate).ereslrBut: if (*pState).resulttype ne 1 then begin   
            (*pState).resulttype = 1
            dm_toggle_menubut,check=(*pState).ereslrBut,uncheck=[(*pState).convrBut,(*pState).qreslrBut]
            if float(!version.release) ge 5.6 then begin
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['convolution of the ','',''])[(*pState).resulttype]+'DCS '+$
                    (['E','Q',(['Q and E','E','Q'])[(*pState).resulttype]])[(*pState).resltype]+' resolution '+(['with','of','of'])[(*pState).resulttype]+' the loaded data.'
               widget_control,(*pState).saveBut,tooltip='Save the last '+(['convolution','E resolution','Q resolution'])[(*pState).resulttype]+' result in an ascii '+$
                    (['','','SPE '])[(*pState).resltype]+'file.'      
            endif
            ptr_free,(*pState).convPtr
            dm_ereslconv_resetbutton,pState
          end   
       (*pstate).qreslrBut: if (*pState).resulttype ne 2 then begin   
            (*pState).resulttype = 2
            dm_toggle_menubut,check=(*pState).qreslrBut,uncheck=[(*pState).convrBut,(*pState).ereslrBut]
            if float(!version.release) ge 5.6 then begin
               widget_control,(*pState).calcBut,tooltip='Calculate the '+(['convolution of the ','',''])[(*pState).resulttype]+'DCS '+$
                    (['E','Q',(['Q and E','E','Q'])[(*pState).resulttype]])[(*pState).resltype]+' resolution '+(['with','of','of'])[(*pState).resulttype]+' the loaded data.'
               widget_control,(*pState).saveBut,tooltip='Save the last '+(['convolution','E resolution','Q resolution'])[(*pState).resulttype]+' result in an ascii '+$
                    (['','','SPE '])[(*pState).resltype]+'file.'      
            endif
            ptr_free,(*pState).convPtr
            dm_ereslconv_resetbutton,pState
          end   
       (*pState).helpBut: begin
            info = ['Choose either the Q and/or E resolution in Option->Resolution Type menu. Choose the Result Type to be either the convolution or the fwhm resolution only.',$
                    '',$
                    'Load the data to be convoluted by selecting the data file on the left first, then click Load Data button. The data file could be "E (or Q), Intensity" '+$
                    '2-column ascii file, or "E (or Q), Intensity, Error (ignored)" 3-column ascii file, or "Q, E, Intensity" ascii SPE file. Column data need to be '+$
                    'separated by comma or space. SPE file is described in the appendix of "How to Add a New File Type" in the mslice help menu. Use -1e20 as the intensity of empty grids. ',$
                    '',$
                    'For the 1D Q vs Intensity file or 2D Q&E SPE file, the Q can be either absolute Q values (powder) or reciprocal space hkl values (single crystal). In the latter case, the lattice parameters, '+$
                    'Q-axis definition, and the Q-offset values need to be specified. For example, if the Q axis is along (h,h,0.5), then Q-axis is (1,1,0) and Q-offset is (0,0,0.5).',$
                    '',$
                    'Enter the wavelength, DCS resolution mode, and master speed (which is almost always 20000) the same way as setting a DCS wavelength. For Q resolution convolution, '+$
                    'the corresponding single energy transfer value also needs to be entered.',$
                    '',$
                    'Click Calculate button to calculate the resolution convolution or the fwhm resolution itself.',$
                    '',$
                    'Click Save Result button to save the last convolution result or resolution as "E (or Q), Convoluted Intensity or Resolution" 2-column ascii file or '+$
                    '"Q, E, Convoluted Intensity or Resolution" SPE file. Note that the Q resolution for single crystal is still the corresponding Q resolution for absolute Q values.',$
                    '',$
                    'Right click over the upper left corner of the plot window to access the plot menu.'] 
            ok = dialog_message(info,/center,/info,dialog_parent=(*pState).tlb,title='About DCS Resolution Convolution Utility')
          end   
       (*pState).loadBut: begin
            (*pState).filesel->getproperty,path=open_path,file=open_file,sep=sep
            if n_elements(open_file) eq 0 then begin
               ok = dialog_message('Please select '+(['an (E, I) 2-column ascii','a (Q, I) 2-column ascii','a (Q, E, Intensity) SPE'])[(*pState).resltype]+' data file from the left first.',$
                    /error,/center,dialog_parent=(*pState).tlb)
               break
            endif
            if ((*pState).path ne open_path) or ((*pState).filename ne open_file[0]) then (*pState).iran[*] = !values.f_nan
            (*pState).path     = open_path
            (*pState).filename = open_file[0]
            lattparm = (*pState).lattparm & qaxis = (*pState).qaxis  & qoffset = (*pState). qoffset
            dm_ereslconv_readfile,open_path+sep+open_file[0],xdat=xdat,ydat=ydat,idat=idat,ierr=ierr,error=error,dialog_parent=(*pState).tlb,type=(*pState).resltype,qtype=qtype,qaxis=qaxis,qoffset=qoffset,lattparm=lattparm
            nxdat = n_elements(xdat) & nydat = n_elements(ydat) & nidat = n_elements(idat) & nierr = n_elements(ierr)
            if (error eq 0) and (nxdat gt 1) then begin 
               if (*pState).resltype ne 0 then begin
                  (*pState).qtype = qtype
                  if qtype eq 1 then begin
                     (*pState).lattparm = lattparm
                     (*pState).qaxis    = qaxis
                     (*pState). qoffset = qoffset
                     if qaxis[0] ne 0 then      label = 'H' $
                     else if qaxis[1] ne 0 then label = 'K' $
                     else if qaxis[2] ne 0 then label = 'L'
                     label = dm_set_viewlabel(label,qaxis,qoffset)
                  endif else label = ''
                  (*pState).xtit[(*pState).resltype] = (['|Q| (\AA!U-1!N)',label])[qtype]
               endif
               if (*pState).resltype eq 2 then begin
                  (*pState).plotWin->setproperty,xdat=xdat,ydat=ydat,zdat=idat,/usepolygon,title='',iran=(*pState).iran,/nodraw
               endif else begin
                  if ptr_valid((*pState).xPtr) then begin       
                     if nierr eq nxdat then begin
                        (*pState).plotWin->add_plot,xdat,idat,yerr=ierr,linestyle='no line',color=(*pState).cols[(*pState).colind],legend='Input',psym='circle',layer=0 
                     endif else begin
                        (*pState).plotWin->add_plot,xdat,idat,linestyle='no line',color=(*pState).cols[(*pState).colind],legend='Input',psym='circle',layer=0
                     endelse
                  endif else begin
                     if nierr eq nxdat then begin
                        (*pState).plotWin->setproperty,xdat=xdat,ydat=idat,yerr=ierr,linestyle='no line',color=(*pState).cols[(*pState).colind],legend='Input',psym='circle',layer=0,/nodraw
                     endif else begin
                        (*pState).plotWin->setproperty,xdat=xdat,ydat=idat,linestyle='no line',color=(*pState).cols[(*pState).colind],legend='Input',psym='circle',layer=0,/nodraw 
                     endelse
                  endelse
                  (*pState).colind = ((*pState).colind+1) mod 6
               endelse
               ptr_free,(*pState).xPtr,(*pState).yPtr,(*pState).iPtr
               (*pState).xPtr = ptr_new(xdat)
               (*pState).iPtr = ptr_new(idat)
               if n_elements(ydat) ne 0 then (*pState).yPtr = ptr_new(ydat)
               (*pState).plotWin->setproperty,xtit=(*pState).xtit[(*pState).resltype],/nodraw
               (*pState).plotWin->draw
               dm_ereslconv_resetbutton,pState
            endif
          end
       (*pState).calcBut: begin
            if finite ((*pState).wl,/nan) then begin
               ok = dialog_message('Please enter the wavelength.',/error,/center,dialog_parent=(*pState).tlb)
               break
            endif else if (*pState).wl le 0 then begin
               ok = dialog_message('Invalid wavelength.',/error,/center,dialog_parent=(*pState).tlb)
               break
            endif 
            ei = 81.8042/((*pState).wl)^2 ;angstrom->meV
            if finite((*pState).mspeed,/nan) then begin
               ok = dialog_message('Please enter the master speed.',/error,/center,dialog_parent=(*pState).tlb)
               break
            endif else if (*pState).mspeed le 0 then begin
               ok = dialog_message('Invalid master speed.',/error,/center,dialog_parent=(*pState).tlb)
               break
            endif
            if (*pState).resltype eq 1 then begin
               if finite((*pState).en,/nan) then begin
                  ok = dialog_message('Please enter the energy transfer.',/error,/center,dialog_parent=(*pState).tlb)
                  break
               endif else if (*pState).en ge ei then begin
                  ok = dialog_message('Energy transfer is larger than incident energy '+dm_to_string(ei,resolution=3)+' meV.',/error,/center,dialog_parent=(*pState).tlb)
                  break
               endif
            endif 
            
            x = *(*pState).xptr
            i = *(*pState).iptr
            if ptr_valid((*pState).yptr) then y = *(*pState).yptr
            
            case (*pState).resltype of 
              0: begin      ;E resolution
                 ;select the energy range <Ei
                 index = where(x lt ei,count)
                 if count eq 0 then return
                 x = x[index] & qs = x
                 i = i[index]
                 ;calculate the fwhm  
                 ef   = ei-x
                 wlf  = sqrt(81.8042/ef)
                 fwhm = dcs_jcc_resn((*pState).wl,wlf,(*pState).reslmode,(*pState).mspeed)/1000. ;ueV->meV
                 end
              1: begin      ;Q resolution
                 ef   = ei-(*pState).en
                 wlf  = sqrt(81.8042/ef)
                 ki   = 2.*!pi/(*pState).wl
                 kf   = 2.*!pi/wlf
                 qs   = x
                 if (*pState).qtype eq 1 then begin
                    for j=0L,n_elements(x)-1 do qs[j] = dcs_cryst_align_qlength1(qs[j]*(*pState).qaxis+(*pState).qoffset,(*pState).lattparm)
                 endif
                 ;select q transfer   |ki-kf| <q<ki+kf
                 index = where((qs lt (ki+kf)) and (qs gt abs(ki-kf)),count)
                 if count eq 0 then return
                 x  = x[index]
                 i  = i[index]
                 qs = qs[index]
                 
                 ;calculate the fwhm
                 fwhm   = dcs_jcc_resn((*pState).wl,wlf,(*pState).reslmode,(*pState).mspeed,eifwhm=eifwhm)
                 dei    = eifwhm/1000.                 ;ueV->meV
                 def    = sqrt(fwhm^2-eifwhm^2)/1000.  ;ueV->meV
                 dki    = ki/2./ei*dei
                 dkf    = kf/2./ef*def
                 cos2th = (ki^2+kf^2-qs^2)/2./ki/kf
                 d2th   = 0.25*!dtor   ;uncertainty for 2theta is a quarter of a degree
                 fwhm   = sqrt((ki-kf*cos2th)^2*dki^2+(kf-ki*cos2th)^2*dkf^2+(ki*kf*d2th)^2*(1-cos2th^2))/qs 
                 ;Q^2 = ki^2+kf^2-2.*ki*kf*cos(2th)
                 end
              2: begin      ;Q&E resolution
                 ;select the energy range <Ei
                 index = where(y lt ei,ny)
                 if ny eq 0 then return
                 y = y[index]
                 i = i[*,index]
                 nx = n_elements(x)
                 ;calculate energy resolution yfwhm
                 ef  = ei-y
                 wlf = sqrt(81.8042/ef)
                 yfwhm = dcs_jcc_resn((*pState).wl,wlf,(*pState).reslmode,(*pState).mspeed)/1000. ;ueV->meV
                 ;calculate q resolution xfwhm
                 xfwhm = replicate(!values.f_nan,nx,ny)
                 ki    = 2.*!pi/(*pState).wl
                 d2th  = 0.25*!dtor   ;uncertainty for 2theta is a quarter of a degree
                 qs    = x
                 if (*pState).qtype eq 1 then begin
                    for j=0L,nx-1 do qs[j] = dcs_cryst_align_qlength1(qs[j]*(*pState).qaxis+(*pState).qoffset,(*pState).lattparm)
                 endif
                 for j=0L,ny-1 do begin
                     ef  = ei-y[j]
                     wlf = sqrt(81.8042/ef)
                     kf  = 2.*!pi/wlf
                     ;select q transfer   |ki-kf| <q<ki+kf
                     index = where((qs lt (ki+kf)) and (qs gt abs(ki-kf)),count,complement=index1,ncomplement=count1)
                     if count1 ne 0 then i[index1,j] = !values.f_nan
                     if count eq 0 then continue
                     ;calculate the fwhm
                     fwhm   = dcs_jcc_resn((*pState).wl,wlf,(*pState).reslmode,(*pState).mspeed,eifwhm=eifwhm)
                     dei    = eifwhm/1000.                 ;ueV->meV
                     def    = sqrt(fwhm^2-eifwhm^2)/1000.  ;ueV->meV
                     dki    = ki/2./ei*dei
                     dkf    = kf/2./ef*def
                     cos2th = (ki^2+kf^2-qs[index]^2)/2./ki/kf
                     xfwhm[index,j] = sqrt((ki-kf*cos2th)^2*dki^2+(kf-ki*cos2th)^2*dkf^2+(ki*kf*d2th)^2*(1-cos2th^2))/qs[index]
                 endfor
                 end
              else:return
            endcase
            
            if (*pState).resltype le 1 then begin   ;1D convolution 
               if (*pState).resulttype eq 0 then begin  ;convolution result
                  ;1D convolution
                  result = dm_1dres_convol(qs,i,fwhm,error=error)
                  if error ne 0 then break
               endif else begin                         ;resolution result only
                  result = fwhm
               endelse
               (*pState).plotWin->add_plot,x,result,linestyle='solid',color=(*pState).cols[(*pState).colind],legend=(['resl convoluted','fwhm '+['E resl (meV)','Q resl (\AA!U-1!N)']])[(*pState).resulttype],layer=0 
               (*pState).colind = ((*pState).colind+1) mod 6
               ptr_free,(*pState).convPtr
               (*pState).convPtr = ptr_new({xdat:x,idat:result})
            endif else begin                        ;2D convolution
               case (*pState).resulttype of
                 0: begin ;convolution result
                    (*pState).plotWin->getproperty,iran=iran 
                    if (total(finite((*pState).iran)) ne 2) or ptr_valid((*pState).convPtr) then (*pState).iran = iran 
                    ;2D convolution
                    result = dm_2dres_convol(qs,y,i,xfwhm,yfwhm,error=error,message='Calculate Q&E resolution convolution. Please wait ...',dialog_parent=(*pState).tlb)
                    if error ne 0 then break
                    iran = (*pState).iran
                    end
                 1: begin ;E-resolution only
                    result = transpose(rebin(yfwhm,ny,nx,/sample))
                    end
                 2: begin ;Q-resolution only
                    result = xfwhm  
                    end
                 else:
               endcase
               ind = where(finite(i,/nan),count)
               if count ne 0 then result[ind] = !values.f_nan
               (*pState).plotWin->setproperty,xdat=x,ydat=y,zdat=result,iran=iran,/usepolygon,title=(['Convoluted','',''])[(*pState).resulttype],$
                     ititle=(['Intensity (arb. unit)','E Resolution (meV)','Q Resolution (\AA!U-1!N)'])[(*pState).resulttype]
               ptr_free,(*pState).convPtr
               (*pState).convPtr = ptr_new({xdat:x,ydat:y,idat:result})
            endelse   
            (*pState).plotWin->draw
            dm_ereslconv_resetbutton,pState
          end   
       (*pState).saveBut: begin
            if (*pState).resltype le 1 then begin
               dm_ereslconv_writefile,file=(['conv_','eresl_','qresl_'])[(*pState).resulttype]+(*pState).filename,xdat=(*(*pState).convPtr).xdat,idat=(*(*pState).convPtr).idat,path=(*pState).path,$
                  resulttype=(*pState).resulttype,dialog_parent=(*pState).tlb 
            endif else begin
               dm_ereslconv_writefile,file=(['conv_','eresl_','qresl_'])[(*pState).resulttype]+(*pState).filename,xdat=(*(*pState).convPtr).xdat,ydat=(*(*pState).convPtr).ydat,idat=(*(*pState).convPtr).idat,$
                  path=(*pState).path,resulttype=(*pState).resulttype,dialog_parent=(*pState).tlb,xtit=(*pState).xtit[(*pState).resltype]
            endelse
            (*pState).filesel->set_path
            (*pState).filesel->getproperty,file=file
            if n_elements(file) ne 0 then (*pState).filesel->clear_file,fileloc=file[0]    
          end
       (*pState).clearBut: begin
            dm_ereslconv_resetbutton,pState,/clear
          end
       (*pState).printBut: begin
            if ptr_valid((*pState).xPtr) then (*pState).plotWin->saveas,'printer'
          end
       (*pState).exitBut: begin
            widget_control,event.handler,/destroy
            return
          end
       (*pState).wlTxt:     (*pState).wl       = dm_to_number(tmp_value,/float)
       (*pState).enTxt:     (*pState).en       = dm_to_number(tmp_value,/float)
       (*pState).reslList:  (*pState).reslmode = tmp_value+1
       (*pState).speedTxt:  (*pState).mspeed   = dm_to_number(tmp_value,/float)   
       else:
    endcase
end

pro dm_ereslconv_Exit,tlb
    widget_control,tlb,get_uvalue=pState,/no_copy
    obj_destroy,(*pState).filesel
    obj_destroy,(*pState).plotWin
    cd,(*pState).root
    ptr_free,(*pState).xPtr,(*pState).yPtr,(*pState).iPtr,(*pState).convPtr
    ptr_free,pState
    widget_control,tlb,/destroy
end

;keywords:
;   workdir:    starting file selector directory
;   bgcolor:    plot window background color, 'white' or 'black'
;   noinfo:     if set, no initial info dialog message
;   notooltip:  if set, no tooltips for the plot window
;   resltype:   0-E resl 1-Q resl 2-Q&E resl
;   resulttype: 0-convolution 1-e resolution only 2-q resolution only
pro dm_ereslconv,event,workdir=workdir,bgcolor=bgcolor,noinfo=noinfo,notooltip=notooltip,resltype=resltype,resulttype=resulttype,lattparm=lattparm
    state={group_leader:                 0L, $  ;group leader
           tlb:                          0L, $  ;top level base
           ccol:                         0L, $  ;place holder
           geom:                    [0L,0L], $  ;geometry
           resltype:                     0b, $  ;resolution type 0-e resl 1-q resl 2-q&e resl
           qtype:                        0b, $  ;q-type for resltype=2, 0-powder,1-single crystal
           lattparm:              fltarr(6), $  ;lattice parameter for qtype = 1
           qaxis:                 fltarr(3), $  ;q-axis hkl value
           qoffset:               fltarr(3), $  ;q-offset hkl value
           ereslBut:                     0L, $  ;e resl menu button
           qreslBut:                     0L, $  ;q resl menu button
           qereslBut:                    0L, $  ;q & e resl menu button
           resulttype:                   0b, $  ;result type 0-convolution, 1-e resolution only, 2-q resolution only
           convrBut:                     0L, $  ;convolution result button
           ereslrBut:                    0L, $  ;E resolution result only button
           qreslrBut:                    0L, $  ;Q resolution result only button
           helpBut:                      0L, $  ;help button
           filesel:               obj_new(), $  ;file selector
           plotWin:               obj_new(), $  ;draw widget
           iran:                    [0e,0e], $  ;save the intensity range
           root:                         '', $  ;starting directory
           path:                         '', $  ;save the current directory 
           filename:                     '', $  ;save the current file name
           cols: ['green','red','blue','magenta','cyan','yellow'], $  ;colors
           colind:                       0s, $  ;index of current color
           loadBut:                      0L, $  ;load file button
           calcBut:                      0L, $  ;calculate button
           saveBut:                      0L, $  ;save result button
           clearBut:                     0L, $  ;clear button
           printBut:                     0L, $  ;print plot button
           exitBut:                      0L, $  ;exit button
           enLab:                        0L, $  ;E transfer label
           enTxt:                        0L, $  ;E transfer text widget
           en:                !values.f_nan, $  ;E transfer in meV
           wlTxt:                        0L, $  ;wavelength text widget
           wl:                !values.f_nan, $  ;wavelength in angstrom
           reslList:                     0L, $  ;resolution mode drop list
           reslmode:                     1s, $  ;resolution mode 1:low, 2:medium
           speedTxt:                     0L, $  ;master speed text widget
           mspeed:                   20000e, $  ;master speed
           xPtr:                  ptr_new(), $  ;pointer to x data
           yPtr:                  ptr_new(), $  ;pointer to y data
           iPtr:                  ptr_new(), $  ;pointer to i data
           convPtr:               ptr_new(), $  ;pointer to calculated convolution data
           xtit:                  strarr(3), $  ;['E (meV)','|Q| (\AA!U-1!N)','|Q| (\AA!U-1!N)']
           title:                        '', $  ;plot title
           enotebook:                    0b  $  ;use Rob's enotebook
    }
    registerName = 'dm_ereslconv'
    if xregistered(registerName) then return  ;only allow one copy to be running at one time
    cd,current = current
    state.root = current
    if n_elements(workdir) eq 0 then workdir=current
    if n_elements(lattparm) eq 6 then state.lattparm = lattparm else state.lattparm[*] = !values.f_nan
    state.qaxis[*]   = !values.f_nan
    state.qoffset[*] = !values.f_nan
    if n_elements(resltype) eq 1 then state.resltype = 0>(resltype)<2 
    if n_elements(resulttype) eq 1 then state.resulttype = 0>(resulttype)<2 
    title = 'DCS '+(['Energy','Q','Q&E 2D'])[state.resltype]+' Resolution Convolution Utility'
    if n_elements(event) ne 0 then begin
       state.group_leader = event.top
       state.tlb  = widget_base(title=title,/row,kill_notify='dm_ereslconv_Exit',/tlb_size_event,group_leader=event.top,map=0,mbar=bar)
    endif else $
       state.tlb  = widget_base(title=title,/row,kill_notify='dm_ereslconv_Exit',/tlb_size_event,map=0,mbar=bar)
    
    ;option menu
    optnmenu = widget_button(bar,value='Option',/menu)   
    typemenu = widget_button(optnmenu,value='Resolution Type',/menu)
    state.ereslBut  = dm_widget_button(typemenu,value='E Resolution (1D)')
    state.qreslBut  = dm_widget_button(typemenu,value='Q Resolution (1D)')
    state.qereslBut = dm_widget_button(typemenu,value='Q and E Resolution (2D)') 
    dm_toggle_menubut,check=([state.ereslBut,state.qreslBut,state.qereslBut])[state.resltype],uncheck=([state.ereslBut,state.qreslBut,state.qereslBut])[(state.resltype+[1,2]) mod 3]
    if (state.resltype eq 0)  and (state.resulttype gt 1) then state.resulttype = 0
    if (state.resltype eq 1)  and (state.resulttype eq 1) then state.resulttype = 0
    typemenu = widget_button(optnmenu,value='Result Type',/menu)
    state.convrBut  = dm_widget_button(typemenu,value='Convolution')
    state.ereslrBut = dm_widget_button(typemenu,value='E Resolution (FWHM)')
    state.qreslrBut = dm_widget_button(typemenu,value='Q Resolution (FWHM)')
    dm_toggle_menubut,check=([state.convrBut,state.ereslrBut,state.qreslrBut])[state.resulttype],uncheck=([state.convrBut,state.ereslrBut,state.qreslrBut])[(state.resulttype+[1,2]) mod 3]
    widget_control,state.ereslrBut,sensitive=(state.resltype ne 1)
    widget_control,state.qreslrBut,sensitive=(state.resltype ne 0)
    ;help menu
    helpmenu = widget_button(bar,value='Help',/menu)   
    state.helpBut = dm_widget_button(helpmenu,value='Help')
    
    if !version.os_family eq 'Windows' then $
       mbar_sep = widget_label(state.tlb,sensitive=0,value=' ',/dynamic_resize,scr_ysize=4)
    
    left = widget_base(state.tlb,/col)
    state.filesel = obj_new('dm_filesel',parent=left,xsize=160,ysize=20,/frame,path=workdir,group_leader=state.tlb)
    if !version.os_family eq 'unix' then $
       center = widget_base(state.tlb,/col,/align_bottom,xpad=0,ypad=0,space=0) $
    else $
       center = widget_base(state.tlb,/col,/align_bottom)
    void = widget_label(center,value='Wavelength ('+string('c5'XB)+')')
    state.wlTxt    = widget_text(center,value=dm_to_string(state.wl),/editable,/all_events)
    void = widget_label(center,value='Resolution')
    state.reslList = dm_widget_droplist(center,value=['Low','Medium'],select=state.reslmode-1)
    void = widget_label(center,value='Master Speed')
    state.speedTxt = widget_text(center,value=dm_to_string(state.mspeed),/editable,/all_events)
    state.ccol = widget_base(center,/col,xpad=0,space=0,ypad=0,/align_bottom)
    if state.resltype then begin  ;Q resolution
       state.enLab = widget_label(state.ccol,value='E transfer (meV)')
       state.enTxt = widget_text(state.ccol,value=dm_to_string(state.en),/editable,/all_events)
    endif
    void = widget_label(center,value=' ')

    state.loadBut  = widget_button(center,value='Load Data')
    state.calcBut  = widget_button(center,value='Calculate')
    state.saveBut  = widget_button(center,value='Save Result')
    state.clearBut = widget_button(center,value='Clear Data')
    state.printBut = widget_button(center,value='Print')
    state.exitBut  = widget_button(center,value='Exit')
    
    if float(!version.release) ge 5.6 then begin  ;tips for some widget buttons
       widget_control,state.loadBut,tooltip='Select the file from the left, then load the '+(['(E, I) 2-column ascii','(Q, I) 2-column ascii','(Q, E, Intensity) SPE'])[state.resltype]+' data file.'
       widget_control,state.calcBut,tooltip='Calculate the '+(['convolution of the ','',''])[state.resulttype]+'DCS '+$
              (['E','Q',(['Q and E','E','Q'])[state.resulttype]])[state.resltype]+' resolution '+(['with','of','of'])[state.resulttype]+' the loaded data.'
       widget_control,state.saveBut,tooltip='Save the last '+(['convolution','E resolution','Q resolution'])[state.resulttype]+' result in an ascii '+(['','','SPE '])[state.resltype]+'file.'      
       widget_control,state.clearBut,tooltip='Clear all the plots and data.'
       widget_control,state.printBut,tooltip='Print the plot.' 
    endif

    geom  = widget_info(state.exitBut,/geometry)
    if state.resltype then widget_control,state.enTxt,scr_xsize=geom.scr_xsize
    widget_control,state.wlTxt,scr_xsize=geom.scr_xsize
    widget_control,state.reslList,xsize=geom.scr_xsize
    widget_control,state.speedTxt,scr_xsize=geom.scr_xsize

    geom  = widget_info(left,/geometry)
    ysize = geom.scr_ysize
    xsize = fix(1.35*ysize,type=3)
    state.xtit = ['E (meV)','|Q| (\AA!U-1!N)','|Q| (\AA!U-1!N)']
    state.plotWin  = obj_new('dm_plot',xsize=xsize,ysize=ysize,widgetbase=state.tlb,group_leader=state.tlb,/isolatin1,/compound,background=bgcolor,notooltip=notooltip,$
        xtit=(state.xtit)[state.resltype],ytit=(['Intensity (arb. unit)','Intensity (arb. unit)','E (meV)'])[state.resltype],$
        itit='Intensity (arb. unit)',legdpos=[0.05,0.85])

    dm_center_kid, state.tlb,state.group_leader,/side   ;centering the top level base
    widget_control,state.tlb,/realize,/map

    geom  = widget_info(state.tlb,/geometry)
    state.geom = [geom.scr_xsize,geom.scr_ysize]
    
    state.iran[*] = !values.f_nan

    pState = ptr_new(state,/no_copy)
    widget_control,(*pState).tlb,set_uvalue=pState

    dm_ereslconv_resetbutton,pState

    if xregistered('dave',/noshow) and (n_elements(event) ne 0) then begin
       (*pState).enotebook = 1b
       RET = DAVE_SET_FOCUS((*pState).tlb)
    endif

    xmanager,registerName,(*pState).tlb,cleanup='dm_ereslconv_Exit',/no_block
    if ~keyword_set(noinfo) then begin
       ok = dialog_message(['Current convolution mode is '+(['1D E','1D Q','2D Q&E'])[(*pState).resltype]+' resolution convolution.',$
             'Use the Option menu to switch to other mode.','For more information, please check the Help menu.'],/center,/info,dialog_parent=(*pState).tlb)
    endif
end