; $Id: dm_dialog_input.pro,v 1.31 2017/03/16 19:34:55 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_dialog_input
;
; PURPOSE:
;  a multipurpose dialog input utility
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################
 @dialog_colorpicker  ;for dialog_colorpickerGetAccumulatedWidgetOffset
pro dm_dialog_input_Exit,tlb,tmp=tmp
    ;do nothing, leave the job to obj_destroy
end

;widget event handler
pro dm_dialog_input_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;object event handler
pro dm_dialog_input::event, event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    ;error handling
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,/center)
       return
    end
    
    eventname = widget_info(event.id,/uname)
    eventname = strsplit(eventname,'_',/extract)
    if n_elements(eventname) eq 2 then id=dm_to_number(eventname[1],/int)
    eventname = strlowcase(eventname[0])
    ;make the dialog window always on top of the screen
    case strupcase(tag_names(event,/STRUCTURE_NAME)) of
         'WIDGET_KBRD_FOCUS': begin
                              if ~self.nottotop and (event.id eq self.tlb) then self->totop
                              if strupcase(widget_info(event.id,/name)) eq 'TEXT' then eventname = 'text_focus'
                              end
         'WIDGET_TEXT_SEL':   if self.txtfocusevent then begin
                              widget_control,event.id,get_value=tmp
                              if n_elements(tmp) eq 1 then widget_control,event.id,set_text_select=[0,strlen(tmp[0])] 
                              endif                     
         else:
    endcase
    if eventname ne 'text_focus' then WIDGET_CONTROL,/HOURGLASS
    self.txtfocusevent = 0b
    case eventname of
       'text_focus': begin
            if event.enter then $
               self.txtfocusevent = 1b $
            else $
               widget_control,event.id,set_text_select=[0,0]
            end
       'cancel': begin
            self.cancel = (1b)
            widget_control,self.tlb,/destroy
            return
            end     
       'ok': begin
            widget_control,self.tlb,/destroy
            return
            end
       'clear': begin
            for i=0,self.n_item-1 do begin
                wid = widget_info(self.tlb,find_by_uname='input_'+dm_to_string(i,/int))
                if ~widget_info(wid,/sensitive) then continue
                widget_type = widget_info(wid,/type)
                if (widget_type eq 3) then begin
                   widget_control,wid,set_value=''
                   (*self.value)[i] = ''
                endif
            endfor
            end
       'apply': begin
            ;set up the property keywords
            self->getproperty,value=value
            if n_elements(*self.property) eq 1 then begin
               if keyword_set((*self.isdefine)[0]) and stregex(value[0],'user define',/boolean,/fold_case) then value = (*self.coldefine)[*,0]
               extra = create_struct((*self.property),value)
            endif else begin
               extra = create_struct((*self.property)[0],value[0])
               for i=1,n_elements(value)-1 do extra = create_struct(extra,(*self.property)[i],value[i])
            endelse
            if ptr_valid(self.extra) then extra = create_struct(extra,*self.extra)     ;add extra keywords
            call_method,self.parenthandler,self.parentobj,_extra=extra
            self.apply = (1b)
            end
       'input':begin
            widget_control,event.id,get_uvalue=value
            widget_type = widget_info(event.id,/type)
            case widget_type of
                 2: begin     ;slider
                    widget_control,event.id,get_value=value
                    (*self.value)[id] = value
                    end
                 3: begin     ;text box
                    widget_control,event.id,get_value=value
                    ind = where(strlen(strtrim(value,2)) gt 0,count)
                    if count gt 0 then begin
                       (*self.value)[id] = value[ind[0]]
                       for i=1,count-1 do begin
                           (*self.value)[id] = (*self.value)[id]+string('A4'xb)+value[ind[i]]
                       endfor 
                    endif else begin
                       (*self.value)[id] = '' 
                    endelse
                    if event.type eq 0 then begin
                       if event.ch eq 10B then begin ;return
                          if id eq self.n_item-1 then begin
                             widget_control,self.tlb,/destroy
                             return
                          endif else begin                  
                             next_id  = (id+1) mod self.n_item
                             wid = widget_info(self.tlb,find_by_uname='input_'+dm_to_string(next_id,/int))
                             if widget_info(wid,/type) eq 3 then begin  ;text
                                widget_control,wid,/input_focus,set_text_select=[0,strlen((*self.value)[next_id])]
                             endif
                          endelse
                       endif
                    endif
                    info_id = widget_info(self.tlb,find_by_uname='info')
                    if (*self.isenergy)[id] then begin
                       if self.elambda eq 1 then begin             ;e2lambda keyword set
                          if stregex((*self.property)[id],'^Ei',/boolean,/fold_case) then name = 'Ei' $
                          else if stregex((*self.property)[id],'^Ef',/boolean,/fold_case) then name = 'Ef' $
                          else name = 'E'
                          tmp0 = '9.04/sqrt('+name+') '
                          tmp1 = dm_to_number((*self.value)[id])
                          if finite(tmp1) then begin
                             if tmp1 ne 0 then tmp0 = dm_to_string(sqrt(81.8042/tmp1),res=4)+' '
                          endif
                          widget_control,info_id,set_value='wavelength='+tmp0+string('c5'XB)
                       endif else if self.elambda eq 2 then begin  ;lambda2e keyword set
                          tmp0 = '81.8/lambda^2'
                          tmp1 = dm_to_number((*self.value)[id])
                          if finite(tmp1) then begin
                             if tmp1 ne 0 then tmp0 = dm_to_string(81.8042/(tmp1^2),res=4)
                          endif
                          widget_control,info_id,set_value='energy='+tmp0+' meV'
                       endif
                    endif else if info_id gt 0 then widget_control,info_id,set_value=''
                    if ptr_valid(self.duplicate) and (event.type ne 3) then begin
                       if (*self.duplicate)[0] eq id then begin
                          for i=1,n_elements((*self.duplicate))-1 do begin
                              wid = widget_info(self.tlb,find_by_uname='input_'+dm_to_string((*self.duplicate)[i],/int))
                              if wid ne 0 then begin
                                 if widget_info(wid,/type) eq 3 then begin
                                    widget_control,wid,set_value=value
                                    (*self.value)[(*self.duplicate)[i]] = (*self.value)[id]
                                 endif
                              endif
                          endfor
                       endif
                    endif
                    end
                 8: begin     ;droplist
                    tmp = widget_info(event.id,/droplist_select)
                    if self.dl_ret_number then begin
                       (*self.value)[id] = tmp 
                       if self.type eq 0 then (*self.value)[id] = strtrim((*self.value)[id],2)
                    endif else (*self.value)[id]=value[tmp]
                    if (*self.isdefine)[id] gt 0 then is_define = stregex(value[tmp],'user define',/boolean,/fold_case)*(*self.isdefine)[id]
                    end
                12: begin    ;combobox
                    tmp = event.index
                    if self.dl_ret_number then begin
                       (*self.value)[id] = tmp 
                       if self.type eq 0 then (*self.value)[id] = strtrim((*self.value)[id],2)
                    endif else (*self.value)[id]=value[tmp]
                    if (*self.isdefine)[id] gt 0 then is_define = stregex(value[tmp],'user define',/boolean,/fold_case)*(*self.isdefine)[id]
                    end
              else:
            endcase
            if obj_valid(self.oWin) then self->updatecbar,event
            if keyword_set(is_define) then begin
               self.nottotop = 1
               if (is_define eq 1) or (self.idl_version lt 8.5) then begin
                  if ptr_valid(self.bgcolor) and (is_define eq 1) then $
                     define = dm_plot.getdefinecolor(wCaller=event.id,default=(*self.coldefine)[*,id],bgcolor=*self.bgcolor,parent=self.tlb) $
                  else $
                     define = dm_plot.getdefinecolor(wCaller=event.id,default=(*self.coldefine)[*,id],parent=self.tlb)
               endif else begin
                  if self.idl_version ge 8.51 then $
                     define = dialog_colorpicker(wcaller=event.id,/modal,initial_color=(*self.coldefine)[*,id]) $
                  else $
                     define = dialog_colorpicker(wcaller=event.id,/modal)
               endelse
               if n_elements(define) eq 3 then (*self.coldefine)[*,id] = define
               self.nottotop = 0
            endif
            end
       'browse': begin ;browse button
            widget_control,event.id,get_uvalue=uval
            id = uval[0] & directory=((uval[1] eq 2) or (uval[1] eq 3)) & read = (uval[1] ne 3) & write = (uval[1] eq 3) 
            if ptr_valid(self.f_ptr) then begin
               type = strsplit((*self.f_ptr)[id],'.',/extract)
               type = type[n_elements(type)-1]
            endif else $
               type = ''
            if (strlen((*self.value)[id]) ne 0) and directory then path=file_dirname((*self.value)[id]) else path=self.path
            file = dm_choose_file(type,dialog_parent=event.handler,path=path,read=read,write=write,multiple_files=self.multiple_files,directory=directory)
            for i=1,n_elements(file)-1 do file[0] = file[0]+'&&'+file[i]
            file = file[0]
            if strlen(file) ne 0 then begin
               if uval[1] eq 4 then begin ;need to read its content
                  openr,unit,file,/get_lun,error=error
                  if error ne 0 then begin
                     ok = dialog_message("Can't read file "+file+'.',dialog_parent=self.tlb,/center)
                     return
                  endif
                  tmp = '' & file = ''
                  while (~eof(unit)) do begin
                     readf,unit,tmp 
                     if uval[2] gt 1 then begin
                        file = [file,tmp]
                     endif else begin
                        tmp = strtrim(tmp,2)
                        if strlen(tmp) eq 0 then continue
                        file = file+tmp
                     endelse
                  endwhile
                  free_lun,unit
               endif
               if total(strlen(file)) eq 0 then return
               if n_elements(file) gt 2 then file = file[1:*]
               self.path = path
               wid = widget_info(self.tlb,find_by_uname='input_'+dm_to_string(id,/int))
               widget_control,wid,set_value=file 
               ind = where(strlen(strtrim(file,2)) gt 0,count)
               if count gt 0 then begin
                  (*self.value)[id] = file[ind[0]]
                  for i=1,count-1 do begin
                      (*self.value)[id] = (*self.value)[id]+string('A4'xb)+file[ind[i]]
                  endfor
               endif else begin
                  (*self.value)[id] = ''
               endelse
            endif
            end
       'colorbar': begin ;expose event in some unix system for the color bar
            if obj_valid(self.oWin) then self->updatecbar
            end
       else:
    endcase
end

;get properties
pro dm_dialog_input::getproperty,value=value,apply=apply,cancel=cancel,path=path,coldefine=coldefine
    if arg_present(cancel)    then cancel=self.cancel
    if arg_present(apply)     then apply=self.apply
    if arg_present(path)      then path=self.path
    if arg_present(coldefine) then coldefine=*self.coldefine 
    if arg_present(value)  then begin
       case self.type of
            1:    value = dm_to_number(*self.value,/int)    ;int
            2:    value = dm_to_number(*self.value,/long)   ;long
            3:    value = dm_to_number(*self.value,/float)  ;float
            4:    value = dm_to_number(*self.value,/double) ;double
            else: value = *self.value
       endcase
       if self.n_item eq 1 then begin
          value = value[0]
          if self.dl_ret_number and (size(value,/type) eq 7) then value = dm_to_number(value)
       endif
    endif
end

pro dm_dialog_input::totop
    widget_control,self.tlb,kbrd_focus_event=0
    widget_control,self.tlb,/show,iconify=0
    wid = widget_info(self.tlb,find_by_uname='ok')
    if wid gt 0 then widget_control,wid,/input_focus
    widget_control,self.tlb,/kbrd_focus_event
end

;update colorbar
pro dm_dialog_input::updatecbar,event
    if ~obj_valid(self.oWin) then return
    ;figure out whether it is color table or just an rgb color
    wid = widget_info(self.tlb,find_by_uname='input_0')
    if wid eq 0 then return
    type = widget_info(wid,/type)
    if (self.n_item eq 3) and ((type eq 3) or (type eq 2)) then begin
       r_index = 0 & g_index = 1 & b_index = 2
       name = strlowcase(*self.property)
       ind = where(strmid(name,0,1) eq 'r',count) & if count eq 1 then r_index = ind
       ind = where(strmid(name,0,1) eq 'g',count) & if count eq 1 then g_index = ind
       ind = where(strmid(name,0,1) eq 'b',count) & if count eq 1 then b_index = ind
       self.oPal->setProperty,red_values=replicate((0>dm_to_number((*self.value)[r_index],/int)<255),256), $
            green_values=replicate((0>dm_to_number((*self.value)[g_index],/int)<255),256), $
            blue_values=replicate((0>dm_to_number((*self.value)[b_index],/int)<255),256)   
    endif else begin
       id = 0
       if n_elements(event) ne 0 then begin
          eventname = widget_info(event.id,/uname)
          eventname = strsplit(eventname,'_',/extract)
          if n_elements(eventname) eq 2 then id=dm_to_number(eventname[1],/int)
       endif
       dm_plot.getcolormap,dm_to_number((*self.value)[id],/int),red_values=r,green_values=g,blue_values=b
       self.oPal->setproperty,red_values=r,green_values=g,blue_values=b
    endelse
    self.oWin->draw,self.oView
end

;destroy the object,life cycle method
pro dm_dialog_input::Cleanup
    if widget_info(self.extbase,/valid_id) then widget_control,self.extbase,/destroy
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    ptr_free,self.extra,self.value,self.f_ptr,self.property,self.duplicate,self.isenergy,self.bgcolor,self.isdefine,self.coldefine
    obj_destroy,[self.oWin,self.oView,self.oPal]
end

;constructor,lifecycle method
;syntax:
;   obj=obj_new('dm_dialog_input','color',default=1,droplist_content=['red','blue','green'],is_droplist=[1],/return_number)
;parameter:
;   name:               a string array of names
;keyword:
;   align_center:       if set, the info messages will be aligned on center
;   bgcolor:            background color for user define color, [r,g,b], if is_define=1, and user define color=bgcolor, 255-bgcolor will be used, use is_define=2 to bypass this
;   bitmap:             a string of the name of bitmap file for the tlb icon
;   cancel:             if arg_presnt, a cancel button will be available
;   cbstring:           if present, use this string instead of "Cancel" for the cancel button
;   coltable:           if set, a color bar will show on the top of the widget
;   default:            default value, or rather starting value
;   dialog_parent:      dialog parent, tlb will be centered on dialog parent
;   double:             if set, all values will be double
;   droplist_content:   a pointer or pointer arrary points to string array(s) containing the droplist values
;                       pointers will be destroyed by the program
;   duplicate:          an index array, the entry of the first index will be copied to the entries of the other indexes, only for textboxes                    
;   e2lambda:           if set, calculate wavelength from energy and show it in the first line of info
;   lambda2e:           if set, calculate energy from wavelength and show it in the first line of info
;   filter:             filters for file open dialog
;   float:              if set, all values will be float
;   gamma:              color palette gamma value, applicable when coltable keyword is set
;   info:               informations to be shown at the bottom of tlb
;   int:                if set, all values will be int
;   is_droplist:        an array of 0 and 1 to show which quest is a droplist
;   is_file:            an array of 0, 1, 2, 3, 4 to show which quest is a file (1, 4 read the content) or directory (2 read, 3 write), and will have an open button next to it
;   is_sensitive:       an array of 0 and 1 to show which item is sensitive, default is all sensitive
;   is_slider:          an array of 0 and 1 to show which item is a slider
;   long:               if set,all values will be long
;   multiline:          number of lines for the input, used with caution, the returning string use string('A4'xb) to seperate lines
;   multiple_files:     flag for browsing multiple files, multiple file names will be spearated by string '&&'
;   scroll:             show scroll, for multiline only
;   parenthandler:      parent event handler name, default is 'setproperty'
;   parentobject:       parent object
;   path:               initial file or directory path
;   property:           properties to be set in calling parentobject->parenthandler, default is name
;   return_number:      return index number of droplist instead of string values, if set and only single item, the returned value will be a float if no other types set
;   slidermaximum:      maximum value for slider, can be an array
;   sliderminimum:      minimum value for slider, can be an array
;   title:              title of tlb
;   wCaller:            caller widget id, if present, the dialog window will appear right below the widget, similar to dialog_colorpicker 
;   xsize:              xsize of text widget
;   _extra:             extra keywords used in calling parentobject->parenthandler,name=value
function dm_dialog_input::Init,name,align_center=align_center,bgcolor=bgcolor,bitmap=bitmap,cancel=cancel,cbstring=cbstring,coldefine=coldefine,coltable=coltable,default=default,$
    dialog_parent=dialog_parent,droplist_content=droplist_content,duplicate=duplicate,e2lambda=e2lambda,lambda2e=lambda2e,filter=filter,float=float,gamma=gamma,info=info,int=int,$
    is_define=is_define,is_droplist=is_droplist,is_energy=is_energy,is_file=is_file,is_sensitive=is_sensitive,is_slider=is_slider,multiline=multiline,multiple_files=multiple_files,$
    parenthandler=parenthandler,parentobject=parentobject,path=path,property=property,return_number=return_number,scroll=scroll,slidermaximum=slidermaximum,sliderminimum=sliderminimum,$
    title=title,wCaller=wCaller,xsize=xsize,_extra=extra
    self.idl_version = dm_to_number(!version.release)
    self.type = 0     ;default type is string
    self.multiple_files = keyword_set(multiple_files)
    if n_elements(multiline) eq 0 then multiline = 1 else multiline = 1>(multiline)
    if n_elements(name)  eq 0 then name = ''
    self.n_item = n_elements(name)
    if n_elements(title) eq 0 then title = 'Input:'
    if n_elements(default) ne 0 then begin
        type = size(default,/type)
        switch type of
          1:
          2:
          3:begin
            mydefault = dm_to_string(default,/int)
            self.type = (2>type)-1                  ;integer,long
            break
            end
          4:
          5:begin
            mydefault = dm_to_string(default)
            self.type = type-1                      ;float,double
            break
            end
          else:begin
            if type eq 7 then mydefault = default else mydefault = dm_to_string(default)
            end
        endswitch
    endif else mydefault = ''
    self.elambda = 0
    if keyword_set(e2lambda) then self.elambda = 1
    if keyword_set(lambda2e) then self.elambda = 2
    if keyword_set(int)      then self.type = 1
    if keyword_set(long)     then self.type = 2
    if keyword_set(float)    then self.type = 3
    if keyword_set(double)   then self.type = 4
    ;exception for drolist
    if n_elements(is_droplist) ne 0 then begin
       if ~keyword_set(return_number) then self.type = 0
    endif
    if n_elements(duplicate) ne 0     then self.duplicate = ptr_new(duplicate)
    if n_elements(extra) ne 0         then self.extra = ptr_new(extra)
    if n_elements(parenthandler) ne 0 then self.parenthandler = parenthandler else self.parenthandler = 'setproperty'
    if n_elements(parentobject) ne 0  then self.parentobj = parentobject
    if n_elements(path) ne 0          then begin
       if file_test(path[0],/directory) then self.path = path[0]
    endif
    n_prop = n_elements(property)
    if n_prop ne 0 then begin
       if n_prop ne 1 then begin
          tmp = strarr(self.n_item)
          for i=0,self.n_item-1 do tmp[i] = property[i mod n_prop]
       endif else tmp = property
       self.property = ptr_new(tmp,/no_copy)
    endif else self.property=ptr_new(name)
    if keyword_set(return_number)     then self.dl_ret_number=1
    if n_elements(xsize) eq 0         then xsize=60
    n_default  = n_elements(mydefault)
    self.value = ptr_new(strarr(self.n_item))
    isdroplist = bytarr(self.n_item)
    isfile     = bytarr(self.n_item)
    isensitive = bytarr(self.n_item)+(1b)
    isenergy   = bytarr(self.n_item)
    isdefine   = bytarr(self.n_item)
    isslider   = bytarr(self.n_item)
    col_define = bytarr(3,self.n_item)
    if n_elements(bgcolor) eq 3 then self.bgcolor = ptr_new(bgcolor) else bgcolor = [255b,255b,255b]
    if n_elements(is_define) ne 0 then begin
       ind = where(is_define gt 0,cnt)
       if cnt gt 0 then begin
          if n_elements(coldefine) eq 0 then begin
             if is_define[ind[0]] eq 1 then coldefine = 255b-bgcolor else coldefine = bgcolor
             for i=1,cnt-1 do coldefine = [[coldefine],[(is_define[ind[i]] eq 1)?(255b-bgcolor):bgcolor]]
          endif
          coldefine = reform(coldefine,3,n_elements(coldefine)/3)
          for i=0,cnt-1 do col_define[*,ind[i]] = coldefine[*,i<(n_elements(coldefine)/3-1)]
       endif
       isdefine[0:(self.n_item<n_elements(is_define))-1] = is_define[0:(self.n_item<n_elements(is_define))-1]
    endif
    if (n_elements(is_droplist) ne 0) and (n_elements(droplist_content) ne 0) then begin
       isdroplist[0:((n_elements(is_droplist)<(self.n_item))-1)] = is_droplist[0:((n_elements(is_droplist)<(self.n_item))-1)]
    endif
    if n_elements(is_file) ne 0 then begin
       isfile[0:((n_elements(is_file)<(self.n_item))-1)] = 0>(is_file[0:((n_elements(is_file)<(self.n_item))-1)])<4
    endif
    if n_elements(is_sensitive) ne 0 then begin
       isensitive[0:((n_elements(is_sensitive)<(self.n_item))-1)] = is_sensitive[0:((n_elements(is_sensitive)<(self.n_item))-1)]
    endif
    if n_elements(is_slider) ne 0 then begin
       isslider[0:((n_elements(is_slider)<(self.n_item))-1)] = is_slider[0:((n_elements(is_slider)<(self.n_item))-1)]
    endif
    if n_elements(is_energy) ne 0 then begin
       isenergy[0:((n_elements(is_energy)<(self.n_item))-1)] = is_energy[0:((n_elements(is_energy)<(self.n_item))-1)]
    endif else if self.elambda gt 0 then isenergy[0] = 1b
    self.isenergy  = ptr_new(isenergy)
    self.isdefine  = ptr_new(isdefine)
    self.coldefine = ptr_new(col_define)
    if n_elements(bgcolor) eq 3 then self.bgcolor = ptr_new(bgcolor)
    for i=0,self.n_item-1 do (*self.value)[i]=mydefault[i mod n_default]
    n_filt1 = n_elements(filter)
    ind_fil = where(isfile,n_filt2)
    if (n_filt1 ne 0) and (n_filt2 ne 0) then begin
       f_val = strarr(self.n_item)
       f_val[ind_fil[0:(n_filt1<n_filt2)-1]] = filter[0:(n_filt1<n_filt2)-1]
       self.f_ptr = ptr_new(f_val,/no_copy)
    endif
    if (self.idl_version ge 6.4) and (n_elements(bitmap) ne 0) then icon_extra={bitmap:bitmap[0],mask:1}
    self.extbase = widget_base(group_leader=dialog_parent,map=0,floating=(n_elements(dialog_parent) ne 0))
    self.tlb     = widget_base(title=title,/col,group_leader=self.extbase,/modal,/align_center,_extra=icon_extra)
    ntext        = 0
    if keyword_set(coltable) then begin
       dm_plot.getcolormap,dm_to_number(mydefault[0],/int),names=names,red_values=r,green_values=g,blue_values=b
       draw = widget_draw(self.tlb,graphics_level=2,scr_ysize=20,renderer=1,retain=0,/expose_events,uname='colorbar')
       self.oView = obj_new('IDLgrView',location=[0,0],viewplane_rect=[0,0,1,1])
       self.oPal  = obj_new('IDLgrPalette',gamma=gamma,red_values=r,green_values=g,blue_values=b)
       model      = obj_new('IDLgrModel')
       cbar       = obj_new('IDLgrColorbar',palette=self.oPal)
       self.oView->add,model
       model->add,cbar
    endif
    col   = widget_base(self.tlb,/col,/align_center)
    label = lonarr(1>(self.n_item))
    lsize = 0
    for i=0,self.n_item-1 do begin
        line = widget_base(col,/row,/align_left,sensitive=isensitive[i])
        label[i] = widget_label(line,value=name[i],/align_center)
        geom = widget_info(label[i],/geometry)
        lsize = lsize>(geom.scr_xsize)
        if isslider[i] then begin
           tmp = where(isslider[0:i],nslider)
           if n_elements(slidermaximum) ne 0 then maximum = slidermaximum[(nslider<n_elements(slidermaximum))-1]
           if n_elements(sliderminimum) ne 0 then minimum = sliderminimum[(nslider<n_elements(slidermaximum))-1]
           void = widget_slider(line,value=(*self.value)[i],maximum=maximum,minimum=minimum,/drag,scr_xsize=xsize,uname='input_'+dm_to_string(i,/int),uvalue=i)       
        endif else if isdroplist[i] then begin
           tmp   = where(isdroplist[0:i],ndroplist)
           value = *droplist_content[(ndroplist<n_elements(droplist_content))-1]
           void  = dm_widget_droplist(line,value=value,scr_xsize=xsize,uname='input_'+dm_to_string(i,/int),select=(*self.value)[i],uvalue=value)
           if (~ keyword_set(return_number)) then (*self.value)[i]=value[dm_to_number((*self.value)[i],/int)] $
           else (*self.value)[i]=dm_to_string(dm_to_number((*self.value)[i],/int))
        endif else begin
           if multiline gt 1 then begin
              value = (*self.value)[i]
              value = strsplit(value,string('A4'xb),/extract)
              void  = widget_text(line,value=value,/all_events,scr_xsize=xsize,uname='input_'+dm_to_string(i,/int),ysize=multiline,uvalue=i,scroll=scroll,/editable,/kbrd_focus_events)       
           endif else begin
              void  = widget_text(line,value=(*self.value)[i],/editable,/all_events,/kbrd_focus_events,scr_xsize=xsize,uname='input_'+dm_to_string(i,/int),/align_center,uvalue=i)
           endelse
           if n_elements(text0id) eq 0 then begin
              text0id = void
              text0sel = [0,strlen((*self.value)[i])]
           endif
           ntext = ntext+1
           if isfile[i] ne 0 then begin
              if n_elements(bitmap_filename) eq 0 then begin
                 defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
                 if exists then bitmap_filename = !DAVE_AUXILIARY_DIR+'open.bmp' $
                 else bitmap_filename = file_which('open.bmp',/include_current_dir)
              endif
              tmpline = widget_base(line,/row,/align_left,/base_align_center)
              browse_but = widget_button(tmpline,value=bitmap_filename,/bitmap,/dynamic_resize,uname='browse',uvalue=[i,isfile[i],multiline])
              if float(!version.release) ge 5.6 then widget_control,browse_but,tooltip='Click to select a '+(['','file','directory to read from','directory to write to',$
                 'file to read'])[isfile[i]]+'.'
           endif
        endelse
    endfor
    if n_elements(droplist_content) ne 0 then ptr_free,droplist_content
    for i=0,self.n_item-1 do begin
        widget_control,label[i],scr_xsize=lsize+5 
        if keyword_set(isdefine[i]) and stregex((*self.value)[i],'user define',/boolean,/fold_case) then $
           tmpdefstr = 'Reselect '+(*self.value)[i]+' to change the user define value.'
    endfor
    if n_elements(tmpdefstr) ne 0 then begin
       if n_elements(info) eq 0 then info = tmpdefstr else info = [tmpdefstr,info]
    endif
    n_info = n_elements(info)
    if (n_info ne 0) or (self.elambda gt 0) then begin
       col = widget_base(self.tlb,/col,/align_center)
       if self.elambda eq 1 then begin
          id = 0>((where(isenergy))[0])
          if stregex((*self.property)[id],'^Ei',/boolean,/fold_case) then tmpname = 'Ei' $
          else if stregex((*self.property)[id],'^Ef',/boolean,/fold_case) then tmpname = 'Ef' $
          else tmpname = 'E'
          tmp0  = '9.04/sqrt('+tmpname+') '
          label = widget_label(col,value='wavelength='+tmp0+string('c5'XB),align_left=~keyword_set(align_center),/dynamic_resize,uname='info')
          tmp1  = dm_to_number(mydefault[0])
          if finite(tmp1) then begin
             if tmp1 ne 0 then begin
                tmp0 = dm_to_string(sqrt(81.8042/tmp1),res=4)+' '
                widget_control,label,set_value='wavelength='+tmp0+string('c5'XB)
             endif
          endif
       endif else if self.elambda eq 2 then begin
          tmp0  = '81.8/lambda^2 '
          label = widget_label(col,value='energy='+tmp0+'meV',align_left=~keyword_set(align_center),/dynamic_resize,uname='info')
          tmp1  = dm_to_number(mydefault[0])
          if finite(tmp1) then begin
             if tmp1 ne 0 then begin
                tmp0 = dm_to_string(81.8042/(tmp1^2),res=4)+' meV'
                widget_control,label,set_value='energy='+tmp0
             endif
          endif
       endif
       for i=0,n_info-1 do label = widget_label(col,value=info[i],align_left=~keyword_set(align_center))
    endif
    row  = widget_base(self.tlb,/row,/align_center)
    if ntext gt 0 then begin
       void = widget_button(row,value='Clear',uname='clear')
       void = widget_label(row,value='',xsize=15)
    endif
    if n_elements(parentobject) ne 0 then begin
       void = widget_button(row,value='Apply',uname='apply')
       void = widget_label(row,value='',xsize=15)
    endif
    if arg_present(cancel) then begin
       if n_elements(cbstring) eq 0 then $
          cbstring = 'Cancel' $
       else if strlen(cbstring) eq 0 then $
          cbstring = 'Cancel'
       void = widget_button(row,value=cbstring,uname='cancel')
       void = widget_label(row,value='',xsize=15)
    endif
    void = widget_button(row,value='OK',uname='ok')
    if (n_elements(wCaller) eq 1) and (n_elements(dialog_parent) eq 1) then begin
       if widget_info(wCaller,/valid_id) and widget_info(dialog_parent,/valid_id) then begin
          if widget_info(wCaller,/visible) then begin
             tmp = dialog_colorpickerGetAccumulatedWidgetOffset(wCaller,offset=offset)
             geom = Widget_Info(wCaller,/GEOMETRY)
             offsetAdjust = ((!version.os_family eq 'Windows') ? 5 : 0)
             widget_control,self.tlb,xoffset=offset[0]-offsetAdjust,yoffset=offset[1]+geom.scr_ysize-offsetAdjust
             centered = 1b
          endif
       endif
    endif 
    if ~keyword_set(centered) then dm_center_kid,self.tlb,dialog_parent   ;centering over the dialog_parent widget
    if keyword_set(coltable) then begin
       geom  = widget_info(self.tlb,/geometry)
       xsize = geom.scr_xsize-(geom.space+geom.xpad+geom.margin)*2
       widget_control,draw,scr_xsize=xsize
    endif
    widget_control,self.tlb,/realize
    if keyword_set(coltable) then begin
       widget_control,draw,get_value=oWin
       oWin->getproperty,dimension=dim
       self.oWin = oWin
       cbar->setproperty,dimension=dim,xcoord_conv=norm_coord([0,dim[0]]),ycoord_conv=norm_coord([0,dim[1]])
       self->updatecbar
    endif
    widget_control,self.tlb,set_uvalue=self,/kbrd_focus_event
    if n_elements(text0id) ne 0 then widget_control,text0id,/input_focus,set_text_select=text0sel
    xmanager,'dm_dialog_input',self.tlb,cleanup='dm_dialog_input_Exit'
    return,1
end

;class definition
pro dm_dialog_input__define
    void={dm_dialog_input        , $    ;thie object class name
        extbase:               0L, $    ;group leader of the modal tlb
        tlb:                   0L, $    ;top level base
        idl_version:           0e, $    ;save the IDL version number
        parentobj:      obj_new(), $    ;parent object
        parenthandler:         '', $    ;event handler of parent object,default is 'event'
        n_item:                0s, $    ;number of items
        type:                  0s, $    ;data type 0:int 1:float 2:string
        path:                  '', $    ;save browse path
        nottotop:              0b, $    ;to avoid blinking if another dialog_input is called from within
        multiple_files:        0b, $    ;flag for browsing for multiple files
        bgcolor:        ptr_new(), $    ;store bgcolor for the user define color
        isdefine:       ptr_new(), $
        coldefine:      ptr_new(), $
        duplicate:      ptr_new(), $
        value:          ptr_new(), $    ;save the reults
        f_ptr:          ptr_new(), $    ;save the filter values
        isenergy:       ptr_new(), $    ;
        extra:          ptr_new(), $    ;extra keyword used in calling parentobj->parenthandler,name=value
        property:       ptr_new(), $    ;property names,default is name
        oWin:           obj_new(), $    ;window object
        oView:          obj_new(), $    ;view object
        oPal:           obj_new(), $    ;palette object
        dl_ret_number:         0b, $    ;if 1 then return droplist choice by number
        elambda:               0s, $    ;if 1 then calculate lambda and show it in the info label
                                   $    ;if 2 then calculate energy from lambda and show it in the info label
        cancel:                0b, $    ;flag to show if cancel button has been pressed
        apply:                 0b, $    ;flag to show if apply button has been pressed
        txtfocusevent:         0b  $    ;flag for text focus event
    }
end

;example
;   result=dm_dialog_input('color',default=1,droplist_content=ptr_new(['red','blue','green']),is_droplist=[1],/return_number)
function dm_dialog_input,name,_ref_extra=extra
    tmp=obj_new('dm_dialog_input',name,_extra=extra)
    tmp->getproperty,value=value,_extra=extra
    obj_destroy,tmp
    if n_elements(value) ne 0 then return,value
end