; $Id: $
;#######################################################################
;
; NAME:
;  dm_choose_trajdata
;
; PURPOSE:
;  choose the data according to plot trajectory type
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  May, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;check thickness range 
;   ndat:       total number of data, for comparison
;   checkfold:  flag for folding to both sides, if set foldcenter needs to be present 
;   qoffset:    for adjusting axis title according to thickness range
pro dcs_mslice::dm_check_thickness,id,range=range,ndat=ndat,index=index,daindex=daindex,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
    extra_tit=extra_tit,checkfold=checkfold,foldcenter=foldcenter,qoffset=qoffset,debug=debug
    if arg_present(daindex) then dark_angle = 1b
    traninvalid = 0b
    tmp_range = range
    if keyword_set(dark_angle) then $
       self->dm_choose_trajdata,id,index=index,daindex=daindex,range=range,empty=empty,debug=debug,unit=unit,label=label,is_E=is_E,is_hkl=is_hkl $
    else $
       self->dm_choose_trajdata,id,index=index,range=range,empty=empty,debug=debug,unit=unit,label=label,is_E=is_E,is_hkl=is_hkl
    if empty then begin
       index = -1 & tmp = temporary(index)
       if keyword_set(dark_angle) then begin
          daindex = -1 & tmp = temporary(daindex)
       endif
       traninvalid = 1b
       is_E = 0b
    endif else if n_elements(index) eq ndat then begin
       index = -1 & tmp = temporary(index)
    endif
    if keyword_set(checkfold) then begin  ;fold to both sides
       if keyword_set(dark_angle) then $
          self->dm_choose_trajdata,id,index=index_f,daindex=daindex_f,range=2.0*foldcenter-reverse(range),empty=empty,debug=debug,is_E=is_E_f $
       else $
          self->dm_choose_trajdata,id,index=index_f,range=2.0*foldcenter-reverse(range),empty=empty,debug=debug,is_E=is_E_f
       if ~empty then begin
          if traninvalid then begin
             is_E = is_E_f
             traninvalid = 0b
             tmp_range = 2.0*foldcenter-reverse(range)
             index = index_f
             if keyword_set(dark_angle) then daindex = daindex_f
          endif else begin
             if n_elements(index) eq 0 then index= index_f else if index[0] ne -1 then index = [index,index_f] else index = index_f
             if keyword_set(dark_angle) then begin
                if n_elements(daindex) eq 0 then daindex= daindex_f else if daindex[0] ne -1 then daindex = [daindex,daindex_f] else daindex = daindex_f
             endif
          endelse
          if n_elements(index_f) eq ndat then begin
             index = -1 & tmp = temporary(index)
          endif
       endif
    endif
    if traninvalid then begin
       if n_elements(emptyitem) eq 0 then emptyitem = label else emptyitem = emptyitem+' and '+label
    endif else begin
       extra_tit = self->range_string(extra_tit,label,tmp_range,unit)
       if is_hkl and (total(finite(range,/nan)) eq 0) and (n_elements(qoffset) ne 0) then begin
          newid = self->dm_adjust_trajid(id)
          ;adjust axis title only when the offset range is less than 0.5
          if (newid ge 0) and (newid le 2) and (abs(range[1]-range[0]) le self.qoffsetthreshold) then qoffset = qoffset+([[self.view_u1],[self.view_u2],[self.view_u3]])[*,newid]*mean(tmp_range)
       endif
    endelse
end

;adjust the trajectory ID number according to sample type and extra viewing axis flag
function dcs_mslice::dm_adjust_trajid,id,is_E=is_E
    is_E = 0b
    if n_elements(id) eq 0 then return,0
    if (self.instrname eq 'macs') and (total([self.extravaxis_yn,keyword_set(self.macshistomode)]) gt 0) then begin
       if keyword_set(self.macshistomode) then Cid = ([4,3])[self.samp_typ] else Cid = -10
       if self.extravaxis_yn[1] then Tid = 4+keyword_set(self.macshistomode)-self.samp_typ else Tid = -10
       if self.extravaxis_yn[2] then Hid = 4+keyword_set(self.macshistomode)-self.samp_typ+self.extravaxis_yn[1] else Hid = -10
       if self.extravaxis_yn[0] then Sid = 4+keyword_set(self.macshistomode)-self.samp_typ+self.extravaxis_yn[1]+self.extravaxis_yn[2] else Sid = -10
    endif
    if (self.instrname eq 'dcs') and (total(self.extravaxis_yn[1:*]) gt 0) then begin
       tmp = [4+(self.intn_typ eq n_elements(self.intn_typ_str)-2),[2,1,2]+self.extravaxis_yn[0]]
       if self.extravaxis_yn[1] then Tid = tmp[self.samp_typ]+1 else Tid = -10
       if self.extravaxis_yn[2] then Hid = tmp[self.samp_typ]+1+self.extravaxis_yn[1] else Hid = -10
    endif
    if (self.samp_typ gt 0) and (id gt 1) then begin
       Eid = [-10,2,-10,2]
       if self.extravaxis_yn[0] and (self.instrname ne 'macs') then Eid = Eid+1
       if id eq Eid[self.samp_typ] then begin     ;Eid is always 3
          id = 3 
          is_E = 1b
       endif else begin                           ;Tid is always 5
          if (self.instrname eq 'macs') and (total([self.extravaxis_yn,keyword_set(self.macshistomode)]) gt 0) then begin
             if id eq Cid then id = 4  $          ;histo time id  is always 4
             else if id eq Sid then id = 7 $      ;specified axis is always 7
             else if id eq Hid then id = 6 $      ;magnetic field is always 6
             else if id eq Tid then id = 5        ;temperature id is always 5
          endif else if self.instrname eq 'wand' then begin
             if self.extravaxis_yn[0] and (id eq 2) then id = 5   ;Tid is always 5
          endif else if (self.instrname eq 'dcs') and (total(self.extravaxis_yn[1:2]) gt 0) then begin
             if id eq Hid then id = 6  $          ;magnetic field is always 6
             else if id eq Tid then id = 5        ;temperature id is always 5
          endif
       endelse
    endif else if (self.samp_typ eq 0) then begin ;powder type
       if id eq 0 then is_E = 1b 
       if (self.instrname eq 'macs') and (total([self.extravaxis_yn,keyword_set(self.macshistomode)]) gt 0) then begin
          if id eq Cid then id = 4  $             ;histo time id  is always 4
          else if id eq Sid then id = 7 $         ;specified axis is always 7
          else if id eq Hid then id = 6 $         ;magnetic field is always 6
          else if id eq Tid then id = 5           ;temperature id is always 5
       endif else if (self.instrname eq 'dcs') and (total(self.extravaxis_yn[1:2]) gt 0) then begin
          if id eq Hid then id = 6      $         ;magnetic field is always 6
          else if id eq Tid then id = 5           ;temperature id is always 5
       endif
       if (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) and (id ge 4) then begin ;y axis for J(Q,y) intensity type 
          Cid = ([0,5])[self.instrname eq 'dcs']
          Yid = ([4,4])[self.instrname eq 'dcs']
          if id eq Yid then id = 8                ;y id is always 8
          if id eq Cid then id = 4                ;time channel id is always 4
       endif
    endif
    if id eq 4 then is_E = 1b                     ;time channel id
    return,id
end

;adjust axis title due to qoffset
function dcs_mslice::dm_adjust_xyztit,qoffset,title=title,hkl_info=hkl_info
    if n_elements(hkl_info) eq 0 then hkl_info = {vaxis:[0.,0.,0.],label:(n_elements(title) eq 1)?title:'',offset:[0.,0.,0.],unit:''}
    if n_elements(qoffset) eq 3 then hkl_info.offset = hkl_info.offset+qoffset 
    return,dm_set_viewlabel(hkl_info.label,hkl_info.vaxis,hkl_info.offset,hkl_info.unit)
end

; parameter:
;     id:           an integer
; keywords:
;  input:
;     range:        if=[NAN, NAN], then assume all possible data will be returned, and return empty=0
;     macsa3:       if set, return a3 as the data, id parameter is ignored
;     macsa4:       if set, return a4 as the data, id parameter is ignored
;     macstime:     if set, return histo time as the data, id parameter is ignored
;  output:
;     data:         qty
;     error:        dqty
;     dadata:       dark angle trajectory data
;     hkl_info:     for hkl axes, return {vaxis:[3],label:title,offset:[3]}
;     is_hkl:       1 if the data is from one of u1,u2,(or u3 if allowed) hkl axes, 0 otherwise
;     is_E:         1 if the data is energy(or has unique values for general multiple-d data), 0 otherwise
;     label:        label
;     unit:         unit
;     title:        title=label (unit)
;     index:
;     daindex:      use the same range for dadata index, if empty, daindex=-1
;     empty:        a flag to show whether data is empty for given range, if empty=1, then index will be all possible data index
;     macshistonchan: number of time channel for macs histo mode
pro dcs_mslice::dm_choose_trajdata,id,data=data,error=error,label=label,unit=unit,title=title,index=index,range=range,hkl_info=hkl_info,$
    is_hkl=is_hkl,is_E=is_E,empty=empty,debug=debug,dadata=dadata,daindex=daindex,macsa3=macsa3,macsa4=macsa4,macstime=macstime,$
    macshistonchan=macshistonchan  
    if keyword_set(debug) then begin
       print,'[[choose traj data...'
       current = systime(/sec)
    endif
    if n_elements(id) ne 0 then old_id = id ;to save original id since id will be adjusted later
    samp_typ = self.samp_typ
    negative2th = self.negative2th
    if samp_typ eq -1 then begin
       if id lt n_elements((*self.dataPtr).is_uniq) then is_E = (*self.dataPtr).is_uniq[0>(id)] else is_E = 0b
       ndat = n_elements((*self.dataPtr).data[*,0])
    endif else begin
       nen  = n_elements((*self.projStrPtr).en)
       n2th = n_elements((*self.projStrPtr).two_theta)
       id   = self->dm_adjust_trajid(id,is_E=is_E)
       if (self.instrname eq 'macs') then begin
          macshistonchan = self.macshistonchan-(self.macshistocut?dm_macs_histotail(self.macshistonchan):0)
          if keyword_set(macsa3) then begin
             samp_typ = 1
             id = -3
          endif else if keyword_set(macsa4) then begin
             negative2th = 1b
             samp_typ = 0
             id = 2
          endif else if keyword_set(macstime) then begin
             id = 4 
             is_E = 1b
          endif
       endif
    endelse
    label = ''
    unit  = '' 
    title = ''
    is_hkl = (samp_typ ge 1) and (id ge 0) and (id le 2)
    if is_hkl then hkl_info =  {vaxis:([[self.view_u1],[self.view_u2],[self.view_u3]])[*,id],label:([self.view_u1lab[1],self.view_u2lab[1],self.view_u3lab])[id],offset:[0.,0.,0.],unit:''} $
    else hkl_info = {vaxis:[0.,0.,0.],label:'',offset:[0.,0.,0.],unit:''}
    if (id eq -4) and (samp_typ ge 0) then begin
       weight = (*self.projStrPtr).weight
       if size(weight,/n_dim) eq 0 then weight = reform(weight,1)  ;force it to be an array
    endif
    case samp_typ of
    -1:begin   ;volume data
       unit  = (*self.dataPtr).unit[0>(id)]
       label = (*self.dataPtr).label[0>(id)]
       case id of
         -4:begin ;weight
            if ptr_valid(self.weightPtr[0]) then data = *(self.weightPtr[0])
            return
            end
         -1:begin
            data  = reform((*self.dataPtr).data[*,0])
            error = reform((*self.dataPtr).data[*,1])
            end
         else:begin
            data  = reform((*self.dataPtr).data[*,id])
            hkl_info = (*self.dataPtr).axis_info[0>(id-2)]
            is_hkl = (total(abs(hkl_info.vaxis)) ne 0)
            hkl_info = create_struct(hkl_info,'unit',unit)         
            end 
       endcase
       end
     0:begin   ;powder
       if (self.instrname eq 'dcs') and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
          tmp = size((*self.projStrPtr).qty)
          if tmp[0] eq 3 then nfile = tmp[3] else nfile = 1L
       endif else nfile = 1L
       case id of
         -4:begin ;weight
            if self.instrname eq 'macs' then begin
               if keyword_set(self.macshistomode) then begin
                  data = (replicate(1,macshistonchan)#weight)*(*self.projStrPtr).histo_weight
                  data = rebin(data,macshistonchan,nen,n2th)
                  if self.macs_rawintn then begin  ;efficiency is applied to weight
                     for i=0,n2th-1 do data[*,*,i] = data[*,*,i]/(*self.eff)[self.macsintn_typ,i]
                  endif
               endif else data = rebin(weight,nen,n2th)
            endif else data = rebin(weight,nen,n2th)
            end
         -3:begin ;A3 for MACS
            if self.instrname eq 'macs' then $
               data  = (90.0-(*self.projStrPtr).psi)#(fltarr(n2th)+1.0) $
            else $
               data  = fltarr(nen,n2th)
            label = 'A3'
            unit  = string('B0'XB)
            end
         -2:begin
            data  = reform(rebin((*self.projStrPtr).ewid,nen,n2th,nfile))       
            label = 'E Width'
            unit  = 'meV'
            end
         -1:begin
            if self.instrname eq 'macs' then begin
               if self.macsintn_typ eq 2 then begin
                  if keyword_set(self.macshistomode) then begin
                     data  = (*self.projStrPtr).qty[*,*,2*indgen(n2th)]+(*self.projStrPtr).qty[*,*,2*indgen(n2th)+1]
                     error = sqrt(((*self.projStrPtr).err[*,*,2*indgen(n2th)])^2+((*self.projStrPtr).err[*,*,2*indgen(n2th)+1])^2)
                  endif else begin
                     data  = reform((*self.projStrPtr).qty[0,*,*]+(*self.projStrPtr).qty[1,*,*])
                     error = reform(sqrt(((*self.projStrPtr).err[0,*,*])^2+((*self.projStrPtr).err[1,*,*])^2))
                  endelse
               endif else begin
                  if keyword_set(self.macshistomode) then begin
                     data  = (*self.projStrPtr).qty[*,*,2*indgen(n2th)+self.macsintn_typ]
                     error = (*self.projStrPtr).err[*,*,2*indgen(n2th)+self.macsintn_typ]
                  endif else begin
                     data  = reform((*self.projStrPtr).qty[self.macsintn_typ,*,*])
                     error = reform((*self.projStrPtr).err[self.macsintn_typ,*,*])
                  endelse
               endelse
            endif else begin
               data  = (*self.projStrPtr).qty
               error = (*self.projStrPtr).err
            endelse
            label = 'Intensity'
            if strlen(self.intn_lab) ne 0 then label = self.intn_lab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.absp_info) and self.absp_asint then label = 'Transmission'
            end
         0: begin
            data = reform(rebin((*self.projStrPtr).en,nen,n2th,nfile))
            if (self.instrname eq 'macs') and (self.macsintn_typ ne 0) then data[*] = 0.0
            end
         1: begin
            if self.instrname eq 'macs' then $
               data = reform((*self.projStrPtr).Q[self.macsintn_typ ne 0,*,*]) $
            else begin
               data = (*self.projStrPtr).Q
               if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            endelse          
            end
         2: begin
            if self.instrname eq 'macs' then begin
               data = (*self.projStrPtr).kidney#(fltarr(n2th)+1.0)
               for i=0,n2th-1 do begin
                   data[*,i] = data[*,i]+(*self.projStrPtr).two_theta[i]-76.
               endfor
               if ~negative2th then data = abs(data)
            endif else begin
               data = (fltarr(nen)+1)#(*self.projStrPtr).two_theta
               if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            endelse
            end
         3: begin
            data  = (fltarr(nen)+1)#(findgen(n2th)+1.0)
            if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            end
         4: begin ;Time for DCS and MACS
            if self.instrname eq 'dcs' then begin
               data  = reform(rebin((*self.projStrPtr).time,nen,n2th,nfile))
            endif else if self.instrname eq 'macs' then begin
               data  = (findgen(macshistonchan)+0.5)*self.macshistowidth
               if (self.macshistomode eq 2) and (self.macshistot0[0] ne 0) then data = (data+self.macshistot0[0]) mod (self.macshistowidth*macshistonchan)
               data  = rebin(data/1000.,macshistonchan,nen,n2th)
               label = 'Time'
               unit  = (['ms','s'])[self.macshistounit]
            endif else begin
               id    = 0
               data  = (*self.projStrPtr).en#(fltarr(n2th)+1.0)
            endelse
            end
         5: begin   ;Temperture, DCS & MACS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).temperature[self.macstemp_typ,i]
               endfor
            endif else if (self.instrname eq 'macs') and self.extravaxis_yn[1] then $
               data = (*self.projStrPtr).temperature[*,self.macstemp_typ]#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = 'T'
            unit  = 'K'
            end
         6: begin   ;magnetic field, DCS & MACS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[2] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).hfield[i]
               endfor
            endif else if (self.instrname eq 'macs') and self.extravaxis_yn[2] then $
               data = (*self.projStrPtr).hfield#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = '\mu!d0!nH'
            unit  = 'T'
            end
         7: begin   ;specified quantity, MACS only
            if (self.instrname eq 'macs') and self.extravaxis_yn[0] then $
               data = (*self.projStrPtr).specify#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = self.extravaxis_spec[0]
            unit  = self.extravaxis_spec[1]
            end
         8: begin   ;y for J(Q,y) intensity type
            data = (*self.projStrPtr).y
            if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            id   = 5
            end
         else:begin
            data  = reform(fltarr(nen,n2th,nfile))
            label = 'None'
            end
       endcase
       if (id ge 0) and (id le 5) and (strlen(label) eq 0) then begin
          label = self.view_label[id]
          unit  = self.view_unit[id]
       endif
       end
     1:begin   ;single crystal
       if (self.instrname eq 'dcs') and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
          tmp = size((*self.projStrPtr).qty)
          if tmp[0] eq 3 then nfile = tmp[3] else nfile = 1L
       endif else nfile = 1L
       case id of
         -4:begin ;weight
            if self.instrname eq 'macs' then begin
               if keyword_set(self.macshistomode) then begin
                  data = (replicate(1,macshistonchan)#weight)*(*self.projStrPtr).histo_weight
                  data = rebin(temporary(data),macshistonchan,nen,n2th)
                  if self.macs_rawintn then begin  ;efficiency is applied to weight
                     for i=0,n2th-1 do data[*,*,i] = data[*,*,i]/(*self.eff)[self.macsintn_typ,i]
                  endif
               endif else data = rebin(weight,nen,n2th)
            endif else data = rebin(weight,nen,n2th)
            end
         -3:begin
            if (self.instrname eq 'macs') then $
               data = (90.0-(*self.projStrPtr).psi)#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = 'A3'
            unit  = string('B0'XB)
            end
         -2:begin
            data  = reform(rebin((*self.projStrPtr).ewid,nen,n2th,nfile))
            label = 'E Width'
            unit  = 'meV'
            end
         -1:begin
            if self.instrname eq 'macs' then begin
               if self.macsintn_typ eq 2 then begin
                  if keyword_set(self.macshistomode) then begin
                     data  = (*self.projStrPtr).qty[*,*,2*indgen(n2th)]+(*self.projStrPtr).qty[*,*,2*indgen(n2th)+1]
                     error = sqrt(((*self.projStrPtr).err[*,*,2*indgen(n2th)])^2+((*self.projStrPtr).err[*,*,2*indgen(n2th)+1])^2)
                  endif else begin                    
                     data  = reform((*self.projStrPtr).qty[0,*,*]+(*self.projStrPtr).qty[1,*,*])
                     error = reform(sqrt(((*self.projStrPtr).err[0,*,*])^2+((*self.projStrPtr).err[1,*,*])^2))
                  endelse
               endif else begin
                  if keyword_set(self.macshistomode) then begin
                     data  = (*self.projStrPtr).qty[*,*,2*indgen(n2th)+self.macsintn_typ]
                     error = (*self.projStrPtr).err[*,*,2*indgen(n2th)+self.macsintn_typ]
                 endif else begin
                     data  = reform((*self.projStrPtr).qty[self.macsintn_typ,*,*])
                     error = reform((*self.projStrPtr).err[self.macsintn_typ,*,*])
                 endelse
               endelse
            endif else begin
               data  = (*self.projStrPtr).qty
               error = (*self.projStrPtr).err
            endelse
            label = 'Intensity'
            if strlen(self.intn_lab) ne 0 then label = self.intn_lab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.absp_info) and self.absp_asint then label = 'Transmission'
            end
         0: begin
            data  = reform((*self.projStrPtr).V[0,*,*])
            if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            label = self.traj_type[0]
            end
         1: begin
            data  = reform((*self.projStrPtr).V[1,*,*])
            if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            label = self.traj_type[1]
            end
         2: begin
            data  = reform((*self.projStrPtr).V[2,*,*])
            if nfile gt 1 then data = rebin(data,nen,n2th,nfile)
            label = self.traj_type[2]
            end
         3: begin
            data  = reform(rebin((*self.projStrPtr).en,nen,n2th,nfile))    
            label = 'E'
            unit  = 'meV'
            end
         4: begin ;histo time for MACS
            if (self.instrname eq 'macs') then begin               
               data  = (findgen(macshistonchan)+0.5)*self.macshistowidth
               if (self.macshistomode eq 2) and (self.macshistot0[0] ne 0) then data = (data+self.macshistot0[0]) mod (self.macshistowidth*macshistonchan)
               data  = rebin(data/1000.,macshistonchan,nen,n2th)
            endif else $
               data  = reform(fltarr(nen,n2th,nfile))
            label = 'Time'
            unit  = (['ms','s'])[self.macshistounit]
            end 
         5: begin   ;Temperture, DCS & MACS only 
            if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).temperature[self.macstemp_typ,i]
               endfor
            endif else if (self.instrname eq 'macs') and self.extravaxis_yn[1] then $
               data = (*self.projStrPtr).temperature[*,self.macstemp_typ]#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = 'T'
            unit  = 'K'
            end    
         6: begin   ;magnetic field, DCS & MACS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[2] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).hfield[i]
               endfor
            endif else if (self.instrname eq 'macs') and self.extravaxis_yn[2] then $
               data = (*self.projStrPtr).hfield#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = '\mu!d0!nH'
            unit  = 'T'
            end
         7: begin   ;specified quantity, MACS only
            if (self.instrname eq 'macs') and self.extravaxis_yn[0] then $
               data = (*self.projStrPtr).specify#(fltarr(n2th)+1.0) $
            else $
               data = reform(fltarr(nen,n2th,nfile))
            label = self.extravaxis_spec[0]
            unit  = self.extravaxis_spec[1]
            end   
         else:begin
            data  = reform(fltarr(nen,n2th,nfile))
            label = 'None'
            end
       endcase
       end
     2:begin   ;diffuse scattering single E
       tmp = size((*self.projStrPtr).V)
       if tmp[0] eq 2 then nfile=1L else nfile=tmp[3]
       case id of
         -4:begin ;weight
            data = transpose(rebin(weight,nfile,n2th))
            end
         -1:begin
            data  = (*self.projStrPtr).qty
            error = (*self.projStrPtr).err
            label = 'Intensity'
            if strlen(self.intn_lab) ne 0 then label = self.intn_lab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.absp_info) and self.absp_asint then label = 'Transmission'
            end
         0: begin
            data  = reform((*self.projStrPtr).V[0,*,*])
            label = self.traj_type[0]
            end
         1: begin
            data  = reform((*self.projStrPtr).V[1,*,*])
            label = self.traj_type[1]
            end
         2: begin
            data  = reform((*self.projStrPtr).V[2,*,*])
            label = self.traj_type[2]
            end
         5: begin   ;Temperture, DCS & WAND only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then $ 
               data = (fltarr(n2th)+1.0)#reform((*self.projStrPtr).temperature[self.macstemp_typ,*]) $
            else if ((self.instrname eq 'wand') and self.extravaxis_yn[0]) then $
               data = (fltarr(n2th)+1.0)#(*self.projStrPtr).temperature $
            else $
               data = fltarr(n2th,nfile)
            label = 'T'
            unit  = 'K'
            end
         6: begin   ;magnetic field, DCS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[2] then $
               data = (fltarr(n2th)+1.0)#(*self.projStrPtr).hfield $
            else $
               data = fltarr(n2th,nfile)
            label = '\mu!d0!nH'
            unit  = 'T'
            end     
         else:begin
            data  = fltarr(n2th,nfile)
            label = 'None'
            end
       endcase
       end
     3:begin   ;diffuse scattering multiple E
       tmp = size((*self.projStrPtr).V)
       if tmp[0] eq 3 then nfile=1L else nfile=tmp[4]
       case id of
         -4:begin ;weight
            data = rebin(weight,nfile,nen,n2th)
            data = transpose(data,[1,2,0])
            end
         -2:begin
            data  = reform(rebin((*self.projStrPtr).ewid,nen,n2th,nfile))
            label = 'E Width'
            unit  = 'meV'
            end
         -1:begin
            data  = (*self.projStrPtr).qty
            error = (*self.projStrPtr).err
            label = 'Intensity'
            if strlen(self.intn_lab) ne 0 then label = self.intn_lab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.absp_info) and self.absp_asint then label = 'Transmission'
            end
         0: begin
            data  = reform((*self.projStrPtr).V[0,*,*,*])
            label = self.traj_type[0]
            end
         1: begin
            data  = reform((*self.projStrPtr).V[1,*,*,*])
            label = self.traj_type[1]
            end
         2: begin
            data  = reform((*self.projStrPtr).V[2,*,*,*])
            label = self.traj_type[2]
            end
         3: begin
            data  = reform(rebin((*self.projStrPtr).en,nen,n2th,nfile))
            label = 'E'
            unit  = 'meV'
            end
         5: begin   ;Temperture, DCS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).temperature[self.macstemp_typ,i]
               endfor
            endif else begin
                data = fltarr(nen,n2th,nfile)
            endelse
            label = 'T'
            unit  = 'K'
            end
         6: begin   ;magnetic field, DCS only
            if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then begin
               data = fltarr(nen,n2th,nfile)
               for i=0L,nfile-1L do begin
                   data[*,*,i] = (*self.projStrPtr).hfield[i]
               endfor
            endif else begin
               data = fltarr(nen,n2th,nfile)
            endelse
            label = '\mu!d0!nH'
            unit  = 'T'
            end   
         else:begin
            data  = fltarr(nen,n2th,nfile)
            label = 'None'
            end
       endcase
       end
     else:
    endcase
    title = label+((strlen(unit) ne 0)?' ('+unit+')':'')
    count = 0
    empty = 0b
    if (self.instrname eq 'macs') and keyword_set(self.macshistomode) and ptr_valid(self.dataStrPtr) and ((id ne -1) and (id ne 4) and (id ne -4)) then begin
       if nen eq 1 then data = reform(data,nen,n2th)
       if macshistonchan eq 1 then data = reform(data,nen,n2th,macshistonchan) $
       else data = rebin(data,nen,n2th,macshistonchan)
       data = transpose(data,[2,0,1])
    endif
    if (n_elements(range) eq 2) and (label ne 'None') then begin
       tmp_ran = range
       if total(finite(tmp_ran)) ne 0 then begin
          if finite(tmp_ran[0],/nan) then tmp_ran[0]=-1e30 ;assign a very large negative number
          if finite(tmp_ran[1],/nan) then tmp_ran[1]=1e30  ;assign a very large positive number
          index = where((data ge min(tmp_ran)) and (data le max(tmp_ran)),count)
          if count eq 0 then empty=1b
       endif
    endif
    if (count eq 0) or (label eq 'None') then begin
       if samp_typ eq -1 then index=lindgen(ndat) $                ;volume data
       else if samp_typ eq 2 then index=lindgen(nfile*n2th)  $     ;diffuse single E
       else if samp_typ eq 3 then index=lindgen(nfile*n2th*nen) $  ;diffuse multiple E
       else index=lindgen(nen*n2th*nfile)                               ;powder,single crystal
    endif
    if ((samp_typ ge 2) or ((self.instrname eq 'macs') and (samp_typ eq 1))) and (arg_present(dadata) or arg_present(daindex)) and ((id ge 0) and (id le 3)) then begin
       field = tag_names(*self.projStrPtr)
       loc   = where(strupcase(field) eq 'DAQV',existed)
       if (existed gt 0) then begin
          if is_E then begin
             if self.instrname eq 'macs' then begin
                dadata = (*self.projStrPtr).daen
             endif else begin
                en = (*self.projStrPtr).en
                nen = n_elements(en)
                ndapsi = n_elements((*self.projStrPtr).daQV[0,*])/nen
                dadata = rebin(en,nen,ndapsi)
                dadata = reform(dadata,nen*ndapsi,/overwrite)
             endelse
          endif else begin
             dadata = reform((*self.projStrPtr).daQV[id,*])
          endelse
       endif
       if arg_present(daindex) and (n_elements(tmp_ran) eq 2) and (n_elements(dadata) ne 0) then begin
          daindex = where((dadata ge min(tmp_ran)) and (dadata le max(tmp_ran)),count)
       endif
    endif
    if n_elements(old_id) ne 0 then id = old_id
    if keyword_set(debug) then begin
       print,'[[choose traj data finished in ',systime(/sec)-current,' sec.'
    endif
end