; $Id: $
;#######################################################################
;
; NAME:
;  dm_choose_slicedata
;
; PURPOSE:
;  choose slice data
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   None
; keywords:
;   (xyz)dat:     data to be returned
;   zerr:         error bar of zdat
;   a3,a4:        for macs only, only when self.macs_savea4=1
;   (xy)info:     additional axis information {vaxis:[3],label:'',offset:[3]}
;   (xy)tit:      label of the data
;   zran:         for z data only, contour plot color bar range
;   zmin:         for z data only
;   extra_tit:    extra title info
;   surfplot:     if set, xdat=[nx],ydat=[ny],zdat=[nx,ny]
;   weight:       weight for avgsum=-1
pro dcs_mslice::dm_choose_slicedata,xdat=xdat,ydat=ydat,zdat=zdat,zerr=zerr,a3=a3,a4=a4,zran=zran,xinfo=xinfo,xtit=xtit,ytit=ytit,yinfo=yinfo,ztit=ztit,zmin=zmin,zmax=zmax,$
    tran_invalid=tran_invalid,extra_tit=extra_tit,surfplot=surfplot,error=error,weight=weight,debug=debug
    if keyword_set(debug) then begin
       print,'&&choose slice data...'
       current=systime(/sec)
    endif
    
    ;weight
    if (self.bin_avgsum eq -1) or ((self.instrname eq 'macs') and keyword_set(self.macshistomode) and self.macs_rawintn) then self->dm_choose_trajdata,-4,data=weight
    
    ;intensity & error bar
    if arg_present(zerr) then begin
       ok_zerr = 1b
       self->dm_choose_trajdata,-1,data=zdat,error=zerr,title=ztit,debug=debug 
    endif else begin
       ok_zerr = 0b
       self->dm_choose_trajdata,-1,data=zdat,title=ztit,debug=debug 
    endelse
    ndat = n_elements(zdat)
    
    extra_tit = ''
    
    nogrid = [0b,0b]
    
    ;choose x and y data
    self->dm_choose_trajdata,self.slice_x[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=xdat,title=xtit,is_E=is_E_x,debug=debug,is_hkl=is_hkl_x,hkl_info=xinfo
    self->dm_choose_trajdata,self.slice_y[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=ydat,title=ytit,is_E=is_E_y,debug=debug,is_hkl=is_hkl_y,hkl_info=yinfo
    
    if is_hkl_x or is_hkl_y then qoffset = 0.
 
    ;choose a3, a4
    if (self.instrname eq 'macs') and self.macs_savea4 then begin
       if arg_present(a3) then self->dm_choose_trajdata,data=a3,/macsa3,debug=debug
       if arg_present(a4) then self->dm_choose_trajdata,data=a4,/macsa4,debug=debug
    endif
    ok_a3 = (n_elements(a3) eq ndat)
    ok_a4 = (n_elements(a4) eq ndat)
    ok_wt = (n_elements(weight) eq ndat)

    nogrid[0] = finite(self.slice_xstep,/nan) or (self.slice_xstep eq 0)
    nogrid[1] = finite(self.slice_ystep,/nan) or (self.slice_ystep eq 0)
       
    if self.samp_typ eq -1 then begin ;general multiple-d data
       ok_ewid = 0b
       ;thickness  
       ndat     = n_elements((*self.dataPtr).data[*,0])
       nitem    = n_elements((*self.dataPtr).label)
       if nitem gt 4 then tran_invalid = bytarr(nitem-4)
       for i=0,nitem-5 do begin
           is_hkl_t = 0b
           if (total(finite((*self.slice_tft)[i,*])) ne 0) then begin
              self->dm_choose_trajdata,(*self.slice_t)[i]+2,index=index1,range=reform((*self.slice_tft)[i,*]),empty=empty,debug=debug,unit=unit,label=label,is_hkl=is_hkl_t,hkl_info=hkl_info_t
              if empty then begin
                 if n_elements(emptyitem) eq 0 then $
                    emptyitem = label $
                 else $
                    emptyitem = emptyitem+' and '+label 
                 index1 = -1 & tmp = temporary(index1)
                 tran_invalid[i] = 1b
              endif else if n_elements(index1) eq ndat then begin
                 index1 = -1 & tmp = temporary(index1)
              endif else begin
                 if n_elements(index) eq 0 then index = index1 $
                 else begin
                    index = dm_common(index,index1,/no_copy)
                 endelse
              endelse
           endif
           if ~tran_invalid[i] then begin
              extra_tit = self->range_string(extra_tit,label,(*self.slice_tft)[i,*],unit)
              if is_hkl_t and (total(finite((*self.slice_tft)[i,*],/nan)) eq 0) and (n_elements(qoffset) ne 0) then begin
                 if (abs((*self.slice_tft)[i,1]-(*self.slice_tft)[i,0]) le self.qoffsetthreshold) then qoffset = qoffset+hkl_info_t.vaxis*mean((*self.slice_tft)[i,*])
              endif
           endif
       endfor
       if n_elements(index) ne 0 then begin
          if index[0] eq -1 then begin
             emptyitem = 'combined thickness'
             if n_elements(tran_invalid) ne 0 then begin
                tran_invalid[*] = 1b
                extra_tit = ''
             endif
          endif else begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_zerr then zerr = zerr[index]
             if ok_a3   then a3 = a3[index]
             if ok_a4   then a4 = a4[index]
             index = -1
          endelse
       endif
       if is_E_x and nogrid[0] then uniq_E_x = xdat
       if is_E_y and nogrid[1] then uniq_E_y = ydat
    endif else begin     
       ;ewid
       if self.eint_yn and (self.samp_typ ne 2) then begin
          self->dm_choose_trajdata,-2,data=ewid
          emax = max((*self.projStrPtr).en,min=emin)
          dE   = emax-emin
       endif
       ok_ewid = (n_elements(ewid) eq ndat)
    
       if is_E_x and nogrid[0] then uniq_E_x = xdat[*,0,0]
       if is_E_y and nogrid[1] then uniq_E_y = ydat[*,0,0]
    
       tran_invalid = bytarr(4)
       is_E_t = bytarr(4)
       check_fold = bytarr(10) ;u1,u2,u3,E,T,H,...
       is_fold2 = ((self.samp_typ gt 0) and (self.fold_type eq 1))
       is_u3 = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
       check_fold[0] = (is_fold2 and self.view_u1fold and finite((*(self.view_u1foldcenter))[0]))
       check_fold[1] = (is_fold2 and self.view_u2fold and finite((*(self.view_u2foldcenter))[0]))
       check_fold[2] = (is_fold2 and is_u3 and self.view_u3fold and finite((*(self.view_u3foldcenter))[0]))
       fold_center = [(*(self.view_u1foldcenter))[0],(*(self.view_u2foldcenter))[0],(*(self.view_u3foldcenter))[0],fltarr(7)]
       
       ;thickness
       ;t1 thickness
       if (total(finite(self.slice_t1ran)) ne 0) and self.disp_flag[2] then begin
          self->dm_check_thickness,self.slice_t1[self.samp_typ],range=self.slice_t1ran,ndat=ndat,index=index1,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.slice_t1[self.samp_typ]],foldcenter=fold_center[self.slice_t1[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[0] = is_E
          tran_invalid[0] = traninvalid
       endif
       ;t2 thickness
       if (total(finite(self.slice_t2ran)) ne 0) and self.disp_flag[3] then begin
          self->dm_check_thickness,self.slice_t2[self.samp_typ],range=self.slice_t2ran,ndat=ndat,index=index2,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.slice_t2[self.samp_typ]],foldcenter=fold_center[self.slice_t2[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[1] = is_E
          tran_invalid[1] = traninvalid
       endif
       ;t3 thickness
       if (total(finite(self.slice_t3ran)) ne 0) and self.disp_flag[4] then begin
          self->dm_check_thickness,self.slice_t3[self.samp_typ],range=self.slice_t3ran,ndat=ndat,index=index3,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.slice_t3[self.samp_typ]],foldcenter=fold_center[self.slice_t3[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[2] = is_E
          tran_invalid[2] = traninvalid
       endif
       ;t4 thickness
       if (total(finite(self.slice_t4ran)) ne 0) and self.disp_flag[5] then begin
          self->dm_check_thickness,self.slice_t4[self.samp_typ],range=self.slice_t4ran,ndat=ndat,index=index4,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.slice_t4[self.samp_typ]],foldcenter=fold_center[self.slice_t4[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[3] = is_E
          tran_invalid[3] = traninvalid
       endif

       if n_elements(index1) ne 0 then index = temporary(index1)
       if n_elements(index2) ne 0 then begin
          if n_elements(index) ne 0 then index = dm_common(index,index2,/no_copy) $
          else index = temporary(index2)
       endif
       if n_elements(index3) ne 0 then begin
          if n_elements(index) ne 0 then index = dm_common(index,index3,/no_copy) $
          else index = temporary(index3)
       endif
       if n_elements(index4) ne 0 then begin
          if n_elements(index) ne 0 then index = dm_common(index,index4,/no_copy) $
          else index = temporary(index4)
       endif
      
       if n_elements(index) ne 0 then begin
          if index[0] eq -1 then begin
             emptyitem = 'combined thickness'
             tran_invalid[*] = 1b
             extra_tit = ''
          endif else if n_elements(zdat) ne n_elements(index) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_zerr then zerr = zerr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_a3   then a3   = a3[index]
             if ok_a4   then a4   = a4[index]
             if ok_wt   then weight = weight[index]
          endif
          index = -1
       endif

       ;fold to both sides
       if check_fold[self.slice_x[self.samp_typ]] then begin
          xdat = [xdat,2.0*fold_center[self.slice_x[self.samp_typ]]-xdat]
          ydat = [ydat,ydat]
          zdat = [zdat,zdat]
          if ok_zerr then zerr = [zerr,zerr]
          if ok_ewid then ewid = [ewid,ewid]
          if ok_a3   then a3   = [a3,a3]
          if ok_a4   then a4   = [a4,a4]
          if ok_wt   then weight = [weight,weight]
       endif
       if check_fold[self.slice_y[self.samp_typ]] then begin
          xdat = [xdat,xdat]
          ydat = [ydat,2.0*fold_center[self.slice_y[self.samp_typ]]-ydat]
          zdat = [zdat,zdat]
          if ok_zerr then zerr = [zerr,zerr]
          if ok_ewid then ewid = [ewid,ewid]
          if ok_a3   then a3   = [a3,a3]
          if ok_a4   then a4   = [a4,a4]
          if ok_wt   then weight = [weight,weight]
       endif
       
       ;extra title info
       if strlen(extra_tit) eq 0 then sep='' else sep=', '
       if ((self.samp_typ eq 2) and (self.instrname ne 'wand')) or ((self.samp_typ eq 3) and (strlen(extra_tit) eq 0) and (~is_E_x) and (~is_E_y)) then begin
          case self.e_bin[self.samp_typ] of
             0:  extra_tit = extra_tit+''
             1:  extra_tit = extra_tit+sep+'Elastic'
             2:  extra_tit = self->range_string(extra_tit,'E',self.e_range,'meV')
          endcase
       endif
       if strlen(extra_tit) eq 0 then sep='' else sep=', '
       if (self.instrname eq 'dcs') and (total(self.dcsdetmask[1:3]) ne 0) then begin  ;add mask info for DCS
          ind = where(1-self.dcsdetmask[1:3],count)
          if count gt 0 then begin
             extra_tit = extra_tit+sep+(['Lower','Central','Upper'])[ind[0]]
             for i=1,count-1 do extra_tit = extra_tit+'+'+(['Lower','Central','Upper'])[ind[i]]
             extra_tit = extra_tit+' Bank'+(['','s'])[count gt 1]
          endif
       endif
       if (self.instrname eq 'macs') and (self.macsintn_typ ne 0) then begin
          extra_tit = extra_tit+sep+'Detector='+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]
       endif
       
       ;zero error for dcs, macs, and wand data
       if total(finite(self.zero_error,/nan)) eq 0 then zero_error = self.zero_error[0]*self.zero_error[2]
       bin_zeroerror = self.bin_zeroerror*((self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand'))
    endelse
    
    if n_elements(qoffset) eq 3 then begin
       if is_hkl_x then xtit = self->dm_adjust_xyztit(qoffset,title=xtit,hkl_info=xinfo)
       if is_hkl_y then ytit = self->dm_adjust_xyztit(qoffset,title=ytit,hkl_info=yinfo)
    endif

    if n_elements(emptyitem) ne 0 then $
       ok = dialog_message(['The specified '+emptyitem+' range covers no data.','The entire possible range is used instead.'],dialog_parent=self.tlb,/center)

    ;horizontal range and vertical range
    n_xran = total(finite(self.slice_xran))
    n_yran = total(finite(self.slice_yran))
    if n_xran eq 2 then begin
       index = where((xdat le max(self.slice_xran)) and (xdat ge min(self.slice_xran)),count)
       if (count gt 0) and (count ne n_elements(zdat)) then begin
          xdat = xdat[index]
          ydat = ydat[index]
          zdat = zdat[index]
          if ok_ewid then ewid = ewid[index]
          if ok_zerr then zerr = zerr[index]
          if ok_a3   then a3 = a3[index]
          if ok_a4   then a4 = a4[index]
          if ok_wt   then weight = weight[index]
          index = -1
          xstart = min(self.slice_xran)
       endif
    endif else begin
       if finite(self.slice_xran[0]) then begin
          index = where(xdat ge self.slice_xran[0],count)
          if (count gt 0) and (count ne n_elements(zdat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_ewid then ewid = ewid[index]
             if ok_zerr then zerr = zerr[index]
             if ok_a3   then a3 = a3[index]
             if ok_a4   then a4 = a4[index]
             if ok_wt   then weight = weight[index]
             index = -1
             xstart = self.slice_xran[0]
          endif
       endif else if finite(self.slice_xran[1]) then begin
          index = where(xdat le self.slice_xran[1],count)
          if (count gt 0) and (count ne n_elements(zdat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_ewid then ewid = ewid[index]
             if ok_zerr then zerr = zerr[index]
             if ok_a3   then a3 = a3[index]
             if ok_a4   then a4 = a4[index]
             if ok_wt   then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_yran eq 2 then begin
       index  = where((ydat le max(self.slice_yran)) and (ydat ge min(self.slice_yran)),count)
       if (count gt 0) and (count ne n_elements(zdat)) then begin
          xdat = xdat[index]
          ydat = ydat[index]
          zdat = zdat[index]
          if ok_ewid then ewid = ewid[index]
          if ok_zerr then zerr = zerr[index]
          if ok_a3   then a3 = a3[index]
          if ok_a4   then a4 = a4[index]
          if ok_wt   then weight = weight[index]
          index = -1
          ystart = min(self.slice_yran)
       endif
    endif else begin
       if finite(self.slice_yran[0]) then begin
          index = where(ydat ge self.slice_yran[0],count)
          if (count gt 0) and (count ne n_elements(zdat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_ewid then ewid = ewid[index]
             if ok_zerr then zerr = zerr[index]
             if ok_a3   then a3 = a3[index]
             if ok_a4   then a4 = a4[index]
             if ok_wt   then weight = weight[index]
             index = -1
             ystart = self.slice_yran[0]
          endif
       endif else if finite(self.slice_yran[1]) then begin
          index = where(ydat le self.slice_yran[1],count)
          if (count gt 0) and (count ne n_elements(zdat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             if ok_ewid then ewid = ewid[index]
             if ok_zerr then zerr = zerr[index]
             if ok_a3   then a3 = a3[index]
             if ok_a4   then a4 = a4[index]
             if ok_wt   then weight = weight[index]
             index = -1
          endif
       endif
    endelse

    ;rebinning according to the steps
    ;warning for slow Voronoi diagrams
    if nogrid[0] and nogrid[1] and (~keyword_set(surfplot) and self.allow_voronoi) and ~self.ftoolflag[2] then begin ;no rebinning
       self.ftoolflag[2] = 1b
       ans = dialog_message('The slice will be rendered by Voronoi diagram when both steps are empty. The plotting could be very slow. Do you want to disable the Voronoi diagram?',/question,dialog_parent=self.tlb,/center)
       if strlowcase(ans) eq 'yes' then begin
          self.allow_voronoi = 0b
          self->my_widget_control,['saveslicexyie','saveslicespe'],/sensitive
          self->my_menu_toggle,'voronoi'+['No','Yes'],self.allow_voronoi
       endif
    endif
    if nogrid[0] and nogrid[1] and (~keyword_set(surfplot) and self.allow_voronoi) then begin ;no rebinning
       ndat = n_elements(xdat)
       xdat = reform(xdat,ndat,/overwrite)
       ydat = reform(ydat,ndat,/overwrite)
       zdat = reform(zdat,ndat,/overwrite)
       if ok_zerr then zerr = reform(zerr,ndat,/overwrite)
       if ok_wt then weight = reform(weight,ndat,/overwrite)
       ok_ewid = 0b
    endif else begin
       xmax  = max(xdat,min=xmin)
       ymax  = max(ydat,min=ymin)
       hstep = (xmax-xmin)/400.
       vstep = (ymax-ymin)/400.
       if is_E_x and nogrid[0] then begin
          index = where((uniq_E_x ge xmin) and (uniq_E_x le xmax),count)
          if count ne 0 then uniq_E_x = uniq_E_x[index]
          if self.instrname eq 'macs' then begin
             xdat = round(temporary(xdat)*(1000d),/L64)*0.001
             uniq_E_x = round(temporary(uniq_E_x)*(1000d),/L64)*0.001
          endif
          extra = {uniq_xval:uniq_E_x} 
       endif
       if is_E_y and nogrid[1] then begin
          index = where((uniq_E_y ge ymin) and (uniq_E_y le ymax),count)
          if count ne 0 then uniq_E_y = uniq_E_y[index]
          if self.instrname eq 'macs' then begin
             ydat = round(temporary(ydat)*(1000d),/L64)*0.001
             uniq_E_y = round(temporary(uniq_E_y)*(1000d),/L64)*0.001
          endif
          if n_elements(extra) eq 0 then $
             extra = {uniq_yval:uniq_E_y} $
          else $
             extra = create_struct('uniq_yval',uniq_E_y,extra)
       endif
       index = -1
       if finite(self.slice_xstep) and (self.slice_xstep ne 0) then begin
          hstep = abs(self.slice_xstep)
          if self.bin_checkstep then error = dm_check_step(xdat,hstep,label=xtit,group_leader=self.tlb,nmax=1e5)
          if keyword_set(error) then return
       endif
       if finite(self.slice_ystep) and (self.slice_ystep ne 0) then begin
          vstep = abs(self.slice_ystep)
          if self.bin_checkstep then error = dm_check_step(ydat,vstep,label=ytit,group_leader=self.tlb,nmax=1e5)
          if keyword_set(error) then return
       endif
       
       checkfinite = self.ftypecheckmask[0] or (strlen(self.userf_str) ne 0)
       if ok_a3 then begin
          tmpx = xdat & tmpy = ydat
          dm_stepgrid_bin,hstep,vstep,tmpx,tmpy,a3,xstart=xstart,ystart=ystart,conststep=self.bin_conststep,debug=debug,checkfinite=checkfinite,extlib=self.bin_extlib,$
              group_leader=self.tlb,_extra=extra
          tmpx = 0 & tmpy = 0
       endif
       if ok_a4 then begin
          tmpx = xdat & tmpy = ydat
          dm_stepgrid_bin,hstep,vstep,tmpx,tmpy,a4,xstart=xstart,ystart=ystart,conststep=self.bin_conststep,debug=debug,checkfinite=checkfinite,extlib=self.bin_extlib,$
              group_leader=self.tlb,_extra=extra
          tmpx = 0 & tmpy = 0
       endif
       dm_stepgrid_bin,hstep,vstep,xdat,ydat,zdat,zerr=zerr,avgsum=self.bin_avgsum,bintr=self.binthreshold,xstart=xstart,ystart=ystart,conststep=self.bin_conststep,$
           ewid=ewid,debug=debug,checkfinite=checkfinite,extlib=self.bin_extlib,group_leader=self.tlb,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,$
           zoerrestrange=self.bin_statrange[0]*self.bin_statrange[1],estapplyall=self.bin_statrange[2],foundrange=foundrange,_extra=extra

       if n_elements(foundrange) ne 0 then self.bin_statrange[3:4] = [foundrange,1]

       ;smooth regular grid data
       index  = where(finite(zdat,/nan),count)  ;bins with no data
       refill = (count ne 0)                    ;flag for refilling
       if self.slice_smooth gt 0 then begin     ;smooth data 
          tmp = size(zdat,/dimension)
          if (n_elements(tmp) gt 1) and (min(tmp) gt 3) then begin
             if ok_zerr then zerr = zerr^2
             if self.idl_version ge 6.2 then begin
                for i=0, self.slice_smooth-1 do begin
                    zdat = convol(zdat,self.slice_sm_kernel,/edge_truncate,/nan,/normalize)
                    if ok_zerr then zerr = convol(zerr,self.slice_sm_kernel,/edge_truncate,/nan,/normalize)
                    if refill then begin
                       zdat[index] = !values.f_nan
                       if ok_zerr then zerr[index] = !values.f_nan
                    endif
                endfor
             endif else begin
                for i=0, self.slice_smooth-1 do begin
                    ;put 0 in bins with no data
                    if refill then begin
                       zdat[index] = 0.0
                       if ok_zerr then zerr[index] = 0.0
                    endif
                    zdat = convol(zdat,self.slice_sm_kernel,total(self.slice_sm_kernel),/edge_truncate)
                    if ok_zerr  then zerr = convol(zerr,self.slice_sm_kernel,total(self.slice_sm_kernel),/edge_truncate)
                endfor
                if refill then begin
                   zdat[index] = !values.f_nan
                   if ok_zerr then zerr[index] = !values.f_nan
                endif
             endelse
             if ok_zerr then zerr = sqrt(zerr)
          endif
       endif
       index = -1      
    endelse
    
    if ok_ewid then begin
       if is_E_x then begin
          dE = (finite(self.slice_xstep)?(abs(self.slice_xstep)):dE)
       endif else if is_E_y then begin
          dE = (finite(self.slice_ystep)?(abs(self.slice_ystep)):dE)
       endif else if self.disp_flag[2] then begin
          tmp_ran = [!values.f_nan,!values.f_nan]
          if self.disp_flag[2] and is_E_t[0] then tmp_ran = self.slice_t1ran
          if self.disp_flag[3] and is_E_t[1] then tmp_ran = self.slice_t2ran
          if self.disp_flag[4] and is_E_t[2] then tmp_ran = self.slice_t3ran
          if self.disp_flag[5] and is_E_t[3] then tmp_ran = self.slice_t4ran
          if finite(tmp_ran[0],/nan) then tmp_ran[0]=-1e30
          if finite(tmp_ran[1],/nan) then tmp_ran[1]=1e30
          tmp_ran = [min(tmp_ran),max(tmp_ran)]
          if tmp_ran[0] lt emin then tmp_ran[0]=emin
          if tmp_ran[1] gt emax then tmp_ran[1]=emax
          if tmp_ran[1] gt tmp_ran[0] then dE=tmp_ran[1]-tmp_ran[0]
       endif
       if dE le 0 then dE=1.0
       zdat = zdat*dE
       if ok_zerr then zerr = zerr*dE
    endif

    ;intensity range
    oldzmax = max(zdat,min=oldzmin,/nan)
    zmax = oldzmax & zmin = oldzmin
    if finite(self.slice_iran[0]) then $
       if self.slice_iran[0] lt oldzmax then zmin=self.slice_iran[0]
    if finite(self.slice_iran[1]) then $
       if self.slice_iran[1] gt oldzmin then zmax=self.slice_iran[1]
    zran = [zmin,zmax]
    if keyword_set(debug) then begin
       help,xdat,nogrid,uniq_E_x,ydat,uniq_E_y
       print,'&&choose slice data finished in ',systime(/sec)-current,' sec.'
    endif
end