; $Id: $
;#######################################################################
;
; NAME:
;  dcs_mslice
;
; PURPOSE:
;  Viewing and analyzing neutron time-of-flight data.
;
; CATEGORY:
;  dcs time-of-flight
;
; HISTORY:
;  06/2003: IDL5.6 compatible, include all functionality of matlab mslice, and windows .hlp help file
;  10/2003: IDL6.0 compatible, add diffuse scattering type, and dcs_mslice.pdf help file
;  09/2004: Make IDL dcs_mslice extendable to other TOF instruments, documentation in dcs_mslice_addftype.pdf
;  06/2005: IDL6.1 compatible, add diffuse scattering multiple energy type, rearange sample types
;  11/2005: Remove some items in the droplists, and add extra HKL range limitation
;  03/2006: Add INX file type
;  03/2007: Add grouped data fitting in powder mode, i.e. add QENS data reduction
;           modifications in background subtraction, especially the treatment of dark count files
;  07/2007: Add empty can subtraction from vanadium file in the first two sample types
;           Add automatic Bragg peak masking
;  02/2008: Compatible with inverse geometry TOF instruments
;  11/2008: more flexible fold x-axis operation, add energy resolution convolution tool
;  05/2009: revamp viewing axes, allow more flexbility in folding and viewing
;  06/2009: add stand-alone interface for general volume data
;  07/2009: add volume plot, and change the display window to tab widgets
;  11/2009: allow two sets of empty can backgrounds, rebuild the context sensitive menu 
;  12/2009: more robust automatic masking in SPE files
;  02/2010: added NXSPE support - Andrei Savici (saviciat@ornl.gov)
;  04/2010: possible to discard bins that have few pixels mapped into them 
;           keep track of correct ki/kf powers for NXSPE files - Andrei Savici (saviciat@ornl.gov)
;  05/2010: modify the way to handle the masking in SPE and NXSPE files      
;  05/2010: add user function option 
;  09/2010: IDL 8.0 compatible
;  11/2010: add dark angle trajectory
;  03/2011: add HFIR WAND instrument, allow user function to be executed before projection, 
;  04/2011: add script file execution
;  05/2011: Move "Option->Allow Extra Viewing Axis" to "Option->Viewing Axis->Allow Extra Viewing Axis".
;  11/2011: Allow WAND user to change Ei and detector offset angle; revamp the popup status menu
;  11/2011: Proper flat plate absorption calculation in diffuse scattering mode.
;  12/2011: Add cuts and slices in dm_plot for IDL 8.0 and after
;  06/2012: IDL 8.2 compatible
;  08/2012: Add PSUTXT keyword in save_slice
;  09/2012: Add more LaTex commands in dm_plot text input, and more font choices
;  11/2012: Add DCS Q&E-resolution convolution tool
;  12/2012: Add multiple slice output in saving volume data
;  02/2013: Add MACS, fully functional
;  03/2013: Add ftp file server to file list in the file selection window
;  04/2013: Add Temperature as extra viewing axis for MACS and WAND
;  05/2013: Add Calculator Tab
;  06/2013: Add magnetic field and a specified quantity to extra viewing axis
;  07/2013: Add Voronoi diagram for plotting slice with both steps empty
;  08/2013: Add MACS experiment sequence planner
;  01/2014: Add plotting MACS instrument alignment files in dcs_cryst_align; Add directory copy in dm_copyftp
;  03/2014: Add rotating to the first symmetric section option 
;  04/2014: Add saving saving cuts and slices as animation movie.
;  05/2014: Add J(Q,y) as one of the intensity type.
;  10/2014: Add weighted mean as the default binning method.
;  10/2014: Add INX data file energy transfer option in the popup menu.
;  12/2014: Add DCS jumpy file scan to the popup menu; Add masking dark angle.
;  04/2015: Add folding to both sides and expand to 360° options
;  08/2015: Add first symmetric section starting vector for symmetry rotation folding
;  09/2015: Add cutting along arbitrary direction in 2d slice plot window.
;  01/2016: Add He recoil simulation for MACS
;  02/2016: Add single crystal Q specification to DCS Q&E resolution convolution Tool 
;  02/2016: Add loading ascii data file in main dcs_mslice (already existed in data mslice).
;  03/2016: Add DCS white beam transmission calculation as a standalone file tool.
;  02/2017: Allow three sets of empty can subtraction.
;  03/2017: Add Temperature and magnetic field as extra axes to DCS file type
;  03/2017: Add histogram data mode for MACS
;  06/2017: Add background subtraction for general data slicer
;  07/2017: Add showing transmission as intensity option
;  04/2018: Add qoffset info in axis titles
;  05/2018: Add more general energy scan vanadium file for MACS detector efficiency
;  09/2018: Add ghoston viewer
;  12/2018: Add event histogram mode for MACS
;  04/2019: Add expanding single crystal projection of powder data in popup menu
;  09/2019: Add detector dependent self-shielding factor
;  12/2019: Add L0&L1 calculation for absorption correction
;  04/2020: Use monitor information for the weighted mean binning
;  02/2021: Add 4d binning and output tab
;  11/2021: Add statisitical estimate for zero inensity error bar calculation.
;  06/2022: Add polarization correction capability for DCS data.
;  06/2022: Add out-of-plane absorption correction calculation.
;  08/2022: Add overplotting to other mslice plot option.
;  04/2023: Add raw intensity option for MACS event mode data.
;  06/2025: Use davehttprequest to restore data repository access
;  
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  August, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;files need to be included
@dm_calc_projection
@dm_choose_cutdata
@dm_choose_slicedata
@dm_choose_volumedata
@dm_choose_4ddata
@dm_choose_trajdata
@dm_det_spurion
@dm_filemenu
@dm_filetools
@dm_macs_spurion
@dm_load_parameter
@dm_mask_detbanks
@dm_plot_cut
@dm_plot_slice
@dm_plot_volume
@dm_plot_trajectory
@dm_save_parameter
@dm_to_fit
@dcs_plotdetmask    ;for plotting masks
@dcs_cryst_align
@dm_macs_sequencer  ;for removing comments in strings

;kill the object
pro dcs_mslice_Exit,tlb
    widget_control,tlb,get_uvalue=self
    if obj_valid(self) then obj_destroy,self
end

;widget event handler
pro dcs_mslice_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;this function returns [li, la] of neutron mean free path in units of cm
;input parameters:
;  formula:   string  describing a chemical formula, elements need to be separated by a space, isotope number goes before the element name: eg "Yb Na O2" or "U Ni4 11B"
;  n_form:    number of chemical formula per unit cell, default is 1
;  rho:       density in unit of g/cm^3
;  lattparm:  [a,b,c,alpha,beta,gamma], optional for calculating rho
;  incohxs:   if set, use incoherent cross section instead of total scattering cross section to calculate L0
function dm_calc_lila,formula,nform=nform,rho=rho,lattparm=lattparm,incohxs=incohxs
    restore,filename=!DAVE_AUXILIARY_DIR+'xsecsdb.dat'
    xsecs.atsym = strtrim(xsecs.atsym,2)
    element = strsplit(strtrim(formula,2),' ',/extract)
    n_elem  = fltarr(n_elements(element))+1.0
    mass    = dblarr(n_elements(element))
    cs_a    = mass
    cs_i    = mass
    for i=0,n_elements(element)-1 do begin
        tmp = stregex(element[i],'^([0-9]*[A-Za-z]+)([0-9]*)',/extract,/sub)
        element[i] = tmp[1]
        if ~stregex(element[i],'^[0-9]+',/boolean) then element[i] = 'nat-'+element[i]
        if strlen(tmp[2]) ne 0 then n_elem[i] = dm_to_number(tmp[2])
        ind = where(strmatch(xsecs.atsym,element[i],/fold_case),count)
        if count eq 0 then message,level=0,'Element '+element[i]+' is not found in the cross section database https://www.ncnr.nist.gov/resources/n-lengths/list.html.'
        mass[i] = dm_to_number(xsecs.atwt[ind[0]],/double)
        cs_a[i] = dm_to_number(xsecs.xsabs[ind[0]],/double)
        if keyword_set(incohxs) then cs_i[i] = dm_to_number(xsecs.xsinc[ind[0]],/double) else cs_i[i] = dm_to_number(xsecs.xssct[ind[0]],/double)
    endfor
    atm = 1.660539  ;atomic mass unit x10^-27 kg
    mu_rho_a = total(n_elem*cs_a,/nan)/atm/total(n_elem*mass)
    mu_rho_i = total(n_elem*cs_i,/nan)/atm/total(n_elem*mass)
    if n_elements(rho) eq 0 then begin
       if n_elements(lattparm) eq 6 then begin
          if n_elements(nform) eq 0 then nform = 1.0
          cosabg = cos(lattparm[3:5]*!ddtor)
          vol = lattparm[0]*lattparm[1]*lattparm[2]*sqrt((1d)+(2d)*cosabg[0]*cosabg[1]*cosabg[2]-total(cosabg[0:2]^2))
          rho = nform*total(n_elem*mass)*atm/vol
       endif
    endif
    if n_elements(rho) eq 0 then message,level=0,'Please enter density keyword rho.'
    li = 1./mu_rho_i/rho
    la = 1./mu_rho_a/rho
    return,float([li,la])
end

;returns 1 if the step is too small, and one doesn't want to continue
function dm_check_step,data,step,label=label,nmax=nmax,group_leader=group_leader
    if step eq 0 then return,1
    if n_elements(nmax) eq 0 then nmax = 1e3
    xmin = min(data,max=xmax)
    if (xmax-xmin)/abs(step) gt nmax then begin
       if n_elements(label) eq 0 then label = 'data'
       ok = dialog_message('The step '+dm_to_string(step)+' for '+label+' range of ['+dm_to_string([xmin,xmax],separator=', ')+'] might be too small, do you want to continue?',/question,/default_no,$
            dialog_parent=group_leader,/center,title='Please confirm')
       if strupcase(ok) eq 'NO' then return,1
    endif
    return,0
end

;returns tooltip keyword pair or no_copy=0 for older idl version
function dm_tooltip,tooltip
    if dm_to_number(!version.release) ge 5.6 then return, {tooltip:tooltip} else return, {no_copy:0}
end

;absorption correction
;varible:
;   type: -1-none 0-cylinder 1-annulus 2-sphere 3-flat plate
pro dcs_mslice::absorption,type
    old = -1
    if type lt -1 or type gt 3 then return
    default = [0,1,1,!values.f_nan,!values.f_nan,!values.f_nan,!values.f_nan,!values.f_nan]
    if (self.samp_typ eq 2) or (self.instrname eq 'macs') then default[1]=0 ;no need to interpolate for diffuse scattering single E or macs data
    if ptr_valid(self.absp_info) then begin
       old   = (*(self.absp_info))[0]
       ninfo = n_elements(*(self.absp_info))
       default[0:(ninfo-2)] = (*(self.absp_info))[1:(ninfo-1)]
    endif
    if finite(default[3],/nan) and (strlen(self.absp_l0l1[3]) ne 0) then default[3:4] = dm_to_number(self.absp_l0l1[3:4])
    case type of
          0 :   begin   ;cylinder
                names  = ['L0(cm):','L1(cm):','R(cm):','Sampling Points:','Interpolate:']
                title  = 'Cylindrical geometry'
                end
          1 :   begin   ;annulus
                names  = ['L0(cm):','L1(cm):','Rout(cm):','Rin(cm):','Sampling Points:','Interpolate:']
                title  = 'Annular geometry'
                end
          2 :   begin   ;sphere
                names  = ['L0(cm):','L1(cm):','R(cm):','Sampling Points:','Interpolate:']
                title  = 'Spherical geometry'
                end
          3 :   begin   ;flat plate
                names  = ['L0(cm):','L1(cm):','Width(cm):','Thickness(cm):','Angle('+string('B0'XB)+'):','Sampling Points:','Interpolate:']
                title  = 'Flat plate geometry'
                end
        else:           ;none
    endcase
    nitem = n_elements(names)
    if nitem ne 0 then begin
       nptst = *self.absp_nptst[type]
       mesg = ['L0 is the energy independent mean free path;',$
               'L1 is the energy dependent mean free path for',$
               '2200m/s or 25.3 meV neutron.',$
               'The effective mean free path is:',$
               'L=1/{1/L0+1/(L1*sqrt[E(meV)/25.3])}' ]
       if type eq 3 then mesg = [mesg,'','The angle is from ki to the normal of the plate'+([(['.',' for the first A3 angle.'])[self.instrname eq 'macs'],$
               ' for the first file.'])[self.samp_typ ge 2],'Counterclockwise rotation is positive.']
       if self.psi_0only then begin ;always equatorial scattering
          default = [default[3:nitem],default[0:1]]
          is_droplist = intarr(nitem) & is_droplist[(nitem-2):(nitem-1)] = 1
       endif else begin
          names   = [names,'Equator Only:']
          default = [default[3:nitem],default[0:2]]
          nitem   = nitem+1
          is_droplist = intarr(nitem) & is_droplist[(nitem-3):(nitem-1)] = 1
          mesg = [mesg,'','If out of plane scattering direction needs to be considered, set equator only to No.']
          if self.instrname ne 'dcs' then begin
             if ptr_valid(self.detPosPtr) then begin
                tmppsi = (*self.detPosPtr).psi
                if n_elements(uniq(tmppsi,sort(tmppsi))) ge 5 then $  ;detector angle will still be interpolated if det_psi contains less than 5 unique values
                   mesg = [mesg,'The detector angles will not be interpolated, resulting in slower calculation.']
             endif else $
                mesg = [mesg,'The detector angles might not be interpolated, resulting in slower calculation.']
          endif 
       endelse
       info = dm_dialog_input(names,default=default,/float,info=mesg,is_droplist=is_droplist,droplist_content=[ptr_new(nptst),ptr_new(['No','Yes']),ptr_new(['No','Yes'])],$
                /return_number,title=title,xsize=140,dialog_parent=self.tlb,cancel=cancel)
       if keyword_set(cancel) then return         
       if total(finite(info)) eq nitem then begin
          ptr_free,self.absp_info
          if self.psi_0only then begin
             info = [info,1]
             nitem = nitem+1
          endif
          self.absp_info = ptr_new([type,info[(nitem-3):(nitem-1)],info[0:(nitem-4)]])
          if self.absp_l1factor ne 1 then ok = dialog_message('Absorption correction incident beam path factor is '+dm_to_string(self.absp_l1factor)+'. Set it to 1 if no adjustment is needed.',dialog_parent=self.tlb,/center)
       endif else begin
          type = -1
          if old ne -1 then ok = dialog_message((keyword_set(cancel)?(''):('Invalid inputs. '))+'Absorption correction is disabled.',dialog_parent=self.tlb,/error,/center)
       endelse
    endif
    self->my_menu_toggle,'absp'+['None','Cylinder','Annulus','Sphere','Flat'],type+1
    dm_set_button,widget_info(self.tlb,find_by_uname='abspCorr'),(type ne -1)
    self->my_widget_control,'abspInt',sensitive=(type ne -1)
    if type eq -1 then ptr_free,self.absp_info
    if ((type ne -1) or (old ne -1)) and ptr_valid(self.projStrPtr) and ptr_valid(self.dataStrPtr) then self->set_display_button,/clearprojection ;need to recalculate projections
end

;adjust the viewing axis label according to hkl value
pro dcs_mslice::adjust_viewlabel,uid
    for i=0,n_elements(uid)-1 do begin
        ind = where(([[self.view_u1],[self.view_u2],[self.view_u3]])[*,uid[i]] ne 0,cnt)
        if cnt ne 0 then self->script,'set_parameter',name=(['viewU1lab','viewU2lab','viewU3lab'])[uid[i]],value=(['H','K','L'])[min(ind)]
    endfor
end

;append data to the existing data for self.samp_typ=-1
pro dcs_mslice::append_data,data=data,no_copy=no_copy
    if (self.samp_typ ne -1) or (n_elements(data) eq 0) then return  
    o_item = n_elements((*self.dataPtr).label)
    n_item = n_elements(data[0,*])
    if n_item ne o_item then begin
       help,(*self.dataPtr).data,output=tmp0 & tmp0 = strtrim(tmp0,2) & pos = strpos(tmp0,' ') & tmp0='EXISTING DATA '+strmid(tmp0,pos+1)
       help,data,output=tmp1
       ok = dialog_message(['The data are incompatible with the existing data.',tmp0,'NEW '+tmp1],dialog_parent=self.tlb,/center,/error)
       return
    endif
    if keyword_set(no_copy) then new_data = transpose(temporary(data)) else new_data=transpose(data)
    if ptr_valid(self.dataStrPtr) then new_data = transpose([[transpose((*self.dataStrPtr).data)],[new_data]]) $
    else new_data = transpose([[transpose((*self.dataPtr).data)],[new_data]])
    self->reset_data,data=new_data,/no_copy
end

pro dcs_mslice::calculator_insert,command,clear=clear,totop=totop
    self->my_widget_control,'calcInput',self.dispBas,get_value=calcstr
    if keyword_set(clear) then begin
       calcstr = ''
    endif else begin
       if n_elements(command) eq 0 then return
       while (strlen(strtrim(calcstr[n_elements(calcstr)-1],2)) eq 0) and (n_elements(calcstr) gt 1) do calcstr = calcstr[0:(n_elements(calcstr)-2)]  ;remove trailing empty strings
       if (n_elements(calcstr) eq 1) and (strlen(strtrim(calcstr[0],2)) eq 0) then calcstr = command else calcstr = [calcstr,'',command]
    endelse
    self->my_widget_control,'calcInput',self.dispBas,set_value=calcstr
    if n_elements(calcstr) ne 0 then sensitive = (max(strlen(strtrim(calcstr,2))) gt 0) else sensitive=0
    self->my_widget_control,['calcCal','clearCal'],self.dispBas,sensitive=sensitive
    if keyword_set(totop) then self->my_widget_control,'dispTab',self.dispBas,set_tab_current=([3,4,5])[self.disp_flag[2]+self.disp_flag[3]]
end

;change show powder line option
pro dcs_mslice::change_powderline,option,menuonly=menuonly
    if n_elements(option) eq 0 then return
    if (self.powd_typ ne 0) and (self.powd_typ ne option) and (option ne 0)then begin
       self.powd_latta = !values.f_nan ;reset saved value
       self.powd_bcc   = 0b
    endif 
    self.powd_typ = 0>(option)<4
    self->my_menu_toggle,'powder'+['None','Al','Cu','St','Sp'],self.powd_typ
    dm_set_button,widget_info(self.tlb,find_by_uname='powdershowall'),self.powd_showall,onstring='Show Closed Powder Line',offstring='Show Actual Powder Line',sensitive=(self.powd_typ ne 0)
    dm_set_button,widget_info(self.tlb,find_by_uname='showPowder'),(self.powd_typ ne 0)
    if keyword_set(menuonly) then return
    default = dm_to_string([0,4.05,3.6148,3.594,!values.f_nan]) ; _, Al, Cu, SS, specify
    if ptr_valid(self.powd_qvalue) then default[4]=dm_to_string(*self.powd_qvalue,separator=', ')
    default = default[self.powd_typ]
    ptr_free,self.powd_qvalue,self.powd_line    
    if self.powd_typ gt 0 and self.powd_typ lt 4 then begin ;ask for the lattice parameter
       if (self.powd_typ eq 3) and self.powd_bcc then begin
          str_str = 'BCC'
          default = '2.87'
       endif else $
          str_str = 'FCC'
       if finite(self.powd_latta) then begin
          if self.powd_latta ne dm_to_number(default) then begin
             info = 'Try a = '+default+' '+string('c5'XB)+' for '+str_str+([' aluminum',' copper',' stainless steel'])[self.powd_typ-1]+'.'
             default = dm_to_string(self.powd_latta)
          endif
       endif 
       if n_elements(info) eq 0 then info = 'Press OK to accept the default '+str_str+' lattice parameter. 
       name   = 'a ('+string('c5'XB)+'):'
       title  = (['Al','Cu','Stainless Steel'])[self.powd_typ-1]+' Lattice Parameter'
       latt_a = !values.f_nan
       if self.powd_typ eq 3 then begin
          info    = [info,'Try a = '+(['2.87 ','3.594 '])[self.powd_bcc]+string('c5'XB)+' for '+(['BCC','FCC'])[self.powd_bcc]+' stainless steel.']
          default = [default,dm_to_string(self.powd_bcc,/int)]
          name    = [name,'structure:']
       endif  
       while(finite(latt_a,/nan)) do begin
          if self.powd_typ ne 3 then begin
             latt_a = dm_dialog_input(name,xsize=80,dialog_parent=self.tlb,info=info,default=default,/float,title=title) 
          endif else begin
             latt_a = dm_dialog_input(name,xsize=80,dialog_parent=self.tlb,info=info,default=default,/float,title=title,droplist_content=ptr_new(['FCC','BCC']),is_droplist=[0,1],/return_number) 
             is_bcc = (latt_a[1] eq 1)    ;BCC structure
             latt_a = latt_a[0]
             self.powd_bcc = is_bcc
          endelse
       endwhile
       self.powd_latta = latt_a
       if keyword_set(is_bcc) then $  ;BCC structure, enough for 2518 meV for BCC SS  
          qsqrt = sqrt([2.,4,6,8,10,12,14,16,18,20,22,24,26,30,32,34,36,38,40,42,44,46,48,50,52,54,56,58,62,64,66,68,70,$
                   72,74,76,78,80,82,84,86,88,90,94,96,98,100,102,104,106,108,110,114,116,118,120,122,126,128,130,$
                   132,134,136,138,140,142,144,146,148,150,152,154,158,160,162,164,166,168,170,172,174,176,178,180,$
                   182,184,186,190,192,194,196,198,200,202,204,206,208,210,212,214,216,218,222,224,226,228,230,232,$
                   234,236,238,242,244,246,248,250,254,256,258,260,262,264,266,268,270,272,274,276,278,280,282,286,$
                   288,290,292,294,296,298,300,302,304,306,308,310,312,314,318,320,322,324,326,328,330,332,334,336,$
                   338,340,342,344,346,350,352,354,356,358,360,362,364,366,370,372,374,376,378,382,384,386,388,390,$
                   392,394,396,398,400,402,404,406,408,410,414,416,418,420,422,424,426,428,430,432,434,436,438,440,$
                   442,446,450,452,454,456,458,460,462,464,466,468,470,472,474,478,480,482,484,486,488,490,492,494,$
                   498,500,502,504,506,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,542,544,546,548,$
                   550,552,554,556,558,560,562,564,566,568,570,574,576,578,580,582,584,586,588,590,592,594,596,598,$
                   600,602,606,608,610,612,614,616,618,620,622,626,628,630,632,634,638,640,642,644,646,648,650,652,$
                   654,656,658,660,662,664,666,670,672,674,676,678,680,682,684,686,688,690,692,694,696,698,702,704,$
                   706,708,710,712,714,716,718,720,722,724,726,728,730,734,736,738,740,742,744,746,748,750,754,756,$
                   758,760,762,766,768,770,772,774,776,778,780,782,784,786,788,790,792,794,798,800,802,804,806,808,$
                   810,812,814,816,818,820,822,824,826,830,832,834,836,838,840,842,844,846,848,850,852,854,856,858,$
                   862,864,866,868,870,872,874,876,878,882,884,886,888,890,894,896,898,900,902,904,906,908,910,912,$
                   914,916,918,920,922,926,928,930,932,934,936,938,940,942,944,946,948,950,952,954,958,962,964,966,$
                   968,970,972,974,976,978,980,982,984,986,990,992,994,996,998,1000,1002,1004,1006,1010,1012,1014]) $      
       else $  ;FCC structure, enough for 2005 meV for FCC Al 
          qsqrt = sqrt([3.,4,8,11,12,16,19,20,24,27,32,35,36,40,43,44,48,51,52,56,59,64,67,68,72,75,76,80,83,84,88,91,$
                   96,99,100,104,107,108,115,116,120,123,128,131,132,136,139,140,144,147,148,152,155,160,163,164,$
                   168,171,172,176,179,180,184,187,192,195,196,200,203,204,208,211,212,216,219,224,227,228,232,235,$
                   236,243,244,248,251,256,259,260,264,267,268,272,275,276,280,283,288,291,292,296,299,300,304,307,$
                   308,312,315,320,323,324,328,331,332,336,339,340,344,347,352,355,356,360,363,364,371,372,376,379,$
                   384,387,388,392,395,396,400,403,404,408,411,416,419,420,424,427,428,432,435,436,440,443,451,452,$
                   456,459,460,464,467,468,472,475,480,483,484,488,491,492,499,500,504,507,512,515,516,520,523,524,$
                   528,531,532,536,539,544,547,548,552,555,556,560,563,564,568,571,576,579,580,584,587,588,592,595,$
                   596,600,603,608,611,612,616,619,620,627,628,632,635,640,643,644,648,651,652,656,659,660,664,667,$
                   672,675,676,680,683,684,688,691,692,696,699,704,707,708,712,715,716,720,723,724,728,731,736,739,$
                   740,744,747,748,755,756,760,763,768,771,772,776,779,780,784,787,788,792,795,800,803,804,808,811,$
                   812,816,819,820,824,827,832,835,836,840,843,844,848,851,852,856,859,864,867,868,872,875,876,883,$
                   884,888,891,896,899,900,904,907,908,912,915,916,920,923,928,931,932,936,939,940,944,947,948,952,$
                   955,963,964,968,971,972,976,979,980,984,987,992,995,996,1000,1003,1004,1011,1012,1016,1019,1024,$
                   1027,1028,1032,1035,1036,1040,1043,1044,1048,1051,1056,1059,1060,1064,1067,1068,1072,1075,1076,$
                   1080,1083,1088,1091,1092,1096,1099,1100,1104,1107,1108,1112,1115,1120,1123,1124,1128,1131,1132,$
                   1139,1140,1144,1147,1152,1155,1156,1160,1163,1164,1168,1171,1172,1176,1179,1184,1187,1188,1192,$
                   1195,1196,1200,1203,1204,1208,1211,1216,1219,1220,1224,1227,1228,1232,1235,1236,1240,1243,1248,$
                   1251,1252,1256,1259,1260,1267,1268,1272,1275,1280,1283,1284,1288,1291,1292,1296,1299,1300,1304,$
                   1307,1312,1315,1316,1320,1323,1324,1328,1331,1332,1336,1339,1344,1347,1348,1352,1355,1356,1360,$
                   1363,1364,1368,1371,1376,1379,1380,1384,1387,1388,1395,1396,1400,1403,1408,1411,1412,1416,1419,$
                   1420,1424,1427,1428,1432,1435,1440,1443,1444,1448,1451,1452,1456,1459,1460,1464,1467,1475,1476,$
                   1480,1483,1484,1488,1491,1492,1496,1499,1504,1507,1508,1512,1515,1516,1523,1524,1528,1531,1536,$
                   1539,1540,1544,1547,1548,1552,1555,1556,1560,1563,1568,1571,1572,1576,1579,1580,1584,1587,1588,$
                   1592,1595,1600,1603,1604,1608])
    endif        
    case self.powd_typ of
          1: qvalue = qsqrt*2*!pi/latt_a     ;Al a=4.05    FCC
          2: qvalue = qsqrt*2*!pi/latt_a     ;Cu a=3.6148  FCC
          3: qvalue = qsqrt*2*!pi/latt_a     ;SS a=3.594   FCC, a=2.87 BCC
          4: begin
             info   = 'Numbers should be separated by a space or comma.'
             qvalue = dm_dialog_input('Q ('+string('c5'XB)+'^-1):',xsize=180,dialog_parent=self.tlb,info=info,default=default)
             if (~ finite(dm_to_number(qvalue))) then begin
                if (strlen(default) ne 0) or (strlen(qvalue) ne 0) then $
                   ok = dialog_message((['Invalid inputs.','Empty input.'])[strlen(qvalue) eq 0]+' No powder line is specified.',dialog_parent=self.tlb,/error,/center)
                self.powd_typ = 0
                self->my_menu_toggle,'powder'+['None','Al','Cu','St','Sp'],self.powd_typ
                dm_set_button,widget_info(self.tlb,find_by_uname='powdershowall'),self.powd_showall,onstring='Show Closed Powder Line',offstring='Show Actual Powder Line',sensitive=(self.powd_typ ne 0)
             endif else begin
                qvalue = dm_to_number(strsplit(qvalue,', '+string(9b),/extract))   
                qvalue = qvalue[sort(qvalue)]
             endelse
             end 
          else:
    endcase
    if n_elements(qvalue) ne 0 and self.powd_typ gt 0 then self.powd_qvalue=ptr_new(qvalue)
    if ptr_valid(self.projStrPtr) then self->set_display_button,/clearprojection ;need to recalculate projections
end

;keyword:
;   init: if not set, free some data pointers
pro dcs_mslice::change_samptyp,new_typ,dcsmaskwarning=dcsmaskwarning,init=init,pdt_reset=pdt_reset
    old_typ = self.samp_typ
    self.samp_typ = new_typ
    self->my_widget_control,['bgtolBut','addtolBut','empttolMenu'],sensitive=((self.samp_typ ge 2) or (self.instrname eq 'macs'))
    sensitive = ((self.samp_typ ge 2) or ((self.instrname eq 'macs') and (self.samp_typ ne 0)))
    self->my_widget_control,'darkTraj',sensitive=sensitive
    if ~sensitive then begin
       self.mask_darkangle = 0b
       self.ftypecheckmask[0] = (((self.instrname eq 'macs') and (self.macs_ptaionly or (self.macshistomode eq 2) or ptr_valid(self.mask_ar) or (ptr_valid(self.mask) and (self.masktype eq 1)))) or self.mask_darkangle or self.ftypecheckmask[1])
    endif
    dm_set_button,widget_info(self.tlb,find_by_uname='mkdarkBut'),self.mask_darkangle,onstring='Mask Dark Angle...',offstring='Not Mask Dark Angle...',sensitive=sensitive
    self->my_widget_control,'darkCorr',sensitive=((self.samp_typ ge 2) and (self.instrname eq 'dcs'))
    self->my_widget_control,['vaxis'+['Ortho','Arbi'],'vfoldMenu'],sensitive=(self.samp_typ ne 0)
    self->my_widget_control,'vaxisMenu',sensitive=((self.samp_typ ne 0) or (self.instrname eq 'macs') or (self.instrname eq 'dcs'))
    if self.instrname eq 'dcs' then begin
       self->my_widget_control,'extraVaxis',sensitive=(self.samp_typ gt 0)
       if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=(total(self.extravaxis_yn[(self.samp_typ eq 0):*]) ne 0) 
    endif
    self->my_widget_control,'bgtcBut',sensitive=((self.samp_typ ne 2) and (self.instrname ne 'wand') and (self.instrname ne 'macs'))
    if old_typ eq new_typ then return
    self->my_menu_toggle,['powdTyp','crystTyp1','crystTyp2','crystTyp3'],self.samp_typ,base=self.fileBas
    if self.idl_version ge 5.6 then self->my_widget_control,'crystTyp',set_button=(self.samp_typ ge 1)  
    sptypes = ['Powder','Single Crystal','Diffuse Scattering Single E','Diffuse Scattering Multiple E']
    self->my_widget_control,'sampTyp',set_value=sptypes[self.samp_typ]
    if keyword_set(init) then return
    if (self.samp_typ ge 2) or (old_typ ge 2) then begin    ;if changed from/to diffuse scattering type
       ;clear data
       ptr_free,self.kfactor,self.dataStrPtr,self.powd_line
       self->my_widget_control,['addBut','clearDat'],sensitive=0          ;disallow adding data and clear data
       self->my_widget_control,'titlLab',set_value='' & self.titl_lab=''  ;clear title label
       self->my_widget_control,'legdLab',set_value='' & self.legd_lab=''  ;clear legend label
       ;clear background
       ptr_free,self.bgdata,self.bgratePtr,self.bgtchnPtr,self.bgdetsPtr
       self->reset_menu,/backgroundmenuonly
    endif
    if (self.instrname eq 'macs') and (self.samp_typ eq 1) and (self.macsintn_typ ne 0) then begin  ;switch to SPEC detector automatically in crystal mode
       self.macsintn_typ = 0
       self->my_menu_toggle,'macsint'+['SPEC','DIFF','SPDF'],self.macsintn_typ
       self->my_widget_control,'macsintMenu',set_value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')'
       ok = dialog_message('Detectors have been switched to SPEC.',dialog_parent=self.tlb,/center,title='Please note:')  
    endif
    if (self.instrname eq 'dcs') and (total(self.dcsdetmask) ne 0) and keyword_set(dcsmaskwarning) then begin
       mask = self.dcsdetmask[0:3]
       if self.dcsdetmask[4] then mask[[1,3]] = 1b
       mask = (['negative angle','lower bank','central bank','upper bank'])[where(mask)]
       if n_elements(mask) gt 2 then begin
          mesg = ''
          for i=0,n_elements(mask)-2 do mesg = mesg+mask[i]+', '
          mesg = mesg+'and '+mask[n_elements(mask)-1]+' detectors are masked.'
       endif else if n_elements(mask) eq 2 then begin
          mesg = mask[0]+' and '+mask[1]+' detectors are masked.'
       endif else mesg = mask[0]+' detectors are masked.'
       strput,mesg,strupcase(strmid(mesg,0,1)),0
       ok = dialog_message(mesg,dialog_parent=self.tlb,/center)  
    endif
    ptr_free,self.projStrPtr
    self->reset_file
    self->reset_pdt
    pdt_reset = 1b
end
 
;search for jumpy files in a directory
pro dcs_mslice::check_jumpy
    dir   = dm_choose_file(/dir,path=self.dirs[0],title='Please select a directory to scan for jumpy files:',dialog_parent=self.tlb)
    if strlen(dir) eq 0 then return
    known = ['20060209_002','20060209_003','20090618_009','20090624_007','20090624_010','20090805_011','20090805_012','20090810_002','20090827_022','20100228_009','20100303_008','20100809_020',$
             '20100926_002','20101008_011','20110130_006','20110224_034','20110301_025','20110301_147','20110304_001','20120427_003','20120427_022','20120427_025','20120826_007','20120922_008',$
             '20121025_082','20130606_019','20140805_021','20140805_022','20140806_011','20140806_079','20140806_084','20140808_051','20140809_080','20140810_038','20140810_122','20140816_007',$
             '20150421_011','20150421_065','20150421_069','20150423_003','20150501_014','20150501_020','20150502_014','20150503_010','20150503_014','20150503_024','20150504_015','20150607_020',$
             '20150723_004','20150814_018','20150925_028','20151015_017','20151018_009','20151116_019','20151116_035','20151116_069','20151118_204','20151126_003','20160303_004','20160303_017',$
             '20160304_022','20160305_022','20160427_008','20160514_007','20160517_003','20160712_018','20170414_005','20170414_010','20170414_028','20170414_029','20170415_010','20170415_014',$
             '20170416_052','20170417_001','20170417_002','20170417_004','20170417_008','20170417_103','20170418_026','20170418_088','20170420_007','20170420_039','20170426_122','20170503_100',$
             '20170706_030','20170806_016','20170905_005','20180621_016','20180707_008','20180707_010','20180707_031','20180720_017','20180721_026','20181106_003','20181215_015','20181218_016',$
             '20181221_016','20190326_013','20190328_006','20190408_039']+'.dcs.gz'
    flast =  '20190603_007'
    mesg  =  obj_new('dm_progress',group_leader=self.tlb)
    mesg->update,message='reading '+dir,title=dir
    files = file_search(dir,'*.dcs.gz',count=count)
    njump = 0
    dPtr  = dm_define_pointer() & omithisto=0 & printinfo=0 & filetype='binary'
    this  = dm_locate_datastrptr(dPtr)
    for i=0L,count-1L do begin
        mesg->update,message='reading '+dir+': '+dm_to_string(i+1)+'/'+dm_to_string(count)
        tmp = file_basename(files[i])
        if tmp gt flast then begin
           dcs_readdatafile,files[i],omithisto,printinfo,filetype,davePtr=dPtr
           qty = (*(*this).commonStr.histPtr).qty
           if n_elements(qty) gt 2 then begin
              qty  = total(qty,1)
              qty_max = max(qty) & qty_mean = mean(qty)
              m_ind   = where(qty ge 10*qty_mean,count1)
              if count1 ne 0 then qty[m_ind] = qty_max
              tmp = a_correlate(qty,[31,248])
              if min(tmp) gt 0.3 then files[njump++] = files[i]
           endif
           tmp = temporary(qty)  ;remove qty
        endif else begin
           tmp = where(known eq tmp,count0)
           if count0 ne 0 then files[njump++] = files[i]
        endelse
        if ~obj_valid(mesg) then break
    endfor
    if obj_valid(mesg) then obj_destroy,mesg
    if njump eq 0 then begin
       title = 'No jumpy files!'
       ok = dialog_message([dir+'  total: '+dm_to_string(count,/int)+' DCS files','',title],/info,title=title,dialog_parent=self.tlb,/center)
    endif else begin
       self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep
       self.current = obj_new('dm_plot',parentobj=self,background=(['black','white'])[self.pbgcol_bw],/isolatin1,xsize=(*self.pdim_list)[0,self.pdim_choice],notooltip=self.pnotooltip,$
                      ysize=(*self.pdim_list)[1,self.pdim_choice],/legdshowfill,legdfsize=([12,10])[njump gt 10],/legdshowoutline,xtit='Detector',ytit='Intensity',title=dir+'   jumpy files',/ylog)
       self->dm_add_plotwindow,type=-1,title=dir+' jumpy files'
       pcol = self.current->getcolor(/list)
       pcol = pcol[1:n_elements(pcol)-2] ;remove the user define color
       psym = self.current->getpsym(/list)
       psym = psym[0:4] & psym = [psym,'filled '+psym]
       yran = [1e20,0.]
       for i=0,njump-1 do begin
           dcs_readdatafile,files[i],omithisto,printinfo,filetype,davePtr=dPtr
           qty = total((*(*this).commonStr.histPtr).qty,1,/double)
           self.current->add_plot,indgen(n_elements(qty)),qty,psym=psym[i mod n_elements(psym)],color=pcol[i mod n_elements(pcol)],legend=file_basename(files[i])
           yran[0] = (yran[0])<(min(qty[where(qty gt 0)],max=ymax))
           yran[1] = (yran[1])>(ymax)
       endfor
       self.current->setproperty,yran=yran*[1/1.5,1.5],xran=[-10,923]
       self->keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep,/reset
    endelse
    heap_free,dPtr
end

;clear data
pro dcs_mslice::clear_data,infomsg=infomsg,clear_bg=clear_bg
    if ptr_valid(self.dataStrPtr) then begin
       reload_data = 1b
       cid = widget_info(self.tlb,find_by_uname='clearDat') 
       if cid ne 0 then self->event,{WIDGET_BUTTON,ID:cid,TOP:self.tlb,HANDLER:self.tlb,SELECT:1} $
       else ptr_free,self.dataStrPtr
    endif
    if keyword_set(clear_bg) and (ptr_valid(self.bgdata[0]) or ptr_valid(self.bgratePtr) or ptr_valid(self.bgtchnPtr) or ptr_valid(self.bgdetsPtr)) then begin
       reload_bg = 1b
       cid = widget_info(self.tlb,find_by_uname='bgclBut')
       if cid ne 0 then self->event,{WIDGET_BUTTON,ID:cid,TOP:self.tlb,HANDLER:self.tlb,SELECT:1}
    endif
    if keyword_set(reload_data) or keyword_set(reload_bg) then begin
       mesg = 'Please reload '+(['','','empty can ','data and empty can '])[keyword_set(reload_data)+keyword_set(reload_bg)*2]+'data.'
       if n_elements(infomsg) eq 0 then ok = dialog_message(dialog_parent=self.tlb,mesg,/info,/center) else infomsg = [infomsg,mesg]
    endif 
end

;clear mask pointer and set the menu items accordingly
pro dcs_mslice::clear_mask,message=message,a3=a3,bga3=bga3,keep_ar=keep_ar,keep_mask=keep_mask,ptaionly=ptaionly,mask_darkangle=mask_darkangle
    if keyword_set(bga3) then begin
       ptr_free,self.bga3mask
       dm_set_button,widget_info(self.tlb,find_by_uname='mkbga3But'),0
       self->my_widget_control,'mkclbga3But',sensitive=0
    endif else if keyword_set(a3) then begin
       ptr_free,self.a3mask
       dm_set_button,widget_info(self.tlb,find_by_uname='mka3But'),0
       self->my_widget_control,'mkcla3But',sensitive=0
       if keyword_set(message) then ok = dialog_message('A3 mask has been cleared.',dialog_parent=self.tlb,/center)  
    endif else begin
       if ptr_valid(self.mask) or ptr_valid(self.mask_ar) then begin
          if ~keyword_set(keep_mask) then begin
             ptr_free,self.mask
             self.maskstr = ''
             if self.instrname eq 'dcs' then begin
                self.dcsdetmask[*] = 0
                self->dm_mask_detbanks
             endif
             if keyword_set(message) then ok = dialog_message('Detector number mask has been cleared.',dialog_parent=self.tlb,/center)  
          endif
          if ~keyword_set(keep_ar) then begin
             ptr_free,self.mask_ar
             if keyword_set(message) then ok = dialog_message('Detector angular range mask has been cleared.',dialog_parent=self.tlb,/center)  
          endif
       endif
       if self.instrname eq 'macs' then begin
          self.macs_ptaionly = keyword_set(ptaionly)
          dm_set_button,widget_info(self.tlb,find_by_uname='mkptaiBut'),self.macs_ptaionly,onstring='View PTAI Only',offstring='View All'
          self->my_widget_control,'ptaionly',self.fileBas,set_button=self.macs_ptaionly
       endif
       self.mask_darkangle = keyword_set(mask_darkangle)
       dm_set_button,widget_info(self.tlb,find_by_uname='mkstangl'),ptr_valid(self.mask_ar)
       dm_set_button,widget_info(self.tlb,find_by_uname='mkdarkBut'),self.mask_darkangle,onstring='Mask Dark Angle...',offstring='Not Mask Dark Angle...'
       self.ftypecheckmask[0] = (((self.instrname eq 'macs') and (self.macs_ptaionly or (self.macshistomode eq 2) or ptr_valid(self.mask_ar) or (ptr_valid(self.mask) and (self.masktype eq 1)))) or self.mask_darkangle or self.ftypecheckmask[1])
       self->my_widget_control,'mkclBut',sensitive=(((self.instrname eq 'macs') and self.macs_ptaionly) or ptr_valid(self.mask_ar) or ptr_valid(self.mask)) or self.mask_darkangle
       if ~ptr_valid(self.mask) then begin
          names = ['mkstdnum','mkpldnum','mkplangl','mkbpBut']
          for i=0,n_elements(names)-1 do dm_set_button,widget_info(self.tlb,find_by_uname=names[i]),0
       endif
    endelse
    self->my_widget_control,'mksvBut',sensitive=(ptr_valid(self.mask) or ptr_valid(self.a3mask) or ptr_valid(self.mask_ar) or ptr_valid(self.bga3mask))
    if ptr_valid(self.projStrPtr) then self->set_display_button,/clearprojection ;need to recalculate projections
end

;for general data slicer, masking data or simply restoring original data
pro dcs_mslice::data_mask
    ;mask signal data and weight
    if ptr_valid(self.dataStrPtr) then (*self.dataPtr) = (*self.dataStrPtr)
    if ptr_valid(self.weightPtr[1]) then (*self.weightPtr[0]) = (*self.weightPtr[1])
    if total(finite(self.mask_filler[0:1])) ne 0 then begin
       if finite(self.mask_filler[0]) then begin
          if finite(self.mask_filler[1]) then begin
             if self.mask_filler[0] gt self.mask_filler[1] then index = where(((*self.dataPtr).data[*,0] gt self.mask_filler[0]) or ((*self.dataPtr).data[*,0] lt self.mask_filler[1]),count) $
             else index = where(((*self.dataPtr).data[*,0] gt self.mask_filler[0]) and ((*self.dataPtr).data[*,0] lt self.mask_filler[1]),count)
          endif else index = where((*self.dataPtr).data[*,0] gt self.mask_filler[0],count)
       endif else index = where((*self.dataPtr).data[*,0] lt self.mask_filler[1],count)
       if count ne 0 then begin
          if ~ptr_valid(self.dataStrPtr) then self.dataStrPtr = ptr_new((*self.dataPtr))
          (*self.dataPtr) = {data:(*self.dataStrPtr).data[index,*],label:(*self.dataStrPtr).label,unit:(*self.dataStrPtr).unit,title:(*self.dataStrPtr).title,$
                             legend:(*self.dataStrPtr).legend,is_uniq:(*self.dataStrPtr).is_uniq,instrument:(*self.dataStrPtr).instrument,axis_info:(*self.dataStrPtr).axis_info}
          if ptr_valid(self.weightPtr[0]) then begin
             if ~ptr_valid(self.weightPtr[1]) then self.weightPtr[1] = ptr_new((*self.weightPtr[0]))
             (*self.weightPtr[0]) = (*self.weightPtr[0])[index]
          endif
       endif else begin
          ok = dialog_message('All data have been masked by the given value. Mask is disabled.',dialog_parent=self.tlb,/center)
          self.mask_filler[0:1] = !values.f_nan
       endelse
    endif 
    if (total(finite(self.mask_filler[0:1])) eq 0) and ptr_valid(self.dataStrPtr) then ptr_free,self.dataStrPtr
    ;mask background
    if ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]) then begin
       data = *self.bgdata[ptr_valid(self.bgdata[1])]
       if ptr_valid(self.bgweightptr[0]) or ptr_valid(self.bgweightptr[1]) then bgweight = *self.bgweightptr[ptr_valid(self.bgweightptr[1])]
       ptr_free,self.bgdata[0],self.bgweightptr[0]
       if (total(finite(self.mask_filler[0:1])) ne 0) and self.mask_filler[2] then begin
          if ~ptr_valid(self.bgdata[1]) then self.bgdata[1] = ptr_new(data)
          if finite(self.mask_filler[0]) then begin
             if finite(self.mask_filler[1]) then begin
                if self.mask_filler[0] gt self.mask_filler[1] then index = where((data[*,0] gt self.mask_filler[0]) or (data[*,0] lt self.mask_filler[1]),count) $
                else index = where((data[*,0] gt self.mask_filler[0]) and (data[*,0] lt self.mask_filler[1]),count)
             endif else index = where(data[*,0] gt self.mask_filler[0],count)
          endif else index = where(data[*,0] lt self.mask_filler[1],count)
          if count ne 0 then begin
             self.bgdata[0] = ptr_new(data[index,*])
             if n_elements(bgweight) ne 0 then self.bgweightptr[0] = ptr_new(bgweight[index])
          endif else ok = dialog_message('All background data have been masked.',dialog_parent=self.tlb,/center)
       endif else begin
          self.bgdata[0] = ptr_new(data,/no_copy)
          if n_elements(bgweight) ne 0 then self.bgweightptr[0] = ptr_new(bgweight,/no_copy)
          if ptr_valid(self.bgdata[1]) then ptr_free,self.bgdata[1]
          if ptr_valid(self.bgweightptr[1]) then ptr_free,self.bgweightptr[1]
       endelse
    endif
    self->my_widget_control,'maskFiller',set_button=(total(finite(self.mask_filler[0:1])) ne 0)
end

;for gerneal data slicer, subtract background
pro dcs_mslice::data_subbg
    self->data_mask ;mask data or restore original data
    if ~ptr_valid(self.dataPtr) or ~ptr_valid(self.bgdata[0]) then return
    if ~ptr_valid(self.dataStrPtr) then self.dataStrPtr = ptr_new(*self.dataPtr)
    widget_control,/hourglass
    if finite(self.ssfac) then ssfac = self.ssfac else ssfac = 1.0
    tol = abs((*self.data_tolerance))
    ind = where(tol eq 0,count,complement=ind1,ncomplement=count1)
    if count ne 0 then tol[ind] = ((count1 eq 0)?1e-6:(min(tol[ind1])<1e-6))  ;make sure no tolerance values are 0
    cmp_str1 = 'index = where('
    cmp_str2 = 'tmp1 = min(
    cmp_str3 = 'index = where('
    cmp_str4 = 'index = where('
    for i=2,n_elements(tol)+1 do begin
        cmp_str1 = cmp_str1+'(abs((*self.bgdata[0])[*,'+dm_to_string(i)+']-(*self.dataPtr).data[i,'+dm_to_string(i)+']) le tol['+dm_to_string(i-2)+'])' 
        cmp_str2 = cmp_str2+'abs((*self.bgdata[0])[*,'+dm_to_string(i)+']-(*self.dataPtr).data[i,'+dm_to_string(i)+'])/tol['+dm_to_string(i-2)+']'
        cmp_str3 = cmp_str3+'(abs((*self.bgdata[0])[*,'+dm_to_string(i)+']-(*self.dataPtr).data[i,'+dm_to_string(i)+']) eq (abs((*self.bgdata[0])[imin,'+dm_to_string(i)+']-(*self.dataPtr).data[i,'+dm_to_string(i)+'])))'  
        cmp_str4 = cmp_str4+'(abs((*self.bgdata[0])[*,'+dm_to_string(i)+']-(*self.bgdata[0])[imin,'+dm_to_string(i)+']) le tol['+dm_to_string(i-2)+'])' 
        if i ne n_elements(tol)+1 then begin
           cmp_str1 = cmp_str1+' and '
           cmp_str2 = cmp_str2+'+'
           cmp_str3 = cmp_str3+' and '
           cmp_str4 = cmp_str4+' and '
        endif
    endfor
    cmp_str1 = cmp_str1+',count)'
    cmp_str2 = cmp_str2+',imin)'
    cmp_str3 = cmp_str3+',count)'
    cmp_str4 = cmp_str4+',count)'
    if self.empt_tol_optn eq 0 then begin ;within tolerance or closest
       for i=0LL,n_elements((*self.dataPtr).data[*,0])-1 do begin
           ok = execute(cmp_str1,1,1) 
           if count eq 0 then begin
              ok = execute(cmp_str2,1,1)
              ok = execute(cmp_str3,1,1)
           endif
           if count eq 1 then begin
              (*self.dataPtr).data[i,0] = (*self.dataPtr).data[i,0]-(*self.bgdata[0])[index,0]*ssfac
              (*self.dataPtr).data[i,1] = sqrt(((*self.dataPtr).data[i,1])^2+((*self.bgdata[0])[index,1]*ssfac)^2)
           endif else begin
              tmp_qty = total((*self.bgdata[0])[index,0],nan=self.ftypecheckmask[1])
              zro_ind = where((*self.bgdata[0])[index,1] lt 0,zro_count) 
              tmp_err = ((*self.bgdata[0])[index,1])^2
              if zro_count ne 0 then begin
                 zro_err = -min(tmp_err[zro_ind])
                 tmp_err[zro_ind] = 0.0
              endif
              tmp_err = total(tmp_err,nan=self.ftypecheckmask[1])
              if (tmp_err eq 0) and (n_elements(zro_err) ne 0) then tmp_err = temporary(zro_err) 
              (*self.dataPtr).data[i,0] = (*self.dataPtr).data[i,0]-tmp_qty/count*ssfac
              if ((*self.dataPtr).data[i,1] lt 0) and (tmp_err lt 0) then (*self.dataPtr).data[i,1] = ((*self.dataPtr).data[i,1])<(-sqrt(abs(tmp_err))/count*ssfac) $               
              else (*self.dataPtr).data[i,1] = sqrt((0.0>((*self.dataPtr).data[i,1]))^2+(0.0>(tmp_err))/(count^2)*(ssfac^2))
           endelse
       endfor 
    endif else begin  ;within tolerance of the closest
       for i=0LL,n_elements((*self.dataPtr).data[*,0])-1 do begin
           ok = execute(cmp_str2,1,1)
           ok = execute(cmp_str4,1,1)
           if count eq 1 then begin
              (*self.dataPtr).data[i,0] = (*self.dataPtr).data[i,0]-(*self.bgdata[0])[index,0]*ssfac
              (*self.dataPtr).data[i,1] = sqrt(((*self.dataPtr).data[i,1])^2+((*self.bgdata[0])[index,1]*ssfac)^2)
           endif else begin
              tmp_qty = total((*self.bgdata[0])[index,0],nan=self.ftypecheckmask[1])
              zro_ind = where((*self.bgdata[0])[index,1] lt 0,zro_count)
              tmp_err = ((*self.bgdata[0])[index,1])^2
              if zro_count ne 0 then begin
                 zro_err = -min(tmp_err[zro_ind])
                 tmp_err[zro_ind] = 0.0
              endif
              tmp_err = total(tmp_err,nan=self.ftypecheckmask[1])
              if (tmp_err eq 0) and (n_elements(zro_err) ne 0) then tmp_err = temporary(zro_err)
              (*self.dataPtr).data[i,0] = (*self.dataPtr).data[i,0]-tmp_qty/count*ssfac
              if ((*self.dataPtr).data[i,1] lt 0) and (tmp_err lt 0) then (*self.dataPtr).data[i,1] = ((*self.dataPtr).data[i,1])<(-sqrt(abs(tmp_err))/count*ssfac) $
              else (*self.dataPtr).data[i,1] = sqrt((0.0>((*self.dataPtr).data[i,1]))^2+(0.0>(tmp_err))/(count^2)*(ssfac^2))
           endelse          
       endfor
    endelse
end

; adds an object to the list and shows its title to the view menu or deletes an object from the list
; and removes the corresponding item in the view menu
; parameter:
;   opcode:   0: add plotwindow , this is the default operation
;             1: make plotwindow current, object keyword must be given
;             2: keep the plotwindow, object keyword must be given
;             3: delete plotwindow, object keyword must be given
;             4: close all plot windows
;             5: change title, object keyword must be given
; keyword:
;   type:     0:slice 1:cut 2:trajectory 3:volume
;   title:    string of the button name to be added or changed to the view menu
;   wtitle:   base title of the plot window
;   object:   opcode=1,2--object chosen; opcode=3--object to be deleted
pro dcs_mslice::dm_add_plotwindow,opcode,type=type,title=title,wtitle=wtitle,object=object
    if n_elements(opcode) eq 0 then opcode=0   ;default add plot window
    viewMenu = widget_info(self.tlb,find_by_uname='viewMenu')
    if ~widget_info(viewMenu,/valid_id) then return
    case opcode of
       0:begin
         if (~ obj_valid(self.current)) then return
         ;create the unique id for the view menu item
         id  = self->dm_get_id(self.current)
         ;check if the current object is already in the list, if not add it to the list
         all = self.gr_list->get(/all,count=count)
         in  = 0b
         for i=0,count-1 do begin
             if all[i] eq self.current then begin
                in = 1b
                break
             endif
         endfor
         if in eq 0 then begin    ;not in the list
            self.gr_list->add,self.current            ;add the object to the list
            id = dm_widget_button(viewMenu,value='',uname=id,separator=(count eq 0)) ;add the item to the view menu
         endif else begin     ;in the list, find the menu item id
            id = widget_info(viewMenu,find_by_uname=id)
         endelse
         ;generate proper string for the menu item
         if n_elements(type) eq 0 then type=0
         case type of
            0:   value = 'Slice: '
            1:   value = 'Cut: '
            2:   value = 'Trajectory: '
            3:   value = 'Volume: '
            else:value = 'Plot: '
         endcase
         if n_elements(title) ne 0 then value=value+strtrim(title,2)
         if widget_info(id,/valid_id) then begin
            if self.idl_version ge 5.6 then $
               widget_control,id,set_value=value,set_button=1 $
            else $
               widget_control,id,set_value=value
         endif
         self->my_widget_control,['killAll','arrPlot'],viewMenu,/sensitive
         if self.idl_version ge 6.1 then $
            self->my_widget_control,'viewCurr',viewMenu,/sensitive 
         end
       1:begin 
         if obj_valid(self.current) and (self.idl_version ge 5.6) then begin   ;remove the current check mark
            id = self->dm_get_id(self.current) ;locate the corresponding view menu item id
            self->my_widget_control,id,viewMenu,set_button=0
         endif
         all = self.gr_list->get(/all,count=count)
         for i=0,count-1 do begin
             if all[i] eq object then self.current=all[i]
         endfor
         if self.idl_version ge 5.6 then begin ;add check mark to the current plot window
            id = self->dm_get_id(self.current)
            self->my_widget_control,id,viewMenu,set_button=1
         endif
         if self.idl_version ge 6.1 then $
            self->my_widget_control,'viewCurr',viewMenu,/sensitive 
         end
       2:begin          
         self.current = obj_new()
         if self.idl_version ge 5.6 then begin ;remove check mark from the view menu
            id = self->dm_get_id(object)  ;locate the corresponding view menu item id
            self->my_widget_control,id,viewMenu,set_button=0
         endif
         if self.idl_version ge 6.1 then $
            self->my_widget_control,'viewCurr',viewMenu,sensitive=0 
         end
       3:begin       
         all = self.gr_list->get(/all,count=count)
         if count eq 0 then return
         for i=0,count-1 do begin   ;destroy all the plot menu items
             id = self->dm_get_id(all[i])
             self->my_widget_control,id,viewMenu,get_value=value,/destroy  ;remove the item in the view menu
             if n_elements(info) eq 0 then info = [id,value] $  ;save the names
             else info = [[info],[[id,value]]]
         endfor      
         self.gr_list->remove,object       ;remove the object from the list
         all = self.gr_list->get(/all,count=count)
         for i=0,count-1 do begin   ;reconstruct the plot menu items
             id  = self->dm_get_id(all[i])
             all[i]->getproperty,keep=keep,title=title
             value = title ;change the menu name to the title of the plot
             id1 = where(info[0,*] eq id,existed)
             if existed ne 0 then begin
                tmp = info[1,id1[0]]
                tmp = strsplit(tmp,':',/extract,/preserve_null)
                if n_elements(tmp) ne 0 then value=tmp[0]+': '+title ;include the type string
             endif
             id = dm_widget_button(viewMenu,uname=id,value=value,separator=(i eq 0),set_button=(1-keep))
         endfor
         self->my_widget_control,['killAll','arrPlot'],viewMenu,sensitive=(count gt 0)
         if self.idl_version ge 6.1 then $
            self->my_widget_control,'viewCurr',viewMenu,sensitive=(self.current ne object)
         end
       4:begin
         ok = dialog_message('Are you sure you want to close all plot windows?',/question,dialog_parent=self.tlb,/center,title='Please confirm:')
         if strupcase(ok) eq 'NO' then return
         all = self.gr_list->get(/all,count=count)
         self.gr_list->remove,/all
         if count gt 0 then begin
            obj_destroy,all
             ;destroy all the plot menu items
            for i=0,count-1 do begin  
                id = self->dm_get_id(all[i])
                self->my_widget_control,id,viewMenu,get_value=value,/destroy  ;remove the item in the view menu
            endfor
            self->my_widget_control,['killAll','arrPlot'],viewMenu,sensitive=0
            if self.idl_version ge 6.1 then $
               self->my_widget_control,'viewCurr',viewMenu,sensitive=0 
         endif
         end
       5:begin          
         if n_elements(object) ne 0 and n_elements(title) ne 0 then begin
            id = self->dm_get_id(object)  ;locate the corresponding view menu item id
            self->my_widget_control,id,viewMenu,get_value=oldtitle
            if n_elements(wtitle) eq 0 then wtitle = ''
            if strlen(wtitle) eq 0 then wtitle = (strsplit(oldtitle,':',/extract))[0]
            if strlen(wtitle) ne 0 then wtitle = wtitle+': '
            self->my_widget_control,id,viewMenu,set_value=wtitle+strtrim(title,2)
         endif
         end
       else:
    endcase
end

;arrange the plot windows
pro dcs_mslice::dm_arrange_plot,oplots,cascade=cascade
    count = n_elements(oplots)
    if count eq 0 then oplots = self.gr_list->get(/all,count=count)
    if count eq 0 then return
    for i=0,count-1 do oplots[i]->totop
    if count eq 1 then return
    if self.idl_version ge 5.6 then scr_rect = get_screen_size() else device,get_screen_size=scr_rect
    scr_rect = [0,0,scr_rect]
    if self.idl_version ge 6.3 then begin  ;for mulitple screens
       geom = widget_info(self.tlb,/geometry)
       oinfo = OBJ_NEW('IDLsysMonitorInfo')
       rects = oInfo->GetRectangles()
       if oinfo->IsExtendedDesktop() then begin
          curMon = 0>(where(rects[0,*]+rects[2,*] ge geom.xoffset+geom.scr_xsize/2))[0]
          scr_rect = rects[*,curMon]
       endif
       obj_destroy,oinfo
    endif
    if keyword_set(cascade) then begin
       oplots[count-1]->getproperty,xsize=xsize,ysize=ysize & ysize=ysize+20
       xstep = (40<((scr_rect[2]-xsize)/count))
       ystep = (40<((scr_rect[3]-ysize)/count))
       xoff  = scr_rect[0]+(5>((scr_rect[2]-xstep*count-xsize)/2))
       yoff  = scr_rect[1]+(5>((scr_rect[3]-ystep*count-ysize)/2))
       for i=0,count-1 do begin
           oplots[i]->getproperty,tlb=tlb
           widget_control,tlb,xoffset=((xoff+xstep*i)<(scr_rect[0]+scr_rect[2]-10)),yoffset=((yoff+ystep*i)<(scr_rect[1]+scr_rect[3]-10))
       endfor
    endif else begin
       if count le 4 then ncol=2 $
       else if count le 9 then ncol=3 $
       else ncol=4
       xsize = intarr(count) & ysize = intarr(count) & wid = lonarr(count)
       for i=0,count-1 do begin
           oplots[i]->getproperty,tlb=tlb
           tmp = widget_info(tlb,/geometry)
           wid[i] = tlb & xsize[i] = tmp.scr_xsize+2 & ysize[i] = tmp.scr_ysize+2
       endfor
       nrow = (count+ncol-1)/ncol
       roww = intarr(nrow) & rowh = intarr(nrow) & xoff = intarr(nrow)
       for i=0,nrow-1 do begin
           roww[i] = total(xsize[ncol*i:((ncol*(i+1)-1)<(count-1))])
           rowh[i] = max(ysize[ncol*i:((ncol*(i+1)-1)<(count-1))])
           xoff[i] = scr_rect[0]+(0>((scr_rect[2]-roww[i])/2))
       endfor
       yoff = scr_rect[1]+(0>((scr_rect[3]-total(rowh))/2))
       for i=0,count-1 do begin
           row = i/ncol
           col = (i mod ncol)
           if xoff[row] eq 0 then xoffset=scr_rect[0]+scr_rect[2]/ncol*col $
           else begin
              xoffset = xoff[row]
              if col ne 0 then xoffset = (xoffset+total(xsize[row*ncol:row*ncol+col-1]))<(scr_rect[0]+scr_rect[2]-20)
           endelse
           if yoff eq 0 then yoffset=scr_rect[1]+scr_rect[3]/nrow*row $
           else begin
              yoffset = yoff
              if row ne 0 then yoffset = (yoffset+total(rowh[0:row-1]))<(scr_rect[1]+scr_rect[3]-20)
           endelse
           widget_control,wid[i],xoffset=xoffset,yoffset=yoffset
       endfor
    endelse
end

;create a unique id for an object
function dcs_mslice::dm_get_id,object
    help,object,output=id & id=strsplit(id,' ',/extract) &  id=id[n_elements(id)-1]
    tmp = strsplit(id,'()',/extract) & if n_elements(tmp) eq 3 then id=tmp[0]+tmp[2]
    return,id
end

;object event handler
pro dcs_mslice::event,event,resetpsioffset=resetpsioffset
    compile_opt IDL2    ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       errmesg = !error_state.msg 
       if obj_valid(self) then begin
          if (self.samp_typ eq 3) and (!error_state.name eq 'IDL_M_CNTGETMEM') then $
             errmesg = [errmesg,'','Please reduce the energy range, or use single E option.']
          if strpos(errmesg[0],'rebinnd') ne -1 then begin
             if (lmgr(/runtime) eq 0) and (!version.os_family eq 'Windows') then begin
                is64bit = (!version.memory_bits eq 64)
                errmesg = 'Please copy msvcr120'+(['','_64'])[is64bit]+'.dll from \dave\programs\auxiliary\rebin_nd\ to your IDL bin\bin.x86'+(['','_64'])[is64bit]+' directory'+$
                          (['.',', and rename it to msvcr120.dll.'])[is64bit]
             endif 
             errmesg = [errmesg,'','The external binning library has since been disabled. You can restore it from the menu Option->Binning Method->Use External Binning Library.']
             self->setproperty,binextlib=0
          endif 
       endif
       ok = dialog_message(errmesg,/error,dialog_parent=self.tlb,/center)
       if obj_valid(self) then self.error = 1b
       if widget_info(self.tlb,/update) eq 0 then widget_control,self.tlb,/map,/update
       return
    endif

    widget_type = widget_info(event.id,/type)
    case widget_type of
       1:tmp_value = event.select                               ;button
       3:begin                                                  ;text box
         widget_control,event.id,get_value=tmp_value
         tmp_value = strtrim(tmp_value,2)
         end
       8:tmp_value = widget_info(event.id,/droplist_select)     ;drop list
       12:begin                                                 ;combobox
         tmp_value = event.index
         widget_control,event.id,get_value=tmp_list
         if tmp_list[tmp_value] ne event.str then begin         ;to avoid events from set_value in unix operating system
            return
         endif
         end
       else:
    endcase
    if n_elements(tmp_value) eq 1 then tmp_value=tmp_value[0]
   
    case self.instrname of  ;for monitor events
       'dcs':  mon_instr = 0
       'macs': mon_instr = 1
       else:   mon_instr = 0
    endcase

    uname = widget_info(event.id,/uname)
    type  = tag_names(event,/structure)
    if self.samp_typ eq -1 then nitem = n_elements((*self.dataPtr).label)
    case strlowcase(type) of
        'dm_macs_sequencer_size': uname = type
        'dm_plot_make_current':   uname = type
        'dm_plot_keep':           uname = type
        'dm_plot_kill':           uname = type
        'dm_plot_changetitle':    uname = type
        'dm_filesel_newfilter':   uname = type
        'widget_context':         uname = 'status'
        'widget_kbrd_focus':      if widget_info(event.id,/name) eq 'TEXT' then uname = 'text_focus'
        'widget_text_sel':        if self.txtfocusevent then begin
                                  tmp = widget_event(event.id,/nowait)
                                  widget_control,event.id,get_value=tmp
                                  if n_elements(tmp) eq 1 then begin
                                     if strlen(strtrim(tmp,2)) ne 0 then widget_control,event.id,set_text_select=[0,strlen(tmp[0])]
                                  endif   
                                  endif
        else:
    endcase
    tmp = strmid(strlowcase(uname),0,8)
    ind_sel = dm_to_number(strmid(strlowcase(uname),0,1,/reverse),/int)
    case tmp of
        'cutsel_t': uname = 'cutsel_t'
        'cutselfr': uname = 'cutsel_from'
        'cutselto': uname = 'cutsel_to'
        'dispsel_': uname = 'dispsel_t'
        'dispself': uname = 'dispsel_from'
        'dispselt': uname = 'dispsel_to'
        'volsel_t': uname = 'volsel_t'
        'volselfr': uname = 'volsel_from'
        'volselto': uname = 'volsel_to'
        'pdchoice': uname = 'pdim_choice'
        'calc_com': begin
                    latts = strjoin(['a','b','c','alpha','beta','gamma']+'= '+dm_to_string(float(self.latt_parm)),' & ')
                    abgs  = strjoin(['alpha','beta','gamma']+'= '+dm_to_string(float(self.latt_parm[3:5])),' & ')
                    uvstr = 'U=['+dm_to_string(self.orie_u,sep=',')+'] & V=['+dm_to_string(self.orie_v,sep=',')+']'
                    end
        else:
    endcase
    if strmid(strlowcase(uname),0,4) eq 'user' then begin
       if !version.os_family eq 'Windows' then begin
          sep  = string(9b)+': ' 
          tail = '             '
       endif else begin 
          sep  = ': '
          tail = ''
       endelse
       mextra = ['','']
       if self.instrname eq 'macs' then begin
          if ptr_valid(self.dataStrPtr) then begin
             if self.extravaxis_yn[1] then mextra[0] = mextra[0]+' data_temperature,'
             if self.extravaxis_yn[2] then mextra = mextra+' data_hfield,'
             if self.extravaxis_yn[0] then mextra = mextra+' data_specify,'
             if keyword_set(self.macshistomode) then mextra[0] = mextra[0]+' histo_weight,'
          endif
          tmp_info = 'en, Ei, Ef, psi, kidney, weight,'+mextra[0]+' det_tt.'
          mextra[1] = ','+mextra[1]
          if strmid(mextra[1],0,1,/reverse) eq ',' then mextra[1] = strmid(mextra[1],0,strlen(mextra[1])-1)
       endif else begin
          if (self.instrname eq 'wand') and self.extravaxis_yn[0] and ptr_valid(self.dataStrPtr) then begin
             tmp_info = 'eief, en, psi, weight, data_temperature, det_tt.' 
          endif else if (self.instrname eq 'dcs') and ptr_valid(self.dataStrPtr) then begin
             if self.extravaxis_yn[1] then mextra[0] = mextra[0]+' data_temperature,'
             if self.extravaxis_yn[2] then mextra = mextra+' data_hfield,'
             tmp_info =  'eief, en,'+(['',' psi,'])[self.samp_typ gt 0]+mextra[0]+' weight, det_tt.'
             mextra[1] = ','+mextra[1]
             if strmid(mextra[1],0,1,/reverse) eq ',' then mextra[1] = strmid(mextra[1],0,strlen(mextra[1])-1)
          endif else begin
             tmp_info = 'eief, en, '+(['','psi, '])[self.samp_typ gt 0]+'weight, det_tt.'
          endelse
       endelse
       userf_info = 'Available variables after projection:'
       if self.samp_typ eq 0 then begin ;powder
          userf_info = userf_info+' qty, err, Q, '+tmp_info
          userf_info = [userf_info,'Available variables before projection: qty, err, '+tmp_info]
          if ptr_valid(self.dataStrPtr) then begin
             help,(*self.dataStrPtr).qty,output=tmp
             tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ') 
             userf_info = [userf_info,'qty,err'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]        
             if self.instrname eq 'macs' then begin
                if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).en,output=tmp $
                else help,(*self.dataStrPtr).Ei,output=tmp 
                tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'en,Ei,Ef,psi,weight,kidney'+mextra[1]+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                if keyword_set(self.macshistomode) then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).histo_weight,output=tmp $
                   else help,(*self.dataStrPtr).histo_weight,output=tmp
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'histo_weight'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endif
                if self.extravaxis_yn[1] then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).temperature,output=tmp $
                   else help,(*self.dataStrPtr).temperature,output=tmp 
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'data_temperature'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail] 
                endif
                if self.extravaxis_yn[2] then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).hfield,output=tmp $
                   else help,(*self.dataStrPtr).hfield,output=tmp 
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'data_hfield'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endif
             endif else begin
                help,self.eief,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'eief'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                help,(*self.dataStrPtr).e,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'en'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                help,(*self.dataStrPtr).weight,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'weight'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                if (self.instrname eq 'dcs') then begin
                   if self.extravaxis_yn[1] then begin 
                      if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).temperature,output=tmp $
                      else help,(*self.dataStrPtr).temperature,output=tmp 
                      tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                      userf_info = [userf_info,'data_temperature'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                   endif
                   if self.extravaxis_yn[2] then begin
                      if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).hfield,output=tmp $
                      else help,(*self.dataStrPtr).hfield,output=tmp 
                      tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                      userf_info = [userf_info,'data_hfield'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                   endif
                endif
             endelse
          endif else if ptr_valid(self.projStrPtr) then begin
             if self.instrname ne 'macs' then begin
             help,self.eief,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
             userf_info = [userf_info,'eief'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             endif
             help,(*self.projStrPtr).en,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
             if self.instrname eq 'macs' then begin
             userf_info = [userf_info,'en,Ei,Ef,psi,weight,kidney'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail] 
             endif else begin
             userf_info = [userf_info,'en'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             endelse
          endif
          if ptr_valid(self.projStrPtr) then begin
             help,(*self.projStrPtr).Q,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
             userf_info = [userf_info,'Q'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail] 
          endif
          if ptr_valid(self.detPosPtr) then begin 
             help,(*self.detPosPtr).two_theta,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ') 
             userf_info = [userf_info,'det_tt'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
          endif
       endif else begin                 ;single crystal
          userf_info = userf_info+' qty, err, Qv, '+tmp_info
          userf_info = [userf_info,'Available variables before projection: qty, err, '+tmp_info]
          if ptr_valid(self.dataStrPtr) then begin 
             help,(*self.dataStrPtr).qty,output=tmp
             tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ') 
             userf_info = [userf_info,'qty,err'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             if self.instrname eq 'macs' then begin
                if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).en,output=tmp $
                else help,(*self.dataStrPtr).Ei,output=tmp 
                tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'en,Ei,Ef,psi,weight,kidney'+mextra[1]+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                if keyword_set(self.macshistomode) then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).histo_weight,output=tmp $
                   else help,(*self.dataStrPtr).histo_weight,output=tmp
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'histo_weight'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endif
                if self.extravaxis_yn[1] then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).temperature,output=tmp $
                   else help,(*self.dataStrPtr).temperature,output=tmp 
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'data_temperature'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endif
                if self.extravaxis_yn[2] then begin
                   if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).hfield,output=tmp $
                   else help,(*self.dataStrPtr).hfield,output=tmp 
                   tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                   userf_info = [userf_info,'data_hfield'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endif    
             endif else begin
                help,self.eief,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'eief'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                help,(*self.dataStrPtr).e,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'en'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                help,(*self.dataStrPtr).rotation,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                if (self.instrname eq 'wand') and self.extravaxis_yn[0] then $
                   userf_info = [userf_info,'psi,weight,data_temperature'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail] $
                else begin
                   if size((*self.dataStrPtr).rotation,/n_dim) eq 0 then begin
                      userf_info = [userf_info,'psi'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                      help,(*self.dataStrPtr).weight,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                      userf_info = [userf_info,'weight'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                   endif else userf_info = [userf_info,'psi,weight'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                endelse
                if (self.instrname eq 'dcs') then begin
                   if self.extravaxis_yn[1] then begin
                      if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).temperature,output=tmp $
                      else help,(*self.dataStrPtr).temperature,output=tmp
                      tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                      userf_info = [userf_info,'data_temperature'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                   endif
                   if self.extravaxis_yn[2] then begin
                      if ptr_valid(self.projStrPtr) then help,(*self.projStrPtr).hfield,output=tmp $
                      else help,(*self.dataStrPtr).hfield,output=tmp 
                      tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                      userf_info = [userf_info,'data_hfield'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
                   endif
                endif
             endelse
          endif else if ptr_valid(self.projStrPtr) then begin
             if self.instrname ne 'macs' then begin
                help,self.eief,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                userf_info = [userf_info,'eief'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             endif
             help,(*self.projStrPtr).en,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
             if self.instrname eq 'macs' then begin
                userf_info = [userf_info,'en,Ei,Ef,psi,kidney'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail] 
             endif else begin
             userf_info = [userf_info,'en'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             help,reform((*self.projStrPtr).V[0,0,0,*]),output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
             userf_info = [userf_info,'psi'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
             endelse
          endif
          if ptr_valid(self.projStrPtr) then begin
             help,(*self.projStrPtr).V,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ') & pos1 = strpos(strlowcase(tmp),'array[')
             userf_info = [userf_info,'Qv'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
          endif
          if ptr_valid(self.detPosPtr) then begin 
             help,(*self.detPosPtr).two_theta,output=tmp & tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ') 
             userf_info = [userf_info,'det_tt'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)+tail]
          endif
       endelse
       if (self.instrname eq 'macs') and (self.macsintn_typ gt 0) then userf_info = [userf_info,'Ef = Ei when detector choice is "'+(['DIFF','SPEC+DIFF'])[self.macsintn_typ-1]+'".'] 
    endif
    if uname ne 'text_focus' then WIDGET_CONTROL,/HOURGLASS
    self.txtfocusevent = 0b
    case strlowcase(uname) of
        'dm_macs_sequencer_size': self.macs_seq_geom = [event.textboxsize,event.handedness]
        'dm_plot_make_current':   self->dm_add_plotwindow,1,object=event.object   ;this event is sent from dm_plot window
        'dm_plot_keep':           self->dm_add_plotwindow,2,object=event.object   ;this event is sent from dm_plot window
        'dm_plot_kill':           self->dm_add_plotwindow,3,object=event.object   ;this event is sent from dm_plot window
        'dm_plot_changetitle':    self->dm_add_plotwindow,5,object=event.object,title=event.title,wtitle=event.wtitle   ;this event is sent from dm_plot_window
        'text_focus': begin
            if event.enter then $
               self.txtfocusevent = 1b $
            else $
               widget_control,event.id,set_text_select=[0,0]
          end
        'tlb':begin
            if self.idl_version ge 5.5 then $
               widget_control,self.tlb,xsize=self.geom[0],ysize=self.geom[1] $
            else $
               widget_control,self.tlb,scr_xsize=self.geom[0],scr_ysize=self.geom[1]
          end
        'savevolumetxt':      self->save_volume
        'savevolxslicespe':   self->save_volume,slice=0,/spe
        'savevolxsliceiexy':  self->save_volume,slice=0,/iexy
        'savevolxslicexyie':  self->save_volume,slice=0,/xyie
        'savevolyslicespe':   self->save_volume,slice=1,/spe
        'savevolysliceiexy':  self->save_volume,slice=1,/iexy
        'savevolyslicexyie':  self->save_volume,slice=1,/xyie
        'savevolzslicespe':   self->save_volume,slice=2,/spe
        'savevolzsliceiexy':  self->save_volume,slice=2,/iexy
        'savevolzslicexyie':  self->save_volume,slice=2,/xyie
        'saveslicedave':      self->save_slice,/dave
        'saveslicespe':       self->save_slice,/spe
        'savesliceiexy':      self->save_slice,/iexy
        'saveslicexyie':      self->save_slice,/xyie
        'sendslicedave': begin
            self->save_slice,dPtr=dPtr,error=error
            if keyword_set(error) then return
            (self.DAVETool)->AddDavePtrToDataManager, dPtr, 'slice from mslice'  
            heap_free,dPtr  
          end              
        'savecutxye':    self->save_cut,/xye
        'savecutdave':   self->save_cut,/dave
        'sendcutdave': begin
            self->save_cut,dPtr=dPtr,error=error
            if keyword_set(error) then return    
            (self.DAVETool)->AddDavePtrToDataManager, dPtr, 'cut from mslice'
            heap_free,dPtr
          end
        'loaddata':      self->load_data
        'loadbgdata':    self->load_data,/background
        'readascii': begin
            path = self.dirs[1]
            datafile = dm_choose_file('*',dialog_parent=self.tlb,/read,path=path,title='Please choose an ascii data file',/multi)
            if strlen(datafile[0]) eq 0 then return
            self->setproperty,workDir=path
            WIDGET_CONTROL,/HOURGLASS
            tmp = obj_new('dcs_mslice',dataFile=datafile,group_leader=self.tlb)
            tmp->setproperty,tickdir=self.tdir_choice
          end
        'saveprojtxt':   self->save_projection
        'd4save':        self->save_4d
        'runscript': begin
            path = self.dirs[1]
            scriptfile = dm_choose_file('*',dialog_parent=self.tlb,/read,path=path,title='Please choose a script file')
            if strlen(scriptfile) eq 0 then return
            self->setproperty,workDir=path
            WIDGET_CONTROL,/HOURGLASS
            tmp = ''
            script = ''
            openr,unit,scriptfile,/get_lun
            while(~ eof(unit)) do begin
                readf,unit,tmp
                script=[script,tmp]
            endwhile
            free_lun,unit
            if n_elements(script) gt 1 then begin
               script = script[1:*]
               ok = execute(dm_strip_comments(script,/combine),1,1)
               if ~ok then begin ;figure out on which line the error occurs, and show error message
                  dm_check_syntax,errmsg=errmsg,name='the script file',sequence=script,self=self
                  ok = dialog_message(errmsg,/error,dialog_parent=self.tlb,/center)
               endif
            endif
          end
        'hist_0':  self->set_filehistory,choose=0
        'hist_1':  self->set_filehistory,choose=1
        'hist_2':  self->set_filehistory,choose=2
        'hist_3':  self->set_filehistory,choose=3
        'hist_4':  self->set_filehistory,choose=4
        'hist_5':  self->set_filehistory,choose=5
        'hist_6':  self->set_filehistory,choose=6
        'hist_7':  self->set_filehistory,choose=7
        'hist_8':  self->set_filehistory,choose=8
        'hist_9':  self->set_filehistory,choose=9
        'clearhist': begin
            ptr_free,self.file_hist[0:9]
            self.n_history = 0
            self->set_filehistory
          end
        'exitbut':begin
            widget_control,self.tlb,/destroy
            return
          end
        'ldsesbut':begin  ;load session file
            path = self.dirs[1]
            sesfile = dm_choose_file('ses',dialog_parent=self.tlb,/read,path=path)
            if strlen(sesfile) ne 0 then self->dm_load_parameter,sesfile,/session
            self->setproperty,workDir=path
            if ptr_valid(self.dataStrPtr) then self->dm_calc_projection
          end
        'svsesbut':begin ;save session
            path = self.dirs[1]
            sesfile = dm_choose_file('ses',dialog_parent=self.tlb,/write,path=path)
            if strlen(sesfile) ne 0 then self->dm_save_parameter,sesfile,/session
            self->setproperty,workDir=path
          end          
        'ldpmbut':begin  ;load parameter file
            path = self.dirs[1]
            parmfile = dm_choose_file('prm',dialog_parent=self.tlb,/read,path=path)
            if strlen(parmfile) ne 0 then self->dm_load_parameter,parmfile
            self->setproperty,workDir=path
          end
        'svpmbut':begin ;save parameters into a file
            path = self.dirs[1]
            parmfile = dm_choose_file('prm',dialog_parent=self.tlb,/write,path=path)
            if strlen(parmfile) ne 0 then self->dm_save_parameter,parmfile
            self->setproperty,workDir=path
          end
        'knpmbut':begin  ;change smoothing kernel
            old = dm_to_string(float(self.slice_sm_kernel))
            old = '[['+old[0]+' ,'+old[1]+' ,'+old[2]+'],['+old[3]+' ,'+old[4]+' ,'+old[5]+'],['+old[6]+' ,'+old[7]+' ,'+old[8]+']]'
            new = dm_dialog_input('kernel matrix:',title='Change smoothing kernel',default=old,xsize=250,dialog_parent=self.tlb,info='The smoothing kernel must be a 3x3 matrix.',cancel=cancel)
            if keyword_set(cancel) then return
            new = dm_to_number(strsplit(new,'[], '+string(9b),/extract),/float)
            if n_elements(new) ne 9 then begin
               ok = dialog_message('Please input a 3x3 matrix.',/error,dialog_parent=self.tlb,/center)
               break
            endif
            if max(abs(new[indgen(3,3)]-new[transpose(indgen(3,3))])) ne 0.0 then begin
               ok = dialog_message('Please input a symmetric matrix.',/error,dialog_parent=self.tlb,/center)
               break
            endif
            if total(new[0:8]) ne 0 then self.slice_sm_kernel=new[indgen(3,3)]
          end
        'bnthbut':begin  ;change binning threshold
            old   = self.binthreshold
            info  = ['Bins with fewer number of data points than (filling factor)*(NMAX)','will be discarded, where NMAX is the maximum number of points in a bin.']
            if self.samp_typ ne -1 then info=[info,'This factor will be reset to 0 every time a file is loaded.']
            repeat begin
              new = dm_dialog_input('filling factor:',title='Change binning threshold',dialog_parent=self.tlb,default=old,info=info,/float,cancel=cancel)
              if keyword_set(cancel) then return
              if finite(new) then begin
                 if (new ge 0) and (new lt 1) then begin
                     ok2 = 1b
                 endif else begin
                     old  = new
                     info = '0.0<= filling factor <1.0'
                 endelse
              endif
            endrep until keyword_set(ok2)
            self.binthreshold = new           
          end
        'ssfcbut':begin  ;change self-shielding factor
            old = self.ssfac
            name = (['self-shielding','background subtraction'])[self.samp_typ eq -1]+' factor'
            if self.samp_typ eq -1 then begin
               info = ['I = I_signal - factor*I_background']
            endif else begin
               n_emp = total(ptr_valid(self.bgdata[1:*]))
               info  = 'I = I_sample - ssfac*'+(['I_emptycan','('+strjoin('factor'+dm_to_string(indgen(1+n_emp)+1)+'*I_can'+dm_to_string(indgen(1+n_emp)+1),' + ')+')'])[n_emp gt 0]
               if n_emp gt 0 then info = [info+',','Currently '+strjoin('factor'+dm_to_string(indgen(1+n_emp)+1)+'='+dm_to_string(self.empcanfactor[0:n_emp]),', ')+'.']
            endelse
            new = dm_dialog_input(name+':',title=name,default=old,/float,info=info,dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(new) then begin
               if new ne self.ssfac then begin
                  self.ssfac = new
                  if (self.samp_typ ge 0) and ptr_valid(self.projStrPtr) and ptr_valid(self.bgdata[0]) then recalc_proj = 1b    ;need to recalculate projections
                  if (self.samp_typ eq -1) then self->data_subbg
               endif
               if self.samp_typ ne -1 then begin
                  ptr_free,self.ssfacPtr
                  self->my_menu_toggle,['','readdet','readen']+'ssfcBut',ptr_valid(self.ssfacPtr)
               endif
            endif else ok = dialog_message('Invalid input.',/error,dialog_parent=self.tlb,/center)
          end
        'readdetssfcbut':begin  ;read detector dependent self-shielding factor from a file
            self->load_ssfac,file,ssfac=ssfac
            if n_elements(ssfac) ne 0 then begin
               ptr_free,self.ssfacPtr
               self.ssfacPtr = ptr_new({file:file,ssfac:ssfac,det_or_en:0})
            endif
            self->my_menu_toggle,['','readdet','readen']+'ssfcBut',ptr_valid(self.ssfacPtr)
          end
        'readenssfcbut':begin  ;read energy dependent self-shielding factor from a file
            self->load_ssfac,file,ssfac=ssfac
            if n_elements(ssfac) ne 0 then begin
               ptr_free,self.ssfacPtr
               self.ssfacPtr = ptr_new({file:file,ssfac:ssfac,det_or_en:1})
            endif
            i_ssfc = ptr_valid(self.ssfacPtr)
            if i_ssfc then i_ssfc = i_ssfc+(*self.ssfacPtr).det_or_en
            self->my_menu_toggle,['','readdet','readen']+'ssfcBut',i_ssfc
          end  
        'detoffsetbut':begin
            case self.instrname of 
                 'wand':  old = -((*self.detPosPtr).two_theta)[0]
                 else:    old = ((*self.detPosPtr).two_theta)[0]
            endcase
            new = dm_dialog_input('Offset Angle:',title='Change detector offset angle',default=old,/float,dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return 
            if finite(new) then begin
               case self.instrname of 
                    'wand':   (*self.detPosPtr).two_theta = (*self.detPosPtr).two_theta+old-new 
                    else:     (*self.detPosPtr).two_theta = (*self.detPosPtr).two_theta+new-old
               endcase
            endif else $
               ok = dialog_message('Invalid input.',/error,dialog_parent=self.tlb,/center)  
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end
        'macsdeteffbut':begin
            if ~ptr_valid(self.eff) then begin
               widget_control,event.id,sensitive=0
               return
            endif
            new = dm_dialog_listinput('det '+dm_to_string(1+indgen(n_elements((*self.detPosPtr).two_theta))),title='Please input new efficiencies:',default=(*self.eff),$
                 label=['SPEC','DIFF'],info=['Intensity will be multiplied by these values.'],cancel=cancel,dialog_parent=self.tlb)
            if keyword_set(cancel) or (total(finite(new,/nan)) gt 0) then return
            if total(abs((*self.eff)-new)) eq 0 then return
            (*self.eff) = new     
            if ~ptr_valid(self.eff1) then self.ftoolflag[1] = 1b
          end
        'addtolbut':begin
            info = 'The tolerance value'+([' is','s are'])[self.instrname eq 'macs']+' used for combining data in file loading.'
            if ptr_valid(self.dataStrPtr) then begin
               help,(*self.dataStrPtr).qty,output=tmp
               tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
               info = [info,'Currently qty,err: '+strtrim(strmid(tmp,pos+1,strlen(tmp)-1-pos),2)+'.']
            end
            if self.instrname eq 'macs' then begin
               name = (['energy tolerance (meV)',['kidney','A3']+' tolerance ('+string('b0'xb)+')'])[0:(1+self.samp_typ)]
               default = self.macs_tolerance[0:(1+self.samp_typ),1]
               new = dm_dialog_input(name+':',/float,default=default,dialog_parent=self.tlb,title='Tolerance value',cancel=cancel,info=info)
            endif else begin
               default = self.psi_tolerance[1]
               new = dm_dialog_input(['psi tolerance ('+string('b0'xb)+'):'],/float,default=default,dialog_parent=self.tlb,title='Tolerance value',cancel=cancel,info=info)
            endelse
            if keyword_set(cancel) then return
            if total(finite(new,/nan)) gt 0 then begin
               ok = dialog_message('Invalid input.',/error,dialog_parent=self.tlb,/center)
               return
            endif
            if total(abs(new-default)) eq 0 then return ;no change
            if self.instrname eq 'macs' then begin
               if (self.macs_tolerance[1+self.samp_typ,1] ne abs(new[1+self.samp_typ])) and (self.samp_typ eq 1) then self.ftoolflag[6] = 0b  ;reset the warning flag
               self.macs_tolerance[0:(1+self.samp_typ),1] = abs(new)
            endif else begin
               self.psi_tolerance[1] = abs(new)
               self.ftoolflag[6] = 0b  ;reset the warning flag
            endelse
            tmp = ptr_valid(self.dataStrPtr)+2*ptr_valid(self.bgdata[0])
            if tmp gt 0 then ok = dialog_message('You might need to reload the'+(['',' data',' background',' data and background'])[tmp]+' files.',dialog_parent=self.tlb,/center)
          end
        'bgtolbut':begin
            if self.samp_typ eq -1 then begin
               default = (*self.data_tolerance)
               name = (*self.dataPtr).label[2:*]+' tolerance'
               ind = where(strlen( (*self.dataPtr).unit[2:*]) ne 0,count)
               if count ne 0 then name[ind] = name[ind]+' ('+(*self.dataPtr).unit[2+ind]+')'
               new = dm_dialog_input(name+':',/float,default=default,dialog_parent=self.tlb,title='Tolerance value',cancel=cancel,$
                     info='The tolerance values are used in choosing corresponding background data for subtraction.')
            endif else if self.instrname eq 'macs' then begin
               name = (['energy tolerance (meV)',['kidney','A3']+' tolerance ('+string('b0'xb)+')'])[0:(1+self.samp_typ)]
               default = self.macs_tolerance[0:(1+self.samp_typ),0]
               new = dm_dialog_input(name+':',/float,default=default,dialog_parent=self.tlb,title='Tolerance value',cancel=cancel,$
                     info='The tolerance values are used in choosing corresponding empty can data for subtraction.')
            endif else begin
               default = self.psi_tolerance[0]
               new = dm_dialog_input(['psi tolerance ('+string('b0'xb)+'):'],/float,default=default,dialog_parent=self.tlb,title='Tolerance value',cancel=cancel,$
                     info='The tolerance value is used in choosing corresponding empty can data for subtraction.')
            endelse
            if keyword_set(cancel) then return 
            if total(finite(new,/nan)) gt 0 then begin
               ok = dialog_message('Invalid input.',/error,dialog_parent=self.tlb,/center)
               return
            endif
            if total(abs(new-default)) eq 0 then return ;no change
            if self.samp_typ eq -1 then begin
               (*self.data_tolerance) = new
               self->data_subbg
            endif else if self.instrname eq 'macs' then begin
               if (self.macs_tolerance[1+self.samp_typ,0] ne abs(new[1+self.samp_typ])) and (self.samp_typ eq 1) then self.ftoolflag[5] = 0b  ;reset the warning flag
               self.macs_tolerance[0:(1+self.samp_typ),0] = abs(new)
            endif else begin
               self.psi_tolerance[0] = abs(new)
               self.ftoolflag[5] = 0b  ;reset the warning flag
            endelse
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.bgdata[0]) and ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections   
          end 
        'wandeiefbut':begin
            old = dm_e2lambda(self.eief)
            new = dm_dialog_input('wavelength ('+string('c5'XB)+'):',/float,default=finite(old)?round(old*1000.0,/L64)/1000.0:old,dialog_parent=self.tlb,/lambda2e,title='Change wavelength',cancel=cancel)
            if keyword_set(cancel) then return 
            if finite(new) then begin
               self.eief = dm_lambda2e(new)
               if ptr_valid(self.dataStrPtr) then (*self.dataStrPtr).eief = self.eief
            endif else $
               ok = dialog_message('Invalid input.',/error,dialog_parent=self.tlb,/center)  
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end  
        'emfcbut':begin  ;change multiple-set empty can factors
            n_emp = total(ptr_valid(self.bgdata[1:*]))
            if ptr_valid(self.ssfacPtr) then ssfac_str = 'read from '+file_basename((*self.ssfacPtr).file) else ssfac_str = 'ssfac = '+dm_to_string(self.ssfac)
            repeat begin
                tmp = dm_dialog_input('empty can '+dm_to_string(1+indgen(n_emp+1))+' factor:',/float,default=self.empcanfactor[0:n_emp],$
                      info=[(['','I=I_samp-ssfac*('+strjoin('factor'+dm_to_string(indgen(1+n_emp)+1)+'*I_can'+dm_to_string(indgen(1+n_emp)+1),' + ')+'),'])[n_emp gt 0],$
                      'ssfac: self-shielding factor, currently '+ssfac_str+'.'],dialog_parent=self.tlb,cancel=cancel)
                if keyword_set(cancel) then return
                if total(finite(tmp)) eq 1+n_emp then begin
                   self.empcanfactor[0:n_emp] = tmp
                   ok2 = 1b
                endif
            endrep until keyword_set(ok2)
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end  
        'ddirbut':begin    ;set data directory
            old = self.dirs[0]
            new = dialog_pickfile(/directory,path=old,dialog_parent=self.tlb)
            if (strlen(new) ne 0) then begin
               self.fileSel->set_path,new
               self->setproperty,dataDir=new
            endif
          end
        'wdirbut':begin  ;set working directory
            old = self.dirs[1]
            new = dialog_pickfile(/directory,path=old,dialog_parent=self.tlb)
            if (strlen(new) ne 0) then self->setproperty,workDir=new
          end
        'bgovbut':begin  ;overall background rate
            value = !values.f_nan & nvalue = 1
            if ptr_valid(self.bgratePtr) then begin
               value  = mean((*self.bgratePtr)[*,0])
               nvalue = n_elements(*self.bgratePtr)   ;dark count file existed
            endif
            tmp_value = dm_dialog_input(dialog_parent=self.tlb,'background rate:',info='in (cts/hr/det)',/float,default=value,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(tmp_value) then begin
               if tmp_value ge 0 then begin
                  if nvalue gt 1 then begin
                     ok = dialog_message('Do you want to replace the existing dark count file data?',/question,/default_no,dialog_parent=self.tlb,/center)
                     if strupcase(ok) eq 'NO' then break
                  endif
                  ptr_free,self.bgratePtr
                  self.bgratePtr = ptr_new(tmp_value)
                  bkgrmenu = widget_info(self.tlb,find_by_uname='bkgrMenu')
                  dm_set_button,widget_info(bkgrmenu,find_by_uname='bgdkBut'),0
                  dm_set_button,widget_info(bkgrmenu,find_by_uname='bgovBut'),1
                  self->my_widget_control,'bgclBut',bkgrmenu,/sensitive
                  if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
               endif else $
                  ok = dialog_message('Negative number is entered. The overall background rate has not been changed.',dialog_parent=self.tlb,/center)
            endif else $
               ok = dialog_message('Invalid input. The overall background rate has not been changed.',dialog_parent=self.tlb,/center)
          end
        'bgclbut':begin
            ptr_free,self.bgdata,self.bgweightPtr,self.bgratePtr,self.bgtchnPtr,self.bgdetsPtr
            if self.samp_typ ge 0 then begin
               self->reset_menu,/backgroundmenuonly
               if (self.idl_version ge 5.6) and (strlowcase(!version.os_family) eq 'windows') then self->my_widget_control,'clearDat',tooltip=''
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
            endif else begin
               self->my_widget_control,'loadbgData',set_button=0
               self->my_widget_control,'bgclBut',sensitive=0
               self->data_subbg  ;restore original signal data
            endelse
          end
        'vdclbut':begin
            widget_control,event.id,sensitive=0
            bkgrmenu = widget_info(self.tlb,find_by_uname='bkgrMenu')
            dm_set_button,widget_info(bkgrmenu,find_by_uname='vandBut'),0
            if self.instrname eq 'macs' then begin
               (*self.eff) = (*self.eff1)
               ptr_free,self.eff1 
               self.ftoolflag[1] = 0b
               dm_set_button,widget_info(bkgrmenu,find_by_uname='vandEscanBut'),0
            endif else $
               ptr_free,self.eff
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end
        'maskfiller':begin   
            tmp = dm_dialog_input(['mask values '+['less','greater']+' than or equal to:','mask applied to:'],/float,default=self.mask_filler,droplist_content=ptr_new(['signal only','both signal and background']),$
                   is_droplist=[0,0,1],xsize=160,info=['Not both fields need to be specified. Clear both fields to disable the masking.'],dialog_parent=self.tlb,cancel=cancel,/return_number)
            if keyword_set(cancel) then return
            if dm_to_string(tmp,separator='&') eq dm_to_string(self.mask_filler,separator='&') then return  ;no change
            self.mask_filler = tmp
            self->data_subbg
          end
        'mkngbut': self->dm_mask_detbanks, 'negative'   ;mask negative angle detectors
        'mkulbut': self->dm_mask_detbanks, 'upperlower' ;mask upper and lower banks
        'mkcnbut': self->dm_mask_detbanks, 'center'     ;mask central bank
        'mklobut': self->dm_mask_detbanks, 'lower'      ;mask lower bank
        'mkupbut': self->dm_mask_detbanks, 'upper'      ;mask upper bank
        'mkptaibut': self->clear_mask,ptaionly=(~self.macs_ptaionly),mask_darkangle=self.mask_darkangle
        'ptaionly':  self->clear_mask,ptaionly=(~self.macs_ptaionly),mask_darkangle=self.mask_darkangle
        'mkdarkbut': begin
            if ~self.mask_darkangle then begin
               name   = ['Dark Angle:','Dark Angle Width:']
               dainfo = ['Dark angle position is relative to the crystal axis u.','Counterclockwise rotation is positive.']
               notok  = 1b
               values = [self.datr_psi,self.datr_width]
               while notok do begin 
                     values = dm_dialog_input(name,title='Dark angle:',default=values,/float,dialog_parent=self.tlb,info=dainfo,cancel=cancel)
                     if keyword_set(cancel) then return
                     notok  = (total(finite(values,/nan)) ne 0)
               endwhile
               self.datr_psi = values[0]
               self.datr_width = abs(values[1])
            endif
            self->clear_mask,/keep_ar,/keep_mask,ptaionly=self.macs_ptaionly,mask_darkangle=(~self.mask_darkangle)
          end
        'mkstdnum':begin ;set mask by detector number
            if ptr_valid(self.detPosPtr) then ndet=n_elements((*self.detPosPtr).two_theta) else ndet=0 
            default = self.maskstr
            ;if the number of exisiting masked detectors is less than 20, save it to the default
            ;if the existing mask is of incompatible type, clear the mask
            if ptr_valid(self.mask) then begin
               exist = 1b
               if (self.masktype eq 0) and (strlen(default) eq 0) then begin
                  n_mask = n_elements(*self.mask)
                  maxmsk = max(*self.mask,min=minmsk)
                  if (n_mask eq (maxmsk-minmsk+1)) and (n_mask gt 3) then $
                      default = dm_to_string(minmsk)+'-'+dm_to_string(maxmsk) $
                  else if n_mask le 20 then $
                      default = dm_to_string(*self.mask,separator=',')
               endif else self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle    ;clear the mask
            endif
            info = 'Detector number starts from 0'+(['',' to '+dm_to_string(ndet-1)+' for '+(*self.ftypename)[self.ftype]])[ndet gt 0]+'.'
            case self.instrname of 
               'dcs':   info = [info,'eg: 370,373 or 370-374c']
               'wand':  info = [info,'eg: 0-8,633-639']
               'macs':  info = [info,'eg: 0-3,6,10']
               else:    info = [info,'eg: 20-28,30,32']
            endcase
            if ptr_valid(self.mask_ar) then info = [info,'','This operation will not remove the existing angular range mask','set in Mask->Set Mask->By Detector Angle.']
            tmp = dm_dialog_input(dialog_parent=self.tlb,'Det num:',title='Masking detectors',default=default,info=info,xsize=200)
            if strlen(tmp) eq 0 then begin
               if keyword_set(exist) then begin
                  self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle
                  ok = dialog_message(['No detector is specified.','Previously set detector number mask has been cleared.'],dialog_parent=self.tlb,/center)
               endif
               break
            endif else begin
               ;search for :- range detectors
               tmpstr  = tmp
               tmpmask = -1L
               repeat begin
                   tmp1 = stregex(tmp,'([0-9]+)[ '+string(9b)+']*[:-][ '+string(9b)+']*([0-9]+)[ '+string(9b)+']*([cCuUlL]*)',/subexpr,length=len)
                   if len[0] gt 0 then begin
                      tmp2 = dm_to_number(strmid(tmp,tmp1[1],len[1]),/long)
                      tmp3 = dm_to_number(strmid(tmp,tmp1[2],len[2]),/long)
                      if len[3] gt 0 then status = strlowcase(strmid(tmp,tmp1[3],1)) else status=''
                      tmp4 = min([tmp2,tmp3])+lindgen(abs(tmp3-tmp2)+1)
                      if (self.instrname eq 'dcs') and (status ne '') then begin
                         tmp4 = dm_common(tmp4,dm_mask_detbanks((where(['l','c','u'] eq status))[0]-1),/no_copy)
                      endif
                      tmpmask = [tmpmask,tmp4]
                      tmp = strmid(tmp,0,tmp1[0])+strmid(tmp,tmp1[0]+len[0],strlen(tmp)-tmp1[0]-len[0])
                   endif
               endrep until len[0] le 0 
               ;individual detectors
               tmp1 = strsplit(tmp,'[a-zA-Z '+string(9b)+',:-]',/regex,/extract)
               tmp2 = dm_to_number(tmp1)
               ind2 = where(finite(tmp2),count2)
               if count2 gt 0 then tmpmask = [tmpmask,dm_to_number(tmp1[ind2],/long)]
               index = where(tmpmask ge 0,count)
               if count gt 0 then mask = tmpmask[index]
            endelse
            if n_elements(mask) gt 0 then begin
               mask = mask[uniq(mask,sort(mask))]
               if ndet gt 0 then ind  = where((mask lt ndet) and (mask ge 0),count) $
               else ind  = where(mask ge 0,count)
               if count ne 0 then begin
                  if ptr_valid(self.mask) then begin
                     if (n_elements(*self.mask) eq count) and (total(abs((*self.mask)-mask[ind])) eq 0) then break
                  endif
                  self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle
                  self.mask = ptr_new(mask[ind])
                  self.maskstr = tmpstr
                  self->my_widget_control,['mksvBut','mkclBut'],/sensitive
                  dm_set_button,event.id,1
               endif 
            endif
            self.masktype = 0b
          end
        'mkstangl':begin ;set mask by detector angle
            if ptr_valid(self.mask_ar) then default = *(self.mask_ar) else default = ''
            if ptr_valid(self.detPosPtr) then det_tt = (*self.detPosPtr).two_theta
            info = 'Use ":" to denote range, "-" only as the negative sign.'
            case self.instrname of 
               'dcs':   info = [info,'eg: <0, 10:20, >120']
               'wand':  info = [info,'eg: <-100, -60:-50,>-20']
               'macs':  begin
                        det_tt = det_tt-76.0  ;kidney=0.0
                        if ptr_valid(self.dataStrPtr) then begin
                           min_kid = min((*self.dataStrPtr).kidney,max=max_kid)
                           det_tt = [det_tt+min_kid,det_tt+max_kid]
                        endif
                        info = [info,'eg: <-10, 20:30,>100']
                        end
               else:    info = [info,'eg: <20, 20:28,>100']
            endcase
            if n_elements(det_tt) ne 0 then begin
               min_tt = min(det_tt,max=max_tt)
               if self.instrname eq 'macs' and ~ptr_valid(self.dataStrPtr) then $
                  info = [info,'Detector angle starts from '+dm_to_string(min_tt)+string('b0'XB)+' to '+dm_to_string(max_tt)+string('b0'XB)+' with kidney=0'+string('b0'XB)+'.'] $
               else $
                  info = [info,'Detector angle starts from '+dm_to_string(min_tt)+string('b0'XB)+' to '+dm_to_string(max_tt)+string('b0'XB)+'.']
            endif
            if ptr_valid(self.mask) then info = [info,'','This operation will not remove the existing detector number mask','set by other masking options.']
            tmp = dm_dialog_input(dialog_parent=self.tlb,'masked angular range:',title='Masking detectors',default=default,info=info,xsize=200,cancel=cancel)
            if keyword_set(cancel) then return
            if strlen(tmp) eq 0 then begin
               if ptr_valid(self.mask_ar) then begin
                  self->clear_mask,/keep_mask,mask_darkangle=self.mask_darkangle    ;clear the mask
                  ok = dialog_message(['No angular range is specified.','Previously set angular range mask has been cleared.'],dialog_parent=self.tlb,/center)
               endif
               break
            endif else begin
               if ptr_valid(self.mask_ar) then begin
                  if (*self.mask_ar) eq tmp then break
                  self->clear_mask,/keep_mask,mask_darkangle=self.mask_darkangle
               endif
               self.mask_ar = ptr_new(tmp)
               dm_set_button,event.id,1
            endelse
            if self.instrname eq 'macs' then self.ftypecheckmask[0] = 1b
            self->my_widget_control,['mksvBut','mkclBut'],/sensitive
          end
        'mka3but':begin ;mask a3 by angle, macs only
            default = ''
            if ptr_valid(self.a3mask) then begin
               exist = 1b
               default = (*self.a3mask)
            endif
            info = ['Use ":" to denote range, "-" only as the negative sign.','eg: <-10, 20:30, >100, 2n+1']
            tmp = dm_dialog_input(dialog_parent=self.tlb,'masked angular range:',title='Masking A3',default=default,info=info,xsize=200,cancel=cancel)
            if keyword_set(cancel) then return
            if strlen(tmp) eq 0 then begin
               if keyword_set(exist) then begin
                  self->clear_mask,/a3
                  ok = dialog_message(['No angular range is specified.','Previously set a3 mask has been cleared.'],dialog_parent=self.tlb,/center)
               endif
               break
            endif else begin
               if keyword_set(exist) then begin
                  if (*self.a3mask) eq tmp then return
               endif
               self->clear_mask,/a3
               self.a3mask = ptr_new(tmp)
               dm_set_button,event.id,1
            endelse
            self->my_widget_control,['mksvBut','mkcla3But'],/sensitive
          end
        'mkbga3but':begin ;mask backgrounda3 by angle, macs only
            default = ''
            if ptr_valid(self.bga3mask) then begin
               exist = 1b
               default = (*self.bga3mask)
            endif
            info = ['Use ":" to denote range, "-" only as the negative sign.','eg: <-10, 20:30, >100, 2n+1','Change the a3 subtraction tolerance value in the Parameter menu.']
            tmp = dm_dialog_input(dialog_parent=self.tlb,'masked angular range:',title='Masking Background A3',default=default,info=info,xsize=200,cancel=cancel)
            if keyword_set(cancel) then return
            if strlen(tmp) eq 0 then begin
               if keyword_set(exist) then begin
                  self->clear_mask,/bga3
                  ok = dialog_message(['No angular range is specified.','Previously set background a3 mask has been cleared.'],dialog_parent=self.tlb,/center)
               endif
               break
            endif else begin
               if keyword_set(exist) then begin
                  if (*self.bga3mask) eq tmp then return
               endif
               self->clear_mask,/bga3
               self.bga3mask = ptr_new(tmp)
               dm_set_button,event.id,1
            endelse
            self->my_widget_control,['mksvBut','mkclbga3But'],/sensitive
          end  
        'mkpldnum':begin ;plot and set mask by detector number
            if (~ ptr_valid(self.dataStrPtr)) then begin  ;must have data
               ok = dialog_message('Please load data files first.',dialog_parent=self.tlb,/center)
               widget_control,event.id,sensitive=0
               break
            endif
            if ptr_valid(self.mask) and ((self.masktype eq 1) and (self.instrname eq 'macs')) then begin
               self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle
               exist = 1b
            endif 
            if self.instrname eq 'macs' then begin
               if keyword_set(self.macshistomode) then begin
                  qty = total((*self.dataStrPtr).qty,2)
                  if self.macsintn_typ eq 2 then $
                     qty = reform(transpose(qty[2*indgen(20),*]+qty[2*indgen(20)+1,*])) $
                  else $
                     qty = reform(transpose(qty[2*indgen(20)+self.macsintn_typ,*]))
               endif else begin
                  if self.macsintn_typ eq 2 then $
                     qty = reform(transpose((*self.dataStrPtr).qty[0,*,*]+(*self.dataStrPtr).qty[1,*,*])) $
                  else $
                     qty = reform(transpose((*self.dataStrPtr).qty[self.macsintn_typ,*,*]))
               endelse
            endif else $
               qty = (*self.dataStrPtr).qty
            case self.samp_typ of
                 0: detsum = total(qty,1,/nan)
                 1: detsum = total(qty,1,/nan)
                 2: detsum = qty[*,0]
                 3: detsum = total(qty[*,*,0],1,/nan)
            endcase
            qty   = 0
            ndet  = n_elements(detsum)
            blist = lonarr(ndet)
            if ptr_valid(self.mask) then begin
               ind = where(*self.mask lt ndet,count)
               if count gt 0 then blist[0:count-1] = (*self.mask)[ind]+1
            endif
            self->plot_mask,detsum,blist,exist=exist
            self.masktype = 0b
            self.maskstr = ''
            dm_set_button,event.id,ptr_valid(self.mask)
            if ptr_valid(self.mask_ar) then ok = dialog_message(['This operation does not remove the existing angular range mask','set in Mask->Set Mask->By Detector Angle.'],dialog_parent=self.tlb,/center,/info)
          end
        'mkplangl':begin ;plot and set mask by angle, only available for DCS, MACS, and WAND
            if (~ ptr_valid(self.dataStrPtr)) then begin  ;must have data
               ok = dialog_message('Please load data files first.',dialog_parent=self.tlb,/center)
               widget_control,event.id,sensitive=0
               break
            endif
            if ptr_valid(self.mask) and ((self.masktype eq 0) and (self.instrname eq 'macs')) then begin
               self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle
               exist = 1b
            endif 
            case self.samp_typ of
                 0: begin
                    if self.instrname eq 'macs' then begin
                       if keyword_set(self.macshistomode) then begin
                          qty = total((*self.dataStrPtr).qty,2)
                          if self.macsintn_typ eq 2 then $
                             detsum = qty[2*indgen(20),*]+qty[2*indgen(20)+1,*] $
                          else $
                             detsum = qty[2*indgen(20)+self.macsintn_typ,*]
                          qty = 0
                       endif else begin
                          if self.macsintn_typ eq 2 then $
                             detsum = (*self.dataStrPtr).qty[0,*,*]+(*self.dataStrPtr).qty[1,*,*] $
                          else $
                             detsum = (*self.dataStrPtr).qty[self.macsintn_typ,*,*]
                       endelse
                    endif else begin
                       detsum = total((*self.dataStrPtr).qty,1,/nan)
                    endelse
                    end
                 1: begin
                    if self.instrname eq 'macs' then begin
                       if keyword_set(self.macshistomode) then begin
                          qty = total((*self.dataStrPtr).qty,2)
                          if self.macsintn_typ eq 2 then $
                             detsum = qty[2*indgen(20),*]+qty[2*indgen(20)+1,*] $
                          else $
                             detsum = qty[2*indgen(20)+self.macsintn_typ,*]
                          qty = 0
                       endif else begin
                          if self.macsintn_typ eq 2 then $
                             detsum = (*self.dataStrPtr).qty[0,*,*]+(*self.dataStrPtr).qty[1,*,*] $
                          else $
                             detsum = (*self.dataStrPtr).qty[self.macsintn_typ,*,*]
                       endelse
                    endif else begin
                       detsum = total((*self.dataStrPtr).qty,1,/nan)
                    endelse
                    end
                 2: detsum = (*self.dataStrPtr).qty[*,0]
                 3: detsum = total((*self.dataStrPtr).qty[*,*,0],1,/nan)
            endcase
            ndet  = n_elements(detsum)
            blist = lonarr(ndet)
            if ptr_valid(self.mask) then begin
               ind = where(*self.mask lt ndet,count)
               if count gt 0 then blist[0:count-1] = (*self.mask)[ind]+1
            endif
            if self.instrname eq 'dcs' then begin
               self->plot_mask,detsum,blist,/angle,exist=exist
            endif else begin
               case self.instrname of
                   'macs':begin
                          n_kid = n_elements((*self.dataStrPtr).kidney)
                          n_det = n_elements((*self.detPosPtr).two_theta)
                          det_tt = ((*self.detPosPtr).two_theta)#(fltarr(n_kid)+1)
                          for i=0L,n_det-1 do det_tt[i,*] = (*self.dataStrPtr).kidney-76.0+(*self.detPosPtr).two_theta[i]
                          detsum = reform(detsum,n_det*n_kid,/overwrite)
                          det_tt = reform(det_tt,n_det*n_kid,/overwrite)
                          end 
                   else:  det_tt = (*self.detPosPtr).two_theta
               endcase
               self->plot_mask,detsum,blist,/angle,det_tt=det_tt,exist=exist
            endelse
            self.masktype = 1b
            dm_set_button,event.id,ptr_valid(self.mask)
            if self.instrname eq 'macs' then self.ftypecheckmask[0] = (self.macs_ptaionly or (self.macshistomode eq 2) or ptr_valid(self.mask_ar) or ptr_valid(self.mask) or self.mask_darkangle)
            if ptr_valid(self.mask_ar) then ok = dialog_message(['This operation does not remove the existing angular range mask','set in Mask->Set Mask->By Detector Angle.'],dialog_parent=self.tlb,/center,/info)
          end
        'mkbpbut':begin ;automatically mask bragg peaks
            if (~ ptr_valid(self.dataStrPtr)) then begin  ;must have data
               ok = dialog_message('Please load data files first.',dialog_parent=self.tlb,/center)
               widget_control,event.id,sensitive=0
               break
            endif
            if ptr_valid(self.mask) or ptr_valid(self.mask_ar) then begin
               ok = dialog_message('This operation will remove all the existing masks. Do you want to continue?',/question,dialog_parent=self.tlb,/center)
               if strmatch(ok,'No',/fold_case) then break
               self->clear_mask     ;clear the mask
               exist = 1b
            endif
            case self.samp_typ of
                 0: detsum = total((*self.dataStrPtr).qty,1,/nan)
                 1: detsum = total((*self.dataStrPtr).qty,1,/nan)
                 2: detsum = (*self.dataStrPtr).qty[*,0]
                 3: detsum = total((*self.dataStrPtr).qty[*,*,0],1,/nan)
            endcase
            case self.samp_typ of
                 0: errsum = sqrt(total((*self.dataStrPtr).err^2,1,/nan))
                 1: errsum = sqrt(total((*self.dataStrPtr).err^2,1,/nan))
                 2: errsum = (*self.dataStrPtr).err[*,0]
                 3: errsum = sqrt(total((*self.dataStrPtr).err[*,*,0]^2,1,/nan))
            endcase
            ;sort detectors by 2theta angles
            if (self.instrname ne 'dcs') and ptr_valid(self.detPosPtr) then begin
               tmpind = sort((*self.detPosPtr).two_theta)
               oldsum = detsum
               detsum = detsum[tmpind]
               errsum = errsum[tmpind]
            endif
            ;masking
            self->maskbraggpeak,detsum,errsum,mask
            if (self.instrname ne 'dcs') and ptr_valid(self.detPosPtr) then begin
               detsum = temporary(oldsum)
               if n_elements(mask) ne 0 then  mask = tmpind[mask] 
               tmpind = 0
            endif
            if n_elements(mask) eq 0  then begin
               mask = [0]     ; no mask
            endif else begin
               mask = mask+1l ;+1 because the mask starts from 1 in plot_mask
            endelse
            if n_elements(mask) eq n_elements(detsum) then mask = [0] ; no mask
            self->plot_mask,detsum,mask,exist=exist
            self.masktype = 0b
            self.maskstr = ''
            dm_set_button,event.id,ptr_valid(self.mask)
            dm_set_button,widget_info(self.tlb,find_by_uname='mkstdnum'),ptr_valid(self.mask)
          end
        'mkrdbut':begin ;read mask file
            path = self.dirs[1]
            file = dm_choose_file('',/read,dialog_parent=self.tlb,title='Read mask file:',path=path)
            if strlen(file) eq 0 then return
            self->setproperty,workDir=path
            if ptr_valid(self.mask) or ptr_valid(self.mask_ar) then exist = 1b
            if ptr_valid(self.a3mask) then a3exist = 1b
            if ptr_valid(self.bga3mask) then bga3exist = 1b
            self->clear_mask        ;clear the detector mask
            self->clear_mask,/a3    ;clear the a3 mask
            self->clear_mask,/bga3  ;clear the background a3 mask
            openr,unit,file,/get_lun
            tmp = ''
            self.masktype = 255b  
            masktype = 0
            while(~ eof(unit)) do begin
                readf,unit,tmp
                tmp = strtrim(tmp,2)
                if strlen(tmp) ne 0 then begin
                   if stregex(tmp,'#type',/boolean,/fold_case) then begin
                      masktype = dm_to_number(strmid(tmp,0,1,/reverse),/int)
                      if masktype lt 2 then self.masktype = 0>(masktype)<1 
                      continue
                   endif
                   if masktype eq 2 then begin
                      if n_elements(armask) eq 0 then armask=tmp else armask=armask+' '+tmp 
                   endif else if masktype eq 3 then begin
                      if n_elements(a3mask) eq 0 then a3mask=tmp else a3mask=a3mask+' '+tmp
                   endif else if masktype eq 4 then begin
                      tmp1 = strsplit(tmp,' ,;'+string(9b),/extract,count=count1)
                      self.mask_darkangle = dm_to_number(tmp1[0],/int)
                      if count1 ge 3 then begin
                         self.datr_psi = dm_to_number(tmp1[1])
                         self.datr_width = dm_to_number(tmp1[2])
                      endif
                   endif else if masktype eq 5 then begin
                      self.macs_ptaionly = dm_to_number(tmp,/int)
                   endif else if masktype eq 6 then begin
                      if n_elements(bga3mask) eq 0 then bga3mask=tmp else bga3mask = bga3mask+' '+tmp
                   endif else begin
                      if ~stregex(tmp,'^ *[0-9><.,-]+',/boolean) then continue  ;make sure it's not a comment line
                      if n_elements(mask) eq 0 then mask=tmp else mask=[mask,tmp]
                   endelse         
                endif
            endwhile
            free_lun,unit
            if n_elements(mask) ne 0 then begin
               if (self.masktype eq 255) then begin  ;ask for the mask type
                  self.masktype = dm_dialog_input('mask type:',default=0,droplist_content=ptr_new(['detector number','detector angle index','detector angular range']),is_droplist=[1],$
                     /return_number,xsize=200,dialog_parent=self.tlb,info=['Detector angle index type is from Mask->Plot Mask->By Detector Angle.',$
                     'Detector angular range type is from Mask->Set Mask->By Detector Angle.'])
               endif
               if (self.masktype ne 2) then begin  
                  tmp = temporary(mask)
                  for j=0L,n_elements(tmp)-1 do begin
                      tmp1 = strsplit(tmp[j],' ,;'+string(9b),/extract,count=count1)
                      if finite(dm_to_number(tmp1[0])) then begin 
                         for i=0L,count1-1L do begin
                             tmp2 = strsplit(tmp1[i],'-:',/extract,count=count2)
                             tmp2 = dm_to_number(tmp2,/long)
                             if count2 eq 2 then $
                                tmp2 = min(tmp2)+lindgen(abs(tmp2[1]-tmp2[0])+1)
                             if n_elements(mask) eq 0 then mask=tmp2 else mask=[mask,tmp2]
                         endfor
                      endif 
                  endfor
                  if n_elements(mask) ne 0 then begin
                     ok = dm_dialog_input('det number starts from:',default=0,droplist_content=ptr_new(['0','1']),is_droplist=[1],/return_number,dialog_parent=self.tlb)
                     mask = mask-ok
                  endif
               endif else begin
                  for i=1L,n_elements(mask)-1 do mask[0] = mask[0]+' '+mask[i]
                  armask = mask[0] & tmp = temporary(mask)
               endelse
            endif 
            if (n_elements(mask) ne 0) or (n_elements(armask) ne 0) then begin
               if n_elements(mask) ne 0 then self.mask = ptr_new(mask)
               if n_elements(armask) ne 0 then self.mask_ar = ptr_new(armask)
            endif else if keyword_set(exist) then begin
               ok = dialog_message('Previously set detector mask has been cleared.',dialog_parent=self.tlb,/center)
            endif
            if n_elements(a3mask) ne 0 then begin
               self.a3mask = ptr_new(a3mask)
            endif else if keyword_set(a3exist) then begin
               ok = dialog_message('Previously set a3 mask has been cleared.',dialog_parent=self.tlb,/center)
            endif
            if n_elements(bga3mask) ne 0 then begin
               self.bga3mask = ptr_new(bga3mask)
            endif else if keyword_set(bga3exist) then begin
               ok = dialog_message('Previously set background a3 mask has been cleared.',dialog_parent=self.tlb,/center)
            endif
            self->my_widget_control,'mksvBut',sensitive=(ptr_valid(self.mask) or ptr_valid(self.a3mask) or ptr_valid(self.mask_ar))
            self->my_widget_control,'mkclBut',sensitive=(((self.instrname eq 'macs') and self.macs_ptaionly) or ptr_valid(self.mask_ar) or ptr_valid(self.mask)) or self.mask_darkangle
            if self.instrname eq 'macs' then begin
               self->my_widget_control,'mkcla3But',sensitive=ptr_valid(self.a3mask)
               self->my_widget_control,'mkclbga3But',sensitive=ptr_valid(self.bga3mask)
               dm_set_button,widget_info(self.tlb,find_by_uname='mka3But'),ptr_valid(self.a3mask)
               dm_set_button,widget_info(self.tlb,find_by_uname='mkbga3But'),ptr_valid(self.bga3mask)
               dm_set_button,widget_info(self.tlb,find_by_uname='mkptaiBut'),self.macs_ptaionly,onstring='View PTAI Only',offstring='View All'
               self->my_widget_control,'ptaionly',self.fileBas,set_button=self.macs_ptaionly
            endif
            self.maskstr = ''
            sensitive = ((self.samp_typ ge 2) or ((self.instrname eq 'macs') and (self.samp_typ ne 0)))
            if ~sensitive and self.mask_darkangle then self.mask_darkangle = 0b
            dm_set_button,widget_info(self.tlb,find_by_uname='mkstangl'),ptr_valid(self.mask_ar)
            dm_set_button,widget_info(self.tlb,find_by_uname='mkdarkBut'),self.mask_darkangle,onstring='Mask Dark Angle...',offstring='Not Mask Dark Angle...'
            if self.masktype eq 0 then dm_set_button,widget_info(self.tlb,find_by_uname='mkstdnum'),ptr_valid(self.mask)
            if self.masktype eq 1 then dm_set_button,widget_info(self.tlb,find_by_uname='mkplangl'),ptr_valid(self.mask)
            self.ftypecheckmask[0] = (((self.instrname eq 'macs') and (self.macs_ptaionly or (self.macshistomode eq 2) or ptr_valid(self.mask_ar) or (ptr_valid(self.mask) and (self.masktype eq 1)))) or self.mask_darkangle or self.ftypecheckmask[1])
          end
        'mksvbut':if ptr_valid(self.mask) or ptr_valid(self.mask_ar) or ptr_valid(self.a3mask) or ptr_valid(self.bga3mask) then begin ;save mask in a file, including a3 mask   
            path = self.dirs[1]
            file = dm_choose_file('msk',/write,dialog_parent=self.tlb,title='Save mask in an ASCII file.',path=path)
            if strlen(file) eq 0 then break
            self->setproperty,workDir=path
            openw,unit,file,/get_lun,error=openerr
            if openerr ne 0 then begin
               ok = dialog_message("Can't write in "+file[0],/error,dialog_parent=self.tlb,/center)
               break
            endif else begin
               widget_control,/hourglass  ;busy signal
            endelse
            if ptr_valid(self.mask) then begin
               printf,unit,'#type '+dm_to_string(self.masktype)
               printf,unit,*(self.mask)   
            endif
            if ptr_valid(self.mask_ar) then begin
               printf,unit,'#type 2'  ;2 to denote anuglar range mask
               printf,unit,*(self.mask_ar)
            endif
            if ptr_valid(self.a3mask) then begin
               printf,unit,'#type 3'  ;3 to denote a3 mask
               printf,unit,*(self.a3mask)   
            endif
            printf,unit,'#type 4'     ;4 to denote mask dark angle
            printf,unit,self.mask_darkangle,self.datr_psi,self.datr_width
            if (self.instrname eq 'macs') then begin
               printf,unit,'#type 5'  ;5 to denote ptaionly mask
               printf,unit,self.macs_ptaionly
            endif
            if ptr_valid(self.bga3mask) then begin
               printf,unit,'#type 6'  ;6 to denote background a3 mask
               printf,unit,*(self.bga3mask) 
            endif
            free_lun,unit
          end
        'mkclbut':    self->clear_mask         ;clear the mask
        'mkcla3but':  self->clear_mask,/a3     ;clear the a3 mask
        'mkclbga3but':self->clear_mask,/bga3   ;clear the background a3 mask
        'directgeom': if self.instrgeom ne 0 then self->setproperty,instrgeom=0
        'inversegeom':if self.instrgeom ne 1 then self->setproperty,instrgeom=1
        'intntyp0':   if self.intn_typ ne 0  then self->setproperty,intntype=0
        'intntyp1':   if self.intn_typ ne 1  then self->setproperty,intntype=1
        'intntyp2':   if self.intn_typ ne 2  then self->setproperty,intntype=2
        'intntyp3':   if self.intn_typ ne 3  then self->setproperty,intntype=3
        'intntyp4':   if self.intn_typ ne 4  then self->setproperty,intntype=4
        'intntyp5':   if self.intn_typ ne 5  then self->setproperty,intntype=5
        'intntyp6':   if self.intn_typ ne 6  then self->setproperty,intntype=6
        'intntyp7':   if self.intn_typ ne 7  then self->setproperty,intntype=7
        'intneint': begin
            self.eint_yn = ~self.eint_yn
            dm_set_button,event.id,self.eint_yn,onstring='E-Integrated Intensity Enabled',offstring='E-Integrated Intensity Disabled'
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end
        'macsintspec': if self.macsintn_typ ne 0 then begin
            self.macsintn_typ = 0
            self->my_menu_toggle,'macsint'+['SPEC','DIFF','SPDF'],self.macsintn_typ
            self->my_widget_control,'macsintMenu',set_value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')'
            self->set_display_button,clearprojection=(self.samp_typ eq 1)
          end  
        'macsintdiff': if self.macsintn_typ ne 1 then begin
            self.macsintn_typ = 1
            self->my_menu_toggle,'macsint'+['SPEC','DIFF','SPDF'],self.macsintn_typ
            self->my_widget_control,'macsintMenu',set_value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')'
            self->set_display_button,clearprojection=(self.samp_typ eq 1)
          end  
        'macsintspdf': if self.macsintn_typ ne 2 then begin
            self.macsintn_typ = 2
            self->my_menu_toggle,'macsint'+['SPEC','DIFF','SPDF'],self.macsintn_typ
            self->my_widget_control,'macsintMenu',set_value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')'
            self->set_display_button,clearprojection=(self.samp_typ eq 1)
          end
        'macshisto0': if self.macshistomode ne 0 then begin
             mesg = ['Reverting to regular data mode. Histogram data in the data files will be discarded.','']
             self.macshistomode = 0
             self->my_menu_toggle,'macshisto'+['0','1','2'],self.macshistomode
             self->my_widget_control,'macshisMenu',set_value='MACS Data Mode ('+(['Normal','Histogram','Event'])[self.macshistomode]+')'
             self->my_widget_control,'macshistowidBut',sensitive=(self.macshistomode ne 0)
             if ptr_valid(self.dataStrPtr) or ptr_valid(self.bgdata[0]) then self->clear_data,/clear_bg,infomsg=mesg
             ptr_free,self.projStrPtr
             if self.mon_typ[1] eq 2 then begin ;pulse number can't be the monitor
                self.mon_typ[1] = 1
                self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[1]
                self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')'
                self->reset_intnlab
             endif
             self->my_widget_control,'monMACSPULSE',sensitive=0
             self->reset_pdt
             ok = dialog_message(mesg,dialog_parent=self.tlb,/center,/info)
          end
        'macshisto1': if self.macshistomode ne 1 then begin
            new = [self.macshistowidth,self.macshistounit,self.macshistonchan,self.macshistocut]
            repeat begin
               new = dm_dialog_input(['histogram channel '+['width:','unit:','number:'],'throwing away trailing channels:'],default=new,dialog_parent=self.tlb,cancel=cancel,is_droplist=[0,1,0,1],xsize=90,$
                     droplist_content=[ptr_new(['microsecond','millisecond']),ptr_new(['no','yes'])],info=['Regular SPEC & DIFF data in the data files will be discarded','in histogram mode.'],/return_number)
               if keyword_set(cancel) then return
               if finite(new[0]) and finite(new[2]) then ok = ((new[0] gt 0) and (new[2] gt 0))
            endrep until keyword_set(ok)
            self.macshistot0[*] = 0
            self.macshistowidth = new[0]
            self.macshistounit  = round(new[1])
            self.macshistonchan = round(new[2])
            self.macshistocut   = round(new[3])
            self.macshistomode  = 1
            self->my_menu_toggle,'macshisto'+['0','1','2'],self.macshistomode
            self->my_widget_control,'macshisMenu',set_value='MACS Data Mode ('+(['Normal','Histogram','Event'])[self.macshistomode]+')'
            self->my_widget_control,'macshistowidBut',sensitive=(self.macshistomode ne 0)
            if ptr_valid(self.dataStrPtr) or ptr_valid(self.bgdata[0]) then self->clear_data,/clear_bg
            ptr_free,self.projStrPtr
            self->my_widget_control,'monMACSPULSE',/sensitive
            self->reset_pdt            
          end
        'macshisto2': if self.macshistomode ne 2 then begin
            self.macshistocut   = 0   ;keep all channels
            self.macshistounit  = 1   ;millisecond
            self.macshistomode  = 2
            self->my_menu_toggle,'macshisto'+['0','1','2'],self.macshistomode
            self->my_widget_control,'macshisMenu',set_value='MACS Data Mode ('+(['Normal','Histogram','Event'])[self.macshistomode]+')'
            self->my_widget_control,'macshistowidBut',sensitive=(self.macshistomode ne 0)
            if ptr_valid(self.dataStrPtr) or ptr_valid(self.bgdata[0]) then self->clear_data,/clear_bg
            ptr_free,self.projStrPtr
            self->my_widget_control,'monMACSPULSE',/sensitive
            self->reset_pdt
            mesg = '2D binary .rgo histogram data files must be present in the data directory.'
            if self.bin_avgsum ne ([-1,1])[self.macs_rawintn] then begin
               mesg = [mesg,'Binning method has been switched from '+(['Weighted Mean','Arithmetic Mean','Sum'])[self.bin_avgsum+1]+' to '+(['Weighted Mean','Sum'])[self.macs_rawintn]+'.']
               self.bin_avgsum = ([-1,1])[self.macs_rawintn]
               self.bin_extlib = 0b
               self->my_menu_toggle,'bin'+['WeighMean','ArithMean','Sum'],self.bin_avgsum+1
               dm_set_button,widget_info(self.tlb,find_by_uname='binAvg'),(self.bin_avgsum ne 1),set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
               dm_set_button,widget_info(self.tlb,find_by_uname='binexLib'),self.bin_extlib,onstring='Use External Binning Library',offstring='Use Internal Binning Routine'
            endif
            if (self.bin_zeroerror ne 1) and (self.macs_rawintn eq 0) then begin
               mesg = [mesg,'Zero intensity error bar calculation has been switched to statistical estimate.']
               self.bin_zeroerror = 1
               self->my_menu_toggle,'zerr'+['userspec','statest'],self.bin_zeroerror
               self->my_widget_control,'zerrMenu',set_value='Zero Intensity Error ('+(['User Specify','Statistical Estimate'])[self.bin_zeroerror]+')'
            endif
            if self.macs_rawintn then $
               mesg = [mesg,'','The intensity is not normalized. The detector efficiency is applied to the weight. The rebinned intensity and weight are for output only. You can switch to normalized intensity in the popup menu.'] 
            ok = dialog_message(mesg,dialog_parent=self.tlb,/center,/info)
          end     
        'macshistowidbut': begin
            new = [self.macshistowidth,self.macshistounit,self.macshistonchan,self.macshistocut]
            old1 = dm_to_string(new[0:1],sep='&') & old2 = dm_to_string(new[2:3],sep='&')
            repeat begin
               new = dm_dialog_input(['histogram channel '+['width:','unit:','number:'],'throwing away trailing channels:'],default=new,dialog_parent=self.tlb,cancel=cancel,is_droplist=[0,1,0,1],xsize=90,$
                     droplist_content=[ptr_new(['microsecond','millisecond']),ptr_new(['no','yes'])],info=['Regular SPEC & DIFF data in the data files will be discarded','in histogram mode.'],/return_number)
               if keyword_set(cancel) then return
               if finite(new[0]) and finite(new[2]) then ok = ((new[0] gt 0) and (new[2] gt 0))
            endrep until keyword_set(ok)
            if keyword_set(ok) then begin
               reset = (self.macshistounit ne new[1]) ;unit has changed
               self.macshistowidth = new[0]
               self.macshistounit  = round(new[1])
               self.macshistonchan = round(new[2])
               self.macshistocut   = round(new[3])
               if reset then begin
                  self->my_widget_control,'calcInput',self.dispBas,get_value=calcstr
                  self->reset_display,/reset,calcstr=calcstr
               endif
               if (dm_to_string([self.macshistonchan,self.macshistocut],sep='&') ne old2) and (ptr_valid(self.dataStrPtr) or ptr_valid(self.bgdata[0])) then self->clear_data,/clear_bg $
               else if dm_to_string([self.macshistowidth,self.macshistounit],sep='&') ne old1 then recalc_proj = 1b
            endif
          end   
        'binweighmean': if self.bin_avgsum ne -1 then begin
            if self.bin_extlib then begin ;weighted mean not implemented in external library
               ok = dialog_message(['Weighted mean is currently not implemented in the external library.','Do you want to disable the external library?'],/question,dialog_parent=self.tlb,/center)
               if ok eq 'No' then begin
                  ok = dialog_message('The binning method has not been changed.',dialog_parent=self.tlb,/center,/info)
                  break
               endif
               self.bin_extlib = 0
               dm_set_button,widget_info(self.tlb,find_by_uname='binexLib'),self.bin_extlib,onstring='Use External Binning Library',offstring='Use Internal Binning Routine'
            endif
            self.bin_avgsum = -1
            self->my_menu_toggle,'bin'+['Avg','Sum'],0>(self.bin_avgsum)
            self->my_menu_toggle,'bin'+['WeighMean','ArithMean'],(self.bin_avgsum+1)
            self->my_widget_control,'binAvg',set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1] 
          end  
        'binarithmean': if self.bin_avgsum ne 0 then begin
            self.bin_avgsum = 0
            self->my_menu_toggle,'bin'+['Avg','Sum'],0>(self.bin_avgsum)
            self->my_menu_toggle,'bin'+['WeighMean','ArithMean'],(self.bin_avgsum+1)
            self->my_widget_control,'binAvg',set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
          end
        'binsum':if self.bin_avgsum ne 1 then begin
            self.bin_avgsum = 1
            self->my_menu_toggle,'bin'+['Sum','Avg','ArithMean','WeighMean'],0
            self->my_widget_control,'binAvg',set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
          endif
        'bincs': begin
            self.bin_conststep = ~self.bin_conststep
            dm_set_button,event.id,self.bin_conststep,onstring='Constant Intervals',offstring='Nonconstant Intervals'
          end 
        'binexlib': self->setproperty,binextlib=(~self.bin_extlib)
        'zerruserspec': begin
            new = dm_dialog_input('zero error:',title='Change zero error',dialog_parent=self.tlb,default=self.zero_error[0],info='This value is the error bar for zero count in the raw data.',/float,cancel=cancel)
            if keyword_set(cancel) or finite(new,/nan) then return
            self.bin_zeroerror = 0
            self->my_menu_toggle,'zerr'+['userspec','statest'],self.bin_zeroerror
            self->my_widget_control,'zerrMenu',set_value='Zero Intensity Error ('+(['User Specify','Statistical Estimate'])[self.bin_zeroerror]+')'
            self.zero_error[0] = abs(new)
            widget_control,event.id,set_value='User Specify'+(['',' ('+dm_to_string(self.zero_error[0])+')'])[finite(self.zero_error[0])]
            self->my_widget_control,['statranMenu','statall'],sensitive=self.bin_zeroerror
          end
        'zerrstatest': if self.bin_zeroerror eq 0 then begin
            self.bin_zeroerror = 1
            self->my_menu_toggle,'zerr'+['userspec','statest'],self.bin_zeroerror
            self->my_widget_control,'zerrMenu',set_value='Zero Intensity Error ('+(['User Specify','Statistical Estimate'])[self.bin_zeroerror]+')'
            self->my_widget_control,['statranMenu','statall'],sensitive=self.bin_zeroerror
          endif
        'statranauto': if self.bin_statrange[0] ne 0 then begin
            self.bin_statrange[0] = 0
            self->my_menu_toggle,'statran'+['auto','userspec'],self.bin_statrange[0]
            self->my_widget_control,'statranMenu',set_value='Estimate Size ('+(['Automatic','User Specify'])[self.bin_statrange[0]]+')
          endif
        'statranuserspec': begin
            info = 'This value is the number of neighboring pixels in each direction. It must be no less than 1.'
            if self.bin_statrange[3] ne 0 then info = [info,'The suggested value from previous auto estimate in '+(['cut','slice','volume','4d'])[self.bin_statrange[4]]+' rebinning is '+dm_to_string(self.bin_statrange[3])+'.']
            new = dm_dialog_input('estimate size:',title='Please specify',dialog_parent=self.tlb,default=self.bin_statrange[1],/int,info=info,cancel=cancel)
            if keyword_set(cancel) or (new eq 0) then return
            self.bin_statrange[0] = 1
            self.bin_statrange[1] = abs(new)
            self->my_menu_toggle,'statran'+['auto','userspec'],self.bin_statrange[0]
            self->my_widget_control,'statranMenu',set_value='Estimate Size ('+(['Automatic','User Specify'])[self.bin_statrange[0]]+')
            widget_control,event.id,set_value='User Specify ('+dm_to_string(self.bin_statrange[1])+')'
          end
        'statall': begin
            self.bin_statrange[2] = 1-self.bin_statrange[2]
            widget_control,event.id,set_button=self.bin_statrange[2]
          end
        'monsum':if self.mon_sumfit[mon_instr] ne 0 then begin
            self.mon_sumfit[mon_instr] = 0
            self->my_menu_toggle,'mon'+['Sum','Fit'],self.mon_sumfit[mon_instr]
            if (self.instrname eq 'macs') then begin
               if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
               self->reset_intnlab
            endif
            if (self.instrname eq 'dcs') then self->clear_data
          endif
        'monfit':begin
            if self.mon_sumfit[mon_instr] eq 0 then begin
               if (self.instrname eq 'macs') and ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
               if (self.instrname eq 'dcs') then self->clear_data
            endif
            self.mon_sumfit[mon_instr] = 1
            self->my_menu_toggle,'mon'+['Sum','Fit'],self.mon_sumfit[mon_instr]
            if self.instrname eq 'macs' then begin
               old = self.macs_mon[1,self.mon_typ[mon_instr]]
               tmp = (['.',' at normalized intensity mode.'])[keyword_set(self.macshistomode) and self.macs_rawintn]
               repeat begin
                  case self.mon_typ[mon_instr] of
                       0: new = dm_dialog_input('monitor time (sec):',default=dm_to_string(old),dialog_parent=self.tlb,info='Intensity is normalized to this monitor time'+tmp,/float)
                       1: new = dm_dialog_input('monitor count:',default=dm_to_string(old,/exponent),dialog_parent=self.tlb,info='Intensity is normalized to this monitor count'+tmp,/float)
                       2: new = dm_dialog_input('pulse number:',default=dm_to_string(old,/int),dialog_parent=self.tlb,info='Intensity is normalized to this pulse number'+tmp,/int)
                       else:
                  endcase
                  if finite(new) then begin
                     if new gt 0 then ok2 = 1b
                  endif
               endrep until keyword_set(ok2)
               if new ne old then begin
                  self.macs_mon[1,self.mon_typ[mon_instr]] = new
                  if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
               endif
               self->reset_intnlab
            endif
          end
        'monfc0':if self.mon_typ[mon_instr] ne 0 then begin
            self.mon_typ[mon_instr] = 0
            self->my_menu_toggle,'mon'+['FC0','BM1','TIME'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['FC0','BM1','Duration'])[self.mon_typ[mon_instr]]+')'
            self->clear_data
          endif
        'monbm1':if self.mon_typ[mon_instr] ne 1 then begin
            self.mon_typ[mon_instr] = 1
            self->my_menu_toggle,'mon'+['FC0','BM1','TIME'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['FC0','BM1','Duration'])[self.mon_typ[mon_instr]]+')'
            self->clear_data
          endif
        'montime':if self.mon_typ[mon_instr] ne 2 then begin
            self.mon_typ[mon_instr] = 2
            self->my_menu_toggle,'mon'+['FC0','BM1','TIME'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['FC0','BM1','Duration'])[self.mon_typ[mon_instr]]+')'
            self->clear_data
          endif
        'monmacstime':if self.mon_typ[mon_instr] ne 0 then begin
            self.mon_typ[mon_instr] = 0
            self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')'
            self->reset_intnlab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
          endif 
        'monmacsmonitor':if self.mon_typ[mon_instr] ne 1 then begin
            self.mon_typ[mon_instr] = 1
            self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')'
            self->reset_intnlab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
          endif
        'monmacspulse':if self.mon_typ[mon_instr] ne 2 then begin
            self.mon_typ[mon_instr] = 2
            self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[mon_instr]
            self->my_widget_control,'mondMenu',set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')'
            self->reset_intnlab
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
          endif     
        'monlamb2': begin
            self.macs_lamb2 = ~self.macs_lamb2
            dm_set_button,event.id,self.macs_lamb2,onstring='Apply Monitor Lambda/2 Correction',offstring='No Monitor Lambda/2 Correction'
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
          end
        'abspnone':     self->absorption,-1
        'abspcylinder': self->absorption,0
        'abspannulus':  self->absorption,1
        'abspsphere':   self->absorption,2
        'abspflat':     self->absorption,3
        'abspl1factor': begin
            new = dm_dialog_input('incident beam path factor:',default=self.absp_l1factor,/float,info='This factor will be multiplied to the incident beam path length in absorption calculation.',dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(new) then self.absp_l1factor = abs(new)
          end
        'abspint':  begin
            self.absp_asint = ~self.absp_asint
            dm_set_button,event.id,self.absp_asint
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
          end
        'calcl0l1': begin
            if self.samp_typ gt 0 then default = [self.absp_l0l1[0:2],dm_to_string(self.latt_parm,sep=', ',res=4,/nozeroend)] else default = [self.absp_l0l1[0:2],'']
            answer = dm_dialog_input(['chemical formula','density (g/cm^3)','# of formula in unit cell','lattice parameters']+':',default=default,dialog_parent=self.tlb,xsize=200,$
                     info=['Elements in the chemical formula need to be separated by space. Isotope is','represented by the mass number preceding the element name, e.g., U Ni4 11B.',$
                           'Number of formula in a unit cell and the lattice parameters are used to calculate','the density if the density value is not given.'],cancel=cancel)
            if keyword_set(cancel) or (strlen(strtrim(answer[0])) eq 0 ) then return
            self.absp_l0l1[0:2] = answer[0:2] 
            nform = dm_to_number(answer[2]) 
            rho   = dm_to_number(answer[1])
            if finite(nform,/nan) then tmp = temporary(nform)
            if finite(rho,/nan)   then tmp = temporary(rho)
            self.absp_l0l1[3:4] = dm_calc_lila(answer[0],nform=nform,rho=rho,lattparm=dm_to_number(strsplit(answer[3],' ,',/extract)))
            ok = dialog_message(['L0 = '+dm_to_string(self.absp_l0l1[3])+' cm','L1 =  '+dm_to_string(self.absp_l0l1[4])+' cm','rho = '+dm_to_string(rho,res=4)+' g/cm^3'],$
                 dialog_parent=self.tlb,/info,title='Result:')
          end  
        'vaxisarbi': if self.view_ortharbi ne 1 then begin
            self.view_ortharbi = 1
            self->my_menu_toggle,'vaxis'+['Ortho','Arbi'],self.view_ortharbi 
            self->my_widget_control,'viewAxis',set_value='  Viewing Axes:'
          endif
        'vaxisortho':if self.view_ortharbi ne 0 then begin
            self.view_ortharbi = 0
            self->my_menu_toggle,'vaxis'+['Ortho','Arbi'],self.view_ortharbi 
            self->my_widget_control,'viewAxis',set_value='  Orthogonal Viewing Axes:'
          endif          
        'mrrf_one': if self.fold_type ne 0 then begin
            reset_fold = (1b)+(self.fold_type gt 1)
            self.fold_type = 0
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          endif
        'mrrf_both': if self.fold_type ne 1 then begin
            reset_fold = (1b)+(self.fold_type gt 1)
            self.fold_type = 1
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b
          endif  
        'symrot_section':if self.fold_type ne 2 then begin
            reset_fold = 2*(self.fold_type lt 2)
            self.fold_type = 2
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          endif
        'symrot_quadrant':if self.fold_type ne 3 then begin
            reset_fold = 2*(self.fold_type lt 2)
            self.fold_type = 3
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          endif
        'symrot_fullmap':if self.fold_type ne 4 then begin
            reset_fold = 2*(self.fold_type lt 2)
            self.fold_type = 4
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          endif 
        'symvec_u1': if self.symrot_startvec ne 0 then begin
            self.symrot_startvec = 0.0
            self->my_menu_toggle,'symvec_'+['U1','Specify'],0
            if (self.fold_type eq 2) and ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end
        'symvec_specify': begin
            old = self.symrot_startvec
            self.symrot_startvec = dm_dialog_input('angle from u1:',default=old,dialog_parent=self.tlb,info='Counterclockwise angle is positive.')
            if finite(self.symrot_startvec,/nan) then self.symrot_startvec = 0
            self->my_menu_toggle,'symvec_'+['U1','Specify'],(self.symrot_startvec ne 0)
            if (self.fold_type eq 2) and ptr_valid(self.projStrPtr) and (self.symrot_startvec ne old) then recalc_proj = 1b 
          end
        'nmrt_min': if self.numrot_optn ne 0 then begin
            self.numrot_optn = 0
            self->my_menu_toggle,'nmrt_'+['Min','Nminus1'],self.numrot_optn
            if (self.fold_type eq 4) and ptr_valid(self.projStrPtr) then recalc_proj = 1b  
          end
        'nmrt_nminus1': if self.numrot_optn ne 1 then begin
            self.numrot_optn = 1
            self->my_menu_toggle,'nmrt_'+['Min','Nminus1'],self.numrot_optn
            if (self.fold_type eq 4) and ptr_valid(self.projStrPtr) then recalc_proj = 1b
          end      
        'powdernone': self->change_powderline,0
        'powderal':   self->change_powderline,1
        'powdercu':   self->change_powderline,2
        'powderst':   self->change_powderline,3
        'powdersp':   self->change_powderline,4
        'powdershowall': begin
            self.powd_showall = ~self.powd_showall
            dm_set_button,event.id,self.powd_showall,onstring='Show Closed Powder Line',offstring='Show Actual Powder Line'
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end
        'epadnone': self->eadjust,0
        'epadauto': self->eadjust,1
        'epadspec': self->eadjust,2
        'empttol_cwt': if self.empt_tol_optn ne 0 then begin
            self.empt_tol_optn = 0b
            self->my_menu_toggle,'empttol_'+['cwt','wtc'],self.empt_tol_optn
            if ptr_valid(self.bgdata[0]) and ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b
            if self.samp_typ eq -1 then self->data_subbg 
          end
        'empttol_wtc': if self.empt_tol_optn ne 1 then begin
            self.empt_tol_optn = 1b
            self->my_menu_toggle,'empttol_'+['cwt','wtc'],self.empt_tol_optn
            if ptr_valid(self.bgdata[0]) and ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b 
            if self.samp_typ eq -1 then self->data_subbg
          end
        'epadcheckfirst':begin
            self.eadjust_checkfirst = ~self.eadjust_checkfirst  
            dm_set_button,event.id,self.eadjust_checkfirst,onstring='Check the First File Only',offstring='Check All Files'
          end
        'crossplotyes': if ~self.crossplot then begin
            self.crossplot = 1b
            self->my_menu_toggle,'crossplot'+['No','Yes'],self.crossplot
            dm_set_button,widget_info(self.tlb,find_by_uname='allowCrossplot'),self.crossplot
          end  
        'crossplotno': if self.crossplot then begin
            self.crossplot = 0b
            self->my_menu_toggle,'crossplot'+['No','Yes'],self.crossplot
            dm_set_button,widget_info(self.tlb,find_by_uname='allowCrossplot'),self.crossplot
          end
        'voronoiyes': if ~self.allow_voronoi then begin
            self.allow_voronoi = 1b
            self.ftoolflag[2]  = 1b ;no warning needed
            info = ['Voronoi diagram is enabled.','Slice plot will be rendered as Voronoi diagrams if both slice steps are left blank. The rendering is much slower.'] 
            ok = dialog_message(info,dialog_parent=self.tlb,/info,/center)
            self->my_widget_control,['saveslicexyie','saveslicespe'],sensitive=(finite(self.slice_xstep) or finite(self.slice_xstep))
            self->my_menu_toggle,'voronoi'+['No','Yes'],self.allow_voronoi
            dm_set_button,widget_info(self.tlb,find_by_uname='allowVoronoi'),self.allow_Voronoi
          end 
        'voronoino': if self.allow_voronoi then begin
            self.allow_voronoi = 0b
            self->my_widget_control,['saveslicexyie','saveslicespe'],/sensitive
            self->my_menu_toggle,'voronoi'+['No','Yes'],self.allow_voronoi
            dm_set_button,widget_info(self.tlb,find_by_uname='allowVoronoi'),self.allow_Voronoi
          end     
        'pbgblack': if self.pbgcol_bw ne 0 then begin
            self.pbgcol_bw = 0
            self->my_menu_toggle,'pbg'+['Black','White'],self.pbgcol_bw 
            (*self.colors)[0] = 'white'
            if total(self.coldefine) eq 0 then self.coldefine = [255b,255b,255b]
            dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutColor'),select=self.cut_color,set_value=(*self.colors)
            self->set_allplots,bgcolor=0,/totop
          endif    
        'pbgwhite': if self.pbgcol_bw ne 1 then begin
            self.pbgcol_bw = 1
            self->my_menu_toggle,'pbg'+['Black','White'],self.pbgcol_bw 
            (*self.colors)[0] = 'black'
            if total(255-self.coldefine) eq 0 then self.coldefine = [0b,0b,0b]
            dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutColor'),select=self.cut_color,set_value=(*self.colors)
            self->set_allplots,bgcolor=1,/totop
          endif 
        'textrm_0': if self.ptrm_choice ne 0 then begin
            self.ptrm_choice = 0
            self->my_menu_toggle,'textrm_'+['0','1'],self.ptrm_choice
            self->set_allplots,render_method=self.ptrm_choice,/totop
          endif 
        'textrm_1': if self.ptrm_choice ne 1 then begin
            self.ptrm_choice = 1
            self->my_menu_toggle,'textrm_'+['0','1'],self.ptrm_choice
            self->set_allplots,render_method=self.ptrm_choice,/totop
          endif
        'tdirinside':if self.tdir_choice ne 0 then self->setproperty,tickdir=0
        'tdiroutside':if self.tdir_choice ne 1 then self->setproperty,tickdir=1
        'ppvcnone':if self.ppvc_choice ne 0 then begin
            self.ppvc_choice = 0
            self->my_menu_toggle,'ppvc'+['None','Avg2','Avg3'],self.ppvc_choice
            self->set_allplots,vt_col=0,/totop
          endif
        'ppvcavg2':if self.ppvc_choice ne 1 then begin
            self.ppvc_choice = 1
            self->my_menu_toggle,'ppvc'+['None','Avg2','Avg3'],self.ppvc_choice
            self->set_allplots,vt_col=1,/totop
          endif
        'ppvcavg3':if self.ppvc_choice ne 2 then begin
            self.ppvc_choice = 2
            self->my_menu_toggle,'ppvc'+['None','Avg2','Avg3'],self.ppvc_choice
            self->set_allplots,vt_col=2,/totop
          endif  
        'pdim_choice':  begin
            self.pdim_choice = ind_sel
            pdimmenu = widget_info(self.tlb,find_by_uname='pdimMenu')
            self->my_menu_toggle,'pdchoice_'+dm_to_string(indgen(n_elements((*self.pdim_list)[0,*])),/int),self.pdim_choice,base=pdimmenu
          end 
        'userfunction': begin
             self.userfwarning = 0b ;clear the warning flag
             if lmgr(/runtime) then begin
                name  = ['user macro:','execution time:'] 
                title = 'Specify user macro'
                examp = ['Example: index = where(qty lt 0,count)','              if count gt 0 then qty[index]=!values.f_nan'] 
             endif else begin
                name  = ['user macro/call:','execution time:']
                title = 'Specify user macro or calls'
                examp = ['Example 1: index = where(qty lt 0,count)',$
                         '                if count gt 0 then qty[index]=!values.f_nan',$
                         'Example 2: qty=my_function(qty,err,en=en,'+((self.samp_typ gt 0)?'Qv=Qv)':'Q=Q)'),$
                         'Example 3: my_procedure,qty,err,en,det_tt=det_tt']
             endelse
             default = self.userf_str
             ;replace '&' with string('A4'xb)
             while((index=strpos(default,'&')) ge 0) do strput,default,string('A4'xb),index
             default = [default,dm_to_string(self.userf_before)]
             path    = self.dirs[1]
             answer  = dm_dialog_input(name,title=title,default=default,dialog_parent=self.tlb,$
                       info=[examp,userf_info,'Clear the string to disable this option. Do not use comment lines.'],xsize=200,multiline=4,is_file=[4],path=path,$
                       droplist_content=ptr_new([['After','Before']+' Projection to Viewing Axes','Before Empty Can Subtraction']),is_droplist=[0,1],/return_number,/scroll,cancel=cancel)
             if keyword_set(cancel) then return
             if path ne self.dirs[1] then self->setproperty,workDir=path
             ;replace possible string('A4'xb) with '&'
             b_a = dm_to_number(answer[1],/int)
             str = answer[0]
             while((index=strpos(str,string('A4'xb))) ge 0) do strput,str,'&',index
             ;remove possible '$&'
             while((index=strpos(str,'$&')) ge 0) do strput,str,'  ',index
             changed = (self.userf_str ne str) or (self.userf_before ne b_a)
             self.userf_str    = str
             self.userf_before = b_a
             widget_control,event.id, set_button=(strlen(str) ne 0)
             if ptr_valid(self.projStrPtr) and changed then recalc_proj = 1b    ;need to recalculate projections
          end
        'extravaxis':begin
            self.extravaxis_yn[0] = ~self.extravaxis_yn[0]
            onstr  = ([(['Allow Extra Viewing Axis','Crystal Axis U3'])[self.instrname eq 'dcs'],'Allow Temperature as Extra Viewing Axis'])[self.instrname eq 'wand']
            offstr = ([(['Disallow Extra Viewing Axis','Disallow Crystal Axis U3'])[self.instrname eq 'dcs'],'Disallow Temperature as Extra Viewing Axis'])[self.instrname eq 'wand']
            dm_set_button,event.id,self.extravaxis_yn[0],onstring=onstr,offstring=offstr
            if self.instrname eq 'wand' then self->my_widget_control,'avgTemp',sensitive=self.extravaxis_yn[0]
            if self.instrname eq 'dcs' then begin
               if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=(total(self.extravaxis_yn[((self.instrname eq 'macs')?0:(self.samp_typ eq 0)):*]) ne 0)
               dm_set_button,widget_info(self.tlb,find_by_uname='evaxsNone'),(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0)
            endif
            if self.extravaxis_yn[0] then ptr_free,self.projStrPtr  ;need to recalculate projection
            self->reset_pdt
          end
        'evaxstemp':begin  
            self.extravaxis_yn[1] = ~self.extravaxis_yn[1]
            dm_set_button,event.id,self.extravaxis_yn[1],onstring='Temperature',offstring='No Temperature'
            if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=(total(self.extravaxis_yn[((self.instrname eq 'macs')?0:(self.samp_typ eq 0)):*]) ne 0)
            self->my_widget_control,'taxisMenu',sensitive=self.extravaxis_yn[1]
            dm_set_button,widget_info(self.tlb,find_by_uname='evaxsNone'),(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0)
            if self.extravaxis_yn[1] then begin
               info = (['Sample','Control','Setpoint'])[self.macstemp_typ]+' temperautre is selected.'
               if self.tempavg_yn or (self.instrname eq 'dcs') then info = info+' Temperature of each file will be averaged.' 
               if ptr_valid(self.dataStrPtr) then begin  ;may need to reload data
                  ind = where((*self.dataStrPtr).temperature lt 0,count)
                  if (count ne 0) or ((self.instrname eq 'dcs') and (self.samp_typ lt 2)) or ((self.instrname eq 'macs') and self.ftoolflag[4]) then self->clear_data,infomsg = info ;missing some temperature
               endif
               ok = dialog_message(info,dialog_parent=self.tlb,/info,/center) 
            endif
            if self.extravaxis_yn[1] then ptr_free,self.projStrPtr
            self->reset_pdt
          end
        'evaxshfld':begin  
            self.extravaxis_yn[2] = ~self.extravaxis_yn[2]
            dm_set_button,event.id,self.extravaxis_yn[2],onstring='Magnetic Field',offstring='No Magnetic Field'
            if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=(total(self.extravaxis_yn[((self.instrname eq 'macs')?0:(self.samp_typ eq 0)):*]) ne 0)
            dm_set_button,widget_info(self.tlb,find_by_uname='evaxsNone'),(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0)
            if self.extravaxis_yn[2] then begin
               self->clear_data
               ptr_free,self.projStrPtr
            endif
            self->reset_pdt
          end
        'evaxsspec':begin  
            new = dm_dialog_input(['name:','unit:','identifier:'],title='Please specify',dialog_parent=self.tlb,default=self.extravaxis_spec,xsize=80,$
                     cancel=cancel,info=['Only the name is required. The identifier is the column name in the data file.','Clear all fields to disable it.'])
            if keyword_set(cancel) then return
            if total(strlen(new)) ne 0 then begin
               self.extravaxis_yn[0] = 1b
               self.extravaxis_spec  = new
            endif else self.extravaxis_yn[0] = 0b
            dm_set_button,event.id,self.extravaxis_yn[0],onstring='Specify',offstring='No Specify'
            if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=(total(self.extravaxis_yn[((self.instrname eq 'macs')?0:(self.samp_typ eq 0)):*]) ne 0)
            dm_set_button,widget_info(self.tlb,find_by_uname='evaxsNone'),(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0)
            if self.extravaxis_yn[0] then begin
               self->clear_data
               ptr_free,self.projStrPtr
            endif
            self->reset_pdt
          end    
        'evaxsnone':begin
            self.extravaxis_yn[*] = 0
            dm_set_button,widget_info(self.tlb,find_by_uname=(['extraVaxis','evaxsSpec'])[self.instrname eq 'macs']),self.extravaxis_yn[0],onstring=(['Crystal Axis U3','Specify'])[self.instrname eq 'macs'],offstring=(['Disallow Crystal Axis U3','No Specify'])[self.instrname eq 'macs']
            dm_set_button,widget_info(self.tlb,find_by_uname='evaxsTemp'),self.extravaxis_yn[1],onstring='Temperature',offstring='No Temperature'
            dm_set_button,widget_info(self.tlb,find_by_uname='evaxsHfld'),self.extravaxis_yn[2],onstring='Magnetic Field',offstring='No Magnetic Field'
            dm_set_button,event.id,1,sensitive=0
            if self.idl_version ge 5.6 then self->my_widget_control,'evaxsMenu',set_button=0
            self->reset_pdt
          end
        'taxissamp':if self.macstemp_typ ne 0 then begin
            self.macstemp_typ = 0
            self->my_menu_toggle,'taxis'+['Samp','Ctrl','Setp'],self.macstemp_typ
            self->my_widget_control,'taxisMenu',set_value='Temperature Choice ('+(['Sample','Control','Setpoint'])[self.macstemp_typ]+')'
            if ptr_valid(self.projStrPtr) and ptr_valid(self.dataStrPtr) and ((self.intn_typ eq 1) or (self.intn_typ eq 3) or (self.intn_typ eq 4) or (self.intn_typ eq 5)) then recalc_proj = 1b
          end  
        'taxisctrl':if self.macstemp_typ ne 1 then begin
            self.macstemp_typ = 1
            self->my_menu_toggle,'taxis'+['Samp','Ctrl','Setp'],self.macstemp_typ
            self->my_widget_control,'taxisMenu',set_value='Temperature Choice ('+(['Sample','Control','Setpoint'])[self.macstemp_typ]+')'
            if ptr_valid(self.projStrPtr) and ptr_valid(self.dataStrPtr) and ((self.intn_typ eq 1) or (self.intn_typ eq 3) or (self.intn_typ eq 4) or (self.intn_typ eq 5)) then recalc_proj = 1b
          end  
        'taxissetp':if self.macstemp_typ ne 2 then begin
            self.macstemp_typ = 2
            self->my_menu_toggle,'taxis'+['Samp','Ctrl','Setp'],self.macstemp_typ
            self->my_widget_control,'taxisMenu',set_value='Temperature Choice ('+(['Sample','Control','Setpoint'])[self.macstemp_typ]+')'
            if ptr_valid(self.projStrPtr) and ptr_valid(self.dataStrPtr) and ((self.intn_typ eq 1) or (self.intn_typ eq 3) or (self.intn_typ eq 4) or (self.intn_typ eq 5)) then recalc_proj = 1b
          end   
        'avgtemp':begin
            self.tempavg_yn = ~self.tempavg_yn
            dm_set_button,event.id,self.tempavg_yn,onstring='Average Temperature of Each File',offstring='Keep Individual Temperature'
            self->clear_data
          end     
        'darkcorr':begin
            new_flag = ~self.dcor_yn
            if new_flag then begin
               self.filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir
               if n_elements(open_file) eq 0 then begin
                  ok = dialog_message(dialog_parent=self.tlb,'Please select dark angle correction files first.',/error,/center)
                  new_flag = 0b
               endif else begin
                  default = self.dcor_eqran
                  eqran   = dm_dialog_input(['E_start (meV):','E_end (meV):','2theta_start:','2theta_end:'],title='Specify E, 2theta range:',xsize=80,cancel=cancel,$
                            default=default,info=['The total intensity in the specified range will','be used to calculate the dark angle correction.'],dialog_parent=self.tlb)
                  if keyword_set(cancel) then return
                  if total(finite(eqran)) ne 4 then $
                     new_flag = 0b $
                  else if (eqran[0] ge eqran[1]) or (eqran[2] ge eqran[3]) then $
                     new_flag = 0b
                  if new_flag eq 0 then $
                     ok = dialog_message('Invalid input. Dark angle correction is disabled.',dialog_parent=self.tlb,/center) $
                  else begin
                     self.dcor_eqran = eqran
                     det_tt = (*self.detPosPtr).two_theta
                     index  = where((abs(det_tt) ge self.dcor_eqran[2]) and (abs(det_tt) le self.dcor_eqran[3]),count)
                     if count eq 0 then begin
                        ok  = dialog_message(['The specified 2theta angle is beyond the detector range.','Dark angle correction is skipped.'],dialog_parent=self.tlb,/center)
                        new_flag = 0b
                     endif else begin
                        dm_load_dcsdata,open_path,open_file,qty=tmp_qty,t_chan=2,e_range=self.dcor_eqran[0:1],ang=ang,parent=self.tlb,/raw,title='Load Dark Angle Files',eadjust=self.eadjust,$
                              epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,montype=self.mon_typ[mon_instr],ftpobj=ftpobj,ftpbufferdir=ftpbufferdir
                        tmp_qty = tmp_qty[index,*]
                        tmp_qty = total(tmp_qty,1)
                        tmp_qty = max(tmp_qty)/tmp_qty
                        ptr_free,self.dcor_info
                        self.dcor_info = ptr_new({factor:tmp_qty,psi:reform(ang[0,*])})
                     endelse
                  endelse
               endelse
            endif
            if self.dcor_yn ne new_flag then begin
               self.dcor_yn = new_flag
               dm_set_button,event.id,self.dcor_yn,onstring='Dark Angle Correction Enabled',offstring='Dark Angle Correction Disabled'
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
            endif
          end
        'darktraj':begin
            self.datr_yn = ~self.datr_yn
            dm_set_button,event.id,self.datr_yn,onstring='Show Dark Angle Trajectory',offstring='Disable Dark Angle Trajectory'
            if ptr_valid(self.projStrPtr) and (~ptr_valid(self.dataStrPtr)) then recalc_proj = 1b
          end 
        'edepeff':begin
            new_flag = ~self.edep_yn
            if new_flag then begin
               case self.instrname of 
                    'dcs':  info = 'The default values for DCS detecotors are 1 cm and 6 atm.'
                    'macs': info = 'The default values for MACS detectors are 3.773 cm and 3 atm.'
                    else:   info = '1 cm and 6 atm are the default parameters for DCS detectors.'
               endcase
               repeat begin
                  new = dm_dialog_input('He3 detector '+['thickness(cm):','pressure(atm):'],title='Please confirm detector info',dialog_parent=self.tlb,default=self.edep_det,info=info,/float,cancel=cancel)
               endrep until ((total(finite(new,/nan)) eq 0) or keyword_set(cancel))
               if keyword_set(cancel) then return
               ind = where(new eq 0,count)
               if count ne 0 then begin
                  new_flag = 0
                  ok = dialog_message('Invalid He3 detector '+strjoin((['thickness','pressure']+ ' ('+dm_to_string(new)+')')[ind],' and ')+'. E-dependent detector efficiency correction is disabled.',dialog_parent=self.tlb,/center)
               endif else self.edep_det = new
            end
            if self.edep_yn ne new_flag then begin
               self.edep_yn = new_flag
               dm_set_button,event.id,self.edep_yn,onstring='E-Dependent Detector Efficiency Correction Enabled',offstring='E-Dependent Detector Efficiency Correction Disabled'
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
            endif
          end
        'errbar':begin
            self.errbar = ~self.errbar
            dm_set_button,event.id,1-self.errbar,onstring='Error Bar Disabled',offstring='Error Bar Enabled'
          end
        'ncnrftp': begin 
            self.fileSel->set_ftpbuffer,(['',self.dirs[2]])[1-self.ncnrftp]
            self.fileSel->getproperty,ncnrftp=ncnrftp
            if ncnrftp ne 1-self.ncnrftp then ok = dialog_message('Cannot establish connection to NCNR Data Repository.',dialog_parent=self.tlb,/center,/error)
            self.ncnrftp = ncnrftp
            dm_set_button,event.id,self.ncnrftp,onstring='Allow NCNR Data Repository',offstring='Disallow NCNR Data Repository'
            if self.ncnrftp then self.fileSel->set_path,/ftp
          end
        'viewspurion': if self.instrname eq 'macs' then self->dm_macs_spurion,/gasrecoil else self->dm_det_spurion
        'viewghoston': dm_ghoston,event,lattparm=self.latt_parm,u=self.orie_u,v=self.orie_v,workdir=self.dirs[1],tickdir=self.tdir_choice
        'useitool': begin
            self.useitool = ~self.useitool
            dm_set_button,event.id,self.useitool,onstring='Use Classical Plotting',offstring='Use iTools Plotting'
          end
        'viewproj':begin
            self.view[0] = (~self.view[0])
            dm_set_button,event.id,self.view[0],onstring='Hide Projection',offstring='Show Projection'
            self->reset_pdt
          end
        'viewdisp':begin
            self.view[1] = (~self.view[1])
            dm_set_button,event.id,self.view[1],onstring='Hide Display',offstring='Show Display'
            self->reset_pdt
          end
        'viewcurr':   if obj_valid(self.current) then self.current->totop
        'killall':    self->dm_add_plotwindow,4
        'arrcascade': self->dm_arrange_plot,/cascade
        'arrtable':   self->dm_arrange_plot
        'hpctbut':    if strlen(self.helpfile[0]) ne 0 then online_help,book=self.helpfile[0],/full_path
        'hpftbut':    if strlen(self.helpfile[1]) ne 0 then online_help,book=self.helpfile[1],/full_path
        'idlhpbut':   dm_idl_help,group=self.tlb
        'hpnewbut':   ok = dialog_message(*self.whatsnew,/information,dialog_parent=self.tlb,title="What's new in Mslice",/center)
        'selfbut':  if ptr_valid(self.dataPtr) then $
               ok = dialog_message(*self.info,/information,dialog_parent=self.tlb,title='About Mslice',/center) $
            else begin
               self->calculator_insert,*self.info,/totop
               ok = dialog_message(*self.info,/information,dialog_parent=self.tlb,title='About Mslice for '+(*self.ftypename)[self.ftype],/center)
            endelse
        'cleardat':  begin
            self.monrate = 0d
            ptr_free,self.kfactor,self.dataStrPtr,self.projStrPtr
            self->my_widget_control,['addBut','mkplBut'],sensitive=0  ;disallow adding data & plotting mask
            widget_control,event.id,sensitive=0
            if ~self.lock_title then begin
               self->my_widget_control,'titlLab',set_value='' & self.titl_lab=''   ;clear title label
               self->my_widget_control,'legdLab',set_value='' & self.legd_lab=''   ;clear legend label
            endif
            if self.instrname eq 'macs' then begin
               self.macs_mon[0,*] = !values.f_nan
               self->my_widget_control,'monSum',set_value='Automatic'
               self->reset_intnlab
               if ptr_valid(self.mask) and (self.masktype eq 1) then self->clear_mask,/message,/keep_ar,mask_darkangle=self.mask_darkangle
            endif else begin
               if self.samp_typ ge 1 then self->my_widget_control,'oriepsiLab',self.projBas,set_value='   <- '+((self.samp_typ eq 1)?'':'first ')+'angle of ki^u'
            endelse
            self->set_display_button
            self.ftoolflag[3] = 0b  ;clear bad detector warning flag for macs
          end
        'locktitlbut':begin
            self.lock_title = ~self.lock_title
            widget_control,event.id,/bitmap,set_value=self.iconfiles[8+self.lock_title],_extra=dm_tooltip('Click to '+(['lock','unlock'])[self.lock_title]+' the title label so that loading files will '+(['not',''])[self.lock_title]+' reset the title and legend labels.')
          end  
        'intnlab':  self.intn_lab = tmp_value   ;intensity label
        'titllab':  self.titl_lab = tmp_value   ;title label
        'legdlab':  self.legd_lab = tmp_value   ;legend label
        'powdtyp':  self->change_samptyp,0,/dcsmaskwarning
        'crysttyp1':self->change_samptyp,1,/dcsmaskwarning
        'crysttyp2':self->change_samptyp,2,/dcsmaskwarning
        'crysttyp3':self->change_samptyp,3,/dcsmaskwarning
        'ebin':  begin    ;ebin size /energy range
            if self.instrname eq 'macs' then begin
               self.mon_sumfit[mon_instr] = tmp_value
               self->my_menu_toggle,'mon'+['Sum','Fit'],self.mon_sumfit[mon_instr]
               if tmp_value eq 1 then begin ;specify
                  old = self.macs_mon[1,self.mon_typ[mon_instr]]
                  tmp = (['.',' at normalized intensity mode.'])[keyword_set(self.macshistomode) and self.macs_rawintn]
                  repeat begin
                     case self.mon_typ[mon_instr] of
                          0: new = dm_dialog_input('monitor time (sec):',default=dm_to_string(old),dialog_parent=self.tlb,info='Intensity is normalized to this monitor time'+tmp,/float)
                          1: new = dm_dialog_input('monitor count:',default=dm_to_string(old,/exponent),dialog_parent=self.tlb,info='Intensity is normalized to this monitor count'+tmp,/float)
                          2: new = dm_dialog_input('pulse number:',default=dm_to_string(old,/int),dialog_parent=self.tlb,info='Intensity is normalized to this pulse number'+tmp,/int)
                          else:
                     endcase   
                     if finite(new) then begin
                        if new gt 0 then ok2 = 1b
                     endif
                  endrep until keyword_set(ok2)
                  if self.mon_typ[mon_instr] eq 2 then new = round(new)
                  if new ne old then self.macs_mon[1,self.mon_typ[mon_instr]] = new
               endif
               if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and ~(keyword_set(self.macshistomode) and self.macs_rawintn) then recalc_proj = 1b    ;need to recalculate projections
               self->reset_intnlab
            endif else begin
               changed  = 0b
               old_ebin = [self.e_bin[self.samp_typ],self.e_bin[4]]
               old_eran = self.e_range 
               if self.samp_typ ge 2 then begin   ;diffuse scattering
                  self.e_bin[self.samp_typ] = tmp_value
                  if tmp_value eq 2 then begin    ;specify energy range
                     default = dm_to_string(self.e_range)
                     items   = ['from:','to:']
                     info    = ['Energy range is in units of meV.','Data files need to be reloaded.']
                     if self.samp_typ eq 3 then begin
                        default = [default,dm_to_string(self.e_bin[4])]
                        items   = [items,'E rebin size:']
                     endif
                     new = dm_dialog_input(items,title='Energy range',default=default,/float,dialog_parent=self.tlb,info=info)
                     self.e_bin[self.samp_typ] = 1
                     self.e_range[*] = !values.f_nan
                     if total(finite(new[0:1])) eq 2 then begin
                        if new[1] gt new[0] then begin
                           self.e_range = new[0:1]
                           self.e_bin[self.samp_typ] = 2
                        endif
                     endif
                     if self.e_bin[self.samp_typ] eq 1 then begin
                        ok = dialog_message('Invalid energy range. Elastic peak option is used.',dialog_parent=self.tlb,/center)
                        dm_set_droplist,event.id,select=1
                     endif
                     if self.samp_typ eq 3 then begin  ;rebin size
                        new = fix(new[2])
                        if new le 0 then begin
                           new = 1
                           ok = dialog_message('Invalid rebin size. 1 is used instead.',dialog_parent=self.tlb,/center)
                        endif
                        self.e_bin[4] = new
                     end
                  endif else if self.samp_typ eq 3 then begin  ;need to ask for rebin size for multiple E type
                     new = dm_to_string(self.e_bin[4])
                     new = dm_dialog_input('E rebin size:', title='Rebin size',default=new,/int,dialog_parent=self.tlb)
                     if new le 0 then begin
                        new = 1
                        ok = dialog_message('Invalid rebin size. 1 is used instead.',dialog_parent=self.tlb,/center)
                     endif
                     self.e_bin[4] = new   
                  endif
               endif else begin    ;single crystal or powder
                  self.e_bin[self.samp_typ] = (*self.rebinsize)[tmp_value]
               endelse
               if self.e_bin[self.samp_typ] ne old_ebin[0] then changed = 1b else begin
                  if self.samp_typ ge 2 then begin
                     if self.e_bin[self.samp_typ] eq 2 then begin
                        if (total(finite(self.e_range)) ne 2) or (total(finite(old_eran)) ne 2) then changed=1b else begin
                           if total(abs(self.e_range-old_eran)) gt 1e-7 then changed = 1b
                        endelse
                     endif
                     if self.samp_typ eq 3 then begin
                        if self.e_bin[4] ne old_ebin[1] then changed = 1b
                     endif
                  endif
               endelse
               if ptr_valid(self.bgdata[0]) and changed then begin
                  ptr_free,self.bgdata
                  self->reset_menu,/backgroundmenuonly
                  ok = dialog_message('Please reload empty can files.',dialog_parent=self.tlb,/center)
               endif
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
            endelse    
          end
        'latta': self.latt_parm[0] = dm_to_number(tmp_value,/float)   ;a
        'lattb': self.latt_parm[1] = dm_to_number(tmp_value,/float)   ;b
        'lattc': self.latt_parm[2] = dm_to_number(tmp_value,/float)   ;c
        'lattaa':self.latt_parm[3] = dm_to_number(tmp_value,/float)   ;alpha
        'lattbb':self.latt_parm[4] = dm_to_number(tmp_value,/float)   ;beta
        'lattcc':self.latt_parm[5] = dm_to_number(tmp_value,/float)   ;gama
        'orieuh':self.orie_u[0]    = dm_to_number(tmp_value,/float)   ;uh
        'orieuk':self.orie_u[1]    = dm_to_number(tmp_value,/float)   ;uk
        'orieul':self.orie_u[2]    = dm_to_number(tmp_value,/float)   ;ul
        'orievh':self.orie_v[0]    = dm_to_number(tmp_value,/float)   ;vh
        'orievk':self.orie_v[1]    = dm_to_number(tmp_value,/float)   ;vk
        'orievl':self.orie_v[2]    = dm_to_number(tmp_value,/float)   ;vl
        'oriepsi': begin  ;psi
            tmp_value = dm_to_number(tmp_value,/float)
            if self.orie_psi ne tmp_value then begin
               self.orie_psi = tmp_value   
               if (n_elements(resetpsioffset) eq 0) or keyword_set(resetpsioffset) then self.orie_offset = !values.f_nan
            endif
          end
        'viewu1typ':if self.view_u1typ ne tmp_value then begin
            self.view_u1typ = tmp_value
            label = self.view_label[0:3]                                
            if self.intn_typ eq n_elements(self.intn_typ_str)-2 then label = [label,self.view_label[5]]
            if self.instrname eq 'dcs' then label = [label,self.view_label[4]]                      
            self->script,'set_parameter',name='viewU1lab',value=label[tmp_value]
            resetu1 = 1 & islabel = 1
          endif
        'viewu1h': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu1 = finite(tmp)?(finite(self.view_u1[0])?((self.view_u1[0] eq tmp)?0:1):(1)):(0)
            self.view_u1[0] = tmp
          end
        'viewu1k': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu1 = finite(tmp)?(finite(self.view_u1[1])?((self.view_u1[1] eq tmp)?0:1):(1)):(0)
            self.view_u1[1] = tmp
          end
        'viewu1l': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu1 = finite(tmp)?(finite(self.view_u1[2])?((self.view_u1[2] eq tmp)?0:1):(1)):(0)
            self.view_u1[2] = tmp
          end
        'viewu1lab':if self.view_u1lab[(self.samp_typ ge 1)] ne tmp_value then begin
            self.view_u1lab[(self.samp_typ ge 1)] = (strlen(tmp_value) eq 0)?' ':tmp_value
            if self.samp_typ eq 0 then begin
               if (self.view_u1typ ge 4) and (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) then begin
                  if self.instrname eq 'dcs' then begin
                     if self.view_u1typ eq 4 then id = 5 else id = 4
                  endif else id = self.view_u1typ+1 
               endif else id = self.view_u1typ
               self.view_label[id] = self.view_u1lab[0]
            endif
            resetu1 = 1 & islabel = 1
          endif
        'viewu1fold':  begin
            self.view_u1fold = ~self.view_u1fold
            self->reset_projection,/reset,/foldonly
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end 
        'viewu1foldcenter': begin
            old = dm_to_string(*(self.view_u1foldcenter),separator=',')
            *(self.view_u1foldcenter) = dm_to_number(strsplit(tmp_value,' ,;',/extract),/float)
            if dm_to_string(*(self.view_u1foldcenter),separator=',') ne old then begin
               if self.fold_type eq 0 then begin
                  *(self.view_u1folddir) = (*(self.view_u1folddir))[0] ;reset folding direction to the first value
                  dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU1Folddir'),select=(*self.view_u1folddir)[0]+1,set_value=['-','auto','+']
               endif
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b
            endif
          end
        'viewu1folddir': begin
            old = dm_to_string(*(self.view_u1folddir),separator=',')
            nfc = n_elements(*(self.view_u1foldcenter))
            if nfc le 1 then begin
               *(self.view_u1folddir) = tmp_value-1
            endif else begin
               *(self.view_u1folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u1foldcenter))+':',title='Set u1 folding direction',$
                    default =*(self.view_u1folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
               value = (['-','auto','+'])[(*self.view_u1folddir)[0]+1]
               for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u1folddir)[i]+1]
               dm_set_droplist,event.id,select=0,set_value=[value,'auto','+']
            endelse
            if (dm_to_string(*(self.view_u1folddir),separator=',') ne old) and ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end
        'viewu2typ':if self.view_u2typ ne tmp_value then begin
            self.view_u2typ = tmp_value
            label = self.view_label[0:3]                                
            if self.intn_typ eq n_elements(self.intn_typ_str)-2 then label = [label,self.view_label[5]]
            if self.instrname eq 'dcs' then label = [label,self.view_label[4]]
            self->script,'set_parameter',name='viewU2lab',value=label[tmp_value]
            resetu2 = 1 & islabel = 1
          endif
        'viewu2h': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu2 = finite(tmp)?(finite(self.view_u2[0])?((self.view_u2[0] eq tmp)?0:1):(1)):(0)
            self.view_u2[0] = tmp
          end
        'viewu2k': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu2 = finite(tmp)?(finite(self.view_u2[1])?((self.view_u2[1] eq tmp)?0:1):(1)):(0)
            self.view_u2[1] = tmp
          end
        'viewu2l': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu2 = finite(tmp)?(finite(self.view_u2[2])?((self.view_u2[2] eq tmp)?0:1):(1)):(0)
            self.view_u2[2] = tmp
          end
        'viewu2lab':if self.view_u2lab[(self.samp_typ ge 1)] ne tmp_value then begin
            self.view_u2lab[(self.samp_typ ge 1)] = (strlen(tmp_value) eq 0)?' ':tmp_value
            if self.samp_typ eq 0 then begin
               if (self.view_u2typ ge 4) and (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) then begin
                  if self.instrname eq 'dcs' then begin
                     if self.view_u2typ eq 4 then id = 5 else id = 4
                  endif else id = self.view_u2typ+1 
               endif else id = self.view_u2typ
               self.view_label[id] = self.view_u2lab[0]
            endif
            resetu2 = 1 & islabel = 1
          endif
        'viewu2fold':  begin
            self.view_u2fold = ~self.view_u2fold
            self->reset_projection,/reset,/foldonly
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end 
        'viewu2foldcenter': begin
            old = dm_to_string(*(self.view_u2foldcenter),separator=',')
            *(self.view_u2foldcenter) = dm_to_number(strsplit(tmp_value,' ,;',/extract),/float)
            if (dm_to_string(*(self.view_u2foldcenter),separator=',') ne old)  and (self.fold_type eq 0) then begin
               *(self.view_u2folddir) = (*(self.view_u2folddir))[0] ;reset folding direction to the first value
               dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU2Folddir'),select=(*self.view_u2folddir)[0]+1,set_value=['-','auto','+']
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
            endif
          end
        'viewu2folddir': begin
            old = dm_to_string(*(self.view_u2folddir),separator=',')
            nfc = n_elements(*(self.view_u2foldcenter))
            if nfc le 1 then begin
               *(self.view_u2folddir) = tmp_value-1
            endif else begin
               *(self.view_u2folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u2foldcenter))+':',title='Set u2 folding direction',$
                    default =*(self.view_u2folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
               value = (['-','auto','+'])[(*self.view_u2folddir)[0]+1]
               for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u2folddir)[i]+1]
               dm_set_droplist,event.id,select=0,set_value=[value,'auto','+']
            endelse
            if (dm_to_string(*(self.view_u2folddir),separator=',') ne old) and ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end
        'viewu3h': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu3 = finite(tmp)?(finite(self.view_u3[0])?((self.view_u3[0] eq tmp)?0:1):(1)):(0)
            self.view_u3[0] = tmp
          end
        'viewu3k': begin
            tmp     = dm_to_number(tmp_value,/float)
            resetu3 = finite(tmp)?(finite(self.view_u3[1])?((self.view_u3[1] eq tmp)?0:1):(1)):(0)
            self.view_u3[1] = tmp
          end
        'viewu3l': begin
            tmp     =  dm_to_number(tmp_value,/float)
            resetu3 = finite(tmp)?(finite(self.view_u3[2])?((self.view_u3[2] eq tmp)?0:1):(1)):(0)
            self.view_u3[2] = tmp
          end
        'viewu3lab':if self.view_u3lab ne tmp_value then begin
            self.view_u3lab = (strlen(tmp_value) eq 0)?' ':tmp_value
            resetu3 = 1 & islabel = 1
          endif
        'viewu3fold':  begin
            self.view_u3fold = ~self.view_u3fold
            self->reset_projection,/reset,/foldonly
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end 
        'viewu3foldcenter': begin
            old = dm_to_string(*(self.view_u3foldcenter),separator=',')
            *(self.view_u3foldcenter) = dm_to_number(strsplit(tmp_value,' ,;',/extract),/float)
            if (dm_to_string(*(self.view_u3foldcenter),separator=',') ne old)  and (self.fold_type eq 0) then begin
               *(self.view_u3folddir) = (*(self.view_u3folddir))[0] ;reset folding direction to the first value
               dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU3Folddir'),select=(*self.view_u3folddir)[0]+1,set_value=['-','auto','+']
               if ptr_valid(self.projStrPtr) then recalc_proj = 1b 
            endif
          end
        'viewu3folddir': begin
            old = dm_to_string(*(self.view_u3folddir),separator=',')
            nfc = n_elements(*(self.view_u3foldcenter))
            if nfc le 1 then begin
               *(self.view_u3folddir) = tmp_value-1
            endif else begin
               *(self.view_u3folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u3foldcenter))+':',title='Set u3 folding direction',$
                    default =*(self.view_u3folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
               value = (['-','auto','+'])[(*self.view_u3folddir)[0]+1]
               for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u3folddir)[i]+1]
               dm_set_droplist,event.id,select=0,set_value=[value,'auto','+']
            endelse
            if (dm_to_string(*(self.view_u3folddir),separator=',') ne old) and ptr_valid(self.projStrPtr) then recalc_proj = 1b 
          end
        'calcprj':      self->dm_calc_projection    ;calculate projection
        'd4selx1':      if self.d4sel_x1[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/d4
        'd4selx2':      if self.d4sel_x2[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/d4
        'd4selx3':      if self.d4sel_x3[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/d4
        'd4selx4':      if self.d4sel_x4[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/d4
        'd4sel_t0':     self.d4sel_t1[0>(self.samp_typ)] = tmp_value
        'd4sel_t1':     self.d4sel_t2[0>(self.samp_typ)] = tmp_value                
        'd4x1from':     self.d4_x1ran[0]   = dm_to_number(tmp_value,/float)
        'd4x1to':       self.d4_x1ran[1]   = dm_to_number(tmp_value,/float)
        'd4x1step':     self.d4_x1step     = dm_to_number(tmp_value,/float)
        'd4x2from':     self.d4_x2ran[0]   = dm_to_number(tmp_value,/float)
        'd4x2to':       self.d4_x2ran[1]   = dm_to_number(tmp_value,/float)
        'd4x2step':     self.d4_x2step     = dm_to_number(tmp_value,/float)        
        'd4x3from':     self.d4_x3ran[0]   = dm_to_number(tmp_value,/float)
        'd4x3to':       self.d4_x3ran[1]   = dm_to_number(tmp_value,/float)
        'd4x3step':     self.d4_x3step     = dm_to_number(tmp_value,/float)
        'd4x4from':     self.d4_x4ran[0]   = dm_to_number(tmp_value,/float)
        'd4x4to':       self.d4_x4ran[1]   = dm_to_number(tmp_value,/float)
        'd4x4step':     self.d4_x4step     = dm_to_number(tmp_value,/float)
        'd4selfrom_t0': self.d4sel_tft1[0] = dm_to_number(tmp_value,/float)
        'd4selto_t0':   self.d4sel_tft1[1] = dm_to_number(tmp_value,/float)
        'd4selfrom_t1': self.d4sel_tft2[0] = dm_to_number(tmp_value,/float)
        'd4selto_t1':   self.d4sel_tft2[1] = dm_to_number(tmp_value,/float)
        'volselx':      if self.volsel_x[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/vol
        'volsely':      if self.volsel_y[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/vol
        'volselz':      if self.volsel_z[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/vol        
        'volxfrom':     self.vol_xran[0]  = dm_to_number(tmp_value,/float)
        'volxto':       self.vol_xran[1]  = dm_to_number(tmp_value,/float)
        'volxstep':     self.vol_xstep    = dm_to_number(tmp_value,/float)
        'volyfrom':     self.vol_yran[0]  = dm_to_number(tmp_value,/float)
        'volyto':       self.vol_yran[1]  = dm_to_number(tmp_value,/float)
        'volystep':     self.vol_ystep    = dm_to_number(tmp_value,/float)
        'volzfrom':     self.vol_zran[0]  = dm_to_number(tmp_value,/float)
        'volzto':       self.vol_zran[1]  = dm_to_number(tmp_value,/float)
        'volzstep':     self.vol_zstep    = dm_to_number(tmp_value,/float)
        'volsel_t':     if self.samp_typ eq -1 then begin
                           (*self.volsel_t)[ind_sel] = tmp_value 
                        endif else begin
                           case ind_sel of 
                                0: self.volsel_t1[self.samp_typ] = tmp_value
                                1: self.volsel_t2[self.samp_typ] = tmp_value
                                2: self.volsel_t3[self.samp_typ] = tmp_value
                                else:
                           endcase
                        endelse
        'volsel_from':  if self.samp_typ eq -1 then begin
                           (*self.volsel_tft)[ind_sel,0] = dm_to_number(tmp_value,/float)
                        endif else begin
                           case ind_sel of 
                                0: self.volsel_tft1[0] = dm_to_number(tmp_value,/float)
                                1: self.volsel_tft2[0] = dm_to_number(tmp_value,/float)
                                2: self.volsel_tft3[0] = dm_to_number(tmp_value,/float)
                                else:
                           endcase
                        endelse
        'volsel_to':    if self.samp_typ eq -1 then begin
                           (*self.volsel_tft)[ind_sel,1] = dm_to_number(tmp_value,/float)
                        endif else begin
                           case ind_sel of 
                                0: self.volsel_tft1[1] = dm_to_number(tmp_value,/float)
                                1: self.volsel_tft2[1] = dm_to_number(tmp_value,/float)
                                2: self.volsel_tft3[1] = dm_to_number(tmp_value,/float)
                                else:
                           endcase
                        endelse
        'volifrom':     self.vol_iran[0]    = dm_to_number(tmp_value,/float)
        'volito':       self.vol_iran[1]    = dm_to_number(tmp_value,/float)
        'volsmooth':    self.vol_smooth     = dm_to_number(tmp_value,/int)
        'volinterp':    self.vol_interp     = tmp_value
        'volcolor':     self.vol_color      = tmp_value
        'volplot':      self->dm_plot_volume             ;plot volume
        'voloplot':     self->dm_plot_volume,/overplot   ;oplot volume        
        'slicesel_x':   if self.slice_x[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/slice
        'slicesel_y':   if self.slice_y[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/slice
        'dispxfrom':    self.slice_xran[0]  = dm_to_number(tmp_value,/float)
        'dispxto':      self.slice_xran[1]  = dm_to_number(tmp_value,/float)
        'dispxstep':    begin
            self.slice_xstep = dm_to_number(tmp_value,/float)
            self->my_widget_control,['saveslicexyie','saveslicespe'],sensitive=(finite(self.slice_xstep) or finite(self.slice_xstep) or (~self.allow_voronoi))
          end  
        'dispyfrom':    self.slice_yran[0]  = dm_to_number(tmp_value,/float)
        'dispyto':      self.slice_yran[1]  = dm_to_number(tmp_value,/float)
        'dispystep':    begin
            self.slice_ystep = dm_to_number(tmp_value,/float)
            self->my_widget_control,['saveslicexyie','saveslicespe'],sensitive=(finite(self.slice_xstep) or finite(self.slice_xstep) or (~self.allow_voronoi))
          end
        'dispifrom':    self.slice_iran[0]  = dm_to_number(tmp_value,/float)
        'dispito':      self.slice_iran[1]  = dm_to_number(tmp_value,/float)
        'dispsmooth':   self.slice_smooth   = dm_to_number(tmp_value,/int)
        'dispshading':  self.slice_shading  = tmp_value
        'dispcolor':    self.slice_color    = tmp_value
        'dispsel_t': begin
            if self.samp_typ ge 0 then begin
               case ind_sel of 
                 0: self.slice_t1[self.samp_typ] = tmp_value
                 1: self.slice_t2[self.samp_typ] = tmp_value
                 2: self.slice_t3[self.samp_typ] = tmp_value
                 3: self.slice_t4[self.samp_typ] = tmp_value
                 else: 
               endcase
            endif else begin
               (*self.slice_t)[ind_sel] = tmp_value 
            endelse
          end
        'dispsel_from': begin
            if self.samp_typ ge 0 then begin
               case ind_sel of 
                 0: self.slice_t1ran[0] = dm_to_number(tmp_value,/float)
                 1: self.slice_t2ran[0] = dm_to_number(tmp_value,/float)
                 2: self.slice_t3ran[0] = dm_to_number(tmp_value,/float)
                 3: self.slice_t4ran[0] = dm_to_number(tmp_value,/float)
                 else: 
               endcase
            endif else begin
               (*self.slice_tft)[ind_sel,0] = dm_to_number(tmp_value,/float)
            endelse
          end
        'dispsel_to': begin
            if self.samp_typ ge 0 then begin
               case ind_sel of 
                 0: self.slice_t1ran[1] = dm_to_number(tmp_value,/float)
                 1: self.slice_t2ran[1] = dm_to_number(tmp_value,/float)
                 2: self.slice_t3ran[1] = dm_to_number(tmp_value,/float)
                 3: self.slice_t4ran[1] = dm_to_number(tmp_value,/float)
                 else: 
               endcase
            endif else begin  
               (*self.slice_tft)[ind_sel,1] = dm_to_number(tmp_value,/float)
            endelse
          end
        'dispfitslice':  self->dm_to_fit,/slice
        'dispplotslice': self->dm_plot_slice             ;plot slice
        'dispopltslice': self->dm_plot_slice,/overplot   ;oplot slice
        'dispsurfslice': self->dm_plot_slice,/surfplot   ;surf slice
        'cutsel_t': begin
            if self.samp_typ eq -1 then begin
               (*self.cutsel_t)[ind_sel] = tmp_value 
            endif else begin
               case ind_sel of
                 0: self.cut_t1[self.samp_typ] = tmp_value
                 1: self.cut_t2[self.samp_typ] = tmp_value
                 2: self.cut_t3[self.samp_typ] = tmp_value
                 3: self.cut_t4[self.samp_typ] = tmp_value
                 3: self.cut_t5[self.samp_typ] = tmp_value
                 else:
               endcase
            endelse
          end
        'cutsel_from': begin
            if self.samp_typ eq -1 then begin
               (*self.cutsel_tft)[ind_sel,0] = dm_to_number(tmp_value,/float)  
            endif else begin
               case ind_sel of
                  0: self.cut_t1ran[0] = dm_to_number(tmp_value,/float)
                  1: self.cut_t2ran[0] = dm_to_number(tmp_value,/float)
                  2: self.cut_t3ran[0] = dm_to_number(tmp_value,/float)
                  3: self.cut_t4ran[0] = dm_to_number(tmp_value,/float)
                  4: self.cut_t5ran[0] = dm_to_number(tmp_value,/float)
                  else:
               endcase
            endelse
          end
        'cutsel_to': begin
            if self.samp_typ eq -1 then begin
               (*self.cutsel_tft)[ind_sel,1] = dm_to_number(tmp_value,/float)
            endif else begin
               case ind_sel of
                  0: self.cut_t1ran[1] = dm_to_number(tmp_value,/float)
                  1: self.cut_t2ran[1] = dm_to_number(tmp_value,/float)
                  2: self.cut_t3ran[1] = dm_to_number(tmp_value,/float)
                  3: self.cut_t4ran[1] = dm_to_number(tmp_value,/float)
                  4: self.cut_t5ran[1] = dm_to_number(tmp_value,/float)
                  else:
               endcase 
            endelse
          end
        'cutx':      if self.cut_x[0>(self.samp_typ)] ne tmp_value then self->select_droplist,event.id,tmp_value,/cut
        'cutfrom':   self.cut_ran[0]  = dm_to_number(tmp_value,/float)
        'cutto':     self.cut_ran[1]  = dm_to_number(tmp_value,/float)
        'cutstep':   self.cut_step    = dm_to_number(tmp_value,/float)
        'cuty':      self.cut_y[0>(self.samp_typ)] = tmp_value
        'cutyfrom':  self.cut_yran[0] = dm_to_number(tmp_value,/float)
        'cutyto':    self.cut_yran[1] = dm_to_number(tmp_value,/float)
        'cutsymbol': self.cut_symbol  = tmp_value
        'cutcolor':  begin
            self.cut_color = tmp_value
            if (*self.colors)[self.cut_color] eq 'user define' then begin
               define = dm_plot.getdefinecolor(default=self.coldefine,title='Define color',bgcolor=[255b,255b,255b]*self.pbgcol_bw,parent=self.tlb)
               if total(define-[255b,255b,255b]*(1-self.pbgcol_bw)) eq 0 then begin
                  self.cut_color = 0
                  dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutColor'),select=self.cut_color
               endif
               self.coldefine = define
            endif
          end
        'cutline':    self.cut_line = tmp_value
        'cutfit':     self->dm_to_fit
        'cutplot':    self->dm_plot_cut
        'cutoplot':   self->dm_plot_cut,/overplot
        'trajx':      if self.traj_x[self.samp_typ] ne tmp_value then self->select_droplist,event.id,tmp_value,/traj
        'trajxfrom':  self.traj_xran[0] = dm_to_number(tmp_value,/float)
        'trajxto':    self.traj_xran[1] = dm_to_number(tmp_value,/float)
        'trajy':      if self.traj_y[self.samp_typ] ne tmp_value then self->select_droplist,event.id,tmp_value,/traj
        'trajyfrom':  self.traj_yran[0] = dm_to_number(tmp_value,/float)
        'trajyto':    self.traj_yran[1] = dm_to_number(tmp_value,/float)
        'trajz1':     self.traj_z1[self.samp_typ] = tmp_value
        'trajz1from': self.traj_z1ran[0] = dm_to_number(tmp_value,/float)
        'trajz1to':   self.traj_z1ran[1] = dm_to_number(tmp_value,/float)
        'trajz2':     self.traj_z2[self.samp_typ] = tmp_value
        'trajz2from': self.traj_z2ran[0] = dm_to_number(tmp_value,/float)
        'trajz2to':   self.traj_z2ran[1] = dm_to_number(tmp_value,/float)
        'plottra':    self->dm_plot_trajectory               ;plot trajectory
        'oplottra':   self->dm_plot_trajectory,/overplot     ;overplot trajectory
        'calcinput':  if event.type ne 3 then self->my_widget_control,['calc','clear']+'Cal',self.dispBas,sensitive=(max(strlen(tmp_value)) gt 0)
        'calc_com_e2l':      self->calculator_insert,[';enter E value below','print, dm_e2lambda( E )          ;E (meV) ->lambda ('+string('c5'xb)+')']
        'calc_com_l2e':      self->calculator_insert,[';enter lambda value below','print, dm_lambda2e( lambda )       ;lambda ('+string('c5'xb)+') -> E (meV)']
        'calc_com_e2v':      self->calculator_insert,[';enter E value below','print, dm_e2velocity( E )          ;E (meV) -> velocity (m/s)']
        'calc_com_qlength1': self->calculator_insert,[';enter h, k, l, a, b, c, alpha, beta, gamma values below',latts,'print, dcs_cryst_align_qlength1( [ h, k, l ], [ a, b, c, alpha, beta, gamma ] )']
        'calc_com_qlength2': self->calculator_insert,[';enter Ei (Ef optional) and twotheta values below','print, dcs_cryst_align_qlength2( Ei, twotheta, Ef=Ef )']
        'calc_com_2theta1':  self->calculator_insert,[';enter Ei (Ef optional), h, k, l, a, b, c, alpha, beta, gamma values below',latts,'print, dcs_cryst_align_2theta1( Ei, [ h, k, l ], [ a, b, c, alpha, beta, gamma ], Ef=Ef )']
        'calc_com_2theta2':  self->calculator_insert,[';enter Ei (Ef optional) and Q values below','print, dcs_cryst_align_2theta2( Ei, Q, Ef=Ef )']
        'calc_com_hklangle': self->calculator_insert,[';enter h, k, l, a, b, c, alpha, beta, gamma values below',latts,'print, dcs_cryst_align_angle1( [ h1, k1, l1 ], [ h2, k2, l2 ], [ a, b, c, alpha, beta, gamma ] )']
        'calc_com_hkl':      if self.instrname eq 'macs' then self->calculator_insert,[';enter a3, a4, and E values below',latts,uvstr,'print, dcs_cryst_align_hkl(Ei, Ef, a3, a4, lattparm=[ a, b, c, alpha, beta, gamma ], U=U, V=V)'] $
                             else self->calculator_insert,[';enter psi, twotheta, and E values below',latts,uvstr,'print, dcs_cryst_align_hkl(Ei, Ef, psi, twotheta, lattparm=[ a, b, c, alpha, beta, gamma ], U=U, V=V, /dcs)']
        'calc_com_a3a4':     if self.instrname eq 'macs' then self->calculator_insert,[';enter hkl vector and E values below',latts,uvstr,'print, dcs_cryst_align_a3a4(Ei, Ef, [h, k, l],lattparm=[ a, b, c, alpha, beta, gamma ],U=U,V=V,lefthand=0)'] $
                             else self->calculator_insert,[';enter hkl vector and E values below',latts,uvstr,'print, dcs_cryst_align_a3a4(Ei, Ef, [h, k, l],lattparm=[ a, b, c, alpha, beta, gamma ],U=U,V=V,lefthand=0,/dcs)']
        'calc_com_vecangle': if self.instrname eq 'macs' then self->calculator_insert,[';A3s and A4s are both 2-element arrays','print, dcs_cryst_align_angle2( A3s, A4s )'] $
                             else self->calculator_insert,[';psis and twothetas are both 2-element arrays','print, dcs_cryst_align_angle2( psis, twothetas, /dcs )'] 
        'calc_com_latta':    self->calculator_insert,[';enter Ei, twotheta, h, k, l, alpha, beta, gamma values below',';if a lattice parameter needs to be fixed, enter it as a keyword',abgs,'print, dcs_cryst_align_latta( Ei, twotheta, [ h, k, l ], [ alpha, beta, gamma ], a=a, b=b, c=c )']
        'clearcal':          self->calculator_insert,/clear
        'calccal':    begin
            self->my_widget_control,'calcInput',self.dispBas,get_value=str
            space = '[ '+string(9b)+']'
            str = dm_strip_comments(str)
            line = '' & n_str = n_elements(str)
            hpPtr = ptrarr(n_str)  ;store help and print results
            for i=0,n_str-1 do begin
                ptrexp0 = 'if ptr_valid(hpPtr['+dm_to_string(i)+']) then *(hpPtr['+dm_to_string(i)+'])=[*(hpPtr['+dm_to_string(i)+']),'
                ptrexp1 = '] else hpPtr['+dm_to_string(i)+']=ptr_new('
                tmp = str[i]
                if strlen(tmp) eq 0 then continue
                if strmid(line,0,1,/reverse) eq '$' then begin  ;remove trailing $
                   line = strmid(line,0,strlen(line)-1)+tmp            
                endif else begin
                   if stregex(tmp,'^help'+space+'*,',/fold_case,/boolean) then begin
                      tmp = tmp+',output=dcs_mslice_calchelp & '+ptrexp0+'dcs_mslice_calchelp'+ptrexp1+'dcs_mslice_calchelp)'
                   endif else begin
                      tmp2 = stregex(tmp,'^print'+space+'*,',/fold_case,length=len)
                      if len[0] gt 0 then begin
                         tmp = 'dcs_mslice_calcprint=strjoin(dm_to_string(['+strmid(tmp,len[0])+']),",  ")& '+ptrexp0+'dcs_mslice_calcprint'+ptrexp1+'dcs_mslice_calcprint)'
                      endif else if strmid(tmp,0,1,/reverse) eq '=' then begin
                         tmp = 'dcs_mslice_ycal='+strmid(tmp,0,strlen(tmp)-1)+'&ny=n_elements(dcs_mslice_ycal)&hpPtr['+dm_to_string(i)+$
                            ']=ptr_new("="+(ny gt 1?"[":"")+dm_to_string(dcs_mslice_ycal,separator=", ")+(ny gt 1?"]":""))'
                      endif else begin
                         tmp1 = tmp
                         repeat begin
                            tmp2 = stregex(tmp1,'(\([^(]*[=,]+[^(]*\))',length=len)
                            if len[0] gt 0 then tmp1 = strmid(tmp1,0,tmp2[0])+strmid(tmp1,tmp2[0]+len[0],strlen(tmp1)-tmp2[0]-len[0])
                         endrep until len[0] le 0 
                         idlword = '^(if[\( ]+|end|goto +|case +|for +|while[\( ]+|repeat[\( ]+|switch +|continue$|break$)'
                         if (strpos(tmp1,'=') eq -1) and (strpos(tmp1,',') eq -1) and (strpos(tmp1,'->') eq -1) and (~stregex(tmp1,idlword,/fold_case,/boolean)) $
                            and (~stregex(tmp1,' +begin$',/fold_case,/boolean)) and (~stregex(tmp1,'^end$',/fold_case,/boolean)) then begin
                            tmp = 'dcs_mslice_ycal='+tmp+'&ny=n_elements(dcs_mslice_ycal)&hpPtr['+dm_to_string(i)+$
                                  ']=ptr_new("="+(ny gt 1?"[":"")+dm_to_string(dcs_mslice_ycal,separator=", ")+(ny gt 1?"]":""))'
                         endif
                      endelse
                   endelse
                   line = line+'&'+tmp
                endelse
            endfor
            if strlen(line) eq 0 then return
            ok = execute(line,1,1)
            if ok then begin
               ind_hp = where(ptr_valid(hpPtr),n_hp)
               for i=n_hp-1,0,-1 do begin
                   if strmid((*(hpPtr[ind_hp[i]]))[0],0,1) eq '=' then begin
                      while strmid(str[ind_hp[i]],0,1,/reverse) eq '=' do str[ind_hp[i]] = strmid(str[ind_hp[i]],0,strlen(str[ind_hp[i]])-1)
                      str[ind_hp[i]] = '; '+str[ind_hp[i]]+*(hpPtr[ind_hp[i]])
                   endif else begin 
                      if ind_hp[i] eq n_str-1 then str = [str[0:ind_hp[i]],'; '+*(hpPtr[ind_hp[i]])] $
                      else str = [str[0:ind_hp[i]],'; '+*(hpPtr[ind_hp[i]]),str[(ind_hp[i]+1):(n_elements(str)-1)]]
                   endelse
               endfor
               if n_hp gt 0 then self->my_widget_control,'calcInput',self.dispBas,set_value=str
               widget_control,event.id,sensitive=0
            endif else ok = dialog_message(!error_state.msg,/error,dialog_parent=self.tlb,/center)
            ptr_free,hpPtr
          end
        'status':  begin  ;context menu
            if ptr_valid(self.dataStrPtr) then qty = 'qty' else qty = ''
            extravaxis = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
            if self.extravaxis_yn[1] then temp = 'temp' else temp = ''
            if keyword_set(self.macshistomode) then begin
               hischan = 'hischan'
               rawint  = 'rawint' 
            endif else begin
               hischan = ''
               rawint  = ''
            endelse
            if (self.samp_typ ge 2) then dcor = 'dcor' else dcor = ''
            if (self.samp_typ eq 0) then neg2th = 'neg2th' else neg2th = ''
            if (self.samp_typ gt 0) then qoffth = 'qoffth' else qoffth = ''
            if (self.samp_typ gt 0) and (self.view_u1fold or ((self.fold_type lt 2) and (self.view_u2fold or (extravaxis and self.view_u3fold)))) then fold = 'fold' else fold = ''
            if (self.samp_typ ge 2) or ((self.samp_typ ne 0) and (self.instrname eq 'macs')) then pow2cry = 'pow2cry' else pow2cry = ''
            if (self.samp_typ ge 2) or ((self.samp_typ ne 0) and (self.instrname eq 'macs')) then datr = 'datr' else datr = ''
            if (self.samp_typ ge 2) or (self.instrname eq 'macs') or (self.samp_typ eq -1)   then bgtol = 'bgtol' else bgtol = ''
            if (self.samp_typ ge 2) or (self.instrname eq 'macs') or (self.samp_typ eq -1)   then bgtolmd = 'bgtolmd' else bgtolmd = ''
            if self.bin_zeroerror and ((self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand')) then binzero =['zroerr','statsize','statall'] else binzero = 'zroerr'
            if ptr_valid(self.absp_info) then absp = ['absp','l1fac'] else absp = 'absp'
            case self.instrname of 
               'dcs':  begin
                       status = [qty,'eief','int',temp,'userf','bin','bincs','extb','binth',binzero,'montyp','bgrate','bgdet','bgtch','bgcan','ssfac',bgtol,bgtolmd,'van','mask','eint',absp,fold,'eadj','edep',dcor,datr,'egc']
                       option = ['bitf','2emp','binstep',neg2th,pow2cry,'2thc',qoffth]
                       ftools = ['jumpy']
                       end
               'macs': begin
                       status = [qty,'geom','eief','int','det','histo',hischan,temp,'doff','montyp','moncnt','monlm2','userf','bin','bincs','extb','binth',binzero,'bgcan','ssfac',bgtol,bgtolmd,'van','mask','a3mask',absp,fold,datr,'egc']
                       option = ['2emp',rawint,'binstep',neg2th,'lockluv','checkluv',pow2cry,'savea4','koff','kback',qoffth]
                       end
               'spe':  begin
                       status = [qty,'geom','eief','int','userf','bin','bincs','extb','binth','bgrate','bgdet','bgtch','bgcan','ssfac',bgtol,bgtolmd,'van','mask','eint',absp,fold,'eadj','edep',datr,'egc']
                       option = ['2emp','binstep',neg2th,pow2cry,qoffth]
                       end
               'nxspe':begin
                       status = [qty,'geom','eief','int','lamcor','userf','bin','bincs','extb','binth','bgrate','bgdet','bgtch','bgcan','ssfac',bgtol,bgtolmd,'van','mask','eint',absp,fold,'eadj','edep',datr,'egc']
                       option = ['2emp','binstep',neg2th,pow2cry,qoffth]
                       end
               'inx':  begin
                       status = [qty,'geom','eief','int','userf','bin','bincs','extb','binth','bgrate','bgdet','bgtch','bgcan','ssfac',bgtol,bgtolmd,'van','mask','eint',absp,fold,'eadj','edep',datr,'egc']
                       option = ['inxerev','2emp','binstep',neg2th,pow2cry,qoffth]
                       end
               'wand': begin
                       status = [qty,'eief','doff','userf','bin','bincs','extb','binth',binzero,'bgcan','ssfac',bgtol,bgtolmd,'van','mask',absp,fold,datr,'egc']
                       option = ['2emp','binstep','2thc',pow2cry,qoffth]
                       end
               'mslicedata':begin
                       status = ['bin','bincs','extb','binth','bgcan','ssfac',bgtol,bgtolmd,'mask','voron','egc']
                       ftools = ['binstep','label','unit']
                       end
               else:
            endcase
            if n_elements(status) ne 0 then status = 'status_'+status[where(strlen(status) ne 0)] else return
            if n_elements(option) ne 0 then option = 'status_'+option[where(strlen(option) ne 0)]
            if n_elements(ftools) ne 0 then ftools = 'status_'+ftools 
            self->my_widget_control,'contextMenu',event.id,/destroy
            ;reconstruct context sensitive menu
            menuid = widget_base(event.id,/CONTEXT_MENU,UNAME='contextMenu')
            for i=0,n_elements(status)-1 do void = dm_widget_button(menuid,_extra=self->getstatus(status[i]),uname=status[i])
            for i=0,n_elements(option)-1 do void = dm_widget_button(menuid,_extra=self->getstatus(option[i]),uname=option[i],separator=((i eq 0) and (n_elements(void) ne 0)))
            for i=0,n_elements(ftools)-1 do void = dm_widget_button(menuid,_extra=self->getstatus(ftools[i]),uname=ftools[i],separator=((i eq 0) and (n_elements(void) ne 0)))
            widget_displaycontextmenu,event.id,event.x,event.y,menuid
          end
        'status_qoffth': begin
            new = dm_dialog_input('thickness range threshold:',title='Specify axis thickness range threshold',default=self.qoffsetthreshold,/float,info=$
                  ['This value is for adjusting q-axis title. If the thickness range is less than the threshold,','the axis title will be adjusted accordingly with an offset.'],$
                  dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(new) then begin
               self.qoffsetthreshold = abs(new)
               self->set_allplots,qoffsetthreshold=self.qoffsetthreshold
            endif
          end
        'status_koff':   begin
            new = dm_dialog_input('kidney offset angle:',title='Specify kidney offset angle',default=self.kidney_offset,/float,info=$
                  ['This value is the kidney hardware angle when software angle is 0.','It is only used for checking the kidney angle limit, not for adjusting detector 2theta angle.'],$
                  dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(new) then self.kidney_offset = new
          end
        'status_kback':   begin
            new = dm_dialog_input('kidney backlash:',title='Specify kidney backlash',default=self.kidney_backlash,/float,info=$
              ['This value used for experiment planning. Default is 2 degrees.'],dialog_parent=self.tlb,cancel=cancel)
            if keyword_set(cancel) then return
            if finite(new) then self.kidney_backlash = new else self.kidney_backlash = 2
          end
        'status_bitf':   begin
            self.bitflip = ~self.bitflip
            ok = dialog_message('Bitflip error checking has now been turned '+(['off.','on.'])[self.bitflip],dialog_parent=self.tlb,/info,/center)
          end
        'status_binstep': begin
            self.bin_checkstep = ~self.bin_checkstep
            ok = dialog_message(['Check binning step flag has now been turned '+(['off.','on.'])[self.bin_checkstep],'Binning step will '+(['not ',''])[self.bin_checkstep]+'be checked when making cut, slice and volume plots.'],dialog_parent=self.tlb,/info,/center)
          end
        'status_jumpy':  self->check_jumpy
        'status_inxerev': begin
            self.inxerev = ~self.inxerev
            self.fileSel->setproperty,inxerev=self.inxerev
            info = 'The energy transfer in the INX data file is now defined as '+(['neutron energy gain/loss.','sample energy gain/loss.'])[self.inxerev]
            self->clear_data,infomsg=info
            ok = dialog_message(info,dialog_parent=self.tlb,/info,/center)
          end
        'status_neg2th': begin
            self.negative2th = ~self.negative2th
            ;need to recalculate projections if negative 2theta exists
            if ptr_valid(self.detPosPtr) then begin
               if (min((*self.detPosPtr).two_theta) lt 0) and ptr_valid(self.projStrPtr) then recalc_proj = 1b
            endif
            ok = dialog_message((['Absolute 2theta angle value is now used.','Negative 2theta angle value is now preserved.'])[self.negative2th],dialog_parent=self.tlb,/info,/center)
          end
        'status_2thc': begin
            if (self.instrname ne 'dcs') and (self.instrname ne 'wand') then return
            self.tthcorr = ~self.tthcorr
            ptr_free,self.detPosPtr
            case self.instrname of
                 'dcs':  self.detPosPtr = dm_load_detpos(/dcs, adjust2th=self.tthcorr) 
                 'wand': self.detPosPtr = dm_load_detpos(/wand,adjust2th=self.tthcorr)
                 else:
            endcase
            if ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections
            ok = dialog_message('Detector 2theta correction has now been '+(['disabled.','enabled.'])[self.tthcorr],dialog_parent=self.tlb,/info,/center)
          end
        'status_2emp': begin
            self.extraempcan_yn = ~self.extraempcan_yn
            n_str = dm_to_string(indgen(n_elements(self.bgdata))+1)
            ptr_free,self.bgdata[1:*]
            self->reset_menu,/backgroundmenuonly  ;reset background menu
            ok = dialog_message([(['Only one set of empty can files are now allowed.','Multiple sets of empty can files are now allowed.'])[self.extraempcan_yn],$
                 '','Empty can subtraction will be:',(['I = Isample - ssfactor*Iemptycan,','I = Isample - ssfactor*('+strjoin('factor'+n_str+'*Iemptycan'+n_str,' + ')+$
                 '),'])[self.extraempcan_yn],'where ssfactor is the self-shielding factor.','',(['ssfactor','All factors'])[self.extraempcan_yn]+$
                 ' can be changed from the Parameter menu.'],dialog_parent=self.tlb,/info,/center)
          end
        'status_rawint':begin
            self.macs_rawintn = ~self.macs_rawintn
            info = 'Switch to '+(['normalized intensity.','unnormalized intensity.'])[self.macs_rawintn]
            if (self.macs_rawintn and (self.bin_avgsum ne 1)) or ((~self.macs_rawintn) and (self.bin_avgsum ne -1)) then begin
               self.bin_avgsum = ([-1,1])[self.macs_rawintn]
               info = [info,'The binning method is switched to '+(['Weighted Mean','Arithmetic Mean','Sum'])[self.bin_avgsum+1]+'.']
               self->my_menu_toggle,'bin'+['WeighMean','ArithMean','Sum'],self.bin_avgsum+1
               dm_set_button,widget_info(self.tlb,find_by_uname='binAvg'),(self.bin_avgsum ne 1),set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
            endif
            if self.macs_rawintn then begin
               if self.macsintn_typ eq 2 then begin
                  self.macsintn_typ = 0
                  self->my_menu_toggle,'macsint'+['SPEC','DIFF','SPDF'],self.macsintn_typ
                  self->my_widget_control,'macsintMenu',set_value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')'
                  info = [info,'The detector choice is switched from SPEC+DIFF to SPEC.']
               endif
               info = [info,'The detector efficiency is applied to the weight.','The rebinned intensity and weight are for output only.']
            endif
            self->my_widget_control,'macsintSPDF',sensitive=~self.macs_rawintn
            if ptr_valid(self.dataStrPtr) then self->clear_data,/clear_bg,info=info
            ok = dialog_message(dialog_parent=self.tlb,info,/info,/center)
          end
        'status_lockluv':begin
            self.locklattuv = ~self.locklattuv
            ok = dialog_message(['The lattice parameters and crystal orientation axes have now been '+(['unlocked','locked'])[self.locklattuv]+$
                 '. Loading MACS files will '+(['','not '])[self.locklattuv]+'reset the lattice parameters and crystal orientation axes.'],dialog_parent=self.tlb,/info,/center)
            widget_control,self.projBas,update=0
            geom_old = widget_info(self.projBas,/geometry)
            if self.locklattuv then begin
               extra = dm_tooltip('Use the popup menu to unlock the lattice parameters and crystal orientation.')
               void = widget_button(self.projrid[0],value=self.iconfiles[9],uname='lock_latt',/bitmap,/dynamic_resize,/flat,sensitive=0,_extra=extra)
               void = widget_button(self.projrid[1],value=self.iconfiles[9],uname='lock_uv',/bitmap,/dynamic_resize,/flat,sensitive=0,_extra=extra)
            endif else self->my_widget_control,['lock_latt','lock_uv'],self.projBas,/destroy
            self->reset_geom,proj_old=geom_old
          end
        'status_checkluv':begin
            self.checklattuv = ~self.checklattuv
            ok = dialog_message(['Check flag has now been '+(['unset.','set.'])[self.checklattuv],'Loading files will '+(['not ',''])[self.checklattuv]+$
                 'check for the consistency of lattice parameters and crystal orientation axes among files.'],dialog_parent=self.tlb,/info,/center)
          end
        'status_savea4':begin
            self.macs_savea4 = ~self.macs_savea4
            self->my_widget_control,'savesliceiexy',set_value=(['I,Err,X,Y 4-Column File...','I,Err,X,Y,A3,A4 6-Column File...'])[self.macs_savea4]
            self->my_widget_control,'saveslicexyie',set_value=(['X,Y,I,Err Matrix File...','X,Y,A3,A4,I,Err Matrix File...'])[self.macs_savea4]
            self->my_widget_control,'savecutxye',set_value=(['X,Y,Err 3-Column File...','X,A3,A4,Y,Err 5-Column File...'])[self.macs_savea4] 
            ok = dialog_message('A3 and A4 will '+(['not','now'])[self.macs_savea4]+' be saved in the text ouput of slice, cut, and projection data.',dialog_parent=self.tlb,/info,/center)
          end
        'status_pow2cry':begin
            self.pow2cryexpan = ~self.pow2cryexpan
            if ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) then recalc_proj = 1b    ;need to recalculate projections  
          end  
        'status_label':begin
            name = ['Intensity',(*self.dataPtr).label[2+indgen(n_elements((*self.dataPtr).label)-2)]]+':'
            old  = (*self.dataPtr).label[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]]
            new  = dm_dialog_input(name,title='Set axis labels',default=old,xsize=100,dialog_parent=self.tlb,cancel=cancel,info='Do not include the unit in the label.')
            if keyword_set(cancel) then return
            for i=0,n_elements(new)-1 do if strlen(new[i]) eq 0 then new[i] = old[i]
            (*self.dataPtr).label[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]] = new
            if ptr_valid(self.dataStrPtr) then (*self.dataStrPtr).label[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]] = new
            self->reset_display,/reset
          end
        'status_unit':begin
            name = ['Intensity',(*self.dataPtr).label[2+indgen(n_elements((*self.dataPtr).label)-2)]]+':'
            old  = (*self.dataPtr).unit[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]]
            new  = dm_dialog_input(name,title='Set axis units',default=old,xsize=100,dialog_parent=self.tlb,info = '\AA for angstrom, \HW for hbar omega ...',cancel=cancel)
            if keyword_set(cancel) then return
            (*self.dataPtr).unit[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]] = new
            if ptr_valid(self.dataStrPtr) then (*self.dataStrPtr).unit[[0,2+indgen(n_elements((*self.dataPtr).label)-2)]] = new
          end   
        else:begin
            if stregex(uname,'^status_',/boolean,/fold_case) then ok = self->getstatus(strlowcase(uname),/help) else begin
               passed = self->dm_filetools(uname,event=event)  ;check for file type and file tools events
               if (~ passed) then begin            ;view specified plot window
                  all = self.gr_list->get(/all,count=count)
                  for i=0,count-1 do begin
                      id = self->dm_get_id(all[i])
                      if id eq uname then begin
                         all[i]->totop
                         break
                      endif
                  endfor
               endif
            endelse
          end
    endcase
    if keyword_set(resetu1) then begin
       if (~ keyword_set(islabel)) then self->adjust_viewlabel,0
       if self.samp_typ gt 0 then self.traj_type[0] = dm_set_viewlabel(self.view_u1lab[1],self.view_u1)
    endif
    if keyword_set(resetu2) then begin
       if (~ keyword_set(islabel)) then self->adjust_viewlabel,1
       if self.samp_typ gt 0 then self.traj_type[1] = dm_set_viewlabel(self.view_u2lab[1],self.view_u2)
    endif
    if keyword_set(resetu3) then begin
       if (~ keyword_set(islabel)) then self->adjust_viewlabel,2
       self.traj_type[2] = dm_set_viewlabel(self.view_u3lab,self.view_u3)
    endif
    if keyword_set(resetu1) or keyword_set(resetu2) or keyword_set(resetu3) then begin
       self->reset_droplist
       self->reset_display,/reset
    endif
    if n_elements(reset_fold) ne 0 then begin
       self->my_menu_toggle,'vfold'+['Mirror','Rotation'],(self.fold_type gt 1)
       self->my_menu_toggle,['mrrf_'+['One','Both'],'symrot_'+['Section','Quadrant','Fullmap']],self.fold_type
       if reset_fold eq 2 then begin
          self.view_u1fold = 0 & self.view_u2fold = 0 & self.view_u3fold = 0
       endif
       if keyword_set(reset_fold) then self->reset_projection,/reset,/foldonly
    endif
    if keyword_set(recalc_proj) then self->set_display_button,/clearprojection ;need to recalculate projections
    if strlowcase(uname) ne 'calcprj' then self.error = 0b    ;reset the error flag   
end

pro dcs_mslice::eadjust,option
    old = self.eadjust
    self.eadjust = 0>(option)<2
    if self.eadjust eq 2 then begin
       new  = dm_dialog_input('specify the apparent elastic peak position:',title='Specify elastic peak position',default=self.eadjust_spec,/float,info='in meV',dialog_parent=self.tlb)
       if finite(new) then $
          self.eadjust_spec = new $
       else begin
          self.eadjust = 0 & old = 2
          ok = dialog_message('No elastic peak adjustment is specified.',dialog_parent=self.tlb,/error,/center)
       endelse
    endif
    if old eq self.eadjust then return
    self->my_menu_toggle,'epad'+['None','Auto','Spec'],self.eadjust
    dm_set_button,widget_info(self.tlb,find_by_uname='eAdjust'),(self.eadjust ne 0)
    self->my_widget_control,'epadCheckfirst',sensitive=(self.eadjust ne 0)
    self->clear_data ;need to reload data
end

pro dcs_mslice::getproperty,dirs=dirs,error=error,ftype=ftype,instrgeom=instrgeom,latticeparm=latticeparm,dataPtr=dataPtr,macshistomode=macshistomode,tlb=tlb,uname=uname
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,dialog_parent=self.tlb,/center)
       error = 1
       return
    endif
    error = 0
    if arg_present(tlb)           then tlb = self.tlb
    if arg_present(uname)         then uname = self.uname
    if arg_present(instrgeom)     then instrgeom = self.instrgeom
    if arg_present(dirs)          then dirs = self.dirs
    if arg_present(ftype)         then ftype = self.instrname
    if arg_present(latticeparm)   then latticeparm = [self.latt_parm,self.orie_u,self.orie_v]
    if arg_present(macshistomode) then macshistomode = self.macshistomode
    if arg_present(dataPtr)       then begin
       if self.samp_typ eq -1 then begin
          dataPtr = self.dataPtr 
       endif else begin
          error = 1
          dataPtr = ptr_new()
       endelse
    endif
end

;get the status string or keywords for context sensitive menu
function dcs_mslice::getstatus,name,help=help
    status = ''
    name = strlowcase(name) 
    if keyword_set(help) then begin
       case name of
           'status_qty':    begin
                            help,(*self.dataStrPtr),/struct,output=status
                            status = ['*** Data pointer (*self.dataStrPtr):',status[1:*]]
                            if ptr_valid(self.projStrPtr) then begin
                               help,(*self.projStrPtr),/struct,output=tmp
                               tmp = ['*** Projection pointer (*self.projStrPtr):',tmp[1:*]]
                               status = [status,tmp]
                            endif
                            for i=0,total(ptr_valid(self.bgdata),/int)-1 do begin
                                help,(*self.bgdata[i]),/struct,output=tmp
                                tmp = ['*** Empty can data pointer (*self.bgdata['+dm_to_string(i)+']):',tmp[1:*]]
                                status = [status,tmp]
                            endfor
                            end
           'status_userf':  status = ['Go to "Option->'+(lmgr(/runtime)?'User Macro':'User Macro/Function/Procedure')+'" to change this setting.']
           'status_ssfac':  status = ['Go to "Parameter->Change '+(['Self-shielding Factor','Background Subtraction Factor...'])[self.samp_typ eq -1]+'" to change this setting.']
           'status_zroerr': status = ['Go to "Option->Binning Method->Zero Intensity Error" to change this setting.']
           'status_statsize':status= ['Go to "Option->Binning Method->Zero Intensity Error->Estimate Size" to change this setting.']
           'status_statall':status = ['Go to "Option->Binning Method->Zero Intensity Error->Apply Estimate to All Intensities" to change this setting.']
           'status_geom':   status = ['Go to "Option->Instrument Geometry" to change this setting.']
           'status_eief':   if self.instrname eq 'wand' then $
                            status = ['Go to "Parameter->Change WAND Incident Wavelength..." to change this setting.'] $
                            else $
                            status = ['fixed '+(['Ei','Ef'])[self.instrgeom]]
           'status_histo':  status = ['Go to "Option->View Intensity As->MACS Data Mode" to switch to '+(['histogram','normal'])[self.macshistomode ne 0]+' mode.']   
           'status_hischan':status = ['Go to "Parameter->Change MACS Histogram Channel Width..." to change this setting.']                
           'status_montyp': status = ['Go to "Option->Monitor Count->Monitor Choice" to change this setting.']
           'status_moncnt': status = ['Go to "Option->Monitor Count" to change this value.']
           'status_monlm2': status = ['Go to "Option->Monitor Count->Apply Monitor Lambda/2 Correction" to change this value.']
           'status_bgrate': status = ['Go to "Background->Set Overall Background Rate..." to change this value',' or',$
                                      'go to "Background->Load Dark Count File(s)" to change this value',' or',$
                                      'go to "Background->Clear Background" to clear all backgrounds.']
           'status_bgtch':  status = ['Go to "Background->Load Time Channel Background File(s)" to change this value',' or',$
                                      'go to "Background->Clear Background" to clear all backgrounds.']
           'status_bgdet':  status = ['Go to "Background->Load Detector Background File(s)" to change this value',' or',$
                                      'go to "Background->Clear Background" to clear all backgrounds.']
           'status_bgtol':  status = ['Go to "Parameter->Change '+(['Empty Can','Background'])[self.samp_typ eq -1]+' Subtraction Tolerance Value..." to change the tolerance value.']
           'status_bgtolmd':status = ['Go to "Option->'+(['Empty Can','Background'])[self.samp_typ eq -1]+' Subtraction Tolerance Method" to change this option.']
           'status_bgcan':  status = ['Go to "Background->Load '+(['Empty Can File(s)','ASCII Background Data File...'])[self.samp_typ eq -1]+'" to change this value',' or',$
                                      'go to "Background->Clear Background" to clear all backgrounds.']
           'status_van':    begin
                            if self.instrname eq 'macs' then $
                            status = ['Go to "Parameter->Change MACS Detector Efficiency..." to change the detector efficiencies,',' or',$
                                      'go to "Background->Load Vanadium File(s)" to calculate the detector efficiencies from data files,',' or',$
                                      'go to "Background->Clear Vanadium" to reset back to the default values.'] $  
                            else $
                            status = ['Go to "Background->Load Vanadium File(s)" to change this value',' or',$
                                      'go to "Background->Clear Vanadium" to clear it.']
                            
                            end
           'status_mask':   begin
                            if self.instrname eq 'mslicedata' then $
                            status = ['Go to Mask menu to change this setting.'] $  
                            else $
                            status = ['Go to Mask menu to change this setting',' or',$
                                      'go to "Mask->Clear '+(['','Detector'])[self.instrname eq 'macs']+'Mask" to clear the existing detector mask.']
                            end
           'status_a3mask': status = ['Go to "Mask->Set Mask->Mask A3 by Angle" to set it',' or',$
                                      'go to "Mask->Clear a3 Mask" to clear the existing A3 mask.']                          
           'status_doff':   status = ['Go to "Parameter->Change '+(*self.ftypename)[self.ftype]+' Detector Offset Angle..." to change this setting.']                        
           'status_int':    status = ['Go to "Option->View Intensity As" to change this setting.']
           'status_det':    status = ['Go to "Option->View Intensity As->MACS Detector Choice" to change this setting.'] 
           'status_temp':   status = ['Go to "Option->Viewing Axis->Temperature Choice" to change this setting.'] 
           'status_eint':   status = ['Go to "Option->View Intensity As" to change this setting.']
           'status_lamcor': begin
                            status = (['0: Constant time bins with no ki/kf correction.',$
                                       '1: Constant time bins with ki/kf correction.',$
                                       '2: Constant energy bins with no ki/kf correction.',$
                                       '3: Constant energy bins with ki/kf correction.'])
                            if ptr_valid(self.dataStrPtr) and (self.lamcorrnxspe ge 0) and (self.lamcorrnxspe le 3) then $
                            status = status[self.lamcorrnxspe] 
                            end
           'status_bin':    status = ['Go to "Option->Binning Method" to change this setting.']
           'status_bincs':  status = ['Go to "Option->Binning Method->Constant Intervals" to change this setting.']
           'status_extb':   status = ['Go to "Option->Binning Method->Use External Binning Library" to change this setting.']
           'status_binth':  status = ['Go to "Parameter->Change Binning Threshold..." to change this setting.']
           'status_eadj':   status = ['Go to "Option->Elastic Peak Position Adjustment" to change this setting.']
           'status_edep':   status = ['Go to "Option->E-Dependent Detector Efficiency Correction" to change this setting.']
           'status_dcor':   status = ['Go to "Option->Dark Angle Correction" to change this setting.']
           'status_datr':   status = ['Go to "Option->Show Dark Angle Trajectory" to change this setting.']
           'status_absp':   status = ['Go to "Option->Apsorption Correction" to change this setting.']
           'status_l1fac':  status = ['Go to "Option->Apsorption Correction" to change this value.']
           'status_fold':   status = ['Go to "Option->Folding Type" to change this setting.']
           'status_voron':  status = ['Go to "Option->Allow Voronoi Diagram" to change this setting.']
           'status_egc':    status = ['Go to "Option->Slice Plot Empty Grid Color" to change this setting.']
           else:
       endcase
       ok = dialog_message(dialog_parent=self.tlb,status,/info,/center)
       return,status
    endif
    if !version.os_family eq 'Windows' then sep=string(9b)+': ' else sep=': '
    case name of
        'status_qty':     begin
                          help,(*self.dataStrPtr).qty,output=tmp
                          tmp = strtrim(tmp[0],2) & pos = strpos(tmp,' ')
                          status = 'qty,err'+sep+strmid(tmp,pos+1,strlen(tmp)-1-pos)
                          end
        'status_userf':   status = (lmgr(/runtime)?'user macro':'user macro/function/procedure')+sep+((strlen(self.userf_str) ne 0)?'yes':'none')
        'status_ssfac':   status = (['self-shielding factor','background subtraction factor'])[self.samp_typ eq -1]+sep+(ptr_valid(self.ssfacPtr)?file_basename((*self.ssfacPtr).file):dm_to_string(self.ssfac))
        'status_zroerr':  status = 'zero intensity error'+sep+(['user specify'+(['',' ('+dm_to_string(self.zero_error[0])+')'])[finite(self.zero_error[0])],'statistical estimate'])[self.bin_zeroerror]  
        'status_statsize':status = 'statistical estimate size'+sep+(['automatic','user specify ('+dm_to_string(self.bin_statrange[1])+')'])[self.bin_statrange[0]]
        'status_statall': status = 'apply estimate to all intensities'+sep+(['no','yes'])[self.bin_zeroerror and self.bin_statrange[2]]
        'status_geom':    status = 'instrument geometry'+sep+(['direct','inverse'])[self.instrgeom]
        'status_eief':    begin
                          status = 'fixed '+(['Ei','Ef'])[self.instrgeom]+sep+(finite(self.eief)?(dm_to_string(self.eief)+' meV'):('unknown'))
                          if finite(self.eief) and ((self.instrname eq 'dcs') or (self.instrname eq 'wand')) then begin
                             status = status+' ('+dm_to_string(dm_e2lambda(self.eief),resolution=3)+' '+string('c5'XB)+')'
                          endif
                          end
        'status_montyp':  begin
                          case self.instrname of 
                             'dcs':  status = 'monitor type'+sep+(['FC0','BM1','duration'])[self.mon_typ[0]]+'/'+(['sum','integrated intensity'])[self.mon_sumfit[0]] 
                             'macs': status = 'monitor type'+sep+(['time','monitor','pulse number'])[self.mon_typ[1]]+'/'+(['automatic','specify'])[self.mon_sumfit[1]] 
                             else:   status = 'monitor type'+sep+'none'
                          endcase
                          end
        'status_moncnt':  begin
                          if self.mon_typ[1] eq 0 then $
                          status = 'monitor count'+sep+([finite(self.macs_mon[0,0])?dm_to_string(self.macs_mon[0,0])+' sec':'n/a',dm_to_string(self.macs_mon[1,0])+' sec'])[self.mon_sumfit[1]] $   
                          else $
                          status = 'monitor count'+sep+([finite(self.macs_mon[0,1])?dm_to_string(self.macs_mon[0,1],/exponent):'n/a',dm_to_string(self.macs_mon[1,1],/exponent)])[self.mon_sumfit[1]] 
                          end
        'status_monlm2':  status = 'monitor lambda/2 correction'+sep+(['no','yes'])[self.macs_lamb2]
        'status_bgrate':  begin
                          status = 'dark count'+sep
                          if ptr_valid(self.bgratePtr) then $
                             status = status+(['File',dm_to_string(*self.bgratePtr)+' cts/hr/det'])[n_elements(*self.bgratePtr) eq 1] $
                          else $
                             status = status+'none'
                          end
        'status_bgtch':   status = 'time channel background'+sep+(ptr_valid(self.bgtchnPtr)?'yes':'none')
        'status_bgdet':   status = 'detector background'+sep+(ptr_valid(self.bgdetsPtr)?'yes':'none')
        'status_bgtol':   begin
                          if self.samp_typ eq -1 then tmp = (*self.data_tolerance) else if self.instrname eq 'macs' then tmp = self.macs_tolerance[0:1+self.samp_typ,0] else tmp = self.psi_tolerance[0]
                          status = (['empty can','background'])[self.samp_typ eq -1]+' subtraction tolerance value'+sep+dm_to_string(tmp,separator=', ')
                          end
        'status_bgtolmd': status = (['empty can','background'])[self.samp_typ eq -1]+' subtraction tolerance method'+sep+(['within tolerance or closest','within tolerance of the closest'])[self.empt_tol_optn]
        'status_bgcan':   if self.samp_typ eq -1 then $
                          status = 'background'+sep+((ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]))?'yes':'none') $
                          else $
                          status = 'empty can'+sep+(['none','yes',dm_to_string(total(ptr_valid(self.bgdata)))+' sets'])[total(ptr_valid(self.bgdata))<2]
        'status_van':     begin
                          if self.instrname eq 'macs' then $
                          status = 'detector efficiency'+sep+(ptr_valid(self.eff1)?'vanadium file':'default') $
                          else $
                          status = 'detector efficiency'+sep+(ptr_valid(self.eff)?'yes':'none')
                          end
        'status_mask':    begin
                          status = 'detector mask'+sep+((ptr_valid(self.mask) or ptr_valid(self.mask_ar))?'yes':'none')
                          if (self.instrname eq 'dcs') and (total(self.dcsdetmask) ne 0) then begin
                             status = 'dectector mask'+sep
                             if self.dcsdetmask[0] then status = status+'neg,'
                             if self.dcsdetmask[4] then status = status+'upper,lower,' else begin
                                if self.dcsdetmask[1] then status = status+'lower,'
                                if self.dcsdetmask[3] then status = status+'upper'
                             endelse
                             if self.dcsdetmask[2] then status = status+'central,'
                             if strmid(status,0,1,/reverse) eq ',' then status=strmid(status,0,strlen(status)-1)
                          endif 
                          if (self.instrname eq 'macs') and self.macs_ptaionly then status = 'dectector mask'+sep+'PTAI only'
                          if self.mask_darkangle then if strmid(status,3,4,/reverse) eq 'none' then status = 'detector mask'+sep+'dark angle' else status = status+'/dark angle'
                          if self.instrname eq 'mslicedata' then begin
                             if ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]) then begin
                                if total(finite(self.mask_filler[0:1])) gt 0 then status= 'data masking'+sep+'yes, '+(['signal data only','signal+background'])[self.mask_filler[2]] $
                                else status = 'data masking'+sep+'no'
                             endif else status = 'data masking'+sep+((total(finite(self.mask_filler[0:1])) gt 0)?'yes':'no')
                          endif
                          end
        'status_a3mask':  status = 'a3 mask'+sep+(ptr_valid(self.a3mask)?'yes':'none')     
        'status_histo':   status = 'data mode'+sep+(['normal','histogram','event'])[self.macshistomode]
        'status_hischan': status = 'histogram channel width and number'+sep+dm_to_string(self.macshistowidth)+([' us',' ms'])[self.macshistounit]+', '+dm_to_string(self.macshistonchan)+' chan'+(['','s'])[self.macshistonchan gt 1]+(['',', no trailing chans'])[self.macshistocut]
        'status_doff':    begin
                          if self.instrname eq 'wand' then doff = -(*self.detPosPtr).two_theta[0] $
                          else if self.instrname eq 'macs' then doff = (*self.detPosPtr).two_theta[0]
                          if n_elements(doff) ne 0 then $
                             status = 'detector offset angle'+sep+dm_to_string(doff,resolution=3)+string('b0'XB) $
                          else $
                             status = 'detector offset angle'+sep+'none'
                          end
        'status_int':     begin
                          status = strlowcase(self.intn_typ_str[self.intn_typ])
                          if ptr_valid(self.absp_info) then begin
                             if ((*self.absp_info)[0] ge 0) and ((*self.absp_info)[0] lt n_elements(self.absp_npts)) and self.absp_asint then status = 'transmission'
                          endif
                          status = 'intensity type'+sep+status
                          end
        'status_det':     status = 'intensity detector'+sep+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]
        'status_temp':    status = 'temperature choice'+sep+(['sample','control','setpoint'])[self.macstemp_typ]+(self.tempavg_yn?'/avg':'')
        'status_eint':    status = 'E-integrated intensity'+sep+(['no','yes'])[self.eint_yn]
        'status_lamcor':  status = 'NXSPE ki/kf correction'+sep+(ptr_valid(self.dataStrPtr)?dm_to_string(self.lamcorrnxspe):'none')
        'status_bin':     status = 'binning method'+sep+(['weighted mean','arithmetic mean','sum'])[self.bin_avgsum+1]
        'status_bincs':   status = 'constant binning interval'+sep+(['no','yes'])[self.bin_conststep]
        'status_extb':    status = 'use external binning library'+sep+(['no','yes'])[self.bin_extlib]
        'status_binth':   status = 'binning threshold'+sep+dm_to_string(self.binthreshold) 
        'status_eadj':    begin
                          status = 'elastic peak adjustment'+sep+(['no','automatic','specified:'+dm_to_string(self.eadjust_spec)])[self.eadjust]
                          if self.eadjust ne 0 then status = status+'('+(['check all','check 1st only'])[self.eadjust_checkfirst]+')'
                          end
        'status_edep':    status = 'E-dep efficiency correction'+sep+(['no','yes'])[self.edep_yn]
        'status_dcor':    status = 'dark angle correction'+sep+(['no','yes'])[self.dcor_yn]
        'status_datr':    status = 'dark angle trajectory'+sep+(['no','yes'])[self.datr_yn]
        'status_voron':   status = 'allow Voronoi diagram'+sep+(['no','yes'])[self.allow_voronoi]
        'status_egc':     status = 'slice empty grid color'+sep+(['none','average of minimum 2 neighbors','average of minimum 3 neighbors'])[self.ppvc_choice]
        'status_absp':    begin
                          status = 'absorption correction'+sep
                          if ptr_valid(self.absp_info) then begin
                             if (*self.absp_info)[0] ge 0 and (*self.absp_info)[0] lt n_elements(self.absp_npts) then $
                                status = status+(['cylinder','annulus','sphere','flat plate'])[(*self.absp_info)[0]] $
                             else $
                                status = status+'none'
                          endif else $
                             status = status+'none'
                          end
        'status_l1fac':   status = 'incident beam path factor'+sep+dm_to_string(self.absp_l1factor)
        'status_fold':    begin
                          status = 'folding type'+sep+(['mirror reflection','symmetry rotation'])[self.fold_type gt 1]
                          status = status+'/'+(['one side','both sides','1st section','1st quadrant','360'+string('b0'xb)])[self.fold_type]
                          if self.fold_type eq 2 then status = status+'/'+(['u1',dm_to_string(self.symrot_startvec)+string('b0'xb)])[self.symrot_startvec ne 0] $
                          else if self.fold_type eq 4 then status = status+'/'+(['minimum','n_fold-1'])[self.numrot_optn]
                          end                  
        'status_bitf':    status = 'Check Bitflip Error'+sep+(['no','yes'])[self.bitflip]  
        'status_binstep': status = 'Check Binning Step'+sep+(['no','yes'])[self.bin_checkstep]
        'status_jumpy':   status = 'Scan for Jumpy Files...'
        'status_inxerev': status = 'Data File Energy Transfer Definition'+sep+(['neutron energy gain/loss','sample energy gain/loss'])[self.inxerev]    
        'status_neg2th':  status = 'Preserve Negative 2theta Angle Value'+sep+(['no','yes'])[self.negative2th]
        'status_2thc':    status = 'Detector 2theta Correction'+sep+(['no','yes'])[self.tthcorr]
        'status_2emp':    status = 'Allow Multiple Sets of Empty Can Files'+sep+(['no','yes'])[self.extraempcan_yn]
        'status_rawint':  status = 'Use Unnormalized Intensity for Event Mode Data'+sep+(['no','yes'])[self.macs_rawintn]
        'status_lockluv': status = 'Lock Lattice Parameters and Crystal Oreintation Axes'+sep+(['no','yes'])[self.locklattuv]
        'status_checkluv':status = 'Check Lattice Parameters and Crystal Oreintation Axes in Files'+sep+(['no','yes'])[self.checklattuv]
        'status_savea4':  status = 'Save A3 && A4 in Text Output of Slice, Cut, and Projection'+sep+(['no','yes'])[self.macs_savea4]
        'status_pow2cry': status = 'Expand Single Crystal Projection of Powder Data'+sep+(['no','yes'])[self.pow2cryexpan]
        'status_label':   status = 'Change Axis Labels...'
        'status_unit':    status = 'Change Axis Units...'
        'status_qoffth':  status = 'Q-axis Renaming Thickness Range Threshold...'+sep+dm_to_string(self.qoffsetthreshold)
        'status_koff':    status = 'Kidney Offset Angle...'+sep+dm_to_string(self.kidney_offset,resolution=3)+string('b0'XB)
        'status_kback':   status = 'Kidney Backlash...'+sep+dm_to_string(self.kidney_backlash,resolution=3)+string('b0'XB)
        else:
    endcase
    ;change the string to keywords
    result = {value:status}
    if self.idl_version ge 5.6 then begin
       tmp = strsplit(status,':',/extract,count=count)
       tmp = strtrim(tmp[n_elements(tmp)-1],2)
       set_button = ((tmp ne 'no') and (tmp ne 'none') and (tmp ne 'n/a'))
       if name eq 'status_eief'   then set_button = finite(self.eief)
       if name eq 'status_ssfac'  then set_button = ((self.ssfac ne 1.0) or ptr_valid(self.ssfacPtr))
       if name eq 'status_zroerr' then set_button = ((self.zero_error[0] ne 0.0) or self.bin_zeroerror)
       if name eq 'status_binth'  then set_button = (self.binthreshold ne 0.0)
       if name eq 'status_doff'   then set_button = ((*self.detPosPtr).two_theta[0] ne 0)
       if name eq 'status_histo'  then set_button = (self.macshistomode ne 0)
       if name eq 'status_bgtol'  then set_button = (self.samp_typ eq -1)?(max(*self.data_tolerance) gt 0):(self.instrname eq 'macs'?(max(self.macs_tolerance[0:1+self.samp_typ,0]) gt 0):(self.psi_tolerance[0] gt 0))
       if name eq 'status_koff'   then set_button = 0
       if name eq 'status_kback'  then set_button = 0
       if name eq 'status_qoffth' then set_button = 0
       if name eq 'status_l1fac'  then set_button = (self.absp_l1factor ne 1)
       if count eq 1 then set_button = 0
       result = create_struct(result,'set_button',set_button)
    endif
    if (self.idl_version ge 8.2) and (!version.os_family eq 'Windows') then begin
       if name eq 'status_jumpy'   then result = create_struct(result,'imagefile',self.iconfiles[1])
       if name eq 'status_lockluv' then result = create_struct(result,'imagefile',self.iconfiles[8+self.locklattuv])
    endif
    return,result
end

;get the device symbol font in different operating systems
function dcs_mslice::getsymbolfont,fsize
    fonts=['symbol*',$
           '-adobe-symbol-medium-r-normal--14-140-75-75-p-85-adobe-fontspecific',$
           '-adobe-symbol-medium-r-normal--14-100-100-100-p-85-adobe-fontspecific']
    fonts[0] = fonts[0]+dm_to_string(fsize+2,/int)
    catch, myerror
    if myerror ne 0 then begin
       return,fonts[0]
    end
    result = fonts[0]
    switch !version.os_family of
        'Windows':break
        'unix':
        'MacOS':begin
                spawn,'xlsfonts',result,errresult,exit_status=error
                if error ne 0 then begin
                   old_win = !d.window
                   device,get_fontnames=result,set_font='*'    ;will pop up a window, reason for using xlsfonts first
                   if !d.window ne old_win then wdelete, !d.window
                endif
                nlist = n_elements(result)
                tmp   = bytarr(nlist)
                for i=0,nlist-1 do begin
                    tmp[i] = ((strpos(strlowcase(result[i]),'symbol'))<0)+1
                endfor
                index = where(tmp,count)
                if count ne 0 then f_list=result[index]
                nlist = n_elements(f_list)
                if nlist ne 0 then begin
                   tmp = intarr(nlist)
                   for i=0,nlist-1 do tmp[i]=dm_to_number((strsplit(f_list[i],'[A-Za-z-]',/regex,/extract))[0],/int)
                   void = min(abs(tmp-fsize+2),index)
                   fonts[1] = f_list[index[0]]
                endif
                result = fonts[1]
                break
                end
        else:
    endswitch
    return,result
end

;save the current plot window to open a new plot window
pro dcs_mslice::keepcurrentplot,oldcurrent=oldcurrent,oldkeep=oldkeep,reset=reset
    if keyword_set(reset) then begin
       if n_elements(oldcurrent) ne 0 then begin
          self.current = oldcurrent
          self.current->setproperty,keep=oldkeep
       endif
    endif else if obj_valid(self.current) then begin
       oldcurrent = self.current
       self.current->getproperty,keep=oldkeep
       self.current->setproperty,/keep
    endif
end

;load data file for self.samp_typ=-1, if background keyword is set, load it as background
pro dcs_mslice::load_data,file,data=data,weight=weight,label=label,unit=unit,is_uniq=is_uniq,init=init,background=background
    if self.tlb ne 0 then tlb = self.tlb
    if n_elements(file) eq 0 then begin
       path = self.dirs[1]
       file = dm_choose_file('*',/read,dialog_parent=tlb,title='Please choose an ascii data file',path=path,/multi)
       if strlen(file[0]) eq 0 then return
       self->setproperty,workDir=path
    endif
    for i_file=0,n_elements(file)-1 do begin
        if n_elements(xdat) ne 0 then tmp = temporary(xdat)
        if n_elements(ydat) ne 0 then tmp = temporary(ydat)
        ftype = ''
        if strmatch(file[i_file],'*.inx',/fold_case) then ftype = 'inx' $
        else if strmatch(file[i_file],'*.spe',/fold_case) then ftype = 'spe'
        case ftype of 
             'inx':begin
                   dm_load_inx,file[i_file],zdat=zdat,zerr=zerr,xdat=xdat,ydat=ydat,comment=title,temperature=temperature,error=error,group_leader=tlb
                   if error ne 0 then return
                   if temperature ne 0 then legend = 'T='+dm_to_string(temperature)+' K'           
                   end
             'spe':begin
                   dm_load_spe,file[i_file],zdat=zdat,zerr=zerr,xdat=xdat,ydat=ydat,xhist=xhist,yhist=yhist,xtit=xtit,ytit=ytit,ztit=ztit,error=error,group_leader=tlb
                   if error ne 0 then return
                   if xhist then xdat = (xdat[0:(n_elements(xdat)-2)]+xdat[1:(n_elements(xdat)-1)])/2.0
                   if yhist then ydat = (ydat[0:(n_elements(ydat)-2)]+ydat[1:(n_elements(ydat)-1)])/2.0
                   if xtit eq 'Phi Grid' then xtit='2theta'
                   if ytit eq 'Energy Grid' then ytit='E'
                   label = [ztit,'dI',xtit,ytit]
                   unit  = ['','','','']
                   for i=0,3 do begin
                       tmp = stregex(label[i],'(.*)\((.*)\) *',/extract,/subexpr)
                       if strlen(tmp[0]) ne 0 then begin
                          label[i] = tmp[1]
                          unit[i]  = tmp[2]
                       endif
                   endfor  
                   if strlowcase(xtit) eq '2theta' then unit[2] = '\deg'
                   if stregex(ytit,'^ *(e|energy) *$',/boolean,/fold_case) then unit[3] = 'meV'
                   end
             else: begin  ;multi-column data
                   openr,funit,file[i_file],/get_lun
                   str = '' & info = '' & nskip = 0L
                   while ~ eof(funit) do begin
                      readf,funit,str
                      tmp = dm_to_number(strsplit(str,' ,'+string(9b),/extract))
                      if total(finite(tmp,/nan)) eq 0 then begin
                         ncol = n_elements(tmp)
                         break
                      endif else begin
                         nskip = nskip+1
                         info  = [info,str] 
                      endelse
                   endwhile
                   free_lun,funit
                   nrow0 = file_lines(file[i_file])-nskip
                   if nrow0 le 0 then return
                   data0 = fltarr(ncol,nrow0,/nozero)
                   openr,funit,file[i_file],/get_lun
                   for i=0,nskip-1 do readf,funit,str
                   readf,funit,data0
                   free_lun,funit
                   for i=1,nskip do begin
                       while strmid(info[i],0,1) eq ';' do info[i] = strmid(info[i],1)
                       info[i] = strtrim(info[i],2)
                       tmp = strpos(info[i],string(9b))
                       while tmp ne -1 do begin
                           tmp1 = info[i]
                           strput,tmp1,' ',tmp
                           info[i] = tmp1
                           tmp = strpos(info[i],string(9b))
                       endwhile
                       tmp = strsplit(info[i],' ',/extract,count=count)
                       if count ge ncol then begin
                          label = strarr(ncol)
                          unit  = strarr(ncol)
                          tmpi  = 0
                          for j=0,count-1 do begin
                              tmp1 = stregex(tmp[j],'(.*)\((.*)\) *',/extract,/subexpr)
                              if strlen(tmp1[0]) ne 0 then begin
                                 if strlen(tmp1[1]) gt 0 then begin
                                    if tmpi ge ncol then break
                                    label[tmpi] = tmp1[1]
                                    tmpi=tmpi+1
                                 endif
                                 if strlen(tmp1[2]) gt 0 then begin
                                    unit[0>(tmpi-1)] = tmp1[2]
                                    if tmpi eq ncol then break
                                 endif
                              endif else begin
                                 if tmpi ge ncol then break
                                 label[tmpi] = tmp[j]
                                 tmpi=tmpi+1
                              endelse
                          endfor
                          break
                       endif
                   endfor
                   if nskip ne 0 then info[0] = 'file header:'
                   if n_elements(label) ne 0 then default = 2*stregex(label[0],'^ *(i|intensity) *$',/boolean,/fold_case)+stregex(label[ncol-1],'weight *$',/boolean,/fold_case)
                   type = dm_dialog_input('data structure:',default=default,dialog_parent=tlb,droplist_content=$
                          ptr_new(['X1, X2 ... Intensity, Error','X1, X2 ... Intensity, Error, Weight','Intensity, Error, X1, X2 ...','Intensity, Error, X1, X2 ... Weight']),$
                          is_droplist=1,xsize=180,/return_number,title=file_basename(file[i_file]),/align_center,info=info)
                   switch type of
                      0:
                      1: begin
                         if type eq 1 then begin
                            ncol = ncol-1
                            weight0 = reform(data0[ncol,*])
                            data0 = data0[0:(ncol-1),*]
                            if n_elements(label) eq ncol+1 then label = label[0:(ncol-1)]
                            if n_elements(unit)  eq ncol+1 then unit  = unit[0:(ncol-1)]
                         endif
                         if n_elements(label) eq ncol then label = shift(label,2)
                         if n_elements(unit)  eq ncol then unit  = shift(unit,2)
                         data0 = shift(data0,[2,0])
                         break
                         end
                      3: begin
                         ncol = ncol-1
                         weight0 = reform(data0[ncol,*])
                         data0 = data0[0:(ncol-1),*]
                         if n_elements(label) eq ncol+1 then label = label[0:(ncol-1)]
                         if n_elements(unit)  eq ncol+1 then unit  = unit[0:(ncol-1)]
                         end
                   endswitch
                   end
        endcase
        if n_elements(xdat) ne 0 and n_elements(ydat) ne 0 then begin
           nx    = n_elements(xdat) & ny = n_elements(ydat) & nrow0 = long64(nx)*long64(ny)
           xdat  = xdat#(fltarr(ny)+1)
           ydat  = (fltarr(nx)+1)#ydat
           data0 = transpose([[reform(temporary(zdat),nrow0)],[reform(temporary(zerr),nrow0)],[reform(temporary(xdat),nrow0)],[reform(temporary(ydat),nrow0)]])
           ncol  = 4
        endif
        if i_file eq 0 then begin
           data = temporary(data0) 
           nrow = temporary(nrow0)
           if n_elements(weight0) ne 0 then weight = temporary(weight0)
        endif else begin
           data = [[temporary(data)],[temporary(data0)]]
           nrow = nrow+temporary(nrow0)
           if n_elements(weight0) ne 0 then weight = [temporary(weight),temporary(weight0)]
        endelse
    endfor
    data = transpose(data)
    if keyword_set(background) then begin
       if ~keyword_set(init) then self->reset_data,data=data,weight=weight,/background,/no_copy
    endif else begin
       if n_elements(label) ne 0 then begin
          name = ['Intensity',label[2+indgen(n_elements(label)-2)]]+':'
          old  = label[[0,2+indgen(n_elements(label)-2)]]
       endif else begin
          name = ['Intensity','x'+dm_to_string(indgen(ncol-2)+1)]+':'
          old  = ['Intensity',strarr(ncol-2)]
       endelse
       new  = dm_dialog_input(name,title='Set axis labels',default=old,xsize=100,dialog_parent=tlb,cancel=cancel,info=['Do not include the unit in the label.','Every label needs to be specified to be valid.','The labels can be changed in the popup menu.'])
       if ~keyword_set(cancel) then begin
          ind = where(strlen(new) ne 0,count)
          if count eq ncol-1 then label = [new[0],'dI',new[1:*]]
       endif
       if n_elements(label) ne 0 then begin
          name = ['Intensity',label[2+indgen(n_elements(label)-2)]]+':'
          if n_elements(unit) ne 0 then old  = unit[[0,2+indgen(n_elements(unit)-2)]] else old = strarr(ncol-1)
       endif else begin
          name = ['Intensity','x'+dm_to_string(indgen(ncol-2)+1)]+':'
          old  = strarr(ncol-1)
       endelse
       new  = dm_dialog_input(name,title='Set axis units',default=old,xsize=100,dialog_parent=tlb,cancel=cancel,info = ['\AA for angstrom, \HW for hbar omega ...','The units can be changed in the popup menu.'])
       if ~keyword_set(cancel) then unit = [new[0],'',new[1:*]]
       is_uniq = intarr(ncol)
       for i=2,ncol-1 do begin
           tmp = data[*,i]
           tmp = tmp[uniq(tmp,sort(tmp))]
           if n_elements(tmp) le 1000<(nrow/20^(ncol-3)) then is_uniq[i] = 1
       endfor
       if total(is_uniq) eq 0 then tmp = temporary(is_uniq) else tmp = 0
       if (n_elements(title) ne 0) and (n_elements(legend) eq 0) then legend = dm_getlegend(title)
       if ~keyword_set(init) then self->reset_data,label=label,unit=unit,data=data,weight=weight,is_uniq=is_uniq,title=title,legend=legend,/no_copy
    endelse
end

;load self-shielding factor from a file
pro dcs_mslice::load_ssfac,file,ssfac=ssfac
    if n_elements(file) eq 0 then begin
       path = self.dirs[1]
       if ptr_valid(self.ssfacPtr) then default = (*self.ssfacPtr).file
       file = dm_choose_file('*',/read,dialog_parent=tlb,title='Please choose the self-shielding factor file',path=path,file=default)
       self->setproperty,workDir=path
    endif
    openr,unit,file,/get_lun,error=err
    if err ne 0 then return
    ssfac = '' & str = ''
    while ~eof(unit) do begin
       readf,unit,str
       str = (strsplit(str,' ,'+string(9b),/extract))[-1]
       if strlen(str) ne 0 then ssfac = [temporary(ssfac),str]
    endwhile
    free_lun,unit
    if n_elements(ssfac) gt 1 then ssfac = double(ssfac[1:*]) else tmp = temporary(ssfac)
end

;mask Bragg peaks
;method:   1. find the highest peak, fit it with a Gaussian, width w0
;          2. replace the peak with background value
;          3. repeat 1 & 2, if w<0.25w0 or w>4w0, not a valid peak(default)
;          4. false peak if peak intensity is less than 4% of the strongest peak, and less than 3 sigma of the fit(default)
;          5. stop when two consecutive false peaks
;          6. if the strongest peak is at the left edge, use the second one as the strongest peak
;parameters:
;   detsum:  [ndet] array, detector intensity
;   errsum:  [ndet] array, error bar
;   mask:    saving returned sorted unique mask, if mask is not empty when called, result will be combined to the existing mask
;keywords:
;   fit_fwhm: passing the fitted fwhm of the strongest peak, if existed when calling, use it as the fwhm of the strongest peak
;   fit_max:  passing the fitted peak intensity of the strongest peak, if existed when calling, use it as the peak intensity
pro dcs_mslice::maskbraggpeak,detsum,errsum,mask,fit_fwhm=fit_fwhm,fit_max=fit_max
    ;smooth data and remove background
    sm1 = 25 & sm2 = 3 & smrep = 10
    ndet = n_elements(detsum) & x = lindgen(ndet) & y = detsum
    for i=0,smrep do  y = (detsum)<(smooth(y,sm1,/nan,/edge_truncate))
    y = detsum-y & y = smooth(y,sm2,/nan,/edge_truncate)
    ;search for peaks        
    wlim0 = self.mbparm[0] & wlim1 = self.mbparm[1] & plim = self.mbparm[2] & slim = self.mbparm[3]
    false_peak = 0
    if n_elements(fit_fwhm) eq 0 then fit_fwhm = !values.f_nan
    if n_elements(fit_max) eq 0 then  fit_max  = !values.f_nan
    nrep = 0L & replim = ndet/2 ;in case of a dead loop
    while (false_peak lt 2) and (nrep lt replim) do begin   ;search for peaks
          nrep = nrep+1
          ;find the maximum
          ymax = max(y,ind_ymax,min=ymin,/nan) 
          if finite(fit_fwhm) then $
             fwhm = fit_fwhm $
          else $
             fwhm = 10<(ndet/20)
          yave = mean(y)
          ;find peak boundaries    
          ip_0 = where((y[0:ind_ymax] le (ymax-1./2*(ymax-ymin))) and (y[0:ind_ymax] ge (ymin+1./3*(yave-ymin))),count0)
          ip_1 = where((y[ind_ymax:ndet-1] le (ymax-1./2*(ymax-ymin))) and (y[ind_ymax:ndet-1] ge (ymin+1./3*(yave-ymin))),count1)
          if count0 ne 0 then begin
             if count1 ne 0 then begin
                if ~finite(fit_fwhm) then fwhm = ip_1[0]+(ind_ymax-ip_0[count0-1])
                ip_1 = (ind_ymax+ip_1[0]*4)<(ndet-1)
             endif else begin
                if ~finite(fit_fwhm) then fwhm = (ind_ymax-ip_0[count0-1])
                ip_1 = (ind_ymax+(ind_ymax-ip_0[count0-1])*4)<(ndet-1)
             endelse
             ip_0 = 0>(ind_ymax-(ind_ymax-ip_0[count0-1])*4)
          endif else begin
             if count1 ne 0 then begin
                if ~finite(fit_fwhm) then fwhm = (ip_1[0])
                ip_0 = 0>(ind_ymax-ip_1[0]*4)   
                ip_1 = (ind_ymax+ip_1[0]*4)<(ndet-1) 
             endif else begin
                ip_0 = 0
                ip_1 = ndet-1
             endelse
          endelse
          if (ip_1-ip_0 ge ndet-1) or (ip_1-ip_0 lt 4) then break
          ;fit the peak with a Gaussian
          estimates = [ymax,x[ind_ymax],fwhm/sqrt(8.*alog(2)),ymin] 
          result = gaussfit(x[ip_0:ip_1],y[ip_0:ip_1],params,estimates=estimates,measure_errors=errsum[ip_0:ip_1],nterms=4,sigma=sigma) 
          if ~finite(sigma[0]) then sigma[0] = 0  
          fwhm = abs(params[2])*sqrt(8.*alog(2))
          if (~ finite(fit_max) and (params[1] gt fwhm))  then fit_max  = params[0]
          if (~ finite(fit_fwhm) and (params[1] gt fwhm)) then fit_fwhm = fwhm
          ip_0 = (0>(fix(params[1]-fwhm*1.2,type=3))<(ndet-1))
          ip_1 = (0>(fix(params[1]+fwhm*1.2,type=3))<(ndet-1))
          if (ip_0 gt ip_1) or (ip_1-ip_0 ge ndet-1) then break
          if (ip_1-ip_0 lt 1) then begin
             false_peak = false_peak+1
          endif else begin
             if ~finite(fit_fwhm) or ((fwhm gt wlim0*fit_fwhm) and (fwhm lt wlim1*fit_fwhm)) then begin
                if ~finite(fit_max) or ((params[0] gt plim*fit_max) and (params[0] gt slim*sigma[0])) then begin
                   false_peak = 0   ;clear the false peak flag
                   if n_elements(mask) eq 0 then $
                      mask = ip_0+lindgen(ip_1-ip_0+1) $
                   else $
                      mask = [mask,ip_0+lindgen(ip_1-ip_0+1)]
                endif else $
                   false_peak = false_peak+1
             endif else $
                false_peak = false_peak+1
          endelse
          ;replace the peak with the background
          y[ip_0:ip_1] = ymin
    endwhile
    ;sort the mask
    if n_elements(mask) ne 0 then mask = mask[uniq(mask,sort(mask))]
end        

;dummy calls to the following IDL library functions
;   amoeba:    used once for NCNR summer school
pro dcs_mslice::my_dummycalls
    ;amoeba
    parm = [-0.42d,-0.05,-0.32,-0.26,0.07,-0.43,-0.13,0.02,-0.26,-0.15,0.02, 0.0, 0.0, 0.0, 0.08] & scale = 0.5d
    parm = amoeba(1d-7,function_name='dm_mnwo4eigen',nmax=1e6,p0=parm,scale=scale,ncalls=ncalls,function_value=err)
end

;toggle a group of menu buttons according to uname and a number
pro dcs_mslice::my_menu_toggle,uname,flag,base=base
    nuname = n_elements(uname)
    if (nuname eq 0) or (n_elements(flag) eq 0) then return
    if n_elements(base) eq 0 then base=self.tlb
    ids = lonarr(nuname) 
    for i=0,nuname-1 do ids[i] = widget_info(base,find_by_uname=uname[i])
    ind = where(ids gt 0,n_valid)
    if n_valid eq 0 then return
    check = where(ind eq flag[0],n_check,complement=uncheck,ncomplement=n_uncheck)
    if n_check eq 0 then tmp = temporary(check) else check = ids[ind[check]]
    if n_uncheck eq 0 then tmp = temporary(uncheck) else uncheck = ids[ind[uncheck]]
    dm_toggle_menubut,check=check,uncheck=uncheck
end

;a widget_control use uname instead of id as identifier
pro dcs_mslice::my_widget_control,uname,base,_ref_extra=extra
    if n_elements(base) eq 0 then base=self.tlb
    if ~widget_info(base,/valid_id) then base=self.tlb
    if ~widget_info(base,/valid_id) then return
    if n_elements(extra) ne 0 then begin
       for i=0,n_elements(uname)-1 do begin
           wid = widget_info(base,find_by_uname=uname[i])
           if wid ne 0 then widget_control,wid,_extra=extra
       endfor
    endif
end

;widget_tex with tab_mode=1,/editable,/all_events,/kbrd_focus_events
function dcs_mslice::my_widget_text,base,ysize=ysize,_ref_extra=extra
    if n_elements(ysize) eq 0 then ysize=1
    return,widget_text(base,tab_mode=1,/editable,/all_events,/kbrd_focus_events,ysize=ysize,_extra=extra)
end

;check if both projection and qty data are available when nomessage keyword is set;
;check if general volume data or both projection and qty data are available when nomessage keyword is not set
;      if not available, give a warning
;returns 0 if not available, 1 if available
function dcs_mslice::pd_valid,nomessage=nomessage
    pd_valid = 0b
    if ptr_valid(self.projStrPtr) then begin
       if n_elements((*self.projStrPtr).qty) gt 1 then pd_valid=1b
    endif
    if keyword_set(nomessage) then return, pd_valid
    pd_valid = (ptr_valid(self.dataPtr)) or (pd_valid and (~self.error))
    if (~pd_valid) then ok = dialog_message((['No data.','Please load data and calculate projections first.'])[self.samp_typ ge 0],dialog_parent=self.tlb,/center)
    return, pd_valid
end


;plot and set mask
;parameter:
;   detsum: [ndet], sum of the intensity at each detector
;   blist:  [ndet], non 0 elements are the index numbers of the masked detectors
;keyword:
;   angle:  if set, plot vs angle instead of vs det number
;   det_tt: angle set, det_tt is the two theta angle
;   exist:  whether the mask pointer exist before
pro dcs_mslice::plot_mask,detsum,blist,angle=angle,det_tt=det_tt,exist=exist
    pblist = ptr_new(blist,/no_copy)
    ;help string
    helptext = [    $
	  "Use the mouse left button to zoom, the right button to mask/unmask. ", $
		"If ''XZOOM'' is displayed on a button in the top row, zooming only ", $
		"occurs in x. If ''XYZOOM'' is displayed zooming occurs in x and y. ", $
		"Click on the ''XZOOM/XYZOOM'' button to toggle between these states. ", $
		"If ''MASK'' is displayed on the other button in the top row, the right ", $
		"button is used to mask one or more detectors. If ''UNMASK'' is displayed ", $
		"the right button unmasks detectors. ", $
		"Click on the ''MASK/UNMASK'' button to toggle between these states. ", $
		"The sliders may also be used to define the displayed detector range. ", $
		"The most recent zoom may be ''undone'' using the ''Undo zoom'' button. ", $
		"To return to the previous user interface click the ''PROCEED'' button."]
    if keyword_set(angle) then $
       dcs_plotdetmask_2th,self.tlb,'',detsum,pblist,helptext,angle=det_tt $
    else $
       dcs_plotdetmask,self.tlb,'',detsum,pblist,helptext
    ind = where(*pblist gt 0,count)
    if ptr_valid(self.mask) then exist = 1b
    self->clear_mask,/keep_ar,mask_darkangle=self.mask_darkangle    ;clear the mask
    if count ne 0 then begin
       self.mask = ptr_new((*pblist)[ind]-1)
       self->my_widget_control,['mksvBut','mkclBut'],/sensitive
       if ptr_valid(self.projStrPtr) then self->set_display_button,/clearprojection ;need to recalculate projections
    endif else if keyword_set(exist) then $
       ok = dialog_message('Previously set detector number mask has been cleared.',dialog_parent=self.tlb,/center)
    ptr_free,pblist
end

;construct a range string
function dcs_mslice::range_string,in_str,name,range,unit
    out_str = strtrim(in_str,2)
    if n_params() lt 3 then return,out_str
    if strlen(out_str) eq 0 then sep = '' else sep = ', '
    rangfin = finite(range)
    if total(rangfin) ne 0 then begin
       if total(rangfin) eq 2 then begin
          out_str = out_str+sep+name+'=['+dm_to_string(min(range))+','+dm_to_string(max(range))+']'
       endif else begin
          if rangfin[0] then $
             out_str = out_str+sep+name+'\geq'+dm_to_string(range[0]) $
          else $
             out_str = out_str+sep+name+'\leq'+dm_to_string(range[1])   
       endelse
       if n_elements(unit) ne 0 then begin
          tmp = strtrim(unit[0],2)
          if strlen(tmp) gt 0 then out_str = out_str+' '+tmp 
       endif
    endif
    return,out_str 
end

;reset droplist values according to the labels
pro dcs_mslice::reset_droplist
    if self.samp_typ gt 0 then begin
       self.traj_type[0] = dm_set_viewlabel(self.view_u1lab[1],self.view_u1)
       self.traj_type[1] = dm_set_viewlabel(self.view_u2lab[1],self.view_u2)
       self.traj_type[2] = dm_set_viewlabel(self.view_u3lab,self.view_u3)
    endif
end

;reset intensity label, for macs instrument, change the monitor count accordingly in the intensity label and monitor choice labels and options
pro dcs_mslice::reset_intnlab
    if self.instrname eq 'macs' then begin
       tmp = stregex(self.intn_lab,'(.+)[ '+string(9b)+']+\(.+\)',/subexpr,/extract)
       if strlen(tmp[1]) gt 0 then intn_lab = tmp[1] else intn_lab = 'Intensity'
       ptmp = ['','']
       if finite(self.macs_mon[0,2]) then ptmp[0] = dm_to_string(self.macs_mon[0,2])+([' pulse',' pulses'])[self.macs_mon[0,2] gt 1]
       if finite(self.macs_mon[1,2]) then ptmp[1] = dm_to_string(self.macs_mon[1,2])+([' pulse',' pulses'])[self.macs_mon[1,2] gt 1]
       case self.mon_typ[1] of 
            0:  monlab = (finite(self.macs_mon[self.mon_sumfit[1],0])?dm_to_string(self.macs_mon[self.mon_sumfit[1],0])+' sec':'')            ;time
            1:  monlab = (finite(self.macs_mon[self.mon_sumfit[1],1])?'mon='+dm_to_string(self.macs_mon[self.mon_sumfit[1],1],/exponent):'')  ;monitor
            2:  monlab = ptmp[self.mon_sumfit[1]]                                                                                             ;pulse
            else:
       endcase
       if self.intn_typ eq n_elements(self.intn_typ_str)-1 then begin
          self.intn_lab = intn_lab+' (counts'+(['','/'])[strlen(monlab) gt 0]+monlab+')' 
       endif else begin
          if finite(self.macs_mon[self.mon_sumfit[1],self.mon_typ[1]]) then begin
             self.intn_lab = intn_lab+' ('+(['','mon=',''])[self.mon_typ[1] eq 0]+monlab+')'
          endif else begin
             self.intn_lab = intn_lab+' (arb. unit)'
          endelse
       endelse
       case self.mon_typ[1] of 
            0: monlab = [(finite(self.macs_mon[0,0])?' ('+dm_to_string(self.macs_mon[0,0])+' sec)':''),'('+dm_to_string(self.macs_mon[1,0])+' sec)...' ]
            1: monlab = [(finite(self.macs_mon[0,1])?' ('+dm_to_string(self.macs_mon[0,1],/exponent)+')':''),'('+dm_to_string(self.macs_mon[1,1],/exponent)+')...' ]
            2: monlab = [(finite(self.macs_mon[0,2])?' ('+ptmp[0]+')':''),'('+ptmp[1]+')...' ]
            else:
       endcase
       ebinbut = widget_info(self.fileBas,find_by_uname='eBin')
       widget_control,ebinbut,set_value=['Automatic','Specify ']+monlab
       dm_set_droplist,ebinbut,select=self.mon_sumfit[1]
       self->my_widget_control,'monSum',set_value='Automatic'+monlab[0]
       self->my_widget_control,'monFit',set_value='Specify '+monlab[1]
       if keyword_set(self.macshistomode) and self.macs_rawintn then self.intn_lab = intn_lab+' (counts)'
    endif
    self->my_widget_control,'intnLab',self.fileBas,set_value=self.intn_lab
end

;(re)create all the data file widgets
pro dcs_mslice::reset_file
    if (~ obj_valid(self.fileSel)) then begin ;create the widgets
       self.fileBas = widget_base(self.tlb,/row,/frame,ypad=1,xpad=0,space=0,map=0)
       widget_control,self.fileBas,update=0
       if !version.os_family eq 'Windows' then begin
          xsize = 280 & ysize = 9
       endif else begin
          xsize = 300 & ysize = 11 
       endelse
       self.fileSel = obj_new('dm_filesel',parent=self.fileBas,pobj=self,filter='.'+(*self.ftypeext)[self.ftype],xsize=xsize,ysize=ysize,ypad=0,group_leader=self.tlb,path=self.dirs[0],$
                      ncnrftp=self.ncnrftp,bufferdir=self.dirs[2],workdir=self.dirs[1],inxerev=self.inxerev)
       self.fileSel->getproperty,dates=dates,ncnrftp=ncnrftp
       self.macs_CFXdate = dates[0:1]
       self.ncnrftp = ncnrftp
       fileCol   = widget_base(self.fileBas,/col,/align_bottom,ypad=0,xpad=0,space=0,/context_events)
       sampleRow = widget_base(fileCol,/row,ypad=0,xpad=0,space=0)
       samplab   = widget_label(sampleRow,value='Sample Type',/align_center,uname='sampLab')
       geom      = widget_info(samplab,/geometry)
       if strlen(self.symfont) eq 0 then self.symfont=self->getsymbolfont(2*(fix(geom.scr_ysize)/2))
       sampTyp   = dm_widget_button(sampleRow,value='Powder',/menu,xsize=168,uname='sampTyp',_extra=dm_tooltip('Press this button to select a sample type.'))
       powdBut   = dm_widget_button(sampTyp,value='Powder',uname='powdTyp')
       crystTyp  = dm_widget_button(sampTyp,value='Single Crystal',uname='crystTyp',/menu)
       sct1But   = dm_widget_button(crystTyp,value='Single Crystal',uname='crystTyp1')
       sct2But   = dm_widget_button(crystTyp,value='Diffuse Scattering Single E',uname='crystTyp2')
       sct3But   = dm_widget_button(crystTyp,value='Diffuse Scattering Multiple E',uname='crystTyp3')
       ebinRow   = widget_base(fileCol,/row,ypad=0,xpad=0,space=0)
       ebinLab   = widget_label(ebinRow,value='E rebin size',/align_center,uname='ebinLab')
       eBin      = dm_widget_droplist(ebinRow,xsize=170,uname='eBin')
       labelRow  = widget_base(fileCol,/row,ypad=0,xpad=0,space=0)
       intn_lab  = widget_label(labelRow,value='Intensity Label',/align_center)
       intnLab   = self->my_widget_text(labelRow,value='',uname='intnLab')
       labelRow  = widget_base(fileCol,/row,ypad=0,xpad=0,space=0)
       titl_lab  = widget_label(labelRow,value='Title Label',/align_center)
       lockBut   = widget_button(labelRow,value=self.iconfiles[8+self.lock_title],uname='locktitlBut',/bitmap,/dynamic_resize,/flat,$
                   _extra=dm_tooltip('Click to '+(['lock','unlock'])[self.lock_title]+' the title label so that loading files will '+(['not',''])[self.lock_title]+' reset the title and legend labels.'))
       titlvoid  = widget_label(labelRow,value='',xsize=10)
       titlLab   = self->my_widget_text(labelRow,value='',uname='titlLab')
       labelRow  = widget_base(fileCol,/row,ypad=0,xpad=0,space=0)
       legd_lab  = widget_label(labelRow,value='Legend Label',/align_center)
       legdLab   = self->my_widget_text(labelRow,value='',uname='legdLab')
       loadRow   = widget_base(fileCol,/row,ypad=0,xpad=0,space=0,/align_right)
       loadBut   = widget_button(loadRow,value='Load Data',uname='loadBut',xsize=self.xsize[4],_extra=dm_tooltip('Select the files first on the left then press this button.'))
       addBut    = widget_button(loadRow,value='Add Data',uname='addBut',sensitive=0,xsize=self.xsize[4])
       geom      = widget_info(intn_lab,/geometry)
       xsize     = geom.xsize+40
       widget_control,samplab,xsize=xsize
       widget_control,intn_lab,xsize=xsize
       widget_control,titl_lab,xsize=xsize-(widget_info(lockBut,/geometry)).xsize-(widget_info(titlvoid,/geometry)).xsize
       widget_control,legd_lab,xsize=xsize
       widget_control,ebinLab,xsize=xsize
       geom      = widget_info(loadRow,/geometry)
       widget_control,sampTyp,scr_xsize=geom.xsize
       widget_control,eBin,   scr_xsize=geom.xsize
       widget_control,intnLab,scr_xsize=geom.xsize 
       widget_control,titlLab,scr_xsize=geom.xsize
       widget_control,legdLab,scr_xsize=geom.xsize
       geom      = widget_info(self.fileBas,/geometry)
       self.xsize[0] = geom.scr_xsize & self.xsize[1] = (self.xsize[0])/12-([3,2])[!version.os_family eq 'Windows'] 
       row       = widget_base(fileCol,/row,xpad=0,ypad=0,/align_right,space=0,/align_bottom)
       exclbas   = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,uname='ptaibase')
       if self.instrname eq 'macs' then begin
          ptaionly = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',uname='ptaionly',/align_center,_extra=dm_tooltip('View PTAI detector only (triple-axis mode)'))
       endif else ptaionly = 0
       void      = widget_label(row,value=([' ','PTAI Only'])[self.instrname eq 'macs'],/align_left,uname='ptaionlyLab',xsize=xsize/2+10)
       clearDat  = widget_button(row,value='Clear Data',uname='clearDat',sensitive=0,xsize=self.xsize[4]*2)
       widget_control,self.fileBas,/update,/map
       self.xsize[4] = 84-2*(!version.os_family eq 'Windows')
    endif else begin
       sampTyp   = widget_info(self.fileBas,find_by_uname='sampTyp')
       crystTyp  = widget_info(self.fileBas,find_by_uname='crystTyp')
       exclbas   = widget_info(self.fileBas,find_by_uname='ptaibase')
       ptaionly  = widget_info(self.fileBas,find_by_uname='ptaionly')
       self.fileSel->getproperty,ncnrftp=ncnrftp
       if self.ncnrftp ne ncnrftp then begin
          self.fileSel->set_ftpbuffer,(['',self.dirs[2]])[self.ncnrftp]
          self.fileSel->getproperty,ncnrftp=ncnrftp
          self.ncnrftp = ncnrftp
       endif
    endelse
    ;reset all values
    ;limited sample types fo MACS & WAND
    self->my_widget_control,['sampLab','sampTyp','ebinLab','eBin'],self.fileBas,sensitive=(self.instrname ne 'wand')
    if self.instrname eq 'macs' then begin
       if crystTyp ne 0 then begin
          widget_control,crystTyp,/destroy
          sct1But = dm_widget_button(sampTyp,value='Single Crystal',uname='crystTyp1')
       endif
       if ptaionly eq 0 then $
          ptaionly = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',uname='ptaionly',/align_center,_extra=dm_tooltip('View PTAI detector only (triple-axis mode)'))
       widget_control,ptaionly,set_button=self.macs_ptaionly
    endif else begin
       if crystTyp eq 0 then begin
          self->my_widget_control,'crystTyp1',sampTyp,/destroy
          crystTyp = dm_widget_button(sampTyp,value='Single Crystal',uname='crystTyp',/menu)
          sct1But = dm_widget_button(crystTyp,value='Single Crystal',uname='crystTyp1')
          sct2But = dm_widget_button(crystTyp,value='Diffuse Scattering Single E',uname='crystTyp2')
          sct3But = dm_widget_button(crystTyp,value='Diffuse Scattering Multiple E',uname='crystTyp3')
       endif
       if ptaionly ne 0 then widget_control,ptaionly,/destroy
    endelse
    tmp = self.samp_typ & self.samp_typ = -1
    self->change_samptyp,tmp,/init
    if self.samp_typ ge 2 then begin    ;diffuse scattering
       self->my_widget_control,'ebinLab',self.fileBas,set_value='Energy Range'
       self->my_widget_control,'eBin',self.fileBas,set_value=['All','Elastic Peak','Specify']
       tmp = self.e_bin[self.samp_typ]
       if tmp gt 2 then begin
          tmp = 1
          self.e_bin[self.samp_typ] = 1    ;default elastic peak
       end
    endif else begin
       self->my_widget_control,'ebinLab',self.fileBas,set_value=(['E Rebin Size','Monitor Count'])[self.instrname eq 'macs']
       if self.instrname ne 'macs' then self->my_widget_control,'eBin',self.fileBas,set_value=dm_to_string(*(self.rebinsize))
       tmp = where(*(self.rebinsize) eq self.e_bin[1-self.samp_typ],count)
       if count eq 0 then self.e_bin[1-self.samp_typ]=1
       tmp = where(*(self.rebinsize) eq self.e_bin[self.samp_typ],count)
       if count eq 0 then self.e_bin[self.samp_typ]=1
       tmp = (0>(tmp[0]))
    endelse
    if self.instrname ne 'macs' then begin
       dm_set_droplist,widget_info(self.tlb,find_by_uname='eBin'),select=tmp
       self->my_widget_control,'ptaionly',self.fileBas,set_button=self.macs_ptaionly
    endif
    self->reset_intnlab
    self->my_widget_control,'titlLab',self.fileBas,set_value=self.titl_lab
    self->my_widget_control,'legdLab',self.fileBas,set_value=self.legd_lab
    self->my_widget_control,'ptaionlyLab',self.fileBas,set_value=([' ','PTAI Only'])[self.instrname eq 'macs']
end

;(re)create and set values of all the widgets in self.projBas
;keyword:
;   reset:    if set, only set the existing widget values, otherwise regenerate all the widgets
;   foldonly: if set, only set the existing folding related values
pro dcs_mslice::reset_projection,reset=reset,foldonly=foldonly
    extravaxis = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
    if (~ keyword_set(reset)) then begin ;(re)generate all the widgets
       if ~widget_info(self.projBas,/valid_id) then begin
          self.projBas = widget_base(self.tlb,/frame,xpad=0,ypad=0,map=0,space=0,xsize=self.xsize[0])
       endif
       widget_control,self.projBas,update=0
       if ~self.view[0] then return
       extra = widget_base(self.projBas,/col,ypad=1,xpad=0)
       switch self.samp_typ of
         0:begin ;powder
           row    = widget_base(extra,/row,ypad=0,xpad=0,space=0)
           left   = widget_base(row,/col,xpad=0,ypad=0)
           label  = widget_label(left,value='  Viewing Axes:',/align_left,uname='viewAxis',/dynamic_resize)
           row    = widget_base(left,/row,ypad=0,xpad=0,space=0)
           label  = widget_label(row,value='',xsize=self.xsize[1]+self.xsize[2])
           label2 = widget_label(row,value='Label',/align_center)
           row    = widget_base(left,/row,ypad=0,xpad=0)
           label  = widget_label(row,value='u1',xsize=self.xsize[1],/align_center)
           viewU1typ = dm_widget_droplist(row,value=self.view_type,xsize=self.xsize[2],uname='viewU1typ')
           viewU1lab = self->my_widget_text(row,value='',xsize=self.xsize[3],uname='viewU1lab')
           geom   = widget_info(viewU1lab,/geometry)
           widget_control,label2,xsize=geom.scr_xsize
           row    = widget_base(left,/row,ypad=0,xpad=0)
           label  = widget_label(row,value='u2',xsize=self.xsize[1],/align_center)
           viewU2typ = dm_widget_droplist(row,value=self.view_type,xsize=self.xsize[2],uname='viewU2typ')
           viewU2lab = self->my_widget_text(row,value='',xsize=self.xsize[3],uname='viewU2lab')
           break
           end
         1:         ;single crystal
         2:         ;diffuse scattering single E
         3:begin    ;diffuse scattering multiple E
           toprow  = widget_base(extra,/row,ypad=0,xpad=0,space=0)
           left    = widget_base(toprow,/col,ypad=0,xpad=0,tab_mode=1)
           self.projrid[0] = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(self.projrid[0],value='  Unit Cell Lattice Parameters:')
           row     = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='a ('+string('c5'XB)+')',xsize=self.xsize[1],/align_center)
           lattA   = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattA')
           label   = widget_label(row,value='b ('+string('c5'XB)+')',xsize=self.xsize[1],/align_center)
           lattB   = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattB')
           label   = widget_label(row,value='c ('+string('c5'XB)+')',xsize=self.xsize[1],/align_center)
           lattC   = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattC')
           row     = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='a ('+string('B0'XB)+')',xsize=self.xsize[1],/align_center,font=self.symfont)
           lattAA  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattAA')
           label   = widget_label(row,value='b ('+string('B0'XB)+')',xsize=self.xsize[1],/align_center,font=self.symfont)
           lattBB  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattBB')
           label   = widget_label(row,value='g ('+string('B0'XB)+')',xsize=self.xsize[1],/align_center,font=self.symfont)
           lattCC  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='lattCC')
           geom    = widget_info(left,/geometry)
           xsize   = self.xsize[0]-geom.xsize
           right   = widget_base(toprow,/col,ypad=0,xpad=0,xsize=xsize,tab_mode=1)
           self.projrid[1] = widget_base(right,/row,ypad=0,xpad=0)
           label   = widget_label(self.projrid[1],value='  Crystal Orientation:')
           row     = widget_base(right,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='uh',xsize=self.xsize[1],/align_center)
           orieUh  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieUh')
           label   = widget_label(row,value='uk',xsize=self.xsize[1],/align_center)
           orieUk  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieUk')
           label   = widget_label(row,value='ul',xsize=self.xsize[1],/align_center)
           orieUl  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieUl')
           row     = widget_base(right,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='vh',xsize=self.xsize[1],/align_center)
           orieVh  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieVh')
           label   = widget_label(row,value='vk',xsize=self.xsize[1],/align_center)
           orieVk  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieVk')
           label   = widget_label(row,value='vl',xsize=self.xsize[1],/align_center)
           orieVl  = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='orieVl')
           row     = widget_base(right,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='Psi',/align_center,xsize=self.xsize[1],uname='psiLab')
           oriePsi = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='oriePsi')
           oriepsiLab = widget_label(row,value='   <- angle of ki^u',uname='oriepsiLab',/dynamic_resize)   ; positive if (ki x u)||Qz (up)
           row     = widget_base(extra,/row,ypad=0,xpad=0,space=0)
           left    = widget_base(row,/col,xpad=0,ypad=0,tab_mode=1)
           label   = widget_label(left,value='  Orthogonal Viewing Axes:',/align_left,uname='viewAxis',/dynamic_resize)
           row     = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='',scr_xsize=self.xsize[1],/align_center)
           label   = widget_label(row,value='a*',scr_xsize=self.xsize[1],/align_center)
           label   = widget_label(row,value='b*',scr_xsize=self.xsize[1],/align_center)
           label   = widget_label(row,value='c*',scR_xsize=self.xsize[1],/align_center)
           label   = widget_label(row,value='Label',scr_xsize=self.xsize[1]+1,/align_center)
           fold    = widget_label(row,value='Fold',scr_xsize=self.xsize[1]-1,/align_center)
           label   = widget_label(row,value='Center',scr_xsize=self.xsize[1],/align_center,uname='viewFoldcenter')
           label   = widget_label(row,value='Direction',scr_xsize=self.xsize[1]+5,/align_center,uname='viewFolddir')
           row     = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='u1',scr_xsize=self.xsize[1],/align_center)
           viewU1h = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU1h')
           viewU1k = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU1k')
           viewU1l = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU1l')
           viewU1lab = self->my_widget_text(row,value='',scr_xsize=self.xsize[1]+1,uname='viewU1lab')
           tmp       = widget_label(row,value='')
           exclbas   = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,/align_bottom)
           viewU1fd  = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',uname='viewU1Fold',/align_center)
           geom      = widget_info(viewU1fd,/geom)
           if !version.os_family eq 'Windows' then $
              widget_control,tmp,scr_xsize=0>(self.xsize[1]-geom.scr_xsize-4) $
           else $
              widget_control,fold,scr_xsize=(self.xsize[1]+geom.scr_xsize)/2
           viewU1fcn = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU1Foldcenter')
           viewU1fdr = dm_widget_droplist(row,value=['-','auto','+'],xsize=self.xsize[1]+([20,5])[!version.os_family eq 'Windows'],uname='viewU1Folddir',select=(*(self.view_u1folddir))[0]+1)
           row     = widget_base(left,/row,ypad=0,xpad=0)
           label   = widget_label(row,value='u2',scr_xsize=self.xsize[1],/align_center)
           viewU2h = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU2h')
           viewU2k = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU2k')
           viewU2l = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU2l')
           viewU2lab = self->my_widget_text(row,value='',scr_xsize=self.xsize[1]+1,uname='viewU2lab')
           tmp       = widget_label(row,value='')
           exclbas   = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,/align_bottom)
           viewU2fd  = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',uname='viewU2Fold',/align_center)
           if !version.os_family eq 'Windows' then widget_control,tmp,scr_xsize=0>(self.xsize[1]-geom.scr_xsize-4)
           viewU2fcn = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU2Foldcenter')
           viewU2fdr = dm_widget_droplist(row,value=['-','auto','+'],xsize=self.xsize[1]+([20,5])[!version.os_family eq 'Windows'],uname='viewU2Folddir',select=(*(self.view_u2folddir))[0]+1)
           if extravaxis then begin
              row     = widget_base(left,/row,ypad=0,xpad=0)
              label   = widget_label(row,value='u3',scr_xsize=self.xsize[1],/align_center)
              viewU3h = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU3h')
              viewU3k = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU3k')
              viewU3l = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU3l')
              viewU3lab = self->my_widget_text(row,value='',scr_xsize=self.xsize[1]+1,uname='viewU3lab')
              tmp       = widget_label(row,value='')
              exclbas   = widget_base(row,/row,xpad=0,ypad=0,space=0,/nonexclusive,/align_bottom)
              viewU3fd  = widget_button(exclbas,value=(!version.os_family eq 'Windows')?'':' ',uname='viewU3Fold',/align_center)
              if !version.os_family eq 'Windows' then widget_control,tmp,scr_xsize=0>(self.xsize[1]-geom.scr_xsize-4)
              viewU3fcn = self->my_widget_text(row,value='',scr_xsize=self.xsize[1],uname='viewU3Foldcenter')
              viewU3fdr = dm_widget_droplist(row,value=['-','auto','+'],xsize=self.xsize[1]+([20,5])[!version.os_family eq 'Windows'],uname='viewU3Folddir',select=(*(self.view_u3folddir))[0]+1)
           endif
           break
           end
         else:
       endswitch
       right = widget_base(row,/col,xpad=0,ypad=0,/align_right)
       geom  = widget_info(left,/geom)
       widget_control,right,scr_xsize=self.xsize[0]-geom.xsize-1
       row   = widget_base(right,/row,xpad=0,ypad=0,/align_right,space=0)
       calcPrj = widget_button(row,value='Calculate Projections',uname='calcPrj',xsize=((self.xsize[4]*2)<(self.xsize[0]-geom.xsize-1)),/align_center,$
                 _extra=dm_tooltip('Calculate the projection before any plots can be made.'))
    endif
    ;now set all the values
    if ~widget_info(self.projBas,/valid_id) then return
    if widget_info(self.projrid[0],/valid_id) then void = widget_info(self.projrid[0],find_by_uname='lock_latt') else void = 0
    if ((self.instrname ne 'macs') or (~self.locklattuv)) and (void ne 0) then self->my_widget_control,['lock_latt','lock_uv'],self.projBas,/destroy $
    else if (self.instrname eq 'macs') and self.locklattuv and (void eq 0) then begin
       extra = dm_tooltip('Use the popup menu to unlock the lattice parameters and crystal orientation.')
       void = widget_button(self.projrid[0],value=self.iconfiles[9],uname='lock_latt',/bitmap,/dynamic_resize,/flat,sensitive=0,_extra=extra)
       void = widget_button(self.projrid[1],value=self.iconfiles[9],uname='lock_uv',/bitmap,/dynamic_resize,/flat,sensitive=0,_extra=extra)
    endif
    ;folding first, single crystal only
    if self.samp_typ gt 0 then begin
       tmp = '. Specify the '+(['folding center and direction','folding center','n-fold symmetry'])[self.fold_type<2]+' to the right. Go to Options menu to change the folding type.'
       self->my_widget_control,'viewFoldcenter',self.projBas,set_value=(['Center','n-fold'])[self.fold_type gt 1],$
          sensitive=(self.view_u1fold or ((self.fold_type lt 2) and (self.view_u2fold or (extravaxis and self.view_u3fold))))
       self->my_widget_control,'viewFolddir',self.projBas,sensitive=((self.fold_type eq 0) and (self.view_u1fold or self.view_u2fold or (extravaxis and self.view_u3fold)))   
       self->my_widget_control,'viewU1Fold',self.projBas,set_button=self.view_u1fold,_extra=dm_tooltip('Click to '+(['enable','disable'])[self.view_u1fold]+tmp)
       self->my_widget_control,'viewU1Foldcenter',self.projBas,set_value=dm_to_string(*self.view_u1foldcenter,separator=','),sensitive=self.view_u1fold
       nfd = n_elements(*self.view_u1folddir)
       if nfd le 1 then begin
          select = (*self.view_u1folddir)[0]+1
          value  = ['-','auto','+']
       endif else begin
          select = 0
          value = (['-','auto','+'])[(*self.view_u1folddir)[0]+1]
          for i=1,nfd-1 do value = value+','+(['-','auto','+'])[(*self.view_u1folddir)[i]+1]
       endelse 
       dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU1Folddir'),select=select,set_value=value,sensitive=((self.fold_type eq 0) and self.view_u1fold) 
       self->my_widget_control,'viewU2Fold',self.projBas,set_button=self.view_u2fold,sensitive=(self.fold_type lt 2),_extra=dm_tooltip('Click to '+(['enable','disable'])[self.view_u2fold]+tmp)
       self->my_widget_control,'viewU2Foldcenter',self.projBas,set_value=dm_to_string(*(self.view_u2foldcenter),separator=','),sensitive=((self.fold_type lt 2) and self.view_u2fold)
       nfd = n_elements(*self.view_u2folddir)
       if nfd le 1 then begin
          select = (*self.view_u2folddir)[0]+1
          value  = ['-','auto','+']
       endif else begin
          select = 0
          value = (['-','auto','+'])[(*self.view_u2folddir)[0]+1]
          for i=1,nfd-1 do value = value+','+(['-','auto','+'])[(*self.view_u2folddir)[i]+1]
       endelse   
       dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU2Folddir'),select=select,set_value=value,sensitive=((self.fold_type eq 0) and self.view_u2fold)
       if extravaxis then begin
          self->my_widget_control,'viewU3Fold',self.projBas,set_button=self.view_u3fold,sensitive=(self.fold_type lt 2),_extra=dm_tooltip('Click to '+(['enable','disable'])[self.view_u3fold]+tmp)
          self->my_widget_control,'viewU3Foldcenter',self.projBas,set_value=dm_to_string(*(self.view_u3foldcenter),separator=','),sensitive=((self.fold_type lt 2) and self.view_u3fold)
          nfd = n_elements(*self.view_u3folddir)
          if nfd le 1 then begin
             select = (*self.view_u3folddir)[0]+1
             value  = ['-','auto','+']
          endif else begin
             select = 0
             value = (['-','auto','+'])[(*self.view_u3folddir)[0]+1]
             for i=1,nfd-1 do value = value+','+(['-','auto','+'])[(*self.view_u3folddir)[i]+1]
          endelse   
          dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU3Folddir'),select=select,set_value=value,sensitive=((self.fold_type eq 0) and self.view_u3fold)
       endif
    endif
    if keyword_set(foldonly) then return
    switch self.samp_typ of
      0:begin   ;powder
        value = self.view_type[0:3] & label = self.view_label[0:3]                                
        if self.intn_typ eq n_elements(self.intn_typ_str)-2 then begin  ;for J(Q,y) intensity type, allow y
           value = [value,self.view_type[5]]
           label = [label,self.view_label[5]]
        endif  
        if self.instrname eq 'dcs' then begin                           ;for dcs, allow time channel
           value = [value,self.view_type[4]]
           label = [label,self.view_label[4]]                           
        endif
        if self.view_u1typ ge n_elements(value) then begin
           self.view_u1typ = 1
           self.view_u1lab[0] = label[self.view_u1typ]
        endif
        if self.view_u2typ ge n_elements(value) then begin
           self.view_u2typ = 0
           self.view_u2lab[0] = label[self.view_u2typ]
        endif
        dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU1typ'),set_value=value,select=self.view_u1typ
        dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU2typ'),set_value=value,select=self.view_u2typ
        break
        end
      1:        ;single crystal
      2:        ;diffuse scattering single energy
      3:begin   ;diffuse scattering multiple energy   
        self->my_widget_control,'viewAxis',self.projBas,set_value=(self.view_ortharbi?'  Viewing Axes:':'  Orthogonal Viewing Axes:')
        self->my_widget_control,'lattA',self.projBas,set_value=dm_to_string(self.latt_parm[0],resolution=5)
        self->my_widget_control,'lattB',self.projBas,set_value=dm_to_string(self.latt_parm[1],resolution=5)
        self->my_widget_control,'lattC',self.projBas,set_value=dm_to_string(self.latt_parm[2],resolution=5)
        self->my_widget_control,'lattAA',self.projBas,set_value=dm_to_string(self.latt_parm[3],resolution=4)
        self->my_widget_control,'lattBB',self.projBas,set_value=dm_to_string(self.latt_parm[4],resolution=4)
        self->my_widget_control,'lattCC',self.projBas,set_value=dm_to_string(self.latt_parm[5],resolution=4)
        self->my_widget_control,'orieUh',self.projBas,set_value=dm_to_string(self.orie_u[0])
        self->my_widget_control,'orieUk',self.projBas,set_value=dm_to_string(self.orie_u[1])
        self->my_widget_control,'orieUl',self.projBas,set_value=dm_to_string(self.orie_u[2])
        self->my_widget_control,'orieVh',self.projBas,set_value=dm_to_string(self.orie_v[0])
        self->my_widget_control,'orieVk',self.projBas,set_value=dm_to_string(self.orie_v[1])
        self->my_widget_control,'orieVl',self.projBas,set_value=dm_to_string(self.orie_v[2])
        self->my_widget_control,'oriePsi',self.projBas,set_value=dm_to_string(self.orie_psi,resolution=3)
        if (total(abs(self.view_u1)) eq 0) or (total(abs(self.view_u2)) eq 0) then begin
           self.view_u1 = self.orie_u
           self.view_u2 = self.orie_v
           self->adjust_viewlabel,[0,1]
        endif
        if extravaxis  and (total(abs(self.view_u3)) eq 0) then begin
           self.view_u3 = [0,0,1]
           self.view_u3lab = 'L'
        endif
        self->reset_droplist
        self->my_widget_control,'viewU1h',self.projBas,set_value=dm_to_string(self.view_u1[0])
        self->my_widget_control,'viewU1k',self.projBas,set_value=dm_to_string(self.view_u1[1])
        self->my_widget_control,'viewU1l',self.projBas,set_value=dm_to_string(self.view_u1[2])
        self->my_widget_control,'viewU2h',self.projBas,set_value=dm_to_string(self.view_u2[0])
        self->my_widget_control,'viewU2k',self.projBas,set_value=dm_to_string(self.view_u2[1])
        self->my_widget_control,'viewU2l',self.projBas,set_value=dm_to_string(self.view_u2[2])
        if extravaxis then begin
           self->my_widget_control,'viewU3h',self.projBas,set_value=dm_to_string(self.view_u3[0])
           self->my_widget_control,'viewU3k',self.projBas,set_value=dm_to_string(self.view_u3[1])
           self->my_widget_control,'viewU3l',self.projBas,set_value=dm_to_string(self.view_u3[2])
           self->my_widget_control,'viewU3lab',self.projBas,set_value=self.view_u3lab
        endif
        self->my_widget_control,'psiLab',self.projBas,set_value=(['Psi','dA3'])[self.instrname eq 'macs'] 
        self->my_widget_control,'oriepsiLab',self.projBas,set_value=(self.instrname eq 'macs')?'   <- A3 offset angle (A3=A3+dA3)':(((self.samp_typ eq 1)?'   <- ':'   <- first ')+'angle of ki^u')      ; positive if (ki x u)||Qz (up)
        break
        end
      else:
    endswitch
    self->my_widget_control,'viewU1lab',self.projBas,set_value=self.view_u1lab[(self.samp_typ ge 1)]
    self->my_widget_control,'viewU2lab',self.projBas,set_value=self.view_u2lab[(self.samp_typ ge 1)]
end

;reset the data content for self.samp_typ=-1
pro dcs_mslice::reset_data,data=data,weight=weight,author=author,axis_info=axis_info,label=label,instrument=instrument,is_uniq=is_uniq,unit=unit,title=title,legend=legend,background=background,no_copy=no_copy
    if self.samp_typ ne -1 then return
    reset   = 1b
    n_item  = n_elements((*self.dataPtr).label)
    n_label = n_elements(label)
    n_unit  = n_elements(unit)
    n_uniq  = n_elements(is_uniq)
    n_instr = n_elements(instrument)
    n_titl  = n_elements(title)
    n_legd  = n_elements(legend)
    n_info  = n_elements(axis_info) 
    if keyword_set(background) then begin
       if n_elements(data) eq 0 then return
       n_col = n_elements(data[0,*])
       if n_item ne n_col then begin
          background = temporary(data)  & help,background,output=output1
          signal = (*self.dataPtr).data & help,signal,output=output2
          ok = dialog_message(['The background data are incompatible with the signal data.',output1,output2],/error,dialog_parent=self.tlb,/center)
          return
       endif
       ptr_free,self.bgdata,self.bgweightPtr
       self.ftypecheckmask[1] = (total(finite(data,/nan)) gt 0)
       if n_elements(weight) eq n_elements(data[*,0]) then self.bgweightPtr[0] = ptr_new(weight,no_copy=no_copy)
       self.bgdata[0] = ptr_new(data,no_copy=no_copy)
       ;check tolerance values
       new  = [(*self.data_tolerance),self.empt_tol_optn]
       name = (*self.dataPtr).label[2:*]+' tolerance'
       ind = where(strlen( (*self.dataPtr).unit[2:*]) ne 0,count)
       if count ne 0 then name[ind] = name[ind]+' ('+(*self.dataPtr).unit[2+ind]+')'
       name = [name,'background subtraction tolerance method'] & irep = 0
       repeat begin
          new = dm_dialog_input(name+':',/float,default=new,dialog_parent=self.tlb,title='Background subtraction tolerance value',cancel=cancel,/return_number,is_droplist=[intarr(n_col-2),1],xsize=180,$
                droplist_content=ptr_new(['Within Tolerance or Closest','Within Tolerance of the Closest']),info=[(['','All fileds must be specified.'])[irep],$
                'The tolerance values can be changed in the Parameter menu.','The tolerance method can be changed in the options menu.'])
          if keyword_set(cancel) then break
          irep = 1
       endrep until (total(finite(new,/nan)) eq 0)
       if ~keyword_set(cancel) then begin
          (*self.data_tolerance) = new[0:n_col-3]
          self.empt_tol_optn = new[n_col-2]
          self->my_menu_toggle,'empttol_'+['cwt','wtc'],self.empt_tol_optn  
       endif
    endif else begin
       if n_elements(author) eq 1 then (*self.info)[-1] = 'Yiming Qiu (yiming.qiu@nist.gov), '+author+'   '+self.mod_date
       
       if n_elements(data) ne 0 then begin      
          self.ftypecheckmask[0] = (total(finite(data,/nan)) gt 0)
          o_item = n_item
          n_item = n_elements(data[0,*])
          if n_item lt 4 then begin
             ok = dialog_message('The data must be [ndat,nitem] with nitem>3.',dialog_parent=self.tlb,/center)
             return
          endif
          instrument1   = (*self.dataPtr).instrument
          label1        = (*self.dataPtr).label
          unit1         = (*self.dataPtr).unit
          is_uniq1      = bytarr(n_item)
          title1        = (*self.dataPtr).title
          legend1       = (*self.dataPtr).legend
          axis_info1    = (*self.dataPtr).axis_info
          if n_item ne o_item then begin
             reset      = 0b   ;need to reconstruct display
             label1     = ['I','dI','x'+dm_to_string(indgen(n_item-2)+1)]
             unit1      = strarr(n_item)
             o_tol      = (*self.data_tolerance)
             axis_info1 = replicate({vaxis:[0.,0.,0.],label:'',offset:[0.,0.,0.]},n_item-2)
             (*self.data_tolerance) = fltarr(n_item-2)
             (*self.data_tolerance)[0:((o_item<n_item)-3)] = o_tol[0:((o_item<n_item)-3)] 
          endif
          ptr_free,self.volsel_t,self.volsel_tft,self.slice_t,self.slice_tft,self.cutsel_t,self.cutsel_tft
          if n_item gt 5 then begin
             self.volsel_t   = ptr_new(indgen(n_item-5)+3)
             self.volsel_tft = ptr_new(fltarr(n_item-5,2))
             (*self.volsel_tft)[*] = !values.f_nan 
          endif
          if n_item gt 4 then begin
             self.slice_t   = ptr_new(indgen(n_item-4)+2)
             self.slice_tft = ptr_new(fltarr(n_item-4,2))
             (*self.slice_tft)[*] = !values.f_nan 
             if self.volsel_x[0] gt n_item-3 then self.volsel_x[0] = 0
             if self.volsel_y[0] gt n_item-3 then self.volsel_y[0] = 1
             if self.volsel_z[0] gt n_item-3 then self.volsel_z[0] = 2
          endif
          if n_item gt 3 then begin
             self.cutsel_t    = ptr_new(indgen(n_item-3)+1)
             self.cutsel_tft  = ptr_new(fltarr(n_item-3,2))
             (*self.cutsel_tft)[*] = !values.f_nan 
             if self.slice_x[0] gt n_item-3 then self.slice_x[0] = 0
             if self.slice_y[0] gt n_item-3 then self.slice_y[0] = 1
             if self.cut_x[0]   gt n_item-3 then self.cut_x[0]   = 0
             if self.cut_y[0]   gt n_item-2 then self.cut_y[0]   = 0
          endif
          ptr_free,self.dataStrPtr,self.weightPtr
          if n_elements(weight) eq n_elements(data[*,0]) then self.weightPtr[0] = ptr_new(weight,no_copy=no_copy)
          (*self.dataPtr) = create_struct('data',keyword_set(no_copy)?temporary(data):data,'label',label1,'unit',unit1,'is_uniq',is_uniq1,'title',title1,'legend',legend1,'instrument',instrument1,'axis_info',axis_info1)
          if ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]) then begin
             if ptr_valid(self.bgdata[0]) then n_col = n_elements((*self.bgdata[0])[0,*]) else n_col = n_elements((*self.bgdata[1])[0,*])
             if n_item ne n_col then begin
                ok = dialog_message(['The background data are incompatible with the new signal data.','The background data have been removed.'],/error,dialog_parent=tlb,/center)
                ptr_free,self.bgdata
             endif
          endif
       endif
       
       if n_label ne 0 then (*self.dataPtr).label[0:(n_item<n_label)-1] = label[0:(n_item<n_label)-1]
       if n_unit  ne 0 then (*self.dataPtr).unit[0:(n_item<n_unit)-1] = unit[0:(n_item<n_unit)-1]
       if n_uniq  ne 0 then (*self.dataPtr).is_uniq[0:(n_item<n_uniq)-1] = is_uniq[0:(n_item<n_uniq)-1]
       if n_info  ne 0 then (*self.dataPtr).axis_info[0:((n_item-2)<n_info)-1] = axis_info[0:((n_item-2)<n_info)-1]
       if n_titl  ne 0 then (*self.dataPtr).title = title[0]
       if n_legd  ne 0 then (*self.dataPtr).legend = legend[0]
       if n_instr ne 0 then begin
          (*self.info)[0] = '--Mslice'+(strlen(instrument[0]) eq 0?'':' for ')+instrument[0]+' --'       
          widget_control,self.tlb,tlb_set_title=(*self.info)[0]
          (*self.dataPtr).instrument = instrument[0]
       endif
   
      if (~ keyword_set(reset)) then begin
         geom_old = widget_info(self.dispBas,/geometry)
         widget_control,self.tlb,update=0
         widget_control,self.dispBas,/destroy
      endif
      self->reset_display,reset=reset
      self->my_widget_control,'savevolumeMenu',sensitive=(n_item ge 5)

      if (~ keyword_set(reset)) then self->reset_geom,disp_old=geom_old
      widget_control,self.tlb,/show,/realize,iconify=0
    endelse
    self->data_subbg
    self->my_widget_control,'loadbgData',set_button=(ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]))
    self->my_widget_control,'bgclBut',sensitive=(ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1]))
end

;(re)create and set values of all the widgets in self.dispBas
;keyword:
;   reset:    if set, only set the existing widget values, otherwise regenerate all the widgets
;   calcstr:  calculator string
pro dcs_mslice::reset_display,reset=reset,calcstr=calcstr
    if self.samp_typ ge 0 then begin
       z2_yn  = (self.extravaxis_yn[0] and (self.instrname ne 'macs') and (self.samp_typ eq 1 or self.samp_typ eq 3))  ;flag for z2 axis in trajectory
       z1_yn  = ((self.extravaxis_yn[0] and (self.instrname ne 'wand')) or (self.samp_typ ne 2))                       ;flag for z1 axis in trajectory
       nitem  = [2,3,2,3]+self.extravaxis_yn[0]*[0,1,1,1]
       if self.instrname eq 'macs' then nitem[0:1] = [2,3]+total(self.extravaxis_yn)+(self.macshistomode ne 0)
       if self.instrname eq 'wand' then nitem[0]   = 2+self.extravaxis_yn[0]  ;not used yet
       if self.instrname eq 'dcs'  then nitem[*]   = nitem[*]+total(self.extravaxis_yn[1:*])
       vol_yn = (nitem[self.samp_typ] ge 3)   ;flag for volume tab, slice t1, cut t2
       st2_yn = (nitem[self.samp_typ] ge 4)   ;flag for 4d tab, volume t1, slice t2, cut t3
       st3_yn = (nitem[self.samp_typ] ge 5)   ;flag for 4d t1,  volume t2, slice t3, cut t4
       st4_yn = (nitem[self.samp_typ] ge 6)   ;flag for 4d t2,  volume t3, slice t4, cut t5
       self.disp_flag = [z1_yn,z2_yn,vol_yn,st2_yn,st3_yn,st4_yn]
    endif
    if (~ keyword_set(reset)) then begin   ;(re)generate all the widgets
       if ~widget_info(self.dispBas,/valid_id) then begin
          if self.samp_typ eq -1 then $ ;non-tab window
             self.dispBas = widget_base(self.tlb,/frame,xpad=0,ypad=0,map=0,space=0) $
          else $
             self.dispBas = widget_base(self.tlb,/frame,xpad=0,ypad=0,map=0,space=0,xsize=self.xsize[0])
       endif
       widget_control,self.dispBas,update=0
       if (~self.view[1]) and (self.samp_typ ne -1) then return
       xsize = fix((self.xsize[0]-3.*self.xsize[1]-self.xsize[2])/4.,type=3)
       self.xsize[4] = xsize
       oldxsize1 = self.xsize[1]
       if self.samp_typ eq -1 then begin ;non-tab window
          extra = widget_base(self.dispBas,/col,xpad=0,ypad=1,/context_events) 
          nitem = n_elements((*self.dataPtr).label)
          sel_list = (*self.dataPtr).label[2:nitem-1]
          row   = widget_base(extra,/row,xpad=5,ypad=0,space=0,/align_left)
          label= widget_label(row,value='sha  ding',/align_left) 
          geom  = widget_info(label,/geometry)
          self.xsize[1] = geom.xsize
          widget_control,label,set_value='Title Label:',xsize=self.xsize[1]*1.85
          wtitl = self->my_widget_text(row,value='',uname='titlLab',scr_xsize=self.xsize[1]*1.15+self.xsize[2]+xsize*2-geom.xoffset)
          row   = widget_base(extra,/row,xpad=5,ypad=0,space=0,/align_left)
          label = widget_label(row,value='Legend Label:',/align_left,xsize=self.xsize[1]*1.85)
          wlegd = self->my_widget_text(row,value='',uname='legdLab',scr_xsize=self.xsize[1]*1.15+self.xsize[2]+xsize*2-geom.xoffset)
          row   = widget_base(extra,/row,xpad=5,ypad=1,space=0)
          if nitem ge 5 then begin
             label      = widget_label(row,value='Volume:',/align_center) 
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='x',xsize=self.xsize[1],/align_center)
             volselx    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselX',select=self.volsel_x[0])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volxfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volxfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volxto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volxto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volxstep   = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='volxstep')   
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
             volsely    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselY',select=self.volsel_y[0])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volyfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volyfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volyto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volyto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volystep   = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='volystep')   
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='z',xsize=self.xsize[1],/align_center)
             volselz    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselZ',select=self.volsel_z[0])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volzfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volzfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volzto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volzto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volzstep   = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='volzstep')   
             for i=0,nitem-6 do begin
                 row    = widget_base(extra,/row,xpad=0,ypad=0,space=0)
                 label  = widget_label(row,value='',xsize=self.xsize[1],/align_center)
                 vtsel  = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volsel_T'+dm_to_string(i),select=(*self.volsel_t)[i])
                 label  = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
                 vtfrom = self->my_widget_text(row,value=dm_to_string((*self.volsel_tft)[i,0]),scr_xsize=xsize,uname='volselfrom_T'+dm_to_string(i))
                 label  = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
                 vtto   = self->my_widget_text(row,value=dm_to_string((*self.volsel_tft)[i,1]),scr_xsize=xsize,uname='volselto_T'+dm_to_string(i))
             endfor
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=self.xsize[1],/align_center)
             label      = widget_label(row,value='intensity range',xsize=self.xsize[1]+self.xsize[2],/align_center)
             volIfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volIfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volIto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volIto')
             label      = widget_label(row,value='color    table ',xsize=xsize*1.5,/align_center)
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=self.xsize[1],/align_center)
             label      = widget_label(row,value='smoothing level',xsize=self.xsize[1]+self.xsize[2],/align_center)
             volSmooth  = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volSmooth')
             label      = widget_label(row,value='interp.',xsize=self.xsize[1],/align_center)
             volInterp  = dm_widget_droplist(row,value=['Nearest Neighb','Trilinear'],xsize=xsize,uname='volInterp')
             volColor   = dm_widget_droplist(row,value=(*self.slice_cmap),xsize=xsize*2,uname='volColor')
             row        = widget_base(extra,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=3*self.xsize[1]+self.xsize[2]+xsize*2,/align_center)
             volPlot    = widget_button(row,value='Plot Volume',xsize=xsize,uname='volPlot')
             voloPlot   = widget_button(row,value='Oplot Volume',xsize=xsize,uname='voloPlot')
             row        = widget_base(extra,/row,xpad=5,ypad=1,space=0)
          endif
          label         = widget_label(row,value='Slice:',/align_center) 
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='x',xsize=self.xsize[1],/align_center)
          dispselH      = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='slicesel_x',select=self.slice_x[0])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          dispHfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispxfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispHto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispxto')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          dispHstep     = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='dispxstep')   
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
          dispselV      = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='slicesel_y',select=self.slice_y[0])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          dispVfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispyfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispVto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispyto')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          dispVstep     = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='dispystep')
          for i=0,nitem-5 do begin
              row       = widget_base(extra,/row,xpad=0,ypad=0,space=0)
              label     = widget_label(row,value='',xsize=self.xsize[1],/align_center)
              disptsel  = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='dispsel_T'+dm_to_string(i),select=(*self.slice_t)[i])
              label     = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
              disptfrom = self->my_widget_text(row,value=dm_to_string((*self.slice_tft)[i,0]),scr_xsize=xsize,uname='dispselfrom_T'+dm_to_string(i))
              label     = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
              disptto   = self->my_widget_text(row,value=dm_to_string((*self.slice_tft)[i,1]),scr_xsize=xsize,uname='dispselto_T'+dm_to_string(i))
          endfor
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=self.xsize[1],/align_center)
          label         = widget_label(row,value='intensity range',xsize=self.xsize[1]+self.xsize[2],/align_center)
          dispIfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispIfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispIto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispIto')
          label         = widget_label(row,value='color    table ',xsize=xsize*1.5,/align_center)
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=self.xsize[1],/align_center)
          label         = widget_label(row,value='smoothing level',xsize=self.xsize[1]+self.xsize[2],/align_center)
          dispSmooth    = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispSmooth')
          label         = widget_label(row,value='shading',xsize=self.xsize[1],/align_center)
          dispShading   = dm_widget_droplist(row,value=['Flat','Gouraud'],xsize=xsize,uname='dispShading')
          dispColor     = dm_widget_droplist(row,value=(*self.slice_cmap),xsize=xsize*2,uname='dispColor')
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=2*self.xsize[1]+self.xsize[2]+xsize,/align_center)
          dispFitSlice  = widget_button(row,value='Fit',xsize=self.xsize[1],uname='dispFitSlice')
          dispPlotSlice = widget_button(row,value='Plot Slice',xsize=xsize,uname='dispPlotSlice')
          dispoPltSlice = widget_button(row,value='Oplot Slice',xsize=xsize,uname='dispoPltSlice')
          dispSurfSlice = widget_button(row,value='Surf Slice',xsize=xsize,uname='dispSurfSlice')
          row           = widget_base(extra,/row,xpad=5,ypad=1,space=0)
          label         = widget_label(row,value='Cut:',/align_center)
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='along',xsize=self.xsize[1],/align_center)
          cutX          = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='cutX',select=self.cut_x[0])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          cutFrom       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutFrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          cutTo         = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutTo')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          cutStep       = self->my_widget_text(row,value='',scr_xsize=xsize-1,uname='cutStep')   
          for i=0,nitem-4 do begin
              row       = widget_base(extra,/row,xpad=0,ypad=0,space=0)
              label     = widget_label(row,value='',xsize=self.xsize[1],/align_center)
              cuttsel   = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='cutsel_T'+dm_to_string(i),select=(*self.cutsel_t)[i])
              label     = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
              cuttfrom  = self->my_widget_text(row,value=dm_to_string((*self.cutsel_tft)[i,0]),scr_xsize=xsize,uname='cutselfrom_T'+dm_to_string(i))
              label     = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
              cuttto    = self->my_widget_text(row,value=dm_to_string((*self.cutsel_tft)[i,1]),scr_xsize=xsize,uname='cutselto_T'+dm_to_string(i))
          endfor
          label         = widget_label(row,value='symbol',xsize=xsize,/align_center)
          cutColor      = dm_widget_droplist(row,value=(*self.colors),xsize=xsize,uname='cutColor')
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
          cutY          = dm_widget_droplist(row,value=['Intensity',sel_list],xsize=self.xsize[2],uname='cutY',select=self.cut_y[0]) ;volume data
          label         = widget_label(row,value='range',xsize=self.xsize[1],/align_center)
          cutYfrom      = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutYfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          cutYto        = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutYto')
          cutSymbol     = dm_widget_droplist(row,value=(*self.symbols),xsize=xsize,uname='cutSymbol')
          cutLine       = dm_widget_droplist(row,value=(*self.lines),xsize=xsize,uname='cutLine')
          row           = widget_base(extra,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=3*self.xsize[1]+self.xsize[2]+xsize,/align_center)
          cutFit        = widget_button(row,value='Fit',xsize=xsize,uname='cutFit')
          cutPlot       = widget_button(row,value='Plot Cut',xsize=xsize,uname='cutPlot')
          cutoPlot      = widget_button(row,value='Oplot Cut',xsize=xsize,uname='cutoPlot')
       endif else begin ;tab window
          case self.samp_typ of
            0: sel_list = self.view_type[0:3]    ;powder
            1: sel_list = self.traj_type[0:2]    ;single crystal
            2: sel_list = self.traj_type[0:1]    ;diffuse scattering single E
            3: sel_list = self.traj_type[0:2]    ;diffuse scattering multipe E
            else:
          endcase
          tabpad = 3
          ;adjust button size
          if self.samp_typ eq 0 then self->my_widget_control,'calcPrj',self.projBas,scr_xsize=self.xsize[4]*2
          extra         = widget_tab(self.dispBas,tab_mode=0,uname='dispTab')
          sliTab        = widget_base(extra,/col,title='Slice',uname='sliceTab',xpad=0,ypad=tabpad)
          cutTab        = widget_base(extra,/col,title='Cut',uname='cutTab',xpad=0,ypad=tabpad)
          tmp           = widget_info(sliTab,/geometry)
          self.xsize[1] = self.xsize[1]-tmp.space
          ;volume
          if vol_yn then begin
             volTab     = widget_base(extra,/col,title='Volume',uname='volTab',xpad=0,ypad=tabpad)
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='x',xsize=self.xsize[1],/align_center)
             volselx    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselX',select=self.volsel_x[self.samp_typ])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volxfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volxfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volxto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volxto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volxstep   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volxstep')   
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
             volsely    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselY',select=self.volsel_y[self.samp_typ])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volyfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volyfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volyto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volyto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volystep   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volystep')   
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='z',xsize=self.xsize[1],/align_center)
             volselz    = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='volselZ',select=self.volsel_z[self.samp_typ])
             label      = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
             volzfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volzfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volzto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volzto')
             label      = widget_label(row,value='step',xsize=xsize,/align_center)
             volzstep   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volzstep')
             for i=0,total([st2_yn,st3_yn,st4_yn])-1 do begin
                 row    = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
                 label  = widget_label(row,value='',xsize=self.xsize[1],/align_center)
                 vtsel  = dm_widget_droplist(row,xsize=self.xsize[2],uname='volsel_T'+dm_to_string(i))
                 label  = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
                 vtfrom = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volselfrom_T'+dm_to_string(i))
                 label  = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
                 vtto   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volselto_T'+dm_to_string(i))
             endfor
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=self.xsize[1],/align_center)
             label      = widget_label(row,value='intensity range',xsize=self.xsize[1]+self.xsize[2],/align_center)
             volIfrom   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volIfrom')
             label      = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
             volIto     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volIto')
             label      = widget_label(row,value='color    table ',xsize=xsize*1.5,/align_center)
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=self.xsize[1],/align_center)
             label      = widget_label(row,value='smoothing level',xsize=self.xsize[1]+self.xsize[2],/align_center)
             volSmooth  = self->my_widget_text(row,value='',scr_xsize=xsize,uname='volSmooth')
             label      = widget_label(row,value='interp.',xsize=self.xsize[1],/align_center)
             volInterp  = dm_widget_droplist(row,value=['Nearest Neighb','Trilinear'],xsize=xsize,uname='volInterp')
             volColor   = dm_widget_droplist(row,value=(*self.slice_cmap),xsize=xsize*2,uname='volColor')
             row        = widget_base(volTab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=3*self.xsize[1]+self.xsize[2]+xsize*2,/align_center)
             volPlot    = widget_button(row,value='Plot Volume',xsize=xsize,uname='volPlot')  
             voloPlot   = widget_button(row,value='Oplot Volume',xsize=xsize,uname='voloPlot',_extra=dm_tooltip('Overplot the volume in the current plot window.'))
          endif
          ;4d
          if st2_yn then begin
             d4Tab      = widget_base(extra,/col,title='4D',uname='4dTab',xpad=0,ypad=tabpad)
             for i=1,4 do begin
                 row    = widget_base(d4Tab,/row,xpad=0,ypad=0,space=0)
                 label  = widget_label(row,value='x'+dm_to_string(i),xsize=self.xsize[1],/align_center)
                 d4selx = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='d4selX'+dm_to_string(i))
                 label  = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
                 d4from = self->my_widget_text(row,value='',scr_xsize=xsize,uname='d4x'+dm_to_string(i)+'from')
                 label  = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
                 d4to   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='d4x'+dm_to_string(i)+'to')
                 label  = widget_label(row,value='step',xsize=xsize,/align_center)
                 d4step = self->my_widget_text(row,value='',scr_xsize=xsize,uname='d4x'+dm_to_string(i)+'step')
             endfor
             for i=0,total([st3_yn,st4_yn])-1 do begin
                 row    = widget_base(d4Tab,/row,xpad=0,ypad=0,space=0)
                 label  = widget_label(row,value='',xsize=self.xsize[1],/align_center)
                 dtsel  = dm_widget_droplist(row,xsize=self.xsize[2],uname='d4sel_T'+dm_to_string(i))
                 label  = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
                 dtfrom = self->my_widget_text(row,value='',scr_xsize=xsize,uname='d4selfrom_T'+dm_to_string(i))
                 label  = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
                 dtto   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='d4selto_T'+dm_to_string(i))
             endfor
             row        = widget_base(d4Tab,/row,xpad=0,ypad=0,space=0)
             label      = widget_label(row,value='',xsize=3*self.xsize[1]+self.xsize[2]+xsize*2,/align_center)
             d4save     = widget_button(row,value='Save 4D Data',xsize=xsize*2,uname='d4Save')
          endif
          ;slice
          row           = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='x',xsize=self.xsize[1],/align_center)
          dispselH      = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='slicesel_x',select=self.slice_x[self.samp_typ])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          dispHfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispxfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispHto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispxto')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          dispHstep     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispxstep')   
          row           = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
          dispselV      = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='slicesel_y',select=self.slice_y[self.samp_typ])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          dispVfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispyfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispVto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispyto')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          dispVstep     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispystep')
          select        = [self.slice_t1[self.samp_typ],self.slice_t2[self.samp_typ],self.slice_t3[self.samp_typ],self.slice_t4[self.samp_typ]]
          for i=0,total([vol_yn,st2_yn,st3_yn,st4_yn])-1 do begin
              row       = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
              label     = widget_label(row,value='',xsize=self.xsize[1],/align_center)
              disptsel  = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='dispsel_T'+dm_to_string(i),select=select[i])
              label     = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
              disptfrom = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispselfrom_T'+dm_to_string(i))
              label     = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
              disptto   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispselto_T'+dm_to_string(i))
          endfor
          row           = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=self.xsize[1],/align_center)
          label         = widget_label(row,value='intensity range',xsize=self.xsize[1]+self.xsize[2],/align_center)
          dispIfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispIfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          dispIto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispIto')
          label         = widget_label(row,value='color    table ',xsize=xsize*1.5,/align_center)
          row           = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=self.xsize[1],/align_center)
          label         = widget_label(row,value='smoothing level',xsize=self.xsize[1]+self.xsize[2],/align_center)
          dispSmooth    = self->my_widget_text(row,value='',scr_xsize=xsize,uname='dispSmooth')
          label         = widget_label(row,value='shading',xsize=self.xsize[1],/align_center)
          dispShading   = dm_widget_droplist(row,value=['Flat','Gouraud'],xsize=xsize,uname='dispShading')
          dispColor     = dm_widget_droplist(row,value=(*self.slice_cmap),xsize=xsize*2,uname='dispColor')
          row           = widget_base(sliTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=2*self.xsize[1]+self.xsize[2]+xsize,/align_center)
          dispFitSlice  = widget_button(row,value='Fit',xsize=self.xsize[1],uname='dispFitSlice',_extra=dm_tooltip('Fit the grouped slice data in PAN (Peak Analysis).'))
          dispPlotSlice = widget_button(row,value='Plot Slice',xsize=xsize,uname='dispPlotSlice',_extra=dm_tooltip('Contour plot the slice.'))
          dispoPltSlice = widget_button(row,value='Oplot Slice',xsize=xsize,uname='dispoPltSlice',_extra=dm_tooltip('Overplot the slice in the current plot window.'))
          dispSurfSlice = widget_button(row,value='Surf Slice',xsize=xsize,uname='dispSurfSlice',_extra=dm_tooltip('Surface plot the slice.'))
          ;cut
          row           = widget_base(cutTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='along',xsize=self.xsize[1],/align_center)
          cutX          = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='cutX',select=self.cut_x[self.samp_typ])
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          cutFrom       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutFrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          cutTo         = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutTo')
          label         = widget_label(row,value='step',xsize=xsize,/align_center)
          cutStep       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutStep')
          select        = [self.cut_t1[self.samp_typ],self.cut_t2[self.samp_typ],self.cut_t3[self.samp_typ],self.cut_t4[self.samp_typ],self.cut_t5[self.samp_typ]]
          for i=0,total([vol_yn,st2_yn,st3_yn,st4_yn]) do begin
              row       = widget_base(cutTab,/row,xpad=0,ypad=0,space=0)
              label     = widget_label(row,value='',xsize=self.xsize[1],/align_center)
              cuttsel   = dm_widget_droplist(row,value=sel_list,xsize=self.xsize[2],uname='cutsel_T'+dm_to_string(i),select=select[i])
              label     = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
              cuttfrom  = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutselfrom_T'+dm_to_string(i))
              label     = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
              cuttto    = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutselto_T'+dm_to_string(i))
          endfor
          label         = widget_label(row,value='symbol',xsize=xsize,/align_center)
          cutColor      = dm_widget_droplist(row,value=(*self.colors),xsize=xsize,uname='cutColor')
          row           = widget_base(cutTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
          cutY          = dm_widget_droplist(row,value=['Intensity',sel_list],xsize=self.xsize[2],uname='cutY',select=self.cut_y[self.samp_typ])
          label         = widget_label(row,value='range',xsize=self.xsize[1],/align_center)
          cutYfrom      = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutYfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          cutYto        = self->my_widget_text(row,value='',scr_xsize=xsize,uname='cutYto')
          cutSymbol     = dm_widget_droplist(row,value=(*self.symbols),xsize=xsize,uname='cutSymbol')
          cutLine       = dm_widget_droplist(row,value=(*self.lines),xsize=xsize,uname='cutLine')
          row           = widget_base(cutTab,/row,xpad=0,ypad=0,space=0)
          label         = widget_label(row,value='',xsize=3*self.xsize[1]+self.xsize[2]+xsize,/align_center)
          cutFit        = widget_button(row,value='Fit',xsize=xsize,uname='cutFit',_extra=dm_tooltip('Fit the cut data in PAN (Peak Analysis).'))
          cutPlot       = widget_button(row,value='Plot Cut',xsize=xsize,uname='cutPlot')
          cutoPlot      = widget_button(row,value='Oplot Cut',xsize=xsize,uname='cutoPlot',_extra=dm_tooltip('Overplot the cut in the current plot window.'))
          ;trajectory
          trajTab       = widget_base(extra,/row,title='Detector Trajectory',uname='trajTab',xpad=0,ypad=tabpad)
          left          = widget_base(trajTab,/col,xpad=0,ypad=0)
          row           = widget_base(left,/row,ypad=0,xpad=0,space=0)
          label         = widget_label(row,value='x',xsize=self.xsize[1],/align_center)
          trajX         = dm_widget_droplist(row,value=self.traj_type,xsize=self.xsize[2],uname='trajX')
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          trajXfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajXfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          trajXto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajXto')
          row           = widget_base(left,/row,ypad=0,xpad=0,space=0)
          label         = widget_label(row,value='y',xsize=self.xsize[1],/align_center)
          trajY         = dm_widget_droplist(row,value=self.traj_type,xsize=self.xsize[2],uname='trajY')
          label         = widget_label(row,value='from:',xsize=self.xsize[1],/align_center)
          trajYfrom     = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajYfrom')
          label         = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
          trajYto       = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajYto')
          for i=1,z1_yn+z2_yn do begin
              row       = widget_base(left,/row,ypad=0,xpad=0,space=0)
              label     = widget_label(row,value='',xsize=self.xsize[1],/align_center)
              trajZ     = dm_widget_droplist(row,value=self.traj_type,xsize=self.xsize[2],uname='trajZ'+dm_to_string(i))
              label     = widget_label(row,value='thick',xsize=self.xsize[1],/align_center)
              trajZfrom = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajZ'+dm_to_string(i)+'from')
              label     = widget_label(row,value='to:',xsize=self.xsize[1],/align_center)
              trajZto   = self->my_widget_text(row,value='',scr_xsize=xsize,uname='trajZ'+dm_to_string(i)+'to')
          endfor
          geom          = widget_info(left,/geom)
          right         = widget_base(trajTab,/col,/align_bottom,xpad=0,xsize=self.xsize[0]-geom.xsize-8,ypad=0)
          plotTra       = widget_button(right,value='Plot Trajectories',uname='plotTra',sensitive=0)
          oplotTra      = widget_button(right,value='Oplot Trajectories',uname='oplotTra',sensitive=0,_extra=dm_tooltip('Overplot the trajectories in the current plot window.'))
          ;calculator
          if ~lmgr(/vm) then begin
             if n_elements(calcstr) ne 0 then sensitive = (max(strlen(strtrim(calcstr,2))) gt 0) else sensitive=0
             calcTab    = widget_base(extra,/row,title='Calculator',uname='calcTab',xpad=0,ypad=tabpad)
             tmp1       = widget_info(sliTab,/geometry)
             tmp2       = widget_info(plotTra,/geometry)
             ysize      = tmp1.scr_ysize-tmp2.scr_ysize & line = 3
             if !version.os_family eq 'Windows' then begin
                font    = widget_info(plotTra,/fontname)
                tmp     = stregex(font,'\*-?([0-9]+)',/subexpr,len=len)
                if len[1] ne 0 then font = strmid(font,0,tmp[1])+dm_to_string(dm_to_number(strmid(font,tmp[1],len[1]))+2)+strmid(font,tmp[1]+len[1])
             endif
             repeat begin
                line    = line+1
                void    = self->my_widget_text(calcTab,uname='calcInput',ysize=line,scr_xsize=self.xsize[2]+3*self.xsize[1]+2*xsize,font=font) 
                tmp     = widget_info(void,/geometry) 
                widget_control,void,/destroy
             endrep until tmp.scr_ysize gt ysize
             void       = self->my_widget_text(calcTab,uname='calcInput',value=calcstr,ysize=line-1,scr_xsize=self.xsize[2]+3*self.xsize[1]+2*xsize,/scroll,font=font) 
             tmp        = widget_info(void,/geometry) & ysize = tmp.scr_ysize
             right      = widget_base(calcTab,/col,/align_bottom,xpad=0,xsize=self.xsize[0]-geom.xsize-8,ypad=0)
             cmenu      = widget_button(right,value='Useful Routines',/menu,_extra=dm_tooltip('Click to see available routines. Use IDL programming language.'))
             void       = widget_button(cmenu,value='E to lambda',uname='calc_com_e2l')
             void       = widget_button(cmenu,value='lambda to E',uname='calc_com_l2e')
             void       = widget_button(cmenu,value='E to velocity',uname='calc_com_e2v')
             void       = widget_button(cmenu,value='Q length from hkl',uname='calc_com_qlength1')
             void       = widget_button(cmenu,value='Q length from Ei, Ef, and 2theta',uname='calc_com_qlength2')
             void       = widget_button(cmenu,value='2theta from hkl',uname='calc_com_2theta1')
             void       = widget_button(cmenu,value='2theta from Q and E',uname='calc_com_2theta2')
             void       = widget_button(cmenu,value=(['psi and 2theta','A3 and A4'])[self.instrname eq 'macs']+' from hkl and E',uname='calc_com_a3a4')
             void       = widget_button(cmenu,value='hkl from '+(['psi and 2theta','A3 and A4'])[self.instrname eq 'macs'],uname='calc_com_hkl')
             void       = widget_button(cmenu,value='angle between two hkl vectors',uname='calc_com_hklangle')
             void       = widget_button(cmenu,value='angle between two vectors from two sets of '+(['psis and 2thetas','A3s and A4s'])[self.instrname eq 'macs'],uname='calc_com_vecangle')
             void       = widget_button(cmenu,value='lattice constant from 2theta',uname='calc_com_latta')            
             void       = widget_label(right,value=' ')
             clcCal     = widget_button(right,value='Clear',uname='clearCal',sensitive=sensitive)
             calCal     = widget_button(right,value='Calculate',uname='calcCal',sensitive=sensitive)
          endif
       endelse
       self.xsize[1] = oldxsize1
    endif
    if ~widget_info(self.dispBas,/valid_id) then return
    ;set all the values
    if (~ ptr_valid(self.dataPtr)) then begin ;tab window
       extravaxis = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))   ;excluding T-axis and other MACS extra axes
       self->set_display_button    ;include some menu buttons and all display buttons
       case self.samp_typ of
         0:begin  ;powder
           sel_list = self.view_type[0:3]                                                                     ;Not including DCS time channel yet
           if self.intn_typ eq n_elements(self.intn_typ_str)-2 then sel_list=[sel_list,self.view_type[5]]     ;for J(Q,y) intensity type, allow y
           end
         1:begin  ;single crystal
           if extravaxis then sel_list = self.traj_type[[0,1,2,3]] else sel_list = self.traj_type[[0,1,3]]
           end
         2:begin  ;diffuse scattering single E
           if extravaxis then sel_list = self.traj_type[[0,1,2]] else sel_list = self.traj_type[[0,1]]
           end
         3:begin  ;diffuse scattering multiple E
           if extravaxis then sel_list = self.traj_type[[0,1,2,3]] else sel_list = self.traj_type[[0,1,3]]
           end
         else:
       endcase
       ;"intelligent" cut step
       if ptr_valid(self.dataStrPtr) then begin
          if self.instrname eq 'macs' then begin
             en = (*self.dataStrPtr).Ei-(*self.dataStrPtr).Ef
             en = en[uniq(en,sort(en))] 
          endif else begin
             en = (*self.dataStrPtr).e
          endelse
          emax = max(en,min=emin)
       endif
       notE = [([self.cut_x[0] ne 0,(self.cut_x[0] ne 0) and (self.cut_x[0] ne 4)])[self.instrname eq 'dcs'],self.cut_x[1] ne 2+extravaxis,1b,self.cut_x[3] ne 2+extravaxis] 
       if notE[self.samp_typ] then begin
          if (~finite(self.cut_step)) then self.cut_step=0.01
       endif else begin
          if finite(self.cut_step) and ptr_valid(self.dataStrPtr) and ((self.instrname eq 'dcs') or (self.instrname eq 'macs')) then begin
             nen = ((finite(self.cut_ran[1])?(self.cut_ran[1]):(emax))-(finite(self.cut_ran[0])?(self.cut_ran[0]):(emin)))/(1e-7>(self.cut_step))
             tmp = where((en le (finite(self.cut_ran[1])?(self.cut_ran[1]):(emax))) and (en ge (finite(self.cut_ran[0])?(self.cut_ran[0]):(emin))), nen1)
             if nen gt nen1 then self.cut_step = !values.f_nan
          endif
       endelse
       ;"intelligent" slice step
       notE = [([self.slice_x[0] ne 0,(self.slice_x[0] ne 0) and (self.slice_x[0] ne 4)])[self.instrname eq 'dcs'],self.slice_x[1] ne 2+extravaxis,1b,self.slice_x[3] ne 2+extravaxis] 
       if notE[self.samp_typ] then begin
          if ~finite(self.slice_xstep) then self.slice_xstep = 0.035
       endif else if ptr_valid(self.dataStrPtr) and ((self.instrname eq 'dcs') or (self.instrname eq 'macs')) and finite(self.slice_xstep) then begin
          nen = ((finite(self.slice_xran[1])?(self.slice_xran[1]):(emax))-(finite(self.slice_xran[0])?(self.slice_xran[0]):(emin)))/(1e-7>(self.slice_xstep))
          tmp = where((en le (finite(self.slice_xran[1])?(self.slice_xran[1]):(emax))) and (en ge (finite(self.slice_xran[0])?(self.slice_xran[0]):(emin))), nen1)
          if nen gt nen1 then self.slice_xstep = !values.f_nan
       endif
       notE = [([self.slice_y[0] ne 0,(self.slice_y[0] ne 0) and (self.slice_y[0] ne 4)])[self.instrname eq 'dcs'],self.slice_y[1] ne 2+extravaxis,1b,self.slice_y[3] ne 2+extravaxis] 
       if notE[self.samp_typ] then begin
          if ~finite(self.slice_ystep) then self.slice_ystep = 0.035
       endif else if ptr_valid(self.dataStrPtr) and ((self.instrname eq 'dcs') or (self.instrname eq 'macs')) and finite(self.slice_ystep) then begin
          nen = ((finite(self.slice_yran[1])?(self.slice_yran[1]):(emax))-(finite(self.slice_yran[0])?(self.slice_yran[0]):(emin)))/(1e-7>(self.slice_ystep))
          tmp = where((en le (finite(self.slice_yran[1])?(self.slice_yran[1]):(emax))) and (en ge (finite(self.slice_yran[0])?(self.slice_yran[0]):(emin))), nen1)
          if nen gt nen1 then self.slice_ystep = !values.f_nan
       endif
       ;"intelligent" volume step
       if vol_yn then begin
          if self.extravaxis_yn[0] and (self.instrname ne 'macs') then index = [0b,0b,0b,1b,0b,0b,0b] else index = [0b,0b,1b,0b,0b,0b,0b]
          if self.samp_typ eq 0 then index = [1b,0b,0b,0b,0b,0b,0b]
          if (~ finite(self.vol_xstep)) then begin
             isE = [index[self.volsel_x[0]],index[self.volsel_x[1]],0b,index[self.volsel_x[3]]]
             self.vol_xstep = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.vol_xstep = ((finite(self.vol_xran[1])?(self.vol_xran[1]):(emax))-(finite(self.vol_xran[0])?(self.vol_xran[0]):(emin)))/150.
                self.vol_xstep = 0.005>(round(self.vol_xstep*100.)/100.)
             endif
          endif
          if (~ finite(self.vol_ystep)) then begin
             isE = [index[self.volsel_y[0]],index[self.volsel_y[1]],0b,index[self.volsel_y[3]]]
             self.vol_ystep = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.vol_ystep = ((finite(self.vol_yran[1])?(self.vol_yran[1]):(emax))-(finite(self.vol_yran[0])?(self.vol_yran[0]):(emin)))/150.
                self.vol_ystep = 0.005>(round(self.vol_ystep*100.)/100.)
             endif
          endif
          if (~ finite(self.vol_zstep)) then begin
             isE = [index[self.volsel_z[0]],index[self.volsel_z[1]],0b,index[self.volsel_z[3]]]
             self.vol_zstep = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.vol_zstep = ((finite(self.vol_zran[1])?(self.vol_zran[1]):(emax))-(finite(self.vol_zran[0])?(self.vol_zran[0]):(emin)))/150.
                self.vol_zstep = 0.005>(round(self.vol_zstep*100.)/100.)
             endif
          endif
       endif
       ;"intelligent" 4d step
       if st2_yn then begin
          if self.extravaxis_yn[0] and (self.instrname ne 'macs') then index = [0b,0b,0b,1b,0b,0b,0b] else index = [0b,0b,1b,0b,0b,0b,0b]
          if self.samp_typ eq 0 then index = [1b,0b,0b,0b,0b,0b,0b] 
          if (~ finite(self.d4_x1step)) then begin
             isE = [index[self.d4sel_x1[0]],index[self.d4sel_x1[1]],0b,index[self.d4sel_x1[3]]]
             self.d4_x1step = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.d4_x1step = ((finite(self.d4_x1ran[1])?(self.d4_x1ran[1]):(emax))-(finite(self.d4_x1ran[0])?(self.d4_x1ran[0]):(emin)))/150.
                self.d4_x1step = 0.005>(round(self.d4_x1step*100.)/100.)
             endif
          endif
          if (~ finite(self.d4_x2step)) then begin
             isE = [index[self.d4sel_x2[0]],index[self.d4sel_x2[1]],0b,index[self.d4sel_x2[3]]]
             self.d4_x2step = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.d4_x2step = ((finite(self.d4_x2ran[1])?(self.d4_x2ran[1]):(emax))-(finite(self.d4_x2ran[0])?(self.d4_x2ran[0]):(emin)))/150.
                self.d4_x2step = 0.005>(round(self.d4_x2step*100.)/100.)
             endif
          endif
          if (~ finite(self.d4_x3step)) then begin
             isE = [index[self.d4sel_x3[0]],index[self.d4sel_x3[1]],0b,index[self.d4sel_x3[3]]]
             self.d4_x3step = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.d4_x3step = ((finite(self.d4_x3ran[1])?(self.d4_x3ran[1]):(emax))-(finite(self.d4_x3ran[0])?(self.d4_x3ran[0]):(emin)))/150.
                self.d4_x3step = 0.005>(round(self.d4_x3step*100.)/100.)
             endif
          endif
          if (~ finite(self.d4_x4step)) then begin
             isE = [index[self.d4sel_x4[0]],index[self.d4sel_x4[1]],0b,index[self.d4sel_x4[3]]]
             self.d4_x4step = 0.05
             if isE[self.samp_typ] and ptr_valid(self.dataStrPtr) then begin
                self.d4_x4step = ((finite(self.d4_x4ran[1])?(self.d4_x4ran[1]):(emax))-(finite(self.d4_x4ran[0])?(self.d4_x4ran[0]):(emin)))/150.
                self.d4_x4step = 0.005>(round(self.d4_x4step*100.)/100.)
             endif
          endif  
       endif
       ;trajectory, doesn't include T and time channel(DCS powder) in the list, add 'None' at the end of the list for thickness    
       n_list = n_elements(sel_list)
       ind_list = indgen(n_list+1)
       if self.traj_x[self.samp_typ] ge n_list  then self.traj_x[self.samp_typ] = ([1,0])[self.samp_typ gt 0]
       if self.traj_y[self.samp_typ] ge n_list  then self.traj_y[self.samp_typ] = ([0,1])[self.samp_typ gt 0]
       if self.traj_x[self.samp_typ] eq self.traj_y[self.samp_typ] then begin
          self.traj_x[self.samp_typ] = ([1,0])[self.samp_typ gt 0]
          self.traj_y[self.samp_typ] = ([0,1])[self.samp_typ gt 0]
       endif
       if z1_yn and self.traj_z1[self.samp_typ] gt n_list then self.traj_z1[self.samp_typ] = n_list
       if z2_yn and self.traj_z2[self.samp_typ] gt n_list then self.traj_z2[self.samp_typ] = n_list
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajX'),set_value=[sel_list],select=self.traj_x[self.samp_typ]
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajY'),set_value=[sel_list],select=self.traj_y[self.samp_typ]
       self->my_widget_control,'trajXfrom', self.dispBas,set_value=dm_to_string(self.traj_xran[0])
       self->my_widget_control,'trajXto',   self.dispBas,set_value=dm_to_string(self.traj_xran[1])
       self->my_widget_control,'trajYfrom', self.dispBas,set_value=dm_to_string(self.traj_yran[0])
       self->my_widget_control,'trajYto',   self.dispBas,set_value=dm_to_string(self.traj_yran[1])
       if z1_yn then begin
          list = where((ind_list ne self.traj_x[self.samp_typ]) and (ind_list ne self.traj_y[self.samp_typ]),count)
          if count ne 0 then self.traj_z1[self.samp_typ] = list[0>((where(list eq self.traj_z1[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajZ1'),set_value=[sel_list,'None'],select=self.traj_z1[self.samp_typ]
          self->my_widget_control,'trajZ1from',self.dispBas,set_value=dm_to_string(self.traj_z1ran[0])
          self->my_widget_control,'trajZ1to',  self.dispBas,set_value=dm_to_string(self.traj_z1ran[1])
       endif
       if z2_yn then begin
          list = where((ind_list ne self.traj_x[self.samp_typ]) and (ind_list ne self.traj_y[self.samp_typ]) and (ind_list ne self.traj_z1[self.samp_typ]),count)
          if count ne 0 then self.traj_z2[self.samp_typ] = list[0>((where(list eq self.traj_z2[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajZ2'),set_value=[sel_list,'None'],select=self.traj_z2[self.samp_typ]
          self->my_widget_control,'trajZ2from',self.dispBas,set_value=dm_to_string(self.traj_z2ran[0])
          self->my_widget_control,'trajZ2to',  self.dispBas,set_value=dm_to_string(self.traj_z2ran[1])
       endif 
       if (self.samp_typ eq 0) and (self.instrname eq 'dcs') then sel_list = [sel_list,self.view_type[4]] ;add time channel back to the selection list for DCS powder
       if (self.instrname eq 'wand') and self.extravaxis_yn[0] then sel_list = [sel_list,'T (K)'] 
       if self.instrname eq 'dcs' then begin
          if self.extravaxis_yn[1] then sel_list = [sel_list,'T (K)']
          if self.extravaxis_yn[2] then sel_list = [sel_list,'H (T)']
       endif
       if self.instrname eq 'macs' then begin
          if self.macshistomode ne 0 then sel_list = [sel_list,'Time ('+(['ms','s'])[self.macshistounit]+')']
          if self.extravaxis_yn[1] then sel_list = [sel_list,'T (K)'] 
          if self.extravaxis_yn[2] then sel_list = [sel_list,'H (T)']
          if self.extravaxis_yn[0] then sel_list = [sel_list,self.extravaxis_spec[0]+((strlen(self.extravaxis_spec[1]) eq 0)?'':(' ('+self.extravaxis_spec[1]+')'))] 
       endif
       n_list   = n_elements(sel_list)
       ind_list = indgen(n_list)
       ;slice
       if self.slice_x[self.samp_typ] ge n_list then self.slice_x[self.samp_typ] = ([1,0])[self.samp_typ ne 0]
       if self.slice_y[self.samp_typ] ge n_list then self.slice_y[self.samp_typ] = ([0,1])[self.samp_typ ne 0]
       if self.slice_x[self.samp_typ] eq self.slice_y[self.samp_typ] then begin
          self.slice_x[self.samp_typ] = ([1,0])[self.samp_typ ne 0]
          self.slice_y[self.samp_typ] = ([0,1])[self.samp_typ ne 0]
       endif
       ;cut 
       if self.cut_x[self.samp_typ] ge n_list then self.cut_x[self.samp_typ] = 0
       if (self.cut_T1[self.samp_typ] eq self.cut_x[self.samp_typ]) or (self.cut_T1[self.samp_typ] ge n_list) then self.cut_T1[self.samp_typ] = (where(ind_list ne self.cut_x[self.samp_typ]))[0]
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T0'),set_value=sel_list,select=self.cut_T1[self.samp_typ]
       self->my_widget_control,'cutselfrom_T0',self.dispBas,set_value=dm_to_string(self.cut_t1ran[0])
       self->my_widget_control,'cutselto_T0',  self.dispBas,set_value=dm_to_string(self.cut_t1ran[1])
       if vol_yn then begin
          ;slice t1
          list = where((ind_list ne self.slice_x[self.samp_typ]) and (ind_list ne self.slice_y[self.samp_typ]),count)
          if count ne 0 then self.slice_t1[self.samp_typ] = list[0>((where(list eq self.slice_t1[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T0'),set_value=sel_list,select=self.slice_t1[self.samp_typ]
          self->my_widget_control,'dispselfrom_T0',self.dispBas,set_value=dm_to_string(self.slice_t1ran[0])
          self->my_widget_control,'dispselto_T0',  self.dispBas,set_value=dm_to_string(self.slice_t1ran[1])
          ;volume
          vol_existed = 1b
          if self.volsel_x[self.samp_typ] ge n_list then self.volsel_x[self.samp_typ] = 0
          if (self.volsel_y[self.samp_typ] ge n_list) or (self.volsel_y[self.samp_typ] eq self.volsel_x[self.samp_typ]) then $
             self.volsel_y[self.samp_typ] = (where(ind_list ne self.volsel_x[self.samp_typ]))[0]
          if (self.volsel_z[self.samp_typ] ge n_list) or (self.volsel_z[self.samp_typ] eq self.volsel_x[self.samp_typ]) or (self.volsel_z[self.samp_typ] eq self.volsel_y[self.samp_typ]) then $
             self.volsel_z[self.samp_typ] = (where((ind_list ne self.volsel_x[self.samp_typ]) and (ind_list ne self.volsel_y[self.samp_typ])))[0]
          ;cut t2
          list = where((ind_list ne self.cut_x[self.samp_typ]) and (ind_list ne self.cut_T1[self.samp_typ]),count)
          if count ne 0 then self.cut_T2[self.samp_typ] = list[0>((where(list eq self.cut_T2[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T1'),set_value=sel_list,select=self.cut_T2[self.samp_typ]
          self->my_widget_control,'cutselfrom_T1',self.dispBas,set_value=dm_to_string(self.cut_t2ran[0])
          self->my_widget_control,'cutselto_T1',  self.dispBas,set_value=dm_to_string(self.cut_t2ran[1])
       endif
       if st2_yn then begin
          ;4d
          if self.d4sel_x1[self.samp_typ] ge n_list then self.d4sel_x1[self.samp_typ] = 0
          if (self.d4sel_x2[self.samp_typ] ge n_list) or (self.d4sel_x2[self.samp_typ] eq self.d4sel_x1[self.samp_typ]) then $
             self.d4sel_x2[self.samp_typ] = (where(ind_list ne self.d4sel_x1[self.samp_typ]))[0]
          if (self.d4sel_x3[self.samp_typ] ge n_list) or (self.d4sel_x3[self.samp_typ] eq self.d4sel_x1[self.samp_typ]) or (self.d4sel_x3[self.samp_typ] eq self.d4sel_x2[self.samp_typ]) then $
             self.d4sel_x3[self.samp_typ] = (where((ind_list ne self.d4sel_x1[self.samp_typ]) and (ind_list ne self.d4sel_x2[self.samp_typ])))[0]
          if (self.d4sel_x4[self.samp_typ] ge n_list) or (self.d4sel_x4[self.samp_typ] eq self.d4sel_x1[self.samp_typ]) or (self.d4sel_x4[self.samp_typ] eq self.d4sel_x2[self.samp_typ]) or (self.d4sel_x4[self.samp_typ] eq self.d4sel_x3[self.samp_typ]) then $
             self.d4sel_x4[self.samp_typ] = (where((ind_list ne self.d4sel_x1[self.samp_typ]) and (ind_list ne self.d4sel_x2[self.samp_typ]) and (ind_list ne self.d4sel_x3[self.samp_typ])))[0]  
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4selX1'),set_value=sel_list,select=self.d4sel_x1[self.samp_typ]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4selX2'),set_value=sel_list,select=self.d4sel_x2[self.samp_typ]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4selX3'),set_value=sel_list,select=self.d4sel_x3[self.samp_typ]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4selX4'),set_value=sel_list,select=self.d4sel_x4[self.samp_typ]
          self->my_widget_control,'d4x1from',self.dispBas,set_value=dm_to_string(self.d4_x1ran[0])
          self->my_widget_control,'d4x1to',  self.dispBas,set_value=dm_to_string(self.d4_x1ran[1])
          self->my_widget_control,'d4x1step',self.dispBas,set_value=dm_to_string(self.d4_x1step)
          self->my_widget_control,'d4x2from',self.dispBas,set_value=dm_to_string(self.d4_x2ran[0])
          self->my_widget_control,'d4x2to',  self.dispBas,set_value=dm_to_string(self.d4_x2ran[1])
          self->my_widget_control,'d4x2step',self.dispBas,set_value=dm_to_string(self.d4_x2step)
          self->my_widget_control,'d4x3from',self.dispBas,set_value=dm_to_string(self.d4_x3ran[0])
          self->my_widget_control,'d4x3to',  self.dispBas,set_value=dm_to_string(self.d4_x3ran[1])
          self->my_widget_control,'d4x3step',self.dispBas,set_value=dm_to_string(self.d4_x3step)
          self->my_widget_control,'d4x4from',self.dispBas,set_value=dm_to_string(self.d4_x4ran[0])
          self->my_widget_control,'d4x4to',  self.dispBas,set_value=dm_to_string(self.d4_x4ran[1])
          self->my_widget_control,'d4x4step',self.dispBas,set_value=dm_to_string(self.d4_x4step)
          ;slice t2
          list = where((ind_list ne self.slice_x[self.samp_typ]) and (ind_list ne self.slice_y[self.samp_typ]) and (ind_list ne self.slice_t1[self.samp_typ]),count)
          if count ne 0 then self.slice_t2[self.samp_typ] = list[0>((where(list eq self.slice_t2[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T1'),set_value=sel_list,select=self.slice_t2[self.samp_typ]
          self->my_widget_control,'dispselfrom_T1',self.dispBas,set_value=dm_to_string(self.slice_t2ran[0])
          self->my_widget_control,'dispselto_T1',  self.dispBas,set_value=dm_to_string(self.slice_t2ran[1])
          ;volume t1
          list = where((ind_list ne self.volsel_x[self.samp_typ]) and (ind_list ne self.volsel_y[self.samp_typ]) and (ind_list ne self.volsel_z[self.samp_typ]),count)
          if count ne 0 then self.volsel_t1[self.samp_typ] = list[0>((where(list eq self.volsel_t1[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T0'),set_value=sel_list,select=self.volsel_t1[self.samp_typ]
          self->my_widget_control,'volselfrom_T0',self.dispBas,set_value=dm_to_string(self.volsel_tft1[0])
          self->my_widget_control,'volselto_T0',  self.dispBas,set_value=dm_to_string(self.volsel_tft1[1])
          ;cut t3
          list = where((ind_list ne self.cut_x[self.samp_typ]) and (ind_list ne self.cut_T1[self.samp_typ]) and (ind_list ne self.cut_T2[self.samp_typ]),count)
          if count ne 0 then self.cut_T3[self.samp_typ] = list[0>((where(list eq self.cut_T3[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T2'),set_value=sel_list,select=self.cut_T3[self.samp_typ]
          self->my_widget_control,'cutselfrom_T2',self.dispBas,set_value=dm_to_string(self.cut_t3ran[0])
          self->my_widget_control,'cutselto_T2',  self.dispBas,set_value=dm_to_string(self.cut_t3ran[1])
       endif
       if st3_yn then begin
          ;4d t1
          list = where((ind_list ne self.d4sel_x1[self.samp_typ]) and (ind_list ne self.d4sel_x2[self.samp_typ]) and (ind_list ne self.d4sel_x3[self.samp_typ]) and $
                       (ind_list ne self.d4sel_x4[self.samp_typ]),count)
          if count ne 0 then self.d4sel_t1[self.samp_typ] = list[0>((where(list eq self.d4sel_t1[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4sel_T0'),set_value=sel_list,select=self.d4sel_t1[self.samp_typ]
          self->my_widget_control,'d4selfrom_T0',self.dispBas,set_value=dm_to_string(self.d4sel_tft1[0])
          self->my_widget_control,'d4selto_T0',  self.dispBas,set_value=dm_to_string(self.d4sel_tft1[1])  
          ;slice t3
          list = where((ind_list ne self.slice_x[self.samp_typ]) and (ind_list ne self.slice_y[self.samp_typ]) and (ind_list ne self.slice_t1[self.samp_typ]) and $
                       (ind_list ne self.slice_t2[self.samp_typ]),count)
          if count ne 0 then self.slice_t3[self.samp_typ] = list[0>((where(list eq self.slice_t3[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T2'),set_value=sel_list,select=self.slice_t3[self.samp_typ]
          self->my_widget_control,'dispselfrom_T2',self.dispBas,set_value=dm_to_string(self.slice_t3ran[0])
          self->my_widget_control,'dispselto_T2',  self.dispBas,set_value=dm_to_string(self.slice_t3ran[1])
          ;volume t2
          list = where((ind_list ne self.volsel_x[self.samp_typ]) and (ind_list ne self.volsel_y[self.samp_typ]) and (ind_list ne self.volsel_z[self.samp_typ]) and $
                       (ind_list ne self.volsel_t1[self.samp_typ]),count)
          if count ne 0 then self.volsel_t2[self.samp_typ] = list[0>((where(list eq self.volsel_t2[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T1'),set_value=sel_list,select=self.volsel_t2[self.samp_typ]
          self->my_widget_control,'volselfrom_T1',self.dispBas,set_value=dm_to_string(self.volsel_tft2[0])
          self->my_widget_control,'volselto_T1',  self.dispBas,set_value=dm_to_string(self.volsel_tft2[1])
          ;cut t4
          list = where((ind_list ne self.cut_x[self.samp_typ]) and (ind_list ne self.cut_T1[self.samp_typ]) and (ind_list ne self.cut_T2[self.samp_typ]) and $
                       (ind_list ne self.cut_T3[self.samp_typ]),count)
          if count ne 0 then self.cut_T4[self.samp_typ] = list[0>((where(list eq self.cut_T4[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T3'),set_value=sel_list,select=self.cut_T4[self.samp_typ]
          self->my_widget_control,'cutselfrom_T3',self.dispBas,set_value=dm_to_string(self.cut_t4ran[0])
          self->my_widget_control,'cutselto_T3',  self.dispBas,set_value=dm_to_string(self.cut_t4ran[1])
       endif
       if st4_yn then begin
          ;4d t2
          list = where((ind_list ne self.d4sel_x1[self.samp_typ]) and (ind_list ne self.d4sel_x2[self.samp_typ]) and (ind_list ne self.d4sel_x3[self.samp_typ]) and $
                       (ind_list ne self.d4sel_x4[self.samp_typ]) and (ind_list ne self.d4sel_t1[self.samp_typ]),count)
          if count ne 0 then self.d4sel_t2[self.samp_typ] = list[0>((where(list eq self.d4sel_t2[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4sel_T1'),set_value=sel_list,select=self.d4sel_t2[self.samp_typ]
          self->my_widget_control,'d4selfrom_T1',self.dispBas,set_value=dm_to_string(self.d4sel_tft2[0])
          self->my_widget_control,'d4selto_T1',  self.dispBas,set_value=dm_to_string(self.d4sel_tft2[1])
          ;slice t4
          list = where((ind_list ne self.slice_x[self.samp_typ]) and (ind_list ne self.slice_y[self.samp_typ]) and (ind_list ne self.slice_t1[self.samp_typ]) and $
                       (ind_list ne self.slice_t2[self.samp_typ]) and (ind_list ne self.slice_t3[self.samp_typ]),count)
          if count ne 0 then self.slice_t4[self.samp_typ] = list[0>((where(list eq self.slice_t4[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T3'),set_value=sel_list,select=self.slice_t4[self.samp_typ]
          self->my_widget_control,'dispselfrom_T3',self.dispBas,set_value=dm_to_string(self.slice_t4ran[0])
          self->my_widget_control,'dispselto_T3',  self.dispBas,set_value=dm_to_string(self.slice_t4ran[1])
          ;volume t3
          list = where((ind_list ne self.volsel_x[self.samp_typ]) and (ind_list ne self.volsel_y[self.samp_typ]) and (ind_list ne self.volsel_z[self.samp_typ]) and $
                       (ind_list ne self.volsel_t1[self.samp_typ]) and (ind_list ne self.volsel_t2[self.samp_typ]),count)
          if count ne 0 then self.volsel_t3[self.samp_typ] = list[0>((where(list eq self.volsel_t3[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T2'),set_value=sel_list,select=self.volsel_t3[self.samp_typ]
          self->my_widget_control,'volselfrom_T2',self.dispBas,set_value=dm_to_string(self.volsel_tft3[0])
          self->my_widget_control,'volselto_T2',  self.dispBas,set_value=dm_to_string(self.volsel_tft3[1])
          ;cut t5 
          list = where((ind_list ne self.cut_x[self.samp_typ]) and (ind_list ne self.cut_T1[self.samp_typ]) and (ind_list ne self.cut_T2[self.samp_typ]) and $
                       (ind_list ne self.cut_T3[self.samp_typ]) and (ind_list ne self.cut_T4[self.samp_typ]),count)
          if count ne 0 then self.cut_T5[self.samp_typ] = list[0>((where(list eq self.cut_T5[self.samp_typ]))[0])]
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T4'),set_value=sel_list,select=self.cut_T5[self.samp_typ]
          self->my_widget_control,'cutselfrom_T4',self.dispBas,set_value=dm_to_string(self.cut_t5ran[0])
          self->my_widget_control,'cutselto_T4',  self.dispBas,set_value=dm_to_string(self.cut_t5ran[1])
       endif
       self->my_widget_control,'calc_com_vecangle',self.dispBas,set_value='angle between two vectors from two sets of '+(['psis and 2thetas','A3s and A4s'])[self.instrname eq 'macs']
       self->my_widget_control,'calc_com_a3a4',self.dispBas,set_value=(['psi and 2theta','A3 and A4'])[self.instrname eq 'macs']+' from hkl and E'
    endif else begin
       nitem    = n_elements((*self.dataPtr).label)
       sel_list = (*self.dataPtr).label[2:nitem-1]
       n_list   = n_elements(sel_list)
       ind_list = indgen(n_list)
       ;title and legend labels
       self.titl_lab = (*self.dataPtr).title
       self.legd_lab = (*self.dataPtr).legend
       self->my_widget_control,'titlLab',set_value=self.titl_lab
       self->my_widget_control,'legdLab',set_value=self.legd_lab
       ;volume 
       if nitem ge 5 then begin 
          vol_existed = 1b
          if self.volsel_x[0] ge n_list then self.volsel_x[0] = 0
          if (self.volsel_y[0] ge n_list) or (self.volsel_y[0] eq self.volsel_x[0]) then $
             self.volsel_y[0] = (where(ind_list ne self.volsel_x[0]))[0]
          if (self.volsel_z[0] ge n_list) or (self.volsel_z[0] eq self.volsel_x[0]) or (self.volsel_z[0] eq self.volsel_y[0]) then $
             self.volsel_z[0] = (where((ind_list ne self.volsel_x[0]) and (ind_list ne self.volsel_y[0])))[0]
          for i=0,nitem-6 do begin  
              tmp = bytarr(n_list)+1b
              tmp[0>([self.volsel_x[0],self.volsel_y[0],self.volsel_z[0]])<(n_list-1)] = 0b
              if i gt 0 then tmp[0>((*self.volsel_t)[0:i-1])<(n_list-1)] = 0b
              list = where(tmp,count)
              if count gt 0 then begin
                 list = ind_list[list]
                 (*self.volsel_t)[i] = list[0>((where(list eq (*self.volsel_t)[i]))[0])]
              endif 
              dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T'+dm_to_string(i)),set_value=sel_list,select=(*self.volsel_t)[i]
              self->my_widget_control,'volselfrom_T'+dm_to_string(i),self.dispBas,set_value=dm_to_string((*self.volsel_tft)[i,0])
              self->my_widget_control,'volselto_T'+dm_to_string(i),self.dispBas,set_value=dm_to_string((*self.volsel_tft)[i,1])
          endfor
       endif
       ;slice
       if self.slice_x[0] ge n_list then self.slice_x[0] = 0
       if self.slice_y[0] ge n_list then self.slice_y[0] = 1
       if self.slice_x[0] eq self.slice_y[0] then begin
          self.slice_x[0] = 0
          self.slice_y[0] = 1
       endif
       for i=0,nitem-5 do begin   
           tmp = bytarr(n_list)+1b
           tmp[0>([self.slice_x[0],self.slice_y[0]])<(n_list-1)] = 0b
           if i gt 0 then tmp[0>((*self.slice_t)[0:i-1])<(n_list-1)] = 0b
           list = where(tmp,count)
           if count gt 0 then begin
              list = ind_list[list]
              (*self.slice_t)[i] = list[0>((where(list eq (*self.slice_t)[i]))[0])]
           endif 
           dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T'+dm_to_string(i)),set_value=sel_list,select=(*self.slice_t)[i]
           self->my_widget_control,'dispselfrom_T'+dm_to_string(i),self.dispBas,set_value=dm_to_string((*self.slice_tft)[i,0])
           self->my_widget_control,'dispselto_T'+dm_to_string(i),  self.dispBas,set_value=dm_to_string((*self.slice_tft)[i,1])
       endfor
       ;cut
       if self.cut_x[0] ge n_list then self.cut_x[0] = 0
       for i=0,nitem-4 do begin  
           tmp = bytarr(n_list)+1b
           tmp[0>(self.cut_x[0])<(n_list-1)] = 0b
           if i gt 0 then tmp[0>((*self.cutsel_t)[0:i-1])<(n_list-1)] = 0b
           list = where(tmp,count)
           if count gt 0 then begin
              list = ind_list[list]
              (*self.cutsel_t)[i] = list[0>((where(list eq (*self.cutsel_t)[i]))[0])]
           endif 
           dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T'+dm_to_string(i)),set_value=sel_list,select=(*self.cutsel_t)[i]
           self->my_widget_control,'cutselfrom_T'+dm_to_string(i),self.dispBas,set_value=dm_to_string((*self.cutsel_tft)[i,0])
           self->my_widget_control,'cutselto_T'+dm_to_string(i),  self.dispBas,set_value=dm_to_string((*self.cutsel_tft)[i,1])
       endfor
    endelse
    ;cut
    if self.cut_y[0>(self.samp_typ)] gt n_list then self.cut_y[0>(self.samp_typ)] = 0
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutX'),set_value=sel_list,select=self.cut_x[0>(self.samp_typ)]
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutY'),set_value=['Intensity',sel_list],select=self.cut_y[0>(self.samp_typ)]
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutColor'), select=self.cut_color
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutSymbol'),select=self.cut_symbol
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutLine'),  select=self.cut_line
    self->my_widget_control,'cutFrom', self.dispBas,set_value=dm_to_string(self.cut_ran[0])
    self->my_widget_control,'cutTo',   self.dispBas,set_value=dm_to_string(self.cut_ran[1])
    self->my_widget_control,'cutStep', self.dispBas,set_value=dm_to_string(self.cut_step)
    self->my_widget_control,'cutYfrom',self.dispBas,set_value=dm_to_string(self.cut_yran[0])
    self->my_widget_control,'cutYto',  self.dispBas,set_value=dm_to_string(self.cut_yran[1])
    ;slice
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='slicesel_x'),set_value=sel_list,select=self.slice_x[0>(self.samp_typ)]
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='slicesel_y'),set_value=sel_list,select=self.slice_y[0>(self.samp_typ)]
    self->my_widget_control,'dispxfrom', self.dispBas,set_value=dm_to_string(self.slice_xran[0])
    self->my_widget_control,'dispxto',   self.dispBas,set_value=dm_to_string(self.slice_xran[1])
    self->my_widget_control,'dispxstep', self.dispBas,set_value=dm_to_string(self.slice_xstep)
    self->my_widget_control,'dispyfrom', self.dispBas,set_value=dm_to_string(self.slice_yran[0])
    self->my_widget_control,'dispyto',   self.dispBas,set_value=dm_to_string(self.slice_yran[1])
    self->my_widget_control,'dispystep', self.dispBas,set_value=dm_to_string(self.slice_ystep)
    self->my_widget_control,'dispIfrom', self.dispBas,set_value=dm_to_string(self.slice_iran[0])
    self->my_widget_control,'dispIto',   self.dispBas,set_value=dm_to_string(self.slice_iran[1])
    self->my_widget_control,'dispSmooth',self.dispBas,set_value=dm_to_string(self.slice_smooth,/int)
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispColor'),  select=self.slice_color
    dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispShading'),select=self.slice_shading
    ;volume
    if keyword_set(vol_existed) then begin
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='volselX'),set_value=sel_list,select=self.volsel_x[0>(self.samp_typ)]
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='volselY'),set_value=sel_list,select=self.volsel_y[0>(self.samp_typ)]
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='volselZ'),set_value=sel_list,select=self.volsel_z[0>(self.samp_typ)]       
       self->my_widget_control,'volxfrom', self.dispBas,set_value=dm_to_string(self.vol_xran[0])
       self->my_widget_control,'volxto',   self.dispBas,set_value=dm_to_string(self.vol_xran[1])
       self->my_widget_control,'volxstep', self.dispBas,set_value=dm_to_string(self.vol_xstep)
       self->my_widget_control,'volyfrom', self.dispBas,set_value=dm_to_string(self.vol_yran[0])
       self->my_widget_control,'volyto',   self.dispBas,set_value=dm_to_string(self.vol_yran[1])
       self->my_widget_control,'volystep', self.dispBas,set_value=dm_to_string(self.vol_ystep)
       self->my_widget_control,'volzfrom', self.dispBas,set_value=dm_to_string(self.vol_zran[0])
       self->my_widget_control,'volzto',   self.dispBas,set_value=dm_to_string(self.vol_zran[1])
       self->my_widget_control,'volzstep', self.dispBas,set_value=dm_to_string(self.vol_zstep)
       self->my_widget_control,'volIfrom', self.dispBas,set_value=dm_to_string(self.vol_iran[0])
       self->my_widget_control,'volIto',   self.dispBas,set_value=dm_to_string(self.vol_iran[1])
       self->my_widget_control,'volSmooth',self.dispBas,set_value=dm_to_string(self.vol_smooth,/int)
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='volColor'), select=self.vol_color
       dm_set_droplist,widget_info(self.dispBas,find_by_uname='volInterp'),select=self.vol_interp
    endif
end

pro dcs_mslice::reset_geom,proj_old=proj_old,disp_old=disp_old
    if n_elements(proj_old) ne 0 then begin
       widget_control,self.projBas,/update,/map
       geom1  = widget_info(self.projBas,/geometry)
       margin = total(1-self.view)*geom1.margin*2*(!version.os_family eq 'Windows')
    endif else margin = 0
    if n_elements(disp_old) ne 0 then begin
       widget_control,self.dispBas,/update,/map
       geom2 = widget_info(self.dispBas,/geometry)
    endif
    if self.idl_version ge 5.5 then begin
       if n_elements(proj_old) ne 0 then self.geom[1] = self.geom[1]+geom1.ysize-proj_old.ysize
       if n_elements(disp_old) ne 0 then self.geom[1] = self.geom[1]+geom2.ysize-disp_old.ysize
       widget_control,self.tlb,ysize=self.geom[1]-margin
    endif else begin
       if n_elements(proj_old) ne 0 then self.geom[1] = self.geom[1]+geom1.scr_ysize-proj_old.scr_ysize
       if n_elements(disp_old) ne 0 then self.geom[1] = self.geom[1]+geom2.scr_ysize-disp_old.scr_ysize
       widget_control,self.tlb,scr_ysize=self.geom[1]-margin
    endelse
    widget_control,self.tlb,/update
end

;reset menu item sensitivity and ticks accordingly
;keywords:
;   geomsensitive: if set to 1: instrument geometry menus will be sensitive
;                  if set to 0: instrument geometry menus will be insensitive
;                  if not set, no action
;   backgroundmenuonly: if set, only reset the background menu
pro dcs_mslice::reset_menu,geomsensitive=geomsensitive,backgroundmenuonly=backgroundmenuonly,pdt_reset=pdt_reset
    if self.samp_typ eq -1 then begin 
       optnmenu = widget_info(self.tlb,find_by_uname='optnMenu')
       self->my_menu_toggle,'bin'+['WeighMean','ArithMean','Sum'],self.bin_avgsum+1,base=optnmenu
       dm_set_button,widget_info(optnmenu,find_by_uname='binAvg'),(self.bin_avgsum ne 1),set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
       self->my_menu_toggle,'empttol_'+['cwt','wtc'],self.empt_tol_optn
       self->my_menu_toggle,'pbg'+['Black','White'],self.pbgcol_bw,base=optnmenu
       self->my_menu_toggle,'ppvc'+['None','Avg2','Avg3'],self.ppvc_choice,base=optnmenu
       self->my_menu_toggle,'tdir'+['Inside','Outside'],self.tdir_choice,base=optnmenu
       dm_set_button,widget_info(optnmenu,find_by_uname='binCS'),self.bin_conststep,onstring='Constant Intervals',offstring='Nonconstant Intervals'
       dm_set_button,widget_info(optnmenu,find_by_uname='binexLib'),self.bin_extlib,onstring='Use External Binning Library',offstring='Use Internal Binning Routine'
       self->my_menu_toggle,'pdchoice_'+dm_to_string(indgen(n_elements((*self.pdim_list)[0,*])),/int),self.pdim_choice,base=optnmenu
       return
    endif 
    ;background menu and parameter menu
    bkgrmenu = widget_info(self.tlb,find_by_uname='bkgrMenu')
    parmmenu = widget_info(self.tlb,find_by_uname='parmMenu')
    emptBut  = widget_info(self.tlb,find_by_uname='emptBut')
    emptBut2 = widget_info(self.tlb,find_by_uname='emptBut2')
    macsdBut = widget_info(self.tlb,find_by_uname='macsdeteffBut')
    wandeBut = widget_info(self.tlb,find_by_uname='wandeiefBut')
    if ((self.instrname eq 'macs') and (macsdBut eq 0)) or ((self.instrname ne 'macs') and (macsdBut ne 0)) or $
       ((self.instrname eq 'wand') and (wandeBut eq 0)) or ((self.instrname ne 'wand') and (wandeBut ne 0)) or $
       ((self.extraempcan_yn eq 1) and (emptBut2 eq 0)) or ((self.extraempcan_yn eq 0) and (emptBut2 ne 0)) then begin
       ;rebuild the background menu and parameter menu
       self->my_widget_control,['vdclBut','vandEscanBut','vandBut','bgclBut','bgovBut','bgdkBut','bgtcBut','bgflBut','emptBut'+dm_to_string(reverse(indgen(n_elements(self.bgdata)-1)+2))],bkgrmenu,/destroy
       self->my_widget_control,['wdirBut','ddirBut','macshistowidBut','macsdeteffBut','wandeiefBut','detoffsetBut','emfcBut'],parmmenu,/destroy
       self->my_widget_control,'emptBut',bkgrmenu,set_value='Load Empty Can File(s)'+(['',' 1'])[self.extraempcan_yn]
       if self.extraempcan_yn then begin
          for i=2,n_elements(self.bgdata) do $
          void = dm_widget_button(bkgrmenu,value='Load Empty Can File(s) '+dm_to_string(i),uname='emptBut'+dm_to_string(i))
          void = widget_button(parmmenu,value='Change Multiple-Set Empty Can Factors...',uname='emfcBut',sensitive=0)
          if self.instrname eq 'wand' then begin
             void = widget_button(parmmenu,value='Change WAND Detector Offset Angle...',uname='detoffsetBut',/separator)
             void = widget_button(parmmenu,value='Change WAND Incident Wavelength...',uname='wandeiefBut')
          endif
          if self.instrname eq 'macs' then begin
             void = widget_button(parmmenu,value='Change MACS Detector Offset Angle...',uname='detoffsetBut',/separator)
             void = widget_button(parmmenu,value='Change MACS Detector Efficiency...',uname='macsdeteffBut')
             void = widget_button(parmmenu,value='Change MACS Histogram Channel Width...',uname='macshistowidBut')
          endif
          void = widget_button(parmmenu,value='Set Data Directory...',uname='ddirBut',/separator)
          void = widget_button(parmmenu,value='Set Working Directory...',uname='wdirBut')
       endif else begin
          if self.instrname eq 'wand' then begin
             void = widget_button(parmmenu,value='Change WAND Detector Offset Angle...',uname='detoffsetBut',/separator)
             void = widget_button(parmmenu,value='Change WAND Incident Wavelength...',uname='wandeiefBut')
          endif
          if self.instrname eq 'macs' then begin
             void = widget_button(parmmenu,value='Change MACS Detector Offset Angle...',uname='detoffsetBut',/separator)
             void = widget_button(parmmenu,value='Change MACS Detector Efficiency...',uname='macsdeteffBut')
             void = widget_button(parmmenu,value='Change MACS Histogram Channel Width...',uname='macshistowidBut')
          endif
          void = widget_button(parmmenu,value='Set Data Directory...',uname='ddirBut',/separator)
          void = widget_button(parmmenu,value='Set Working Directory...',uname='wdirBut')
       endelse
       void    = dm_widget_button(bkgrmenu,value='Load Detector Background File(s)',uname='bgflBut')
       void    = dm_widget_button(bkgrmenu,value='Load Time Channel Background File(s)',uname='bgtcBut')
       void    = dm_widget_button(bkgrmenu,value='Load Dark Count File(s)',uname='bgdkBut')
       void    = dm_widget_button(bkgrmenu,value='Set Overall Background Rate...',uname='bgovBut')
       void    = widget_button(bkgrmenu,value='Clear Background',uname='bgclBut')
       if self.instrname eq 'macs' then begin
          void = dm_widget_button(bkgrmenu,value='Load Kidney Scan Vanadium File(s)',uname='vandBut',/separator)
          void = dm_widget_button(bkgrmenu,value='Load Energy Scan Vanadium File(s)',uname='vandEscanBut')
       endif else $
          void = dm_widget_button(bkgrmenu,value='Load Vanadium File(s)',uname='vandBut',/separator)
       void    = widget_button(bkgrmenu,value='Clear Vanadium',uname='vdclBut')
    endif
    if self.extraempcan_yn then begin
       self->my_widget_control,'emfcBut',parmmenu,sensitive=ptr_valid(self.bgdata[1])
       for i=2,n_elements(self.bgdata) do dm_set_button,widget_info(bkgrmenu,find_by_uname='emptBut'+dm_to_string(i)),ptr_valid(self.bgdata[i-1]),sensitive=ptr_valid(self.bgdata[i-2])
    endif
    dm_set_button,widget_info(bkgrmenu,find_by_uname='emptBut'),ptr_valid(self.bgdata[0])
    dm_set_button,widget_info(bkgrmenu,find_by_uname='bgflBut'),ptr_valid(self.bgdetsPtr)
    dm_set_button,widget_info(bkgrmenu,find_by_uname='bgtcBut'),ptr_valid(self.bgtchnPtr)
    if ptr_valid(self.bgratePtr) then begin
       dm_set_button,widget_info(bkgrmenu,find_by_uname='bgdkBut'),(n_elements(*self.bgratePtr) gt 1)
       dm_set_button,widget_info(bkgrmenu,find_by_uname='bgovBut'),(n_elements(*self.bgratePtr) eq 1)
    endif else begin
       dm_set_button,widget_info(bkgrmenu,find_by_uname='bgdkBut'),0
       dm_set_button,widget_info(bkgrmenu,find_by_uname='bgovBut'),0
    endelse
    self->my_widget_control,['bgflBut','bgdkBut'],bkgrmenu,sensitive=((self.instrname ne 'wand') and (self.instrname ne 'macs'))
    self->my_widget_control,'bgovBut',bkgrmenu,sensitive=(self.instrname eq 'dcs')
    self->my_widget_control,'bgtcBut',bkgrmenu,sensitive=((self.samp_typ ne 2) and (self.instrname ne 'wand') and (self.instrname ne 'macs'))
    if self.instrname eq 'macs' then begin
       self->my_widget_control,'macsdeteffBut',parmmenu,sensitive=ptr_valid(self.eff)
       self->my_widget_control,'macshistowidBut',parmmenu,sensitive=keyword_set(self.macshistomode)
       if ~ptr_valid(self.eff1) then begin
          dm_set_button,widget_info(bkgrmenu,find_by_uname='vandBut'),0
          dm_set_button,widget_info(bkgrmenu,find_by_uname='vandEscanBut'),0
       endif
    endif else dm_set_button,widget_info(bkgrmenu,find_by_uname='vandBut'),ptr_valid(self.eff)
    self->my_widget_control,'vdclBut',bkgrmenu,sensitive=((ptr_valid(self.eff) and (self.instrname ne 'macs')) or (ptr_valid(self.eff1) and (self.instrname eq 'macs')))
    self->my_widget_control,'bgclBut',bkgrmenu,sensitive=(total(ptr_valid([self.bgtchnPtr,self.bgratePtr,self.bgdetsPtr,self.bgdata[0]])) ne 0)
    i_ssfc = ptr_valid(self.ssfacPtr)
    if i_ssfc then i_ssfc = i_ssfc+(*self.ssfacPtr).det_or_en
    self->my_menu_toggle,['','readdet','readen']+'ssfcBut',i_ssfc,base=parmmenu
    self->my_widget_control,['readdetssfcBut','readenssfcBut'],parmmenu,sensitive=(self.instrname ne 'macs')
    if keyword_set(backgroundmenuonly) then return
    ;filemenu
    ftoolmenu = widget_info(self.tlb,find_by_uname='ftoolMenu')
    tmp = widget_info(ftoolmenu,/n_children)
    widget_control,ftoolmenu,sensitive=(tmp ne 0)
    self->my_widget_control,'nxspe_savephxBut',ftoolmenu,sensitive=ptr_valid(self.detPosPtr)
    self->my_widget_control,'savesliceiexy',set_value=(['I,Err,X,Y 4-Column File...','I,Err,X,Y,A3,A4 6-Column File...'])[(self.instrname eq 'macs') and self.macs_savea4]
    self->my_widget_control,'saveslicexyie',set_value=(['X,Y,I,Err Matrix File...','X,Y,A3,A4,I,Err Matrix File...'])[(self.instrname eq 'macs') and self.macs_savea4],$
       sensitive=(finite(self.slice_xstep) or finite(self.slice_xstep) or (~self.allow_voronoi))
    self->my_widget_control,'saveslicespe',sensitive=(finite(self.slice_xstep) or finite(self.slice_xstep) or (~self.allow_voronoi))
    self->my_widget_control,'savecutxye',set_value=(['X,Y,Err 3-Column File...','X,A3,A4,Y,Err 5-Column File...'])[(self.instrname eq 'macs') and self.macs_savea4] 
    ;file menu: add a click to the current file type
    ftypemenu = widget_info(self.tlb,find_by_uname='ftypeMenu')
    self->my_menu_toggle,'ftype_'+dm_to_string(indgen(n_elements(*self.ftypename)),/int),self.ftype,base=ftypemenu
    self->my_widget_control,['savesliceMenu','savecutMenu','saveprojMenu'],sensitive=(ptr_valid(self.projStrPtr) and ptr_valid(self.dataStrPtr))
    ;mask menu
    maskmenu = widget_info(self.tlb,find_by_uname='maskMenu')
    self->my_widget_control,['mkbga3But','mka3But','mkclbga3But','mkcla3But','mkptaiBut','mkdarkBut'],maskmenu,/destroy
    stmkmenu = widget_info(self.tlb,find_by_uname='stmkMenu')
    sensitive=((self.samp_typ ge 2) or ((self.instrname eq 'macs') and (self.samp_typ ne 0)))
    if ~sensitive then self.mask_darkangle = 0b
    if self.instrname eq 'macs' then begin
       void = dm_widget_button(stmkmenu,value='View PTAI Only',uname='mkptaiBut',onstring='View PTAI Only',offstring='View All',set_button=self.macs_ptaionly,/separator)
       void = dm_widget_button(stmkmenu,value='Mask Dark Angle...',uname='mkdarkBut',onstring='Mask Dark Angle...',offstring='Not Mask Dark Angle...',set_button=self.mask_darkangle,sensitive=sensitive)
       void = dm_widget_button(stmkmenu,value='Mask A3 By Angle...',uname='mka3But',/separator,set_button=ptr_valid(self.a3mask))
       void = dm_widget_button(stmkmenu,value='Mask Background A3 by Angle...',uname='mkbga3But',set_button=ptr_valid(self.bga3mask))
       void = widget_button(maskmenu,value='Clear A3 Mask',uname='mkcla3But',sensitive=ptr_valid(self.a3mask))
       void = widget_button(maskmenu,value='Clear Background A3 Mask',uname='mkclbga3But',sensitive=ptr_valid(self.bga3mask))
    endif else begin
       void = dm_widget_button(stmkmenu,value='Mask Dark Angle...',uname='mkdarkBut',onstring='Mask Dark Angle...',offstring='Not Mask Dark Angle...',set_button=self.mask_darkangle,sensitive=sensitive,/separator)
    endelse
    self->my_widget_control,'mkdbMenu',sensitive=(self.instrname eq 'dcs')
    if self.instrname eq 'dcs' then self->dm_mask_detbanks
    self->my_widget_control,'mkclBut',maskmenu,sensitive=(((self.instrname eq 'macs') and self.macs_ptaionly) or ptr_valid(self.mask_ar) or ptr_valid(self.mask)) or self.mask_darkangle,$
          set_value=(['Clear Mask','Clear Detector Mask'])[(self.instrname eq 'macs')]
    self->my_widget_control,'mksvBut',maskmenu,sensitive=(ptr_valid(self.mask) or ptr_valid(self.a3mask) or ptr_valid(self.mask_ar))
    self->my_widget_control,'mkplBut',maskmenu,sensitive=ptr_valid(self.dataStrPtr)
    self->my_widget_control,'mkbpBut',maskmenu,sensitive=(self.instrname ne 'macs')
    self->my_widget_control,'mkplangl',maskmenu,sensitive=((self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand'))
    dm_set_button,widget_info(maskmenu,find_by_uname='mkstangl'),ptr_valid(self.mask_ar)
    if ~ptr_valid(self.mask) then begin
       names = ['mkstdnum','mkpldnum','mkplangl','mkbpBut']
       for i=0,n_elements(names)-1 do dm_set_button,widget_info(self.tlb,find_by_uname=names[i]),0
    endif
    ;option menu
    self->setproperty,instrgeom=self.instrgeom,geomsensitive=geomsensitive
    ntype = n_elements(self.intn_typ_str)
    if self.intn_typ ge ntype or self.intn_typ lt 0 then self.intn_typ=0
    self->my_menu_toggle,'intnTyp'+dm_to_string(indgen(ntype),/int),self.intn_typ
    self->my_widget_control,'intnTyp'+dm_to_string(indgen(ntype-1)),sensitive=((self.instrname ne 'wand') and (self.instrname ne 'macs'))
    self->my_widget_control,'intnTyp'+dm_to_string([0,1,3]),sensitive=(self.instrname ne 'wand')  ;allow S(q,omega),symmetrized S(q,omega), and Chi(q,omega) for macs
    dm_set_button,widget_info(self.tlb,find_by_uname='intnEint'),self.eint_yn,onstring='E-Integrated Intensity Enabled',offstring='E-Integrated Intensity Disabled',$
       sensitive=((self.instrname ne 'wand') and (self.instrname ne 'macs'))  
    zerrmenu = widget_info(self.tlb,find_by_uname='zerrMenu')
    d_m_w    = (self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand')
    userspec1 = 'User Specify'+(['',' ('+dm_to_string(self.zero_error[0])+')'])[finite(self.zero_error[0])]
    userspec2 = 'User Specify'+(['',' ('+dm_to_string(self.bin_statrange[1])+')'])[self.bin_statrange[0]]
    if (d_m_w and (zerrmenu eq 0)) or (~d_m_w and (zerrmenu ne 0)) then begin  ;zerror error bar option for dcs,macs,and wand
       binmmenu = widget_info(self.tlb,find_by_uname='binmMenu')
       if zerrmenu ne 0 then self->my_widget_control,'zerrMenu',binmmenu,/destroy
       if d_m_w then begin
          zerrmenu = widget_button(binmmenu,value='Zero Intensity Error ('+(['User Specify','Statistical Estimate'])[self.bin_zeroerror]+')',uname='zerrMenu',/menu,/separator)
          void     = dm_widget_button(zerrmenu,value=userspec1,uname='zerruserspec',set_button=(self.bin_zeroerror eq 0))
          void     = dm_widget_button(zerrmenu,value='Statistical Estimate',uname='zerrstatest',set_button=(self.bin_zeroerror eq 1))
          statmenu = widget_button(zerrmenu,value='Estimate Size ('+(['Automatic','User Specify'])[self.bin_statrange[0]]+')',uname='statranMenu',/menu,/separator,sensitive=self.bin_zeroerror)
          void     = dm_widget_button(statmenu,value='Automatic',uname='statranauto',set_button=1-self.bin_statrange[0])
          void     = dm_widget_button(statmenu,value=userspec2,uname='statranuserspec',set_button=self.bin_statrange[0])
          void     = dm_widget_button(zerrmenu,value='Apply Estimate to All Intensities',uname='statall',set_button=self.bin_statrange[2],sensitive=self.bin_zeroerror)
       endif
    endif else if d_m_w then begin
       widget_control,zerrmenu,set_value='Zero Intensity Error ('+(['User Specify','Statistical Estimate'])[self.bin_zeroerror]+')'
       self->my_menu_toggle,'zerr'+['userspec','statest'],self.bin_zeroerror,base=zerrmenu
       self->my_widget_control,'zerruserspec',zerrmenu,set_value=userspec1
       self->my_widget_control,'statranMenu',zerrmenu,set_value='Estimate Size ('+(['Automatic','User Specify'])[self.bin_statrange[0]]+')',sensitive=self.bin_zeroerror
       self->my_menu_toggle,'statran'+['auto','userspec'],self.bin_statrange[0],base=zerrmenu
       self->my_widget_control,'statranuserspec',zerrmenu,set_value=userspec2
       self->my_widget_control,'statall',set_button=self.bin_statrange[2],sensitive=self.bin_zeroerror
    endif
    ;construct macs intensity type and extra viewing axis option
    macsintmenu = widget_info(self.tlb,find_by_uname='macsintMenu')
    macshismenu = widget_info(self.tlb,find_by_uname='macshisMenu')
    vaxismenu   = widget_info(self.tlb,find_by_uname='vaxisMenu')
    taxismenu   = widget_info(self.tlb,find_by_uname='taxisMenu')
    evaxsmenu   = widget_info(self.tlb,find_by_uname='evaxsMenu')
    if macshismenu ne 0 then begin
       widget_control,macshismenu,/destroy
       widget_control,macsintmenu,/destroy
    endif
    if taxismenu ne 0 then begin
       widget_control,taxismenu,/destroy
       widget_control,evaxsmenu,/destroy
    endif else self->my_widget_control,['extraVaxis','avgTemp'],/destroy
    if self.instrname eq 'dcs' then begin
       evaxsmenu = dm_widget_button(vaxismenu,value='Allow Extra Viewing Axis',/menu,uname='evaxsMenu',/separator)
       taxismenu = widget_button(vaxismenu,value='Temperature Choice ('+(['Sample','Control','Setpoint'])[self.macstemp_typ]+')',sensitive=self.extravaxis_yn[1],/menu,uname='taxisMenu')
       void = dm_widget_button(evaxsmenu,value='Crystal Axis U3',set_button=self.extravaxis_yn[0],onstring='Crystal Axis U3',offstring='Disallow Crystal Axis U3',uname='extraVaxis',sensitive=(self.samp_typ gt 0))
       void = dm_widget_button(evaxsmenu,value='Temerature',set_button=self.extravaxis_yn[1],onstring='Temperature',offstring='No Temperature',uname='evaxsTemp')
       void = dm_widget_button(evaxsmenu,value='Magnetic Field',set_button=self.extravaxis_yn[2],onstring='Magnetic Field',offstring='No Magnetic Field',uname='evaxsHfld')
       void = dm_widget_button(evaxsmenu,value='None',uname='evaxsNone',set_button=(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0),/separator)
       void = dm_widget_button(taxismenu,value='Sample',set_button=(self.macstemp_typ eq 0),uname='taxisSamp')
       void = dm_widget_button(taxismenu,value='Control',set_button=(self.macstemp_typ eq 1),uname='taxisCtrl')
       void = dm_widget_button(taxismenu,value='Setpoint',set_button=(self.macstemp_typ eq 2),uname='taxisSetp')
       void = dm_widget_button(taxismenu,value='Average Temperature of Each File',/set_button,onstring='Average Temperature of Each File',offstring='Keep Individual Temperature',uname='avgTemp',sensitive=0,/separator)
       if self.idl_version ge 5.6 then widget_control,evaxsmenu,set_button=(total(self.extravaxis_yn[(self.samp_typ eq 0):*]) ne 0) 
    endif else if self.instrname eq 'macs' then begin
       itypmenu = widget_info(self.tlb,find_by_uname='itypMenu')
       if keyword_set(self.macshistomode) and self.macs_rawintn and (self.macsintn_typ eq 2) then self.macsintn_typ = 0
       macsintmenu = widget_button(itypmenu,value='MACS Detector Choice ('+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]+')',/menu,uname='macsintMenu')
       macshismenu = widget_button(itypmenu,value='MACS Data Mode ('+(['Normal','Histogram','Event'])[self.macshistomode]+')',/menu,uname='macshisMenu')
       void = dm_widget_button(macsintmenu,value='SPEC',set_button=(self.macsintn_typ eq 0),uname='macsintSPEC')
       void = dm_widget_button(macsintmenu,value='DIFF',set_button=(self.macsintn_typ eq 1),uname='macsintDIFF')
       void = dm_widget_button(macsintmenu,value='SPEC+DIFF',set_button=(self.macsintn_typ eq 2),uname='macsintSPDF',sensitive=(~keyword_set(self.macshistomode)) or (~self.macs_rawintn))
       void = dm_widget_button(macshismenu,value='Normal',set_button=(self.macshistomode eq 0),uname='macshisto0')
       void = dm_widget_button(macshismenu,value='Histogram',set_button=(self.macshistomode eq 1),uname='macshisto1')
       void = dm_widget_button(macshismenu,value='Event',set_button=(self.macshistomode eq 2),uname='macshisto2')
       evaxsmenu = dm_widget_button(vaxismenu,value='Allow Extra Viewing Axis',/menu,uname='evaxsMenu',/separator)
       taxismenu = widget_button(vaxismenu,value='Temperature Choice ('+(['Sample','Control','Setpoint'])[self.macstemp_typ]+')',sensitive=self.extravaxis_yn[1],/menu,uname='taxisMenu')
       void = dm_widget_button(evaxsmenu,value='Temerature',set_button=self.extravaxis_yn[1],onstring='Temperature',offstring='No Temperature',uname='evaxsTemp')
       void = dm_widget_button(evaxsmenu,value='Magnetic Field',set_button=self.extravaxis_yn[2],onstring='Magnetic Field',offstring='No Magnetic Field',uname='evaxsHfld')
       void = dm_widget_button(evaxsmenu,value='Specify',set_button=self.extravaxis_yn[0],onstring='Specify',offstring='No Specify',uname='evaxsSpec')
       void = dm_widget_button(evaxsmenu,value='None',uname='evaxsNone',set_button=(total(self.extravaxis_yn) eq 0),sensitive=(total(self.extravaxis_yn) ne 0),/separator)
       void = dm_widget_button(taxismenu,value='Sample',set_button=(self.macstemp_typ eq 0),uname='taxisSamp')
       void = dm_widget_button(taxismenu,value='Control',set_button=(self.macstemp_typ eq 1),uname='taxisCtrl')
       void = dm_widget_button(taxismenu,value='Setpoint',set_button=(self.macstemp_typ eq 2),uname='taxisSetp')
       void = dm_widget_button(taxismenu,value='Average Temperature of Each File',set_button=self.tempavg_yn,onstring='Average Temperature of Each File',offstring='Keep Individual Temperature',uname='avgTemp',/separator)
       if self.idl_version ge 5.6 then widget_control,evaxsmenu,set_button=(total(self.extravaxis_yn) ne 0)  
    endif else begin
       onstr  = (['Allow Extra Viewing Axis','Allow Temperature as Extra Viewing Axis'])[ (self.instrname eq 'wand')]
       offstr = (['Disallow Extra Viewing Axis','Disallow Temperature as Extra Viewing Axis'])[ (self.instrname eq 'wand')]
       void = dm_widget_button(vaxismenu,value=onstr,uname='extraVaxis',set_button=self.extravaxis_yn[0],onstring=onstr,offstring=offstr,/separator)
       if self.instrname eq 'wand' then void = dm_widget_button(vaxismenu,value='Average Temperature of Each File',set_button=self.tempavg_yn,onstring='Average Temperature of Each File',offstring='Keep Individual Temperature',uname='avgTemp')
    endelse
    self->my_menu_toggle,'bin'+['WeighMean','ArithMean','Sum'],self.bin_avgsum+1 
    dm_set_button,widget_info(self.tlb,find_by_uname='binAvg'),(self.bin_avgsum ne 1),set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]    
    dm_set_button,widget_info(self.tlb,find_by_uname='binCS'),self.bin_conststep,onstring='Constant Intervals',offstring='Nonconstant Intervals'      
    dm_set_button,widget_info(self.tlb,find_by_uname='binexLib'),self.bin_extlib,onstring='Use External Binning Library',offstring='Use Internal Binning Routine'     
    if (self.instrname eq 'dcs') or (self.instrname eq 'macs') then begin ;reconstruct monitor count option menu
       mon_instr = ([0,1])[self.instrname eq 'macs']
       self->my_menu_toggle,'mon'+['Sum','Fit'],self.mon_sumfit[mon_instr]
       self->my_widget_control,'monSum',set_value=(['Sum','Automatic'])[self.instrname eq 'macs']
       self->my_widget_control,'monFit',set_value=(['Integrated Intensity','Specify...'])[self.instrname eq 'macs']
       moncMenu = widget_info(self.tlb,find_by_uname='moncMenu')
       mondMenu = widget_info(moncMenu,find_by_uname='mondMenu')
       monlamb2 = widget_info(moncMenu,find_by_uname='monlamb2')
       widget_control,moncMenu,/sensitive
       if self.instrname eq 'dcs' then begin
          self.mon_typ[mon_instr] = 0>(self.mon_typ[mon_instr])<2
          if monlamb2 ne 0 then begin
             widget_control,monlamb2,/destroy
             widget_control,mondmenu,/destroy
             mondmenu = widget_button(moncmenu,value='Monitor Choice ('+(['FC0','BM1','Duration'])[self.mon_typ[mon_instr]]+')',/menu,/separator,uname='mondMenu')
             void     = dm_widget_button(mondmenu,value='FC0',uname='monFC0')
             void     = dm_widget_button(mondmenu,value='BM1 (default)',uname='monBM1')
             void     = dm_widget_button(mondmenu,value='Duration',uname='monTIME')
          endif else widget_control,mondmenu,set_value='Monitor Choice ('+(['FC0','BM1','Duration'])[self.mon_typ[mon_instr]]+')'
          self->my_menu_toggle,'mon'+['FC0','BM1','TIME'],self.mon_typ[mon_instr],base=mondmenu
       endif else begin
          if monlamb2 eq 0 then begin
             widget_control,mondMenu,/destroy
             mondmenu = widget_button(moncmenu,value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')',/menu,/separator,uname='mondMenu')
             void     = dm_widget_button(mondmenu,value='Time',uname='monMACSTIME')
             void     = dm_widget_button(mondmenu,value='Monitor',uname='monMACSMONITOR')
             void     = dm_widget_button(mondmenu,value='Pulse Number',uname='monMACSPULSE')
             monlamb2 = dm_widget_button(moncmenu,value='Apply Monitor Lambda/2 Correction',/separator,uname='monlamb2')
          endif else widget_control,mondmenu,set_value='Monitor Choice ('+(['Time','Monitor','Pulse Number'])[self.mon_typ[mon_instr]]+')'
          self->my_menu_toggle,'monMACS'+['TIME','MONITOR','PULSE'],self.mon_typ[mon_instr],base=mondmenu
          self->my_widget_control,'monMACSPULSE',mondmenu,sensitive=(self.macshistomode ne 0)
          dm_set_button,monlamb2,self.macs_lamb2,onstring='Apply Monitor Lambda/2 Correction',offstring='No Monitor Lambda/2 Correction'
       endelse
    endif else $
       self->my_widget_control,'moncMenu',sensitive=0
    if ptr_valid(self.absp_info) then $
       self->my_menu_toggle,'absp'+['None','Cylinder','Annulus','Sphere','Flat'],(*(self.absp_info))[0]+1 $
    else $
       self->my_menu_toggle,'absp'+['None','Cylinder','Annulus','Sphere','Flat'],0
    dm_set_button,widget_info(self.tlb,find_by_uname='abspCorr'),ptr_valid(self.absp_info)?((*(self.absp_info))[0] ne -1):0
    self->my_widget_control,'abspInt',sensitive=(ptr_valid(self.absp_info)?((*(self.absp_info))[0] ne -1):0),set_button=self.absp_asint
    self->my_menu_toggle,'vaxis'+['Ortho','Arbi'],self.view_ortharbi
    self->my_menu_toggle,'vfold'+['Mirror','Rotation'],(self.fold_type gt 1)
    self->my_menu_toggle,['mrrf_'+['One','Both'],'symrot_'+['Section','Quadrant','Fullmap']],self.fold_type
    self->my_menu_toggle,'symvec_'+['U1','Specify'],(self.symrot_startvec ne 0)
    self->my_menu_toggle,'nmrt_'+['Min','Nminus1'],self.numrot_optn
    self->change_powderline,self.powd_typ,/menuonly
    self->my_menu_toggle,'epad'+['None','Auto','Spec'],self.eadjust
    dm_set_button,widget_info(self.tlb,find_by_uname='eAdjust'),(self.eadjust ne 0)
    self->my_menu_toggle,'empttol_'+['cwt','wtc'],self.empt_tol_optn
    self->my_menu_toggle,'pbg'+['Black','White'],self.pbgcol_bw
    self->my_menu_toggle,'ppvc'+['None','Avg2','Avg3'],self.ppvc_choice
    self->my_menu_toggle,'tdir'+['Inside','Outside'],self.tdir_choice,base=optnmenu
    self->my_menu_toggle,'crossplot'+['No','Yes'],self.crossplot
    self->my_menu_toggle,'voronoi'+['No','Yes'],self.allow_voronoi
    if !version.os_family eq 'Windows' then self->my_menu_toggle,'textrm_'+['0','1'],self.ptrm_choice    
    dm_set_button,widget_info(self.tlb,find_by_uname='allowCrossplot'),self.crossplot
    dm_set_button,widget_info(self.tlb,find_by_uname='allowVoronoi'),self.allow_voronoi
    pdimmenu = widget_info(self.tlb,find_by_uname='pdimMenu')
    self->my_menu_toggle,'pdchoice_'+dm_to_string(indgen(n_elements((*self.pdim_list)[0,*])),/int),self.pdim_choice,base=pdimmenu
    self->my_widget_control,'userFunction',set_button=(strlen(self.userf_str) ne 0)
    dm_set_button,widget_info(self.tlb,find_by_uname='epadCheckfirst'),self.eadjust_checkfirst,onstring='Check the First File Only',offstring='Check All Files',sensitive=(self.eadjust ne 0)
    dm_set_button,widget_info(self.tlb,find_by_uname='darkCorr'),self.dcor_yn,onstring='Dark Angle Correction Enabled',offstring='Dark Angle Correction Disabled'
    dm_set_button,widget_info(self.tlb,find_by_uname='darkTraj'),self.datr_yn,onstring='Show Dark Angle Trajectory',offstring='Disable Dark Angle Trajectory'
    dm_set_button,widget_info(self.tlb,find_by_uname='edepEff'),self.edep_yn,onstring='E-Dependent Detector Efficiency Correction Enabled',offstring='E-Dependent Detector Efficiency Correction Disabled'
    dm_set_button,widget_info(self.tlb,find_by_uname='errBar'),1-self.errbar,onstring='Error Bar Disabled',offstring='Error Bar Enabled'
    dm_set_button,widget_info(self.tlb,find_by_uname='ncnrftp'),self.ncnrftp,onstring='Allow NCNR Data Repository',offstring='Disallow NCNR Data Repository',sensitive=((self.instrname eq 'dcs') or (self.instrname eq 'macs'))
    self->my_widget_control,['eAdjust','edepEff'],sensitive=((self.instrname ne 'wand') and (self.instrname ne 'macs'))
    spurmenu = widget_info(self.tlb,find_by_uname='spurionMenu')
    ghostbut = widget_info(spurmenu,find_by_uname='viewGhoston')
    widget_control,spurmenu,sensitive=(self.instrname ne 'wand')
    self->my_widget_control,'viewSpurion',spurmenu,set_value=(['View Spurion Caused by Detector','View Spurion Caused by Gas Recoil...'])[self.instrname eq 'macs']
    if self.instrname eq 'macs' then begin
       if ghostbut eq 0 then ghostbut = widget_button(spurmenu,value='View Ghoston',uname='viewGhoston')
    endif else if ghostbut ne 0 then widget_control,ghostbut,/destroy
    self->change_samptyp,self.samp_typ,pdt_reset=pdt_reset
    ;window menu
    dm_set_button,widget_info(self.tlb,find_by_uname='viewProj'),self.view[0],onstring='Hide Projection',offstring='Show Projection'
    dm_set_button,widget_info(self.tlb,find_by_uname='viewDisp'),self.view[1],onstring='Hide Display',offstring='Show Display'
end

;reset projection and display window
pro dcs_mslice::reset_pdt
    WIDGET_CONTROL,/HOURGLASS

    if ~widget_info(self.projBas,/valid_id) then return
    
    widget_control,self.tlb,update=0
    geom_proj = widget_info(self.projBas,/geometry)
    geom_disp = widget_info(self.dispBas,/geometry)
    self->my_widget_control,'calcInput',self.dispBas,get_value=calcstr
    widget_control,self.dispBas,/destroy
    widget_control,self.projBas,/destroy
    self->reset_projection
    self->reset_display,calcstr=calcstr
    self->reset_geom,proj_old=geom_proj,disp_old=geom_disp
end

;save the 4d rebinned data as a text file
;keywords:
;   file:   full filename, including the directory, if absent, a dialog window will show up
pro dcs_mslice::save_4d,file=file,error=error
    ext = 'txt'
    path = self.dirs[1]
    if n_elements(file) eq 0 then begin
       file = dm_choose_file(ext,/write,dialog_parent=self.tlb,title='Save 4d rebinned data as '+ext+'  file.',path=path)
       file = file[0]
    endif
    if strlen(file) eq 0 then return
    self->setproperty,workDir=path

    ;select data
    if (self.samp_typ ge 0) and self.absp_asint then $
       self->dm_choose_4ddata,x1dat=x1dat,x2dat=x2dat,x3dat=x3dat,x4dat=x4dat,idat=idat,x1tit=x1tit,x2tit=x2tit,x3tit=x3tit,x4tit=x4tit,itit=itit,weight=weight,error=error $
    else $
       self->dm_choose_4ddata,x1dat=x1dat,x2dat=x2dat,x3dat=x3dat,x4dat=x4dat,idat=idat,ierr=ierr,x1tit=x1tit,x2tit=x2tit,x3tit=x3tit,x4tit=x4tit,itit=itit,weight=weight,error=error
    if keyword_set(error) then return
    if n_elements(ierr) eq 0 then ierr = fltarr(size(idat,/dimensions))

    widget_control,/hourglass  ;busy signal
  
    openw,unit,file,/get_lun,error=openerr
    if openerr ne 0 then begin
       ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
       return
    endif
    ok_wght = (n_elements(weight) eq n_elements(idat))
    fmt  = '('+dm_to_string(6+ok_wght,/int)+'g)'
    header = dm_to_string(x1tit,data=x1dat[0],over=over,/addsemicolon)
    header = header+dm_to_string(x2tit,data=x2dat[0],over=over)
    header = header+dm_to_string(x3tit,data=x3dat[0],over=over)
    header = header+dm_to_string(x4tit,data=x4dat[0],over=over)
    header = header+dm_to_string([itit,'dI'],data=idat[0],over=over)
    if ok_wght then header = header+dm_to_string('weight',data=weight[0],over=over)
    printf,unit,header
    if ok_wght then begin
       for l=0L,n_elements(x4dat)-1 do begin
           for k=0L,n_elements(x3dat)-1 do begin
               for j=0L,n_elements(x2dat)-1 do begin
                   for i=0L,n_elements(x1dat)-1 do begin
                       if finite(idat[i,j,k,l]) then printf,unit,x1dat[i],x2dat[j],x3dat[k],x4dat[l],idat[i,j,k,l],ierr[i,j,k,l],weight[i,j,k,l],format=fmt
                   endfor
               endfor
           endfor
       endfor
    endif else begin        
       for l=0L,n_elements(x4dat)-1 do begin
           for k=0L,n_elements(x3dat)-1 do begin
               for j=0L,n_elements(x2dat)-1 do begin
                   for i=0L,n_elements(x1dat)-1 do begin
                       if finite(idat[i,j,k,l]) then printf,unit,x1dat[i],x2dat[j],x3dat[k],x4dat[l],idat[i,j,k,l],ierr[i,j,k,l],format=fmt
                   endfor
               endfor
           endfor
       endfor
    endelse
    free_lun,unit
end

;save cuts in files
;keyword:
;   dave:   if set, save cut as dave files
;   dPtr:   if present, will pass the cut data as a dave pointer
;   xye:    if set, save cut as x, y, yerr 3-column file
;   file:   full filename, including the directory, if absent, a dialog window will show up
pro dcs_mslice::save_cut,dave=dave,dPtr=dPtr,xye=xye,file=file,error=error
    if (~ self->pd_valid()) then return
    ext = ''
    if keyword_set(dave) then ext = 'dave'
    if keyword_set(xye)  then ext = 'xye'
    if (~ arg_present(dPtr)) then begin
       if strlen(ext) eq 0 then return
       path = self.dirs[1]
       if n_elements(file) eq 0 then begin
          file = dm_choose_file(ext,/write,dialog_parent=self.tlb,title='Save cut as '+ext+' file.',path=path)
          file = file[0]
       endif
       if strlen(file) eq 0 then return
       self->setproperty,workDir=path
       openw,unit,file,/get_lun,error=openerr
       if openerr ne 0 then begin
          ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
          return
       endif
    endif

    ;select data
    if (~self.errbar) or ((self.samp_typ ge 0) and self.absp_asint) then $
       self->dm_choose_cutdata,xdat=xdat,ydat=ydat,a3=a3,a4=a4,xtit=xtit,ytit=ytit,weight=weight,error=error $
    else $
       self->dm_choose_cutdata,xdat=xdat,ydat=ydat,yerr=yerr,a3=a3,a4=a4,xtit=xtit,ytit=ytit,weight=weight,error=error
    if keyword_set(error) then begin
       if n_elements(unit) ne 0 then free_lun,unit
       return
    endif

    if keyword_set(xye) then begin
       ok_wght = (n_elements(weight) eq n_elements(ydat))
       ok_yerr = (n_elements(yerr) eq n_elements(ydat))
       ok_a3   = (n_elements(a3) eq n_elements(ydat))
       fmt     = '('+dm_to_string(2+2*ok_a3+ok_yerr+ok_wght,/int)+'g)'
       header  = dm_to_string(xtit,data=xdat[0],over=over,/addsemicolon)
       if ok_a3 then header = header+dm_to_string(['a3(/deg)','a4(/deg)'],data=a3[0],over=over)
       header = header+dm_to_string(ytit,data=ydat[0],over=over)
       if ok_yerr then header = header+dm_to_string('dI',data=yerr[0],over=over)
       if ok_wght then header = header+dm_to_string('weight',data=weight[0],over=over)
       printf,unit,header
       if ok_yerr then begin
          if ok_a3 then begin
             if ok_wght then begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],a3[i],a4[i],ydat[i],yerr[i],weight[i],format=fmt
             endif else begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],a3[i],a4[i],ydat[i],yerr[i],format=fmt 
             endelse
          endif else begin
             if ok_wght then begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],ydat[i],yerr[i],weight[i],format=fmt
             endif else begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],ydat[i],yerr[i],format=fmt
             endelse
          endelse
       endif else begin
          if ok_a3 then begin
             if ok_wght then begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],a3[i],a4[i],ydat[i],weight[i],format=fmt
             endif else begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],a3[i],a4[i],ydat[i],format=fmt
             endelse
          endif else begin
             if ok_wght then begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],ydat[i],weight[i],format=fmt
             endif else begin
                for i=0L,n_elements(xdat)-1 do printf,unit,xdat[i],ydat[i],format=fmt
             endelse
          endelse
       endelse
    endif

    if keyword_set(dave) or arg_present(dPtr) then begin
       if n_elements(yerr) ne 0 then $
          ret = create_dave_pointer(    $
                dPtr,                   $
                qty     = ydat,         $
                qtlabel = ytit,         $
                err     = yerr,         $
                xvals   = xdat,         $
                xtype   = 'POINT',      $
                xlabel  = xtit          $
          ) $
       else $
          ret = create_dave_pointer(    $
                dPtr,                   $
                qty     = ydat,         $
                qtlabel = ytit,         $
                xvals   = xdat,         $
                xtype   = 'POINT',      $
                xlabel  = xtit          $
          )
       info = 'Cut from Mslice'+((self.samp_typ lt 0)?'':(' for '+(*self.ftypename)[self.ftype]))
       if self.samp_typ ge 0 then info = [info,'data file name: '+((n_elements(*self.file_recent) eq 2)?(*self.file_recent)[1]:(*self.file_recent)[1]+' to '+(*self.file_recent)[-1]),$
                                          'data directory: '+(*self.file_recent)[0]]
       *(*(*dPtr).dataStrPtr).commonStr.treatmentPtr = info
       if n_elements(unit) ne 0 then begin
          davePtr = dPtr
          save,davePtr,filename=file,/compress
          heap_free,dPtr
       endif
    endif

    if n_elements(unit) ne 0 then free_lun,unit
end

;save detector info as a phx file
pro dcs_mslice::save_phx
    if ~ptr_valid(self.detposptr) then begin
       ok = dialog_message('The detector info is unavailable.',/error,dialog_parent=self.tlb,/center)
       self->my_widget_control,'nxspe_savephxBut',widget_info(self.tlb,find_by_uname='ftoolMenu'),sensitive=0
       return
    endif
    path = self.dirs[1]
    file = dm_choose_file('phx',/write,dialog_parent=self.tlb,title='Save detecotor info:',path=path)
    file = file[0]
    if strlen(file) eq 0 then return
    self->setproperty,workDir=path
    widget_control,/hourglass
    if ~self.psi_spe then begin
       ok = dialog_message('The psi definition is not spe type. The information saved in the phx file might not be correct.',dialog_parent=self.tlb,/center)
    endif
    dm_write_phx,file,(*self.detposptr).two_theta,(*self.detposptr).dtwo_theta,(*self.detposptr).psi,(*self.detposptr).dpsi
end

;save the projection volume as a text file
;keywords:
;   file:   full filename, including the directory, if absent, a dialog window will show up
pro dcs_mslice::save_projection,file=file
    ext = 'txt'
    if self.samp_typ eq - 1 then begin
       title = 'Save all data as a '+ext+' file.' 
       nitem = n_elements((*self.dataPtr).label)
       ndat  = n_elements((*self.dataPtr).data[*,0])
    endif else begin
       title = 'Save projection as a '+ext+' file.'
       nen   = n_elements((*self.projStrPtr).en)
       n2th  = n_elements((*self.projStrPtr).two_theta)
    endelse
    
    path = self.dirs[1]
    if n_elements(file) eq 0 then begin
       file = dm_choose_file(ext,/write,dialog_parent=self.tlb,title=title,path=path)
       file = file[0]
    endif
    if strlen(file) eq 0 then return
    self->setproperty,workDir=path
    
    openw,unit,file,/get_lun,error=openerr
    if openerr ne 0 then begin
       ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
       return
    endif
    
    widget_control,/hourglass
    
    ok_y = ((self.samp_typ eq 0) and (self.intn_typ eq (n_elements(self.intn_typ_str)-2)))  ;J(Q,y) intensity type & powder mode, MACS doesn't support this mode
    
    ;a4 for MACS
    if (self.instrname eq 'macs') then begin
       if self.macs_savea4 then begin
          self->dm_choose_trajdata,data=a4,/macsa4
          self->dm_choose_trajdata,data=a3,/macsa3
       endif
       ok_a4 = (n_elements(a4) ne 0)
       ok_ht = keyword_set(self.macshistomode)
       intst = (['Ispec','Idiff','Ispec+diff'])[self.macsintn_typ]
       self->dm_choose_trajdata,-1,data=macs_int,error=macs_err
       fmt = '('+dm_to_string(4+total([self.extravaxis_yn,self.samp_typ])+2*(n_elements(a4) ne 0)+ok_ht*2,/int)+'g)'
       if self.samp_typ eq 0 then header = dm_to_string('|Q|(\AA!U-1!N)',data=(*self.projStrPtr).Q[0],over=over,/addsemicolon) $
       else header = dm_to_string(self.traj_type[0:1],data=(*self.projStrPtr).V[0],over=over,/addsemicolon)
       if ok_a4 then header = header+dm_to_string(['A3(\deg)','A4(\deg)'],data=a3[0],over=over)
       header = header+dm_to_string('E(meV)',data=(*self.projStrPtr).en[0],over=over)
       if ok_ht then begin
          self->dm_choose_trajdata,4,data=time,unit=tunit,macshistonchan=ntchan
          self->dm_choose_trajdata,-4,data=weight
          header = header+dm_to_string('Time('+tunit+')',data=time[0],over=over)
       endif
       if self.extravaxis_yn[1] then header = header+dm_to_string('T(K)',data=(*self.projStrPtr).temperature[0],over=over)
       if self.extravaxis_yn[2] then header = header+dm_to_string('H(T)',data=(*self.projStrPtr).hfield[0],over=over)
       if self.extravaxis_yn[0] then header = header+dm_to_string(self.extravaxis_spec[0]+((strlen(self.extravaxis_spec[1]) eq 0)?'':('('+self.extravaxis_spec[1]+')')),data=(*self.projStrPtr).specify[0],over=over)
       header = header+dm_to_string([intst,'d'+intst],data=macs_int[0],over=over)
       if ok_ht then header = header+dm_to_string('weight',data=weight[0],over=over)
    endif else if self.instrname eq 'dcs' then begin
       fmt = '('+dm_to_string(4+total(self.extravaxis_yn[(self.samp_typ eq 0):*])+((self.samp_typ eq 1) or (self.samp_typ eq 3))+keyword_set(ok_y),/int)+'g)'
       if self.samp_typ eq 0 then begin
          header = dm_to_string('|Q|(\AA!U-1!N)',data=(*self.projStrPtr).Q[0],over=over,/addsemicolon)
          if ok_y then header = header+dm_to_string('y(\AA!U-1!N)',data=(*self.projStrPtr).y[0],over=over)
       endif else begin
          header = dm_to_string(self.traj_type[0:1+self.extravaxis_yn[0]],data=(*self.projStrPtr).V[0],over=over,/addsemicolon)
       endelse
       if self.samp_typ ne 2 then header = header+dm_to_string('E(meV)',data=(*self.projStrPtr).en[0],over=over)
       if self.extravaxis_yn[1] then header = header+dm_to_string('T(K)',data=(*self.projStrPtr).temperature[0],over=over)
       if self.extravaxis_yn[2] then header = header+dm_to_string('H(T)',data=(*self.projStrPtr).hfield[0],over=over)
       header = header+dm_to_string(['I','dI'],data=(*self.projStrPtr).qty[0],over=over)
    endif else begin
       if self.samp_typ eq -1 then begin
          tmpstr = (*self.dataPtr).label[2:*]+'('+(*self.dataPtr).unit[2:*]+')'
          tmpstr = [tmpstr,(*self.dataPtr).label[0:1]+'('+(*self.dataPtr).unit[0:1]+')']
          header = dm_to_string(tmpstr,data=(*self.dataPtr).data[0],over=over,/addsemicolon)
       endif else if self.samp_typ eq 0 then begin
          header = dm_to_string('|Q|(\AA!U-1!N)',data=(*self.projStrPtr).Q[0],over=over,/addsemicolon)
          if ok_y then header = header+dm_to_string('y(\AA!U-1!N)',data=(*self.projStrPtr).y[0],over=over)
          header = header+dm_to_string('E(meV)',data=(*self.projStrPtr).en[0],over=over)
          header = header+dm_to_string(['I','dI'],data=(*self.projStrPtr).qty[0],over=over)
       endif else if self.samp_typ eq 1 then begin
          header = dm_to_string(self.traj_type[0:1+self.extravaxis_yn[0]],data=(*self.projStrPtr).V[0],over=over,/addsemicolon)
          header = header+dm_to_string('E(meV)',data=(*self.projStrPtr).en[0],over=over)
          header = header+dm_to_string(['I','dI'],data=(*self.projStrPtr).qty[0],over=over)
       endif else if self.samp_typ eq 2 then begin
          header = dm_to_string(self.traj_type[0:1],data=(*self.projStrPtr).V[0],over=over,/addsemicolon)
          if self.extravaxis_yn[0] then begin
             if self.instrname eq 'wand' then header = header+dm_to_string('T (K)',data=(*self.projStrPtr).temperature[0],over=over) $
             else header = dm_to_string(self.traj_type[2],data=(*self.projStrPtr).V[0],over=over)
          endif
          header = header+dm_to_string(['I','dI'],data=(*self.projStrPtr).qty[0],over=over)
       endif else if self.samp_typ eq 3 then begin
          header = dm_to_string(self.traj_type[0:1+self.extravaxis_yn[0]],data=(*self.projStrPtr).V[0],over=over,/addsemicolon)
          header = header+dm_to_string('E(meV)',data=(*self.projStrPtr).en[0],over=over)
          header = header+dm_to_string(['I','dI'],data=(*self.projStrPtr).qty[0],over=over)
       endif
    endelse
    
    evaxs = self.extravaxis_yn[1]+2*self.extravaxis_yn[2]+4*(((self.instrname eq 'dcs') and (self.samp_typ eq 0))?0:self.extravaxis_yn[0])+10*(keyword_set(ok_a4) or ok_y)
    
    if self.samp_typ eq 0 or self.samp_typ eq 1 then begin
       if (self.instrname eq 'dcs') and (self.extravaxis_yn[1] or self.extravaxis_yn[2]) then begin
          tmp = size((*self.projStrPtr).qty)
          if tmp[0] eq 3 then nfile = tmp[3] else nfile = 1L
       endif else nfile = 1L
    endif else if self.samp_typ ge 2 then begin
       tmp = size((*self.projStrPtr).V)
       if tmp[0] eq self.samp_typ then nfile=1L else nfile=tmp[self.samp_typ]
    endif
    
    if n_elements(header) ne 0 then printf,unit,header
    case self.samp_typ of
      -1: begin
          fmt = '('+dm_to_string(nitem,/int)+'g)'
          for i=0L,ndat-1L do $
              printf,unit,reform((*self.dataPtr).data[i,[2+indgen(nitem-2),0,1]]),format=fmt
          end
       0: begin
          if self.instrname eq 'macs' then begin  ;with T axis
             qid = 0>(self.macsintn_typ)<1
             if ok_ht then begin  ;histogram data
                for j=0L,n2th-1L do begin
                    for i=0L,nen-1L do begin
                        for k=0L,ntchan-1L do begin
                            if ~finite(macs_int[k,i,j]) then continue
                            case evaxs of  ;could be slow, bearable, to save code space
                                 0: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 1: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 2: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 3: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 4: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 5: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 6: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 7: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                10: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                11: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                12: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                13: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                14: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                15: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                16: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                17: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                              else:
                            endcase
                        endfor
                    endfor
                endfor
             endif else begin
                for j=0L,n2th-1L do begin
                    for i=0L,nen-1L do begin
                        if ~finite(macs_int[i,j]) then continue
                        case evaxs of  ;could be slow, bearable, to save code space 
                             0: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],macs_int[i,j],macs_err[i,j],format=fmt
                             1: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             2: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             3: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             4: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             5: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             6: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             7: printf,unit,(*self.projStrPtr).Q[qid,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            10: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],macs_int[i,j],macs_err[i,j],format=fmt
                            11: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            12: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            13: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            14: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            15: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            16: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            17: printf,unit,(*self.projStrPtr).Q[qid,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt
                          else: 
                        endcase
                    endfor
                endfor
             endelse
          endif else if self.instrname eq 'dcs' then begin
             for k=0L,nfile-1L do begin
                 for j=0L,n2th-1L do begin
                     for i=0L,nen-1L do begin                 
                         if ~finite((*self.projStrPtr).qty[i,j,k]) then continue
                         case evaxs of 
                              0: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              1: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              2: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              3: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                             10: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).y[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                             11: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).y[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                             12: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).y[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                             13: printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).y[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                           else:
                         endcase
                     endfor
                 endfor
             endfor
          endif else begin
             fmt = '('+dm_to_string(4+ok_y,/int)+'g)'
             if ok_y then begin
                for j=0L,n2th-1 do begin
                    for i=0L,nen-1L do begin
                        if ~finite((*self.projStrPtr).qty[i,j]) then continue
                        printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).y[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                    endfor
                endfor
             endif else begin
                for j=0L,n2th-1 do begin
                    for i=0L,nen-1L do begin                
                        if ~finite((*self.projStrPtr).qty[i,j]) then continue
                        printf,unit,(*self.projStrPtr).Q[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                    endfor
                endfor
             endelse
          endelse
          end
       1: begin
          if self.instrname eq 'macs' then begin
             if ok_ht then begin  ;histogram data
                for j=0L,n2th-1L do begin
                    for i=0L,nen-1L do begin
                        for k=0L,ntchan-1L do begin
                            if ~finite(macs_int[k,i,j]) then continue
                            case evaxs of
                                 0: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 1: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 2: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 3: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 4: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 5: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 6: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                 7: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                10: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                11: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                12: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                13: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                14: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                15: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                16: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                                17: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[k,i,j],a4[k,i,j],(*self.projStrPtr).en[i],time[k,i,j],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[k,i,j],macs_err[k,i,j],weight[k,i,j],format=fmt
                              else:
                            endcase
                        endfor
                    endfor
                endfor  
             endif else begin
                for j=0L,n2th-1 do begin
                    for i=0L,nen-1 do begin
                        if ~finite(macs_int[i,j]) then continue
                        case evaxs of 
                             0: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],macs_int[i,j],macs_err[i,j],format=fmt
                             1: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             2: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             3: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             4: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             5: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             6: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                             7: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            10: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],macs_int[i,j],macs_err[i,j],format=fmt
                            11: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            12: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            13: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            14: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            15: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            16: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                            17: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],a3[i,j],a4[i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[i],(*self.projStrPtr).hfield[i],(*self.projStrPtr).specify[i],macs_int[i,j],macs_err[i,j],format=fmt 
                          else:
                        endcase
                    endfor
                endfor
             endelse
          endif else if self.instrname eq 'dcs' then begin
             for k=0L,nfile-1L do begin
                 for j=0L,n2th-1L do begin
                     for i=0L,nen-1L do begin
                         if ~finite((*self.projStrPtr).qty[i,j,k]) then continue
                         case evaxs of 
                              0: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              1: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              2: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              3: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              4: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              5: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              6: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              7: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt 
                           else:
                         endcase 
                     endfor
                 endfor    
             endfor     
          endif else begin
             if self.extravaxis_yn[0] then begin
                fmt = '(6g)'
                for j=0L,n2th-1 do begin
                    for i=0L,nen-1 do begin
                        if ~finite((*self.projStrPtr).qty[i,j]) then continue
                        printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                    endfor
                endfor
             endif else begin
                fmt = '(5g)'
                for j=0L,n2th-1 do begin
                    for i=0L,nen-1 do begin
                        if ~finite((*self.projStrPtr).qty[i,j]) then continue
                        printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                    endfor
                endfor
             endelse
          endelse
          end
       2: begin
          if self.instrname eq 'dcs' then begin
             for j=0L,nfile-1L do begin
                 for i=0L,n2th-1L do begin
                     if ~finite((*self.projStrPtr).qty[i,j]) then continue
                     case evaxs of
                          0: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          1: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).temperature[self.macstemp_typ,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          2: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).hfield[j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          3: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).temperature[self.macstemp_typ,j],(*self.projStrPtr).hfield[j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          4: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          5: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).temperature[self.macstemp_typ,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          6: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).hfield[j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                          7: printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).temperature[self.macstemp_typ,j],(*self.projStrPtr).hfield[j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                       else:
                     endcase
                 endfor
             endfor             
          endif else begin            
             if self.extravaxis_yn[0] then begin
                fmt = '(5g)'
                if self.instrname eq 'wand' then begin
                   for j=0L,nfile-1 do begin
                       for i=0L,n2th-1 do begin
                           if ~finite((*self.projStrPtr).qty[i,j]) then continue
                           printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).temperature[j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                       endfor
                   endfor
                endif else begin
                   for j=0L,nfile-1 do begin
                       for i=0L,n2th-1 do begin
                           if ~finite((*self.projStrPtr).qty[i,j]) then continue
                           printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).V[2,i,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                       endfor
                   endfor
                endelse
             endif else begin
                fmt = '(4g)'
                for j=0L,nfile-1 do begin
                    for i=0L,n2th-1 do begin
                        if ~finite((*self.projStrPtr).qty[i,j]) then continue
                        printf,unit,(*self.projStrPtr).V[0,i,j],(*self.projStrPtr).V[1,i,j],(*self.projStrPtr).qty[i,j],(*self.projStrPtr).err[i,j],format=fmt
                    endfor
                endfor
             endelse
          endelse
          end
       3: begin
          if self.instrname eq 'dcs' then begin
             for k=0L,nfile-1L do begin
                 for j=0L,n2th-1L do begin
                     for i=0L,nen-1L do begin
                         if ~finite((*self.projStrPtr).qty[i,j,k]) then continue
                         case evaxs of
                              0: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              1: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              2: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              3: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              4: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).V[2,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              5: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).V[2,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              6: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).V[2,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                              7: printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).V[2,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).temperature[self.macstemp_typ,k],(*self.projStrPtr).hfield[k],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                           else:
                         endcase
                     endfor
                 endfor
             endfor          
          endif else begin            
             if self.extravaxis_yn[0] then begin
                fmt = '(6g)'
                for k=0L,nfile-1 do begin
                    for j=0L,n2th-1 do begin
                        for i=0L,nen-1 do begin
                            if ~finite((*self.projStrPtr).qty[i,j,k]) then continue
                            printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).V[2,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                        endfor
                    endfor
                endfor
             endif else begin
                fmt = '(5g)'
                for k=0L,nfile-1 do begin
                    for j=0L,n2th-1 do begin
                        for i=0L,nen-1 do begin
                            if ~finite((*self.projStrPtr).qty[i,j,k]) then continue
                            printf,unit,(*self.projStrPtr).V[0,i,j,k],(*self.projStrPtr).V[1,i,j,k],(*self.projStrPtr).en[i],(*self.projStrPtr).qty[i,j,k],(*self.projStrPtr).err[i,j,k],format=fmt
                        endfor
                    endfor
                endfor
             endelse
          endelse
          end   
    endcase   
    
    free_lun,unit
end

;save slice in files
;keywords:
;   dave:   if set, save slice as grouped DAVE file, for PAN fitting
;   dPtr:   if present, will pass the slice data as a dave pointer
;   iexy:   if set, save slice as Int, Err, X, Y 4-column file
;   psutxt: for PSU, see Craig Brown's 08/03/2012 email
;   spe:    if set, save slice as spe file
;   xyie:   if set, save slice as X, Y, Int, Err matrix file
;   file:   full filename, including the directory, if absent, a dialog window will show up
;   smooth: if present, smooth data according to the smooth value
;   filler: if present, used as the filler value for empty data, useful in script mode
pro dcs_mslice::save_slice,dave=dave,dPtr=dPtr,file=file,filler=filler,iexy=iexy,psutxt=psutxt,spe=spe,xyie=xyie,smooth=smooth,error=error
    if (~ self->pd_valid()) then return
    ext = ''
    if keyword_set(dave)        then ext = 'dave' $
    else if keyword_set(iexy)   then ext = 'iexy' $
    else if keyword_set(spe)    then ext = 'spe'  $
    else if keyword_set(xyie)   then ext = 'xyie' $
    else if keyword_set(psutxt) then ext = 'txt'  $
    else if ~arg_present(dPtr)  then message,'A file type keyword needs to be set.'

    dcs_filler = -1e20
    if finite(self.mask_filler[0]) and (self.samp_typ ne -1) then dcs_filler = self.mask_filler[0]

    if (~ arg_present(dPtr)) then begin
       if strlen(ext) eq 0 then return
       path = self.dirs[1]
       if n_elements(file) eq 0 then begin
          file = dm_choose_file(ext,/write,dialog_parent=self.tlb,title='Save slice as '+ext+' file.',path=path)
          file = file[0]
       endif
       if strlen(file) eq 0 then return
       self->setproperty,workDir=path
       if (~ keyword_set(spe)) then begin
          openw,unit,file,/get_lun,error=openerr
          if openerr ne 0 then begin
             ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
             return
          endif
       endif
    endif

    ;select data
    tmp = self.slice_smooth
    if finite(self.slice_xstep,/nan) and finite(self.slice_xstep,/nan) and (self.allow_voronoi) then smooth = 0 ;no smoothing
    if n_elements(smooth) eq 0 then begin
       if self.slice_smooth gt 0 then begin
          info = 'Do you want to '+(['save','send'])[arg_present(dPtr)]+' the smoothed data?'
          ok = dialog_message(info,/question,dialog_parent=self.tlb,/center)
          if strlowcase(ok) eq 'no' then self.slice_smooth = 0
       endif
    endif else self.slice_smooth = smooth
    if (self.samp_typ ge 0) and self.absp_asint then $
       self->dm_choose_slicedata,xdat=xdat,ydat=ydat,a3=a3,a4=a4,zdat=zdat,xtit=xtit,ytit=ytit,ztit=ztit,zmin=zmin,weight=weight,error=error $
    else $
       self->dm_choose_slicedata,xdat=xdat,ydat=ydat,a3=a3,a4=a4,zdat=zdat,zerr=zerr,xtit=xtit,ytit=ytit,ztit=ztit,zmin=zmin,weight=weight,error=error
    if keyword_set(error) then begin
       if n_elements(unit) ne 0 then free_lun,unit 
       return
    endif
    if n_elements(zerr) eq 0 then zerr = fltarr(size(zdat,/dimensions))
    self.slice_smooth = tmp
    ;need filler for empty grid
    index = where(finite(zdat,/nan),count)
    if count ne 0 then begin
       if n_elements(filler) ne 0 then begin
          dcs_filler = filler[0]
          notok = 0b
       endif else notok = 1b
       while(notok) do begin
          dcs_filler = dm_dialog_input('filler value:',/float,default=dcs_filler,info='This value is used for empty grids that contain no data.',dialog_parent=self.tlb,cancel=error)
          if keyword_set(error) then return
          notok = finite(dcs_filler,/nan)
       endwhile
       if self.samp_typ ne -1 then self.mask_filler[0] = dcs_filler
       zdat[index] = dcs_filler
       zerr[index] = -1.0               ;signifies missing data
       if n_elements(a4) eq n_elements(zdat) then a4[index] = dcs_filler
    endif

    widget_control,/hourglass

    if keyword_set(iexy)   then self->write_iexy,unit,xdat,ydat,zdat,zerr,a3=a3,a4=a4,xtit=xtit,ytit=ytit,ztit=ztit,weight=weight
    if keyword_set(spe)    then dm_write_spe,file,xdat,ydat,zdat,zerr,xtit=xtit,ytit=ytit,ztit=ztit,group_leader=self.tlb
    if keyword_set(xyie)   then self->write_xyie,unit,xdat,ydat,zdat,zerr,a3=a3,a4=a4
    if keyword_set(psutxt) then self->write_psutxt,unit,xdat,ydat,zdat,zerr
    if keyword_set(dave) or arg_present(dPtr) then begin
       nx = n_elements(xdat) & ny = n_elements(ydat)
       if count ne 0 then begin ;creat mask
          mask = bytarr(nx,ny)+1b
          mask[index] = 0b
          mask = bits2bytes(transpose(mask))
       endif
       ;contruct DAVE pointer
       xunit = ''
       yunit = ''
       if self.samp_typ eq 0 then begin
          tmp = stregex(xtit,'^[ '+string(9b)+']*(.+)[ '+string(9b)+']*\((.+)\)[ '+string(9b)+']*$',/subexpr,/extract)
          if strlen(tmp[2]) gt 0 then begin
             xtit  = tmp[1]
             xunit = tmp[2]
             pos   = strpos(xunit,'\AA') ;replace \AA with string('c5'XB)
             if pos ne -1 then xunit = strmid(xunit,0,pos)+string('c5'XB)+strmid(xunit,pos+3)
          endif
          tmp = stregex(ytit,'^[ '+string(9b)+']*(.+)[ '+string(9b)+']*\((.+)\)[ '+string(9b)+']*$',/subexpr,/extract)
          if strlen(tmp[2]) gt 0 then begin
             ytit  = tmp[1]
             yunit = tmp[2]
             pos   = strpos(yunit,'\AA') ;replace \AA with string('c5'XB)
             if pos ne -1 then yunit = strmid(yunit,0,pos)+string('c5'XB)+strmid(yunit,pos+3)
          endif
       endif
       dPtr = dm_define_pointer()
       ptr_free,(*(*dPtr).dataStrPtr).commonStr.histPtr
       (*(*dPtr).dataStrPtr).commonStr.histPtr   = ptr_new({qty:transpose(temporary(zdat)),err:transpose(temporary(abs(zerr))),x:ydat,y:xdat})
       (*(*dPtr).dataStrPtr).commonStr.xunits    = yunit
       (*(*dPtr).dataStrPtr).commonStr.xlabel    = ytit
       (*(*dPtr).dataStrPtr).commonStr.yunits    = xunit
       (*(*dPtr).dataStrPtr).commonStr.ylabel    = xtit
       (*(*dPtr).dataStrPtr).commonStr.histlabel = ztit
       info = 'Slice from Mslice'+((self.samp_typ lt 0)?'':(' for '+(*self.ftypename)[self.ftype]))
       if self.samp_typ ge 0 then info = [info,'data file name: '+((n_elements(*self.file_recent) eq 2)?(*self.file_recent)[1]:(*self.file_recent)[1]+' to '+(*self.file_recent)[-1]),$
                                          'data directory: '+(*self.file_recent)[0]]
       *(*(*dPtr).dataStrPtr).commonStr.treatmentPtr = info
       if count ne 0 then begin ;mask
          ;add PAN_MASK and DCS_FILLER info to specificPtr
          specific = (*(*(*dPtr).dataStrPtr).specificPtr)
          tags = strupcase(tag_names(specific))
          loc = where(tags eq 'PAN_MASK',count)
          if count eq 0 then specific = create_struct('PAN_MASK',ptr_new(mask),specific) $
          else *specific.pan_mask = mask
          loc = where(tags eq 'DCS_FILLER',count)
          if count eq 0 then specific = create_struct('DCS_FILLER',dcs_filler,specific) $
          else specific.dcs_filler = dcs_filler
          (*(*(*dPtr).dataStrPtr).specificPtr) = specific
       endif
       if n_elements(unit) ne 0 then begin
          davePtr = dPtr
          save,davePtr,filename=file,/compress
          heap_free,dPtr
       endif
    endif

    if n_elements(unit) ne 0 then free_lun,unit
end

;save the volume data
;keywords:
;   slice:  if present,save slices(0: x slice, 1: y slice, 2: z slice), otherwise save all volume data in one text file
;   file:   full filename, including the directory, if absent, a dialog window will show up  
;   filler: if present, used as the filler value for empty data, useful in script mode 
;   iexy:   if set, save slice as Int, Err, X, Y 4-column file
;   psutxt: for PSU, see Craig Brown's 08/03/2012 email
;   spe:    if set, save slice as spe file
;   xyie:   if set, save slice as X, Y, Int, Err matrix file
;   smooth: if present, smooth data according to the smooth value 
pro dcs_mslice::save_volume,slice=slice,file=file,filler=filler,iexy=iexy,psutxt=psutxt,spe=spe,xyie=xyie,smooth=smooth,error=error
    ext = 'multi-column text'
    if keyword_set(iexy)   then ext = 'iexy'
    if keyword_set(spe)    then ext = 'spe'
    if keyword_set(xyie)   then ext = 'xyie'
    if keyword_set(psutxt) then ext = 'txt'
    if n_elements(slice) ne 0 then begin
       if (slice[0] lt 0) or (slice[0] gt 2) then return
       if ext eq 'multi-column text' then begin
          spe = 1b
          ext = 'spe'   ;default spe file type for slices
       endif
    endif
     
    path = self.dirs[1]
    if n_elements(file) eq 0 then begin
       file = dm_choose_file((ext eq 'multi-column text')?'txt':ext,/write,dialog_parent=self.tlb,title='Save volume data as '+ext+'  file.',path=path)
       file = file[0]
    endif
    if strlen(file) eq 0 then return
    self->setproperty,workDir=path
    
    dcs_filler = -1e20
    if finite(self.mask_filler[0]) and (self.samp_typ ne -1) then dcs_filler = self.mask_filler[0] 
    
    ;select data
    tmp = self.vol_smooth
    if n_elements(smooth) eq 0 then begin
       if self.vol_smooth gt 0 then begin
          info = 'Do you want to save the smoothed data?'
          ok = dialog_message(info,/question,dialog_parent=self.tlb,/center)
          if strlowcase(ok) eq 'no' then self.vol_smooth = 0
       endif  
    endif else self.vol_smooth = smooth
    
    if (self.samp_typ ge 0) and self.absp_asint then $
       self->dm_choose_volumedata,xdat=xdat,ydat=ydat,zdat=zdat,ierr=ierr,xtit=xtit,ytit=ytit,ztit=ztit,itit=itit,weight=weight,error=error $
    else $
       self->dm_choose_volumedata,xdat=xdat,ydat=ydat,zdat=zdat,idat=idat,ierr=ierr,xtit=xtit,ytit=ytit,ztit=ztit,itit=itit,weight=weight,error=error
    if keyword_set(error) then return
    if n_elements(idat) eq 0 then idat = fltarr(size(zdat,/dimensions))

    self.vol_smooth = tmp
    if (n_elements(slice) ne 0) and ~keyword_set(iexy) then begin
       itmp = where(finite(idat,/nan),count)
       if count ne 0 then begin
          if n_elements(filler) ne 0 then begin
             dcs_filler = filler[0]
             notok = 0b
          endif else notok = 1b
          while(notok) do begin
             dcs_filler = dm_dialog_input('filler value:',/float,default=dcs_filler,info='This value is used for empty grids that contain no data.',dialog_parent=self.tlb)
             notok = finite(dcs_filler,/nan)
          endwhile
          if self.samp_typ ne -1 then self.mask_filler[0] = dcs_filler
          idat[itmp] = dcs_filler
          ierr[itmp] = -1 
          itmp = -1
       endif
    endif
    widget_control,/hourglass  ;busy signal
    
    if n_elements(slice) eq 0 then begin  ;single text file
       openw,unit,file,/get_lun,error=openerr
       if openerr ne 0 then begin
          ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
          return
       endif
       ok_wght = (n_elements(weight) eq n_elements(idat))
       header = dm_to_string(xtit,data=xdat[0],over=over,/addsemicolon)
       header = header+dm_to_string(ytit,data=ydat[0],over=over)
       header = header+dm_to_string(ztit,data=zdat[0],over=over)
       header = header+dm_to_string([itit,'dI'],data=idat[0],over=over)
       if ok_wght then header = header+dm_to_string('weight',data=weight[0],over=over)
       fmt = '('+dm_to_string(5+ok_wght,/int)+'g)'
       printf,unit,header
       if ok_wght then begin
          for k=0L,n_elements(zdat)-1 do begin
              for j=0L,n_elements(ydat)-1 do begin
                  for i=0L,n_elements(xdat)-1 do begin
                      if finite(idat[i,j,k]) then printf,unit,xdat[i],ydat[j],zdat[k],idat[i,j,k],ierr[i,j,k],weight[i,j,k],format=fmt
                  endfor
              endfor
          endfor
       endif else begin
          for k=0L,n_elements(zdat)-1 do begin
              for j=0L,n_elements(ydat)-1 do begin
                  for i=0L,n_elements(xdat)-1 do begin
                      if finite(idat[i,j,k]) then printf,unit,xdat[i],ydat[j],zdat[k],idat[i,j,k],ierr[i,j,k],format=fmt
                  endfor
              endfor    
          endfor
       endelse
       free_lun,unit
    endif else begin
       tmpfile = file_basename(file,'.'+ext,/fold_case)
       nxyz = size(idat,/dim)
       ok = dialog_message(['The volume slices will be saved as '+dm_to_string(nxyz[slice])+' '+tmpfile+'_'+(['x','y','z'])[slice]+'=####.'+ext+' files.',$
               'Do you want to continue?'],/question,dialog_parent=self.tlb,/center,title='Please confirm')
       if ok eq 'No' then return
       file = file_dirname(file)+self.pathsep+tmpfile
       case slice of 
         0: begin
            xx = ydat
            yy = zdat      
            xtit = ytit & ytit = ztit
            end
         1: begin
            xx = xdat
            yy = zdat
            ytit = ztit
            end
         2: begin
            xx = xdat
            yy = ydat    
            end
       endcase 
       for i=0L,nxyz[slice]-1L do begin
           case slice of 
             0: begin
                zz = reform(idat[i,*,*])
                ee = reform(ierr[i,*,*])
                tt = xdat[i]      
                end
             1: begin
                zz = reform(idat[*,i,*])
                ee = reform(ierr[*,i,*])    
                tt = ydat[i]      
                end
             2: begin
                zz = reform(idat[*,*,i])
                ee = reform(ierr[*,*,i]) 
                tt = zdat[i]         
                end
           endcase 
           tmpfile = file+'_'+(['x','y','z'])[slice]+'='+dm_to_string(tt,resolution=4)+'.'+ext
           if (~keyword_set(spe)) then begin
              openw,unit,tmpfile,/get_lun,error=openerr
              if openerr ne 0 then begin
                 ok = dialog_message("Can't write in "+file,/error,dialog_parent=self.tlb,/center)
                 return
              endif
           endif
           if keyword_set(iexy)   then self->write_iexy,unit,xx,yy,zz,ee
           if keyword_set(spe)    then begin
              dm_write_spe,tmpfile,xx,yy,zz,ee,xtit=xtit,ytit=ytit,ztit=itit,group_leader=self.tlb,error=error
              if error ne 0 then return
           endif
           if keyword_set(xyie)   then self->write_xyie,unit,xx,yy,zz,ee
           if keyword_set(psutxt) then self->write_psutxt,unit,xx,yy,zzt,ee
           if n_elements(unit) ne 0 then free_lun,unit
       endfor
    endelse
end

;script commands
; parameter: 
;     command:   a string or string array of commands. Accepted commands are:
;                'load_data' :     filename, directory, and eief keywords need to be present
;                'load_emptycan':  filename and directory keywords need to be present
;                'load_parameter': filename and directory keywords need to be present
;                'calc_projection':temperature keywords need to be present if used, dwu2 keywords are optional
;                'save_cut':       filename and directory keyword need to be present, extra keywords are xye=xye,dave=dave
;                'save_slice':     filename, directory keywords need to be present, smooth and filler keywords are optional,
;                                  extra keywords are dave=dave,iexy=iexy,spe=spe,xyie=xyie,psutxt=psutxt
;                'save_volume':    filename, directory keywords need to be present, smooth and filler keywords are optional
;                                  extra keywords are slice=slice,iexy=iexy,spe=spe,xyie=xyie,psutxt=psutxt
;                'set_parameter':  name and value keywords need to be present
;                'keep_current_plot': keep the current plot window, new plots will be in a new plot window
;                'plot_cut':       plotting cut, legend and title keywords can be used
;                'plot_slice':     plotting slice, title keyword can be used
;                'plot_volume':    plotting volume, title keyword can be used
; keyword:
;     angle:     a float array or file name containing the sample rotation angle for SPE files, for load_emptycan and load_data events
;     directory: a string
;     dwu2:      u^2 value in the Debye-Waller factor correction,default=0
;     filename:  a string or string array of file names, excluding directory
;     eief:      efixed for loading files that don't have the eief information
;     filler:    the filler value for empty grid data, default=-1e20
;     legend:    a string for the plot legend
;     name:      name of the wid to be set_parameter
;     overplot:  overplot flag of 'plot_cut' command
;     phxfile:   a string of of phxfile name, excluding directory, for loading spe files
;     resetpsioffset: keyword passed to event for resetting self.orie_offset
;     resolution:used when command is 'set_parameter', in dm_to_string()
;     smooth:    smooth level for saving slice data and volume data, default is 0
;     surfplot:  surface plot flag for 'plot_slice' command
;     temperature: sample temperature for intensity types that requires a temperature in projection calculation
;     title:     a string for the plot title
;     value:     vlaue of the set_parameter
pro dcs_mslice::script,command,angle=angle,directory=directory,dwu2=dwu2,eief=eief,filename=filename,filler=filler,legend=legend,name=name,overplot=overplot,$
    phxfile=phxfile,resetpsioffset=resetpsioffset,resolution=resolution,smooth=smooth,surfplot=surfplot,temperature=temperature,title=title,value=value,_ref_extra=_extra   
    if n_elements(smooth) eq 0 then smooth = 0
    if n_elements(filler) eq 0 then filler = -1e20
    if n_elements(dwu2)   eq 0 then dwu2   = 0.0
    if n_elements(title) ne 0 then begin
       self.titl_lab = strtrim(title[0],2)
       self->my_widget_control,'titlLab',set_value=self.titl_lab
    endif
    if n_elements(legend) ne 0 then begin
       self.legd_lab = strtrim(legend[0],2)
       self->my_widget_control,'legdLab',set_value=self.legd_lab
    endif
    if keyword_set(overplot) then begin ;shift the color to the next one in the list
       self.cut_color = (self.cut_color+1) mod (n_elements(*self.colors))
       if self.view[1] or (self.samp_typ eq -1) then dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutColor'),select=self.cut_color
    endif
    for ii=0,n_elements(command)-1 do begin
        case strlowcase(command[ii]) of
          'load_data':       ok = self->dm_filetools('loadbut',filename=filename,directory=directory,angle=angle,eief=eief,phxfile=phxfile,/scriptmode)
          'add_data':        ok = self->dm_filetools('addbut',filename=filename,directory=directory,angle=angle,eief=eief,phxfile=phxfile,/scriptmode)
          'load_emptycan':   ok = self->dm_filetools('emptbut',filename=filename,directory=directory,angle=angle,/scriptmode)
          'load_emptycan2':  ok = self->dm_filetools('emptbut2',filename=filename,directory=directory,angle=angle,/scriptmode)
          'load_emptycan3':  ok = self->dm_filetools('emptbut3',filename=filename,directory=directory,angle=angle,/scriptmode)
          'load_parameter': begin
              if n_elements(filename)  eq 0 then break
              if n_elements(directory) eq 0 then directory=self.dirs[1]
              self->dm_load_parameter,directory[0]+self.pathsep+filename[0]
              end
          'calc_projection': self->dm_calc_projection,temperature=temperature,dwu2=dwu2
          'save_cut': begin
              if n_elements(filename)  eq 0 then break
              if n_elements(directory) eq 0 then directory=self.dirs[1]
              self->save_cut,file=directory[0]+self.pathsep+filename[0],_extra=_extra
              end
          'save_slice': begin
              if n_elements(filename)  eq 0 then break
              if n_elements(directory) eq 0 then directory=self.dirs[1]
              self->save_slice,file=directory[0]+self.pathsep+filename[0],smooth=smooth,filler=filler,_extra=_extra
              end
          'save_volume': begin
              if n_elements(filename)  eq 0 then break
              if n_elements(directory) eq 0 then directory=self.dirs[1]
              self->save_volume,file=directory[0]+self.pathsep+filename[0],smooth=smooth,filler=filler,_extra=_extra
              end
          'save_projection': begin
                if n_elements(filename)  eq 0 then break
                if n_elements(directory) eq 0 then directory=self.dirs[1]
                self->save_projection,file=directory[0]+self.pathsep+filename[0]
              end    
          'set_parameter':begin
              n_name = n_elements(name) & n_value = n_elements(value)
              if (n_name eq 0) or (n_value eq 0) then break
              txtevent = {widget_text_ch} & txtevent.top = self.tlb & txtevent.handler = self.tlb
              ;recognized unames
              ft = ['from','to'] & fts = [ft,'step']
              unames = ['intn','titl','legd']+'Lab' 
              params = ['intn','titl','legd']+'_lab'
              n_fbas = n_elements(unames)                                   ;in self.fileBas
              unames = [unames,'latt'+['A','B','C','AA','BB','CC'],'orie'+['U'+['h','k','l'],'V'+['h','k','l'],'Psi'],$
                        'viewU'+['1'+['h','k','l','lab'],'2'+['h','k','l','lab'],'3'+['h','k','l','lab']]]
              params = [params,'latt_parm['+['0','1','2','3','4','5']+']','orie_'+['u['+['0','1','2'],'v['+['0','1','2']]+']','orie_psi',$
                        'view_u'+['1'+['[0]','[1]','[2]','lab[(self.samp_typ ge 1)]'],'2'+['[0]','[1]','[2]','lab[(self.samp_typ ge 1)]'],'3'+['[0]','[1]','[2]','lab']]]
              n_pbas = n_elements(unames)                                   ;in self.projBas
              unames = [unames,'dispsel'+ft+'_T0','dispsel'+ft+'_T1','dispx'+fts,'dispy'+fts,'dispI'+ft,'dispSmooth','cut'+['From','To','Step'],'cutsel'+ft+'_T0','cutsel'+ft+'_T1',$
                        'cutsel'+ft+'_T2','cutY'+ft,'volx'+fts,'voly'+fts,'volz'+fts,'volsel'+ft+'_T0','volI'+ft,'volSmooth','d4x1'+fts,'d4x2'+fts,'d4x3'+fts,'d4x4'+fts] 
              params = [params,'slice_t1ran['+['0','1']+']','slice_t2ran['+['0','1']+']','slice_xran['+['0','1']+']','slice_xstep','slice_yran['+['0','1']+']','slice_ystep',$
                        'slice_iran['+['0','1']+']','slice_smooth','cut_ran['+['0','1']+']','cut_step','cut_t1ran['+['0','1']+']','cut_t2ran['+['0','1']+']','cut_t3ran['+['0','1']+']',$
                        'cut_yran['+['0','1']+']','vol_xran['+['0','1']+']','vol_xstep','vol_yran['+['0','1']+']','vol_ystep','vol_zran['+['0','1']+']','vol_zstep','volsel_tft1['+['0','1']+']',$
                        'vol_iran['+['0','1']+']','vol_smooth','d4_x1ran['+['0','1']+']','d4_x1step','d4_x2ran['+['0','1']+']','d4_x2step','d4_x3ran['+['0','1']+']','d4_x3step',$
                        'd4_x4ran['+['0','1']+']','d4_x4step']
              n_dbas = n_elements(unames)                                   ;in self.dispBas
              unamel = strlowcase(unames) ;lower case version
              bvalid = widget_info([self.fileBas,self.projBas,self.dispBas],/valid_id)
              for i=0L,n_name-1 do begin
                  ind = where(unamel eq strlowcase(name[i]),count)
                  if count eq 0 then continue
                  tmpvalue = value[i<(n_value-1)]
                  txtevent.id = 0
                  if (ind lt n_fbas) and bvalid[0] then begin
                     txtevent.id = widget_info(self.fileBas,find_by_uname=unames[ind[0]])
                  endif else if (ind lt n_pbas) and bvalid[1] then begin
                     txtevent.id = widget_info(self.projBas,find_by_uname=unames[ind[0]])
                  endif else if bvalid[2] then begin
                     txtevent.id = widget_info(self.dispBas,find_by_uname=unames[ind[0]])
                  endif
                  if txtevent.id ne 0 then begin  ;prefered
                     widget_control,txtevent.id,set_value=dm_to_string(tmpvalue,resolution=resolution),_extra=_extra
                     self->event,txtevent,resetpsioffset=resetpsioffset
                  endif else begin  ;not desired
                     ok = execute('self.'+params[ind[0]]+'=tmpvalue',1,1)
                     if ~ok then ok = dialog_message(!error_state.msg,/error,dialog_parent=self.tlb,/center)
                  endelse
              endfor
              end
          'keep_current_plot': if obj_valid(self.current) then begin
              self.current->setproperty,/keep
              self->dm_add_plotwindow,2,object=self.current
              end    
          'plot_cut':        self->dm_plot_cut,overplot=overplot
          'plot_slice':      self->dm_plot_slice,surfplot=surfplot
          'plot_volume':     self->dm_plot_volume
          else:
        endcase
    endfor
end

;return the first available current plot from all mslices
function dcs_mslice::searchcurrent,selfonly=selfonly,crossplot=crossplot,keep=keep,ylog=ylog,group_leader=group_leader
    current = self.current
    if obj_valid(current) then current->getproperty,keep=keep,ylog=ylog,group_leader=group_leader else keep = 1b
    if (~keyword_set(selfonly)) and self.crossplot and keep then begin
       tmp = obj_valid()
       ind = where(obj_isa(tmp,'dcs_mslice'),count)
       for i=0,count-1 do begin
           current = tmp[ind[i]]->searchcurrent(/selfonly,keep=keep,group_leader=group_leader)
           if ~keep then break
       endfor
    endif
    if obj_valid(current) then crossplot = (group_leader ne self.tlb)
    return,current
end

;select a droplist axis in display
pro dcs_mslice::select_droplist,id,value,d4=d4,vol=vol,slice=slice,cut=cut,traj=traj
    if self.samp_typ lt 0 then nitem = n_elements((*self.dataPtr).label)
    fts = ['from','to','step']
    if keyword_set(d4) then begin
       nams = 'd4selX'+['1','2','3','4']
       wids = [widget_info(self.dispBas,find_by_uname=nams[0]),widget_info(self.dispBas,find_by_uname=nams[1]),$
               widget_info(self.dispBas,find_by_uname=nams[2]),widget_info(self.dispBas,find_by_uname=nams[3])]
       indx = (where(wids eq id,count))[0]
       if count eq 0 then return
       d4sel = reform(([[self.d4sel_x1],[self.d4sel_x2],[self.d4sel_x3],[self.d4sel_x4]])[0>(self.samp_typ),*])
       d4ran = [[self.d4_x1ran],[self.d4_x2ran],[self.d4_x3ran],[self.d4_x4ran]]
       d4stp = [self.d4_x1step,self.d4_x2step,self.d4_x3step,self.d4_x4step]
       old   = d4sel[indx]
       case indx of
            0 : self.d4sel_x1[0>(self.samp_typ)] = value
            1 : self.d4sel_x2[0>(self.samp_typ)] = value
            2 : self.d4sel_x3[0>(self.samp_typ)] = value
            3 : self.d4sel_x4[0>(self.samp_typ)] = value
       endcase
       ;swap the other axes that has the same axis number
       nams = 'd4x'+['1','2','3','4']
       for i=indx+1,indx+3 do begin
           j = (i mod 4)
           if d4sel[j] eq value then begin
              case j of
                 0 : self.d4sel_x1[0>(self.samp_typ)] = old
                 1 : self.d4sel_x2[0>(self.samp_typ)] = old
                 2 : self.d4sel_x3[0>(self.samp_typ)] = old
                 3 : self.d4sel_x4[0>(self.samp_typ)] = old
              endcase
              dm_set_droplist,wids[j],select=old
              self->script,'set_parameter',name=[nams[indx]+fts,nams[j]+fts],value=[d4ran[*,j],d4stp[j],d4ran[*,indx],d4stp[indx]]
              swapped = 1b
              break
           endif
       endfor
       if ~keyword_set(swapped) then begin
          if self.samp_typ ge 0 then begin
             if self.disp_flag[4] and (self.d4sel_t1[self.samp_typ] eq value) then begin
                self.d4sel_t1[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4sel_T0'),select=old
             endif else if self.disp_flag[5] and (self.d4sel_t2[self.samp_typ] eq value) then begin
                self.d4sel_t2[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='d4sel_T1'),select=old
             endif
          endif
       endif
    endif else if keyword_set(vol) then begin
       nams = 'volsel'+['X','Y','Z']
       wids = [widget_info(self.dispBas,find_by_uname=nams[0]),widget_info(self.dispBas,find_by_uname=nams[1]),$
               widget_info(self.dispBas,find_by_uname=nams[2])]
       indx = (where(wids eq id,count))[0]
       if count eq 0 then return
       volsel = reform(([[self.volsel_x],[self.volsel_y],[self.volsel_z]])[0>(self.samp_typ),*])
       volran = [[self.vol_xran],[self.vol_yran],[self.vol_zran]]
       volstp = [self.vol_xstep,self.vol_ystep,self.vol_zstep]
       old    = volsel[indx]
       case indx of 
            0: self.volsel_x[0>(self.samp_typ)] = value
            1: self.volsel_y[0>(self.samp_typ)] = value
            2: self.volsel_z[0>(self.samp_typ)] = value
       endcase
       ;swap the other axis that has the same axis number
       nams = 'vol'+['x','y','z']
       for i=indx+1,indx+2 do begin
           j = (i mod 3)
           if volsel[j] eq value then begin
              case j of
                   0 : self.volsel_x[0>(self.samp_typ)] = old
                   1 : self.volsel_y[0>(self.samp_typ)] = old
                   2 : self.volsel_z[0>(self.samp_typ)] = old
              endcase
              dm_set_droplist,wids[j],select=old
              self->script,'set_parameter',name=[nams[indx]+fts,nams[j]+fts],value=[volran[*,j],volstp[j],volran[*,indx],volstp[indx]]
              swapped = 1b
              break
           endif
       endfor
       if ~keyword_set(swapped) then begin
          if self.samp_typ ge 0 then begin
             if self.disp_flag[3] and (self.volsel_t1[self.samp_typ] eq value) then begin
                self.volsel_t1[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T0'),select=old
             endif else if self.disp_flag[4] and (self.volsel_t2[self.samp_typ] eq value) then begin
                self.volsel_t2[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T1'),select=old
             endif else if self.disp_flag[5] and (self.volsel_t3[self.samp_typ] eq value) then begin
                self.volsel_t3[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T2'),select=old
             endif
          endif else if nitem ge 6 then begin
             ind = where((*self.volsel_t) eq value,count)
             for i=0,count-1 do begin
                 (*self.volsel_t)[ind[i]] = old
                 dm_set_droplist,widget_info(self.dispBas,find_by_uname='volsel_T'+dm_to_string(ind[i])),select=old
             endfor
          endif
       endif
    endif else if keyword_set(slice) then begin
       nams = 'slicesel_'+['x','y']
       wids = [widget_info(self.dispBas,find_by_uname=nams[0]),widget_info(self.dispBas,find_by_uname=nams[1])]
       indx = (where(wids eq id,count))[0]
       if count eq 0 then return
       slisel = reform(([[self.slice_x],[self.slice_y]])[0>(self.samp_typ),*])
       sliran = [[self.slice_xran],[self.slice_yran]]
       slistp = [self.slice_xstep,self.slice_ystep]
       old    = slisel[indx]
       case indx of
            0: self.slice_x[0>(self.samp_typ)] = value
            1: self.slice_y[0>(self.samp_typ)] = value
       endcase
       ;swap the other axis that has the same axis number
       nams = 'disp'+['x','y'] & j = ((indx+1) mod 2)
       if slisel[j] eq value then begin
          case j of
               0: self.slice_x[0>(self.samp_typ)] = old
               1: self.slice_y[0>(self.samp_typ)] = old
          endcase
          dm_set_droplist,wids[j],select=old
          self->script,'set_parameter',name=[nams[indx]+fts,nams[j]+fts],value=[sliran[*,j],slistp[j],sliran[*,indx],slistp[indx]]
       endif else begin
          if self.samp_typ ge 0 then begin
             if self.disp_flag[2] and (self.slice_t1[self.samp_typ] eq value) then begin
                self.slice_t1[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T0'),select=old
             endif else if self.disp_flag[3] and (self.slice_t2[self.samp_typ] eq value) then begin
                self.slice_t2[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T1'),select=old
             endif else if self.disp_flag[4] and (self.slice_t3[self.samp_typ] eq value) then begin
                self.slice_t3[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T2'),select=old
             endif else if self.disp_flag[5] and (self.slice_t4[self.samp_typ] eq value) then begin
                self.slice_t4[self.samp_typ] = old
                dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T3'),select=old
             endif
          endif else if nitem ge 5 then begin
             ind = where((*self.slice_t) eq value,count)
             for i=0,count-1 do begin
                 (*self.slice_t)[ind[i]] = old
                 dm_set_droplist,widget_info(self.dispBas,find_by_uname='dispsel_T'+dm_to_string(ind[i])),select=old
             endfor
          endif
       endelse
    endif else if keyword_set(cut) then begin
       old = self.cut_x[0>(self.samp_typ)]
       self.cut_x[0>(self.samp_typ)] = value
       ;swap any thickness axis that has the same axis number
       if self.samp_typ ge 0 then begin
          if self.cut_t1[self.samp_typ] eq value then begin
             self.cut_t1[self.samp_typ] = old
             dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T0'),select=old
          endif else if self.disp_flag[2] and (self.cut_t2[self.samp_typ] eq value) then begin
             self.cut_t2[self.samp_typ] = old
             dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T1'),select=old
          endif else if self.disp_flag[3] and (self.cut_t3[self.samp_typ] eq value) then begin
             self.cut_t3[self.samp_typ] = old
             dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T2'),select=old
          endif else if self.disp_flag[4] and (self.cut_t4[self.samp_typ] eq value) then begin
             self.cut_t4[self.samp_typ] = old
             dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T3'),select=old
          endif else if self.disp_flag[5] and (self.cut_t5[self.samp_typ] eq value) then begin
             self.cut_t5[self.samp_typ] = old
             dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T4'),select=old
          endif
       endif else if nitem ge 4 then begin
          ind = where((*self.cutsel_t) eq value,count)
          for i=0,count-1 do begin
              (*self.cutsel_t)[ind[i]] = old
              dm_set_droplist,widget_info(self.dispBas,find_by_uname='cutsel_T'+dm_to_string(ind[i])),select=old
          endfor
       endif
    endif else if keyword_set(traj) then begin
       nams = 'traj'+['X','Y']
       wids = [widget_info(self.dispBas,find_by_uname=nams[0]),widget_info(self.dispBas,find_by_uname=nams[1])]
       indx = (where(wids eq id,count))[0]
       if count eq 0 then return
       trasel = reform(([[self.traj_x],[self.traj_y]])[self.samp_typ,*])
       old    = trasel[indx]
       case indx of 
            0: self.traj_x[self.samp_typ] = value
            1: self.traj_y[self.samp_typ] = value
       endcase
       ;swap the other axis that has the same axis number
       j = ((indx+1) mod 2)
       if trasel[j] eq value then begin
          case j of
               0: self.traj_x[self.samp_typ] = old
               1: self.traj_y[self.samp_typ] = old
          endcase
          dm_set_droplist,wids[j],select=old
       endif else if self.disp_flag[0] and (self.traj_z1[self.samp_typ] eq value) then begin
          self.traj_z1[self.samp_typ] = old
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajZ1'),select=old
       endif else if self.disp_flag[1] and (self.traj_z2[self.samp_typ] eq value) then begin
          self.traj_z2[self.samp_typ] = old
          dm_set_droplist,widget_info(self.dispBas,find_by_uname='trajZ2'),select=old
       endif
    endif
end

;set property for all plots
pro dcs_mslice::set_allplots,totop=totop,_extra=extra
    oplots = self.gr_list->get(/all,count=count)
    for i=0,count-1 do oplots[i]->setproperty,_extra=extra
    if keyword_set(totop) and obj_valid(self.current) and (count gt 1) then self.current->totop
end

;set the buttons after projection is changed
;keyword:
;   clearprojection: if set, clear self.projStrPtr pointer
pro dcs_mslice::set_display_button,clearprojection=clearprojection
    if keyword_set(clearprojection) then ptr_free,self.projStrPtr
    pd_valid = self->pd_valid(/nomessage)
    self->my_widget_control,'savevolumeMenu',sensitive=(pd_valid and self.disp_flag[2])
    self->my_widget_control,['savesliceMenu','savecutMenu','saveprojMenu'],sensitive=pd_valid
    self->my_widget_control,'viewSpurion',sensitive=((ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and (self.samp_typ lt 2) and (self.instrgeom eq 0)) or (self.instrname eq 'macs'))
    if ~self.view[1] then return
    self->my_widget_control,['dispFitSlice','dispPlotSlice','dispoPltSlice','dispSurfSlice','cutFit','cutPlot','cutoPlot','volPlot','voloPlot','d4Save'],self.dispBas,sensitive=pd_valid
    self->my_widget_control,'plotTra', self.dispBas,sensitive=ptr_valid(self.projStrPtr)
    self->my_widget_control,'oplotTra',self.dispBas,sensitive=(ptr_valid(self.projStrPtr) and (self.samp_typ ne 0))
    if ~widget_info(self.dispBas,/valid_id) then return
    wid = widget_info(self.dispBas,find_by_uname='dispTab')
    if wid eq 0 then return
    if (self.instrname eq 'macs') and (self.samp_typ eq 0) then begin
       if self.disp_flag[2] and ~self.disp_flag[3] then begin
          if self.macsintn_typ ne 0 then self->my_widget_control,['volPlot','voloPlot'],self.dispBas,sensitive=0
       endif else if ~self.disp_flag[2] then begin
          if self.macsintn_typ ne 0 then dfcur = 1
          if pd_valid then begin
             minen = min((*self.projStrPtr).en,max=maxen)
             if minen eq maxen then dfcur = 1
          endif
          if n_elements(dfcur) ne 0 then self->my_widget_control,['dispFitSlice','dispPlotSlice','dispoPltSlice','dispSurfSlice'],self.dispBas,sensitive=0
       endif
    endif 
    if (~ ptr_valid(self.dataStrPtr)) and (ptr_valid(self.projStrPtr)) then begin
       widget_control,wid,set_tab_current=([2,3,4])[self.disp_flag[2]+self.disp_flag[3]]
    endif else begin
       current = widget_info(wid,/tab_current)
       if (current eq ([2,3,4])[self.disp_flag[2]+self.disp_flag[3]]) or (current eq ([3,4,5])[self.disp_flag[2]+self.disp_flag[3]]) and pd_valid then dfcur=0
       if n_elements(dfcur) ne 0 then widget_control,wid,set_tab_current=dfcur
    endelse
end

pro dcs_mslice::set_filehistory,info,dir,files,choose=choose,ftpobj=ftpobj,eventname=eventname,emptset=emptset
    self.fileSel->getproperty,ftpserver=ftpserver
    if n_elements(choose) eq 1 then begin
       ftype = strmid((*(self.file_hist[choose]))[0],6,strlen((*(self.file_hist[choose]))[0])-6)
       if ftype ne self.ftype then ok=self->dm_filetools((*(self.file_hist[choose]))[0])
       self.filesel->set_path,(*(self.file_hist[choose]))[2]
       self.filesel->getproperty,dFiles=dFiles
       for i=3,n_elements(*(self.file_hist[choose]))-1 do begin
           ind = where(dFiles eq (*(self.file_hist[choose]))[i],count)
           if count ne 0 then begin
              if n_elements(selected) eq 0 then selected=ind $
              else selected=[selected,ind]
           endif
       endfor
       if n_elements(selected) ne 0 then self.filesel->set_file,selected,/highlight
       return
    endif
    num = n_elements(self.file_hist)/2    ;capacity
    if (n_elements(info) ne 0) and (n_elements(dir) ne 0) and (n_elements(files) ne 0) then begin
       if obj_valid(ftpobj) then dir = ftpserver+dir
       ;save for session file
       if n_elements(eventname) eq 0 then eventname = ''
       case strlowcase(eventname) of
            'loadbut': begin
                       ptr_free,self.file_hist[13:*]
                       self.file_hist[13] = ptr_new(['ftype_'+dm_to_string(self.ftype,/int),info,dir,files])
                       end
            'addbut':  begin
                       tmpind = ((where(ptr_valid(self.file_hist)))[-1]+1)<(n_elements(self.file_hist)-1)
                       self.file_hist[tmpind] = ptr_new(['ftype_'+dm_to_string(self.ftype,/int),info,dir,files])
                       end
            'emptbut': begin
                       ptr_free,self.file_hist[10+emptset]
                       self.file_hist[10+emptset] = ptr_new(['ftype_'+dm_to_string(self.ftype,/int),info,dir,files])
                       end
            else:
       endcase
       ;entry to the file history menu       
       for i=0,self.n_history-1 do begin  ;check if this entry already exists
           if dir ne (*(self.file_hist[i]))[2] then continue
           if n_elements(files) ne n_elements(*(self.file_hist[i]))-3 then continue
           same = 1b
           for j=0, n_elements(files)-1 do begin
               if files[j] ne (*(self.file_hist[i]))[3+j] then same=0b
           endfor
           if same then begin ;entry exists, move that entry to the top
              if i ne 0 then begin
                 self.file_hist[0:i] = shift(self.file_hist[0:i],1)
                 for j=0,i do self->my_widget_control,'hist_'+dm_to_string(j),set_value=dm_to_string(j,/int)+' '+(*(self.file_hist[j]))[1]
              endif
              return 
           endif
       endfor
       if self.n_history lt num then begin ;empty room available
          if self.n_history ne 0 then self.file_hist[1:self.n_history] = self.file_hist[0:self.n_history-1]
          self.file_hist[0] = ptr_new(['ftype_'+dm_to_string(self.ftype,/int),info,dir,files])
          self.n_history = self.n_history+1
       endif else begin                    ;full
          ptr_free,self.file_hist[num-1]
          self.file_hist[1:num-1] = self.file_hist[0:num-2]
          self.file_hist[0] = ptr_new(['ftype_'+dm_to_string(self.ftype,/int),info,dir,files])
       endelse
    endif
    histmenu = widget_info(self.tlb,find_by_uname='history')
    widget_control,histmenu,sensitive=(self.n_history gt 0)
    self->my_widget_control,'clearHist',sensitive=(self.n_history gt 0)
    for i=0,num-1 do self->my_widget_control,'hist_'+dm_to_string(i,/int),histmenu,/destroy
    for i=0,self.n_history-1 do begin
        void = widget_button(histmenu,value=dm_to_string(i,/int)+' '+(*(self.file_hist[i]))[1],uname='hist_'+dm_to_string(i,/int))
    endfor
end

;setting work directory will also set the ftp buffer directory if the current one is invalid
pro dcs_mslice::setproperty,binextlib=binextlib,dataDir=dataDir,geomsensitive=geomsensitive,instrgeom=instrgeom,intntype=intntype,slice_cutcolor=slice_cutcolor,tickdir=tickdir,workDir=workDir
    if n_elements(slice_cutcolor) ne 0 then begin
       self.slice_cutcolor = keyword_set(slice_cutcolor)
       self->set_allplots,cutcolor=self.slice_cutcolor
    endif
    if n_elements(tickdir) ne 0 then begin
       self.tdir_choice = keyword_set(tickdir)
       self->my_menu_toggle,'tdir'+['Inside','Outside'],self.tdir_choice
       self->set_allplots,tickdir=self.tdir_choice,/totop
    endif
    if n_elements(binextlib) ne 0 then begin
       if keyword_set(binextlib) and (self.bin_avgsum eq -1) then begin ;weighted mean not implemented in external library
          ok = dialog_message(['The binning method is weighted mean, which is currently not implemented in the external library.','Do you want to switch to arithmetic mean?'],/question,dialog_parent=self.tlb,/center)
          if ok eq 'No' then begin
             ok = dialog_message('The external binning library is disabled.',dialog_parent=self.tlb,/center,/info)
             return
          endif
          self.bin_avgsum = 0
          self->my_menu_toggle,'bin'+['Avg','Sum'],0>(self.bin_avgsum)
          self->my_menu_toggle,'bin'+['WeighMean','ArithMean'],(self.bin_avgsum+1)
          self->my_widget_control,'binAvg',set_value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1]
       endif
       self.bin_extlib = keyword_set(binextlib)
       dm_set_button,widget_info(self.tlb,find_by_uname='binexLib'),self.bin_extlib,onstring='Use External Binning Library',offstring='Use Internal Binning Routine'
    endif
    if (n_elements(dataDir) ne 0) or (n_elements(workDir) ne 0) then begin
       defsysv,'!dave_defaults',exists=exists
       if exists then exists = ptr_valid(!dave_defaults)
    endif
    if n_elements(dataDir) ne 0 then begin
       if file_test(dataDir[0],/directory,/read) or stregex(dataDir[0],'^(ftp|https)://',/boolean,/fold_case) then begin
          self.dirs[0] = dataDir[0]
          if exists then (*!dave_defaults).datDir = dataDir[0]
       endif
    endif
    if n_elements(workDir) ne 0 then begin
       if file_test(workDir[0],/directory,/write) then begin
          self.dirs[1] = workDir[0]
          if exists then (*!dave_defaults).workDir = workDir[0]
          if ~file_test(self.dirs[2],/directory,/write) then self.dirs[2] = self.dirs[1]
          if obj_valid(self.fileSel) then self.fileSel->setproperty,workdir=self.dirs[1]
       endif
    endif
    if n_elements(geomsensitive) ne 0 then self->my_widget_control,['direct','inverse']+'Geom',sensitive=geomsensitive
    if n_elements(instrgeom) ne 0 then begin  ;set instrument geometry 
       self.instrgeom = keyword_set(instrgeom)
       if ~self.instrgeom then begin          ;direct geometry
          self->my_widget_control,'edepEff',/sensitive
          self->my_widget_control,'viewSpurion',sensitive=(ptr_valid(self.dataStrPtr) and ptr_valid(self.projStrPtr) and (self.samp_typ lt 2))
       endif else begin                       ;inverse geometry
          self->my_widget_control,['edepEff','viewSpurion'],sensitive=0
       endelse
       if self.instrname eq 'macs' then self->my_widget_control,'viewSpurion',/sensitive
       self->my_menu_toggle,['direct','inverse']+'Geom',self.instrgeom
    endif
    if n_elements(intntype) ne 0 then begin   ;set intensity type
       ntype = n_elements(self.intn_typ_str)
       oldtype = self.intn_typ
       self.intn_typ = 0>(intntype)<(ntype-1)
       self->my_menu_toggle,'intnTyp'+dm_to_string(indgen(ntype),/int),self.intn_typ
       if ((oldtype eq ntype-2) or (self.intn_typ eq ntype-2)) and (self.samp_typ eq 0) then self->reset_pdt ;J(Q,y) menu is different from others
       self->set_display_button,/clearprojection ;need to recalculate projections
       ;for MACS, chisq needs temperature
       if (self.instrname eq 'macs') then begin
          self->reset_intnlab
          if ((self.intn_typ eq 1) or (self.intn_typ eq 3)) and (~self.extravaxis_yn[1]) then  $
             ok = dialog_message('Please enable the temperature viewing axis to ensure that temperature information is not missing from the data.',dialog_parent=self.tlb,/center)
       endif
    endif
end

;check the current viewing axes to see if they are both on the scattering plane, only the first two viewing axes are checked
function dcs_mslice::viewonscattplane
    if total(finite(self.view_u1,/nan)) ne 0 then return,0b
    if total(finite(self.view_u2,/nan)) ne 0 then return,0b
    if total(abs(self.view_u1)) eq 0         then return,0b
    if total(abs(self.view_u2)) eq 0         then return,0b
    if total(finite(self.orie_u,/nan))  ne 0 then return,1b    ;can't check, assume it is ok
    if total(finite(self.orie_v,/nan))  ne 0 then return,1b    ;can't check, assume it is ok
    tmp = double(crossp(self.orie_u,self.orie_v))
    if transpose(self.view_u1)#tmp ne 0      then return,0b
    if transpose(self.view_u2)#tmp ne 0      then return,0b 
    return,1b
end

;save a slice as iexy file
pro dcs_mslice::write_iexy,unit,xdat,ydat,zdat,zerr,a3=a3,a4=a4,xtit=xtit,ytit=ytit,ztit=ztit,weight=weight
    nx = n_elements(xdat) & ny = n_elements(ydat) & nz = n_elements(zdat)
    ok_wght = (n_elements(weight) eq nz)
    if n_elements(xtit)+n_elements(ytit)+n_elements(ztit) eq 3 then begin
       header = dm_to_string([ztit,'dI'],data=zdat[0],over=over,/addsemicolon)
       header = header+dm_to_string(xtit,data=xdat[0],over=over)
       header = header+dm_to_string(ytit,data=ydat[0],over=over)
       if (n_elements(a3) eq nz) and (n_elements(a4) eq nz) then header = header+dm_to_string(['a3(\deg)','a4(\deg)'],data=a3[0],over=over)
       if ok_wght then header = header+dm_to_string('weight',data=weight[0],over=over)
       printf,unit,header
    endif
    if nx eq nz then begin
       if (n_elements(a3) eq nz) and (n_elements(a4) eq nz) then begin
          a3 = reform(a3,nx,/overwrite) & a4 = reform(a4,nx,/overwrite)
          if ok_wght then begin
             for i=0L,nx-1L do printf,unit,zdat[i],zerr[i],xdat[i],ydat[i],a3[i],a4[i],weight[i],format='(7g)'
          endif else begin
             for i=0L,nx-1L do printf,unit,zdat[i],zerr[i],xdat[i],ydat[i],a3[i],a4[i],format='(6g)'
          endelse
       endif else begin
          if ok_wght then begin
             for i=0L,nx-1L do printf,unit,zdat[i],zerr[i],xdat[i],ydat[i],weight[i],format='(5g)'
          endif else begin
             for i=0L,nx-1L do printf,unit,zdat[i],zerr[i],xdat[i],ydat[i],format='(4g)'
          endelse
       endelse
    endif else begin
       if (n_elements(a3) eq nz) and (n_elements(a4) eq nz) then begin
          if ok_wght then begin
             for i=0L,nx-1L do begin
                 for j=0L,ny-1L do printf,unit,zdat[i,j],zerr[i,j],xdat[i],ydat[j],a3[i,j],a4[i,j],weight[i,j],format='(7g)'
             endfor
          endif else begin
             for i=0L,nx-1L do begin
                 for j=0L,ny-1L do printf,unit,zdat[i,j],zerr[i,j],xdat[i],ydat[j],a3[i,j],a4[i,j],format='(6g)'
             endfor
          endelse
       endif else begin
          if ok_wght then begin
             for i=0L,nx-1L do begin
                 for j=0L,ny-1L do printf,unit,zdat[i,j],zerr[i,j],xdat[i],ydat[j],weight[i,j],format='(5g)'
             endfor
          endif else begin
             for i=0L,nx-1L do begin
                 for j=0L,ny-1L do printf,unit,zdat[i,j],zerr[i,j],xdat[i],ydat[j],format='(4g)'
             endfor
          endelse
       endelse
    endelse
end
    
;save a slice as xyie matrix file
pro dcs_mslice::write_xyie,unit,xdat,ydat,zdat,zerr,a3=a3,a4=a4
    nx = n_elements(xdat) & ny = n_elements(ydat)
    fmt = '('+dm_to_string(nx,/int)+'(g13.6," "))'
    printf,unit,nx,ny
    printf,unit,' '
    for i=0L,nx-1L do printf,unit,xdat[i]
    printf,unit,' '
    for i=0L,ny-1L do printf,unit,ydat[i]
    if n_elements(a3) eq n_elements(zdat) then begin
       printf,unit,' '
       printf,unit,a3,format=fmt
    endif
    if n_elements(a4) eq n_elements(zdat) then begin
       printf,unit,' '
       printf,unit,a4,format=fmt
    endif
    printf,unit,' '
    printf,unit,zdat,format=fmt
    printf,unit,' '
    printf,unit,zerr,format=fmt
end

;save a slice as psutxt file
pro dcs_mslice::write_psutxt,unit,xdat,ydat,zdat,zerr
    nx = n_elements(xdat)    ;# of Q
    ny = n_elements(ydat)    ;# of E
    printf,unit,'# Number of energy transfer values'
    printf,unit,dm_to_string(ny)
    printf,unit,'# Number of Q transfer values'
    printf,unit,dm_to_string(nx)
    printf,unit,'# energy transfer (meV) Values:'
    for i=0L,ny-1 do printf,unit,dm_to_string(ydat[i])
    printf,unit,'# Q transfer (1/Angstroms) Values:'
    for i=0L,nx-1 do printf,unit,dm_to_string(xdat[i])
    for i=0L,nx-1 do begin
        printf,unit,'# Group '+dm_to_string(i)
        for j=0L,ny-1 do printf,unit,dm_to_string(zdat[i,j])+'   '+dm_to_string(zerr[i,j]) 
    endfor
end

;cleanup, life cycle method
pro dcs_mslice::Cleanup
    self->dm_save_parameter,self.parmname,/quiet
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    obj_destroy,self.gr_list
    obj_destroy,self.fileSel
    ptr_free,self.colors,self.slice_cmap,self.lines,self.symbols,self.pdim_list,self.data_tolerance,self.ssfacPtr,self.ftypename,self.ftypeext,self.file_recent,self.file_hist,self.rebinsize
    ptr_free,self.bgratePtr,self.bgtchnPtr,self.bgdetsPtr,self.bgdata,self.powd_line,self.powd_qvalue,self.dcor_info,self.a3mask,self.bga3mask,self.mask,self.mask_ar,self.eff,self.eff1,self.kfactor
    ptr_free,self.absp_info,self.absp_npts,self.absp_nptst,self.view_u1foldcenter,self.view_u1folddir,self.view_u2foldcenter,self.view_u2folddir,self.view_u3foldcenter,self.view_u3folddir,self.whatsnew 
    ptr_free,self.detPosPtr,self.dataStrPtr,self.projStrPtr,self.dataPtr,self.weightPtr,self.bgweightPtr,self.info,self.slice_t,self.slice_tft,self.cutsel_t,self.cutsel_tft,self.volsel_t,self.volsel_tft
end

;initialization, life cycle method
;usage:
;   void=obj_new('dcs_mslice',group_leader=group_leader)
;parameters:
;   NONE
;keyword:
;   uname:          an optional identifier, can be retrieved by getproperty method
;   group_leader:   groupleader widget ID
;   dataFile:       if present, load data from the dataFile, and start it as a volume data viewer
;   dataPtr:        if present, starts it as a volume data viewer
;   backgroundFile: if present, load background from backgroundFile for the volume data viewer, dataFile or dataPtr must be present, not availabe if run in virtual machine
;   backgroundPtr:  if present, use it as background, dataFile or dataPtr must be present, not availabe if run in virtual machine 
;   instrname:      default starting instrument name
;   no_parmfile:    if set, don't load mslice.prm
;   weight,bgweight:for volume data viewer, [ndat]
function dcs_mslice::Init,uname=uname,dataFile=dataFile,backgroundFile=backgroundFile,dataPtr=dataPtr,weight=weight,backgroundPtr=backgroundPtr,bgweight=bgweight,group_leader=group_leader,$
    workDir=workDir,dataDir=dataDir,DAVETool=DAVETool,instrname=instrname,no_parmfile=no_parmfile
    if (n_elements(dataFile) ne 0) or (n_elements(dataPtr) ne 0) then begin
       ;load data from data file
       if n_elements(dataFile) ne 0 then begin
          self->load_data,dataFile,data=data,label=label,unit=unit,is_uniq=is_uniq,weight=weight,/init
          if n_elements(data) eq 0 then return,0
          nitem = n_elements(data[0,*]) & nlabel = n_elements(label) & nunit = n_elements(unit) & nuniq = n_elements(is_uniq)
          dataPtr = ptr_new(create_struct('data',temporary(data)))
          if nlabel ne 0 then begin
             if nlabel lt nitem then label = [label,(['Intensity','Error','x'+dm_to_string(indgen(nitem-2)+1)])[nlabel:*]]
             *dataPtr = create_struct(*dataPtr,'label',label[0:nitem-1])
          endif
          if nunit ne 0  then begin
             if nunit lt nitem then unit = [unit,strarr(nitem-nunit)]
             *dataPtr = create_struct(*dataPtr,'unit',unit[0:nitem-1])
          endif
          if nuniq ne 0 then begin
             if nuniq lt nitem then is_uniq = [is_uniq,bytarr(nitem-nuniq)]
             *dataPtr = create_struct(*dataPtr,'is_uniq',is_uniq[0:nitem-1])
          endif
       endif
       ;check dataPtr
       if (~ ptr_valid(dataPtr)) then $
          errmesg = 'Invalid data pointer. ' $
       else begin
          field = strupcase(tag_names(*dataPtr))    
          loc   = where(field eq 'DATA',count)
          if count eq 0 then begin
             errmesg = "The data pointer doesn't contain a data field. " 
          endif else begin
             nitem = n_elements((*dataPtr).data[0,*])
             if nitem lt 4 then errmesg = 'The data need at least two independent variables.'
             self.ftypecheckmask[0] = (total(finite((*dataPtr).data,/nan)) gt 0)
             self.data_tolerance = ptr_new(fltarr(nitem-2))
          endelse
       endelse
       if n_elements(errmesg) ne 0 then begin
          ok = dialog_message(errmesg,/error,dialog_parent=group_leader,/center)
          return,0
       endif
       loc = where(field eq 'LABEL',count)
       if count eq 0 then begin
          loc = where(field eq 'LABELS',count)
          if count ne 0 then label = (*dataPtr).labels $
          else label = ['Intensity','Error','x'+dm_to_string(indgen(nitem-2)+1)]
          *dataPtr = create_struct(*dataPtr,'label',label)
       endif
       loc = where(field eq 'INSTRUMENT',count)
       if count eq 0 then *dataPtr = create_struct(*dataPtr,'instrument','')
       loc = where(field eq 'UNIT',count)
       if count eq 0 then begin
          loc = where(field eq 'UNITS',count)
          if count ne 0 then unit = (*dataPtr).units $
          else unit = strarr(nitem)
          *dataPtr = create_struct(*dataPtr,'unit',unit)
       endif
       loc = where(field eq 'IS_UNIQ',count)
       if count eq 0 then begin
          is_uniq = bytarr(nitem)
          *dataPtr = create_struct(*dataPtr,'is_uniq',is_uniq)
       endif
       loc = where(field eq 'AXIS_INFO',count)
       if count eq 0 then begin
          axis_info = replicate({vaxis:[0.,0.,0.],label:'',offset:[0.,0.,0.]},nitem-2)
          *dataPtr = create_struct(*dataPtr,'axis_info',axis_info)
       endif
       loc = where(field eq 'TITLE',count)
       if count eq 0 then *dataPtr = create_struct(*dataPtr,'title','')
       loc = where(field eq 'LEGEND',count)
       if count eq 0 then *dataPtr = create_struct(*dataPtr,'legend','')
       loc = where(field eq 'AUTHOR',count)
       if count ne 0 then author = ', '+(*dataPtr).author[0] else author = ''
       self.dataPtr = dataPtr 
       if n_elements(weight) eq n_elements((*dataPtr).data[*,0]) then self.weightPtr[0] = ptr_new(weight)
       ;background data, not availabe if run in virtual machine
       if (n_elements(backgroundFile) ne 0) or ptr_valid(backgroundPtr) then begin
          if ~lmgr(/vm) then begin
             if n_elements(backgroundFile) ne 0 then self->load_data,backgroundFile,data=bgdata,/init,/background,weight=bgweight $
             else bgdata = *backgroundPtr 
          endif else ok = dialog_message("Background subtraction is not availabe in virtual machine mode.",dialog_parent=group_leader,/center)
          if ptr_valid(backgroundPtr) then ptr_free,backgroundPtr
       endif
       if n_elements(bgdata) ne 0 then begin
          if n_elements(bgdata[0,*]) eq nitem then begin
             self.ftypecheckmask[1] = (total(finite(bgdata,/nan)) gt 0)
             self.bgdata[0] = ptr_new(bgdata,/no_copy)
             if n_elements(bgweight) eq n_elements(bgdata[*,0]) then self.bgweightPtr[0] = ptr_new(bgweight)
          endif else ok = dialog_message("The background data are not compatible with the signal data. The background data are discarded.",/error,dialog_parent=group_leader,/center)
       endif
       self.instrname = 'mslicedata'
    endif
    if n_elements(group_leader) ne 0 then self.group_leader=group_leader
    if n_elements(uname) ne 0 then self.uname = uname[0]
    registerName = (['dcs_mslice','mslice'])[ptr_valid(dataPtr)] 
    if xregistered(registerName) and (~ ptr_valid(dataPtr)) then begin ;only allow one dcs_mslice running
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0
       return,0
    endif
    
    ;error handler
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,dialog_parent=group_leader,/center)
       self->Cleanup
       if obj_valid(tmpplotobj) then obj_destroy,tmpplotobj
       return, 0
    end

    tmpplotobj = obj_new('dm_plot',render_method=0)   ;to get symbol and linestyle and color list

    ;initialize the data
    self.idl_version        = dm_to_number(!version.release)
    accel_yn                = (self.idl_version ge 6.1)
    tmp = dm_check_macskidney(0,5,offset=offset)
    self.ncnrftp            = 0b
    self.kidney_offset      = offset
    self.kidney_backlash    = 2.0
    self.instrgeom          = 0     ;default direct geometry
    self.view               = [1b,1b] 
    self.userfwarning       = 1b
    self.dcor_yn            = 0
    self.edep_yn            = 1
    self.bitflip            = 1
    self.tthcorr            = 0
    self.mask_filler        = [-1e20,!values.f_nan,0]
    self.mbparm             = [0.25,4.0,0.04,3.0]
    self.pbgcol_bw          = 1     ;default white
    self.ppvc_choice        = 1     ;default average of minimum 2 neighbors
    self.pdim_list          = ptr_new([3,2]#[150,200,250,300,400])
    self.ptrm_choice        = tmpplotobj->textrendermethod()
    if self.idl_version ge 5.6 then scr_size = get_screen_size() else device,get_screen_size=scr_size
    self.pdim_choice        = 0>(max(where((*self.pdim_list)[1,*] le scr_size[1]/2.5)))
    self.eadjust            = 0
    self.eadjust_spec       = !values.f_nan
    self.eadjust_checkfirst = 0
    self.eint_yn            = 0
    self.intn_typ           = 0     ;default S(q,w)
    self.macsintn_typ       = 0     ;'SPEC'
    self.macs_tolerance     = [[0.01,0.1,0.01],[0.01,0.1,0.01]] ;default tolerance for macs
    self.macs_lamb2         = 1
    self.macs_effmask       = [-10.,10.]
    self.macshistowidth     = !values.f_nan
    self.macshistonchan     = 1024
    self.macs_seq_geom      = [!values.f_nan,!values.f_nan,0]
    self.psi_tolerance      = [0.01,0.01]             ;default tolerance for others
    self.intn_typ_str       = ['S(Q,omega)','Symmetrized S(Q,omega)','d2(sigma)/d(Omega)d(Ef)','Chi(Q,omega)','GDOS','VDOS','J(Q,y)','Unadjusted by ki/kf factor']
    self.powd_typ           = 0     ;no powder line
    self.powd_showall       = 1     ;show closed powder line
    self.powd_latta         = !values.f_nan
    self.dcor_eqran[*]      = !values.f_nan
    self.datr_psi           = !values.f_nan
    self.datr_yn            = 0b
    self.datr_width         = 10.0  ;degree
    self.xsize              = [589L,47L,120L,9L,92L]
    self.ssfac              = 1.0
    self.view_ortharbi      = 1     ;arbitrary
    if !version.memory_bits eq 64 then $
    self.bin_extlib         = 1b    ;use external library
    self.bin_avgsum         = 0     ;arithmetic mean
    self.binthreshold       = 0.0   ;threshold level
    self.bin_conststep      = 0     ;average value
    self.bin_checkstep      = 1b    ;check binnning step
    self.bin_statrange      = [0,1,0,0,0] ;default automatic, initial user specified size is 1, do not apply estimate for all error bars.
    self.zero_error         = [0.0,0.0,!values.f_nan] ;error bar for zero count for dcs,macs,wand
    self.mon_sumfit         = [1,0] ;dcs-fit, macs-auto
    self.mon_typ            = [1,1] ;use bm1 as the dcs monitor, and monitor for macs
    self.macs_mon           = [[!values.f_nan,30],[!values.f_nan,1e6],[!values.f_nan,100]]  ;[[automatic time, specify time],[automatic monitor, specify monitor],[automatic pulse#, specify pulse#]]
    self.temperature        = 300.0
    self.bgeran             = [-20,-10]
    self.intn_lab           = 'Intensity (arb. unit)'
    self.samp_typ           = (ptr_valid(dataPtr)?(-1):0)     ;default powder
    self.rebinsize          = ptr_new([1s,2s,4s,5s,8s,10s,20s,25s,40s,50s])
    self.e_bin              = [1,1,1,1,1]
    self.e_range[*]         = !values.f_nan
    self.latt_parm          = [10d,10d,10d,90d,90d,90d]
    self.orie_u             = [1e,0e,0e]
    self.orie_v             = [0e,1e,0e]
    self.checklattuv        = 1b    ;default checking
    self.view_type          = ['Energy','|Q|','2Theta','Det Group Number','Time Channel','y']
    self.view_label         = ['E','|Q|','2theta','Detector Group','Time','y']
    self.view_unit          = ['meV','\AA!U-1!N','\deg','','\mus','\AA!U-1!N']
    self.view_u1typ         = 1
    self.view_u2typ         = 0
    self.view_u1            = [1,0,0]
    self.view_u2            = [0,1,0]
    self.view_u3            = [0,0,1]
    self.view_u1lab         = ['|Q|','H']
    self.view_u2lab         = ['E','K']
    self.view_u3lab         = 'L'
    self.view_u1folddir     = ptr_new(0)
    self.view_u2folddir     = ptr_new(0)
    self.view_u3folddir     = ptr_new(0)
    self.view_u1foldcenter  = ptr_new(0)
    self.view_u2foldcenter  = ptr_new(0)
    self.view_u3foldcenter  = ptr_new(0)
    self.slice_x            = [0,0,0,0]
    self.slice_y            = [1,1,1,1]
    self.slice_t1           = [2,2,2,2]
    self.slice_t2           = [3,3,3,3]
    self.slice_t3           = [4,4,4,4]
    self.slice_t4           = [5,5,5,5]
    self.slice_xran[*]      = !values.f_nan
    self.slice_yran[*]      = !values.f_nan
    self.slice_t1ran[*]     = !values.f_nan
    self.slice_t2ran[*]     = !values.f_nan
    self.slice_t3ran[*]     = !values.f_nan
    self.slice_t4ran[*]     = !values.f_nan
    self.slice_xstep        = 0.035
    self.slice_ystep        = !values.f_nan
    self.slice_iran[*]      = !values.f_nan
    self.slice_sm_kernel    = [[0.1e,0.2e,0.1e],[0.2e,0.8e,0.2e],[0.1e,0.2e,0.1e]]
    self.slice_cmap         = ptr_new(tmpplotobj->getpalette(/list))
    self.slice_color        = (where(strmatch(*self.slice_cmap,'CB-Spectral',/fold_case)))[0]    ;CB-Spectral
    if self.slice_color lt 0 then $    
    self.slice_color        = (where(strmatch(*self.slice_cmap,'MPL-Viridis',/fold_case)))[0]    ;mpl-viridis    
    self.cut_x              = [0s,0s,0s,0s]
    self.cut_y              = [0s,0s,0s,0s]
    self.cut_t1             = [1s,1s,1s,1s]
    self.cut_t2             = [2s,2s,2s,2s]
    self.cut_t3             = [3s,3s,3s,3s]
    self.cut_t4             = [4s,4s,4s,4s]
    self.cut_t5             = [5s,5s,5s,5s]
    self.cut_ran[*]         = !values.f_nan
    self.cut_step           = 0.035
    self.cut_t1ran[*]       = !values.f_nan
    self.cut_t2ran[*]       = !values.f_nan
    self.cut_t3ran[*]       = !values.f_nan
    self.cut_t4ran[*]       = !values.f_nan
    self.cut_t5ran[*]       = !values.f_nan
    self.cut_yran[*]        = !values.f_nan
    self.vol_smooth         = 1
    self.vol_interp         = 1     ;0:nearest neighbor 1:trilinear
    self.vol_color          = self.slice_color    ;mpl-viridis
    self.volsel_x           = [0,0,0,0]
    self.volsel_y           = [1,1,1,1]
    self.volsel_z           = [2,2,2,2]
    self.vol_xran[*]        = !values.f_nan
    self.vol_xstep          = !values.f_nan
    self.vol_yran[*]        = !values.f_nan
    self.vol_ystep          = !values.f_nan
    self.vol_zran[*]        = !values.f_nan
    self.vol_zstep          = !values.f_nan
    self.vol_iran[*]        = !values.f_nan
    self.volsel_t1          = [3,3,3,3]
    self.volsel_t2          = [4,4,4,4]
    self.volsel_t3          = [5,5,5,5]
    self.volsel_tft1[*]     = !values.f_nan
    self.volsel_tft2[*]     = !values.f_nan
    self.volsel_tft3[*]     = !values.f_nan  
    self.d4sel_x1           = [0,0,0,0]
    self.d4sel_x2           = [1,1,1,1]
    self.d4sel_x3           = [2,2,2,2]
    self.d4sel_x4           = [3,3,3,3]  
    if self.samp_typ eq 0 then begin  ;MACS powder with temperature axis
       self.d4sel_x1[0]     = 1
       self.d4sel_x2[0]     = 0
       self.d4sel_x3[0]     = 4
       self.d4sel_x4[0]     = 5
       self.volsel_x[0]     = 1
       self.volsel_y[0]     = 0
       self.volsel_z[0]     = 4 
       self.slice_x[0]      = 1
       self.slice_y[0]      = 0
       self.slice_t1[0]     = 4
       self.cut_t2[0]       = 4
    endif
    self.d4_x1ran[*]        = !values.f_nan
    self.d4_x2ran[*]        = !values.f_nan
    self.d4_x3ran[*]        = !values.f_nan
    self.d4_x4ran[*]        = !values.f_nan
    self.d4_x1step          = !values.f_nan
    self.d4_x2step          = !values.f_nan
    self.d4_x3step          = !values.f_nan
    self.d4_x4step          = !values.f_nan
    self.d4sel_tft1[*]      = !values.f_nan
    self.d4sel_tft2[*]      = !values.f_nan
    self.qoffsetthreshold   = 0.5   ;adjust axis title only when the offset range is less than 0.5
    self.cut_symbol         = 0s
    self.symbols            = ptr_new(tmpplotobj->getpsym(/list))
    self.cut_color          = 0
    self.colors             = ptr_new(tmpplotobj->getcolor(/list))
    (*self.colors)[0]       = (['white','black'])[self.pbgcol_bw]
    self.coldefine          = 255b-[255b,255b,255b]*self.pbgcol_bw
    self.cut_line           = 0
    self.lines              = ptr_new(tmpplotobj->getlinestyle(/list))
    self.traj_type          = ['[H,0,0]','[0,K,0]','[0,0,L]','E (meV)','None']
    self.traj_x             = [1,0,0,0]
    self.traj_xran          = [!values.f_nan,!values.f_nan]
    self.traj_y             = [0,1,1,1]
    self.traj_yran          = [!values.f_nan,!values.f_nan]
    self.traj_z1            = [5,4,3,4]
    self.traj_z1ran         = [!values.f_nan,!values.f_nan]
    self.traj_z2            = [0,4,3,4]
    self.traj_z2ran         = [!values.f_nan,!values.f_nan]
    self.gr_list            = obj_new('IDL_Container')
    self.errbar             = 1b
    self.useitool           = 0b
    self.negative2th        = 0b
    self.empcanfactor[*]    = 1.0
    self.mjqy               = !values.f_nan
    self.eief               = !values.f_nan
    self.estartend          = [!values.f_nan,!values.f_nan,81.]
    self.psi[*]             = !values.f_nan
    self.absp_l1factor      = 1.0
    self.absp_npts          = [ptr_new([[30,60],[60,120],[90,180]]),$
                               ptr_new([[30,60],[60,120],[90,180]]),$
                               ptr_new([[30,60,60],[60,120,120],[90,180,180]]),$
                               ptr_new([[60,30],[120,60],[180,90]])]
    self.absp_nptst         = [ptr_new(['nr=30, nphi=60','nr=60, nphi=120','nr=90, nphi=180']),$
                               ptr_new(['nr=30, nphi=60','nr=60, nphi=120','nr=90, nphi=180']),$
                               ptr_new(['nr=30, nphi=60, ntheta=60','nr=60, nphi=120, ntheta=120','nr=90, nphi=180, ntheta=180']),$
                               ptr_new(['nw=60, nt=30','nw=120, nt=60','nw=180, nt=90'])]
    self.mod_date           = '8/2025'
    self.info               = ptr_new(['--Mslice for DCS --', $
                               'Based on the NCNR DCS data reduction programs by John Copley and the matlab version of mslice by Radu Coldea.','',$
                               'Please cite the following paper in your publication:',$
                               '"DAVE: A comprehensive software suite for the reduction, visualization, and analysis of low energy neutron spectroscopic data", '+$
                               'R.T. Azuah, L.R. Kneller, Y. Qiu, P.L.W. Tregenna-Piggott, C.M. Brown, J.R.D. Copley, and R.M. Dimeo, '+$
                               'J. Res. Natl. Inst. Stan. Technol. 114, 341 (2009).','','Yiming Qiu (yiming.qiu@nist.gov)   '+self.mod_date])                                 
    self.whatsnew           = ptr_new([$
                               '4/2023: Add raw intensity option for MACS event mode data.',$
                               '8/2022: Add overplotting to other mslice plot option.',$
                               '6/2022: Add polarization correction for DCS data.',$
                               '6/2022: Add out-of-plane absorption correction calculation.',$
                               '10/2021: Add statisitical estimate for zero inensity error bar calculation.',$
                               '2/2021: Add 4d binning and output tab.',$
                               '4/2020: Use monitor information for the weighted mean binning.',$
                               '12/2019: Add neutron mean free path calculation for absorption correction.',$
                               '9/2019: Add detector-dependent self-shielding factor for TOF data.',$
                               '4/2019: Add expanding single crystal projection of powder data in popup menu.',$
                               '12/2018: Add event histogram data format for MACS.',$
                               '9/2018: Add ghoston spurion viewer for MACS.',$ 
                               '','For more information, please check the Mslice help manual.'])
    obj_destroy,tmpplotobj
    self.pathsep = dm_define_pointer(/getpathsep)
    cd,current = current
    self.dirs = [current,current,dm_define_pointer(/gettempdir)]
    defsysv,'!dave_defaults',exists=exists
    if exists then exists = ptr_valid(!dave_defaults)
    if (n_elements(dataDir) eq 0) and (n_elements(workDir) eq 0) and exists then begin
       dataDir = (*!dave_defaults).datDir
       workDir = (*!dave_defaults).workDir
    endif
    self->setproperty,dataDir=dataDir,workDir=workDir
    self.helpfile = ['dcs_mslice.pdf','dcs_mslice_addftype.pdf']
    self.parmname = self.dirs[2]+self.pathsep+registerName+'.prm'
    defsysv,'!DAVE_PDFHELP_DIR',exists=exists
    if exists then self.helpfile = !DAVE_PDFHELP_DIR+self.pathsep+self.helpfile $
    else begin
       self.helpfile[0] = file_which(self.helpfile[0],/include)
       self.helpfile[1] = file_which(self.helpfile[1],/include)
    endelse
    defsysv,'!ddtor',exists=exists
    if (~exists) then defsysv,'!ddtor',!dpi/(180d) ;double version of !dtor
    defsysv,'!dcs_hsq2mn',exists=exists
    if (~exists) then defsysv,'!dcs_hsq2mn',81.80421d ;h^2/2./m_n/e*1e23
    defsysv,'!screensize',exists=exists   ;used in dcs_plotdetmask
    if (~exists) then defsysv,'!screensize',scr_size,1
    defsysv,'!dave_colors',exists=exists  ;used in dcs_plotdetmask
    if (~exists) then begin
       color = {color, $ 
            red:   [255,0,255,  0,  0,  0,255,255,152,128,128,128,255,178,255],$ ; red values
            green: [255,0,  0,255,  0,255,  0,255,120,128,128,128, 54,192,128],$ ; green values
            blue:  [255,0,  0,  0,255,255,255,  0,103,220, 60,128, 54,220,128] $ ; blue values
        }
       defsysv, '!dave_colors',color,1
    endif
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists ;should exist if starts from dave
    if (~ exists) then begin
       file = file_which('dcs_DetPos2.dat',/INCLUDE_CURRENT_DIR)
       if float(!version.release) ge 6.0 then path = file_dirname(file,/mark_directory) $
       else begin
          id   = strpos(file,pathsep,/reverse_search)
          path = strmid(file,0,id+1)
       endelse
       defsysv,'!DAVE_AUXILIARY_DIR',path
    endif
    
    self.iconfiles = !DAVE_AUXILIARY_DIR+['mslice.ico','mslice_scan.bmp','align.ico','ftp.ico','macs.ico','mslice_save.bmp','mslice_open.bmp','idl.ico','unlock.bmp','lock.bmp']
    if dm_to_number(!version.release) ge 6.4 then icon_extra={bitmap:self.iconfiles[0]}

    if n_elements(DAVETool) ne 0 then begin
       if obj_valid(DAVETool) then self.DAVETool = DAVETool
    endif
    
    ;generate widgets
    if ptr_valid(self.dataPtr) then begin
       self.mask_filler   = [!values.f_nan,!values.f_nan,0]
       self.samp_typ      = -1
       n_item             = n_elements((*dataPtr).label)
       if n_item gt 5 then begin
          self.volsel_t   = ptr_new(indgen(n_item-5)+3)
          self.volsel_tft = ptr_new(fltarr(n_item-5,2))
          (*self.volsel_tft)[*] = !values.f_nan 
       endif
       if n_item gt 4 then begin
          self.slice_t    = ptr_new(indgen(n_item-4)+2)
          self.slice_tft  = ptr_new(fltarr(n_item-4,2))
          (*self.slice_tft)[*] = !values.f_nan 
       endif
       if n_item gt 3 then begin
          self.cutsel_t   = ptr_new(indgen(n_item-3)+1)
          self.cutsel_tft = ptr_new(fltarr(n_item-3,2))
          (*self.cutsel_tft)[*] = !values.f_nan 
       endif
       
       ;load mslice.prm if existed
       if ~keyword_set(no_parmfile) then begin
          if file_test(self.parmname,/read) then self->dm_load_parameter,self.parmname,/init
       endif
       
       *(self.info) = ['--Mslice'+(strlen((*dataPtr).instrument) eq 0?'':' for '+(*dataPtr).instrument)+' --','Yiming Qiu (yiming.qiu@nist.gov)'+author+'   '+self.mod_date]
       self.tlb = widget_base(title=(*self.info)[0],/column,kill_notify='dcs_mslice_Exit',mbar=bar,group_leader=self.group_leader,ypad=0,xpad=0,uname='tlb',/tlb_size_event,_extra=icon_extra,map=0)
       filemenu = widget_button(bar,value='File',/menu)
       parmmenu = widget_button(bar,value='Parameter',/menu)
       if ~lmgr(/vm) then begin
       bkgrmenu = widget_button(bar,value='Background',/menu)
       void     = dm_widget_button(bkgrmenu,value='Load ASCII Background Data File...',uname='loadbgData',set_button=(ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1])),imagefile=self.iconfiles[6])
       void     = widget_button(bkgrmenu,value='Clear Background',uname='bgclBut',sensitive=(ptr_valid(self.bgdata[0]) or ptr_valid(self.bgdata[1])))
       endif
       maskmenu = widget_button(bar,value='Mask',/menu)
       optnmenu = widget_button(bar,value='Options',uname='optnMenu',/menu)
       viewmenu = widget_button(bar,value='Window',/menu,uname='viewMenu') 
       helpmenu = widget_button(bar,value='Help',/menu) 
       binmmenu = widget_button(optnmenu,value='Binning Method',/menu,uname='binmMenu')
       if ~lmgr(/vm) then begin
       eptlmenu = widget_button(optnmenu,value='Background Subtraction Tolerance Method',/menu,uname='empttolMenu')
       tol_cwt  = dm_widget_button(eptlmenu,value='Within Tolerance or Closest',uname='empttol_cwt',set_button=~self.empt_tol_optn)
       tol_wtc  = dm_widget_button(eptlmenu,value='Within Tolerance of the Closest',uname='empttol_wtc',set_button=self.empt_tol_optn)
       endif
       crosmenu = dm_widget_button(optnmenu,value='Allow Overplotting to Other Mslice Plots',/menu,uname='allowCrossplot',/separator,set_button=self.crossplot)
       voromenu = dm_widget_button(optnmenu,value='Allow Voronoi Diagram',/menu,uname='allowVoronoi',set_button=self.allow_voronoi)
       ppvcmenu = widget_button(optnmenu,value='Slice Plot Empty Grid Color',uname='ppvcMenu',/menu)
       tdirmenu = widget_button(optnmenu,value='Plot Axis Tick Direction',uname='tdirMenu',/menu)
       pbgcmenu = widget_button(optnmenu,value='Plot Window Background Color',/menu)
       pdimmenu = widget_button(optnmenu,value='Plot Window Default Size',uname='pdimMenu',/menu)
       if !version.os_family eq 'Windows' then begin
       pfntmenu = widget_button(optnmenu,value='Plot Window Text Rendering Method',uname='ptextrmMenu',/menu)
       fntrm0   = dm_widget_button(pfntmenu,value='Texture',uname='textrm_0',set_button=~self.ptrm_choice)
       fntrm1   = dm_widget_button(pfntmenu,value='Triangles',uname='textrm_1',set_button=self.ptrm_choice)
       endif
       void     = dm_widget_button(filemenu,value='Load ASCII Data File...',uname='loadData',imagefile=self.iconfiles[6],/notchecked)
       fcutmenu = dm_widget_button(filemenu,value='Save Cut Data As',uname='savecutMenu',/separator,/menu,imagefile=self.iconfiles[5],/notchecked)
       fslimenu = dm_widget_button(filemenu,value='Save Slice Data As',uname='savesliceMenu',/menu,imagefile=self.iconfiles[5],/notchecked)
       fvolmenu = dm_widget_button(filemenu,value='Save Volume Data As',uname='savevolumeMenu',/menu,sensitive=(n_item ge 5),imagefile=self.iconfiles[5],/notchecked)
       fpromenu = dm_widget_button(filemenu,value='Save All Data As',uname='saveprojMenu',/menu,imagefile=self.iconfiles[5],/notchecked)   
       void     = widget_button(filemenu,value='Exit',uname='exitBut',/separator)
       void     = widget_button(fvolmenu,value='Single Text File...',uname='savevolumetxt')
       vslimenu = widget_button(fvolmenu,value='Multiple Slice Files',uname='savevolsliceMenu',/menu)
       vxslmenu = widget_button(vslimenu,value='X Slices',uname='savevolxsliceMenu',/menu)
       vyslmenu = widget_button(vslimenu,value='Y Slices',uname='savevolysliceMenu',/menu)
       vzslmenu = widget_button(vslimenu,value='Z Slices',uname='savevolzsliceMenu',/menu)
       void     = widget_button(vxslmenu,value='SPE File...',uname='savevolxslicespe')
       void     = widget_button(vxslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolxsliceiexy')
       void     = widget_button(vxslmenu,value='X,Y,I,Err Matrix File...',uname='savevolxslicexyie')
       void     = widget_button(vyslmenu,value='SPE File...',uname='savevolyslicespe')
       void     = widget_button(vyslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolysliceiexy')
       void     = widget_button(vyslmenu,value='X,Y,I,Err Matrix File...',uname='savevolyslicexyie')
       void     = widget_button(vzslmenu,value='SPE File...',uname='savevolzslicespe')
       void     = widget_button(vzslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolzsliceiexy')
       void     = widget_button(vzslmenu,value='X,Y,I,Err Matrix File...',uname='savevolzslicexyie')
       void     = widget_button(fslimenu,value='DAVE File...',uname='saveslicedave')
       void     = widget_button(fslimenu,value='SPE File...',uname='saveslicespe')
       void     = widget_button(fslimenu,value='I,Err,X,Y 4-Column File...',uname='savesliceiexy')
       void     = widget_button(fslimenu,value='X,Y,I,Err Matrix File...',uname='saveslicexyie')
       if obj_valid(self.DAVETool) then $
          void  = widget_button(fslimenu,value='Send to DAVE Data Manager',uname='sendslicedave',/separator)
       void     = widget_button(fcutmenu,value='X,Y,Err 3-Column File...',uname='savecutxye')
       void     = widget_button(fcutmenu,value='DAVE File...',uname='savecutdave')
       if obj_valid(self.DAVETool) then $
          void  = widget_button(fcutmenu,value='Send to DAVE Data Manager',uname='sendcutdave',/separator)
       void     = widget_button(fpromenu,value='Text File...',uname='saveprojtxt')
       void     = dm_widget_button(parmmenu,value='Load Parameters...',uname='ldpmBut',imagefile=self.iconfiles[6],accelerator='Ctrl+L',/notchecked)
       void     = dm_widget_button(parmmenu,value='Save Parameters...',uname='svpmBut',imagefile=self.iconfiles[5],accelerator='Ctrl+S',/notchecked)
       void     = widget_button(parmmenu,value='Change Background Subtraction Factor...',uname='ssfcBut',/separator)
       void     = widget_button(parmmenu,value='Change Smoothing Kernel...',uname='knpmBut')
       void     = widget_button(parmmenu,value='Change Binning Threshold...',uname='bnthBut')
       if ~lmgr(/vm) then $
          void  = widget_button(parmmenu,value='Change Background Subtraction Tolerance Value...',uname='bgtolBut')
       void     = widget_button(parmmenu,value='Set Working Directory...',uname='wdirBut',/separator)
       void     = dm_widget_button(maskmenu,value='Mask Data by Value...',uname='maskFiller')  
       binAvg   = dm_widget_button(binmmenu,value='Average'+([' (Weighted Mean)',' (Arithmetic Mean)',''])[self.bin_avgsum+1],uname='binAvg',/menu,set_button=(self.bin_avgsum ne 1))
       binArith = dm_widget_button(binAvg,value='Arithmetic Mean',uname='binArithMean',set_button=(self.bin_avgsum eq 0))
       binWeigh = dm_widget_button(binAvg,value='Weighted Mean',uname='binWeighMean',set_button=(self.bin_avgsum eq -1))
       binSum   = dm_widget_button(binmmenu,value='Sum',uname='binSum',set_button=(self.bin_avgsum eq 1))
       crosYes  = dm_widget_button(crosmenu,value='Yes',uname='crossplotYes',set_button=self.crossplot)
       crosNo   = dm_widget_button(crosmenu,value='No',uname='crossplotNo',set_button=(~self.crossplot))
       vorYes   = dm_widget_button(voromenu,value='Yes',uname='voronoiYes',set_button=self.allow_voronoi)
       vorNo    = dm_widget_button(voromenu,value='No',uname='voronoiNo',set_button=(~self.allow_voronoi))
       bgWhite  = dm_widget_button(pbgcmenu,value='White',uname='pbgWhite',set_button=self.pbgcol_bw,_extra=accel_yn?{accelerator:'Ctrl+W'}:{no_copy:1})
       bgBlack  = dm_widget_button(pbgcmenu,value='Black',uname='pbgBlack',set_button=(~self.pbgcol_bw),_extra=accel_yn?{accelerator:'Ctrl+B'}:{no_copy:1})
       for i=0,n_elements((*self.pdim_list)[0,*])-1 do void = dm_widget_button(pdimmenu,value=dm_to_string((*self.pdim_list)[*,i],separator='x'),uname='pdchoice_'+dm_to_string(i,/int),set_button=(self.pdim_choice eq i)) 
       pvcnone  = dm_widget_button(ppvcmenu,set_button=(self.ppvc_choice eq 0),value='None',uname='ppvcNone')
       pvcavg2  = dm_widget_button(ppvcmenu,set_button=(self.ppvc_choice eq 1),value='Average of Minimum 2 Neighbors',uname='ppvcAvg2')
       pvcavg3  = dm_widget_button(ppvcmenu,set_button=(self.ppvc_choice eq 2),value='Average of Minimum 3 Neighbors',uname='ppvcAvg3')
       void     = dm_widget_button(tdirmenu,value='Inside',uname='tdirInside',set_button=~self.tdir_choice)
       void     = dm_widget_button(tdirmenu,value='Outside',uname='tdirOutside',set_button=self.tdir_choice)
       bincstep = dm_widget_button(binmmenu,value='Constant Intervals',/separator,uname='binCS',set_button=self.bin_conststep,onstring='Constant Intervals',offstring='Nonconstant Intervals')
       binexlib = dm_widget_button(binmmenu,value='Use External Binning Library',uname='binexLib',onstring='Use External Binning Library',offstring='Use Internal Binning Routine',set_button=self.bin_extlib)
       void     = widget_button(viewmenu,value='Current Plot Window',uname='viewCurr',sensitive=0,accelerator='Ctrl+Tab')
       void     = widget_button(viewmenu,value='Close All Plot Windows',uname='killAll',sensitive=0)
       arrpmenu = widget_button(viewmenu,value='Arrange Plots',uname='arrPlot',sensitive=0,/menu)
       void     = widget_button(arrpmenu,value='Cascade',uname='arrCascade')
       void     = widget_button(arrpmenu,value='Table',uname='arrTable')
       void     = widget_button(helpmenu,value='About Mslice',uname='selfBut')
       
       ;menu bar separator for windows system
       if !version.os_family eq 'Windows' then $
          mbar_sep = widget_label(self.tlb,value=' ',sensitive=0,/dynamic_resize,/align_top,scr_ysize=5) $
       else $
          self.xsize[2] = self.xsize[2]+20
       
       self->data_subbg ;apply mask and subtract background
       
       self->reset_display
       widget_control,self.dispBas,/update,/map
    endif else begin
       self.tlb = widget_base(title=(*self.info)[0],/column,kill_notify='dcs_mslice_Exit',mbar=bar,group_leader=self.group_leader,ypad=0,xpad=0,uname='tlb',/tlb_size_event,_extra=icon_extra,map=0)
       filemenu = widget_button(bar,value='File',/menu)
       parmmenu = widget_button(bar,value='Parameter',/menu,uname='parmMenu')
       bkgrmenu = widget_button(bar,value='Background',/menu,uname='bkgrMenu')
       maskmenu = widget_button(bar,value='Mask',/menu,uname='maskMenu')
       optnmenu = widget_button(bar,value='Options',uname='optnMenu',/menu)
       viewmenu = widget_button(bar,value='Window',/menu,uname='viewMenu')
       helpmenu = widget_button(bar,value='Help',/menu)
       igeomenu = widget_button(optnmenu,value='Instrument Geometry',uname='igeomMenu',/menu)
       itypmenu = widget_button(optnmenu,value='View Intensity As',/menu,uname='itypMenu')
       binmmenu = widget_button(optnmenu,value='Binning Method',/menu,uname='binmMenu')
       moncmenu = widget_button(optnmenu,value='Monitor Count',uname='moncMenu',/menu)
       abspmenu = dm_widget_button(optnmenu,value='Absorption Correction',/menu,uname='abspCorr')
       vaxsmenu = widget_button(optnmenu,value='Viewing Axis',/menu,uname='vaxisMenu',sensitive=0)
       vfldmenu = widget_button(optnmenu,value='Folding Type',/menu,uname='vfoldMenu',sensitive=0)
       powdmenu = dm_widget_button(optnmenu,value='Show Powder Line',/menu,uname='showPowder')
       epadmenu = dm_widget_button(optnmenu,value='Elastic Peak Position Adjustment',/menu,uname='eAdjust')
       eptlmenu = widget_button(optnmenu,value='Empty Can Subtraction Tolerance Method',/menu,uname='empttolMenu')
       crosmenu = dm_widget_button(optnmenu,value='Allow Overplotting to Other Mslice Plots',/menu,uname='allowCrossplot',/separator)
       voromenu = dm_widget_button(optnmenu,value='Allow Voronoi Diagram',/menu,uname='allowVoronoi')
       ppvcmenu = widget_button(optnmenu,value='Slice Plot Empty Grid Color',uname='ppvcMenu',/menu)
       tdirmenu = widget_button(optnmenu,value='Plot Axis Tick Direction',uname='tdirMenu',/menu)
       pbgcmenu = widget_button(optnmenu,value='Plot Window Background Color',/menu)
       pdimmenu = widget_button(optnmenu,value='Plot Window Default Size',uname='pdimMenu',/menu)
       if !version.os_family eq 'Windows' then begin
       pfntmenu = widget_button(optnmenu,value='Plot Window Text Rendering Method',uname='ptextrmMenu',/menu)
       void     = dm_widget_button(pfntmenu,value='Texture',uname='textrm_0')
       void     = dm_widget_button(pfntmenu,value='Triangles',uname='textrm_1') 
       end
       void     = widget_button(filemenu,value='File Type',uname='ftypeMenu',/menu)
       void     = widget_button(filemenu,value='File Tools',uname='ftoolMenu',/menu,sensitive=0)
       fcutmenu = dm_widget_button(filemenu,value='Save Cut Data As',uname='savecutMenu',/menu,/separator,sensitive=0,imagefile=self.iconfiles[5],/notchecked)
       fslimenu = dm_widget_button(filemenu,value='Save Slice Data As',uname='savesliceMenu',/menu,sensitive=0,imagefile=self.iconfiles[5],/notchecked)
       fvolmenu = dm_widget_button(filemenu,value='Save Volume Data As',uname='savevolumeMenu',/menu,sensitive=0,imagefile=self.iconfiles[5],/notchecked)
       fpromenu = dm_widget_button(filemenu,value='Save Projection As',uname='saveprojMenu',/menu,sensitive=0,imagefile=self.iconfiles[5],/notchecked)
       void     = dm_widget_button(filemenu,value='Read and Plot ASCII Data File...',uname='readascii',/separator,imagefile=self.iconfiles[0],/notchecked)
       if ~lmgr(/vm) then $
          void  = widget_button(filemenu,value='Execute Script File...',uname='runScript',/separator)
       void     = dm_widget_button(filemenu,value='Load Session...',uname='ldsesBut',/separator,imagefile=self.iconfiles[6],/notchecked)
       void     = dm_widget_button(filemenu,value='Save Session...',uname='svsesBut',imagefile=self.iconfiles[5],/notchecked)
       void     = widget_button(filemenu,value='Recent Files',uname='history',sensitive=0,/menu,/separator)
       void     = widget_button(filemenu,value='Clear File History',uname='clearHist',sensitive=0)
       void     = widget_button(filemenu,value='Exit',uname='exitBut',/separator)
       void     = widget_button(fvolmenu,value='Single Text File...',uname='savevolumetxt')
       vslimenu = widget_button(fvolmenu,value='Multiple Slice Files',uname='savevolsliceMenu',/menu)
       vxslmenu = widget_button(vslimenu,value='X Slices',uname='savevolxsliceMenu',/menu)
       vyslmenu = widget_button(vslimenu,value='Y Slices',uname='savevolysliceMenu',/menu)
       vzslmenu = widget_button(vslimenu,value='Z Slices',uname='savevolzsliceMenu',/menu)
       void     = widget_button(vxslmenu,value='SPE File...',uname='savevolxslicespe')
       void     = widget_button(vxslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolxsliceiexy')
       void     = widget_button(vxslmenu,value='X,Y,I,Err Matrix File...',uname='savevolxslicexyie')
       void     = widget_button(vyslmenu,value='SPE File...',uname='savevolyslicespe')
       void     = widget_button(vyslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolysliceiexy')
       void     = widget_button(vyslmenu,value='X,Y,I,Err Matrix File...',uname='savevolyslicexyie')
       void     = widget_button(vzslmenu,value='SPE File...',uname='savevolzslicespe')
       void     = widget_button(vzslmenu,value='I,Err,X,Y 4-Column File...',uname='savevolzsliceiexy')
       void     = widget_button(vzslmenu,value='X,Y,I,Err Matrix File...',uname='savevolzslicexyie')
       void     = widget_button(fslimenu,value='DAVE File...',uname='saveslicedave')
       void     = widget_button(fslimenu,value='SPE File...',uname='saveslicespe')
       void     = widget_button(fslimenu,value='I,Err,X,Y 4-Column File...',uname='savesliceiexy')
       void     = widget_button(fslimenu,value='X,Y,I,Err Matrix File...',uname='saveslicexyie')
       if obj_valid(self.DAVETool) then $
          void  = widget_button(fslimenu,value='Send to DAVE Data Manager',uname='sendslicedave',/separator)
       void     = widget_button(fcutmenu,value='X,Y,Err 3-Column File...',uname='savecutxye')
       void     = widget_button(fcutmenu,value='DAVE File...',uname='savecutdave')
       if obj_valid(self.DAVETool) then $
          void  = widget_button(fcutmenu,value='Send to DAVE Data Manager',uname='sendcutdave',/separator)
       void     = widget_button(fpromenu,value='Text File...',uname='saveprojtxt')
       void     = dm_widget_button(parmmenu,value='Load Parameters...',uname='ldpmBut',imagefile=self.iconfiles[6],accelerator='Ctrl+L',/notchecked)
       void     = dm_widget_button(parmmenu,value='Save Parameters...',uname='svpmBut',imagefile=self.iconfiles[5],accelerator='Ctrl+S',/notchecked)
       ssfcmenu = widget_button(parmmenu,value='Change Self-shielding Factor',/separator,/menu)
       void     = dm_widget_button(ssfcmenu,value='Single Value Self-shileding Factor...',uname='ssfcBut')
       void     = dm_widget_button(ssfcmenu,value='Read Detector Dependent Self-shileding Factor...',uname='readdetssfcBut')
       void     = dm_widget_button(ssfcmenu,value='Read Energy Dependent Self-shileding Factor...',uname='readenssfcBut')
       void     = widget_button(parmmenu,value='Change Smoothing Kernel...',uname='knpmBut')
       void     = widget_button(parmmenu,value='Change Binning Threshold...',uname='bnthBut')
       void     = widget_button(parmmenu,value='Change File Combination Tolerance Value...',uname='addtolBut')
       void     = widget_button(parmmenu,value='Change Empty Can Subtraction Tolerance Value...',uname='bgtolBut')
       if self.extraempcan_yn then begin
          void  = widget_button(parmmenu,value='Change Multiple-Set Empty Can Factors...',uname='emfcBut',sensitive=0)
       endif
       if self.instrname eq 'wand' then begin
          void  = widget_button(parmmenu,value='Change WAND Detector Offset Angle...',uname='detoffsetBut',/separator)
          void  = widget_button(parmmenu,value='Change WAND Incident Wavelength...',uname='wandeiefBut')
       endif
       if self.instrname eq 'macs' then begin
          void  = widget_button(parmmenu,value='Change MACS Detector Offset Angle...',uname='detoffsetBut',/separator)
          void  = widget_button(parmmenu,value='Change MACS Detector Efficiency...',uname='macsdeteffBut')
       endif
       void     = widget_button(parmmenu,value='Set Data Directory...',uname='ddirBut',/separator)
       void     = widget_button(parmmenu,value='Set Working Directory...',uname='wdirBut')
       void     = dm_widget_button(bkgrmenu,value='Load Empty Can File(s)'+(['',' 1'])[self.extraempcan_yn],uname='emptBut')
       if self.extraempcan_yn then for i=2,n_elements(self.bgdata) do $
          void  = dm_widget_button(bkgrmenu,value='Load Empty Can File(s) '+dm_to_string(i),uname='emptBut'+dm_to_string(i),sensitive=0)
       void     = dm_widget_button(bkgrmenu,value='Load Detector Background File(s)...',uname='bgflBut')
       void     = dm_widget_button(bkgrmenu,value='Load Time Channel Background File(s)...',uname='bgtcBut')
       void     = dm_widget_button(bkgrmenu,value='Load Dark Count File(s)',uname='bgdkBut')
       void     = dm_widget_button(bkgrmenu,value='Set Overall Background Rate...',uname='bgovBut')
       void     = widget_button(bkgrmenu,value='Clear Background',uname='bgclBut',sensitive=0)
       void     = dm_widget_button(bkgrmenu,value='Load Vanadium File(s)...',uname='vandBut',/separator)
       void     = widget_button(bkgrmenu,value='Clear Vanadium',uname='vdclBut',sensitive=0)
       mkdbmenu = dm_widget_button(maskmenu,value='DCS Mask',uname='mkdbMenu',/menu)
       void     = dm_widget_button(mkdbmenu,value='Mask Negative Angle Detectors',uname='mkngBut')
       void     = dm_widget_button(mkdbmenu,value='Mask Upper and Lower Banks',uname='mkulBut')
       void     = dm_widget_button(mkdbmenu,value='Mask Lower Bank',uname='mkloBut',/separator)
       void     = dm_widget_button(mkdbmenu,value='Mask Central Bank',uname='mkcnBut')
       void     = dm_widget_button(mkdbmenu,value='Mask Upper Bank',uname='mkupBut')
       stmkmenu = widget_button(maskmenu,value='Set Mask',/menu,uname='stmkMenu')
       void     = dm_widget_button(stmkmenu,value='By Detector Number...',uname='mkstdnum')
       void     = dm_widget_button(stmkmenu,value='By Detector Angle...',uname='mkstangl')
       plmkmenu = widget_button(maskmenu,value='Plot Mask',uname='mkplBut',sensitive=0,/menu)
       void     = dm_widget_button(plmkmenu,value='By Detector Number...',uname='mkpldnum')
       void     = dm_widget_button(plmkmenu,value='By Detector Angle...',uname='mkplangl')
       void     = dm_widget_button(plmkmenu,value='Mask Bragg Peaks...',uname='mkbpBut',sensitive=0)
       void     = dm_widget_button(maskmenu,value='Read Mask File...',uname='mkrdBut',/separator,imagefile=self.iconfiles[6],/notchecked)
       void     = dm_widget_button(maskmenu,value='Save Mask File...',uname='mksvBut',sensitive=0,imagefile=self.iconfiles[5],/notchecked)
       void     = widget_button(maskmenu,value='Clear Mask',uname='mkclBut',sensitive=0,/separator)
       void     = dm_widget_button(igeomenu,value='Direct',uname='directGeom')
       void     = dm_widget_button(igeomenu,value='Inverse',uname='inverseGeom')
       for i=0,n_elements(self.intn_typ_str)-1 do $
           void = dm_widget_button(itypmenu,value=self.intn_typ_str[i],uname='intnTyp'+dm_to_string(i,/int))
       void     = dm_widget_button(itypmenu,value='E-Integrated Intensity',uname='intnEint',set_button=self.eint_yn,onstring='E-Integrated Intensity Enabled',offstring='E-Integrated Intensity Disabled',/separator)
       bavgmenu = dm_widget_button(binmmenu,value='Average',uname='binAvg',/menu)
       void     = dm_widget_button(bavgmenu,value='Arithmetic Mean',uname='binArithMean')
       void     = dm_widget_button(bavgmenu,value='Weighted Mean',uname='binWeighMean')
       void     = dm_widget_button(binmmenu,value='Sum',uname='binSum')
       void     = dm_widget_button(binmmenu,value='Constant Intervals',/separator,uname='binCS')
       void     = dm_widget_button(binmmenu,value='Use External Binning Library',uname='binexLib')
       void     = dm_widget_button(moncmenu,value='Sum',uname='monSum')
       void     = dm_widget_button(moncmenu,value='Integrated Intensity',uname='monFit')
       mondmenu = widget_button(moncmenu,value='Monitor Choice',/menu,/separator,uname='mondMenu')
       void     = dm_widget_button(mondmenu,value='FC0',uname='monFC0')
       void     = dm_widget_button(mondmenu,value='BM1 (default)',uname='monBM1')
       void     = dm_widget_button(mondmenu,value='Duration',uname='monTIME')
       if lmgr(/runtime) then usertmp = 'User Macro...' else usertmp = 'User Macro/Function/Procedure...'
       if ~lmgr(/vm) then $
          void  = dm_widget_button(optnmenu,value=usertmp,uname='userFunction',set_button=strlen(self.userf_str),onstring='Enable '+usertmp,offstring='Disable '+usertmp,/separator)
       void     = dm_widget_button(optnmenu,value='Dark Angle Correction...',uname='darkCorr',set_button=self.dcor_yn,onstring='Dark Angle Correction Enabled',$
                  offstring='Dark Angle Correction Disabled',sensitive=0,separator=lmgr(/vm))
       void     = dm_widget_button(optnmenu,value='Show Dark Angle Trajectory',uname='darkTraj',set_button=self.datr_yn,onstring='Show Dark Angle Trajectory',$
                  offstring='Disable Dark Angle Trajectory')           
       void     = dm_widget_button(optnmenu,value='E-Dependent Detector Efficiency Correction',uname='edepEff',set_button=self.edep_yn,onstring='E-Dependent Detector Efficiency Correction Enabled',$
                  offstring='E-Dependent Detector Efficiency Correction Disabled')
       void     = dm_widget_button(optnmenu,value='Hide Error Bar',uname='errBar',set_button=1-self.errbar,onstring='Error Bar Disabled',offstring='Error Bar Enabled')
       void     = dm_widget_button(optnmenu,value='Allow NCNR Data Repository',uname='ncnrftp',set_button=self.ncnrftp,onstring='Allow NCNR Data Repository',offstring='Disallow NCNR Data Repository')
       spurmenu = widget_button(optnmenu,value='View Spurions',/menu,/separator,uname='spurionMenu',sensitive=0)
       void     = widget_button(spurmenu,value='View Spurion Caused by Detector',uname='viewSpurion')
       void     = dm_widget_button(viewmenu,value='Projection',uname='viewProj',set_button=self.view[0],onstring='Hide Projection',offstring='Show Projection',_extra=accel_yn?{accelerator:'Ctrl+P'}:{no_copy:1})
       void     = dm_widget_button(viewmenu,value='Display',uname='viewDisp',set_button=self.view[1],onstring='Hide Display',offstring='Show Display',_extra=accel_yn?{accelerator:'Ctrl+D'}:{no_copy:1})
       void     = widget_button(viewmenu,value='Current Plot Window',uname='viewCurr',sensitive=0,accelerator='Ctrl+Tab',/separator)
       void     = widget_button(viewmenu,value='Close All Plot Windows',uname='killAll',sensitive=0,separator=(self.idl_version lt 6.1))
       arrpmenu = widget_button(viewmenu,value='Arrange Plots',uname='arrPlot',sensitive=0,/menu)
       void     = dm_widget_button(powdmenu,value='None',uname='powderNone')
       void     = dm_widget_button(powdmenu,value='Al',uname='powderAl')
       void     = dm_widget_button(powdmenu,value='Cu',uname='powderCu')
       void     = dm_widget_button(powdmenu,value='Stainless Steel',uname='powderSt')
       void     = dm_widget_button(powdmenu,value='Specify...',uname='powderSp')
       void     = dm_widget_button(powdmenu,value='Show Closed Powder Line',uname='powdershowall',set_button=self.powd_showall,onstring='Show Closed Powder Line',offstring='Show Actual Powder Line',/separator)
       void     = dm_widget_button(vaxsmenu,value='Orthogonal',uname='vaxisOrtho')
       void     = dm_widget_button(vaxsmenu,value='Arbitrary',uname='vaxisArbi')
       void     = dm_widget_button(vaxsmenu,value='Allow Extra Viewing Axis',uname='extraVaxis',/separator,set_button=self.extravaxis_yn[0],onstring='Allow Extra Viewing Axis',offstring='Disallow Extra Viewing Axis')
       mrrfmenu = dm_widget_button(vfldmenu,value='Mirror Reflection',/menu,uname='vfoldMirror')
       void     = dm_widget_button(mrrfmenu,value='Fold to One Side',uname='mrrf_One')
       void     = dm_widget_button(mrrfmenu,value='Fold to Both Sides',uname='mrrf_Both')
       smrtmenu = dm_widget_button(vfldmenu,value='Symmetry Rotation',/menu,uname='vfoldRotation')
       void     = dm_widget_button(smrtmenu,value='To the First Symmetric Section',uname='symrot_Section')
       void     = dm_widget_button(smrtmenu,value='To the First Quadrant',uname='symrot_Quadrant')
       void     = dm_widget_button(smrtmenu,value='Expand to 360'+string('b0'xb),uname='symrot_Fullmap')
       smvcmenu = widget_button(smrtmenu,value='First Symmetric Section Starting Vector',/menu,uname='smvcMenu',/separator)
       void     = dm_widget_button(smvcmenu,value='Default (u1)',uname='symvec_U1')
       void     = dm_widget_button(smvcmenu,value='Specify',uname='symvec_Specify')
       nmrtmenu = widget_button(smrtmenu,value='Number of Rotations for 360'+string('b0'xb)+' Expansion',/menu,uname='nmrtMenu')
       void     = dm_widget_button(nmrtmenu,value='Minimum',uname='nmrt_Min')
       void     = dm_widget_button(nmrtmenu,value='N_fold-1',uname='nmrt_Nminus1')
       void     = dm_widget_button(abspmenu,value='None',uname='abspNone')
       void     = dm_widget_button(abspmenu,value='Cylinder...',uname='abspCylinder')
       void     = dm_widget_button(abspmenu,value='Annulus...',uname='abspAnnulus')
       void     = dm_widget_button(abspmenu,value='Sphere...',uname='abspSphere')
       void     = dm_widget_button(abspmenu,value='Flat Plate...',uname='abspFlat')
       void     = widget_button(abspmenu,value='Set Incident Beam Path Factor...',uname='abspL1Factor',/separator)
       void     = dm_widget_button(abspmenu,value='Show Transmission as Intensity',uname='abspInt')
       void     = widget_button(abspmenu,value='Calculate Neutron Mean Free Path L0 && L1...',uname='calcl0l1',/separator)
       void     = dm_widget_button(epadmenu,value='None',uname='epadNone')
       void     = dm_widget_button(epadmenu,value='Automatic',uname='epadAuto')
       void     = dm_widget_button(epadmenu,value='Specify...',uname='epadSpec')
       void     = dm_widget_button(epadmenu,value='Check the First File Only',uname='epadCheckfirst',set_button=self.eadjust_checkfirst,$
                  onstring='Check the First File Only',offstring='Check All Files',/separator)
       void     = dm_widget_button(eptlmenu,value='Within Tolerance or Closest',uname='empttol_cwt')
       void     = dm_widget_button(eptlmenu,value='Within Tolerance of the Closest',uname='empttol_wtc')
       void     = dm_widget_button(crosmenu,value='Yes',uname='crossplotYes')
       void     = dm_widget_button(crosmenu,value='No',uname='crossplotNo')
       void     = dm_widget_button(voromenu,value='Yes',uname='voronoiYes')
       void     = dm_widget_button(voromenu,value='No',uname='voronoiNo')
       void     = dm_widget_button(pbgcmenu,value='White',uname='pbgWhite',_extra=accel_yn?{accelerator:'Ctrl+W'}:{no_copy:1})
       void     = dm_widget_button(pbgcmenu,value='Black',uname='pbgBlack',_extra=accel_yn?{accelerator:'Ctrl+B'}:{no_copy:1})
       for i=0,n_elements((*self.pdim_list)[0,*])-1 do void = dm_widget_button(pdimmenu,value=dm_to_string((*self.pdim_list)[*,i],separator='x'),uname='pdchoice_'+dm_to_string(i,/int))
       pvcnone  = dm_widget_button(ppvcmenu,value='None',uname='ppvcNone')
       pvcavg2  = dm_widget_button(ppvcmenu,value='Average of Minimum 2 Neighbors',uname='ppvcAvg2')
       pvcavg3  = dm_widget_button(ppvcmenu,value='Average of Minimum 3 Neighbors',uname='ppvcAvg3')
       void     = dm_widget_button(tdirmenu,value='Inside',uname='tdirInside')
       void     = dm_widget_button(tdirmenu,value='Outside',uname='tdirOutside')
       void     = widget_button(arrpmenu,value='Cascade',uname='arrCascade')
       void     = widget_button(arrpmenu,value='Table',uname='arrTable')
       if strlen(self.helpfile[0]) ne 0 then $
          void  = dm_widget_button(helpmenu,value='Mslice Help',uname='hpctBut',imagefile=self.iconfiles[0],accelerator='F1',/notchecked)
       void     = dm_widget_button(helpmenu,value='IDL Help',uname='idlhpBut',imagefile=self.iconfiles[7],/notchecked)
       if strlen(self.helpfile[1]) ne 0 then $
       void     = widget_button(helpmenu,value='How to Add a New File Type',uname='hpftBut')
       void     = widget_button(helpmenu,value="What's New in Mslice",uname='hpnewBut',/separator)
       void     = widget_button(helpmenu,value='About Mslice for DCS',uname='selfBut',/separator)

       ;menu bar separator for windows system
       if !version.os_family eq 'Windows' then $
          mbar_sep = widget_label(self.tlb,value=' ',sensitive=0,/dynamic_resize,/align_top,scr_ysize=5) $
       else $
          self.xsize[2] = self.xsize[2]+20
       
       self->dm_filemenu,/init ;also set up menu items
       self->reset_file
       
       ;load dcs_mslice.prm if existed
       if file_test(self.parmname,/read) then begin
          self->dm_load_parameter,self.parmname,/init
          parm_loaded = 1b
       endif else if n_elements(instrname) eq 0 then begin ;only ask if the default parameter file is unavailable and the instrument is not specified
          ftype = dm_dialog_input('file type:',default=self.ftype,droplist_content=ptr_new(*self.ftypename),is_droplist=[1],/return_number,title='Please choose')
       endif
       if n_elements(instrname) ne 0 then begin
          ftype = where(strlowcase(*self.ftypename) eq strlowcase(strtrim(instrname[0],2)),count)
          if count eq 0 then ftype = self.ftype 
       endif
       if n_elements(ftype) ne 0 then begin
          if self.ftype ne ftype[0] then ok = self->dm_filetools('ftype_'+dm_to_string(ftype))
       endif
       if (self.instrname ne 'dcs') and (self.instrname ne 'macs') then self.fileSel->set_ftpbuffer,''
       self.fileSel->getproperty,ncnrftp=ncnrftp
       self.ncnrftp = ncnrftp
       dm_set_button,widget_info(self.tlb,find_by_uname='ncnrftp'),self.ncnrftp,onstring='Allow NCNR Data Repository',offstring='Disallow NCNR Data Repository'

       self->reset_projection
       self->reset_display
       widget_control,self.projBas,/update,/map
       widget_control,self.dispBas,/update,/map
    endelse
    
    dm_center_kid,self.tlb,self.group_leader
    widget_control,self.tlb,/realize,/update
    geom  = widget_info(self.tlb,/geometry)
    space = total(1-self.view)*geom.space*(!version.os_family eq 'Windows')
    if self.idl_version ge 5.5 then $
       self.geom = [geom.xsize,geom.ysize]+[0,space] $
    else begin
       self.geom = [geom.scr_xsize,geom.scr_ysize]+[0,space]
       widget_control,self.tlb,scr_xsize=self.geom[0],scr_ysize=self.geom[1]-space
    endelse
    
    widget_control,self.tlb,set_uvalue=self,/map
    
    xmanager,registerName,self.tlb,event_handler='dcs_mslice_event',cleanup='dcs_mslice_Exit',/no_block
    if (geom.scr_ysize gt scr_size[1]) and (registerName eq 'dcs_mslice') and (~keyword_set(parm_loaded)) then $  ;hint only once
       ok = dialog_message(['The monitor size is too small for mslice.','Click mslice menu Window->Projection or Window->Display to hide part of the mslice window.'],title='Small screen size:',/info,dialog_parent=self.tlb,/center)

    return,1
end

;definition of dcs_mslice class
pro dcs_mslice__define
    n_emp = 3                            ;number of empty cans allowed
    void={dcs_mslice,                  $
         group_leader:     0L,         $ ;group leader
         tlb:              0L,         $ ;container base of most of the widgets
         ncnrftp:          0b,         $ ;flag for allowing ncnr ftp, introduced after ftp server issue in 2021
         uname:            '',         $ ;an additional identifier
         mod_date:         '',         $ ;latest update date
         xsize:            lonarr(5),  $ ;to record the xsize
         geom:             [0L,0L],    $ ;xsize,ysize
         iconfiles:        strarr(10), $ ;to save icon file names   ['mslice.ico','mslice_scan.bmp','align.ico','ftp.ico','macs.ico','mslice_save.bmp','mslice_open.bmp','idl.ico','unlock.bmp','lock.bmp']
         view:             [1b,1b],    $ ;flag for viewing the Projection and Display section
         ssfac:            1.0e,       $ ;self shielding factor
         ssfacPtr:         ptr_new(),  $ ;if valid, precedes ssfac, points to a structure {file:'',ssfac:[ndet or nen], det_or_en: 0(det) or 1(en)}.
         temperature:      300e,       $ ;sample temperature for detailed balance calculation
         instrgeom:        0s,         $ ;instrument geometry  0:direct    1:inverse
         ftype:            0s,         $ ;file types: 0-DCS 1-SPE ...
         ftypename:        ptr_new(),  $ ;store file type names: 0-"DCS" 1-"SPE" ...
         ftypeext:         ptr_new(),  $ ;store file type extensions
         ftypecheckmask:   [0b,0b],    $ ;flag for checking mask in data files 0-no, 1-yes; for general volume data [0] for signal, [1] for background; otherwise [0] for projection [1] for raw data
         instrname:        '',         $ ;store the lower case of the current instrment name
         bgratePtr:        ptr_new(),  $ ;a pointer to the overall background rate or detector dark count array [[bgrate],[dbgrate]]
         bgeran:           [0e,0e],    $ ;background energy range
         bgtchnPtr:        ptr_new(),  $ ;pointer which points to {tchan:[start,end] qty:[end-start+1]} background information
         bgdetsPtr:        ptr_new(),  $ ;pointer which points to an [ndet] array of background
         bgdata:       ptrarr(n_emp),  $ ;pointer which points to a data structure which contains the background information, [1:2] used when extraempcan_yn = 1
                                       $ ;-single crystal,powder
                                       $ ; {qty:[ntchan,ndet], err:[ntchan,ndet]}
                                       $ ;-diffuse scattering
                                       $ ; {qty:[ndet],err:[ndet]} or {qty:[nen,ndet],err[nen,ndet]}
                                       $ ;-macs
                                       $ ; {qty:[2,ndet,ndat],err:[2,ndet,ndat],rotation:[ndat],kidney:[ndat],weight:[ndat],cfx:[ndat],Ei:[ndat],Ef:[ndat]}
                                       $ ;-general volume data  [1] for unmasked [ndat,nitem]
                                       $ ; [ndat,nitem]
         empcanfactor: fltarr(n_emp),  $ ;factors to be applied to the empty can files, used only when multiple sets are available
         intn_typ:         0s,         $ ;intensity type
                                       $ ;0-S(Q,omega) 1-Symmetrized S(Q,omega) 2-d2sigma/dOmegadEf 3-Chi(Q,omega) 4-GDOS 5-VDOS 6-raw data
         macsintn_typ:     0s,         $ ;0-spec 1-diff 2-spec+diff
         macshistomode:    0b,         $ ;0-normal 1-multi channel histogram 2-event mode histogram
         macshistowidth:   0e,         $ ;histo time channel width in the unit defined by self.macshistounit
         macshistounit:    0b,         $ ;0-us 1-ms
         macshistonchan:   0l,         $ ;number of channels
         macshistocut:     0b,         $ ;flag for throwing away the trailing channels
         macshistot0:      [0e,0e],    $ ;starting time for the histogram time channel, only used for macshistomode=2 (event mode)  [signal, empty can]
         macs_rawintn:     0b,         $ ;flag for not normalizing MACS event mode intensity, per Tim Reeder's email on 4/26/2023
                                       $ ;use sum for binning, output data will have correct summed intensity and weight, mslice plots may not make sense since they are not normalized.
         macstemp_typ:     0s,         $ ;0-sample 1-control 2-set point
         tempavg_yn:       0b,         $ ;0-no 1-yes if set, the tempertures in each file will be averaged, for MACS & WAND only
         data_tolerance:   ptr_new(),  $ ;tolerance value for general volume data
         macs_tolerance:   fltarr(3,2),$ ;macs tolerance values [energy, kidney angle, a3 angle] in [[empty can subtraction],[file loading combination]]
         macs_lamb2:       0b,         $ ;flag for MACS monitor lambda/2 correction, if this flag is on, and the CFX filter is none, lamba/2 correction will be done during projection calculation
         macs_CFXdate:     [0ll,0ll],  $ ;[date string when CFX status is included in the file, date when mgf2 filter is installed], read from dm_filesel
         macs_effmask:     [0e,0e],    $ ;a4 angle range to be ignored in the detector efficiency calculation
         macs_savea4:      0b,         $ ;flag for saving a4 in the slice,cut,and projection ascii output
         macs_ptaionly:    0b,         $ ;flag for showing PTAI only
         macs_seq_geom:    [0e,0e,0e], $ ;save the macs_sequencer text box size and layout handedness 0-right 1-left
         psi_tolerance:    [0e,0e],    $ ;tolerance value for other instruments in [empty can subtraction,file loading combination]
         empt_tol_optn:    0b,         $ ;empty can subtraction tolerance option 0:within tolerance or closest 1:within the tolerace of the closest
         intn_typ_str:     strarr(8),  $ ;save the intensity type names
         lamcorrnxspe:     0s,         $ ;flag for ki/kf correction in NXSPE files 0: no (const t bins) 1: yes (const t bins)
                                       $ ; 2 no (const E bins) 3 yes (const E bins)
         mon_sumfit:       [1s,0s],    $ ;flag for[dcs,macs] monitor count calculation method 0:sum 1:gauss fit area, for MACS 0-automatic monitor 1-specify
         mon_typ:          [1s,1s],    $ ;monitor choice for [dcs,macs], DCS: 0-fc0 1-bm1 2-duration; if 2, overall background rate is disabled MACS: 0-time 1-monitor 2-pulse number (histogram mode)
         macs_mon:         fltarr(2,3),$ ;macs monitor counts [[automatic time, specify time],[automatic monitor, specify monitor],[automatic pulse#, specify pulse#]]
         bin_avgsum:       0e,         $ ;flag for binning method -1:weighted mean 0:arithmetic mean 1:sum
         binthreshold:     0e,         $ ;treshold level for binning
         bin_conststep:    0b,         $ ;flag for constant binning intervals 0:no  1:yes
         bin_extlib:       0b,         $ ;flag for using external binning libraries 0:no 1:yes
         bin_checkstep:    1b,         $ ;flag for checking binning steps
         bin_zeroerror:    0b,         $ ;flag for how to set zero error 0: User Specify (default) 1: Statistical estimate from within binning size (default for MACS Event data only)
         bin_statrange:    intarr(5),  $ ;size for searching for non-zero counts in statistical estimate, [flag (0-auto range, 1-user specify), user specify value, flag for applying estimate to all intensities(0-no 1-yes), save the auto range found range, [cut,slice,volume,4d] auto range search index]
         zero_error:       fltarr(3),  $ ;[error bar for zero count intensity,error bar for sqrt(I=1) from data file,error bar after projection calculation]
         view_ortharbi:    0s,         $ ;flag for ortho or arbitrary viewing axis 0:orthogonal 1:arbitrary
         fold_type:        0s,         $ ;flag for folding type [0:mirror-one side 1:mirror-both sides 2:symmetry rotation-first section 3: sym rot-first quadrant 4: sym rot-360 expansion
         symrot_startvec:  0e,         $ ;angle of symmetry rotation folding starting vector away from u1, for fold_type=2
         numrot_optn:      0b,         $ ;0: minimum number of rotation to achieve 360 expansion 1: n-fold-1 rotations, for fold_type=4
         powd_typ:         0s,         $ ;0-none 1-Al 2-Cu 3-Stainless Steel 4-specify
         powd_qvalue:      ptr_new(),  $ ;save the q values for the powder line
         powd_latta:       0e,         $ ;save input lattice a value for powder line
         powd_bcc:         0b,         $ ;save stainless steel structure choice of BCC
         powd_showall:     0b,         $ ;flag for showing the powder line in full circle
         eint_yn:          0b,         $ ;flag for E-Integrated intensity
                                       $ ;if set, then in Q-cut, I=sum(I[i,j].dE[i,j])/sum(dE[i,j])*E_width
         negative2th:      0b,         $ ;flag for perserving negative 2th in powder mode
         extraempcan_yn:   0b,         $ ;flag for allowing 2 sets of empty can files
         pow2cryexpan:     0b,         $ ;flag for powder to crystal projection
         extravaxis_yn:    bytarr(3),  $ ;MACS: flag for extra viewing axis for [specify, temperature, magnetic field]
                                       $ ;DCS:  flag for extra viewing axis for [U3, temperature, magnetic field] 
                                       $ ;extravaxis_yn[0]: WAND:flag for extra viewing axis for temperature; others: flag for allowing extra viewing axis in single crystal modes
         extravaxis_spec:  ['','',''], $ ;MACS speficy extra viewing axis [name,unit,ICE file identifier]
         pbgcol_bw:        0b,         $ ;0:black 1:white for the plotting window background color
         pdim_list:        ptr_new(),  $ ;list of plot window size options
         pdim_choice:      0s,         $ ;default plot window size choice
         ppvc_choice:      0b,         $ ;plot window empty grid color choice 0-none 1-average of minimum 2 neighbors 2-average of minimum 3 neighbors
         tdir_choice:      0b,         $ ;plot axis tick direction, 0-inside 1-outside,and suppress the ticks of the other same axis
         ptrm_choice:      0b,         $ ;0-texture rendering method, 1-triangles rendering method, to address windows 10 font issue
         pnotooltip:       0b,         $ ;notooltip flag for dm_plot 0: first time starting mslice 1: after the parameter file is loaded 
         eadjust:          0s,         $ ;flag for elastic peak adjustment 0:no 1:auto, 2:specify
         eadjust_spec:     0e,         $ ;specified elastic peak position
         eadjust_checkfirst:0b,        $ ;flag for using the first file shift for all, default is no(check all files)
         userf_str:        '',         $ ;save user function strings:[macro string, user function string, user procedure string] 
         userf_before:     0b,         $ ;flag for executing the user function before the projection 0-after projection 1-before projection 2-before empty can subtraction
         userfwarning:     0b,         $ ;warning for userfunction
         errbar:           1b,         $ ;flag for showing error bar
         useitool:         0b,         $ ;flag for using itools
         datr_yn:          0b,         $ ;flag for showing dark angle trajactory
         datr_psi:         0e,         $ ;position of the dark angle with respect to u
         datr_width:       0e,         $ ;width of the dark angle, default 5 degree
         dcor_yn:          0b,         $ ;flag for dark angle correction in diffuse scattering modes
         dcor_eqran:       fltarr(4),  $ ;E and 2theta range for the dark angle correction or tchan background
         dcor_info:        ptr_new(),  $ ;a pointer points to {factor:[n_psi], psi:[n_psi]}
         edep_yn:          0b,         $ ;flag for calculating energy dep. det. efficiency 0:no 1:yes
         edep_det:         [0e,0e],    $ ;[thickness, he3 pressure] of the detector for e-dep det efficiency calculation
         bitflip:          0b,         $ ;flag for checking bitflip errors 0:no 1:yes for DCS type only
         inxerev:          0b,         $ ;flag for reverse the energey transfer definition in INX files 0:neutron energy gain/loss 1:sample energy gain/loss, for INX type only
         tthcorr:          0b,         $ ;flag for correcting dcs 2theta   0:no 1:yes for DCS type only
         mask_filler:      [0e,0e,0e], $ ;mask_filler[0]: a filler to indicate empty data in grid data such as slices or smaller than intensity mask for data browser
                                       $ ;mask_filler[1]: greater than intensity mask, mask_filler[2]: flag for masking signal only or both signal & background
         mbparm:           fltarr(4),  $ ;automatic bragg peak masking parameter [width lower limit, width upper limit, intensity lower limit, intensity lower limit(sigma)], default [0.25,4.0,0.04,3.0]
         dcsdetmask:       bytarr(5),  $ ;flag for dcs detector bank mask [negative, lower bank, central bank, upper bank,upper and lower]
         a3mask:           ptr_new(),  $ ;macs only, masking a3, an angulra range string, similar to mask when masktype = 2
         bga3mask:         ptr_new(),  $ ;macs only, masking a3 for empty can
         mask:             ptr_new(),  $ ;detector mask pointer
         maskstr:          '',         $ ;save set mask by detector number string
         masktype:         0b,         $ ;0-by det number 1-by angle    0,1-self.mask save index
         mask_ar:          ptr_new(),  $ ;detector angular mask, save angular range
         mask_darkangle:   0b,         $ ;flag for masking dark angle
         eff:              ptr_new(),  $ ;detector efficiency pointer
         eff1:             ptr_new(),  $ ;save the default macs detector efficiency originally saved in data file, only used when Load Vandadium button is pressed
                                       $ ;clear vanadium will restore its content to eff pointer
         fileSel:          obj_new(),  $ ;data file selection
         fileBas:          0L,         $ ;file selection base
         intn_lab:         '',         $ ;intensity label
         legd_lab:         '',         $ ;legend label for 1d plot
         titl_lab:         '',         $ ;title label
         lock_title:       0b,         $ ;flag for locking the title legend, if locked, loading file will not alter the title legend, can only be changed in the popup menu
         monrate:          0e,         $ ;monitor rate from dcs data file
         kfactor:          ptr_new(),  $ ;(ki/kf)
                                       $ ;-powder,single crystal,diffuse scattering multiple E
                                       $ ; [ntchan] not rebinned
                                       $ ;-diffuse scattering sinegle E
                                       $ ; 1 for elastic and all time channels ki/kf_ave for the others
         samp_typ:         1s,         $ ;0-powder, 1-single crystal, 2-diffuse scattering single E, 3-diffuse multiple E, -1:stand alone volume data slicer
         rebinsize:        ptr_new(),  $ ;allowed rebin size
         e_bin:            intarr(5),  $ ;ebin size/erange choice
                                       $ ; [0:1] e_bin size [2:3] erange choice [4] e_bin size for diffuse scattering mult. E
         e_range:          [0e,0e],    $ ;diffuse scattering energy range
         projBas:          0L,         $ ;projection base
         projrid:          [0L,0L],    $ ;saving lattice param & crystal orientation label row ids
         dispBas:          0L,         $ ;display base
         disp_flag:        bytarr(6),  $ ;flag for display items: [z1_yn,z2_yn,vol_yn,st2_yn,st3_yn,st4_yn]
         latt_parm:        dblarr(6),  $ ;lattice parameters [a,b,c,alpha,beta,gamma]
         orie_u:           [1e,0e,0e], $ ;[uh,uk,ul]
         orie_v:           [0e,1e,0e], $ ;[vh,vk,vl]
         locklattuv:       0b,         $ ;flag for keeping lattice parameters and u&v from changed by loading macs files, can only be changed in the popup menu
         checklattuv:      0b,         $ ;flag for checking lattice parameters and u&v in files for consistency, can only be changed in the popup menu
         orie_psi:         0e,         $ ;psi
         orie_offset:      0e,         $ ;save the offset angle of psi
         view_type:        strarr(6),  $ ;view  type ['Energy','|Q|','2Theta','Det Group Number','Time Channel','y']
         view_label:       strarr(6),  $ ;labels     ['E','|Q|','2Theta','Det Group', 'T', 'y']
         view_unit:        strarr(6),  $ ;units      ['(meV)','(A^-1)','(^o)','','(usec)','(A^-1)']
         view_u1typ:       1s,         $ ;powder u1 type 0-E, 1-Q, 2-2theta, 3-Det Group
         view_u1:          fltarr(3),  $ ;[u1h,u1k,u1l]
         view_u1lab:       strarr(2),  $ ;['|Q|','Qh'] -u1 label
         view_u1fold:      0b,         $ ;flag for folding u1 axis, single crystal only
         view_u1folddir:   ptr_new(),  $ ;folding direction, -1:minus, 0:auto, 1:plus
         view_u1foldcenter:ptr_new(),  $ ;folding center for u1, default is 0
         view_u2typ:       0s,         $ ;powder u2 type 0-E, 1-Q, 2-2theta, 3-Det Group
         view_u2:          fltarr(3),  $ ;[u2h,u2k,u2l]
         view_u2lab:       strarr(2),  $ ;['E','Qk']   -u2 label
         view_u2fold:      0b,         $ ;flag for folding u2 axis, single crystal only
         view_u2folddir:   ptr_new(),  $ ;folding direction, -1:minus, 0:auto, 1:plus
         view_u2foldcenter:ptr_new(),  $ ;folding center for u2, default is 0
         view_u3:          fltarr(3),  $ ;[u3h,u3k,u3l]
         view_u3lab:       '',         $ ;u3 label
         view_u3fold:      0b,         $ ;flag for folding u3 axis, single crystal only
         view_u3folddir:   ptr_new(),  $ ;folding direction, -1:minus, 0:auto, 1:plus
         view_u3foldcenter:ptr_new(),  $ ;folding center for u3, default is 0    
         d4sel_x1:         intarr(4),  $ ;x1-axis selection for 4d tab
         d4sel_x2:         intarr(4),  $ ;x2-axis selection for 4d tab
         d4sel_x3:         intarr(4),  $ ;x3-axis selection for 4d tab
         d4sel_x4:         intarr(4),  $ ;x4-axis selection for 4d tab
         d4sel_t1:         intarr(4),  $ ;thickness 1 selection for 4d tab
         d4sel_t2:         intarr(4),  $ ;thickness 2 selection for 4d tab
         d4sel_tft1:       fltarr(2),  $ ;thickness 1 range for 4d tab
         d4sel_tft2:       fltarr(2),  $ ;thickness 2 range for 4d tab   
         d4_x1ran:         fltarr(2),  $ ;x1-ran
         d4_x1step:        0e,         $ ;x1-step
         d4_x2ran:         fltarr(2),  $ ;x2-ran
         d4_x2step:        0e,         $ ;x2-step
         d4_x3ran:         fltarr(2),  $ ;x3-ran
         d4_x3step:        0e,         $ ;x3-step
         d4_x4ran:         fltarr(2),  $ ;x4-ran
         d4_x4step:        0e,         $ ;x4-step
         volsel_x:         intarr(4),  $ ;x-axis selection for volume plot
         volsel_y:         intarr(4),  $ ;y-axis selection for volume plot
         volsel_z:         intarr(4),  $ ;z-axis selection for volume plot
         volsel_t:         ptr_new(),  $ ;thickness selection for volume plot, for samp_typ=-1
         volsel_tft:       ptr_new(),  $ ;thickness range for volume plot,for samp_typ=-1
         volsel_t1:        intarr(4),  $ ;thickness 1 selection for volume tab
         volsel_t2:        intarr(4),  $ ;thickness 2 selection for volume tab
         volsel_t3:        intarr(4),  $ ;thickness 3 selection for volume tab
         volsel_tft1:      fltarr(2),  $ ;thickness 1 range for volume tab
         volsel_tft2:      fltarr(2),  $ ;thickness 2 range for volume tab
         volsel_tft3:      fltarr(2),  $ ;thickness 3 range for volume tab
         vol_xran:         fltarr(2),  $ ;x-ran
         vol_xstep:        0e,         $ ;x-step
         vol_yran:         fltarr(2),  $ ;y-ran
         vol_ystep:        0e,         $ ;y-step
         vol_zran:         fltarr(2),  $ ;z-ran
         vol_zstep:        0e,         $ ;z-step
         vol_iran:         fltarr(2),  $ ;i-ran
         vol_smooth:       1s,         $ ;smoothing level
         vol_interp:       0s,         $ ;interpolation 0: nearest neighbor 1:trilinear
         vol_color:        4s,         $ ;color map number 4-Blue/Green/Red/Yellow
         slice_x:          intarr(4),  $ ;x-axis selection of contour plot
         slice_y:          intarr(4),  $ ;y-axis selection of contour plot
         slice_t:          ptr_new(),  $ ;thickness selection of the slice for volume data
         slice_tft:        ptr_new(),  $ ;thickness range of the slice for volume data
         slice_xran:       fltarr(2),  $ ;x range [from,to]
         slice_xstep:      0e,         $ ;x step
         slice_yran:       fltarr(2),  $ ;y range [from,to]
         slice_ystep:      0e,         $ ;y step
         slice_t1:         intarr(4),  $ ;slice thickness 1 slection for tab
         slice_t2:         intarr(4),  $ ;slice thickness 2 slection for tab
         slice_t3:         intarr(4),  $ ;slice thickness 3 slection for tab
         slice_t4:         intarr(4),  $ ;slice thickness 4 slection for tab
         slice_t1ran:      fltarr(2),  $ ;thickness 1 range[from,to]
         slice_t2ran:      fltarr(2),  $ ;thickness 2 range[from,to]
         slice_t3ran:      fltarr(2),  $ ;thickness 3 range[from,to]
         slice_t4ran:      fltarr(2),  $ ;thickness 4 range[from,to]
         slice_iran:       fltarr(2),  $ ;intensity range[from,to]
         slice_smooth:     0s,         $ ;smooth level
         slice_sm_kernel:  fltarr(3,3),$ ;default smooth kernel
         slice_shading:    0s,         $ ;shading 0-flat 1-Gouraud
         slice_cmap:       ptr_new(),  $ ;color map names
         slice_color:      13s,        $ ;color map number 13-RAINBOW
         allow_voronoi:    0b,         $ ;flag for using Voronoi diagram
         slice_cutcolor:   0b,         $ ;0-use default, 1-use black or white
         cut_x:            intarr(4),  $ ;cut along axis number
         cut_ran:          fltarr(2),  $ ;cut range
         cut_step:         0e,         $ ;cut step
         cutsel_t:         ptr_new(),  $ ;thickness selection of the cut for volume data
         cutsel_tft:       ptr_new(),  $ ;thickness range of the cut for volume data
         cut_t1:           intarr(4),  $ ;thickness 1 selection
         cut_t2:           intarr(4),  $ ;thickness 2 selection
         cut_t3:           intarr(4),  $ ;thickness 3 selection
         cut_t4:           intarr(4),  $ ;thickness 4 selection
         cut_t5:           intarr(4),  $ ;thickness 5 selection
         cut_t1ran:        fltarr(2),  $ ;thickness 1 range
         cut_t2ran:        fltarr(2),  $ ;thickness 2 range
         cut_t3ran:        fltarr(2),  $ ;thickness 3 range
         cut_t4ran:        fltarr(2),  $ ;thickness 4 range
         cut_t5ran:        fltarr(2),  $ ;thickness 5 range
         cut_y:            intarr(4),  $ ;y axis number
         cut_yran:         fltarr(2),  $ ;y range
         cut_symbol:       0s,         $ ;plot symbol number
         qoffsetthreshold: 0e,         $ ;qoffset range threshold for renaming axis titles,default 0.5
         symbols:          ptr_new(),  $ ;symbol names
         cut_color:        0s,         $ ;color number
         colors:           ptr_new(),  $ ;color names
         coldefine:        bytarr(3),  $ ;user defined color
         cut_line:         0s,         $ ;linestyle number,default solid line
         lines:            ptr_new(),  $ ;linestyle names
         traj_type:        strarr(5),  $ ;traj type names
         traj_x:           intarr(4),  $ ;trajectory x type
         traj_xran:        fltarr(2),  $ ;[xfrom,xto]
         traj_y:           intarr(4),  $ ;trajectory y type
         traj_yran:        fltarr(2),  $ ;[yfrom,yto]
         traj_z1:          intarr(4),  $ ;trajectory z type
         traj_z1ran:       fltarr(2),  $ ;[zfrom,zto]
         traj_z2:          intarr(4),  $ ;trajectory z type
         traj_z2ran:       fltarr(2),  $ ;[zfrom,zto]
         traj_layer:       0s,         $ ;for trajectory overplot coloring
         psi_spe:          0b,         $ ;flag indicating the definition of psi in the detPosPtr is SPE type or not, 1: yes 0: no
         psi_0only:        0b,         $ ;flag indicating whether there's out-of-plane detector
         detPosPtr:        ptr_new(),  $ ;pointer which points to a structure
                                       $ ;{two_theta:[ndet], dtwo_theta:[ndet], psi:[ndet], dpsi:[ndet]}
         dataStrPtr:       ptr_new(),  $ ;pointer which points to a data structure
                                       $ ;-single crystal,powder
                                       $ ; {qty:[ntchan,ndet], err:[ntchan,ndet], e:[ntchan],weight:weight,eief:eief,ewid:ewid[ntchan],$
                                       $ ;  info:{ch_wl:ch_wl,ch_ms:ch_ms,ch_srdenom:ch_srdenom,ch_res:ch_res,tsdmin:tsdmin,$
                                       $ ;  temperature:temperature,time:[ntchan]}}    time is only for dcs data
                                       $ ;-diffuse scattering
                                       $ ; {qty:[ndet,nfiles],err:[ndet,nfiles],rotation:[nfiles],weight:[nfiles],eief:eief,ewid:ewid[ntchan],$
                                       $ ;  info:{ch_wl:ch_wl,ch_ms:ch_ms,ch_srdenom:ch_srdenom,ch_res:ch_res,tsdmin:tsdmin,$
                                       $ ;  temperature:temperature,filename:open_file,filepath:open_path}
                                       $ ;-MACS
                                       $ ; {qty:[2,ndet,ndat],err:[2,ndet,ndat],rotation:[ndat],kidney:[ndat],weight:[ndat],cfx:[ndat],ptai:[ndat],Ei:[ndat],Ef:[ndat],temperature:[ndat],$
                                       $ ;  hfield:[ndat],info:{latticeparm:latticeparm,latticeori:latticeori}}
                                       $ ;-general volume data, a copy of unmasked dataPtr
         projStrPtr:       ptr_new(),  $ ;pointer which points to a structure
                                       $ ;-single crystal
                                       $ ; {V:[3,nen,ndet],en:[nen],two_theta:[ndet],qty:[nen,ndet],err:[nen,ndet],ewid:[nen],temperature:[nen]} 
                                       $ ; temperature is only for MACS with extra view axis
                                       $ ;-powder
                                       $ ; {Q:[nen,ndet],en:[nen],two_theta:[ndet],qty:[nen,ndet],err:[nen,ndet],ewid:[nen],time:[nen]}
                                       $ ;-diffuse scattering
                                       $ ; single E
                                       $ ; {V:[3,ndet,nfile],en:[1],two_theta:[ndet],qty:[ndet,nfile],err:[ndet,nfile],ewid:[1],temperature:[nfile],<daQV:[nproj,npt]>}
                                       $ ; temperature is only for WAND with extra view axis
                                       $ ; multiple E
                                       $ ; {V:[3,nen,ndet,nfile],en:[nen],two_theta:[ndet],qty:[nen,ndet,nfile],err:[nen,ndet,nfile],ewid:[nen],<daQV:[nproj,npt]>}
         dataPtr:          ptr_new(),  $ ;general volume data pointer
                                       $ ;{data:[ndat,nitem],label:[nitem],unit:[nitem],is_uniq:[nitem],instrument:'',title:'',legend:'',axis_info:[nitem-2],weight:[ndat]} 
                                       $ ; the first two item of data needs to be I and dI
         weightPtr:        ptrarr(2),  $ ;save weight data for general volume data  [1] for unmasked
         bgweightPtr:      ptrarr(2),  $ ;save background weight data for general volume data  [1] for unmasked
         powd_line:        ptr_new(),  $ ;powder line for diffuse scattering mode
         absp_info:        ptr_new(),  $ ;absorption information [type,sampling pts,interoplate,equator_only,l0,l1,Rin/width,Rout/thickness,angle]
                                       $ ; type: 0-cylinder 1-annulus 2-sphere 3-flat plate
         absp_npts:        ptrarr(4),  $ ;saving npts choice option numbers
         absp_nptst:       ptrarr(4),  $ ;saving npts choice option strings
         absp_asint:       0b,         $ ;if set, use transmission as intensity
         absp_l0l1:        strarr(5),  $ ;save l0 & l1 calculation defaults [formula,rho,n_formula, l0, l1] 
         absp_l1factor:    0e,         $ ;additional overall incident beam path factor, calculated incident beam path will be multiplied by this factor
         file_hist:        ptrarr(20), $ ;save the latest 10 file selections.(10:19 for session file history, 10:12 bgfiles, 13:19 load and add buttons)   
         file_recent:      ptr_new(),  $ ;save the file names of the most recent load, [0] save the file directory
         n_history:        0s,         $ ;number of histories
         gr_list:          obj_new(),  $ ;container of all the graphic objects
         current:          obj_new(),  $ ;current plot object
         crossplot:        0b,         $ ;flag for allowing overplotting from different mslices
         error:            0b,         $ ;error flag
         helpfile:         ['',''],    $ ;help file
         dirs:             ['','',''], $ ;[data directory, work directory, parameter file directory]
         parmname:         '',         $ ;default parameter file name
         pathsep:          '',         $ ;path separator
         symfont:          '',         $ ;symbol font name
         mjqy:             0e,         $ ;save the mass number for J(Q,y) intensity type
         eief:             0e,         $ ;save fixed energy, Ei for direct geometry, Ef for inverse geometry
         estartend:        fltarr(3),  $ ;save [e_start, e_end, n_en] for projection calculation, only used when user function is specified 
         phxfile:          '',         $ ;save spe detector info phx file name
         psi:              fltarr(4),  $ ;save psi start,end,step info, psi[3] saves the first file angle
         kidney:           strarr(4),  $ ;save macs kidney [start,end,step,optimization flag] info for projection calculation
         kidney_offset:    0e,         $ ;save macs kidney offset angle
         kidney_backlash:  0e,         $ ;save macs kidney backlash angle, if NAN, use 2 degrees
         whatsnew:         ptr_new(),  $ ;what's new infomation
         info:             ptr_new(),  $ ;self information
         idl_version:      0e,         $ ;IDL version number
         ftoolflag:        bytarr(7),  $ ;flag for filetools: [0] resolution convolution tool 0-never started 1-started; [1] macs deteff changed; [2] voronoi warning; [3] macs bad detector warning; 
                                       $ ;[4] data file combined; [5] empty can subtraction a3 tolerance too large for macs event mode data; [6] add data tolerance for macs event mode data
         txtfocusevent:    0b,         $ ;flag for text focus event
         DAVETool:         obj_new()   $ ;from DAVE
    }
end

pro dcs_mslice,group_leader=group_leader,workDir=workDir,dataDir=dataDir,DAVETool=DAVETool,dataFile=dataFile,dataPtr=dataPtr,instrname=instrname,no_parmfile=no_parmfile,obj_mslice=obj_mslice,uname=uname,_extra=extra
    if n_elements(DAVETool) eq 0 then davesysvars   ;for necessary DAVE system variables
    obj_mslice = obj_new('dcs_mslice',group_leader=group_leader,uname=uname,workDir=workDir,dataFile=dataFile,dataDir=dataDir,DAVETool=DAVETool,dataPtr=dataPtr,instrname=instrname,no_parmfile=no_parmfile)
end