; $Id: $
;#######################################################################
;
; NAME:
;  dcs_cryst_align
;
; PURPOSE:
;  This program plots crystal alignment files
;
; CATEGORY:
;  DCS data tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  September, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

@dm_load_macs         ;for loading phades file
@dm_macs_sequencer    ;for several MACS utilities
@dm_calc_projection   ;for several routines defined there

pro dcs_cryst_align_e2lambda_event,event
    widget_control,/hourglass
    widget_control,event.handler,get_uvalue=state

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       return
    end
    
    case event.id of
       state.eTxt:     dcs_cryst_align_e2lambda_update,state,/E
       state.lTxt:     dcs_cryst_align_e2lambda_update,state,/lambda
       state.a2Txt:    dcs_cryst_align_e2lambda_update,state,/A2
       state.plotBut:  DCSToolsPerformance,group_leader=event.top
       state.exitBut:  begin
                       widget_control,event.handler,/destroy
                       return
                       end
       else:
    endcase
 
    widget_control,event.handler,set_uvalue=state
end

pro dcs_cryst_align_e2lambda_update,state,E=E,lambda=lambda,A2=A2
    eng = !values.f_nan & wav = !values.f_nan  & atu = !values.f_nan ;initialization
    ;retrieve values from text boxes
    current_id = 0>(keyword_set(lambda)+2*keyword_set(A2))<2
    widget_control,([state.eTxt,state.lTxt,state.a2Txt])[current_id],get_value=str
    if n_elements(str) eq 0 then return
    if state.current[current_id] eq strtrim(str,2) then return
    tmp = dm_to_number(strsplit(str,' ,'+string(9b),/extract))
    ind0 = where(finite(tmp),count0)
    if count0 gt 0 then begin
       ind1 = where(tmp[ind0] gt 0,count1)
       if count1 gt 0 then begin
          if keyword_set(E) then begin
             eng = tmp[ind0[ind1]]
             wav = dm_e2lambda(eng)
             atu = asin(wav/2.0/3.35416)/!dtor*2
             widget_control,state.lTxt,set_value=dm_to_string(wav,resolution=3,separator=', ')
          endif
          if keyword_set(lambda) then begin
             wav = tmp[ind0[ind1]]
             eng = dm_lambda2e(wav) 
             atu = asin(wav/2.0/3.35416)/!dtor*2
             widget_control,state.eTxt,set_value=dm_to_string(eng,resolution=3,separator=', ')
          endif
          if keyword_set(a2) then begin
             atu = tmp[ind0[ind1]]
             wav = sin(atu/2.0*!dtor)*3.35416*2
             eng = dm_lambda2e(wav)
             widget_control,state.lTxt,set_value=dm_to_string(wav,resolution=3,separator=', ')
             widget_control,state.eTxt,set_value=dm_to_string(eng,resolution=3,separator=', ')
          endif
       endif
       if (state.instr eq 2) and (~keyword_set(A2)) then widget_control,state.a2Txt,set_value=dm_to_string(atu,res=3,separator=', ') ;MACS
    endif
    
    wav = wav[0] & eng = eng[0] & atu = atu[0] ;only the first one  
    if finite(eng) and finite(wav) then begin
       if (state.instr eq 1) then begin ;for DCS, figure out the energy resolution and flux 
          f_low  = interpol((*(state.flPtr))[1,*],(*(state.flPtr))[0,*],wav)/state.fmax*100.0
          f_med  = interpol((*(state.fmPtr))[1,*],(*(state.fmPtr))[0,*],wav)/state.fmax*100.0
          flux   = 'low:'+strtrim(string(0>(f_low)<100,format='(f6.2)'),2)+', '+$
                   'med:'+strtrim(string(0>(f_med)<100,format='(f6.2)'),2)
          widget_control,state.fxTxt,set_value=flux
          er_low = 13220./wav^2.947
          er_med = 3948./wav^2.656
          eresl  = 'low:'+strtrim(string(er_low,format='(f9.1)'),2)+', '+$
                   'med:'+strtrim(string(er_med,format='(f9.1)'),2)
          widget_control,state.erTxt,set_value=eresl 
       endif   
    endif else begin
       if (state.instr eq 1) then begin ;DCS
          widget_control,state.fxTxt,set_value=''
          widget_control,state.erTxt,set_value=''
       endif
       if keyword_set(E) then      widget_control,state.lTxt,set_value=''
       if keyword_set(lambda) then widget_control,state.eTxt,set_value=''
       if (state.instr eq 2) and (~keyword_set(A2)) then widget_control,state.a2Txt,set_value=''
    endelse
    if (state.instr eq 2) and keyword_set(A2) and ~finite(atu) then begin
       widget_control,state.lTxt,set_value=''
       widget_control,state.eTxt,set_value=''
    endif
    widget_control,state.eTxt,get_value=str & state.current[0] = strtrim(str,2)
    widget_control,state.lTxt,get_value=str & state.current[1] = strtrim(str,2)
    if state.instr eq 2 then begin
       widget_control,state.a2Txt,get_value=str & state.current[2] = strtrim(str,2)
    endif
end

pro dcs_cryst_align_e2lambda_Exit,tlb
    widget_control,tlb,get_uvalue=state
    ptr_free,state.flPtr,state.fmPtr
    widget_control,tlb,/destroy
end

;E to Lambda conversion
pro dcs_cryst_align_e2lambda,event,lambda=lambda,ei=ei,dcs=dcs,macs=macs
    state={group_leader:    0L, $   ;group leader
           tlb:             0L, $   ;top level base
           instr:           0b, $   ;flag for instrument type: 0-other 1-dcs 2-macs
           eTxt:            0L, $   ;E text box
           lTxt:            0L, $   ;lambda text box
           erTxt:           0L, $   ;E resolution text box
           fxTxt:           0L, $   ;flux text box
           a2Txt:           0L, $   ;A2 text box
           current:   strarr(3),$   ;save current E, lambda, A2 string to avoid unnecessary updates
           plotBut:         0L, $   ;start performance button
           exitBut:         0L, $   ;exit button
           fmax:            0e, $   ;maximum flux
           flPtr:    ptr_new(), $   ;low resl mode flux
           fmPtr:    ptr_new()  $   ;medium resl mode flux
          }
    registerName = 'dcs_cryst_align_e2lambda'
    if xregistered(registerName) then begin ;only allow one copy running
       id = LookupManagedWidget(registername)
       widget_control,id,iconify=0
       return
    endif
    state.instr = keyword_set(dcs)
    if keyword_set(macs) then state.instr = 2
    if n_elements(ei) eq 0 then ei = !values.f_nan
    if n_elements(lambda) eq 0 then lambda = !values.f_nan
    if n_elements(event) ne 0 then state.group_leader = event.top
    state.tlb   = widget_base(title='Convert Energy to Wavelength',/col,kill_notify='dcs_cryst_align_e2lambda_Exit',tlb_frame_attr=1,group_leader=state.group_leader)
    row1        = widget_base(state.tlb,/row,/grid_layout)
    col1        = widget_base(row1,/col,/grid_layout)
    col2        = widget_base(row1,/col,/grid_layout)
    if (state.instr eq 2) then begin
    col3        = widget_base(row1,/col,/grid_layout)
    void        = widget_label(col3,value='A2('+string(176b)+')',/align_center)
    state.a2Txt = widget_text(col3,/editable,/all_events)
    void        = widget_label(state.tlb,value='A2 is calculated using PG (002) d-spacing of 3.35416 '+string('c5'xb)+'.',/align_center)
    endif
    void        = widget_label(col1,value='E (meV)',/align_center)
    state.eTxt  = widget_text(col1,/editable,/all_events,value=dm_to_string(ei))
    void        = widget_label(col2,value='Lambda ('+string('c5'xb)+')',/align_center)
    state.lTxt  = widget_text(col2,/editable,/all_events,value=dm_to_string(lambda))
    if (state.instr eq 1) then begin
    void        = widget_label(col1,value='FWHM ('+string('b5'xb)+'eV)',/align_center)
    state.erTxt = widget_text(col1)
    void        = widget_label(col2,value='Flux (% of maximum)',/align_center)
    state.fxTxt = widget_text(col2)
    endif
    row2        = widget_base(state.tlb,/row,/align_center)
    if (state.instr eq 1) then begin
    state.plotBut = widget_button(row2,value='  Plot  ',sensitive=(xregistered('dave',/noshow) or xregistered('wd_DAVETool',/noshow)))
    void        = widget_label(row2,value= '   ')
    endif
    state.exitBut = widget_button(row2,value='  Exit  ')
    ;dcs flux info
    state.flPtr = ptr_new([[1.25,357], [1.3,1799.4], [1.35,30630], [1.36,44460], [1.38,69780], [1.4,86595], [1.42,86820], [1.44,59370], [1.45,57570], [1.46,73200],$
                    [1.48,117900], [1.5,137498], [1.52,143280], [1.54,145380], [1.55,135750], [1.56,106530], [1.58,33600], [1.6,34440], [1.62,122010], [1.64,177630],$
                    [1.65,172500], [1.66,158280], [1.68,167760], [1.7,212850], [1.72,165270], [1.74,68310], [1.75,51450], [1.76,71100], [1.78,186870], [1.8,266385],$
                    [1.82,265980], [1.84,216420], [1.85,177360], [1.86,131010], [1.88,43860], [1.9,4959], [1.95,3192], [1.96,7602], [1.98,32730], [2,87915],$
                    [2.02,116220], [2.04,60900], [2.05,28993.5], [2.06,10188], [2.08,820.8], [2.1,167.22], [2.15,385.5], [2.2,196260], [2.25,541500], [2.3,720400],$
                    [2.4,803100], [2.5,875400], [2.6,905700], [2.7,912600], [2.8,902400], [2.9,898500], [3,901350], [3.1,888300], [3.2,876600], [3.25,855900],$
                    [3.27,826800], [3.3,760800], [3.33,390900], [3.35,346200], [3.38,731100], [3.4,831900], [3.42,839400], [3.45,846000], [3.5,861300], [3.6,870300],$
                    [3.7,877800], [3.8,859500], [3.85,844500], [3.9,832200], [3.95,807000], [3.98,781200], [4,763800], [4.03,734700], [4.05,787800], [4.08,850500],$
                    [4.1,872400], [4.15,857400], [4.2,847200], [4.3,828900], [4.4,798300], [4.5,765900], [4.6,738000], [4.62,719400], [4.65,716400], [4.67,746400],$
                    [4.7,780000], [4.72,777900], [4.75,772800], [4.78,766200], [4.8,763950], [5,721650], [5.5,645900], [6,558000], [6.3,508200], [6.4,487500],$
                    [6.45,466500], [6.5,453600], [6.55,405600], [6.6,297180], [6.65,33900], [6.7,24780], [6.75,279690], [6.8,384600], [6.85,402600], [6.9,413700],$
                    [6.95,406200], [7,404850], [7.5,349800], [8,292040], [9,199770], [10,142250], [12,69260], [14,34890], [15,25104], [16,17742],$
                    [18,9459], [20,5184]])
    state.fmPtr = ptr_new([[1.35,3894], [1.4,11083.5], [1.45,4354.5], [1.5,17070], [1.55,17355], [1.6,1409.55], [1.65,22560], [1.7,27450], [1.75,3330], [1.8,34275],$
                    [1.85,22425], [1.9,207.45], [1.95,322.2], [2,11304], [2.05,1801.5], [2.1,130.185], [2.15,150.9], [2.2,28005], [2.25,72900], [2.3,92160],$
                    [2.4,102165], [2.5,112905], [2.6,117765], [2.7,120540], [2.8,117645], [2.9,119475], [3,122535], [3.2,117090], [3.25,114390], [3.3,103335],$
                    [3.35,31950], [3.4,112305], [3.5,116445], [3.6,120810], [3.7,120810], [3.8,119025], [3.85,118320], [3.9,115035], [3.95,111540], [4,105735],$
                    [4.05,106050], [4.1,123960], [4.15,121905], [4.2,121185], [4.3,117795], [4.4,113085], [4.6,103245], [4.8,110460], [5,103703], [5.5,91095],$
                    [6,77842.5], [6.5,59820], [6.6,36780], [6.7,952.5], [6.8,49335], [6.9,53190], [7,54127.5], [7.5,43965], [8,36540], [9,24315],$
                    [10,16657.5], [12,7679.25], [14,3831], [16,1926], [18,1006.58], [20,554.55]])
    state.fmax = max((*(state.flPtr))[1,*])
    dm_center_kid,state.tlb,state.group_leader
    widget_control,state.tlb,/realize
    if finite(ei) then dcs_cryst_align_e2lambda_update,state,/e $
    else if finite(lambda) then dcs_cryst_align_e2lambda_update,state,/lambda
    widget_control,state.tlb,set_uvalue=state
    xmanager,registerName,state.tlb,cleanup='dcs_cryst_align_e2lambda_Exit',/no_block
end

;build context sensitive menu
pro dcs_cryst_align_csm,pState
    center = widget_info((*pState).tlb,find_by_uname='center')
    for i=0,1 do begin
        csm = widget_info(center,find_by_uname='contextevent'+dm_to_string(i))
        if csm eq 0 then csm = widget_base(center,/context_events,uname='contextevent'+dm_to_string(i),xpad=0,ypad=0,space=0)
        void = widget_label(csm,value=' ',sensitive=0)
        mid  = widget_base(csm,/context_menu,uname='contextmenu')
        void = dm_widget_button(mid,value='Show Cursor Position',uname='optnshowcursor')
        void = dm_widget_button(mid,value='Show Fitting Parameters',uname='optnshowfit')
        void = dm_widget_button(mid,value='Fix Fitting Width',uname='optnfixwidth')
        void = dm_widget_button(mid,value='Extrapolate Multi-peak Backgrounds',uname='fit_extrap')
        srtm = widget_button(mid,value='Sort Files by',/menu,/separator,uname='srtmenu')
        void = dm_widget_button(srtm,value='File Number',uname='sortnumb')
        void = dm_widget_button(srtm,value='File Name',uname='sortname')
        fitm = widget_button(mid,value='Fitting Function',/menu,uname='fitmenu')
        void = dm_widget_button(fitm,value='Gaussian',uname='fit_gaussian')
        void = dm_widget_button(fitm,value='Lorentzian',uname='fit_lorentzian',sensitive=(*pState).mpfit_exist)
        void = dm_widget_button(fitm,value='Moffat',uname='fit_moffat',sensitive=(*pState).mpfit_exist)
        for i=0,3 do void = dm_widget_button(fitm,value=(*pState).fitfunname[i,(*pState).fitfunchoice],uname='fitfun_'+dm_to_string(i+3),separator=(i eq 0))
    endfor
end

;return detector number for a 2theta, bank keyword for dcs (-1:l, 0:c, 1:u) and preference over positive central bank det
function dcs_cryst_align_detnum,pState,tth,bank=bank,tolerance=tolerance
    abstth = abs(*(*pState).tthPtr)
    mintth = min(abstth,max=maxtth)
    if n_elements(tolerance) eq 0 then tolerance = 0.
    if (abs(tth) lt mintth-tolerance) or (abs(tth) gt maxtth+tolerance) then return,-1
    tmp1 = min(abs(abstth-abs(tth)),idet)
    tmp2 = min(abs(*(*pState).tthPtr-tth),idet1)  ;same sign of angles
    if tmp2 le tmp1 then idet = idet1
    if (*pState).ftype eq 'DCS' then begin
       tmp1 = [idet[0],idet[0]+1,idet[0]+2]<(n_elements(abstth)-1)
       tmp2 = dm_mask_detbanks(detnum=tmp1)
       tmp3 = where(tmp2 eq 0,count)
       idet = tmp1[tmp3[0]*(count ne 0)]
       bank = tmp2[tmp3[0]*(count ne 0)]
    endif
    return,idet[0]
end

;return text string of fit info
function dcs_cryst_align_fitinfo,pState
    if ~ptr_valid((*pState).fitfunstack) then return,''
    (*pState).plotWin->getproperty,xtit=xtit
    tmp = stregex(xtit,'[a-z0-9]+.*\((.+)\) *$',/subexpr,/fold_case,/extract)
    if strlen(tmp[1]) ne 0 then unit = ' '+tmp[1] else unit = ''
    if (*pState).fitfunchoice eq 0 then fwhm = (*(*pState).fitfunstack)[*].param[2]*sqrt(8.*alog(2)) $          ;gaussian
    else if (*pState).fitfunchoice eq 1 then fwhm = (*(*pState).fitfunstack)[*].param[2]*2.0 $                  ;lorentzian
    else fwhm = (*(*pState).fitfunstack)[*].param[2]*2.0*sqrt(2^(1/(*(*pState).fitfunstack)[*].param[3])-1.0)   ;moffat
    if n_elements(fwhm) eq 1 then tmp = ['',unit] else tmp = ['[',']']
    text = 'peak!l !n='+tmp[0]+dm_to_string((*(*pState).fitfunstack)[*].param[1],resolution=3,sep=',')+tmp[1]+'!Cfwhm='+tmp[0]+dm_to_string(fwhm,resolution=3,sep=',')+tmp[1]
    if ((*pState).xaxis eq ([0,4])[((*pState).ftype eq 'MACS') or ((*pState).ftype eq 'TAS')]) and finite((*pState).tth[(*pState).ftype eq 'MACS']) then begin
       tth = (*pState).tth[(*pState).ftype eq 'MACS']
       if (*pState).ftype eq 'WAND' then tth = -tth  ;WAND
       if tth ge 0 then a2 = 90.-tth/2. else a2 = -(90.+tth/2.)
       if (*pState).ftype eq 'MACS' then a2 = 90-a2  ;MACS
       if ((*pState).ftype eq 'DCS') or ((*pState).ftype eq 'WAND') then $  ;only show reset angle for DCS and WAND for now
          text = text+'!C'+(*(*pState).psiname)[(where(*(*pState).fname eq (*pState).ftype))[0]]+' reset to '+dm_to_string(a2,resolution=3)+string('b0'xb)
    endif  
    if (*pState).showfit then begin
       if n_elements((*(*pState).fitfunstack)) eq 1 then begin
          res1 = (1>(ceil(2-alog10(min(abs((*(*pState).fitfunstack).sigma[where((*(*pState).fitfunstack).sigma ne 0)]))+1e-7))))
          res2 = (1>(ceil(2-alog10(abs((*(*pState).fitfunstack).chisq)+1e-7))))
          text = text+'!C!Dparams=['+dm_to_string((*(*pState).fitfunstack).param,sep=', ')+']!C!Derror=['+dm_to_string((*(*pState).fitfunstack).sigma,sep=', ',res=res1)+$
                 ']!C!D\chi!s!e2!r!ired!d='+dm_to_string((*(*pState).fitfunstack).chisq,res=res2)+'!n'
          if (*(*pState).fitfunstack).peaktime ne 0 then text = text+'!C!Dpeak time: '+dm_to_string((*(*pState).fitfunstack).peaktime,/date,/est)
       endif else begin
          for i=0,n_elements((*(*pState).fitfunstack))-1 do text = text+'!C!Dparams=['+dm_to_string((*(*pState).fitfunstack)[i].param,sep=', ')+']'
       endelse
    endif 
    return,text
end

;calculate the fitting line data
pro dcs_cryst_align_fitline,pState,params,x,xx=xx,yy=yy,xmin=xmin,xmax=xmax
    nfit = n_elements(params[0,*]) & is_moffat = ((*pState).fitfunchoice eq 2) & nx = 60 
    if n_elements(xx) eq 0 then begin
       if (n_elements(xmin) eq 0) or (n_elements(xmax) eq 0) then xmax = max(x,min=xmin)
       x0 = xmin+findgen(nx)/(nx-1.0)*(xmax-xmin)*1.2-0.1*(xmax-xmin)
       for i=0,nfit-1 do begin
           fwhm = ([params[2,i]*sqrt(8.*alog(2)),params[2,i]*2.0,params[2,i]*2.0*sqrt(2^(1/params[2+is_moffat,i])-1.0)])[(*pState).fitfunchoice]
           if (params[2,i] ne 0) and (abs(fwhm) lt abs(xmax-xmin)) then begin
              tmpxx = params[1,i]-2*fwhm+findgen(nx)/(nx-1.)*4*fwhm
              if n_elements(xx) eq 0 then xx = tmpxx else xx=[xx,tmpxx]
              id = where(x0 lt min(xx,max=tmpmax),count)
              if count gt 0 then xx = [x0[id],xx]
              id = where(x0 gt tmpmax,count)
              if count gt 0 then xx = [xx,x0[id]]
           endif
       endfor
       if n_elements(xx) eq 0 then xx = x0 else xx = xx[uniq(xx,sort(xx))]
    endif
    yy = fltarr(n_elements(xx))
    if nfit gt 1 then begin  ;sort peak positions, and rearange fitfunstack
       tmp = sort(params[1,*])
       if n_elements((*(*pState).fitfunstack)) eq nfit then (*(*pState).fitfunstack) = (*(*pState).fitfunstack)[tmp]
    endif
    for i=0,nfit-1 do begin
        if params[2,i] ne 0 then begin
           case (*pState).fitfunchoice of
              0:  yy = yy+params[0,i]*exp(-0.5*((xx-params[1,i])/params[2,i])^2)
              1:  yy = yy+params[0,i]/(1.+((xx-params[1,i])/params[2,i])^2)
              2:  yy = yy+params[0,i]/(1.+((xx-params[1,i])/params[2,i])^2)^params[3,i]
           endcase
        endif
        if (nfit eq 1) or (~(*pState).fit_extrap) then begin
           if (*pState).fitfunnterm ge 4 then yy = yy+params[3+is_moffat,i]
           if (*pState).fitfunnterm ge 5 then yy = yy+params[4+is_moffat,i]*xx
           if (*pState).fitfunnterm ge 6 then yy = yy+params[5+is_moffat,i]*xx^2
        endif
    endfor
    if (nfit gt 1) and (*pState).fit_extrap then begin
       if (*pState).fitfunnterm ge 4 then yy = yy+interpol(params[3+is_moffat,*],params[1,*],xx)
       if (*pState).fitfunnterm ge 5 then yy = yy+interpol(params[4+is_moffat,*],params[1,*],xx)*xx
       if (*pState).fitfunnterm ge 6 then yy = yy+interpol(params[5+is_moffat,*],params[1,*],xx)*xx^2
    endif
end

pro dcs_cryst_align_loadmacs,filename,hs=hs,ks=ks,ls=ls,en=en,a1=a1,a2=a2,a3=a3,a4=a4,a5=a5,a6=a6,pt=pt,mr=mr,mt=mt,ds=ds,mb=mb,bs=bs,as=as,sx=sx,sy=sy,sz=sz,sl=sl,su=su,ht=ht,vt=vt,dm=dm,fc=fc,$
    b1=b1,b2=b2,vh=vh,vv=vv,kn=kn,hf=hf,t1=t1,t2=t2,t3=t3,ts=ts,te=te,wt=wt,it=it,dit=dit,cfx=cfx,duration=duration,pu=pu,xid=xid,yid=yid,did=did,macs_cfxdate=macs_cfxdate,maspacing=maspacing,$
    deteff=deteff,error=error,avgtemp=avgtemp,montype=montype,latticeparm=latticeparm,latticeori=latticeori,group_leader=group_leader,histodata=histodata
    if n_elements(it)  ne 0 then tmp = temporary(it)
    if n_elements(dit) ne 0 then tmp = temporary(dit)
    ;read the macs file
    dm_load_macs,filename,data,colinfo,maspacing=maspacing,deteff=deteff,header=header,latticeparm=latticeparm,latticeori=latticeori,error=error,group_leader=group_leader,histodata=histodata,$
       data_histo=data_histo,dqty_histo=dqty_histo,pulse_histo=pu,/cut_tail
    if keyword_set(histodata) and (n_elements(data_histo) eq 0) then tmp = temporary(data)
    if n_elements(data) eq 0 then return
    n_data  = n_elements(data[*,0])
    x_info  = ['H','K','L','E','A3','A4','A1','A2','A5','A6','MonRot','MonTrans','DFMDTS','MBTSlide','MonBlade'+string(1+indgen(21),format='(i02)'),'AnalyzerTheta'+string(1+indgen(20),format='(i02)'),$
               'Smpl'+['X','Y','Z','LTilt','UTilt'],'HSlit','VSlit','DMBT','Focus','Beta1','Beta2','VBAH','VBAV','MagField','Temp']
    ind_qty = lonarr(2,21)
    ind_err = lonarr(21)
    for j=1,20 do begin
        tmp = string(j,format='(i02)')
        ind_qty[0,j-1] = (where(colinfo eq 'spec'+tmp))[0]
        ind_qty[1,j-1] = (where(colinfo eq 'diff'+tmp))[0]
        ind_err[j-1]   = (where(colinfo eq 'spec'+tmp+'err'))[0]
    endfor
    readerr = (total(ind_err[0:19] ge 0) eq 20)   ;flag for reading intensity error bars
    ind_qty[0,20] = (where(colinfo eq 'spec'))[0]
    ind_qty[1,20] = (where(colinfo eq 'diff'))[0]
    ind_err[20]   = ind_err[0]
    ind_hs = (where(colinfo eq 'h',cnt_hs))[0]
    ind_ks = (where(colinfo eq 'k',cnt_ks))[0]
    ind_ls = (where(colinfo eq 'l',cnt_ls))[0]
    ind_en = (where(colinfo eq 'e',cnt_en))[0]
    ind_a1 = (where(colinfo eq 'a1',cnt_a1))[0]
    ind_a2 = (where(colinfo eq 'a2',cnt_a2))[0]
    ind_a3 = (where(colinfo eq 'a3',cnt_a3))[0]
    ind_a4 = (where(colinfo eq 'a4',cnt_a4))[0]
    ind_a5 = (where(colinfo eq 'a5',cnt_a5))[0]
    ind_a6 = (where(colinfo eq 'a6',cnt_a6))[0]
    ind_sl = (where(colinfo eq 'smplltilt',cnt_sl))[0]
    ind_su = (where(colinfo eq 'smplutilt',cnt_su))[0]
    ind_ht = (where(colinfo eq 'hslit',cnt_ht))[0]
    ind_vt = (where(colinfo eq 'vslit',cnt_vt))[0]
    ind_wt = (where(colinfo eq 'monitor',cnt_wt))[0]
    ind_te = (where(colinfo eq 'time',cnt_te))[0]
    ind_t1 = (where(colinfo eq 'temp',cnt_t1))[0]
    ind_t2 = (where(colinfo eq 'temperaturecontrolreading',cnt_t2))[0]
    ind_t3 = (where(colinfo eq 'temperaturesetpoint',cnt_t3))[0]
    ind_ts = (where(colinfo eq 'timestamp',cnt_ts))[0]
    ind_pt = (where(colinfo eq 'ptai',cnt_pt))[0]
    ind_mr = (where(colinfo eq 'monrot',cnt_mr))[0]
    ind_mt = (where(colinfo eq 'montrans',cnt_mt))[0]
    ind_ds = (where(colinfo eq 'dfmdts',cnt_ds))[0]
    ind_mb = (where(colinfo eq 'mbtslide',cnt_mb))[0]
    ind_bs = lonarr(21)
    ind_as = lonarr(20)
    for j=1,21 do begin
        tmp = string(j,format='(i02)')
        ind_bs[j-1] = (where(colinfo eq 'monblade'+tmp,cnt_bs))[0]
        if j le 20 then ind_as[j-1] = (where(colinfo eq 'analyzertheta'+tmp,cnt_as))[0]
    endfor
    ind_sx = (where(colinfo eq 'smplx',cnt_sx))[0]
    ind_sy = (where(colinfo eq 'smply',cnt_sy))[0]
    ind_sz = (where(colinfo eq 'smplz',cnt_sz))[0]
    ind_dm = (where(colinfo eq 'dmbt',cnt_dm))[0]
    ind_fc = (where(colinfo eq 'focus',cnt_fc))[0]
    ind_b1 = (where(colinfo eq 'beta1',cnt_b1))[0]
    ind_b2 = (where(colinfo eq 'beta2',cnt_b2))[0]
    ind_vh = (where(colinfo eq 'vbah',cnt_vh))[0]
    ind_vv = (where(colinfo eq 'vbav',cnt_vv))[0]
    ind_kn = (where(colinfo eq 'kidney',cnt_kn))[0]
    ind_hf = (where(colinfo eq 'magfield',cnt_hf))[0]
    if cnt_hf eq 0 then ind_hf = (where(colinfo eq 'hfield',cnt_hf))[0]
    if cnt_hs gt 0 then hs = data[*,ind_hs] else hs = fltarr(n_data)
    if cnt_ks gt 0 then ks = data[*,ind_ks] else ks = fltarr(n_data)
    if cnt_ls gt 0 then ls = data[*,ind_ls] else ls = fltarr(n_data)
    if cnt_en gt 0 then en = data[*,ind_en] else en = fltarr(n_data)
    if cnt_a1 gt 0 then a1 = data[*,ind_a1] else a1 = fltarr(n_data)
    if cnt_a2 gt 0 then a2 = data[*,ind_a2] else a2 = fltarr(n_data)
    if cnt_a3 gt 0 then a3 = data[*,ind_a3] else a3 = fltarr(n_data)
    if cnt_a4 gt 0 then a4 = data[*,ind_a4] else a4 = fltarr(n_data)
    if cnt_a5 gt 0 then a5 = data[*,ind_a5] else a5 = fltarr(n_data)
    if cnt_a6 gt 0 then a6 = data[*,ind_a6] else a6 = fltarr(n_data)
    if cnt_sx gt 0 then sx = data[*,ind_sx] else sx = fltarr(n_data)
    if cnt_sy gt 0 then sy = data[*,ind_sy] else sy = fltarr(n_data)
    if cnt_sz gt 0 then sz = data[*,ind_sz] else sz = fltarr(n_data)
    if cnt_sl gt 0 then sl = data[*,ind_sl] else sl = fltarr(n_data)
    if cnt_su gt 0 then su = data[*,ind_su] else su = fltarr(n_data)
    if cnt_ht gt 0 then ht = data[*,ind_ht] else ht = fltarr(n_data)
    if cnt_vt gt 0 then vt = data[*,ind_vt] else vt = fltarr(n_data)
    if cnt_mr gt 0 then mr = data[*,ind_mr] else mr = fltarr(n_data)
    if cnt_wt gt 0 then wt = data[*,ind_wt] else wt = fltarr(n_data)
    if cnt_te gt 0 then te = data[*,ind_te] else te = fltarr(n_data)
    if cnt_t1 gt 0 then t1 = data[*,ind_t1] else t1 = fltarr(n_data)
    if cnt_t2 gt 0 then t2 = data[*,ind_t2] else t2 = fltarr(n_data)
    if cnt_t3 gt 0 then t3 = data[*,ind_t3] else t3 = fltarr(n_data)
    if cnt_hf gt 0 then hf = data[*,ind_hf] else hf = fltarr(n_data)
    if cnt_ts gt 0 then ts = data[*,ind_ts] else ts = fltarr(n_data)
    if cnt_pt gt 0 then pt = data[*,ind_pt] else pt = fltarr(n_data)
    if cnt_mt gt 0 then mt = data[*,ind_mt] else mt = fltarr(n_data)
    if cnt_ds gt 0 then ds = data[*,ind_ds] else ds = fltarr(n_data)
    if cnt_mb gt 0 then mb = data[*,ind_mb] else mb = fltarr(n_data)
    if cnt_bs gt 0 then bs = data[*,ind_bs] else bs = fltarr(n_data,21)
    if cnt_as gt 0 then as = data[*,ind_as] else as = fltarr(n_data,20)
    if cnt_dm gt 0 then dm = data[*,ind_dm] else dm = fltarr(n_data)
    if cnt_fc gt 0 then fc = data[*,ind_fc] else fc = fltarr(n_data)
    if cnt_b1 gt 0 then b1 = data[*,ind_b1] else b1 = fltarr(n_data)
    if cnt_b2 gt 0 then b2 = data[*,ind_b2] else b2 = fltarr(n_data)
    if cnt_vh gt 0 then vh = data[*,ind_vh] else vh = fltarr(n_data)
    if cnt_vv gt 0 then vv = data[*,ind_vv] else vv = fltarr(n_data)
    if cnt_kn gt 0 then kn = data[*,ind_kn] else kn = fltarr(n_data)
    if n_elements(pu) eq 0 then pu = fltarr(n_data)
    if keyword_set(histodata) then begin
       data_histo = total(data_histo,1,/nan)
       tmpsize = size(data_histo,/dim)
       if n_elements(tmpsize) eq 1 then tmpsize = [tmpsize,1]
       data_histo = reform(data_histo,[2,20,tmpsize[1]])
       it = fltarr(2,21,tmpsize[1])
       it[*,0:19,*] = data_histo[*,*,*]
       for i=0L,n_data-1L do it[*,20,i] = data_histo[*,pt[i]-1,i]
       if n_elements(dqty_histo) ne 0 then begin
          dqty_histo = sqrt(total(dqty_histo^2,1,/nan))
          dqty_histo = reform(dqty_histo,[2,20,tmpsize[1]])
          dit = fltarr(2,21,tmpsize[1])
          dit[*,0:19,*] = dqty_histo[*,*,*]
          for i=0L,n_data-1L do dit[*,20,i] = dqty_histo[*,pt[i]-1,i]
       endif
    endif else begin
       it = transpose([[[data[*,ind_qty[0,*]]]],[[data[*,ind_qty[1,*]]]]])
       if readerr then begin
          dit = transpose([[[data[*,ind_err]]],[[sqrt(data[*,ind_qty[1,*]])]]])
          for i=0L,n_elements(pt)-1 do begin
              it[*,20,i]  = it[*,pt[i]-1,i]
              dit[*,20,i] = dit[*,pt[i]-1,i]
          endfor
       endif
    endelse 
    nstr = 'cfx'+['be','hopg','mgf']
    ;check if the date is before CFXdate
    tmp = stregex(header,'#Date(.*)',/fold_case,/subexpr,length=len)
    ind = (where(len[1,*] gt 0,count))[0]
    if count gt 0 then begin
       date = dm_to_number(strmid(header[ind],tmp[1,ind],len[1,ind]),/date)
       if date lt macs_cfxdate then nstr = 'cfx'+['be','hopg']
    endif
    cnt_cfx = 0
    for j=0,n_elements(nstr)-1 do begin
        ind_cfx = (where(colinfo eq nstr[j],count))[0]
        if count ne 0 then begin
           if finite(data[0,ind_cfx]) then begin
              if round(data[0,ind_cfx]) then cnt_cfx = cnt_cfx+2^j
           endif
        endif
    endfor
    cfx = cnt_cfx*(intarr(n_data)+1)
    if keyword_set(avgtemp) then begin
       t1[*] = mean(t1)
       t2[*] = mean(t2)
       t3[*] = mean(t3)
    endif
    if n_elements(header) ne 0 then begin
       if n_elements(duration) eq 0 then begin
          duration = wt[0]
          ;figure out the monitor is time or monitor count
          index = (where(stregex(header, '#reference',/boolean,/fold_case),count))[0]
          if count gt 0 then begin
             tmp = strlowcase(strsplit(header[index],' '+string(9b),/extract))
             if n_elements(montype) eq 0 then begin
                if tmp[n_elements(tmp)-1] eq 'time'    then montype = 0
                if tmp[n_elements(tmp)-1] eq 'monitor' then montype = 1
             endif
             index = where(colinfo eq tmp[n_elements(tmp)-1],count)
             if (count gt 0) and (tmp[n_elements(tmp)-1] eq 'time') then duration = -abs(te[0]) ;negative value to denote time
          endif
       endif
       ;figure out scanning which device and detector id
       if (n_elements(xid) eq 0) or (n_elements(yid) eq 0) then begin
          tmp = '' 
          ind = (where(stregex(header, '#scan +[0-9]+ +',/boolean,/fold_case),count))[0]
          if count gt 0 then begin
             tmp = strsplit(header[ind],/extract)
             tmp = strlowcase(tmp[n_elements(tmp)-1])
             if tmp eq 'kidney' then tmp = 'a4' $
             else if tmp eq 'singledet' then tmp = 'a6' $
             else if tmp eq 'qx' then tmp = 'h' $
             else if tmp eq 'qy' then tmp = 'k' $
             else if tmp eq 'qz' then tmp = 'l'
          endif else if n_data gt 1 then begin
             test = regress(lindgen(n_data),hf,correlation=corr)
             if finite(corr) then begin
                if abs(corr) gt 0.9 then tmp = 'magfield'
             endif
             if strlen(tmp) eq 0 then begin
                test = regress(lindgen(n_data),t1,correlation=corr)
                if finite(corr) then begin
                   if abs(corr) gt 0.9 then tmp = 'temp'
                endif
             endif
          endif
          if n_elements(xid) eq 0 and (strlen(tmp) ne 0) then xid = 0>((where(strlowcase(x_info) eq tmp))[0])
          if n_elements(yid) eq 0 then begin
             ind = (where(stregex(header, '#signal +[0-9]* +monitor',/boolean,/fold_case),count))[0]
             tmp = strmid(tmp,0,3)
             yid = 3-((count eq 1) and ((tmp eq 'mon') or (tmp eq 'dfm')))
             ind = (where(stregex(header, '#signal +[0-9]* +',/boolean,/fold_case),count))[0]
             if count gt 0 then begin
                tmp = strsplit(header[ind],/extract)
                tmp = tmp[n_elements(tmp)-1]
                if stregex(tmp,'spec',/fold_case,/boolean) then did = 0
                if stregex(tmp,'diff',/fold_case,/boolean) then did = 1
             endif 
          endif
       endif
    endif
end

pro dcs_cryst_align_loadtas,filename,hs=hs,ks=ks,ls=ls,en=en,a3=a3,a4=a4,a1=a1,a2=a2,a5=a5,a6=a6,mr=mr,ar=ar,as=as,ps=ps,sx=sx,sy=sy,sz=sz,sw=sw,sh=sh,sl=sl,su=su,bw=bw,bh=bh,hf=hf,$
    t1=t1,t2=t2,t3=t3,ts=ts,te=te,wt=wt,it=it,ms=ms,m_id=m_id,duration=duration,xid=xid,yid=yid,is_bt4=is_bt4,is_bt7=is_bt7,is_nice=is_nice,avgtemp=avgtemp,latticeparm=latticeparm,$
    latticeori=latticeori,bufferdir=bufferdir,tmpfile=tmpfile,error=error,group_leader=group_leader
    error = 0b
    is_bt4 = stregex(filename,'\.b[ta-d][249]$',/boolean,/fold_case)
    is_nice = stregex(filename,'^.+\.[a-z0-9]{2,4}\.[a-z0-9]{2,4}$',/fold_case,/boolean)
    is_bt7 = stregex(filename,'\.b[ta-d]7$',/boolean,/fold_case)
    if n_elements(it) ne 0 then tmp = temporary(it)
    if n_elements(ms) ne 0 then tmp = temporary(ms)
    if n_elements(m_id) ne 0 then tmp = temporary(m_id)
    if ~is_nice then begin
       tmpobj = obj_new('dtas_data',filename=filename)
       if ~obj_valid(tmpobj) then begin ;attempt to read the file again with known work-arounds
          dm_fix_tasfile,filename,fixedfile=tmpfile,bufferdir=bufferdir,group_leader=group_leader
          if n_elements(tmpfile) ne 0 then begin
             tmpobj = obj_new('dtas_data',filename=tmpfile)
             file_delete,tmpfile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
          endif
       endif
    endif
    if obj_valid(tmpobj) then begin
       ret = tmpobj->get_property(isIce=isIce,var_names=var_name,data=data,plot_str=plot_str,header=header,contents=contents)
       if n_elements(header) ne 0 then isNeut = stregex(header[0],"'neut'",/boolean,/fold_case) else isNeut = 0
       n_data = n_elements(*(data.index))
       if n_data gt 0 then begin
          if n_elements(header) gt 0 then begin  ;n_data might be more than actual #pts in certain tas files with empty lines at the end
             ind = where(stregex(header,' #pts ',/boolean),cnt)
             if cnt gt 0 then begin
                tmp = strsplit(header[ind[0]],' ',/extract)
                ind1 = where(strmatch(tmp,'#pts'),cnt1)
                offset = n_elements(tmp)-ind1[cnt1-1]
                tmp = strsplit(header[ind[0]-1],' ',/extract)
                n_data1 = dm_to_number(tmp[n_elements(tmp)-offset])
                if (n_data gt n_data1) and (n_data1 gt 0) then n_data = fix(n_data1)
             endif
          endif
          if n_data gt 1 then tmp = temporary(contents)
          hs = dblarr(n_data) & as = dblarr(n_data,13)
          ks = hs & ls = hs & en = hs & a3 = hs & a4 = hs & a1 = hs & a2 = hs & a5 = hs & a6 = hs & mr = hs & ar = hs & ps = hs & sx = hs & sy = hs & sz = hs 
          sw = hs & sh = hs & sl = hs & su = hs & bw = hs & bh = hs & hf = hs & t1 = hs & t2 = hs & t3 = hs & ts = hs & te = hs & wt = hs & ms = hs
          if keyword_set(isIce) and is_bt7 then it = dblarr(n_data,1+3+11+48+3) else it = hs
          tag_name = strlowcase(tag_names(data))
          tmp_form = '^'+['analyzerblade([0-9]{2})','ddc([0-2]{1})','tdc([0-9]{2})','psdc([0-9]{2})','sdc([0-2]{1})']+'$'
          tmp_indx = [0,1,4,15,63]
          var_name = strlowcase(var_name)
          for i=0,n_elements(var_name)-1 do begin
              case var_name[i] of
                   'mon' :                      begin
                                                wt = (*(data.mon))[0:(n_data-1)]
                                                if n_elements(duration) eq 0 then duration = wt[0]*(keyword_set(isNeut)*2-1)
                                                end
                   'monitor':                   begin
                                                if keyword_set(isIce) then begin
                                                   wt = (*(data.monitor))[0:(n_data-1)]
                                                endif else begin
                                                   te = (*(data.monitor))[0:(n_data-1)]
                                                   if n_elements(duration) eq 0 then duration = -te[0]
                                                endelse
                                                end
                   'min':                       te = (*(data.min))[0:(n_data-1)]*60.0
                   'time':                      te = (*(data.time))[0:(n_data-1)]
                   'qx':                        hs = (*(data.qx))[0:(n_data-1)]
                   'qy':                        ks = (*(data.qy))[0:(n_data-1)]
                   'qz':                        ls = (*(data.qz))[0:(n_data-1)]
                   'h':                         hs = (*(data.h))[0:(n_data-1)]
                   'k':                         ks = (*(data.k))[0:(n_data-1)]
                   'l':                         ls = (*(data.l))[0:(n_data-1)]
                   'e':                         en = (*(data.e))[0:(n_data-1)]
                   'a1':                        a1 = (*(data.a1))[0:(n_data-1)]
                   'a2':                        a2 = (*(data.a2))[0:(n_data-1)]
                   'a3':                        a3 = (*(data.a3))[0:(n_data-1)]
                   'a4':                        a4 = (*(data.a4))[0:(n_data-1)]
                   'a5':                        a5 = (*(data.a5))[0:(n_data-1)]
                   'a6':                        a6 = (*(data.a6))[0:(n_data-1)]
                   'temp':                      t1 = (*(data.temp))[0:(n_data-1)]
                   'tact':                      t1 = (*(data.tact))[0:(n_data-1)]
                   'temperaturecontrolreading': t2 = (*(data.temperaturecontrolreading))[0:(n_data-1)]
                   'temperaturesetpoint':       t3 = (*(data.temperaturesetpoint))[0:(n_data-1)]
                   'dfmrot':                    mr = (*(data.dfmrot))[0:(n_data-1)]
                   'analyzerrotation':          ar = (*(data.analyzerrotation))[0:(n_data-1)]
                   'psdet':                     ps = (*(data.psdet))[0:(n_data-1)]
                   'detector':                  it[*,0] = (*(data.detector))[0:(n_data-1)]
                   'counts':                    it[*,0] = (*(data.counts))[0:(n_data-1)]
                   'intensity':                 it[*,0] = (*(data.intensity))[0:(n_data-1)]
                   'motor_1':                   a1 = (*(data.motor_1))[0:(n_data-1)]
                   'motor_2':                   begin
                                                a2 = (*(data.motor_2))[0:(n_data-1)]
                                                if (strlowcase(plot_str.xtitle) eq 'motor_1') and (mean(a2) ne a2[0]) then plot_str.xtitle = 'motor_2' ;theta-two theta scan, plot vs two theta
                                                end
                   'motor_3':                   a3 = (*(data.motor_3))[0:(n_data-1)]
                   'motor_4':                   begin
                                                a4 = (*(data.motor_4))[0:(n_data-1)]
                                                if (strlowcase(plot_str.xtitle) eq 'motor_3') and (mean(a4) ne a4[0]) then plot_str.xtitle = 'motor_4' ;theta-two theta scan, plot vs two theta
                                                end
                   'motor_5':                   a5 = (*(data.motor_5))[0:(n_data-1)]
                   'motor_6':                   begin
                                                a6 = (*(data.motor_6))[0:(n_data-1)]
                                                if (strlowcase(plot_str.xtitle) eq 'motor_5') and (mean(a6) ne a6[0]) then plot_str.xtitle = 'motor_6' ;theta-two theta scan, plot vs two theta
                                                end
                   'motor_7':                   if is_bt4 then su = (*(data.motor_7))[0:(n_data-1)]
                   'motor_8':                   if is_bt4 then sl = (*(data.motor_8))[0:(n_data-1)]
                   'motor_9':                   if is_bt4 then sx = (*(data.motor_9))[0:(n_data-1)]
                   'motor_10':                  if is_bt4 then sy = (*(data.motor_10))[0:(n_data-1)]
                   'smplutrn':                  sx = (*(data.smplutrn))[0:(n_data-1)]
                   'smplltrn':                  sy = (*(data.smplltrn))[0:(n_data-1)]
                   'smplelev':                  sz = (*(data.smplelev))[0:(n_data-1)]
                   'motor_11':                  if is_bt4 then sh = (*(data.motor_11))[0:(n_data-1)] else sl = (*(data.motor_11))[0:(n_data-1)]
                   'motor_12':                  if is_bt4 then sw = (*(data.motor_12))[0:(n_data-1)] else su = (*(data.motor_12))[0:(n_data-1)]
                   'smplltilt':                 sl = (*(data.smplltilt))[0:(n_data-1)]
                   'smplutilt':                 su = (*(data.smplutilt))[0:(n_data-1)]
                   'smplwdth':                  sw = (*(data.smplwdth))[0:(n_data-1)]
                   'smplhght':                  sh = (*(data.smplhght))[0:(n_data-1)]
                   'bksltwdth':                 bw = (*(data.bksltwdth))[0:(n_data-1)]
                   'bkslthght':                 bh = (*(data.bkslthght))[0:(n_data-1)]
                   'hfield':                    hf = (*(data.hfield))[0:(n_data-1)]
                   'magfield':                  hf = (*(data.magfield))[0:(n_data-1)]
                   'timestamp':                 ts = (*(data.timestamp))[0:(n_data-1)]
                   else:                        begin
                                                ind_tag = where(tag_name eq var_name[i],cnt)
                                                if cnt gt 0 then begin
                                                   if stregex(var_name[i],'^motor_[12][0-9]$',/boolean) then begin ;unaccounted motor #
                                                      m_id = dm_to_number(strmid(var_name[i],strpos(var_name[i],'_')+1))
                                                      ms = (*(data.(ind_tag)))[0:(n_data-1)]
                                                   endif else begin
                                                      for j=0,n_elements(tmp_form)-1 do begin
                                                          tmp = stregex(var_name[i],tmp_form[j],/subexpr,length=len)
                                                          if len[1] gt 0 then begin
                                                             tmp_num = dm_to_number(strmid(var_name[i],tmp[1]),/int)
                                                             if j eq 0 then as[*,tmp_num-1] = (*(data.(ind_tag)))[0:(n_data-1)] $
                                                             else it[*,tmp_indx[j]+tmp_num] = (*(data.(ind_tag)))[0:(n_data-1)]
                                                             break 
                                                          endif
                                                      endfor
                                                   endelse
                                                endif
                                                end
              endcase
          endfor
          max_hf = max(hf,min=min_hf)
          if max_hf gt 1000 then begin
             hf = hf/1e4 ;gauss->tesla
             if max_hf gt min_hf then plot_str.xtitle = 'hfield'
          endif
          ret = tmpobj->get_private(lattice_ptr=lattice_ptr,orientation_ptr=orientation_ptr,xtal_orientation_ptr=xtal_orientation_ptr)
          if ptr_valid(lattice_ptr) then begin
             latticeparm = (*lattice_ptr)
             if size(latticeparm,/type) eq 8 then latticeparm = [latticeparm.a,latticeparm.b,latticeparm.c,latticeparm.alpha,latticeparm.beta,latticeparm.gamma]
          endif
          if ptr_valid(orientation_ptr) then latticeori = (*orientation_ptr)
          if ptr_valid(xtal_orientation_ptr) then begin 
             latticeori = (*xtal_orientation_ptr)
             latticeori = [latticeori.hu,latticeori.ku,latticeori.lu,latticeori.hv,latticeori.kv,latticeori.lv]
          endif
          if keyword_set(isIce) and (n_elements(header) ne 0) then begin
             if n_elements(duration) eq 0 then begin
                duration = wt[0]
                ;figure out the monitor is time or monitor count
                yn = stregex(header, '#reference',/boolean,/fold_case)
                index = where(yn,count)
                if count gt 0 then begin
                   tmp = strlowcase(strsplit(header[index[0]],' '+string(9b),/extract))
                   if (tmp[n_elements(tmp)-1] eq 'time') then duration = -abs(te[0]) ;negative value to denote time
                endif
             endif
             ;figure out scanning which device and detector id
             if (n_elements(xid) eq 0) or (n_elements(yid) eq 0) then begin
                ind = where(stregex(header, '#scan +[0-9]+ +',/boolean,/fold_case),count)
                if count eq 1 then begin
                   tmp = strsplit(header[ind],/extract)
                   tmp = strlowcase(tmp[n_elements(tmp)-1])
                   if tmp eq 'singledet' then begin
                      tmp = 'a6'
                      ind = where(stregex(header,'#Fixed +',/boolean,/fold_case),count)
                      if count eq 1 then begin
                         tmp1 = strsplit(header[ind[0]],' ',/extract)
                         ind  = where(strmatch(tmp1,'A6',/fold_case),count)
                         if count eq 1 then tmp = -1  ;A6 is fixed, used point as xaxis
                      endif
                   endif
                   if n_elements(xid) eq 0 then xid = tmp
                   if n_elements(yid) eq 0 then begin
                      yid = 1
                      ind = where(stregex(header, '#signal +[0-9]+ +[^ ]+',/boolean,/fold_case),count)
                      if count eq 1 then begin
                         tmp1 = stregex(header[ind[0]],'#signal +[0-9]+ +([^ ]+)',/fold_case,/extract,/subexpr)
                         if strlen(tmp1[0]) ne 0 then begin
                            if (strlowcase(tmp1[1]) eq 'monitor') then begin
                               tmp = strmid(tmp,0,3)
                               if (tmp eq 'mon') or (tmp eq 'dfm') then yid = tmp1[1] 
                            endif else yid = tmp1[1]
                         endif
                      endif
                   endif
                endif else if n_data gt 1 then begin
                   test = regress(lindgen(n_data),hf,correlation=corr)
                   if finite(corr) then begin
                      if abs(corr) gt 0.9 then xid = 'magfield'
                   endif
                   if n_elements(xid) eq 0 then begin
                      test = regress(lindgen(n_data),t1,correlation=corr)
                      if finite(corr) then begin
                         if abs(corr) gt 0.9 then xid = 'temp'
                      endif
                   endif
                endif
             endif
          endif else begin
             t2 = t1 & t3 = t1
             if (n_data eq 1) and (n_elements(contents) gt 0) then begin  ;plot_str.xtitle might be incorrect 
                ind = where(finite(dm_to_number(contents)),cnt)
                if cnt gt 0 then begin
                   tmp = (strsplit(contents[ind[cnt-1]-1],' ',/extract))[0]
                   tmp2 =  stregex(tmp,'Q.*([xyz])',/subexp,/extract,/fold_case)
                   if strlen(tmp2[1]) gt 0 then tmp = 'q'+strlowcase(tmp2[1])
                   plot_str.xtitle = tmp
                endif
             end
             if n_elements(xid) eq 0 then xid = plot_str.xtitle
             if n_elements(yid) eq 0 then yid = 1
          endelse
       endif
    endif else if is_nice then begin
       if H5F_IS_HDF5(filename) then dm_load_macsnexus,filename,data,info,header=header,error=error,group_leader=group_leader,units=units $ ;NICE nexus file
       else dm_load_macs,filename,data,info,header=header,error=error,group_leader=group_leader,units=units
       if (n_elements(data) ne 0) and ~keyword_set(error) then begin
           n_data = n_elements(data[*,0])
           hs = dblarr(n_data) & as = dblarr(n_data,13)
           ks = hs & ls = hs & en = hs & a3 = hs & a4 = hs & a1 = hs & a2 = hs & a5 = hs & a6 = hs & mr = hs & ar = hs & ps = hs & sx = hs & sy = hs & sz = hs
           sw = hs & sh = hs & sl = hs & su = hs & bw = hs & bh = hs & hf = hs & t1 = hs & t2 = hs & t3 = hs & ts = hs & te = hs & wt = hs & it = hs
           if is_bt7 then it = dblarr(n_data,1+3+11+48+3)
           tmp_id1 = ['a3','a4','a5','a6','smplltilt','smplutilt','smplltrn','smplutrn','h','k','l','e','temp','temperaturecontrolreading', 'temperaturesetpoint','magfield','hslit','vslit','bksltwdth','bkslthght'] ;displayed name of tmp_id2
           tmp_id2 = ['^sampletheta(motor)*$','^sampletwotheta(motor)*$','^ana(lyzer)*theta(motor)*$','^ana(lyzer)twotheta(motor)*$','lowertilt$','uppertilt$','lower[a-z]*tran','upper[a-z]*tran','\.(set)*h$','\.(set)*k$',$
                      '\.(set)*l$','^et','^temp$','^temp.*control','^temp.*setpoint','^magfield','^(presample)*slitwidth','^(presample)*slitheight','^postsampleslitwidth','^postsampleslitheight']  ;must be regular expression         
           tmp_ind = lonarr(n_elements(tmp_id1))
           tmp_cnt = tmp_ind
           for i=0,n_elements(tmp_id1)-1 do begin
               tmp_ind[i] = (where(stregex(info,tmp_id2[i],/boolean),cnt))[0]
               if cnt eq 0 then $
               tmp_ind[i] = (where(strmatch(info,tmp_id1[i]),cnt))[0]
               tmp_cnt[i] = cnt 
           endfor
           ind_wt = (where(strmatch(info,'monitor'),cnt_wt))[0]
           ind_it = (where(strmatch(info,'counts'),cnt_it))[0]
           if cnt_it eq 0 then $
           ind_it = (where(strmatch(info,'detector'),cnt_it))[0]
           ind_dd = (where(strmatch(info,'dd0'),cnt_dd))[0]
           ind_dr = (where(strmatch(info,'doord0'),cnt_dr))[0]
           ind_ps = (where(strmatch(info,'psd00'),cnt_ps))[0]
           ind_sd = (where(strmatch(info,'sd0'),cnt_sd))[0]
           ind_te = (where(strmatch(info,'time'),cnt_te))[0]
           ind_ts = (where(strmatch(info,'timestamp'),cnt_ts))[0]
           factor = [1.0,1.0,1.0]
           if n_elements(units) ne 0 then begin
              ind = where(strmatch(units[0],'cm',/fold_case),cnt)
              if cnt gt 0 then factor[ind] = 10.0 ;cm->mm
           endif
           if tmp_cnt[0]  ne 0 then a3[*] = data[*,tmp_ind[0]]
           if tmp_cnt[1]  ne 0 then a4[*] = data[*,tmp_ind[1]]
           if tmp_cnt[2]  ne 0 then a5[*] = data[*,tmp_ind[2]]
           if tmp_cnt[3]  ne 0 then a6[*] = data[*,tmp_ind[3]]
           if tmp_cnt[4]  ne 0 then sl[*] = data[*,tmp_ind[4]]
           if tmp_cnt[5]  ne 0 then su[*] = data[*,tmp_ind[5]]
           if tmp_cnt[6]  ne 0 then sy[*] = data[*,tmp_ind[6]]*factor[0]
           if tmp_cnt[7]  ne 0 then sx[*] = data[*,tmp_ind[7]]*factor[0] 
           if tmp_cnt[8]  ne 0 then hs[*] = data[*,tmp_ind[8]]
           if tmp_cnt[9]  ne 0 then ks[*] = data[*,tmp_ind[9]]
           if tmp_cnt[10] ne 0 then ls[*] = data[*,tmp_ind[10]]
           if tmp_cnt[11] ne 0 then en[*] = data[*,tmp_ind[11]]
           if tmp_cnt[12] ne 0 then t1[*] = data[*,tmp_ind[12]]
           if tmp_cnt[13] ne 0 then t2[*] = data[*,tmp_ind[13]]
           if tmp_cnt[14] ne 0 then t3[*] = data[*,tmp_ind[14]]
           if tmp_cnt[15] ne 0 then hf[*] = data[*,tmp_ind[15]]
           if tmp_cnt[16] ne 0 then sw[*] = data[*,tmp_ind[16]]*factor[1]
           if tmp_cnt[17] ne 0 then sh[*] = data[*,tmp_ind[17]]*factor[1]
           if tmp_cnt[18] ne 0 then bw[*] = data[*,tmp_ind[18]]*factor[2]
           if tmp_cnt[19] ne 0 then bh[*] = data[*,tmp_ind[19]]*factor[2]
           if cnt_wt ne 0 then wt[*] = data[*,ind_wt]
           if cnt_it ne 0 then it[*,0] = data[*,ind_it]
           if is_bt7 and (cnt_dd ne 0) then it[*,1:3] = data[*,ind_dd:(ind_dd+2)]
           if is_bt7 and (cnt_dr ne 0) then it[*,4:14] = data[*,ind_dr:(ind_dr+10)]
           if is_bt7 and (cnt_ps ne 0) then it[*,15:62] = data[*,ind_ps:(ind_ps+47)]
           if is_bt7 and (cnt_sd ne 0) then it[*,63:65] = data[*,ind_sd:(ind_sd+2)]           
           if cnt_te ne 0 then te[*] = data[*,ind_te]
           if cnt_ts ne 0 then ts[*] = data[*,ind_ts]
           if ts[0] gt 1e12 then  ts = ts/1d3  ;ms->sec
           ;figure out scanning which device and detector id
           if n_elements(xid) eq 0 then begin
              if is_bt7 then begin
                 tmp_ind = where(tmp_id1 eq 'hslit')
                 tmp_id1[tmp_ind:tmp_ind+1] = ['smplwdth','smplhght']
              endif
              ind = where(stregex(header, '#scan +[0-9]* +',/boolean,/fold_case),count)
              if count eq 1 then begin
                 tmp = strsplit(header[ind],/extract)
                 tmp = strlowcase(tmp[n_elements(tmp)-1])
                 if tmp eq 'singledet' then tmp = 'a6'
                 ind = where(tmp_id1 eq strlowcase(tmp),cnt)
                 if cnt gt 0 then xid = tmp_id1[ind[0]] else begin
                    for i=0,n_elements(tmp_id2)-1 do begin
                        ind = where(stregex(tmp,tmp_id2[i],/boolean,/fold_case),cnt)
                        if cnt eq 1 then begin
                           xid = tmp_id1[i]
                           break
                        endif
                    endfor
                 endelse
              endif else begin
                 ind = where(tmp_ind eq 1,cnt) ;xid is info[1]
                 if cnt ne 0 then begin
                    tmp_max = max(data[*,tmp_ind[ind[0]]],min=tmp_min)
                    if tmp_max gt tmp_min then xid = tmp_id1[ind[0]]
                 endif
              endelse
           endif
           ;figure out the monitor is time or monitor count
           if n_elements(duration) eq 0 then begin
              duration = double(long(mean(wt[where(wt ge 0)])))
              if (max(te) eq min(te)) and (te[0] ne 0) then duration = -abs(te[0]) ;negative value to denote time
              if duration eq 0 then duration = 1.0  ;in case monitor = 0
           endif
       endif
    endif
    if obj_valid(tmpobj) then obj_destroy,tmpobj
    if keyword_set(avgtemp) and (n_elements(t1) ne 0) then begin
       t1[*] = mean(t1)
       t2[*] = mean(t2)
       t3[*] = mean(t3)
    endif
    is_bt7 = is_bt7 and (is_nice or keyword_set(isIce))
end

;adjust the precision of values for xtit, return the first element of the unique values
function dcs_cryst_align_uniq,val,n_val=n_val,ret_str=ret_str,nan=nan
    ind = where(finite(val),n_val,ncomplement=n_nan)
    if ((~keyword_set(nan)) and (n_nan gt 0)) or (n_val eq 0) then tmpval = val[0] else begin
       val1 = val[ind]
       tmpval = round(val1*1d4,/L64)/1d4 & tmpval = tmpval[uniq(tmpval,sort(tmpval))] & n_val = n_elements(tmpval)
       if (n_val ne 1) and (n_val le n_elements(val1)/2+1) then begin
          tmpval = round(val1*1d3,/L64)/1d3 & tmpval = tmpval[uniq(tmpval,sort(tmpval))] & n_val = n_elements(tmpval)
       endif
       if n_val eq 1 then tmpval[0] = round(mean(val1)*1d4)/1d4
    endelse
    return,keyword_set(ret_str)?dm_to_string(tmpval[0]):(tmpval[0])
end

;construct plot xtit for H,K,L scans of MACS and TAS
function dcs_cryst_align_xtit,pState
    if (*pState).xaxis lt 0 then return,'Point'
    if (((*pState).ftype ne 'MACS') && ((*pState).ftype ne 'TAS')) || ((*pState).xaxis gt 2) then return,(*(*pState).xtit)[(*pState).xaxis]
    npoint = n_elements((*(*pState).xPtr)[0,*]) & hkl_sco = dblarr(3,3)   ;[*,0]-step, [*,1]-hkl coefficient, [*,2]-hkl offset
    if npoint eq 1 then begin
       hkl_sco[*,2] = (*(*pState).xPtr)[0:2,0]
       tmp_id = where(hkl_sco[*,2] eq hkl_sco[(*pState).xaxis,2])
    endif else begin
       for i=0,2 do begin
           tmp = dcs_cryst_align_uniq((*(*pState).xPtr)[i,1:(npoint-1)]-(*(*pState).xPtr)[i,0:(npoint-2)],n_val=ntmp)
           if (ntmp gt 1) || (tmp eq 0) then begin    ;no constant step
              tmp = dcs_cryst_align_uniq((*(*pState).xPtr)[i,*],n_val=ntmp)
              if ntmp eq 1 then hkl_sco[i,2] = tmp else tmp_id = (n_elements(tmp_id) eq 0)?i:[tmp_id,i]
           endif else hkl_sco[i,0] = tmp
       endfor
       if n_elements(tmp_id) gt 1 then return,(['H','K','L'])[(*pState).xaxis]     ;can't construct [h,k,l] type xtit
       if n_elements(tmp_id) eq 1 then (*pState).xaxis = tmp_id
       ind0 = where(hkl_sco[*,0] ne 0,cnt0)
       if cnt0 gt 0 then begin
          if hkl_sco[(*pState).xaxis,0] eq 0 then (*pState).xaxis = ind0[0]
          ind1 = where(ind0 ne (*pState).xaxis,cnt1)
          if cnt1 gt 0 then begin
             hkl_sco[ind0[ind1],1] = round(hkl_sco[ind0[ind1],0]/hkl_sco[(*pState).xaxis,0]*1d3)/1d3
             hkl_sco[ind0[ind1],2] = round(((*(*pState).xPtr)[ind0[ind1],0]-(*(*pState).xPtr)[(*pState).xaxis,0]*hkl_sco[ind0[ind1],1])*1d4)/1d4
          endif
       endif else if (max(abs(hkl_sco[*,2])) eq 0) and (n_elements(tmp_id) eq 0) then return,(['H','K','L'])[(*pState).xaxis]   ;[h,k,l] data missing
       tmp_id = (*pState).xaxis
    endelse
    hkl_sco[tmp_id,*] = replicate(1.0,n_elements(tmp_id))#transpose([0.0,1.0,0.0])
    return,dm_set_viewlabel((['H','K','L'])[(*pState).xaxis],hkl_sco[*,1],hkl_sco[*,2])
end

;plotting
pro dcs_cryst_align_plot,pState,dataonly=dataonly,xs=xs,ys=ys,dys=dys,xtit=xtit,ytit=ytit,params=params,sigma=sigma,fixedwidth=fixedwidth,peaktime=peaktime
    if (*pState).fit then (*pState).plotWin->getproperty,xran=oldxran,iszoom=iszoom ;save the xrange
    if n_elements(*(*pState).yPtr) eq 0 then return ;no data
    tit    = (*pState).title
    xtit   = dcs_cryst_align_xtit(pState)
    ytit   = 'Neutron Counts'
    n_file = n_elements((*(*pState).xPtr)[0,*]) 
    n_det  = n_elements(*(*pState).detPtr)
    t_axis = stregex((*(*pState).xtit)[(*pState).xaxis],'^T *\(.*K\)',/boolean,/fold_case)
    textra = (t_axis and ((*pState).ftype ne 'WAND'))*(*pState).tchoice
    subbg  = (n_elements(*(*pState).bgPtr) ne 0)
    case (*pState).ftype of 
         'DCS':   begin
              ytit = ytit+'/'+dm_to_string((*pState).duration,res=1)+' Sec'+(['','s'])[(*pState).duration ge 1.5]
              if subbg then dcsbgfactor = (*pState).dcs_monitor[0]/(*pState).dcs_monitor[1]
              end
         'MACS':  begin
              xA4 = stregex(xtit,'A4',/fold_case,/boolean)  ;Is x axis A4?
              yMn = ((*(*pState).detPtr)[0] eq -1)          ;Is y axis monitor?
              yPt = ((*(*pState).detPtr)[0] eq -2)          ;Is detector PTAI?
              iA4 = (where(stregex((*(*pState).xtit),'A4',/fold_case,/boolean)))[0]
              ikn = iA4+2 & iPt = iA4+1
              mon = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-1,*])
              tim = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-2,*])
              pul = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-5,*])
              ief = 2*((*pState).macs_eff-1)                ;detector efficiency index
              macs_ydat = (*(*pState).yPtr)
              macs_yerr = (*(*pState).dyPtr)
              timestamp = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-4,*])
              if (*pState).macs_normchoice eq 0 then monfac = mon[0]/(1>mon) $    ;monitor normalization
              else if (*pState).macs_normchoice eq 1 then monfac = tim[0]/tim $   ;time normalization
              else monfac = pul[0]/(1>pul)                                        ;pulse normalization
              for j=0L,n_elements(monfac)-1 do begin
                  macs_ydat[*,*,j] = macs_ydat[*,*,j]*monfac[j]
                  macs_yerr[*,*,j] = macs_yerr[*,*,j]*monfac[j]
              endfor
              if subbg then begin
                 n_bgfile = n_elements((*(*pState).bgPtr)[0,0,*])
                 if n_bgfile lt n_file then begin
                    ok = dialog_message('There are not enough empty can data ('+dm_to_string(n_bgfile)+' instead of '+dm_to_string(n_file)+'). Only background subtracted data are shown.',$
                         dialog_parent=(*pState).tlb,/center)
                    n_file = n_bgfile
                    macs_ydat = macs_ydat[*,*,0:(n_file-1)]
                    macs_yerr = macs_yerr[*,*,0:(n_file-1)]
                 endif 
                 monfac = ([mon[0],tim[0],pul[0]])[(*pState).macs_normchoice]/reform((*(*pState).bgmonPtr)[(*pState).macs_normchoice,*])
                 bg_ydat = *(*pState).bgPtr
                 bg_yerr = *(*pState).dbgPtr
                 for j=0L,n_elements(monfac)-1 do begin
                     bg_ydat[*,*,j] = bg_ydat[*,*,j]*monfac[j]
                     bg_yerr[*,*,j] = bg_yerr[*,*,j]*monfac[j]
                 endfor
                 macs_ydat = macs_ydat-bg_ydat[*,*,0:(n_file-1)]
                 macs_yerr = sqrt(macs_yerr^2+bg_yerr[*,*,0:(n_file-1)]^2)
              endif
              if (*pState).macs_eff ne 0 then begin         ;apply detector efficiency
                 for i=0L,n_file-1 do begin
                     for j=0,19 do begin
                         macs_ydat[0,j,i] = macs_ydat[0,j,i]*((*pState).macs_deteff)[ief,j]
                         macs_ydat[1,j,i] = macs_ydat[1,j,i]*((*pState).macs_deteff)[ief+1,j]
                         macs_yerr[0,j,i] = macs_yerr[0,j,i]*((*pState).macs_deteff)[ief,j]
                         macs_yerr[1,j,i] = macs_yerr[1,j,i]*((*pState).macs_deteff)[ief+1,j]
                     endfor  
                 endfor
                 macs_ydat[0,20,*] = macs_ydat[0,20,*]*((*pState).macs_deteff)[ief,(*(*pState).xPtr)[iPt,*]-1]
                 macs_ydat[1,20,*] = macs_ydat[1,20,*]*((*pState).macs_deteff)[ief+1,(*(*pState).xPtr)[iPt,*]-1]
                 macs_yerr[0,20,*] = macs_yerr[0,20,*]*((*pState).macs_deteff)[ief,(*(*pState).xPtr)[iPt,*]-1]
                 macs_yerr[1,20,*] = macs_yerr[1,20,*]*((*pState).macs_deteff)[ief+1,(*(*pState).xPtr)[iPt,*]-1]
              endif
              if (*pState).macs_lamb2 then begin            ;apply monitor lambda/2 correction
                 iA2  = (where(stregex((*(*pState).xtit),'A2',/fold_case,/boolean)))[0] 
                 Ei   = reform(dm_lambda2e(2*(*pState).macs_maspacing[0]*sin(!dtor*(*(*pState).xPtr)[iA2+2,*]/2)))
                 cfx  = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-3,*])
                 mon1 = dm_macs_lambda2correction(Ei,mon,cfx)
                 fac  = mon/(1>(mon1))  
                 for i=0L,n_file-1 do begin
                     macs_ydat[*,*,i] = macs_ydat[*,*,i]*fac[i]
                     macs_yerr[*,*,i] = macs_yerr[*,*,i]*fac[i]
                 endfor
              endif
              titind = stregex(tit,'\(2th=([0-9.]+)',/subexpr,length=titlen)
              if titlen[1] gt 0 then tittth = dm_to_number(strmid(tit,titind[1],titlen[1])) else begin
                 titind = stregex(tit,'\(<2th>=([0-9.]+)',/subexpr,length=titlen)
                 if titlen[1] gt 0 then tittth = dm_to_number(strmid(tit,titind[1],titlen[1]))
              endelse
              if (*pState).duration lt 0 then $
                 ytit = ytit+'/'+dm_to_string(-(*pState).duration,/int)+' Sec'+(['s',''])[(*pState).duration ge -1] $
              else $
                 ytit = ytit+'/mon='+dm_to_string((*pState).duration,exp=(((*pState).duration mod 10) eq 0))
              end
         'TAS':   begin
              if (*pState).duration lt 0 then begin
                 if (*(*pState).yPtr)[0,0] ne 0 then ytit = ytit+'/'+dm_to_string(-(*pState).duration,/int)+' Sec'+(['s',''])[(*pState).duration ge -1]
              endif else ytit = ytit+'/mon='+dm_to_string((*pState).duration,exp=(((*pState).duration mod 10) eq 0))
              timestamp = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-1,*])
              if timestamp[0] eq 0 then tmp = temporary(timestamp)
              if (*(*pState).detPtr)[0] eq 0 then subbg = 0 ;Monitor as intensity, no need to subtract background
              if subbg then begin
                 bgfac = abs((*pState).duration/*(*pState).bgmonPtr)
                 if finite(bgfac,/nan) then begin
                    ok = dialog_message('The empty can monitor count is 0, no normalization factor is used in empty can subtraction.',dialog_parent=(*pState).tlb,/center)
                    bgfac = 1.0
                 endif
              endif
              if (~(*pState).is_bt7) and stregex(xtit,'motor *[1-3][0-9]',/fold_case,/boolean) then textra = 3  ;these are for motor # higher than motor_12
              end
         'WAND':  if subbg then dcsbgfactor = 1.0     
         else:
    endcase
    showindvdet = (*pState).indvdet
    if (strpos(tit,'det=all') ne -1) and ((*pState).ftype ne 'MACS') then showindvdet = 0b 
    if strpos(tit,'det=upper') ne -1   then showindvdet = 0b 
    if strpos(tit,'det=central') ne -1 then showindvdet = 0b 
    if strpos(tit,'det=lower') ne -1   then showindvdet = 0b 
    if keyword_set(dataonly)           then showindvdet = 0b 
    hidelegend = (n_elements(*(*pState).detPtr) eq 1) or (~showindvdet)
    colrs = (*pState).plotWin->getcolor(/list,linecolor=colr0)  ;colr0 is the first line color
    colrs = colrs[0:n_elements(colrs)-2]                        ;remove the user defined color
    (*pState).plotWin->getproperty,textpos=textpos,textfsize=textfsize
    if n_elements(textpos) eq 2 then (*pState).textposfsize[0:1] = textpos
    if n_elements(textfsize) eq 1 then (*pState).textposfsize[2] = textfsize
    for ii = 0L, 0>(showindvdet*n_det-1) do begin
        if (*pState).xaxis lt 0 then x = lindgen(n_file)+1 else x = reform((*(*pState).xPtr)[(*pState).xaxis+textra,*])
        if (*pState).ftype eq 'MACS' then begin
           if ((*pState).xaxis gt iA4) or (((*pState).xaxis eq iA4) and (~yPt)) then x = reform((*(*pState).xPtr)[(*pState).xaxis+2+textra,*])
        endif
        y = fltarr(n_file) & dy = y
        if n_elements(dcsbgfactor) ne 0 then begin  ;DCS empty can subtraction
           y_bg  = total((*(*pState).bgPtr)[*(*pState).detPtr],/nan)
           dy_bg = sqrt(total(((*(*pState).dbgPtr)[*(*pState).detPtr])^2,/nan))
        endif
        for i=0,n_file-1 do begin
            if showindvdet then begin
               if (*pState).ftype eq 'MACS' then begin
                  if ~yMn and ~yPt then begin
                     if (*pState).y_typ eq 2 then begin ;spec+diff
                        y[i]  = macs_ydat[0,(*(*pState).detPtr)[ii],i]+macs_ydat[1,(*(*pState).detPtr)[ii],i]
                        dy[i] = sqrt((macs_yerr[0,(*(*pState).detPtr)[ii],i])^2+(macs_yerr[1,(*(*pState).detPtr)[ii],i])^2)
                     endif else begin                   ;spec or diff
                        y[i]  = macs_ydat[(*pState).y_typ,(*(*pState).detPtr)[ii],i]
                        dy[i] = macs_yerr[(*pState).y_typ,(*(*pState).detPtr)[ii],i]
                     endelse
                     if xA4 then x[i] = x[i]-76+(*(*pState).tthPtr)[(*(*pState).detPtr)[ii]]
                  endif
               endif else begin
                  y[i]  = (*(*pState).yPtr)[(*(*pState).detPtr)[ii],i]
                  dy[i] = (*(*pState).dyPtr)[(*(*pState).detPtr)[ii],i]
                  if n_elements(dcsbgfactor) ne 0 then begin  ;DCS empty can subtraction
                     y[i]  = y[i]-dcsbgfactor*(*(*pState).bgPtr)[(*(*pState).detPtr)[ii]]
                     dy[i] = sqrt((dy[i])^2+(dcsbgfactor*((*(*pState).dbgPtr)[(*(*pState).detPtr)[ii]]))^2)
                  endif
               endelse
            endif else begin
               if (*pState).ftype eq 'MACS' then begin
                  if ~yMn and ~yPt and ~xA4 then begin
                     if (*pState).y_typ eq 2 then begin ;spec+diff
                        y[i]  = total(macs_ydat[0,*(*pState).detPtr,i],/nan)+total(macs_ydat[1,*(*pState).detPtr,i],/nan)
                        dy[i] = sqrt(total((macs_yerr[0,*(*pState).detPtr,i])^2,/nan)+total((macs_yerr[1,*(*pState).detPtr,i])^2,/nan))
                     endif else begin                   ;spec or diff
                        y[i]  = total(macs_ydat[(*pState).y_typ,*(*pState).detPtr,i],/nan)
                        dy[i] = sqrt(total((macs_yerr[(*pState).y_typ,*(*pState).detPtr,i])^2,/nan))
                     endelse
                  endif
               endif else begin             
                  if ~keyword_set(yMn) then begin
                     y[i]  = total((*(*pState).yPtr)[*(*pState).detPtr,i],/nan)
                     dy[i] = sqrt(total(((*(*pState).dyPtr)[*(*pState).detPtr,i])^2,/nan))
                     if n_elements(dcsbgfactor) ne 0 then begin  ;DCS and WAND empty can subtraction
                        y[i]  = y[i]-dcsbgfactor*y_bg
                        dy[i] = sqrt((dy[i])^2+dcsbgfactor^2*(dy_bg)^2)
                     endif else if subbg then begin ;TAS empty can subtraction
                        if i ge n_elements((*(*pState).bgPtr)[0,*]) then begin
                           n_bgfile = n_elements((*(*pState).bgPtr)[0,*])
                           ok = dialog_message('There are not enough empty can data ('+dm_to_string(n_bgfile)+' instead of '+dm_to_string(n_file)+'). Only background subtracted data are shown.',$
                                dialog_parent=(*pState).tlb,/center)
                           n_file = n_bgfile     
                           break
                        endif
                        y_bg  = total((*(*pState).bgPtr)[*(*pState).detPtr,i],/nan)
                        dy_bg = sqrt(total(((*(*pState).dbgPtr)[*(*pState).detPtr,i])^2,/nan))
                        y[i]  = y[i]-bgfac*y_bg
                        dy[i] = sqrt(dy[i]^2+bgfac^2*dy_bg^2)                     
                     endif
                  endif
               endelse
            endelse
        endfor
        if keyword_set(yMn) then begin
           y  = mon
           dy = sqrt(mon)
           if keyword_set(xA4) then begin
              x  = (fltarr(20)+1)#x
              timestamp = (fltarr(20)+1)#timestamp
              for i=0,19 do begin
                  x[i,*] = x[i,*]-76+(*(*pState).tthPtr)[i]
              endfor
              y  = (fltarr(20)+1)#y
              dy = (fltarr(20)+1)#dy
           endif
        endif 
        if keyword_set(yPt) then begin
           if (*pState).y_typ eq 2 then begin ;spec+diff
              y  = reform(macs_ydat[0,20,*]+macs_ydat[1,20,*])
              dy = sqrt((reform((macs_yerr[0,20,*])))^2+(reform((macs_yerr[1,20,*])))^2)        
           endif else begin                   ;spec or diff
              y  = reform(macs_ydat[(*pState).y_typ,20,*])
              dy = reform(macs_yerr[(*pState).y_typ,20,*])
           endelse
        endif 
        if ~showindvdet and keyword_set(xA4) then begin ;special case of MACS A4
           if ~yMn and ~yPt then begin
              x = (fltarr(n_det)+1)#x
              timestamp = (fltarr(20)+1)#timestamp
              for i=0,n_det-1 do begin
                  x[i,*] = x[i,*]-76+(*(*pState).tthPtr)[(*(*pState).detPtr)[i]]
              endfor
              x  = reform(x)
              if (*pState).y_typ eq 2 then begin ;spec+diff
                 y  = reform(macs_ydat[0,*(*pState).detPtr,*]+macs_ydat[1,*(*pState).detPtr,*])
                 dy = sqrt(reform((macs_yerr[0,*(*pState).detPtr,*]))^2+reform((macs_yerr[1,*(*pState).detPtr,*]))^2)
              endif else begin                   ;spec or diff
                 y  = reform(macs_ydat[(*pState).y_typ,*(*pState).detPtr,*])
                 dy = reform(macs_yerr[(*pState).y_typ,*(*pState).detPtr,*])
              endelse
           endif
        endif
        macsptai = (((*pState).ftype eq 'MACS') and (n_elements(*(*pState).detPtr) eq 1) and ((*(*pState).detPtr)[0] eq -2))
        if (*pState).tempreb_yn and t_axis and (n_file gt 1) and ~keyword_set(dataonly) then begin  ;combine data with same T_set
           case (*pState).ftype of 
                'DCS':  if (*pState).tempcom_yn then T_set = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-2,*]) else T_set = round(x*100.0)/100.0
                'MACS': if (*pState).tempcom_yn then T_set = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-6,*]) else T_set = round(x*100.0)/100.0
                'TAS':  if (*pState).tempcom_yn then T_set = reform((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-2,*]) else T_set = round(x*100.0)/100.0
             else:      T_set = round(x*100.0)/100.0
           endcase
           tmp = T_set  
           uniq_val = tmp[uniq(tmp,sort(tmp))]
           dm_step_bin,0,T_set,ydat=x,uniq_val=uniq_val  
           dm_step_bin,0,tmp,ydat=y,yerr=dy,uniq_val=uniq_val
        endif else if (macsptai or (*pState).is_bt7) and ~keyword_set(dataonly) then begin
           tmpx = dcs_cryst_align_uniq(x,n_val=n_x)
           if n_x eq 1 then x[*] = tmpx
        endif
        lsty = (['no line',(['solid','dashed'])[(ii/15) mod 2]])[~hidelegend]
        if showindvdet then begin
           if (*pState).ftype eq 'DCS' then begin
              if (*(*pState).detPtr)[ii] eq 913 or (*(*pState).detPtr)[ii] eq 914 then begin
                 legd = (['FC2','FC3'])[(*(*pState).detPtr)[ii]-913]
              endif else begin
                 legd = dm_to_string((*(*pState).detPtr)[ii]+1)+([' L',' C',' U'])[dm_mask_detbanks(detnum=(*(*pState).detPtr)[ii])+1]
              endelse 
           endif else legd = dm_to_string((*(*pState).detPtr)[ii]+1)
        endif else legd = ''
        if n_elements(x) gt n_elements(y) then x = x[0:(n_elements(y)-1)] ;may happen if bgdata is less than sample data
        if ii eq 0 then begin
           xs  = x
           ys  = y
           dys = dy
           if ~keyword_set(dataonly) then $
              (*pState).plotWin->setproperty,xdat=x,ydat=y,yerr=dy,/nodraw,xtit=xtit,ytit=ytit,psym='circle',hidelegend=hidelegend,linestyle=lsty,color=colr0,legend=legd
        endif else begin
           xs  = [xs,x]
           ys  = [ys,y]
           dys = [dys,dy]
           if ~keyword_set(dataonly) then $
              (*pState).plotWin->add_plot,x,y,yerr=dy,psym='circle',linestyle=lsty,color=colrs[ii mod n_elements(colrs)],legend=legd
        endelse
    endfor
    if keyword_set(dataonly) then return
    xmax = max(xs,min=xmin) 
    ymax = max(ys+abs(dys),imax) & imax = imax[0]
    ymin = min(ys-abs(dys))
    xran = [xmin,xmax]+[-0.1,0.1]*((xmax eq xmin)?(xmax eq 0?1:abs(xmax)):(xmax-xmin))
    yran = [ymin,ymax]+[-0.1,0.1]*((ymax eq ymin)?(ymax eq 0?1:abs(ymax)):(ymax-ymin))
    if (*pState).fit and (n_elements(y) gt ((*pState).fitfunnterm+((*pState).fitfunchoice eq 2))) and ((~(*pState).indvdet) or (n_elements(*(*pState).detPtr) eq 1)) then begin
       if total(finite(oldxran,/nan)) eq 0 then begin
          index = where((x ge oldxran[0]) and (x le oldxran[1]),count)
          if count gt ((*pState).fitfunnterm+((*pState).fitfunchoice eq 2)) then begin
             x = x[index]
             y = y[index]
             dy = dy[index]
             if n_elements(timestamp) ne 0 then timestamp = timestamp[index]
          endif else if iszoom then ok = dialog_message('There are not enough data points in the zoomed range to fit. All data are used.',dialog_parent=(*pState).tlb,/center)
       endif
       index = where(dy le 0,count)
       if count ne 0 then dy[index] = 1
       index = sort(x) & x = x[index] & y = y[index] & dy = dy[index]
       if n_elements(timestamp) ne 0 then timestamp = timestamp[index]
       is_moffat = ((*pState).fitfunchoice eq 2)
       if (*pState).fixfitwidth then begin
          if (n_elements(fixedwidth) eq 0) then begin
             widget_control,widget_info((*pState).tlb,find_by_uname='fixwidth'),get_value=fixedwidth
             fixedwidth = dm_to_number(fixedwidth)
          endif
       endif else fixedwidth = !values.f_nan
       if ~iszoom then ptr_free,(*pState).fitfunstack
       if ptr_valid((*pState).fitfunstack) and ~(*pState).fit_extrap then begin ;subtract existing peaks before the next fit
          dcs_cryst_align_fitline,pState,(*(*pState).fitfunstack)[*].param,xx=x,yy=yy
          y = (n_elements(dcsbgfactor) ne 0)?(y-yy):(0.0>(y-yy))
       endif
       dm_gaussfit,y,x=x,params=params,lorentzian=((*pState).fitfunchoice eq 1),moffat=is_moffat,nterms=(*pState).fitfunnterm+is_moffat,measure_errors=abs(dy),chisq=chisq,sigma=sigma,/auto,fwhm=fwhm,fixedwidth=fixedwidth
       if (*pState).fixfitwidth and (params[2] ne 0) then begin
          if finite(fixedwidth,/nan) then begin
             wid = widget_info((*pState).tlb,find_by_uname='fixwidth')
             if wid ne 0 then widget_control,wid,set_value=dm_to_string(params[2])
          endif
       endif
       if (total(finite(params,/nan)) eq 0) and (params[2] ne 0) then begin
          if (n_elements(timestamp) ne 0) and (params[1] ge x[0]) and (params[1] le x[n_elements(x)-1]) then begin
             ind = (where(x ge params[1]))[0]
             if ind eq 0 then timestamp = timestamp[0] else timestamp = interpol(timestamp[ind-1:ind],x[ind-1:ind],params[1])
             if arg_present(peaktime) then peaktime = timestamp
          endif else timestamp = 0d
          if ptr_valid((*pState).fitfunstack) then begin
             for i=0,n_elements((*(*pState).fitfunstack))-1 do begin  ;check if the peak has already existed
                 if abs((*(*pState).fitfunstack)[i].param[1]-params[1]) lt max([sigma[1],(*(*pState).fitfunstack)[i].sigma[1]]) and $
                    abs((*(*pState).fitfunstack)[i].param[2]-params[2]) lt max([sigma[2],(*(*pState).fitfunstack)[i].sigma[2]]) then existed = 1b
             endfor
             if ~keyword_set(existed) then *((*pState).fitfunstack) = [*((*pState).fitfunstack),{param:params,sigma:sigma,chisq:chisq,peaktime:timestamp}] 
          endif else (*pState).fitfunstack = ptr_new({param:params,sigma:sigma,chisq:chisq,peaktime:timestamp})  
       endif else ok = dialog_message('Unable to fit the data.',dialog_parent=(*pState).tlb,/center)
       if ptr_valid((*pState).fitfunstack) then begin
          dcs_cryst_align_fitline,pState,(*(*pState).fitfunstack)[*].param,x,xx=xx,yy=yy,xmin=xmin,xmax=xmax
          (*pState).plotWin->add_plot,xx,yy,linestyle='solid',color=(['red','cyan'])[total(abs([255,0,0]-colr0)) eq 0],legend=(['gaussian','lorentzian','moffat'])[(*pState).fitfunchoice]+' fit',layer=0,thick=2
          if ((*pState).ftype eq 'MACS') and (n_elements(tittth) ne 0) then begin
             tmp = min(abs(x-params[1]),ind)
             if ~xA4 then tittth = mean((*(*pState).xPtr)[ikn,ind[0]])-76.0+tittth else tittth = mean((*(*pState).xPtr)[ikn,*])-76.0+tittth 
             tit = strmid(tit,0,titind[1])+dm_to_string(tittth,res=3)+strmid(tit,titind[1]+titlen[1])
             (*pState).tth[1] = tittth
          endif
          (*pState).plotWin->add_text,dcs_cryst_align_fitinfo(pState),(*pState).textposfsize[0],(*pState).textposfsize[1],fontsize=(*pState).textposfsize[2],description='fitinfo'
       endif else (*pState).fit = 0
    endif else begin
       if ((*pState).ftype eq 'MACS') and (n_elements(tittth) ne 0) then begin
          tittth = mean((*(*pState).xPtr)[ikn,*])-76.0+tittth 
          tit = strmid(tit,0,titind[1])+dm_to_string(tittth,res=3)+strmid(tit,titind[1]+titlen[1])
          (*pState).tth[1] = tittth
       endif
       if (n_elements(timestamp) eq 1) and arg_present(peaktime) then peaktime = timestamp[0] 
    endelse
    if ~(*pState).showfname then filename = '' else begin
       filename = (*pState).filename[0]
       if subbg then filename = '('+filename+')-('+(*pState).filename[1]+')'
    endelse
    (*pState).plotWin->setproperty,xran=xran,yran=yran,/nodraw,title=tit,cornertxt=filename,legdcolumns=1,legdshowoutline=0
    (*pState).plotWin->draw
end

;calculate the 2theta angle from q&E
function dcs_cryst_align_2theta2,Ei,q,Ef=Ef
    ki = 2*!dpi/dm_e2lambda(Ei,/double)
    if n_elements(Ef) ne 0 then kf = 2*!dpi/dm_e2lambda(Ef,/double) else kf = ki
    if q gt ki+kf or q lt abs(ki-kf) then return,!values.f_nan else return,acos((ki^2+kf^2-q^2)/2./ki/kf)/!ddtor
end

;calculate the 2theta angle from hkl
function dcs_cryst_align_2theta1,Ei,hkl,dir_latt_parm,Ef=Ef
    q = dcs_cryst_align_qlength1(hkl,dir_latt_parm)
    return,dcs_cryst_align_2theta2(Ei,q,Ef=Ef)
end

;returns a3 and a4, kihkl and kfhkl keywords store the hkl of ki and kf in the lattice frame
;if dcs keyword is set, A3 is tof psi, and the sense of rotation (ccw) is opposite to that of TAS (cw)
;lefthand keyword is used to return negative a4
;Ei,Ef: [n_E], Q:[3]  returns [2,n_E], kihkl,kfhkl:[3,n_E]
function dcs_cryst_align_a3a4,Ei,Ef,Q,dcs=dcs,lattparm=lattparm,lefthand=lefthand,kihkl=kihkl,kfhkl=kfhkl,U=U,V=V,projU1=projU1,projU2=projU2,projU3=projU3,abc_r=abc_r,error=error
    n_E = n_elements(Ei)
    if n_elements(abc_r) eq 0 then begin
       dm_u1u2u3,lattparm,U,V,U,V,U1=projU1,U2=projU2,U3=projU3,abc_r=abc_r,error=error
       projU3 = crossp(projU1,projU2)
       if keyword_set(error) then return,[0.,0]
    endif
    qlen = norm(abc_r#Q)
    ki   = 2.*!pi/dm_e2lambda(Ei,/double)
    kf   = 2.*!pi/dm_e2lambda(Ef,/double)
    a1   = acos((qlen^2+ki^2-kf^2)/2./qlen/ki)/!dtor
    a2   = dcs_cryst_align_angle1(Q,U,lattparm,abc_r=abc_r,hkl3=V)
    if keyword_set(lefthand) then begin
       a3 = (90d)+a2+a1
       a4 = -acos((ki^2+kf^2-qlen^2)/2./ki/kf)/!dtor
    endif else begin
       a3 = (90d)+a2-a1
       a4 = acos((ki^2+kf^2-qlen^2)/2./ki/kf)/!dtor
    endelse
    error = (finite(a4,/nan) or finite(a1,/nan))
    a3 = a3+(a3 lt -180)*(360d)-(a3 gt 180)*(360d)  ;range [-180,180]
    if arg_present(kihkl) or arg_present(kfhkl) then begin
       ;now rotate the instrument to the sample frame, a3 clockwise is positive
       r_a3  = a3*!dtor
       tmpki = dm_q2hkl(transpose([[ki*sin(r_a3)],[-ki*cos(r_a3)],[replicate(0.0,n_E)]]),[[projU1],[projU2],[projU3]])
       kihkl = matrix_multiply(U,reform(tmpki[0,*]))+matrix_multiply(V,reform(tmpki[1,*]))
       kfhkl = kihkl-Q#replicate(1,n_E)       
    endif
    if keyword_set(dcs) then a3=90.0-temporary(a3)
    return,reform(transpose([[temporary(a3)],[temporary(a4)]]))
end

;calculate the angle between hkl1 and hkl2, hkl1=[3], hkl2=[3]
;if hkl3=[3] is present,the angle is negative if hkl3 and hkl1 are not on the same side of hkl2
function dcs_cryst_align_angle1,hkl1,hkl2,dir_latt_parm,abc_r=abc_r,hkl3=hkl3
    if n_elements(abc_r) eq 0 then abc_r = dm_basis_r(dir_latt_parm)
    q1  = abc_r#hkl1
    q2  = abc_r#hkl2
    ang = (acos((-1.0)>(matrix_multiply(q2,q1,/atranspose)/norm(q1)/norm(q2))<1.0)/!ddtor)[0]
    if n_elements(hkl3) eq 3 then begin
       q3 = abc_r#hkl3
       if matrix_multiply(crossp(q2,q1),crossp(q2,q3),/atran) lt 0 then ang=ang*(-1d)
    endif
    return,ang
end

;calculate the angle between two vectors from 2 sets of A3s & A4s
;if DCS keyword is set, A3 is tof psi, and the sense of rotation (ccw) is opposite to that of TAS (cw)
function dcs_cryst_align_angle2,A3s,A4s,DCS=DCS
    A1  = (-1)^(~keyword_set(DCS))*A3s[0]+(90.0-abs(A4s[0])/2.0)*((A4s[0] lt 0)*2.0-1.0)
    A2  = (-1)^(~keyword_set(DCS))*A3s[1]+(90.0-abs(A4s[1])/2.0)*((A4s[1] lt 0)*2.0-1.0)
    tmp = abs(A1-A2)
    return,([tmp,360-tmp])[tmp gt 180]
end

;calculate hkl values from a3 and a4
;if dcs keyword is set, A3 is tof psi, and the sense of rotation (ccw) is opposite to that of TAS (cw)
function dcs_cryst_align_hkl,Ei,Ef,a3,a4,lattparm=lattparm,U=U,V=V,dcs=dcs,error=error
    n_Ei = n_elements(Ei)
    n_Ef = n_elements(Ef)
    n_a3 = n_elements(a3)
    if (n_Ei eq 0) or (n_Ef eq 0) or (n_a3 eq 0) or (n_a3 ne n_elements(a4)) or (n_elements(lattparm) ne 6) or (n_elements(U) ne 3) or (n_elements(V) ne 3) then error = 1b $
    else dm_u1u2u3,lattparm,U,V,U,V,U1=U1,U2=U2,U3=U3,error=error
    if keyword_set(error) then return,[0,0,0.]
    if keyword_set(dcs) then psi= a3 else psi = 90.0-a3
    hkl = fltarr(3,n_a3)
    for i=0L,n_a3-1 do begin
        Q  = reform(dm_proj_spec_samp(Ei[i<(n_Ei-1)],Ei[i<(n_Ei-1)]-Ef[i<(n_Ef-1)],a4[i],0,psi=psi[i],type=1,is_spe=0,instrgeom=0))
        QV = dm_proj_view(Q,U1,U2,U3)
        hkl[*,i] = QV[0]*U+QV[1]*V
    endfor
    return,reform(hkl)
end

;calculate the lattice parameter from two theta and hkl, two solutions possible
function dcs_cryst_align_latta,Ei,tth,hkl,abg,a=a,b=b,c=c
    if (n_elements(Ei) ne 1) or (n_elements(tth) ne 1) or (n_elements(hkl) ne 3) or (n_elements(abg) ne 3) then $
       message,['Incorrect number of arguments. ','Usuage: latt = dcs_cryst_align_latta(Ei, tth, [h, k, l], [alpha, beta, gamma], a=a, b=b, c=c) ','Ei and tth are scalars.']
    abc = [!values.d_nan,!values.d_nan,!values.d_nan]
    if n_elements(a) ne 0 then abc[0] = a[0]
    if n_elements(b) ne 0 then abc[1] = b[0]
    if n_elements(c) ne 0 then abc[2] = c[0]
    hkl0 = [0d,0d,0d]
    hkl1 = double(hkl)
    abc0 = [1d,1d,1d]
    qlen = dcs_cryst_align_qlength2(Ei,tth)
    ind0 = where(finite(abc,/nan),cnt0,comp=ind1,ncomp=cnt1)
    if cnt1 gt 0 then begin ;subtract the fixed part from hkl
       hkl0[ind1] = hkl[ind1]
       abc0[ind1] = abc[ind1]
       hkl1[ind1] = 0d
       ang1  = dcs_cryst_align_angle1(hkl0,hkl1,[1d,1d,1d,abg])*!ddtor
       qlen0 = dcs_cryst_align_qlength1(hkl0,[abc0,abg])
       ang2  = asin(qlen0*sin(ang1)/qlen)*[1d,-1d]+[0d,!dpi]
       ind2  = where(ang2 lt ang1,cnt2)
       if finite(ang1) and finite(ang2[0]) and (cnt2 gt 0) then begin
          qlen = sin(ang1-ang2[ind2])/sin(ang1)*qlen
          if n_elements(qlen) eq 2 then abc = abc#[1d,1d]
       endif else return,abc
    endif
    if cnt0 gt 0 then begin
       latta = dcs_cryst_align_qlength1(hkl1,[1d,1d,1d,abg])/qlen
       if (latta[0] lt 2) or (latta[0] gt 15) then latta = reverse(latta) ;push unlikely solution to the back  
       for i=0,n_elements(qlen)-1 do abc[ind0,i] = latta[i]
    endif
    return,abc
end

;calculate q length from direct crystal lattice parameters, hkl = [3,n_q]
function dcs_cryst_align_qlength1,hkl,cryst
    sinabg  = sin(cryst[3:5]*!ddtor)
    cosabg  = cos(cryst[3:5]*!ddtor)
    v       = cryst[0]*cryst[1]*cryst[2]*sqrt((1d)+(2d)*cosabg[0]*cosabg[1]*cosabg[2]-total(cosabg[0:2]^2))
    rec_abc = (2d)*!dpi/v*[cryst[1]*cryst[2]*sinabg[0],cryst[0]*cryst[2]*sinabg[1],cryst[0]*cryst[1]*sinabg[2]]
    tmp     = (4d)*!dpi*!dpi/v/v*cryst[0]*cryst[1]*cryst[2]
    aadotbb = tmp*cryst[2]*(cosabg[0]*cosabg[1]-cosabg[2])
    bbdotcc = tmp*cryst[0]*(cosabg[1]*cosabg[2]-cosabg[0])
    ccdotaa = tmp*cryst[1]*(cosabg[2]*cosabg[0]-cosabg[1])
    result  = sqrt(matrix_multiply(hkl^2,rec_abc^2,/atranspose)+(2d)*(hkl[0,*]*hkl[1,*]*aadotbb+hkl[1,*]*hkl[2,*]*bbdotcc+hkl[2,*]*hkl[0,*]*ccdotaa))
    if n_elements(result) eq 1 then return,result[0] else return,result
end

;calculate q length from Ei, Ef and 2theta
function dcs_cryst_align_qlength2,Ei,tth,Ef=Ef
    ki = 2*!dpi/dm_e2lambda(Ei,/double)
    if n_elements(Ef) ne 0 then kf = 2*!dpi/dm_e2lambda(Ef,/double) else kf = ki
    return,sqrt(ki^2+kf^2-2*ki*kf*cos(tth*!ddtor))
end

;change file type
pro dcs_cryst_align_changeftype,event,init=init,pState=pState,det_info=det_info,x_info=x_info,keepcurrentdir=keepcurrentdir
    if n_elements(event) ne 0 then begin
       widget_control,event.handler,get_uvalue=pState
       uname = widget_info(event.id,/uname)
       if strmid(uname,0,5) ne 'ftype' then return
       ftype = strmid(uname,6)
       if (*pState).ftype eq ftype then return else (*pState).ftype = ftype
    endif
    ;initialize the instrument info
    (*pState).t_chan = 1
    (*pState).e_ran  = [!values.f_nan,!values.f_nan]
    (*pState).tth    = [!values.f_nan,!values.f_nan]
    (*pState).xaxis  = 0
    (*pState).fit    = 0
    if ptr_valid((*pState).fitfunstack) then ptr_free,(*pState).fitfunstack
    (*pState).textposfsize = [0.05,0.93,14] ;set them to default every time the file type is changed
    case (*pState).ftype of   ;temperature needs to be the last entry of xtit
         'DCS': begin     ;DCS
                alldetPtr  = dm_load_detpos(/dcs)
                det_info   = ['All','Specify','Central Bank','Upper Bank','Lower Bank','FC2','FC3']
                x_info     = ['Rotation','Height','X-translation','Y-translation','X-tilt','Y-tilt','File Index','Magnetic Field','Temperature']
                *(*pState).xtit = ['Rotation (\deg)','Height (mm)','X-translation (mm)','Y-translation (mm)','X-tilt (\deg)','Y-tilt (\deg)','File Index','\mu!i0!nH (T)','T (K)']
                (*pState).tch_tit = 'tchan='+(['all','elastic','specify'])[(*pState).t_chan]
                end
         'MACS':begin    ;MACS
                alldetPtr  = dm_load_detpos(/macs)
                det_info   = ['All','Specify','Monitor','PTAI']
                x_info     = ['H','K','L','E','A3','A4','A1','A2','A5','A6','MonRot','MonTrans','DFMDTS','MBTSlide','MonBlade'+string(1+indgen(21),format='(i02)'),'AnalyzerTheta'+string(1+indgen(20),format='(i02)'),$
                              'Smpl'+['X','Y','Z','LTilt','UTilt'],'HSlit','VSlit','DMBT','Focus','Beta1','Beta2','VBAH','VBAV','MagField','Temperature']
                *(*pState).xtit = ['H','K','L','E (meV)',['A3','A4','A1','A2','A5','A6','MonRot']+' (\deg)',['MonTrans','DFMDTS']+' (mm)','MBTSlide (cm)','MonBlade'+string(1+indgen(21),format='(i02)')+' (\deg)',$
                              'AnalyzerTheta'+string(1+indgen(20),format='(i02)')+' (\deg)',['X','Y','Z']+' (mm)',['Lower','Upper']+'-tilt (\deg)',['H','V']+'Slit (mm)','DMBT (mm)','Focus',['Beta1','Beta2']+' (\deg)',$
                              ['VBAH','VBAV']+' (mm)','\mu!i0!nH (T)','T (K)']
                (*pState).tch_tit = ''
                end
         'WAND':begin    ;WAND
                alldetPtr  = dm_load_detpos(/wand)
                det_info   = ['All','Specify']
                x_info     = ['Psi','Temperature']
                *(*pState).xtit = ['Psi (\deg)','T (K)']
                (*alldetPtr).two_theta  = (*alldetPtr).two_theta*(-1)
                (*(*pState).infoPtr)[0] = 1.46 ;fixed wavelength
                (*pState).tch_tit = ''
                (*pState).ncnrftp = 0
                end
         'TAS': begin    ;TAS
                det_info   = ['Monitor','Detector']
                x_info     = ['H','K','L','E','A3','A4','A1','A2','A5','A6','Smpl'+['LTrn','UTrn','Wdth','Hght','LTilt','UTilt'],'MagField','Temp']
                *(*pState).xtit = ['H','K','L','E (meV)',['A3','A4','A1','A2','A5','A6']+' (\deg)',[['Lower','Upper']+'-translation','Beam Width','Beam Height']+' (mm)',['Lower','Upper']+'-tilt (\deg)',$
                              '\mu!i0!nH (T)','T (K)']
                (*pState).tch_tit = ''
                (*pState).indvdet = 0
                (*pState).macs_normchoice = 0 ;referenced in calculate flipping ratio tool
                end
         else:
    endcase
    iftype = (where(*(*pState).fname eq (*pState).ftype))[0]
    if ptr_valid(alldetPtr) then begin
       *(*pState).tthPtr = (*alldetPtr).two_theta 
       ptr_free,alldetPtr
    endif else *(*pState).tthPtr = 0
    (*pState).ndet    = n_elements(*(*pState).tthPtr)
    *(*pState).detPtr = indgen(1>(*pState).ndet)  ;default all detectors
    (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det=all'
    (*pState).title   = (*pState).det_tit
    if strlen((*pState).tch_tit) ne 0 then (*pState).title=(*pState).title+', '+(*pState).tch_tit

    ;construct detector droplist
    if (*pState).ftype eq 'DCS' then begin ;DCS det bank info
       tmp   = string(*(*pState).tthPtr,format='(f8.2)')
       tmp   = string(indgen((*pState).ndet)+1,format='(i3)')+tmp+string('b0'xb)
       index = dm_mask_detbanks(0)
       tmp[index] = tmp[index]+'  c' & index=-1  ;central bank
       index = dm_mask_detbanks(1)
       tmp[index] = tmp[index]+'  u' & index=-1  ;upper bank
       index = dm_mask_detbanks(-1)
       tmp[index] = tmp[index]+'  l' & index=-1  ;lower bank
    endif else begin
       tmp   = string(indgen((*pState).ndet)+1,format='(i'+dm_to_string(ceil(alog10((*pState).ndet)))+')')+'   '+dm_to_string(*(*pState).tthPtr,res=3)+string('b0'xb)
    endelse
    if (*pState).ftype ne 'TAS' then det_info = [det_info,tmp]

    if keyword_set(init) then begin
       *(*pState).fname   = ['DCS','MACS','WAND','TAS']
       *(*pState).fextn   = ['.dcs','.ng0,.bt9','.wand','.bt2,.ba2,.bb2,.bc2,.bd2,.bt4,.bt7,.ba7,.bb7,.bc7,.bd7,.bt9,.ba9,.bb9,.bc9,.bd9,.ng5,.phd']
       *(*pState).psiname = ['A2','A3','Psi','A3']
    endif else begin
       widget_control,(*pState).tlb,update=0
       ftypemenu = widget_info((*pState).mbar,find_by_uname='ftypeMenu')
       checked   = widget_info(ftypemenu,find_by_uname='ftype_'+(*pState).ftype)
       for i=0,n_elements(*(*pState).fname)-1 do begin
           if (*(*pState).fname)[i] ne (*pState).ftype then begin
              tmp = widget_info(ftypemenu,find_by_uname='ftype_'+(*(*pState).fname)[i])
              unchecked = (n_elements(unchecked) eq 0)?tmp:([unchecked,tmp])
           endif
       endfor
       dm_toggle_menubut,check=checked,uncheck=unchecked
       widget_control,widget_info((*pState).mbar,find_by_uname='optnbitflip'),sensitive=((*pState).ftype eq 'DCS')      
       widget_control,widget_info((*pState).mbar,find_by_uname='optnindvdet'),sensitive=((*pState).ftype ne 'TAS')
       dm_set_button,widget_info((*pState).mbar,find_by_uname='optnindvdet'),(*pState).indvdet,onstring='Show Individual Detector',offstring='Sum Detectors'
       widget_control,widget_info((*pState).mbar,find_by_uname='temp_samp'),sensitive=((*pState).ftype ne 'WAND')
       widget_control,widget_info((*pState).mbar,find_by_uname='temp_ctrl'),sensitive=((*pState).ftype ne 'WAND')
       widget_control,widget_info((*pState).mbar,find_by_uname='temp_setp'),sensitive=((*pState).ftype ne 'WAND')
       widget_control,widget_info((*pState).mbar,find_by_uname='temp_comb'),sensitive=(((*pState).ftype ne 'WAND') and (*pState).tempreb_yn)
       widget_control,widget_info((*pState).mbar,find_by_uname='temp_avg'),sensitive=((*pState).ftype ne 'DCS')
       dcs_cryst_align_setoption,pState,'tempchoice',(*pState).tchoice  ;set up temperature choice menu name
       optnmenu = widget_info((*pState).mbar,find_by_uname='optnMenu')
       toolmenu = widget_info((*pState).mbar,find_by_uname='toolMenu')
       kidlimitBut = widget_info(toolmenu,find_by_uname='kidlimitbut')
       flipratBut  = widget_info(toolmenu,find_by_uname='macsflipratio')
       ncnrftpBut  = widget_info(optnmenu,find_by_uname='ncnrftp')
       if flipratBut ne 0 then widget_control,flipratBut,/destroy
       if ncnrftpBut ne 0 then widget_control,ncnrftpBut,/destroy
       if (*pstate).ftype eq 'MACS' then begin  ;MACS
          if ((*pState).macs_histo eq 0) and ((*pState).macs_normchoice eq 2) then (*pState).macs_normchoice = 0
          normmenu = widget_button(optnmenu,value='Intensity Normalization Choice ('+(['Monitor','Time','Pulse Number'])[(*pState).macs_normchoice]+')',uname='normMenu',/menu)
          void = dm_widget_button(normmenu,value='Divided by Monitor Count',uname='norm_monitor',set_button=((*pState).macs_normchoice eq 0))
          void = dm_widget_button(normmenu,value='Divided by Counting Time',uname='norm_time',set_button=((*pState).macs_normchoice eq 1))
          void = dm_widget_button(normmenu,value='Divided by Pulse Number',uname='norm_pulse',set_button=((*pState).macs_normchoice eq 2),sensitive=keyword_set((*pState).macs_histo))
          void = dm_widget_button(normmenu,value='Apply Monitor Lambda/2 Correction',set_button=(*pState).macs_lamb2,onstring='Apply Monitor Lambda/2 Correction',$
                                  offstring='No Monitor Lambda/2 Correction',uname='optnapplylamb2',/separator)
          macseffmenu = dm_widget_button(optnmenu,value='Apply Detector Efficiency ('+(['None','From File','Specify'])[(*pState).macs_eff]+')',set_button=((*pState).macs_eff ne 0),uname='macseffMenu',/menu)
          void = dm_widget_button(macseffmenu,value='From File',uname='macs_eff_file',set_button=((*pState).macs_eff eq 1))
          void = dm_widget_button(macseffmenu,value='Specify...',uname='macs_eff_spec',set_button=((*pState).macs_eff eq 2))
          void = dm_widget_button(macseffmenu,value='None',uname='macs_eff_none',set_button=((*pState).macs_eff eq 0),sensitive=((*pState).macs_eff ne 0),/separator)
          macshistomenu = widget_button(optnmenu,value='MACS Data Mode ('+(['Normal','Histogram','Event'])[(*pState).macs_histo]+')',uname='macshistoMenu',/menu)
          void = dm_widget_button(macshistomenu,value='Normal',uname='macs_histo0',set_button=((*pState).macs_histo eq 0))
          void = dm_widget_button(macshistomenu,value='Histogram',uname='macs_histo1',set_button=((*pState).macs_histo eq 1))
          void = dm_widget_button(macshistomenu,value='Event',uname='macs_histo2',set_button=((*pState).macs_histo eq 2))
          if kidlimitBut eq 0 then begin
             void = widget_button(toolmenu,value='Kidney Angle Limit',uname='kidlimitbut')
             void = widget_button(toolmenu,value='Fit and Plot Instrument Alignment Files',uname='macsalignment',/separator)
             void = widget_button(toolmenu,value='Fit and Plot Monochromator Alignment Files',uname='macsalignmonchr')
             void = widget_button(toolmenu,value='Fit and Plot Analyzer Alignment Files',uname='macsalignanalyzer')
             void = widget_button(toolmenu,value='Fit and Plot Kidney Alignment File',uname='macsalignkidney')
             void = widget_button(toolmenu,value='Calculate Flipping Ratio',uname='macsflipratio',/separator)
          endif
       endif else begin
          normmenu = widget_info(optnmenu,find_by_uname='normMenu')
          macseffmenu = widget_info(optnmenu,find_by_uname='macseffMenu')  
          macshistomenu = widget_info(optnmenu,find_by_uname='macshistoMenu')
          if macshistomenu ne 0 then widget_control,macshistomenu,/destroy   
          if macseffmenu ne 0 then widget_control,macseffmenu,/destroy
          if normmenu ne 0 then widget_control,normmenu,/destroy         
          if kidlimitBut ne 0 then begin
             widget_control,widget_info(toolmenu,find_by_uname='macsalignkidney'),/destroy
             widget_control,widget_info(toolmenu,find_by_uname='macsalignanalyzer'),/destroy
             widget_control,widget_info(toolmenu,find_by_uname='macsalignmonchr'),/destroy
             widget_control,widget_info(toolmenu,find_by_uname='macsalignment'),/destroy
             widget_control,kidlimitBut,/destroy
          endif
          if (*pState).ftype eq 'TAS' then void = widget_button(toolmenu,value='Calculate Flipping Ratio',uname='macsflipratio',/separator)
       endelse
       if (*pState).ftype ne 'WAND' then void = dm_widget_button(optnmenu,value='Allow NCNR Data Repository',uname='ncnrftp',set_button=(*pState).ncnrftp,/separator)
       widget_control,(*pState).tlb,tlb_set_title=(*pState).ftype+([' Single Crystal',''])[(*pState).ftype eq 'MACS']+' Alignment Utility'
        widget_control,widget_info((*pState).mbar,find_by_uname='clearbgfile'),sensitive=0
       widget_control,widget_info((*pState).mbar,find_by_uname='calca2but'),set_value='Calculate '+(*(*pState).psiname)[iftype]+' Reset Value...'
       widget_control,widget_info((*pState).mbar,find_by_uname='calca2rot'),set_value='Calculate '+(*(*pState).psiname)[iftype]+' Rotation Angle...'
       widget_control,widget_info((*pState).mbar,find_by_uname='calca3a4'),set_value='Calculate '+$
          (['A3 && A4','A2 && 2theta','Psi && 2theta'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]+'...'
       widget_control,widget_info((*pState).mbar,find_by_uname='calchkl'),set_value='Calculate hkl from '+$
          (['A3 && A4','A2 && 2theta','Psi && 2theta'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]+'...'
       widget_control,widget_info((*pState).mbar,find_by_uname='calcvecang'),set_value='Calculate Angle between Two Vectors from '+$
          (['A3s && A4s','A2s && 2thetas','Psis && 2thetas'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]+'...'
       widget_control,widget_info((*pState).tlb,find_by_uname='detlabel'),set_value=(['Det. Number','Detector'])[(*pState).ftype eq 'TAS']
       dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=([0,1])[(*pState).ftype eq 'TAS'],set_value=det_info
       sensitive = (((*pState).ftype ne 'WAND') and ((*pState).ftype ne 'TAS'))
       tmpind = ((*pState).ftype eq 'MACS')+2*(1-sensitive)
       widget_control,widget_info((*pState).tlb,find_by_uname='timelabel'),set_value=(['Time Channel','Det. Type',' '])[tmpind],sensitive=sensitive
       dm_set_droplist,widget_info((*pState).tlb,find_by_uname='timelist'),select=([(*pState).t_chan,(*pState).y_typ,0])[tmpind],set_value=$
          ([['All','Elastic','Specify',' '],['SPEC','DIFF','SPEC+DIFF','Monitor'],[' ',' ',' ',' ']])[0:([2,3,0])[tmpind],tmpind],sensitive=sensitive
       dm_set_droplist,widget_info((*pState).tlb,find_by_uname='xaxislist'),select=(*pState).xaxis,set_value=x_info
       ptr_free,(*pState).xPtr,(*pState).yPtr,(*pState).dyPtr,(*pState).bgPtr,(*pState).dbgPtr,(*pState).bgmonPtr
       (*pState).xPtr     = ptr_new(/allocate_heap)
       (*pState).yPtr     = ptr_new(/allocate_heap)
       (*pState).dyPtr    = ptr_new(/allocate_heap)
       (*pState).bgPtr    = ptr_new(/allocate_heap)
       (*pState).dbgPtr   = ptr_new(/allocate_heap)
       (*pState).bgmonPtr = ptr_new(/allocate_heap)
       (*pState).filesel->set_filter,(*(*pState).fextn)[iftype],keepcurrentdir=keepcurrentdir
       (*pState).filesel->getproperty,ncnrftp=ncnrftp
       if (*pState).ncnrftp ne ncnrftp then (*pState).filesel->set_ftpbuffer,(['',dm_define_pointer(/gettempdir)])[(*pState).ncnrftp]
       (*pState).plotWin->erase
       widget_control,(*pState).tlb,/map,/update
    endelse
end

;generate a load button event
pro dcs_cryst_align_reload,pState,uname=uname,_extra=extra
    if n_elements(uname) eq 0 then uname='loadbut'
    load = widget_info((*pState).tlb,find_by_uname=uname)
    dcs_cryst_align_event,{WIDGET_BUTTON,ID:load,TOP:(*pState).tlb,HANDLER:(*pState).tlb,SELECT:1},_extra=extra
end

pro dcs_cryst_align_setfilehistory,pState,dir,files,choose=choose,ftpobj=ftpobj
    (*pState).fileSel->getproperty,ftpserver=ftpserver
    if n_elements(choose) eq 1 then begin
       ftype = strmid((*((*pState).file_hist[choose]))[0],6)
       if ftype ne (*pState).ftype then begin
          wid = widget_info((*pState).tlb,find_by_uname='ftype_'+ftype)
          dcs_cryst_align_changeftype,{WIDGET_BUTTON,ID:wid,TOP:(*pState).tlb,HANDLER:(*pState).tlb,SELECT:1}
       endif
       (*pState).filesel->set_path,(*((*pState).file_hist[choose]))[2]
       (*pState).filesel->getproperty,dFiles=dFiles
       for i=3,n_elements(*((*pState).file_hist[choose]))-1 do begin
           ind = where(dFiles eq (*((*pState).file_hist[choose]))[i],count)
           if count ne 0 then begin
              if n_elements(selected) eq 0 then selected=ind $
              else selected=[selected,ind]
           endif
       endfor
       if n_elements(selected) ne 0 then (*pState).filesel->set_file,selected,/highlight
       return
    endif
    num = n_elements((*pState).file_hist)    ;capacity
    nfiles = n_elements(files)
    if (n_elements(dir) ne 0) and (nfiles ne 0) then begin
       if obj_valid(ftpobj) then dir1 = ftpserver+dir else dir1 = dir
       info = files[0] & if nfiles gt 1 then info = info+' to '+files[nfiles-1]
       for i=0,(*pState).n_history-1 do begin  ;check if this entry already exists, if existed move to the top
           if dir1 ne (*((*pState).file_hist[i]))[2] then continue
           if n_elements(files) ne n_elements(*((*pState).file_hist[i]))-3 then continue
           same = 1b
           for j=0, n_elements(files)-1 do begin
               if files[j] ne (*((*pState).file_hist[i]))[3+j] then same=0b
           endfor
           if same then begin ;entry exists, move that entry to the top
              if i ne 0 then begin
                 (*pState).file_hist[0:i] = shift((*pState).file_hist[0:i],1)
                 for j=0,i do widget_control,widget_info((*pState).tlb,find_by_uname='fhist_'+dm_to_string(j,/int)),set_value=dm_to_string(j,/int)+' '+(*((*pState).file_hist[j]))[1]
              endif
              return
           endif
       endfor
       if (*pState).n_history lt num then begin ;empty room available
          if (*pState).n_history ne 0 then $
             (*pState).file_hist[1:(*pState).n_history] = (*pState).file_hist[0:(*pState).n_history-1]
          (*pState).file_hist[0] = ptr_new(['ftype_'+(*pState).ftype,info,dir1,files])
          (*pState).n_history = (*pState).n_history+1
       endif else begin                    ;full
          ptr_free,(*pState).file_hist[num-1]
          (*pState).file_hist[1:num-1] = (*pState).file_hist[0:num-2]
          (*pState).file_hist[0] = ptr_new(['ftype_'+(*pState).ftype,info,dir1,files])
       endelse
    endif
    histmenu = widget_info((*pState).tlb,find_by_uname='fhistMenu')
    widget_control,histmenu,sensitive=((*pState).n_history gt 0)
    widget_control,widget_info((*pState).tlb,find_by_uname='clearHistory'),sensitive=((*pState).n_history gt 0)
    for i=0,num-1 do begin
        tmp = widget_info(histmenu,find_by_uname='fhist_'+dm_to_string(i,/int))
        if tmp ne 0 then widget_control,tmp,/destroy
    endfor
    for i=0,(*pState).n_history-1 do begin
        void = widget_button(histmenu,value=dm_to_string(i,/int)+' '+(*((*pState).file_hist[i]))[1],uname='fhist_'+dm_to_string(i))
    endfor
end

;set up options menu
pro dcs_cryst_align_setoption,pState,option,choice,tlb=tlb,plot=plot,clearfitstack=clearfitstack,calcfr=calcfr,setplotcursor=setplotcursor
    if n_elements(tlb) eq 0 then tlb = (*pState).tlb
    if keyword_set(clearfitstack) and ptr_valid((*pState).fitfunstack) then ptr_free,(*pState).fitfunstack
    case option of 
         'fixwidth': begin
              (*pState).fixfitwidth = (0>(choice)<1)
              dm_set_button,widget_info(tlb,find_by_uname='optnfixwidth'),(*pState).fixfitwidth,onstring='Fix Fitting Width',offstring='Not Fix Fitting Width'
              end
         'showfit': begin
              (*pState).showfit = (0>(choice)<1)
              dm_set_button,widget_info(tlb,find_by_uname='optnshowfit'),(*pState).showfit,onstring='Show Fitting Parameters',offstring='Hide Fitting Parameters'
              if tlb eq (*pState).tlb then (*pState).plotWin->change_title,0,textobjdescp='fitinfo',new=dcs_cryst_align_fitinfo(pState)
              end
         'showcpos': begin
              (*pState).showcursor = (0>(choice)<1)
              dm_set_button,widget_info(tlb,find_by_uname='optnshowcursor'),(*pState).showcursor,onstring='Show Cursor Position',offstring='Hide Cursor Position'   
              if keyword_set(setplotcursor) then (*pState).plotWin->setproperty,showcursorpos=(*pState).showcursor
              end
         'fit_extrap':begin
             (*pState).fit_extrap = (0>(choice)<1)
             dm_set_button,widget_info(tlb,find_by_uname='fit_extrap'),(*pState).fit_extrap 
             end
         'funcchoice': begin
             (*pState).fitfunchoice = (0>(choice)<2)
             ids = [widget_info(tlb,find_by_uname='fit_gaussian'),widget_info(tlb,find_by_uname='fit_lorentzian'),widget_info(tlb,find_by_uname='fit_moffat')]
             termid = [widget_info(tlb,find_by_uname='fitfun_3'),widget_info(tlb,find_by_uname='fitfun_4'),widget_info(tlb,find_by_uname='fitfun_5'),widget_info(tlb,find_by_uname='fitfun_6')]
             dm_toggle_menubut,check=ids[(*pState).fitfunchoice],uncheck=ids[where([0,1,2] ne (*pState).fitfunchoice)]
             widget_control,widget_info(tlb,find_by_uname=(['fitfcMenu','fitmenu'])[tlb ne (*pState).tlb]),set_value='Fitting Function ('+(['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+')'
             for i=0,3 do widget_control,termid[i],set_value=(*pState).fitfunname[i,(*pState).fitfunchoice],sensitive=((i ne 3) or (((*pState).fitfunchoice eq 0) and (~(*pState).fixfitwidth)))
             if ((*pState).fitfunnterm eq 6) and ((*pState).fitfunchoice ne 0) then begin
                (*pState).fitfunnterm = 5
                dm_toggle_menubut,check=termid[2],uncheck=termid[3]
             endif
             if tlb eq (*pState).tlb then widget_control,widget_info(tlb,find_by_uname='fitbut'),set_value=(['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+' Fit'
             end    
         'macseff': begin
             if choice eq 2 then begin
                new = dm_dialog_listinput('det '+dm_to_string(1+indgen(20)),title='Please input new efficiencies:',default=(*pState).macs_deteff[2:3,*],cancel=cancel,$
                      label=['SPEC','DIFF'],info=['Intensity will be multiplied by these values.'],dialog_parent=tlb)
                if keyword_set(cancel) then return
                ind = where(finite(new,/nan),cnt)
                if cnt ne 0 then begin
                   ok = dialog_message((['Invalid input for '+(['SPEC ','DIFF '])[ind[0] mod 2]+dm_to_string(ind[0]/2+1)+'.','There are invalid inputs.'])[cnt gt 1],/error,dialog_parent=tlb,/center)
                   return
                endif
                (*pState).macs_deteff[2:3,*] = new
             endif
             (*pState).macs_eff = (0>(choice)<2)
             ids = [widget_info(tlb,find_by_uname='macs_eff_none'),widget_info(tlb,find_by_uname='macs_eff_file'),widget_info(tlb,find_by_uname='macs_eff_spec')]
             widget_control,ids[0],sensitive=((*pState).macs_eff ne 0)
             dm_toggle_menubut,check=ids[(*pState).macs_eff],uncheck=ids[where([0,1,2] ne (*pState).macs_eff)]
             dm_set_button,widget_info(tlb,find_by_uname='macseffMenu'),((*pState).macs_eff ne 0),set_value='Apply Detector Efficiency ('+(['None','From File','Specify'])[(*pState).macs_eff]+')'
             end
         'macshisto': begin
             (*pState).macs_histo = (0>(choice)<2)
             ok = dialog_message((['Reverting to regular data mode. Histogram data in the files will be discarded.',$
                  'Switching to histogram data mode. Regular SPEC & DIFF  data in the files will be discarded. Data are summed over time channel.',$
                  'Switching to event data mode. 2D binary .rgo histogram data files must be present in the data directory. Data are summed over time channel.'])[(*pState).macs_histo],$
                  dialog_parent=tlb,/center,/info)
             ids = [widget_info(tlb,find_by_uname='macs_histo0'),widget_info(tlb,find_by_uname='macs_histo1'),widget_info(tlb,find_by_uname='macs_histo2')]
             dm_toggle_menubut,uncheck=ids[where([0,1,2] ne (*pState).macs_histo)],check=ids[(*pState).macs_histo]
             widget_control,widget_info(tlb,find_by_uname='norm_pulse'),sensitive=keyword_set((*pState).macs_histo)
             widget_control,widget_info(tlb,find_by_uname='macshistoMenu'),set_value='MACS Data Mode ('+(['Normal','Histogram','Event'])[(*pState).macs_histo]+')'
             if ((*pState).macs_normchoice eq 2) and ((*pState).macs_histo eq 0) then dcs_cryst_align_setoption,pState,'macsnorm',0
             (*pState).filesel->getproperty,file=open_file
             if n_elements(open_file) ne 0 then dcs_cryst_align_reload,pState,/nofilehistory ;reload file
             end
         'macsnorm': begin
             (*pState).macs_normchoice = (0>(choice)<2)
             ids = [widget_info(tlb,find_by_uname='norm_monitor'),widget_info(tlb,find_by_uname='norm_time'),widget_info(tlb,find_by_uname='norm_pulse')]
             dm_toggle_menubut,uncheck=ids[where([0,1,2] ne (*pState).macs_normchoice)],check=ids[(*pState).macs_normchoice]
             widget_control,widget_info(tlb,find_by_uname='normMenu'),set_value='Intensity Normalization Choice ('+(['Monitor','Time','Pulse Number'])[(*pState).macs_normchoice]+')'
             end
         'sortfile': begin
             ids = [widget_info(tlb,find_by_uname='sortnumb'),widget_info(tlb,find_by_uname='sortname')]
             dm_toggle_menubut,check=ids[choice],uncheck=ids[1-choice]
             widget_control,widget_info(tlb,find_by_uname=(['srtmenu','sortMenu'])[tlb eq (*pState).tlb]),set_value='Sort Files by ('+(['File Number','File Name'])[choice]+')'
             end
         'tempchoice': begin
             (*pState).tchoice = (0>(choice)<2)
             ids = [widget_info(tlb,find_by_uname='temp_samp'),widget_info(tlb,find_by_uname='temp_ctrl'),widget_info(tlb,find_by_uname='temp_setp')]
             dm_toggle_menubut,uncheck=ids[where([0,1,2] ne (*pState).tchoice)],check=ids[(*pState).tchoice]
             widget_control, widget_info(tlb,find_by_uname='tempMenu'),set_value='Temperature Choice'+((*pState).ftype eq 'WAND'?'':' ('+(['Sample','Control','Setpoint'])[(*pState).tchoice]+')') 
             end
         'termchoice': begin
             termid = [widget_info(tlb,find_by_uname='fitfun_3'),widget_info(tlb,find_by_uname='fitfun_4'),widget_info(tlb,find_by_uname='fitfun_5'),widget_info(tlb,find_by_uname='fitfun_6')]
             (*pState).fitfunnterm = (3>(choice)<6)
             dm_toggle_menubut,check=termid[(*pState).fitfunnterm-3],uncheck=termid[where([3,4,5,6] ne (*pState).fitfunnterm)]         
             end
         'fr_fitorder': if ((*pState).ftype eq 'TAS') or ((*pState).ftype eq 'MACS') then begin
             ids = [widget_info(tlb,find_by_uname='frfo12'),widget_info(tlb,find_by_uname='frfo21')]
             (*pState).fr_fitorder = (0>(choice)<1)
             dm_toggle_menubut,check=ids[(*pState).fr_fitorder],uncheck=ids[1-(*pState).fr_fitorder]
             widget_control, widget_info(tlb,find_by_uname='frfomenu'),set_value='Flipping Ratio Fitting Order ('+(['1->2','2->1'])[(*pState).fr_fitorder]+')'
             if keyword_set(calcfr) then begin
                (*pState).filesel->getproperty,path=open_path,file=open_file
                if n_elements(open_file) eq 2 then dcs_cryst_align_reload,pState,uname='macsflipratio'
             endif
             end
         else:
    endcase
    if dm_to_number(!version.release) ge 5.6 then $
       widget_control,widget_info((*pState).tlb,find_by_uname='fitbut'),tooltip='Fit the highest peak or zoom in to fit a local peak to a '+$
          (['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+' function:'+string(10b)+'y='+(*pState).fitfunname[(*pState).fitfunnterm-3,(*pState).fitfunchoice]+'.'
    if keyword_set(plot) then dcs_cryst_align_plot,pState
end

pro dcs_cryst_align_event,event,background=background,nofilehistory=nofilehistory
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=pState

    if (*pState).enotebook then $
       IF DAVE_SET_FOCUS(EVENT) THEN RETURN

    widget_type = widget_info(event.id,/type)
    case widget_type of
       8:    tmp_value = widget_info(event.id,/droplist_select)  ;droplist
       12:   tmp_value = event.index                             ;combobox
       else: tmp_value = 0
    endcase

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=(*pState).tlb,!error_state.msg,/error,/center)
       ;restore internal parameters
       if n_elements(tmp_macs_eff) ne 0 then (*pState).macs_eff = tmp_macs_eff
       if n_elements(tmp_fixfitwidth) ne 0 then (*pState).fixfitwidth = tmp_fixfitwidth
       if n_elements(tmp_showfit) ne 0 then (*pState).showfit = tmp_showfit
       if n_elements(tmpfile) ne 0 then file_delete,tmpfile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
       catch,/cancel
       if n_elements(mesg) ne 0 then begin
          if obj_valid(mesg[0]) then obj_destroy,mesg[0]
       endif
       if widget_info((*pState).tlb,/update) eq 0 then widget_control,(*pState).tlb,/map,/update
       return
    end
    
    tab     = string(9b)
    dgsym   = '!e !n\deg' ;string('b0'xb)
    uname   = strlowcase(widget_info(event.id,/uname))
    type    = strlowcase(tag_names(event,/structure))
    iftype  = (where(*(*pState).fname eq (*pState).ftype))[0]
    psiname = (*(*pState).psiname)[iftype]
    use_E   = (((*pState).ftype eq 'MACS') or ((*pState).ftype eq 'TAS'))
    case strmid(type,0,7) of
        'dm_plot': uname = type
        'dm_file': uname = type
        else:
    endcase
    
    ind = where(['loadbut','loadbgfile','macsalignment','macsalignmonchr','macsalignanalyzer','macsalignkidney','macsflipratio'] eq uname,count)
    if count ne 0 then begin
       date_str = strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)  ;date string
       colrs = (*pState).plotWin->getcolor(/list,linecolor=colr0)  ;colr0 is the first line color
       (*pState).filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir,sep=sep,dFiles=dFiles,sortfile=sortfile
       n_files = n_elements(open_file)
       if n_files le 0 then begin
          name = ([(*pState).ftype,'empty can','instrument alignment','monochromator alignment','two analyzer alignment','kidney center alignment','two flipping ratio measurement'])[ind[0]]+' file'+(['s',''])[ind[0] eq 5]
          ok = dialog_message('Please select '+name+' first.',/error,dialog_parent=event.handler,/center)
          return
       endif else begin
          if (ind[0] eq 4) and (n_files ne 2) then begin
             ok = dialog_message('Please select two analyzer alignment files'+(['.',' only.'])[n_files gt 2],/error,dialog_parent=event.handler,/center)
             return 
          endif else if (ind[0] eq 6) and (n_files ne 2) then begin
             ok = dialog_message('Please select two flipping ratio files'+(['.',' only.'])[n_files gt 2],/error,dialog_parent=event.handler,/center)
             return
          endif
       endelse
       selected = lonarr(n_files)
       if ~keyword_set(nofilehistory) then dcs_cryst_align_setfilehistory,pState,open_path,open_file,ftpobj=ftpobj
       if total(stregex(open_file,'^.+\.[a-z0-9]{2,4}\.[a-z0-9]{2,4}$',/fold_case,/boolean)) eq n_files then extension=strmid(open_file[0],strpos(open_file[0],'.',5,/reverse_search,/reverse_offset)) $
       else extension = strmid(open_file[0],strpos(open_file[0],'.',/reverse_search))
       tmp = (*pState).filesel->filebasename(open_file,extension=extension)
       if n_elements(open_file) gt 3 then filename = tmp[0]+'->'+tmp[n_elements(tmp)-1] else filename = strjoin(tmp,',')
    endif
    if uname eq 'contextevent0' or uname eq 'contextevent1' then uname = 'contextevent'
    ind = where(['contextevent','sortnumb','sortname','dm_filesel_sortfile'] eq uname,count)
    if count ne 0 then (*pState).filesel->getproperty,sortfile=sortfile
    case uname of
       'tlb':    begin
          widget_control,event.id,scr_xsize=(*pState).geom[0],scr_ysize=(*pState).geom[1]
          (*pState).plotWin->draw,/view ;avoid darkscreen
          end
       'contextevent': if ~(*pState).fixfitwidth then begin
          menuid = widget_info(event.id,find_by_uname='contextmenu')
          ffom = widget_info(menuid,find_by_uname='frfomenu') ;extra item for MACS and TAS
          if (((*pState).ftype eq 'TAS') or ((*pState).ftype eq 'MACS')) and (ffom eq 0) then begin
             ffom = widget_button(menuid,value='Flipping Ratio Fitting Order',/menu,uname='frfomenu')
             void = dm_widget_button(ffom,value='1->2',uname='frfo12')
             void = dm_widget_button(ffom,value='2->1',uname='frfo21')
          endif else if ((*pState).ftype ne 'MACS') and ((*pState).ftype ne 'TAS') and (ffom ne 0) then widget_control,ffom,/destroy    
          dcs_cryst_align_setoption,pState,'showfit',(*pState).showfit,tlb=menuid
          dcs_cryst_align_setoption,pState,'fixwidth',(*pState).fixfitwidth,tlb=menuid
          dcs_cryst_align_setoption,pState,'showcpos',(*pState).showcursor,tlb=menuid
          dcs_cryst_align_setoption,pState,'sortfile',sortfile,tlb=menuid
          dcs_cryst_align_setoption,pState,'funcchoice',(*pState).fitfunchoice,tlb=menuid
          dcs_cryst_align_setoption,pState,'termchoice',(*pState).fitfunnterm,tlb=menuid
          dcs_cryst_align_setoption,pState,'fit_extrap',(*pState).fit_extrap,tlb=menuid
          dcs_cryst_align_setoption,pState,'fr_fitorder',(*pState).fr_fitorder,tlb=menuid
          widget_displaycontextmenu,event.id,event.x,event.y,menuid   
          end
       'dm_plot_showcpos': dcs_cryst_align_setoption,pState,'showcpos',event.status
       'loadbut': begin
          (*pState).fit = 0
          if ptr_valid((*pState).fitfunstack) then ptr_free,(*pState).fitfunstack
          e_range = (*pState).e_ran
          case (*pState).ftype of 
              'DCS': begin   ;DCS
                 dm_load_dcsdata,open_path,open_file,ang=x,qty=y,dqty=dy,t_chan=(*pState).t_chan,e_range=e_range,monrate=monitor,$
                      parent=(*pState).tlb,duration=duration,/raw,eadjust=1,montype=1,/fissionchamber,bitflip=(*pState).bitflip,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir
                 if n_elements(y) ne 0 then begin
                    if keyword_set(background) then begin
                       nfile = n_elements(y[0,*])
                       (*pState).dcs_monitor[1] = monitor*nfile
                       if nfile gt 1 then begin ;combine files
                          y  = total(y,2)
                          dy = sqrt(total(dy^2,2))  ;not quite accurate
                       endif
                    endif else (*pState).dcs_monitor[0] = monitor
                 endif
                 end
              'MACS': begin  ;MACS
                 for i=0L,n_files-1L do begin
                     file = open_path+sep+open_file[i]
                     if i eq 0 then $
                        mesg = obj_new('dm_progress',title='Loading '+(['data','background'])[keyword_set(background)],message='Loading '+file,group_leader=(*pState).tlb) $
                     else $
                        mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)
                     if obj_valid(ftpobj) then begin
                        ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+dm_define_pointer(/getpathsep)+'ftptmp'+open_file[i])
                        file = ftpbufferdir+dm_define_pointer(/getpathsep)+'ftptmp'+open_file[i]
                     endif
                     dcs_cryst_align_loadmacs,file,hs=hs,ks=ks,ls=ls,en=en,a3=a3,a4=a4,a1=a1,a2=a2,a5=a5,a6=a6,pt=pt,mr=mr,mt=mt,ds=ds,mb=mb,bs=bs,as=as,sx=sx,sy=sy,sz=sz,sl=sl,ht=ht,vt=vt,$
                        su=su,dm=dm,fc=fc,b1=b1,b2=b2,vh=vh,vv=vv,kn=kn,hf=hf,t1=t1,t2=t2,t3=t3,ts=ts,te=te,wt=wt,it=it,dit=dit,pu=pu,cfx=cfx,duration=duration,xid=xid,yid=yid,did=did,$
                        macs_cfxdate=(*pState).macs_cfxdate,maspacing=maspacing,deteff=deteff,group_leader=(*pState).tlb,error=error,avgtemp=(*pState).tempavg_yn,latticeparm=latticeparm,$
                        latticeori=latticeori,montype=montype,histodata=(*pState).macs_histo
                     (*pState).macs_deteff[0:1,*] = deteff
                     (*pState).macs_maspacing = maspacing
                     if total(finite((*pState).macs_deteff[2:3,*],/nan)) ne 0 then (*pState).macs_deteff[2:3,*] = deteff
                     if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                     if keyword_set(error) then begin
                        if obj_valid(mesg) then obj_destroy,mesg
                        if n_elements(x) eq 0 then return else break
                     endif
                     if n_elements(it) eq 0 then continue
                     if n_elements(x) eq 0 then begin
                        x  = transpose([[hs],[ks],[ls],[en],[a3],[a4],[pt],[kn],[a1],[a2],[a5],[a6],[mr],[mt],[ds],[mb],[bs],[as],[sx],[sy],[sz],[sl],[su],[ht],[vt],[dm],[fc],[b1],[b2],[vh],$
                              [vv],[hf],[t1],[t2],[t3],[pu],[ts],[cfx],[te],[wt]])
                        y  = it
                        if n_elements(dit) eq 0 then dy = sqrt(it) else dy = dit
                     endif else begin
                        x  = [[x],[transpose([[hs],[ks],[ls],[en],[a3],[a4],[pt],[kn],[a1],[a2],[a5],[a6],[mr],[mt],[ds],[mb],[bs],[as],[sx],[sy],[sz],[sl],[su],[ht],[vt],[dm],[fc],[b1],[b2],$
                              [vh],[vv],[hf],[t1],[t2],[t3],[pu],[ts],[cfx],[te],[wt]])]]
                        y  = [[[y]],[[it]]] 
                        if n_elements(dit) eq 0 then dy = [[[dy]],[[sqrt(it)]]] else dy = [[[dy]],[[dit]]]
                     endelse
                     if (~ obj_valid(mesg)) then break
                 endfor
                 if n_elements(x) eq 0 then break
                 if ~keyword_set(background) then begin
                    ;check monitor counts
                    if (min(x[n_elements(x[*,0])-1,*]) lt 1000) and ((*pState).macs_normchoice eq 0)  and ~keyword_set(montype) then begin
                       ans = dialog_message(['Warning: Low monitor counts.','','Do you want to switch the intensity normalization choice from divided by monitor counts to divided by time?'],$
                             dialog_parent=(*pState).tlb,/question,/center)
                       if strupcase(ans) eq 'YES' then dcs_cryst_align_setoption,pState,'macsnorm',1
                    endif
                    if keyword_set(montype) and ((*pState).macs_normchoice eq 1) then begin
                       ans = dialog_message(['The intensity normalization choice is divided by time.','Do you want to switch to divided by monitor counts?'],dialog_parent=(*pState).tlb,$
                             /question,/center)
                       if strupcase(ans) eq 'YES' then dcs_cryst_align_setoption,pState,'macsnorm',0
                    endif
                    if n_elements(did) ne 0 then (*pState).y_typ = did
                    if yid eq 2 then begin ;monitor
                       *(*pState).detPtr = -1
                       (*pState).det_tit = 'det=monitor'
                    endif else begin       ;ptai
                       *(*pState).detPtr = -2
                       (*pState).det_tit = (['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det=ptai'
                    endelse
                    dm_set_droplist,widget_info((*pState).tlb,find_by_uname='timelist'),select=([(*pState).y_typ,3])[(*(*pState).detPtr)[0] eq -1]
                    widget_control,widget_info((*pState).tlb,find_by_uname='fitbut'),sensitive=(~(*pState).indvdet or (n_elements(*(*pState).detPtr) eq 1))
                 endif
                 end   
              'WAND': begin  ;WAND
                 for i=0L,n_files-1L do begin
                     file = open_path+sep+open_file[i]
                     if i eq 0 then $
                        mesg = obj_new('dm_progress',title='Loading '+(['data','background'])[keyword_set(background)],message='Loading '+file,group_leader=(*pState).tlb) $
                     else $
                        mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)
                     ;read the wand file
                     dm_load_wand,file,data=data,psi=psi,temperature=temperature,error=error,group_leader=(*pState).tlb
                     if keyword_set(error) then begin
                        if obj_valid(mesg) then obj_destroy,mesg
                        if n_elements(x) eq 0 then return else break
                     endif
                     mask = where(data[*,0] lt 0,mskcount)
                     if mskcount gt 0 then data[mask,*] = 0.0
                     if (*pState).tempavg_yn then temperature[*] = mean(temperature)
                     if i eq 0 then begin
                        x = transpose([[psi],[temperature]])
                        y = data 
                     endif else begin
                        x = [[x],[transpose([[psi],[temperature]])]]
                        y = [[y],[data]] 
                     endelse
                     if (~ obj_valid(mesg)) then break
                 endfor
                 if n_elements(y) ne 0 then begin
                    n_data = n_elements(y[0,*])
                    if keyword_set(background) then y = total(y,2)
                    dy = sqrt(y)
                    if keyword_set(background) then begin
                       y  = y/n_data
                       dy = dy/n_data
                    endif
                 endif
                 end
              'TAS': begin
                 filter = ['.b[ta-d]7','.phd','.bt4','.ng5','.b[ta-d]9','.b[ta-d]2']  ;recognized filter type
                 ind = bytarr(n_elements(filter))
                 for i=0L,n_files-1L do begin
                     for j=0,n_elements(filter)-1 do ind[j] = stregex(open_file[i],'\'+filter[j]+'$',/boolean,/fold_case)
                     ftype = (['',filter])[(where(ind))[0]+1]
                     if ftype eq '' then continue ;not supported yet
                     file = open_path+sep+open_file[i]
                     if i eq 0 then $
                        mesg = obj_new('dm_progress',title='Loading '+(['data','background'])[keyword_set(background)],message='Loading '+file,group_leader=(*pState).tlb) $
                     else $
                        mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)
                     if obj_valid(ftpobj) then begin
                        ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+dm_define_pointer(/getpathsep)+'ftptmp'+open_file[i])
                        file = ftpbufferdir+dm_define_pointer(/getpathsep)+'ftptmp'+open_file[i]
                     endif
                     dcs_cryst_align_loadtas,file,hs=hs,ks=ks,ls=ls,en=en,a3=a3,a4=a4,a1=a1,a2=a2,a5=a5,a6=a6,mr=mr,ar=ar,as=as,ps=ps,sy=sy,sx=sx,sz=sz,sw=sw,sh=sh,sl=sl,su=su,bw=bw,$
                         bh=bh,hf=hf,t1=t1,t2=t2,t3=t3,ts=ts,te=te,wt=wt,it=it,ms=ms,m_id=m_id,duration=duration,yid=yid,xid=xid,is_bt4=is_bt4,is_bt7=is_bt7,is_nice=is_nice,$
                         avgtemp=(*pState).tempavg_yn,latticeparm=latticeparm,latticeori=latticeori,tmpfile=tmpfile,bufferdir=ftpbufferdir,error=error,group_leader=(*pState).tlb
                     if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                     if keyword_set(error) then begin
                        if obj_valid(mesg) then obj_destroy,mesg
                        if n_elements(x) eq 0 then return else break
                     endif
                     if n_elements(it) eq 0 then continue
                     qty = transpose([[wt],[it]])
                     err = sqrt(qty)
                     if n_elements(ms) gt 0 then tmp_x = transpose([[hs],[ks],[ls],[en],[a3],[a4],[a1],[a2],[a5],[a6],[mr],[ar],[as],[sy],[sx],[sz],[sw],[sh],[sl],[su],[bw],[bh],[hf],[t1],[t2],[t3],[ts],[ms]]) $
                     else tmp_x = transpose([[hs],[ks],[ls],[en],[a3],[a4],[a1],[a2],[a5],[a6],[mr],[ar],[as],[sy],[sx],[sz],[sw],[sh],[sl],[su],[bw],[bh],[hf],[t1],[t2],[t3],[ts]])
                     if n_elements(x) eq 0 then begin
                        x  = temporary(tmp_x)
                        y  = qty
                        dy = err
                     endif else begin
                        x  = [[x],[temporary(tmp_x)]]
                        if duration lt 0 then monfac = abs(duration)/te $
                        else monfac = double(duration)/(1>(wt))
                        for j=0,n_elements(qty[*,0])-1 do begin
                            qty[j,*] = qty[j,*]*monfac
                            err[j,*] = err[j,*]*monfac
                        endfor
                        y  = [[y],[qty]]
                        dy = [[dy],[err]]
                     endelse
                 endfor
                 if n_elements(x) eq 0 then break
                 if ~keyword_set(background) then begin
                    (*pState).is_bt7 = keyword_set(is_bt7)
                    if (*pState).is_bt7 then begin
                       det_info = ['Monitor','Detector','DDC'+dm_to_string([0,1,2]),'TDC0'+dm_to_string(indgen(9)),'PSDC'+string(indgen(48),format='(i02)'),'SDC'+dm_to_string([0,1,2])]
                       x_info   = ['H','K','L','E','A3','A4','A1','A2','A5','A6','DFMRot','AnalyzerRotation','AnalyzerBlade'+string(1+indgen(13),format='(i02)'),$
                                   'Smpl'+['LTrn','UTrn','Elev','Wdth','Hght','LTilt','UTilt'],'Bkslt'+['Wdth','Hght'],'MagField','Temp']
                       *(*pState).xtit = ['H','K','L','E (meV)',['A3','A4','A1','A2','A5','A6','DFMRot','AnalyzerRotation','AnalyzerBlade'+string(1+indgen(13),format='(i02)')]+' (\deg)',$
                                   [['Lower','Upper']+'-translation','Elevation','Beam Width','Beam Height']+' (mm)',['Lower','Upper']+'-tilt (\deg)','BackSlit-'+['width','height']+' (mm)','\mu!i0!nH (T)','T (K)']             
                    endif else begin
                       det_info = ['Monitor','Detector']
                       x_info   = ['H','K','L','E','A3','A4','A1','A2','A5','A6','Smpl'+['LTrn','UTrn','Wdth','Hght','LTilt','UTilt'],'MagField','Temp']
                       *(*pState).xtit = ['H','K','L','E (meV)',['A3','A4','A1','A2','A5','A6']+' (\deg)',[['Lower','Upper']+'-translation','Beam Width','Beam Height']+' (mm)',['Lower','Upper']+'-tilt (\deg)',$
                                   '\mu!i0!nH (T)','T (K)']
                       x_ind = [indgen(10),25,26,28,29,30,31,34,35,36,37,38]
                       if (n_elements(ms) gt 0) and (n_elements(m_id) gt 0) then begin
                          x_ind = [x_ind,39]
                          x_info = [x_info,'Motor_'+dm_to_string(m_id)]
                          *(*pState).xtit = [*(*pState).xtit, 'Motor '+dm_to_string(m_id)]
                       endif
                       x = x[x_ind,*]
                    endelse
                    if n_elements(xid) eq 0 then xid = -1
                    if n_elements(yid) eq 0 then yid = 1
                    if size(xid,/type) eq size('a',/type) then begin
                       if stregex(xid,'motor_[0-9]+',/boolean,/fold_case) then begin
                          motor_id = 0>(dm_to_number(strmid(xid,strpos(xid,'_')+1),/int))<20
                          xid = (['','a1','a2','a3','a4','a5','a6',is_bt4?'Smpl'+['UTilt','LTilt','Ltrn','UTrn','Hght','Wdth']:[xid,xid,xid,xid,'SmplLTilt','SmplUTilt'],$
                                 strarr(10)+xid])[motor_id]
                       endif
                       if strlowcase(xid) eq 'tact'   then xid = 'Temp'
                       if strlowcase(xid) eq 'hfield' then xid = 'MagField'
                       if strmatch(xid,'q[x-z]',/fold_case) then xid = (['H','K','L'])[(byte(strlowcase(xid)))[1]-byte('x')]
                       tmp = where(strlowcase(x_info) eq strlowcase(xid[0]),count)
                       if count ne 0 then xid = tmp[0] else xid = -1
                    endif
                    if size(yid,/type) eq size('a',/type) then begin
                       tmp = where(strlowcase(det_info) eq strlowcase(yid),count)
                       if count eq 0 then yid = 1 else yid = tmp[0]
                    endif
                    *(*pState).detPtr = yid
                    (*pState).det_tit = (['det=monitor','','det='+['ddc'+dm_to_string([0,1,2]),'tdc0'+dm_to_string(indgen(9)),'psdc'+string(indgen(48),format='(i02)'),'sdc'+dm_to_string([0,1,2])]])[yid]
                    if (~(*pState).is_bt7) and (~keyword_set(sortfile)) and (~keyword_set(is_nice)) and (~(*pState).answered) then begin
                       ind_file = where(~((*pState).filesel->isDir(dFiles,/fast)),nfile)
                       if nfile gt 2 then begin
                          ans = dialog_message(['Do you want to sort the files by name?','','The sorting preference can be changed in the options menu.'],dialog_parent=(*pState).tlb,/question,/center)
                          if strupcase(ans) eq 'YES' then (*pState).filesel->changesort
                          (*pState).answered = 1b
                       endif
                    endif
                 endif
                 end   
              else:
          endcase
          if obj_valid(mesg) then obj_destroy,mesg
          if n_elements(x) eq 0 then begin
             datstr = (['','background '])[keyword_set(background)]+(['data','histogram data','event data'])[((*pState).ftype eq 'MACS')*(*pState).macs_histo]
             ok = dialog_message(([open_file[0]+' contains','The files contain'])[n_files gt 1]+' no valid '+datstr+'.',dialog_parent=event.handler,/center,title='Empty '+datstr)
             return
          endif
          (*pState).filename[keyword_set(background)] = filename
          (*pState).filename[2] = open_path
          if keyword_set(background) then begin
             *(*pState).bgPtr  = temporary(y)
             *(*pState).dbgPtr = temporary(dy)
             case (*pState).ftype of 
                  'MACS' : *(*pState).bgmonPtr = x[n_elements(x[*,0])-[1,2,5],*]
                  'TAS'  : *(*pState).bgmonPtr = duration
                  else:
             endcase
          endif else begin
             if n_elements(xid) ne 0 then begin
                (*pState).xaxis = (-1)>xid
                dm_set_droplist,widget_info((*pState).tlb,find_by_uname='xaxislist'),select=(*pState).xaxis,set_value=x_info
             endif
             if n_elements(yid) ne 0 then begin
                dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=yid,set_value=det_info
                (*pState).title = (*pState).det_tit
             endif
             if n_elements(e_range) ne 0 then (*pState).e_ran = e_range
             *(*pState).xPtr  = temporary(x)
             *(*pState).yPtr  = temporary(y)
             *(*pState).dyPtr = temporary(dy)
             if n_elements(duration) ne 0 then (*pState).duration = duration
             if n_elements(latticeparm) eq 6 then (*(*pState).infoPtr)[2:7] = latticeparm   
             if n_elements(latticeori) eq 6 then (*(*pState).infoPtr)[8:13] = latticeori
          endelse
          if n_elements(*(*pState).yPtr) ne 0 then dcs_cryst_align_plot,pState
         end
       'loadbgfile': begin
          if ((*pState).ftype eq 'DCS') and ((*pState).t_chan eq 1) and (n_elements(*(*pState).xPtr) eq 0) then begin
             ok = dialog_message('Please load sample data files first to get the same energy range for empty can files.',dialog_parent=(*pState).tlb,/center)
             return
          endif          
          dcs_cryst_align_reload,pState,/nofilehistory,/background
          widget_control,widget_info((*pState).mbar,find_by_uname='clearbgfile'),sensitive=(n_elements(*(*pState).bgPtr) ne 0)
          dm_set_button,event.id,(n_elements(*(*pState).bgPtr) ne 0)
         end  
       'clearbgfile': begin
          ptr_free,(*pState).bgPtr,(*pState).dbgPtr,(*pState).bgmonPtr
          (*pState).bgPtr    = ptr_new(/allocate_heap)
          (*pState).dbgPtr   = ptr_new(/allocate_heap)
          (*pState).bgmonPtr = ptr_new(/allocate_heap)
          widget_control,event.id,sensitive=0
          dm_set_button,widget_info((*pState).tlb,find_by_uname='loadbgfile'),0
          if n_elements(*(*pState).yPtr) ne 0 then dcs_cryst_align_plot,pState 
         end
       'optnbitflip': begin
          (*pState).bitflip = 1-(*pState).bitflip
          dm_set_button,event.id,(*pState).bitflip,onstring='Bitflip Error Correction Enabled',offstring='Bitflip Error Correction Disabled'
         end  
       'optnindvdet': begin
          (*pState).indvdet = 1-(*pState).indvdet
          dm_set_button,event.id,(*pState).indvdet,onstring='Show Individual Detector',offstring='Sum Detectors'
          widget_control,widget_info((*pState).tlb,find_by_uname='fitbut'),sensitive=(~(*pState).indvdet or (n_elements(*(*pState).detPtr) eq 1))
          dcs_cryst_align_plot,pState
         end
       'optnshowfit': dcs_cryst_align_setoption,pState,'showfit',1-(*pState).showfit
       'optnfixwidth':begin
          dcs_cryst_align_setoption,pState,'fixwidth',1-(*pState).fixfitwidth
          cnt  = widget_info((*pState).tlb,find_by_uname='center')
          csms = [widget_info(cnt,find_by_uname='contextevent0'),widget_info(cnt,find_by_uname='contextevent1')]
          for i=0,1 do begin
              kids = widget_info(csms[i],/all_children)
              for j=0,n_elements(kids)-1 do begin
                  if kids[j] ne 0 then widget_control,kids[j],/destroy
              endfor
          endfor
          if (*pState).fixfitwidth then begin
             geom0 = widget_info(widget_info(cnt,find_by_uname='loadbut'),/geometry)
             geom1 = widget_info(cnt,/geometry)
             void = widget_label(csms[0],value='Fixed Width (C2)',/align_center,scr_xsize=geom1.scr_xsize-2*geom1.xpad,uname='fixwidthlabel')
             void = widget_text(csms[1],uname='fixwidth',/editable,scr_xsize=geom1.scr_xsize-2*geom1.xpad,frame=0)
             if !version.os_family eq 'unix' then begin
                wids = ['detlabel','detlist','timelabel','timelist','xaxislabel','xaxislist','fixwidthlabel',['load','fit','clpb','save','data','print']+'but']
                for i=0,n_elements(wids)-1 do widget_control,widget_info(cnt,find_by_uname=wids[i]),scr_ysize=geom0.scr_ysize
             endif
             if ((*pState).fitfunnterm eq 6) and ((*pState).fitfunchoice eq 0) then begin ;mpfit doesn't support nterm=6
                dcs_cryst_align_setoption,pState,'termchoice',5
                ok = dialog_message("Fixed width fitting currently doesn't work with nterm=6 for Gaussian. The Gaussian has been switched to nterm=5.",dialog_parent=event.handler,/center)
             endif
          endif else begin
             dcs_cryst_align_csm,pState
          endelse
          widget_control,widget_info((*pState).tlb,find_by_uname='fitfun_6'),sensitive=(((*pState).fitfunchoice eq 0) and (~(*pState).fixfitwidth))
          dcs_cryst_align_plot,pState
         end   
       'optnshowfname': begin
          (*pState).showfname = 1-(*pState).showfname
          dm_set_button,event.id,(*pState).showfname,onstring='Show File Name',offstring='Hide File Name'
          if ~(*pState).showfname then filename = '' else begin
             filename = (*pState).filename[0]
             if n_elements(*(*pState).bgPtr) ne 0 then filename = '('+filename+')-('+(*pState).filename[1]+')'
          endelse
          (*pState).plotWin->setproperty,cornertxt=filename
         end  
       'optnshowcursor': dcs_cryst_align_setoption,pState,'showcpos',1-(*pState).showcursor,/setplotcursor
       'optnapplylamb2': begin
          (*pState).macs_lamb2 = 1-(*pState).macs_lamb2
          dm_set_button,event.id,(*pState).macs_lamb2,onstring='Apply Monitor Lambda/2 Correction',offstring='No Monitor Lambda/2 Correction'
          dcs_cryst_align_plot,pState
         end
       'ncnrftp': begin    
          (*pState).filesel->set_ftpbuffer,(['',dm_define_pointer(/gettempdir)])[1-(*pState).ncnrftp]
          (*pState).filesel->getproperty,ncnrftp=ncnrftp
          if ncnrftp ne 1-(*pState).ncnrftp then ok = dialog_message('Cannot establish connection to NCNR Data Repository.',dialog_parent=(*pState).tlb,/center,/error)
          (*pState).ncnrftp = ncnrftp
          dm_set_button,event.id,(*pState).ncnrftp
          if (*pState).ncnrftp then (*pState).filesel->set_path,/ftp
         end  
       'norm_monitor':if (*pState).macs_normchoice ne 0 then dcs_cryst_align_setoption,pState,'macsnorm',0,/plot
       'norm_time':   if (*pState).macs_normchoice ne 1 then dcs_cryst_align_setoption,pState,'macsnorm',1,/plot
       'norm_pulse':  if (*pState).macs_normchoice ne 2 then dcs_cryst_align_setoption,pState,'macsnorm',2,/plot
       'temp_samp':   if (*pState).tchoice ne 0 then dcs_cryst_align_setoption,pState,'tempchoice',0,plot=((*pState).xaxis eq (n_elements(*(*pState).xtit)-1))
       'temp_ctrl':   if (*pState).tchoice ne 1 then dcs_cryst_align_setoption,pState,'tempchoice',1,plot=((*pState).xaxis eq (n_elements(*(*pState).xtit)-1))
       'temp_setp':   if (*pState).tchoice ne 2 then dcs_cryst_align_setoption,pState,'tempchoice',2,plot=((*pState).xaxis eq (n_elements(*(*pState).xtit)-1))
       'temp_rebin':begin
          (*pState).tempreb_yn = ~(*pState).tempreb_yn
          dm_set_button,event.id,(*pState).tempreb_yn
          widget_control,widget_info((*pState).mbar,find_by_uname='temp_comb'),sensitive=(((*pState).ftype ne 'WAND') and (*pState).tempreb_yn)
          if stregex((*(*pState).xtit)[(*pState).xaxis],'^T *\(.*K\)',/boolean,/fold_case) then dcs_cryst_align_plot,pState 
         end
       'temp_comb': begin
          (*pState).tempcom_yn = ~(*pState).tempcom_yn
          dm_set_button,event.id,(*pState).tempcom_yn
          if stregex((*(*pState).xtit)[(*pState).xaxis],'^T *\(.*K\)',/boolean,/fold_case) then dcs_cryst_align_plot,pState
         end  
       'temp_avg': begin
          (*pState).tempavg_yn = ~(*pState).tempavg_yn
          dm_set_button,event.id,(*pState).tempavg_yn
          (*pState).filesel->getproperty,file=open_file
          if n_elements(open_file) ne 0 then dcs_cryst_align_reload,pState,/nofilehistory ;reload file
         end        
       'sortnumb': if sortfile ne 0 then (*pState).filesel->changesort
       'sortname': if sortfile ne 1 then (*pState).filesel->changesort
       'fit_pan':  begin
          dcs_cryst_align_plot,pState,/dataonly,xs=xs,ys=ys,dys=dys,xtit=xtit,ytit=ytit
          if n_elements(xs) ne 0 then begin
             ;replace \AA with string('c5'XB)
             pos = strpos(xtit,'\AA')
             if pos ne -1 then xtit = strmid(xtit,0,pos)+string('c5'XB)+strmid(xtit,pos+3)
             pos = strpos(ytit,'\AA')
             if pos ne -1 then ytit = strmid(ytit,0,pos)+string('c5'XB)+strmid(ytit,pos+3)
             if n_elements(dys) ne n_elements(ys) then begin
                ok  = dialog_message('No yerr. 0.1*y is used as yerr.',dialog_parent=(*pState).tlb,/center)
                dys = 0.1*ys
             endif
             ;make sure that there's no 0's in dys
             index = where(dys eq 0.0,count,complement=index1,ncomplement=count1)
             if count ne 0 then begin
                if count1 eq 0 then err = 0.01 else err = min(abs(dys[index1]))
                dys[index] = err
             endif
             ;to PAN
             filename = (*pState).filename[0]
             if n_elements(*(*pState).bgPtr) ne 0 then filename = '('+filename+')-('+(*pState).filename[1]+')'
             info = ['Data from '+(*pState).ftype+([' Single Crystal',''])[(*pState).ftype eq 'MACS']+' Alignment Utility.','File name: '+filename,'File directory:'+(*pState).filename[2]]
             opan = obj_new('opan',mydata=ys,error=dys,xval=xs,yval=0,xtit=xtit,ztit=ytit,group_leader=(*pState).tlb,workDir=(*pState).workDir,header=info)
          endif else ok = dialog_message('Please load data file first.',dialog_parent=(*pState).tlb,/error,/center)
         end
       'fit_gaussian':   if (*pState).fitfunchoice ne 0 then dcs_cryst_align_setoption,pState,'funcchoice',0,/plot,/clearfitstack
       'fit_lorentzian': if (*pState).fitfunchoice ne 1 then dcs_cryst_align_setoption,pState,'funcchoice',1,/plot,/clearfitstack
       'fit_moffat':     if (*pState).fitfunchoice ne 2 then dcs_cryst_align_setoption,pState,'funcchoice',2,/plot,/clearfitstack
       'fitfun_3':       if (*pState).fitfunnterm ne 3  then dcs_cryst_align_setoption,pState,'termchoice',3,/plot,/clearfitstack    
       'fitfun_4':       if (*pState).fitfunnterm ne 4  then dcs_cryst_align_setoption,pState,'termchoice',4,/plot,/clearfitstack
       'fitfun_5':       if (*pState).fitfunnterm ne 5  then dcs_cryst_align_setoption,pState,'termchoice',5,/plot,/clearfitstack
       'fitfun_6':       if (*pState).fitfunnterm ne 6  then dcs_cryst_align_setoption,pState,'termchoice',6,/plot,/clearfitstack
       'fit_extrap':                                         dcs_cryst_align_setoption,pState,'fit_extrap',1-(*pState).fit_extrap,/clearfitstack
       'macs_eff_none':  if (*pState).macs_eff ne 0     then dcs_cryst_align_setoption,pState,'macseff',0,/plot
       'macs_eff_file':  if (*pState).macs_eff ne 1     then dcs_cryst_align_setoption,pState,'macseff',1,/plot
       'macs_eff_spec':                                      dcs_cryst_align_setoption,pState,'macseff',2,/plot 
       'macs_histo0':    if (*pState).macs_histo ne 0   then dcs_cryst_align_setoption,pState,'macshisto',0
       'macs_histo1':    if (*pState).macs_histo ne 1   then dcs_cryst_align_setoption,pState,'macshisto',1
       'macs_histo2':    if (*pState).macs_histo ne 2   then dcs_cryst_align_setoption,pState,'macshisto',2
       'frfo12':         if (*pState).fr_fitorder ne 0  then dcs_cryst_align_setoption,pState,'fr_fitorder',0,/calcfr
       'frfo21':         if (*pState).fr_fitorder ne 1  then dcs_cryst_align_setoption,pState,'fr_fitorder',1,/calcfr
       'fitbut':  if n_elements(*(*pState).yPtr) ne 0 then begin
          dcs_cryst_align_plot,pState,/dataonly,xs=xs,ys=ys
          if n_elements(xs) gt (*pState).fitfunnterm+((*pState).fitfunchoice eq 2) then begin
             (*pState).fit = 1
             dcs_cryst_align_plot,pState
          endif else ok = dialog_message('There are not enough data points to fit.',dialog_parent=(*pState).tlb,/center)
         endif else ok = dialog_message('Please load data file first.',dialog_parent=(*pState).tlb,/error,/center)
       'savebut': if n_elements(*(*pState).xPtr) ne 0 then begin
          if dm_to_number(!version.release) ge 8.0 then $          
             ftypename = ['Postscript','PDF','JPEG','GIF','TIFF','BMP','PNG','PPM'] $
          else $
             ftypename = ['Postscript','JPEG','GIF','TIFF','BMP','PNG','PPM']
          ftype = dm_dialog_input('file type:',default=0,droplist_content=ptr_new(ftypename),is_droplist=[1],$
              /return_number,title='Please choose',dialog_parent=(*pState).tlb,xsize=80,cancel=cancel)
          if keyword_set(cancel) then return
          ftypename[0] = 'PS'    
          (*pState).plotWin->saveas,ftypename[ftype]
         end
       'databut': if n_elements(*(*pState).xPtr) ne 0 then begin
          path = (*pState).workDir
          file = dm_choose_file('txt',/write,dialog_parent=(*pState).tlb,title='Save plot data as an ASCII file.',path=path)
          if strlen(file) eq 0 then break
          if file_test(path,/directory) then (*pState).workDir=path
          openw,unit,file,/get_lun,error=openerr
          if openerr ne 0 then begin
             ok = dialog_message("Can't write in "+file[0],/error,dialog_parent=(*pState).tlb,/center)
          endif else begin
             dcs_cryst_align_plot,pState,/dataonly,xs=x,ys=y,dys=dy,xtit=xtit,ytit=ytit  ;get data
             widget_control,/hourglass  ;busy signal
             header = dm_to_string(xtit,data=x[0],over=over,/addsemicolon)
             header = header+dm_to_string(ytit,data=y[0],over=over)
             header = header+dm_to_string('dy',data=dy[0],over=over)
             printf,unit,header
             for i=0L,n_elements(x)-1L do $
                 printf,unit,x[i],y[i],dy[i],format='(3g)'
             free_lun,unit
          endelse
         end
       'printbut': if n_elements(*(*pState).xPtr) ne 0 then (*pState).plotWin->saveas,'printer'
       'clpbbut':  if n_elements(*(*pState).xPtr) ne 0 then (*pState).plotWin->saveas,'clipboard'
       'exitbut': begin
          widget_control,event.handler,/destroy
          return
         end
       'detlist': begin
          old = *(*pState).detPtr
          (*pState).tth = [!values.f_nan,!values.f_nan]
          widget_control,event.id,get_value=det_list
          case tmp_value of
              0: begin  ;all
                 if (*pState).ftype eq 'TAS' then begin
                    *(*pState).detPtr = 0
                    (*pState).det_tit = 'det=monitor'
                    break
                 endif
                 *(*pState).detPtr = indgen((*pState).ndet)
                 (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det=all'     
                 end
              1: begin  ;specify
                 if (*pState).ftype eq 'TAS' then begin
                    *(*pState).detPtr = 1
                    (*pState).det_tit = ''
                    break
                 endif
                 n_index = n_elements(*(*pState).detPtr)
                 maxdet  = max(*(*pState).detPtr,min=mindet)
                 if (n_index eq (maxdet-mindet+1)) and (n_index gt 3) then $
                    default = dm_to_string(mindet+1)+'-'+dm_to_string(maxdet+1) $
                 else begin
                    default = dm_to_string(*(*pState).detPtr+1,separator=',')
                    if ((*pState).ftype eq 'DCS') then begin
                       if ((*(*pState).detPtr)[0] eq 913) then default = 'FC2'
                       if ((*(*pState).detPtr)[0] eq 914) then default = 'FC3'
                       if n_elements((*(*pState).detPtr)) eq 294 then begin
                          if total(abs((*(*pState).detPtr)-dm_mask_detbanks(-1))) eq 0 then default = 'Lower Bank' $
                          else if total(abs((*(*pState).detPtr)-dm_mask_detbanks(1))) eq 0 then default = 'Upper Bank'
                       endif else if n_elements((*(*pState).detPtr)) eq 325 then begin
                          if total(abs((*(*pState).detPtr)-dm_mask_detbanks(0))) eq 0 then default = 'Central Bank'
                       endif
                    endif else if ((*pState).ftype eq 'MACS') then begin
                       if ((*(*pState).detPtr)[0] eq -1) then default = 'monitor'
                       if ((*(*pState).detPtr)[0] eq -2) then default = 'ptai'
                    endif
                 endelse
                 tmp = dm_dialog_input(dialog_parent=event.handler,'Det num:',default=default,title='Specifying detectors',$
                       info='Detector number starts from 1 to '+dm_to_string((*pState).ndet)+'. eg:'+(['371,374 or 370-374 c','6,7,8 or 6-8','120,121,122 or 120-122'])[iftype<2],xsize=200)
                 (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det='+tmp
                 if strlen(tmp) ne 0 then begin
                    if ((*pState).ftype eq 'DCS') then begin
                       if strlowcase(tmp) eq 'central bank' then  tmp_det = 0
                       if strlowcase(tmp) eq 'lower bank'   then  tmp_det = -1
                       if strlowcase(tmp) eq 'upper bank'   then  tmp_det = 1
                       if strlowcase(tmp) eq 'fc2'          then  tmp_det = 913
                       if strlowcase(tmp) eq 'fc3'          then  tmp_det = 914
                       if n_elements(tmp_det) ne 0 then begin
                           (*pState).det_tit = (tmp_det gt 2)?((['FC2','FC3'])[tmp_det-913]):('det='+(['lower','central','upper'])[tmp_det+1])
                           dm_set_droplist,event.id,select=(tmp_det gt 2)?(5+tmp_det-913):(([4,2,3])[tmp_det+1])
                           *(*pState).detPtr = (tmp_det gt 2)?tmp_det:(dm_mask_detbanks(tmp_det))
                           break
                        endif
                    endif else if ((*pState).ftype eq 'MACS') then begin
                       if strlowcase(tmp) eq 'monitor' then  tmp_det = -1
                       if strlowcase(tmp) eq 'ptai'    then  tmp_det = -2
                       if n_elements(tmp_det) ne 0 then begin
                           (*pState).det_tit = ([(['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det=ptai','det=monitor'])[2+tmp_det]
                           dm_set_droplist,event.id,select=n_elements(det_list)-23-tmp_det
                           *(*pState).detPtr = tmp_det
                           break
                        endif
                    endif
                    ;search for :- range detectors
                    len = [1]
                    det_tmp = -1L
                    while (len[0] gt 0) do begin
                        tmp1 = stregex(tmp,'([0-9]+)[ '+tab+']*[:-][ '+tab+']*([0-9]+)[ '+tab+']*([cCuUlL]*)',/subexpr,length=len)
                        if len[0] gt 0 then begin
                           tmp2 = dm_to_number(strmid(tmp,tmp1[1],len[1]),/long)
                           tmp3 = dm_to_number(strmid(tmp,tmp1[2],len[2]),/long)
                           if len[3] gt 0 then status = strlowcase(strmid(tmp,tmp1[3],1)) else status=''
                           tmp4 = min([tmp2,tmp3])+lindgen(abs(tmp3-tmp2)+1)
                           if status ne '' then $
                              tmp4 = dm_common(tmp4,dm_mask_detbanks((where(['l','c','u'] eq status))[0]-1)+1,/no_copy)
                           det_tmp = [det_tmp,tmp4]
                           tmp = strmid(tmp,0,tmp1[0])+strmid(tmp,tmp1[0]+len[0],strlen(tmp)-tmp1[0]-len[0])
                        endif
                    endwhile 
                    ;individual detectors
                    tmp1 = strsplit(tmp,'[a-zA-Z '+tab+',:-]',/regex,/extract)
                    tmp2 = dm_to_number(tmp1)
                    ind2 = where(finite(tmp2),count2)
                    if count2 gt 0 then det_tmp = [det_tmp,dm_to_number(tmp1[ind2],/long)]
                    index = where(det_tmp gt 0,count)
                    if count gt 0 then dets = det_tmp[index]-1
                 endif  
                 ndet = n_elements(dets)
                 if ndet gt 0 then dets = dets[uniq(dets,sort(dets))]
                 if n_elements(dets) ne ndet then (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det='+dm_to_string(dets+1,separator=',')
                 ndet = n_elements(dets)
                 if (ndet eq 0) or (ndet eq (*pState).ndet) then begin
                    *(*pState).detPtr = indgen((*pState).ndet)
                    (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det=all'
                    dm_set_droplist,event.id,select=0
                 endif else begin
                    *(*pState).detPtr = dets
                    maxdet = max(dets,min=mindet)
                    if (ndet eq (maxdet-mindet+1)) and (ndet gt 3) then $
                       (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det='+dm_to_string(mindet+1)+'-'+dm_to_string(maxdet+1)
                    (*pState).tth[0] = mean((*(*pState).tthPtr)[dets])
                    tth = dm_to_string((*pState).tth[0],resolution=3)+dgsym
                    if ndet gt 1 then tth = '(<2th>='+tth+')' else tth='(2th='+tth+')'
                    (*pState).det_tit = (*pState).det_tit+tth
                 endelse
                 end
           else: begin
                 case (*pState).ftype of
                      'DCS':  begin
                              if tmp_value le 4 then *(*pState).detPtr = dm_mask_detbanks(([0,1,-1])[tmp_value-2]) $
                              else if tmp_value le 6 then  *(*pState).detPtr = ([913,914])[tmp_value-5] $
                              else *(*pState).detPtr = tmp_value-7
                              if tmp_value le 6 then (*pState).det_tit = (['det='+['central','upper','lower'],'FC2','FC3'])[tmp_value-2] $
                              else begin
                                 (*pState).tth[0]  = (*(*pState).tthPtr)[tmp_value-7]
                                 tth = dm_to_string((*pState).tth[0])+dgsym
                                 (*pState).det_tit = 'det='+strtrim(string(tmp_value-6),2)+'(2th='+tth+')'
                              endelse
                              end
                      'MACS': begin
                              if tmp_value lt 4 then begin
                                 *(*pState).detPtr = ([-1,-2])[tmp_value-2]  ;monitor and ptai
                                 (*pState).det_tit = (['det=monitor',(['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det=ptai'])[tmp_value-2]
                              endif else begin
                                 *(*pState).detPtr = tmp_value-4
                                 (*pState).det_tit = (['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det='+strtrim(string(tmp_value-3),2)
                              endelse 
                              end
                      'TAS':  begin
                              *(*pState).detPtr = tmp_value
                              (*pState).det_tit = (['det=monitor','','det='+['ddc'+dm_to_string([0,1,2]),'tdc0'+dm_to_string(indgen(9)),'psdc'+string(indgen(48),format='(i02)'),'sdc'+dm_to_string([0,1,2])]])[tmp_value]
                              end       
                      else:   begin
                              *(*pState).detPtr = tmp_value-2
                              (*pState).tth[0]  = (*(*pState).tthPtr)[tmp_value-2]
                              tth = dm_to_string((*pState).tth[0])+dgsym
                              (*pState).det_tit = (['',(['spec ','diff ','spec+diff '])[(*pState).y_typ]])[(*pState).ftype eq 'MACS']+'det='+strtrim(string(tmp_value-1),2)+'(2th='+tth+')'
                              end
                 endcase
                 end
          endcase
          if (*pState).ftype eq 'DCS' then $
             (*pState).title = (*pState).det_tit+', '+(*pState).tch_tit $
          else begin
             (*pState).title = (*pState).det_tit
             if (*pState).ftype eq 'MACS' then begin
                dm_set_droplist,widget_info((*pState).tlb,find_by_uname='timelist'),select=([(*pState).y_typ,3])[(*(*pState).detPtr)[0] eq -1] 
             endif
          endelse
          n_old = n_elements(old)
          n_new = n_elements(*(*pState).detPtr)
          comm  = dm_common(old,*(*pState).detPtr)
          same  = (n_old eq n_new) and (n_old eq n_elements(comm)) and (comm[0] ne -1)
          old   = -1 & comm = -1
          widget_control,widget_info((*pState).tlb,find_by_uname='fitbut'),sensitive=(~(*pState).indvdet or (n_elements(*(*pState).detPtr) eq 1))
          if (n_elements(*(*pState).yPtr) ne 0) and (~ same) then dcs_cryst_align_plot,pState   ;reload data
         end
       'timelist': if (((*pState).ftype eq 'DCS') and (((*pState).t_chan ne tmp_value) or (tmp_value eq 2))) or ((*pState).ftype eq 'MACS') then begin
          if (*pState).ftype eq 'DCS' then begin
             (*pState).t_chan = tmp_value
             if (*pState).t_chan eq 0 then (*pState).tch_tit='tchan=all' $
             else if (*pState).t_chan eq 1 then (*pState).tch_tit='tchan=elastic' $
             else begin
                default = dm_to_string((*pState).e_ran)
                new     = dm_dialog_input(['from:','to:'],title='Energy range',xsize=100,default=default,/float,$
                          dialog_parent=(*pState).tlb,info='In units of meV')
                (*pState).t_chan = 1
                (*pState).e_ran[*] = !values.f_nan
                if total(finite(new)) eq 2 then $
                   if new[1] gt new[0] then begin
                      (*pState).e_ran   = new
                      (*pState).t_chan  = 2
                      (*pState).tch_tit = 'E=['+dm_to_string(new[0])+','+dm_to_string(new[1])+']meV'
                   endif
                if (*pState).t_chan eq 1 then begin
                   ok = dialog_message('Invalid energy range. Elastic peak option is used.',dialog_parent=(*pState).tlb,/center)
                   dm_set_droplist,event.id,select=1
                   (*pState).tch_tit = 'tchan=elastic'
                endif
             endelse
             (*pState).title = (*pState).det_tit+', '+(*pState).tch_tit
             (*pState).filesel->getproperty,file=open_file
             if n_elements(open_file) ne 0 then dcs_cryst_align_reload,pState,/nofilehistory ;reload file
             if n_elements(*(*pState).bgPtr) ne 0 then begin
                ptr_free,(*pState).bgPtr,(*pState).dbgPtr
                (*pState).bgPtr = ptr_new(/allocate_heap)
                (*pState).dbgPtr = ptr_new(/allocate_heap)
                widget_control,widget_info((*pState).mbar,find_by_uname='clearbgfile'),sensitive=0
                ok = dialog_message('Please reload the empty can file.',dialog_parent=(*pState).tlb,/center)
             endif
          endif else if (*pState).ftype eq 'MACS' then begin
             if tmp_value eq 3 then begin
                *(*pState).detPtr = -1
                (*pState).det_tit = 'det=monitor'
                (*pState).title   = (*pState).det_tit
                dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=2
             endif else begin
                (*pState).y_typ = tmp_value
                if (*(*pState).detPtr)[0] eq -1 then begin
                   *(*pState).detPtr = -2  ;ptai
                   (*pState).det_tit = (['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det=ptai'
                   (*pState).title   = (*pState).det_tit
                   dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=3
                endif else begin     
                   str1 = (['(spec *\+ *)*diff','spec( *\+ *diff)*','(spec )*(diff *)*'])[(*pState).y_typ]
                   str2 = (['spec','diff','spec+diff '])[(*pState).y_typ]
                   tmp  = stregex((*pState).title,str1,/fold_case,len=len)
                   if len gt 0 then (*pState).title = strmid((*pState).title,0,tmp)+str2+strmid((*pState).title,tmp+len)
                endelse
             endelse
             dcs_cryst_align_plot,pState
          endif
         endif
       'xaxislist':if (*pState).xaxis ne tmp_value then begin
          (*pState).xaxis = tmp_value
          (*pState).fit   = 0
          if ptr_valid((*pState).fitfunstack) then ptr_free,(*pState).fitfunstack
          if n_elements(*(*pState).xPtr) ne 0 then dcs_cryst_align_plot,pState
         end
       'calctthbut':begin
          default = (float(*(*pState).infoPtr))[[0,2+indgen(6),14,15,16]]
          tmpname = [(['lambda ('+string('c5'XB)+')','E (meV)'])[use_E],['a','b','c']+' ('+string('c5'XB)+')','alpha','beta','gamma','H','K','L']+':'
          if use_E then begin ;macs or tas, use energy
             if finite(default[0]) then default[0] = dm_lambda2e(default[0],/double)   
          endif
          info = dm_dialog_input(tmpname,default=default,/float,dialog_parent=(*pState).tlb,xsize=80,title='Calculate 2theta',e2lambda=use_E,lambda2e=~use_E,cancel=cancel)
          if keyword_set(cancel) then return
          ind = where(finite(info,/nan),count)
          if count ne 0 then message,'Missing '+strjoin(([(['lambda','E'])[use_E],'a','b','c','alpha','beta','gamma','H','K','L'])[ind],', ')+'.',/noname
          if use_E then begin
             if (*pState).ftype eq 'MACS' then kidlim  = dm_getkidneyrange(info[0])  ;kidney range
             info[0] = dm_e2lambda(info[0],/double)
          endif
          (*(*pState).infoPtr)[[0,2+indgen(6),14,15,16]] = double(info[0:9])
          Ei  = dm_lambda2e(info[0],/double)
          q   = dcs_cryst_align_qlength1(info[7:9],info[1:6])
          tth = dcs_cryst_align_2theta1(Ei,info[7:9],info[1:6])
          if finite(tth) then begin
             if (*pState).ftype eq 'MACS' then begin
                det_l = (([tth,-tth]-kidlim[0]+76.)/8)<19
                det_u = (([tth,-tth]-kidlim[1]+76.)/8)>0 
                ptais = [-1,-1]
                if det_l[0] ge det_u[0] then ptais[0] = 1>(round((det_l[0]+det_u[0])/2)+1)<20
                if det_l[1] ge det_u[1] then ptais[1] = 1>(round((det_l[1]+det_u[1])/2)+1)<20
                ind = where(ptais gt 0,count)
                if count eq 0 then begin
                   mesg = 'It is outside the detector range.'
                endif else begin
                   mesg = ''
                   for i=0,count-1 do begin
                       ptai   = ptais[ind[i]]
                       kidney = ([tth,-tth])[ind[i]]+76.0-8*(ptai-1)
                       mesg   = [mesg,((count eq 2)?'For 2theta='+dm_to_string(([tth,-tth])[ind[i]],resolution=3)+string('b0'xb)+', ':'')+'PTAI='+dm_to_string(ptai)+', kidney='+$
                                dm_to_string(kidney,resolution=3)+string('b0'xb)+', rate meter channel='+dm_to_string(2*(ptai-1))+' (SPEC) or '+dm_to_string(2*ptai-1)+' (DIFF).']
                   endfor
                   if ind[0] eq 1 then tth = -tth  ;negative side only
                endelse
             endif else if (*pState).ftype ne 'TAS' then begin
                idet = dcs_cryst_align_detnum(pState,tth,bank=bank)
                if idet[0] ne -1 then begin
                   if n_elements(bank) ne 0 then mesg = 'The corresponding detector number is '+dm_to_string(idet[0]+1,/int)+' ('+(['lower','central','upper'])[bank+1]+' bank).' $
                   else mesg = 'The corresponding detector number is '+dm_to_string(idet[0]+1,/int)+'.'
                   if (*pState).ftype eq 'DCS' then mesg = [mesg,'','(Detector number starts from 1. DCS liveplot detector number starts from 0.)'] $
                   else mesg = [mesg,'(Detector number starts from 1.)'] 
                endif else $
                   mesg = 'It is outside the detector range.'
             endif else mesg = ''
             mesg = ['Two theta angle for ('+dm_to_string(info[7:9],separator=', ')+') is '+dm_to_string(tth,resolution=3)+string('b0'xb)+'.',mesg]
          endif else begin
             lmax = 4.*!dpi/q
             if use_E then begin
                emin = dm_lambda2e(lmax)
                mesg = 'Incident energy must be great than '+dm_to_string(emin,resolution=2)+' meV to view ('+dm_to_string(info[7:9],separator=', ')+') reflection.'
             endif else begin
                lmax = dm_to_string(lmax,resolution=3)+' '+string('c5'XB)
                mesg = 'Incident wavelength must be smaller than '+lmax+' to view ('+dm_to_string(info[7:9],separator=', ')+') reflection.'
             endelse
          endelse
          ok = dialog_message(mesg,/info,dialog_parent=(*pState).tlb,title='Result',/center)
         end
       'calclattbut':begin
          default = [abs((*pState).tth[(*pState).ftype eq 'MACS']),float((*(*pState).infoPtr)[[0,5,6,7,14,15,16]]),!values.f_nan,!values.f_nan,!values.f_nan]
          tmpname = ['2theta',(['lambda ('+string('c5'XB)+')','E (meV)'])[use_E],'alpha','beta','gamma','H','K','L','a','b','c']+':'
          if use_E and finite(default[1]) then default[1] = dm_lambda2e(default[1],/double) ;macs or tas, use energy 
          if ptr_valid((*pState).fitfunstack) and ((*pState).ftype eq 'MACS' or (*pState).ftype eq 'TAS') and (*pState).fit and ((*pState).xaxis eq 5) then begin ;get A4 from fit
             default[0] = abs((*(*pState).fitfunstack)[0].param[1])
             a2 = (*(*pState).xPtr)[7+2*((*pState).ftype eq 'MACS'),0]
             if a2 gt 0 then default[1] = round(dm_lambda2e(sin(a2/2.0*!dtor)*3.35416*2)*100.)/100.
          endif         
          info = dm_dialog_input(tmpname,title='Calculate lattice parameter',info='a, b, c values are optional. Enter fixed values only.',default=default,/float,dialog_parent=(*pState).tlb,xsize=80,cancel=cancel)
          if keyword_set(cancel) then return
          ind = where(finite(info[0:7],/nan),count)
          if count ne 0 then message,'Missing '+strjoin((['2theta',(['lambda','E'])[use_E],'alpha','beta','gamma','H','K','L'])[ind],', ')+'.',/noname
          info = double(info)
          if ~use_E then info[1] = dm_lambda2e(info[1],/double)
          abcs = dcs_cryst_align_latta(info[1],info[0],info[5:7],info[2:4],a=info[8],b=info[9],c=info[10])
          if total(finite(abcs,/nan)) eq 0 then begin
             ind = where(finite(info[8:10],/nan),count) 
             if count gt 1 then tmpstr = ' assuming '+strjoin((['a','b','c'])[ind],'=')+'.' else tmpstr = '.'
             mesg = "The lattice paramters are ["+dm_to_string(abcs[*,0],resolution=4,sep=', ')+'] '+string('c5'xB)
             if n_elements(abcs[0,*]) eq 2 then mesg = [mesg,'or ['+dm_to_string(abcs[*,1],resolution=4,sep=', ')+'] '+string('c5'xB)]
             mesg[n_elements(mesg)-1] = mesg[n_elements(mesg)-1]+tmpstr
             ok = dialog_message(mesg,/info,dialog_parent=(*pState).tlb,title='Result',/center)
             info[1] = dm_e2lambda(info[1],/double) ;E->lambda
             (*(*pState).infoPtr)[[0,5,6,7,14,15,16,2,3,4]] = [info[1:7],abcs[*,0]]
          endif else ok = dialog_message("Sorry, no valid solution can be found.",/info,dialog_parent=(*pState).tlb,title='Result',/center)     
         end
       'calca2but':begin
          case (*pState).ftype of 
               'DCS':   info = psiname+['=90-theta, 2theta>0','=-(90+theta), 2theta<0']
               'WAND':  info = psiname+'=theta-90'
               else:    info = psiname+['=theta, 2theta>0','=180+theta, 2theta<0']
          endcase
          tth = dm_dialog_input('two theta:',default=(*pState).tth[(*pState).ftype eq 'MACS'],info=info,xsize=80,dialog_parent=(*pState).tlb,title='Calculate '+psiname+' reset value',cancel=cancel)
          if keyword_set(cancel) then return
          if finite(tth) then begin
             if (*pState).ftype eq 'WAND' then tth = -tth  ;WAND
             if tth ge 0 then a2 = 90.0-tth/2. else a2=-(90.0+tth/2.)
             if ((*pState).ftype eq 'MACS') or ((*pState).ftype eq 'TAS') then a2 = 90-a2  ;MACS or TAS
             mesg = 'Set '+psiname+' to '+dm_to_string(a2,resolution=3)+string('b0'xb)+'.'
             ok = dialog_message(mesg,/info,dialog_parent=(*pState).tlb,title='Result',/center)
          endif
         end
       'calca2rot':begin  
          default = (float(*(*pState).infoPtr))[[0,2+indgen(12)]]
          tmpname = [(['lambda ('+string('c5'XB)+')','E (meV)'])[use_E],['a','b','c']+' ('+string('c5'XB)+')','alpha','beta','gamma','H1','K1','L1','H2','K2','L2']+':'
          if use_E then begin  ;macs or tas, use energy
             if finite(default[0]) then default[0] = dm_lambda2e(default[0],/double)
          endif
          if (*pState).ftype eq 'MACS' then begin
             tmpname = [tmpname,'Scatter Mode']
             default = [default,0]
             droplist_content = ptr_new(['auto','positive','negative'])
             is_droplist = [intarr(n_elements(tmpname)-1),1]
          endif
          info = dm_dialog_input(tmpname,default=default,/float,dialog_parent=(*pState).tlb,droplist_content=droplist_content,is_droplist=is_droplist,/return_number,xsize=80,title='Calculate '+psiname+' Rotation',info='1->2 counterclockwise',cancel=cancel)
          if keyword_set(cancel) then return
          ind = where(finite(info,/nan),count)
          if count ne 0 then message,'Missing '+strjoin(([(['lambda','E'])[use_E],'a','b','c','alpha','beta','gamma','H1','K1','L1','H2','K2','L2'])[ind],', ')+'.',/noname
          if use_E then begin
             if (*pState).ftype eq 'MACS' then kidlim  = dm_getkidneyrange(info[0])  ;kidney range
             info[0] = dm_e2lambda(info[0],/double)
          endif
          (*(*pState).infoPtr)[[0,2+indgen(12)]] = double(info[0:12])
          ki = 2.*!dpi/info[0]
          q1 = dcs_cryst_align_qlength1(info[7:9],info[1:6])
          q2 = dcs_cryst_align_qlength1(info[10:12],info[1:6])
          if max([q1,q2])/2. lt ki then begin
             mesg = ''
             tth1 = asin(q1/2./ki)*2/!dtor
             tth2 = asin(q2/2./ki)*2/!dtor
             if (*pState).ftype eq 'MACS' then begin
                det_l = (([tth1,-tth1]-kidlim[0]+76.)/8)<19
                det_u = (([tth1,-tth1]-kidlim[1]+76.)/8)>0
                if det_l[0] lt det_u[0] then begin
                   if det_l[1] ge det_u[1] then tth1 = -tth1 $ 
                   else mesg = 'Two theta angle for ('+dm_to_string(info[7:9],separator=', ')+') is outside the detector range.'
                endif
                if strlen(mesg) eq 0 then begin
                   det_l = (([tth2,-tth2]-kidlim[0]+76.)/8)<19
                   det_u = (([tth2,-tth2]-kidlim[1]+76.)/8)>0
                   if det_l[0] lt det_u[0] then begin
                      if det_l[1] ge det_u[1] then tth2 = -tth2 $ 
                      else mesg = 'Two theta angle for ('+dm_to_string(info[10:12],separator=', ')+') is outside the detector range.'
                   endif  
                endif 
                if info[13] eq 1 then begin              ;scatter mode is positive
                   tth1 = abs(tth1) & tth2 = abs(tth2)
                endif else if info[13] eq 2 then begin   ;scatter mode is negative
                   tth1 = -abs(tth1) & tth2 = -abs(tth2)
                endif
             endif else if (*pState).ftype ne 'TAS' then begin
                idet = where(*(*pState).tthPtr ge max([tth1,tth2]),count)
                if count ne 0 then begin
                   if (*pState).ftype eq 'WAND' then begin
                      tth1 = -tth1 & tth2 = -tth2
                   endif
                endif else $
                   mesg = 'The vectors are beyond the detector range.'
             endif
             if strlen(mesg) eq 0 then begin
                abet = dcs_cryst_align_angle1(info[7:9],info[10:12],info[1:6]) ;angle between q1 and q2
                if tth1 ge 0 then ang1 = 90.0-tth1/2. else ang1 = -(90.0+tth1/2.)
                if tth2 ge 0 then ang2 = 90.0-tth2/2. else ang2 = -(90.0+tth2/2.)
                ang2 = ang2-abet
                if use_E then begin
                   ang1 = 90-ang1
                   ang2 = 90-ang2
                endif
                mesg = psiname+' needs to rotate '+dm_to_string(ang2-ang1,resolution=2)+string('b0'xb)+'.'
                if ((*pState).ftype eq 'MACS') and ((tth1 gt 0) ne (tth2 gt 0)) then begin
                   mesg = [mesg,'A4 of vect_1 is '+dm_to_string(tth1,res=3)+string('b0'xb)+' and A4 of vect_2 is '+dm_to_string(tth2,res=3)+string('b0'xb)+'.'] 
                endif
             endif
          endif else begin
             lmax = 4.*!dpi/(max([q1,q2]))
             if use_E then begin
                mesg = 'Incident energy must be greater than '+dm_to_string(dm_lambda2e(lmax),resolution=3)+' meV.'
             endif else begin
                mesg = 'Incident wavelength must be smaller than '+dm_to_string(lmax,resolution=3)+' '+string('c5'XB)+'.'
             endelse
          endelse
          ok = dialog_message(mesg,/info,dialog_parent=(*pState).tlb,title='Result',/center)
         end
       'calca3a4':begin
          default = float(((*(*pState).infoPtr))[0:16])
          tmpname = ['Ei (meV):','Ef (meV):',['a','b','c']+' ('+string('c5'XB)+'):','alpha:','beta:','gamma:','U:','','','V:','','','Q:','','']
          default[0] = dm_lambda2e(default[0])
          title = 'Calculate '+(['A3 & A4','A2 & 2theta','Psi & 2theta'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]
          mesg = ['Leave Ef blank for Ef=Ei.','All other fields must be entered.','U and V define the scattering plane.']
          info = dm_dialog_input(tmpname,default=default,/float,dialog_parent=(*pState).tlb,xsize=80,title=title,cancel=cancel,info=mesg,is_e=[1,1,bytarr(15)],/e2lambda)
          if keyword_set(cancel) then return
          ind = where(finite(info[[0,indgen(15)+2]],/nan),count)
          if count ne 0 then message,'Missing '+strjoin((['Ei','a','b','c','alpha','beta','gamma','Uh','Uk','Ul','Vh','Vk','Vl','Qh','Qk','Ql'])[ind],', ')+'.',/noname
          if transpose(info[14:16])#(crossp(info[8:10],info[11:13])) ne 0 then begin
             ok = dialog_message('Q=('+dm_to_string(info[14:16],sep=', ')+') is not in the scattering plane.',/error,dialog_parent=(*pState).tlb,title='Result',/center)
             return
          endif
          a4lim = [-180.,180]
          dcsorwand = (((*pState).ftype eq 'DCS') or ((*pState).ftype eq 'WAND'))
          if dcsorwand then a4lim = [min(*(*pState).tthPtr ),max(*(*pState).tthPtr)]
          if (*pState).ftype eq 'MACS' then begin
             kidlim = dm_getkidneyrange(info[0])+[-1.5,1.5]  ;kidney range
             a4lim  = kidlim+[-76.,76.] 
          endif
          (*(*pState).infoPtr)[0:16] = [dm_e2lambda(info[0],/double),info[1:16]]
          if finite(info[1],/nan) then info[1] = info[0]
          result = ''
          a4name = (['A4','2theta'])[dcsorwand]
          for lefthand=0,1 do begin
              a3a4 = float(dcs_cryst_align_a3a4(info[0],info[1],info[14:16],lattparm=info[2:7],U=info[8:10],V=info[11:13],lefthand=lefthand,error=error,dcs=dcsorwand))
              if keyword_set(error) and (lefthand eq 0) then return
              if (a3a4[1] le a4lim[1]) and (a3a4[1] ge a4lim[0]) then begin
                 if (*pState).ftype eq 'DCS' then begin
                    idet = dcs_cryst_align_detnum(pState,a3a4[1])
                    if idet[0] ne -1 then result = [result,a4name+'='+dm_to_string(a3a4[1])+' (det number='+dm_to_string(idet[0]+1)+'), '+psiname+'='+dm_to_string(a3a4[0])] $
                    else result = [result,a4name+'='+dm_to_string(a3a4[1])+', '+psiname+'='+dm_to_string(a3a4[0])]
                 endif else if (*pState).ftype eq 'MACS' then begin
                    ptai = round(mean(0>(((a3a4[1]-kidlim+76.)/8))<19))+1
                    kidney = a3a4[1]+76.-(ptai-1)*8
                    result = [result,a4name+'='+dm_to_string(a3a4[1])+' (PTAI='+dm_to_string(ptai)+', kidney='+dm_to_string(kidney)+'), '+psiname+'='+dm_to_string(a3a4[0])]
                 endif else begin
                    result = [result,a4name+'='+dm_to_string(a3a4[1])+', '+psiname+'='+dm_to_string(a3a4[0])]
                 endelse
              endif
          endfor
          if n_elements(result) eq 3 then result = [result,'',a4name+' is positive for scattering to the right, and negative to the left.']
          if n_elements(result) eq 1 then result = [result,'No '+psiname+' & '+a4name+' were found.'] $
          else result = ['','Ei='+dm_to_string(info[0])+' meV, Ef='+dm_to_string(info[1])+' meV, Q=('+dm_to_string(info[14:16],sep=', ')+'):',result]
          ok = dialog_message(result[1:*],/info,dialog_parent=(*pState).tlb,title='Result',/center)
         end
       'calchkl':begin
          default = float(((*(*pState).infoPtr))[[17,19,indgen(14)]])
          a3 = (['A3','A2','Psi'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]
          a4 = (['A4','2theta'])[((*pState).ftype eq 'DCS') or ((*pState).ftype eq 'WAND')]
          tmpname = [a3+':',a4+':','Ei (meV):','Ef (meV):',['a','b','c']+' ('+string('c5'XB)+'):','alpha:','beta:','gamma:','U:','','','V:','','']
          default[2] = dm_lambda2e(default[2])
          title = 'Calculate hkl'
          mesg = ['Leave Ef blank for Ef=Ei.','All other fields must be entered.','U and V define the scattering plane.']
          info = dm_dialog_input(tmpname,default=default,/float,dialog_parent=(*pState).tlb,xsize=80,title=title,cancel=cancel,info=mesg,is_e=[0,0,1,1,bytarr(12)],/e2lambda)
          if keyword_set(cancel) then return
          ind = where(finite(info[[0,1,2,indgen(12)+4]],/nan),count)
          if count ne 0 then message,'Missing '+strjoin(([a3,a4,'Ei','a','b','c','alpha','beta','gamma','Uh','Uk','Ul','Vh','Vk','Vl'])[ind],', ')+'.',/noname
          dcsorwand = (((*pState).ftype eq 'DCS') or ((*pState).ftype eq 'WAND'))
          (*(*pState).infoPtr)[[17,19,indgen(14)]] = [info[0:1],dm_e2lambda(info[2],/double),info[3:15]]
          if finite(info[3],/nan) then info[3] = info[2]
          hkl = dcs_cryst_align_hkl(info[2],info[3],info[0],info[1],lattparm=info[4:9],U=info[10:12],V=info[13:15],error=error,dcs=dcsorwand)
          if keyword_set(error) then return
          ok = dialog_message('hkl = ('+dm_to_string(hkl,sep=', ')+') for '+a3+' = '+dm_to_string(info[0])+', '+a4+' = '+dm_to_string(info[1])+'.',/info,$
               dialog_parent=(*pState).tlb,title='Result',/center)
         end  
       'calcvecang': begin
          default = (float(*(*pState).infoPtr))[17:20]
          if (*pState).ftype eq 'DCS' then name = ['A2_1','A2_2','2theta_1','2theta_2'] $
          else if (*pState).ftype eq 'WAND' then name = ['Psi_1','Psi_2','2theta_1','2theta_2'] $
          else name = ['A3_1','A3_2','A4_1','A4_2']
          a3a4 = dm_dialog_input(name+':',xsize=80,dialog_parent=(*pState).tlb,cancel=cancel,default=default)
          if keyword_set(cancel) then return
          ind = where(finite(a3a4,/nan),count)
          if count ne 0 then message,'Missing '+strjoin(name[ind],', ')+'.',/noname
          (*(*pState).infoPtr)[17:20] = double(a3a4)
          ang = dcs_cryst_align_angle2(a3a4[0:1],a3a4[2:3],dcs=(((*pState).ftype eq 'DCS') or ((*pState).ftype eq 'WAND')))
          ok = dialog_message('The angle between the two vectors is '+dm_to_string(ang)+'.',/info,dialog_parent=(*pState).tlb,title='Result',/center)
         end  
       'e2lambdabut':   dcs_cryst_align_e2lambda,event,lambda=float((*(*pState).infoPtr)[0]),dcs=((*pState).ftype eq 'DCS'),macs=(((*pState).ftype eq 'MACS') or ((*pState).ftype eq 'TAS'))
       'kidlimitbut':   dm_macs_kidlimit,event,bgcolor=(*pState).bgcolor
       'macsalignment': begin ;20140103/24394-24425
          ind_dts = (where(stregex(*(*pState).xtit,'DFMDTS',/fold_case,/boolean)))[0] 
          ind_b11 = (where(stregex(*(*pState).xtit,'MonBlade11',/fold_case,/boolean)))[0] 
          ind_a2  = (where(stregex(*(*pState).xtit,'A2',/fold_case,/boolean)))[0]
          ind_mbt = (where(stregex(*(*pState).xtit,'MBTSlide',/fold_case,/boolean)))[0]
          for i=0,n_files-1 do begin
              selected[i] = where(dFiles eq open_file[i])
              (*pState).filesel->set_file,selected[i],/highlight
              dcs_cryst_align_reload,pState,/nofilehistory  ;load data first        
              if ((*pState).xaxis eq ind_dts) or ((*pState).xaxis eq ind_b11) then begin
                 (*pState).fit = 1
                 dcs_cryst_align_plot,pState,params=params
                 if n_elements(angles) eq 0 then begin
                    angles   = (*(*pState).xPtr)[ind_a2+2,0] 
                    mbtslide = (*(*pState).xPtr)[ind_mbt+2,0]  
                    dfmdts   = (((*pState).xaxis eq ind_dts)?(params[1]):0.0)
                    blade11  = (((*pState).xaxis eq ind_b11)?(params[1]):0.0)
                 endif else begin
                    ind = where(abs(angles-(*(*pState).xPtr)[ind_a2+2,0]) le 0.1,cnt)
                    if cnt eq 0 then begin
                       angles   = [angles,(*(*pState).xPtr)[ind_a2+2,0]]
                       mbtslide = [mbtslide,(*(*pState).xPtr)[ind_mbt+2,0]]  
                       dfmdts   = [dfmdts,(((*pState).xaxis eq ind_dts)?(params[1]):0.0)]
                       blade11  = [blade11,(((*pState).xaxis eq ind_b11)?(params[1]):0.0)]
                    endif else begin
                       angles[ind[0]]   = (*(*pState).xPtr)[ind_a2+2,0]
                       mbtslide[ind[0]] = (*(*pState).xPtr)[ind_mbt+2,0]  
                       if ((*pState).xaxis eq ind_dts) then dfmdts[ind[0]] = params[1]
                       if ((*pState).xaxis eq ind_b11) then blade11[ind[0]] = params[1]
                    endelse 
                 endelse
              endif
              wait,0.5
          endfor 
          (*pState).filesel->set_file,selected,/highlight 
          (*pState).filename[0] = filename
          if ~(*pState).showfname then filename = ''
          dm_fit_macs_alignment,angles,mbtslide,dfmdts,blade11,plotobj=(*pState).plotWin,filename=filename,colr0=colr0,group_leader=(*pState).tlb
         end
       'macsalignmonchr': begin ;20140103/24452-24472
          a1center = fltarr(21)
          da1      = fltarr(21)
          datPtr   = ptrarr(21*2)   ;odd for fit
          ind_bld0 = (where(stregex(*(*pState).xtit,'MonBlade01',/fold_case,/boolean)))[0]
          for i=0,n_files-1 do begin
              selected[i] = where(dFiles eq open_file[i])
              (*pState).filesel->set_file,selected[i],/highlight
              dcs_cryst_align_reload,pState,/nofilehistory  ;load data first        
              i_mon = (*pState).xaxis-ind_bld0
              if i_mon ge 0 and i_mon lt 21 then begin
                 (*pState).fit = 1
                 dcs_cryst_align_plot,pState,params=params,sigma=sigma,xs=xs,ys=ys,dys=dys,ytit=ytit
                 a1center[i_mon] = params[1]
                 da1[i_mon]      = sigma[1]
                 ptr_free,datPtr[(2*i_mon):(2*i_mon+1)]
                 datPtr[2*i_mon] = ptr_new([[xs],[ys],[dys]])
                 dcs_cryst_align_fitline,pState,params,xs,xx=xx,yy=yy
                 if n_elements(monpos) eq 0 then monpos = fix(mean(xs))
                 datPtr[2*i_mon+1] = ptr_new([[xx],[yy]])
              endif
              wait,0.5
          endfor
          (*pState).filesel->set_file,selected,/highlight
          (*pState).filename[0] = filename
          if ~(*pState).showfname then filename = ''
          (*pState).plotWin->setproperty,xdat=indgen(21)+1,xran=[0,22],ydat=a1center,yerr=da1,ytit='MonBlade Center (\deg)',xtit='MonBlade Number',/nodraw,$
              title='MACS monochromator alignment',psym='circle',/hidelegend,linestyle='no line',color=colr0,cornertxt=filename
          (*pState).plotWin->draw  
          ok = dialog_message('Do you want to save the fit results?',title='Save?',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin
             file = dm_choose_file('txt',/write,dialog_parent=(*pState).tlb,title='Please select a file name for saving the results',file='macs_monblade_'+date_str+'.txt')
             if strlen(file) ne 0 then begin
                openw,lun,file,/get_lun,error=openerr
                if openerr ne 0 then message,"Can't write in "+file+'.',/noname
                for j=0,20 do printf,lun,j+1,a1center[j]
                free_lun,lun
             endif    
          endif 
          ;move to the center and reset
          ok = dialog_message('Do you want to move to the fit center and reset monochrmator blade positions in ICE?',title='Talk to ICE?',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin
             resetpos = dm_dialog_input('monblade reset position:',default=monpos,dialog_parent=(*pState).tlb,/return_number,cancel=cancel)
             if ~keyword_set(cancel) then begin
                if finite(resetpos,/nan) then begin
                   ok = dialog_message('The entered value is invalid. '+dm_to_string(monpos)+' is used instead.',dialog_parent=(*pState).tlb,/center)
                   resetpos = monpos
                endif       
                (*pState).plotWin->add_plot,[0,22],[resetpos,resetpos],linestyle='dashed',color='grey',/draw
                dm_sendcom,'wait 66666'
                for nn=0,20 do begin
                    blades = 'monblade'+string(nn+1,format='(i02)') 
                    dm_move,blades,a1center[nn]
                    dm_sendcom,'device sethard '+blades + ' '+dm_to_string(resetpos)
                endfor
             endif
          endif
          ;plot all scans
          ok = dialog_message('Do you want to plot all the scan files?',title='Overplot?',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin
             ytit = 'Monitor Counts'
             if (*pState).duration lt 0 then ytit = ytit+'/'+dm_to_string(-(*pState).duration,/int)+' Sec'+(['s',''])[(*pState).duration ge -1]
             psym = ((*pState).plotWin->getpsym(/list))[0:4] & psym = [psym,'filled '+psym]
             cols = ((*pState).plotWin->getcolor(/list))[0:(-2)]
             (*pState).plotWin->erase
             yran = [1e6,-1]
             for i=0,20 do begin
                 if ~ptr_valid(datPtr[2*i]) then continue
                 (*pState).plotWin->add_plot,(*datPtr[2*i])[*,0],(*datPtr[2*i])[*,1],yerr=(*datPtr[2*i])[*,2],linestyle='no line',psym=psym[i mod n_elements(psym)],$
                                    color=cols[i mod n_elements(cols)],legend=dm_to_string(i+1)
                 (*pState).plotWin->add_plot,(*datPtr[2*i+1])[*,0],(*datPtr[2*i+1])[*,1],linestyle='solid',psym='no symbol',color=cols[i mod n_elements(cols)]                   
                 yran[0] = (yran[0])<(min((*datPtr[2*i])[*,1])) & yran[1] = (yran[1])>(max((*datPtr[2*i])[*,1]))
                 xran = [min((*datPtr[2*i])[*,0]),max((*datPtr[2*i])[*,0])]           
             endfor
             (*pState).plotWin->setproperty,xtit='MonBlade (\deg)',ytit=ytit,xran=xran+[-0.1,0.1]*(xran[1]-xran[0]),yran=yran+[-0.1,0.1]*(yran[1]-yran[0]),legdcolumns=2,/legdshowoutline,$
                 hidelegend=0,legdpos=[0.035,0.95]
          endif
          ptr_free,datPtr
         end
       'macsalignanalyzer': begin
          ;ask for the a4 mask range
          mask = dm_dialog_input(['ignore A4 from:','to:'],default=[-12.0,12.0],dialog_parent=(*pState).tlb,/return_number,$
                    info=['Data with A4 within the ignored range will not be used in averaging','the analyzer center position.'],cancel=cancel)
          if keyword_set(cancel) then break
          if total(finite(mask,/nan)) gt 0 then begin
             ok = dialog_message('Invalid range. New detector efficiency is not calculated.',dialog_parent=(*pState).tlb,/center)
             break
          endif
          mask = [min(mask),max(mask)]
          a5center = fltarr(2,20)
          a5weight = fltarr(2,20)
          da5cnt   = fltarr(2,20)
          kidney   = fltarr(2)
          moncnt   = fltarr(2)
          a2       = fltarr(2)
          ind_det0 = 4
          ind_ana0 = (where(stregex(*(*pState).xtit,'AnalyzerTheta01',/fold_case,/boolean)))[0]
          ind_kid  = (where(stregex((*(*pState).xtit),'A4',/fold_case,/boolean)))[0]+2
          ind_a2   = (where(stregex((*(*pState).xtit),'A2',/fold_case,/boolean)))[0]+2
          (*pState).y_typ = 0
          dm_set_droplist,widget_info((*pState).tlb,find_by_uname='timelist'),select=0
          tmp_macs_eff = (*pState).macs_eff ;save efficiency method
          (*pState).macs_eff = 0            ;do not apply efficiency
          for i=0,1 do begin
              selected[i] = where(dFiles eq open_file[i])
              (*pState).filesel->set_file,selected[i],/highlight
              dcs_cryst_align_reload,pState,/nofilehistory ;load data first
              moncnt[i] = (*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-1,0]
              kidney[i] = (*(*pState).xPtr)[ind_kid,0]
              a2[i]     = (*(*pState).xPtr)[ind_a2,0]
              for j=0,19 do begin
                  (*pState).det_tit = 'spec det='+dm_to_string(j+1)
                  (*pState).title   = (*pState).det_tit
                  (*pState).xaxis   = ind_ana0+j
                  (*pState).fit     = 1 
                  *(*pState).detPtr = j
                  dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=ind_det0+j
                  dm_set_droplist,widget_info((*pState).tlb,find_by_uname='xaxislist'),select=ind_ana0+j
                  dcs_cryst_align_plot,pState,params=params,sigma=sigma,xs=xs,ys=ys,dy=dy
                  a5center[i,j] = params[1]
                  a5weight[i,j] = sqrt(2*!PI)*params[0]*params[2]*((i eq 1)?(moncnt[0]/moncnt[1]):1.0)
                  da5cnt[i,j]   = sigma[1]
                  wait,0.5
              endfor
          endfor
          (*pState).filesel->set_file,selected,/highlight
          (*pState).filename[0] = filename
          if ~(*pState).showfname then filename = ''
          (*pState).macs_eff = tmp_macs_eff ;restore efficiency method
          a4     = kidney[0]-76.0+findgen(20)*8.0
          ign_k0 = where((a4 le mask[1]) and (a4 ge mask[0]),cnt0)
          a4     = kidney[1]-76.0+findgen(20)*8.0
          ign_k1 = where((a4 le mask[1]) and (a4 ge mask[0]),cnt1)
          a5_ave = reform((a5center[0,*]+a5center[1,*])/2.)
          da5    = reform(sqrt(da5cnt[0,*]^2+da5cnt[1,*]^2)/2.0) 
          if cnt0 gt 0 then begin
             a5_ave[ign_k0]     = a5center[1,ign_k0]
             da5[ign_k0]        = da5cnt[1,ign_k0]
             a5weight[0,ign_k0] = !values.f_nan 
          endif
          if cnt1 gt 0 then begin
             a5_ave[ign_k1]     = a5center[0,ign_k1]
             da5[ign_k1]        = da5cnt[0,ign_k1]
             a5weight[1,ign_k1] = !values.f_nan 
          endif
          wgh_ave = total(a5weight,1,/nan)/(1>total(finite(a5weight),1)) ;average of two file
          ind     = where(wgh_ave gt 0,cnt,complement=ind1,ncomplement=cnt1)
          (*pState).plotWin->setproperty,xdat=indgen(20)+1,ydat=a5_ave,yerr=da5,ytit='Average Center (\deg)',xtit='Analyzer Number',/nodraw,xran=[0,21],$
              title='A5 alignment @ kidney=('+dm_to_string(kidney,sep=dgsym+', ')+dgsym+')',psym='circle',/hidelegend,linestyle='no line',color=colr0,cornertxt=filename  
          (*pState).plotWin->add_text,'ignore!d1!n = ('+dm_to_string(ign_k0+1,sep=',')+')!Cignore!d2!n = ('+dm_to_string(ign_k1+1,sep=',')+')',0.05,0.93,fontsize=10
          (*pState).plotWin->draw  
          ok = dialog_message('Do you want to save the fit results?',title='Please confirm:',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin
             file = dm_choose_file('txt',/write,dialog_parent=(*pState).tlb,title='Please select a file name for saving the results',file='macs_aligna5_'+date_str+'.txt')
             if strlen(file) eq 0 then break   
             openw,lun,file,/get_lun,error=openerr
             if openerr ne 0 then message,"Can't write in "+file+'.',/noname
             det_eff = fltarr(20)+1.0
             det_eff[ind] = 1.0/wgh_ave[ind]
             det_eff[ind] = det_eff[ind]*cnt/total(det_eff[ind])
             if cnt ne 20 then det_eff = det_eff*20./total(det_eff)
             printf,lun,'det#'+tab+'center1'+tab+'     center2'+tab+'center_average'+tab+tab+'int_int1'+tab+'int_int2'+tab+'int_average'+tab+'det_eff'
             for j=0,19 do printf,lun,string(j+1,format='(i02)'),a5center[0,j],a5center[1,j],a5_ave[j],a5weight[0,j],a5weight[1,j],wgh_ave[j],det_eff[j],format='(i2, 7g)'
             free_lun,lun    
          endif 
          ;move to the center and reset
          ok = dialog_message('Do you want to move to the fit center and reset analyzer positions in ICE?',title='Talk to ICE?',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin
             a5pos = mean(a2)/2.0
             resetpos = dm_dialog_input('analyzer reset position:',default=a5pos,dialog_parent=(*pState).tlb,/return_number,cancel=cancel)
             if keyword_set(cancel) then break
             if finite(resetpos,/nan) then begin
                ok = dialog_message('The entered value is invalid. '+dm_to_string(a5pos)+' is used instead.',dialog_parent=(*pState).tlb,/center)
                resetpos = a5pos
             endif  
             (*pState).plotWin->add_plot,[0,21],[a5pos,a5pos],linestyle='dashed',color='grey',/draw
             dm_sendcom,'wait 66666'
             for i=1,20 do begin
                 analyzer = 'analyzertheta'+string(i,format='(i02)')
                 dm_move,analyzer,a5_ave[i-1]
                 dm_sendcom,'device set '+analyzer+' '+dm_to_string(resetpos)
             endfor
             if cnt1 ne 0 then begin
                ok = dialog_message('Detector '+dm_to_string(ind1+1,sep=', ')+([' are',' is'])[cnt1 eq 1]+' reset even though '+(['they are','it is'])[cnt1 eq 1]+ ' outside the masking range.',dialog_parent=(*pState).tlb,/center)
             endif
          endif
         end
       'macsalignkidney': begin
          dcs_cryst_align_reload,pState,/nofilehistory ;load data first
          *(*pState).detPtr = indgen((*pState).ndet)
          (*pState).det_tit = (['spec ','diff ','spec+diff '])[(*pState).y_typ]+'det=all'
          (*pState).title   = (*pState).det_tit
          (*pState).xaxis   = (where(stregex(*(*pState).xtit,'^A4',/fold_case,/boolean)))[0] 
          dm_set_droplist,widget_info((*pState).tlb,find_by_uname='detlist'),select=0
          dm_set_droplist,widget_info((*pState).tlb,find_by_uname='xaxislist'),select=(*pState).xaxis
          tmp_macs_eff = (*pState).macs_eff ;save efficiency method
          (*pState).macs_eff = 1            ;apply efficiency from file
          dcs_cryst_align_plot,pState,xs=x,ys=y,dys=dy
          (*pState).macs_eff = tmp_macs_eff ;restore efficiency method
          if n_elements(x) eq 0 then break
          ind  = sort(x) & x = x[ind] & y = y[ind]& dy = dy[ind]
          ind1 = where(x le -30,count1)
          ind2 = where(x ge 30,count2)
          dm_gaussfit,y[ind1],x=x[ind1],params=params1,nterms=4,measure_errors=dy[ind1]
          dm_gaussfit,y[ind2],x=x[ind2],params=params2,nterms=4,measure_errors=dy[ind2]
          yy = [params1[0]*exp(-0.5*((x[ind1]-params1[1])/params1[2])^2)+params1[3],params2[0]*exp(-0.5*((x[ind2]-params2[1])/params2[2])^2)+params2[3]]
          (*pState).plotWin->add_plot,[x[ind1],x[ind2]] ,yy,linestyle='solid',color=(['red','cyan'])[total(abs([255,0,0]-colr0)) eq 0],legend='gauss fit',layer=0,thick=2
          (*pState).plotWin->add_text,'peak1 = '+dm_to_string(params1[1],res=3)+dgsym+'!Cpeak2 = '+dm_to_string(params2[1],res=3)+dgsym+'!Ccenter = '+dm_to_string((params1[1]+params2[1])/2.0,res=3)+dgsym,$
             0.05,0.93,fontsize=(*pState).textposfsize[2]
          (*pState).plotWin->draw
          ok = dialog_message('Do you want to redefine kidney angle in ICE?',title='Talk to ICE?',/question,dialog_parent=(*pState).tlb,/center)
          if ok eq 'Yes' then begin  
             dm_sendcom,'wait 66666'
             kidney = dm_dev_pos('kidney',/number)
             dm_move,'kidney',kidney
             dm_sendcom,'device set kidney '+dm_to_string(kidney-(params1[1]+params2[1])/2.0,res=3)
          endif
         end
       'macsflipratio': begin
          moncnt = fltarr(2,2)    ;[*,0] save duration [*,1] save monitor counts
          f_numb = strarr(2)
          tmp_fixfitwidth = (*pState).fixfitwidth
          tmp_showfit = (*pState).showfit
          (*pState).fixfitwidth = 1
          (*pState).showfit = 1
          if (*pState).fr_fitorder then open_file = reverse(open_file)          
          for i=0,1 do begin
              selected[i] = where(dFiles eq open_file[i])
              tmp = stregex(open_file[i],'([0-9]{5,8})\.(ng|b[ta-d])[0-9]$',/extract,/subexpr)
              if strlen(tmp[0]) ne 0 then f_numb[i] = tmp[0] else f_numb[i] = open_file[i]
              (*pState).filesel->set_file,selected[i],/highlight
              dcs_cryst_align_reload,pState,/nofilehistory ;load data first
              moncnt[i,0] = (*pState).duration
              if (*pState).ftype eq 'MACS' then moncnt[i,1] = mean((*(*pState).xPtr)[n_elements((*(*pState).xPtr)[*,0])-1,*]) else moncnt[i,1] =mean((*(*pState).yPtr)[0,*]) 
              (*pState).fit = 1
              if i eq 0 then $
                 dcs_cryst_align_plot,pState,params=params1,sigma=sigma1,xs=xs1,ys=ys1,dy=dy1,fixedwidth=!values.f_nan,peaktime=peaktime1 $
              else $
                 dcs_cryst_align_plot,pState,params=params2,sigma=sigma2,xs=xs2,ys=ys2,dy=dy2,fixedwidth=(n_elements(params1) eq 0?!values.f_nan:params1[2]),peaktime=peaktime2     
              if (n_elements(params1) ne 0) or (n_elements(params2) ne 0) then wait,1
          endfor
          ytit = 'Neutron Counts'
          if moncnt[0,0]*moncnt[1,0] gt 0 then begin
             if moncnt[0,0] eq moncnt[1,0] then ytit = ytit+((moncnt[0,0] lt 0)?('/'+dm_to_string(abs(moncnt[0,0]),/int)+' Sec'+(['s',''])[moncnt[0,0] ge -1]):('/mon='+dm_to_string(moncnt[0,0],/exp))) 
             moncnt = abs(moncnt[*,0])
          endif else moncnt = moncnt[*,1]
          (*pState).fixfitwidth = tmp_fixfitwidth
          (*pState).showfit = tmp_showfit
          (*pState).filesel->set_file,selected,/highlight
          if (n_elements(params1) eq 0) or (n_elements(params2) eq 0) then begin
             if n_elements(xs1) ne n_elements(xs2) then break
             fr = total(ys1)/total(ys2)*moncnt[1]/moncnt[0]
             if fr lt 1 then fr = 1./fr
             dfr = fr*sqrt(1./total(ys1)+1./total(ys2))
          endif else begin
             fr  = params1[0]/params2[0]*moncnt[1]/moncnt[0]
             if fr lt 1 then fr = 1./fr
             dfr = fr*sqrt((sigma1[0]/params1[0])^2+(sigma2[0]/params2[0])^2)
             dcs_cryst_align_fitline,pState,params1,xs1,xx=xx1,yy=yy1
             dcs_cryst_align_fitline,pState,params2,xs2,xx=xx2,yy=yy2
          endelse
          (*pState).plotWin->setproperty,xdat=xs1,ydat=ys1,yerr=dy1,/nodraw,psym='circle',/hidelegend,linestyle='no line',color='black'
          if n_elements(yy1) ne 0 then (*pState).plotWin->add_plot,xx1,yy1,linestyle='solid',color='red',layer=0,thick=2
          (*pState).plotWin->add_plot,xs2,ys2,yerr=dy2,linestyle='no line',psym='circle',color=(['black','green'])[n_elements(params2) eq 0]
          if n_elements(yy2) ne 0 then (*pState).plotWin->add_plot,xx2,yy2,linestyle='solid',color='green',layer=2,thick=2
          xmin  = min([xs1,xs2],max=xmax) & ymin = min([ys1-dy1,ys2-dy2]) & ymax = max([ys1+dy1,ys2+dy2])
          title = 'flipping ratio='+dm_to_string(fr,resolution=3)+'\pm'+dm_to_string(dfr,resolution=3)
          text  = '!C!D'+f_numb[0]+': '+(['',dm_to_string(peaktime1,/date,/est)])[n_elements(peaktime1) ne 0]+' ('+(['red','black'])[n_elements(params1) eq 0]+')'
          text  = text+'!C!D'+f_numb[1]+': '+(['',dm_to_string(peaktime2,/date,/est)])[n_elements(peaktime2) ne 0]+' (green)'
          if (n_elements(peaktime2) ne 0) and (n_elements(peaktime1) ne 0) then text  = text+'!C!D'+'average : '+dm_to_string(mean([peaktime2,peaktime1],/double),/date,/est)
          (*pState).plotWin->add_text,text,0.04,([0.94,0.84])[(n_elements(params1) eq 0) or (n_elements(params2) eq 0)],fontsize=(*pState).textposfsize[2]
          if (*pState).fr_fitorder then f_numb = reverse(f_numb)
          (*pState).plotWin->setproperty,xran=[xmin,xmax]+([-0.1,0.1])*(xmax-xmin),yran=[ymin,ymax]+([-0.1,0.1])*(ymax-ymin),ytit=ytit,title=title,cornertxt=f_numb[0]+','+f_numb[1]
         end  
       'clearhistory': begin
          ptr_free,(*pState).file_hist
          (*pState).n_history = 0
          dcs_cryst_align_setfilehistory,pState
         end
       'dm_filesel_sortfile': dcs_cryst_align_setoption,pState,'sortfile',sortfile
       'dm_filesel_newfilter': begin
          tlb = (*pState).tlb
          if stregex(event.filter,'\.dcs',/fold_case,/boolean)  then new_ftype = 'DCS' else $
          if stregex(event.filter,'\.ng0,* *(\.bt9)*',/fold_case,/boolean) then new_ftype = 'MACS' else $
          if stregex(event.filter,'\.wand',/fold_case,/boolean) then new_ftype = 'WAND' else $
          if stregex(event.filter,'\.(bt[247]|ng5|phd)',/fold_case,/boolean) then new_ftype = 'TAS'
          if n_elements(new_ftype) ne 0 then dcs_cryst_align_changeftype,{WIDGET_BUTTON,ID:widget_info(tlb,find_by_uname='ftype_'+new_ftype),TOP:tlb,HANDLER:tlb,SELECT:1},keepcurrentdir=event.keepcurrentdir 
         end 
       else: begin
          choose = dm_to_number(strmid(uname,6,strlen(uname)-6),/int) 
          uname  = strmid(uname,0,5) 
          if uname eq 'ftype' then dcs_cryst_align_changeftype,event                     ;change file type
          if uname eq 'fhist' then dcs_cryst_align_setfilehistory,pState,choose=choose   ;file history
         end
    endcase
end

pro dcs_cryst_align_Exit,tlb
    widget_control,tlb,get_uvalue=pState,/no_copy
    obj_destroy,(*pState).filesel
    obj_destroy,(*pState).plotWin
    cd,(*pState).root
    ptr_free,(*pState).fname,(*pState).fextn,(*pState).psiname,(*pState).xtit,(*pState).file_hist,(*pState).fitfunstack
    ptr_free,(*pState).detPtr,(*pState).xPtr,(*pState).yPtr,(*pState).dyPtr,(*pState).bgPtr,(*pState).dbgPtr,(*pState).bgmonPtr,(*pState).tthPtr,(*pState).infoPtr
    ptr_free,pState
    widget_control,tlb,/destroy
end

;initparms=[a,b,c,alpha,beta,gamma,uh,uk,ul,vh,vk,vl]
pro dcs_cryst_align,event,initparms,bgcolor=bgcolor,datadir=datadir,workdir=workdir,wand=wand,macs=macs,tas=tas,tickdir=tickdir,render_method=render_method,ncnrftp=ncnrftp,group_leader=group_leader
    state={group_leader:                 0L, $  ;group leader
           tlb:                          0L, $  ;top level base
           mbar:                         0L, $  ;menu bar
           ftype:                     'DCS', $  ;instrument type 'DCS', 'MACS', 'WAND', or 'TAS'
           is_bt7:                       0b, $  ;for BT7 ice and nice data files, including multiple detectors
           fname:   ptr_new(/allocate_heap), $  ;instrument name 
           fextn:   ptr_new(/allocate_heap), $  ;file extension
           psiname: ptr_new(/allocate_heap), $  ;'A2' for DCS, 'A3' for MACS, 'Psi' for WAND
           geom:                    [0L,0L], $  ;geometry
           filesel:               obj_new(), $  ;file selector
           ncnrftp:                      0b, $  ;flag for allowing access to NCNR ftp server
           file_hist:            ptrarr(10), $  ;save the latest 10 file selections
           filename:             ['','',''], $  ;save file name cornertxt string [file name, bgfile name, directory name]
           n_history:                    0s, $  ;number of histories
           infoPtr: ptr_new(/allocate_heap), $  ;store the info for calculating 2theta, [lambda,Ef,a,b,c,alpha,beta,gamma,h1,k1,l1,h2,k2,l2,Qh,Qk,Ql,a3_1,a3_2,a4_1,a4_2]
           fit:                          0b, $  ;flag for fit
           tchoice:                      0b, $  ;temperature choice 0-sample 1-control 2-setpoint
           tempreb_yn:                   1b, $  ;0-no 1-yes if set, rebin temperature axis, default yes
           tempcom_yn:                   0b, $  ;0-no 1-yes if set, combine temperature with the same set point if temperature is the x-axis
           tempavg_yn:                   0b, $  ;0-no 1-yes if set, the tempertures in each file will be averaged, for MACS & WAND & TAS only
           fitfunname:          strarr(4,3), $  ;function formula: [*,0]-gaussian,[*,1]-lorentzian,[*,2]-moffat
           fitfunchoice:                 0b, $  ;0-gaussian, 1-lorentzian, 2-moffat
           fitfunnterm:                  4b, $  ;fit function nterms gaussian-3,4,5,6 lorentzian-3,4,5 moffat-3,4,5 (nterms=fitfunnterm+1 for moffat)
           fit_extrap:                   1b, $  ;if set, extrapolate the fit background for multiple peak, otherwise subtract existing peaks then fit
           fr_fitorder:                  0b, $  ;flipping ratio fitting order 0:1->2 1:2->1
           mpfit_exist:                  0b, $  ;flag for mpfit function existing or not
           fitfunstack:           ptr_new(), $  ;pointer to fit parameters [nparm, nfit]
           showfit:                      0b, $  ;flag for showing fitting parameters
           textposfsize:          fltarr(3), $  ;save the info text position and font size
           fixfitwidth:                  0b, $  ;flag for fix fitting width
           showfname:                    1b, $  ;flag for showing file name at the plot corner
           showcursor:                   0b, $  ;flag for showing cursor position
           ndet:                         0L, $  ;number of detectors
           bitflip:                      1b, $  ;flag for checking the bitflip error
           indvdet:                      0b, $  ;flag for showing individual detectors
           macs_eff:                     0b, $  ;flag for apply macs detector efficiency 0:do not apply 1:apply from file 2:apply specified
           macs_deteff:        fltarr(4,20), $  ;save the macs detector efficiency [0:1,*] file [2:3,*] specify
           macs_cfxdate:                0ll, $  ;date MgF2 CFX filter installed, read from dm_filesel
           macs_lamb2:                   0b, $  ;flag for applying monitor lamda/2 correction
           macs_normchoice:              0b, $  ;flag for normalizing choice for macs, 0-divided by monitor count, 1-divided by time
           macs_maspacing:        fltarr(2), $  ;[monochromator,analyzer] spacing
           macs_histo:                   0b, $  ;macs data mode 0-normal 1-histogram 2-event
           tthPtr:  ptr_new(/allocate_heap), $  ;save the two theta value
           detPtr:  ptr_new(/allocate_heap), $  ;pointer to detector array
           tth:                     [0e,0e], $  ;save the tth vaule [det list 2th, macs actual 2th]
           t_chan:                       1s, $  ;0-all 1-elastic 2-specify
           e_ran:                   [0e,0e], $  ;energy range for t_chan=2
           xaxis:                        0s, $  ;xaxis number
           y_typ:                        0s, $  ;for macs, 0-spec 1-diff 2-spec+diff
           xtit:    ptr_new(/allocate_heap), $  ;xtit strings
           xPtr:    ptr_new(/allocate_heap), $  ;pointer to x data
           yPtr:    ptr_new(/allocate_heap), $  ;pointer to y data
           dyPtr:   ptr_new(/allocate_heap), $  ;pointer to dy data
           bgPtr:   ptr_new(/allocate_heap), $  ;pointer to background data, DCS only
           dbgPtr:  ptr_new(/allocate_heap), $  ;pointer to background error data, DCS only
           bgmonPtr:ptr_new(/allocate_heap), $  ;pointer to background monitor for TAS and MACS 
           bgcolor:                      '', $  ;save plot window background color
           plotWin:               obj_new(), $  ;draw widget
           duration:                     0e, $  ;duration of measurement in secs
           dcs_monitor:             [0e,0e], $  ;dcs monitor counts, [sample,background]
           det_tit:                      '', $  ;output detector string
           tch_tit:                      '', $  ;output time channel string
           title:                        '', $  ;plot title
           root:                         '', $  ;starting directory
           workDir:                      '', $  ;work directory
           datDir:                       '', $  ;data directory
           answered:                     0b, $  ;flag for answering user input for file sorting
           enotebook:                    0b  $  ;use Rob's enotebook
    }
    registerName = 'dcs_cryst_align'
    cd,current = current
    state.root = current
    state.workDir = current
    state.datDir  = current
    state.ncnrftp = (keyword_set(ncnrftp) and ~keyword_set(wand))
    if n_elements(workdir) ne 0 then state.workDir=workdir[0]
    if n_elements(datadir) ne 0 then state.datDir=datadir[0]
    if n_elements(bgcolor) ne 0 then state.bgcolor=bgcolor[0] else state.bgcolor = 'white'

    *state.infoPtr = dblarr(21) & (*state.infoPtr)[*] = !values.d_nan
    if (n_elements(initparms) le 12) and (n_elements(initparms) ne 0) then (*state.infoPtr)[2:(n_elements(initparms)+1)] = initParms
    
    ;handle instrument choice. To add a new instrument, modify dcs_cryst_align_changeftype
    if keyword_set(macs) then state.ftype = 'MACS' $
    else if keyword_set(wand) then state.ftype = 'WAND' $
    else if keyword_set(tas)  then state.ftype = 'TAS' $
    else if xregistered('dcs_mslice',/noshow) then begin  ;if started on IDL
         FORWARD_FUNCTION LookupManagedWidget
         id = LookupManagedWidget('dcs_mslice')
         if id ne 0 then begin
            widget_control,id,get_uvalue=obj_Mslice
            if obj_valid(obj_Mslice) then begin
               obj_Mslice->getproperty,ftype=ftype,latticeparm=latticeparm,dirs=dirs & ftype = strupcase(ftype)
               if (ftype eq 'MACS') or (ftype eq 'WAND') then state.ftype = ftype
               if finite((*state.infoPtr)[2],/nan) then (*state.infoPtr)[2:13] = latticeparm
               if n_elements(datadir) eq 0 then state.datDir = dirs[0]
            endif
         endif
    endif
 
    state.macs_deteff[2:3,*] = !values.f_nan
    
    pState = ptr_new(state,/no_copy)
    
    dcs_cryst_align_changeftype,/init,pState=pState,det_info=det_info,x_info=x_info

    defsysv,'!ddtor',exists=exists
    if (~exists) then defsysv,'!ddtor',!dpi/(180d) ;double version of !dtor
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists ;should exist if starts from dave
    if (~ exists) then begin
       file = file_which('dcs_DetPos2.dat',/INCLUDE_CURRENT_DIR)
       if float(!version.release) ge 6.0 then path = file_dirname(file,/mark_directory) $
       else begin
          id   = strpos(file,pathsep,/reverse_search)
          path = strmid(file,0,id+1)
       endelse
       defsysv,'!DAVE_AUXILIARY_DIR',path
    endif
    icon = !DAVE_AUXILIARY_DIR+'align.ico'
    if dm_to_number(!version.release) ge 6.4 then icon_extra={bitmap:icon}
    
    if n_elements(event) ne 0 then begin
       if n_elements(group_leader) eq 0 then group_leader = event.top
    endif
    if n_elements(group_leader) ne 0 then (*pState).group_leader = group_leader
    (*pState).tlb  = widget_base(title=(*pState).ftype+([' Single Crystal',''])[(*pState).ftype eq 'MACS']+' Alignment Utility',/column,kill_notify='dcs_cryst_align_Exit',$
          /tlb_size_event,group_leader=group_leader,_extra=icon_extra,map=0,mbar=bar,uname='tlb')
    (*pState).mbar = bar      
    
    ;check if mpfitpeak routine exists
    (*pState).mpfit_exist = 1b
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       (*pState).mpfit_exist = 0b
       (*pState).fitfunchoice = 0
    end
    if (*pState).mpfit_exist then resolve_routine,'mpfitpeak',/no_recompile,/is_function  ;check if mpfitpeak function exists, for Lorentzian and Moffat fit

    ;file menu
    filemenu    = widget_button(bar,value='File',/menu)
    ftypemenu   = widget_button(filemenu,value='File Type',/menu,uname='ftypeMenu')
    fhistmenu   = widget_button(filemenu,value='Recent Files',/menu,uname='fhistMenu',sensitive=0,/separator)
    clearhist   = widget_button(filemenu,value='Clear File History',uname='clearHistory',sensitive=0)
    fileExit    = widget_button(filemenu,value='Exit',uname='exitbut',/separator)
    for i=0,n_elements(*(*pState).fname)-1 do $
        void    = dm_widget_button(ftypemenu,value=(*(*pState).fname)[i],uname='ftype_'+(*(*pState).fname)[i],set_button=strmatch((*(*pState).fname)[i],(*pState).ftype),/fold_case)
    
    ;option menu
    optnmenu    = widget_button(bar,value='Options',/menu,uname='optnMenu')   
    optnBitflip = dm_widget_button(optnmenu,value='Check Bitflip Error',set_button=(*pState).bitflip,onstring='Bitflip Error Correction Enabled',$
                  offstring='Bitflip Error Correction Disabled',uname='optnbitflip',sensitive=((*pState).ftype eq 'DCS'))
    optnIndvDet = dm_widget_button(optnmenu,value='Show Individual Detector',set_button=(*pState).indvdet,onstring='Show Individual Detector',$
                  offstring='Sum Detectors',uname='optnindvdet',sensitive=((*pState).ftype ne 'TAS'))   
    optnShowFnm = dm_widget_button(optnmenu,value='Show File Name',set_button=(*pState).showfname,onstring='Show File Name',$
                  offstring='Hide File Name',uname='optnshowfname')    
    optnShowCur = dm_widget_button(optnmenu,value='Show Cursor Position',set_button=(*pState).showcursor,onstring='Show Cursor Position',$
                  offstring='Hide Cursor Position',uname='optnshowcursor')
    optnShowFit = dm_widget_button(optnmenu,value='Show Fitting Parameters',set_button=(*pState).showfit,onstring='Show Fitting Parameters',$
                  offstring='Hide Fitting Parameters',uname='optnshowfit')
    optnFixWid  = dm_widget_button(optnmenu,value='Fix Fitting Width',set_button=(*pState).fixfitwidth,onstring='Fix Fitting Width',$
                  offstring='Not Fix Fitting Width',uname='optnfixwidth')
    void        = dm_widget_button(optnmenu,value='Extrapolate Multi-peak Backgrounds',uname='fit_extrap',set_button=(*pState).fit_extrap)
    sortmenu    = widget_button(optnmenu,value='Sort Files by (File Number)',uname='sortMenu',/menu,/separator)
    void        = dm_widget_button(sortmenu,value='File Number',uname='sortnumb',/set_button)
    void        = dm_widget_button(sortmenu,value='File Name',uname='sortname')               
    tempmenu    = widget_button(optnmenu,value='Temperature Choice'+((*pState).ftype eq 'WAND'?'':' ('+(['Sample','Control','Setpoint'])[(*pState).tchoice]+')'),uname='tempMenu',/menu)
    void        = dm_widget_button(tempmenu,value='Sample',uname='temp_samp',set_button=((*pState).tchoice eq 0),sensitive=((*pState).ftype ne 'WAND')) 
    void        = dm_widget_button(tempmenu,value='Control',uname='temp_ctrl',set_button=((*pState).tchoice eq 1),sensitive=((*pState).ftype ne 'WAND'))
    void        = dm_widget_button(tempmenu,value='Setpoint',uname='temp_setp',set_button=((*pState).tchoice eq 2),sensitive=((*pState).ftype ne 'WAND'))
    void        = dm_widget_button(tempmenu,value='Rebin Temperature',uname='temp_rebin',/separator,set_button=(*pState).tempreb_yn)
    void        = dm_widget_button(tempmenu,value='Combine Temperature with Same Set Point',uname='temp_comb',set_button=(*pState).tempcom_yn,sensitive=(((*pState).ftype ne 'WAND') and (*pState).tempreb_yn))
    void        = dm_widget_button(tempmenu,value='Average Temperature of Each File',uname='temp_avg',set_button=(*pState).tempavg_yn)
    fitfcmenu   = widget_button(optnmenu,value='Fitting Function ('+(['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+')',uname='fitfcMenu',/menu)
    void        = dm_widget_button(fitfcmenu,value='Gaussian',uname='fit_gaussian',set_button=((*pState).fitfunchoice eq 0))
    void        = dm_widget_button(fitfcmenu,value='Lorentzian',uname='fit_lorentzian',set_button=((*pState).fitfunchoice eq 1),sensitive=(*pState).mpfit_exist) 
    void        = dm_widget_button(fitfcmenu,value='Moffat',uname='fit_moffat',set_button=((*pState).fitfunchoice eq 2),sensitive=(*pState).mpfit_exist)
    (*pState).fitfunname = [['C0*exp[-(x-C1)^2/(2*C2^2)]'+['','+C3','+C3+C4*x','+C3+C4*x+C5*x^2']],['C0/(1+(x-C1)^2/C2^2)'+['','+C3','+C3+C4*x','+C3+C4*x+C5*x^2']],['C0/(1+(x-C1)^2/C2^2)^C3'+['','+C4','+C4+C5*x','+C4+C5*x+C6*x^2']]]
    for i=0,3 do void = dm_widget_button(fitfcmenu,value=(*pState).fitfunname[i,(*pState).fitfunchoice],uname='fitfun_'+dm_to_string(3+i),set_button=((*pState).fitfunnterm eq 3+i),separator=(i eq 0),$
                 sensitive=((i ne 3) or (((*pState).fitfunchoice eq 0) and (~(*pState).fixfitwidth))))
    if (*pState).ftype eq 'MACS' then begin  ;MACS
       normmenu = widget_button(optnmenu,value='Intensity Normalization Choice ('+(['Monitor','Time','Pulse Number'])[(*pState).macs_normchoice]+')',uname='normMenu',/menu)
       void     = dm_widget_button(normmenu,value='Divided by Monitor Count',uname='norm_monitor',set_button=((*pState).macs_normchoice eq 0))
       void     = dm_widget_button(normmenu,value='Divided by Counting Time',uname='norm_time',set_button=((*pState).macs_normchoice eq 1))
       void     = dm_widget_button(normmenu,value='Divided by Pulse Number',uname='norm_pulse',set_button=((*pState).macs_normchoice eq 2),sensitive=keyword_set((*pState).macs_histo))
       void     = dm_widget_button(normmenu,value='Apply Monitor Lambda/2 Correction',set_button=(*pState).macs_lamb2,onstring='Apply Monitor Lambda/2 Correction',$
                                   offstring='No Monitor Lambda/2 Correction',uname='optnapplylamb2',/separator)
       macseffmenu = dm_widget_button(optnmenu,value='Apply Detector Efficiency ('+(['None','From File','Specify'])[(*pState).macs_eff]+')',set_button=((*pState).macs_eff ne 0),uname='macseffMenu',/menu)
       void     = dm_widget_button(macseffmenu,value='From File',uname='macs_eff_file',set_button=((*pState).macs_eff eq 1))
       void     = dm_widget_button(macseffmenu,value='Specify...',uname='macs_eff_spec',set_button=((*pState).macs_eff eq 2))
       void     = dm_widget_button(macseffmenu,value='None',uname='macs_eff_none',set_button=((*pState).macs_eff eq 0),sensitive=((*pState).macs_eff ne 0),/separator)
       macshistomenu = widget_button(optnmenu,value='MACS Data Mode ('+(['Normal','Histogram','Event'])[(*pState).macs_histo]+')',uname='macshistoMenu',/menu)
       void     = dm_widget_button(macshistomenu,value='Normal',uname='macs_histo0',set_button=((*pState).macs_histo eq 0))
       void     = dm_widget_button(macshistomenu,value='Histogram',uname='macs_histo1',set_button=((*pState).macs_histo eq 1))
       void     = dm_widget_button(macshistomenu,value='Event',uname='macs_histo2',set_button=((*pState).macs_histo eq 2))
    endif
    if (*pState).ftype ne 'WAND' then ftpbutton = dm_widget_button(optnmenu,value='Allow NCNR Data Repository',uname='ncnrftp',set_button=(*pState).ncnrftp,/separator)
        
    ;tools menu
    iftype      = (where(*(*pState).fname eq (*pState).ftype))[0]
    toolmenu    = widget_button(bar,value='Tools',/menu,uname='toolMenu')
    calctthBut  = widget_button(toolmenu,value='Calculate 2theta...',uname='calctthbut')
    calclattBut = widget_button(toolmenu,value='Calculate Lattice Parameter...',uname='calclattbut')
    calca2But   = widget_button(toolmenu,value='Calculate '+(*(*pState).psiname)[iftype]+' Reset Value...',uname='calca2but')
    calca2Rot   = widget_button(toolmenu,value='Calculate '+(*(*pState).psiname)[iftype]+' Rotation Angle...',uname='calca2rot')
    calca3a4But = widget_button(toolmenu,value='Calculate '+(['A3 && A4','A2 && 2theta','Psi && 2theta'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]+'...',uname='calca3a4')
    calchklBut  = widget_button(toolmenu,value='Calculate hkl from '+(['A3 && A4','A2 && 2theta','Psi && 2theta'])[((*pState).ftype eq 'DCS')+2*((*pState).ftype eq 'WAND')]+'...',uname='calchkl')
    calcvkAng   = widget_button(toolmenu,value='Calculate Angle between Two Vectors from '+(['A3s && A4s','A2s && 2thetas','Psis && 2thetas'])[((*pState).ftype eq 'DCS')+$
                  2*((*pState).ftype eq 'WAND')]+'...',uname='calcvecang')
    panfit      = widget_button(toolmenu,value='PAN Fit Data',uname='fit_pan',/separator)
    e2lambdaBut = widget_button(toolmenu,value='E to Wavelength Conversion',/separator,uname='e2lambdabut')
    if (*pState).ftype eq 'MACS' then begin  ;MACS
       void     = widget_button(toolmenu,value='Kidney Angle Limit',uname='kidlimitbut')
       void     = widget_button(toolmenu,value='Fit and Plot Instrument Alignment Files',uname='macsalignment',/separator)
       void     = widget_button(toolmenu,value='Fit and Plot Monochromator Alignment Files',uname='macsalignmonchr')
       void     = widget_button(toolmenu,value='Fit and Plot Analyzer Alignment Files',uname='macsalignanalyzer')
       void     = widget_button(toolmenu,value='Fit and Plot Kidney Alignment File',uname='macsalignkidney')
    endif
    if ((*pState).ftype eq 'TAS') or ((*pState).ftype eq 'MACS') then $
       void     = widget_button(toolmenu,value='Calculate Flipping Ratio',uname='macsflipratio',/separator)
    
    ;background menu
    backmenu    = widget_button(bar,value='Background',/menu)
    void        = dm_widget_button(backmenu,value='Load Empty Can File(s)',uname='loadbgfile')
    void        = widget_button(backmenu,value='Clear Background',uname='clearbgfile',/separator,sensitive=0)
    
    if !version.os_family eq 'Windows' then $
       mbar_sep = widget_label((*pState).tlb,sensitive=0,value=' ',/dynamic_resize,scr_ysize=4)
    
    tmpbase     = widget_base((*pState).tlb,/row)   
    left        = widget_base(tmpbase,/col)
    (*pState).filesel = obj_new('dm_filesel',parent=left,xsize=160,ysize=20,/frame,filter=(*(*pState).fextn)[iftype],path=(*pState).datDir,group_leader=(*pState).tlb,$
                  ncnrftp=(*pState).ncnrftp,workdir=(*pState).workdir,/sortevent)
    if (*pState).ncnrftp then begin
       (*pState).filesel->getproperty,ncnrftp=ncnrftp
       if (*pState).ncnrftp ne ncnrftp then dm_set_button,ftpbutton,ncnrftp
       (*pState).ncnrftp = ncnrftp
    endif
    if !version.os_family eq 'unix' then $
       center   = widget_base(tmpbase,/col,/align_bottom,xpad=0,ypad=0,space=0,uname='center') $
    else $
       center   = widget_base(tmpbase,/grid_layout,/col,/align_bottom,uname='center')

    void        = widget_label(center,value=(['Det. Number','Detector'])[(*pState).ftype eq 'TAS'],uname='detlabel')
    detList     = dm_widget_droplist(center,value=[det_info,'SmplUTranslation'],uname='detlist')
    void        = widget_label(center,value=(['Time Channel','Det. Type'])[(*pState).ftype eq 'MACS'],uname='timelabel',sensitive=(((*pState).ftype ne 'WAND') and ((*pState).ftype ne 'TAS')))
    timeList    = dm_widget_droplist(center,value=([['All','Elastic','Specify',' '],['SPEC','DIFF','SPEC+DIFF','Monitor']])[0:([2,3])[(*pState).ftype eq 'MACS'],((*pState).ftype eq 'MACS')],$
                  select=([(*pState).t_chan,(*pState).y_typ])[(*pState).ftype eq 'MACS'],sensitive=(((*pState).ftype ne 'WAND') and ((*pState).ftype ne 'TAS')),uname='timelist')
    void        = widget_label(center,value='X-axis',uname='xaxislabel')
    xaxisList   = dm_widget_droplist(center,value=x_info,uname='xaxislist')
    ;for context sensitive menu
    dcs_cryst_align_csm,pState
    loadBut     = widget_button(center,value='Load Data',uname='loadbut')
    fitBut      = widget_button(center,value=(['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+' Fit',uname='fitbut')
    clpbBut     = widget_button(center,value='Copy to Clipboard',uname='clpbbut')
    saveBut     = widget_button(center,value='Save Plot',uname='savebut')
    dataBut     = widget_button(center,value='Save Data',uname='databut')
    printBut    = widget_button(center,value='Print',uname='printbut')
    if dm_to_number(!version.release) ge 5.6 then begin
       widget_control,fitBut,tooltip='Fit the highest peak or zoom in to fit a local peak to a '+(['Gaussian','Lorentzian','Moffat'])[(*pState).fitfunchoice]+' function:'+$
          string(10b)+'y='+(*pState).fitfunname[(*pState).fitfunnterm-3,(*pState).fitfunchoice]+'.'
       widget_control,dataBut,tooltip='Save the plot data in an ascii file.'
    endif
    
    geom  = widget_info(printBut,/geometry)
    widget_control,detList,xsize=geom.scr_xsize,set_value=det_info
    widget_control,timeList,xsize=geom.scr_xsize
    widget_control,xaxisList,xsize=geom.scr_xsize

    geom  = widget_info(left,/geometry)
    ysize = geom.scr_ysize
    xsize = fix(1.4*ysize,type=3)
    (*pState).plotWin = obj_new('dm_plot',xsize=xsize,ysize=ysize,widgetbase=tmpbase,group_leader=(*pState).tlb,/compound,background=(*pState).bgcolor,/isolatin1,ctxtfsize=8,$
       showcursorpos=(*pState).showcursor,render_method=render_method,tickdir=tickdir)

    geom  = widget_info(center,/geometry)
    widget_control,center,xsize=geom.scr_xsize
       
    dm_center_kid, (*pState).tlb,(*pState).group_leader,/side   ;centering the top level base
    widget_control,(*pState).tlb,/realize,/map
    
    geom  = widget_info((*pState).tlb,/geometry)
    (*pState).geom = [geom.scr_xsize,geom.scr_ysize]
    
    (*pState).filesel->getproperty,dates=dates
    (*pState).macs_cfxdate = dates[1] ;date mgf2 cfx filter installed
    
    widget_control,(*pState).tlb,set_uvalue=pState

    if xregistered('dave',/noshow) and (n_elements(event) ne 0) then begin
       (*pState).enotebook = 1b
       RET = DAVE_SET_FOCUS((*pState).tlb)
    endif
    
    xmanager,registerName,(*pState).tlb,cleanup='dcs_cryst_align_Exit',/no_block
end