; $Id$
;+
; CLASS_NAME:
;   DAVEGeneralSettings
;
; PURPOSE:
;   Extend IDLitGeneralSettings class in order to add additional
;   general settings that can be modified by a user at runtime from the
;   Preferences dialog
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitComponent
;   IDLitIMessaging
;
; METHODS:
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; March 2007
;-


function DAVEGeneralSettings::Init, _REF_EXTRA=_extra
compile_opt idl2

; call base class init
if (~Self->IDLitGeneralSettings::Init(_extra=_extra)) then $
  return,0

;; Register additional properties
Self->RegisterProperty, 'DATA_DIRECTORY', USERDEF='', $
  NAME='Raw data directory', $
  DESCRIPTION='Default raw data directory used for file selection'

Self->Registerproperty, 'FTKWWFLAG', $
  ENUMLIST=['NO','YES'], $
  NAME="Use FT_KWW in PAN?", $
  DESCRIPTION='Control use of FT_KWW in PAN'

Self->RegisterProperty, 'DEBUGFLAG', $
  ENUMLIST=['OFF','ON'], $
  NAME="Debugging (!debug)", $
  DESCRIPTION='Set error handling behaviour', sensitive=~lmgr(/runtime)

;; Default values.
;Self._printVec = 0              ; 0=bitmap, 1=vector
;Self._clipVec = 0               ; bitmap, 1=vector
Self._debugFlag = 0
Self._ftkwwFlag = 0              ; 0=>No to using ft_kww function in PAN
defaultDir = ''
found = IDLitGetResource('',strName,/User) ; find location of .idl resource directory
if (found) then defaultDir = strmid(strname,0,strpos(strname,'.idl')-1) ; strip the .idl... part to get 'home' dir
Self->SetProperty, working_directory=defaultDir, data_directory=defaultDir

if (N_ELEMENTS(_extra) gt 0) then $
  self->SetProperty, _EXTRA=_extra

return, 1
end

;;---------------------------------------------------------------------------
;; DAVEGeneralSettings::Cleanup
;;
;; Purpose:
;; The destructor for the class.
;;
;; Parameters:
;; None.
;;
;pro DAVEGeneralSettings::Cleanup
;
;    compile_opt idl2, hidden
;
;    ;; Cleanup superclass
;    self->IDLitComponent::Cleanup
;end


;;---------------------------------------------------------------------------
;; Property Management
;;---------------------------------------------------------------------------
;; DAVEGeneralSettings::GetProperty
;;
;; Purpose:
;;   Used to get the value of the properties associated with this class.
;;
;; Keywords:
;;   DATA_DIRECTORY  - The raw data directory
;;
;;   debugFlag     - will 1 if debugging or 0 otherwise
;;
pro DAVEGeneralSettings::GetProperty, $
  DATA_DIRECTORY=dataDirectory, $
  debugFlag=debugFlag, $
  ftkwwFlag=ftkwwFlag, $
  propToBeModified=prop2mod, $
  _REF_EXTRA=_super
compile_opt idl2

; The property to be modified
if (arg_present(prop2mod)) then prop2mod = self._propToBeModified

; Data directory
if (ARG_PRESENT(dataDirectory)) then $
    dataDirectory = self._strDataDirectory

; Debug flag
if (arg_present(debugFlag)) then debugFlag=self._debugFlag
    
; ft_kww flag
if (Arg_present(ftkwwFlag)) then ftkwwFlag=self._ftkwwFlag

;; Call the superclass
if(n_elements(_super) gt 0)then $
    self->IDLitGeneralSettings::GetProperty, _EXTRA=_super

end

;;---------------------------------------------------------------------------
;; DAVEGeneralSettings::SetProperty
;;
;; Purpose:
;;   Used to set the value of the properties associated with this class.
;;
;; Properties:
;;   DATA_DIRECTORY  - The raw data directory
;;
;;   debugFlag     -  1 if debugging or 0 otherwise
;;
pro DAVEGeneralSettings::SetProperty, $
    DATA_DIRECTORY=dataDirectoryIn, $
    propToBeModified=prop2mod, $
    FTKWWFLAG=ftkwwFlag, $
    DEBUGFLAG=debugflag, $
    _EXTRA=_super

compile_opt idl2, hidden

if (n_elements(prop2mod) gt 0) then self._propToBeModified = prop2mod

if (N_ELEMENTS(dataDirectoryIn) gt 0) then begin
   Self._strDataDirectory = dataDirectoryIn
   Self->SetPropertyAttribute, 'DATA_DIRECTORY', $
       USERDEF=self._strDataDirectory

   dataDirectory = dataDirectoryIn

   ; syncronize change with all live tools
   Self->_UpdateCurrentTools,/update_tool, data_directory=dataDirectory
endif

if (n_elements(debugFlag) GT 0) then begin
    self._debugFlag = debugFlag
    defsysv,'!debug',exist=isDefined
    case isDefined of
        1: !debug = self._debugFlag
        else: defsysv,'!debug', self._debugFlag
    endcase
endif

if (n_elements(ftkwwFlag) gt 0) then begin
  self._ftkwwFlag = ftkwwFlag
  Defsysv,'!ftkwwFlag',exist=isDefined
  case isDefined of
    1: !ftkwwFlag = self._ftkwwFlag
    else: Defsysv,'!ftkwwFlag', self._ftkwwFlag
  endcase
  if (!ftkwwFlag) then begin
    msg = []
    ;msg = 'DISCLAIMER'
    msg = [msg,'Fourier transform of the Kohlrausch-Wlliams-Watts function']
    msg = [msg,'in Peak Fitting']
    msg = [msg,'']
    msg = [msg,'WARNING: This model function uses a numerical Fourier Transform']
    msg = [msg,'to treat data. There are numerous sources of systematic error ']
    msg = [msg,'associated with performing such Fourier Transformations. Testing']
    msg = [msg,'of this function was limited to specific cases. Especially, if']
    msg = [msg,'the energy window of the data is large compared to the resolution']
    msg = [msg,'function, the tau values are large, or the Beta values are low ']
    msg = [msg,'spurious effect might be present. By clicking "YES" below you are']
    msg = [msg,'accepting complete responsibility for knowing all of the pitfalls']
    msg = [msg,'and error sources associated with this model function, and you ']
    msg = [msg,'are responsible for all results that come from using it to model']
    msg = [msg,'your data.']
    
    
    msg = [msg,'']
    msg = [msg,'Do you accept this disclaimer?']
    title = 'FT_KWW Disclaimer                                                               '
    status = Self->Promptuseryesno(msg,answer,title=title)
    if (status eq 0 || answer eq 0) then begin
      ;; disclaimer not accepted so flag should remain switched off
      self._ftkwwFlag = 0
      !ftkwwFlag = self._ftkwwFlag
    endif
;    answer = Dialog_message(dialog_parent = self.tlb,msg,/question,  $
;      title = title)
  endif
endif
    
 

if (N_ELEMENTS(_super) gt 0) then $
    self->IDLitGeneralSettings::SetProperty, _EXTRA=_super

end


;----------------------------------------------------------------------------
; Purpose:
;   This function method is used to edit a user-defined property.
;
; Arguments:
;   Tool: Object reference to the tool.
;
;   PropertyIdentifier: String giving the name of the userdef property.
;
; Keywords:
;   None.
;
function DAVEGeneralSettings::EditUserDefProperty, oTool, identifier
compile_opt idl2

self->SetProperty, propTobeModified=identifier

; call base class
void = Self->IDLitGeneralSettings::EditUserDefProperty(oTool, identifier)

case identifier of

    'DATA_DIRECTORY': begin
        status = oTool->DoUIService('SelectDirectory', self)
        return, status               ; returning 0 seems to activate undo; don't need to undo/redo
    end

    else:

endcase

return, 1                           ; returning 0 seems to activate undo sometimes

end


;---------------------------------------------------------------------------
; Purpose:
;   Internal method to update a single tool with the tool settings.
;
;   WORKING_DIRECTORY will either come in as the current member value,
;   either from _UpdateToolSettings or SetProperty, or it will come
;   in undefined if it wasn't set in SetProperty.
;
pro DAVEGeneralSettings::_UpdateMyToolSettings, oTool, $
  WORKING_DIRECTORY=workingDirectory, DATA_DIRECTORY=dataDirectory
compile_opt idl2

; Tool settings.
oTool->SetProperty, $
  CHANGE_DIRECTORY=self._bChangeDirectory, $
  WORKING_DIRECTORY=workingDirectory, $
  DATA_DIRECTORY=dataDirectory

end


;---------------------------------------------------------------------------
; Purpose:
;   Internal method to update a single tool with the current settings.
;   Called only from the System object on tool creation.
;
pro DAVEGeneralSettings::_InitialToolSettings, oTool
compile_opt idl2

self->_UpdateBufferSettings, oTool
self->_UpdateWindowSettings, oTool
self->_UpdateMyToolSettings, oTool, $
  WORKING_DIRECTORY=self._strWorkingDirectory, DATA_DIRECTORY=self._strDataDirectory
end


;;---------------------------------------------------------------------------
;; DAVEGeneralSettings::_UpdateCurrentTools
;;
;; Purpose:
;;   This routine will update the buffersettings on all currently
;;   active tools with the current settings in this system.
;;
pro DAVEGeneralSettings::_UpdateCurrentTools, $
  UPDATE_BUFFER=updateBuffer, $
  UPDATE_WINDOW=updateWindow, $
  UPDATE_TOOL=updateTool, $
  WORKING_DIRECTORY=workingDirectory, $
  DATA_DIRECTORY=dataDirectory
compile_opt idl2

;; Get the current set of tools and loop over them.
oSystem = self->GetTool()
if (~OBJ_VALID(oSystem)) then $
  return

oToolsCon = oSystem->GetByIdentifier("/TOOLS")
if (~OBJ_VALID(oToolsCon)) then $
  return
oTools = oToolsCon->Get(/ALL, COUNT=nTools)

for i=0, nTools-1 do begin
    if (KEYWORD_SET(updateBuffer)) then $
      self->_UpdateBufferSettings, oTools[i]
    if (KEYWORD_SET(updateWindow)) then $
      self->_UpdateWindowSettings, oTools[i]
    if (KEYWORD_SET(updateTool)) then $
      self->_UpdateMyToolSettings, oTools[i], $
        WORKING_DIRECTORY=workingDirectory, DATA_DIRECTORY=dataDirectory
endfor

end


;;---------------------------------------------------------------------------
;; Definition
;;---------------------------------------------------------------------------
;; DAVEGeneralSettings__Define
;;
;; Purpose:
;; Class definition for the DAVEGeneralSettings class
;;

pro DAVEGeneralSettings__Define
;; Pragmas
compile_opt idl2

void = {DAVEGeneralSettings, $
        inherits   IDLitGeneralSettings,    $
        _strDataDirectory : '', $ ; Current data directory
        _debugFlag : 0, $
        _ftkwwFlag : 0, $
        _propToBeModified: '' $
       }
end
