;+
; Purpose:
;   DaveHTTPRequest is a wrapper class for the built in HttpRequest class used to make 
;   GET, POST, PUT, and DELETE requests to an HTTP or HTTPS server.
;   
;   It is an effective replacement for DaveFtpUrl class since the NCNR public data
;   repository no longer supports access through the ftp protocol.
;-
;


;==============================================================================
pro DaveHTTPRequest::Cleanup
  compile_opt idl2

  Obj_destroy, Self.ourl

end

;==============================================================================
pro DaveHTTPRequest::CloseConnections
  compile_opt idl2
  ; does nothing but implemented to maintain same interface like for DaveFtpUrl class that this replaces
  ;Self.ourl->Closeconnections

end

;+
; Purpose:
;   Retrieve the file specified by the filename parameter from a remote server
;
; Parameters:
;   filename  - a string specifying the full path of the remote file to be retrieved.
;
; Keywords:
;   localfilename - a string specifying the full path of a local filename which will
;                   contain a copy of the remote file
;
;   string -  if this keyword is set, the contents of filename is returned as a string array.
;
;   buffer - if this keyword is set, the contents of filename is returned as a byte array.
;
; Return:
;   The return value can be one of the following:-
;     - A scalar string specifing the local filename on the local computer
;       where the remote file was copied. This is the default behaviour.
;     - A string array containing the contents of the remote file if the STRING
;       keyword is set.
;     - A byte array containing the contents of the remote file if the BUFFER
;       keyword is set.
;-
;==============================================================================
function DaveHTTPRequest::GetFileContent, filename, localfilename=localfile, string=string, buffer=buffer
  compile_opt idl2

  Catch, catchError
  if (catchError ne 0) then begin
    Print,!error_state.msg
    Catch, /cancel
    Return, ''
  endif
  
  ; first, retrieve the content of the remote file specified by filename
  Self->Getproperty, baseFullPath=bfpath
  response = Httprequest.Get(bfpath+filename)
  byteBuffer = response.content   ; these are raw bytes, thus requiring further processing

  ; if localfile was not specified then define a temporary one (ehich will be deleted later, below)
  usingTmpfile = 0
  if (~keyword_set(localfile) || strlen(strtrim(localfile)) eq 0) then begin
    localfile = !home_dir+'ncnrdataTmpfile.dat'
    usingTmpfile = 1
  end

  ;; first write the remote raw bytes to a local file on disk
  ;; the localfile is either a requested file or a temporary file
  Openw,lun,localfile,/get_lun
  status = fstat(lun)
  if (status.open) then begin
    Writeu, lun, byteBuffer
    Free_lun,lun,/force
  endif
  
  ; and then read contents from localfile to buffer if the string keyword was set
  if (keyword_set(string)) then begin
    status = Dave_read_filecontents(localfile,dstring = stringBuffer,errmsg = errmsg)
    retValue = stringBuffer
  endif else if (keyword_set(buffer)) then begin
    retValue=byteBuffer
  endif else retValue=localfile

  if (usingTmpfile) then File_delete, localfile ; was only needed to write a byteBuffer and read it back into a stringBuffer
  
  return, retValue

end


;==============================================================================
function DaveHTTPRequest::GetListing, status=status,status_code=status_code
  compile_opt idl2
  ; need to execute a HTTPREQUEST::POST request on the server to retrieve contents of the specified directory.
  ; This is achieved by executing a custom php script (https://ncnr.nist.gov/ncnrdata/listftpfiles_pg.php) 
  ; on the server that goes to the directory, scans the content and returns that to the caller.
  ; 
  ; Keywords:
  ; 
  ;  json: use to pass json encoded data. In this case an argument called "pathlist" 
  ;       providing the desired subdirectory whose directory listing is to be retrieved.
  ;       Use hash() to generate the json format where: 
  ;       the key is "pathlist" and 
  ;       the value is a list (string array) of directories starting with 'ncnrdata'. 
  ;       So /ncnrdata/bt7/202102/ will be specified as ["ncnrdata","bt7","202102"]
  ;
  ;  header: set to a hash, dictionary or structure of name/value pairs containing http request headers. 
  ;         In this case, we use it to define the content-type to be returned as application/json

  key = "pathlist"

  ; currentDir is the full path specification in the server https://www.ncnr.nist.gov/
  ; and the top-level directory in the server is "/pub/ncnrdata"
  ; However "pathlist" considers /ncnrdata" as the top-level because this info is used within the phpfile 
  ; so we need to strip "/pub" from currentDir
  currentDir = Self.currentDir
  currentDir = currentDir.Replace("/pub","",/fold); remove the "/pub" portion of the path
  value = strsplit(currentDir,'/',/extract)    ; convert unix filepath into an array of subdirectories
  jsonData = Hash(key,value)    ; eg: '{"pathlist":["ncnrdata","bt7","202102"]}.
                                ; jsonData is passed parameter in the POST call below and the target is the phpfile
  header=Hash("Content-Type", "application/json")
  
  response = Httprequest.Post(Self.phpfile,json=jsonData,headers=header)

  ; check for errors and return an empty hash() if unsuccessful!
  status = response.ok
  status_code = response.status_code
  Self.response = response
  if (~status) then return, OrderedHash()
  
  ; if successful then 
  listings = response.Json()    ; returns an OrderedHash()
                                ; contains 3 entries
                                ; "subdirs", "files_metadata", "pathlist"

  return, listings
end


;==============================================================================
pro DaveHTTPRequest::GetProperty, server=server, baseDir=baseDir, currentDir=currentDir $
  , response=response, basefullpath=basefullpath, phpfile=phpfile $
  ,_REF_EXTRA=etc
  compile_opt idl2
  if (Arg_present(server)) then Self.ourl->Getproperty, URL_HOSTNAME =  server
  if (Arg_present(baseDir)) then baseDir = Self.basedir
  if (Arg_present(response)) then response = Self.response
  if (Arg_present(basefullpath)) then basefullpath = Self.basefullpath
  if (Arg_present(currentDir)) then currentDir = Self.currentdir
  if (Arg_present(phpfile)) then phpfile = Self.phpfile

end


;==============================================================================
pro DaveHTTPRequest::SetProperty, server=server, baseDir=baseDir, username=user, password=passwd $
  , currentDir=currentDir, callback_function=callback_function $
  , callback_data = callback_data $
  , response=response, basefullpath=basefullpath $
  ,_EXTRA=etc
  compile_opt idl2
  if (N_elements(server) gt 0) then begin
    Self.server = server
  endif
  if (N_elements(baseDir) gt 0) then begin
    if (Strmid(baseDir,0,1,/reverse) ne '/') then baseDir = baseDir + '/'
    Self.basedir = baseDir
  endif
  if (N_elements(currentDir) gt 0) then begin
    if (Strmid(currentDir,0,1,/reverse) ne '/') then currentDir = currentDir + '/'
    Self.currentdir = currentDir
  endif
  if (N_elements(basefullpath) gt 0) then  Self.basefullpath = basefullpath
  
  if (N_elements(etc) gt 0 ) then Self.ourl->Setproperty, _EXTRA=etc

end


;==============================================================================
;+
; Keywords:
;   server - set to scalar string specifying the URL of the server to connect to.
;            Default is 'https://ncnr.nist.gov/'.
;
;   baseDir - set to scalar string specifying the base directory on the server.
;             Default is '/pub/ncnrdata/'.
;
;   phpfile - custom php script (currently https://ncnr.nist.gov/ncnrdata/listftpfiles_pg.php) on the server 
;             that scans a specified directory and returns a directory listing found there. It basically 
;             acts as an ftp->GetFtpDirList() on a remote server. The returned information is an 
;             IDL dictionary object containing entries for any directories and files found in the 
;             specified directory on the server.
;             
;   baseFullPath - is base URL added to the relative directory on the server for a file to be 
;             retrieved or downloaded from the server. This is used in the GetFileContent() method
;
;-
function DaveHTTPRequest::Init, server=server, baseDir=baseDir, phpfile=phpfile, _REF_EXTRA=etc
  compile_opt idl2
  if (N_elements(server) eq 0) then server = 'https://ncnr.nist.gov/';'ftp.ncnr.nist.gov'
  if (N_elements(phpfile) eq 0) then phpfile = "https://ncnr.nist.gov/ncnrdata/listftpfiles_pg.php"
  if (N_elements(header) eq 0) then header=Hash("Content-Type", "application/json")
  if (N_elements(baseDir) eq 0) then baseDir = '/pub/ncnrdata/'
  if (N_elements(baseUrl) eq 0) then baseFullPath = 'https://ncnr.nist.gov' ;is base URL added to the relative directory 
                                                                            ;on the server for a file to be retrieved or 
                                                                            ;downloaded from the server. This is used in 
                                                                            ;the GetFileContent() method
  Self.basedir = baseDir
  Self.currentdir = baseDir
  Self.phpfile = phpfile
  self.server = server
  Self.baseFullPath = baseFullPath

  Return, 1
end

;==============================================================================
pro DaveHTTPRequest__define

  void = {DaveHTTPRequest $
    ,ourl:Obj_new() $
    ,response:obj_new() $
    ,phpfile:'' $
    ,server:'' $
    ,basedir:'' $
    ,baseFullPath:'' $
    ,currentdir:'' $
  }
end


;-------------------------------------------------
; Void Procedure
; Included here to force a call to json_serialize which is used internally by HttpRequest::Post()
; If this call is not made then the dave executable will not include json_serialize and hence will not function properly
; 
function void_json_serialize
myhash = HASH("Planet", "Jupiter", "Index", 5, "Mass", 1.9d27, "Units", "kg")
json = JSON_SERIALIZE(myhash)
return, json
end
