;+
; Purpose:
;   Wrapper for IDLnetURL class used to connect to a remote ftp server, send and receive
;   ftp requests such as directory listings and retrieve contents of remote files
;-
;


;==============================================================================
pro DAVEftpURL::Cleanup
compile_opt idl2

obj_destroy, Self.oUrl

end

;==============================================================================
pro DAVEftpURL::CloseConnections
compile_opt idl2

Self.oUrl->CloseConnections

end

;+
; Purpose:
;   Retrieve the file specified by the filename parameter from the server
; 
; Parameters:
;   filename  - a string specifying the full path of the remote file to be retrieved.
;   
; Keywords:
;   localfilename - a string specifying the full path of a local filename which will
;                   contain a copy of the remote file
;   
;   string -  if this keyword is set, the contents of filename is returned as a string array.
;   
;   buffer - if this keyword is set, the contents of filename is returned as a byte array.
;   
; Return:
;   The return value can be one of the following:-
;     - A scalar string specifing the local filename on the local computer 
;       where the remote file was copied. This is the default behaviour.
;     - A string array containing the contents of the remote file if the STRING 
;       keyword is set.
;     - A byte array containing the contents of the remote file if the BUFFER 
;       keyword is set.
;-
;==============================================================================
function DAVEftpURL::GetFileContent, filename, localfilename=localfile, string=string, buffer=buffer
compile_opt idl2

catch, catchError
if (catchError ne 0) then begin
    print,!error_state.msg
    catch, /cancel
    return, ''
endif

; set directory
Self.oUrl->SetProperty, URL_PATH = filename

; get contents of filename from server
return, Self.oUrl->Get(string=string, buffer=buffer,filename=localfile)

end

;==============================================================================
function DAVEftpURL::GetListing
compile_opt idl2

; set directory
Self.oUrl->SetProperty, URL_PATH = Self.currentDir

; get dirctory listing
return, Self.oUrl->GetFtpDirList()

end


;==============================================================================
pro DAVEftpURL::GetProperty, server=server, baseDir=baseDir, currentDir=currentDir $
                           ,_REF_EXTRA=etc
compile_opt idl2
if (arg_present(server)) then Self.oURL->GetProperty, URL_HOSTNAME =  server
if (arg_present(baseDir)) then baseDir = Self.baseDir
if (arg_present(currentDir)) then begin
  Self.oURL->GetProperty, URL_PATH = currentDir
  ; sometimes the URL_PATH bay be a server filename. In this case, strip
  ; the base file and return only the directory component
  n = strlen(currentDir)
  if (strmid(currentDir,n-1,1) ne '/') then begin
    ; Note: can't use file_dirname() for this b/c path_sep() is OS-dependent
    ; while the ftp server is unix. So '/' will be switch to '\' on windows which is 
    ; not a desired outcome. 
    toks = strsplit(currentDir,'/',/extract,count=ntoks)
    currentDir = '/'+strjoin(toks[0:ntoks-2],'/')+'/'
  endif
endif

if (n_elements(etc) gt 0 ) then Self.oURL->GetProperty, _EXTRA=etc 

end

;==============================================================================
pro DAVEftpURL::SetProperty, server=server, baseDir=baseDir, username=user, password=passwd $
                           , currentDir=currentDir, callback_function=callback_function $
                           , callback_data = callback_data $
                           ,_EXTRA=etc
compile_opt idl2
if (n_elements(server) gt 0) then begin
  Self.server = server
  Self.oURL->SetProperty, URL_HOSTNAME_ =  server
endif
if (n_elements(baseDir) gt 0) then begin
  if (strmid(baseDir,0,1,/reverse) ne '/') then baseDir = baseDir + '/'
  Self.baseDir = baseDir
  Self.oURL->SetProperty, URL_PATH = baseDir
endif
if (n_elements(currentDir) gt 0) then begin
  if (strmid(currentDir,0,1,/reverse) ne '/') then currentDir = currentDir + '/'
  Self.currentDir = currentDir
  Self.oURL->SetProperty, URL_PATH = currentDir
endif
if (n_elements(user) gt 0) then Self.oURL->SetProperty, URL_USERNAME=user 
if (n_elements(passwd) gt 0) then Self.oURL->SetProperty, URL_PASSWORD=passwd
if (n_elements(callback_function) gt 0) then Self.oURL->SetProperty, callback_function =  callback_function
if (n_elements(callback_data) gt 0) then Self.oURL->SetProperty, callback_data =  callback_data

if (n_elements(etc) gt 0 ) then Self.oURL->SetProperty, _EXTRA=etc 

end


;==============================================================================
;+
; Keywords:
;   server - set to scalar string specifying the URL of the server to connect to.
;            Default is 'ftp.ncnr.nist.gov'.
;   
;   baseDir - set to scalar string specifying the base directory in the server.
;             Default is '/pub/ncnrdata/'.
;   
;   username - set to scalar string specifying the username used to logon to the 
;              server. Default is ''.
;
;   password - set to scalar string specifying the password used to logon to the 
;              server. Default is ''.
;-
function DAVEftpURL::Init, server=server, baseDir=baseDir, username=user, password=passwd, _REF_EXTRA=etc
compile_opt idl2

if (n_elements(server) eq 0) then server = '44.207.113.19';'ftp.ncnr.nist.gov'
if (n_elements(baseDir) eq 0) then baseDir = '/pub/ncnrdata/'
if (n_elements(user) eq 0) then user = ''
if (n_elements(passwd) eq 0) then passwd = ''


Self.oURL = IDLnetURL(URL_SCHEME = 'ftp', URL_HOSTNAME=server, URL_PATH=baseDir $
                     ,URL_USERNAME=user, URL_PASSWORD=passwd, FTP_CONNECTION_MODE=0)

if (~obj_valid(Self.oURL)) then return, 0

if (n_elements(etc) gt 0 ) then Self.oURL->SetProperty, _EXTRA=etc 

Self.baseDir = baseDir
Self.currentdir = baseDir

return, 1
end

;==============================================================================
pro DAVEftpURL__define

void = {DAVEftpURL $
        ,oURL:obj_new() $
        ,server:'' $
        ,baseDir:'' $
        ,currentDir:'' $
       }
end