; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   GenericDataset
;
; PURPOSE:
;   Data Class for representing of DAVE 1.x formated datasets
;
; CATEGORY:
;
; SUPERCLASSES:
;   IDLitParameterSet
;
; SUBCLASSES:
;
; CREATION:
;   See GenericDataset::Init
;
; METHODS:
;   Clone (interface only - not implemented)
;   Transpose
;   GetProperty
;   SetProperty
;   ToDavePtr
;   ToASCIISPE
;   ToASCIIGrp
;   ToASCIIColumn
;   _AddComponents (private)
;   _GetDescriPtrStr (private)
;   _GetSpecificStr (private)
;   _MakeSpecificObj (private)
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; December 2008
;-
;###############################################################################


;===============================================================================


;===============================================================================


;===============================================================================
;+
; GenericDataset::toASCIIColumn
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Column data.
;
; PARAMETERS
;    data - a nxm array of the data where n is the nos of columns and and m the nos of rows.
;           Where for a n=3 (ie 3 column dataset) (if no errors, then the error column would be absent)
;              column 1 is the independent data
;              column 2 is the dependent data
;              column 3 is the uncertainty in the dependent data
;              
;           If n gt 3 then it is assummed that there are multiple data groups all with the same
;           independent data as follows (if no errors, then the error columns would be absent):
;              column 1 is the independent data
;              column 2 is the dependent data for grp 1
;              column 3 is the uncentainty in the dependent data for group 1
;              column 4 is the dependent data for grp 2
;              column 5 is the uncentainty in the dependent data for group 2
;              ...
;    
;    header - is a string vector of size n (ie n columns) containing textual annotation (label) for the data block, 
;             including info about group values
;    
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function GenericDataset::toASCIIColumn, data, header, errorMsg
compile_opt idl2

; need at least the first parameter to proceed
if (n_params() lt 1) then return, 0

Self->GetProperty, axis1Value=xData, axis2Value=yData, dataValue=qty, errorValue=err $
                 ,axis1Distribution=xType, axis2Distribution=yType

; all relevant data must be present
xExist = n_elements(xData) gt 0
yExist = n_elements(yData) gt 0
qtyExist = n_elements(qty) gt 0
errPresent = n_elements(err) gt 0
if (~xExist || ~qtyExist) then begin
   errorMsg = 'Inconplete dataset - the dependent or independent data was not found in dataset'
   return, 0
endif

qtySize= size(qty)
ndims = qtySize[0]
d1 = qtySize[1]
d2 = qtySize[2]

nx = n_elements(xData)
if (nx ne d1) then begin
   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
   if (isHistogram && (nx eq d1+1)) then begin
      ;; convert to point mode
      index = lindgen(d1)
      xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
      nx = n_elements(xData)
   endif else begin
      errormsg = 'Size of dependent and independent data do not match!'
      ;self->SignalError, msg, severity=2
      return, 0
   endelse
endif

if (ndims eq 2) then begin
   ny = n_elements(yData)
   if (ny ne d2) then begin
      isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
      if (isHistogram && (ny eq d2+1)) then begin
         ;; convert to point mode
         index = lindgen(d2)
         yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
         ny = n_elements(yData)
      endif else begin
         errormsg = 'Size of dependent and independent data do not match!'
         ;self->SignalError, msg, severity=2
         return, 0
      endelse
   endif
endif

if (ndims eq 1) then begin
   dataColumns = [transpose(xData),transpose(qty)]
   ncols = 2
   if (errPresent) then begin
      dataColumns = [dataColumns,transpose(err)]
      ncols = 3
   endif
   header = ['X       ','Intensity','Error    ']
endif else begin
   dataColumns = transpose(xData)
   ncols = 1
   header = 'X       '
   if (errPresent eq 0) then begin
      for j=0,d2-1 do begin
         buf = 'I'+strtrim(string(j+1),2)+' (Y='+strtrim(string(yData[j],format='(G10.4)'),2)+')'
         header = [header,buf]
         dataColumns = [dataColumns,transpose(qty[*,j])]
         ncols++
      endfor
   endif else begin
      for j=0,d2-1 do begin
         buf1 = 'I'+strtrim(string(j+1),2)+' (Y='+strtrim(string(yData[j],format='(G10.4)'),2)+')'
         buf2 = 'Error'+strtrim(string(j+1),2)+'    '
         header = [header,buf1,buf2]
         dataColumns = [dataColumns,transpose(qty[*,j]),transpose(err[*,j])]
         ncols += 2 
      endfor                 
   endelse
endelse

data = temporary(dataColumns)

return, 1
end



;===============================================================================
;+
; GenericDataset::toASCIIGrp
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Grp data format variables.
;
; PARAMETERS
;   dataStr - a data structure containing a SPE dataset This is an anonymous data structure
;             containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function GenericDataset::toASCIIGrp, dataStr, errorMsg
compile_opt idl2

; require the single output parameter to proceed
if (n_params() lt 1) then return, 0

Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
                 ,axis1Distribution=xType, axis2Distribution=yType

; all relevant data must be present
xExist = n_elements(xData) gt 0
yExist = n_elements(yData) gt 0
qtyExist = n_elements(qty) gt 0
errPresent = n_elements(err) gt 0
if (~xExist || ~yExist || ~qtyExist) then begin
   errorMsg = 'Inconplete dataset - the dependent or independent data was not found in dataset'
   return, 0
endif

; must be a 2D dataset
ndim = size(qty,/n_dimensions)
if (ndim ne 2) then begin
   errorMsg = 'The dataset has the wrong dimension - expected 2D!'
   return, 0
endif

nx = n_elements(xData)
ny = n_elements(yData)
nqty1 = (size(qty))[1]
nqty2 = (size(qty))[2]
if (nx ne nqty1) then begin
   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
   if (isHistogram && (nx eq nqty1+1)) then begin
      ;; convert to point mode. ASCII grp format only stores point
      ;; mode data
     index = lindgen(nqty1)
     xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
     nx = n_elements(xData)
   endif else begin
      errormsg = 'Size of dependent and first independent data do not match!'
      ;self->SignalError, msg, severity=2
      return, 0
   endelse
endif
if (ny ne nqty2) then begin
   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
   if (isHistogram && (ny eq nqty2+1)) then begin
      ;; convert to point mode
     index = lindgen(nqty2)
     yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
     ny = n_elements(yData)
   endif else begin
      errormsg = 'Size of dependent and second independent data do not match!'
      ;self->SignalError, msg, severity=2
      return, 0
   endelse
endif

; Generate errors if not present
if (~errPresent) then begin
   index = where(qty le 0, cnt)
   err = qty
   if (cnt gt 0) then err[index] = 1.0
   err = temporary(sqrt(err))
endif

dataStr = {xData: temporary(xData), $
              yData: temporary(yData), $
              data: temporary(qty), $
              error: temporary(err), $
              xTitle: xLabel, $
              yTitle: yLabel, $
              dataTitle: qtyLabel $
             }


return, 1

end


;===============================================================================
;+
; GenericDataset::Clone
; 
; PURPOSE:
;   return an object containing an axact copy of the data in the current object
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - a copy of the current object
;    failure - a null object
;-
function GenericDataset::Clone
compile_opt idl2

;; To be implemented by derived classes

return, obj_new()

end


;===============================================================================
;+
; GenericDataset::Transpose
; 
; PURPOSE:
;   If object contains a 2D dataset, then swap the dimensions of the dependent data
;   and error. Also swap the data in the first and second independent axes.
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function GenericDataset::Transpose
compile_opt idl2

; Only applicable to 2D datasets
if (Self.ndims ne 2) then return, 0

; transpose the dependent data
status = (Self.dataRef)->GetData(data)
if (~status) then return, 0
status = (Self.dataRef)->SetData(transpose(temporary(data)),/no_copy,/no_notify)
if (~status) then return, 0

; Transpose the error data
if (obj_valid(Self.errorRef))then begin
   status = (Self.errorRef)->GetData(data)
   if (~status) then return, 0
   status = (Self.errorRef)->SetData(transpose(temporary(data)),/no_copy,/no_notify)
endif

; Swap the contents of the first and second independent axes
Self->GetProperty, axis1Value=v1, axis1Name=n1, axis1Label=l1, axis1Units=u1, axis1Distribution=d1
Self->GetProperty, axis2Value=v2, axis2Name=n2, axis2Label=l2, axis2Units=u2, axis2Distribution=d2
Self->SetProperty,axis2Value=v1,axis2Name=n1,axis2Label=l1,axis2Units=u1,axis2Distribution=d1,/no_copy,/no_notify
Self->SetProperty,axis1Value=v2,axis1Name=n2,axis1Label=l2,axis1Units=u2,axis1Distribution=d2,/no_copy,/no_notify

; Send notification messagees to observers since the /no_notify was used earlier
; Though several data objects were modified a single notification message will surfice
(Self.dataRef)->NotifyDataChange
(Self.axis1Ref)->NotifyDataChange
(Self.axis2Ref)->NotifyDataChange
(Self.dataRef)->NotifyDataComplete
(Self.axis1Ref)->NotifyDataComplete
(Self.axis2Ref)->NotifyDataComplete

return, 1
end


;===============================================================================
;+
; GenericDataset::toASCIISPE
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII SPE data format variables.
;
; PARAMETERS
;   dataStr - a data structure containing a SPE dataset This is an anonymous data structure
;             containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function GenericDataset::toASCIISPE, dataStr, errorMsg
compile_opt idl2

; require the single output parameter to proceed
if (n_params() lt 1) then return, 0

Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
                 ,axis1Distribution=xType, axis2Distribution=yType

; all relevant data must be present
xExist = n_elements(xData) gt 0
yExist = n_elements(yData) gt 0
qtyExist = n_elements(qty) gt 0
errPresent = n_elements(err) gt 0
if (~xExist || ~yExist || ~qtyExist) then begin
   errorMsg = 'Missing independent or dependent data so cannot procceed'
   return, 0
endif

; must be a 2D dataset
ndim = size(qty,/n_dimensions)
if (ndim ne 2) then return, 0

nx = n_elements(xData)
ny = n_elements(yData)
nqty1 = (size(qty))[1]
nqty2 = (size(qty))[2]
if (nx ne nqty1) then begin
   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
   if (isHistogram && (nx ne nqty1+1)) then begin
      errormsg = 'Size of dependent and first independent data do not match!'
      ;self->SignalError, msg, severity=2
      return, 0
   endif
endif

if (ny ne nqty2) then begin
   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
   if (isHistogram && (ny ne nqty2+1)) then begin
      errormsg = 'Size of dependent and second independent data do not match!'
      ;self->SignalError, msg, severity=2
      return, 0
   endif
endif

; Generate errors if not present
if (~errPresent) then begin
   index = where(qty le 0, cnt)
   err = qty
   if (cnt gt 0) then err[index] = 1.0
   err = temporary(sqrt(err))
endif

dataStr = {xData: temporary(xData), $
              yData: temporary(yData), $
              data: temporary(qty), $
              error: temporary(err), $
              xTitle: xLabel, $
              yTitle: yLabel, $
              dataTitle: qtyLabel $
             }

return, 1

end



;===============================================================================
;+
; GenericDataset::toDavePtr
; 
; PURPOSE:
;   function used to convert an object of this class into davePtr data struture.
;
; PARAMETERS
;    davePTr - pointer to the DAVE data format structure
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function GenericDataset::toDavePtr, davePtr, errorMsg
compile_opt idl2

if (n_params() lt 1) then return, 0

Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
                 ,axis1Units=xUnits, axis2Units=yUnits, dataUnits=qtyUnits $
                 ,axis1Distribution=xType, axis2Distribution=yType


retVal = create_dave_pointer(davePtr $
                           ,instrument = inst   $
                           ,qty = qty $
                           ,qtunits = qtyUnits $
                           ,qtlabel = qtyLabel $
                           ,err = err $
                           ,xvals = xdata $
                           ,xtype = xtype $
                           ,xunits = xunits $
                           ,xlabel = xlabel $
                           ,yvals = ydata $
                           ,ytype = ytype $
                           ,yunits = yunits $
                           ,ylabel = ylabel $
                           ,specificstr = specificstr $
                           ,treatment = trmt $
                           ,dname = dptrName $
                           ,dunits = dptrUnits $
                           ,dlegend = dptrDesc $
                           ,dqty = dptrQty $
                           ,derr = dptrErr $
                           ,ermsg = errormsg $
                          )

if (retVal eq 0) then begin
    ;; cleanup before returning
   if (n_elements(specificStr) gt 0) then heap_free, specificStr
   if (ptr_valid(davePtr)) then heap_free, davePtr
endif

return, retVal

end


;===============================================================================
;+
; GenericDataset::SetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;
;   instrument - the instrument from which this dataset was obtained from (string)
;   
;   treatment - the treatment history for this dataset (string array)
;   
;   descriPtr - a pointer to the descriPtr component of the dataset (consult davePtr description)
;   
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis, if a 2D dataset - a vector
;   
;   dataValue - the actual values for the dependent data - either 1D or 2D array
;   
;   errorValue - the actual errors in the dependent data - either 1D or 2D array
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;-
pro GenericDataset::SetProperty, instrument=instrument, treatment=treatment, descriPtr=descriPtr $
   ,axis1Value=axis1Value,axis2Value=axis2Value,dataValue=dataValue,errorValue=errorValue $
   ,axis1Name=axis1Name, axis2Name=axis2Name, dataName=dataName $
   ,axis1Label=axis1Label, axis2Label=axis2Label, dataLabel=dataLabel $
   ,axis1Units=axis1Units, axis2Unitsl=axis2Units, dataUnits=dataUnits $
   ,axis1Distribution=axis1Distribution, axis2Distribution=axis2Distribution $
   ,filename=filename $
   ,no_notify=no_notify, no_copy=no_copy $
   ,_EXTRA=etc
compile_opt idl2

;
if (n_elements(instrument) gt 0  && obj_valid(Self.instrumentRef)) then begin
   void = (Self.instrumentRef)->SetData(instrument)
endif 
;
if (n_elements(treatment) gt 0  && obj_valid(Self.treatmentRef)) then begin
   void = (Self.treatmentRef)->SetData(treatment)
endif 
;
if (n_elements(filename) gt 0) then Self.filename = filename
;
if (n_elements(axis1Value) gt 0 && obj_valid(Self.axis1Ref)) then begin
   void = (Self.axis1Ref)->SetData(axis1Value, no_copy=keyword_set(no_copy), no_notify=keyword_set(no_notify))
endif 
;
if (n_elements(axis2Value) gt 0 && (Self.ndims eq 2) && obj_valid(Self.axis2Ref)) then begin
   void = (Self.axis2Ref)->SetData(axis2Value, no_copy=keyword_set(no_copy), no_notify=keyword_set(no_notify))
endif 
;
if (n_elements(dataValue) gt 0 && obj_valid(Self.dataRef)) then begin
   void = (Self.dataRef)->SetData(dataValue, no_copy=keyword_set(no_copy), no_notify=keyword_set(no_notify))
endif 
;
if (n_elements(errorValue) gt 0  && obj_valid(Self.errorRef)) then begin
   void = (Self.errorRef)->SetData(errorValue, no_copy=keyword_set(no_copy), no_notify=keyword_set(no_notify))
endif 
;
if (obj_valid(Self.axis1Ref)) then begin
   if (n_elements(axis1Label) gt 0) then (Self.axis1Ref)->AddMetaData, 'Long_name', axis1Label
   if (n_elements(axis1Units) gt 0) then (Self.axis1Ref)->AddMetaData, 'Units', axis1Units
   if (n_elements(axis1Distribution) gt 0) then (Self.axis1Ref)->AddMetaData, 'Distribution', axis1Distribution
   if (n_elements(axis1Name) gt 0) then (Self.axis1Ref)->SetProperty, name=axis1Name
endif
;
if (obj_valid(Self.axis2Ref)) then begin
   if (n_elements(axis2Label) gt 0) then (Self.axis2Ref)->AddMetaData, 'Long_name', axis2Label
   if (n_elements(axis2Units) gt 0) then (Self.axis2Ref)->AddMetaData, 'Units', axis2Units
   if (n_elements(axis2Distribution) gt 0) then (Self.axis2Ref)->AddMetaData, 'Distribution', axis2Distribution
   if (n_elements(axis2Name) gt 0) then (Self.axis2Ref)->SetProperty, name=axis2Name
endif
;
if (obj_valid(Self.dataRef)) then begin
   if (n_elements(dataLabel) gt 0) then (Self.dataRef)->AddMetaData, 'Long_name', dataLabel
   if (n_elements(dataUnits) gt 0) then (Self.dataRef)->AddMetaData, 'Units', dataUnits
   if (n_elements(dataName) gt 0) then (Self.dataRef)->SetProperty, name=dataName
endif


; call base class mutator
if (n_elements(etc) gt 0) then Self->IDLitParameterSet::SetProperty, _EXTRA=etc

end


;===============================================================================
;+
; GenericDataset::GetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;   axis1Ref - the object reference to the first dependent axis of the dataset
;   
;   axis2Ref - the object reference to the second dependent axis of the dataset, if it is 2D
;   
;   dataRef - the object reference to the dependent data
;   
;   errorRef - the object reference to the error in the dependent data
;   
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis, if a 2D dataset - a vector
;   
;   dataValue - the actual values for the dependent data - either 1D or 2D array
;   
;   errorValue - the actual errors in the dependent data - either 1D or 2D array
;   
;   nDimensions - 1 | 2 if a 1D | 2D dataset
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;-
pro GenericDataset::GetProperty, axis1Ref=axis1Ref,axis2Ref=axis2Ref,dataRef=dataRef,errorRef=errorRef, plotDataRef=plotDataRef $
   ,axis1Value=axis1Value,axis2Value=axis2Value,dataValue=dataValue,errorValue=errorValue, nDimensions=nDimensions $
   ,axis1Name=axis1Name, axis2Name=axis2Name, dataName=dataName $
   ,axis1Label=axis1Label, axis2Label=axis2Label, dataLabel=dataLabel $
   ,axis1Units=axis1Units, axis2Units=axis2Units, dataUnits=dataUnits $
   ,axis1Distribution=axis1Distribution, axis2Distribution=axis2Distribution $
   ,axis1Dim=axis1Dim,axis2Dim=axis2Dim,dataDim=dataDim,errorDim=errorDim $
   ,instrument=instrument, treatment=treatment, trmtRef=trmtRef, filename=filename $
   ,usedCount= usedCnt $
   ,_REF_EXTRA=etc
compile_opt idl2


;
if (arg_present(instrument)) then begin
   if (obj_valid(Self.instrumentRef)) then $
       void = (Self.instrumentRef)->GetData(instrument) else $
       instrument = 'UNKNOWN'
endif
;
if (arg_present(treatment)) then begin
   void = (Self.treatmentRef)->GetData(treatment)
endif
;
if (arg_present(axis1Name) || arg_present(axis1Label) || arg_present(axis1Units) || arg_present(axis1Distribution)) then begin
   if (obj_valid(Self.axis1Ref)) then begin
      if ((void = (Self.axis1Ref)->GetMetaData('Long_name',axis1Label)) eq 0 ) then axis1Label=''
      if ((void = (Self.axis1Ref)->GetMetaData('Units',axis1Units)) eq 0 ) then axis1Units=''
      if ((void = (Self.axis1Ref)->GetMetaData('Distribution',axis1Distribution)) eq 0 ) then axis1Distribution='POINTS'
      (Self.axis1Ref)->GetProperty, name=axis1Name
   endif else begin
      axis1Label = ''
      axis1Units = ''
      axis1Distribution = 'POINTS'
      axis1Name = ''
   endelse
endif
;
if (arg_present(axis2Name) || arg_present(axis2Label) || arg_present(axis2Units) || arg_present(axis2Distribution)) then begin
   if (obj_valid(Self.axis2Ref)) then begin
      if ((void = (Self.axis2Ref)->GetMetaData('Long_name',axis2Label)) eq 0 ) then axis2Label=''
      if ((void = (Self.axis2Ref)->GetMetaData('Units',axis2Units)) eq 0 ) then axis2Units=''
      if ((void = (Self.axis2Ref)->GetMetaData('Distribution',axis2Distribution)) eq 0 ) then axis2Distribution='POINTS'
      (Self.axis2Ref)->GetProperty, name=axis2Name
   endif else begin
      axis2Label = ''
      axis2Units = ''
      axis2Distribution = 'POINTS'
      axis2Name = ''
   endelse
endif
;
if (arg_present(dataName) || arg_present(dataLabel) || arg_present(dataUnits)) then begin
   if (obj_valid(Self.dataRef)) then begin
      if ((void = (Self.dataRef)->GetMetaData('Long_name',dataLabel)) eq 0 ) then dataLabel=''
      if ((void = (Self.dataRef)->GetMetaData('Units',dataUnits)) eq 0 ) then dataUnits=''
      (Self.dataRef)->GetProperty, name=dataName
   endif else begin
      dataLabel = ''
      dataUnits = ''
      dataName = ''
   endelse
endif
;
if (arg_present(axis1Value) || arg_present(axis1Dim)) then begin
   void = (Self.axis1Ref)->GetData(axis1Value)
   axis1Dim = n_elements(axis1Value)
endif
;
if ((arg_present(axis2Value) || arg_present(axis2Dim)) && (Self.ndims eq 2)) then begin
   void = (Self.axis2Ref)->GetData(axis2Value)
   axis2Dim = n_elements(axis2Value)
endif
;
if (arg_present(dataValue) || arg_present(dataDim)) then begin
   void = (Self.dataRef)->GetData(dataValue)
   dataDim = size(dataValue,/dimensions)
endif
;
if (arg_present(errorValue) || arg_present(errorDim)) then begin
   void = (Self.errorRef)->GetData(errorValue)
   errorDim = size(errorValue,/dimensions)
endif
;
axis1Ref = Self.axis1Ref
;
axis2Ref = Self.axis2Ref
;
dataRef = Self.dataRef
;
errorRef = Self.errorRef
;
plotDataRef = Self.plotDataRef
;
trmtRef = Self.treatmentRef
;
nDimensions = Self.ndims
;
filename = Self.filename

if (arg_present(usedCnt)) then begin
   ;; determine if the dependent or independent data are being used in a visualization
   dCnt = 0
   if (obj_valid(dataRef)) then $
      void = dataRef->GetDataObservers(/ALL,isa='IDLitVisualization',count=dCnt)
   xCnt = 0
   if (obj_valid(axis1Ref)) then $
      void = axis1Ref->GetDataObservers(/ALL,isa='IDLitVisualization',count=xCnt)
   yCnt = 0
   if (obj_valid(axis2Ref)) then $
      void = axis2Ref->GetDataObservers(/ALL,isa='IDLitVisualization',count=yCnt)
   usedCnt = dCnt > xCnt > yCnt   ; return the largest of the three
endif

; call base class accessor
if (n_elements(etc) gt 0) then Self->IDLitParameterSet::GetProperty, _EXTRA=etc

end




;===============================================================================
;+
; GenericDataset::Init
; 
; PURPOSE:
;   Initialization method for objects of GenericDataset class.
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;-
function GenericDataset::Init, _REF_EXTRA=etc
compile_opt idl2

; Basic error handler
catch, theError
if (theError ne 0) then begin
   catch, /cancel
   return, 0
endif


; Call the base class init method to instantiate container
return, self->IDLitParameterSet::Init(_EXTRA=etc)

end


;===============================================================================
pro GenericDataset__Define
compile_opt idl2

void = {GenericDataset $
        ,inherits IDLitParameterSet $
        ,axis1Ref:obj_new() $
        ,axis2Ref:obj_new() $
        ,dataRef:obj_new() $
        ,errorRef:obj_new() $
        ,plotDataRef:obj_new() $
        ,instrumentRef:obj_new() $
        ,treatmentRef:obj_new() $
        ,filename:'' $
        ,ndims:0 $
        }

end
