; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEDataset
;
; PURPOSE:
;   Data Class for representing of DAVE 1.x formated datasets
;
; CATEGORY:
;
; SUPERCLASSES:
;   GenericDataset
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEDataset::Init
;
; METHODS:
;   GetProperty
;   SetProperty
;   ToDavePtr
;   ToASCIISPE
;   ToASCIIGrp
;   ToASCIIColumn
;   _AddComponents (private)
;   _GetDescriPtrStr (private)
;   _GetSpecificStr (private)
;   _MakeSpecificObj (private)
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; December 2008
;-
;###############################################################################


;===============================================================================


;===============================================================================


;===============================================================================
;+
; DAVEDataset::toASCIIColumn
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Column data.
;
; PARAMETERS
;    data - a nxm array of the data where n is the nos of columns and and m the nos of rows.
;           Where for a n=3 (ie 3 column dataset) (if no errors, then the error column would be absent)
;              column 1 is the independent data
;              column 2 is the dependent data
;              column 3 is the uncertainty in the dependent data
;              
;           If n gt 3 then it is assummed that there are multiple data groups all with the same
;           independent data as follows (if no errors, then the error columns would be absent):
;              column 1 is the independent data
;              column 2 is the dependent data for grp 1
;              column 3 is the uncentainty in the dependent data for group 1
;              column 4 is the dependent data for grp 2
;              column 5 is the uncentainty in the dependent data for group 2
;              ...
;    
;    header - is a string vector of size n (ie n columns) containing textual annotation (label) for the data block, 
;             including info about group values
;    
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
;function DAVEDataset::toASCIIColumn, data, header
;compile_opt idl2
;
;; need at least the first parameter to proceed
;if (n_params() lt 1) then return, 0
;
;Self->GetProperty, axis1Value=xData, axis2Value=yData, dataValue=qty, errorValue=err $
;                 ,axis1Distribution=xType, axis2Distribution=yType
;
;; all relevant data must be present
;xExist = n_elements(xData) gt 0
;yExist = n_elements(yData) gt 0
;qtyExist = n_elements(qty) gt 0
;errPresent = n_elements(err) gt 0
;if (~xExist || ~qtyExist) then return, 0
;
;qtySize= size(qty)
;ndims = qtySize[0]
;d1 = qtySize[1]
;d2 = qtySize[2]
;
;nx = n_elements(xData)
;if (nx ne d1) then begin
;   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (nx eq d1+1)) then begin
;      ;; convert to point mode
;      index = lindgen(d1)
;      xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
;      nx = n_elements(xData)
;   endif else begin
;      ;msg = 'Size of dependent and independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endelse
;endif
;
;if (ndims eq 2) then begin
;   ny = n_elements(yData)
;   if (ny ne d2) then begin
;      isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
;      if (isHistogram && (ny eq d2+1)) then begin
;         ;; convert to point mode
;         index = lindgen(d2)
;         yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
;         ny = n_elements(yData)
;      endif else begin
;         ;msg = 'Size of dependent and independent data do not match!'
;         ;self->SignalError, msg, severity=2
;         return, 0
;      endelse
;   endif
;endif
;
;if (ndims eq 1) then begin
;   dataColumns = [transpose(xData),transpose(qty)]
;   ncols = 2
;   if (errPresent) then begin
;      dataColumns = [dataColumns,transpose(err)]
;      ncols = 3
;   endif
;   header = ['X       ','Intensity','Error    ']
;endif else begin
;   dataColumns = transpose(xData)
;   ncols = 1
;   header = 'X       '
;   if (errPresent eq 0) then begin
;      for j=0,d2-1 do begin
;         buf = 'I'+strtrim(string(j+1),2)+' (Y='+strtrim(string(yData[j],format='(G10.4)'),2)+')'
;         header = [header,buf]
;         dataColumns = [dataColumns,transpose(qty[*,j])]
;         ncols++
;      endfor
;   endif else begin
;      for j=0,d2-1 do begin
;         buf1 = 'I'+strtrim(string(j+1),2)+' (Y='+strtrim(string(yData[j],format='(G10.4)'),2)+')'
;         buf2 = 'Error'+strtrim(string(j+1),2)+'    '
;         header = [header,buf1,buf2]
;         dataColumns = [dataColumns,transpose(qty[*,j]),transpose(err[*,j])]
;         ncols += 2 
;      endfor                 
;   endelse
;endelse
;
;data = temporary(dataColumns)
;
;return, 1
;end
;


;===============================================================================
; DAVEDataset::toASCIIGrp
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Grp data format variables.
;
; PARAMETERS
;    xValue - the first independent variable
;    
;    yValue - the second independent variable
;    
;    dataValue - the dependent data
;    
;    errorValue - the uncertainty in the dependent data
;    
;    xTitle - the first independent axis label/title
;    
;    yTitle - the second indenpendent axis label/title
;    
;    dataTitle - the dependent label/title
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
;function DAVEDataset::toASCIIGrp, xValue,yValue,dataValue,errorValue,xTitle,yTitle,dataTitle
;compile_opt idl2
;
;; need at least the first 3 parameters to proceed
;if (n_params() lt 3) then return, 0
;
;Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
;                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
;                 ,axis1Distribution=xType, axis2Distribution=yType
;
;; all relevant data must be present
;xExist = n_elements(xData) gt 0
;yExist = n_elements(yData) gt 0
;qtyExist = n_elements(qty) gt 0
;errPresent = n_elements(err) gt 0
;if (~xExist || ~yExist || ~qtyExist) then return, 0
;
;; must be a 2D dataset
;ndim = size(qty,/n_dimensions)
;if (ndim ne 2) then return, 0
;
;nx = n_elements(xData)
;ny = n_elements(yData)
;nqty1 = (size(qty))[1]
;nqty2 = (size(qty))[2]
;if (nx ne nqty1) then begin
;   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (nx eq nqty1+1)) then begin
;      ;; convert to point mode. ASCII grp format only stores point
;      ;; mode data
;     index = lindgen(nqty1)
;     xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
;     nx = n_elements(xData)
;   endif else begin
;      ;msg = 'Size of dependent and first independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endelse
;endif
;if (ny ne nqty2) then begin
;   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (ny eq nqty2+1)) then begin
;      ;; convert to point mode
;     index = lindgen(nqty2)
;     yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
;     ny = n_elements(yData)
;   endif else begin
;      ;msg = 'Size of dependent and second independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endelse
;endif
;
;; Generate errors if not present
;if (~errPresent) then begin
;   index = where(qty le 0, cnt)
;   err = qty
;   if (cnt gt 0) then err[index] = 1.0
;   err = temporary(sqrt(err))
;endif
;
;xValue = temporary(xData)
;yValue = temporary(yData)
;dataValue = temporary(qty)
;errorValue = temporary(err)
;xTitle = xLabel
;yTitle = yLabel
;dataTitle = qtyLabel
;
;return, 1
;
;end



;===============================================================================
; DAVEDataset::toASCIISPE
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII SPE data format variables.
;
; PARAMETERS
;    xValue - the first independent variable
;    
;    yValue - the second independent variable
;    
;    dataValue - the dependent data
;    
;    errorValue - the uncertainty in the dependent data
;    
;    xTitle - the first independent axis label/title
;    
;    yTitle - the second indenpendent axis label/title
;    
;    dataTitle - the dependent label/title
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
;function DAVEDataset::toASCIISPE, xValue,yValue,dataValue,errorValue,xTitle,yTitle,dataTitle
;compile_opt idl2
;
;; need at least the first 3 parameters to proceed
;if (n_params() lt 3) then return, 0
;
;retVal = 1
;
;Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
;                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
;                 ,axis1Distribution=xType, axis2Distribution=yType
;
;; all relevant data must be present
;xExist = n_elements(xData) gt 0
;yExist = n_elements(yData) gt 0
;qtyExist = n_elements(qty) gt 0
;errPresent = n_elements(err) gt 0
;if (~xExist || ~yExist || ~qtyExist) then return, 0
;
;; must be a 2D dataset
;ndim = size(qty,/n_dimensions)
;if (ndim ne 2) then return, 0
;
;nx = n_elements(xData)
;ny = n_elements(yData)
;nqty1 = (size(qty))[1]
;nqty2 = (size(qty))[2]
;if (nx ne nqty1) then begin
;   isHistogram = strcmp(xType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (nx ne nqty1+1)) then begin
;      ;msg = 'Size of dependent and first independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endif
;endif
;
;;; If necessary, add a null value to xData to make it a 'false histogram'
;;; with one extra point relative to the dependent data.
;xIsPoint = strcmp(xType,'POINTS',4,/FOLD_CASE)
;if (xIsPoint) then xData = [xData,0.0]
;
;;; If necessary, add a null value to yData to make it a 'false histogram'
;;; with one extra point relative to the dependent data.
;yIsPoint = strcmp(yType,'POINTS',4,/FOLD_CASE)
;if (yIsPoint) then yData = [yData,0.0]
;
;if (ny ne nqty2) then begin
;   isHistogram = strcmp(yType,'HISTOGRAM',4,/FOLD_CASE)
;   if (isHistogram && (ny ne nqty2+1)) then begin
;      ;msg = 'Size of dependent and second independent data do not match!'
;      ;self->SignalError, msg, severity=2
;      return, 0
;   endif
;endif
;
;; Generate errors if not present
;if (~errPresent) then begin
;   index = where(qty le 0, cnt)
;   err = qty
;   if (cnt gt 0) then err[index] = 1.0
;   err = temporary(sqrt(err))
;endif
;
;xValue = temporary(xData)
;yValue = temporary(yData)
;dataValue = temporary(qty)
;errorValue = temporary(err)
;xTitle = xLabel
;yTitle = yLabel
;dataTitle = qtyLabel
;
;return, retVal
;end



;===============================================================================
;+
; DAVEDataset::Transpose
; 
; PURPOSE:
;   If object contains a 2D dataset, then swap the dimensions of the dependent data
;   and error. Also swap the data in the first and second independent axes.
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function DAVEDataset::Transpose
compile_opt idl2

; simply call the base class to do most of the work
if (~Self->GenericDataset::Transpose()) then return, 0

; Now handle mask property
Self->updateMask

return, 1
end


;===============================================================================
;+
; DAVEDataset::UpdateMask
; 
; PURPOSE:
;   Ensure the bit mask is updated and consistent with the data.
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;-
pro DAVEDataset::UpdateMask
compile_opt idl2

if (~obj_valid(Self.specificPtrRef)) then return

; From the specificPtr container, retrieve all children
; and locate the 'PAN_MASK' object.
oSpec = Self.specificPtrRef
oCont = oSpec->Get(/all,count=nCont)
maskFound = 0
for i=0,nCont-1 do begin
   ;if (maskFound) then break
   oCont[i]->GetProperty, name=name
   if (maskFound = strcmp(name,'PAN_MASK')) then break
endfor
if (~maskFound) then return

; Retrieve the dependent data and determine the mask
; NaNs signify masked data!
Self->GetProperty, dataValue=data
void = oCont[i]->SetData(bits2bytes(finite(data)))

end


;===============================================================================
;+
; DAVEDataset::toDavePtr
; 
; PURPOSE:
;   function used to convert an object of this class into davePtr data struture.
;
; PARAMETERS
;    davePtr - pointer to the DAVE data format structure
;
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function DAVEDataset::toDavePtr, davePtr, errorMsg
compile_opt idl2

if (n_params() eq 0) then return, 0

retVal = 1

Self->UpdateMask

Self->GetProperty, axis1Value=xData, axis1Label=xLabel, axis2Value=yData, axis2Label=yLabel $
                 ,axis1Units=xUnits, axis2Units=yUnits, dataUnits=qtyUnits $
                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
                 ,instrument=inst, treatment=trmt $
                 ,specificPtr=specificPtr, descriPtr=descriPtr $
                 ,axis1Distribution=xType, axis2Distribution=yType

if (ptr_valid(specificPtr) && (n_elements(*specificPtr) gt 0)) then specificStr = (*specificPtr)
if (ptr_valid(descriPtr)) then begin
   dName = (*descriPtr).name
   dUnits = (*descriPtr).units
   dLegend = (*descriPtr).legend
   dQty = (*descriPtr).qty
   dErr = (*descriPtr).err
endif

retVal = create_dave_pointer(davePtr $
                           ,instrument = inst   $
                           ,qty = qty $
                           ,qtunits = qtyUnits $
                           ,qtlabel = qtyLabel $
                           ,err = err $
                           ,xvals = xData $
                           ,xtype = xType $
                           ,xunits = xUnits $
                           ,xlabel = xLabel $
                           ,yvals = yData $
                           ,ytype = yType $
                           ,yunits = yUnits $
                           ,ylabel = yLabel $
                           ,specificstr = specificStr $
                           ,treatment = trmt $
                           ,dName = dName $
                           ,dUnits = dUnits $
                           ,dLegend = dLegend $
                           ,dQty = dQty $
                           ,dErr = dErr $
                           ,ermsg = errormsg $
                          )

if (retVal eq 0) then begin
    ;; cleanup before returning
    if (ptr_valid(specificPtr)) then heap_free, specificPtr
    if (ptr_valid(davePtr)) then heap_free, davePtr
    davePtr = ptr_new()
endif 
if (ptr_valid(descriPtr)) then ptr_free, descriPtr
if (ptr_valid(specificPtr)) then ptr_free, specificPtr


return, retVal

end


;===============================================================================
;+
; DAVEDataset::SetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;
;   instrument - the instrument from which this dataset was obtained from (string)
;   
;   treatment - the treatment history for this dataset (string array)
;   
;   descriPtr - a pointer to the descriPtr component of the dataset (consult davePtr description)
;   
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis, if a 2D dataset - a vector
;   
;   dataValue - the actual values for the dependent data - either 1D or 2D array
;   
;   errorValue - the actual errors in the dependent data - either 1D or 2D array
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;-
pro DAVEDataset::SetProperty, descriPtr=descriPtr $
   ,_EXTRA=etc
compile_opt idl2

;
if (n_elements(descriPtr) gt 0 && ptr_valid(descriPtr) && obj_valid(Self.descriPtrRef)) then begin
                  ;{  descriptrstr,  $
                  ;   name:'',    $
                  ;   units:'',      $
                  ;   legend:'',     $
                  ;   qty:0.0,    $
                  ;   err:0.0        $
                  ; }
   obj = (Self.descriPtrRef)->getByName('Value')
   if (obj_valid(obj)) then begin
      void = obj->SetData((*descriPtr).qty)
      obj->SetProperty, name=(*descriPtr).name
   endif
   
   obj = (Self.descriPtrRef)->getByName('Description')
   if (obj_valid(obj)) then void = obj->SetData((*descriPtr).legend)
   
   obj = (Self.descriPtrRef)->getByName('Error')
   if (obj_valid(obj)) then begin
      void = obj->SetData((*descriPtr).err)
      obj->SetProperty, name='Error in '+(*descriPtr).name
   endif
   
   obj = (Self.descriPtrRef)->getByName('Units')
   if (obj_valid(obj)) then void = obj->SetData((*descriPtr).units)

endif 

; call base class mutator
if (n_elements(etc) gt 0) then Self->GenericDataset::SetProperty, _EXTRA=etc

end


;===============================================================================
;+
; DAVEDataset::GetProperty
; 
; PURPOSE:
;   
;
; PARAMETERS
;
; KEYWORDS:
;   instrument - the instrument from which this dataset was obtained from (string)
;   
;   treatment - the treatment history for this dataset (string array)
;   
;   specificPtr - a pointer to the specificPtr component contained in the dataset (consult davePtr description)
;   
;   descriPtr - a pointer to the descriPtr component of the dataset (consult davePtr description)
;   
;   axis1Ref - the object reference to the first dependent axis of the dataset
;   
;   axis2Ref - the object reference to the second dependent axis of the dataset, if it is 2D
;   
;   dataRef - the object reference to the dependent data
;   
;   errorRef - the object reference to the error in the dependent data
;   
;   axis1Value - the actual values of the first independent axis - a vector
;   
;   axis2Value - the actual values of the second independent axis, if a 2D dataset - a vector
;   
;   dataValue - the actual values for the dependent data - either 1D or 2D array
;   
;   errorValue - the actual errors in the dependent data - either 1D or 2D array
;   
;   nDimensions - 1 | 2 if a 1D | 2D dataset
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;-
pro DAVEDataset::GetProperty, specificPtr=specificPtr, descriPtr=descriPtr $
   ,_REF_EXTRA=etc
compile_opt idl2

;
if (arg_present(descriPtr)) then begin
   status = Self->_GetDescriPtrStr(descriPtrStr)
   descriPtr = (status)? ptr_new(descriPtrStr, /no_copy) : ptr_new()
endif
;
if (arg_present(specificPtr)) then begin
   status = Self->_GetSpecificStr(specificStr, Self.specificPtrRef)
   specificPtr = (status)? ptr_new(specificStr, /no_copy) : ptr_new()
endif

; call base class accessor
if (n_elements(etc) gt 0) then Self->GenericDataset::GetProperty, _EXTRA=etc

end


;===============================================================================
;+
; DAVEDataset::_GetDescriPtrStr
; 
; PURPOSE:
;   function used to retrieve the descriPtr structure component 
;   as defined by the davePtr data struture.
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function DAVEDataset::_GetDescriPtrStr, descriPtrStr
compile_opt idl2

status = 0

cont = Self.descriPtrRef
if (~obj_valid(cont)) then return, status

obj = cont->getByName('Value')
if (~obj_valid(obj)) then return, status
obj->getProperty, name=dName
void = obj->getData(dQty)

obj = cont->getByName('Description')
if (~obj_valid(obj)) then return, status
void = obj->getData(dLegend)

obj = cont->getByName('Error')
if (~obj_valid(obj)) then return, status
void = obj->getData(dErr)

obj = cont->getByName('Units')
if (~obj_valid(obj)) then return, status
void = obj->getData(dUnits)


if (n_elements(dName) ne 0) and (n_elements(dUnits) ne 0) and $
   (n_elements(dLegend) ne 0) and (n_elements(dQty) ne 0) and $
   (n_elements(derr) ne 0) then begin
   descriPtrStr =  {  descriptrstr,  $
                  name:'',    $
                  units:'',      $
                  legend:'',     $
                  qty:0.0,    $
                  err:0.0        $
               }

   if (n_elements(dname) ne 0) and (n_elements(dunits) ne 0) and $
      (n_elements(dlegend) ne 0) and (n_elements(dqty) ne 0) and $
      (n_elements(derr) ne 0) then begin
      descriPtrStr.name = dname
      descriPtrStr.units = dunits
      descriPtrStr.legend = dlegend
      descriPtrStr.qty = dqty
      descriPtrStr.err = derr      
      status = 1
   endif
endif

return, status
end

;===============================================================================
;+
; DAVEDataset::_GetSpecificStr
; 
; PURPOSE:
;   Private recursive function used to retrieve the specificPtr structure component 
;   as defined by the davePtr data struture.
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;-
function DAVEDataset::_GetSpecificStr, dataStr, obj
compile_opt idl2

if (~obj_valid(obj)) then return, 0

obj->getProperty, type=type, name=tag

switch type of
    '': break

    'STRUCT':
    'DAVE1SPECIFICPTR':
    'POINTER': begin
        if (obj_isa(obj,'IDL_CONTAINER')) then begin
            cont = obj->get(/all,count=cnt)
            if (cnt eq 0) then break 
            for i = 0,cnt-1 do retVal = self->_GetSpecificStr(dataStr,cont[i])
            
            
            ;; store in heap if required
            tmpStr = (type eq 'POINTER')? ptr_new(dataStr) : dataStr

            if (type ne 'DAVE1SPECIFICPTR') then $
              dataStr = (n_elements(dataStr) eq 0)? $
                        create_struct(tag,tmpStr) : $
                        create_struct(dataStr,tag,tmpStr)
            
            
        endif else begin
            if (~obj->getData(value)) then break
            if (type ne 'STRUCT') then  $
              value = ptr_new(value)
            
            dataStr = (n_elements(dataStr) eq 0)? $
                      create_struct(tag,value) : $
                      create_struct(dataStr,tag,value)
            
        endelse
        
        break
    end
    
    else: begin
        if (obj_isa(obj,'IDLITDATA')) then begin        
            if (~obj->getData(value)) then break
            obj->GetProperty, extraType=extraType   ;   only defined for IDLitDataDave (subclass of IDLitData) objects
            if (n_elements(extraType) eq 0) then extraType=''
            if (strcmp(extraType,'POINTER')) then value=ptr_new(value)   ; => value was originally a heap var!
            dataStr = (n_elements(dataStr) eq 0)? $
                      create_struct(tag,value) : $
                      create_struct(dataStr,tag,value)
        endif else if (obj_isa(obj,'IDL_CONTAINER')) then begin
            cont = obj->get(/all,count=cnt)
            for i = 0,cnt-1 do retVal = self->_GetSpecificStr(dataStr,cont[i])
        endif
        
        break
    end
    
endswitch

return, 1

end


;===============================================================================
;+
; DAVEDataset::_MakeSpecificObj
; 
; PURPOSE:
;   Convert the contents of the Specific pointer into an object hierarchy.
;   This method is called recursively!
;
; PARAMETERS
;   dataStr - The data structure to be converted
;
;   Container - The parent container in which object(s) of the dataStr
;               are to be placed.
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;-
function DAVEDataset::_MakeSpecificObj, dataStr, parent, type=type, _EXTRA=etc
compile_opt idl2

switch size(dataStr,/tname) of
    'UNDEFINED':
    'OBJREF': break
    
    'POINTER': begin
        if ptr_valid(dataStr) then begin
            if (~keyword_set(type)) then type='POINTER'
            void = self->_MakeSpecificObj(*dataStr,parent,type=type,_EXTRA=etc)
        endif
        break
    end
    
    'STRUCT': begin
        tags = tag_names(dataStr)
        ntags = n_tags(dataStr)
        if (ntags le 0) then break

        if (~keyword_set(type)) then type='STRUCTURE'
        child = obj_new('IDLitDataContainer',type=type, _EXTRA=etc)
        if (~obj_valid(child)) then break
        
        parent->add, child
        if (strcmp(type,'DAVE1SPECIFICPTR')) then Self.specificPtrRef = child  ; register main specific container
        
        for i=0,ntags-1 do $
          void = self->_MakeSpecificObj(dataStr.(i),child,name=tags[i] $
                                ,description=tags[i]) ; ,type=vartype(dataStr.(i))
        
        break
    end
    
    else: begin
        if (n_elements(dataStr) gt 0) then begin
            axisType = 4   ; 'UNDEFINED' by default
            extraType = ''   ; unspecified
            if (~keyword_set(type)) then type=partype(dataStr)
            if (strcmp(type,'POINTER')) then begin
               extraType=type   ; store as extraType property
               type=partype(dataStr)   ; and retrieve the proper type of the data stored in the heap
            endif
            void = where(size(dataStr,/type) eq [1,2,3,4,5,12,13,14,15],validNumericType)   ; what is the intrinsic data type?
            void = where(size(dataStr,/n_dimensions) eq [1,2],oneORtwoDims)   ; is it a vector/array?
            if (validNumericType && oneORtwoDims) then axisType=2   ; set as dependent data vector/array suitable for plotting

            child = obj_new('IDLitDataDave',dataStr,type=type, extraType=extraType, axisType=axisType,_EXTRA=etc)
            if (obj_valid(child)) then parent->add, child
        endif
        break
    end

endswitch




return, 1
end


;===============================================================================
;+
; DAVEDataset::_AddComponents
; 
; PURPOSE:
;   Move contents from the specified davePtr heap variable into the class
;
; PARAMETERS
;   davePtr -  A ponter to data structure that contains a dave format dataset.
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;-
function DAVEDataset::_AddComponents, davePtr
compile_opt idl2

if (~ptr_valid(davePtr) || (n_elements(*davePtr) eq 0)) then return, 0

; The instrument name and treatment history
childData = (*(*davePtr).dataStrPtr).commonStr.instrument
child = obj_new('IDLitData',childData,name='Instrument',type='DAVE1INST'$ 
                ,description='Instrument Name')
if (obj_valid(child)) then begin
   Self->add, child
   self.instrumentRef = child
endif

childData = (*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr)
child = obj_new('IDLitData',childData,name='Treatment History',type='DAVE1HISTORY' $
                ,description='Treatment history')
if (obj_valid(child)) then begin
   Self->add, child
   Self.treatmentRef = child
endif


;_________________________________________________
; Deal with the descriPtr
void = where(tag_names(*davePtr) eq 'DESCRIPTR', cnt)
if (cnt eq 1) then begin 
    ;; A descriPtr entry is present.
    ;; If it is valid, create an entry for it
    if (ptr_valid((*davePtr).descriPtr) && $
        (n_elements(*(*davePtr).descriPtr) gt 0)) then begin
        
        descriPtrStr = (*(*davePtr).descriPtr)
        ;; Create a container for the descriPtr and fill it with the
        ;; descriPtr contents
        oDescriPtr = obj_new('IDLitParameterSet', name='Sample Information Tag' $
                             ,description='Sample information tag (descriPtr)'  $
                             ,type='DESCRIPTR')
        oQty = obj_new('IDLitData',descriPtrStr.qty,name=descriPtrStr.name,type=partype(descriPtrStr.qty),uvalue=0.0)
        oErr = obj_new('IDLitData',descriPtrStr.err,name='Error in '+descriPtrStr.name,type=partype(descriPtrStr.err))
        oUnits = obj_new('IDLitData',descriPtrStr.units,name='Units',type=partype(descriPtrStr.units))
        oLegend = obj_new('IDLitData',descriPtrStr.legend,name='Description',type=partype(descriPtrStr.legend))
        parameter_names = ['Description','Value','Error','Units']
        oDescriPtr->add, [oLegend,oQty,oErr,oUnits],parameter_name=parameter_names
        Self->add, oDescriPtr
        Self.descriPtrRef = oDescriPtr
    endif
endif

;_________________________________________________
; Deal with commonStr

; Create a container for the data (x,y,z axes)
dims = (size((*(*(*davePtr).dataStrPtr).commonStr.histPtr).qty))[0]
Self.ndims = dims
icon = (dims eq 1)? 'plot' : 'surface'
oCommonCon = obj_new('IDLitParameterSet',name='Experimental Data' $
                     ,description='Experimental Data Section',type='DAVE1COMMONSTR',icon=icon)
oCommonCon->AddMetaData,'DatasetName',Self.filename ; give it same name as the filename
Self->Add, oCommonCon
Self.plotDataRef = oCommonCon

; First Independent Axis
; Store in IDLitDataDave object and include additional metadata
desc = (*(*davePtr).dataStrPtr).commonStr.xlabel
xname = (strtrim(desc) eq '')? 'X Axis' : desc
xAxisType = '0'
data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).x
dist = (strcmp((*(*davePtr).dataStrPtr).commonStr.xtype,'HISTOGRAM',4,/FOLD_CASE) gt 0)? $
       'HISTOGRAM' : 'POINTS'
oX = obj_new('IDLitDataDave',data,name=xname,type=partype(data),description=desc,axisType=xAxisType)
oX->AddMetaData,'Long_name',desc
oX->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.xunits
oX->AddMetaData,'Distribution',dist

oCommonCon->Add, oX, parameter_name=xAxisType
Self.axis1Ref = oX

if (dims eq 2) then begin
    ;; Second Independent Axis
    ;; Store in IDLitDataDave object and include additional metadata
    desc = (*(*davePtr).dataStrPtr).commonStr.ylabel
    yname = (strtrim(desc) eq '')? 'Y Axis' : desc
    yAxisType = '1'
    data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).y
    dist = (strcmp((*(*davePtr).dataStrPtr).commonStr.ytype,'HISTOGRAM',4,/FOLD_CASE) gt 0)? $
           'HISTOGRAM' : 'POINTS'
    oY = obj_new('IDLitDataDave',data,name=yname,type=partype(data),description=desc,axisType=yAxisType)
    oY->AddMetaData,'Long_name',desc
    oY->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.yunits
    oY->AddMetaData,'Distribution',dist
    
    oCommonCon->Add, oY, parameter_name=yAxistype
    Self.axis2Ref = oY
endif

; The data
; Store in IDLitDataDave object using a suitable type and additional metadata.
desc = (*(*davePtr).dataStrPtr).commonStr.histlabel
zname = (strtrim(desc) eq '')? 'Data' : desc
zAxisType = '2'
axes = (dims eq 1)? xname : [xname,yname]
data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).qty

; Check for masked data points 
; Masked points have the value -1e20 on a NaN
; For consistentcy, replace masked points with a NaN
index = where(data eq -1e20, nMasked)
if (nMasked gt 0) then data[index] = !values.F_NAN

oZ = obj_new('IDLitDataDave',data,name=zname,type=partype(data),description=desc,axisType=zAxisType)
oZ->AddMetaData,'Signal',1
oZ->AddMetaData,'Axes',axes
oZ->AddMetaData,'Long_name',desc
oZ->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.histunits

oCommonCon->Add, oZ, parameter_name=zAxisType
Self.dataRef = oZ

; The data error
; Store in an IDLitDataDave object
ename = 'Error'
eAxisType = '3'
desc = 'Uncertainty in '+zname
error = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).err
oErr = obj_new('IDLitDataDave',error,name=ename,type=partype(error),description=desc,axisType=eAxisType)
oCommonCon->Add, oErr, parameter_name=eAxisType
Self.errorRef = oErr

;_________________________________________________
; Deal with specficPtr
specificPtr = (*(*davePtr).dataStrPtr).specificPtr
if (ptr_valid(specificPtr) && $
    (n_elements(*specificPtr) gt 0)) then begin
    
    ; Is pan_mask field present? 
    ; If not, create one. It is expected that NaNs are used as filler values for masked data!
    tags = tag_names(*specificPtr)
    index = where(tags eq 'PAN_MASK', foundMask)
    if (~foundMask) then begin
      *specificPtr = create_struct(*specificPtr $
                                 ,'pan_filler',!VALUES.F_NAN $
                                 ,'pan_mask',ptr_new(bits2bytes(finite(data))) $
                                 )
    endif
    
    ; convert structure into an object hierarchy
    void = self->_MakeSpecificObj(specificPtr,Self,name='Miscellaneous' $
                          ,description='Instrument Specific Details' $
                          ,type='DAVE1SPECIFICPTR')
endif

return, 1
end


;;===============================================================================
; DAVEDataset::Clone
; 
; PURPOSE:
;   return an object containing an axact copy of the data in the current object
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - a copy of the current object
;    failure - a null object
;
function DAVEDataset::Clone
compile_opt idl2

status = Self->toDAVEPtr(davePtr)

if (status && ptr_valid(davePtr)) then begin
   filename = 'Copy of '+Self.filename
   Self->GetProperty, name=nameTag
   nameTag = 'Copy of '+nameTag
   oData = obj_new('DAVEDataset',davePtr=davePtr,nameTag=nameTag)
   heap_free, davePtr
   if (obj_valid(oData)) then oData->SetProperty, filename=filename
   return, oData
end

return, obj_new() 

end


;===============================================================================
; DAVEDataset::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEDataset class. For the initialization to 
;   complete successfully, at least a filename or the davePtr keyword must be supplied.
;
; PARAMETERS
;   filename - The name of a dave format file from which to retrieve the dataset. If the
;              davePtr keyword is supplied, the filename is simply used as the name property 
;              of the instantiated object of this class.
;
;   davePtr -  A pointer to data structure that contains a dave format dataset.
;
;   nameTag -  A label to be associated with objects of this class. If undefined, it will 
;              default to the basename of the filename keyword, if that is set, or to 'UNTITLED'
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function DAVEDataset::Init, filename=filename, davePtr=davePtr, nameTag=nameTag, _REF_EXTRA=etc
compile_opt idl2

; Basic error handler
catch, theError
if (theError ne 0) then begin
   catch, /cancel
   return, 0
endif

filenameIsSet = n_elements(filename)
davePtrIsSet = n_elements(davePtr)
nameTagIsSet = n_elements(nameTag)

; Can't proceed if both filename and davePtr are undefined
if (~filenameIsSet && ~davePtrIsSet) then return, 0

; If starting from filename, retrieve the davePtr, if it is not specified
if (filenameIsSet and ~davePtrIsSet) then begin
   ; Check file contents to make sure this is a valid DAVE 1.x file.
   ; Using the IDL_Savefile class avoids the need to restore the file first
   oSaveFile = obj_new('IDL_Savefile', filename)
   if (~obj_valid(oSaveFile)) then return, 0
   varNames = oSaveFile->Names()   ; retrieve variable names found in sav file
   obj_destroy, oSaveFile
   
   ; A valid DAVE file must contain a variable called davePtr
   res = where(varNames eq 'DAVEPTR', count)
   if (count eq 0) then return, 0

   ; use IDL restore to retrieve info from file
   restore,filename, /relaxed_structure_assignment, restored_objects=objects

   ; No davePtr? Can't continue; This should not really happen because of previous check above!
   if (~ptr_valid(davePtr) || (n_elements(*davePtr) eq 0)) then begin
      obj_destroy, objects
      return, 0
   endif

   ; strip dir path and extension from the filename and assign to nameTag, if required
   if (~nameTagisSet) then nameTag = file_basename(filename,'.dave',/fold_case)
endif

; If nameTag is not yet specified, set it to 'Untitled'
if (n_elements(nameTag) eq 0) then nameTag = 'UNTITLED'
description = (filenameIsSet)? filename : nameTag
Self.filename=description

; iTools does not handle '_[0-9]' characters in nameTags well eg _0, _1, etc
; replace all '_' with '-'
;while ((pos = stregex(nameTag,'_[0-9]')) ne -1) do begin
;    strput, nameTag,'-',pos
;endwhile

; Call the base class init method to instantiate container
status = self->GenericDataset::Init(name=nameTag $
                ,description=description,identifier='id'+file_basename(nameTag) $
                ,type='DAVE1DATASET', _EXTRA=etc)
if (status eq 0) then return, status

; Construct remainder of object and return
return, self->_AddComponents(davePtr)

end


;===============================================================================
pro DAVEDataset__Define
compile_opt idl2

void = {DAVEDataset $
        ,inherits GenericDataset $
        ,descriPtrRef:obj_new() $
        ,specificPtrRef:obj_new() $
        }

end
