; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   ASCIIColumnDataset
;
; PURPOSE:
;   Data Class for representing ASCII Column formated datasets
;
; CATEGORY:
;
; SUPERCLASSES:
;   GenericDataset
;
; SUBCLASSES:
;
; CREATION:
;   See ASCIIColumnDataset::Init
;
; METHODS:
;   GetProperty
;   SetProperty
;   ToDavePtr
;   ToASCIISPE
;   ToASCIIGrp
;   ToASCIIColumn
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; December 2008
;-
;###############################################################################


;===============================================================================
; ASCIIColumnDataset::toASCIIColumn
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Column data.
;
; PARAMETERS
;    data - a nxm array of the data where n is the nos of columns and and m the nos of rows.
;           Where for a n=3 (ie 3 column dataset) (if no errors, then the error column would be absent)
;              column 1 is the independent data
;              column 2 is the dependent data
;              column 3 is the uncertainty in the dependent data
;              
;           If n gt 3 then it is assummed that there are multiple data groups all with the same
;           independent data as follows (if no errors, then the error columns would be absent):
;              column 1 is the independent data
;              column 2 is the dependent data for grp 1
;              column 3 is the uncentainty in the dependent data for group 1
;              column 4 is the dependent data for grp 2
;              column 5 is the uncentainty in the dependent data for group 2
;              ...
;    
;    header - is a string vector of size n (ie n columns) containing textual annotation (label) for the data block, 
;             including info about group values
;    
;
;   errorMsg - if unsuccessful, this string may containing an error message
;   
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
function ASCIIColumnDataset::toASCIIColumn, data, header, errorMsg
compile_opt idl2

; need at least the first parameter to proceed
if (n_params() lt 1) then return, 0


;; Get the top level contents of dataset
objs = Self->Get(/all,count=cnt)
if (cnt lt 1) then  begin
   errorMsg = ['Empty dataset!','Cannot write selected dataset']
   return, 0
endif

ncols=0
Self->GetProperty, AXIS1REF=obj
if (obj_valid(obj)) then begin
   void = obj->getData(xData)
   if (n_elements(xData) le 0) then begin
      errormsg = ['Incomplete dataset!','Cannot write selected dataset']
      ;self->SignalError, msg,severity=2
      return, 0
   endif
   obj->GetProperty, name=header
   dataColumns = transpose(xData)
   nx = n_elements(xData)
   ncols++
   ;if ((void = obj->GetMetaData('Long_name',xLabel)) eq 0 ) then xLabel=''
   ;if ((void = obj->GetMetaData('Units',xUnits)) eq 0 ) then xUnits=''
   ;if ((void = obj->GetMetaData('Distribution',xType)) eq 0 ) then xType='POINTS'
endif


Self->GetProperty, dataREF=obj ; the dependent axis
if ((retVal = obj_valid(obj)) eq 0) then begin
   errormsg = ['Incomplete dataset!','Cannot write selected dataset']
   ;self->SignalError, msg,severity=2
   return, 0
endif
void = obj->getData(qty)
nqty = n_elements(qty) 
if (nqty le 0) then begin
   errormsg = ['Incomplete dataset!','Cannot write selected dataset']
   ;self->SignalError, msg,severity=2
   return, 0
endif
;if ((void = obj->GetMetaData('Long_name',qtyLabel)) eq 0 ) then qtyLabel=''
;if ((void = obj->GetMetaData('Units',qtyUnits)) eq 0 ) then qtyUnits=''
;if ((void = obj->GetMetaData('Axes',axes)) eq 0 ) then axes=''
dataColumns = (n_elements(dataColumns) eq 0)? transpose(qty) : $
              [dataColumns,transpose(qty)]
obj->GetProperty, name=name
header = [header,name]
ncols++

Self->GetProperty, errorREF=obj ; the error in the dependent axis
if (obj_valid(obj)) then begin
   void = obj->getData(err)
   if (n_elements(err) gt 0) then begin
      dataColumns = [dataColumns,transpose(err)]
      obj->GetProperty, name=name
      header = [header,name]
      ncols++
   endif
endif

;; get additional columns, if present
if (cnt gt 3) then begin
   processedTypes = ['0','2','3']
   for i=0,cnt-1 do begin
      if (~obj_valid(objs[i])) then continue
      objs[i]->getProperty, axisType=axisType
      res = where(processedTypes eq axisType, processed)
      if (processed) then continue
      ok = objs[i]->getData(data)
      if (~ok || (n_elements(data) ne nqty)) then continue

      objs[i]->GetProperty, name=name
      dataColumns = [dataColumns,transpose(data)]
      header = [header,name]
      ncols++
   endfor
endif 

data = temporary(dataColumns)

return, 1
end



;===============================================================================
; ASCIIColumnDataset::Clone
; 
; PURPOSE:
;   return an object containing an axact copy of the data in the current object
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - a copy of the current object
;    failure - a null object
;
function ASCIIColumnDataset::Clone
compile_opt idl2

status = Self->toASCIIColumn(data,header)

if (status) then begin
   filename = 'Copy of '+Self.filename
   Self->GetProperty, name=nameTag, treatment=trmt
   nameTag = 'Copy of '+nameTag
   oData = obj_new('ASCIIColumnDataset',data=data,header=header,nameTag=nameTag)
   if (obj_valid(oData)) then oData->SetProperty, filename=filename, treatment=trmt
   return, oData   
end

return, obj_new() 

end



;===============================================================================
; ASCIIColumnDataset::toASCIIGrp
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII Grp data format variables.
;
; PARAMETERS
;   dataStr - a data structure containing a SPE dataset This is an anonymous data structure
;             containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
;
;   errorMsg - if unsuccessful, this string may containing an error message
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
function ASCIIColumnDataset::toASCIIGrp, dataStr, errorMsg
compile_opt idl2

; Don't yet know how to convert myself to ASCII Grp
errorMsg = "Cannot convert ASCII Column datasets to ASCII GRP format"
return, 0

end



;===============================================================================
; ASCIIColumnDataset::toASCIISPE
; 
; PURPOSE:
;   function used to convert an object of this class into ASCII SPE data format variables.
;
; PARAMETERS
;   dataStr - a data structure containing a SPE dataset This is an anonymous data structure
;             containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
;
;   errorMsg - if unsuccessful, this string may containing an error message
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
function ASCIIColumnDataset::toASCIISPE, dataStr, errorMsg
compile_opt idl2

; Don't yet know how to convert myself to ASCII SPE
errorMsg = "Cannot convert ASCII Column datasets to ASCII SPE format"
return, 0

end



;===============================================================================
; ASCIIColumnDataset::toDavePtr
; 
; PURPOSE:
;   function used to convert an object of this class into davePtr data struture.
;
; PARAMETERS
;    davePTr - pointer to the DAVE data format structure
;    
;    errorMsg - an error message string or array in case of error
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - 1
;    failure - 0
;
function ASCIIColumnDataset::toDavePtr, davePtr, errormsg
compile_opt idl2

if (n_params() lt 1) then return, 0

retVal = 1


; Only a 3 column x,y,e dataset can be stored in a 1D davePtr
; It is not obvious to convert a multi-column dataset
if (Self.nColumns ne 3) then begin
   errormsg = 'Only a 3 column ASCII Column file can be converted to DAVE format'
   return, 0
endif


Self->GetProperty, axis1Value=xData, axis1Label=xLabel $
                 ,dataValue=qty, dataLabel=qtyLabel, errorValue=err $
                 ,axis1Distribution=xType

retVal = create_dave_pointer(davePtr $
                           ,qty = qty $
                           ,qtlabel = qtyLabel $
                           ,err = err $
                           ,xvals = xdata $
                           ,xtype = xtype $
                           ,xlabel = xlabel $
                          )

if (retVal eq 0) then begin
   ;; cleanup before returning
   errormsg = 'Failed to successfully create a davePtr data structure' 
   if (ptr_valid(davePtr)) then heap_free, davePtr
endif

return, retVal

end



;===============================================================================
; ASCIIColumnDataset::Init
; 
; PURPOSE:
;   Initialization method for objects of ASCIIColumnDataset class. For the initialization to 
;   complete successfully, the data keyword must be supplied.
;
; PARAMETERS
;   filename   - The name of a dave format file from which the dataset was retrieved. The data must have 
;                been retrieved already and stored in the data keyword variable below.
;
;   nameTag    - A label to be associated with objects of this class. If undefined, it will 
;                default to the basename of the filename keyword, if that is set, or to 'UNTITLED'
;
;   data  - a nxm array of the data where n is the nos of columns and and m the nos of rows.
;           Where for a n=3 (ie 3 column dataset) (if no errors, then the error column would be absent)
;              column 1 is the independent data
;              column 2 is the dependent data
;              column 3 is the uncertainty in the dependent data
;              
;           If n gt 3 then it is assummed that there are multiple data groups all with the same
;           independent data as follows (if no errors, then the error columns would be absent):
;              column 1 is the independent data
;              column 2 is the dependent data for grp 1
;              column 3 is the uncentainty in the dependent data for group 1
;              column 4 is the dependent data for grp 2
;              column 5 is the uncentainty in the dependent data for group 2
;              ...
;    
;   header - is a string vector of size n (ie n columns) containing textual annotation (label) for the data block, 
;             including info about group values
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function ASCIIColumnDataset::Init, filename=filename, data=data, header=fieldnames, nameTag=nameTag, _REF_EXTRA=etc
compile_opt idl2

filenameIsSet = n_elements(filename)
dataIsSet = n_elements(data)
nameTagIsSet = n_elements(nameTag)
fieldNamesIsSet = n_elements(fieldNames)

; can't proceed otherwise
if (~dataIsSet) then return, 0

nFields = (size(data,/dimensions))[0]

if (nFields lt 1) then return, 0

if (~fieldNamesIsSet) then $
   fieldNames = strarr(nFields)+'field'+strtrim(string(indgen(nFields)+1),2)   ; 'field1', 'field2', etc
nf = n_elements(fieldNames)
if (nf lt nFields) then begin
   buffer = strarr(nFields)+'field'+strtrim(string(indgen(nFields)+1),2)   ; 'field1', 'field2', etc
   filedNames = [filedNames,buffer[nf:nFields-1]]
endif

if (filenameIsSet) then begin
   i = strpos(filename,'.',/reverse_search)
   ext = (i lt 0)? '' : strmid(filename,i)
   ; strip dir path and extension from the filename and assign to nameTag, if required
   if (~nameTagisSet) then nameTag = file_basename(filename,ext,/fold_case)
endif

; If nameTag is not yet specified, set it to 'Untitled'
if (n_elements(nameTag) eq 0) then nameTag = 'UNTITLED'
description = (filenameIsSet)? filename : nameTag
Self.filename = description


; iTools does not handle '_[0-9]' characters in name property of objects
; replace all '_' with '-'
;while ((pos = stregex(nameTag,'_[0-9]')) ne -1) do begin
;    strput, nameTag,'-',pos
;endwhile

; Call the base class init method to instantiate container
status = self->GenericDataset::Init(name=nameTag,identifier='id'+file_basename(nameTag) $
                ,ICON='plot',type='ASCIICOL',DESCRIPTION=description,_EXTRA=etc)
                
Self->AddMetaData,'DatasetName',nameTag 

; Check for masked data points 
; Replace masked points with a -1e20 filler value with a NaN
index = where(data eq -1e20, nMasked)
if (nMasked gt 0) then data[index] = !values.F_NAN


for i=0,nFields-1 do begin
    axisType = (i eq 0)? i : i+1
    if (nFields eq 1) then axisType = 2 ; ie single column only ==> dependent axis
    type = partype(reform(data[i,*]))
    oItem = OBJ_NEW('IDLitDataDAVE', reform(data[i,*]),axisType=axisType,type=type, $
                    NAME=fieldNames[i],description=fieldNames[i])
    oItem->AddMetaData,'Long_name',fieldNames[i]
    oItem->AddMetaData,'Units',''
    if (axisType eq 0) then begin
      oItem->AddMetaData,'Distribution','POINT'
      Self.axis1Ref = oItem
    endif
    if (axisType eq 2) then Self.dataRef = oItem
    if (axisType eq 3) then Self.errorRef = oItem
 
    Self->Add, oItem, parameter_name=strtrim(axisType,2)
endfor

; Add a blank treatment history object
child = obj_new('IDLitData','',name='Treatment History',type='HISTORY' $
                ,description='Treatment history')
if (obj_valid(child)) then begin
   Self->add, child
   Self.treatmentRef = child
endif

Self.ndims = 1   ; indicate that column data are 1D though not strictly correct!
Self.plotDataRef = Self
Self.nColumns = nFields

return, 1

end


;===============================================================================
pro ASCIIColumnDataset__Define
compile_opt idl2

void = {ASCIIColumnDataset $
        ,inherits GenericDataset $
        ,nColumns:0 $
        }

end