; $Id$
;###############################################################################
;
; NAME:
;   fans_datreduc
;
; PURPOSE:
;   FANS data redunction module. It performs the following tasks
;   ** Load one or more sample files
;   ** Load one or more background files
;   ** Load one or more Fast bkgd files
;   ** Allows user to mask (remove) detectors after visual inspection
;      of data
;   ** Correct data for fast bkgd (using a polynomial fit to it)
;   ** Perform bkgd subtraction if required.
;
; CATEGORY:
;   FANS Data Reduction
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   May, 2002
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################





;===============================================================================
; wd_fansfitfast_plotClick
;
; PURPOSE:
;   Event handler for button events dispatched from the plot
;   window. When detector display mode is on (ie samp, bkgd views)
;   then deal with left-mouse clicks to mask/unmask dets else
;   zoom/reset display. Mark all masked detectors with grey bars in
;   the display window.
;
; PARAMETERS:
;   event [in] - event structure
;
pro wd_fansfitfast_plotClick, event
compile_opt idl2

widget_control, event.top, get_uvalue=sPtr

case event.type of
    0: begin                ; button press
        (*sPtr).mouseType = event.press
        if ((*sPtr).mouseType eq 4) then begin ; right-hand mb so reset plot to full range
            wd_fansfitfast_plotData, sPtr
            (*sPtr).mouseType = 0 ; set mouse type to undefined value
            return
        endif
        if ((*sPtr).mouseType eq 1) then begin ; left-hand mb so record mouse location
            (*sPtr).zp1 = [event.x,event.y]
        endif
    end
    1: begin                ; button release
        if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
            zp1 = (*sPtr).zp1
            zp2 = (*sPtr).zp2
            x = min([zp1[0], zp2[0]])
            y = min([zp1[1], zp2[1]])
            w = abs(zp1[0] - zp2[0])
            h = abs(zp1[1] - zp2[1])
            x2 = x+w
            y2 = y+h
            lc = convert_coord(x, y, /device, /to_data)
            uc = convert_coord(x2, y2, /device, /to_data)
            wd_fansfitfast_plotData, sPtr,xrange=[lc[0],uc[0]], yrange=[lc[1],uc[1]]
            (*sPtr).zp2 = [0.0,0.0] ; reset to zero
        endif
        (*sPtr).mouseType = 0
    end
    2: begin                ; mouse motion
        if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
            (*sPtr).zp2 = [event.x, event.y]
            zp1 = (*sPtr).zp1
            zp2 = (*sPtr).zp2
            xc = [zp1[0], event.x, event.x, zp1[0], zp1[0]]
            yc = [zp1[1], zp1[1], event.y, event.y, zp1[1]]
            device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
            plots, xc, yc, color=(*sPtr).colors.red, /device
        endif
    end
    else:
endcase    
    

end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_fansfitfast_plotData
;
; PURPOSE:
;   Plot data in the draw widget.
;
; PARAMETERS:
;   sPtr [in] - pointer to state structure
;
;
; KEYWORDS:
;
pro wd_fansfitfast_plotData, sPtr, _EXTRA=etc ;xrange=xrange, yrange=yrange
compile_opt idl2

dataPtr = (*sPtr).dataPtr 
date = (*dataPtr).date
title = (*dataPtr).title
subtitle = 'Expt date: '+date
xtitle = (*dataPtr).xtitle
ytitle = (*dataPtr).ytitle

omega = (*dataPtr).energy
counts = (*dataPtr).counts
sigma = (*dataPtr).errors

;; set winPix as destination window initially
wset, (*sPtr).winPix
  
;; plot detector counts vs detector number
plot,omega,counts, $
     xstyle=1,/ynozero,psym=-4,symsize=1.5,linestyle=0,charsize=1.4, $
     xtitle=xtitle,subtitle=subtitle,ytitle=ytitle,title=title, $
     _EXTRA=etc

;; plot errors
;sigma = counts
;index = where(sigma le 0.0, zeroCnts)
;if (zeroCnts gt 0) then sigma[index] = 1.0
;sigma = sqrt(sigma)
errplot, omega, counts-sigma, counts+sigma ; plot errors

;; display fit
(*sPtr).props->GetProperty, params=fpar, func=funcIndex
case funcIndex of
    0: yfit=(fans_Arcsin(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar
    1: yfit=(fans_Linear(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
    2: yfit=(fans_Quad(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
    3: yfit=(fans_Cubic(omega,fpar))[*,0] ;; funct returns eval + partial derivatives wrt fpar 
endcase
oplot, omega,yfit,linestyle=2,thick=4,color=(*sPtr).colors.magenta ; overplot fit

wset, (*sPtr).winVis    ; set winVis as destination window
device, copy=[0,0,(*sPtr).winWidth,(*sPtr).winHeight,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis


end



;===============================================================================
; wd_fansfitfast_fitData
;
; PURPOSE:
;   Event handler; Initiate a new fast bkgd fit.
;
; PARAMETERS:
;   event [in] - event structure to be handled
;
; KEYWORDS:
;   sPtr [in] - pointer to state structure
;
pro wd_fansfitfast_fitData, sPtr
compile_opt idl2

dataPtr = (*sPtr).dataPtr
dspace = (*dataPtr).dSpace ; get the monochromator d-space value
omega = (*dataPtr).energy
counts = (*dataPtr).counts
sigma = (*dataPtr).errors
;sigma = counts
;index = where(sigma le 0.0, zeroCnts)
;if (zeroCnts gt 0) then sigma[index] = 1.0

(*sPtr).Props->GetProperty, func=funcIndex

case funcIndex of
    0: begin
        funcName = 'fans_Arcsin'        
        par = [1.0,1.0,1.0,0.0,0.0,0.0]
        par[4] = dspace
        par[5] = ((*sPtr).xunits eq 0)? 1.0 : (*sPtr).mev2wnos
        fixpar=[1,1,1,1,0,0]    ; <== fifth and sixth are fixed during fitting
    end
    1: begin
        par = [1.0,1.0]
        fixpar=[1,1]
        funcName = 'fans_Linear'        
    end
    2: begin
        par = [1.0,1.0,1.0]
        fixpar=[1,1,1]
        funcName = 'fans_Quad'        
    end
    3: begin
        par = [1.0,1.0,1.0,1.0]
        fixpar=[1,1,1,1]
        funcName = 'fans_Cubic'        
    end
endcase

; do a non-linear lsqfit using a Levenberg-Marquardt algorithm
nfree = n_elements(counts) - n_elements(par)
if (nfree le 0) then begin
   msg = 'Insufficient data points to fit function'
   msg = [msg,'Select another function or Try a manual fit']
   msg = [msg,'For a manual fit, adjust parameters in property sheet until fit is satisfactory']
   title = 'FANS Data Reduction - Fit Fast Background'
   (*sPtr).Tool->ErrorMessage, msg, severity=0, title=title
   (*sPtr).Props->SetProperty, params=par
   return
endif
yfit = lmfit(omega,counts,par,fita=fixpar,measure_errors=sigma,function_name=funcName,/double)

(*sPtr).Props->SetProperty, params=par

;switch funcIndex of
;    0:          ; Arcsin
;    3: begin    ; Cubic
;      (*sPtr).Props->SetProperty, p0=par[0],p1=par[1],p2=par[2],p3=par[3]
;      break
;    end
;    1: begin    ; Linear
;      (*sPtr).Props->SetProperty, p0=par[0],p1=par[1]
;      break
;    end
;    2: begin    ; Quad
;      (*sPtr).Props->SetProperty, p0=par[0],p1=par[1],p2=par[2]
;      break
;    end
;endswitch

end
;-------------------------------------------------------------------------------


;===============================================================================
; fans_Help
;
; PURPOSE:
;   Event handler. Display FANS data reduction manual.
;
; PARAMETERS:
;   event [in] - event structure.
;
pro wd_fansfitfast_Help,event
compile_opt idl2

widget_control,event.top,get_uvalue = pState
void = launch_help(!DAVE_PDFHELP_DIR+'fansmaskdetectors_manual.pdf',tlb = event.top)
return
end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_resize
;
; PURPOSE:
;   Event handler. Resize the tlb widget.
;
; PARAMETERS:
;   event [in] - event structure
;
pro wd_fansfitfast_resize, event
compile_opt idl2

if dave_set_focus(event) then return

widget_control, event.top, get_uvalue=sPtr

; new plot window width is maximum of the two r-values
(*sPtr).winWidth = (event.x - (*sPtr).restOfWidth) > (*sPtr).minXSize

; new plot window height is maximum of two r-values
(*sPtr).winHeight = (event.y - (*sPtr).restOfHeight) > (*sPtr).minYSize


; Delete and recreate pixmap window using new dimensions
wdelete, (*sPtr).winPix
window,/free,/pixmap,xsize=(*sPtr).winWidth,ysize=(*sPtr).winHeight
(*sPtr).winPix = !d.window
; Also resize visible window using new dimensions 
widget_control, (*sPtr).winVisID, xsize=(*sPtr).winWidth, ysize=(*sPtr).winHeight

widget_control, (*sPtr).wTLB, /realize

wd_fansfitfast_plotData, sPtr

end



;===============================================================================
; wd_fansfitfast_cleanup
;
; PURPOSE:
;   Cleanup routine called when the main widget is destroyed
;
; PARAMETERS:
;   wTLB [in] - the widget id of the top-level-base that is being
;                killed.
;
pro wd_fansfitfast_cleanup, wTLB
compile_opt idl2

widget_control, wTLB, get_uvalue=sPtr

device, decomposed = (*sPtr).olddc
tvlct,(*sPtr).oldrgb.r,(*sPtr).oldrgb.g,(*sPtr).oldrgb.b
wdelete, (*sPtr).winPix         ; delete pixmap window


;
;; deal with the remainder of the state pointer
obj_destroy, (*sPtr).Props
ptr_free, sPtr

end
;-------------------------------------------------------------------------------


;===============================================================================
; wd_fansfitfast_PS_event
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
function wd_fansfitfast_PS_event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_fansfitfast_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;; Get our state var
widget_control, event.top, get_uvalue=sPtr 
uname = widget_info(event.id,/uname)
sname = tag_names(event,/structure_name)

oTool = (*sPtr).Tool


case sname of
   'WIDGET_PROPSHEET_CHANGE': begin    
      ; Get the value of the changed property
      value = widget_info(event.id, component=event.component,property_value=event.identifier)
      ; Set the component's property value. 
      event.component->SetPropertyByIdentifier, event.identifier, value 

      id = event.identifier
      
      wd_fansFitFast_plotData, sPtr

      widget_control, (*sPtr).wPS, /refresh_property
   end
   
   else:
endcase

return, 1
end


;===============================================================================
; wd_fansfitfast_event
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
pro wd_fansfitfast_event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_fansfitfast_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


widget_control, event.top, get_uvalue=sPtr 
uname = widget_info(event.id,/uname)
sname = tag_names(event,/structure_name)

;; resize event
if (strcmp(sname,'WIDGET_BASE')) then begin

   return
endif

;;
case strupcase(uname) of
   'DISCARD': begin
      widget_control, event.top, /destroy
   end
   
   'ACCEPT': begin
      (*sPtr).Props->GetProperty, func=funcIndex, params=params, nparams=np

      ;; this is a USERDEF property type hence need to set both the
      ;; property value and the userdef attribute (the userdef attribute
      ;; determines what gets displayed in propertysheet!)

      sparams = strjoin(strtrim(string(params[0:np-1],format='(G12.6)'),2),', ')
      (*sPtr).tool->SetProperty, fastparams = sparams
      (*sPtr).tool->SetPropertyAttribute, 'FASTPARAMS', userdef=sparams

      (*sPtr).tool->SetProperty, fastfunc = funcIndex

      widget_control, event.top, /destroy
   end
   
   'DRAW': begin
      wd_fansfitfast_plotClick, event
   end
   
   'DOFIT': begin
      wd_fansfitfast_fitData, sPtr
      wd_fansfitfast_plotData, sPtr
      widget_control, (*sPtr).wPS, /refresh_property
   end
   
   'HELP': wd_fansfitfast_Help, event
   

   else:
endcase

end


;===============================================================================
; wd_fansfitfast
;
; PURPOSE:
;   Main entry point and dialog builder for FANS data reduction application.
;
; PARAMETERS:
;
; KEYWORDS:
;
;   group_leader [in] - the ID of the parent widget from which this
;                       dialog was launched.
;
;   dataDir [in] - The default location for raw data files
;
;   workDir [in] - The default location for saving reduced files
;
;   DAVETool [in] - Reference to the DAVE tool object
;
pro wd_fansfitfast, oUI, dataPtr=dataPtr, _EXTRA=etc
compile_opt idl2

if (~obj_valid(oUI)) then return
if (~ptr_valid(dataPtr)) then return

tvlct,rorig,gorig,borig,/get
oldrgb = {r:rorig,g:gorig,b:borig}
device, get_decomposed = olddc
device,decomposed = 0
colors = hfbs_GetColor(/Load, Start=1) ; custom (16) color table

oUI->GetProperty, group_leader = group_leader
oTool = oUI->GetTool()
oTool->GetProperty, fastparams=paramsString,funcs=funcs,fastfunc=fastfunc,energyUnitFlag=xunits
params = [0.0,0.0]
if (~strcmp(paramsString,'')) then $
   params = float(strsplit(paramsString,',',/extract,count=nParams))

; define container base widgets; make it resizeable
wTLB = widget_base(group_leader=group_leader,/col,title='FANS Data Reduction - Fit Fast Background' $
                  ,uname='FANSFITFAST',/tlb_size_events,/modal)

wCB = widget_base(wTLB,/col,frame=1);,/align_center)
wCBa = widget_base(wCB,/row,/align_center)

wCBb = widget_base(wCB,col=2,/grid,/base_align_center,/frame)
wCBc = widget_base(wCBb,/col,/base_align_center)
wCBd = widget_base(wCBb,/col,/base_align_left)

xSize = 500
ySize = 350
wVis = widget_draw(wCBa,xsize=xsize,ysize=ysize,/button_events,/motion_events,uname='Draw')

oProps = obj_new('FANS_FUNCPROPS',name='Preferences',func=fastfunc,unit=xunits,params=params)
wPS = widget_propertysheet(wCBc $
                     ,value=oProps $
                     ,uname='PROPSHEET' $
                     ,scr_xsize=xSize*0.46 $
                     ,scr_ysize=ysize*0.4 $
                     ,event_func='wd_FANSFITFAST_PS_event' $
                    )


;wFunc = widget_combobox(wCBc,value=funcs, xsize=120, uname='FUNCS')
;widget_control, wFunc, set_combobox_select=fastfunc   ; select correct function

;voID = widget_label(wCBa,value='Fitted Parameters:',/dynamic_resize)
;vInfo = widget_info(voID,/geometry)
;wParams = widget_text(wCBa,value=paramsString,scr_xsize=0.98*(xSize-vInfo.xsize))


void = widget_button(wCBd,value='Manual',uname='man',sensitive=0,xsize=100)
void = widget_button(wCBd,value='Perform Fit',uname='DoFit',xsize=100)
void = widget_button(wCBd,value='Accept Fit',uname='Accept',xsize=100)
void = widget_button(wCBd,value='Discard Fit',uname='Discard',xsize=100)


; realise widgets
widget_control, wTLB, /realize

tlbInfo = widget_info(wTLB,/geometry)
restOfHeight = tlbInfo.scr_ysize - ySize
restOfWidth = tlbInfo.scr_xsize - xSize

; create a pixmap window with same size as visible window
; and note its number
window,/free,/pixmap,xsize=xsize,ysize=ysize
winPix = !d.window 

; get the visible window nos and set it as default plot window;
widget_control, wVis, get_value=winVis;,sensitive=1
wset, winVis

; define state structure
;energy = (*(*dataPtr).energyPtr)
;mskListPtr = (n_elements(maskedList) gt 0)? ptr_new(maskedList) : ptr_new()
state = {wTLB:wTLB $
        ,wPS:wPS $ 
;        ,paramsPtr:ptr_new(params) $ ; ptr to fitted parameters
        ,wVis:wVis $   ; visible plot window widget ID
        ,winVis:winVis $       ; visible window number
        ,winPix:winPix $       ; pixmap window number
        ,winWidth:xsize $      ; visible/pixmap window width
        ,winHeight:ysize $     ; visible/pixmap window height
        ,restOfHeight:restOfHeight $ ; height of tlb - plot window height
        ,restofWidth:restofWidth $   ; width of tlb - plot window width
        ,minXSize:0.5*xSize $      ; minimum width for plot window: 0.5 of initial value 
        ,minYSize:0.5*ySize $      ; minimum plotwindow height: 0.5 of initial value
        ,dataPtr:dataPtr $     ; ptr to dataset
        ,zp1:[0.0,0.0] $       ; zoom selection - first point
        ,zp2:[0.0,0.0] $       ; zoom selection - second point
        ,mouseType:0 $         ; mouse, 1==left, 2==middle, 4==right
;        ,fitfunc:fastfunc $
        ,xunits:xunits    $         ; 0=>meV, 1=>cm-1
        ,mev2wnos:8.065541        $   ; meV to wavenumber conversion factor
        ,accept:0 $
        ,tool:oTool $
        ,props:oProps $
        ,oldrgb:oldrgb $       ; rbg values outside fans_datreduc
        ,olddc:olddc   $       ; previous device decompose setting
        ,colors:colors $       ; colors within fans_datreduc from a custom 16-color table 
        }
sPtr = ptr_new(state,/no_copy)
widget_control, wTLB, set_uvalue=sPtr

paramsZero = (params[0] eq 0.0 && params[1] eq 0.0) 
if (paramsZero) then begin
   wd_fansfitfast_fitData, sPtr   ; fit data
   wd_fansfitfast_plotData, sPtr   ; Plot data
   widget_control, wPS, /refresh_property
endif else begin
   wd_fansfitfast_plotData, sPtr   ; Plot data
endelse

widget_control, wTLB, kill_notify='wd_fansfitfast_cleanup'
xmanager, 'wd_fansfitfast',wTLB, /no_block

;help, maskList

end



;===============================================================================
pro fans_funcProps::GetProperty, p0=p0,p1=p1,p2=p2,p3=p3,p4=p4,p5=p5,defn=defn,nparams=np $
                               , func=funcindex,xunit=xunit,params=params,_REF_EXTRA=etc
compile_opt idl2
;
if (arg_present(p0)) then p0 = Self.params[0]
;
if (arg_present(p1)) then p1 = Self.params[1]
;
if (arg_present(p2)) then p2 = Self.params[2]
;
if (arg_present(p3)) then p3 = Self.params[3]
;
if (arg_present(p4)) then p4 = Self.params[4]
;
if (arg_present(p5)) then p5 = Self.params[5]
;
if (arg_present(defn)) then defn = Self.defn
;
if (arg_present(funcIndex)) then funcIndex = Self.func
;
if (arg_present(xunit)) then xunit = Self.xunit
;
if (arg_present(params)) then params = Self.params
;
if (arg_present(np)) then np = Self.np

if (n_elements(etc) gt 0) then Self->IDLitComponent::GetProperty, _EXTRA=etc

end



;===============================================================================
pro fans_funcProps::SetProperty, p0=p0,p1=p1,p2=p2,p3=p3,p4=p4,p5=p5 $
                               , func=funcindex, params=params, _EXTRA=etc
compile_opt idl2
;
if (n_elements(p0) gt 0) then Self.params[0] = p0 
;
if (n_elements(p1) gt 0) then Self.params[1] = p1 
;
if (n_elements(p2) gt 0) then Self.params[2] = p2 
;
if (n_elements(p3) gt 0) then Self.params[3] = p3 
;
if (n_elements(p4) gt 0) then Self.params[4] = p4 
;
if (n_elements(p5) gt 0) then Self.params[5] = p5
;
if (n_elements(xunit) gt 0) then Self.xunit=xunit
;
if (n_elements(params) gt 0) then begin
   for i=0,n_elements(params)-1 do Self.params[i]=params[i]
endif
;
if (n_elements(funcIndex) gt 0) then begin
   Self.func = funcIndex
   func = Self.funcs[funcIndex]
   Self.defn = Self.defns[funcIndex]
   case strlowcase(func) of
      'linear': begin
         Self->SetPropertyAttribute, 'p2', hide=1
         Self->SetPropertyAttribute, 'p3', hide=1
         Self->SetPropertyAttribute, 'p4', hide=1
         Self.np = 2
      end
      'quadratic': begin
         Self->SetPropertyAttribute, 'p2', hide=0
         Self->SetPropertyAttribute, 'p3', hide=1
         Self->SetPropertyAttribute, 'p4', hide=1    
         Self.np = 3
      end
      'cubic': begin
         Self->SetPropertyAttribute, 'p2', hide=0
         Self->SetPropertyAttribute, 'p3', hide=0
         Self->SetPropertyAttribute, 'p4', hide=1    
         Self.np = 4
      end
      'arcsin': begin
         Self->SetPropertyAttribute, 'p2', hide=0
         Self->SetPropertyAttribute, 'p3', hide=0
         Self->SetPropertyAttribute, 'p4', hide=0
         Self.np = 6
      end
      else:
   endcase

endif 

if (n_elements(etc) gt 0) then Self->IDLitComponent::SetProperty, _EXTRA=etc

end


;===============================================================================
function fans_funcProps::Init, params=params, _EXTRA=etc
compile_opt idl2

; initilize base class
if (~Self->IDLitComponent::Init(_EXTRA=etc)) then return, 0

;if (~ptr_valid(sPtr)) then return, 0
;Self.sPtr = sPtr

Self.funcs = ['Arcsin','Linear','Quadratic','Cubic']

Self.defns = ["p0 + p1/x + p2/x^2 + p3*ASIN(SQRT(20.45/x)/p4)" $
             ,"p0 + p1*x" $
             ,"p0 + p1*x + p2*x^2" $
             ,"p0 + p1*x + p2*x^2 + p3x^3" $
             ]

Self.func = 0
Self.defn = Self.defns[0]

if (n_elements(params) gt 0) then begin
   for i=0,n_elements(params)-1 do Self.params[i]=params[i]
endif

Self->RegisterProperty, 'func', enumlist=Self.funcs, name='Function'
Self->RegisterProperty, 'defn', /string, name='Definition'
Self->RegisterProperty, 'p0',/float
Self->RegisterProperty, 'p1',/float
Self->RegisterProperty, 'p2',/float
Self->RegisterProperty, 'p3',/float
Self->RegisterProperty, 'p4',/float

Self->SetPropertyAttribute, 'NAME', /hide
Self->SetPropertyAttribute, 'DESCRIPTION', /hide

if (n_elements(etc) gt 0) then Self->SetProperty, _EXTRA=etc 

return, 1
end


;===============================================================================
pro fans_funcProps__define
compile_opt idl2
struct = {fans_funcProps  $
         ,inherits IDLitComponent $
         ,defn:'' $
         ,func:0 $
         ,np:0 $
         ,xunit:0 $
         ,params:fltarr(6) $
         ,funcs:['','','',''] $
         ,defns:['','','',''] $
;         ,sptr:ptr_new() $
         }
end
