; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEvisImage
;
; PURPOSE:
;   This file simply inherits and extends IDL's IDLitVisImage visualization class
;
; CATEGORY:
;   DAVE Visualization
;
; SUPERCLASSES:
;   IDLitVisImage
;
; SUBCLASSES:
;
; CREATION:
;
; METHODS:
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; September 2004
;-
;###############################################################################




;===============================================================================
; DAVEvisImage::EnsurePalette
; 
; PURPOSE:
;   This procedure method verifies that valid palette parameter is
;   associated with this image. If not, it creates one. Overides
;   IDLitVisImage::EnsurePalette and is essentially the same except
;   that any data that are automatically created are stored in the
;   system 'Data Manager' and not underneath the parent of the
;   'IMAGEPIXELS' parameter.
;
;   This method is only intended to be called if the image is a
;   single-channel image.
;
; PARAMETERS:
;   red [out]   - the red values for the palette object
;
;   green [out] - the green values for the palette object
;
;   blue [out]  - the blue values for the palette object
;
; KEYWORDS:
;   none
;
pro DAVEvisImage::EnsurePalette, red, green, blue
compile_opt idl2, hidden

oPalette = self->IDLitParameter::GetParameter('PALETTE')
;; Do we have a valid palette?
if (OBJ_VALID(oPalette) && oPalette->GetData(palette)) then begin
    if (ARG_PRESENT(red)) then begin
        red = REFORM(palette[0,*])
        green = REFORM(palette[1,*])
        blue = REFORM(palette[2,*])
    endif
    return
endif

; If no palette exists, create a custom palette from IDL color table
; 13 (rainbow). The rgb values are extracted from a temporal
; IDLgrPalette object.
oVoidPalette = obj_new('IDLgrPalette')
oVoidPalette->LoadCT,13         ; use color table 13 - rainbow
oVoidPalette->GetProperty, red_values=red,green_values=green,blue_values=blue
obj_destroy,oVoidPalette

oRainbowPalette = OBJ_NEW('IDLitDataIDLPalette', $
                       TRANSPOSE([[red],[green],[blue]]), $
                       NAME='Palette')
oRainbowPalette->SetProperty,/AUTO_DELETE ; automatically deleted data_observers < 1
success = self->IDLitParameter::SetData(oRainbowPalette, $
                                        PARAMETER_NAME='PALETTE');, /NO_UPDATE)
;; Send a notification message to update UI
self->DoOnNotify, self->GetFullIdentifier(),"ADDITEMS", ''

oPalette = self->GetParameter('PALETTE')

; Add to the system 'Data Manager'
self->AddByIdentifier,'/DATA MANAGER',oPalette

self->SetPropertyAttribute, 'VISUALIZATION_PALETTE', SENSITIVE=1

end



;===============================================================================
; DAVEvisImage::EnsureXYParameters
; 
; PURPOSE:
;   Ensure that X and Y parameters exist, based on the image dimensions,
;   origin and pixel size. Overides IDLitVisImage::EnsureXYParameters and is
;   essentially the same except that any data that are automatically
;   created are stored in the system 'Data Manager' and not underneath
;   the parent of the 'IMAGEPIXELS' parameter.
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
pro DAVEvisImage::EnsureXYParameters

paramNames = ['X', 'Y']
for i=0,1 do begin
    ;; Check if parameter already exists.
    oParam = self->GetParameter(paramNames[i])
    if ~obj_valid(oParam) then begin
        ;; Create and set the parameter.
        data = DINDGEN(self._gridDims[i])*self._userStep[i] + $
               self._userOrigin[i]
        oData = OBJ_NEW('IDLitDataIDLVector', data, NAME=paramNames[i])
        oData->SetProperty,/AUTO_DELETE
        self->SetParameter, paramNames[i], oData, /NO_UPDATE

        ;; Add to data manager.
        oData = self->GetParameter(paramNames[i])
        self->AddByIdentifier,'/DATA MANAGER',oData
    endif
endfor

; Send a notification message to update UI
self->DoOnNotify, self->GetFullIdentifier(),"ADDITEMS", ''

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEvisImage::OnDataChangeUpdate
; 
; PURPOSE:
;   Extend IDLitVisImage::OnDataChangeUpdate.
;   This method is called when the data associated with a parameter
;   has changed. When called, the visualization performs the
;   desired updates to reflect the change.
;
; PARAMETERS:
;   oSubject - The data item for the parameter that has changed.
;
;   parmName - The name of the parameter that has changed.
;
; KEYWORDS:
;   none
;
pro DAVEvisImage::OnDataChangeUpdate, oSubject, parmName, skipBaseClass=skipBaseClass
compile_opt idl2

; Call base class method
if (~keyword_set(skipBaseClass)) then $
  self->IDLitVisImage::OnDataChangeUpdate, oSubject, parmName

; if (self.newlyCreated) then begin
;     self.newlyCreated = 0
;     ;; Request box axes.
;     ;;self->SetAxesStyleRequest, 2
;     oDataspace = self->GetDataspace()
;     oAxesCon = (obj_valid(oDataSpace))? oDataspace->GetAxes(/container) : obj_new()
;     if (obj_valid(oAxesCon)) then oAxesCon->SetProperty, style=2
; endif

; NB. The implementation of the base class method is such that
; recursive calls to it are explicit for <parameter_set>
; (ie called as self->IDLitVisPlot::OnDataChangeUpdate from within the
; base class method). Thus this method will never be recursively
; called. Thus we have to handle this situation again below.

;; Perform additional tasks not handled by the base class method
case strupcase(parmName) of
    '<PARAMETER SET>':begin
        ;; Get our data
        position = oSubject->Get(/all, count=nCount, name=name)
        for i=0, nCount-1 do begin
            oData = oSubject->GetByName(name[i])
            if (~obj_valid(oData)) then continue
            
            ;; Make a recursive call but skip base class call in other
            ;; to handle cases below
            self->OnDataChangeUpdate, oData, name[i], /skipBaseClass
        endfor
    end

    'IMAGEPIXELS': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label
            zLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then zLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              zLabel = (strlen(units) gt 0)? zLabel + ' ('+units+')' : zLabel
            
            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)
            
            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,ztitle=strtrim(zLabel)
        endif
        ;self->SetProperty, /pal_color ; Use palette
    end
    
    'Y': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label
            yLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then yLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              yLabel = (strlen(units) gt 0)? yLabel + ' ('+units+')' : yLabel
            
            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)
            
            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,ytitle=strtrim(yLabel)
        endif
    end
    
    'X': begin
        help,self
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; X label
            xLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then xLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              xLabel = (strlen(units) gt 0)? xLabel + ' ('+units+')' : xLabel

            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)

            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,xtitle=strtrim(xLabel)
            
        endif                    
    end
    
    else:
endcase

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEvisImage::Init
; 
; PURPOSE:
;   Register visualization parameters
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
function DAVEvisImage::Init, _REF_EXTRA=etc

compile_opt idl2

; Call base class init method
if (~self->IDLitVisImage::Init(_EXTRA=etc)) then return, 0

; Set sensible default properties
self->SetProperty, interpolate=1 ; bilinear interpolation

; Request box axes.
self->SetAxesRequest, 1, /always
self->SetAxesStyleRequest, 2

;self.newlyCreated = 1

return, 1
end


;===============================================================================
; DAVEvisImage__define
; 
; PURPOSE:
;   DAVEvisImage class structure definition
;
pro DAVEvisImage__define

compile_opt idl2

struc = {DAVEvisImage, $
         inherits IDLitVisImage $
;         ,newlyCreated:0b $
        }

end
