; $Id$
;###############################################################################
;+
; NAME:
;   SPINSPSDCalibPreferences
;
; PURPOSE:
;   Retrieve or save user preferences for SPINS PSD Data Reduction module
;
; CATEGORY:
;   SPINS Data Reduction

; PARAMETERS:
;
; KEYWORDS:
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; November 2011
;-
;###############################################################################




;===============================================================================
;+
; NAME:
;   SPINSPSDCalibPreferences
;
; PURPOSE:
;   Main function
;
; PARAMETERS
;   preferences [in|out] - the prefrences structure to be save or updated if passed in or
;                          that was created or updated on exit.
; KEYWORDS:
;   save [in] - force the input preferences structure to be saved to disk
;   
;   reset [in] - force the preferences structure to be reset to their default values and then 
;                saved to disk
;   
;   updateVersion [in] - force a check of the preferenceVersion field of the input preferences
;                        structure. If it is not up to date, then update preferences structure
;                        to the current definition.
;
; RETURN:
;   Return values are 1 if successful or 0 otherwise
;-
function SPINSPSDCalibPreferences, preferences, save=save, reset=reset, updateVersion=updateVersion
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        print,'SPINSPSDCalibPreferences: Error Encountered'
        print,!ERROR_STATE.MSG
        catch, /cancel
        return, 0
    endif
endif

; ** Version number **
; ** Increment by 1 whenever the preferences structure content changes **
; Ver 1  : Initial
; Ver 2  : Add instrumentName
; Ver 3  : Add A4Value and A6Value
; Ver 4  : Add maskThreshold, fill_background, fill_color, fill_level, fill_transparency

versionNumber = 4

; Retrieve the app directory
appDir = getDAVEConfigDir()
if (n_elements(appDir) le 0) then return, 0
if (~file_test(appDir,/write,/read)) then return, 0
prefFile = appDir+path_sep()+'TASPSDCalibPreferences.txt'

preferenceIsDefined = strcmp(size(preferences,/tname),'STRUCT')

; this updateVersion variable supercedes the updateVersion keyword but the keyword is retain for clarity of use!
updateVersion = 0
if (preferenceIsDefined) then begin
   ; old preferences structures may be missing the preferenceVersion field!
   oldTags = tag_names(preferences)
   void = where(strcmp(oldTags,'preferenceVersion',/fold),prefVersionIsDefined)
   oldVersion = (prefVersionIsDefined)? fix(preferences.preferenceVersion) : 0
   updateVersion = (versionNumber ne oldVersion)?  1 : 0
endif

if (keyword_set(save) && preferenceIsDefined) then begin
   ; save and exit
   ntags = n_tags(preferences)
   tags = tag_names(preferences)
   if (ntags lt 1) then return, 0

   openw, lun, prefFile, /get_lun
   ;printf, lun, 'PreferenceVersion : ',strtrim(string(versionNumber),2)
   for i=0,ntags-1 do printf, lun, tags[i] + ' : '+preferences.(i)
   
   close, lun, /force
   
   return, 1
endif

; Check version number
fileVersionNumber = 0
if (file_test(prefFile,/read)) then begin
   buffer = ''
   openr, lun, prefFile, /get_lun
   readf, lun, buffer
   close, lun, /force

   toks = strtrim(strsplit(buffer[0],':',count=ntok,/extract), 2)
   if (ntok eq 2) then fileVersionNumber = fix(toks[1])
endif
if (versionNumber ne fileVersionNumber) then reset = 1   ; force a reset of the preference file

if (keyword_set(reset) || ~file_test(prefFile,/read) || keyword_set(updateVersion)) then begin
   ; Set preferences to their default values, save to file and return to caller
   ; Also update structure if required and return to caller
   defPreferences = {PreferenceVersion:     strtrim(string(versionNumber),2) $
                  ,instrumentName:        '0'           $               ; 0=BT7, 1=SPINS
                  ,ekFac:                 '2.072194'    $
                  ,logFlag:               '0'           $
                  ,normalizeTo:           '0'           $
                  ,anaModeFlag:           '0'           $               ; Analyzer mode: 0='Flat', 1='See Through', 2='Fixed Ef'
                  ,a4Flag:                '0'           $               ; how to determine a4: 0=Calculation, 1=Measurement
                  ,plotFlag:              '0'           $               ; View one of 0=Raw Calib, 1=Raw A4Calib, 2=Ef  3=dA4  4=Sens
                  ,dpdefCorFlag:          '0'           $               ; Apply dpdEf correction? 0=No, 1=Yes
                  ,theta_gValue:          '0.0'         $               ; Required when calculating delta A4 in Fixed Ef mode
                  ,pixelNumber:           '0'           $               ; specifies which pixel to view when 'plotFlag' is set to examine fits
                  ,thetagValue:           '0.0'         $               ; Required when calculating delta A4 in Fixed Ef mode
                  ,defaultIntensityFlag:  '1'           $               ; use default intensity range 1 = yes, 0 = no
                  ,A4Value:               '0.0'         $
                  ,A6Value:               '0.0'         $
                  ,maskThreshold:         '5'           $
                  ,s_errorbar_capsize:    '0.1'         $
                  ,s_transparency:        '50'          $
                  ,s_antialias:           '3'           $
                  ,s_color:               '0,0,255'     $
                  ,s_errorbar_color:      '0,0,255'     $
                  ,s_use_default_color:   '1'           $
                  ,s_sym_increment:       '1'           $
                  ,s_sym_index:           '4'           $
                  ,s_sym_size:            '1.25'        $               ; 
                  ,s_sym_color:           '0,0,255'     $               ;
                  ,s_sym_thick:           '1.0'         $               ;
                  ,s_y_errorbars:         '1'           $               ; show y errors
                  ,s_linestyle:           '6'           $               ;
                  ,s_nsum:                '1'           $               ; no point everaging
                  ,s_thick:               '3.0'         $               ; line thickness
                  ,s_sym_filled:          '1'           $               ; use filled symbols
                  ,s_sym_fill_color:      '0,0,255'     $               ; symbol fill color
                  ,s_font_name:           'Helvetica'   $               ;
                  ,s_font_size:           '13'          $               ;
                  ,s_text_color:          '0,0,0'       $               ;
                  ,b_errorbar_capsize:    '0.1'         $
                  ,b_transparency:        '50'          $
                  ,b_antialias:           '3'           $
                  ,b_color:               '0,255,255'   $
                  ,b_errorbar_color:      '0,255,255'   $
                  ,b_use_default_color:   '1'           $
                  ,b_sym_increment:       '1'           $
                  ,b_sym_index:           '0'           $
                  ,b_sym_size:            '1.25'        $               ; 
                  ,b_sym_color:           '0,255,255'   $               ;
                  ,b_sym_thick:           '1.0'         $               ;
                  ,b_y_errorbars:         '1'           $               ; show y errors
                  ,b_linestyle:           '0'           $               ;
                  ,b_nsum:                '1'           $               ; no point everaging
                  ,b_thick:               '3.0'         $               ; line thickness
                  ,b_sym_filled:          '1'           $               ; use filled symbols
                  ,b_sym_fill_color:      '0,255,255'   $               ; symbol fill color
                  ,b_font_name:           'Helvetica'   $               ;
                  ,b_font_size:           '13'          $               ;
                  ,b_text_color:          '0,0,0'       $               ;
                  ,fill_background:       '1'           $
                  ,fill_level:            '0.0'         $
                  ,fill_color:            '128,128,128' $
                  ,fill_transparency:     '50'          $
                 }

   ntags = n_tags(defPreferences)
   tags = tag_names(defPreferences)

   if (keyword_set(reset) || ~file_test(prefFile,/read)) then begin
      openw, lun, prefFile, /get_lun
      ;printf, lun, 'PreferenceVersion : ',strtrim(string(versionNumber),2)
      for i=0,ntags-1 do printf, lun, tags[i] + ' : '+defPreferences.(i)
      
      close, lun, /force
   endif
   if (keyword_set(updateVersion)) then begin
      ; if the version number of the input preference structure is not current, then
      ; we need to update the input structure to match the current definition by taking the
      ; default preferences structure and updating any matching fields with values from the input structure
      oTags = tag_names(preferences)
      oNtags = n_tags(preferences)
      for i = 0,oNtags-1 do begin   
         if (strcmp(oTags[i],'PreferenceVersion',/fold)) then continue ; ignore the old preferenceVersion field
         index = where(strcmp(tags,oTags[i]), found)
         if (found) then defPreferences.(index) = preferences.(i)
      endfor
   endif
   preferences = defPreferences
endif else begin
   ; Retrieve the current preferences from file and return to caller
   lines = file_lines(prefFile)
   buffer = strarr(lines)
   openr, lun, prefFile, /get_lun
   readf, lun, buffer
   close, lun, /force
   
   for i=0,lines-1 do begin
      toks = strtrim(strsplit(buffer[i],':',count=ntok,/extract), 2)
      ;toks = strsplit(buffer[i],':',count=ntok,/extract)
      if (ntok gt 1) then begin
         value = (ntok eq 2)? toks[1] : ''
         preferences = (i eq 0)? create_struct(toks[0],value) : $
            create_struct(preferences,toks[0],value)
      endif
   endfor
endelse

return, 1

end

