
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataUserdefined
;
; PURPOSE:
;   A simple operation that rebins the plottable data of an
;   IDLitData object. The plottable data must consist of an dependent
;   component with at least one independent component.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataUserdefined::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataUserdefined::GetProperty, xvals0=xvals0,yvals0=yvals0,zvals0=zvals0,evals0=evals0,xlab0=xlab0,ylab0=ylab0 $
  ,zlab0=zlab0,xunit0=xunit0,yunit0=yunit0,zunit0=zunit0 $
  ,xvals1=xvals1,yvals1=yvals1,zvals1=zvals1,evals1=evals1,xlab1=xlab1,ylab1=ylab1,zlab1=zlab1,xunit1=xunit1,yunit1=yunit1,zunit1=zunit1 $
  ,_REF_EXTRA=etc
compile_opt idl2

if (arg_present(xvals0)) then xvals0 = self.xvals0
if (Arg_present(yvals0)) then yvals0 = self.yvals0
if (Arg_present(zvals0)) then zvals0 = self.zvals0
if (Arg_present(evals0)) then evals0 = self.evals0
if (Arg_present(xlab0)) then xlab0 = self.xlab0
if (Arg_present(ylab0)) then ylab0 = self.ylab0
if (Arg_present(zlab0)) then zlab0 = self.zlab0
if (Arg_present(xunit0)) then xunit0 = self.xunit0
if (Arg_present(yunit0)) then yunit0 = self.yunit0
if (Arg_present(zunit0)) then zunit0 = self.zunit0

if (Arg_present(xvals1)) then xvals1 = self.xvals1
if (Arg_present(yvals1)) then yvals1 = self.yvals1
if (Arg_present(zvals1)) then zvals1 = self.zvals1
if (Arg_present(evals1)) then evals1 = self.evals1
if (Arg_present(xlab1)) then xlab1 = self.xlab1
if (Arg_present(ylab1)) then ylab1 = self.ylab1
if (Arg_present(zlab1)) then zlab1 = self.zlab1
if (Arg_present(xunit1)) then xunit1 = self.xunit1
if (Arg_present(yunit1)) then yunit1 = self.yunit1
if (Arg_present(zunit1)) then zunit1 = self.zunit1

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataUserdefined::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataUserdefined::SetProperty,xvals0=xvals0,yvals0=yvals0,zvals0=zvals0,evals0=evals0,xlab0=xlab0,ylab0=ylab0 $
  ,zlab0=zlab0,xunit0=xunit0,yunit0=yunit0,zunit0=zunit0 $
  ,xvals1=xvals1,yvals1=yvals1,zvals1=zvals1,evals1=evals1,xlab1=xlab1,ylab1=ylab1,zlab1=zlab1,xunit1=xunit1,yunit1=yunit1,zunit1=zunit1 $
  ,_EXTRA=etc
compile_opt idl2

; Call base class mutator
if(N_elements(etc) gt 0) then $
  self->Idlitoperation::setproperty, _EXTRA=etc

oTool = self->GetTool()
title = 'User Defined Transform'
oParent = getmaindataobjectparent(self.oTarget)
if (~obj_valid(oParent)) then return

oParent->Getproperty, axis1Value=x, nDimensions=ndim, dataValue=z, errorValue=e
if (ndim eq 2) then oParent->Getproperty, axis2Value=y else y = !null

; evaluate the expression and and ensure they evaluate without errors before retaining
if (N_elements(xvals1) && ((xvals1.Trim()).Strlen() gt 0)) then begin
  ; verify that the expression can execute without error and notify the user if not
  if (~execute('void = '+xvals1,1)) then begin
    msg = ['The specified expression:',xvals1,'has a syntax/logical error!','Please correct it!','Resetting to an empty string']
    oTool->ErrorMessage, msg, severity=0, title=title
    Self.xvals1 = ''
  endif else self.xvals1=xvals1 
endif
if (N_elements(yvals1) && ((yvals1.Trim()).Strlen() gt 0)) then begin
  ; verify that the expression can execute without error and notify the user if not
  if (~Execute('void = '+yvals1,1)) then begin
    msg = ['The specified expression:',yvals1,'has a syntax/logical error!','Please correct it!','Resetting to an empty string']
    oTool->Errormessage, msg, severity=0, title=title
    Self.yvals1 = ''
  endif else self.yvals1=yvals1
endif

if (N_elements(zvals1) && ((zvals1.Trim()).Strlen() gt 0)) then begin
  ; verify that the expression can execute without error and notify the user if not
  if (~Execute('void = '+zvals1,1)) then begin
    msg = ['The specified expression:',zvals1,'has a syntax/logical error!','Please correct it!','Resetting to an empty string']
    oTool->Errormessage, msg, severity=0, title=title
    Self.zvals1 = ''
  endif else self.zvals1=zvals1
endif

if (N_elements(evals1) && ((evals1.Trim()).Strlen() gt 0)) then begin
  ; verify that the expression can execute without error and notify the user if not
  if (~Execute('void = '+evals1,1)) then begin
    msg = ['The specified expression:',evals1,'has a syntax/logical error!','Please correct it!','Resetting to an empty string']
    oTool->Errormessage, msg, severity=0, title=title
    Self.evals1 = ''
  endif else self.evals1=evals1
endif

; these are not modifiable from the UI
if (N_elements(xvals0) ) then self.xvals0=xvals0
if (N_elements(yvals0) ) then self.yvals0=yvals0
if (N_elements(zvals0) ) then self.zvals0=zvals0
if (N_elements(xlab0) ) then self.xlab0=xlab0
if (N_elements(ylab0) ) then self.ylab0=ylab0
if (N_elements(zlab0) ) then self.zlab0=zlab0
if (N_elements(xunit0) ) then self.xunit0=xunit0
if (N_elements(yunit0) ) then self.yunit0=yunit0
if (N_elements(zunit0) ) then self.zunit0=zunit0

; these are modifiable from the UI
; any value is acceptable including an empty string
if (N_elements(xlab1) && xlab1.strlen() gt 0) then self.xlab1=xlab1
if (N_elements(ylab1) && ylab1.strlen() gt 0) then self.ylab1=ylab1
if (N_elements(zlab1) && zlab1.strlen() gt 0) then self.zlab1=zlab1
if (N_elements(xunit1) && xunit1.strlen() gt 0) then self.xunit1=xunit1
if (N_elements(yunit1) && yunit1.strlen() gt 0) then self.yunit1=yunit1
if (N_elements(zunit1) && zunit1.strlen() gt 0) then self.zunit1=zunit1
;if (N_elements()) then self.=

; Update the current bin settings as they depend on the ind axis data
;if (~keyword_set(no_update)) then self->InitializeAttributes


end


;===============================================================================
; DAVEopDataUserdefined::InitializeAttributes
; 
; PURPOSE:
;   
;
; PARAMETERS:
;
; KEYWORDS:
;
;
pro DAVEopDataUserdefined::InitializeAttributes
compile_opt idl2


end



;===============================================================================
; DAVEopDataUserdefined::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataUserdefined::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values, This is the first one
; that is created for this oCmdSet
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

oParent = getmaindataobjectparent(oTarget)

;Retrieve details from target dataset and use that to initialize the current operation properties
oParent->Getproperty, nDimensions=ndim, axis1Label=xLab, axis1Units=xUnit, dataLabel=zLab, dataUnits=zUnit

Self.xvals0 = 'x'
Self.zvals0 = 'z'
Self.evals0 = 'e'
Self.xlab0 = xLab
Self.xunit0=xUnit
Self.zlab0=zLab
Self.zunit0=zUnit

Self.xvals1 = 'x'
Self.zvals1 = 'z'
Self.evals1 = 'e'
Self.xlab1 = xLab
Self.xunit1=xUnit
Self.zlab1=zLab
Self.zunit1=zUnit

if (ndim eq 1) then begin
  ; 1D data has only one independent axis so hide reference to 2nd indep data
  self->SetPropertyAttribute, 'xvals0', description='Current indep var' ,name='Current indep var',sensitive=0
  self->SetPropertyAttribute, 'xlab0', description='Current indep label' ,name='Current indep label',sensitive=0
  self->SetPropertyAttribute, 'xunit0', description='Current indep unit' ,name='Current indep unit',sensitive=0
  self->SetPropertyAttribute, 'yvals0', sensitive=0, hide=1
  self->SetPropertyAttribute, 'ylab0', sensitive=0, hide=1
  self->SetPropertyAttribute, 'yunit0', sensitive=0, hide=1

  self->SetPropertyAttribute, 'xvals1', description='Indep var expression' ,name='Indep var expression',sensitive=1
  self->SetPropertyAttribute, 'xlab1', description='Indep var label' ,name='Indep var label',sensitive=1
  self->SetPropertyAttribute, 'xunit1', description='Indep var unit' ,name='Indep var unit',sensitive=1
  self->SetPropertyAttribute, 'yvals1', sensitive=0, hide=1
  self->SetPropertyAttribute, 'ylab1', sensitive=0, hide=1
  self->SetPropertyAttribute, 'yunit1', sensitive=0, hide=1
endif else begin
  ; 2D data so show 2nd independent axis properties
  self->Setpropertyattribute, 'xvals0', description='Current 1st indep var' ,name='Current 1st indep var',sensitive=0
  self->Setpropertyattribute, 'xlab0', description='Current 1st indep label' ,name='Current 1st indep label',sensitive=0
  self->Setpropertyattribute, 'xunit0', description='Current 1st indep unit' ,name='Current 1st indep unit',sensitive=0
  self->Setpropertyattribute, 'yvals0', sensitive=0, hide=0
  self->Setpropertyattribute, 'ylab0', sensitive=0, hide=0
  self->Setpropertyattribute, 'yunit0', sensitive=0, hide=0

  self->Setpropertyattribute, 'xvals1', description='1st Indep var expression' ,name='1st Indep var expression',sensitive=1
  self->Setpropertyattribute, 'xlab1', description='1st Indep var label' ,name='1st Indep var label',sensitive=1
  self->Setpropertyattribute, 'xunit1', description='1st Indep var unit' ,name='1st Indep var unit',sensitive=1
  self->Setpropertyattribute, 'yvals1', sensitive=1, hide=0
  self->Setpropertyattribute, 'ylab1', sensitive=1, hide=0
  self->Setpropertyattribute, 'yunit1', sensitive=1, hide=0

  oParent->Getproperty, axis2Label=yLab, axis2Units=yUnit
  Self.yvals0 = 'y'
  Self.ylab0 = yLab
  Self.yunit0=yUnit
  Self.yvals1 = 'y'
  Self.ylab1 = yLab
  Self.yunit1=yUnit
  void = oCmd->Additem('OLD_YVALS0',Self.yvals0)
  void = oCmd->Additem('OLD_YLAB0',Self.ylab0)
  void = oCmd->Additem('OLD_YUNIT0',Self.yunit0)
  void = oCmd->Additem('OLD_YVALS1',Self.yvals1)
  void = oCmd->Additem('OLD_YLAB1',Self.ylab1)
  void = oCmd->Additem('OLD_YUNIT1',Self.yunit1)
endelse

void = oCmd->Additem('NDIM',ndim)
void = oCmd->Additem('OLD_XVALS0',Self.xvals0)
void = oCmd->Additem('OLD_ZVALS0',Self.zvals0)
void = oCmd->Additem('OLD_EVALS0',Self.evals0)
void = oCmd->Additem('OLD_XLAB0',Self.xlab0)
void = oCmd->Additem('OLD_ZLAB0',Self.zlab0)
void = oCmd->Additem('OLD_XUNIT0',Self.xunit0)
void = oCmd->Additem('OLD_ZUNIT0',Self.zunit0)
void = oCmd->Additem('OLD_XVALS1',Self.xvals1)
void = oCmd->Additem('OLD_ZVALS1',Self.zvals1)
void = oCmd->Additem('OLD_EVALS1',Self.evals1)
void = oCmd->Additem('OLD_XLAB1',Self.xlab1)
void = oCmd->Additem('OLD_ZLAB1',Self.zlab1)
void = oCmd->Additem('OLD_XUNIT1',Self.xunit1)
void = oCmd->Additem('OLD_ZUNIT1',Self.zunit1)


; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataUserdefined::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataUserdefined::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty,xvals1=xvals1,yvals1=yvals1,zvals1=zvals1,evals1=evals1 $
  ,xlab1=xlab1,ylab1=ylab1,zlab1=zlab1,xunit1=xunit1,yunit1=yunit1,zunit1=zunit1

void = oCmd->AddItem('NEW_XVALS1',xvals1)
void = oCmd->AddItem('NEW_YVALS1',yvals1)
void = oCmd->AddItem('NEW_ZVALS1',zvals1)
void = oCmd->AddItem('NEW_EVALS1',evals1)
void = oCmd->AddItem('NEW_XLAB1',xlab1)
void = oCmd->Additem('NEW_YLAB1',ylab1)
void = oCmd->Additem('NEW_ZLAB1',zlab1)
void = oCmd->Additem('NEW_XUNIT1',xunit1)
void = oCmd->Additem('NEW_YUNIT1',yunit1)
void = oCmd->Additem('NEW_ZUNIT1',zunit1)

return, 1

end


;===============================================================================
; DAVEopDataUserdefined::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataUserdefined::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataUserdefined::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataUserdefined::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0
; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as
; axes data or objects and the treatment buffer.
oParent = Getmaindataobjectparent(Self.oTarget)


; Undo the changes that were made to the target
void = oCmds[1]->GetItem('OLD_X',x)
void = oCmds[1]->Getitem('NDIM',ndim)
void = oCmds[1]->Getitem('OLD_XLAB',xlab)
void = oCmds[1]->Getitem('OLD_XUNIT',xunit)

void = oCmds[1]->Getitem('OLD_Z',z)
void = oCmds[1]->Getitem('OLD_ZLAB',zlab)
void = oCmds[1]->GetItem('OLD_ZUNIT',zunit)
void = oCmds[1]->GetItem('OLD_TREATMENT',trmt)
oParent->Setproperty, axis1Value=x, axis1Label=xLab, axis1Units=xUnit, dataValue=z, dataLabel=zLab, dataUnits=zUnit, treatment=trmt

if (ndim eq 2) then begin
  void = oCmds[1]->Getitem('OLD_Y',y)
  void = oCmds[1]->Getitem('OLD_YLAB',yLab)
  void = oCmds[1]->Getitem('OLD_YUNIT',yUnit)
  oParent->Setproperty, axis2Value=y, axis2Label=yLab, axis2Units=yUnit
endif

void = oCmds[1]->GetItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmds[1]->GetItem('OLD_E',e)
    oParent->Setproperty, errorValue=e
endif

; then reset the operation properties to those before the operation was performed
void = oCmds[0]->GetItem('OLD_XVALS0',xvals0)
Self.xvals0 = xvals0
void = oCmds[0]->GetItem('OLD_ZVALS0',zvals0)
Self.zvals0 = zvals0
void = oCmds[0]->GetItem('OLD_EVALS0',evals0)
Self.evals0 = evals0
void = oCmds[0]->GetItem('OLD_XLAB0',xlab0)
Self.xlab0 = xlab0
void = oCmds[0]->GetItem('OLD_ZLAB0',zlab0)
Self.zlab0 = zlab0
void = oCmds[0]->GetItem('OLD_XUNIT0',xunit0)
Self.xunit0 = xunit0
void = oCmds[0]->GetItem('OLD_ZUNIT0',zunit0)
Self.zunit0 = zunit0
void = oCmds[0]->GetItem('OLD_XVALS1',xvals1)
Self.xvals1 = xvals1
void = oCmds[0]->GetItem('OLD_ZVALS1',zvals1)
Self.zvals1 = zvals1
void = oCmds[0]->GetItem('OLD_EVALS1',evals1)
Self.evals1 = evals1
void = oCmds[0]->GetItem('OLD_XLAB1',xlab1)
Self.xlab1 = xlab1
void = oCmds[0]->GetItem('OLD_ZLAB1',zlab1)
Self.zlab1 = zlab1
void = oCmds[0]->GetItem('OLD_XUNIT1',xunit1)
Self.xunit1 = xunit1
void = oCmds[0]->GetItem('OLD_ZUNIT1',zunit1)
Self.zunit1 = zunit1
if (ndim eq 2) then begin
  void = oCmds[0]->GetItem('OLD_YVALS0',yvals0)
  Self.yvals0 = yvals0
  void = oCmds[0]->GetItem('OLD_YLAB0',ylab0)
  Self.ylab0 = ylab0
  void = oCmds[0]->GetItem('OLD_YUNIT0',yunit0)
  Self.yunit0 = yunit0
  void = oCmds[0]->GetItem('OLD_YVALS1',yvals1)
  Self.yvals1 = yvals1
  void = oCmds[0]->GetItem('OLD_YLAB1',ylab1)
  Self.ylab1 = ylab1
  void = oCmds[0]->GetItem('OLD_YUNIT1',yunit1)
  Self.yunit1 = yunit1
endif
 
return, 1

end


;===============================================================================
; DAVEopDataUserdefined::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataUserdefined::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0
; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as
; axes data or objects and the treatment buffer.
oParent = Getmaindataobjectparent(Self.oTarget)

  
; then reset the operation properties to those before the operation was performed
void = oCmds[0]->Getitem('OLD_XVALS0',xvals0)
Self.xvals0 = xvals0
void = oCmds[0]->Getitem('OLD_ZVALS0',zvals0)
Self.zvals0 = zvals0
void = oCmds[0]->Getitem('OLD_EVALS0',evals0)
Self.evals0 = evals0
void = oCmds[0]->Getitem('OLD_XLAB0',xlab0)
Self.xlab0 = xlab0
void = oCmds[0]->Getitem('OLD_ZLAB0',zlab0)
Self.zlab0 = zlab0
void = oCmds[0]->Getitem('OLD_XUNIT0',xunit0)
Self.xunit0 = xunit0
void = oCmds[0]->Getitem('OLD_ZUNIT0',zunit0)
Self.zunit0 = zunit0
void = oCmds[0]->Getitem('NEW_XVALS1',xvals1)
Self.xvals1 = xvals1
void = oCmds[0]->Getitem('NEW_ZVALS1',zvals1)
Self.zvals1 = zvals1
void = oCmds[0]->Getitem('NEW_EVALS1',evals1)
Self.evals1 = evals1
void = oCmds[0]->Getitem('NEW_XLAB1',xlab1)
Self.xlab1 = xlab1
void = oCmds[0]->Getitem('NEW_ZLAB1',zlab1)
Self.zlab1 = zlab1
void = oCmds[0]->Getitem('NEW_XUNIT1',xunit1)
Self.xunit1 = xunit1
void = oCmds[0]->Getitem('NEW_ZUNIT1',zunit1)
Self.zunit1 = zunit1
void = oCmds[0]->Getitem('NDIM',ndim)
if (ndim eq 2) then begin
  void = oCmds[0]->Getitem('OLD_YVALS0',yvals0)
  Self.yvals0 = yvals0
  void = oCmds[0]->Getitem('OLD_YLAB0',ylab0)
  Self.ylab0 = ylab0
  void = oCmds[0]->Getitem('OLD_YUNIT0',yunit0)
  Self.yunit0 = yunit0
  void = oCmds[0]->Getitem('NEW_YVALS1',yvals1)
  Self.yvals1 = yvals1
  void = oCmds[0]->Getitem('NEW_YLAB1',ylab1)
  Self.ylab1 = ylab1
  void = oCmds[0]->Getitem('NEW_YUNIT1',yunit1)
  Self.yunit1 = yunit1
endif


; then redo the actual data modification
; Get various stored properties and restore the operation with them
if (oCmds[1]->GetItem('NEW_X',_newx_)) then oParent->Setproperty, axis1Value=_newx_
if (oCmds[1]->GetItem('NEW_XLAB',xlab1)) then oParent->Setproperty, axis1Label=xlab1
if (oCmds[1]->GetItem('NEW_XUNIT',xUnit1)) then oParent->Setproperty, axis1Units=xUnit1
if (oCmds[1]->GetItem('NEW_Y',_newy_)) then oParent->Setproperty, axis2Value=_newy_
if (oCmds[1]->GetItem('NEW_YLAB',ylab1)) then oParent->Setproperty, axis2Label=ylab1
if (oCmds[1]->GetItem('NEW_YUNIT',yUnit1)) then oParent->Setproperty, axis2Units=yUnit1
if (oCmds[1]->GetItem('NEW_Z',_newz_)) then oParent->Setproperty, dataValue=_newz_
if (oCmds[1]->GetItem('NEW_ZLAB',zlab1)) then oParent->Setproperty, dataLabel=zlab1
if (oCmds[1]->GetItem('NEW_ZUNIT',zUnit1)) then oParent->Setproperty, dataUnits=zUnit1
if (oCmds[1]->GetItem('NEW_E',_newe_)) then oParent->Setproperty, errorValue=_newe_
if (oCmds[1]->GetItem('NEW_TREATMENT',trmt)) then oParent->Setproperty, treatment=trmt

return, 1

end


;===============================================================================
; DAVEopDataUserdefined::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataUserdefined::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be operated on.
; The valid types are specified when the operation was included in davemenus.xml
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   ;; Valid types are containers with at least one independent and dependent data components
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
;   oTarget = (n_elements(oTarget) gt 0)? [oTarget,oRes] : oRes
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
oTarget = oSel[0]
if (n_elements(oTarget) eq 0) then begin
    oTool->StatusMessage, 'Selection does not contain data that can be rebinned!'
    return, obj_new()
endif


; Make the first valid target the operation's target dataset
self.oTarget = oTarget
;self.oSel = oSel[0]             ; also record the selected data containing the target dataset

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Some operation properties depend on the selected data: update them
    self->InitializeAttributes

    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Rebin the data and record changes in undo/redo buffer
if (~self->DoTheOperation(oTarget, oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataUserdefined::DoTheOperation
; 
; PURPOSE:
;   Transform the data conatined in the specified target according to the user specified expression and record
;   changes in the operation's undo/redo buffer.
;
; PARAMETERS:
;   oTarget [in|out] - the data container whose data is to be
;                      rebinned.
;
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0.
;
function DAVEopDataUserdefined::DoTheOperation, oTarget, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get current operation settings
self->GetProperty,xvals1=xvals1,yvals1=yvals1,zvals1=zvals1,evals1=evals1 $
  ,xlab1=xlab1,ylab1=ylab1,zlab1=zlab1,xunit1=xunit1,yunit1=yunit1,zunit1=zunit1

; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as 
; axes data or objects and the treatment buffer.
oParent = getmaindataobjectparent(oTarget)
if (~obj_valid(oParent)) then return, 0

; 1st independentt axis
oParent->GetProperty, axis1Value=x, nDimensions=ndim, axis1Label=xLab, axis1Units=xUnit
if (n_elements(x) eq 0) then return, 0

; 2nd indep axis, if present
if (ndim eq 2) then begin
  oParent->GetProperty, axis2Value=y, axis2Label=yLab, axis2Units=yUnit
  if (n_elements(y) eq 0) then return, 0
endif else y = !null

; The dependent data
oParent->GetProperty, dataValue=z, dataLabel=zLab, dataUnits=zUnit
if (n_elements(z) eq 0) then return, 0

; The error in the dependent data
oParent->GetProperty, errorValue=e
errExist = (n_elements(e) gt 0)

; verify that the expression can execute without error and notify the user if not
if ((xvals1.trim()).strlen() gt 0) then begin
  if (~Execute('void = '+xvals1,1)) then begin
    msg = ['The specified expression: ',xvals1,' has a syntax/logical error! ','Exiting...!']
    oTool->Errormessage, msg, severity=0, title=title
    oTool->Statusmessage,strjoin(msg)
    Return, 0
  endif
endif

; verify that the expression can execute without error and notify the user if not
if ((yvals1.Trim()).Strlen() gt 0) then begin
  if (~Execute('void = '+yvals1,1)) then begin
    msg = ['The specified expression: ',yvals1, 'has a syntax/logical error! ','Exiting...!']
    oTool->Errormessage, msg, severity=0, title=title
    oTool->Statusmessage,Strjoin(msg)
    Return, 0  
  endif
endif

; verify that the expression can execute without error and notify the user if not
if ((zvals1.Trim()).Strlen() gt 0) then begin
  if (~Execute('void = '+zvals1,1)) then begin
    msg = ['The specified expression: ',zvals1,' has a syntax/logical error! ','Exiting...!']
    oTool->Errormessage, msg, severity=0, title=title
    oTool->Statusmessage,Strjoin(msg)
    Return, 0
  endif
endif

; verify that the expression can execute without error and notify the user if not
if ((evals1.Trim()).Strlen() gt 0) then begin
  if (~Execute('void = '+evals1,1)) then begin
    msg = ['The specified expression: ',evals1,' has a syntax/logical error! ','Exiting...!']
    oTool->Errormessage, msg, severity=0, title=title
    oTool->Statusmessage,Strjoin(msg)
    Return, 0
  endif
endif

; Store copies of original data because the user defined transform operation is not
; reversible.
oCmd = obj_new("IDLitCommand", TARGET_IDENTIFIER=oTarget->GetFullIdentifier())
oCmdSet->Add, oCmd
void = oCmd->AddItem('OLD_X',x)
void = oCmd->AddItem('NDIM',ndim)
void = oCmd->AddItem('OLD_XLAB',xLab)
void = oCmd->Additem('OLD_XUNIT',xUnit)
if (ndim eq 2) then begin
  void = oCmd->Additem('OLD_Y',y)
  void = oCmd->Additem('OLD_YLAB',yLab)
  void = oCmd->Additem('OLD_YUNIT',yUnit)
endif
void = oCmd->AddItem('OLD_Z',z)
void = oCmd->Additem('OLD_ZLAB',zLab)
void = oCmd->Additem('OLD_ZUNIT',zUnit)
void = oCmd->AddItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmd->AddItem('OLD_E',e)
endif



; Get treatment history
oParent->GetProperty, trmtRef = oTrmt    ; can also use: oTrmt = locatetreatmentobject(oTarget)
if (obj_valid(oTrmt)) then begin
   void = oTrmt->GetData(trmt)
   ;void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
   void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
   ;; modify treatment info accordingly
   line = '____________________________________________________'
   trmt = [trmt, $
           line, $
           'Timestamp: '+systime(), $
           'Perform a user-defined operation','']
endif

; Commence user-defined operations
if ((xvals1.trim()).strlen() gt 0) then begin
  status = execute('_newx_='+xvals1,1)
  if (status) then begin
    oParent->SetProperty, axis1Value=_newx_
    void = oCmd->AddItem('NEW_X',_newx_)
    trmt = [trmt,'The independent axis was changed to: ',xvals1,'']
  endif
endif
if (~xlab1.equals(xLab)) then begin
  oParent->Setproperty, axis1Label=xlab1
  void = oCmd->Additem('NEW_XLAB',xlab1)
  trmt = [trmt,'The independent axis label was changed from: ',xlab,' to: ',xlab1,'']
endif
if (~xUnit1.equals(xUnit)) then begin
  oParent->Setproperty, axis1Units=xUnit1
  void = oCmd->Additem('NEW_XUNIT',xUnit1)
  trmt = [trmt,'The independent axis units was changed from: ',xUnit,' to: ',xUnit1,'']
endif

if (ndim eq 2) then begin
  if ((yvals1.Trim()).Strlen() gt 0) then begin
    status = Execute('_newy_='+yvals1,1)
    if (status) then begin
      oParent->Setproperty, axis2Value=_newy_
      void = oCmd->Additem('NEW_Y',_newy_)
      trmt = [trmt,'The 2nd independent axis was changed to: ',yvals1,'']
    endif
  endif
  if (~ylab1.equals(yLab)) then begin
    oParent->Setproperty, axis2Label=ylab1
    void = oCmd->Additem('NEW_YLAB',ylab1)
    trmt = [trmt,'The 2nd independent axis label were changed from: ',ylab,' to: ',ylab1,'']
  endif
  if (~yUnit1.equals(yUnit)) then begin
    oParent->Setproperty, axis2Units=yUnit1
    void = oCmd->Additem('NEW_YUNIT',yUnit1)
    trmt = [trmt,'The 2nd independent axis units were changed from: ',yUnit,' to: ',yUnit1,'']
  endif
endif

if ((zvals1.Trim()).Strlen() gt 0) then begin
  status = Execute('_newz_='+zvals1,1)
  if (status) then begin
    oParent->Setproperty, dataValue=_newz_
    void = oCmd->Additem('NEW_Z',_newz_)
    trmt = [trmt,'The dependent data was changed to: ',zvals1,'']
  endif
endif
if (~zlab1.equals(zLab)) then begin
  oParent->Setproperty, dataLabel=zlab1
  void = oCmd->Additem('NEW_ZLAB',zlab1)
  trmt = [trmt,'The dependent data label was changed from: ',zlab,' to: ',zlab1,'']
endif
if (~zUnit1.equals(zUnit)) then begin
  oParent->Setproperty, dataUnits=zUnit1
  void = oCmd->Additem('NEW_ZUNIT',zUnit1)
  trmt = [trmt,'The dependent data units were changed from: ',zUnit,' to: ',zUnit1,'']
endif
if (errExist) then begin
  if ((evals1.Trim()).Strlen() gt 0) then begin
    status = Execute('_newe_='+evals1,1)
    if (status) then begin
      oParent->Setproperty, errorValue=_newe_
      void = oCmd->Additem('NEW_E',_newe_)
      trmt = [trmt,'The error in the dependent data was changed to: ',evals1,'']
    endif
  endif
endif

; Update treatment information
if (obj_valid(oTrmt)) then begin
    ;trmt = [trmt, $
    ;        'From '+ $
    ;        strtrim(string(lolim_mod),2)+' to '+strtrim(string(uplim_mod),2)+ $
    ;        ' in steps of '+strtrim(string(binw),2)]

    void = oCmd->AddItem('NEW_TREATMENT',trmt) ; save modified treatment info
    void = oTrmt->SetData(trmt,/no_copy,/no_notify)
endif

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataUserdefined::Cleanup
; 
; PURPOSE:
;   DAVEopDataUserdefined class cleanup
;
pro DAVEopDataUserdefined::Cleanup
compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataUserdefined::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataUserdefined::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is not reversible
; This operation is expensive
if (~self->IDLitOperation::Init(NAME='User-defined Operation' $
;                                ,types=['DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP'] $
                                ,reversible_operation=0,expensive_computation=1 $
                                ,_EXTRA=etc)) then return, 0

;self->SetProperty, reversible_operation=0, expensive_computation=1

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register an offset property for this operation
self->Registerproperty, 'xvals0', /string, description='Current 1st indep var' ,name='Current 1st indep var',sensitive=0
self->Registerproperty, 'xlab0', /string, description='Current 1st indep label' ,name='Current 1st indep label',sensitive=0
self->Registerproperty, 'xunit0', /string, description='Current 1st indep unit' ,name='Current 1st indep unit',sensitive=0
self->Registerproperty, 'yvals0', /string, description='Current 2nd indep var' ,name='Current 2nd indep var',sensitive=0
self->Registerproperty, 'ylab0', /string, description='Current 2nd indep label' ,name='Current 2nd indep label',sensitive=0
self->Registerproperty, 'yunit0', /string, description='Current 2nd indep unit' ,name='Current 2nd indep unit',sensitive=0
self->Registerproperty, 'zvals0', /string, description='Current dependent var' ,name='Current dependent var',sensitive=0
self->Registerproperty, 'zlab0', /string, description='Current dependent label' ,name='Current dependent label',sensitive=0
self->Registerproperty, 'zunit0', /string, description='Current dependent unit' ,name='Current dependent unit',sensitive=0
self->Registerproperty, 'evals0', /string, description='Current error variable' ,name='Current error var',sensitive=0

self->Registerproperty, 'xvals1', /string, description='1st indep var expression' ,name='1st indep var expression',sensitive=1
self->Registerproperty, 'xlab1', /string, description='1st indep label' ,name='1st indep label',sensitive=1
self->Registerproperty, 'xunit1', /string, description='1st indep unit' ,name='1st indep unit',sensitive=1
self->Registerproperty, 'yvals1', /string, description='2nd indep var expression' ,name='2nd indep var expression',sensitive=1
self->Registerproperty, 'ylab1', /string, description='2nd indep label' ,name='2nd indep label',sensitive=1
self->Registerproperty, 'yunit1', /string, description='2nd indep unit' ,name='2nd indep unit',sensitive=1
self->Registerproperty, 'zvals1', /string, description='Dependent var expression' ,name='Dependent var expression',sensitive=1
self->Registerproperty, 'zlab1', /string, description='Dependent label' ,name='Dependent label',sensitive=1
self->Registerproperty, 'zunit1', /string, description='Dependent unit' ,name='Dependent unit',sensitive=1
self->Registerproperty, 'evals1', /string, description='Error var expression' ,name='Error var expression',sensitive=1


; init offset to 0.0
self.ndims = 1
;self.truncate = 1
;Self.norebin=0
;self.setdefault = 1


; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataUserdefined__define
; 
; PURPOSE:
;   DAVEopDataUserdefined class structure definition
;
pro DAVEopDataUserdefined__define

compile_opt idl2

struc = {DAVEopDataUserdefined $
         ,inherits IDLitOperation $
         ,ndims:0 $              ; 1==1D || 2==2D
         ,xvals0:''$             ; the 1st independent axis variable
         ,yvals0:''$             ; the 2nd independent axis variable
         ,zvals0:''$             ; the dependent axis variable
         ,evals0:''$             ; the error axis variable
         ,xlab0:''$              ; the 1st independent axis label
         ,ylab0:''$              ; the 2nd independent axis label
         ,zlab0:''$              ; the dependent axis label
         ,xunit0:''$              ; the 1st independent axis unit
         ,yunit0:''$              ; the 2nd independent axis unit
         ,zunit0:''$              ; the dependent axis unit

         ,xvals1:''$             ; the new 1st independent axis expression in terms of the existing data variables
         ,yvals1:''$             ; the new 2nd independent axis expression
         ,zvals1:''$             ; the new dependent axis expression
         ,evals1:''$             ; the new error axis expression
         ,xlab1:''$              ; the new 1st independent axis label
         ,ylab1:''$              ; the new 2nd independent axis label
         ,zlab1:''$              ; the new dependent axis label
         ,xunit1:''$              ; the new 1st independent axis unit
         ,yunit1:''$              ; the new 2nd independent axis unit
         ,zunit1:''$              ; the new dependent axis unit

         ,oTarget:obj_new() $   ; the target object.
;         ,oSel:obj_new() $      ; the selected data item
        }

end
