; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDatasetSort
;
; PURPOSE:
;   A simple operation that multiplies the data of the selected
;   IDLitData object(s) by a scalefactor.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDatasetSort::GetProperty
;
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [out] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDatasetSort::GetProperty, _sortOrder=_sortOrder, _comments=_comments, _sortAxis=_sortAxis, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(_sortOrder)) then _sortOrder =  self._sortOrder
if (arg_present(_comments)) then _comments =  self._comments
if (arg_present(_sortAxis)) then _sortAxis =  self._sortAxis

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDatasetSort::SetProperty
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [in] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDatasetSort::SetProperty, _sortOrder=_sortOrder, _comments=_comments, _sortAxis=_sortAxis, _EXTRA=etc
compile_opt idl2

if (n_elements(_sortOrder) gt 0) then self._sortOrder = _sortOrder
if (n_elements(_comments) gt 0) then self._comments = _comments
if (n_elements(_sortAxis) gt 0) then Self._sortAxis = _sortAxis

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc

end



;===============================================================================
; DAVEopDatasetSort::RecordInitialValues
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetSort::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, _sortAxis=axis, _sortOrder=order
void = oCmd->AddItem('OLD_AXIS',axis)
void = oCmd->AddItem('OLD_ORDER',order)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDatasetSort::RecordFinalValues
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetSort::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, _sortAxis=axis, _sortOrder=order
void = oCmd->AddItem('NEW_AXIS',axis)
void = oCmd->AddItem('NEW_ORDER',order)

return, 1

end


;===============================================================================
; DAVEopDatasetSort::DoExecuteUI
;
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetSort::DoExecuteUI
  compile_opt idl2

  ; Get the tool
  oTool = self->GetTool()
  if (~obj_valid(oTool)) then return, 0

  ; Use the build-in 'PropertySheet' UI service to let the user
  ; customize the operation's property.
  return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDatasetSort::UndoOperation
;
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetSort::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Loop through rest of commands and undo the changes that were made
for i=0,nCmds-1 do begin
  status = oCmds[i]->GetItem('ID_DATA', idData)
  oData = oTool->GetByIdentifier(idData)
  if (~obj_valid(oData)) then begin
    oCmdSet->Remove, oCmds[i] ; This command obviously has a problem so delete it!
    oTool->StatusMessage,'Missing dataset! Undo could not be completed'
    continue
  endif
  status = oCmds[i]->GetItem('ID_AXIS', idAxis)
  oAxis = oTool->GetByIdentifier(idAxis)
  If (~oData->GetData(sortedData)) then continue
  If (~oAxis->GetData(sortedAxis)) then continue
  
  ; get the original unsorted data from the command object and undo the operation
  status = oCmds[i]->GetItem('DEP_DATA', unsortedData)
  status = oCmds[i]->GetItem('AXIS_DATA', unsortedAxis)
  void = oData->SetData(unsortedData,/no_copy,/no_notify)
  void = oAxis->SetData(unsortedAxis,/no_copy,/no_notify)

  ; store the previously sorted data in case we need to redo the operation
  void = oCmds[i]->AddItem('AXIS_DATA',sortedAxis,/overwrite)
  void = oCmds[i]->AddItem('DEP_DATA',sortedData,/overwrite)

  ; process error, if present
  status = oCmds[i]->GetItem('ERREXIST', errExist)
  if (errExist) then begin
    status = oCmds[i]->GetItem('ID_ERR', idErr)
    oErr = oTool->GetByIdentifier(idErr)
    If (~oErr->GetData(sortedError)) then continue
    status = oCmds[i]->GetItem('ERR_DATA', unsortedError)
    void = oErr->SetData(unsortedError,/no_copy,/no_notify)
    void = oCmds[i]->AddItem('ERR_DATA',sortedError,/overwrite)
  endif

  status = oCmds[i]->GetItem('TRMTEXIST', trmtExist)
  if (trmtExist) then begin
    status = oCmds[i]->GetItem('ID_TREATMENT', idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then begin
      status = oCmds[i]->GetItem('OLD_TREATMENT', oldTrmt)
      void = oTrmt->SetData(oldTrmt)
    endif
  endif
  ; Notify Observers
  oData->notifyDataChange
  oData->notifyDataComplete
  oAxis->notifyDataChange
  oAxis->notifyDataComplete

endfor

return, 1

end


;===============================================================================
; DAVEopDatasetSort::RedoOperation
;
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetSort::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Loop through the commands and redo the changes that were made
for i=0,nCmds-1 do begin
  status = oCmds[i]->GetItem('ID_DATA', idData)
  oData = oTool->GetByIdentifier(idData)
  if (~obj_valid(oData)) then begin
    oCmdSet->Remove, oCmds[i] ; This command obviously has a problem so delete it!
    oTool->StatusMessage,'Missing dataset! Redo could not be completed'
    continue
  endif
  status = oCmds[i]->GetItem('ID_AXIS', idAxis)
  oAxis = oTool->GetByIdentifier(idAxis)
  If (~oData->GetData(unsortedData)) then continue
  If (~oAxis->GetData(unsortedAxis)) then continue
  
  ; get the previously sorted data from the command object and redo the operation
  status = oCmds[i]->GetItem('DEP_DATA', sortedData)
  status = oCmds[i]->GetItem('AXIS_DATA', sortedAxis)
  void = oData->SetData(sortedData,/no_copy,/no_notify)
  void = oAxis->SetData(sortedAxis,/no_copy,/no_notify)

  ; store the original unsorted data in case we need to undo the operation
  void = oCmds[i]->AddItem('AXIS_DATA',unsortedAxis,/overwrite)
  void = oCmds[i]->AddItem('DEP_DATA',unsortedData,/overwrite)

  ; process error, if present
  status = oCmds[i]->GetItem('ERREXIST', errExist)
  if (errExist) then begin
    status = oCmds[i]->GetItem('ID_ERR', idErr)
    oErr = oTool->GetByIdentifier(idErr)
    If (~oErr->GetData(unsortedError)) then continue
    status = oCmds[i]->GetItem('ERR_DATA', sortedError)
    void = oErr->SetData(sortedError,/no_copy,/no_notify)
    void = oCmds[i]->AddItem('ERR_DATA',unsortedError,/overwrite)
  endif

  status = oCmds[i]->GetItem('TRMTEXIST', trmtExist)
  if (trmtExist) then begin
    status = oCmds[i]->GetItem('ID_TREATMENT', idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then begin
      status = oCmds[i]->GetItem('NEW_TREATMENT', newTrmt)
      void = oTrmt->SetData(newTrmt)
    endif
  endif
  ; Notify Observers
  oData->notifyDataChange
  oData->notifyDataComplete
  oAxis->notifyDataChange
  oAxis->notifyDataComplete
endfor

return, 1

end


;===============================================================================
; DAVEopDatasetSort::DoAction
;
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's scalefactor to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDatasetSort::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
  oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
  return, obj_new()
endif

; Locate valid dataset containers that can be scaled
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
oSel = []
for i = 0,n_elements(oSelections)-1 do begin
  ;; search for one of the valid types from this selction
  ;; Valid types are containers with at least one independent and dependent data components
  j = 0
  repeat begin
    oRes = oSelections[i]->GetByType(validTypes[j],count=found)
  endrep until (found || ++j ge nValid)

  if (~obj_valid(oRes)) then continue
  oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
nSel = n_elements(oSel)
if (nSel eq 0) then begin
  oTool->StatusMessage, 'Invalid data selection(s)! Please select dataset folder to be sorted'
  return, obj_new()
endif


; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b

;; Record initial value for the scalefactor property of this operation.
if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
  obj_destroy, oCmdSet
  return, obj_new()
endif

; loop through selected datasets and perform scaling
for i = 0,nSel-1 do begin
  oTarget = oSel[i]

  ;; Update properties based on selected dataset
  oTarget->GetProperty, nDimensions=nDim
  Self->SetPropertyAttribute, '_sortAxis', hide=(nDim eq 1), sensitive=1
  oTarget->GetProperty, name=name
  Self->SetProperty, _comment=name

  if (doUI) then begin
    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
      ;obj_destroy, oCmdSet
      ;return, obj_new()
      continue
    endif
  endif

  ;; Perform the sorting
  if (i eq 0) then begin
    ; Retrieve the first command object from the command set (created by RecordInitialValues())
    oCmd = oCmdSet->Get(position=0)
    if (~obj_valid(oCmd)) then return, 0
  endif else begin
    ; create a command to store info about the scaling
    oCmd = obj_new("IDLitCommand", TARGET_IDENTIFIER=oTarget->GetFullIdentifier())
    oCmdSet->Add, oCmd    ; Add the command to command set
  endelse

  if (Self._sortAxis eq 0) then oTarget->GetProperty, axis1Ref=oAxis, axis1Value=axisData $
    else oTarget->GetProperty, axis2Ref=oAxis, axis2Value=axisData
  oTarget->GetProperty, dataRef=oData, dataValue=data, errorRef=oErr, errorValue=error, trmtRef = oTrmt
  errExist = obj_valid(oErr)
  trmtExist = obj_valid(oTrmt)
  
  order = Self._sortOrder
  index = sort(axisData)

  void = oCmd->AddItem('SORTAXIS',Self._sortAxis)
  void = oCmd->AddItem('SORTORDER',Self._sortOrder)
  void = oCmd->AddItem('ID_DATA',oData->GetFullIdentifier())
  void = oCmd->AddItem('DEP_DATA',data) ; sort is irresversible so have to store the old unsorted data
  void = oCmd->AddItem('ID_AXIS',oAxis->GetFullIdentifier())
  void = oCmd->AddItem('AXIS_DATA',axisData)
  void = oCmd->AddItem('ERREXIST',errExist)
  void = oCmd->AddItem('TRMTEXIST',trmtExist)
  if (errExist) then void = oCmd->AddItem('ID_ERR',oErr->GetFullIdentifier())
  if (errExist) then void = oCmd->AddItem('ERR_DATA',error)

  if (order eq 1) then index = reverse(index)
  axisData = axisData[index]
  sortAxis = Self._sortAxis
  if (nDim eq 1) then begin
    n = n_elements(data)
    data = data[index[0:n-1]]       ; use n in case of histogram data!
    if (errExist) then error = error[index[0:n-1]]
  endif else begin
    if (sortAxis eq 0) then begin
      ;; x-axis
      n = (size(data))[1]
      data = data[index[0:n-1],*]
      error = error[index[0:n-1],*]
    endif else begin
      ;; y-axis
      n = (size(data))[2]
      data = data[*,index[0:n-1]]
      error = error[*,index[0:n-1]]
    endelse   
  endelse
  
  if (trmtExist) then begin
    straxis = ['x-axis','y-axis']
    strorder = ['increasing','decreasing']
    void = oTrmt->GetData(trmt)
    line = '____________________________________________________'
    ;; modify treatment info accordingly
    newTrmt = [trmt,line,'Timestamp: '+systime(), 'Perform sorting of data with-respect to ' $
      +straxis[sortAxis]+' in '+strorder[order]+' order']
    void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
    void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
    void = oCmd->AddItem('NEW_TREATMENT',newTrmt)
  endif

  void = oData->setData(data,/no_copy,/no_notify)   ; modify but do not notify obervers
  void = oAxis->setData(axisData,/no_copy,/no_notify)
  if (errExist) then void = oErr->SetData(error,/no_copy,/no_notify)

  ; Notify Observers
  oAxis->notifyDataChange
  oAxis->notifyDataComplete
  oData->notifyDataChange
  oData->notifyDataComplete
  ;if (errExist) then begin
  ;  oErr->notifyDataChange
  ;  oErr->notifyDataComplete
  ;endif
endfor

;; Record final value for the scalefactor property of this operation.
if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
  obj_destroy, oCmdSet
  return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDatasetSort::Cleanup
;
; PURPOSE:
;   DAVEopDatasetSort class cleanup
;
pro DAVEopDatasetSort::Cleanup

  compile_opt idl2

  ; call base class cleanup
  self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDatasetSort::Init
;
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDatasetSort::Init, _REF_EXTRA=etc
  compile_opt idl2

  ; call superclass init
  if (~self->IDLitOperation::Init(NAME='Scalefactor' $
    ,_EXTRA=etc)) then return, 0
  ;types=['IDLVECTOR','IDLARRAY2D','IDLARRAY3D','IDLIMAGE'] $


  ; Unhide the SHOW_EXECUTION_UI property
  self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

  ; This operation is not reversible
  ; This operation is not expensive
  self->SetProperty, reversible_operation=0, expensive_operation=0

  Self->RegisterProperty, '_sortAxis', enum=['x-axis','y-axis'], description='Axis' $
    ,name='Sort Axis',sensitive=1, hide=1

  ; Register an scalefactor property for this operation
  Self->RegisterProperty, '_sortOrder',  enum=['Increasing','Decreasing'], description='Sorting Order' $
    ,name='Sorting Order',sensitive=1, hide=0

  Self->RegisterProperty, '_comments', /string, description='Notes' $
    ,name='Comments',sensitive=0, hide=0

  ; init scalefactor to 1.0
  Self._sortAxis = 0
  Self._sortOrder = 0
  ;Self._comments = 'Specify scalefactors (space delimited) for each group'

  ; return success
  return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDatasetSort__define
;
; PURPOSE:
;   DAVEopDatasetSort class structure definition
;
pro DAVEopDatasetSort__define

  compile_opt idl2

  struc = {DAVEopDatasetSort      $
    ,inherits IDLitOperation      $
    ,_comments:''                 $ ; useful help notes
    ,_sortAxis:0                  $ ; 0 => x-axis, 1 => y-axis
    ,_sortOrder:0                 $ ; sort order: 0=increasing, 1=descreasing
  }

end
