
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataMaskUsingMinValue
;
; PURPOSE:
;   A simple operation that rebins the plottable data of an
;   IDLitData object. The plottable data must consist of an dependent
;   component with at least one independent component.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataMaskUsingMinValue::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataMaskUsingMinValue::GetProperty, oldMask=oldMask, newMask=newMask, _REF_EXTRA=etc
compile_opt idl2

; dmin
if (arg_present(oldMask)) then oldMask = self.oldMask

;
if (arg_present(newMask)) then  newMask = self.newMask


; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataMaskUsingMinValue::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataMaskUsingMinValue::SetProperty, oldMask=oldMask, newMask=newMask, _EXTRA=etc
compile_opt idl2

; newMask
if (n_elements(newMask)) then self.newMask=newMask

; dmax
if (n_elements(oldMask)) then self.oldMask=oldMask


; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end




;===============================================================================
; DAVEopDataMaskUsingMinValue::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataMaskUsingMinValue::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataMaskUsingMinValue::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataMaskUsingMinValue::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmd = oCmdSet->Get(/all,count=nCmd)
if (nCmd ne 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get various stored properties and restore the operation with them
void = oCmd->GetItem('OLDMASK',oldMask)
void = oCmd->GetItem('NEWMASK',newMask)
void = oCmd->GetItem('NANINDEX',nanIndex)
self->SetProperty, newMask=newMask,oldMask=oldMask


; Undo the changes that were made to the target
void = oCmd->GetItem('ID_DEP',idDep)
oDep = oTool->GetByIdentifier(idDep)

if (~obj_valid(oDep)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

if (~oDep->GetData(zdata)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

zdata[nanIndex] = oldMask
void = oDep->SetData(zdata,/no_copy,/no_notify)

void = oCmd->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmd->GetItem('OLD_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif
 
;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataMaskUsingMinValue::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataMaskUsingMinValue::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmd = oCmdSet->Get(/all,count=nCmd)
if (nCmd ne 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get various stored properties and restore the operation with them
void = oCmd->GetItem('NEWMASK',newMask)
void = oCmd->GetItem('NANINDEX',nanIndex)
void = oCmd->GetItem('OLDMASK',oldMask)
self->SetProperty, newMask=newMask,oldMask=oldMask

; Redo the changes that were made
void = oCmd->GetItem('ID_DEP',idDep)
oDep = oTool->GetByIdentifier(idDep)
if (~obj_valid(oDep)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

if (~oDep->GetData(zdata)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

zdata[nanIndex] = newMask
void = oDep->SetData(zdata,/no_copy,/no_notify)

void = oCmd->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmd->GetItem('NEW_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif

;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataMaskUsingMinValue::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataMaskUsingMinValue::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be rebinned
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   ;; Valid types are containers with at least one independent and dependent data components
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
;   oTarget = (n_elements(oTarget) gt 0)? [oTarget,oRes] : oRes
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
oTarget = oSel[0]
if (n_elements(oTarget) eq 0) then begin
    oTool->StatusMessage, 'Selection does not contain data that can be rebinned!'
    return, obj_new()
endif


; Make the first valid target the operation's target dataset
self.oTarget = oTarget

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0
oCmdSet->Add, oCmd
   
; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as 
; axes data or objects and the treatment buffer.
oParent = getmaindataobjectparent(oTarget)


; The dependent data
oParent->GetProperty, dataRef=oDep, dataValue=zin

nanIndex = where(finite(zin,/nan),nanCnt)
if (nanCnt gt 0) then begin
   oldMask = !values.f_nan
   newMask = min(zin,/nan)
   Self->SetProperty, oldMask=oldMask, newMask=newMask
   void = oCmd->AddItem('OLDMASK',oldMask)
   void = oCmd->AddItem('NEWMASK',newMask)
   void = oCmd->AddItem('NANINDEX',nanIndex)
   void = oCmd->AddItem('ID_DEP',oDep->GetFullIdentifier())
endif else begin
   oTool->ErrorMessage,'Intensity data does not contain any NaNs!', severity=0, title='Replace Mask values'
   oTool->StatusMessage,'Fix Mask: Intensity data does not contain any NaNs!'
   
   obj_destroy, oCmdSet
   return, obj_new()
endelse

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b

if (doUI) then begin
    
   

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    Self->GetProperty, newMask=newMask
    void = oCmd->AddItem('NEWMASK',newMask)
endif

;; Change the mask from NaNs to the newMask value
zin[nanIndex] = newMask
if (~oDep->setData(zin,/no_copy,/no_notify)) then begin
   obj_destroy, oCmdSet
   return, obj_new()
endif

; Get treatment history
oParent->GetProperty, trmtRef = oTrmt    ; can also use: oTrmt = locatetreatmentobject(oTarget)
if (obj_valid(oTrmt)) then begin
   void = oTrmt->GetData(trmt)
   void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
   void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
   ;; modify treatment info accordingly
   line = '____________________________________________________'
   trmt = [trmt, $
           line, $
           'Timestamp: '+systime(), $
           'Replace NaN mask values with the value: ' + strtrim(string(newMask),2)]
   void = oCmd->AddItem('NEW_TREATMENT',trmt) ; updated treatment info
   void = oTrmt->SetData(trmt,/no_copy,/no_notify)
endif


; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataMaskUsingMinValue::Cleanup
; 
; PURPOSE:
;   DAVEopDataMaskUsingMinValue class cleanup
;
pro DAVEopDataMaskUsingMinValue::Cleanup
compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataMaskUsingMinValue::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataMaskUsingMinValue::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is not reversible
; This operation is expensive
if (~self->IDLitOperation::Init(NAME='Fix Masking' $
                                ,_EXTRA=etc)) then return, 0

;self->SetProperty, reversible_operation=0, expensive_computation=1

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register an offset property for this operation
self->RegisterProperty, 'oldMask', /float,name='Current Mask value',sensitive=0
self->RegisterProperty, 'newMask', /float, name='Desired Mask value'

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataMaskUsingMinValue__define
; 
; PURPOSE:
;   DAVEopDataMaskUsingMinValue class structure definition
;
pro DAVEopDataMaskUsingMinValue__define

compile_opt idl2

struc = {DAVEopDataMaskUsingMinValue $
         ,inherits IDLitOperation $
;         ,index:ptr_new() $     ; Indices of values that were affected
         ,oldMask:0.0 $         ; previous mask value
         ,newMask:0.0 $         ; current mask value
         ,oTarget:obj_new() $   ; the target object.
        }

end
