; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataConvolute
;
; PURPOSE:
;   A simple operation that performs a convolution of the target dataset using
;   a supplied resolution data. The dataset must contain plottable data with dependent
;   and independent components.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   ResetOpAttributes
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2014
;-
;###############################################################################


;===============================================================================
; DAVEopDataConvolute::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DAVEopDataConvolute::GetProperty, dataset1=dataset1,dataset2=dataset2, normResFlag=normResFlag $
  ,resultFlag=resultFlag,resultName=resultName, _REF_EXTRA=etc
compile_opt idl2

; dataset1
if (arg_present(dataset1)) then dataset1 = self.dataset1

; dataset2
if (arg_present(dataset2)) then dataset2 = self.dataset2

if (arg_present(normResFlag)) then normResFlag = self.normResFlag

if (arg_present(resultFlag)) then resultFlag = self.resultFlag

if (arg_present(resultName)) then resultName = self.resultName

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataConvolute::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DAVEopDataConvolute::SetProperty, dataset1=dataset1,dataset2=dataset2, normResFlag=normResFlag $
  ,resultFlag=resultFlag,resultName=resultName, _EXTRA=etc

compile_opt idl2

; dataset1 index
if (n_elements(dataset1)) then begin
  Self.dataset1=dataset1
  resultname = ptr_valid(Self.selNamesPtr)? (*Self.selNamesPtr)[dataset1]+'-conv' : 'untitled'
  Self.resultName = resultName 
endif

; dataset2 index
if (n_elements(dataset2)) then Self.dataset2 = dataset2

if (n_elements(normResFlag)) then Self.normResFlag = normResFlag

if (n_elements(resultFlag)) then begin
  Self.resultFlag = resultFlag
  Self->SetPropertyAttribute, 'resultName', hide=~resultFlag
endif

if (n_elements(resultName)) then Self.resultName = resultName

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataConvolute::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataConvolute::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store useful info. The target for the
; command is the operation itself. For the convolution operation, only
; one convolution will be performed per command set object created! If
; this were not the case, then it would be better to make a dataset
; object as the target.
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, normResFlag=normResFlag, resultFlag=resultFlag, resultName=resultName
void = oCmd->AddItem('OLD_NORMRESFLAG',normResFlag)
void = oCmd->AddItem('OLD_RESULTFLAG',resultFlag)
void = oCmd->AddItem('OLD_RESULTNAME',resultName)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataConvolute::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataConvolute::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, normResFlag=normResFlag, resultFlag=resultFlag, resultName=resultName
void = oCmd->AddItem('NEW_NORMRESFLAG',normResFlag)
void = oCmd->AddItem('NEW_RESULTFLAG',resultFlag)
void = oCmd->AddItem('NEW_RESULTNAME',resultName)

return, 1

end


;===============================================================================
; DAVEopDataConvolute::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataConvolute::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataConvolute::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataConvolute::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Get the 'old' properties and revert to them
void = oCmds[0]->GetItem('OLD_NORMRESFLAG',normResFlag)
void = oCmds[0]->GetItem('OLD_RESULTFLAG',resultFlag)
void = oCmds[0]->GetItem('OLD_RESULTNAME',resultName)
oCmdTarget->SetProperty, normResFlag=normResFlag, resultFlag=resultFlag, resultName=resultName

void = oCmds[0]->GetItem('NEW_RESULTFLAG',resultFlag)
case resultFlag of
  0: begin
    ;; Convoluted result was stored in the original dataset
        
    ; Get various stored properties and restore the operation with them
    void = oCmds[0]->GetItem('IDDATA_DS',idDep)
    oDep = oTool->GetByIdentifier(idDep)
    trmtExist = oCmds[0]->GetItem('ID_TREATMENT',idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then void = oCmds[0]->GetItem('OLD_TREATMENT',trmt) else trmtExist=0
    
    if (~obj_valid(oDep) || ~obj_isa(oDep,'IDLitData')) then begin
      msg = 'convolution: Missing dependent data 1!'
      oTool->StatusMessage,msg
      return, 0
    endif

    void = oCmds[0]->GetItem('OLD_ZDATA',z1)
    void = oDep->SetData(z1,/no_copy,/no_notify)
    
    ; Treatment info
    if (trmtExist) then $
      void = oTrmt->SetData(trmt,/no_copy,/no_notify) ; modify treatment
      
    ; Notify observers about the change so they can update themselves!
    oDep->notifyDataChange
    oDep->notifyDataComplete
  end
  
  1: begin
    ;; a new dataset was created and the Convoluted results stored in it

    ; Undo previous convolution by deleting the previously generated output from the Data Manager
    void = oCmds[0]->GetItem('FULLOUTPUTID',idDataset)
    ;oDataset = oTool->GetByIdentifier(idDataset)
    oDM = oTool->GetService("DAVEDATA_MANAGER")
    status = oDM->DeleteData(idDataset) ;oDataset->GetFullIdentifier())    
  end
  
  else:
endcase



; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DAVEopDataConvolute::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataConvolute::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Get the 'new' properties and reset operation to them
void = oCmds[0]->GetItem('NEW_NORMRESFLAG',normResFlag)
void = oCmds[0]->GetItem('NEW_RESULTFLAG',resultFlag)
void = oCmds[0]->GetItem('NEW_RESULTNAME',resultName)
oCmdTarget->SetProperty, normResFlag=normResFlag, resultFlag=resultFlag, resultName=resultName


; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('IDDATA_DS',idDep)
oDep = oTool->GetByIdentifier(idDep)
trmtExist = oCmds[0]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then void = oCmds[0]->GetItem('NEW_TREATMENT',trmt) else trmtExist = 0

if (~obj_valid(oDep)) then begin
    msg = 'convolution: Missing dependent data!'
    oTool->StatusMessage,msg
    return, 0
endif
;if (~oDep->GetData(data1)) then return, 0
;
;if (~obj_valid(oDep2) || ~obj_isa(oDep2,'IDLitData')) then begin
;    msg = 'convolution: Missing dependent data!'
;    oTool->StatusMessage,msg
;    return, 0
;endif
;if (~oDep2->GetData(data2)) then return, 0
;data2 = ds2Norm * data2   ; apply normalization factor as before
;
;if (errExist) then begin
;    if (~obj_valid(oErr1) || ~obj_isa(oErr1,'IDLitData')) then begin
;        msg = 'convolution: Missing error in dependent data!'
;        oTool->StatusMessage,msg
;        return, 0
;    endif
;    if (~oErr1->GetData(err1)) then return, 0
;    
;    if (~obj_valid(oErr2) || ~obj_isa(oErr2,'IDLitData')) then begin
;        msg = 'convolution: Missing error in dependent data!'
;        oTool->StatusMessage,msg
;        return, 0
;    endif
;    if (~oErr2->GetData(err2)) then return, 0
;    err2 = err2 * ds2Norm  ; apply normalization factor as before
;endif


case resultFlag of
  0: begin ;; Results should be stored in the first dataset in the convolution eqn
    
  end
  1: begin
    ;; Results should be stored in a new dataset created
    
    ; locate the first dataset used in original convolution eqn
    oTarget1 = getmaindataobjectparent(oDep)
    if (~obj_valid(oTarget1)) then return, 0
    if (~obj_isa(oTarget1,'IDLitParameterSet')) then return, 0
    
    ; Create new dataset to contain Convoluted data
    if (obj_hasmethod(oTarget1,'Clone')) then begin
      ; use clone method to replicate first dataset
      oResult = oTarget1->Clone()
      oResult->SetProperty, name=resultName
      oTool->AddDatasetToDataManager, oResult
    endif else begin
      ; use DM to replicate first dataset
      oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
      if (~oDM->CopyData(oTarget1->GetFullIdentifier(), name=resultName)) then return, 0
    endelse
    
    oResult->GetProperty, trmtRef=oTrmt, dataRef=oDep    ; dependent data and treatment object refs
    trmtExist = oCmds[0]->GetItem('NEW_TREATMENT',trmt)
    
    ; Update 'FULLOUTPUTID' entry
    void = oCmds[0]->AddItem('FULLOUTPUTID',oResult->GetFullIdentifier(),/overwrite)
    
  end
  
  else:
endcase

;; Retrieve the previously performed convolution result and update the data object
void = oCmds[0]->GetItem('NEW_ZDATA',zout)
void = oDep->SetData(zout,/no_copy,/no_notify)

; Update Treatment info
if (trmtExist) then $
  void = oTrmt->SetData(trmt,/no_copy,/no_notify) ; modify treatment


; Notify observers about the change so they can update themselves!
oDep->notifyDataChange
oDep->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DAVEopDataConvolute::ResetOpAttributes
; 
; PURPOSE:
;   Reset attributes for this operation to reflect the currently
;   selected datasets
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function DAVEopDataConvolute::ResetOpAttributes
compile_opt idl2

; Get the selected data and populate the first and second dataset fields
if (self.selCount lt 2) then return, 0

self->SetPropertyAttribute, 'dataset1', enumlist=(*self.selNamesPtr)
self->SetPropertyAttribute, 'dataset2', enumlist=(*self.selNamesPtr)
self->SetProperty, dataset1=0, dataset2=1

return, 1

end


;===============================================================================
; DAVEopDataConvolute::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Take the
;   difference between two supplied datasets and store the results in
;   the first dataset or in a new dataset.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataConvolute::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData(count=cnt)
;void = where(obj_valid(oSelections),cnt)
if (cnt lt 2) then begin
  oSelections = oTool->GetDAVEDataManagerContents(count=cnt)
  if (cnt lt 2) then begin
    oTool->StatusMessage, 'ERROR: Convolution: At least two top-level datasets from the Data Browser tree are required to perform a Convolution.'
    return, obj_new()
  endif
endif

; Locate datasets containing dave1commonstr containers
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be 'DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP',...
                                    ; these are defined when this operation is registered!
nValid = n_elements(validTypes)
validName = 'DAVE Data Manager'
for i = 0,n_elements(oSelections)-1 do begin
    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oResult = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)
    if (~obj_valid(oResult)) then continue

    ; Record the obj ref. and identifier of the valid data as well as the selection it is found in
    ; Also the note the name property of the selection
    selID = oSelections[i]->GetFullIdentifier()
    selIDs = (n_elements(selIDs) gt 0)? [selIDs,selID] : selID
    oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
    oSelections[i]->GetProperty, name=selName
    selNames = (n_elements(selNames) gt 0)? [selNames,selName] : selName
endfor

if (n_elements(oSel) eq 0) then begin
    oTool->StatusMessage, 'ERROR: Convolution: Could not locate two valid datasets to perform convolution!'
    return, obj_new()
endif

; Store the selected datasets and the associated target data objects
if (ptr_valid(self.targetIDsPtr)) then ptr_free, self.targetIDsPtr
if (ptr_valid(self.selNamesPtr)) then ptr_free, self.selNamesPtr

self.targetIDsPtr = ptr_new(selIDs) ;ptr_new(targetIDs)
self.selNamesPtr = ptr_new(selNames)
self.selCount = n_elements(selNames)
oTarget = oSel

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
;hasPropSet = 0b

if (doUI) then begin
    
    ;; Set operation properties based on the selected data
    if (~self->ResetOpAttributes()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Perform convolution and record changes in undo/redo buffer
if (~self->Convolute(oTarget, oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataConvolute::Convolute
; 
; PURPOSE:
;   Perform difference b/n two datasets and record changes in the
;   operation's undo/redo buffer.
;
; PARAMETERS:
;   oTarget [in|out] - the target data - not used in this operation
;
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0
;
function DAVEopDataConvolute::Convolute, oTarget, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Operation settings
ds1 = Self.dataset1
ds2 = Self.dataset2
normResFlag = Self.normResFlag
resFlag=Self.resultFlag
resName=Self.resultName

; Get the data contained in ds1 and ds2
; and check that they have the same size (dims and elements)
targetID1 = (*self.targetIDsPtr)[ds1]
oTarget1 = oTool->GetByIdentifier(targetID1)
if (~obj_valid(oTarget1)) then begin
    msg = 'ERROR: Convolution: Could not retrieve target dataset!'
    oTool->StatusMessage,msg
    return, 0
endif
targetID2 = (*self.targetIDsPtr)[ds2]
oTarget2 = oTool->GetByIdentifier(targetID2)
;targetID = (*self.targetIDsPtr)[ds2]
if (~obj_valid(oTarget2)) then begin
    msg = 'ERROR: Convolution: Could not retrieve resolution dataset!'
    oTool->StatusMessage,msg
    return, 0
endif

; Are the datasets compatible?
oTarget1->GetProperty, nDimensions=nDim1, axis2Dim=ny1
oTarget2->GetProperty, nDimensions=nDim2, axis2Dim=ny2
passed = ((nDim1 eq 2) && (nDim2 eq 2))? ny1 eq ny2 : 1
;s1 = size(data1)
;s2 = size(data2)
;passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2]))
if (~passed) then begin
    msg = "ERROR: Convolution: Data to be convoluted and resolution don't have same number of groups!"
    oTool->StatusMessage,msg
    return, 0
endif

; Get the actual data from the datasets
oTarget1->GetProperty, dataRef=oDep,axis1Value=x1,axis2Value=y1,dataValue=z1 $
  ,axis1Distribution=x1Type, axis2Distribution=y1Type $
  ,treatment=treatment, trmtRef=oTrmt
if (~obj_valid(oDep) || ~obj_isa(oDep,'IDLitData') || ~isa(x1) || ~isa(z1)) then begin
  msg = 'ERROR: Convolution: Could not retrieve any data from dataset to be convoluted!'
  oTool->StatusMessage,msg
  return, 0
endif

oTarget2->GetProperty, dataRef=oDep2,axis1Value=x2,axis2Value=y2,dataValue=z2 $
  ,axis1Distribution=x2Type, axis2Distribution=y2Type $
  ,dataName=resDataName
if (~isa(x2) || ~isa(z2)) then begin
  msg = 'ERROR: Convolution: Could not retrieve any data from resolution dataset!'
  oTool->StatusMessage,msg
  return, 0
endif

; if either x or y values are histograms then convert to point mode
if (strcmp(x1type,'HIST',4,/fold)) then begin
  n = n_elements(x1)
  x1 = 0.5*(x1[0:n-2]+x1[1:n-1])
endif
if (nDim1 eq 2 && strcmp(y1type,'HIST',4,/fold)) then begin
  n = n_elements(y1)
  y1 = 0.5*(y1[0:n-2]+y1[1:n-1])
endif
if (strcmp(x2type,'HIST',4,/fold)) then begin
  n = n_elements(x2)
  x2 = 0.5*(x2[0:n-2]+x2[1:n-1])
endif
if (nDim2 eq 2 && strcmp(y2type,'HIST',4,/fold)) then begin
  n = n_elements(y2)
  y2 = 0.5*(y2[0:n-2]+y2[1:n-1])
endif

; Do the convolution
i = 0
n = (nDim1 eq 1)? 1 : n_elements(y1)
zout = z1
while (i lt n) do begin
  zRes = (nDim2 eq 2)? z2[*,i] : z2
  if (normResFlag) then begin
    index = uniq(x2)    ; int_tabulated() requires unique x values
    zRes = zRes/int_tabulated(x2[index],zres[index],/sort)
  endif
  zDat = (nDim1 eq 2)? z1[*,i] : z1
  case nDim1 of
    1: zout = convolute(x1,zDat,x2,zRes)
    2: zout[*,i] = convolute(x1,zDat,x2,zRes)
  endcase
  i++
endwhile



; Retrieve the first command object from the command set to record
; details of what is done.
; This was created in RecordInitialValues()
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Store identifier of the dependent data object of the original dataset that was convoluted
void = oCmd->AddItem('IDDATA_DS',oDep->GetFullIdentifier())

case resFlag of
  0: begin
    ;; Results should be stored in the original dataset
    
    ; Get the treatment object
    ;trmtExist = 0
    ;if (obj_valid(oTrmt)) then $
    ;  trmtExist = oTrmt->GetData(trmt)     
   
    void = oCmd->AddItem('OLD_ZDATA',z1)
    oResult = oTarget1
  end

  1: begin
    ;; Results should be stored in a new dataset created
    
    ; Create new dataset to contain Convoluted data
    if (obj_hasmethod(oTarget1,'Clone')) then begin
      ; use clone method to replicate first dataset
      oResult = oTarget1->Clone()
      oResult->SetProperty, name=resName
      oTool->AddDatasetToDataManager, oResult
    endif else begin
      ; use DM to replicate first dataset
      oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
      if (~oDM->CopyData(oTarget1->GetFullIdentifier(), name=resName, copyID=copyID)) then return, 0
      oResult = oTool->GetByIdentifier(copyID)
    endelse
    
    ; Update 'FULLOUTPUTID' entry
    void = oCmd->AddItem('FULLOUTPUTID',oResult->GetFullIdentifier(),/overwrite)

  end
  
  else:
endcase

; Get the treatment info and update it
trmtExist = 0
oResult->GetProperty, trmtRef=oTrmt, dataRef=oDep
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then begin
  YesNo = ['No','Yes']
  void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
  void = oCmd->AddItem('OLD_TREATMENT',trmt)
  
  line = '____________________________________________________'
  newTrmt = 'Convolution applied: '
  newTrmt = [newTrmt,'Original dataset name: '+(*self.selNamesPtr)[ds1]]
  newTrmt = [newTrmt,'Resolution dataset name: '+(*self.selNamesPtr)[ds2]]
  newTrmt = [newTrmt,'Unit normalization applied to resolution data: '+YesNo[normResFlag]]  
  newTrmt = [newTrmt,'Convoluted data saved in: '+resName]
  newTrmt = [trmt,line,'Timestamp: '+systime(),newTrmt]
  
  void = oCmd->AddItem('NEW_TREATMENT',newTrmt) ; save modified treatment info
  void = oTrmt->SetData(newTrmt,/no_copy,/no_notify) ; modify treatment obj
endif

; Store convolution result in data object
void = oCmd->AddItem('NEW_ZDATA',zout)
void = oDep->SetData(zout,/no_copy,/no_notify) ; /no_notify ==> don't notify observers!

; Notify observers about the change so they ca update themselves!
oDep->notifyDataChange
oDep->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataConvolute::Cleanup
; 
; PURPOSE:
;   DAVEopDataConvolute class cleanup
;
pro DAVEopDataConvolute::Cleanup
compile_opt idl2

if (ptr_valid(self.selNamesPtr)) then ptr_free, self.selNamesPtr
if (ptr_valid(self.targetIDsPtr)) then ptr_free, self.targetIDsPtr
if (ptr_valid(self.selIDsPtr)) then ptr_free, self.selIDsPtr

;ptr_free, selNamesPtr,targetIDsPtr,selIDsPtr

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataConvolute::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataConvolute::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is reversible
; This operation is not expensive
if (~self->IDLitOperation::Init(NAME='Data convolution' $
                                ,reversible_operation=1,expensive_computation=0 $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']

;self->SetProperty, reversible_operation=1, expensive_computation=0

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register properties for this operation
self->RegisterProperty, 'dataset1', enumlist=['empty'], description='Dataset to convolute' $
  ,name='Dataset to convolute'
self->RegisterProperty, 'dataset2', enumlist=['empty'], description='Resolution dataset"' $
  ,name='Resolution dataset'
self->RegisterProperty, 'normResFlag', enumlist=['No','Yes'], description='Normalize Resolution' $
  ,name='Normalize Resolution'
resList = ['Overwrite Original Dataset','Create New Dataset']
self->RegisterProperty, 'resultFlag',enumlist=resList, description='Destination for subtracted results' $
  ,name='Save convoluted dataset in'
self->RegisterProperty, 'resultName',/string, description='Name of dataset to store result' $
  ,name='Convoluted dataset name',hide=1

self.normResFlag=1

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataConvolute__define
; 
; PURPOSE:
;   DAVEopDataConvolute class structure definition
;
pro DAVEopDataConvolute__define

compile_opt idl2

struc = {DAVEopDataConvolute $
         ,inherits IDLitOperation $
         ,dataset1:0B   $ ; Dataset to convolute
         ,dataset2:1B   $ ; Resolution dataset
         ,resultFlag:0  $ ; destination for convoluted data: 0=overwrite dataset1, 1=create new dataset
         ,resultName:'' $ ; name of dataset to store the sum
         ,normResFlag:1 $    ; normalize resolution data 0=No, 1=Yes
         ,resSourceFlag:0  $    ; 0=from data manager; 1=load new file
         ,resfilename:''   $    ; resolution data filename to be read from disk
         ,selCount:0B $         ; nos of datasets currently selected
         ,selNamesPtr:ptr_new() $ ;  ptr to names of currently selected datasets
         ,targetIDsPtr:ptr_new() $ ; ptr to ids of target objects in selected datasets
         ,selIDsPtr:ptr_new() $ ; ptr to ids of selected datasets
        }

end
