; $Id$
;###############################################################################
;
; NAME:
;  PAN_QFluidJqyCF
;
; PURPOSE:
;  Convolution Function for use in fitting data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;  June 2014
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function pan_QFluidJqyCF,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

; Use common block here to enable persistent storage of the constants
; B, D and F so they can be used between calls to this function. This is
; a way of making B, D and F global variables.
; Note that this common statement declared and used here only!
common pan_QF_Jqy_constants, B, D, F, etc, kc_com, T_com

canDraw = 0
twoDimFlag = 0
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 1           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if n_params() eq 0 then begin
  parmnames = ['Scale Factor','Temperature', $
               'n0','kc','a2','b3','a4','b4','b5','b6','a6','Y Offset']
  initParms = [1.0,1.0,0.07,0.5,0.897,0.097,0.46,0.0,0.1637,0.344,0.38,0.0]
  return,-1
endif

;  drawMessage = strarr(4)
;  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
;  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

;bc = parms[0]
;bs = parms[1]
sf = parms[0]
T  = parms[1]
n0 = parms[2]
kc = parms[3]
yoff = parms[11]

muPtr = ptr_new({a2:parms[4],  $     ; [(1/A)**2]
                 b3:parms[5],  $     ; [(1/A)**3]
                 a4:parms[6],  $     ; [(1/A)**4]
                 b4:parms[7],  $     ; [(1/A)**4]
                 b5:parms[8], $     ; [(1/A)**5]
                 b6:parms[9], $     ; [(1/A)**6]
                 a6:parms[10]})      ; [(1/A)**6]

; Define the constants if necessary
; B/c of the common statement above, this will be done once during an IDL session
; even if this function is called multiple times! Also, whenever T and kc changes 
; the constants wil be re-evaluated.
firstime = n_elements(B) eq 0 || n_elements(D) eq 0 || n_elements(F) eq 0
THasChanged = (n_elements(T_com) eq 1) && (T_com ne T)
kcHasChanged = (n_elements(kc_com) eq 1) && (kc_com ne kc)
if (firstime || THasChanged || kcHasChanged) then begin
  mass = 4.0026           ; rel a. mass of He4
  hbar = 1.0545919        ; *10**(-34) Js
  ec   = 1.6021917        ; *10**(-19) C
  amu  = 1.660531         ; *10**(-27) Kg
  kB   = 1.380622         ; *10**(-23) J/K
  rho  = 0.6022169/27.0   ; 1/A**3         He4 at SVP
  Vs   = 240              ; m/s            He4 at SVP
  B = mass*amu*Vs/(2.0*hbar*(2.0*!pi)^3*rho)/1000.0  ; A**2
  D = hbar*Vs/(2.0*kb*T)/10.0                    ; A
  F = 2.0*kc^2                                     ; (1/A)**2


  ; Evaluate F(s) at several s, and use is as a table for cubic spline interpolation
  ; in subsequent evaluations of F(s) during the Fourier Transform of Jia(s)
  s = [findgen(21)*0.25, findgen(10)*0.5+5.5, findgen(10)+11, findgen(12)*2.5+21, $
    findgen(10)*5+50, findgen(30)*10+100];, findgen(10)*20+600, findgen(6)*50+800];, $
  ;findgen(3)*500+1200, 5000.0, 10000.0, 20000.0, 50000.0, 100000.0]
  fs = fofs(s, nOrder = 1, kc=kc, T=T, B=B, D=D, F=F) ; 1st moment of fofs i.e. when nOrder=1
  splinePtr = ptr_new({x:s,y:fs,y2:spl_init(s,fs,/double)})
  
  I0 = Fofs(0.0,/splineFlag,splinePtr=splinePtr)
  A = 1.0 - n0*(1 + I0)
  
  etc = create_struct('y',0.0,'muPtr',muPtr,'n0',n0,'A',A,'kc',kc,'T',T,'B',B,'D',D,'F',F, $
    'splineFlag',1,'splinePtr',splinePtr)
  
  kc_com = kc
  T_com = T
endif

; Update etc structure
I0 = Fofs(0.0,/splineFlag,splinePtr=splinePtr)
A = 1.0 - n0*(1 + I0)
etc.muPtr = muPtr
etc.n0 = n0
etc.A = A

; Evalute J(y) as Fourier Transform of J(s)
SQ = 1.0
n = n_elements(x)
amp = sf*SQ/!pi
yout = dblarr(n)
for i=0,n-1 do begin
  etc.y = x[i] + yoff   ; add arbitrary y offset parameter
  yout[i] = amp * qpint1d('Jqys',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)
endfor
if (n eq 1) then yout = yout[0]

; Comment out the following code because adding an n0 to the calculation at the peak position is not handle
; properly in any subsequent numerical analysis. The correct way is to add a n0 delta function to represent
; the condensate! This can be done in PAN, for example.

;if ((*muPtr).b6 eq 0.0) then begin
;  ;; In this case, evaluating the FT above is problematic because R(Q,s) does not converge
;  ;; hence the n0*R(Q,s) component of J(Q,s) is left out. This now taken into account by
;  ;; adding n0 to the final FT at the peak location of jiay
;  void = yout.max(index)
;  yout[index] = yout[index] + n0
;endif


return, yout

end


pro test_pan_QFluidJqyCF, plot=plot
compile_opt idl2

parms = [1.0,1.0,0.07,0.5,0.897,0.097,0.46,0.0,0.1637,0.344,0.38,-5.0]
y = findgen(141)*0.1 - 7.0
void = pan_QFluidJqyCF(parmnames=pnames)
jqy = pan_QFluidJqyCF(y,parms)
if (keyword_set(plot)) then begin
  title = ['J(Y)',strjoin(pnames,'; '),strjoin(string(parms,format='(F5.3)'),'; ')]
  p = plot(y,jqy,title=title)
endif
end
