; $Id$
;###############################################################################
;
; NAME:
;  Pan_lorentzian_2D
;
; PURPOSE:
;  Lorentzian distribution for use in fitting data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function Pan_lorGlobalAreaWidthCenter_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 1
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all
if N_params() eq 0 then begin
  single_parmnames = ['Area','FWHM','Center']
  multi_parmnames = ''
  parmnames = single_parmnames

  Return,-1
endif
drawMessage = Strarr(4)
drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

if (Keyword_set(changefirst)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is first release at the location for the function centroid
  ; translate the position into suitable parameters.
  ;Print,'###Changefirst called within function...'
  ;Print,'Changefirst...'
  index = Indgen(nq)
  fwhm = 0.05*(xrange[1]-xrange[0])
  gam = 0.5*fwhm
  area = yMouseClick*gam*!dpi
  parms[0] = area
  parms[1] = fwhm
  parms[2] = xMouseClick
  func_dataHash['ymax'] = yMouseClick
endif

if (Keyword_set(changesecond)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is move to define the function width
  ; translate the position into suitable parameters.
  ;Print,'###Changesecond called within function...'
  
  cen    = parms[2]          ; the initial center determined from changefirst above for current group
  fwhm = 2.0*Abs(xMouseClick-cen)
  gamma = 0.5*fwhm
  area = func_dataHash['ymax']*gamma*!dpi

  ; the function consists of 1 gaussian for each group.
  ; all the gaussians are initialized with the same area and width
  parms[0] = area
  parms[1] = fwhm
endif

; Vectorize
; extrapolote to 2D by multiplying with appropriate unit vector
index = Indgen(nq)
area = ux#uq*parms[0]
gam  = ux#uq*parms[1]*0.5
cen  = ux#uq*parms[2]
xq    = x#uq

yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

Return,yout
end



function Pan_lorGlobalAreaWidth_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 1
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all
if N_params() eq 0 then begin
  single_parmnames = ['Area','FWHM']
  multi_parmnames = ['Center']
  parmnames = single_parmnames
  centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
  parmnames = [parmnames,centerstr]

  Return,-1
endif
drawMessage = Strarr(4)
drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

if (Keyword_set(changefirst)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is first release at the location for the function centroid
  ; translate the position into suitable parameters.
  ;Print,'###Changefirst called within function...'
  ;Print,'Changefirst...'
  index = Indgen(nq)
  fwhm = 0.05*(xrange[1]-xrange[0])
  gam = 0.5*fwhm
  area = yMouseClick*gam*!dpi
  parms[0] = area
  parms[1] = fwhm
  parms[index+2] = xMouseClick
  func_dataHash['ymax'] = yMouseClick
endif

if (Keyword_set(changesecond)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is move to define the function width
  ; translate the position into suitable parameters.
  ;Print,'###Changesecond called within function...'
  
  cen    = parms[(groupNumber-1)+2]          ; the initial center determined from changefirst above for current group
  fwhm = 2.0*Abs(xMouseClick-cen)
  gamma = 0.5*fwhm
  area = func_dataHash['ymax']*gamma*!dpi

  ; the function consists of 1 gaussian for each group.
  ; all the gaussians are initialized with the same area and width
  parms[0] = area
  parms[1] = fwhm
endif

; Vectorize
; extrapolote to 2D by multiplying with appropriate unit vector
index = Indgen(nq)
area = ux#uq*parms[0]
gam  = ux#uq*parms[1]*0.5
cen  = ux#parms[index+2]
xq    = x#uq

yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

Return,yout
end



function Pan_lorGlobalAreaOnly_2d,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 1
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all
  if N_params() eq 0 then begin
    single_parmnames = ['Area']
    multi_parmnames = ['FWHM','Center']

    parmnames = single_parmnames
    fwhmstr = 'FWHM'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,fwhmstr,centerstr]

    Return,-1
  endif
  drawMessage = Strarr(4)
  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

  if (Keyword_set(changefirst)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is first release at the location for the function centroid
    ; translate the position into suitable parameters.
    index = Indgen(nq)
    fwhm = 0.05*(xrange[1]-xrange[0])
    gam = 0.5*fwhm
    area = yMouseClick*gam*!dpi
    parms[0] = area
    parms[index+1] = fwhm
    parms[index+nq+1] = xMouseClick
    func_dataHash['ymax'] = yMouseClick
  endif

  if (Keyword_set(changesecond)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is move to define the function width
    ; translate the position into suitable parameters.
    ;Print,'###Changesecond called within function...'

    cen    = parms[(groupNumber-1)+nq+1]          ; the initial center determined from changefirst above for current group
    fwhm = 2.0*Abs(xMouseClick-cen)
    gamma = 0.5*fwhm
    area = func_dataHash['ymax']*gamma*!dpi

    ; the function consists of 1 gaussian for each group.
    ; all the gaussians are initialized with the same area and width
    index = Indgen(nq)
    parms[index+1] = fwhm
    parms[0] = area
  endif

  ; Vectorize
  ; extrapolote to 2D by multiplying with appropriate unit vector
  index = Indgen(nq)
  area  = ux#uq*parms[0]
  gam = ux#parms[index+1]*0.5
  cen  = ux#parms[index+nq+1]
  xq    = x#uq

  yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

  Return,yout
end



function Pan_lorGlobalWidthOnly_2d,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 1
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all
  if N_params() eq 0 then begin
    single_parmnames = ['FWHM']
    multi_parmnames = ['Area','Center']

    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    Return,-1
  endif
  drawMessage = Strarr(4)
  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

  if (Keyword_set(changefirst)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is first release at the location for the function centroid
    ; translate the position into suitable parameters.
    index = Indgen(nq)
    fwhm = 0.05*(xrange[1]-xrange[0])
    gam = 0.5*fwhm
    area = yMouseClick*gam*!dpi
    parms[0] = fwhm
    parms[index+1] = area
    parms[index+nq+1] = xMouseClick
    func_dataHash['ymax'] = yMouseClick
  endif

  if (Keyword_set(changesecond)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is move to define the function width
    ; translate the position into suitable parameters.
    ;Print,'###Changesecond called within function...'

    cen    = parms[(groupNumber-1)+nq+1]          ; the initial center determined from changefirst above for current group
    fwhm = 2.0*Abs(xMouseClick-cen)
    gamma = 0.5*fwhm
    area = func_dataHash['ymax']*gamma*!dpi

    ; the function consists of 1 gaussian for each group.
    ; all the gaussians are initialized with the same area and width
    index = Indgen(nq)
    parms[index+1] = area
    parms[0] = fwhm
  endif

  ; Vectorize
  ; extrapolote to 2D by multiplying with appropriate unit vector
  index = Indgen(nq)
  gam  = ux#uq*parms[0]*0.5
  area = ux#parms[index+1]
  cen  = ux#parms[index+nq+1]
  xq    = x#uq

  yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

  Return,yout
end



function Pan_lorGlobalWidthQDep_2d,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 1
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all
  if N_params() eq 0 then begin
    single_parmnames = ['FWHM']
    multi_parmnames = ['Area','Center']

    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    Return,-1
  endif
  drawMessage = Strarr(4)
  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

  if (Keyword_set(changefirst)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is first release at the location for the function centroid
    ; translate the position into suitable parameters.
    index = Indgen(nq)
    fwhm = 0.05*(xrange[1]-xrange[0])
    gam = 0.5*fwhm
    area = yMouseClick*gam*!dpi
    func_dataHash['ymax'] = yMouseClick

    qVal   = qVals[groupNumber-1]

    parms[0] = fwhm/qVal    ; enforce a linear Q dependence for width
    parms[index+1] = area
    parms[index+nq+1] = xMouseClick

  endif

  if (Keyword_set(changesecond)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is move to define the function width
    ; translate the position into suitable parameters.
    ;Print,'###Changesecond called within function...'

    cen    = parms[(groupNumber-1)+nq+1]          ; the initial center determined from changefirst above for current group
    fwhm = 2.0*Abs(xMouseClick-cen)
    gamma = 0.5*fwhm
    area = func_dataHash['ymax']*gamma*!dpi

    ; the function consists of 1 gaussian for each group.
    ; all the gaussians are initialized with the same area and width
    qVal   = qVals[groupNumber-1]

    index = Indgen(nq)
    parms[index+1] = area
    parms[0] = fwhm/qVal    ; enforce a linear Q dependence for width
  endif

  ; Vectorize
  ; extrapolote to 2D by multiplying with appropriate unit vector
  index = Indgen(nq)
  gam  = ux#(parms[0]*0.5*qVals)      ; enforce a linear Q dependence for width
  area = ux#parms[index+1]
  cen  = ux#parms[index+nq+1]
  xq    = x#uq

  yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

  Return,yout
end



function Pan_lorGlobalWidthQ2Dep_2d,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 1
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all
  if N_params() eq 0 then begin
    single_parmnames = ['FWHM']
    multi_parmnames = ['Area','Center']

    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    Return,-1
  endif
  drawMessage = Strarr(4)
  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

  if (Keyword_set(changefirst)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is first release at the location for the function centroid
    ; translate the position into suitable parameters.
    index = Indgen(nq)
    fwhm = 0.05*(xrange[1]-xrange[0])
    gam = 0.5*fwhm
    area = yMouseClick*gam*!dpi
    func_dataHash['ymax'] = yMouseClick

    qVal   = qVals[groupNumber-1]

    parms[0] = fwhm/qVal^2    ; enforce a quadratic Q dependence for width
    parms[index+1] = area
    parms[index+nq+1] = xMouseClick

  endif

  if (Keyword_set(changesecond)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is move to define the function width
    ; translate the position into suitable parameters.
    ;Print,'###Changesecond called within function...'

    cen    = parms[(groupNumber-1)+nq+1]          ; the initial center determined from changefirst above for current group
    fwhm = 2.0*Abs(xMouseClick-cen)
    gamma = 0.5*fwhm
    area = func_dataHash['ymax']*gamma*!dpi

    ; the function consists of 1 gaussian for each group.
    ; all the gaussians are initialized with the same area and width
    qVal   = qVals[groupNumber-1]

    index = Indgen(nq)
    parms[index+1] = area
    parms[0] = fwhm/qVal^2    ; enforce a quadratic Q dependence for width
  endif

  ; Vectorize
  ; extrapolote to 2D by multiplying with appropriate unit vector
  index = Indgen(nq)
  gam  = ux#(parms[0]*0.5*qVals^2)      ; enforce a quadratic Q dependence for width
  area = ux#parms[index+1]
  cen  = ux#parms[index+nq+1]
  xq    = x#uq

  yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

  Return,yout
end



function Pan_lorIndepAreaWidthCenter_2d,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 1
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes
  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all
  if N_params() eq 0 then begin
    parmnames = []
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    fwhmstr = 'FWHM'+Strtrim(String(Indgen(nq)),2)
    parmnames = [areastr,centerstr,fwhmstr]
    single_parmnames = ''
    multi_parmnames = ['Area','Center','FWHM']

    Return,-1
  endif
  drawMessage = Strarr(4)
  drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
  drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

  if (Keyword_set(changefirst)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is first release at the location for the function centroid
    ; translate the position into suitable parameters.
    ;Print,'###Changefirst called within function...'
    ;Print,'Changefirst...'
    index = Indgen(nq)
    fwhm = 0.05*(xrange[1]-xrange[0])
    gam = 0.5*fwhm
    area = yMouseClick*gam*!dpi
    parms[index] = area
    parms[index+nq] = xMouseClick
    parms[index+2*nq] = fwhm
    func_dataHash['ymax'] = yMouseClick
  endif

  if (Keyword_set(changesecond)) then begin
    ; user is initializing the function graphically
    ; this block of code is executed when the mouse is move to define the function width
    ; translate the position into suitable parameters.
    ;Print,'###Changesecond called within function...'

    cen    = parms[(groupNumber-1)+nq]          ; the initial center determined from changefirst above for current group
    fwhm = 2.0*Abs(xMouseClick-cen)
    gamma = 0.5*fwhm
    area = func_dataHash['ymax']*gamma*!dpi

    ; the function consists of 1 gaussian for each group.
    ; all the gaussians are initialized with the same area and width
    index = Indgen(nq)
    parms[index] = area
    parms[index+2*nq] = fwhm
  endif

  ; Vectorize
  ; extrapolote to 2D by multiplying with appropriate unit vector
  index = Indgen(nq)
  area = ux#parms[index]
  cen  = ux#parms[index+nq]
  gam  = ux#parms[index+2*nq]*0.5
  xq    = x#uq

  yout = (area*gam/!dpi)/((xq-cen)^2+gam^2)

  Return,yout
end