; $Id$
;###############################################################################
;
; NAME:
;  PAN_BroadenedThreeFoldTunneling_2D
;
; PURPOSE:
;  Implementation of the 3-fold rotational tunneling model with broadened
;  tunneling lines for use in fitting data in RAINS.
;
; CATEGORY:
;  DAVE, Data Analysis, RAINS, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_BroadenedThreeFoldTunneling_2D,x,parms,         $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  parmnames = ['Radius','Tunnel energy','Gamma_t (tunnel)','Gamma_q (elastic)', $
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]
  single_parmnames = parmnames[0:3]
  multi_parmnames = ['Intensity','Center']



	if arg_present(initParms) then begin
    intensity = Max(yVals,dimension = 1,index)
    index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming there are 3 peaks present, estimate the peak pos for these
    ; at each Q and record the highest energy one. 
    ; Et is taken as mean of these
    npeaks = 3
    highest_peak = []
    for i = 0,nq-1 do begin
      xpeaks = Get_peak_pos(xVals,yVals[*,i],npeaks)
      highest_peak = [highest_peak,max(xpeaks)]
    endfor
    Et = mean(highest_peak)
    
    radius = 1.0      ; Ang units

    ; estimate HWHM for the largest peak at the_highest_ Q value
    hm = 0.5*intensity[nq-1]
    res = Where(yVals[*,nq-1] ge hm, found)
    if (found gt 0) then begin
      Gamma = 0.5*xvals[res[found-1]] - xvals[res[0]]
    endif

		initParms = [radius,et,Gamma,Gamma,intensity,center]
	endif

	return,-1
endif

if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
;yout = Fltarr(nx,nq)
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, Fltarr(nx,nq)   ; bail if resolution is not 2D

radius = parms[0]

index = indgen(nq) + 4
intensity = ux#(parms[index])
center = ux#(parms[index+nq])

; Now calculate the EISF and delta term
eisf = (1.0 + 2.0*Sph_bessel(qVals*radius*sqrt(3.0),0))/3.0
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = eisf                ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position


; The 3 Lorentzians - representing 1 elastic and 2 tunneling peaks
Et = parms[1]
gamma_t = parms[2]
gamma_q = parms[3]
uxuq = ux#uq
xuq = x#uq
tun1_center = center - Et
tun2_center = center + Et
Lor_elastic = gamma_q*uxuq/(gamma_q^2*uxuq + (xuq - center)^2)
Lor_tun1    = gamma_t*uxuq/(gamma_t^2*uxuq + (xuq - tun1_center)^2)
Lor_tun2    = gamma_t*uxuq/(gamma_t^2*uxuq + (xuq - tun2_center)^2)
Lor = (1 - ux#eisf)/3.0 * (Lor_elastic + Lor_tun1 + Lor_tun2)

; Convolute with resolution
yout = intensity*pan_convolute(x,resPtr,Lor,pseudo_delta = pseudo_delta)

return,yout
end
