;==============================================================================
pro Plotparms_definevars_event, Event
  compile_opt idl2

  Widget_control, Event.top, get_uvalue=sPtr

  ; Basic error Handler
  if (N_elements(!debug) && (!debug eq 0)) then begin
    Catch, catchError
    if (catchError ne 0) then begin
      ;;print, 'Error handled!'
      eTitle = 'plotParms_DefineVars_Event: Error encountered'
      eMsg = 'An error or unusual condition was encountered!'
      eMsg = [eMsg,'Please, report the following to the DAVE team:']
      eMsg = [eMsg,!error_state.msg]
      void = Dialog_message(/error,eMsg,title=eTitle,dialog_parent=Event.top)
      Catch, /cancel
      Return
    endif
  endif

  firstline = '; PAN: Derived Parameter Macro'
  uname = Widget_info(Event.id,/uname)
  switch uname of
    'OK': begin
      ; No break statement - continue to next cae
    end

    'EVALUATEEXP': begin
      wText = Widget_info(Event.top,find_by_uname='VAREXP')
      Widget_control, Wtext, get_value=_str

      if ((N_elements(_str) eq 1) && Strcmp(Strtrim(_str[0],2),'')) then begin
        msg = ['Please enter an expression in the text area!','For guidance, select an example from the drop down list']
        void = Dialog_message(msg,/error,title='Derived Parameter Macro',dialog_parent=Event.top)
        Return
      endif

      p = (*sPtr).p
      pErr = (*sPtr).pErr
      x0 = (*sPtr).x0
      x = (*sPtr).x
      y = (*sPtr).y
      yErr = (*sPtr).yerr

      nParms = N_elements(p)  ; p is a hash data structure
      nGroups = N_elements(p[0])
      nCurves = (*sPtr).nCurves
      parmNames = (*sPtr).parmNames

      _nn = N_elements(_str)
      _status = Intarr(_nn)
      proceed = 1
      for i=0,_nn-1 do begin
        _status[i] = Execute(_str[i],1,0)
        if (~_status[i]) then begin
          msg = ['Error in your expression on line '+Strtrim(String(i+1),2),_str[i],' ','Please fix the problem and try again!']
          void = Dialog_message(msg,/error,title='Derived Parameter Macro',dialog_parent=Event.top)
          proceed = 0
        endif
      endfor

      if (Total(_status) eq _nn && ~Strcmp(uname,'OK')) then begin
        msg = ['No syntax error found in your expression','If you are done, please proceed by clicking on the ACCEPT button']
        void = Dialog_message(msg,/information,title='Derived Parameter Macro',dialog_parent=Event.top)
      endif
      if (Strcmp(uname,'OK') && proceed) then begin
        (*sPtr).varDefStatus = 1
        Widget_control, Event.top, /destroy
        if (N_elements(x) gt 0) then (*sPtr).x = x
        if (N_elements(y) gt 0) then (*sPtr).y = y
        if (N_elements(yErr) gt 0) then (*sPtr).yErr = yErr
        if (N_elements(xLabel) gt 0) then (*sPtr).xLabel = xLabel
        if (N_elements(yLabel) gt 0) then begin
          if (yLabel eq '') then yLabel='Unspecified'
          (*sPtr).yLabel = yLabel
        endif
      endif
      break
    end

    'EXAMPLESINDEX': begin
      text0 =         firstline
      text0 = [text0, '; All text following a semicolon are comments and hence ignored']
      text0 = [text0, '; However, when saving/restoring macro from file the comment on line 1 must be present exactly as']
      text0 = [text0, '; above. Note that macro is evaluated line by line. So no multi-line control statements or loops!']
      text0 = [text0, '; ']
      wText = Widget_info(Event.top,find_by_uname='VAREXP')
      case Event.index of
        1: begin
          text = [text0,'x = 2*!pi/x0                                    ; inverse of x0']
          text = [Text, 'y = p[3]/p[1]                                   ; Ratio of parameter 3 and parameter 1']
          text = [Text, 'yErr = y*Quadrature(pErr[3]/p[3],pErr[1]/p[1])  ; Fractional errors added in quadrature']
          text = [Text, 'xLabel = "d $\AA$"']
            text = [Text, 'yLabel = ""']
          Widget_control, Wtext, set_value=text
        end

        2: begin
          text = [text0,'x = x0^2                               ; square of x0']
          text = [Text, 'y = (p[2]-p[1])/x0                     ; Subtract parameter 1 from parameter 2 and d']
          text = [Text, 'yErr = Quadrature(pErr[2],pErr[1])/x0  ; Errors are combined in quadrature']
          text = [Text, 'xLabel = "Q $\AA^{-1}$"']
            text = [Text, 'yLabel = "Corrected Area"']
          Widget_control, Wtext, set_value=text
        end

        3: begin
          text = [text0,'x = x0                         ;']
          text = [Text, 'elInt = [p[5]]                 ; elastic intensity for just 1 peak in this case']
          text = [Text, 'elErr = [pErr[5]]              ; error in elastic intensity               ']
          text = [Text, 'qeInt = [[p[2]],[p[8]]]        ; quasi-elastic intensities - 2 peaks. Note how multiple parameters']
          text = [Text, '                               ; are combined to form a 2D array. Each parameter is a vector']
          text = [Text, 'qeErr = [[pErr[2]],[pErr[8]]]  ; error in quasi-elastic intensities']
          text = [Text, 'status = Eisf(elInt,qeInt,elErr,qeErr, RetVal=y, RetErr=yErr) ; Built-in DAVE function - see below']
          text = [Text, 'xLabel = "Q $\AA^{-1}$"']
            text = [Text, 'yLabel = "$A_0(Q)$"']
          text = [Text,' ']
          text = [Text, '; Eisf() is a DAVE function that calculates the Elastic Incoherent structure factor. It calculates']
          text = [Text, '; the ratio of the sum of the elastic to the sum of the elastic and sum of quasielastic intensities']
          text = [Text, '; The four arguments specify the elastic and quasielastic intensities plus their respective errors']
          text = [Text, '; The calculated eisf and error are y and yErr respectively, returned via the "RetVal" and "RetErr"']
          text = [Text, '; keywords. status will be 1 if Eisf() is successful and 0 otherwise']
          text = [Text, '; Please contact the DAVE team if you need a similarly complicated function built into DAVE.']
          Widget_control, Wtext, set_value=text
        end
        else: begin
          Widget_control, Wtext, set_value=''
        end

      endcase
      break
    end

    'CANCEL': begin
      (*sPtr).varDefStatus = 0
      Widget_control, Event.top, /destroy
      break
    end

    'SAVEEXP': begin
      wText = Widget_info(Event.top,find_by_uname='VAREXP')
      Widget_control, Wtext, get_value=text
      if (N_elements(text) eq 0) then break

      if (~Strcmp(text[0], firstline,Strlen(firstline),/fold)) then $
        text = [firstline,Text]

      filename = Dialog_pickfile(dialog_parent = Event.top,$
        title = 'Select File to Save Macro to', /write,/overwrite_prompt ,filter = '*.mf',$
        path = (*sPtr).workDir,default_extension='mf')
      if (Strtrim(filename) eq '') then break
      Openw, lun, filename,/get_lun
      for i=0,n_elements(text)-1 do Printf, lun, Text[i]
      Free_lun, lun, /force
      msg = ['Macro saved in:',filename]
      void = Dialog_message(msg,/information,title='Derived Parameter Macro',dialog_parent=Event.top)
      break
    end

    'LOADEXP': begin
      filename = Dialog_pickfile(dialog_parent = Event.top,$
        title = 'Select File to Restore Macro from', /read,filter = '*.mf',$
        path = (*sPtr).workDir)
      if (Strtrim(filename) eq '') then break
      fileLen = File_lines(filename)
      if (fileLen eq 0) then break
      text = Strarr(fileLen)
      Openr,lun,filename,/get_lun
      Readf, lun, Text
      Free_lun, lun, /force

      ; check contents
      if (~Strcmp(text[0], firstline,Strlen(firstline),/fold)) then begin
        msg = ['File does not contain a Derived Parameter Macro expression!', 'The first line of the file should be:',firstline]
        void = Dialog_message(msg,/error,title='Derived Parameter Macro',dialog_parent=Event.top)
        break
      endif

      wText = Widget_info(Event.top,find_by_uname='VAREXP')
      Widget_control, Wtext, set_value=text
      break
    end

    else: begin

    end
  endswitch

end;plotParms_DefineVars_Event



;==============================================================================
;+
; Purpose
;   Define new variablr/parameter from the fitted parameters obtained from PAN.
;   Creates a widget dialog from which an IDL expression for the new variable can be
;   entered in terms of the existing fitted parameters.
;-
function Plotparms_definevars, Event
  compile_opt idl2

  if (N_elements(event) eq 0) then Return, 0
  if (Tag_names(event,/struct) ne 'WIDGET_BUTTON') then Return, 0
  Widget_control, Event.id, get_uvalue=sPtr
  p = (*sPtr).p
  pErr = (*sPtr).pErr
  x0 = (*sPtr).x0
  x = (*sPtr).x
  y = (*sPtr).y
  yErr = (*sPtr).yerr

  nParms = N_elements(p)  ; p is a hash data structure
  nGroups = N_elements(p[0])
  nCurves = (*sPtr).nCurves
  parmNames = (*sPtr).parmNames

  modal = 1
  title = 'Parameter Plotting Utility: Derived Parameter Macro'
  wTLB = Widget_base(group_leader=Event.top,/tlb_frame_attr, /col,/modal,/base_align_center,title=title,uvalue=sPtr)

  base0 = Widget_base(wTLB,/col,uname='BASE0');,/scroll,y_scroll_size=400)
  base0_1 = Widget_base(base0,/base_align_left,/frame,/col)
  font = (!version.os_family eq 'Windows')? 'Lucida Sans Typewriter*24' : $
    'lucidasanstypewriter-18'
  font1 = (!version.os_family eq 'Windows')? 'Lucida Sans Typewriter*16' : $
    'lucidasanstypewriter-12'
  font2 = (!version.os_family eq 'Windows')? 'Lucida Sans Typewriter*14' : $
    'lucidasanstypewriter-10'

  txt =      "The derived parameter consists of 3 variables that must be defined in your macro: 'x', 'y' and 'yErr'  "
  txt = [txt,"where:                                                                                                 "]
  txt = [txt," x is an independent variable and a vector of size nGroups and analogous to x0 (see below)             "]
  txt = [txt," y is a dependent variable and a vector of size nGroups                                                "]
  txt = [txt," yErr is the error in y and a vector of size nGroups                                                   "]
  txt = [txt,"x, y and yErr should be expressed in terms of the known fitted parameters and valid DAVE/IDL functions:"]
  txt = [txt," p[nParms] - the existing fitted parameters. There are nParms parameters, each of size nGroups         "]
  txt = [txt," pErr[nParms] - the existing errors in the fitted parameters. Same dimensions as p                     "]
  txt = [txt," x0 - the fixed value assigned to each group (usually temp, Q, etc) in the fits. Vector of size nGroups"]
  txt = [txt,'For your current fits, nParms = '+Strtrim(String(nParms),2) +' and nGroups = '+Strtrim(String(nGroups),2)+Strjoin(Replicate(' ',52))+'  ']
  txt = [txt,'For guidance, please use the "Select Example" droplist below to view a few example expressions         ']
  IDLge90 = (Float(!version.release) ge 9.0)? 1 : 0
  IDLge91 = (Float(!version.release) ge 9.1)? 1 : 0
  n = N_elements(txt)
  color = [240,240,240]
  if (IDLge91) then begin
    for i=0,1 do void = widget_label(base0_1,value=txt[i],font=font2,background_color=color)
    for i=2,4 do void = widget_label(base0_1,value=txt[i],foreground_color=[0,50,255],font=font2,background_color=color)
    for i=5,5 do void = widget_label(base0_1,value=txt[i],font=font2,background_color=color)
    for i=6,8 do void = widget_label(base0_1,value=txt[i],foreground_color=[0,100,255],font=font2,background_color=color)
    for i=9,n-1 do void = widget_label(base0_1,value=txt[i],font=font2,background_color=color)
  endif else begin
    for i=0,1 do void = widget_label(base0_1,value=txt[i],font=font2)
    for i=2,4 do void = widget_label(base0_1,value=txt[i],font=font2)
    for i=5,5 do void = widget_label(base0_1,value=txt[i],font=font2)
    for i=6,8 do void = widget_label(base0_1,value=txt[i],font=font2)
    for i=9,n-1 do void = widget_label(base0_1,value=txt[i],font=font2)
  endelse
;  for i=0,1 do void = Cw_coloredlabel(base0_1,value=txt[i],font=font2,background_color=color)
;  for i=2,4 do void = Cw_coloredlabel(base0_1,value=txt[i],foreground_color=[0,50,255],font=font2,background_color=color)
;  for i=5,5 do void = Cw_coloredlabel(base0_1,value=txt[i],font=font2,background_color=color)
;  for i=6,8 do void = Cw_coloredlabel(base0_1,value=txt[i],foreground_color=[0,100,255],font=font2,background_color=color)
;  for i=9,n-1 do void = Cw_coloredlabel(base0_1,value=txt[i],font=font2,background_color=color)

  ;base1 = widget_base(wTLB,/col,/scroll,y_scroll_size=200,xsize=geom.xsize)
  base2 = Widget_base(wTLB,/col,/frame)
  base3 = Widget_base(wTLB,/row,/frame,/grid)

  geom = Widget_info(base0,/geometry)

  void = Widget_text(base2,/editable,uname='VAREXP',/scroll,/wrap $
    ,/no_newline,scr_xsize=geom.scr_xsize - 4.0*geom.xpad, scr_ysize=0.75*geom.scr_ysize,font=font2)

  txt = ['Select Example' $
    ,'Example 1' $
    ,'Example 2' $
    ,'Example 3' $
    ]

  tt0 = 'Evaluate current macro expression to check for syntax errors. Logical errors cannot be identified!'
  tt1 = 'Click to save current macro expression to a text file for later use'
  tt2 = 'Click to load a previously saved macro expression from a text file. This will replace text area contents!'
  tt3 = 'Exit this dialog and retain the new definitions for x, y and yErr'
  tt4 = 'Exit this dialog but discard changes to x, y and yErr'
  base2_1 = Widget_base(base2,/row,/grid,/align_center)
  void = Widget_combobox(base2_1,value=txt,uname='EXAMPLESINDEX',/flat)
  void = Widget_button(base2_1,value='Evaluate Macro',uname='EVALUATEEXP',tooltip=tt0)
  void = Widget_button(base2_1,value='Save Macro',uname='SAVEEXP',tooltip=tt1)
  void = Widget_button(base2_1,value='Load Macro',uname='LOADEXP',tooltip=tt2)
  void = Widget_button(base3,value='ACCEPT',uname='OK',tooltip=tt3)
  void = Widget_button(base3,value='CANCEL',uname='CANCEL',tooltip=tt4)

  Widget_control, wTLB, /realize
  Xmanager, 'plotParms_DefineVars', wTLB, no_block=modal

  Return, 1

end
