; macroEditor is based on IDL's XDISPLAYFILE.PRO
;
;###############################################################################
;
; NAME:
;  macroEditor
;
; PURPOSE:
;  Basic text editor for user-defined macro functions in PAN
;  Based on xdisplayfile procedure in IDL
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO macroEditor_write, wText, filename

  COMPILE_OPT hidden

  WIDGET_CONTROL, /HOURGLASS
  OPENW, unit, FILENAME, /GET_LUN, ERROR=i		;open the file and then
  if i lt 0 then begin		;OK?
	a = [ !error_state.msg, filename + ' could not be opened for writing.']  ;No
	void = DIALOG_MESSAGE(a, /ERROR, DIALOG_PARENT=wText)
  endif else begin
	WIDGET_CONTROL, wText, get_value=txtArray
	ON_IOERROR, done_writing
	; print out each line separately in order to get desired line breaks
	for j = 0, N_ELEMENTS(txtArray)-1 do PRINTF, unit, txtArray[j]
done_writing:
	ON_IOERROR, null
	FREE_LUN, unit				;free the file unit.
  endelse
END



PRO macroEditor_event, event

  COMPILE_OPT hidden

  WIDGET_CONTROL, event.top, GET_UVALUE=state

  CASE TAG_NAMES(event, /STRUCTURE_NAME) OF

    'WIDGET_BASE': BEGIN
      w = (event.x gt state.x_reserve)? (event.x - state.x_reserve) : state.x_reserve
      if (!version.os_family eq 'Windows') then begin
        h=event.y
      endif else begin
        h = (event.y gt state.y_reserve)? (event.y - state.y_reserve) : state.y_reserve
      endelse
      WIDGET_CONTROL, state.textID, scr_xsize=w, scr_ysize=h

      RETURN
      END
      
    'WIDGET_KILL_REQUEST': retval = "EXIT"

    ELSE: WIDGET_CONTROL, event.id, GET_UVALUE = retval
  ENDCASE
  
  loadSampleMacroFlag = 0

  SWITCH retval OF
  
	"SAVE_AS": BEGIN
               if (LMGR(/DEMO)) then begin
                  tmp = DIALOG_MESSAGE( /ERROR, $
                        'Save As: Feature disabled for demo mode.')
                  return
                endif
		filename = DIALOG_PICKFILE(/WRITE,file=file_basename(state.filename),path=state.workDir $
		                          ,DEFAULT_EXTENSION='*.mf',filter='*.mf' $
		                          ,/overwrite_prompt)
		IF (STRLEN(filename) GT 0) THEN BEGIN
		  state.filename = filename
			macroEditor_write, state.textID, state.filename
			WIDGET_CONTROL, event.top, TLB_SET_TITLE = state.title+filename
			Widget_control, Event.top, set_uvalue=state     ; update state var in case it has changed
		ENDIF		
		break
	END

  "LOADSAMPMACRO_RECOIL": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample2DFitMacro_recoil.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_DIFF": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample2DFitMacro.mf'
    if (Strlen(macrofilename) eq 0 || ~file_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end
  
  "LOADSAMPMACRO_2L": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_2L.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_FWFM_vs_Q": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_FWHM_vs_Q.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_FWFM_vs_Q2": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_FWHM_vs_Q2.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_Tau_KWW": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_tau_kww.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_activation_energy": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_activation_energy.mf'
    if (Strlen(macrofilename) eq 0 || ~File_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_GAUSpBKGD": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro.mf'
    if (Strlen(macrofilename) eq 0 || ~file_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_EISFRotTun": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_EISFRotTun.mf'
    if (Strlen(macrofilename) eq 0 || ~file_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

  "LOADSAMPMACRO_OneMinusEISFRotTun": begin
    macrofilename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro_OneMinusEISFRotTun.mf'
    if (Strlen(macrofilename) eq 0 || ~file_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
  end

	
	"LOADMACROFILE": begin
	  macrofilename = Dialog_pickfile(/read,file=File_basename(State.filename),path=State.workdir,filter='*.mf')
    if (Strlen(macrofilename) eq 0 || ~file_test(macrofilename,/read)) then Return
    loadSampleMacroFlag = 1
    break
	end
	
  'CANCEL': (*state.cancelPtr) = 1
	"EXIT": BEGIN
    widget_control, state.textID, get_value = text
    (*state.textptr) = text
    Widget_control, Event.top, set_uvalue=state     ; update state var in case it has changed
		WIDGET_CONTROL, event.top, /DESTROY
		IF (WIDGET_INFO(state.ourGroup, /VALID)) THEN $
			WIDGET_CONTROL, state.ourGroup, /DESTROY
		break
	END
	
	ELSE:
  
  ENDSWITCH

  if (loadSampleMacroFlag) then begin
    nlines = File_lines(macroFilename)
    text = Strarr(nlines)
    Openr,unit, macroFilename, /get_lun
    Readf, unit, Text
    Free_lun, unit, /force

    Widget_control, State.textid, set_value=text
    (*State.textptr) = text
    State.filename = macroFilename
    Widget_control, Event.top, TLB_SET_TITLE = State.title+macroFilename
    Widget_control, Event.top, set_uvalue=state     ; update state var in case it has changed
  endif

END




PRO macroEditorGrowToScreen, tlb, text, height, nlines
; Grow the text widget so that it displays all of the text or
; it is as large as the screen can hold.
  compile_opt hidden

  max_y = (get_screen_size())[1] - 100
  cur_y = (WIDGET_INFO(tlb, /geometry)).scr_ysize

  ; If the display is already long enough, then there's nothing to do.
  ;
  ; We are only filling to grow the display, not shrink it, so if its
  ; already too big, there's nothing to do. This can only happen if
  ; the caller sets a large HEIGHT keyword, which is operator error.
  if ((nlines le height) || (cur_y gt max_y)) then return

  ; The strategy I use is binary divide and conquer. Furthermore,
  ; if nlines is more than 150, I limit the search to that much.
  ; (Consider that a typical screen is 1024 pixels high, and that
  ; using 10pt type, this yields 102 lines). This number may need to
  ; be adjusted as screen resolution grows but that will almost certainly
  ; be a slowly moving and easy to track target.
  ;
  ; Note: The variable cnt should never hit its limit. It is there
  ; as a "deadman switch".
  low = height
  high = MIN([150, nlines+1])
  cnt=0
  while ((low lt high) && (cnt++ lt 100)) do begin
    old_low = low
    old_high = high
    mid = low + ((high - low + 1) / 2)
    WIDGET_CONTROL, text, ysize=mid
    cur_y = (WIDGET_INFO(tlb, /geometry)).scr_ysize
    if (cur_y lt max_y) then low = mid else high = mid
    if ((old_low eq low) && (old_high eq high)) then break
  endwhile

end







PRO macroEditor,  group_leader = group_leader, workDir=workDir, type=type, $ ;FILENAME, 
                  TITLE = TITLE, WIDTH = WIDTH, $
                  HEIGHT = HEIGHT, FONT = font, MODAL=MODAL, $
                  GROW_TO_SCREEN=grow_to_screen, $
                  WTEXT=textID, BLOCK=block, $
                  output=output
;+
; NAME:
;	macroEditor
;
; PURPOSE:
;	Display an ASCII text file using widgets and the widget manager.
;
; CATEGORY:
;	Widgets.
;
; CALLING SEQUENCE:
;	macroEditor, Filename
;
; INPUTS:
;     Filename:	A scalar string that contains the filename of the file
;		to display.  The filename can include a path to that file.
;
; KEYWORD PARAMETERS:
;	BLOCK:  Set this keyword to have XMANAGER block when this
;		application is registered.  By default the Xmanager
;               keyword NO_BLOCK is set to 1 to provide access to the
;               command line if active command 	line processing is available.
;               Note that setting BLOCK for this application will cause
;		all widget applications to block, not only this
;		application.  For more information see the NO_BLOCK keyword
;		to XMANAGER.
;
;	FONT:   The name of the font to use.  If omitted use the default
;		font.
;	group_leader:	The widget ID of the group leader of the widget.  If this
;		keyword is specified, the death of the group leader results in
;		the death of macroEditor.
;
;       GROW_TO_SCREEN: If TRUE, the length of the display area is grown
;		to show as much of the text as possible without being too
;		large to fit on the screen. In this case, HEIGHT sets the
;		lower bound on the size instead of setting the size itself.
;
;	HEIGHT:	The number of text lines that the widget should display at one
;		time.  If this keyword is not specified, 24 lines is the
;		default.
;
;	TITLE:	A string to use as the widget title rather than the file name
;		or "macroEditor".
;
;	WIDTH:	The number of characters wide the widget should be.  If this
;		keyword is not specified, 80 characters is the default.
;
; OUTPUTS:
;	No explicit outputs.  A file viewing widget is created.
;
; SIDE EFFECTS:
;	Triggers the XMANAGER if it is not already in use.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Open a file and create a widget to display its contents.
;
; MODIFICATION HISTORY:
;	Written By Steve Richards, December 1990
;	Graceful error recovery, DMS, Feb, 1992.
;       12 Jan. 1994  - KDB
;               If file was empty, program would crash. Fixed.
;       4 Oct. 1994     MLR Fixed bug if /TEXT was present and /TITLE was not.
;	2 jan 1997	DMS Added DONE_BUTTON keyword, made Done
;			button align on left, removed padding.
;	19 Nov 2004, GROW_TO_SCREEN and RETURN_ID keywords. Allow for
;                       user to resize display. General updating.
;-

; Establish defaults if keywords not specified
IF(NOT(KEYWORD_SET(EDITABLE))) THEN editable = 1
IF(NOT(KEYWORD_SET(HEIGHT))) THEN HEIGHT = 24
IF(NOT(KEYWORD_SET(WIDTH))) THEN WIDTH = 80
IF N_ELEMENTS(block) EQ 0 THEN block=0

if (n_elements(font) eq 0) then begin
  if (!VERSION.OS_FAMILY EQ 'Windows') then $
    font = "Courier*20*Bold" else $
    font = "-adobe-courier-bold-r-normal--18-180-75-75-m-110-iso8859-1"
endif

if (n_elements(workDir) eq 0) then begin
  workDir=''
endif

title = 'PAN Macro Editor - '
noTitle = N_elements(title) EQ 0

if (N_elements(type) eq 0) then type=0
if (type eq 0) then filename = !DAVE_AUXILIARY_DIR+'sample1DFitMacro.mf'
if (type eq 1) then filename = !DAVE_AUXILIARY_DIR+'sample2DFitMacro.mf'
nlines = file_lines(filename)
samplemacro = Strarr(nlines)
Openr,unit, filename, /get_lun
Readf, unit, samplemacro
Free_lun, unit, /force


;IF(NOT(KEYWORD_SET(TEXT))) THEN BEGIN
;  IF noTitle THEN TITLE = FILENAME
;
;  unit = -1
;  CATCH, err
;  if (err ne 0) then begin
;    CATCH,/CANCEL
;    if (unit ne -1) then FREE_LUN, 1
;    a = [ !error_state.msg, ' Unable to display ' + filename]
;    nlines = n_elements(a)
;  endif else begin
;    nlines = MIN([FILE_LINES(filename), 10000])
;    OPENR, unit, FILENAME, /GET_LUN
;    a = strarr(nlines)
;    readf, unit, a
;    CATCH, /CANCEL
;    FREE_LUN, unit
;  endelse
;ENDIF ELSE BEGIN
;  IF(N_ELEMENTS(FILENAME) EQ 0) THEN FILENAME=''
;  IF noTitle THEN TITLE = 'macroEditor'
;  if (Isa(TEXT, /STRING)) then begin
;    a = TEXT
;  endif else begin
;    a = String(TEXT, /IMPLIED_PRINT)
;  endelse
;  nlines = n_elements(a)
;ENDELSE

ourGroup = 0L
if KEYWORD_SET(MODAL) then begin
  ourGroup = (n_elements(group_leader) gt 0)? group_leader : widget_base()  ; modal requires a group leader
  filebase = Widget_base(TITLE = title, $
    /TLB_KILL_REQUEST_EVENTS, TLB_FRAME_ATTR=1, $
    /BASE_ALIGN_LEFT, /COLUMN, $
    /MODAL, GROUP_LEADER = group_leader)
  ;if N_ELEMENTS(group_leader) GT 0 then begin
  ;  filebase = WIDGET_BASE(TITLE = TITLE, $
	;	/TLB_KILL_REQUEST_EVENTS, TLB_FRAME_ATTR=1, $
  ;                    /BASE_ALIGN_LEFT, /COLUMN, $
  ;                    /MODAL, GROUP_LEADER = group_leader)
  ;endif else begin
  ;  ; modal requires a group leader
  ;  ourGroup = WIDGET_BASE()
  ;  filebase = WIDGET_BASE(TITLE = TITLE, $
  ;    		/TLB_KILL_REQUEST_EVENTS, TLB_FRAME_ATTR=1, $
  ;                    /BASE_ALIGN_LEFT, /COLUMN, $
  ;                    /MODAL, GROUP_LEADER = ourGroup)
  ;endelse
  menu_bar = filebase
endif else begin
  filebase = WIDGET_BASE(TITLE = title, $
      		/TLB_KILL_REQUEST_EVENTS, /TLB_SIZE_EVENTS, $
                      /BASE_ALIGN_LEFT, /COLUMN, MBAR=menu_bar, $
		GROUP_LEADER = group_leader)
endelse

extra = ''
IF (menu_bar NE filebase) THEN BEGIN
  IF (!VERSION.OS_FAMILY EQ 'Windows') THEN extra = '&'
  menu_bar = WIDGET_BUTTON(menu_bar, VALUE=extra+'File', /MENU)
ENDIF ELSE $
  menu_bar = WIDGET_BASE(filebase, /ROW)


IF (editable) THEN BEGIN
  ; add 'Save', 'Save as...' buttons here
  void = WIDGET_BUTTON(menu_bar, VALUE = extra+'Save Macro to file', UVALUE = "SAVE_AS")
  if (type eq 1) then begin
    menuLoadSampMacro = Widget_button(menu_bar, VALUE = extra+'Load Macro', /menu)
    void = Widget_button(menuLoadSampMacro, VALUE = extra+'Recoil', UVALUE = "LOADSAMPMACRO_RECOIL")
    void = Widget_button(menuLoadSampMacro, VALUE = extra+'Diffusion', UVALUE = "LOADSAMPMACRO_DIFF")
    ;void = Widget_button(menuLoadSampMacro, VALUE = extra+'Recoil', UVALUE = "LOADSAMPMACRO_")
  endif else begin
    menuLoadSampMacro = Widget_button(menu_bar, VALUE = extra+'Load Macro', /menu)
    void = Widget_button(menuLoadSampMacro, VALUE = extra+'Gaussian+Bkgd', UVALUE = "LOADSAMPMACRO_GAUSpBKGD")
    void = Widget_button(menuLoadSampMacro, VALUE = extra+'EISF (Rot Tunneling)', UVALUE = "LOADSAMPMACRO_EISFRotTun")
    void = Widget_button(menuLoadSampMacro, VALUE = extra+'1-EISF (Rot Tunneling)', UVALUE = "LOADSAMPMACRO_OneMinusEISFRotTun")
    subMenuLSM = Widget_button(menuLoadSampMacro, VALUE = extra+'Summer School', /menu)
    subSubMenuLSM = Widget_button(subMenuLSM, VALUE = extra+'Ethylene Glycol', /menu)
    void = Widget_button(subSubMenuLSM, VALUE = extra+'2 Lorentzians', UVALUE = "LOADSAMPMACRO_2L")
    void = Widget_button(subSubMenuLSM, VALUE = extra+'FWHM vs Q', UVALUE = "LOADSAMPMACRO_FWFM_vs_Q")
    void = Widget_button(subSubMenuLSM, VALUE = extra+'FWHM vs Q Square', UVALUE = "LOADSAMPMACRO_FWFM_vs_Q2")
    void = Widget_button(subSubMenuLSM, VALUE = extra+'Tau_KWW', UVALUE = "LOADSAMPMACRO_Tau_KWW")
    void = Widget_button(subSubMenuLSM, VALUE = extra+'Activation Energy', UVALUE = "LOADSAMPMACRO_activation_energy")
  endelse
  void = WIDGET_BUTTON(menu_bar, VALUE = extra+'Restore Macro from file', UVALUE = "LOADMACROFILE")
  void = WIDGET_BUTTON(menu_bar, VALUE = extra+'Accept and Exit', UVALUE = "EXIT")
  void = Widget_button(menu_bar, VALUE = extra+'Cancel and Exit', UVALUE = "CANCEL")
ENDIF

;; Done button
;if n_elements(done_button) eq 0 then done_button = "Done with " + TITLE
;filequit = WIDGET_BUTTON(menu_bar, SEPARATOR=editable, $
;	VALUE = extra+done_button, UVALUE = "EXIT")

; Create a text widget to display the text
IF n_elements(font) gt 0 then begin
  textID = WIDGET_TEXT(filebase, XSIZE = WIDTH, YSIZE = HEIGHT, $
	EDITABLE = editable, UVALUE='TEXT', /SCROLL, VALUE = sampleMacro, $
	FONT = font)
endif else begin
  textID = WIDGET_TEXT(filebase, XSIZE = WIDTH, YSIZE = HEIGHT, $
	EDITABLE = editable, UVALUE='TEXT', /SCROLL, VALUE = sampleMacro)
endelse

; Add button controls
rowBase = widget_base(filebase, /row,/frame,/align_center)
void = Widget_button(rowBase,value = 'Save Macro',uvalue='SAVE_AS',font=font)
void = Widget_button(rowBase,value = 'Restore Macro',uvalue='LOADMACROFILE',font=font)
void = Widget_button(rowBase,value = 'Accept',uvalue='EXIT',font=font)
void = Widget_button(rowBase,value = 'Cancel',uvalue='CANCEL',font=font)


if (keyword_set(grow_to_screen)) then $
  macroEditorGrowToScreen, filebase, textID, height, nlines

WIDGET_CONTROL, filebase, /REALIZE

geo_base = WIDGET_INFO(filebase, /geometry)
geo_text = WIDGET_INFO(textID, /geometry)

state={ ourGroup:ourGroup, filename: filename, workDir:workDir, $
        textID:textID, notitle:noTitle, title:title, $
        x_reserve:geo_base.scr_xsize - geo_text.scr_xsize, $
        y_reserve:geo_base.scr_ysize - geo_text.scr_ysize, $
        textPtr:ptr_new(sampleMacro), cancelPtr:ptr_new(0) }
WIDGET_CONTROL, filebase, SET_UVALUE = state


xmanager, "macroEditor", filebase, GROUP_LEADER = GROUP, $
	NO_BLOCK=(NOT(FLOAT(block)))
		
output = {macro:(*state.textPtr), cancel:(*state.cancelPtr)}

end



; Test sequence
;fname = !DAVE_AUXILIARY_DIR+'sample2DFitMacro.mf'
;nlines = file_lines(fname)
;text = strarr(nlines)
;Openr,unit, fname, /get_lun
;Readf, unit, text
;Free_lun, unit, /force
;macroEditor,fname,/editable,/modal,title="PAN Fit macro editor"
;or
;macroEditor,text=text,/editable,font=thisfont,/modal,title="PAN Fit macro editor"