; $Id$
;
;###############################################################################
;
; NAME:
;  Phase Transitions Module for DAVE
;
; PURPOSE:
;  Quick analysis of rocking or radial scans as a function of temperature for
;  BT9,BT7, or SPINS (NG5)
;
; AUTHOR:
;   Danny Phelan
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   E-mail: dphelan@nist.gov
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;###############################################################################

pro two_gauss_phase_transitions_fixedwidth, x, p, yfit
; The independent variable is X
; Parameter values are passed in "P"
yfit = abs(p[0])+abs(p[1]/(p[2]*sqrt(2.*!pi)))*exp(-0.5*((x-p[3])/p[2])^2.)+abs(p[4]/(p[2]*sqrt(2.*!pi)))*exp(-0.5*((x-p[5])/p[2])^2.)
END

pro two_gauss_phase_transitions, x, p, yfit
; The independent variable is X
; Parameter values are passed in "P"
yfit = abs(p[0])+abs(p[1]/(p[2]*sqrt(2.*!pi)))*exp(-0.5*((x-p[3])/p[2])^2.)+abs(p[4]/(p[5]*sqrt(2.*!pi)))*exp(-0.5*((x-p[6])/p[5])^2.)
END

pro help_phase_transitions
;help dialog
;create top-level base
tlb_help=widget_base(/col,title='HELP')

explain_label=widget_text(tlb_help,value=['The Phase Transitions module is for analyzing rocking and radial scans performed using the '+$
	'find peak commands in ICP on SPINS or BT-9 or for data taken in the BT-7 file format.  In particular, it can be used to visualize the'+$
	' results of scans performed as a function of temperature.',' ',$
	'The utility can analyze rocking and radial scans together, or either radial or rocking scans only.'+$
	'  The program should only be used to analyze scans through a SINGLE Bragg reflection.  If scans were performed through several Bragg reflections,'+$
	' then the program can be subsequently re-run for each Bragg reflection.  WARNING: IT WOULD NOT BE UNEXPECTED TO GET STRANGE OUTPUT'$
	+' IF DATAFILES FOR TWO DIFFERENT BRAGG REFLECTIONS WERE INPUT SIMULTANEOUSLY.',' ','After pressing the START button on the opening screen,'+$
	' a menu will appear asking if rocking scans will be analyzed or not.  If the ANALYZE ROCKING SCANS button is pressed, a pick-file dialog will'+$
	' appear.  The user should select all rocking scan files (multiple files can be selected using the CTRL or SHIFT keys).  Subsequently, if the BT-9'+$
	' or SPINS data format is recognized, a table will appear which will allow the user to label the rocking scans by temperature.  An optional'+$
	' scaling factor can also be applied if unequal measuring times/monitors were counted for different temperatures.  (For BT-7 files, this'+$
	' table does not appear since the temperature is stored in the data file.)  The filenames column should not be edited, as this will have'+$
	' no effect.  The ENTER key must be pushed after the last field is filled in, or the last field may not be recognized.'+$
	' After this is finished, a similar sequence of events will occur to enter the radial scans.',' ','After all data has been entered,'+$
	' the program will interpolate the entered data to form smoothed contour plots of the rocking/radial scans.  It will numerically calculate'+$
	' the integrated intensity of the rocking/radial scans, and it will fit the scans to a single Gaussian model, extracting out the integrated intensity,'+$
	' FWHM, and peak positions.  It will also attempt to fit the data with a 2-Gaussian model.  A plotting window will appear with a drop-list that'+$
	' allows the user to choose a figure from amongst the contour plots,'+$
	' the calculated quantities as a function of temperature, or the individual Gaussian fits to the scans.  '+$
	'The results of 1 or 2 Gaussian fits can be accessed selected by droplists.  The currently plotted image can be'+$
	' saved as a .jpg file by choosing File->PRINT TO JPEG.  A multi-column ascii file of results can be outputted by choosing File->Output Temperature-'+$
	'Dependent Results to File.'],/wrap,/scroll,xsize=50,ysize=14)
centertlb,tlb_help
;render the widget on the screen
widget_control,tlb_help,/realize
;start the xmanager
xmanager,'phase_transitions_mapper',tlb_help,event_handler='help_phase_transitions_event_handler',/no_block
end

pro help_phase_transitions_event_handler,event
;do absolutely nothing.
end

pro output_file_phase_transitions
;for outputting the file of results
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

fn_output=dialog_pickfile(title='Create Output File Name')
OpenW,lun,fn_output,/GET_LUN
	if radial_scan_yn eq 'y' then begin
		printf,lun,'RADIAL SCAN RESULTS (Single Gaussian)'
		printf,lun,'Temp.','Int.I(Num)','Int.I(Gauss.)','Int.I(Gauss) Error','FWHM(Gauss.)','FWHM(Gauss) Error','Pos.(Gauss.)',$
			'Pos.(Gauss.) Error',format='(50A20)'
		for n=0,n_elements(temps_radial)-1 do begin
			printF,lun,temps_radial[n],numerically_integrated_intensity_radial[n],fitted_integrated_intensity_radial[n],$
			fitted_integrated_intensity_radial_error[n],fitted_width_radial[n],fitted_width_radial_error[n],fitted_position_radial[n],$
			fitted_position_radial_error[n], FORMAT='(50G20.6)'
		endfor
		printf,lun,'RADIAL SCAN RESULTS (2 Gaussians)'
		printf,lun,'Temp.','Int.I(Peak 1)','Int.I(Peak 1) Error','FWHM(Peak 1)','FWHM(Peak 1) Error','Pos.(Peak 1)',$
			'Pos.(Peak 1) Error','Int.I(Peak 2)','Int.I(Peak 2) Error','FWHM(Peak 2)','FWHM(Peak 2) Error','Pos.(Peak 2)',$
			'Pos.(Peak 2) Error',format='(50A20)'
		for n=0,n_elements(temps_radial)-1 do begin
			printF,lun,temps_radial[n],$
			two_gauss_radial_integrated_intensity_peak1[n],$
			two_gauss_radial_integrated_intensity_peak1_error[n],$
			two_gauss_radial_width_peak1[n],$
			two_gauss_radial_width_peak1_error[n],$
			two_gauss_radial_position_peak1[n],$
			two_gauss_radial_position_peak1_error[n],$
			two_gauss_radial_integrated_intensity_peak2[n],$
			two_gauss_radial_integrated_intensity_peak2_error[n],$
			two_gauss_radial_width_peak2[n],$
			two_gauss_radial_width_peak2_error[n],$
			two_gauss_radial_position_peak2[n],$
			two_gauss_radial_position_peak2_error[n],$
			FORMAT='(50G20.6)'
		endfor
	endif
	if rocking_scan_yn eq 'y' then begin
		printf,lun,'ROCKING SCAN RESULTS (Single Gaussian)'
		printf,lun,'Temp.','Int.I(Num)','Int.I(Gauss.)','Int.I(Gauss) Error','FWHM(Gauss.)','FWHM(Gauss) Error','Pos.(Gauss.)',$
			'Pos.(Gauss.) Error',format='(50A20)'
		for n=0,n_elements(temps_rocking)-1 do begin
			printF,lun,temps_rocking[n],numerically_integrated_intensity_rocking[n],fitted_integrated_intensity_rocking[n],$
			fitted_integrated_intensity_rocking_error[n],fitted_width_rocking[n],fitted_width_rocking_error[n],fitted_position_rocking[n],$
			fitted_position_rocking_error[n], FORMAT='(50G20.6)'
		endfor
		printf,lun,'ROCKING SCAN RESULTS (2 Gaussians)'
		printf,lun,'Temp.','Int.I(Peak 1)','Int.I(Peak 1) Error','FWHM(Peak 1)','FWHM(Peak 1) Error','Pos.(Peak 1)',$
			'Pos.(Peak 1) Error','Int.I(Peak 2)','Int.I(Peak 2) Error','FWHM(Peak 2)','FWHM(Peak 2) Error','Pos.(Peak 2)',$
			'Pos.(Peak 2) Error',format='(50A20)'
		for n=0,n_elements(temps_rocking)-1 do begin
			printF,lun,temps_rocking[n],$
			two_gauss_rocking_integrated_intensity_peak1[n],$
			two_gauss_rocking_integrated_intensity_peak1_error[n],$
			two_gauss_rocking_width_peak1[n],$
			two_gauss_rocking_width_peak1_error[n],$
			two_gauss_rocking_position_peak1[n],$
			two_gauss_rocking_position_peak1_error[n],$
			two_gauss_rocking_integrated_intensity_peak2[n],$
			two_gauss_rocking_integrated_intensity_peak2_error[n],$
			two_gauss_rocking_width_peak2[n],$
			two_gauss_rocking_width_peak2_error[n],$
			two_gauss_rocking_position_peak2[n],$
			two_gauss_rocking_position_peak2_error[n],$
			FORMAT='(50G20.6)'
		endfor
	endif
close,lun
Free_Lun, lun
end

;OpenW, 12, 'energy_scan_afm_55K.dat'
;   PrintF, 12, 'hw', 'int','err', FORMAT='(5A15)'
;   for i=0,n_elements(energy_escan_afm_55K)-1 do begin
;   	PrintF, 12, energy_escan_afm_55K[i], i_escan_afm_55K[i], err_escan_afm_55K[i], FORMAT='(5G15.6)'
;   endfor;i
;close,12


pro plot_results_handler_radial_and_rocking,event
;handles events from the main analysis routine if there were both radial and rocking scans
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

;the index of the drop-list is stored in 'event.index'!
uname=widget_info(event.id,/uname)
WSET,WINPIX

device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 1) and (number_of_rocking_gaussians eq 1)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y=fitted_integrated_intensity_radial
		e=fitted_integrated_intensity_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 3 then begin
		x=temps_radial
		y=fitted_width_radial
		e=fitted_width_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 4 then begin
		x=temps_radial
		y=fitted_position_radial
		e=fitted_position_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if ((event.index gt 9) and (event.index lt 10+n_elements(temps_radial))) then begin
		x=(*radial_data_array[event.index-10]).x
		y=(*radial_data_array[event.index-10]).y
		e=(*radial_data_array[event.index-10]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_radial[event.index-10],fitted_integrated_intensity_radial[event.index-10],fitted_width_radial[event.index-10]/sqrt(8.*alog(2.)),$
			fitted_position_radial[event.index-10]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif else if event.index eq 5 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 6 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index eq 7 then begin
		x=temps_rocking
		y=fitted_integrated_intensity_rocking
		e=fitted_integrated_intensity_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 8 then begin
		x=temps_rocking
		y=fitted_width_rocking
		e=fitted_width_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 9 then begin
		x=temps_rocking
		y=fitted_position_rocking
		e=fitted_position_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index ge (10+n_elements(temps_radial)) then begin
		x=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).x
		y=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).y
		e=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_rocking[event.index-10-n_elements(temps_radial)],fitted_integrated_intensity_rocking[event.index-10-n_elements(temps_radial)],fitted_width_rocking[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			fitted_position_rocking[event.index-10-n_elements(temps_radial)]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif else if uname eq 'PRINT' then begin
	WSET,WINVIS
	void = TVRead(/JPEG)
endif else if uname eq 'OUTPUTFILE' then begin
	output_file_phase_transitions
endif else if uname eq 'HELP' then begin
	help_phase_transitions
endif else if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 2) and (number_of_rocking_gaussians eq 1)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y1=two_gauss_radial_integrated_intensity_peak1
		y2=two_gauss_radial_integrated_intensity_peak2
		e1=two_gauss_radial_integrated_intensity_peak1_error
		e2=two_gauss_radial_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 3 then begin
		x=temps_radial
		y1=two_gauss_radial_width_peak1
		y2=two_gauss_radial_width_peak2
		e1=two_gauss_radial_width_peak1_error
		e2=two_gauss_radial_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 4 then begin
		x=temps_radial
		y1=two_gauss_radial_position_peak1
		y2=two_gauss_radial_position_peak2
		e1=two_gauss_radial_position_peak1_error
		e2=two_gauss_radial_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if ((event.index gt 9) and (event.index lt 10+n_elements(temps_radial))) then begin
		x=(*radial_data_array[event.index-10]).x
		y=(*radial_data_array[event.index-10]).y
		e=(*radial_data_array[event.index-10]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
;		print,n_elements(two_gauss_radial_fitted_constant)
		p=[two_gauss_radial_fitted_constant[event.index-10],two_gauss_radial_integrated_intensity_peak1[event.index-10],two_gauss_radial_width_peak1[event.index-10]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak1[event.index-10],two_gauss_radial_integrated_intensity_peak2[event.index-10],two_gauss_radial_width_peak2[event.index-10]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak2[event.index-10]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif else if event.index eq 5 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 6 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index eq 7 then begin
		x=temps_rocking
		y=fitted_integrated_intensity_rocking
		e=fitted_integrated_intensity_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 8 then begin
		x=temps_rocking
		y=fitted_width_rocking
		e=fitted_width_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 9 then begin
		x=temps_rocking
		y=fitted_position_rocking
		e=fitted_position_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index ge (10+n_elements(temps_radial)) then begin
		x=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).x
		y=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).y
		e=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_rocking[event.index-10-n_elements(temps_radial)],fitted_integrated_intensity_rocking[event.index-10-n_elements(temps_radial)],fitted_width_rocking[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			fitted_position_rocking[event.index-10-n_elements(temps_radial)]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif else if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 1) and (number_of_rocking_gaussians eq 2)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y=fitted_integrated_intensity_radial
		e=fitted_integrated_intensity_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 3 then begin
		x=temps_radial
		y=fitted_width_radial
		e=fitted_width_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 4 then begin
		x=temps_radial
		y=fitted_position_radial
		e=fitted_position_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if ((event.index gt 9) and (event.index lt 10+n_elements(temps_radial))) then begin
		x=(*radial_data_array[event.index-10]).x
		y=(*radial_data_array[event.index-10]).y
		e=(*radial_data_array[event.index-10]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_radial[event.index-10],fitted_integrated_intensity_radial[event.index-10],fitted_width_radial[event.index-10]/sqrt(8.*alog(2.)),$
			fitted_position_radial[event.index-10]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif else if event.index eq 5 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 6 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index eq 7 then begin
		x=temps_rocking
		y1=two_gauss_rocking_integrated_intensity_peak1
		y2=two_gauss_rocking_integrated_intensity_peak2
		e1=two_gauss_rocking_integrated_intensity_peak1_error
		e2=two_gauss_rocking_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 8 then begin
		x=temps_rocking
		y1=two_gauss_rocking_width_peak1
		y2=two_gauss_rocking_width_peak2
		e1=two_gauss_rocking_width_peak1_error
		e2=two_gauss_rocking_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 9 then begin
		x=temps_rocking
		y1=two_gauss_rocking_position_peak1
		y2=two_gauss_rocking_position_peak2
		e1=two_gauss_rocking_position_peak1_error
		e2=two_gauss_rocking_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index ge (10+n_elements(temps_radial)) then begin
		x=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).x
		y=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).y
		e=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
;		print,n_elements(two_gauss_rocking_fitted_constant)
		p=[two_gauss_rocking_fitted_constant[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_integrated_intensity_peak1[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_width_peak1[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak1[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_integrated_intensity_peak2[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_width_peak2[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak2[event.index-10-n_elements(temps_radial)]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif else if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 2) and (number_of_rocking_gaussians eq 2)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y1=two_gauss_radial_integrated_intensity_peak1
		y2=two_gauss_radial_integrated_intensity_peak2
		e1=two_gauss_radial_integrated_intensity_peak1_error
		e2=two_gauss_radial_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 3 then begin
		x=temps_radial
		y1=two_gauss_radial_width_peak1
		y2=two_gauss_radial_width_peak2
		e1=two_gauss_radial_width_peak1_error
		e2=two_gauss_radial_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 4 then begin
		x=temps_radial
		y1=two_gauss_radial_position_peak1
		y2=two_gauss_radial_position_peak2
		e1=two_gauss_radial_position_peak1_error
		e2=two_gauss_radial_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if ((event.index gt 9) and (event.index lt 10+n_elements(temps_radial))) then begin
		x=(*radial_data_array[event.index-10]).x
		y=(*radial_data_array[event.index-10]).y
		e=(*radial_data_array[event.index-10]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
	;	print,n_elements(two_gauss_radial_fitted_constant)
		p=[two_gauss_radial_fitted_constant[event.index-10],two_gauss_radial_integrated_intensity_peak1[event.index-10],two_gauss_radial_width_peak1[event.index-10]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak1[event.index-10],two_gauss_radial_integrated_intensity_peak2[event.index-10],two_gauss_radial_width_peak2[event.index-10]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak2[event.index-10]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif else if event.index eq 5 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 6 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index eq 7 then begin
		x=temps_rocking
		y1=two_gauss_rocking_integrated_intensity_peak1
		y2=two_gauss_rocking_integrated_intensity_peak2
		e1=two_gauss_rocking_integrated_intensity_peak1_error
		e2=two_gauss_rocking_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 8 then begin
		x=temps_rocking
		y1=two_gauss_rocking_width_peak1
		y2=two_gauss_rocking_width_peak2
		e1=two_gauss_rocking_width_peak1_error
		e2=two_gauss_rocking_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 9 then begin
		x=temps_rocking
		y1=two_gauss_rocking_position_peak1
		y2=two_gauss_rocking_position_peak2
		e1=two_gauss_rocking_position_peak1_error
		e2=two_gauss_rocking_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index ge (10+n_elements(temps_radial)) then begin
		x=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).x
		y=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).y
		e=(*rocking_data_array[event.index-10-n_elements(temps_radial)]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
	;	print,n_elements(two_gauss_rocking_fitted_constant)
		p=[two_gauss_rocking_fitted_constant[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_integrated_intensity_peak1[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_width_peak1[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak1[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_integrated_intensity_peak2[event.index-10-n_elements(temps_radial)],$
			two_gauss_rocking_width_peak2[event.index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak2[event.index-10-n_elements(temps_radial)]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif else if uname eq 'GAUSSIAN_DROPBOX_ROCKING' then begin
	number_of_rocking_gaussians=event.index+1
	if number_of_rocking_gaussians eq 1 then begin
		values[7]='Integ. Intensity-Rocking Scans (Single Gaussian)'
		values[8]='Width-Rocking Scans (Single Gaussian)'
		values[9]='Peak Position-Rocking Scans (Single Gaussian)'
	 	for n=0,n_elements(temps_rocking)-1 do begin
			values[10+n_elements(temps_radial)+n]=string(temps_rocking[n])+' K rocking fitting (Single Gaussian)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif else if number_of_rocking_gaussians eq 2 then begin
		values[7]='Integ. Intensity-Rocking Scans (2 Gaussians)'
		values[8]='Width-Rocking Scans (2 Gaussians)'
		values[9]='Peak Position-Rocking Scans (2 Gaussians)'
	 	for n=0,n_elements(temps_rocking)-1 do begin
			values[10+n_elements(temps_radial)+n]=string(temps_rocking[n])+' K rocking fitting (2 Gaussians)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif
	widget_control,droplist_id_find,set_combobox_select=0
	droplist_index=0
	plot_contour_radial_scans
endif else if uname eq 'GAUSSIAN_DROPBOX_RADIAL' then begin
	number_of_radial_gaussians=event.index+1
	if number_of_radial_gaussians eq 1 then begin
		values[2]='Integ. Intensity-Radial Scans (Single Gaussian)'
		values[3]='Width-Radial Scans (Single Gaussian)'
		values[4]='Peak Position-Radial Scans (Single Gaussian)'
	 	for n=0,n_elements(temps_radial)-1 do begin
			values[10+n]=string(temps_radial[n])+' K radial fitting (Single Gaussian)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif else if number_of_radial_gaussians eq 2 then begin
		values[2]='Integ. Intensities-Radial Scans (2 Gaussians)'
		values[3]='Widths-Radial Scans (2 Gaussians)'
		values[4]='Positions-Radial Scans (2 Gaussians)'
	 	for n=0,n_elements(temps_radial)-1 do begin
			values[10+n]=string(temps_radial[n])+' K radial fitting (2 Gaussians)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif
	widget_control,droplist_id_find,set_combobox_select=0
	droplist_index=0
	plot_contour_radial_scans
endif
end

pro plot_results_handler_rocking_only,event
;this handles events from the main analysis routine if there were only rocking scans
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

;the index of the drop-list is stored in 'event.index'!
uname=widget_info(event.id,/uname)
WSET,WINPIX
device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

if ((uname eq 'DROPLIST') and (number_of_rocking_gaussians eq 1)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 2 then begin
		x=temps_rocking
		y=fitted_integrated_intensity_rocking
		e=fitted_integrated_intensity_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 3 then begin
		x=temps_rocking
		y=fitted_width_rocking
		e=fitted_width_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 4 then begin
		x=temps_rocking
		y=fitted_position_rocking
		e=fitted_position_rocking_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
;modify the results handler here to add numerically integrated intensity!  and then fit below.
	endif else if event.index eq 1 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index gt 4 then begin
		x=(*rocking_data_array[event.index-5]).x
		y=(*rocking_data_array[event.index-5]).y
		e=(*rocking_data_array[event.index-5]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_rocking[event.index-5],fitted_integrated_intensity_rocking[event.index-5],fitted_width_rocking[event.index-5]/sqrt(8.*alog(2.)),$
			fitted_position_rocking[event.index-5]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
    *(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y

	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]

endif else if uname eq 'PRINT' then begin
	WSET,WINVIS
	void = TVRead(/JPEG)
endif else if uname eq 'OUTPUTFILE' then begin
	output_file_phase_transitions
endif else if uname eq 'HELP' then begin
	help_phase_transitions
endif else if uname eq 'GAUSSIAN_DROPBOX_ROCKING' then begin
	number_of_rocking_gaussians=event.index+1
	if number_of_rocking_gaussians eq 1 then begin
		values=strarr(5+n_elements(temps_rocking))
		values[0]='Rocking Contour'
		values[1]='Integ. Intensity-Rocking Scans (Numerical)'
		values[2]='Integ. Intensity-Rocking Scans (Single Gaussian)'
		values[3]='Width-Rocking Scans (Single Gaussian)'
		values[4]='Peak Position-Rocking Scans (Single Gaussian)'
		for n=5,n_elements(values)-1 do begin
			values[n]=string(temps_rocking[n-5])+' K rocking fitting (Single Gaussian)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif else if number_of_rocking_gaussians eq 2 then begin
		values=strarr(5+n_elements(temps_rocking))
		values[0]='Rocking Contour'
		values[1]='Integ. Intensity-Rocking Scans (Numerical)'
		values[2]='Integ. Intensity-Rocking Scans (2 Gaussians)'
		values[3]='Width-Rocking Scans (2 Gaussians)'
		values[4]='Peak Position-Rocking Scans (2 Gaussians)'
		for n=5,n_elements(values)-1 do begin
			values[n]=string(temps_rocking[n-5])+' K rocking fitting (2 Gaussians)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif
	widget_control,droplist_id_find,set_combobox_select=0
	droplist_index=0
	plot_contour_rocking_scans
endif else if ((uname eq 'DROPLIST') and (number_of_rocking_gaussians eq 2)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_rocking_scans
	endif else if event.index eq 2 then begin
		x=temps_rocking
		y1=two_gauss_rocking_integrated_intensity_peak1
		y2=two_gauss_rocking_integrated_intensity_peak2
		e1=two_gauss_rocking_integrated_intensity_peak1_error
		e2=two_gauss_rocking_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 3 then begin
		x=temps_rocking
		y1=two_gauss_rocking_width_peak1
		y2=two_gauss_rocking_width_peak2
		e1=two_gauss_rocking_width_peak1_error
		e2=two_gauss_rocking_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 4 then begin
		x=temps_rocking
		y1=two_gauss_rocking_position_peak1
		y2=two_gauss_rocking_position_peak2
		e1=two_gauss_rocking_position_peak1_error
		e2=two_gauss_rocking_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 1 then begin
		x=temps_rocking
		y=numerically_integrated_intensity_rocking
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index gt 4 then begin
		x=(*rocking_data_array[event.index-5]).x
		y=(*rocking_data_array[event.index-5]).y
		e=(*rocking_data_array[event.index-5]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
;		print,n_elements(two_gauss_rocking_fitted_constant)
		p=[two_gauss_rocking_fitted_constant[event.index-5],two_gauss_rocking_integrated_intensity_peak1[event.index-5],two_gauss_rocking_width_peak1[event.index-5]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak1[event.index-5],two_gauss_rocking_integrated_intensity_peak2[event.index-5],two_gauss_rocking_width_peak2[event.index-5]/sqrt(8.*alog(2.)),$
			two_gauss_rocking_position_peak2[event.index-5]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5

    *(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y

	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif
end

pro plot_contour_rocking_scans
;plots the rocking scan contour
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

;HELP,PSTATE
WSET,WINPIX
device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

places=where(z_rocking lt 0)
if where(z_rocking lt 0.) ne [-1] then z_rocking[places]=0.
levels = 54
maxz=max(z_rocking)
minz=min(z_rocking)
step = (maxz - minz) / levels
userLevels = IndGen(levels+1) * step + minz
contour,z_rocking,xint_rocking,temps_rocking,/fill,levels=userlevels,$
	xrange=[min(xint_rocking),max(xint_rocking)],/xstyle,background=white,color=black,yrange=[min(temps_rocking),max(temps_rocking)],/ystyle,$
	charsize=2.0,xtitle='A3',ytitle='T (K)',Position=[0.17, 0.14, 0.9, 0.80]
loadct,13
contour,z_rocking,xint_rocking,temps_rocking,/fill,levels=userlevels,/overplot
loadct,39
xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5


ColorBar, NColors=255,$
  		Range=[minz,maxz], Format='(I5)', $
  		Position = [0.5, 0.85, 0.9, 0.9], Color=black,title='Counts'


WSET,WINVIS
DEVICE,COPY=[0,0,600,500,0,0,WINPIX]

end

pro plot_contour_radial_scans
;plots the radial scan contour
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

WSET,WINPIX
device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

places=where(z_radial lt 0)
if where(z_radial lt 0.) ne [-1] then z_radial[places]=0.
levels = 54
maxz=max(z_radial)
minz=min(z_radial)
step = (maxz - minz) / levels
userLevels = IndGen(levels+1) * step + minz
contour,z_radial,xint_radial,temps_radial,/fill,levels=userlevels,$
	xrange=[min(xint_radial),max(xint_radial)],/xstyle,background=white,color=black,yrange=[min(temps_radial),max(temps_radial)],/ystyle,$
	charsize=2.0,xtitle='2 Theta',ytitle='T (K)',Position=[0.17, 0.14, 0.9, 0.80]
loadct,13
contour,z_radial,xint_radial,temps_radial,/fill,levels=userlevels,/overplot
loadct,39
xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
ColorBar, NColors=255,$
  		Range=[minz,maxz], Format='(I5)', $
  		Position = [0.5, 0.85, 0.9, 0.9], Color=black,title='Counts'

WSET,WINVIS
DEVICE,COPY=[0,0,600,500,0,0,WINPIX]

end


pro plot_results_handler_radial_only,event
;handles events from the main analysis routine if there were only radial scans
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians


;the index of the drop-list is stored in 'event.index'!
uname=widget_info(event.id,/uname)
WSET,WINPIX
device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 1)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y=fitted_integrated_intensity_radial
		e=fitted_integrated_intensity_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 3 then begin
		x=temps_radial
		y=fitted_width_radial
		e=fitted_width_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 4 then begin
		x=temps_radial
		y=fitted_position_radial
		e=fitted_position_radial_error
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index gt 4 then begin
		x=(*radial_data_array[event.index-5]).x
		y=(*radial_data_array[event.index-5]).y
		e=(*radial_data_array[event.index-5]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		p=[fitted_constant_radial[event.index-5],fitted_integrated_intensity_radial[event.index-5],fitted_width_radial[event.index-5]/sqrt(8.*alog(2.)),$
			fitted_position_radial[event.index-5]]
		gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5

    *(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y

	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif else if uname eq 'PRINT' then begin
	WSET,WINVIS
	void = TVRead(/JPEG)
endif else if uname eq 'OUTPUTFILE' then begin
	output_file_phase_transitions
endif else if uname eq 'HELP' then begin
	help_phase_transitions
endif else if uname eq 'GAUSSIAN_DROPBOX_RADIAL' then begin
	number_of_radial_gaussians=event.index+1
	if number_of_radial_gaussians eq 1 then begin
;		values=strarr(5+n_elements(temps_radial))
		values[0]='Radial Contour'
		values[1]='Integ. Intensity-Radial Scans (Numerical)'
		values[2]='Integ. Intensity-Radial Scans (Single Gaussian)'
		values[3]='Width-Radial Scans (Single Gaussian)'
		values[4]='Peak Position-Radial Scans (Single Gaussian)'
		for n=5,n_elements(values)-1 do begin
			values[n]=string(temps_radial[n-5])+' K radial fitting (Single Gaussian)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif else if number_of_radial_gaussians eq 2 then begin
;		values=strarr(5+n_elements(temps_radial))
		values[0]='Radial Contour'
		values[1]='Integ. Intensity-Radial Scans (Numerical)'
		values[2]='Integ. Intensities-Radial Scans (2 Gaussians)'
		values[3]='Widths-Radial Scans (2 Gaussians)'
		values[4]='Positions-Radial Scans (2 Gaussians)'
		for n=5,n_elements(values)-1 do begin
			values[n]=string(temps_radial[n-5])+' K radial fitting (2 Gaussians)'
		endfor
		;get the widget id for the droplist
		droplist_id_find=widget_info(event.top,find_by_uname='DROPLIST')
		widget_control,droplist_id_find,set_value=values
	endif
	widget_control,droplist_id_find,set_combobox_select=0
	droplist_index=0
	plot_contour_radial_scans
endif else if ((uname eq 'DROPLIST') and (number_of_radial_gaussians eq 2)) then begin
	droplist_index=event.index
	if event.index eq 0 then begin
		plot_contour_radial_scans
	endif else if event.index eq 2 then begin
		x=temps_radial
		y1=two_gauss_radial_integrated_intensity_peak1
		y2=two_gauss_radial_integrated_intensity_peak2
		e1=two_gauss_radial_integrated_intensity_peak1_error
		e2=two_gauss_radial_integrated_intensity_peak2_error
		y3=y1+y2
		e3=sqrt(e1^2.+e2^2.)
		ymin=min([y1,y2,y3])
		ymax=max([y1,y2,y3])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		oplot,x,y3,color=red,psym=8
		errplot,x,y3-e3,y3+e3,color=red
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
		xyouts,.85,.83,'Sum',color=red,/norm
	endif else if event.index eq 3 then begin
		x=temps_radial
		y1=two_gauss_radial_width_peak1
		y2=two_gauss_radial_width_peak2
		e1=two_gauss_radial_width_peak1_error
		e2=two_gauss_radial_width_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 4 then begin
		x=temps_radial
		y1=two_gauss_radial_position_peak1
		y2=two_gauss_radial_position_peak2
		e1=two_gauss_radial_position_peak1_error
		e2=two_gauss_radial_position_peak2_error
		ymin=min([y1,y2])
		ymax=max([y1,y2])
		plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
			yrange=[ymin-(ymax-ymin)/5.,ymax+(ymax-ymin)/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y1-e1,y1+e1,color=black
		oplot,x,y2,color=blue,psym=8
		errplot,x,y2-e2,y2+e2,color=blue
		xyouts,.85,.89,'Peak 1',color=black,/norm
		xyouts,.85,.86,'Peak 2',color=blue,/norm
	endif else if event.index eq 1 then begin
		x=temps_radial
		y=numerically_integrated_intensity_radial
		plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
	endif else if event.index gt 4 then begin
		x=(*radial_data_array[event.index-5]).x
		y=(*radial_data_array[event.index-5]).y
		e=(*radial_data_array[event.index-5]).e
		plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
			yrange=[min(y)-(max(y)-min(y))/5.,max(y)+(max(y)-min(y))/5.],xrange=[min(x)-(max(x)-min(x))/5.,max(x)+(max(x)-min(x))/5.],/xstyle,/ystyle
		errplot,x,y-e,y+e,color=black
		fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
	;	print,n_elements(two_gauss_radial_fitted_constant)
		p=[two_gauss_radial_fitted_constant[event.index-5],two_gauss_radial_integrated_intensity_peak1[event.index-5],two_gauss_radial_width_peak1[event.index-5]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak1[event.index-5],two_gauss_radial_integrated_intensity_peak2[event.index-5],two_gauss_radial_width_peak2[event.index-5]/sqrt(8.*alog(2.)),$
			two_gauss_radial_position_peak2[event.index-5]]
		two_gauss_phase_transitions,fitx,p,fitted
		oplot,fitx,fitted,color=black,thick=2.0
	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5

    *(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y

	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif
end

pro gauss_phase_transitions, x, p, yfit
; The independent variable is X
; Parameter values are passed in "P"
yfit = p[0]+p[1]/(p[2]*sqrt(2.*!pi))*exp(-0.5*((x-p[3])/p[2])^2.)
END

PRO mouse_event,EVENT
;this procedure handles mouse events for the main drawing window.  ie, it handles the boxed rescaling

common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

WSET,WINPIX

	;PRINT,'mouse_event'
	;HELP,EVENT,/STRUCT
;	PRINT,EVENT.x,EVENT.Y,EVENT.PRESS
	WIDGET_CONTROL,EVENT.ID,GET_UVALUE=PSTATE
	;HELP,PSTATE
	!X = *(*PSTATE).PX
	!Y = *(*PSTATE).PY

IF ((EVENT.PRESS NE 0) and (droplist_index ne 0)) THEN BEGIN
		(*PSTATE).PRESSED = 1
		(*PSTATE).XPRESS = EVENT.X
		(*PSTATE).YPRESS = EVENT.Y
ENDIF

IF ((EVENT.RELEASE NE 0) and (droplist_index ne 0)) THEN BEGIN
	(*PSTATE).PRESSED = 0
	R1 = CONVERT_COORD((*PSTATE).XPRESS,(*PSTATE).YPRESS,/DEVICE,/TO_DATA)
	R2 = CONVERT_COORD(EVENT.X,EVENT.Y,/DEVICE,/TO_DATA)
	X0 = MIN([R1[0],R2[0]])
	X1 = MAX([R1[0],R2[0]])
	Y0 = MIN([R1[1],R2[1]])
	Y1 = MAX([R1[1],R2[1]])
	x0_copy=x0
	y0_copy=y0
	x1_copy=x1
	y1_copy=y1

	;-------------------- here we have to replot with the new x-range or y-range!
	;first determine the state (radial only, rocking only, or both radial and rocking)
	if ((rocking_scan_yn eq 'y') and (radial_scan_yn eq 'n')) then begin
		device,decomposed=0
		loadct,39
		red=254
		green=142
		blue=59
		yellow=191
		black=0
		purple=22
		white=255
		;plotsym,0,/fill
		psym = dave_plotsym(/circle)
		
		if number_of_rocking_gaussians eq 1 then begin
			if droplist_index eq 0 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 2 then begin
				x=temps_rocking
				y=fitted_integrated_intensity_rocking
				e=fitted_integrated_intensity_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 3 then begin
				x=temps_rocking
				y=fitted_width_rocking
				e=fitted_width_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
				yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 4 then begin
				x=temps_rocking
				y=fitted_position_rocking
				e=fitted_position_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 1 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
			endif else if droplist_index gt 4 then begin
				x=(*rocking_data_array[droplist_index-5]).x
				y=(*rocking_data_array[droplist_index-5]).y
				e=(*rocking_data_array[droplist_index-5]).e
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_rocking[droplist_index-5],fitted_integrated_intensity_rocking[droplist_index-5],fitted_width_rocking[droplist_index-5]/sqrt(8.*alog(2.)),$
					fitted_position_rocking[droplist_index-5]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
		xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
    	*(*PSTATE).PX = !X
		*(*PSTATE).PY = !Y
		WSET,WINVIS
		DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
		endif else if number_of_rocking_gaussians eq 2 then begin
			if droplist_index eq 0 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 2 then begin
				x=temps_rocking
				y1=two_gauss_rocking_integrated_intensity_peak1
				y2=two_gauss_rocking_integrated_intensity_peak2
				e1=two_gauss_rocking_integrated_intensity_peak1_error
				e2=two_gauss_rocking_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 3 then begin
				x=temps_rocking
				y1=two_gauss_rocking_width_peak1
				y2=two_gauss_rocking_width_peak2
				e1=two_gauss_rocking_width_peak1_error
				e2=two_gauss_rocking_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 4 then begin
				x=temps_rocking
				y1=two_gauss_rocking_position_peak1
				y2=two_gauss_rocking_position_peak2
				e1=two_gauss_rocking_position_peak1_error
				e2=two_gauss_rocking_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 1 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if droplist_index gt 4 then begin
				x=(*rocking_data_array[droplist_index-5]).x
				y=(*rocking_data_array[droplist_index-5]).y
				e=(*rocking_data_array[droplist_index-5]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[two_gauss_rocking_fitted_constant[droplist_index-5],two_gauss_rocking_integrated_intensity_peak1[droplist_index-5],two_gauss_rocking_width_peak1[droplist_index-5]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak1[droplist_index-5],two_gauss_rocking_integrated_intensity_peak2[droplist_index-5],two_gauss_rocking_width_peak2[droplist_index-5]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak2[droplist_index-5]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
			xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
		    *(*PSTATE).PX = !X
			*(*PSTATE).PY = !Y
			WSET,WINVIS
			DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
		endif
	endif else if ((rocking_scan_yn eq 'n') and (radial_scan_yn eq 'y')) then begin
		device,decomposed=0
		loadct,39
		red=254
		green=142
		blue=59
		yellow=191
		black=0
		purple=22
		white=255
		;plotsym,0,/fill
		psym = dave_plotsym(/circle)
		
  		if number_of_radial_gaussians eq 1 then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y=fitted_integrated_intensity_radial
				e=fitted_integrated_intensity_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y=fitted_width_radial
				e=fitted_width_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y=fitted_position_radial
				e=fitted_position_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
			endif else if droplist_index gt 4 then begin
				x=(*radial_data_array[droplist_index-5]).x
				y=(*radial_data_array[droplist_index-5]).y
				e=(*radial_data_array[droplist_index-5]).e
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_radial[droplist_index-5],fitted_integrated_intensity_radial[droplist_index-5],fitted_width_radial[droplist_index-5]/sqrt(8.*alog(2.)),$
					fitted_position_radial[droplist_index-5]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
  		endif else if number_of_radial_gaussians eq 2 then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y1=two_gauss_radial_integrated_intensity_peak1
				y2=two_gauss_radial_integrated_intensity_peak2
				e1=two_gauss_radial_integrated_intensity_peak1_error
				e2=two_gauss_radial_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y1=two_gauss_radial_width_peak1
				y2=two_gauss_radial_width_peak2
				e1=two_gauss_radial_width_peak1_error
				e2=two_gauss_radial_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y1=two_gauss_radial_position_peak1
				y2=two_gauss_radial_position_peak2
				e1=two_gauss_radial_position_peak1_error
				e2=two_gauss_radial_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					ymin=min(y)
					ymax=max(y)
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if droplist_index gt 4 then begin
				x=(*radial_data_array[droplist_index-5]).x
				y=(*radial_data_array[droplist_index-5]).y
				e=(*radial_data_array[droplist_index-5]).e
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					ymin=min(y)
					ymax=max(y)
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[two_gauss_radial_fitted_constant[droplist_index-5],two_gauss_radial_integrated_intensity_peak1[droplist_index-5],two_gauss_radial_width_peak1[droplist_index-5]/sqrt(8.*alog(2.)),$
				two_gauss_radial_position_peak1[droplist_index-5],two_gauss_radial_integrated_intensity_peak2[droplist_index-5],two_gauss_radial_width_peak2[droplist_index-5]/sqrt(8.*alog(2.)),$
				two_gauss_radial_position_peak2[droplist_index-5]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
		endif
		xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
    	*(*PSTATE).PX = !X
		*(*PSTATE).PY = !Y
		WSET,WINVIS
		DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
	endif else if ((rocking_scan_yn eq 'y') and (radial_scan_yn eq 'y')) then begin
		device,decomposed=0
		loadct,39
		red=254
		green=142
		blue=59
		yellow=191
		black=0
		purple=22
		white=255
		;plotsym,0,/fill
		psym = dave_plotsym(/circle)
		
		if ((number_of_radial_gaussians eq 1) and (number_of_rocking_gaussians eq 1)) then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y=fitted_integrated_intensity_radial
				e=fitted_integrated_intensity_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y=fitted_width_radial
				e=fitted_width_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y=fitted_position_radial
				e=fitted_position_radial_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
			endif else if ((droplist_index gt 9) and (droplist_index lt 10+n_elements(temps_radial))) then begin
				x=(*radial_data_array[droplist_index-10]).x
				y=(*radial_data_array[droplist_index-10]).y
				e=(*radial_data_array[droplist_index-10]).e
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_radial[droplist_index-10],fitted_integrated_intensity_radial[droplist_index-10],fitted_width_radial[droplist_index-10]/sqrt(8.*alog(2.)),$
					fitted_position_radial[droplist_index-10]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif else if droplist_index eq 5 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 6 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
			endif else if droplist_index eq 7 then begin
				x=temps_rocking
				y=fitted_integrated_intensity_rocking
				e=fitted_integrated_intensity_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 8 then begin
				x=temps_rocking
				y=fitted_width_rocking
				e=fitted_width_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 9 then begin
				x=temps_rocking
				y=fitted_position_rocking
				e=fitted_position_rocking_error
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index ge (10+n_elements(temps_radial)) then begin
				x=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).x
				y=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).y
				e=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).e
				if ((x0 eq x1) and (y0 eq y1)) then begin
					x0=min(x)-(max(x)-min(x))/5.
					x1=max(x)+(max(x)-min(x))/5.
					y0=min(y)-(max(y)-min(y))/5.
					y1=max(y)+(max(y)-min(y))/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
					yrange=[y0,y1],xrange=[x0,x1],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_rocking[droplist_index-10-n_elements(temps_radial)],fitted_integrated_intensity_rocking[droplist_index-10-n_elements(temps_radial)],fitted_width_rocking[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					fitted_position_rocking[droplist_index-10-n_elements(temps_radial)]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
  		endif else if ((number_of_radial_gaussians eq 2) and (number_of_rocking_gaussians eq 1)) then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y1=two_gauss_radial_integrated_intensity_peak1
				y2=two_gauss_radial_integrated_intensity_peak2
				e1=two_gauss_radial_integrated_intensity_peak1_error
				e2=two_gauss_radial_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y1=two_gauss_radial_width_peak1
				y2=two_gauss_radial_width_peak2
				e1=two_gauss_radial_width_peak1_error
				e2=two_gauss_radial_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y1=two_gauss_radial_position_peak1
				y2=two_gauss_radial_position_peak2
				e1=two_gauss_radial_position_peak1_error
				e2=two_gauss_radial_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if ((droplist_index gt 9) and (droplist_index lt 10+n_elements(temps_radial))) then begin
				x=(*radial_data_array[droplist_index-10]).x
				y=(*radial_data_array[droplist_index-10]).y
				e=(*radial_data_array[droplist_index-10]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		;		print,n_elements(two_gauss_radial_fitted_constant)
				p=[two_gauss_radial_fitted_constant[droplist_index-10],two_gauss_radial_integrated_intensity_peak1[droplist_index-10],two_gauss_radial_width_peak1[droplist_index-10]/sqrt(8.*alog(2.)),$
					two_gauss_radial_position_peak1[droplist_index-10],two_gauss_radial_integrated_intensity_peak2[droplist_index-10],two_gauss_radial_width_peak2[droplist_index-10]/sqrt(8.*alog(2.)),$
					two_gauss_radial_position_peak2[droplist_index-10]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif else if droplist_index eq 5 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 6 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if droplist_index eq 7 then begin
				x=temps_rocking
				y=fitted_integrated_intensity_rocking
				e=fitted_integrated_intensity_rocking_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 8 then begin
				x=temps_rocking
				y=fitted_width_rocking
				e=fitted_width_rocking_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 9 then begin
				x=temps_rocking
				y=fitted_position_rocking
				e=fitted_position_rocking_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index ge (10+n_elements(temps_radial)) then begin
				x=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).x
				y=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).y
				e=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='A3',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_rocking[droplist_index-10-n_elements(temps_radial)],fitted_integrated_intensity_rocking[droplist_index-10-n_elements(temps_radial)],fitted_width_rocking[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					fitted_position_rocking[droplist_index-10-n_elements(temps_radial)]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
  		endif else if ((number_of_radial_gaussians eq 1) and (number_of_rocking_gaussians eq 2)) then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y=fitted_integrated_intensity_radial
				e=fitted_integrated_intensity_radial_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y=fitted_width_radial
				e=fitted_width_radial_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y=fitted_position_radial
				e=fitted_position_radial_error
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if ((droplist_index gt 9) and (droplist_index lt 10+n_elements(temps_radial))) then begin
				x=(*radial_data_array[droplist_index-10]).x
				y=(*radial_data_array[droplist_index-10]).y
				e=(*radial_data_array[droplist_index-10]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
				p=[fitted_constant_radial[droplist_index-10],fitted_integrated_intensity_radial[droplist_index-10],fitted_width_radial[droplist_index-10]/sqrt(8.*alog(2.)),$
					fitted_position_radial[droplist_index-10]]
				gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif else if droplist_index eq 5 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 6 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if droplist_index eq 7 then begin
				x=temps_rocking
				y1=two_gauss_rocking_integrated_intensity_peak1
				y2=two_gauss_rocking_integrated_intensity_peak2
				e1=two_gauss_rocking_integrated_intensity_peak1_error
				e2=two_gauss_rocking_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 8 then begin
				x=temps_rocking
				y1=two_gauss_rocking_width_peak1
				y2=two_gauss_rocking_width_peak2
				e1=two_gauss_rocking_width_peak1_error
				e2=two_gauss_rocking_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 9 then begin
				x=temps_rocking
				y1=two_gauss_rocking_position_peak1
				y2=two_gauss_rocking_position_peak2
				e1=two_gauss_rocking_position_peak1_error
				e2=two_gauss_rocking_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index ge (10+n_elements(temps_radial)) then begin
				x=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).x
				y=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).y
				e=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
					p=[two_gauss_rocking_fitted_constant[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_integrated_intensity_peak1[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_width_peak1[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak1[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_integrated_intensity_peak2[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_width_peak2[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak2[droplist_index-10-n_elements(temps_radial)]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
  		endif else if ((number_of_radial_gaussians eq 2) and (number_of_rocking_gaussians eq 2)) then begin
			if droplist_index eq 0 then begin
				plot_contour_radial_scans
			endif else if droplist_index eq 2 then begin
				x=temps_radial
				y1=two_gauss_radial_integrated_intensity_peak1
				y2=two_gauss_radial_integrated_intensity_peak2
				e1=two_gauss_radial_integrated_intensity_peak1_error
				e2=two_gauss_radial_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 3 then begin
				x=temps_radial
				y1=two_gauss_radial_width_peak1
				y2=two_gauss_radial_width_peak2
				e1=two_gauss_radial_width_peak1_error
				e2=two_gauss_radial_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 4 then begin
				x=temps_radial
				y1=two_gauss_radial_position_peak1
				y2=two_gauss_radial_position_peak2
				e1=two_gauss_radial_position_peak1_error
				e2=two_gauss_radial_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 1 then begin
				x=temps_radial
				y=numerically_integrated_intensity_radial
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if ((droplist_index gt 9) and (droplist_index lt 10+n_elements(temps_radial))) then begin
				x=(*radial_data_array[droplist_index-10]).x
				y=(*radial_data_array[droplist_index-10]).y
				e=(*radial_data_array[droplist_index-10]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
		;		print,n_elements(two_gauss_radial_fitted_constant)
				p=[two_gauss_radial_fitted_constant[droplist_index-10],two_gauss_radial_integrated_intensity_peak1[droplist_index-10],two_gauss_radial_width_peak1[droplist_index-10]/sqrt(8.*alog(2.)),$
					two_gauss_radial_position_peak1[droplist_index-10],two_gauss_radial_integrated_intensity_peak2[droplist_index-10],two_gauss_radial_width_peak2[droplist_index-10]/sqrt(8.*alog(2.)),$
					two_gauss_radial_position_peak2[droplist_index-10]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif else if droplist_index eq 5 then begin
				plot_contour_rocking_scans
			endif else if droplist_index eq 6 then begin
				x=temps_rocking
				y=numerically_integrated_intensity_rocking
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
			endif else if droplist_index eq 7 then begin
				x=temps_rocking
				y1=two_gauss_rocking_integrated_intensity_peak1
				y2=two_gauss_rocking_integrated_intensity_peak2
				e1=two_gauss_rocking_integrated_intensity_peak1_error
				e2=two_gauss_rocking_integrated_intensity_peak2_error
				y3=y1+y2
				e3=sqrt(e1^2.+e2^2.)
				ymin=min([y1,y2,y3])
				ymax=max([y1,y2,y3])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Integrated Intensity (arb. unit)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				oplot,x,y3,color=red,psym=8
				errplot,x,y3-e3,y3+e3,color=red
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
				xyouts,.85,.83,'Sum',color=red,/norm
			endif else if droplist_index eq 8 then begin
				x=temps_rocking
				y1=two_gauss_rocking_width_peak1
				y2=two_gauss_rocking_width_peak2
				e1=two_gauss_rocking_width_peak1_error
				e2=two_gauss_rocking_width_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='FWHM (deg.)',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index eq 9 then begin
				x=temps_rocking
				y1=two_gauss_rocking_position_peak1
				y2=two_gauss_rocking_position_peak2
				e1=two_gauss_rocking_position_peak1_error
				e2=two_gauss_rocking_position_peak2_error
				ymin=min([y1,y2])
				ymax=max([y1,y2])
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y1,color=black,background=white,psym=8,xtitle='T (K)',ytitle='Peak Position',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y1-e1,y1+e1,color=black
				oplot,x,y2,color=blue,psym=8
				errplot,x,y2-e2,y2+e2,color=blue
				xyouts,.85,.89,'Peak 1',color=black,/norm
				xyouts,.85,.86,'Peak 2',color=blue,/norm
			endif else if droplist_index ge (10+n_elements(temps_radial)) then begin
				x=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).x
				y=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).y
				e=(*rocking_data_array[droplist_index-10-n_elements(temps_radial)]).e
				ymin=min(y)
				ymax=max(y)
				if ((x0_copy eq x1_copy) and (y0_copy eq y1_copy)) then begin
					x0_copy=min(x)-(max(x)-min(x))/5.
					x1_copy=max(x)+(max(x)-min(x))/5.
					y0_copy=ymin-(ymax-ymin)/5.
					y1_copy=ymax+(ymax-ymin)/5.
				endif
				plot,x,y,color=black,background=white,psym=8,xtitle='2 Theta',ytitle='Intensity',charsize=2.0,$
					yrange=[y0_copy,y1_copy],xrange=[x0_copy,x1_copy],/xstyle,/ystyle
				errplot,x,y-e,y+e,color=black
				fitx=makepts(xlo=min(x),xhi=max(x),npts=1000)
	;			print,n_elements(two_gauss_rocking_fitted_constant)
				p=[two_gauss_rocking_fitted_constant[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_integrated_intensity_peak1[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_width_peak1[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak1[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_integrated_intensity_peak2[droplist_index-10-n_elements(temps_radial)],$
					two_gauss_rocking_width_peak2[droplist_index-10-n_elements(temps_radial)]/sqrt(8.*alog(2.)),$
					two_gauss_rocking_position_peak2[droplist_index-10-n_elements(temps_radial)]]
				two_gauss_phase_transitions,fitx,p,fitted
				oplot,fitx,fitted,color=black,thick=2.0
			endif
		endif
  	endif
	xyouts,.1,0.95,values[droplist_index],color=black,/norm,charsize=1.5
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	WSET,WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,WINPIX]
endif

IF (*PSTATE).PRESSED EQ 1 THEN BEGIN
	device,decomposed=0
	loadct,39
	red=254
	green=142
	blue=59
	yellow=191
	black=0
	purple=22
	white=255
	;plotsym,0,/fill
	psym = dave_plotsym(/circle)
	
	;WSET,(*PSTATE).WINPIX
	R2 = CONVERT_COORD(EVENT.X,EVENT.Y,/DEVICE,/TO_DATA)
	R1 = CONVERT_COORD((*PSTATE).XPRESS,(*PSTATE).YPRESS,/DEVICE,/TO_DATA)
	;PRINT,'R=',R
	WSET,(*PSTATE).WINVIS
	DEVICE,COPY=[0,0,600,500,0,0,(*PSTATE).WINPIX]
	XBOX = [R1[0],R2[0],R2[0],R1[0],R1[0]]
	YBOX = [R1[1],R1[1],R2[1],R2[1],R1[1]]
	X0 = MIN([(*PSTATE).XPRESS,EVENT.X])
	X1 = MAX([(*PSTATE).XPRESS,EVENT.X])
	Y0 = MIN([(*PSTATE).YPRESS,EVENT.Y])
	Y1 = MAX([(*PSTATE).YPRESS,EVENT.Y])
	;XBOX = [(*PSTATE).XPRESS,EVENT.X,EVENT.X,(*PSTATE).XPRESS,(*PSTATE).XPRESS]
	;YBOX = [(*PSTATE).YPRESS,(*PSTATE).YPRESS,EVENT.Y,EVENT.Y,(*PSTATE).YPRESS]
	XBOX = [X0,X1,X1,X0,X0]
	YBOX = [Y0,Y0,Y1,Y1,Y0]
;	PRINT,EVENT.X,EVENT.Y,CONVERT_COORD(EVENT.X,EVENT.Y,/DEVICE,/TO_DATA);XBOX,YBOX



	PLOTS,XBOX,YBOX,$
				;/NOERASE,$
				COLOR=red,$
				;XSTYLE=4,YSTYLE=4,$
				/DEVICE

ENDIF
END;mouse_event

PRO mouse_cleanup,ID
	WIDGET_CONTROL,ID,GET_UVALUE=PSTATE
	PTR_FREE,(*PSTATE).PX
	PTR_FREE,(*PSTATE).PY
	PTR_FREE,PSTATE
END;mouse_cleanup

pro phase_transition_main_analysis
;This is the main analysis routine.  After all rocking and radial scan selections have been made it is called.
;The data for both types of scans is extracted here.  The data is fit, the contours are determined, and the main plotting widget is realized.

common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

device,decomposed=0
loadct,39
red=254
green=142
blue=59
yellow=191
black=0
purple=22
white=255
;plotsym,0,/fill
psym = dave_plotsym(/circle)

droplist_index=0 ;this is the initial index of the droplist.
number_of_radial_gaussians=1 ;this is the default number of radial gaussians
number_of_rocking_gaussians=1 ;this is the default number of rocking gaussians
if radial_scan_yn eq 'y' then begin	;do the radial analysis

	;determine what instrument we are deailing with..
	filebasenames=file_basename(fn_radial)
	if strmid(filebasenames[0],2,/reverse) eq 'bt9' then begin
		instrument='bt9'
		temps_radial=float(temps_radial) ;convert the temperatures from string to float
		scale_for_radial=float(scaling_radial) ;get the scale factors.
		endif else if strmid(filebasenames[0],2,/reverse) eq 'ng5' then begin
		instrument='ng5'
		temps_radial=float(temps_radial) ;convert the temperatures from string to float
		scale_for_radial=float(scaling_radial) ;get the scale factors.
		endif else if strmid(filebasenames[0],2,/reverse) eq 'bt7' then begin
		instrument='bt7'
		temps_radial=fltarr(n_elements(fn_radial)) ;create an array for the temperatures, which will be read from the file below
	endif else begin
		;instrument unknown
		return
	endelse


	;make a loop to input data
	for n=0,n_elements(fn_radial)-1 do begin
		;first read the file!
		if (instrument eq 'bt9') or (instrument eq 'ng5') then begin
			read_object=obj_new('dtas_data',filename=fn_radial[n])
			another=read_object->get_property(data=data)
			x=*data.motor_4;no_4
			y=*data.intensity*scale_for_radial[n]
			e=*data.error*scale_for_radial[n]
		endif else if instrument eq 'bt7' then begin
			read_object=obj_new('dtas_data',filename=fn_radial[n])
			another=read_object->get_property(data=data)
			x=*data.a4
			y=*data.signal
			e=*data.error
			monitor=*data.monitor
		    if n eq 0 then begin
				monitor_to_scale_by=monitor[0]
		    endif
			y=y*monitor_to_scale_by/monitor
			e=e*monitor_to_scale_by/monitor
			temps_radial_temporary=*data.temperaturecontrolreading
			temps_radial[n]=temps_radial_temporary[0]
		endif
		data = {x:x,y:y,e:e} ;structure of x,y, and error values for the data
		ptr = ptr_new(data) ;make a pointer to the data
		if n eq 0 then begin
			radial_data_array=[ptr] ;radial_data_array is an array of pointers to the structure of the data.. this gives the first element of the array
		endif else begin
			radial_data_array=[radial_data_array,ptr] ;this gives the subsequent (not first) elements of the array of pointers to the structure of data
		endelse
	endfor

;-----
	;sort by temperature.
	sorted_temps=sort(temps_radial)
	temps_radial=temps_radial[sorted_temps]
	radial_data_array=radial_data_array[sorted_temps]
;--------

	;set-up the contour plot
	;need the minimum a4 value for each set and the maximum a4 value for each set
	a4_minimums=fltarr(n_elements(temps_radial))
	a4_maximums=fltarr(n_elements(temps_radial))
	for n=0,n_elements(temps_radial)-1 do begin
		a4_minimums[n]=min((*radial_data_array[n]).x)
		a4_maximums[n]=max((*radial_data_array[n]).x)
	endfor

	xint_radial=makepts(xlo=max(a4_minimums),xhi=min(a4_maximums),npts=100)
	z_radial=fltarr(n_elements(xint_radial),n_elements(temps_radial))
	for n=0,n_elements(temps_radial)-1 do begin
		z_radial[*,n]=interpol((*radial_data_array[n]).y,(*radial_data_array[n]).x,xint_radial)
	endfor

	;fit with Gaussians.
	;set up the arrays that will store the fitting results
	fitted_integrated_intensity_radial=fltarr(n_elements(temps_radial))
	fitted_integrated_intensity_radial_error=fltarr(n_elements(temps_radial))
	fitted_width_radial=fltarr(n_elements(temps_radial))
	fitted_width_radial_error=fltarr(n_elements(temps_radial))
	fitted_position_radial=fltarr(n_elements(temps_radial))
	fitted_position_radial_error=fltarr(n_elements(temps_radial))
	fitted_constant_radial=fltarr(n_elements(temps_radial))
	numerically_integrated_intensity_radial=fltarr(n_elements(temps_radial))	;calculate numerically integrated intensity

	;set up starting parameters for the fits
	p=[0.,0.,0.,0.]
	p[0]=min((*radial_data_array[0]).y)
	p[1]=(max((*radial_data_array[0]).y)-min((*radial_data_array[0]).y))*.3*sqrt(2.*!pi)
	places_where_maximum_is=where((*radial_data_array[0]).y eq max((*radial_data_array[0]).y))
	p[2]=0.3
	p[3]=(*radial_data_array[0]).x[places_where_maximum_is[0]]
	for n=0,n_elements(temps_radial)-1 do begin
		x=(*radial_data_array[n]).x
		y=(*radial_data_array[n]).y
		e=(*radial_data_array[n]).e
		if (where(e eq 0) ne [-1.]) then begin
			e[where(e eq 0)]=1.
		endif
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		fitted_integrated_intensity_radial[n]=p[1]
		fitted_integrated_intensity_radial_error[n]=csigma[1]
		fitted_width_radial[n]=p[2]*sqrt(8.*alog(2.))
		fitted_width_radial_error[n]=csigma[2]*sqrt(8.*alog(2.))
		fitted_position_radial[n]=p[3]
		fitted_position_radial_error[n]=csigma[3]
		fitted_constant_radial[n]=p[0]
		numerically_integrated_intensity_radial[n]=int_tabulated(x,y)
	endfor

;-------
	;fit 2 Gaussians.
	;set up the arrays that will store the fitting results
	two_gauss_radial_integrated_intensity_peak1=fltarr(n_elements(temps_radial))
	two_gauss_radial_integrated_intensity_peak2=fltarr(n_elements(temps_radial))
	two_gauss_radial_width_peak1=fltarr(n_elements(temps_radial))
	two_gauss_radial_width_peak2=fltarr(n_elements(temps_radial))
	two_gauss_radial_position_peak1=fltarr(n_elements(temps_radial))
	two_gauss_radial_position_peak2=fltarr(n_elements(temps_radial))
	two_gauss_radial_fitted_constant=fltarr(n_elements(temps_radial))
	two_gauss_radial_integrated_intensity_peak1_error=fltarr(n_elements(temps_radial))
	two_gauss_radial_integrated_intensity_peak2_error=fltarr(n_elements(temps_radial))
	two_gauss_radial_width_peak1_error=fltarr(n_elements(temps_radial))
	two_gauss_radial_width_peak2_error=fltarr(n_elements(temps_radial))
	two_gauss_radial_position_peak1_error=fltarr(n_elements(temps_radial))
	two_gauss_radial_position_peak2_error=fltarr(n_elements(temps_radial))


	;set up starting parameters for the 2-Gaussian fits

	;perform the two-Gaussian fit
	for n=0,n_elements(temps_radial)-1 do begin
		x=(*radial_data_array[n]).x
		y=(*radial_data_array[n]).y
		e=(*radial_data_array[n]).e
		if (where(e eq 0) ne [-1.]) then begin
			e[where(e eq 0)]=1.
		endif
		p=[0.,0.,0.,0.,0.,0.,0.]
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.1*sqrt(2.*!pi)
		p[2]=0.1
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
;		print,n,'starting positions',possible_positions
;		first we perform the fit using the positions determined from the get_peak_pos routine.
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		bestchisq=chisq
		if p[6] gt p[3] then begin
			two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
			two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
			two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
			two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
			two_gauss_radial_position_peak1[n]=p[3]
			two_gauss_radial_position_peak1_error[n]=csigma[3]
			two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
			two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
			two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
			two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
			two_gauss_radial_position_peak2[n]=p[6]
			two_gauss_radial_position_peak2_error[n]=csigma[6]
			two_gauss_radial_fitted_constant[n]=p[0]
		endif else begin
			two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
			two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
			two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
			two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
			two_gauss_radial_position_peak2[n]=p[3]
			two_gauss_radial_position_peak2_error[n]=csigma[3]
			two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
			two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
			two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
			two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
			two_gauss_radial_position_peak1[n]=p[6]
			two_gauss_radial_position_peak1_error[n]=csigma[6]
			two_gauss_radial_fitted_constant[n]=p[0]
		endelse
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.5*sqrt(2.*!pi)
		p[2]=0.5
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*1.*sqrt(2.*!pi)
		p[2]=1.0
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform the fit starting both gaussians near the position of maximum peak intensity.
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.2*sqrt(2.*!pi)
		p[2]=0.2
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.2
		p[6]=p[3]-.2
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.2*sqrt(2.*!pi)
		p[2]=0.2
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.2
		p[6]=p[3]+.2
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.3
		p[6]=p[3]-.5
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/2.
		p[5]=0.3
		p[6]=p[3]+.5
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_radial_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[3]
				two_gauss_radial_position_peak1_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_radial_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[6]
				two_gauss_radial_position_peak2_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_radial_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_radial_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_radial_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak2[n]=p[3]
				two_gauss_radial_position_peak2_error[n]=csigma[3]
				two_gauss_radial_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_radial_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_radial_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_radial_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_radial_position_peak1[n]=p[6]
				two_gauss_radial_position_peak1_error[n]=csigma[6]
				two_gauss_radial_fitted_constant[n]=p[0]
			endelse
		endif
	endfor
;----------
endif

if rocking_scan_yn eq 'y' then begin
	;do the rocking scan analysis

	;determine what instrument we are deailing with..
	filebasenames=file_basename(fn_rocking)
	if strmid(filebasenames[0],2,/reverse) eq 'bt9' then begin
		instrument='bt9'
		temps_rocking=float(temps_rocking) ;convert the temperatures from string to float
		scale_for_rocking=float(scaling_rocking) ;get the scale factors.
		endif else if strmid(filebasenames[0],2,/reverse) eq 'ng5' then begin
		instrument='ng5'
		temps_rocking=float(temps_rocking) ;convert the temperatures from string to float
		endif else if strmid(filebasenames[0],2,/reverse) eq 'bt7' then begin
		instrument='bt7'
		temps_rocking=fltarr(n_elements(fn_rocking))
	endif else begin
		;instrument unknown! ADD SOMETHING HERE TO GO BACK TO MAIN MENU!!!
		return
	endelse


	;make a loop to input data
	for n=0,n_elements(fn_rocking)-1 do begin
		;first read the file!
		if (instrument eq 'bt9') or (instrument eq 'ng5') then begin
			read_object=obj_new('dtas_data',filename=fn_rocking[n])
			another=read_object->get_property(data=data)
			x=*data.motor_3;no_3
			y=*data.intensity*scale_for_rocking[n]
			e=*data.error*scale_for_rocking[n]
		endif else if instrument eq 'bt7' then begin
			read_object=obj_new('dtas_data',filename=fn_rocking[n])
			another=read_object->get_property(data=data)
			x=*data.a3
			y=*data.signal
			e=*data.error
			monitor=*data.monitor
		    if n eq 0 then begin
				monitor_to_scale_by=monitor[0]
		    endif
			y=y*monitor_to_scale_by/monitor
			e=e*monitor_to_scale_by/monitor
			temps_rocking_temporary=*data.temperaturecontrolreading
			temps_rocking[n]=temps_rocking_temporary[0]
		endif
		data = {x:x,y:y,e:e} ;structure of data
		ptr = ptr_new(data) ;pointer to structure of data
		if n eq 0 then begin
			rocking_data_array=[ptr] ;this will be an array of pointers to the data structures.  this sets the initial element
		endif else begin
			rocking_data_array=[rocking_data_array,ptr] ;this sets the subsequent elements
		endelse
	endfor

;-----
	;sort by temperature.
	sorted_temps=sort(temps_rocking)
	temps_rocking=temps_rocking[sorted_temps]
	rocking_data_array=rocking_data_array[sorted_temps]
;--------

	;set-up the contour plot
	;need the minimum a3 value for each set and the maximum a3 value for each set
	a3_minimums=fltarr(n_elements(temps_rocking))
	a3_maximums=fltarr(n_elements(temps_rocking))
	for n=0,n_elements(temps_rocking)-1 do begin
		a3_minimums[n]=min((*rocking_data_array[n]).x)
		a3_maximums[n]=max((*rocking_data_array[n]).x)
	endfor

	xint_rocking=makepts(xlo=max(a3_minimums),xhi=min(a3_maximums),npts=100)
	z_rocking=fltarr(n_elements(xint_rocking),n_elements(temps_rocking))
	for n=0,n_elements(temps_rocking)-1 do begin
		z_rocking[*,n]=interpol((*rocking_data_array[n]).y,(*rocking_data_array[n]).x,xint_rocking)
	endfor

	;fit with Gaussians.
	;set up the arrays that will store the fitting results
	fitted_integrated_intensity_rocking=fltarr(n_elements(temps_rocking))
	fitted_integrated_intensity_rocking_error=fltarr(n_elements(temps_rocking))
	fitted_width_rocking=fltarr(n_elements(temps_rocking))
	fitted_width_rocking_error=fltarr(n_elements(temps_rocking))
	fitted_position_rocking=fltarr(n_elements(temps_rocking))
	fitted_position_rocking_error=fltarr(n_elements(temps_rocking))
	fitted_constant_rocking=fltarr(n_elements(temps_rocking))
	numerically_integrated_intensity_rocking=fltarr(n_elements(temps_rocking))
	;set up starting parameters for the fits
	p=[0.,0.,0.,0.]
	p[0]=min((*rocking_data_array[0]).y)
	p[1]=(max((*rocking_data_array[0]).y)-min((*rocking_data_array[0]).y))*.3*sqrt(2.*!pi)
	places_where_maximum_is=where((*rocking_data_array[0]).y eq max((*rocking_data_array[0]).y))
	p[2]=0.3
	p[3]=(*rocking_data_array[0]).x[places_where_maximum_is[0]]
	for n=0,n_elements(temps_rocking)-1 do begin
		x=(*rocking_data_array[n]).x
		y=(*rocking_data_array[n]).y
		e=(*rocking_data_array[n]).e
		if (where(e eq 0) ne [-1.]) then begin
			e[where(e eq 0)]=1.
		endif
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		fitted_integrated_intensity_rocking[n]=p[1]
		fitted_integrated_intensity_rocking_error[n]=csigma[1]
		fitted_width_rocking[n]=p[2]*sqrt(8.*alog(2.))
		fitted_width_rocking_error[n]=csigma[2]*sqrt(8.*alog(2.))
		fitted_position_rocking[n]=p[3]
		fitted_position_rocking_error[n]=csigma[3]
		fitted_constant_rocking[n]=p[0]
		numerically_integrated_intensity_rocking[n]=int_tabulated(x,y)	;calculate numerically integrated intensity
	endfor

;-------
	;fit 2 Gaussians.
	;set up the arrays that will store the fitting results
	two_gauss_rocking_integrated_intensity_peak1=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_integrated_intensity_peak2=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_width_peak1=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_width_peak2=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_position_peak1=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_position_peak2=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_fitted_constant=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_integrated_intensity_peak1_error=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_integrated_intensity_peak2_error=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_width_peak1_error=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_width_peak2_error=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_position_peak1_error=fltarr(n_elements(temps_rocking))
	two_gauss_rocking_position_peak2_error=fltarr(n_elements(temps_rocking))

	;perform the two-Gaussian fit
	for n=0,n_elements(temps_rocking)-1 do begin
		x=(*rocking_data_array[n]).x
		y=(*rocking_data_array[n]).y
		e=(*rocking_data_array[n]).e
		if (where(e eq 0) ne [-1.]) then begin
			e[where(e eq 0)]=1.
		endif
		p=[0.,0.,0.,0.,0.,0.,0.]
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.1*sqrt(2.*!pi)
		p[2]=0.1
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
;		print,n,'starting positions',possible_positions
;		first we perform the fit using the positions determined from the get_peak_pos routine.
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		bestchisq=chisq
		if p[6] gt p[3] then begin
			two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
			two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
			two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
			two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
			two_gauss_rocking_position_peak1[n]=p[3]
			two_gauss_rocking_position_peak1_error[n]=csigma[3]
			two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
			two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
			two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
			two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
			two_gauss_rocking_position_peak2[n]=p[6]
			two_gauss_rocking_position_peak2_error[n]=csigma[6]
			two_gauss_rocking_fitted_constant[n]=p[0]
		endif else begin
			two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
			two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
			two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
			two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
			two_gauss_rocking_position_peak2[n]=p[3]
			two_gauss_rocking_position_peak2_error[n]=csigma[3]
			two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
			two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
			two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
			two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
			two_gauss_rocking_position_peak1[n]=p[6]
			two_gauss_rocking_position_peak1_error[n]=csigma[6]
			two_gauss_rocking_fitted_constant[n]=p[0]
		endelse
		;next perform with different starting widths
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
					two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform with different starting widths
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.5*sqrt(2.*!pi)
		p[2]=0.5
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
					two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform with different starting widths
		possible_positions=get_peak_pos(x,y,2)
		p[0]=min(y)
		p[1]=(max(y)-min(y))*1.*sqrt(2.*!pi)
		p[2]=1.0
		p[3]=possible_positions[1]
		p[4]=p[1]/2.
		p[5]=p[2]
		p[6]=possible_positions[0]
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
					two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform the fit starting both gaussians near the position of maximum peak intensity.
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.2*sqrt(2.*!pi)
		p[2]=0.2
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.2
		p[6]=p[3]-.2
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform the fit starting both gaussians at the position of maximum peak intensity.
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.2*sqrt(2.*!pi)
		p[2]=0.2
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.2
		p[6]=p[3]+.2
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform the fit starting both gaussians at the position of maximum peak intensity.
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/5.
		p[5]=0.3
		p[6]=p[3]-.5
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
		;next perform the fit starting both gaussians at the position of maximum peak intensity.
		places_where_maximum_is=where(y eq max(y))
		p[0]=min(y)
		p[1]=(max(y)-min(y))*.3*sqrt(2.*!pi)
		p[2]=0.3
		p[3]=x[places_where_maximum_is[0]]
		p[4]=p[1]/2.
		p[5]=0.3
		p[6]=p[3]+.5
		yfit=mpcurvefit(x,y,1d/e^2.,p,sigma,/noderivative,$
			function_name = 'two_gauss_phase_transitions',chisq=chisq,dof=dof,/quiet)
		CSIGMA  = SIGMA * SQRT(CHISQ / DOF)     ; scaled uncertainties
		if chisq lt bestchisq then begin
			bestchisq=chisq
			if p[6] gt p[3] then begin
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[1]
				two_gauss_rocking_width_peak1[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[3]
				two_gauss_rocking_position_peak1_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[4]
				two_gauss_rocking_width_peak2[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[6]
				two_gauss_rocking_position_peak2_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endif else begin
				two_gauss_rocking_integrated_intensity_peak2[n]=abs(p[1])
				two_gauss_rocking_integrated_intensity_peak2_error[n]=csigma[1]
				two_gauss_rocking_width_peak2[n]=abs(p[2])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak2_error[n]=csigma[2]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak2[n]=p[3]
				two_gauss_rocking_position_peak2_error[n]=csigma[3]
				two_gauss_rocking_integrated_intensity_peak1[n]=abs(p[4])
				two_gauss_rocking_integrated_intensity_peak1_error[n]=csigma[4]
				two_gauss_rocking_width_peak1[n]=abs(p[5])*sqrt(8.*alog(2.))
				two_gauss_rocking_width_peak1_error[n]=csigma[5]*sqrt(8.*alog(2.))
				two_gauss_rocking_position_peak1[n]=p[6]
				two_gauss_rocking_position_peak1_error[n]=csigma[6]
				two_gauss_rocking_fitted_constant[n]=p[0]
			endelse
		endif
	endfor
endif



;----------------------------------------
;----------------------------------------
;plot stuff
;this might look a little weird.  these are going to form the main plotting results widgets.  it is divided into three possibilities depending on
;whether only radial scans were input, whether only rocking scans were input, or whether both were input.
if ((radial_scan_yn eq 'y') and (rocking_scan_yn eq 'n')) then begin
	;plot if there are radial scans but not rocking scans
	tlb = widget_base(title='PLOT RESULTS',/row,mbar=bar)
	tlb1 = widget_base(tlb,/col)
	tlb2 = widget_base(tlb,/col)
	file=widget_button(bar,value='File',/menu)
	PRINTBUTTON=widget_button(file,value='Print to JPEG',uname='PRINT')
	MAKEFILEBUTTON=widget_button(file,value='Output Temperature-Dependent Results to File',uname='OUTPUTFILE')
	HELPFILEBUTTON=widget_button(file,value='Help',uname='HELP')
	values=strarr(5+n_elements(temps_radial))
	values[0]='Radial Contour'
	values[1]='Integ. Intensity-Radial Scans (Numerical)'
	values[2]='Integ. Intensity-Radial Scans (Single Gaussian)'
	values[3]='Width-Radial Scans (Single Gaussian)'
	values[4]='Peak Position-Radial Scans (Single Gaussian)'

	for n=5,n_elements(values)-1 do begin
		values[n]=string(temps_radial[n-5])+' K radial fitting'
	endfor

;	drop = WIDGET_DROPLIST(tlb1, VALUE=values, uname='DROPLIST')
	drop_id=widget_label(tlb1,value='Data to plot',/align_left)
	drop = WIDGET_COMBOBOX(tlb1, VALUE=values, uname='DROPLIST')
	blank_space=widget_label(tlb1,value=' ')
	gaussian_dropbox_radial_id=widget_label(tlb1,value='# Gaussians for Radial Scans',/align_left)
	values_gaussian_dropbox_radial=['1','2']
	gaussian_dropbox_radial=WIDGET_COMBOBOX(tlb1, VALUE=values_gaussian_dropbox_radial, uname='GAUSSIAN_DROPBOX_RADIAL')


	window,/free,/pixmap,xsize=600,ysize=500
	winpix = !d.window

	drawing_widget = widget_draw(tlb2,xsize=600,ysize=500,/BUTTON_EVENTS,/MOTION_EVENTS,$
								EVENT_PRO='mouse_event',KILL_NOTIFY='mouse_cleanup');cwo_drawplot(llb2,xsize=400,ysize=400,obj=obj)


	centertlb,tlb
	widget_control,tlb,/realize
	analysis_window = !d.window


	widget_control,tlb,set_uvalue=pstate
	wset,analysis_window
	winvis = analysis_window


	wset,winpix
	plot_contour_radial_scans
	PX = PTR_NEW(!X)
	PY = PTR_NEW(!Y)
	wset,winvis
	device,copy=[0,0,600,500,0,0,winpix]

	STATE ={WINPIX:WINPIX,WINVIS:WINVIS,XPRESS:0,YPRESS:0,XRELEASE:0,YRELEASE:0,PRESSED:0,DROP:DROP,PX:PX,PY:PY}
	PSTATE = PTR_NEW(STATE)
	WIDGET_CONTROL,DRAWING_WIDGET,SET_UVALUE=PSTATE
	xmanager,'phase_transition_main_analysis',tlb,event_handler='plot_results_handler_radial_only',/no_block
endif else if ((radial_scan_yn eq 'n') and (rocking_scan_yn eq 'y')) then begin
	;plot if there are rocking scans but not radial scans

	tlb = widget_base(title='PLOT RESULTS',/row,mbar=bar)
	tlb1 = widget_base(tlb,/col)
	tlb2 = widget_base(tlb,/col)
	file=widget_button(bar,value='File',/menu)
	PRINTBUTTON=widget_button(file,value='Print to JPEG',uname='PRINT')
	MAKEFILEBUTTON=widget_button(file,value='Output Temperature-Dependent Results to File',uname='OUTPUTFILE')
	HELPFILEBUTTON=widget_button(file,value='Help',uname='HELP')
	values=strarr(5+n_elements(temps_rocking))
	values[0]='Rocking Contour'
	values[1]='Integ. Intensity-Rocking Scans (Numerical)'
	values[2]='Integ. Intensity-Rocking Scans (Single Gaussian)'
	values[3]='Width-Rocking Scans (Single Gaussian)'
	values[4]='Peak Position-Rocking Scans (Single Gaussian)'
	for n=5,n_elements(values)-1 do begin
		values[n]=string(temps_rocking[n-5])+' K rocking fitting'
	endfor

;	drop = WIDGET_DROPLIST(tlb1, VALUE=values, uname='DROPLIST', scr_ysize=50)
	drop_id=widget_label(tlb1,value='Data to plot',/align_left)
	drop = WIDGET_COMBOBOX(tlb1, VALUE=values, uname='DROPLIST')
	blank_space=widget_label(tlb1,value=' ')
	gaussian_dropbox_rocking_id=widget_label(tlb1,value='# Gaussians for Rocking Scans',/align_left)
	values_gaussian_dropbox_rocking=['1','2']
	gaussian_dropbox_rocking=WIDGET_COMBOBOX(tlb1, VALUE=values_gaussian_dropbox_rocking, uname='GAUSSIAN_DROPBOX_ROCKING')
	window,/free,/pixmap,xsize=600,ysize=500
	winpix = !d.window

	drawing_widget = widget_draw(tlb2,xsize=600,ysize=500,/BUTTON_EVENTS,/MOTION_EVENTS,$
								EVENT_PRO='mouse_event',KILL_NOTIFY='mouse_cleanup');cwo_drawplot(llb2,xsize=400,ysize=400,obj=obj)

	centertlb,tlb
	widget_control,tlb,/realize
	analysis_window = !d.window


	widget_control,tlb,set_uvalue=pstate
	wset,analysis_window

	winvis = analysis_window

	STATE ={WINPIX:WINPIX,WINVIS:WINVIS,XPRESS:0,YPRESS:0,XRELEASE:0,YRELEASE:0,PRESSED:0,DROP:DROP,PX:PTR_NEW(!X),PY:PTR_NEW(!Y)}
	PSTATE = PTR_NEW(STATE)


	wset,winpix
	plot_contour_rocking_scans
	*(*PSTATE).PX = !X
	*(*PSTATE).PY = !Y
	wset,winvis
	device,copy=[0,0,600,500,0,0,winpix]
	WIDGET_CONTROL,DRAWING_WIDGET,SET_UVALUE=PSTATE

	xmanager,'phase_transition_main_analysis',tlb,event_handler='plot_results_handler_rocking_only',/no_block

endif else if ((radial_scan_yn eq 'y') and (rocking_scan_yn eq 'y')) then begin
	;plot if there are rocking scans and also radial scans

	tlb = widget_base(title='PLOT RESULTS',/row,mbar=bar)
	tlb1 = widget_base(tlb,/col)
	tlb2 = widget_base(tlb,/col)
	file=widget_button(bar,value='File',/menu)
	PRINTBUTTON=widget_button(file,value='Print to JPEG',uname='PRINT')
	MAKEFILEBUTTON=widget_button(file,value='Output Temperature-Dependent Results to File',uname='OUTPUTFILE')
	HELPFILEBUTTON=widget_button(file,value='Help',uname='HELP')
	values=strarr(10+n_elements(temps_rocking)+n_elements(temps_radial))
	values[0]='Radial Contour'
	values[1]='Integ. Intensity-Radial Scans (Numerical)'
	values[2]='Integ. Intensity-Radial Scans (Single Gaussian)'
	values[3]='Width-Radial Scans (Single Gaussian)'
	values[4]='Peak Position-Radial Scans (Single Gaussian)'
	values[5]='Rocking Contour'
	values[6]='Integ. Intensity-Rocking Scans (Numerical)'
	values[7]='Integ. Intensity-Rocking Scans (Single Gaussian)'
	values[8]='Width-Rocking Scans (Single Gaussian)'
	values[9]='Peak Position-Rocking Scans (Single Gaussian)'

 	for n=0,n_elements(temps_radial)-1 do begin
		values[10+n]=string(temps_radial[n])+' K radial fitting (Single Gaussian)'
	endfor

 	for n=0,n_elements(temps_rocking)-1 do begin
		values[10+n_elements(temps_radial)+n]=string(temps_rocking[n])+' K rocking fitting (Single Gaussian)'
	endfor

	drop_id=widget_label(tlb1,value='Data to plot',/align_left)
	drop = WIDGET_COMBOBOX(tlb1, VALUE=values, uname='DROPLIST')
	blank_space=widget_label(tlb1,value=' ')
	gaussian_dropbox_radial_id=widget_label(tlb1,value='# Gaussians for Radial Scans',/align_left)
	values_gaussian_dropbox_radial=['1','2']
	gaussian_dropbox_radial=WIDGET_COMBOBOX(tlb1, VALUE=values_gaussian_dropbox_radial, uname='GAUSSIAN_DROPBOX_RADIAL')
	blank_space_2=widget_label(tlb1,value=' ')
	gaussian_dropbox_rocking_id=widget_label(tlb1,value='# Gaussians for Rocking Scans',/align_left)
	values_gaussian_dropbox_rocking=['1','2']
	gaussian_dropbox_rocking=WIDGET_COMBOBOX(tlb1, VALUE=values_gaussian_dropbox_rocking, uname='GAUSSIAN_DROPBOX_ROCKING')

	window,/free,/pixmap,xsize=600,ysize=500
	winpix = !d.window

	drawing_widget = widget_draw(tlb2,xsize=600,ysize=500,/BUTTON_EVENTS,/MOTION_EVENTS,$
								EVENT_PRO='mouse_event',KILL_NOTIFY='mouse_cleanup');cwo_drawplot(llb2,xsize=400,ysize=400,obj=obj)

	centertlb,tlb
	widget_control,tlb,/realize
	analysis_window = !d.window

	widget_control,tlb,set_uvalue=pstate
	wset,analysis_window

	winvis = analysis_window
	wset,winpix
	plot_contour_radial_scans
	PX = PTR_NEW(!X)
	PY = PTR_NEW(!Y)
	wset,winvis
	device,copy=[0,0,600,500,0,0,winpix]
	STATE ={WINPIX:WINPIX,WINVIS:WINVIS,XPRESS:0,YPRESS:0,XRELEASE:0,YRELEASE:0,PRESSED:0,DROP:DROP,PX:PX,PY:PY}
	PSTATE = PTR_NEW(STATE)
	WIDGET_CONTROL,DRAWING_WIDGET,SET_UVALUE=PSTATE

	xmanager,'phase_transition_main_analysis',tlb,event_handler='plot_results_handler_radial_and_rocking',/no_block

endif
end

pro return_to_radial_scan_organize_handler,event
;called after RETURN is pressed if there were not enough temperatures input
widget_control,event.top,/destroy
organize_the_radial_scans
end

pro organize_the_radial_scans_event_handler,event
;handles a call for the CONTINUE button from organize_the_radial_scans.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

uname=widget_info(event.id,/uname)
if uname eq 'CONTINUE' then begin
	id=widget_info(event.top,find_by_uname='TABLE1')
	widget_control,id,get_value=TABLE1
	temps_radial=TABLE1[0,*]
	scaling_radial=TABLE1[2,*]
    widget_control,event.top,/destroy
	places=where(temps_radial eq ' ')
	if places[0] ne -1 then begin
		;at least one file does not have a temperature!
		tlb=widget_base(/col,title='At Least One Temperature Missing')
		return_label_id=widget_label(tlb,value='There is at least one missing temperature.')
		return_label_id=widget_label(tlb,value='Be sure to push return after last temperature is entered.')
		return=widget_button(tlb,value='RETURN',xsize=200,uname='RETURN')
		centertlb,tlb
		widget_control,tlb,/realize
		xmanager,'return_to_radial_scan_organize',tlb,event_handler='return_to_radial_scan_organize_handler',/no_block
	endif else phase_transition_main_analysis
endif
end

pro organize_the_radial_scans
;either bt9 or ng5 files were entered and now temperatures and optional scaling factors need to be input.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

			tlb=widget_base(/col,title='Label Radial Scans',xsize=800,ysize=800)
			;create a label widget
			label=widget_label(tlb, value='Input Temperatures For the Radial Scan Files')
			;create a table widget ofr a 2d Array
			data=strarr(3,n_elements(fn_radial))
			filebasenames=file_basename(fn_radial)
			column_labels = ['T (K)', 'filename **DONT EDIT**', 'Scaling Factor']
;			row_labels=makepts(xlo=1,xhi=temps,npts=temps)
			data[0,*]=' ' ;temperatures
			data[1,*]=filebasenames ;filenames
			data[2,*]='1' ;optional scaling factor, default=1
			table1=widget_table(tlb,value=data,/editable,column_labels=column_labels,column_widths=200,y_scroll_size=10,uname='TABLE1')
			label=widget_label(tlb, value='**BE SURE TO PUSH ENTER AFTER TEMPERATURE INPUT**')
			CONTINUE_BUTTON=widget_button(tlb,value='CONTINUE',uname='CONTINUE')
			centertlb,tlb
			widget_control,tlb,/realize
			;start the xmanager
			xmanager,'input_files_widgetbase',tlb,event_handler='organize_the_radial_scans_event_handler',/no_block
end


pro return_to_rocking_scan_organize_handler,event
;called after RETURN is pressed if there were not enough temperatures input
widget_control,event.top,/destroy
organize_the_rocking_scans
end

pro organize_the_rocking_scans_event_handler,event
;handles a call for the CONTINUE button from organize_the_rocking_scans.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

uname=widget_info(event.id,/uname)
if uname eq 'CONTINUE' then begin
	id=widget_info(event.top,find_by_uname='TABLE1')
	widget_control,id,get_value=TABLE1
	temps_rocking=TABLE1[0,*] ;here an array of strings
	scaling_rocking=TABLE1[2,*] ;here an array of strings
    widget_control,event.top,/destroy
	places=where(temps_rocking eq ' ') ;check for blank temperature inputs
	if places[0] ne -1 then begin
		;at least one file does not have a temperature!
		tlb=widget_base(/col,title='At Least One Temperature Missing')
		return_label_id=widget_label(tlb,value='There is at least one missing temperature.')
		return_label_id=widget_label(tlb,value='Be sure to push return after last temperature is entered.')
		return=widget_button(tlb,value='RETURN',xsize=200,uname='RETURN')
		centertlb,tlb
		widget_control,tlb,/realize
		xmanager,'return_to_rocking_scan_organize',tlb,event_handler='return_to_rocking_scan_organize_handler',/no_block
	endif else radial_scan_yn_event
endif
end

pro organize_the_rocking_scans
;either bt9 or ng5 files were entered and now temperatures and optional scaling factors need to be input.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

			tlb=widget_base(/col,title='Label Rocking Scans',xsize=800,ysize=800)
			;create a label widget
			label=widget_label(tlb, value='Input Temperatures For the Rocking Scan Files')
			;create a table widget ofr a 2d Array
			data=strarr(3,n_elements(fn_rocking))
			filebasenames=file_basename(fn_rocking)
			column_labels = ['T (K)', 'filename **DONT EDIT**', 'Scaling Factor']
;			row_labels=makepts(xlo=1,xhi=temps,npts=temps)
			data[0,*]=' ' ;temperatures
			data[1,*]=filebasenames ;filenames
			data[2,*]='1'	;scaling factors (default=1)
			table1=widget_table(tlb,value=data,/editable,column_labels=column_labels,column_widths=200,y_scroll_size=10,uname='TABLE1')
			label=widget_label(tlb, value='**BE SURE TO PUSH ENTER AFTER TEMPERATURE INPUT**')
			CONTINUE_BUTTON=widget_button(tlb,value='CONTINUE',uname='CONTINUE')
			centertlb,tlb
			widget_control,tlb,/realize
			;start the xmanager
			xmanager,'input_files_widgetbase',tlb,event_handler='organize_the_rocking_scans_event_handler',/no_block
end

pro rocking_not_enough_files_handler,event
;there weren't enough rocking files entered, so return to rocking_scan_yn_event
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

uname=widget_info(event.id,/uname)
case uname of
	'START': begin
			  phase_transitions_open_screen_event_handler,event
			  end
endcase
end

pro radial_not_enough_files_handler,event
;there weren't enough radial files entered, so return to radial_scan_yn_event
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

uname=widget_info(event.id,/uname)
case uname of
	'RETURN': begin
		      widget_control,event.top,/destroy
			  radial_scan_yn_event
			  end
endcase
end

pro radial_scan_yn_event_handler,event
;handles the radial y/n widget.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

uname=widget_info(event.id,/uname)
widget_control,event.top,/destroy
case uname of
	'ANALYZE': begin
			   radial_scan_yn='y'
			   filters=['fpt04*.ng5','fpt04*.bt9','*.bt7']
		       fn_radial=dialog_pickfile(/multiple_files,title='Select All Radial Scan Files',filter=filters)
   			   if (n_elements(fn_radial) eq 1) then begin
					;there must be at least 2 files!
					tlb=widget_base(/col,title='Not Enough Files')
					return_label_id=widget_label(tlb,value='There must be at least 2 files.')
					return=widget_button(tlb,value='RETURN',xsize=200,uname='RETURN')
					centertlb,tlb
					widget_control,tlb,/realize
					xmanager,'return_radial',tlb,event_handler='radial_not_enough_files_handler',/no_block
   		   	   		endif else begin
						filebasenames=file_basename(fn_radial)
						filebasenumbers=fix(strmid(filebasenames,6,3,/reverse_offset))
						sorted_filebasenumbers=sort(filebasenumbers)
						fn_radial=fn_radial[sorted_filebasenumbers]
						if strmid(filebasenames[0],2,/reverse) eq 'bt7' then phase_transition_main_analysis else organize_the_radial_scans
   		   	   		endelse
			   end
	'SKIP':    begin
			   radial_scan_yn='n'
			   phase_transition_main_analysis
			   end
endcase
end

pro radial_scan_yn_event
;widget to input whether or not radial scans will be analyzed.
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians

	tlb=widget_base(/col,title='Radial Scans (Y/N)')
	subbase = widget_base(tlb,/row)
	analyze=widget_button(subbase,value='Analyze Radial Scans',xsize=200,uname='ANALYZE')
	skip=widget_button(subbase,value='Skip Radial Scans',xsize=200,uname='SKIP')
	centertlb,tlb
	;render the widget on the screen
	widget_control,tlb,/realize
	;start the xmanager
	xmanager,'radial_scan_yn',tlb,event_handler='radial_scan_yn_event_handler',/no_block
end

pro rocking_scan_yn_event_handler,event
;this procedure handles the user's choice of analyze or skip for rocking scans
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians


uname=widget_info(event.id,/uname)
widget_control,event.top,/destroy ;destroy the analyze/skip selection widget
case uname of
	'ANALYZE': begin
			   rocking_scan_yn='y' ;marks that rocking scans will be analyzed
			   filters=['fpx03*.ng5','fpx03*.bt9','*.bt7']
		       fn_rocking=dialog_pickfile(/multiple_files,title='Select All Rocking Scan Files',filter=filters) ;select rocking scan files
   			   if (n_elements(fn_rocking) eq 1) then begin
					;there must be at least 2 files!
					tlb=widget_base(/col,title='Not Enough Files')
					return_label_id=widget_label(tlb,value='There must be at least 2 files.')
					return=widget_button(tlb,value='RETURN',xsize=200,uname='START')
					centertlb,tlb
					widget_control,tlb,/realize
					xmanager,'return_rocking',tlb,event_handler='rocking_not_enough_files_handler',/no_block
   		   	   endif else begin
						filebasenames=file_basename(fn_rocking)
						filebasenumbers=fix(strmid(filebasenames,6,3,/reverse_offset)) ;want the file numbers
						sorted_filebasenumbers=sort(filebasenumbers) ;order the file numbers
						fn_rocking=fn_rocking[sorted_filebasenumbers]
						if strmid(filebasenames[0],2,/reverse) eq 'bt7' then radial_scan_yn_event else organize_the_rocking_scans ;if bt7 is selected then
						;temperature and monitor are already included in data file, so the radial scan prompt can be opened.  else if bt9 or ng5, then
						;temperatures and optional scaling factors needed to be added, so organize_the_rocking_scans is called.
			   endelse
			   end
	'SKIP':	   begin
			   rocking_scan_yn='n'
			   radial_scan_yn_event ;if user wants to skip rocking scans, then move on to radial scans
			   end
endcase
end


pro phase_transitions_open_screen_event_handler,event
;this is the handler after the user pushes START on the first page
;it gives the user the option of analyzing rocking scans or skipping rocking scans
;the event is then handled by rocking_scan_yn_event_handler
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians


uname=widget_info(event.id,/uname)
if uname eq 'START' then begin ;if the user wants to start, then create a prompt to determine if rocking scans will be analyzed
	widget_control,event.top,/destroy
	tlb=widget_base(/col,title='Rocking Scans (Y/N)')
	subbase = widget_base(tlb,/row)
	analyze=widget_button(subbase,value='Analyze Rocking Scans',xsize=200,uname='ANALYZE')
	skip=widget_button(subbase,value='Skip Rocking Scans',xsize=200,uname='SKIP')
	centertlb,tlb
	;render the widget on the screen
	widget_control,tlb,/realize
	;start the xmanager
	xmanager,'rocking_scan_yn',tlb,event_handler='rocking_scan_yn_event_handler',/no_block
endif else if uname eq 'HELP' then begin ;if the user wants help then run the help dialog.
	help_phase_transitions
endif
end

pro phase_transitions_mapper_for_rocking_and_radial_triple_axis,group_leader=group_leader,$
                                                                workDir=workDir,$
                                                                dataDir=dataDir,$
                                                                DAVETool=DAVETool,$
                                                                _Extra=extra
;This is the opening screen for the module.  It is a widget with START and HELP buttons.



;make common variables
common phase_transitions,rocking_scan_yn,fn_rocking,radial_scan_yn,fn_radial,temps_rocking,temps_radial,xint_radial,z_radial,radial_data_array,$
	fitted_integrated_intensity_radial,fitted_integrated_intensity_radial_error,fitted_width_radial,fitted_width_radial_error,$
	fitted_position_radial,fitted_position_radial_error,fitted_constant_radial,xint_rocking,z_rocking,rocking_data_array,$
	fitted_integrated_intensity_rocking,fitted_integrated_intensity_rocking_error,fitted_width_rocking,fitted_width_rocking_error,$
	fitted_position_rocking,fitted_position_rocking_error,fitted_constant_rocking,winpix,winvis,PSTATE,droplist_index,$
	numerically_integrated_intensity_rocking,numerically_integrated_intensity_radial,scaling_rocking,scaling_radial,values,$
	two_gauss_radial_integrated_intensity_peak1,two_gauss_radial_integrated_intensity_peak2,$
	two_gauss_radial_width_peak1,two_gauss_radial_width_peak2,$
	two_gauss_radial_position_peak1,two_gauss_radial_position_peak2,$
	two_gauss_radial_fitted_constant,two_gauss_radial_integrated_intensity_peak1_error,$
	two_gauss_radial_integrated_intensity_peak2_error,$
	two_gauss_radial_width_peak1_error,two_gauss_radial_width_peak2_error,$
	two_gauss_radial_position_peak1_error,two_gauss_radial_position_peak2_error,number_of_radial_gaussians,$
	two_gauss_rocking_integrated_intensity_peak1,two_gauss_rocking_integrated_intensity_peak2,$
	two_gauss_rocking_width_peak1,two_gauss_rocking_width_peak2,$
	two_gauss_rocking_position_peak1,two_gauss_rocking_position_peak2,$
	two_gauss_rocking_integrated_intensity_peak1_error,two_gauss_rocking_integrated_intensity_peak2_error,$
	two_gauss_rocking_width_peak1_error,two_gauss_rocking_width_peak2_error,$
	two_gauss_rocking_position_peak1_error,two_gauss_rocking_position_peak2_error,$
	two_gauss_rocking_fitted_constant,number_of_rocking_gaussians




  if n_elements(workdir) eq 0 then workdir = '.'
  if n_elements(datadir) eq 0 then datadir = '.'

  if n_elements(group_leader)  eq 1 then begin
      ;create top-level base
      tlb=widget_base(/col,title='Phase Transition Mapper',group_leader=group_leader)
  endif else begin
      ;create top-level base
      tlb=widget_base(/col,title='Phase Transition Mapper')
  endelse


explain_label=widget_text(tlb,value='In order to determine the temperature of a phase transition of a single crystal, '+$
 'rocking and radial (theta-2theta) scans are often performed through a particular Bragg reflection upon cooling or warming using the find peak'+$
 ' commands in ICP.'+$
 '  The Phase Transition Mapper is a utility that can be used to quickly visualize the results of such measurements.  It generates a 2D contour plot of'+$
 ' the temperature dependence of the scans, calculates the numerically integrated intensity of the scans, and shows the results of single-Gaussian or '+$
 ' 2-Gaussian fits to the scans (integrated intensity, peak width, and peak position).  For SPINS and BT-9, fpx03# and fpt04# can be handled.  BT-7 file formats'+$
 ' are also supported.',/wrap,/scroll,xsize=50,ysize=14)

START_BUTTON=widget_button(tlb,value='START',uname='START')
HELP_BUTTON=widget_button(tlb,value='HELP',uname='HELP')

centertlb,tlb
;render the widget on the screen
widget_control,tlb,/realize
;start the xmanager
xmanager,'phase_transitions_mapper',tlb,event_handler='phase_transitions_open_screen_event_handler',/no_block
end