; $Id$
;-------------------------------------------------------------------------------------------------------------------
; Reads MACS files, and returns data array with appropriate columns extracted from the files.
; 
; Input :
;     files   : Input MACS files
;     
; Output :
;     data    : Output data array
;     columns : Vector with names of columns extracted from the files. These columns are
;               [Monitor, SPEC[1-20], DIFF[1-20], Devices]. 
;               Devices by default are a2, a3, kidney and a5. If devices are specified explicitly, then they are
;               appended to the default list specified above 
;                
; Optional Input :
;     devices     : See columns above
;     no_norm     : Specify of the counts are not to be normalized. Normalized by default to monitor count 1e6
;     concat_devs : Devices which are compared  for concatenation of data. By default : a2, a3, kidney and a5
;     empsc       : Weights of the background files.
;     a3ran       : Range of A3 values to be extracted
;     
; Written by : Vivek Thampy (01/05/2010) 
;-------------------------------------------------------------------------------------------------------------------
pro MACS_extract_data, files, data, columns, devices=devices, weights=weights, a3shift=a3shift, no_norm=no_norm, $  
                  empsc=empsc, concat_devs=concat_devs, a3ran=a3ran, invert=invert, _extra=_extra
   
   normonct = 1.e6
   
   devs = ['A2','A3','KIDNEY','A5']
   if n_elements(concat_devs) eq 0 then concat_devs = devs else concat_devs = strupcase(concat_devs)
   if n_elements(devices)     eq 0 then devices     = devs else devices     = [devs, strupcase(devices)]
   devices = (devices[sort(devices)])[uniq(devices[sort(devices)])]
   
   nfiles = n_elements(files)
   if n_elements(empsc)   ne nfiles then empsc   = (n_elements(empsc)   eq 1) ? fltarr(nfiles)+empsc   : fltarr(nfiles)+1.
   if n_elements(a3shift) ne nfiles then a3shift = (n_elements(a3shift) eq 1) ? fltarr(nfiles)+a3shift : fltarr(nfiles) 
   
   dets      = indgen(20) + 1  &  n_dets  = n_elements(dets)
   spec      = 'SPEC' + string(dets, form='(i02)')
   diff      = 'DIFF' + string(dets, form='(i02)')
   errspec   = 'SPEC' + string(dets, form='(i02)') + '_ERR'
   errdiff   = 'DIFF' + string(dets, form='(i02)') + '_ERR'
   
   columns   = ['MONITOR', spec, diff, devices]
   n_columns = n_elements(columns)
   inds      = intarr(n_columns)
   data      = fltarr(n_columns,1)
   
   for i=0, nfiles-1 do begin
      readmacsfile, files(i), ptr
      for j = 0, n_columns-1 do inds[j] = where(tag_names(ptr) eq strupcase(columns[j])) 
    
      n_points = n_elements(*ptr.a2)
      di = fltarr(n_columns, n_points)
      for j = 0, n_columns-1 do di[j,*] = *ptr.(inds[j])
      
      for j = 1, 40 do begin
         lis = where(finite(di[j,*]) and (di[j,*] ge 0) and (di[j,*] lt 1e10), nlis)
         if nlis gt 0 then di = di[*,lis] $
         else begin
            di = 0 & break
         endelse
      endfor 
      if n_elements(di) le 1 then continue
     
      di[where(columns eq 'A3'), *] += a3shift[i]
      if n_elements(weights) eq n_dets then di[1:n_dets, *] *= rebin(weights, n_dets, n_points)
      
      di = [di[0,*], di[1:2*n_dets,*] * empsc[i], di[2*n_dets+1:n_columns-1,*]]
      data = [[data],[di]]
   end
   
   data = data[*, 1:n_elements(data[0,*])-1]
   
   if n_elements(a3ran) eq 2 then begin
      a3col = where(columns eq 'A3')
      lis   = where((data[a3col, *] ge a3ran[0]) and (data[a3col,*] le a3ran[1]), nlis)
      if nlis gt 0 then data = data[*,lis] else data[1:40, *] = 0
   endif 
   
   if keyword_set(invert) then begin
      inv_data = data 
      inv_data[where(columns eq 'A3'),*] -= 180.
      data = [[data],[inv_data]]
   end
   
   data = MACS_concat_data(data, concat_devs, columns)
   
   if keyword_set(no_norm) then normon = 1 else normon = normonct/(rebin(data[0,*]+1e-10, 2*n_dets, n_elements(data[0,*])))
   data = [data[0,*], normon*data[1:2*n_dets,*], normon*sqrt(data[1:2*n_dets,*]), data[2*n_dets+1:n_columns-1,*]]

   columns = ['MONITOR', spec, diff, errspec, errdiff, devices]
   
end;MACS_extract_data
