; $Id$
; DCS_RUNSINFO.PRO
;
; This program reads in and plots sample temperatures, and a bunch of other
; parameters, for several files.
;
; Written by J.R.D. Copley.
;************************************************************************************************
pro DCS_RunsinfoCleanup,tlb
;************************************************************************************************
;
compile_opt strictarr
;
widget_control,tlb,get_uvalue = pState

; revert to existing colors prior to exiting dcs_multiplot.
if (!d.name eq 'WIN' or !d.name eq 'X') then begin
	tvlct, (*pState).default_colors
	device, decomposed=(*pstate).old_decomp
endif
;
wdelete,(*pState).winState.winPix
ptr_free,(*pState).winState.xPtr,(*pState).winState.yPtr
ptr_free,pState
return
end


;************************************************************************************************
pro dcs_runsinfoQuit,event
;************************************************************************************************
;
compile_opt strictarr
;
;
widget_control,event.top,/destroy
return
end


;;;************************************************************************************************
;;pro dcs_runsinfoText,event
;;;************************************************************************************************
;;;
;;compile_opt strictarr
;;;
;;;
;;widget_control,event.top,get_uvalue = pState
;;ntypes=8
;;fmt="("+"'"+["Temperature (K)","Wavelength (A)","Speed ratio denominator",$
;;	"tsdmin (us)","Time (h)","Beam monitor count rates (c/s)",$
;;	"He-3 count rate (c/s)","Status of ORC (0=stopped,1=oscillating)"]+$
;;	"'"+$
;;	['(f10.2)','(f10.3)','(i3)','(f10.3)','(f10.3)','(f10.4)','(f10.4)','(i3)']+")"
;;index=*(*pState).runindexPtr
;;runnames=*(*pState).runnamesPtr
;;runcomments=*(*pState).runcommentsPtr
;;startdates=*(*pState).startdatesPtr
;;stopdates=*(*pState).stopdatesPtr
;;user=*(*pState).userPtr
;;nruns=n_elements(index)-1
;;for k=0,nruns-1 do begin
;;	print,runnames[k]
;;	print,runcomments[k]
;;	print,startdates[k]
;;	print,stopdates[k]
;;	print,user[k]
;;	for type=0,ntypes-1 do begin
;;		y = *(*pState).dcsStatePtr[type]
;;		nvals=n_elements(y)/index[nruns]
;;		for j=0,nvals-1 do begin
;;			print,y[index[k]:index[k+1]-1,j],format=fmt[type]
;;		endfor
;;	endfor
;;endfor
;;return
;;end


;************************************************************************************************
pro dcs_runsinfoPlot,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfoPlot: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;
widget_control,event.top,get_uvalue = pState
type=where(widget_info((*pState).buttons,/button_set),count)
if (count eq 0) then type=0 else type=type[0]
;
; Generate x and y values.
y = *(*pState).dcsStatePtr[type]
npts=n_elements(y[*,0])
;
if (not (*pState).timeabsc) then begin
	x = findgen(npts)+1
endif else begin
	x = *(*pState).abscissaPtr
endelse
;
; Generate x and y titles.
runnames=*(*pState).runnamesPtr
xt0=runnames[0]
;; pos0=strpos(xt0,'200',/reverse_search)
firstrun=strmid(file_basename(xt0),0,12) ; modified JRDC 1/27/2010
if (strmid(firstrun,11) eq '.') then firstrun=strmid(firstrun,0,11)+" "
xtitle = "First run is "+firstrun

;;if ((*pstate).plotmode) then begin
;;	case type of
;;		0: ytitle = "Temperature (relative)"
;;		1: ytitle = "Wavelength (relative)"
;;		2: ytitle = "Speed ratio denominator (relative)"
;;		3: ytitle = "tsdmin (relative)"
;;		4: ytitle = "Time (relative)"
;;		5: ytitle = "Beam monitor counts (relative)"
;;		6: ytitle = "He-3 counts (relative)"
;;		7: ytitle = "Beam monitor count rates (relative)"
;;		8: ytitle = "He-3 count rate (relative)"
;;		9: ytitle = "Status of ORC (0=stopped,1=oscillating) (relative)"
;;		10: ytitle = "Status of Shutter (0=closed,1=open) (relative)"
;;		11: ytitle = "Run parameters (relative)"
;;		12: ytitle = "Total resets (relative)"
;;		13: ytitle = "Total repeats (relative)"
;;		else:
;;	endcase
;;endif else begin
	case type of
		0: ytitle = "Temperature (K)"
		1: ytitle = "Magnetic Field (T)"
		2: ytitle = "Wavelength (A)"
		3: ytitle = "Speed ratio denominator"
		4: ytitle = "tsdmin (us)"
		5: ytitle = "Time (h)"
		6: ytitle = "Beam monitor counts"
		7: ytitle = "He-3 counts"
		8: ytitle = "Beam monitor count rates (c/s)"
		9: ytitle = "He-3 count rate (c/s)"
		10: ytitle = "Status of ORC (0=stopped,1=oscillating)"
		11: ytitle = "Status of Shutter (0=closed,1=open)"
		12: ytitle = "Run parameters"
		13: ytitle = "Total resets"
		14: ytitle = "Total repeats"
		else:
	endcase
;;endelse
;
;;if ((*pstate).plotmode) then begin
;;	y1=(fltarr(npts)+1)#y[0,*]
;;	index=where(y1 eq 0.0,count)
;;	if (count gt 0) then y1[index]=1.0
;;	y=y/y1
;;endif
;
; Generate x and y ranges (in data units)
if (*pState).winState.autoscale eq 1 or $
	(*pState).winState.autoscale eq 2 then begin
	if (not (*pState).timeabsc) then begin
		(*pState).winState.xrange = [0.99999,float(npts)+1]
	endif else begin
		nstopdates=n_elements(*(*pState).stopdatesPtr)
		maxx=dcs_juldate((*(*pState).stopdatesPtr)[nstopdates-1])
		(*pState).winState.xrange = [min(x)-0.00001,maxx]
	endelse
endif
;
if (*pState).winState.autoscale eq 1 or $
	(*pState).winState.autoscale eq 3 then begin
	dy = 0.05*(max(y,/nan)-min(y,/nan))
	if (dy eq 0.0) then dy = min(y,/nan)*0.2
	(*pState).winState.yrange = [(min(y,/nan)-dy) ,max(y,/nan)+dy > 1]
	if type ne 1 then (*pState).winState.yrange[0] = ((*pState).winState.yrange[0])>0
	if finite((*pState).winState.yrange[1],/nan) then (*pState).winState.yrange[1] = 1
	if finite((*pState).winState.yrange[0],/nan) then (*pState).winState.yrange[0] = -1
endif

;
xrange = (*pState).winState.xrange
yrange = (*pState).winState.yrange
;
; wset,(*pState).winstate.winPix
widget_control,event.top,set_uvalue = pState
;
!x=(*pState).bangx
!y=(*pState).bangy
!p=(*pState).bangp
;
; Setting color table for initial plot.
if (!d.name eq 'WIN' or !d.name eq 'X') then begin
	device, get_decomposed=old_decomp
	(*pState).old_decomp=old_decomp
	default_colors=bindgen(256,3)
	tvlct,default_colors,/get
	(*pState).default_colors=default_colors
	device, decomposed=0
;	colorset
	tvlct,(*pState).amcolor.red,(*pState).amcolor.green,(*pState).amcolor.blue
endif

; Initial plot with no data, no x axes.
plot,x,y[*,0],ytitle=ytitle,yrange=yrange,ystyle=1,$
	position = [0.15,0.15,0.925,0.85],/nodata,color=1,$
	xrange=xrange,xstyle=5,yminor=(*pState).nminor+1,$
	ytick_get=yval
xmin=!x.crange[0]
xmax=!x.crange[1]
ymin=!y.crange[0]
ymax=!y.crange[1]
oplot,[xmin,xmax,xmax,xmin,xmin],[ymin,ymin,ymax,ymax,ymin],thick=2.0,color=1

; If desired, draw horizontal dashed lines for major and minor ticks.
if ((*pState).showgrid[1] eq 1) then begin
	nyval=n_elements(yval)
	if (nyval gt 1) then begin
		ydif=yval[1]-yval[0]
		ydifminor=ydif/((*pState).nminor+1)
		ystart=yval[0]-ydif
		for k=0,(nyval+2)*((*pState).nminor+1) do begin
			yk=ystart+k*ydifminor
			if (yk gt ymin and yk lt ymax) then begin
				oplot,[xmin,xmax],[yk,yk],color=1,thick=1.0,linestyle=2
			endif
		endfor
	endif
endif

; If desired, draw horizontal solid lines for major ticks.
if ((*pState).showgrid[0] eq 1) then begin
	nyval=n_elements(yval)
	for k=0,nyval-1 do begin
		oplot,[xmin,xmax],[yval[k],yval[k]],color=1,thick=1.0
	endfor
endif

(*pState).bangx=!x
(*pState).bangy=!y
(*pState).bangp=!p

index=*(*pState).runindexPtr
nruns=n_elements(runnames)
;
; Define corner coordinates in data units
xl=!x.crange[0]
xr=!x.crange[1]
yb=!y.crange[0]
yt=!y.crange[1]
xw=xr-xl
yw=yt-yb
;
; Plot data for each run
if (!debug) then print,index[nruns]-1,n_elements(x)-1
xfrac=0.75
case type of
	0: begin ; Temperatures
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			xx=x[index1]
			yy=y[index1,0]
			plots,xx,yy,psym=2,color=2
			if (index1 lt index2) then begin
				xx=[x[index1],x[index2]]
				yy=[y[index1,0],y[index2,0]]
				oplot,xx,yy,color=2,thick=2,psym=0
				xx=x[index1:index2]
				yy=y[index1:index2,1]
				oplot,xx,yy,color=3,thick=1,psym=-4
				yy=y[index1:index2,2]
				oplot,xx,yy,color=4,thick=1,psym=-4
			endif else begin
				if ((*pState).timeabsc) then $
					xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
				xx=[x[index1],x[index1]+xfrac]
				yy=[y[index1,0],y[index1,0]]
				oplot,xx,yy,color=2,thick=2,psym=0
				yy=[y[index1,1],y[index1,1]]
				oplot,xx,yy,color=3,thick=1,psym=-4
				yy=[y[index1,2],y[index1,2]]
				oplot,xx,yy,color=4,thick=1,psym=-4
			endelse
		endfor
	end
	5: begin ; Durations
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			xx=x[index1]
			yy=y[index1,0]
			plots,xx,yy,psym=2,color=2
			plots,xx,y[index1,1],psym=2,color=3
			if (index1 lt index2) then begin
				xx=[x[index1],x[index2]]
				yy=[y[index1,0],y[index2,0]]
				oplot,xx,yy,color=2,thick=2,psym=0
				yy=[y[index1,1],y[index2,1]]
				oplot,xx,yy,color=3,thick=2,psym=0
			endif else begin
				if ((*pState).timeabsc) then $
					xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
				xx=[x[index1],x[index1]+xfrac]
				yy=[y[index1,0],y[index1,0]]
				oplot,xx,yy,color=2,thick=2,psym=0
				yy=[y[index1,1],y[index1,1]]
				oplot,xx,yy,color=3,thick=2,psym=0
			endelse
			if (k gt 0) then begin
				xx=x[index1]
				yy=y[index1,2]
;				if (xx gt xl and xx lt xr and yy gt yb and yy lt yt) then $
					plots,xx,yy,psym=2,color=4,symsize=2
			endif
		endfor
	end
	6: begin ; Beam monitor count rates for each cycle
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			xx=x[index1]
			yy=y[index1,0]
			if (index1 lt index2) then begin
				xx=x[index1:index2]
				for m=0,3 do begin
					yy=y[index1:index2,m]
					oplot,xx,yy,color=m+2,thick=1,psym=-4
				endfor
			endif else begin
				if ((*pState).timeabsc) then $
					xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
				xx=[x[index1],x[index1]+xfrac]
				for m=0,3 do begin
					yy=[y[index1,m],y[index1,m]]
					oplot,xx,yy,color=m+2,thick=1,psym=-4
				endfor
			endelse
		endfor
	end
	7: begin ; He-3 count rates for each cycle
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			xx=x[index1]
			yy=y[index1,0]
			if (index1 lt index2) then begin
				xx=x[index1:index2]
				yy=y[index1:index2]
				oplot,xx,yy,color=2,thick=1,psym=-4
			endif else begin
				if ((*pState).timeabsc) then $
					xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
				xx=[x[index1],x[index1]+xfrac]
				yy=[y[index1],y[index1]]
				oplot,xx,yy,color=2,thick=1,psym=-4
			endelse
		endfor
	end
	8: begin ; Beam monitor count rates
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			for m=0,3 do begin
				xx=x[index1]
				yy=y[index1,m]
				if (xx gt xl and xx lt xr and yy gt yb and yy lt yt) then begin
					plots,xx,yy,color=2+m,thick=1,psym=-4
					if (index1 lt index2) then begin
						xx=[x[index1],x[index2]]
						yy=[y[index1,m],y[index2,m]]
						oplot,xx,yy,color=2+m,thick=2,psym=0
					endif else begin
						if ((*pState).timeabsc) then $
							xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
						xx=[x[index1],x[index1]+xfrac]
						yy=[y[index1,m],y[index1,m]]
						oplot,xx,yy,color=2+m,thick=2,psym=0
					endelse
				endif
			endfor
		endfor
	end
	12: begin ; Run parameters
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			for m=0,3 do begin
				xx=x[index1]
				yy=y[index1,m]
				if (xx gt xl and xx lt xr and yy gt yb and yy lt yt) then begin
					plots,xx,yy,color=2+m,thick=1,psym=-4
					if (index1 lt index2) then begin
						xx=[x[index1],x[index2]]
						yy=[y[index1,m],y[index2,m]]
						oplot,xx,yy,color=2+m,thick=2,psym=0
					endif else begin
						if ((*pState).timeabsc) then $
							xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
						xx=[x[index1],x[index1]+xfrac]
						yy=[y[index1,m],y[index1,m]]
						oplot,xx,yy,color=2+m,thick=2,psym=0
					endelse
				endif
			endfor
		endfor
	end
	else: begin
		for k=0,nruns-1 do begin
			index1=index[k]
			index2=index[k+1]-1
			xx=x[index1]
			yy=y[index1,0]
			if (xx gt xl and xx lt xr and yy gt yb and yy lt yt) then begin
				plots,xx,yy,psym=2,color=2
				if (index1 lt index2) then begin
					xx=[x[index1],x[index2]]
					yy=[y[index1,0],y[index2,0]]
					oplot,xx,yy,color=2,thick=2,psym=0
				endif else begin
					if ((*pState).timeabsc) then $
						xfrac=(dcs_juldate((*(*pState).stopdatesPtr)[k])-x[index1])*0.75
					xx=[x[index1],x[index1]+xfrac]
					yy=[y[index1,0],y[index1,0]]
					oplot,xx,yy,color=2,thick=2,psym=0
				endelse
			endif
		endfor
	end
endcase
;
; Plot upper and lower x axes.
oplot,[xl,xr],[yb+yw*0.001,yb+yw*0.001],color=1,thick=2
oplot,[xl,xr],[yt-yw*0.001,yt-yw*0.001],color=1,thick=2
;
; Define x and y offsets and widths in normalized units.
xoff=!x.window[0]
xwid=!x.window[1]-!x.window[0]
yoff=!y.window[0]
ywid=!y.window[1]-!y.window[0]
;
; Plot the title.
xyouts,xoff+xwid*0.5,(yoff+ywid)+0.05,$
	xtitle,alignment = 0.5,charsize=1.5,/normal,color=1
;
; Plot x tick labels and (optionally) vertical lines as x ticks.
yc=yb-0.02*yw
for j = 0,nruns-1 do begin
	runstring=strmid(file_basename(runnames[j]),4,8); modified JRDC, 1/27/2010.
	if (strmid(runstring,7) eq '.') then runstring=strmid(runstring,0,7)+" "
	xc=index[j]+1
	xc=x[index[j]] ; This line has been added.
	if (xc gt xl and xc lt xr) then begin
		xyouts,xc,yc,$
			runstring,alignment = 0.0,charsize=1.0,$
			orientation = -45.0,/data,color=1
		if ((*pState).showvert eq 1) then oplot,[xc,xc],[yb,yt],color=1,thick=1.2,linestyle=1
	endif
endfor
;
; Optionally plot time labels.
if ((*pstate).showtimes) then begin
	startdates=*(*pState).startdatesPtr
	yc2=yt-0.02*yw
	for j = 0,nruns-1 do begin
		timestring=strmid(startdates[j],12,5)
		xc=index[j]+1
		xc=x[index[j]] ; This line has been added.
		if (xc gt xl and xc lt xr) then begin
			xyouts,xc,yc2,$
				timestring,alignment = 0.0,charsize=1.0,$
				orientation = -45.0,/data,color=1
		endif
	endfor
endif
;
; For temperatures plot a legend
if (type eq 0) then begin
	if ((*pstate).tsensorswap) then begin
		descrip=["Set point","Sample","Control"]
	endif else begin
		descrip=["Set point","Control","Sample"]
	endelse
	for j=0,2 do begin
		xs=xl+j*xw/3
		ys=yt+yw*0.02
		plots,[xs,xs+xw*0.08],[ys+yw*0.01,ys+yw*0.01],/data,color=j+2,thick=2
		xyouts,xs+xw*0.1,ys,descrip[j],$
			/data,alignment=0.0,charsize=1.2,charthick=1,color=1
	endfor
endif
;
; For durations plot a legend
if (type eq 5) then begin
	descrip=["Counting time","Elapsed clock time","Time between runs"]
	for j=0,2 do begin
		xs=xl+j*xw/3
		ys=yt+yw*0.02
		case j of
		0: begin
			plots,[xs,xs+xw*0.08],[ys+yw*0.01,ys+yw*0.01],/data,color=2,thick=2
			xyouts,xs+xw*0.1,ys,descrip[j],$
				/data,alignment=0.0,charsize=1.2,charthick=1,color=1
			end
		1: begin
			plots,[xs,xs+xw*0.08],[ys+yw*0.01,ys+yw*0.01],/data,color=3,thick=2
			xyouts,xs+xw*0.1,ys,descrip[j],$
				/data,alignment=0.0,charsize=1.2,charthick=1,color=1
			end
		2: begin
			plots,xs,ys+yw*0.01,psym=2,color=4,symsize=2
			xyouts,xs+xw*0.025,ys,descrip[j],$
				/data,alignment=0.0,charsize=1.2,charthick=1,color=1
			end
		endcase
	endfor
endif
;
; For beam monitor count rates plot a legend
if (type eq 6 or type eq 8) then begin
	descrip=["BM0","BM1","BM2","BM3"]
	for j=0,3 do begin
		xs=xl+j*xw/4
		ys=yt+yw*0.02
		plots,[xs,xs+xw*0.08],[ys+yw*0.01,ys+yw*0.01],/data,color=j+2,thick=2
		xyouts,xs+xw*0.1,ys,descrip[j],$
			/data,alignment=0.0,charsize=1.2,charthick=1,color=1
	endfor
endif
;
; For run parameters plot a legend
if (type eq 12) then begin
	descrip=["# of cycles","Cycle time (m)","Checkstats flag","Rectemp flag"]
	for j=0,3 do begin
		xs=xl+j*xw/4
		ys=yt+yw*0.02
		plots,[xs,xs+xw*0.08],[ys+yw*0.01,ys+yw*0.01],/data,color=j+2,thick=2
		xyouts,xs+xw*0.1,ys,descrip[j],$
			/data,alignment=0.0,charsize=1.2,charthick=1,color=1
	endfor
endif
;
if (!d.name eq 'WIN' or !d.name eq 'X') then begin
	device,decomposed=(*pState).old_decomp
	tvlct, (*pState).default_colors
endif
;
return
end


;************************************************************************************************
pro dcs_runsinfoSave,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfoSave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

widget_control,event.top,get_uvalue = pState
type=where(widget_info((*pState).buttons,/button_set),count)
if (count eq 0) then type=0 else type=type[0]
;
; Get y values.
y = *(*pState).dcsStatePtr[type]
npts=n_elements(y[*,0])
;
; Abbreviate run names.
runnames=*(*pState).runnamesPtr
;; xt0=runnames[0]
;; pos0=strpos(xt0,'200',/reverse_search)
nruns=n_elements(runnames)
for k = 0,nruns-1 do begin
	runstring=strmid(file_basename(runnames[k]),0,12); modified JRDC 1/27/2010.
	if (strmid(runstring,11) eq '.') then runstring=strmid(runstring,0,11)+" "
	runnames[k]=runstring
endfor
;
case type of
	0: print_all=1
	6: print_all=1
	7: print_all=1
	else: print_all=0
endcase
;
startdates=*(*pState).startdatesPtr
;
case type of
	0: ytitle = "Temperature"
	1: ytitle = "Magfield"
	2: ytitle = "Wavelength"
	3: ytitle = "Speedratio"
	4: ytitle = "tsdmin"
	5: ytitle = "Time"
	6: ytitle = "Monitorcounts"
	7: ytitle = "Detectorcounts"
	8: ytitle = "Monitorcountrate"
	9: ytitle = "Detectorcountrate"
	10: ytitle = "ORCstatus"
	11: ytitle = "Shutterstatus"
	12: ytitle = "Runparameters"
	13: ytitle = "Totalresets"
	14: ytitle = "Totalrepeats"
	else:
endcase
;
filename=runnames[0]+'_'+ytitle+'.txt'
filename=dialog_pickfile(file=filename,path=(*pState).workdir,filter='*.txt')
print,filename
if (filename eq '' or filename eq ' ') then return
;
openw,unit,filename,/get_lun
k=0
index=*(*pState).runindexPtr
for j=0,npts-1 do begin
	res=where(j eq index,n)
	if (n eq 1) then begin
		k=0
		output0=runnames[res];+'  '+startdates[res]
	endif
	k=k+1
	output=string(j)+string(k)+output0+strjoin(string(transpose(y[j,*])))
	if (print_all or (~print_all and n eq 1)) then begin
		print,output
		printf,unit,output
	endif
endfor
free_lun,unit
;
end


;************************************************************************************************
pro dcs_runsinfoRead,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfoRead: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;
widget_control,event.top,get_uvalue = pState

davesysvars
dcs_sysvars
;
dave_makeDavePtr,davePtr=davePtr
dcs_definedavestructure,davePtr=davePtr
;
files = dialog_pickfile(dialog_parent = event.top,title = 'Select run files',$
	path=(*pState).dataDir,/multiple_files) ;path=!dcs_data_directory
;
;files=[$
;	'C:\DCS Data\20041203_002.dcs.gz',$
;	'C:\DCS Data\20041202_012.dcs.gz',$
;	'C:\DCS Data\20041202_013.dcs.gz',$
;	'C:\DCS Data\20041202_014.dcs.gz',$
;	'C:\DCS Data\20041203_001.dcs.gz']
;
files=files[sort(files)]
if (files[0] eq '' or files[0] eq ' ') then begin
	widget_control,event.top,/destroy
	return
endif
;
; Throw out files with invalid names (other than "*.dcs" and "*.dcs.gz").
nf=n_elements(files)
ok=bytarr(nf)
for i=0,nf-1 do begin
	end4=strmid(files[i],3,4,/reverse_offset)
	end7=strmid(files[i],6,7,/reverse_offset)
	ok[i]=(end4 eq ".dcs")	or (end7 eq ".dcs.gz")
endfor
files=files[where(ok eq 1)]
;
nf=n_elements(files)
nofirstline=0
printinfo=0
omithisto=2
filetype="binary"
;
files=files[sort(files)]
;
progressBar = Obj_New($
	"SHOWPROGRESS",$
;	/cancelbutton,$;	(How does this work?)
	title=strcompress(string(nf))+" data files to be read ...",$
	xsize=300,$
	ysize=15,$
	message="Reading file 1",$
	color=184)
;
progressBar->Start
;
dcs_readdatafile,files[0],omithisto,printinfo,filetype,davePtr=davePtr
;
incr=100.0/nf
progressBar->Update,incr
;
comments=(*(*(*davePtr).dataStrPtr).specificPtr).comments
startdates=(*(*(*davePtr).dataStrPtr).specificPtr).start_date
stopdates=(*(*(*davePtr).dataStrPtr).specificPtr).stop_date
user=(*(*(*davePtr).dataStrPtr).specificPtr).user
;
; temp_sample and temp_control are initialized in dcs_definedavestructure as vectors of length nvals
; with value -1.0, then in read_binaryoctavefile they become vectors of length nvals, value -1.1
; They are then modified according to the content of the raw data file.	In the comments that follow
;	#cycles is the number of accepted (good) cycles and the descriptions refer to the content of the
;	raw data file.
;
; Case A:	If rectemp=1 but the temperature cannot be read temp_sample and temp_control (in the raw data
;					file) are vectors of length #cycles, value -1.0 (thus temp_sample and temp_control in this
;					procedure, if there were 3 good cycles, would be [-1.0,-1.0,-1.0,-1.1,-1.1,-1.1,-1.1,-1.1,-1.1 ...])
; Case B:	If rectemp=0, temp_sample,temp_control and temp_setpoint (in the raw data file) are scalars,
;					value "300.0", i.e. temp_sample and temp_control in this procedure, would be [300.0,-1.1,-1.1,-1.1,
;					-1.1,-1.1,-1.1,-1.1,-1.1,-1.1,-1.1 ...])
; Case C:	Normally temp_sample and temp_control (in the raw data file) are vectors of length #cycles
;					with positive numbers.
;	Case D:	If the temperature controller were set to <none> temp_sample and temp_control (in the raw data
;					file) would be vectors of length #cycles, value 0.0 (thus temp_sample and temp_control in this
;					procedure, if there were 3 good cycles, would be [0.0,0.0,0.0,-1.1,-1.1,-1.1,-1.1,-1.1,-1.1 ...])
;
; ntemps is the number of meaningful (positive) temperature readings (0, 1, >0, 0 in cases A, B, C, D)
; ncycs is intended to be the number of good cycles.
res_temps=where((*(*(*davePtr).dataStrPtr).specificPtr).temp_sample gt 0.0,ntemps)
res_cycs=where((*(*(*davePtr).dataStrPtr).specificPtr).he3_csum ge 0,ncycs)
;
case ntemps of
	; No temperature reading or no controller defined (cases A and D).
	0: begin
		t_sample=fltarr(ncycs)-100.0
		t_control=fltarr(ncycs)-100.0
		t_setpoint=fltarr(ncycs)-100.0
	end
	; One temperature reading (case C), or else not recording temperature (values of 300) (case B).
	1: begin
		ts=(*(*(*davePtr).dataStrPtr).specificPtr).temp_sample[0]
		tc=(*(*(*davePtr).dataStrPtr).specificPtr).temp_control[0]
		tsp=(*(*(*davePtr).dataStrPtr).specificPtr).temp_setpoint
		if (ts eq 300.0 and tc eq 300.0 and tsp eq 300.0) then begin
			ts=-10.0
			tc=-10.0
			tsp=-10.0
		endif
		t_sample=fltarr(ncycs)+ts
		t_control=fltarr(ncycs)+tc
		t_setpoint=fltarr(ncycs)+tsp
	end
	; Multiple temperature readings.
	else: begin
		t_sample=(*(*(*davePtr).dataStrPtr).specificPtr).temp_sample[0:ncycs-1]
		t_control=(*(*(*davePtr).dataStrPtr).specificPtr).temp_control[0:ncycs-1]
		t_setpoint=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).temp_setpoint
	end
endcase
n2=ncycs
field_setpoint=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).field_setpoint
field_sample=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).field_sample
wavelength=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).ch_wl
srdenom=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).ch_srdenom
tsdmin=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).tsdmin
duration=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).duration/3600.0
runtime=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[921]/432000.
timelost=(fltarr(ncycs)+1)*0.0
;
julstart=dcs_juldate((*(*(*davePtr).dataStrPtr).specificPtr).start_date)
julstop=dcs_juldate((*(*(*davePtr).dataStrPtr).specificPtr).stop_date)
cyctim=(julstop-julstart)/ncycs
abscissa=julstart+findgen(ncycs)*cyctim
;
time_for_run=(fltarr(ncycs)+1)*(julstop-julstart)*86400.0/60.0
;
;;time_counting=(*(*(*davePtr).dataStrPtr).specificPtr).detsum[921]/120.0/60.0
;;time_for_cycle=(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[1]/60.0
;;time_duration=(*(*(*davePtr).dataStrPtr).specificPtr).duration/60.0
;;requested_number_of_cycles=(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[0]
;;actual_number_of_cycles=time_duration/time_for_cycle
;;difference=requested_number_of_cycles-fix(actual_number_of_cycles)
;;remainder=actual_number_of_cycles-fix(actual_number_of_cycles)
;;comment=''
;;if (difference gt 0) then begin
;;	comment='    Run truncated'
;;	if (remainder lt 0.025) then comment='   Cycles lost'
;;endif
;;print,'time_for_run,time_counting,time_for_cycle,time_duration,requested_number_of_cycles',$
;;	' actual_number_of_cycles,difference'
;;print,time_for_run,time_counting,time_for_cycle,time_duration,requested_number_of_cycles,$
;;	actual_number_of_cycles,difference,remainder,files[0],comment,format='(4f8.1,i5,f10.4,i5,f10.4,2a)'
;
cycle_duration=(*(*(*davePtr).dataStrPtr).specificPtr).duration_csum[0:ncycs-1]
if (cycle_duration[0] eq 0.0) then cycle_duration=cycle_duration+1
high_csum_rate=(*(*(*davePtr).dataStrPtr).specificPtr).high_csum[*,0:ncycs-1]/((fltarr(18)+1)#cycle_duration)
he3_csum_rate=(*(*(*davePtr).dataStrPtr).specificPtr).he3_csum[0:ncycs-1]/cycle_duration
;
bm0rate=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[915]/$
	(*(*(*davePtr).dataStrPtr).specificPtr).duration
bm1rate=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[916]/$
	(*(*(*davePtr).dataStrPtr).specificPtr).duration
bm2rate=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[917]/$
	(*(*(*davePtr).dataStrPtr).specificPtr).duration
bm3rate=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[918]/$
	(*(*(*davePtr).dataStrPtr).specificPtr).duration
;
;	Range of detectors to include in detector count rate calculation.
r1=0 & r2=912
;
detrate=(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).detsum[r1:r2])/$
	(*(*(*davePtr).dataStrPtr).specificPtr).duration
collosc=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).coll_osc
shutstat=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).shutter_stat
currcyc=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[0]
cycltim=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[1]/60.0
checkst=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[2]
rectemp=(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[3]
resets=(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).resets)
repeats=(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).repeats)
(*pState).prev_stop_date=(*(*(*davePtr).dataStrPtr).specificPtr).stop_date
;
if (nf gt 1) then begin
	for irun=1,nf-1 do begin
		file_path=files[irun]
		dcs_readdatafile,file_path,omithisto,printinfo,filetype,davePtr=davePtr
		progressBar->Update,irun*incr
		progressBar->SetLabel,'Reading file'+strcompress(string(irun))
		comments=[comments,(*(*(*davePtr).dataStrPtr).specificPtr).comments]
		startdates=[startdates,(*(*(*davePtr).dataStrPtr).specificPtr).start_date]
		stopdates=[stopdates,(*(*(*davePtr).dataStrPtr).specificPtr).stop_date]
		user=[user,(*(*(*davePtr).dataStrPtr).specificPtr).user]
;
;	See comments above.
; ntemps is the number of meaningful (positive) temperature readings (0, 1, >0, 0 in cases A, B, C, D)
; ncycs is intended to be the number of good cycles.
		res_temps=where((*(*(*davePtr).dataStrPtr).specificPtr).temp_sample gt 0.0,ntemps)
		res_cycs=where((*(*(*davePtr).dataStrPtr).specificPtr).he3_csum ge 0,ncycs)
;
		case ntemps of
	; No temperature reading or no controller defined (cases A and D).
			0: begin
				t_sample=[t_sample,fltarr(ncycs)-100.0]
				t_control=[t_control,fltarr(ncycs)-100.0]
				t_setpoint=[t_setpoint,fltarr(ncycs)-100.0]
			end
	; One temperature reading (case C), or else not recording temperature (values of 300) (case B).
			1: begin
				ts=(*(*(*davePtr).dataStrPtr).specificPtr).temp_sample[0]
				tc=(*(*(*davePtr).dataStrPtr).specificPtr).temp_control[0]
				tsp=(*(*(*davePtr).dataStrPtr).specificPtr).temp_setpoint
				if (ts eq 300.0 and tc eq 300.0 and tsp eq 300.0) then begin
					ts=-10.0
					tc=-10.0
					tsp=-10.0
				endif
				t_sample=[t_sample,fltarr(ncycs)+ts]
				t_control=[t_control,fltarr(ncycs)+tc]
				t_setpoint=[t_setpoint,fltarr(ncycs)+tsp]
			end
			; Multiple temperature readings.
			else: begin
				t_sample=[t_sample,$
					(*(*(*davePtr).dataStrPtr).specificPtr).temp_sample[0:ncycs-1]]
				t_control=[t_control,$
					(*(*(*davePtr).dataStrPtr).specificPtr).temp_control[0:ncycs-1]]
				t_setpoint=[t_setpoint,(fltarr(ncycs)+1)*$
					(*(*(*davePtr).dataStrPtr).specificPtr).temp_setpoint]
			end
		endcase
		n2=[n2,n2[irun-1]+ncycs]
		wavelength=[wavelength,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).ch_wl]
	 	field_setpoint=[field_setpoint,(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).field_setpoint]
	  field_sample=[field_sample,(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).field_sample]	
		srdenom=[srdenom,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).ch_srdenom]
		tsdmin=[tsdmin,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).tsdmin]
		duration=[duration,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).duration/3600.0]
		runtime=[runtime,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[921]/432000.]
		timebetween=dcs_interpoldate((*pState).prev_stop_date,$
			(*(*(*davePtr).dataStrPtr).specificPtr).start_date,$
			0.0,return_difference=1)
		timelost=[timelost,$
			(fltarr(ncycs)+1)*timebetween[0]*24 > 0.001]
	;
		julstart=dcs_juldate((*(*(*davePtr).dataStrPtr).specificPtr).start_date)
		julstop=dcs_juldate((*(*(*davePtr).dataStrPtr).specificPtr).stop_date)
		cyctim=(julstop-julstart)/ncycs
		abscissa=[abscissa,julstart+findgen(ncycs)*cyctim]
	;
		time_for_run=[time_for_run,$
			(fltarr(ncycs)+1)*(julstop-julstart)*86400.0/60.0]
;
;;		time_counting=(*(*(*davePtr).dataStrPtr).specificPtr).detsum[921]/120.0/60.0
;;		time_for_cycle=(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[1]/60.0
;;		time_duration=(*(*(*davePtr).dataStrPtr).specificPtr).duration/60.0
;;		requested_number_of_cycles=(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[0]
;;		actual_number_of_cycles=time_duration/time_for_cycle
;;		difference=requested_number_of_cycles-fix(actual_number_of_cycles)
;;		remainder=actual_number_of_cycles-fix(actual_number_of_cycles)
;;		comment=''
;;		if (difference gt 0) then begin
;;			comment='    Run truncated'
;;			if (remainder lt 0.025) then comment='   Cycles lost'
;;		endif
;;		print,time_for_run,time_counting,time_for_cycle,time_duration,requested_number_of_cycles,$
;;			actual_number_of_cycles,difference,remainder,file_path,comment,format='(4f8.1,i5,f10.4,i5,f10.4,2a)'
	;
		cycle_duration=(*(*(*davePtr).dataStrPtr).specificPtr).duration_csum[0:ncycs-1]
		if (cycle_duration[0] eq 0.0) then cycle_duration=cycle_duration+1
		high_csum_rate=[[high_csum_rate],$
			[(*(*(*davePtr).dataStrPtr).specificPtr).high_csum[*,0:ncycs-1]/((fltarr(18)+1)#cycle_duration)]]
		he3_csum_rate=[he3_csum_rate,$
			(*(*(*davePtr).dataStrPtr).specificPtr).he3_csum[0:ncycs-1]/cycle_duration]
	;
		bm0rate=[bm0rate,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[915]/$
				(*(*(*davePtr).dataStrPtr).specificPtr).duration]
		bm1rate=[bm1rate,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[916]/$
				(*(*(*davePtr).dataStrPtr).specificPtr).duration]
		bm2rate=[bm2rate,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[917]/$
				(*(*(*davePtr).dataStrPtr).specificPtr).duration]
		bm3rate=[bm3rate,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).detsum[918]/$
				(*(*(*davePtr).dataStrPtr).specificPtr).duration]
		detrate=[detrate,$
			(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).detsum[r1:r2])/$
				(*(*(*davePtr).dataStrPtr).specificPtr).duration]
		collosc=[collosc,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).coll_osc]
		shutstat=[shutstat,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).shutter_stat]
		currcyc=[currcyc,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[0]]
		cycltim=[cycltim,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[1]/60.0]
		checkst=[checkst,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[2]]
		rectemp=[rectemp,$
			(fltarr(ncycs)+1)*(*(*(*davePtr).dataStrPtr).specificPtr).runinfo[3]]
		resets=[resets,$
			(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).resets)]
		repeats=[repeats,$
			(fltarr(ncycs)+1)*total((*(*(*davePtr).dataStrPtr).specificPtr).repeats)]
		(*pState).prev_stop_date=(*(*(*davePtr).dataStrPtr).specificPtr).stop_date
	endfor
endif
;
progressBar->Destroy
Obj_Destroy, progressBar
;
(*pState).nruns=nf
(*pState).runindexPtr=ptr_new([0,n2])
(*pState).runnamesPtr=ptr_new(files)
(*pState).runcommentsPtr=ptr_new(comments)
(*pState).startdatesPtr=ptr_new(startdates)
(*pState).stopdatesPtr=ptr_new(stopdates)
(*pState).userPtr=ptr_new(user)
(*pState).abscissaPtr=ptr_new(abscissa)
;
(*pState).dcsStatePtr[0] = ptr_new([[t_setpoint],[t_control],[t_sample]])
(*pState).dcsStatePtr[1] = ptr_new(field_sample)
(*pState).dcsStatePtr[2] = ptr_new(wavelength)
(*pState).dcsStatePtr[3] = ptr_new(srdenom)
(*pState).dcsStatePtr[4] = ptr_new(tsdmin)
(*pState).dcsStatePtr[5] = ptr_new([[duration],[time_for_run/60.0],[timelost]])
(*pState).dcsStatePtr[6] = ptr_new(transpose(high_csum_rate[2:5,*]))
(*pState).dcsStatePtr[7] = ptr_new(he3_csum_rate)
(*pState).dcsStatePtr[8] = ptr_new([[bm0rate],[bm1rate],[bm2rate],[bm3rate]])
(*pState).dcsStatePtr[9] = ptr_new(detrate)
(*pState).dcsStatePtr[10] = ptr_new((collosc eq 1)*0.999+(collosc eq 0)*0.001)
(*pState).dcsStatePtr[11] = ptr_new((shutstat eq 1)*0.999+(shutstat eq 0)*0.001-(shutstat eq -1)*0.999)
checkst=(checkst eq 1)*0.999+(checkst eq 0)*0.001-(checkst eq -1)*0.999
rectemp=(rectemp eq 1)*0.995+(rectemp eq 0)*0.005-(rectemp eq -1)*0.995
(*pState).dcsStatePtr[12] = ptr_new([[currcyc],[cycltim],[checkst],[rectemp]])
(*pState).dcsStatePtr[13] = ptr_new(resets+0.001)
(*pState).dcsStatePtr[14] = ptr_new(repeats+0.001)
(*pState).winState.autoscale=1
;
wset,(*pState).winState.winPix
dcs_runsinfoPlot,event
wset,(*pState).winState.winVis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winState.winPix]
empty
return
end


;************************************************************************************************
pro dcs_runsinfoDraw,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfoDraw: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;
widget_control,event.top,get_uvalue = pState
type=where(widget_info((*pState).buttons,/button_set),count)
if (count eq 0) then type=0 else type=type[0]
;
!x=(*pState).bangx
!y=(*pState).bangy
!p=(*pState).bangp
;
wset,(*pState).winState.winVis
;
;
case event.type of
0:	begin		; button press
	  (*pState).winState.mouse = event.press
	  if (*pState).winState.mouse eq 4 then begin
	    (*pState).winState.autoscale = 1
	    return
	  endif
	  if (*pState).winState.mouse eq 1 then begin
	    (*pState).winState.xbox[0] = event.x
	    (*pState).winState.ybox[0] = event.y
	    wset,(*pState).winState.winVis
	    device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winState.winPix]
	    empty
	    (*pState).winState.autoscale = 0
	  endif
	end
1:	begin	; button release
	  xll = (*pState).winState.xbox[0] < (*pState).winState.xbox[1]
	  yll = (*pState).winState.ybox[0] < (*pState).winState.ybox[1]
	  w = abs((*pState).winState.xbox[1] - (*pState).winState.xbox[0])
	  h = abs((*pState).winState.ybox[1] - (*pState).winState.ybox[0])
	  xur = xll + w
	  yur = yll + h
	  ll = convert_coord(xll,yll,/device,/to_data,/double)
	  ur = convert_coord(xur,yur,/device,/to_data,/double)
	  (*pState).winState.xrange = [ll[0],ur[0]]
	  (*pState).winState.yrange = [ll[1],ur[1]]
	  wset,(*pState).winState.winPix
	  dcs_runsinfoPlot,event
	  wset,(*pState).winState.winVis
	  device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winState.winPix]
	  (*pState).winState.mouse = 0B
	end
2:	begin	; mouse motion
	  result=convert_coord(event.x,event.y,/device,/to_data,/double)
	  index=*(*pstate).runindexPtr
	  runnames=*(*pstate).runnamesPtr
	  comments=*(*pstate).runcommentsPtr
	  startdates=*(*pstate).startdatesPtr
	  stopdates=*(*pstate).stopdatesPtr
	  user=*(*pstate).userPtr
;
;
		nruns=n_elements(runnames)
	  if (not (*pState).timeabsc) then begin
	  	xlft=index[0:nruns-1]+1
	  	xrgt=index[1:nruns]+1
	  endif else begin
		  xlft=dcs_juldate(startdates)
		  xrgt=dcs_juldate(stopdates)
		endelse
		runnum=fix(max(where(result[0] gt xlft)))
		info=strarr(4)
	  if (runnum ge 0 and runnum le nruns-1) then begin
		  dt1=startdates[runnum]
		  dt2=stopdates[runnum]
			f=(result[0]-xlft[runnum])/(xrgt[runnum]-xlft[runnum])
			showrun=(f le 1.0)
			dtstring=dcs_interpoldate(dt1,dt2,f)
			info[0]=dtstring
			if (showrun) then begin
;;				cycnum=fix(result[0])-1
				cycnum=index[runnum]+fix(f*(index[runnum+1]-index[runnum])) ; This line added.
;;		  	pos0=strpos(runnames[runnum],'200',/reverse_search)
				runstring=strmid(file_basename(runnames[runnum]),4,8); modified JRDC, 1/27/2010.
				if (strmid(runstring,7) eq '.') then runstring=strmid(runstring,0,7)+" "
				xcurs_info=" [Run "+runstring+": cycle"+$
;;					strcompress(string(result[0]-index[runnum],format="(i3)"))+"]"
					strcompress(string(cycnum-index[runnum]+1,format="(i3)"))+"]"
				if ((*(*pState).dcsStatePtr[0])[cycnum,0] gt 0.0) then $
					tset=string((*(*pState).dcsStatePtr[0])[cycnum,0],format="(f8.2)") else tset=" - "
				if ((*(*pState).dcsStatePtr[0])[cycnum,1] gt 0.0) then $
					tctl=string((*(*pState).dcsStatePtr[0])[cycnum,1],format="(f8.2)") else tctl=" - "
				if ((*(*pState).dcsStatePtr[0])[cycnum,2] gt 0.0) then $
					tsam=string((*(*pState).dcsStatePtr[0])[cycnum,2],format="(f8.2)") else tsam=" - "
;
;
				temp_info="  "+strcompress("Set"+tset+";  Ctl"+tctl+";  Sam"+tsam)
				case type of
					0: ycurs_info=string(result[1],format="(f8.2)")+"K"
					1: ycurs_info=string(result[1],format="(f8.2)")+"T"
					2: ycurs_info=string(result[1],format="(f8.3)")+"A"
					3: ycurs_info=string(result[1],format="(i3)")
					4: ycurs_info=string(result[1],format="(f8.0)")+" us"
					5: begin
						if (result[1] gt 2.0) then ycurs_info=string(result[1],$
							format="(f8.2)")+" hr" else $
							ycurs_info=string(result[1]*60.0,format="(f8.2)")+" min"
						end
					6: ycurs_info=string(result[1],format="(f8.2)")+" c/s"
					7: ycurs_info=string(result[1],format="(f8.2)")+" c/s"
					8: ycurs_info=string(result[1],format="(f8.0)")+" c/s"
					9: ycurs_info=string(result[1],format="(f8.0)")+" c/s"
					10: ycurs_info=string(result[1],format="(i3)")
					11: ycurs_info=string(result[1],format="(i3)")
					12: ycurs_info=string(result[1],format="(i5)")
					13: ycurs_info=string(result[1],format="(i6)")
					14: ycurs_info=string(result[1],format="(i7)")
					else:
				endcase
;
				infoline1=dtstring+xcurs_info+temp_info
				infoline2=""
				lgth=(*pState).curswindow_xsize
				for k=1,lgth do infoline2=infoline2+" "
				lastpart='  ('+strcompress(ycurs_info)+")"
				infoline2=strmid(infoline2,0,lgth-strlen(lastpart))+lastpart
				firstpart=comments[runnum]
				if (user[runnum] ne "xxx") then firstpart=firstpart+" [USER = "+user[runnum]+"]"
				infoline2=firstpart+strmid(infoline2,strlen(firstpart))
				infoline3=" Lambda "+"SRD"+" tsdmin "+"Duration"+" Runtime"+$
					" BM0rate"+" BM1rate"+" BM2rate"+" BM3rate"+" Detrate"+" ORC"+$
					" BS"+"  Ncycs"+" Cyctim"+" St"+" RT"+" Reset"+" Repeat"+"  MagField"
				infoline4=$
					string((*(*pState).dcsStatePtr[2])[cycnum],format="(f7.3)")+$
					string((*(*pState).dcsStatePtr[3])[cycnum],format="(i3)")+$
					string((*(*pState).dcsStatePtr[4])[cycnum],format="(f9.0)")+$
					string((*(*pState).dcsStatePtr[5])[cycnum,0],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[5])[cycnum,1],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[6])[cycnum,0],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[6])[cycnum,1],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[6])[cycnum,2],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[6])[cycnum,3],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[7])[cycnum],format="(f8.2)")+$
					string((*(*pState).dcsStatePtr[10])[cycnum],format="(i3)")+$
					string((*(*pState).dcsStatePtr[11])[cycnum],format="(i4)")+$
					string((*(*pState).dcsStatePtr[12])[cycnum,0],format="(i7)")+$
					string((*(*pState).dcsStatePtr[12])[cycnum,1],format="(f7.2)")+$
					string((*(*pState).dcsStatePtr[12])[cycnum,2],format="(i3)")+$
					string((*(*pState).dcsStatePtr[12])[cycnum,3],format="(i3)")+$
					string((*(*pState).dcsStatePtr[13])[cycnum],format="(i6)")+$
					string((*(*pState).dcsStatePtr[14])[cycnum],format="(i7)")+$
					string((*(*pState).dcsStatePtr[1])[cycnum],format="(f8.4)")
				info=[infoline1,infoline2,infoline3,infoline4]
			endif
		endif
		;
		widget_control,(*pState).curswindow,set_value=info
	  if (*pState).winState.mouse eq 1 then begin
	  	(*pState).winState.xbox[1] = event.x
	  	(*pState).winState.ybox[1] = event.y
	  	xc = [(*pState).winState.xbox[0],event.x,event.x,(*pState).winState.xbox[0],$
	  	      (*pState).winState.xbox[0]]
	  	yc = [(*pState).winState.ybox[0],(*pState).winState.ybox[0],event.y,event.y,$
	  	      (*pState).winState.ybox[0]]
	  	wset,(*pState).winState.winVis
	  	device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winState.winPix]
;
    ; Setting color table plot
    device, get_decomposed=old_decomp
    (*pState).old_decomp=old_decomp
    default_colors=bindgen(256,3)
    tvlct,default_colors,/get
    (*pState).default_colors=default_colors
    device, decomposed=0
;    colorset
    tvlct,(*pState).amcolor.red,(*pState).amcolor.green,(*pState).amcolor.blue
    ; plots
    plots,xc,yc,/device,color=1  ; Plot
    ; Recover default colors
   	device, decomposed=old_decomp
   	tvlct,default_colors
	  	empty
	  endif
	end
else:
endcase
return
end


;************************************************************************************************
pro dcs_runsinfoPrint,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfoPrint: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;
thisDevice = !d.name
;deviceKeywords = ps_form(Cancel = canceled,parent = event.top)
devicekeywords = PSConfig(Cancel=canceled,group_leader = event.top,$
    DefaultSetup="Centered (Landscape)",directory=(*pState).workdir)
if canceled ne 1 then begin
  SET_PLOT, 'PS'
  DEVICE, _Extra = deviceKeywords
  dcs_runsinfoPlot,event
  DEVICE, /CLOSE_FILE
  set_plot,thisDevice
endif
return
end


;************************************************************************************************
pro dcs_runsinfo_event,event
;************************************************************************************************
;
compile_opt strictarr
;

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'dcs_runsinfo_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;
widget_control,event.top,get_uvalue = pState
;
nbuttons=n_elements((*pState).buttons)
for k=0,nbuttons-1 do begin
	if (event.id eq (*pState).buttons[k]) then begin
		(*pState).winState.autoscale = 3
	endif
endfor
;
if event.id eq (*pState).showbox then begin
	if ((*pState).showtimes eq 1) then begin
		(*pState).showtimes=0
		widget_control,(*pstate).showbox,set_value="Show times"
	endif else begin
		(*pState).showtimes=1
		widget_control,(*pstate).showbox,set_value="Hide times"
	endelse
endif
;
if event.id eq (*pState).grid then begin
	if ((*pState).showgrid[0] eq 1) then begin
		(*pState).showgrid[0]=0
		widget_control,(*pstate).grid,set_value="Draw horizontal solid lines at major ticks"
	endif else begin
		(*pState).showgrid[0]=1
		widget_control,(*pstate).grid,set_value="Hide horizontal solid lines at major ticks"
	endelse
endif
;
if event.id eq (*pState).mgrid then begin
	if ((*pState).showgrid[1] eq 1) then begin
		(*pState).showgrid[1]=0
		widget_control,(*pstate).mgrid,set_value="Draw horizontal dashed lines at major+minor ticks"
	endif else begin
		(*pState).showgrid[1]=1
		widget_control,(*pstate).mgrid,set_value="Hide horizontal dashed lines at major+minor ticks"
	endelse
endif
;
if event.id eq (*pState).vert then begin
	if ((*pState).showvert eq 1) then begin
		(*pState).showvert=0
		widget_control,(*pstate).vert,set_value="Show run boundaries"
	endif else begin
		(*pState).showvert=1
		widget_control,(*pstate).vert,set_value="Hide run boundaries"
	endelse
endif
;
if event.id eq (*pState).tsswap then (*pState).tsensorswap=1-(*pState).tsensorswap
;
if event.id eq (*pState).absc then begin
	(*pState).winState.autoscale=1
	if ((*pState).timeabsc eq 1) then begin
		(*pState).timeabsc=0
		widget_control,(*pstate).absc,set_value="Use date/time as abscissa"
	endif else begin
		(*pState).timeabsc=1
		widget_control,(*pstate).absc,set_value="Use run cycle as abscissa"
	endelse
endif
;
for k=0,4 do begin
	if event.id eq (*pState).minor[k] then begin
		for kp=0,4 do begin
			widget_control,(*pState).minor[kp],set_button=0
		endfor
		widget_control,(*pState).minor[k],/set_button
		(*pState).nminor=k
		break
	endif
endfor
;
;;if event.id eq (*pState).modebox then begin
;;	if ((*pState).plotmode eq 1) then begin
;;		(*pState).plotmode=0
;;		widget_control,(*pstate).modebox,set_value="Plot relative values"
;;	endif else begin
;;		(*pState).plotmode=1
;;		widget_control,(*pstate).modebox,set_value="Plot absolute values"
;;	endelse
;;endif
;
widget_control,event.top,set_uvalue = pState
wset,(*pState).winState.winPix
dcs_runsinfoPlot,event
wset,(*pState).winState.winVis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pState).winState.winPix]
empty
;
end


;************************************************************************************************
pro dcs_runsinfo,$
	group_leader = group_leader,$
;	notify_ids = notify_ids,$
	register_name = register_name, $
	dataDir=dataDir,workDir=workDir, $
	_EXTRA=etc
;************************************************************************************************
;
compile_opt strictarr
;
;
;;;;;;;added following 4 lines for non-modal widget action
if n_elements(group_leader) eq 0 then group_leader = 0L
if n_elements(notify_ids) eq 0 then notify_ids = [0L,0L]
if n_elements(register_name) eq 0 then register_name = 'DCS_runsinfo'
if xregistered(register_name) then return
if n_elements(workDir) eq 0 then workDir = ''
if n_elements(dataDir) eq 0 then dataDir = ''

;

; get old_decomp
device, get_decomposed=old_decomp

; Widget definition module
;
; To add a new quantity to the list of those that can be displayed, make changes
; (1) in the definition of 'sigtypes' (below)
; (2) to the size of dcsStatePtr (below)
; (3) in dcs_runsinfoRead to the initial value and to subsequent values
;	(4) in dcs_runsinfoRead define the new element in (*pState).dcsStatePtr
; (5) in dcs_runsinfoDraw to the values of ycurs_info, infoline3, infoline4
; (6) in dcs_runsinfoPlot to define the y label.
; (7) in dcs_runsinfoSave to define the y label.
;
;

default_colors=bindgen(256,3)
colorset
tvlct,default_colors,/get
device, decomposed=0
amcolor = {color, $ ; name
red:   [255,0,255,  0,  0,  0,255,255,152,128,128,128,255,178,255],$ ; red values
green: [255,0,  0,255,  0,255,  0,255,120,128,128,128, 54,192,128],$ ; green values
blue:  [255,0,  0,  0,255,255,255,  0,103,220, 60,128, 54,220,128] $ ; blue values
}

tvlct,amcolor.red,amcolor.green,amcolor.blue
;
tlb = widget_base(/row,title = 'DCS Run Information Display',mbar = bar,$
	group_leader=group_leader)
	filemenu = widget_button(bar,value = 'File',/menu)
		void = widget_button(filemenu,value = 'Select raw data files',$
			event_pro = 'dcs_runsinfoRead')
		void = widget_button(filemenu,value = 'Print this plot to a postscript file',$
       event_pro = 'dcs_runsinfoPrint')
		void = widget_button(filemenu,value = 'Save data to a text file',$
       event_pro = 'dcs_runsinfoSave')
		void = widget_button(filemenu,value = 'Quit',event_pro = 'dcs_runsinfoQuit')
	optmenu = widget_button(bar,value = 'Options',/menu)
		grid = widget_button(optmenu,value = 'Draw horizontal solid lines at major ticks')
		mgrid = widget_button(optmenu,value = 'Draw horizontal dashed lines at major+minor ticks',/separator)
		minor=lonarr(5)
		for k=0,4 do minor[k] = widget_button(optmenu,value = strcompress(k)+' minor ticks',/checked_menu)
		nminor=3
		widget_control,minor[nminor],/set_button
		vert = widget_button(optmenu,value = 'Hide run boundaries',/separator)
		tsswap = widget_button(optmenu,value = 'Swap control and sample legend labels',/separator)
	abscmenu = widget_button(bar,value = 'Abscissa',/menu)
		absc = widget_button(abscmenu,value = 'Use run cycle as abscissa',/separator)
	base1 = widget_base(tlb,/col)
		void=widget_label(base1,value='Selection')
		sigtypes = ['Temperatures','Magnetic Field','Wavelength','S. R. denominator','tsdmin',$
			'Times','Beam monitor rates','3He detector rate','Mean Beam monitor rates','Mean 3He detector rate','Status of ORC',$
			'Status of shutter','Run parameters','Total resets','Total repeats']
;
		nbuttons=n_elements(sigtypes)
		buttons=lonarr(nbuttons)
		butbase=widget_base(base1,/col,/exclusive,frame=1)
		for k=0,nbuttons-1 do begin
			buttons[k]=widget_button(butbase,value=sigtypes[k],/no_release)
		endfor
		widget_control,buttons[0],/set_button
		widget_control,buttons[7],sensitive=1
		widget_control,buttons[8],sensitive=1
;
		showbox = widget_button(base1,value='Hide times')
		modebox = 0l;  widget_button(base1,value='Plot relative values')
		void = widget_button(base1,value='Quit',event_pro='dcs_runsinfoquit')
	base2 = widget_base(tlb,/col)
		base3 = widget_base(base2,/align_center)
			curswindow = widget_text(base3,xsize=20,ysize=4,/editable,font="COURIER*14",/scroll)
			geometry=widget_info(curswindow,/geometry)
			conversion=geometry.xsize/geometry.scr_xsize
			xsize = !screensize[0]*0.8 & ysize = !screensize[1]*0.7
			win = widget_draw(base2,xsize = xsize,ysize = ysize,/button_events,$
	      event_pro = 'dcs_runsinfoDraw',/motion_events)
;
centertlb,tlb
widget_control,tlb,/realize
;
			curswindow_xsize=xsize*conversion
			widget_control,curswindow,xsize=curswindow_xsize
;
widget_control,win,get_value = winVis
window,/free,/pixmap,xsize = xsize,ysize = ysize
winPix = !d.window

winState =  {winPix:winPix,$
             winVis:winVis,$
             win:win,$
             xbox:[0.0,1.0],$
             xrange:[0.0d0,1.0d0],$
             ybox:[0.0,1.0],$
             yrange:[0.0d0,1.0d0],$
             xPtr:ptr_new(/allocate_heap),$
             yPtr:ptr_new(/allocate_heap),$
             xtitle:'',$
             ytitle:'',$
             title:'',$
             mouse:0B,$
             autoscale:1}

state = {curswindow:curswindow,$
         curswindow_xsize:curswindow_xsize,$
         winState:winState,$
         abscissaPtr:ptr_new(/allocate_heap),$
         dcsStatePtr:ptrarr(15,/allocate_heap),$
         nruns:1,$
         grid:grid,$
         mgrid:mgrid,$
         showgrid:[0,0],$
         minor:minor,$
         nminor:nminor,$
         vert:vert,$
         tsswap:tsswap,$
         absc:absc,$
         showvert:1,$
         timeabsc:1,$
         showbox:showbox,$
         showtimes:1,$
         modebox:modebox,$
         tsensorswap:0,$
 ;;        plotmode:0,$
         runcommentsPtr:ptr_new(/allocate_heap),$
    		 buttons:buttons,$
         runindexPtr:ptr_new(/allocate_heap),$
         runnamesPtr:ptr_new(/allocate_heap),$
         userPtr:ptr_new(/allocate_heap),$
         startdatesPtr:ptr_new(/allocate_heap),$
         stopdatesPtr:ptr_new(/allocate_heap),$
         notify_ids:notify_ids,  $              ; for non-modal widget
         prev_stop_date:"",      $
         default_colors:default_colors,  $
         amcolor:amcolor,                $
         old_decomp:old_decomp, $
         bangx:!x,$
         bangy:!y,$
         bangp:!p, $
         workDir:workDir, $
         dataDir:dataDir }

widget_control,tlb,set_uvalue = ptr_new(state,/no_copy)
;
dcs_runsinfoRead,{event,id:void,top:tlb,handler:0l}
;
if (widget_info(tlb,/valid_id)) then $
; added register_name variable to the call to xmanager.
	xmanager,register_name,tlb,/no_block,cleanup='dcs_runsinfocleanup',$
		event_handler="dcs_runsinfo_event"
;
end
