; $Id: dm_to_number.pro,v 1.17 2017/03/16 19:34:56 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_to_number
;
; PURPOSE:
;  convert string to numbers,default is to convert to a float
;  if the string is empty, return NaN for float or double type
;  return 0 for int,long or byte type
;  return 0 if the string matched any zerostring
;  return 1 if the string matched any onestring
;  return 2 if the string matcched any twostring
;  
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  December, 2022
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

function dm_to_number,numstr,float=float,int=int,double=double,byte=byte,long=long,count=count,date=date,zerostring=zerostring,onestring=onestring,twostring=twostring,epoch=epoch
    count = n_elements(numstr)
    if count eq 0 then return, 0
    tmp = size(numstr,/dimension) 
    if count eq 1 then tmp = 1
    str = numstr 
    if size(numstr[0],/type) ne 7 then str = strtrim(string(numstr),2)
    if keyword_set(int)           then val = intarr(tmp)    $
    else if keyword_set(long)     then val = lonarr(tmp)    $
    else if keyword_set(double)   then val = dblarr(tmp)    $
    else if keyword_set(byte)     then val = bytarr(tmp)    $
    else if keyword_set(date)     then val = ulon64arr(tmp) $
    else if keyword_set(epoch)    then val = lon64arr(tmp)  $
    else val = fltarr(tmp)
    if n_elements(zerostring) ne 0 then string0 = strlowcase(strtrim(zerostring,2))
    if n_elements(onestring) ne 0 then  string1 = strlowcase(strtrim(onestring,2))
    if n_elements(twostring) ne 0 then  string2 = strlowcase(strtrim(twostring,2))
    if keyword_set(epoch) or keyword_set(date) then begin
       strformat = ['([0-9]{1,2})[ -]([a-z]{3})[ -]([0-9]{4}) +([0-9]{1,2})\:([0-9]{1,2})\:([0-9]{1,2}) *([a-z]*)',$                            ;DCS  'DD-MON-YYYY HH:mm:ss'
                    '([0-9]{4})\-([0-9]{1,2})\-([0-9]{1,2})[t ]+([0-9]{1,2})\:([0-9]{1,2})\:([0-9]{1,2}(\.[0-9]{3}){0,1}) *(\-*[a-z0-9:+]*)',$  ;ICE  'YYYY-MM-DD HH:mm:ss EST'
                                                                                                                                                ;NICE 'YYYY-MM-DDTHH:mm:ss(.SSS)Z(-05:00 or -0500)
                    '([a-z]{3}) +([0-9]{1,2}) +([0-9]{1,2})\:([0-9]{1,2})\:([0-9]{1,2}) +([0-9]{4}) *([a-z]*)',$                                ;IDL  'DOW MON DD HH:mm:ss YYYY'
                    '([0-9]{1,2})\/([0-9]{1,2})\/([0-9]{4}) +([0-9]{1,2})\:([0-9]{1,2})\:([0-9]{1,2}) *([a-z]*)',$                              ;'MM/DD/YYYY HH:mm:ss'
                    '([0-9]{1,2})[ -]([a-z]{3})[ -]([0-9]{4})','([0-9]{4})\-([0-9]{1,2})\-([0-9]{1,2}) +([0-9]{1,2})','([0-9]{1,2})\/([0-9]{1,2})\/([0-9]{4})'] 
       date_indx = [[6,5,4,3,2,1],[6,5,4,1,2,3],[5,4,3,6,1,2],[6,5,4,3,1,2],[0,0,0,3,2,1],[0,0,0,1,2,3],[0,0,0,3,1,2]]
       monisstr  = [1,0,1,0,1,0,0]
       months    = ['jan','feb','mar','apr','may','jun','jul','aug','sep','oct','nov','dec']
       cdf_epoch,epoch_0,1970,1,1,0,0,0,/compute_epoch & epoch_0 = epoch_0/1000
    endif
    on_ioerror,error
    for i=0L,count-1 do begin
        flag = 1b
        if keyword_set(epoch) or keyword_set(date) then begin ;number of seconds that have elapsed since 00:00:00 Coordinated Universal Time (UTC), 1 January 1970 or construct a ulon64 number yyyymmddhhmmss
           if n_elements(tm_yr) ne 0 then tmp = temporary(tm_yr)
           for j=0,n_elements(strformat)-1 do begin
               tmp = stregex(str[i],strformat[j],/fold_case,/subexpr,len=len)
               if len[0] gt 0 then begin
                  tm_mil = 0l
                  if date_indx[0,j]+1 lt n_elements(len) then begin
                     tmp_str = strmid(str[i],tmp[date_indx[0,j]+1],len[date_indx[0,j]+1])
                     if stregex(tmp_str,'^\.[0-9]+$',/boolean) then tm_mil = fix(dm_to_number(tmp_str)*1000)  ;in case double keyword is set
                  endif
                  if date_indx[0,j] ne 0 then tm_sec = dm_to_number(strmid(str[i],tmp[date_indx[0,j]],len[date_indx[0,j]]),/long) else tm_sec = 0l
                  if date_indx[1,j] ne 0 then tm_min = dm_to_number(strmid(str[i],tmp[date_indx[1,j]],len[date_indx[1,j]]),/long) else tm_min = 0l
                  if date_indx[2,j] ne 0 then tm_hr  = dm_to_number(strmid(str[i],tmp[date_indx[2,j]],len[date_indx[2,j]]),/long) else tm_hr  = 0l
                  tm_yr = dm_to_number(strmid(str[i],tmp[date_indx[3,j]],len[date_indx[3,j]]),/long)
                  if monisstr[j] then $
                     tm_mon = (where(months eq strlowcase(strmid(str[i],tmp[date_indx[4,j]],len[date_indx[4,j]]))))[0]+1 $
                  else $
                     tm_mon = dm_to_number(strmid(str[i],tmp[date_indx[4,j]],len[date_indx[4,j]]),/long) 
                  tm_day = dm_to_number(strmid(str[i],tmp[date_indx[5,j]],len[date_indx[5,j]]),/long)
                  if keyword_set(epoch) and (n_elements(len) ge 8) then begin
                     if len[n_elements(len)-1] ne 0 then begin
                        tmp_str = strlowcase(strmid(str[i],tmp[n_elements(len)-1],len[n_elements(len)-1]))
                        if stregex(tmp_str,'^[a-z]+$',/boolean) then begin
                           us_tzcode = ['est','edt','cst','cdt','mst','mdt','pst','pdt','akst','akdt','ast','adt','hst','hdt']
                           us_offset = [5,4,6,5,7,6,8,7,9,8,4,3,10,9]
                           tmp_ind   = where(us_tzcode eq tmp_str,count)
                           if count ne 0 then tm_hr = tm_hr+us_offset[tmp_ind]
                        endif else if len[n_elements(len)-1] gt 2 then tm_hr = tm_hr-dm_to_number(strmid(str[i],tmp[n_elements(len)-1],len[n_elements(len)-1]-2),/int)
                     endif
                  endif
                  break
               endif
           endfor
           if keyword_set(epoch) and n_elements(tm_yr) ne 0 then begin
              cdf_epoch,epoch_1,tm_yr,tm_mon,tm_day,tm_hr,tm_min,tm_sec,tm_mil,/compute_epoch
              val[i] = epoch_1/1000-epoch_0
           endif 
           if keyword_set(date) and n_elements(tm_yr) ne 0 then val[i] = tm_yr*(10000000000ull)+tm_mon*(100000000ull)+tm_day*(1000000ull)+tm_hr*(10000ull)+tm_min*(100ull)+tm_sec
        endif else begin
           ;get rid of extra ., for example, '8.0.1'->'8.01'
           tmp1 = strpos(str[i],'.') & tmp2 = strpos(str[i],'.',/reverse_search)
           while (tmp1 ne tmp2) do begin
                 str[i] = strmid(str[i],0,tmp2)+strmid(str[i],tmp2+1,strlen(str[i])-tmp2-1)
                 tmp1   = strpos(str[i],'.') & tmp2 = strpos(str[i],'.',/reverse_search)
           endwhile
           tmp = str[i]+' '  ;+' ' to force error with emptry string ''
           if keyword_set(int)       then val[i] = fix(str[i],type=2) $
           else if keyword_set(long) then val[i] = fix(str[i],type=3) $
           else if keyword_set(byte) then begin
              if i eq 0 then val = fix(str[i],type=1) $
              else val = [val,fix(str[i],type=1)]
           endif else if keyword_set(double) then val[i] = double(tmp) $
           else val[i] = float(tmp)
        endelse
        flag = 0b  ;clear error flag
        error: if flag then begin
               if  keyword_set(int)        then val[i] = 0s $
               else if keyword_set(long)   then val[i] = 0l $
               else if keyword_set(double) then val[i] = !values.d_nan $
               else if keyword_set(byte)   then val[i] = 0b $
               else if keyword_set(date)   then val[i] = 0ull  $
               else if keyword_set(epoch)  then val[i] = 0ll $
               else val[i] = !values.f_nan
        endif
        if finite(val[i],/nan) or keyword_set(int) or keyword_set(long) then begin
           tmp = strlowcase(strtrim(str[i],2))
           if n_elements(string0) ne 0 then begin
              ind = where(string0 eq tmp,count1)
              if count1 gt 0 then val[i] = 0
           endif
           if n_elements(string1) ne 0 then begin
              ind = where(string1 eq tmp,count1)
              if count1 gt 0 then val[i] = 1
           endif
           if n_elements(string2) ne 0 then begin
              ind = where(string2 eq tmp,count1)
              if count1 gt 0 then val[i] = 2
           endif
        endif
    endfor
    if keyword_set(byte) then count = n_elements(val)
    return,(count eq 1)?val[0]:val
end