; $Id$
;#######################################################################
;
; NAME:
;  dm_macs_sequencer
;
; PURPOSE:
;  this program implements Jose Rodriguez's IDL porgrams to compile and send experiment sequence to ICE 
;
; CATEGORY:
;  dcs_mslice file tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  January, 2024
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

@dm_check_macskidney

;runs specified a3 scan at different kidney angles at a given Ei
;monitor keyword is a little bit unreliable
pro dm_a3scan,Ei,step,Ef=Ef,beta1=beta1,beta2=beta2,gobetas=gobetas,dmbt=dmbt,keepstep=keepstep,max_step=max_step,shifthalfstep=shifthalfstep,kid_i=kid_i,kid_f=kid_f,n_kid=n_kid,a3_scan_name=a3_scan_name,comment=comment,$
    filename=filename,spinstate=spinstate,monitor=monitor,movea3=movea3,movebasesampletheta=movebasesampletheta,old_step=old_step,temperature=temperature,threeblade=threeblade,vfocus=vfocus,backlash=backlash,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_a3scan runs specified a3 scan at different kidney angles at a given Ei.',$
                 'syntax: dm_a3scan, Ei, step, beta1=beta1, beta2=beta2, Ef=Ef, a3_scan_name=a3_scan_name, comment=comment, filename=filename, spinstate=spinstate, $',$
                 '                  movea3=movea3, movebasesampletheta=movebasesampletheta, gobetas=gobetas, dmbt=dmbt, keepstep=keepstep, max_step=max_step, $',$
                 '                  shifthalfstep=shifthalfstep, threeblade=threeblade, vfocus=vfocus, kid_i=kid_i, kid_f=kid_f, n_kid=n_kid, help=help',$
                 sep+'Ei:'+tab+tab+'incident energy in unit of meV, a scalar.',$
                 sep+'step:'+tab+tab+'kidney step, a scalar. If keepstep keyword is not set, an optimized step will be used instead to achieve evenly spaced A4.',$
                 sep+'beta1, beta2:'+tab+'beta1 and beta2 value, a scalar.',$
                 sep+'Ef:'+tab+tab+'final energy in unit of meV, a scalar. If present, Ef will be set in the sequence.',$
                 sep+'a3_scan_name:'+tab+'a3 scan name, a string or string array. This named scan is set up and stored in ICE.',$
                 sep+'comment:'+tab+'new comment of the scan, a string. The new comment will replace the existing comment in the specified scan. Do not use comma, < or > character.',$
                 sep+'filename:'+(keyword_set(printhelp)?tab:tab+tab)+'new file name of the scan, a string. The new file name will replace the existing file name in the specified scan.',$
                 sep+'spinstate:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, the spin state will be added to comment and the end of the file name if those keywords are specified.',$
                 sep+'movea3:'+(keyword_set(printhelp)?tab:tab+tab)+'a3 value to move to at the beginning and end of each scan.',$
                 sep+'movebasesampletheta:'+tab+'basesampletheta value to move to at the beginning and end of each scan.',$
                 sep+'gobetas:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, beta1 and beta2 calculated from Ei and dmbt will be set. dmbt keyword needs to be given a value.',$
                 sep+'dmbt:'+tab+tab+'dmbt in unit of mm, a scalar. It is combined with gobetas keyword to set beta1 and beta2.',$
                 sep+'keepstep:'+tab+'if set, the specified step will be used, otherwise an optimized step will be used.',$
                 sep+'max_step:'+tab+'if set, use the largest optimized step. The larger the step, the wider the uneven edge.',$
                 sep+'shifthalfstep:'+(keyword_set(printhelp)?'':tab)+'if set, the kidney angles will be shifted by half a step.',$
                 sep+'temperature:'+tab+'keyword for restoring temperature if the heater was turned off during instrument movement. Temperature controller needs to be in the zone mode.',$
                 sep+'threeblade:'+(keyword_set(printhelp)?tab:tab)+'if set, horizontal focus is flat, and only three monblades are used.',$
                 sep+'vfocus:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, in vertical focus mode. threeblade keyword needs to be set.',$
                 sep+'kid_i:'+(keyword_set(printhelp)?tab:tab+tab)+'starting kidney angle, a scalar. Default is the lower limit of the kidney angle.',$
                 sep+'kid_f:'+(keyword_set(printhelp)?tab:tab+tab)+'final kidney angle, a scalar. Default is the upper limit of the kidney angle.',$
                 sep+'backlash:'+tab+'kidney backlash, default is 2 degrees.',$
                 sep+'n_kid:'+(keyword_set(printhelp)?tab:tab+tab)+'save the output number of kidney angles.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       ei_max = !dm_macsseq_dryrun.ei_max
       ei_min = !dm_macsseq_dryrun.ei_min
    endif
    
    dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min
    dm_check_parm,step,name='step',/number,/gezero
    for i=0,n_elements(a3_scan_name)-1 do dm_check_parm,a3_scan_name,name='a3_scan_name',/string
    if n_elements(Ef) ne 0 then dm_check_parm,Ef,name='Ef',/number,/positive
    if n_elements(movea3) ne 0 then dm_check_parm,movea3,name='movea3',/number
    if n_elements(movebasesampletheta) ne 0 then dm_check_parm,movebasesampletheta,name='movebasesampletheta',/number
    if n_elements(beta1) ne 0 then dm_check_parm,beta1,name='beta1',/number
    if n_elements(beta2) ne 0 then dm_check_parm,beta2,name='beta2',/number
    if n_elements(backlash) ne 0 then dm_check_parm,backlash,name='backlash',/number,/positive else backlash = 2.0
    if keyword_set(gobetas) then dm_check_parm,dmbt,name='dmbt',/number,/gezero
    if n_elements(temperature) ne 0 then dm_check_parm,temperature,name='temperature',/number,/positive
    
    kran = dm_getkidneyrange(Ei[0])
    kid0 = kran[0]
    kran[0] = kran[0]+backlash
    
    if n_elements(kid_i) ne 0 then begin
       dm_check_parm,kid_i,name='kid_i',/number,upperlimit=kran[1]
       kran[0] = (kran[0])>(kid_i[0])
    endif
    if n_elements(kid_f) ne 0 then begin
       dm_check_parm,kid_f,name='kid_f',/number,lowerlimit=kran[0]
       kran[1] = (kran[1])<(kid_f[0])
    endif
    if keyword_set(keepstep) then begin
       tmp_step = step[0]
       all_step = dm_optimizekidstep(tmp_step/floor(tmp_step),kran,/all_step,eq_step=all_equv)
       ind = where(abs(all_step-tmp_step) lt 0.0001,cnt)
       if cnt ne 0 then eq_step = all_equv[ind[0]] 
    endif else begin
       if keyword_set(old_step) then tmp_step = dm_optimizekidstep_old(step[0],kran,eq_step=eq_step) $
       else tmp_step = dm_optimizekidstep(step[0],kran,eq_step=eq_step,max_step=max_step)
    endelse
    
    n_kid = (tmp_step eq 0)?1:(floor(abs(kran[1]-kran[0])/tmp_step)+1)-keyword_set(shifthalfstep)
    range = (n_kid-1)*tmp_step
    kcen  = min(kran)+range/2.0+tmp_step/2.0*keyword_set(shifthalfstep)
    
    if Ei[0] gt 16.5 then begin
       comm0 = ['move ei 16 a4 0','move ei 16.5 kidney -51']    
    endif else begin
       comm0 = 'move ei '+dm_to_string(Ei[0])+' a4 0 '
    endelse
    comm0 = [comm0,'move ei '+dm_to_string(Ei[0])+' kidney '+dm_to_string(kcen,res=3)]
    if n_elements(beta1) ne 0 then comm0=[comm0,'move beta1 '+dm_to_string(beta1)]
    if n_elements(beta2) ne 0 then comm0=[comm0,'move beta2 '+dm_to_string(beta2)]
    
    if n_elements(movea3) ne 0 then dm_move,'a3',movea3
    if n_elements(movebasesampletheta) ne 0 then dm_move,'basesampletheta',movebasesampletheta
    dm_sendcom,'autoptai'
    if keyword_set(threeblade) then begin
       dm_sendcom,'action vertifocus flat'
       dm_sendcom,'action horizfocus flat'
    endif
    if n_elements(temperature) ne 0 then begin
       tempstr = ['move temp 1','move temp '+dm_to_string(temperature[0])]
       comm0 = [tempstr[0],comm0,tempstr[1]]
    endif
    for i=0,n_elements(comm0)-1 do dm_sendcom,comm0[i]
    if keyword_set(threeblade) then begin
       dm_move,'beta1',3
       dm_move,'beta2',0
       dm_setblades,3
       if keyword_set(vfocus) then dm_move,'focus',dm_get_focus(Ei[0])
    endif
    if n_elements(Ef) ne 0 then dm_sendcom,'move ef '+dm_to_string(Ef[0])
    if keyword_set(gobetas) then dm_gobetas,Ei[0],dmbt[0]
    for i=0,n_kid-1 do begin
        kid = min(kran)+tmp_step*(i+0.5*keyword_set(shifthalfstep))
        if n_elements(temperature) ne 0 then dm_sendcom,tempstr[0]
        if i eq 0 then dm_move,'kidney',(kid0)>(kid-backlash)   ;to overcome backlash
        dm_move,'kidney',kid
        if n_elements(temperature) ne 0 then dm_sendcom,tempstr[1]
        if i eq 0 then begin
           dm_runscan,a3_scan_name,comment=comment,filename=filename,monitor=monitor,spinstate=spinstate 
        endif else begin
           dm_runscan,a3_scan_name
        endelse
        if n_elements(movea3) ne 0 then dm_move,'a3',movea3
        if n_elements(movebasesampletheta) ne 0 then dm_move,'basesampletheta',movebasesampletheta
    endfor
    if keyword_set(threeblade) and keyword_set(vfocus) then dm_sendcom,'action vertifocus flat'
    dm_print,'A3 scan: Ei='+dm_to_string(Ei[0])+' meV, total kidney number='+dm_to_string(n_kid)+',  actual step='+dm_to_string(tmp_step,res=4)+$
             (n_elements(eq_step) ne 0?', equivalent A4 step='+dm_to_string(eq_step,res=4):'')+', scan name="'+a3_scan_name+'"'

    if exists then begin
       !dm_macsseq_dryrun.nes  = !dm_macsseq_dryrun.nes+1
       !dm_macsseq_dryrun.nkid = !dm_macsseq_dryrun.nkid+n_kid
    endif
end

;align A5 at a given Ef.
pro dm_align_a5,Ef,directory,saveweights=saveweights,help=help,nowait=nowait,output=output,group_leader=group_leader
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_align_a5 aligns A5 at a given Ef.',$
                 'syntax: dm_align_a5, Ef, nowait=nowait, saveweights=saveweights, help=help',$
                 sep+'Ef:'+tab+tab+'final energy in unit of meV, a scalar.',$
                 sep+'directory:'+tab+'optional directory for saving files. Only used if saveweights keyword is set.',$
                 sep+'nowait:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, no wait command will be issued while resetting analyzer positions.',$
                 sep+'saveweights:'+tab+'if set, detector weights will be saved in weights_Ef_'+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
  
    dm_check_parm,Ef,name='Ef',/number,/positive
    if n_elements(directory) eq 0 then cd,current=directory
    dm_check_parm,directory,name='directory',/string
    
    ;if (Ef[0] ne 2.35) and (Ef[0] ne 3.7) and (Ef[0] ne 5) then message,'Ef must be 2.35, 3.7 or 5 meV.'
    Ei   = Ef[0]
    kran = dm_getkidneyrange(Ei)
    kmin = kran[0] 
    kmax = kran[1]
    if kmax-kmin gt 26 then begin
       kmin = kmin+1.0
       kmax = kmax-1.0
    endif 
    ignore_low_k  = where(abs(kmin-76.0+findgen(20)*8.0) le 12)
    ignore_high_k = where(abs(kmax-76.0+findgen(20)*8.0) le 12)
    a5pos = asin(!pi/3.35416/sqrt(Ei/2.0721248))/!dtor

    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj)

    dm_sendcom,'action horizfocus flat'
    dm_sendcom,'action vertifocus sagittal '
    dm_sendcom,'move dmbt 20'
    dm_sendcom,'autoptai'
    dm_sendcom,'move ei '+dm_to_string(Ei)+' a4 0'
    dm_sendcom,'move beta1 0 beta2 0'
    dm_move,'kidney',kmax
    dm_move,'ef',Ei
    dm_findpeak,'a5',5.0,0.2,5.0,/sync,datafile=data1
    dm_print,'finish first scan'
    
    if dryrun then dm_print,'Plotting and fitting first scan data are not available during dryrun.' $
    else dm_plot_macs,data1,'a5',center=a5_k_h,intg_int=weights_h
    
    dm_move,'kidney',kmin,/sync
    dm_findpeak,'a5',5.0,0.2,5.0,/sync,datafile=data2
    dm_print,'finish second scan'
    
    if dryrun then dm_print,['Plotting and fitting the second scan data are not available during dryrun.','Resetting analyzers are not availabel during dryrun.'] $
    else begin
       dm_plot_macs,data2,'a5',center=a5_k_l,intg_int=weights_l
       
       a5_ave  = (a5_k_l + a5_k_h)/2.
       weights = (weights_l+weights_h)/2.
       
       for nn=0,n_elements(ignore_high_k)-1 do begin
           a5_ave[ignore_high_k[nn]]  = a5_k_l[ignore_high_k[nn]]
           weights[ignore_high_k[nn]] = weights_l[ignore_high_k[nn]]
       endfor
       for nn=0,n_elements(ignore_low_k)-1 do begin
           a5_ave[ignore_low_k[nn]]=a5_k_h[ignore_low_k[nn]]
           weights[ignore_low_k[nn]] = weights_h[ignore_low_k[nn]]
       endfor
       
       dm_print,'a5_ave: '+dm_to_string(a5_ave,separator=', ')
       
       ;move to the center and reset
       ok = dialog_message('Do you want to move to the fit center and reset analyzer positions?',title='Please confirm:',/question,dialog_parent=group_leader,/center)
       if ok eq 'Yes' then begin
          if ~keyword_set(nowait) then dm_sendcom,'wait 6666'
          for i=1,20 do begin
              analyzer = 'analyzertheta'+string(i,format='(i02)')
              dm_move,analyzer,a5_ave[i-1]
              dm_sendcom,'device set '+analyzer+' '+dm_to_string(a5pos)
          endfor
       endif
    
       if keyword_set(saveweights) then begin
          ;open file for writing
          Efstr = dm_to_string(Ei)+'meV'
          while ((index=strpos(Efstr,'.')) ge 0) do strput,Efstr,'p',index  ;replace decimal point by letter 'p'
          file = directory+dm_define_pointer(/getpathsep)+'weights_'+Efstr+'_'+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt'
          openw,lun,file,/get_lun,error=openerr
          if openerr ne 0 then message,"Can't write in "+file+'.'
          ;calculate weights
          weights = 1./weights
          weights = weights*20/total(weights) 
          ;print weights
          printf,lun,weights
          free_lun,lun    
       endif
    endelse
end

;align monocorhomator
pro dm_align_monochromator,directory,savecenter=savecenter,setblades=setblades,help=help,nowait=nowait,output=output,group_leader=group_leader
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_align_monochromator aligns the monochormator blades at 45 degrees angle.',$
                 'syntax: dm_align_monochromator, directory, nowait=nowait, savecenter=savecenter, setblades=setblades, help=help',$
                 sep+'directory:'+tab+'optional directory for saving files. Only used if savecenter keyword is set.',$
                 sep+'nowait:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, no wait command will be issued while resetting monochromator blade positions.',$
                 sep+'savecenter:'+tab+'if set, fitted blade centers will be saved in monbladecenter_'+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt.',$
                 sep+'setblades:'+tab+'if set, move the blades to the fitted center and reset them to 45.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
   
    if n_elements(directory) eq 0 then cd,current=directory
    dm_check_parm,directory,name='directory',/string
    
    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj)
    
    ; Preparing macs to align the monochromator
    dm_sendcom,'action vertifocus flat '
    dm_sendcom,'action horizfocus flat'
    dm_sendcom,'move beta1 0 beta2 0'
    dm_move, 'a1', 44
    dm_move, 'dmbt',10
    dm_move, 'vbah',10
    dm_move, 'vbav',360
    dm_sendcom,'autoptai'
    dm_sendcom,'move a2 90 a4 0'
    dm_sendcom,'move a1 45'
    dm_move, 'monrot', 0
    dm_move,'dfmdts',-210,/relative
    blades = 'monblade'+string(indgen(21)+1,format='(i02)') 
    bladecenter = fltarr(21)
    
    if ~dryrun then begin
       pobj = obj_new('dm_plot',xtit='MonBlade (\deg)',ytit='Monitor (cnts)',legdcolumns=2,background='white')
       pcol = pobj->getcolor(/list)
       pcol = pcol[0:n_elements(pcol)-2] ;remove the user define color
       psym = pobj->getpsym(/list)
       psym = psym[0:4] & psym = [psym,'filled '+psym]
    endif
    
    for nn=0,20 do begin
        tmp = temporary(datafile)
        dm_move,blades[nn],45
        dm_findpeak,blades[nn],3,0.2,5,detector='monitor',ffit='gauss',datafile=datafile,/sync    
        dm_move,blades[nn],30      
        
        if ~dryrun then begin
           dm_load_macs,datafile,data,info,header=header,deteff=deteff,error=error,all_neg=all_neg,filename=tmpfilename
           if nn eq 0 then files = tmpfilename
           if error then message,'error encountered reading the file.'
           if all_neg then message,'empty data.'
           xind = where(info eq blades[nn],count) & if count eq 0 then continue
           yind = where(info eq 'monitor',count) & if count eq 0 then continue
           x = data[*,xind[0]] & y = data[*,yind[0]] & yerr = sqrt(y)
           dm_gaussfit,y,x=x,params=params
           bladecenter[nn] = params[1]
           xmin = min(x,max=xmax)
           xx   = xmin+findgen(101)/100*(xmax-xmin)
           yy   = params[0]*exp(-(xx-params[1])^2/(2*params[2]^2))
           ymin = min([y-yerr,yy])
           ymax = max([y+yerr,yy])
           if n_elements(xran) eq 0 then begin
              xran = [xmin,xmax]
              yran = [ymin,ymax]
           endif else begin
              xran[0] = (xran[0])<(xmin)
              xran[1] = (xran[1])>(xmax)
              yran[0] = (yran[0])<(ymin)
              yran[1] = (yran[1])>(ymax)
           endelse
           pobj->add_plot,x,y,yerr=yerr,color=pcol[nn mod n_elements(pcol)],psym=psym[nn mod n_elements(psym)],linestyle='no line',legend=dm_to_string(nn+1)
           pobj->add_plot,xx,yy,color=pcol[nn mod n_elements(pcol)],psym='no symbol',linestyle='solid'
           dm_print,'blade '+dm_to_string(nn+1)+' center='+dm_to_string(params[1])
        endif else dm_print,'Fitting monochromator blade '+dm_to_string(nn+1)+' data is not available in dryrun.'
        
        dm_move,'dfmdts',21,/relative
    endfor

    if ~dryrun then begin
       files = [files,tmpfilename]
       files = file_basename(files,strmid(files[0],strpos(files[0],'.',/reverse_search)))
       pobj->add_plot,[45,45],yran+[-0.05,0.05]*(yran[1]-yran[0]),color='gray',linestyle='dashed'
       pobj->setproperty,xran=xran+[-0.05,0.05]*(xran[1]-xran[0]),yran=yran+[-0.05,0.05]*(yran[1]-yran[0]),/legdshowoutline,cornertxt=files[0]+'-'+files[1]
       ;save center in a file
       if keyword_set(savecenter) then begin
          ;open file for writing
          file = directory+dm_define_pointer(/getpathsep)+'monbladecenter_'+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt'
          openw,lun,file,/get_lun,error=openerr
          if openerr ne 0 then message,"Can't write in "+file+'.'
          for nn=0,20 do printf,lun,bladecenter[nn]
          free_lun,lun    
       endif
       ;set blades
       if keyword_set(setblades) then begin
          ;move to the center and reset
          ok = dialog_message('Do you want to move to the fit center and reset monochromator blade positions?',title='Please confirm:',/question,dialog_parent=group_leader,/center)
          if ok eq 'Yes' then begin
             if ~keyword_set(nowait) then dm_sendcom,'wait 6666'
             for nn=0,20 do begin
                 dm_move,blades[nn],bladecenter[nn]
                 dm_sendcom,'device sethard '+blades[nn] + ' 45'
             endfor
          endif
       endif
    endif else begin
       if keyword_set(setblades) then dm_print,'Resetting blade position is not available in dryrun.'
    endelse  
end

;calculate beta1 and beta2 according to Ei and dmbt value
function dm_betas,Ei,dmbt,plottt=plottt,plotdmbt=plotdmbt,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_betas() calculates beta1 and beta2 according to the value of Ei and dmbt.',$
                 'syntax: beta12 = dm_betas(Ei, dmbt, plotdmbt=plotdmbt, plottt=plottt, help=help)',$
                 sep+'Ei:'+tab+tab+'incident energy in unit of meV, a scalar.',$
                 sep+'dmbt:'+tab+tab+'dmbt in unit of mm, a scalar.',$
                 sep+'plotdmbt:'+tab+'if set, plot all beta1 and beta2 vs dmbt.',$
                 sep+'plottt:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, plot all beta1 and beta2 vs two theta.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,[0,0]
    endif
    
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       ei_max = !dm_macsseq_dryrun.ei_max
       ei_min = !dm_macsseq_dryrun.ei_min
    endif

    dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min
    dm_check_parm,dmbt,name='dmbt',/number,/gezero

    taum       = 1.87325
    ki         = sqrt(Ei[0]/2.072)
    tt         = 2*asin(taum/2./ki)/!dtor ;Ei->tt
    tt_all     = [43.,50.,60.,70.,80.,90.,100.,110.,120.]
    dmbt_all   = [5,10,15,20,25.]  
    ttc_mat_b1 = fltarr(n_elements(dmbt_all))
    ttc_mat_b2 = fltarr(n_elements(dmbt_all))  
    beta1_20   = [0.3,0.45,0.7,0.87,0.87,1.10,1.14,1.20,1.28]
    beta2_20   = [1.66,2.08,2.12,2.20,2.23,2.19,2.17,2.07,2.00]
    beta1_10   = [0.4, 0.63, 0.8, 0.95, 1.22, 1.29, 1.45, 1.57, 1.58]
    beta2_10   = [2.07, 2.02, 2.13, 2.18, 2.32, 2.30, 2.21, 2.07, 2.05]
    beta1_5    = [0.7, 1.19, 1.32, 1.30, 1.40, 1.57, 1.70, 1.71, 2.0]
    beta2_5    = [2.28, 2.55, 2.49, 2.69, 2.52, 2.43, 2.44, 2.21, 2.21]
    beta1_15   = [0.43, 0.41, 0.58, 0.77, 0.95, 1.13, 1.21, 1.43, 1.35]
    beta2_15   = [2.09, 2.12, 2.22, 2.31, 2.28, 2.17, 2.14, 2.11, 2.02]
    beta1_25   = [0.1, 0.16, 0.23, 0.58, 0.68, 0.92, 1.08, 1.10, 1.16]
    beta2_25   = [1.68, 2.14, 2.20, 2.39, 2.23, 2.16, 2.17, 2.01, 1.98]
    dbeta1     = [0.05, 0.03, 0.03, 0.02 ,0.04, 0.03, 0.05, 0.03, 0.04]
    dbeta2     = [0.02, 0.01, 0.01, 0.03, 0.03, 0.02, 0.03, 0.02, 0.02]   
    all_betas1 = [ [beta1_5], [beta1_10], [beta1_15],[beta1_20], [beta1_25]]
    all_betas2 = [ [beta2_5], [beta2_10], [beta2_15],[beta2_20], [beta2_25]]
    if keyword_set(plottt) then begin
       x1 = min(tt_all)+findgen(200)/199*(max(tt_all)-min(tt_all))
       plot1 = obj_new('dm_plot',title='beta vs 2 theta',xtit='2 theta (\deg)',ytit='beta1 & beta2 (\deg)',background='white',legdpos=[0.78,0.3],legdfsize=10)
       cols1 = ['green','cyan','orange','magenta','red']
    endif
    if keyword_set(plotdmbt) then begin
       x2 = min(dmbt_all)+findgen(100)/99*(max(dmbt_all)-min(dmbt_all))
       plot2 = obj_new('dm_plot',title='beta vs dmbt',xtit='dmbt (mm)',ytit='beta1 & beta2 (\deg)',background='white',legdpos=[0.86,0.95],legdfsize=10)
       cols2 = ['green','cyan','blue','orange','magenta','brown','red','turquoise','purple']
    endif
    for nn=0,4 do begin
        pbeta1 = poly_fit(tt_all,all_betas1[*,nn],5,MEASURE_ERRORS=dbeta1)
        pbeta2 = poly_fit(tt_all,all_betas2[*,nn],5,MEASURE_ERRORS=dbeta2)
        ttc_mat_b1[nn] = poly(tt,pbeta1)
        ttc_mat_b2[nn] = poly(tt,pbeta2)
        if keyword_set(plottt) then begin
           plot1->add_plot,tt_all,all_betas1[*,nn],legend='dmbt='+dm_to_string(dmbt_all[nn]),psym='circle',linestyle='no line',color=cols1[nn]
           plot1->add_plot,tt_all,all_betas2[*,nn],psym='filled circle',linestyle='no line',color=cols1[nn]
           plot1->add_plot,x1,poly(x1,pbeta1),psym='no symbol',linestyle='solid',color=cols1[nn]
           plot1->add_plot,x1,poly(x1,pbeta2),psym='no symbol',linestyle='solid',color=cols1[nn]
        endif
    endfor     
    pbeta1 = poly_fit(dmbt_all,ttc_mat_b1,3)
    pbeta2 = poly_fit(dmbt_all,ttc_mat_b2,3)
    beta1c = poly(dmbt[0],pbeta1)
    beta2c = poly(dmbt[0],pbeta2)
    
    if keyword_set(plottt)   then plot1->setproperty,xran=[min(tt_all)-8,max(tt_all)+8],yran=[-0.2,3],/legdshowoutline
    if keyword_set(plotdmbt) then begin
       for nn=0,n_elements(tt_all)-1 do begin
           pbeta1 = poly_fit(dmbt_all,reform(all_betas1[nn,*]),3)
           pbeta2 = poly_fit(dmbt_all,reform(all_betas2[nn,*]),3)
           plot2->add_plot,dmbt_all,all_betas1[nn,*],legend=dm_to_string(tt_all[nn])+'\deg',psym='circle',linestyle='no line',color=cols2[nn]
           plot2->add_plot,dmbt_all,all_betas2[nn,*],psym='filled circle',linestyle='no line',color=cols2[nn]
           plot2->add_plot,x2,poly(x2,pbeta1),psym='no symbol',linestyle='solid',color=cols2[nn]
           plot2->add_plot,x2,poly(x2,pbeta2),psym='no symbol',linestyle='solid',color=cols2[nn]
       endfor
       plot2->setproperty,xran=[min(dmbt_all)-2,max(dmbt_all)+4],yran=[-0.2,3],/legdshowoutline
    endif
    
    return,[beta1c,beta2c]
end

;replace ll with kidlim[0], ul with kidlim[1], cn with mean(kidlim) in the string
function dm_calckidney,string,kidlim
    retstr = (n_elements(kidlim) eq 0)
    if retstr then tmp0 = strtrim(string,2) else tmp0 = 'value='+string
    tmp1 = stregex(tmp0,'ll|ul|cn',/fold_case)
    while tmp1 ne -1 do begin
       tmp2 = strlowcase(strmid(tmp0,tmp1,2))
       case tmp2 of
         'll' : tmp0 = strmid(tmp0,0,tmp1)+(retstr?'l_limit':'('+dm_to_string(kidlim[0])+')')+strmid(tmp0,tmp1+2)
         'ul' : tmp0 = strmid(tmp0,0,tmp1)+(retstr?'u_limit':'('+dm_to_string(kidlim[1])+')')+strmid(tmp0,tmp1+2)
         'cn' : tmp0 = strmid(tmp0,0,tmp1)+(retstr?'center':'('+dm_to_string(mean(kidlim))+')')+strmid(tmp0,tmp1+2)
       endcase
       tmp1 = stregex(tmp0,'ll|ul|cn',/fold_case)
    endwhile
    if retstr then return,tmp0
    ok = execute(tmp0,1,1)
    return,value
end

pro dm_check_parm,parm,name=name,number=number,positive=positive,even=even,odd=odd,gezero=gezero,nonzero=nonzero,lowerlimit=lowerlimit,upperlimit=upperlimit,nochar=nochar,string=string,scanname=scanname
    if n_elements(name) eq 0 then name='variable'
    if n_elements(parm) eq 0 then message,level=-1,name+' is undefined.'
    if keyword_set(number) or keyword_set(positive) or keyword_set(even) or keyword_set(odd) or keyword_set(gezero) or keyword_set(nonzero) then begin
       if size(parm,/type) eq 7 then message,level=-1,name+' must be a number.'
       if keyword_set(positive) then begin
          if (total(~finite(parm)) gt 0) or (total(parm le 0) gt 0) then message,level=-1,'invalid '+name+'. '+name+' needs to be a positive number.'
       endif
       if keyword_set(nonzero) then begin
          if (total(~finite(parm)) gt 0) or (total(parm eq 0) gt 0) then message,level=-1,'invalid '+name+'. '+name+' needs to nonzero.'
       endif
       if keyword_set(gezero) then begin
          if (total(~finite(parm)) gt 0) or (total(parm lt 0) gt 0) then message,level=-1,'invalid '+name+'. '+name+' needs to greater than or equal to zero.'
       endif
       if keyword_set(even) then begin
          if (total(~finite(parm)) gt 0) or (total((parm mod 2) ne 0) gt 0) then message,level=-1,'invalid '+name+'. '+name+' needs to be an even number.'
       endif
       if keyword_set(odd) then begin
          if (total(~finite(parm)) gt 0) or (total((parm mod 2) eq 0) gt 0) then message,level=-1,'invalid '+name+'. '+name+' needs to be an odd number.'
       endif
       if n_elements(lowerlimit) ne 0 then begin
          if total(parm lt lowerlimit[0]) gt 0 then message,level=-1,name+' is less than the lower limit of '+dm_to_string(lowerlimit[0])+'.'
       endif
       if n_elements(upperlimit) ne 0 then begin
          if total(parm gt upperlimit[0]) gt 0 then message,level=-1,name+' is greater than the upper limit of '+dm_to_string(upperlimit[0])+'.'
       endif
    endif
    if keyword_set(string) then begin
       if size(parm[0],/type) ne 7 then message,level=-1,name+' must be a string.'
       for i=0,n_elements(nochar)-1 do begin
           if strpos(parm[0],nochar[i]) ne -1 then message,level=-1,name+' has '+('"'+[nochar[i]+'"','empty space'])[nochar[i] eq ' ']+' in it.'
       endfor
    endif
    if keyword_set(scanname) then begin
       if size(parm[0],/type) ne 7 then message,level=-1,name+' must be a string.'  ;first it must be a string
       if dm_macs_sequencer_isdryrun(mesgobj=mesgobj) then begin                    ;then check it only during dry run
          checked = 0b
          if ptr_valid(!dm_macsseq_dryrun.scanname) then begin
             tmp = where(*(!dm_macsseq_dryrun.scanname) eq strupcase(strtrim(parm[0],2)),count)
             checked = (count ne 0)
          endif
          if ~checked then begin
             !dm_macsseq_dryrun.dryrun = 0b
             dm_sendcom,'Scan ListDescr "'+strtrim(parm[0],2)+'"',cmd_error=error,cmd_response=cmd_response,/immediate,/quiet
             !dm_macsseq_dryrun.dryrun = 1b
             if n_elements(cmd_response) ne 0 then begin
                if stregex(cmd_response,'not found in list',/fold_case,/boolean) then message,level=-1,'Scan "'+strtrim(parm[0],2)+'" is not found in the ICE scan list.'
             endif
             if ptr_valid(!dm_macsseq_dryrun.scanname) then *(!dm_macsseq_dryrun.scanname) = [*(!dm_macsseq_dryrun.scanname),strupcase(strtrim(parm[0],2))] $
             else !dm_macsseq_dryrun.scanname = ptr_new(strupcase(strtrim(parm[0],2)))
          endif
       endif
    endif
end

;check existing ICE sequence to see if any wait command longer than specified length exists
;keywords:
;  waitcmds:  return the command line with the long waits
;  ignore:    wait length to be checked, default is 3600 secs
;  cmds:      save the current sequence array
;  spinflip:  return 1 if there's a spin flip command in the sequence, may be undefined
;  spinstate: return the current spinsate at the end of ice server queue
pro dm_check_seqwait,waitcmds=waitcmds,ignore=ignore,cmds=cmds,spinstate=spinstate,spinflip=spinflip
    if n_elements(waitcmds) ne 0 then tmp = temporary(waitcmds)
    if n_elements(ignore) eq 0 then ignore = 3600.0
    tmp_dryrun = !dm_macsseq_dryrun.dryrun
    !dm_macsseq_dryrun.dryrun = 0b
    dm_sendcom,'ListStack',cmd_error=error,cmd_response=cmd_response,/immediate,/quiet
    !dm_macsseq_dryrun.dryrun = tmp_dryrun
    if n_elements(cmd_response) ne 0 then begin
       cmds = strsplit(cmd_response,string([13B, 10B]),/extract)
       for i=0,n_elements(cmds)-1 do begin
           if arg_present(waitcmds) then begin
              tmp = stregex(cmds[i],'wait +([0-9]+)',/fold_case,/extract,/subexpr)
              if strlen(tmp[0]) ne 0 then begin
                 if dm_to_number(tmp[1]) gt ignore then begin
                    if n_elements(waitcmds) eq 0 then waitcmds = cmds[i] else waitcmds = [waitcmds,cmds[i]]
                 endif
              endif
           endif
           if arg_present(spinstate) then begin
              tmp = stregex(cmds[i],'move +flip +([ab])',/fold_case,/extract,/subexpr)
              if strlen(tmp[0]) ne 0 then begin
                 spinstate = (['NSF','SF'])[strmatch(tmp[1],'b',/fold_case)]
              endif
           endif
           if arg_present(spinflip) then begin
              if stregex(cmds[i],'^ *iftalk.*spinflip.*',/boolean,/fold_case) then spinflip = 1b
           endif
       endfor
    endif
    if arg_present(spinstate) and (n_elements(spinstate) eq 0) then begin
       spinstate = 'NSF'
       !dm_macsseq_dryrun.dryrun = 0b
       dm_sendcom,'print flip',cmd_error=error,cmd_response=cmd_response,/immediate,/quiet
       !dm_macsseq_dryrun.dryrun = tmp_dryrun
       if n_elements(cmd_response) ne 0 then begin
          tmpcmds = strsplit(cmd_response,string([13B, 10B]),/extract)
          for i=0,n_elements(tmpcmds)-1 do begin
              tmp = stregex(tmpcmds[i],'flip: +([ab]+)',/fold_case,/extract,/subexpr)
              if strlen(tmp[0]) ne 0 then begin
                 spinstate = (['NSF','SF'])[strmatch(tmp[1],'b',/fold_case)]
              endif
          endfor
       endif
    endif
end

;check IDL syntax line by line to figure out where error occurs
pro dm_check_syntax,errmsg=errmsg,name=name,sequence=sequence,self=self
    errmsg = !error_state.msg
    tmp0 = ''
    for i_xyyzzz=0L,n_elements(sequence)-1 do begin
        tmp = dm_strip_comments(sequence[i_xyyzzz])
        if strlen(tmp) eq 0 then continue else tmp0 = tmp0+tmp
        if strmid(tmp,0,/reverse_offset) eq '$' then begin
           tmp0 = strmid(tmp0,0,strlen(tmp0)-1)
           continue
        endif
        idlword = '^(if[\( ]+|end|goto +|case +|for +|while[\( ]+|repeat[\( ]+|switch +|continue$|break$)'
        if stregex(tmp0,'^for +',/fold_case,/boolean) then begin
           tmp = stregex(tmp0,'^for (.*),',/subexpr,/extract)
           ok = execute(tmp[1],1,1)
           if ~ok then break
           tmp0 = ''
           continue
        endif
        if stregex(tmp0,idlword,/fold_case,/boolean) then begin
           tmp0 = ''
           continue
        endif
        ok = execute(tmp0,1,1)
        if ~ok then break
        tmp0 = ''
    endfor
    if n_elements(ok) ne 0 then begin
       if ~ok then errmsg = [!error_state.msg,'','line '+dm_to_string(i_xyyzzz+1)+': '+tmp0]
    endif
    if n_elements(name) ne 0 then errmsg[0] = strmid(errmsg[0],0,strlen(errmsg[0])-1)+' in '+strtrim(name[0],2)+'.'
end

;sends the comment command
pro dm_comment,comment,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_comment appends a comment to the end of the current data file. It can also be used as a reminder in the sequence.',$
                 'syntax: dm_comment, comment, help=help',$
                 sep+'comment:'+tab+'a string.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,comment,name='comment',/string
    for i=0,n_elements(comment)-1 do dm_sendcom,'comment "'+strtrim(comment[i],2)+'"'
end

;returns all valid MACS devices
function dm_def_dev,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_def_dev() returns the list of device names of MACS.','syntax: device = dm_def_dev(help=help)']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
    endif
    
    t_axis          = ['A1','A2','A3','A4','A5']
    all_collimators = ['ACOLMON','ACOLMONMOTOR','BCOLMON','BCOLMONMOTOR','COLMON']
    analyzer_theta  = ['ANALYZERTHETA20','ANALYZERTHETA01','ANALYZERTHETA02','ANALYZERTHETA03','ANALYZERTHETA04',$
                       'ANALYZERTHETA05','ANALYZERTHETA06','ANALYZERTHETA07','ANALYZERTHETA08','ANALYZERTHETA09',$
                       'ANALYZERTHETA10','ANALYZERTHETA11','ANALYZERTHETA12','ANALYZERTHETA13','ANALYZERTHETA14',$
                       'ANALYZERTHETA15','ANALYZERTHETA16','ANALYZERTHETA17','ANALYZERTHETA18','ANALYZERTHETA19']
    analyzer_ef     = ['EF','EF20','EF01','EF02','EF03','EF04','EF05','EF06','EF07','EF08','EF09','EF10','EF11', $
                       'EF12','EF13','EF14','EF15','EF16','EF17','EF18','EF19']            
    det_sys_mbt     = ['BASESAMPLETHETA','KIDNEY','MBTSLIDE']
    mon_filters     = ['BEFILMON','BEFILMONMOTOR','MGFILMON','MGFILMONMOTOR','PGFILMON','PGFILMONMOTOR','CFXBE','CFXHOPG','MCFX']
    misc            = ['PTAI','VBAH','VBAV','LAKESHORE331','TEMP','MAGFIELD','EI']
    res_sp          = ['H','K','L','HKL']
    sample_table    = ['SMPLX','SMPLY','SMPLZ','SMPLLTILT','SMPLUTILT','BETA1','BETA2','DMBT']
    dfm_motors      = ['DFMDTS','DFMAXIS','FOCUS1','FOCUS2','FOCUS','FREF','L1','L0','MONROT','MONTRANS']
    dfm_blades      = ['MONBLADE01','MONBLADE02','MONBLADE03','MONBLADE04','MONBLADE05','MONBLADE06',$
                       'MONBLADE07','MONBLADE08','MONBLADE09','MONBLADE10','MONBLADE11','MONBLADE12',$
                       'MONBLADE13','MONBLADE14','MONBLADE15','MONBLADE16','MONBLADE17','MONBLADE18',$
                       'MONBLADE19','MONBLADE20','MONBLADE21']
    RETURN, [t_axis,all_collimators,analyzer_theta,analyzer_ef,det_sys_mbt,mon_filters,misc,res_sp,sample_table,dfm_motors,dfm_blades] 
end     

;read a device posiiton
function dm_dev_pos,device,error=error,hardware=hardware,number=number,help=help,_ref_extra=extra,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_dev_pos() returns the position of a device.','syntax: position = dm_dev_pos(device, number=number, error=error, help=help)',$
                 sep+'device:'+(keyword_set(printhelp)?tab:tab+tab)+'a string of the name of the device.',$
                 sep+'hardware:'+tab+'if set, returns the hardware angle, otherwise return the software angle.',$
                 sep+'number:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, dm_dev_pos() returns a number, otherwise returns a string.',$
                 sep+'error:'+(keyword_set(printhelp)?tab:tab+tab)+'returns the error status.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,!values.f_nan
    endif

    error = 1b
    dm_check_parm,device,name='device',/string
    if n_elements(extra) ne 0  then message,'syntax error, use position = dm_dev_pos(/help) for help.'
    dev = strcompress(strupcase(device[0]),/REMOVE_ALL)
    valid_device = where(dm_def_dev() eq dev,count)
    if count eq 0 then message,'invalid device.'

    ;Build the command
    command = 'DEVICE '+(['READ','GETHARD'])[keyword_set(hardware)]+' '+dev
    dm_sendcom,command,cmd_error=error,cmd_response=cmd_response,/immediate
    if n_elements(cmd_response) eq 0 then cmd_response = ''
    if keyword_set(number) then cmd_response = dm_to_number(cmd_response)
    return,cmd_response
end  

;******************************
;*DFMDTS as a function of blade11
;*
;* Input: blade11
;* A[0]= DFMDTS_0
;* A[1]= fref
;* A[2]= monrot_0
;******************************
pro dm_dfmdts_blade11,blade11,A,F
    F = A[0]-A[1]/tan(2*!dtor*(blade11-A[2]))
end

;******************************
;*DFMDTS as a function of mbtslide
;*
;* Input: mbtslide
;* A[0]= DFMDTS_0 (fixed with dfmdts_blade11 value)
;* A[1]= fref     (fixed with dfmdts_blade11 value)
;* A[2]= mbtpivot
;* A[3]= mbtslide_0
;* A[4]= phi
;******************************
pro dm_dfmdts_mbtslide,mbtslide,A,F
    upval = (-1*A[2]*sin(!dtor*A[4])-mbtslide+A[3])*A[1]*cos(!dtor*A[4])
    loval = A[2]+(-1*A[2]*sin(!dtor*A[4])-mbtslide+A[3])*sin(!dtor*A[4])
    F     = A[0]+upval/loval
end

;scan a device in a certain range with a step size
pro dm_findpeak,device,range,step_size,time,detector=detector,ffit=ffit,help=help,sync=sync,datastr=datastr,datafile=datafile,_ref_Extra=extra,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_findpeak scans a device over a given range to find the device position at which maximum intensity is reached.',$
                 'syntax: dm_findpeak, device, range, step_size, time, detector=detector, ffit=ffit, sync=sync, datastr=datastr, datafile=datafile, help=help',$
                 sep+'device:'+(keyword_set(printhelp)?tab:tab+tab)+'a string of the name of the device.',$
                 sep+'range:'+(keyword_set(printhelp)?tab:tab+tab)+'device range. The scan is centered at current device position.',$
                 sep+'step_size:'+tab+'step size in the scan.',$
                 sep+'time:'+tab+tab+'time per step. If negative, the reference is the MONITOR.',$
                 sep+'detector:'+(keyword_set(printhelp)?tab:tab+tab)+"detector to scan. Default is 'SPEC'.",$
                 sep+'ffit:'+tab+tab+"fitting function. Default is 'Gauss'. Use an empty string to skip fitting.",$
                 sep+'sync:'+tab+tab+'if set, the command is run synchronously.',$
                 sep+'datastr:'+(keyword_set(printhelp)?tab:tab+tab)+'returns a structure of the scan data, sync keyword needs to be set.',$
                 sep+'datafile:'+(keyword_set(printhelp)?tab:tab+tab)+'returns the raw scan data in the form of a string array, sync keyword needs to be set.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,device,name='device',/string
    dm_check_parm,range,name='range',/number,/nonzero
    dm_check_parm,step_size,name='step_size',/number,/nonzero
    dm_check_parm,time,name='time',/number,/nonzero
    if n_elements(extra) ne 0    then message,'syntax error, use dm_findpeak,/help for help.'
    if n_elements(ffit) eq 0     then ffit = 'Gauss'
    if size(ffit,/type) ne 7     then message,'ffit must be a string.'
    if n_elements(detector) eq 0 then detector = 'SPEC'
    if size(detector,/type) ne 7 then message,'detector must be a string.'

    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj)

    if time lt 0 then begin
       reference = 'MONITOR'
       time = (-1)*time
    endif else reference = 'TIME'
    
    command = 'FINDPEAK '+device[0]+' '+dm_to_string(range)+' '+dm_to_string(step_size)+' '+reference+' '+dm_to_string(time)+' '+detector+((strlen(ffit) ne 0)?(' -func '+ffit):'')    
    
    if keyword_set(sync) and (arg_present(datastr) or arg_present(datafile)) then begin
       dm_sendcom,command,cmd_error=error,cmd_response=cmd_response,/sync,data=datafile
       ind = where(stregex(datafile,'#Columns',/boolean,/fold_case),count)
       if count gt 0 then begin ;get rid of trailing lines starting with #
          while (n_elements(datafile) gt ind[0]+1) and stregex(datafile[n_elements(datafile)-1],'^[ '+string(9b)+']*#',/boolean) do datafile = datafile[0:(n_elements(datafile)-2)]
       endif
       if arg_present(datastr) then begin      
          if dryrun then dm_print,'dataStr is not available during dryrun.' $
          else err = dave_parse_ice_macs(datafile,datastr)
       endif            
    endif else $
       dm_sendcom,command,cmd_error=error,cmd_response=cmd_response
end

pro dm_fit_macs_alignment,angles,mbtslide,dfmdts,blade11,colr0=colr0,filename=filename,group_leader=group_leader,ice=ice,nowait=nowait,plotobj=plotobj,savefile=savefile
    if ~obj_valid(plotobj) then plotobj = obj_new('dm_plot',background='white')
    if n_elements(colr0) eq 0 then colr = plotobj->getcolor(/list,linecolor=colr0)
    date_str = strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)  ;date string
    a = fltarr(6)+!values.f_nan
    default = [-2.535,778.629,41.6554,36.972,81.8089,0.325]
    namestr = ['DTS','fref','mbtpivot','mbtslide','a2ref','monrot']
    info    = ['The monrot and DTS values are listed in the MACS Microsoft Access document DFM_MACS,','which is located at the DFM computer in C:\LabVIEW_Work directory.','They are listed at row 23 and 29 as ROTATION and DTS in the INDEXER_ASSIGNMENTS table.']
    if keyword_set(ice) then ok_ice = 'Yes' else ok_ice  = dialog_message('Do you want to allow communication to ICE to read fref, a2ref, mbtpivot, and mbtslide values?',title='Talk to ICE?',/question,dialog_parent=group_leader,/center)
    if ok_ice eq 'Yes' then begin
       tmp  = dm_dialog_input(['monrot:','DTS:'],/float,dialog_parent=group_leader,info=info)
       if keyword_set(ice) then begin
          if finite(tmp[0],/nan) then message,'invalid monrot value.'
          if finite(tmp[1],/nan) then message,'invalid DTS value.'
       endif
       a[[5,0]] = tmp
       a[1] = dm_dev_pos('fref',/number)
       a[3] = dm_dev_pos('mbtslide',/number)
       dm_sendcom,'geta2ref',/sync,cmd_respons=a2ref_str & print,a2ref_str
       dm_sendcom,'getmbtpivot',/sync,cmd_respons=mbtpivot_str & print,mbtpivot_str
       a2ref_str_ext    = strsplit(a2ref_str,/extract)
       mbtpivot_str_ext = strsplit(mbtpivot_str,/extract)
       a2ref            = dm_to_number(a2ref_str_ext[2])
       mbtpivot         = dm_to_number(mbtpivot_str_ext[2])
       a[2]             = mbtpivot
       a[4]             = 90-a2ref
    endif else begin
       a  = dm_dialog_input(namestr+':',/float,dialog_parent=group_leader,default=default,info=info)
    endelse
    ind = where(finite(a,/nan),cnt)
    if cnt ne 0 then begin
       if cnt eq 6 then mesg = 'The entered values are invalid. Default values of ' else begin
          mesg = 'The entered '
          for i=0,cnt-1 do mesg = mesg+([', ',''])[i eq 0]+(['','and '])[i ne 0 and i eq cnt-1]+namestr[ind[i]]
          mesg = mesg+([' values are invalid',' value is invalid'])[cnt eq 1]+'. Default value'+(['s of ',' of '])[cnt eq 1]
       endelse
       for i=0,cnt-1 do mesg = mesg+([', ',''])[i eq 0]+(['','and '])[i ne 0 and i eq cnt-1]+namestr[ind[i]]+'='+dm_to_string(default[ind[i]])
       mesg = mesg+([' are used',' is used'])[cnt eq 1]+' instead.'
       ok = dialog_message(mesg,dialog_parent=group_leader,/center)
       default[4] = 90-default[4]
       a[ind] = default[ind]
    endif
    a_old         = a
    ind           = sort(angles) & n_ang = n_elements(angles)
    angles        = angles[ind]
    mbtslide      = mbtslide[ind]
    dfmdts        = dfmdts[ind]
    blade11       = blade11[ind]
    weights       = fltarr(n_ang)+1.0
    A_blade11     = [a[0],a[1],a[5]]
    yfit_blade11  = curvefit(blade11,dfmdts,weights,A_blade11,SIGMA_blade11,FUNCTION_name='dm_dfmdts_blade11', /NODERIVATIVE)
    A_mbtslide    = [A_blade11[0],A_blade11[1],a[2],a[3],a[4]]
    yfit_mbtslide = curvefit(mbtslide,dfmdts,weights,A_mbtslide,SIGMA_mbtslide,FUNCTION_name='dm_dfmdts_mbtslide', /NODERIVATIVE,FITA=[0,0,1,1,1])
    plotobj->setproperty,xdat=blade11,ydat=dfmdts,ytit='DFMDTS (mm)',xtit='BLADE11 (\deg) or MBTSLIDE (cm)',/nodraw,xran=[min([blade11,mbtslide]),max([blade11,mbtslide])]+[-5,5],$
          yran=[min(dfmdts),max(dfmdts)]+[-60,60],title='BLADE11 & MBTSLIDE vs DFMDTS',psym='circle',hidelegend=0,linestyle='no line',color=colr0,cornertxt=filename,$
          legend='BLADE11',legdpos=[0.45,0.93],/legdshowoutline
    plotobj->add_plot,blade11,yfit_blade11,color=colr0,linestyle='solid'
    plotobj->add_plot,mbtslide,dfmdts,color=(['red','cyan'])[total(abs([255,0,0]-colr0)) eq 0],linestyle='no line',psym='filled circle',legend='MBTSLIDE'
    plotobj->add_plot,mbtslide,yfit_mbtslide,color=(['red','cyan'])[total(abs([255,0,0]-colr0)) eq 0],linestyle='solid'
    plotobj->add_text,'DFMDTS!d0!n = '+dm_to_string(A_blade11[0])+'!Cfref          !d !n= '+dm_to_string(A_blade11[1])+'!Cmonrot!d0!n   = '+dm_to_string(A_blade11[2])+$
          '!Cmbtpivot !d !n= '+dm_to_string(A_mbtslide[2])+'!Cmbtslide!d0!n = '+dm_to_string(A_mbtslide[3])+'!Cphi          = '+dm_to_string(90-A_mbtslide[4]),0.05,0.4,fontsize=10
    plotobj->draw  
    a_new = [A_mbtslide[0:4],A_blade11[2]]
    new_par = a_old
    new_par[0] = a_old[0]+a_new[0]
    new_par[1] = a_new[1]
    new_par[2] = a_new[2]
    new_par[3] = a_old[3]-a_new[3]
    new_par[4] = a_new[4]
    new_par[5] = a_old[5]+a_new[5]
    if ok_ice eq 'Yes' then begin
       ok = dialog_message('Do you want to reset the new instrument parameters in ICE?',title='Please confirm:',/question,dialog_parent=group_leader,/center)
       if ok eq 'Yes' then begin
          setfref     = 'move fref '+dm_to_string(new_par[1]) 
          setmbtpivot = 'setmbtpivot '+dm_to_string(new_par[2])
          setmbtslide = 'device set mbtslide '+dm_to_string(new_par[3])
          seta2ref    = 'seta2ref '+dm_to_string(90-new_par[4])
          if ~keyword_set(nowait) then dm_sendcom,'wait 6666'
          dm_sendcom,setmbtpivot
          dm_sendcom,setmbtslide
          dm_sendcom,seta2ref
          dm_sendcom,setfref
          ok = dialog_message(dialog_parent=group_leader,['Please enter the new monrot='+dm_to_string(new_par[5])+' and DTS='+dm_to_string(new_par[0])+' values in the MACS Microsoft Access document DFM_MACS,',$
                  'which is located at the DFM computer in C:\LabVIEW_Work directory.','They are listed at row 23 and 29 as ROTATION and DTS in the INDEXER_ASSIGNMENTS table.',$
                  'And set fref='+dm_to_string(new_par[1])+' in SYS_PARAMETERS table, row 12.'],/info,/center)
       endif
    endif else ok = dialog_message('The fit parameters may need to be adjusted for actual instrument configuration.',dialog_parent=group_leader,/center)
    if n_elements(savefile) eq 0 then begin
       ok_save = dialog_message('Do you want to save the fit results?',title='Please confirm:',/question,dialog_parent=group_leader,/center)
       if ok_save eq 'Yes' then begin
          savefile = dm_choose_file('txt',/write,dialog_parent=group_leader,title='Please select a file name for saving the results',file='macs_alignment_'+date_str+'.txt')
          if strlen(savefile) eq 0 then return
       endif else return
    endif
    openw,lun,savefile,/get_lun,error=openerr
    if openerr ne 0 then message,"Can't write in "+savefile+'.'
    tab = string(9b)
    printf,lun,'       twotheta        mbtslidepos     dfmdtspos      monblade11pos'
    for i=0,n_ang-1 do printf,lun,angles[i],' ',mbtslide[i],' ',dfmdts[i],' ',blade11[i]
    printf,lun,'  '
    printf,lun,'DFMDTS_0   = '+dm_to_string(a_new[0])
    printf,lun,'fref       = '+dm_to_string(a_new[1])
    printf,lun,'mbtpivot   = '+dm_to_string(a_new[2])
    printf,lun,'mbtslide_0 = '+dm_to_string(a_new[3])
    printf,lun,'phi        = '+dm_to_string(90.0-a_new[4])
    printf,lun,'monrot     = '+dm_to_string(a_new[5])
    printf,lun,' '
    printf,lun,'DFMDTS old value      = '+dm_to_string(a_old[0])+tab+'    new value = '+dm_to_string(new_par[0])
    printf,lun,'FREF old value        = '+dm_to_string(a_old[1])+tab+'    new value = '+dm_to_string(new_par[1])
    printf,lun,'MBTPIVOT old value    = '+dm_to_string(a_old[2])+tab+'    new value = '+dm_to_string(new_par[2])
    printf,lun,'MBTSLIDE old value    = '+dm_to_string(a_old[3])+tab+'    new value = '+dm_to_string(new_par[3])
    printf,lun,'A2REF old value       = '+dm_to_string(90.-a_old[4])+tab+'    new value = '+dm_to_string(90.-new_par[4])
    printf,lun,'MONROT_ZERO old value = '+dm_to_string(a_old[5])+tab+'    new value = '+dm_to_string(new_par[5])   
    free_lun,lun
end

;calculate focus value for Ei
function dm_get_focus,Ei,dfmdts=dfmdts
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then fref = !dm_macsseq_dryrun.fref $
    else fref = dm_dev_pos('fref',/number)
    theta = asin(sqrt(81.8042/Ei)/2/3.35416)
    L0 = 6582.-fref/TAN(2*theta)
    L1 = 900.+fref/SIN(2*theta)
    if arg_present(dfmdts) then dfmdts = - fref/TAN(2*theta)
    return,round(2*L0*L1*SIN(theta)/(L0+L1))
end

;parse the kidney hardware limit from the ICE error message
function dm_get_error_kidlimit,errmsg
    if strlen(errmsg) eq 0 then return,!values.f_nan
    tmpmsg = strsplit(errmsg,'Motor_Move_HARDLIMIT_ERROR:',/regex,/fold_case,/extract,count=ntmp)
    tmp = stregex(tmpmsg[ntmp-1],'KidneyMotor hit a hard limit at +([+.0-9-]+)',/subexpr,/fold_case,length=len)
    if len[1] gt 0 then return,float(strmid(tmpmsg[ntmp-1],tmp[1],len[1])) $   
    else return,!values.f_nan
end

pro dm_get_help,cmd,output=output
    if n_elements(cmd) eq 0 then return
    case strlowcase(cmd) of 
         'dm_a3scan':                dm_a3scan,/help,output=output
         'dm_align_a5':              dm_align_a5,/help,output=output
         'dm_align_monochromator':   dm_align_monochromator,/help,output=output
         'dm_betas':                 ok = dm_betas(/help,output=output)
         'dm_comment':               dm_comment,/help,output=output
         'dm_def_dev':               ok = dm_def_dev(/help,output=output)
         'dm_dev_pos':               ok = dm_dev_pos(/help,output=output)
         'dm_findpeak':              dm_findpeak,/help,output=output
         'dm_geta3offset':           ok = dm_geta3offset(/help,output=output)
         'dm_getkidneyrange':        ok = dm_getkidneyrange(/help,output=output)
         'dm_getlastscanname':       ok = dm_getlastscanname(/help,output=output)
         'dm_gobetas':               dm_gobetas,/help,output=output
         'dm_help':                  dm_help,/help,output=output
         'dm_kidscan':               dm_kidscan,/help,output=output
         'dm_macs_alignment':        dm_macs_alignment,/help,output=output
         'dm_macs_kidneylimitscan':  dm_macs_kidneylimitscan,/help,output=output
         'dm_move':                  dm_move,/help,output=output
         'dm_optimizekidstep':       ok = dm_optimizekidstep(/help,output=output)
         'dm_plot_macs':             dm_plot_macs,/help,output=output
         'dm_print':                 dm_print,/help,output=output
         'dm_runscan':               dm_runscan,/help,output=output
         'dm_runtab':                dm_runtab,/help,output=output
         'dm_sendcom':               dm_sendcom,/help,output=output
         'dm_setslits':              dm_setslits,/help,output=output
         'dm_spinflip':              dm_spinflip,/help,output=output
         'dm_tempstaleness':         dm_tempstaleness,/help,output=output
         'dm_to_string':             output = 'dm_to_string() converts a variable to a string.'
         'dm_wait':                  dm_wait,/help,output=output
         else:
    endcase
end

;get the offset angle between a3 and basesampletheta
function dm_geta3offset,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_geta3offset() returns the offset angle among A3, basesampletheta, and kidney.',sep+'A3 = basesampletheta + kidney + offset',$
                 'syntax: offset = dm_geta3offset(help=help)']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,!values.f_nan
    endif   
    a3 = dm_dev_pos('a3',/number)
    bt = dm_dev_pos('basesampletheta',/number)
    kd = dm_dev_pos('kidney',/number)
    return,a3-bt-kd
end


;get the kidney angle range for a specified Ei
function dm_getkidneyrange,Ei,center=center,eimat=eimat,k_min=k_min,k_max=k_max,offset=offset,plot=plot,help=help,notcheckeimax=notcheckeimax,notcheckeimin=notcheckeimin,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_getkidneyrange() returns the kidney software angle range for a specified Ei, 1.5 degrees within the limit. Returns a two-element array or [2,n_Ei] array.',$
                 'syntax: kran = dm_getkidneyrange(Ei, center=center, eimat=eimat, k_min=k_min, k_max=k_max, plot=plot, help=help)',$
                 sep+'Ei:'+(keyword_set(printhelp)?tab+tab:tab)+'Ei at which the kidney range to be returned, a scalar or array, in unit meV.',$
                 sep+'center:'+tab+'if set, returns the center position of the kidney range.',$
                 sep+'eimat:'+tab+"returns an array of Ei's that have been measured at.",$
                 sep+'k_min:'+tab+'returns the lower limit of kidney software angle for eimat, an array.',$
                 sep+'k_max:'+tab+'returns the upper limit of kidney software angle for eimat, an array.',$
                 sep+'offset:'+tab+'hardware offset angle, it corresponds to the hardware angle when software angle is 0. If absent, use the default offset.',$
                 sep+'plot:'+(keyword_set(printhelp)?tab+tab:tab)+'if set, plots the kidney limits vs. eimat.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,0
    endif   
    
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       if ~keyword_set(notcheckeimax) then ei_max = !dm_macsseq_dryrun.ei_max
       if ~keyword_set(notcheckeimin) then ei_min = !dm_macsseq_dryrun.ei_min
    endif

    dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min
 
    ok = dm_check_macskidney(fltarr(n_elements(Ei)),Ei,limits=limits,llim=llim,ulim=ulim,offset=offset,/spline)
    eimat = limits[*,0]
    k_min = limits[*,1]-offset
    k_max = limits[*,2]-offset
    
    if keyword_set(plot) then begin
       tmp = obj_new('dm_plot',eimat,k_min,legend='lower limit',color='green',psym='filled circle',xtit='E!di!n (meV)',ytit='Kidney Limits (\deg)',background='white',legdpos=[0.6,0.9],legdfsize=10)
       tmp->add_plot,eimat,k_max,legend='upper limit',color='red',psym='filled circle'
       tmp->setproperty,xran=[min(eimat)-0.5,max(eimat)+0.5],yran=[min(k_min)-5,max(k_max)+5],/legdshowoutline
    endif
    
    if keyword_set(center) then return,(llim+ulim)/2.0 $
    else return,reform(transpose([[llim+1.5],[ulim-1.5]]))
end

;get last scan name
function dm_getlastscanname,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_getlastscanname() returns the file name of the last scan.',$
                 'syntax: filename = dm_getlastscanname(help=help)']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,''
    endif   
  
    filename = ''
    dm_sendcom,'',data=data
    ind = where(stregex(data,'#Filename',/boolean,/fold_case),count)
    if count ne 0 then begin
       tmp = stregex(data[ind[0]],'#Filename +(.+)$',/fold_case,/subexpr,length=len)
       if tmp[1] ne -1 then filename = strtrim(strmid(data[ind[0]],tmp[1],len[1]),2)
       if strlen(filename) ne 0 then filename = filename+'.ng0'
    endif
    return,filename
end 

;set beta1 and beta2 according to Ei and dmbt
pro dm_gobetas,Ei,dmbt,plotdmbt=plotdmbt,plottt=plottt,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_gobetas sets beta1 and beta2 according to the value of Ei and dmbt.',$
                 'syntax: dm_gobetas, Ei, dmbt, plotdmbt=plotdmbt, plottt=plottt, help=help',$
                 sep+'Ei:'+(keyword_set(printhelp)?tab+tab:tab)+'incident energy in unit of meV, a scalar.',$
                 sep+'dmbt:'+(keyword_set(printhelp)?tab+tab:tab)+'dmbt in unit of mm, a scalar.',$
                 sep+'plotdmbt:'+tab+'if set, plot all beta1 and beta2 vs dmbt only. No command will be sent to ICE.',$
                 sep+'plottt:'+tab+'if set, plot all beta1 and beta2 vs two theta only. No command will be sent to ICE.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    if keyword_set(plottt) or keyword_set(plotdmbt) then begin
       ok = dm_betas(5,20,plottt=plottt,plotdmbt=plotdmbt)
       return
    endif
    
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       ei_max = !dm_macsseq_dryrun.ei_max
       ei_min = !dm_macsseq_dryrun.ei_min
    endif

    dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min
    dm_check_parm,dmbt,name='dmbt',/number,/gezero
    
    beta12 = dm_betas(Ei[0],dmbt[0])
    dm_sendcom,'move beta1 '+dm_to_string(beta12[0],res=3)+' beta2 '+dm_to_string(beta12[1],res=3)
end

pro dm_help,expression,expression1,expression2,expression3,expression4,expression5,expression6,expression7,expression8,expression9,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_help shows the help info of the expression in the sequencer status box.',$
                 'syntax: dm_help, expression, help=help',$
                 sep+'expression:'+tab+'an IDL expression or variable.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    if n_elements(expression) eq 0 then return
    help,expression,output=strs,level=-1
    if n_elements(expression1) ne 0 then begin
       help,expression1,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression2) ne 0 then begin
       help,expression2,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression3) ne 0 then begin
       help,expression3,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression4) ne 0 then begin
       help,expression4,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression5) ne 0 then begin
       help,expression5,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression6) ne 0 then begin
       help,expression6,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression7) ne 0 then begin
       help,expression7,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression8) ne 0 then begin
       help,expression8,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    if n_elements(expression9) ne 0 then begin
       help,expression9,output=strs1,level=-1 & strs = [strs,strs1]
    endif
    
    help,/traceback,output=output
    if total(stregex(output,'DM_MACS_SEQUENCER_EVENT',/fold_case,/boolean)) gt 0 then begin
       tmp = obj_valid()
       ind = where(obj_isa(tmp,'DM_MACS_SEQUENCER'))
       tmp[ind[0]]->my_widget_control,'statusbox',get_value=txtstr
       if total(strlen(txtstr)) eq 0 then outstr = strs $
       else outstr = [txtstr,strs]
       tmp[ind[0]]->my_widget_control,'statusbox',set_value=outstr,set_text_top_line=0>(n_elements(outstr)-4)
    endif else if total(stregex(output,'DCS_MSLICE_EVENT',/fold_case,/boolean)) gt 0 then begin
       tmp = obj_valid()
       ind = where(obj_isa(tmp,'DCS_MSLICE'),count)
       for i=0,count-1 do begin
           tmp[ind[i]]->getproperty,ftype=ftype
           if ftype ne 'mslicedata' then break
       endfor
       tmp[ind[i]]->calculator_insert,';'+strs
    endif else print,strs
end

;a simple idl syntax list
pro dm_idl_help,group=group,font=font
    tab  = string(9b) & sep = '      '
    text = ['ASSIGNMENT:',sep+'Syntax:'+tab+'variable = expression',sep+'Examples:'+tab+'x = 7',tab+tab+'num = [1,2,3,4]',tab+tab+'y = x^2+x+1','',$
            'IF statement:',sep+'Syntax:'+tab+'if expression then statement',tab+tab+'if expression then statement1 else statement2',sep+'Examples:'+tab+'if x lt 0 then x = x+1',$
            tab+tab+'if x ge 0 then y=2 else y=3',tab+tab+'if x gt 0 then begin',tab+tab+sep+'a=1 & b=2',tab+tab+'endif else begin',tab+tab+sep+'a=2 & b=1',tab+tab+'endelse','',$
            'CASE statement:',sep+'Syntax:'+tab+'case expression of',tab+tab+sep+'expression 1:'+tab+'statement',tab+tab+sep+'expression 2:'+tab+'statement',tab+tab+sep+'expression 3:'+tab+'begin',$
            tab+tab+tab+tab+'statements',tab+tab+tab+tab+'end',tab+tab+sep+'else:'+tab+tab+'statement',tab+tab+'endcase',sep+'Examples:'+tab+'case color of',$
            tab+tab+sep+"'red':"+tab+tab+'rgb = [255b,0b,0b]',tab+tab+sep+"'green':"+tab+'rgb = [0b,255b,0b]',tab+tab+sep+"'blue':"+tab+tab+'rgb = [0b,0b,255b]',tab+tab+sep+'...',$
            tab+tab+sep+'else:'+tab+tab+'rgb = [0b,0b,0b]',tab+tab+'endcase','',$
            'SWITCH statement:',sep+'Syntax:'+tab+'switch expression of',tab+tab+sep+'expression 1:',tab+tab+sep+'expression 2:'+tab+'statement',tab+tab+sep+'expression 3:'+tab+'begin',$
            tab+tab+tab+tab+'statements',tab+tab+tab+tab+'end',tab+tab+sep+'else:'+tab+tab+'statement',tab+tab+'endswitch',sep+'Examples:'+tab+'switch month of',$
            tab+tab+sep+['1:','3:','5:','7:','8:','10:','12:'+tab+'begin'],tab+tab+tab+'num_days = 31',tab+tab+tab+['break','end'],$
            tab+tab+sep+['4:','6:','9:','11:'+tab+'begin'],tab+tab+tab+'num_days = 30',tab+tab+tab+['break','end'],tab+tab+sep+'2:'+tab+'begin',$
            tab+tab+tab+'num_days = 28',tab+tab+tab+'if (year mod 4) eq 0 then begin',tab+tab+tab+sep+'if ((year mod 100) ne 0) or ((year mod 400) eq 0) then num_days = 29',$
            tab+tab+tab+'endif',tab+tab+tab+['break','end'],tab+tab+sep+'else:'+tab+"message, 'Invalid month.'",tab+tab+'endwitch','',$
            'FOR loops:',sep+'Syntax:'+tab+'for variable = init, limit, step do statement',sep+'Examples:'+tab+'for i=0,9 do print,i^2',tab+tab+'for x=1.0,0.1,-0.1 do print, cos(asin(x))',$
            tab+tab+'for i=0L, n_elements(y)-1 do begin',tab+tab+sep+'x[i] = sin(i*!dtor)',tab+tab+sep+'y[i] = sqrt(abs(x[i]))',tab+tab+'endfor','',$
            'WHILE loops:',sep+'Syntax:'+tab+'while expression do statement',sep+'Examples:'+tab+'while x gt 0 do x=x-1',tab+tab+'while ~eof(lun) do begin',$
            tab+tab+sep+'readf,lun,txt',tab+tab+sep+'print,txt',tab+tab+'endwhile','',$
            'REPEAT loops:',sep+'Syntax:'+tab+'repeat statement until expression',sep+'Examples:'+tab+'repeat x=x-1 until x le 0 ','',tab+tab+["line = ''","print, 'Enter text(done to quit):'"],$
            tab+tab+'repeat begin',tab+tab+sep+['read, line','if n_elements(text) eq 0 then text=line else text=[text,line]'],tab+tab+"endrep until strlowcase(line) eq 'done'",'',$
            'OPERATORS:',sep+'Parentheses:'+tab+tab+'()'+tab+'Group expressions or enclose function parameter lists',sep+'Square Brackets:'+tab+'[]'+tab+'Create arrays or enclose array subscripts',$
            sep+'Mathematical Operators:'+tab+'='+tab+'Assignment',tab+tab+tab+['+'+tab+'Addition','-'+tab+'Subtraction and negation','*'+tab+'Multiplication','/'+tab+'Division','^'+tab+'Exponentiation',$
            'MOD'+tab+'Modulus','++'+tab+'Increment by one','--'+tab+'Decrement by one'],sep+'Minimum and Maximum Operators:',tab+tab+tab+['<'+tab+'Minimum, the smaller of two operands. eg A=5<(-6) is the same as A=-6',$
            '>'+tab+'Maximum, the larger of two operands'],sep+'Matrix Multiplication:'+tab+'#'+tab+'Multiply the columns of the first array by the rows of the second array',$
            tab+tab+tab+'##'+tab+'Multiply the rows of the first array by the columns of the second array',sep+'Logical Operators:'+tab+'&&'+tab+'And',tab+tab+tab+['||'+tab+'Or','~'+tab+'Not'],$
            sep+'Bitwise Operators:'+tab+'AND'+tab+'And',tab+tab+tab+['OR'+tab+'Or','NOT'+tab+'Bitwise inverse','XOR'+tab+'Exclusive or'],$
            sep+'Relational Operators:'+tab+'EQ'+tab+'Equal to',tab+tab+tab+['NE'+tab+'Not equal to','LE'+tab+'Less than or equal to','LT'+tab+'Less than',$
            'GE'+tab+'Greater than or equal to','GT'+tab+'Greater than'],'','Visit https://www.nv5geospatialsoftware.com/docs/idl_programming.html for more IDL programming help.']
    xdisplayfile,text=text,group=group,title='IDL Statement Syntax',done_button='Exit',/editable,/grow_to_screen,return_id=wid,width=100,font=font
    dm_center_kid,wid
end

;get IP address of the computer
;returns -1 if unsuccessful
function dm_ip_address
    case !version.os_family of
         'Windows':   spawn,'ipconfig',result,error,/noshell
         else:        spawn,'ifconfig -a',result,error,/noshell
    endcase
    if strlen(error) ne 0 then return,-1
    for i=0,n_elements(result)-1 do begin
        if !version.os_family  eq 'Windows' then begin  
           tmp = stregex(result[i],'ip.* address.+\: +([0-9]+\.[0-9]+\.[0-9]+\.[0-9]+)',/fold_case,/subexpr,length=len)  
        endif else begin
           tmp = stregex(result[i],'inet +([0-9]+\.[0-9]+\.[0-9]+\.[0-9]+)',/fold_case,/subexpr,length=len)
        endelse
        if tmp[1] ne -1 then begin 
            ip_address = strmid(result[i],tmp[1],len[1])
            break
        endif
    endfor
    if n_elements(ip_address) ne 0 then return,ip_address else return,-1
end

;a simple AES encryption of an IP-address string no longer than 16 characters
;return:
;   the encrypted and MIME base64 encoded string (unless base64 keyword is set to 0)
;parameter:
;   ip_address:     IP-address string, not longer than 16 characters
;keyword:
;   byte_str:       save the byte array of the encrypted string
;   keystring:      string of the encryption key, 16 character long, optional
;   base64:         if set to 0, returns the plain encrypted string, otherwise return the MIME base64 encoded string, default is set to 1.
function dm_ip_encrypt,ip_address,byte_str=byte_str,keystring=keystring,base64=base64
    if dm_to_number(!version.release) ge 8.22 then ran1_extra = {ran1:1}
    byte_str = byte(ip_address)
    if n_elements(base64) eq 0 then base64 = 1b
    if dm_to_number(!version.release) lt 7.1 then base64 = 0b
    if n_elements(byte_str) lt 16 then begin
       repeat byte_str = [byte_str,0b] until n_elements(byte_str) eq 16 
    endif else begin
       byte_str = byte_str[0:15]
    endelse
    if n_elements(keystring) eq 0 then begin
       key = byte(97+randomu(byte('S'),16,_extra=ran1_extra)*26)
    endif else begin
       key = byte(keystring)
       if n_elements(key) lt 16 then begin
          repeat key = [key,key] until n_elements(key) ge 16 
       endif
       key = key[0:15]
    endelse 
    ;create Sbox
    ;create log table and inverse
    atab = 1
    for i=0,253 do atab = [atab,(((tmp=(atab[i]*2)) gt 255)?((tmp-256) xor '1b'x):(tmp)) xor atab[i]]
    atab = [atab,1]   & atab[0] = 0
    ltab = sort(atab) & atab[0] = 1
    ;multiplicative inverse
    mi = atab[255-ltab] & mi[0] = 0 & atab=0 & ltab=0 
    Sbox = mi
    for i=0,3 do mi = mi xor (Sbox=(Sbox=Sbox*2)+(Sbox gt 255)*(-255))
    Sbox = byte(mi) xor '63'xb
    rcon = ['1'x,'2'x,'4'x,'8'x,'10'x,'20'x,'40'x,'80'x,'1b'x,'36'x]  
    byte_str = byte_str xor key
    for rnd=0,9 do begin
        t = key[12:15] & t = t[[1,2,3,0]] & t = Sbox[t] & t[0] = (t[0] xor rcon[rnd])
        for i=0,3 do key[i] = key[i] xor t[i]
        for i=1,3 do begin
            t = key[((i-1)*4):((i-1)*4+3)]
            for j=0,3 do key[4*i+j] = key[4*i+j] xor t[j]
        endfor
        byte_str = reform((sbox[byte_str])[[0,5,10,15,4,9,14,3,8,13,2,7,12,1,6,11]],4,4)
        if rnd ne 9 then begin
           b = byte(((b= byte_str*2)-(tmp=(b gt 255))*256) xor ('1b'x*tmp))
           byte_str = b xor byte_str[*,[3,0,1,2]] xor byte_str[*,[2,3,0,1]] xor b[*,[1,2,3,0]] xor byte_str[*,[1,2,3,0]]
        endif
        byte_str = reform(byte_str,16) xor key                                           
    endfor
    if keyword_set(base64) then return,idl_base64(byte_str) else return,string(byte_str)
end

;kidney scan for given Ei
pro dm_kidscan,Ei,step,a3=a3,beta1=beta1,beta2=beta2,comment=comment,Ef=Ef,filename=filename,spinstate=spinstate,gobetas=gobetas,dmbt=dmbt,keepstep=keepstep,max_step=max_step,$
    shifthalfstep=shifthalfstep,counter=counter,cnts=cnts,kid_i=kid_i,kid_f=kid_f,n_kid=n_kid,sync=sync,help=help,old_step=old_step,threeblade=threeblade,vfocus=vfocus,backlash=backlash,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_kidscan sets up a kidney scan at a given Ei.',$
                 'syntax: dm_kidscan, Ei, step, a3=a3, beta1=beta1, beta2=beta2, comment=comment, Ef=Ef, filename=filename ,spinstate=spinstate, gobetas=gobetas, dmbt=dmbt, $',$
                 '                   keepstep=keepstep, max_step=max_step, shifthalfstep=shifthalfstep, counter=counter,cnts=cnts, threeblade=threeblade, vfocus=vfocus, $',$
                 '                   kid_i=kid_i, kid_f=kid_f, n_kid=n_kid, sync=sync, help=help',$
                 sep+'Ei:'+tab+tab+'incident energy in unit of meV, a scalar.',$
                 sep+'step:'+tab+tab+'kidney scan step, a scalar. If keepstep keyword is not set, an optimized step will be used instead to achieve evenly spaced A4.',$
                 sep+'a3:'+tab+tab+'a3 angle position for the kidney scan, a scalar.',$
                 sep+'beta1, beta2:'+tab+'beta1 and beta2 value, a scalar.',$
                 sep+'comment:'+tab+'comment of the scan, a string. filename keyword needs to be specified.',$
                 sep+'Ef:'+tab+tab+'final energy in unit of meV, a scalar. If present, Ef will be set in the sequence.',$
                 sep+'filename:'+(keyword_set(printhelp)?tab:tab+tab)+'filename of the data of the scan. a3 needs to be specified if this keyword is used.',$
                 sep+'spinstate:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, the spin state will be added to comment and the end of the file name if those keywords are specified.',$
                 sep+'gobetas:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, beta1 and beta2 calculated from Ei and dmbt will be set. dmbt keyword needs to be given a value.',$
                 sep+'dmbt:'+tab+tab+'dmbt in unit of mm, a scalar. It is combined with gobetas keyword to set beta1 and beta2.',$
                 sep+'keepstep:'+tab+'if set, the specified step will be used, otherwise an optimized step will be used.',$
                 sep+'max_step:'+tab+'if set, use the largest optimized step. The larger the step, the wider the uneven edge.',$
                 sep+'shifthalfstep:'+(keyword_set(printhelp)?'':tab)+'if set, the kidney angles will be shifted by half a step.',$
                 sep+'counter:'+(keyword_set(printhelp)?tab:tab+tab)+"'Time' or 'Monitor', default is 'Monitor'",$
                 sep+'cnts:'+tab+tab+'preset counter value. Default is 1 sec for time and 2.5e6 for monitor.',$
                 sep+'threeblade:'+(keyword_set(printhelp)?tab:tab)+'if set, horizontal focus is flat, and only three monblades are used.',$
                 sep+'vfocus:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, in vertical focus mode. threeblade keyword needs to be set.',$
                 sep+'kid_i:'+(keyword_set(printhelp)?tab:tab+tab)+'starting kidney angle of the scan, a scalar. Default is the lower limit of the kidney angle.',$
                 sep+'kid_f:'+(keyword_set(printhelp)?tab:tab+tab)+'final kidney angle of the scan, a scalar. Default is the upper limit of the kidney angle.',$
                 sep+'backlash:'+tab+'kidney backlash, default is 2 degrees.',$
                 sep+'n_kid:'+(keyword_set(printhelp)?tab:tab+tab)+'save the output number of kidney angles.',$
                 sep+'sync:'+tab+tab+'if set, the command is run synchronously.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       ei_max = !dm_macsseq_dryrun.ei_max
       ei_min = !dm_macsseq_dryrun.ei_min
       !dm_macsseq_dryrun.nspinstate = !dm_macsseq_dryrun.nspinstate+keyword_set(spinstate)
    endif

    dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min   
    dm_check_parm,step,name='step',/number,/gezero
    if n_elements(Ef) ne 0 then dm_check_parm,Ef,name='Ef',/number,/positive
    if n_elements(backlash) ne 0 then dm_check_parm,backlash,name='backlash',/number,/positive else backlash = 2.0
    if n_elements(beta1) ne 0 then dm_check_parm,beta1,name='beta1',/number
    if n_elements(beta2) ne 0 then dm_check_parm,beta2,name='beta2',/number
    if keyword_set(gobetas) then dm_check_parm,dmbt,name='dmbt',/number,/gezero
    if n_elements(counter) eq 0 then counter = 'Monitor'
    if (strlowcase(counter[0]) ne 'time') and (strlowcase(counter[0]) ne 'monitor') then message,"counter must be either 'Time' or 'Monitor'."
    if n_elements(cnts) eq 0 then begin
       if strlowcase(counter[0]) eq 'time' then cnts = 1 $
       else cnts = 2.5e6
    endif
    dm_check_parm,cnts,name='cnts',/number,/positive
    if n_elements(filename) ne 0 then begin
       dm_check_parm,filename,name='filename',/string,nochar=[' ','=']
       if strmid(strtrim(filename[0],2),0,1,/reverse_offset) ne '_' then filename = strtrim(filename[0],2)+'_'
       if keyword_set(spinstate) then filename = filename+!spinstate+'_'
       dm_check_parm,a3,name='Please specified a3. a3',/number
       if n_elements(comment) ne 0 then begin
          dm_check_parm,comment,name='comment',/string
          tmppos = strpos(comment,'"')
          while tmppos ne -1 do begin
                strput,comment,"'",tmppos
                tmppos = strpos(comment,'"')
          endwhile
          if keyword_set(spinstate) then comment = !spinstate+' '+comment
       endif
    endif else begin
      if n_elements(a3) ne 0 then dm_check_parm,a3,name='a3',/number
    endelse
    kran = dm_getkidneyrange(Ei[0])
    kid0 = kran[0]
    kran[0] = kran[0]+backlash
    tmp_start = kran[0]-0.5
    
    if n_elements(kid_i) ne 0 then begin
       dm_check_parm,kid_i,name='kid_i',/number,upperlimit=kran[1]
       kran[0] = (kran[0])>(kid_i[0])
    endif
    if n_elements(kid_f) ne 0 then begin
       dm_check_parm,kid_f,name='kid_f',/number,lowerlimit=kran[0]
       kran[1] = (kran[1])<(kid_f[0])
    endif
    if keyword_set(keepstep) then begin
       tmp_step = step[0]
       all_step = dm_optimizekidstep(tmp_step/floor(tmp_step),kran,/all_step,eq_step=all_equv)
       ind = where(abs(all_step-tmp_step) lt 0.0001,cnt)
       if cnt ne 0 then eq_step = all_equv[ind[0]] 
    endif else begin
       if keyword_set(old_step) then tmp_step = dm_optimizekidstep_old(step[0],kran,eq_step=eq_step) $
       else tmp_step = dm_optimizekidstep(step[0],kran,eq_step=eq_step,max_step=max_step)
    endelse
      
    n_kid = (tmp_step eq 0)?1:(floor(abs(kran[1]-kran[0])/tmp_step)+1)-keyword_set(shifthalfstep)
    range = (n_kid-1)*tmp_step
    kcen  = min(kran)+range/2.0+tmp_step/2.0*keyword_set(shifthalfstep)
    dm_print,'kidney scan: Ei='+dm_to_string(Ei[0])+' meV, total kidney number='+dm_to_string(n_kid)+',  actual step='+dm_to_string(tmp_step,res=4)+$
             (n_elements(eq_step) ne 0?', equivalent A4 step='+dm_to_string(eq_step,res=4):'')
    if Ei[0] gt 16.5 then begin
       comm0 = ['move ei 16 a4 0','move ei 16.5 kidney -51']
    endif else begin
       comm0 = 'move ei '+dm_to_string(Ei[0])+' a4 0 '
    endelse
    comm0 = [comm0,'move ei '+dm_to_string(Ei[0])+' kidney '+dm_to_string(kcen,res=3)]
    if n_elements(beta1) ne 0 then comm0=[comm0,'move beta1 '+dm_to_string(beta1)]
    if n_elements(beta2) ne 0 then comm0=[comm0,'move beta2 '+dm_to_string(beta2)]
    if n_elements(a3) ne 0 then comm0=['move a3 '+dm_to_string(a3[0]),comm0]
    if n_elements(Ef) ne 0 then comm0=[comm0,'move Ef '+dm_to_string(Ef[0])]
    dm_sendcom,'autoptai'
    if keyword_set(threeblade) then begin
       dm_sendcom,'action vertifocus flat'
       dm_sendcom,'action horizfocus flat'
    endif
    for i=0,n_elements(comm0)-1 do dm_sendcom,comm0[i]
    if keyword_set(threeblade) then begin
       dm_move,'beta1',3
       dm_move,'beta2',0      
       dm_setblades,3
       if keyword_set(vfocus) then dm_move,'focus',dm_get_focus(Ei[0])
    endif
    if keyword_set(gobetas) then dm_gobetas,Ei[0],dmbt[0]
    if n_elements(filename) ne 0 then begin
       cmd_str = 'scan descrtolist "dm_kidscan" Scan:JType=ANGLE:Npts='+dm_to_string(n_kid)+':Counts='+dm_to_string(cnts)+$
                 ':Prefac=1.0:DetectorType=SPEC:CountType='+counter+':Filename='+strtrim(filename[0],2)+':HoldScan=0.0:Range=A3='+dm_to_string(a3)+' '+dm_to_string(a3)+$
                 ' s:Range=Kidney='+dm_to_string(kcen-range/2.0,res=4)+' '+dm_to_string(kcen+range/2.0,res=4)+' s'
       if n_elements(comment) ne 0 then cmd_str = cmd_str+':Comment="'+comment+'"'
       dm_move,'kidney',(kid0)>(kcen-range/2.0-backlash)  ;to overcome backlash
       dm_sendcom,cmd_str
       ;dm_wait,10
       dm_sendcom,'device read temp'
       dm_sendcom,'scan runscan "dm_kidscan"'
    endif else begin
       dm_sendcom,'findpeak kidney '+dm_to_string(range+0.001,res=3)+' '+dm_to_string(tmp_step)+' '+ counter +' '+ dm_to_string(cnts)+' spec',sync=sync
    endelse
    if keyword_set(threeblade) and keyword_set(vfocus) then dm_sendcom,'action vertifocus flat'
    if exists then begin
       !dm_macsseq_dryrun.nes  = !dm_macsseq_dryrun.nes+1
       !dm_macsseq_dryrun.nkid = !dm_macsseq_dryrun.nkid+n_kid
    endif
end

;scan dfmdts and blade11 at different A2 for instrument alignment
pro dm_macs_alignment,filename=filename,group_leader=group_leader,help=help,nowait=nowait,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_macs_alignment scans dfmdts and blade11 at different A2 for instrument alignment.',$
                 'syntax: dm_macs_alignment, filename=filename, group_leader=group_leader, nowait=nowait',$
                 sep+'filename:'+tab+'name of a file to save the alignment results, a string.',$
                 sep+'group_leader:'+tab+'group leader widget ID.',$
                 sep+'nowait:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, no wait command will be issued while resetting monochromator blade positions.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    angles   = [110,100.,90.,80.,70.,60.,50.] & n_ang = n_elements(angles)
    dfmdts   = fltarr(n_ang)
    blade11  = fltarr(n_ang)
    mbtslide = fltarr(n_ang)
    dryrun   = dm_macs_sequencer_isdryrun(mesgobj=mesgobj,group_leader=group_leader)
    dm_sendcom,'action vertifocus flat '    
    dm_sendcom,'action horizfocus flat'
    dm_sendcom,'move beta1 0 beta2 0'
    dm_sendcom,'move dmbt 10'
    dm_sendcom,'move vbah 10'
    dm_sendcom,'move vbav 360'
    dm_sendcom,'autoptai'
    for nn=0,n_ang-1 do begin
        dm_sendcom,'move a2 '+dm_to_string(angles[nn])+' a1 '+ dm_to_string(angles[nn]/2) + ' a4 0' 
        dm_move,'monrot',0
        dm_move,'monblade11',angles[nn]/2.
        dm_findpeak,'dfmdts',80,4,5,detector='monitor',ffit='gauss'      
        dm_sendcom,'acceptfindpeak'
        dm_findpeak,'monblade11',3,0.2,5,detector='monitor',ffit='gauss'     
        dm_sendcom,'acceptfindpeak'
        if n_elements(filename) ne 0 then begin
           if n_elements(datafile) ne 0 then tmp = temporary(datafile)
           dm_findpeak,'dfmdts',80,4,5,detector='monitor',ffit='gauss',datafile=datafile,/sync
           if ~dryrun then begin
              dm_load_macs,datafile,data,info,header=header,deteff=deteff,error=error,all_neg=all_neg,filename=tmpfilename
              if nn eq 0 then files = tmpfilename
              if error then message,'error encountered reading the file.'
              if all_neg then message,'empty data.'
              xind = where(info eq 'dfmdts',count) & if count eq 0 then continue
              yind = where(info eq 'monitor',count) & if count eq 0 then continue
              x = data[*,xind[0]] & y = data[*,yind[0]] & yerr = sqrt(y)
              dm_gaussfit,y,x=x,params=params,/plot
              dfmdts_cen = params[1]
              dm_move,'dfmdts',dfmdts_cen   
              tmp = temporary(datafile)
              dm_findpeak,'monblade11',3,0.2,5,detector='monitor',ffit='gauss',datafile=datafile,/sync
              dm_load_macs,datafile,data,info,header=header,deteff=deteff,error=error,all_neg=all_neg,filename=tmpfilename
              if error then message,'error encountered reading the file.'
              if all_neg then message,'empty data.'
              xind = where(info eq 'monblade11',count) & if count eq 0 then continue
              yind = where(info eq 'monitor',count) & if count eq 0 then continue
              x = data[*,xind[0]] & y = data[*,yind[0]] & yerr = sqrt(y)
              dm_gaussfit,y,x=x,params=params,/plot
              bladecenter = params[1]
              dm_move,'monblade11',bladecenter           
              dfmdts[nn]   = dfmdts_cen
              blade11[nn]  = bladecenter
              mbtslide[nn] = dm_dev_pos('mbtslide',/number)
              dm_print,' '
              dm_print,'a2 = '+dm_to_string(angles[nn])
              dm_print,'mbtslide = '+dm_to_string(mbtslide[nn])
              dm_print,'monblade11 = '+dm_to_string(blade11[nn])
              dm_print,'dfmdts = '+dm_to_string(dfmdts[nn])
           endif else dm_print,'findpeak result for a2='+dm_to_string(angles[nn])+' is not available in dryrun.'
        endif else begin
           dm_wait,6666
           dm_findpeak,'dfmdts',80,4,5,detector='monitor',ffit='gauss'
           dm_sendcom,'acceptfindpeak'
           dm_wait,6666
           dm_findpeak,'monblade11',3,0.2,5,detector='monitor',ffit='gauss'
           dm_sendcom,'acceptfindpeak'     
        endelse
    endfor
    if ~dryrun then begin
       files = [files,tmpfilename]
       files = file_basename(files,strmid(files[0],strpos(files[0],'.',/reverse_search)))
       dm_fit_macs_alignment,angles,mbtslide,dfmdts,blade11,filename=files[0]+'-'+files[1],/ice,savefile=filename,group_leader=group_leader,nowait=nowait
    endif else dm_print,'Fitting and resetting alignment parameters is not available in dryrun.'    
end

pro dm_macs_kidneylimitscan,filename=filename,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_macs_kidneylimitscan scans for the kidney hardware limits at various energies.',$
                 'syntax: dm_macs_kidneylimitscan, filename=filename',$
                 sep+'filename:'+tab+'name of a file to save the scan results, a string.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj)
    q = 2*!pi/3.35416 ;monochromator q VECTOR
    ;get the energies to scan
    ok  = dm_check_macskidney(0,5,limits=limits)
    Eis = limits[*,0] ;or specify a list here
    l_l = limits[*,1] ;existing lower limits
    u_l = limits[*,2] ;existing upper limits
    dm_sendcom,'autoptai'
    for i=0L,n_elements(Eis)-1 do begin
        a2 = asin(q/2./sqrt(Eis[i]/2.0721248))*2./!dtor
        dm_sendcom,'Setmaxelbowwristmoveattempts 7'
        dm_sendcom,'move a2 '+dm_to_string(a2,resolution=3)+' a4 0'  ;move A2 to the corresponding Ei value
        dm_sendcom,'Setmaxelbowwristmoveattempts 1'
        dm_sendcom,'move kidney '+dm_to_string(l_l[i]-10,resolution=3),/sync,cmd_response=cmd_response  ;check lower limit
        tmp = dm_get_error_kidlimit(cmd_response)
        if finite(tmp) then l_l[i] = tmp
        dm_sendcom,'move kidney '+dm_to_string(u_l[i]+10,resolution=3),/sync,cmd_response=cmd_response  ;check upper limit
        tmp = dm_get_error_kidlimit(cmd_response)
        if finite(tmp) then u_l[i] = tmp
        dm_sendcom,'move kidney -2 -relative' ;move kidney away from the upper limit position
    endfor
    dm_sendcom,'Setmaxelbowwristmoveattempts 7'
    
    if dryrun then return
    
    ;save the result
    if n_elements(filename) eq 0 then filename='macs_kidneylimit_'+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt'
    openw,unit,filename,/get_lun,error=openerr
    if openerr ne 0 then message,"Can't write in "+filename+'.'
    for i=0L,n_elements(Eis)-1 do printf,unit,Eis[i],l_l[i],u_l[i]
    free_lun,unit
end

;move a device
pro dm_move,device,target,relative=relative,sync=sync,cmd_success=cmd_success,cmd_error=cmd_error,cmd_response=cmd_response,_ref_Extra=extra,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_move moves a device to a target position.',$
                 'syntax: dm_move, device, target, relative=relative, sync=sync, cmd_success=cmd_success, cmd_error=cmd_error, cmd_response=cmd_response, help=help',$
                 sep+'device:'+(keyword_set(printhelp)?tab:tab+tab)+'a string of the name of the device.',$
                 sep+'target:'+(keyword_set(printhelp)?tab:tab+tab)+'the target position, a scalar.',$
                 sep+'relative:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, a relative move.',$
                 sep+'sync:'+tab+tab+'if set, the command is run synchronously.',$
                 sep+'cmd_success:'+tab+'returns the success status.',$
                 sep+'cmd_error:'+tab+'returns the error status.',$
                 sep+'cmd_response:'+tab+'returns the command response.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,device,name='device',/string
    dm_check_parm,target,name='target'
    if n_elements(extra) ne 0 then message,'syntax error, use dm_move,/help for help.'
    f_pos = target[0]
    if size(f_pos,/type) eq 7 then begin
       f_pos = strtrim(strupcase(f_pos),2)
       case f_pos of 
            'IN' : f_pos = 1
            'ON' : f_pos = 1
            'OUT': f_pos = 0
            'OFF': f_pos = 0
            else:  message,'invalid target.'
       endcase
    endif
    
    dev = strcompress(strupcase(device[0]),/REMOVE_ALL)
    valid_device = where(dm_def_dev() eq dev,count)
    if count eq 0 then message,'invalid device.'

    ;Build the command
    if keyword_set(relative) then relstr=' -relative' else relstr=''
    command = 'move '+dev+' '+dm_to_string(f_pos)+relstr
    dm_sendcom,command,cmd_success=cmd_success,cmd_error=cmd_error,cmd_response=cmd_response,sync=sync
end

;old kidney step optimization method
function dm_optimizekidstep_old,step,kran,eq_step=eq_step,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_optimizekidstep_old() optimize a kidney step to ensure even detector distribution in a given kidney range. Returns the closest optimal step.',$
                 'syntax: newstep = dm_optimizekidstep_old(step, kran, eq_step=eq_step, help=help)',$
                 sep+'step:'+tab+tab+'target step, a scalor number.',$
                 sep+'kran:'+tab+tab+'kidney range, a 2-element array.',$
                 sep+'eq_step:'+(keyword_set(printhelp)?tab:tab+tab)+'save the equivalent A4 step.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,0
    endif

    dm_check_parm,step,name='step',/number
    dm_check_parm,kran,name='kran',/number
    if n_elements(kran) eq 1 then message,'kran must be a 2-element array.'
    if (step eq 0) or (step ge 8) or (abs(kran[1]-kran[0]) lt 8) then begin
       eq_step = !values.f_nan
       return,step
    endif

    N = (1.0+(abs(kran[1]-kran[0])+19*8.)/step)/20.
    s = abs(kran[1]-kran[0])/N

    n = max([floor(abs(kran[1]-kran[0])/8.0),1])
    m = max([floor(8.0*n/s),1])
    
    for i=m,m+100 do begin
        if ((i/8)*8 ne i) and ((i/n)*n ne i) then begin
           out_step = 8.*n/i
           break
        endif
    endfor
    if n_elements(out_step) eq 0 then out_step = 8.*n/m
    if arg_present(eq_step) then begin
       nx = floor(abs(kran[1]-kran[0])/out_step)+1
       x0 = findgen(nx)*out_step mod 8.0
       x0 = x0[UNIQ(x0,sort(x0))]
       if n_elements(x0) eq 1 then eq_step = 8.0 else eq_step = round((x0[1]-x0[0])*1e5)/1e5
    endif
    return,out_step
end

function dm_optimizekidstep,step,kran,eq_step=eq_step,max_step=max_step,n_step=n_step,all_step=all_step,tol_limit=tol_limit,det_span=det_span,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_optimizekidstep() optimize a kidney step to ensure even detector distribution in a given kidney range. Returns the largest optimal step.',$
                 'syntax: newstep = dm_optimizekidstep(step, kran, eq_step=eq_step, max_step=max_step, all_step=all_step, tol_limit=tol_limit, help=help)',$
                 sep+'step:'+tab+tab+'target step, a scalor number.',$
                 sep+'kran:'+tab+tab+'kidney range, a 2-element array.',$
                 sep+'eq_step:'+(keyword_set(printhelp)?tab:tab+tab)+'save the equivalent A4 step.',$
                 sep+'max_step:'+tab+'if set, return the largest optimal step. The larger the step, the wider the uneven edge.',$
                 sep+'n_step:'+(keyword_set(printhelp)?tab:tab+tab)+'save the number of steps using the optimized step for the kidney range.',$
                 sep+'all_step:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, return all possible steps.',$
                 sep+'tol_limit:'+(keyword_set(printhelp)?tab:tab+tab)+'the step angle tolerance limit, default is 0.25*step.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return,0
    endif

    if n_elements(det_span) eq 0 then det_span = 8.0  ;detector interval

    dm_check_parm,step,name='step',/number
    dm_check_parm,kran,name='kran',/number
    if n_elements(kran) eq 1 then message,'kran must be a 2-element array.'
    if (step eq 0) or (step ge det_span) then begin
       n_step = (step eq 0)?0:(1+floor(abs(kran[1]-kran[0])/step))
       eq_step = ((n_step*step mod det_span) eq 0)?(step):!values.f_nan
       return,step
    endif

    nstep = 1+round(det_span/step)
    ss = [0.0,0.0]            ;[eq_step, out_step]
    for i=0,nstep-2 do begin  ;generate a list of all possible evenly spaced steps
        for j=0,1 do begin
            eq_step = det_span/(nstep+((-1)^j)*float(i))
            for k=(nstep<(floor(det_span/eq_step))),1,-1 do begin ;do not use a step greater than detector interval
                out_step = k*eq_step
                x0 = round(((findgen(floor(abs(kran[1]-kran[0])/out_step)+1)*out_step) mod det_span)/eq_step)
                x0 = x0[uniq(x0,sort(x0))]
                if n_elements(x0) eq round(det_span/eq_step) then ss = [[ss],[[eq_step,out_step]]]
            endfor
        endfor
    endfor
    ss0 = round(ss[*,1:*]*1e5)/1e5
    if keyword_set(all_step) then begin
       tmp = ss0[0,*]*100+ss0[1,*]/10
       ss0 = ss0[*,uniq(tmp,sort(tmp))]
       eq_step = reform(ss0[0,*])
       outstep = reform(ss0[1,*])
    endif else begin
       dif = min(abs(ss0[0,*]-step))
       tol = 0.0 
       if n_elements(tol_limit) eq 0 then tol_limit = 0.1>round((0.25*step)*10)/10.0
       while n_elements(outstep) eq 0 do begin 
          ind = where(abs(ss0[0,*]-step) le (tol*(tol le tol_limit)+dif))
          ss  = ss0[*,ind]
          ind = where(ss[0,*] ne ss[1,*],count)   ;try to find a step that is larger than the equivalent step
          if (count ne 0) or (tol gt tol_limit) then begin
             if count ne 0 then ss = ss[*,ind] 
             if ~keyword_set(max_step) then begin
                ind = where(ss[1,*] le det_span/2.,count)
                if count ne 0 then begin
                   ss = ss[*,ind]
                   tmp = max(ss[1,*],ind)
                endif else tmp = min(ss[1,*],ind)
             endif else tmp = max(ss[1,*],ind)
             eq_step = ss[0,ind]
             outstep = ss[1,ind]
          endif
          tol = tol+([0.05,0.1])[tol eq 0]
       endwhile
    endelse
    n_step = 1+floor(abs(kran[1]-kran[0])/outstep)
    return,outstep
end

;plot and fit macs data.
pro dm_plot_macs,file,device,move2fit=move2fit,setpos=setpos,center=center,txtfile=txtfile,all_dat=all_dat,intg_int=intg_int,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_plot_macs plot and fit a MACS data file. It can also move the specified device to the fit positions.',$
                 'syntax: dm_plot_macs, file, device=device, center=center, move2fit=move2fit, setpos=setpos, txtfile=txtfile, all_dat=all_dat, intg_int=intg_int, help=help',$
                 sep+'file:'+tab+tab+'data file name, a scalar string, or an array of data file content.',$
                 sep+'device:'+(keyword_set(printhelp)?tab:tab+tab)+"device name to plot against, 'a1' or 'a5', a scalor string.",$
                 sep+'center:'+(keyword_set(printhelp)?tab:tab+tab)+'returns the fit center of all the specified device.',$
                 sep+'move2fit:'+tab+'an array of device indices starting from 1. If present, the specified devices will be moved to their corresponding fit centers. Only for A5 device.',$
                 sep+'setpos:'+(keyword_set(printhelp)?tab:tab+tab)+'the set position of the anaylzers, a scalar. If present, the devices specified in move2fit will be reset after they are moved to the fit center.',$ 
                 sep+'txtfile:'+(keyword_set(printhelp)?tab:tab+tab)+'file name to save all the fitting parameters, a scalar string.',$
                 sep+'all_dat:'+(keyword_set(printhelp)?tab:tab+tab)+'returns the fitting parameters, a [20,4] array.',$
                 sep+'intg_int:'+(keyword_set(printhelp)?tab:tab+tab)+'returns the integrated intensity, a 20-element array.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,file,name='file',/string
    dm_check_parm,device,name='device',/string
    if n_elements(move2fit) gt 0 then begin
       ind = where((move2fit le 0) or (move2fit gt 20),count)
       if count ne 0 then message,'invalid move2fit.'
    endif
    dev = strcompress(strupcase(device[0]),/REMOVE_ALL)
    valid_device = where(dm_def_dev() eq dev,count)
    if count eq 0 then message,'invalid device.' 
    case dev of 
         'A5': begin
               ind  = 'analyzertheta'+string(indgen(20)+1,format='(i02)')
               dep  = 'spec'+string(indgen(20)+1,format='(i02)')
               xtit = 'Analyzer Theta (\deg)'
               ytit = 'Spec (cts)'
               end
         'A1': begin
               ind  = 'a1'                                        ;'monblade'+string(indgen(21)+1,format='(i02)')
               dep  = 'spec'+string(indgen(20)+1,format='(i02)')  ;'monitor'
               xtit = 'A1 (\deg)'                                 ;'Monochromator Theta (\deg)'
               ytit = 'Spec (cts)'                                ;'Monitor (cts)'
               tmp  = temporary(move2fit)                         ;not allow move2fit
               end
         else: message,'device must be A1 or A5.'
    endcase
    
    dm_load_macs,file,data,info,header=header,deteff=deteff,filename=filename,error=error,all_neg=all_neg
    if error then message,'error encountered reading the file.'
    if all_neg then message,'empty data.'
    
    center   = fltarr(20)
    all_dat  = fltarr(20,4)
    intg_int = fltarr(20)
    
    pobj = obj_new('dm_plot',title=filename,xtit=xtit,ytit=ytit,background='white',legdcolumns=2)
    pcol = pobj->getcolor(/list)
    pcol = pcol[0:n_elements(pcol)-2] ;remove the user define color
    psym = pobj->getpsym(/list)
    psym = psym[0:4] & psym = [psym,'filled '+psym]
    
    for i=0,(n_elements(dep)-1)>(n_elements(ind)-1) do begin
        xind = where(info eq ind[i<(n_elements(ind)-1)],count) & if count eq 0 then continue
        yind = where(info eq dep[i<(n_elements(dep)-1)],count) & if count eq 0 then continue
        x = data[*,xind[0]] & y = data[*,yind[0]] & yerr = sqrt(y)
        dm_gaussfit,y,x=x,params=params
        all_dat[i,*] = [params[0],params[1],params[2]*2*SQRT(2*ALOG(2)),params[3]]  ;params[2]->FWHM
        intg_int[i]  = sqrt(2*!PI)*params[0]*params[2]
        center[i]    = params[1]
        xmin = min(x,max=xmax)
        xx   = xmin+findgen(101)/100*(xmax-xmin)
        yy   = params[0]*exp(-(xx-params[1])^2/(2*params[2]^2))
        ymin = min([y-yerr,yy])
        ymax = max([y+yerr,yy])
        if n_elements(xran) eq 0 then begin
           xran = [xmin,xmax]
           yran = [ymin,ymax]
        endif else begin
           xran[0] = (xran[0])<(xmin)
           xran[1] = (xran[1])>(xmax)
           yran[0] = (yran[0])<(ymin)
           yran[1] = (yran[1])>(ymax)
        endelse
        pobj->add_plot,x,y,yerr=yerr,color=pcol[i mod n_elements(pcol)],psym=psym[i mod n_elements(psym)],linestyle='no line',legend=dm_to_string(i+1)
        pobj->add_plot,xx,yy,color=pcol[i mod n_elements(pcol)],psym='no symbol',linestyle='solid'
    endfor
    pobj->setproperty,xran=xran+[-0.05,0.05]*(xran[1]-xran[0]),yran=yran+[-0.05,0.05]*(yran[1]-yran[0]),/legdshowoutline
    
    if n_elements(move2fit) gt 0 then begin
       for nn=0,n_elements(move2fit)-1 do begin
           analyzer = 'analyzertheta'+string(move2fit[nn],format='(i02)')
           dm_move,analyzer,center[move2fit[nn]-1]
           dm_sendcom,'wait 1'
           if n_elements(setpos) ne 0 then dm_sendcom,'device set '+analyzer+' '+dm_to_string(setpos[0])
       endfor
    endif
    
    if n_elements(txtfile) ne 0 then begin
       openw,lun,txtfile[0],/get_lun,error=openerr
       if openerr ne 0 then message,"Can't write in "+txtfile[0]+'.'
       printf,lun,transpose(all_dat)
       free_lun,lun
    endif
end

pro dm_print,expression,expression1,expression2,expression3,expression4,expression5,expression6,expression7,expression8,expression9,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_print prints on the sequencer status box.',$
                 'syntax: dm_print, expression, help=help',$
                 sep+'expression:'+tab+'string or string array to be printed.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    if n_elements(expression) eq 0 then return
    if size(expression,/type) ne 7 then strs = dm_to_string(expression) else strs = expression
    if n_elements(expression1) ne 0 then strs = [strs,dm_to_string(expression1)]
    if n_elements(expression2) ne 0 then strs = [strs,dm_to_string(expression2)]
    if n_elements(expression3) ne 0 then strs = [strs,dm_to_string(expression3)]
    if n_elements(expression4) ne 0 then strs = [strs,dm_to_string(expression4)]
    if n_elements(expression5) ne 0 then strs = [strs,dm_to_string(expression5)]
    if n_elements(expression6) ne 0 then strs = [strs,dm_to_string(expression6)]
    if n_elements(expression7) ne 0 then strs = [strs,dm_to_string(expression7)]
    if n_elements(expression8) ne 0 then strs = [strs,dm_to_string(expression8)]
    if n_elements(expression9) ne 0 then strs = [strs,dm_to_string(expression9)]
    
    help,/traceback,output=output
    if total(stregex(output,'DM_MACS_SEQUENCER_EVENT',/fold_case,/boolean)) gt 0 then begin
       tmp = obj_valid()
       ind = where(obj_isa(tmp,'DM_MACS_SEQUENCER'))
       tmp[ind[0]]->my_widget_control,'statusbox',get_value=txtstr
       if total(strlen(txtstr)) eq 0 then outstr = strs $
       else outstr = [txtstr,strs]
       tmp[ind[0]]->my_widget_control,'statusbox',set_value=outstr,set_text_top_line=0>(n_elements(outstr)-4)
    endif else if total(stregex(output,'DCS_MSLICE_EVENT',/fold_case,/boolean)) gt 0 then begin
       tmp = obj_valid()
       ind = where(obj_isa(tmp,'DCS_MSLICE'),count)
       for i=0,count-1 do begin
           tmp[ind[i]]->getproperty,ftype=ftype
           if ftype ne 'mslicedata' then break
       endfor
       tmp[ind[i]]->calculator_insert,';'+strs
    endif else print,strs
end

;monitor keyword is a little bit unreliable
pro dm_runscan,scanname,comment=comment,Ei=Ei,Ef=Ef,A3=A3,filename=filename,spinstate=spinstate,beta1=beta1,beta2=beta2,monitor=monitor,ptai=ptai,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_runscan runs a scan, the scan is already set up in ICE.',$
                 'syntax: dm_runscan, scanname, comment=comment, filename=filename ,spinstate=spinstate, Ei=Ei, Ef=Ef, A3=A3, beta1=beta1, beta2=beta2, ptai=ptai, help=help',$
                 sep+'scanname:'+tab+'scan name, a string or string array. This named scan is set up and stored in ICE.',$
                 sep+'comment:'+tab+'new comment of the scan, a string. The new comment will replace the existing comment in the specified scan. Do not use comma, < or > character.',$
                 sep+'filename:'+(keyword_set(printhelp)?tab:tab+tab)+'new file name of the scan, a string. The new file name will replace the existing file name in the specified scan.',$
                 sep+'spinstate:'+(keyword_set(printhelp)?tab:tab+tab)+'if set, the spin state will be added to comment and the end of the file name if those keywords are specified.',$
                 sep+'Ei:'+tab+tab+'incident energy at the start of the scan in unit of meV, a scalar.',$
                 sep+'Ef:'+tab+tab+'final energy at the start of the scan in unit of meV, a scalar.',$
                 sep+'A3:'+tab+tab+'A3 angle at the start of the scan, a scalar.',$
                 sep+'beta1, beta2:'+tab+'beta1 and beta2 value, a scalar.',$
                 sep+'ptai:'+tab+tab+'ptai value for the scan, an integer between 1 and 20. Manual ptai mode will be set. After the scan, ptai mode will be set to auto.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    for i=0,n_elements(scanname)-1 do dm_check_parm,scanname[i],name='scanname',/scanname
    if n_elements(filename) eq 1 then begin
       dm_check_parm,filename[0],name='filename',/string,nochar=[' ','=']
       if strmid(strtrim(filename[0],2),0,1,/reverse_offset) ne '_' then filename = strtrim(filename[0],2)+'_'
       if keyword_set(spinstate) then filename = filename+!spinstate+'_'
       for i=0,n_elements(scanname)-1 do dm_sendcom,'scan operation "'+strtrim(scanname[i],2)+'" Filename='+strtrim(filename[0],2)
    endif
    if n_elements(comment) eq 1 then begin
       dm_check_parm,comment[0],name='comment',/string
       if keyword_set(spinstate) then comment = !spinstate+' '+comment
       for i=0,n_elements(scanname)-1 do dm_sendcom,'scan operation "'+strtrim(scanname[i],2)+'" Comment="'+strtrim(comment[0],2)+'"'
    endif
    if n_elements(monitor) eq 1 then begin
       dm_check_parm,monitor[0],name='monitor',/number,/positive
       for i=0,n_elements(scanname)-1 do begin
           dm_sendcom,'scan operation "'+strtrim(scanname[i],2)+'" Counts='+dm_to_string(monitor[0])
           dm_sendcom,'scan operation "'+strtrim(scanname[i],2)+'" Prefac=1'
       endfor
    endif
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then !dm_macsseq_dryrun.nspinstate = !dm_macsseq_dryrun.nspinstate+keyword_set(spinstate)
    if n_elements(Ei) ne 0 then begin
       if exists then begin
          ei_max = !dm_macsseq_dryrun.ei_max
          ei_min = !dm_macsseq_dryrun.ei_min
       endif
       dm_check_parm,Ei,name='Ei',/number,/positive,upperlimit=ei_max,lowerlimit=ei_min
       dm_sendcom,'autoptai'
       dm_sendcom,'move Ei '+dm_to_string(Ei[0])+' a4 0'
       if n_elements(beta1) ne 0 then dm_sendcom,'move beta1 '+dm_to_string(beta1)
       if n_elements(beta2) ne 0 then dm_sendcom,'move beta2 '+dm_to_string(beta2)
    endif
    if n_elements(Ef) ne 0 then begin
       dm_check_parm,Ef,name='Ef',/number,/positive
       dm_sendcom,'move Ef '+dm_to_string(Ef[0])
    endif
    if n_elements(A3) ne 0 then begin
       dm_check_parm,A3,name='A3',/number
       dm_sendcom,'move A3 '+dm_to_string(A3[0])
    endif
    if n_elements(ptai) eq 1 then begin
       dm_check_parm,ptai[0],name='ptai',/number,lowerlimit=1,upperlimit=20
       dm_sendcom,'manualptai'
       dm_sendcom,'move ptai '+dm_to_string(ptai[0])
    endif
    for i=0,n_elements(scanname)-1 do dm_sendcom,'scan runscan "'+strtrim(scanname[i],2)+'"'
    if n_elements(ptai) eq 1 then dm_sendcom,'autoptai'
end

pro dm_runtab,tabname,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_runtab runs the script in a tab.',$
                 'syntax: dm_runtab, tabname, help=help',$
                 sep+'tabname:'+tab+'tab name, a string.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,tabname,name='tabname',/string
    tmpname = strtrim(tabname,2)
    
    tmp = obj_valid()
    ind = where(obj_isa(tmp,'DM_MACS_SEQUENCER'))
    tmp[ind[0]]->getproperty,tlb=tlb,currenttabtitle=currenttabtitle
    if strmatch(strtrim(currenttabtitle,2),tmpname,/fold_case) then message,'Calling the current tab.'
    
    wid = widget_info(tlb,find_by_uname='tabbase')
    wid = widget_info(wid,/child)
    while wid ne 0 do begin
       widget_control,wid,get_uvalue=tabtitle
       if strmatch(strtrim(tabtitle,2),tmpname,/fold_case) then break
       wid = widget_info(wid,/sibling)
    endwhile
    if wid eq 0 then message,'Could not find a tab with the name '+tmpname+'.'
    tboxwid = widget_info(wid,/child) ;locate the textbox widget id
    widget_control,tboxwid,get_value=sequence
    sequence = dm_strip_comments(sequence,/combine)
    ok = execute(sequence,1,1)
end

;send a command to ICE server
pro dm_sendcom,command,sync=sync,cmd_success=cmd_success,cmd_error=cmd_error,cmd_response=cmd_response,immediate=immediate,data=data,_ref_Extra=extra,help=help,output=output,quiet=quiet
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_sendcom sends an ICE command to the server.',$
                 'syntax: dm_sendcom, command, sync=sync, cmd_success=cmd_success, cmd_error=cmd_error, cmd_response=cmd_response, immediate=immediate, data=data, help=help',$
                 sep+'command:'+tab+'a command string.',$
                 sep+'sync:'+tab+tab+'if set, the command is run synchronously.',$
                 sep+'cmd_success:'+tab+'returns the success status.',$
                 sep+'cmd_error:'+tab+'returns the error status.',$
                 sep+'cmd_response:'+tab+'returns the command response.',$
                 sep+'immediate:'+tab+'if set, the command is an immediate command, otherwise the command is a queued command.',$
                 sep+'data:'+tab+tab+"returns the data of the finished scan in the form of a string array, sync keyword must be set, otherwise the command must be equal to ''."]
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    dm_check_parm,command,name='command',/string
    if n_elements(extra) ne 0 then message,'syntax error, use dm_sendcom,/help for help.'
    
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(!error_state.msg,/error,/center)
       catch,/cancel
       cmd_success = 0b & cmd_error = 1b & cmd_response  = '' & data = ''
       if obj_valid(iceAPI)                then obj_destroy,iceAPI
       if obj_valid(immCmd)                then obj_destroy,immCmd
       if obj_valid(staticClientReference) then obj_destroy,staticClientReference
       if obj_valid(oJBridgeSession)       then obj_destroy,oJBridgeSession
       return
    end

    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj,isexecute=isexecute)

    if stregex(command[0],'^ *wait +[0-9]+ *$',/boolean,/fold_case) then !dm_macsseq_dryrun.nwait = !dm_macsseq_dryrun.nwait+1 $
    else if stregex(command[0],'^ *iftalk.*spinflip.*',/boolean,/fold_case) then !dm_macsseq_dryrun.nspinflip = !dm_macsseq_dryrun.nspinflip+1

    if dryrun then begin
       if ptr_valid(!dm_macsseq_dryrun.sequence) then *(!dm_macsseq_dryrun.sequence) = [*(!dm_macsseq_dryrun.sequence),command[0]] $
       else !dm_macsseq_dryrun.sequence = ptr_new(command[0])
       if keyword_set(sync) then !dm_macsseq_dryrun.sync = 1b
       cmd_success = 1b & cmd_error = 0b & cmd_response  = '' & data = ''
       return
    endif

    if strlen(command[0]) ne 0 then begin
       if keyword_set(isexecute) and (n_elements(mesgobj) eq 0) then begin
          catch,/cancel ;let the error below handled elsewhere
          message,'Sequence stopped.',level=-1,/noprint,/noname
       endif
       if ~keyword_set(quiet) then begin
          if n_elements(mesgobj) ne 0 then mesgobj->update,message='EXECUTING '+command[0]
          dm_print,'EXECUTING '+command[0]
       endif
    endif else if ~arg_present(data) then return
    
    ; Grab the special IDLJavaBridgeSession object
    oJBridgeSession = OBJ_NEW('IDLJavaObject$IDLJAVABRIDGESESSION')
    staticClientReference = OBJ_NEW('IDLjavaObject$Static$ICE_CLIENTAPI', 'ice.ClientAPI')
    
    if ~obj_valid(staticClientReference) then begin
       cmd_success = 0b & cmd_error = 1b & cmd_response  = '' & data = ''
       obj_destroy,oJBridgeSession
       return
    endif

    iceAPI = staticClientReference->getInstance('user', 'macs.ncnr.nist.gov')
    iceAPI->waitForConnection
    if strlen(command[0]) ne 0 then begin
       if keyword_set(immediate) then begin
          immCmd = OBJ_NEW('IDLJavaObject$ICE_COMMANDS_IMMEDIATECOMMAND', 'ice.commands.ImmediateCommand', command[0])
          immCmd->run
       endif else begin
          immCmd = OBJ_NEW('IDLJavaObject$ICE_COMMANDS_QUEUEDCOMMAND', 'ice.commands.QueuedCommand', command[0])
          if keyword_set(sync) then begin
             immCmd->runsynchronous 
             if arg_present(data) then data = iceAPI->getlastdata()
          endif else immCmd->runAsynchronous
       endelse
       cmd_success  = immcmd->IsSuccess()
       cmd_error    = immCmd->IsError()
       cmd_response = immCmd->getResponse()
       obj_destroy,[iceAPI,immCmd,staticClientReference,oJBridgeSession]
    endif else begin
       if arg_present(data) then data = iceAPI->getlastdata()
       obj_destroy,[iceAPI,staticClientReference,oJBridgeSession]
    endelse
    
    if n_elements(data) eq 1 then data = strsplit(data,string(10b),/extract)  ;convert from a single string to a string array, ecah element corresponds to a line
end

pro dm_setblades,nblade
    dm_check_parm,nblade,name='blade number',/number,/odd
    for ii=1,11-(nblade+1)/2 do  dm_move,'monblade0'+dm_to_string(ii,/int), 15
    for ii=11+(nblade+1)/2,21 do dm_move,'monblade'+dm_to_string(ii,/int), -15
end

; calculate the slits given beta1 beta2 (degrees) dmbt (mm) and hsamp (mm)
pro dm_setslits,beta1,beta2,dmbt,hsamp,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_setslits calculates the slits given beta1, beta2 (degrees), dmbt (mm), and hsamp (mm).',$
                 'syntax: dm_setslits, beta1, beta2, dmbt, hsamp, help=help',$
                 sep+'beta1:'+tab+'beta1 in unit of degree, a scalar. beta1 and beta2 can be calculated using beta12 = dm_betas(Ei,dmbt).',$
                 sep+'beta2:'+tab+'beta2 in unit of degree, a scalar.',$
                 sep+'dmbt:'+(keyword_set(printhelp)?tab+tab:tab)+'dmbt in unit of mm, a scalar.',$
                 sep+'hsamp:'+tab+'sample height in unit of mm, a scalar.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    dm_check_parm,beta1,name='beta1',/number
    dm_check_parm,beta2,name='beta2',/number
    dm_check_parm,beta1,name='dmbt',/number,/gezero
    dm_check_parm,hsamp,name='hsamp',/number,/gezero
    
    slit1 = dmbt/2+250.*tan(beta1*!Pi/180)
    slit2 = dmbt/2+250.*tan(beta2*!Pi/180)
    h     = hsamp+(400-hsamp)*250./1670.5       ;vertical divergence +/-6.8°  (atan(200/1670.5)/!dtor)
    dm_print,'slit w1 = '+dm_to_string(slit1,res=2)+' mm, w2 = '+dm_to_string(slit2,res=2)+' mm'
    dm_print,'slit w = '+dm_to_string(2*max([slit1,slit2]),res=2)+' mm'
    dm_print,'slit h = '+dm_to_string(h,res=2)+' mm'
end

pro dm_spinflip,state,initstate=initstate,device=device,open=open,close=close,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_spinflip sends the spin flip command to 3He polarizer to achieve the desired state. No action will be taken if the desired state is the same as the current state.',$
                 'Use dm_spinflip without any parameter or keyword to force a flip. Use !spinstate to access the current spin state.',$
                 'syntax: dm_spinflip, state, initstate=initstate, device=device, open=open, close=close, help=help',$
                 sep+'state:'+tab+'"nsf" or "sf", a string. If the state is the same as the current state, no action is taken. If absent, current state will be flipped.',$
                 sep+'initstate:'+tab+'The current spin polarization state, "nsf" or "sf", a string. If this keyword is absent, the last recorded state is assumed as the current state.',$
                 sep+'device:'+tab+'The spinflip device name, either "spinflip" or "spinfliptemp", default is "spinflip".',$
                 sep+'open:'+(keyword_set(printhelp)?tab+tab:tab)+'If set, send the command to open the spinflip device. Device keyword needs to be present, otherwise the default spinflip device is used.',$
                 sep+'close:'+(keyword_set(printhelp)?tab+tab:tab)+'If set, send the command to close the spinflip device. Device keyword needs to be present, otherwise the default spinflip device is used.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    
    if n_elements(device) ne 0 then begin
      if ~strmatch(strtrim(device,2),'spinflip',/fold_case) and ~strmatch(strtrim(device,2),'spinfliptemp',/fold_case) then message,'device must be either "spinflip" or "spinfliptemp".'
    endif else device = 'spinflip'
    
    if keyword_set(open) then begin
       dm_sendcom,'ifopen '+device
       return
    endif
    
    if keyword_set(close) then begin
       dm_sendcom,'ifclose '+device
       return
    endif
    
    dryrun = dm_macs_sequencer_isdryrun(mesgobj=mesgobj)
    
    if n_elements(initstate) ne 0 then begin
       dm_check_parm,initstate,name='initstate',/string
       if ~stregex(strtrim(initstate,2),'^n?sf$',/fold_case,/boolean) then message, 'initstate must be either "sf" or "nsf".'
       !spinstate = strupcase(strtrim(initstate,2))
    endif else if strlen(!spinstate) eq 0 then begin  ;figure out  the spin state from the current sequence or spin
       dm_check_seqwait,spinstate=spinstate
       !spinstate = spinstate
    endif
    if n_elements(state) ne 0 then dm_check_parm,state,name='state',/string else begin
       state = (['NSF','SF'])[strmatch(!spinstate,'nsf',/fold_case)]
    endelse
    if ~stregex(strtrim(state,2),'^n?sf$',/fold_case,/boolean) then message, 'state must be either "sf" or "nsf".'
    
    if strmatch(strtrim(state,2),!spinstate,/fold_case) then return
    !spinstate = strupcase(strtrim(state,2))
    dm_sendcom,'iftalk '+device+' "spinflip" '
    dm_sendcom,'move flip '+(['A','B'])[strmatch(!spinstate,'sf',/fold_case)]
end

;remove comment lines in command string arrays
;parameter:
;   instr:       the input string array
;keyword:
;   combine:     if set, combine into one string suitable for execute()
function dm_strip_comments,instr,combine=combine
    outstr = strtrim(instr,2)
    space  = '[ '+string(9b)+']'
    ;remove trailing empty strings
    while (strlen(outstr[n_elements(outstr)-1]) eq 0) and (n_elements(outstr) gt 1) do outstr = outstr[0:(n_elements(outstr)-2)]
    ;remove all comment strings lines
    indx = where(~stregex(outstr,'^'+space+'*;',/boolean),count)    
    if count ne 0 then outstr = outstr[indx] else outstr = ''   
    for i=0L,n_elements(outstr)-1 do begin 
        ;remove comment at the end of a line
        if strpos(outstr[i],';') gt 0 then begin
           flag = 0
           for j=0L,strlen(outstr[i])-1 do begin
               letter = strmid(outstr[i],j,1)
               if (letter eq ';') then begin
                  remove = 0b
                  if (flag eq 0) then remove = 1b
                  if (flag eq 1) and (strpos(strmid(outstr[i],j),"'") eq -1) and (strmid(outstr[i],0,1,/reverse) ne '$') then remove = 1b
                  if (flag eq 2) and (strpos(strmid(outstr[i],j),'"') eq -1) and (strmid(outstr[i],0,1,/reverse) ne '$') then remove = 1b
                  if keyword_set(remove) then begin
                     outstr[i] = strmid(outstr[i],0,j)
                     break
                  endif
               endif
               if letter eq '"' then flag = 2-flag
               if (letter eq "'") and (flag ne 2)  then flag = 1-flag
           endfor
           outstr[i] = strtrim(outstr[i],2)
        endif
    endfor
    if (n_elements(outstr) gt 1) and keyword_set(combine) then begin
       for i=1L,n_elements(outstr)-1 do begin
           if strlen(outstr[i]) ne 0 then outstr[0] = outstr[0]+'&'+outstr[i]
       endfor
       outstr = strtrim(outstr[0],2)
       if strmid(outstr,0,1) eq '&' then outstr = strmid(outstr,1,strlen(outstr)-1)     ;remove the first '&' character
       while ((index=strpos(outstr,'$&')) ge 0) do strput,outstr,'  ',index             ;remove possible '$&'
    endif
    if n_elements(outstr) eq 1 then outstr = outstr[0]
    return,outstr
end

;set temperature device staleness
pro dm_tempstaleness,time,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_tempstaleness sets the stale time of the current temperature device.',$
                 'syntax: dm_tempstaleness, time, help=help',$
                 sep+'time:'+tab+'temperature device staleness time in unit of second, a scalar.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    dm_check_parm,time,/number,/positive,name='time'
    tmp = !dm_macsseq_dryrun.dryrun
    !dm_macsseq_dryrun.dryrun = 0b
    dm_sendcom,'device getparent temp',cmd_error=error,cmd_response=device,/immediate,/quiet
    !dm_macsseq_dryrun.dryrun = tmp
    if (error eq 0) and (strlen(device) ne 0) then dm_sendcom,'settempstaleness '+device+' '+dm_to_string(time,/int)
end

pro dm_wait,time,comment=comment,refill=refill,help=help,output=output
    if keyword_set(help) then begin
       tab = string(9b) & sep = '  '
       if ~arg_present(output) then printhelp = 1b
       output = ['dm_wait sends a wait command to the server.',$
                 'syntax: dm_wait, time, comment=comment, help=help',$
                 sep+'time:'+tab+tab+'waiting time in unit of second.',$
                 sep+'comment:'+tab+'a comment string to be shown in the sequence.']
       if keyword_set(printhelp) then for i=0,n_elements(output)-1 do print,output[i]
       return
    endif
    if keyword_set(refill) then begin
       time = 6666
       comment = 'Filling cryogen. Please do not remove the above wait command.'
    endif
    dm_check_parm,time,name='time',/number,/positive
    dm_sendcom,'wait '+dm_to_string(time,/int)
    if n_elements(comment) ne 0 then dm_comment,comment
end

pro dm_macs_a3offset_update,state
    widget_control,state.aTxt,get_value=str_a
    widget_control,state.bTxt,get_value=str_b
    widget_control,state.kTxt,get_value=str_k
    a3 = dm_to_number(str_a)
    bt = dm_to_number(str_b)
    kd = dm_to_number(str_k)
    if total(finite([a3,bt,kd],/nan)) ne 0 then offset=!values.f_nan $
    else offset = a3-bt-kd
    widget_control,state.oTxt,set_value=dm_to_string(offset)
    if finite(offset,/nan) then return
    widget_control,state.k1Txt,get_value=str_k1
    widget_control,state.k2Txt,get_value=str_k2 
    k1 = dm_to_number(str_k1)
    k2 = dm_to_number(str_k2)
    if state.aorb then begin  ;basesampletheta->A3
       as = [state.a1Txt,state.a2Txt]
       bs = [state.b1Txt,state.b2Txt]
       sn = 1.0
    endif else begin          ;A3->basesampletheta
       as = [state.b1Txt,state.b2Txt]
       bs = [state.a1Txt,state.a2Txt]
       sn = -1.0
    endelse
    bv = fltarr(2)
    for i=0,1 do begin
        widget_control,as[i],set_value=''
        widget_control,bs[i],get_value=str_b
        bv[i] = dm_to_number(str_b)
    endfor
    if total(finite([k1,k2])) eq 0 then return
    if total(finite(bv,/nan)) ne 0 then return  
    av = [!values.f_infinity,-!values.f_infinity]
    if finite(k1) then begin
       av[0] = (av[0])<(min(bv)+(k1+offset)*sn)
       av[1] = (av[1])>(max(bv)+(k1+offset)*sn)
    endif
    if finite(k2) then begin
       av[0] = (av[0])<(min(bv)+(k2+offset)*sn)
       av[1] = (av[1])>(max(bv)+(k2+offset)*sn)
    endif
    for i=0,1 do widget_control,as[i],set_value=dm_to_string(av[i])
end

pro dm_macs_a3offset_event,event
    widget_control,/hourglass
    widget_control,event.handler,get_uvalue=state

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       if stregex(!error_state.msg,'Ei is greater than the',/boolean) then widget_control,state.eiTxt,set_value=''
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       return
    end
    
    case event.id of   
         state.a1Txt:    state.aorb = 0b
         state.a2Txt:    state.aorb = 0b
         state.b1Txt:    state.aorb = 1b
         state.b2Txt:    state.aorb = 1b
         state.eiTxt:    begin
                         widget_control,state.eiTxt,get_value=str_e
                         Ei = dm_to_number(str_e)
                         if ~finite(Ei) then return
                         kran = dm_getkidneyrange(Ei,/notcheckeimin)
                         widget_control,state.k1Txt,set_value=dm_to_string(kran[0],res=3)
                         widget_control,state.k2Txt,set_value=dm_to_string(kran[1],res=3)
                         end
         state.riceBut:  begin
                         a3 = dm_dev_pos('a3',/number)
                         bt = dm_dev_pos('basesampletheta',/number)
                         kd = dm_dev_pos('kidney',/number)
                         widget_control,state.aTxt,set_value=dm_to_string(a3)
                         widget_control,state.bTxt,set_value=dm_to_string(bt)
                         widget_control,state.kTxt,set_value=dm_to_string(kd)
                         end
         state.clearBut: begin
                         ids = [state.aTxt,state.bTxt,state.kTxt,state.oTxt,state.a1Txt,state.a2Txt,state.b1Txt,state.b2Txt,state.k1Txt,state.k2Txt,state.eiTxt]
                         for i=0,n_elements(ids)-1 do widget_control,ids[i],set_value=''
                         return
                         end
         state.exitBut:  begin
                         widget_control,event.handler,/destroy
                         return
                         end
         else:           
    endcase
    dm_macs_a3offset_update,state
    widget_control,event.handler,set_uvalue=state
end

pro dm_macs_a3offset_Exit,tlb
    widget_control,tlb,/destroy
end

;calculates the offset angle between A3 and Basesampletheta
pro dm_macs_a3offset,event,allow_ice=allow_ice
    state={group_leader:    0L, $   ;group leader
           tlb:             0L, $   ;top level base
           aTxt:            0L, $   ;A3 text box, editable
           oTxt:            0L, $   ;offset angle text box
           bTxt:            0L, $   ;basesampletheta text box
           kTxt:            0L, $   ;kidney text box
           a1Txt:           0L, $   ;A3 from 
           a2Txt:           0L, $   ;A3 to
           b1Txt:           0L, $   ;basesampletheta from
           b2Txt:           0L, $   ;basesampletheta to
           k1Txt:           0L, $   ;kidney from
           k2Txt:           0L, $   ;kidney to
           eiTxt:           0L, $   ;Ei
           aorb:            0b, $   ;A3 or basesampletheta flag
           riceBut:         0L, $   ;read ice button
           clearBut:        0L, $   ;clear button
           exitBut:         0L  $   ;exit button
          }
    registerName = 'dm_macs_a3offset'
    if xregistered(registerName) then begin ;only allow one copy running
       id = LookupManagedWidget(registername)
       widget_control,id,iconify=0
       return
    endif
    
    if n_elements(event) ne 0 then state.group_leader = event.top
    state.tlb      = widget_base(title='Offset Angle among A3, Basesampletheta, and Kidney',/col,kill_notify='dm_macs_a3offset_Exit',tlb_frame_attr=1,group_leader=state.group_leader)
    degree         = ' ('+string('b0'XB)+')' & margin = 6
    row            = widget_base(state.tlb,/row,xpad=0,space=0,ypad=margin)
    void           = widget_label(row,value='',scr_xsize=margin)
    void           = widget_label(row,value='Basesampletheta'+degree,/align_center)
    tmp            = widget_info(void,/geometry)
    xsize          = 120>(tmp.scr_xsize/2*2+10) 
    widget_control,void,scr_xsize=xsize,set_value='A3'+degree
    void           = widget_label(row,value='',scr_xsize=margin)
    void           = widget_label(row,value='Basesampletheta'+degree,/align_center,scr_xsize=xsize)
    void           = widget_label(row,value='',scr_xsize=margin)
    void           = widget_label(row,value='Kidney'+degree,/align_center,scr_xsize=xsize)
    void           = widget_label(row,value='',scr_xsize=margin)
    void           = widget_label(row,value='Offset Angle'+degree,/align_center,scr_xsize=xsize)
    void           = widget_label(row,value='',scr_xsize=margin)
    row            = widget_base(state.tlb,/row,xpad=0,space=0)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.aTxt     = widget_text(row,/editable,/all_events,scr_xsize=xsize,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.bTxt     = widget_text(row,/editable,/all_events,scr_xsize=xsize,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.kTxt     = widget_text(row,/editable,/all_events,scr_xsize=xsize,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.oTxt     = widget_text(row,scr_xsize=xsize)
    row            = widget_base(state.tlb,/row,xpad=0,space=0)
    void           = widget_label(row,value='',scr_xsize=margin)
    for i=0,2 do begin
        void       = widget_label(row,value='from:',/align_center,scr_xsize=xsize/2)
        void       = widget_label(row,value='to:',/align_center,scr_xsize=xsize/2)
        void       = widget_label(row,value='',scr_xsize=margin)
    endfor
    void           = widget_label(row,value='Ei (meV)',scr_xsize=xsize,/align_center)
    row            = widget_base(state.tlb,/row,xpad=0,space=0)
    void           = widget_label(row,value='',scr_xsize=margin)       
    state.a1Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',/align_center,scr_xsize=2)
    state.a2Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.b1Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',/align_center,scr_xsize=2)
    state.b2Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.k1Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',/align_center,scr_xsize=2)
    state.k2Txt    = widget_text(row,/editable,/all_events,scr_xsize=xsize/2-1,tab_mode=1)
    void           = widget_label(row,value='',scr_xsize=margin)
    state.eiTxt    = widget_text(row,/editable,/all_events,scr_xsize=xsize,tab_mode=1)
    row            = widget_base(state.tlb,/row,/align_center) 
    void           = widget_label(row,value='A3 = Basesampletheta + Kidney + Offset')
    row            = widget_base(state.tlb,/row,/align_center)
    void           = widget_label(row,value='Kidney lower and upper limits for a given Ei are 1.5'+string('b0'xb)+' within the software limit.')
    row            = widget_base(state.tlb,/row,/align_center)
    ok             = dm_getkidneyrange(5,eimat=eimat)
    void           = widget_label(row,value='Ei should be in the range of ['+dm_to_string(min(eimat))+', '+dm_to_string(max(eimat))+'] meV.')
    row            = widget_base(state.tlb,/row,/align_center,ypad=margin) 
    if keyword_set(allow_ice) then begin
       state.riceBut = widget_button(row,value='  Read ICE  ')
       if dm_to_number(!version.release) ge 5.6 then widget_control,state.riceBut,tooltip='Read A3, Basesampletheta, and Kidney angles from the ICE server. Be aware that the motors might be moving.'
       void        = widget_label(row,value= '',scr_xsize=margin*3)
    endif
    state.clearBut = widget_button(row,value='  Clear  ')
    void           = widget_label(row,value= '',scr_xsize=margin*3)
    state.exitBut  = widget_button(row,value='  Exit  ')
    dm_center_kid,state.tlb,state.group_leader
    widget_control,state.tlb,/realize,set_uvalue=state
    if keyword_set(allow_ice) then begin
       tmp = obj_new('dm_progress',title='Please wait...',message='Reading MACS angles from ICE...',group_leader=state.tlb,/nostop) 
       dm_macs_a3offset_event,{id:state.riceBut,top:state.tlb,handler:state.tlb,select:1b} ;initialize
       obj_destroy,tmp
    endif
    xmanager,registerName,state.tlb,cleanup='dm_macs_a3offset_Exit',/no_block
end

pro dm_macs_kidlimit_event,event
    widget_control,/hourglass
    widget_control,event.handler,get_uvalue=state

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       widget_control,event.handler,set_uvalue=state
       catch,/cancel
       return
    end

    case event.id of
       state.eTxt:     begin
                       widget_control,event.id,get_value=str
                       Ei = dm_to_number(strsplit(str,' ,'+string(9b),/extract))
                       ind0 = where(finite(Ei),count0)
                       if count0 gt 0 then begin
                          ind1 = where(Ei[ind0] gt 0,count1)
                          if count1 gt 0 then begin
                             limits = dm_getkidneyrange(Ei[ind0[ind1]],offset=state.offset,/notcheckeimin)
                             ind2   = where(limits[0,*] lt limits[1,*],count2,complement=ind3,ncomplement=count3)
                             if count2 ne 0 then begin
                                if count3 ne 0 then limits[*,ind3] = !values.f_nan
                                widget_control,state.lTxt,set_value=dm_to_string(limits[0,*],resolution=3,separator=', ')
                                widget_control,state.uTxt,set_value=dm_to_string(limits[1,*],resolution=3,separator=', ')
                                valid = 1b
                             endif
                          endif
                       endif 
                       if ~keyword_set(valid) then begin
                          widget_control,state.lTxt,set_value=''
                          widget_control,state.uTxt,set_value=''
                       endif
                       end
       state.oTxt:     begin
                       old_offset = state.offset
                       widget_control,event.id,get_value=str
                       state.offset = dm_to_number(str)
                       if finite(state.offset,/nan) then state.offset = 0.0
                       widget_control,event.handler,set_uvalue=state
                       if state.offset ne old_offset then begin
                          if obj_valid(state.plotobj) then begin
                             event.id = state.plotBut
                             dm_macs_kidlimit_event,event
                          endif
                          event.id = state.eTxt
                          dm_macs_kidlimit_event,event
                          widget_control,state.oTxt,/input_focus
                       endif
                       end
       state.plotBut:  begin
                       if ~obj_valid(state.plotobj) then state.plotobj = obj_new('dm_plot',legdfsize=10,background=state.bgcolor,wtit='MACS Kidney Limits')
                       state.plotobj->setproperty,xdat=(*state.kidlimit)[*,0],ydat=(*state.kidlimit)[*,1]-state.offset,/no_copy,legend='lower limit',color='green',xran=[2,17],yran=[-60,55],$
                            psym='filled circle',xtit='E!di!n (meV)',ytit='Kidney Limits (\deg)',linestyle='solid',/nodraw,hidelegend=0,legdpos=[0.6,0.9],/legdshowoutline,/showcursorpos
                       state.plotobj->add_plot,(*state.kidlimit)[*,0],(*state.kidlimit)[*,2]-state.offset,legend='upper limit',color='red',psym='filled circle',linestyle='solid'
                       state.plotobj->draw
                       end
       state.exitBut:  begin
                       widget_control,event.handler,/destroy
                       return
                       end
       else:
    endcase
    
    widget_control,event.handler,set_uvalue=state
end

pro dm_macs_kidlimit_Exit,tlb
    widget_control,tlb,get_uvalue=state
    ptr_free,state.kidlimit
    if obj_valid(state.plotobj) then obj_destroy,state.plotobj
    widget_control,tlb,/destroy
end

;macs kidney limit
pro dm_macs_kidlimit,event,bgcolor=bgcolor
    state={group_leader:    0L, $   ;group leader
           tlb:             0L, $   ;top level base
           eTxt:            0L, $   ;E text box, editable
           oTxt:            0L, $   ;offset angle text box
           offset:          0e, $   ;offset angle value
           lTxt:            0L, $   ;lower limit text box
           uTxt:            0L, $   ;upper limit text box
           plotBut:         0L, $   ;plot limits button
           exitBut:         0L, $   ;exit button
           kidlimit: ptr_new(), $   ;kidney limits
           bgcolor:         '', $   ;plot window background color
           plotobj:  obj_new()  $   ;dm_plot
          }
    registerName = 'dm_macs_kidlimit'
    if xregistered(registerName) then begin ;only allow one copy running
       id = LookupManagedWidget(registername)
       widget_control,id,iconify=0
       return
    endif
    
    if n_elements(bgcolor) ne 0 then state.bgcolor = bgcolor[0]
    
    tmp = dm_check_macskidney(0,5,limits=limits,offset=offset)
    state.kidlimit = ptr_new(limits)
    state.offset   = offset
    
    if n_elements(event) ne 0 then state.group_leader = event.top
    state.tlb     = widget_base(title='Kidney Angle Limits',/col,kill_notify='dm_macs_kidlimit_Exit',tlb_frame_attr=1,group_leader=state.group_leader)
    row1          = widget_base(state.tlb,/row,/grid_layout)
    col1          = widget_base(row1,/col,/grid_layout)
    col2          = widget_base(row1,/col,/grid_layout)
    col3          = widget_base(row1,/col,/grid_layout)
    col4          = widget_base(row1,/col,/grid_layout)
    void          = widget_label(col1,value='Ei (meV)',/align_center)
    state.eTxt    = widget_text(col1,/editable,/all_events)
    void          = widget_label(col2,value='Lower Limit ('+string('b0'XB)+')',/align_center)
    state.lTxt    = widget_text(col2)
    void          = widget_label(col3,value='Upper Limit ('+string('b0'XB)+')',/align_center)
    state.uTxt    = widget_text(col3)
    void          = widget_label(col4,value='Offset Angle ('+string('b0'XB)+')',/align_center)
    state.oTxt    = widget_text(col4,/editable,/all_events,value=dm_to_string(state.offset,resolution=3))
    row2          = widget_base(state.tlb,/row,/align_center) 
    void          = widget_label(row2,value='The limits are 1.5 degrees within the actual software angle limits calculated from hardware limits using the offset angle.')
    row3          = widget_base(state.tlb,/row,/align_center)
    ok            = dm_getkidneyrange(5,eimat=eimat)
    void          = widget_label(row3,value='Ei should be in the range of ['+dm_to_string(min(eimat))+', '+dm_to_string(max(eimat))+'] meV.')
    row4          = widget_base(state.tlb,/row,/align_center)
    state.plotBut = widget_button(row4,value='  Plot  ')
    void          = widget_label(row4,value= '   ')
    state.exitBut = widget_button(row4,value='  Exit  ')
    dm_center_kid,state.tlb,state.group_leader
    widget_control,state.tlb,/realize
    widget_control,state.tlb,set_uvalue=state
    xmanager,registerName,state.tlb,cleanup='dm_macs_kidlimit_Exit',/no_block
end

pro dm_macs_kidoptimizer_Exit,tlb
    widget_control,tlb,get_uvalue=state
    if widget_info(state.group_leader,/valid_id) then begin
       widget_control,state.group_leader,get_uvalue=topobj
       if obj_isa(topobj,'DM_MACS_SEQUENCER') then topobj->setproperty,kidoptstr=state.vals
    endif
    if obj_valid(state.pobj) then obj_destroy,state.pobj
    widget_control,tlb,/destroy
end

pro dm_macs_kidoptimizer_event,event
    widget_control,/hourglass
    widget_control,event.handler,get_uvalue=state

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       widget_control,event.handler,set_uvalue=state
       catch,/cancel
       return
    end

    case event.id of
         state.optmBut: dm_macs_kidoptimizer_opt,state
         state.saveBut: state.pobj->saveas,'png'
         state.prntBut: state.pobj->saveas,'printer'
         state.exitBut: begin
                widget_control,event.handler,/destroy
                return
                end
         else:  if widget_info(event.id,/type) eq 3 then begin  ;text boxes
                if event.type gt 2 then break
                id = where([state.eTxt,state.lTxt,state.uTxt,state.sTxt,state.bTxt] eq event.id,count)
                if count eq 0 then break
                widget_control,event.id,get_value=str
                state.vals[id] = str
                widget_control,state.optmBut,sensitive=(finite(dm_to_number(state.vals[0])) and finite(dm_to_number(state.vals[3])))
                if finite(dm_to_number(state.vals[0])) and finite(dm_to_number(state.vals[3])) then begin
                   if event.type eq 0 then if event.ch eq 10b then recal = 1b   ;automatically recalculate if enter is pressed
                   if id eq 3 then recal = 1b                                   ;automatically recalculate if step is changed
                   if keyword_set(recal) then dm_macs_kidoptimizer_opt,state 
                endif
                widget_control,event.id,/input_focus
                end
    endcase

    widget_control,event.handler,set_uvalue=state
end

pro dm_macs_kidoptimizer_opt,state
    vals = dm_to_number(state.vals)
    if finite(vals[0],/nan) or finite(vals[3],/nan) or (vals[0] le 0) or (vals[3] eq 0) then begin
       widget_control,state.optmBut,sensitive=0
       return
    endif
    kidran = vals[1:2] & kidstep = abs(vals[3])
    kidlim = dm_getkidneyrange(vals[0],offset=offset)
    if finite(vals[4]) then backlash = vals[4] else backlash = 2.0
    kidlim[0] = kidlim[0]+backlash
    if (~lmgr(/vm)) and stregex(state.vals[1],'ul|ll|cn',/boolean,/fold_case) then kidran[0] = dm_calckidney(state.vals[1],kidlim)
    if (~lmgr(/vm)) and stregex(state.vals[2],'ul|ll|cn',/boolean,/fold_case) then kidran[1] = dm_calckidney(state.vals[2],kidlim)
    if finite(kidran[0],/nan) then kidran[0] = kidlim[0]
    if finite(kidran[1],/nan) then kidran[1] = kidlim[1]
    kidran = (kidlim[0])>(kidran)<(kidlim[1])
    steps = dm_optimizekidstep(kidstep,kidran,/all_step,eq_step=eq_step)
    ;list around 10 steps
    tmp = eq_step[uniq(eq_step,sort(eq_step))]
    dif = min(abs(tmp-kidstep),ind0) & ntmp = n_elements(tmp) & count = 0 & edge = 2
    while (count lt 10) and (edge lt 5) do begin
       ind = where((eq_step ge tmp[0>(ind0-edge)]) and (eq_step le tmp[(ind0+edge)<(ntmp-1)]),count)
       edge = edge+1
    endwhile
    eq_step = eq_step[ind] & steps = steps[ind]
    step1 = dm_optimizekidstep(kidstep,kidran,eq_step=eq_step1,n_step=n_step1)
    step2 = dm_optimizekidstep(kidstep,kidran,eq_step=eq_step2,/max_step,n_step=n_step2,det_span=det_span)
    step3 = dm_optimizekidstep_old(kidstep,kidran,eq_step=eq_step3)
    steps = [steps,step3] & eq_step = [eq_step,eq_step3]
    tmp   = round(eq_step*1d3)/1d1+round(steps*1d3)/1d4
    tmp   = uniq(tmp,sort(tmp))
    eq_step = eq_step[tmp]
    steps = steps[tmp]
    x0    = (det_span+0.001)*findgen(20)-76   ;use 8.001 instead of 8 to shift the points 
    state.pobj->erase,/nodraw
    cols = state.pobj->getcolor(/list) & cols = cols[where(cols ne 'yellow')] & ncol = n_elements(cols)
    syms = ['filled circle','circle'] & nsym = n_elements(syms)
    ind  = where(abs(steps-kidstep) lt 1e-5,count)
    if count eq 0 then begin  ;show the target step also
       steps   = [kidstep,steps]
       eq_step = [!values.f_nan,eq_step]
    endif
    ind = where(abs(steps-step1) lt 1e-5,count)
    if count ne 0 then begin  ;optimized step will be shown in red diamond
       cols[where(cols eq 'red')] = cols[ind[0] mod ncol]
       cols[ind[0] mod ncol] = 'red'
    endif
    if state.idl_version ge 8.2 then begin ;\checkmark only available after IDL version 8.2
       checkmark = '\checkmark' & edge = [-8,15]
    endif else begin
       checkmark = '*' & edge = [-8,8]
    endelse
    xmin = 0 & xmax = 0
    for i=0,n_elements(steps)-1 do begin
        if (abs(steps[i]-step1) lt 1e-5) then begin
           steps[i] = step1 & eq_step[i] = eq_step1
        endif
        nx = floor(abs(kidran[1]-kidran[0])/steps[i])+1
        x1 = min(kidran)+findgen(nx)*steps[i]
        xs = x1+x0[0]
        for j=1,19 do xs = [xs,x1+x0[j]]
        legend = 'step='+dm_to_string(steps[i])+(finite(eq_step[i])?', equv. step='+dm_to_string(eq_step[i]):'')
        psym = syms[i mod nsym]
        if (abs(steps[i]-step3) lt 1e-5) then legend = legend+' (old)'
        if (abs(steps[i]-step1) lt 1e-5) then begin
           legend = legend+' '+checkmark
           psym = 'filled diamond'
           if n_elements(cornertxt) eq 0 then cornertxt = checkmark+' optimized kidney step='+dm_to_string(step1)+' ('+dm_to_string(abs(kidran[0]-kidran[1])/step1+1,/int)+' kids)'
           if state.idl_version ge 8.2 then state.pobj->add_text,checkmark,max(xs)+8,-0.1*(n_elements(steps)/2.0-i),fontsize=20,normalize=0
        endif else if (abs(steps[i]-step2) lt 1e-5) then begin
           legend = legend+' '+checkmark+'*'
           psym = 'filled triangle'
           cornertxt = checkmark+'   optimized kidney step='+dm_to_string(step1)+' ('+dm_to_string(n_step1)+' kids)'
           cornertxt = cornertxt+'!c'+checkmark+'* with max_step keyword set='+dm_to_string(step2)+' ('+dm_to_string(n_step2)+' kids)'
        endif
        state.pobj->add_plot,xs,fltarr(n_elements(xs))-0.1*(n_elements(steps)/2.0-i),color=cols[i mod ncol],psym=psym,legend=legend,linestyle='no line',symsize=0.005
        tmpmin = min(xs,max=tmpmax)
        xmin = (xmin)<(tmpmin) & xmax = (xmax)>(tmpmax)
    endfor
    state.pobj->setproperty,yran=[-0.05,0.05]*n_elements(steps)+[-0.1,0],xran=[xmin,xmax]+edge,cornertxt=cornertxt,/legdshowfill,ytickmajor=0,ytickminor=0,$
                title='Ei='+dm_to_string(vals[0])+' meV, target step='+dm_to_string(kidstep)+', kidney range=['+dm_to_string(kidran,sep=', ',res=3)+']'  
    widget_control,state.saveBut,/sensitive
    widget_control,state.prntBut,/sensitive      
end

;macs kidney step optimization
pro dm_macs_kidoptimizer,event,bgcolor=bgcolor
    state={group_leader:    0L, $   ;group leader
           tlb:             0L, $   ;top level base
           eTxt:            0L, $   ;E text box
           lTxt:            0L, $   ;kidney from text box
           uTxt:            0L, $   ;kidney to text box
           sTxt:            0L, $   ;step text box
           bTxt:            0L, $   ;kidney backlash text box
           vals:     strarr(5), $   ;save e,l,u,s,b values
           optmBut:         0L, $   ;optimize button
           saveBut:         0L, $   ;save button
           prntBut:         0L, $   ;print button
           exitBut:         0L, $   ;exit button
           idl_version:     0e, $   ;save idl version 
           pobj:     obj_new()  $   ;dm_plot
          }
    registerName = 'dm_macs_kidoptimizer'
    if xregistered(registerName) then begin ;only allow one copy running
       id = LookupManagedWidget(registername)
       widget_control,id,iconify=0
       return
    endif
    bmpfiles = strarr(8)
    if n_elements(event) ne 0 then begin
       state.group_leader = event.top
       widget_control,event.top,get_uvalue=topobj
       if obj_isa(topobj,'DM_MACS_SEQUENCER') then begin
          topobj->getproperty,kidoptstr=kidoptstr,bmpfiles=bmpfiles
          state.vals = kidoptstr
       endif
    endif
    state.idl_version = dm_to_number(!version.release)
    state.tlb     = widget_base(title='Kidney Step Optimization',mbar=mbar,/col,kill_notify='dm_macs_kidoptimizer_Exit',tlb_frame_attr=1,map=0,group_leader=state.group_leader)
    filemenu      = widget_button(mbar,value='File',/menu)
    state.saveBut = dm_widget_button(filemenu,value='Save...',sensitive=0,imagefile=bmpfiles[1],/notchecked)
    state.prntBut = widget_button(filemenu,value='Print...',sensitive=0) 
    state.exitBut = widget_button(filemenu,value='Exit',/separator)
    row1          = widget_base(state.tlb,/row)
    void          = widget_label(row1,value='Ei (meV):')
    state.eTxt    = widget_text(row1,value=state.vals[0],/editable,/all_events,scr_xsize=80)
    void          = widget_label(row1,value=' ',scr_xsize=10)
    void          = widget_label(row1,value='kidney from:',/align_center)
    state.lTxt    = widget_text(row1,value=state.vals[1],/editable,/all_events,scr_xsize=80)
    void          = widget_label(row1,value=' ',scr_xsize=10)
    void          = widget_label(row1,value='kidney to:',/align_center)
    state.uTxt    = widget_text(row1,value=state.vals[2],/editable,/all_events,scr_xsize=80)
    void          = widget_label(row1,value=' ',scr_xsize=10)
    void          = widget_label(row1,value='step:',/align_center)
    state.sTxt    = widget_text(row1,value=state.vals[3],/editable,/all_events,scr_xsize=80)
    void          = widget_label(row1,value=' ',scr_xsize=10)
    void          = widget_label(row1,value='backlash:',/align_center)
    state.bTxt    = widget_text(row1,value=state.vals[4],/editable,/all_events,scr_xsize=80)
    void = widget_label(row1,value=' ',scr_xsize=20)
    state.optmBut = widget_button(row1,value='Optimize',sensitive=(finite(dm_to_number(state.vals[0])) and finite(dm_to_number(state.vals[3]))),scr_xsize=100,frame=1)
    info = ['Leave kidney range blank for full allowable kidney range. If backlash is not specified, 2 degrees is used.','Use UL and LL for the upper and lower kidney limit, and CN for the mean kidney center position at specified energy. For example: ul-5, ll+5, cn-4.']
    for i=0,n_elements(info)-1 do void = widget_label(state.tlb,value=info[i],/align_left)
    state.pobj    = obj_new('dm_plot',xtit='A4 (\deg)',ytit='',/legdshowoutline,legdpos=[0.03,0.95],legdfsize=10,fsize=15,ctxtfsize=10,xsize=850,ysize=500,/noparentevent,background=bgcolor,widgetbase=state.tlb,/compound,group_leader=state.tlb)
             dm_macs_kidoptimizer_opt,state   
    dm_center_kid,state.tlb,state.group_leader
    widget_control,state.tlb,/update,/realize,/map
    widget_control,state.tlb,set_uvalue=state
    xmanager,registerName,state.tlb,cleanup='dm_macs_kidoptimizer_Exit',/no_block
end

;check if the computer ip address is authorized to send commands to ICE server
function dm_macs_sequencer_authorizeip
    ip_list = ['I8e9wjs4amAW86riELnUIw','IseUpmU4Ub4B86TfGrkGqg','2Me9lrI4algE86rcELnUOA','rce9IN44am6t86pSI7nUOQ','gMfl0E84vg0o846LnbkB8g','p8e9PuE4agq986qcILnUhQ']+'==' ;Q,M1,U,R,M2,U encrypted to protect the IP-addresses
    ind = where(ip_list eq dm_ip_encrypt(dm_ip_address()),count)
    return,(count eq 1)
end

;get dryrun status
function dm_macs_sequencer_isdryrun,mesgobj=mesgobj,isexecute=isexecute,group_leader=group_leader
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       dryrun = !dm_macsseq_dryrun.dryrun
       isexecute = !dm_macsseq_dryrun.execute 
       if obj_valid(!dm_macsseq_dryrun.message) then mesgobj = !dm_macsseq_dryrun.message
       if arg_present(group_leader) and n_elements(group_leader) eq 0 then begin
          tmp = obj_valid()
          ind = where(obj_isa(tmp,'DM_MACS_SEQUENCER'))
          tmp[ind[0]]->getproperty,tlb=group_leader
       endif 
    endif else dryrun = 0b
    return,dryrun
end

pro dm_macs_sequencer_Exit,tlb
    widget_control,tlb,get_uvalue=self
    if obj_valid(self) then obj_destroy,self
end

;widget event handler
pro dm_macs_sequencer_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;construct commands menu
pro dm_macs_sequencer::commands,menuid,_extra=_extra
    cmenu = widget_button(menuid,value='Commands',/menu,_extra=_extra)
    tmenu = widget_button(menuid,value='Templates',/menu,_extra=_extra)
    void  = widget_button(menuid,value='From File...',/separator,uname='ins_file',_extra=_extra)
    ;commands
    void  = widget_button(cmenu,value='dm_a3can',             uname='ins_com_a3scan')
    void  = widget_button(cmenu,value='dm_comment',           uname='ins_com_comment')
    void  = widget_button(cmenu,value='dm_dev_pos()',         uname='ins_com_dev_pos')
    void  = widget_button(cmenu,value='dm_findpeak',          uname='ins_com_findpeak')
    void  = widget_button(cmenu,value='dm_getkidneyrange()',  uname='ins_com_getkidneyrange')
    void  = widget_button(cmenu,value='dm_getlastscanname()', uname='ins_com_getlastscanname')
    void  = widget_button(cmenu,value='dm_gobetas',           uname='ins_com_gobetas')
    void  = widget_button(cmenu,value='dm_kidscan',           uname='ins_com_kidscan')
    void  = widget_button(cmenu,value='dm_move',              uname='ins_com_move')
    void  = widget_button(cmenu,value='dm_runscan',           uname='ins_com_runscan')
    void  = widget_button(cmenu,value='dm_runtab',            uname='ins_com_runtab')
    void  = widget_button(cmenu,value='dm_sendcom',           uname='ins_com_sendcom')
    void  = widget_button(cmenu,value='dm_spinflip',          uname='ins_com_spinflip')
    void  = widget_button(cmenu,value='dm_tempstaleness',     uname='ins_com_tempstaleness')
    void  = widget_button(cmenu,value='dm_wait',              uname='ins_com_wait')
    lmenu = widget_button(cmenu,value='collimator',/menu,/separator)
    amenu = widget_button(lmenu,value="a (60')",/menu)
    bmenu = widget_button(lmenu,value="b (40')",/menu)
    void  = widget_button(amenu,value='in',                   uname='ins_com_acol_in')
    void  = widget_button(amenu,value='out',                  uname='ins_com_acol_out')
    void  = widget_button(bmenu,value='in',                   uname='ins_com_bcol_in')
    void  = widget_button(bmenu,value='out',                  uname='ins_com_bcol_out')
    fmenu = widget_button(cmenu,value='filter',/menu)
    xmenu = widget_button(fmenu,value='cfx',/menu)
    bmenu = widget_button(xmenu,value='Be',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_cfx_be_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_cfx_be_out')
    bmenu = widget_button(xmenu,value='HOPG',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_cfx_hopg_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_cfx_hopg_out')
    bmenu = widget_button(xmenu,value='MgF2',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_cfx_mgf2_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_cfx_mgf2_out')
    mmenu = widget_button(fmenu,value='mcfx',/menu)
    void  = widget_button(mmenu,value='BeO',                  uname='ins_com_mcfx_beo')
    void  = widget_button(mmenu,value='Be',                   uname='ins_com_mcfx_be')
    void  = widget_button(mmenu,value='HOPG',                 uname='ins_com_mcfx_hopg')
    xmenu = widget_button(fmenu,value='cfx  (MACS computer)',/menu,/separator)
    bmenu = widget_button(xmenu,value='Be',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_macs_be_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_macs_be_out')
    bmenu = widget_button(xmenu,value='HOPG',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_macs_hopg_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_macs_hopg_out')
    bmenu = widget_button(xmenu,value='MgF2',/menu)
    void  = widget_button(bmenu,value='In',                   uname='ins_com_macs_mgf2_in')
    void  = widget_button(bmenu,value='Out',                  uname='ins_com_macs_mgf2_out')
    fmenu = widget_button(cmenu,value='focusing',/menu)
    void  = widget_button(fmenu,value='flat',                 uname='ins_com_hvf_flat')
    void  = widget_button(fmenu,value='doubly focusing',      uname='ins_com_hvf_dfoc')
    hmenu = widget_button(fmenu,value='horizontal',/menu,/separator)
    vmenu = widget_button(fmenu,value='vertical',/menu)
    void  = widget_button(hmenu,value='flat',                 uname='ins_com_hf_flat')
    void  = widget_button(hmenu,value='energy',               uname='ins_com_hf_energy')
    void  = widget_button(hmenu,value='venetian',             uname='ins_com_hf_venetian')
    void  = widget_button(hmenu,value='point',                uname='ins_com_hf_point')
    void  = widget_button(vmenu,value='flat',                 uname='ins_com_vf_flat')
    void  = widget_button(vmenu,value='sagittal',             uname='ins_com_vf_sagittal')
    pmenu = widget_button(cmenu,value='ptai',/menu)
    void  = widget_button(pmenu,value='auto',                 uname='ins_com_ptaiauto')
    void  = widget_button(pmenu,value='manual',               uname='ins_com_ptaimanual')
    smenu = widget_button(cmenu,value='scattermode',/menu)
    void  = widget_button(smenu,value='auto',                 uname='ins_com_smodeauto')
    void  = widget_button(smenu,value='positive',             uname='ins_com_smodepos')
    void  = widget_button(smenu,value='negative',             uname='ins_com_smodeneg')
    smenu = widget_button(cmenu,value='spinflip',/menu)
    void  = widget_button(smenu,value='open',                 uname='ins_com_spinflipopen')
    void  = widget_button(smenu,value='close',                uname='ins_com_spinflipclose')
    ;templates
    void  = widget_button(tmenu,value='align analyzer (A5)',  uname='ins_tem_aligna5')
    void  = widget_button(tmenu,value='align instrument',     uname='ins_tem_aligninstr')
    void  = widget_button(tmenu,value='align monochromator',  uname='ins_tem_alignmono')
    void  = widget_button(tmenu,value='kidney limit scan',    uname='ins_tem_kidlimscan')
    void  = dm_widget_button(tmenu,value='instrument demagnetization',uname='ins_tem_demag',/separator,imagefile=self.bmpfiles[6],/notchecked)
    void  = dm_widget_button(tmenu,value='set slits',         uname='ins_tem_setslit',imagefile=self.bmpfiles[7],/notchecked)
end

;
function dm_macs_sequencer::dryrun,display=display,sequence=sequence,currenttabtitle=currenttabtitle
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if (~exists) then begin
       ok = dm_getkidneyrange(5,eimat=eimat)
       defsysv,'!dm_macsseq_dryrun',{dryrun:1b,execute:0b,sync:0b,sequence:ptr_new(),message:obj_new(),scanname:ptr_new(),nes:0L,nkid:0L,nwait:0L,nspinflip:0L,nspinstate:0L,$
              wid:[0L,0L],ei_max:max(eimat),ei_min:min(eimat),fref:dm_dev_pos('fref',/number)}
    endif else begin
       !dm_macsseq_dryrun.dryrun     = 1b
       !dm_macsseq_dryrun.execute    = 0b
       !dm_macsseq_dryrun.sync       = 0b
       !dm_macsseq_dryrun.nes        = 0
       !dm_macsseq_dryrun.nkid       = 0
       !dm_macsseq_dryrun.nwait      = 0
       !dm_macsseq_dryrun.nspinflip  = 0
       !dm_macsseq_dryrun.nspinstate = 0
       if widget_info(!dm_macsseq_dryrun.wid[0],/valid_id) then begin
          widget_control,!dm_macsseq_dryrun.wid[1],get_value=tmpseq
          if strlen(strtrim(tmpseq[0],2)) ne 0 then widget_control,!dm_macsseq_dryrun.wid[0],/destroy ;do not close the previous display if the first line is an empty line
       endif
       ptr_free,!dm_macsseq_dryrun.sequence,!dm_macsseq_dryrun.scanname
    endelse
    self->getproperty,currentsequence=sequence,currenttabtitle=currenttabtitle
    ;check existing sequence for long waits, and spin states
    dm_check_seqwait,waitcmds=waitcmds,ignore=ignore,spinstate=spinstate,spinflip=spinflip
    !dm_macsseq_dryrun.nspinflip = keyword_set(spinflip)
    !spinstate = spinstate ;reset spinstate for the dryrun
    dm_print,dm_to_string(systime(/sec),/date)+'>   Dry running "'+currenttabtitle+'"'
    ok = execute(dm_strip_comments(sequence,/combine),1,1)
    if ~ok then begin ;figure out on which line the error occurs, and show error message
       dm_check_syntax,errmsg=errmsg,sequence=sequence
       if total(stregex(errmsg,'Ei is greater than the upper limit',/boolean)) ne 0 then errmsg=[errmsg,'','To change the Ei upper limit, set the !dm_macsseq_dryrun.ei_max value.','Note that kidney range might be incorrect if Ei is beyond the current limit.']
       if total(stregex(errmsg,'Ei is less than the lower limit',/boolean)) ne 0 then errmsg=[errmsg,'','To change the Ei lower limit, set the !dm_macsseq_dryrun.ei_min value.','Note that kidney range might be incorrect if Ei is beyond the current limit.']
       error = 1b
    endif else begin
       ;specifically check for ptai and monochromator focus syntax
       name1 = ['ptai','[^ "'+"'v]+focus"]
       name2 = ['PTAI','Monochromator focus']
       sntax = ['(auto|manual|get|move +|print +|, *|" *|'+"' *)ptai( |mode| *=|'|"+'")','action +(horizfocus +(flat|energy)|vertifocus +(flat|sagittal))']
       for i=0,n_elements(name1)-1 do begin
           ok = stregex(sequence,'^[ '+string(9b)+']*dm_.*'+name1[i],/boolean,/fold_case)
           ind = where(ok,count)
           if count ne 0 then begin
              for j=0,count-1 do begin
                  if ~stregex(sequence[ind[j]],sntax[i],/boolean,/fold_case) then begin
                     if n_elements(errmsg) eq 0 then errmsg = name2[i]+' syntax error, line '+dm_to_string(ind[j]+1)+': '+sequence[ind[j]] $
                     else errmsg = [errmsg,name2[i]+' syntax error, line '+dm_to_string(ind[j]+1)+': '+sequence[ind[j]]]
                     error = 1b
                  endif
              endfor
           endif
       endfor
    endelse
    !dm_macsseq_dryrun.dryrun = 0b
    if n_elements(errmsg) ne 0 then ok = dialog_message(errmsg,/error,dialog_parent=self.tlb,/center)
    if keyword_set(error) or (~ptr_valid(!dm_macsseq_dryrun.sequence)) then return, 0
    sequence = *(!dm_macsseq_dryrun.sequence)
    if n_elements(sequence) eq 0 then return,1
    if total(strlen(sequence)) eq 0 then return,1
    if self.addwait and ~(stregex(sequence[0],'^wait +[1-9]+',/boolean,/fold_case)) then sequence = ['wait 6666',sequence]
    if keyword_set(display) then begin
       disptext = sequence
       if n_elements(waitcmds) gt 0 then $
          disptext = ['( Warning: the following wait command'+(['','s'])[n_elements(waitcmds) gt 1]+' in the existing server queue '+(['is','are'])[n_elements(waitcmds) gt 1]+' longer than '+dm_to_string(ignore)+' secs: )',$
                      '(    '+waitcmds+' )','',disptext] 
       if (!dm_macsseq_dryrun.nspinflip gt 0) and (!dm_macsseq_dryrun.nspinstate eq 0) then $
          disptext = ['( Reminder: you can set the spinstate keyword to add spin state to the file name.)','',disptext]
       xdisplayfile,text=disptext,group=self.tlb,title='Compiled MACS Sequence'+(['',' (might be incomplete):'])[!dm_macsseq_dryrun.sync],done_button='Exit',/editable,/grow_to_screen,return_id=wid,wtext=tid,font=self.font[0]
       dm_center_kid,wid,self.tlb,/side
       !dm_macsseq_dryrun.wid = [wid,tid]
    endif
    if (!dm_macsseq_dryrun.nes ne 0) or (!dm_macsseq_dryrun.nkid ne 0) then dm_print,'( total energy number='+dm_to_string(!dm_macsseq_dryrun.nes)+', total kidney number='+dm_to_string(!dm_macsseq_dryrun.nkid)+' )'
    if (!dm_macsseq_dryrun.nwait gt 0) then dm_print,dm_to_string(!dm_macsseq_dryrun.nwait)+' wait command'+(['.','s.'])[!dm_macsseq_dryrun.nwait gt 1]
    if (!dm_macsseq_dryrun.nspinflip gt 0) and (!dm_macsseq_dryrun.nspinstate eq 0) then dm_print,'Set the spinstate keyword to add spin state to the file name.'
    return,1    
end

;object event handler
pro dm_macs_sequencer::event,event
    widget_control,/hourglass

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=self.tlb,!error_state.msg,/error,/center)
       defsysv,'!dm_macsseq_dryrun',exists=exists
       if exists then begin
          if obj_valid(!dm_macsseq_dryrun.message) then obj_destroy,!dm_macsseq_dryrun.message
          !dm_macsseq_dryrun.execute = 0b
       endif
       catch,/cancel
       return
    end
   
    date_str = strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)  ;date string
    uname = widget_info(event.id,/uname)
    type  = tag_names(event,/structure)
    case strlowcase(type) of
        'widget_context':  uname = type
        'widget_kill_request': uname = 'exit'
        else:
    endcase

    case strlowcase(uname) of
         'tlb':     if strlowcase(tag_names(event,/structure)) eq 'widget_base' then begin ;resize event
                    textboxsize = [((event.x-self.xoffset)>(self.xmin)),((event.y-self.yoffset)>(self.ymin))]
                    self->my_widget_control,'statusbox',scr_xsize=self.xoffset+textboxsize[0]
                    self->my_widget_control,(['left','right'])[self.handedness],scr_xsize=textboxsize[0],scr_ysize=textboxsize[1]
                    textboxsize = textboxsize-[self.xmargin,self.ymargin]
                    tabs = widget_info(self.tlb,find_by_uname='tabbase')
                    tabi = widget_info(tabs,/child) 
                    while tabi ne 0 do begin
                        self->my_widget_control,'textbox',tabi,scr_xsize=textboxsize[0],scr_ysize=textboxsize[1]
                        tabi = widget_info(tabi,/sibling)
                    endwhile
                    widget_control,self.tlb,/update
                    sendevent = {dm_macs_sequencer_size,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,TEXTBOXSIZE:textboxsize,HANDEDNESS:self.handedness}
                    if widget_info(self.group_leader,/valid_id) ne 0 then widget_control,self.group_leader,send_event=sendevent,/no_copy  ;allow mslice to remember the sequencer geometry
                    endif     
         'tabbase': self->reset_button,/forcedryrun
         'widget_context': begin
                    self->my_widget_control,'contextmenu',event.id,/destroy
                    ;context sensitive menu
                    uname  = widget_info(event.id,/uname)
                    menuid = widget_base(event.id,/context_menu,uname='contextmenu')
                    tabs   = widget_info(self.tlb,find_by_uname='tabbase')
                    ntab   = widget_info(tabs,/tab_number)
                    widget_control,event.id,get_value=sequence
                    tmp_value = strtrim(sequence,2)
                    void = widget_button(menuid,value='Insert Commands',/menu)
                    self->commands,void
                    void = dm_widget_button(menuid,value='Add Wait',uname='addwait',set_button=self.addwait,/separator)
                    void = dm_widget_button(menuid,value='Autocomplete',uname='autocomplete',set_button=self.autocomplete)
                    void = dm_widget_button(menuid,value='Toggle Comment',uname='togglecomment',imagefile=self.bmpfiles[5],/notchecked,/separator)
                    if self.idl_version ge 8.3 then begin  ;clipboard class only implemented in IDL 8.3 and later
                    text_select = widget_info(event.id,/text_select)
                    void = widget_button(menuid,value='Cut'+string(9b)+'Ctrl+X',sensitive=(text_select[1] gt 0),/separator,uname='txt_cut')
                    void = widget_button(menuid,value='Copy'+string(9b)+'Ctrl+C',sensitive=(text_select[1] gt 0),uname='txt_copy')
                    void = widget_button(menuid,value='Paste'+string(9b)+'Ctrl+V',sensitive=(total(strlen(Clipboard.get())) gt 0),uname='txt_paste')
                    endif
                    void = dm_widget_button(menuid,value='Save',uname='saveidl',sensitive=(max(strlen(tmp_value)) gt 0),imagefile=self.bmpfiles[1],/notchecked,/separator)
                    void = dm_widget_button(menuid,value='Rename',uname='renametab',imagefile=self.bmpfiles[2],/notchecked)
                    if ntab gt 1 then $
                    void = dm_widget_button(menuid,value='Close',uname='closetab',/separator,imagefile=self.bmpfiles[4],/notchecked)
                    if ntab lt n_elements(self.sequenceptr)-1 then $
                    void = dm_widget_button(menuid,value='New Tab',uname='newtab',separator=(n_elements(void) ne 0),imagefile=self.bmpfiles[3],/notchecked)
                    if n_elements(void) ne 0 then widget_displaycontextmenu,event.id,event.x,event.y,menuid
                    end
         'togglecomment':begin  ;behaves like IDL editor toggle comment
                    self->getproperty,currenttboxwid=wid,currentsequence=sequence
                    text_select = widget_info(wid,/text_select)
                    ;figure out the lines
                    tmp = total(strlen(sequence)+1,/cumulative)
                    if text_select[1] gt 0 then begin
                       ind = where(tmp-1 eq text_select[0],cnt)
                       if cnt ne 0 then text_select = 0>(text_select+[1,-1])
                       ind = where(tmp eq text_select[0]+text_select[1],cnt)
                       if cnt ne 0 then text_select[1] = 0>(text_select[1]-1)
                    endif
                    line_i = (where(tmp gt text_select[0]))[0]
                    line_f = (where(tmp gt text_select[0]+text_select[1]))[0]
                    ;add ;?
                    tmp = strtrim(sequence[line_i:line_f],2)
                    if total(strlen(tmp)) eq 0 then add_sc = 1b
                    for i=0,line_f-line_i do begin
                        if strlen(tmp[i]) eq 0 then continue
                        if strmid(tmp[i],0,1) ne ';' then add_sc = 1b
                    endfor
                    if keyword_set(add_sc) then begin
                       for i=line_i,line_f do sequence[i] = ';'+sequence[i]
                       text_select[0] = text_select[0]+1
                       text_select[1] = text_select[1]+line_f-line_i
                    endif else begin
                       for i=line_i,line_f do begin
                           pos = strpos(sequence[i],';')
                           if pos ne -1 then sequence[i] = strmid(sequence[i],0,pos)+strmid(sequence[i],pos+1)
                       endfor
                       tmp = (strlen(tmp) ne 0)
                       text_select[0] = text_select[0]-tmp[0]
                       text_select[1] = 0>(text_select[1]-(total(tmp)-tmp[0]))
                    endelse
                    widget_control,wid,set_value=sequence,set_text_select=text_select
                    self->reset_button,/forcedryrun
                    end
         'txt_cut': begin
                    self->getproperty,currenttboxwid=wid
                    widget_control,wid,get_value=txt,/use_text_select
                    widget_control,wid,set_value='',/use_text_select
                    Clipboard.Set,txt
                    end 
         'txt_copy':begin
                    self->getproperty,currenttboxwid=wid
                    widget_control,wid,get_value=txt,/use_text_select
                    Clipboard.Set,txt
                    end
         'txt_paste':begin
                    self->getproperty,currenttboxwid=wid
                    widget_control,wid,set_value=Clipboard.get(),/use_text_select
                    end                                                         
         'renametab':begin
                    self->getproperty,currenttabnum=currenttabnum,currenttabid=pid,currenttabtitle=tmp
                    tmp = dm_dialog_input('new name:',xsize=120,default=tmp,dialog_parent=self.tlb,cancel=cancel,title='New name:',bitmap=self.bmpfiles[2])
                    if keyword_set(cancel) or (strlen(tmp) eq 0) then break
                    widget_control,pid,set_uvalue=tmp,base_set_title=tmp
                    (*(self.sequenceptr[n_elements(self.sequenceptr)-1]))[currenttabnum]=tmp
                    end
         'closetab':begin
                    self->getproperty,currenttabnum=tcrr,currentsequence=seqcrr,currenttabtitle=titcrr,currenttabid=tid,alltabnum=ntab
                    if ntab eq 1 then break
                    for i=0,n_elements(seqcrr)-1 do begin
                        if strlen(strtrim(seqcrr[i],2)) ne 0 then begin
                           ok = dialog_message('Are you sure you want to close tab "'+titcrr+'"?',title='Please confirm:',/question,dialog_parent=self.tlb,/center)
                           if ok eq 'No' then return else break
                        endif
                    endfor
                    ptr_free,self.sequenceptr[tcrr]
                    tmp  = *(self.sequenceptr[n_elements(self.sequenceptr)-1])
                    case tcrr of 
                         0:       *(self.sequenceptr[n_elements(self.sequenceptr)-1]) = tmp[((ntab-1)<1):ntab-1]
                         ntab-1:  *(self.sequenceptr[n_elements(self.sequenceptr)-1]) = tmp[0:(0>(ntab-2))]
                         else:    *(self.sequenceptr[n_elements(self.sequenceptr)-1]) = [tmp[0:tcrr-1],tmp[tcrr+1:ntab-1]]
                    endcase
                    if tcrr ne ntab-1 then begin
                       self.sequenceptr[tcrr:ntab-2] = self.sequenceptr[tcrr+1:ntab-1]
                       self.sequenceptr[ntab-1] = ptr_new()
                    endif
                    widget_control,tid,/destroy
                    self->save_sequence,/allintempdir
                    end   
         'newtab':  begin
                    tabs = widget_info(self.tlb,find_by_uname='tabbase')
                    geom = widget_info(widget_info(self.tlb,find_by_uname='textbox'),/geometry)
                    ntab = widget_info(tabs,/tab_number)
                    name = dm_dialog_input('tab name:',xsize=120,dialog_parent=self.tlb,cancel=cancel,title='Please enter:',bitmap=self.bmpfiles[3])
                    if keyword_set(cancel) then break
                    if strlen(name) eq 0 then name = 'Untitled'  
                    itab = ([0,1])[name eq 'Untitled']    
                    self->getproperty,alltabtitle=alltabtitle               
                    for i=0,ntab-1 do begin ;avoid the same name
                        ind = stregex(alltabtitle[i],'^'+name+' *([0-9]*)$',/subexpr,length=len,/fold_case)
                        if len[1] ge 0 then itab = itab>(1>(dm_to_number(strmid(alltabtitle[i],ind[1],len[1]),/int))+1)
                    endfor
                    if itab ne 0 then name = name+' '+dm_to_string(itab)
                    tabi = widget_base(tabs,title=name,uname='tab_'+dm_to_string(ntab+1),xpad=0,ypad=0,uvalue=name)
                    tbox = widget_text(tabi,value='',/scroll,/editable,/all_events,scr_xsize=geom.scr_xsize,scr_ysize=geom.scr_ysize,uname='textbox',/context_events,font=self.font[0])
                    widget_control,tabs,set_tab_current=ntab
                    *(self.sequenceptr[n_elements(self.sequenceptr)-1]) = [*(self.sequenceptr[n_elements(self.sequenceptr)-1]),name]
                    if self.addwait then begin
                       wait_cmd = 'dm_wait, 6666  ;a long wait'
                       self->insert_command,commands=wait_cmd,help=''
                    endif else self->reset_button,/forcedryrun
                    end                              
         'textbox': begin
                    self->reset_button,forcedryrun=(event.type ne 3)
                    widget_control,event.id,get_value=sequence
                    if n_elements(sequence) eq 0 then return
                    ;show help info if the cursor is positioned on a command word or append an incomplete command automatically
                    pos0 = (widget_info(event.id,/text_select))[0]
                    pos  = widget_info(event.id,text_offset_to_xy=pos0)
                    if min(pos) ge 0 then begin
                       if pos[1] gt n_elements(sequence)-1 then return
                       if pos[0] gt strlen(sequence[pos[1]]) then return
                       tmp   = string(reverse(byte(strmid(sequence[pos[1]],0,pos[0]))))
                       tmp1  = strmid(sequence[pos[1]],pos[0])
                       pos_s = stregex(tmp,'[, =\+\('+string(9b)+']')
                       pos_e = stregex(tmp1,'[, =\+\('+string(9b)+']')
                       if pos_s eq -1 then pos_s = 0 else pos_s = pos[0]-pos_s
                       if pos_e eq -1 then pos_e = strlen(sequence[pos[1]])-1 else pos_e = pos[0]+pos_e-1
                       cmd = strlowcase(strtrim(strmid(sequence[pos[1]],pos_s,pos_e-pos_s+1),2))
                       case cmd of 
                            '' :          break
                            'dm_a3':      if tmp1 eq '' then append = 'scan, '
                            'dm_al':      if tmp1 eq '' then append = 'ign_'
                            'dm_align_a': if tmp1 eq '' then append = '5, ' 
                            'dm_align_m': if tmp1 eq '' then append = 'onochromator, '
                            'dm_b':       if tmp1 eq '' then append = 'etas('
                            'dm_c':       if tmp1 eq '' then append = 'omment, '
                            'dm_d':       if tmp1 eq '' then append = 'ev_pos('
                            'dm_def':     if tmp1 eq '' then append = '_dev('
                            'dm_f':       if tmp1 eq '' then append = 'indpeak, '
                            'dm_ge':      if tmp1 eq '' then append = 't'
                            'dm_geta':    if tmp1 eq '' then append = '3offset('
                            'dm_getk':    if tmp1 eq '' then append = 'idneyrange('
                            'dm_getl':    if tmp1 eq '' then append = 'astscanname('
                            'dm_go':      if tmp1 eq '' then append = 'betas, '
                            'dm_h':       if tmp1 eq '' then append = 'elp, '
                            'dm_k':       if tmp1 eq '' then append = 'idscan, '
                            'dm_ma':      if tmp1 eq '' then append = 'cs_'
                            'dm_macs_a':  if tmp1 eq '' then append = 'lignment, '
                            'dm_macs_k':  if tmp1 eq '' then append = 'idneylimitscan, '
                            'dm_mo':      if tmp1 eq '' then append = 've, '
                            'dm_o':       if tmp1 eq '' then append = 'ptimizekidstep('
                            'dm_pl':      if tmp1 eq '' then append = 'ot_macs, '
                            'dm_pr':      if tmp1 eq '' then append = 'int, '
                            'dm_runs':    if tmp1 eq '' then append = 'can, '
                            'dm_runt':    if tmp1 eq '' then append = 'ab, '
                            'dm_sen':     if tmp1 eq '' then append = 'dcom, '
                            'dm_set':     if tmp1 eq '' then append = 'slits, '
                            'dm_sp':      if tmp1 eq '' then append = 'inflip, '
                            'dm_te':      if tmp1 eq '' then append = 'mpstaleness, '
                            'dm_to_s':    if tmp1 eq '' then append = 'tring('
                            'dm_w':       if tmp1 eq '' then append = 'ait, '
                            'print':      if tmp1 eq '' and strlen(tmp) eq 5 then begin
                                             pos[0] = pos[0]-5 
                                             pos0   = pos0-5
                                             cmd    = ''
                                             append = 'dm_print'
                                             sequence[pos[1]] = strmid(sequence[pos[1]],0,pos[0])+strmid(sequence[pos[1]],pos[0]+5)
                                          endif
                            'help':       if tmp1 eq '' and strlen(tmp) eq 4 then begin
                                             pos[0] = pos[0]-4 
                                             pos0   = pos0-4
                                             cmd    = ''
                                             append = 'dm_help'
                                             sequence[pos[1]] = strmid(sequence[pos[1]],0,pos[0])+strmid(sequence[pos[1]],pos[0]+4)
                                          endif                            
                            else:         dm_get_help,cmd,output=output
                       endcase
                       if (~self.autocomplete) and (n_elements(append) ne 0) then begin
                          if strmid(append,0,3) ne 'dm_' then tmp1 = temporary(append)
                       endif
                       if (n_elements(append) ne 0) and (event.type ne 2) then begin
                          sequence[pos[1]] = strmid(sequence[pos[1]],0,pos[0])+append+strmid(sequence[pos[1]],pos[0])
                          t_line = widget_info(event.id,/text_top_line) 
                          widget_control,event.id,set_value=sequence,set_text_top_line=t_line,set_text_select=pos0+strlen(append),/input_focus,/all_text_events
                          dm_get_help,(strsplit(strtrim(cmd+append,2),'(, ',/extract))[0],output=output
                       endif
                       if n_elements(output) ne 0 then self->my_widget_control,'statusbox',set_value=output
                    endif
                    if event.type ne 3 then begin
                       self->getproperty,currenttabnum=currenttabnum
                       ptr_free,self.sequenceptr[currenttabnum]
                       self.sequenceptr[currenttabnum] = ptr_new(sequence)
                    endif
                    end
         'ins_com_acol_in':       self->insert_command,command="dm_sendcom,'move acolmon in'",help="Move A collimator (60') in."       
         'ins_com_acol_out':      self->insert_command,command="dm_sendcom,'move acolmon out'",help="Move A collimator (60') out."    
         'ins_com_bcol_in':       self->insert_command,command="dm_sendcom,'move bcolmon in'",help="Move B collimator (40') in."       
         'ins_com_bcol_out':      self->insert_command,command="dm_sendcom,'move bcolmon out'",help="Move B collimator (40') out."                     
         'ins_com_macs_be_in':    self->insert_command,command="; cfx -c 1 -v 0",help="Move Be CFX filter in. Run this command in macs computer. ICE cfxbe state will not be updated."
         'ins_com_macs_be_out':   self->insert_command,command="; cfx -c 1 -v 1",help="Move Be CFX filter out. Run this command in macs computer. ICE cfxbe state will not be updated."
         'ins_com_macs_hopg_in':  self->insert_command,command="; cfx -c 2 -v 0",help="Move HOPG CFX filter in. Run this command in macs computer. ICE cfxhopg state will not be updated."
         'ins_com_macs_hopg_out': self->insert_command,command="; cfx -c 2 -v 1",help="Move HOPG CFX filter out. Run this command in macs computer. ICE cfxhopg state will not be updated."
         'ins_com_macs_mgf2_in':  self->insert_command,command="; cfx -c 0 -v 0",help="Move MgF2 CFX filter in. Run this command in macs computer.  ICE cfxmgf state will not be updated. MgF2 filter is not installed yet."
         'ins_com_macs_mgf2_out': self->insert_command,command="; cfx -c 0 -v 1",help="Move MgF2 CFX filter out. Run this command in macs computer. ICE cfxmgf state will not be updated. MgF2 filter is not installed yet."
         'ins_com_cfx_be_in':     self->insert_command,command="dm_sendcom,'move cfxbe in'",help='Move Be CFX filter in.' 
         'ins_com_cfx_be_out':    self->insert_command,command="dm_sendcom,'move cfxbe out'",help='Move Be CFX filter out.' 
         'ins_com_cfx_hopg_in':   self->insert_command,command="dm_sendcom,'move cfxhopg in'",help='Move HOPG CFX filter in.' 
         'ins_com_cfx_hopg_out':  self->insert_command,command="dm_sendcom,'move cfxhopg out'",help='Move HOPG CFX filter out.' 
         'ins_com_cfx_mgf2_in':   self->insert_command,command="dm_sendcom,'move cfxmgf in'",help='Move MgF2 CFX filter in. MgF2 filter is not installed yet.' 
         'ins_com_cfx_mgf2_out':  self->insert_command,command="dm_sendcom,'move cfxmgf out'",help='Move MgF2 CFX filter out. MgF2 filter is not installed yet.' 
         'ins_com_mcfx_beo':      self->insert_command,command="dm_sendcom,'move mcfx BeO'",help='Use BeO MCFX filter.' 
         'ins_com_mcfx_be':       self->insert_command,command="dm_sendcom,'move mcfx Be'",help='Use Be MCFX filter.' 
         'ins_com_mcfx_hopg':     self->insert_command,command="dm_sendcom,'move mcfx HOPG'",help='Use HOPG (graphite) MCFX filter.'           
         'ins_com_hvf_flat':      self->insert_command,command="dm_sendcom,'action "+['vertifocus','horizfocus']+" flat'",help='Horizontal are vertical focusing modes are both set to flat.' 
         'ins_com_hvf_dfoc':      self->insert_command,command="dm_sendcom,'action "+['horizfocus energy','vertifocus sagittal']+"'",help='Horizontal focusing mode is set to energy, and vertical focusing mode is set to sagittal.'  
         'ins_com_hf_flat':       self->insert_command,command="dm_sendcom,'action horizfocus flat'",help='Horizontal focusing mode is set to flat.' 
         'ins_com_hf_energy':     self->insert_command,command="dm_sendcom,'action horizfocus energy'",help='Horizontal focusing mode is set to energy.'  
         'ins_com_hf_venetian':   self->insert_command,command="dm_sendcom,'action horizfocus venetian'",help='Horizontal focusing mode is set to venetian.'  
         'ins_com_hf_point':      self->insert_command,command="dm_sendcom,'action horizfocus point'",help='Horizontal focusing mode is set to point.' 
         'ins_com_vf_flat':       self->insert_command,command="dm_sendcom,'action vertifocus flat'",help='Vertical focusing mode is set to flat.'   
         'ins_com_vf_sagittal':   self->insert_command,command="dm_sendcom,'action vertifocus sagittal'",help='Vertical focusing mode is set to sagittal.'             
         'ins_com_spinflipopen':  self->insert_command,command="dm_sendcom,'ifopen spinflip'",help=['ifopen device','device can be spinflip or spinfliptemp:',$
                                                               '    spinflip - communication throuth port 3 of the USB serial device at the instrument.',$
                                                               '    spinfliptemp - communication through the temperature serial cable. Device needs to be closed at the end.']
         'ins_com_spinflipclose': self->insert_command,command="dm_sendcom,'ifclose spinflip'",help=['ifclose device','device can be spinflip or spinfliptemp.']
         'ins_com_ptaiauto':      self->insert_command,command="dm_sendcom,'autoptai'",help='Auto PTAI mode'  
         'ins_com_ptaimanual':    self->insert_command,command="dm_sendcom,'manualptai'",help='Manual PTAI mode'    
         'ins_com_smodeauto':     self->insert_command,command="dm_sendcom,'setscattermode auto'",help='Set scatter mode to auto.' 
         'ins_com_smodepos':      self->insert_command,command="dm_sendcom,'setscattermode pos'",help='Set scatter mode to positive.'   
         'ins_com_smodeneg':      self->insert_command,command="dm_sendcom,'setscattermode neg'",help='Set scatter mode to negative.'   
         'ins_tem_aligna5': begin
                    Ef = 5                 
                    path = dialog_pickfile(file=self.workdir,/directory,/write,dialog_parent=self.tlb,title='Please select a directory for saving the alignment files, cancel to skip')
                    command = ['','Ef = '+dm_to_string(Ef)]
                    help = ['Align 20 analyzers at Ef='+dm_to_string(Ef)+' meV.']
                    if strlen(path) ne 0 then begin
                       self.workdir = path
                       command = [command,"directory = '"+path+"'",'dm_align_a5, Ef, directory, /saveweights',''] 
                       help = [help,'The weights are saved in '+path+'weights_'+dm_to_string(Ef)+'meV_'+date_str+'.txt.']
                    endif else begin
                       command = [command,'dm_align_a5, Ef','']
                    endelse
                    self->insert_command,command=command,help=help
                    end
         'ins_tem_aligninstr': begin
                    path = self.workdir
                    file = dm_choose_file('txt',/write,dialog_parent=self.tlb,title='Please select a file name for saving the alignment data',path=path,file='macs_alignment_'+date_str+'.txt')
                    if strlen(file) eq 0 then return
                    self.workdir = path                      
                    self->insert_command,command=['',"filename = '"+file+"'",'dm_macs_alignment, filename=filename',''],$
                          help=['Scan dfmdts and monblade11 at 2theta=[120,110,100,90,80,70,60,50,40] for instrument alignment.','The alignment results are saved in '+file+'.']
                    end           
         'ins_tem_alignmono': begin
                    path = dialog_pickfile(file=self.workdir,/directory,/write,dialog_parent=self.tlb,title='Please select a directory for saving the monochromator blade centers, cancel to skip')
                    help = ['Align monochromator blades at 45 degree. After the scans, move the blades to the fitted center and reset them to 45.']
                    if strlen(path) ne 0 then begin
                       self.workdir = path
                       command = ['',"directory = '"+path+"'",'dm_align_monochromator, directory, /savecenter, /setblades',''] 
                       help = [help,'The fitted monochormator blade centers are saved in '+path+'monbladecenter_'+date_str+'.txt.']
                    endif else begin
                       command = ['','dm_align_monochromator, /setblades','']
                    endelse
                    self->insert_command,command=command,help=help
                    end           
         'ins_tem_kidlimscan': begin
                    path = self.workdir
                    file = dm_choose_file('txt',/write,dialog_parent=self.tlb,title='Please select a file name for saving the limits',path=path,file='macs_kidneylimit_'+date_str+'.txt')
                    if strlen(file) eq 0 then return
                    self.workdir = path                      
                    self->insert_command,command=['',"filename = '"+file+"'",'dm_macs_kidneylimitscan, filename=filename',''],$
                          help=['Scan for the kidney hardware limits at various energies.','The results are saved in '+file+'.']
                    end
         'ins_tem_setslit': begin
                    self->insert_command,command=['','Ei = ','dmbt = ','hsamp = ','beta12 = dm_betas(Ei, dmbt)','dm_setslits, beta12[0], beta12[1], dmbt, hsamp',''],$
                          help=['Calculate slit dimension. dm_setslits will not send any command to the ICE server. Dryrun to print the results.','Ei is in unit of meV.',$
                          'dmbt is in unit of mm.','hsamp is the sample height in unit of mm.']
                    end           
         'ins_tem_demag': begin
                    space = '      '
                    magid = dm_dialog_input('magnet:',default=0,droplist_content=ptr_new(['11T dil fridge','10T dry magnet']),is_droplist=[1],/return_number,title='Please choose',dialog_parent=self.tlb,xsize=100)
                    field = (['[-6,4,-2,1,-0.8,0.4,-0.2,0.1,-0.05,0.02,-0.01,0]','[-4,2,-1,0.8,-0.6,0.4,-0.2,0.1,-0.05,0.02,-0.01,0]'])[magid]
                    helps = ['Demagnetization procedure.','Make sure hoses and cables are safe to move over the entire energy range.']
                    infos = [';Make sure hoses and cables are safe to move over the entire energy range.',';The whole seqeunce takes about '+(['3 hours','1 hour'])[magid]+' to finish.']
                    if magid eq 0 then begin
                       infos = [infos[0],';For 11T dil fridge magnet, keep the switch heater on during the demag procedure.',';Remember to turn off the switch heater at the end.',$
                                ';Liquid helium level should be above 50%.',infos[-1]] 
                       helps = [helps,'Keep the switch heater on during demagnetization, and remember to turn off the switch heater at the end.','Liquid helium level should be above 50%.']
                    endif
                    self->insert_command,command=[infos,'',(["dm_wait, 6666,comment='Remember to keep the switch heater on.'",'dm_wait, 6666'])[magid ne 0],$
                          'Eran = [12.5,2.5]    ;avoid 13 meV',';movea3 = 0','bs = '+field,'',"dm_sendcom, 'action vertifocus flat'","dm_sendcom, 'action horizfocus flat'",$
                          "dm_sendcom, 'autoptai'",'',"dm_sendcom, 'device setlowerlimit magfield '+dm_to_string(min(bs)-1)","dm_wait, 10",'','kcen = dm_getkidneyrange(Eran,/center)',$
                          '',"if n_elements(movea3) ne 0 then dm_move,'a3',movea3","dm_sendcom, 'move ei '+dm_to_string(Eran[0])+' a4 0'",$
                          ";dm_sendcom, 'move ei '+dm_to_string(Eran[0])+' kidney '+dm_to_string(kcen[0],res=3)",'for i=0,n_elements(bs)-1 do begin',$
                          space+(["if bs[i] eq 0 then dm_wait, 6666,comment='Turn off the switch heater flag now.'","dm_sendcom, 'move magfield '+dm_to_string(abs(bs[i])*(-1)^(i+1))",$
                          "if n_elements(movea3) ne 0 then dm_move,'a3',movea3","dm_sendcom, 'move ei '+dm_to_string(Eran[(i+1) mod 2])+' a4 0'",$
                          ";dm_sendcom, 'move ei '+dm_to_string(Eran[(i+1) mod 2])+' kidney '+dm_to_string(kcen[(i+1) mod 2],res=3)"])[magid:*],'endfor','',$
                          "dm_sendcom, 'move ei 3.7 a4 0'","dm_sendcom, 'move kidney 0'"],help=helps
                    self->getproperty,currenttabid=wid,currenttabtitle=title,currenttabnum=currenttabnum,alltabtitle=alltabtitle
                    if ~strmatch(title,'*demag*',/fold_case) then begin ;change the tab name to 'demag'
                       i_tab = 1 & name = 'demag'
                       for i=0,n_elements(alltabtitle)-1 do begin
                           ind = stregex(alltabtitle[i],'^'+name+' *([0-9]*)$',/subexpr,length=len,/fold_case)
                           if len[1] ge 0 then i_tab = i_tab>(1>(dm_to_number(strmid(alltabtitle[i],ind[1],len[1]),/int))+1)
                       endfor
                       if i_tab ne 1 then name = name+' '+dm_to_string(i_tab)
                       widget_control,wid,set_uvalue=name,base_set_title=name
                       (*(self.sequenceptr[n_elements(self.sequenceptr)-1]))[currenttabnum]=name
                    endif                      
                    end
         'ins_file':begin
                    path = self.workdir
                    file = dm_choose_file('*',dialog_parent=self.tlb,/read,path=path,title='Plase choose a command file')
                    if strlen(file) eq 0 then break
                    self.workdir = path
                    self->load_sequence,file,sequence=sequence
                    self->insert_command,commands=sequence
                    end           
         'dryrun':  begin
                    ok = self->dryrun(/display)
                    self->my_widget_control,['saveseq','execute'],sensitive=ok
                    if ok then self->save_sequence,/allintempdir
                    end
         'execute': begin
                    self->getproperty,currentsequence=sequence,currenttabtitle=tabtitle
                    if total(stregex(sequence,'!*spin(state)*',/boolean,/fold_case)) ne 0 then dm_check_seqwait,spinstate=spinstate else spinstate = ''
                    !spinstate = spinstate ;reset spinstate
                    mesg = 'Do you want to send the "'+tabtitle+'" sequence to ICE server?'
                    if self.addwait and ~(stregex((*(!dm_macsseq_dryrun.sequence))[0],'^wait +[1-9]+',/boolean,/fold_case)) then begin
                       sequence = ['dm_wait,6666',sequence]
                       mesg = [mesg,'','A "wait 6666" command is added to the beginning of the sequence.']
                    endif
                    ok = dialog_message(mesg,title='Please confirm:',/question,/default_no,dialog_parent=self.tlb,/center)
                    if ok eq 'No' then return
                    !dm_macsseq_dryrun.dryrun  = 0b
                    !dm_macsseq_dryrun.execute = 1b
                    !dm_macsseq_dryrun.nwait   = 0
                    !dm_macsseq_dryrun.message = obj_new('dm_progress',title='Please wait...',message='Compiling sequence.',group_leader=self.tlb) 
                    sequence = dm_strip_comments(sequence,/combine)
                    !dm_macsseq_dryrun.message->update,message='Sending "'+tabtitle+'" commands to ICE server.'
                    self->my_widget_control,'statusbox',set_value=dm_to_string(systime(/sec),/date)+'>   Sending "'+tabtitle+'" commands to ICE server.'
                    ok = execute(sequence,1,1)
                    if ~ok then begin
                       ok = dialog_message(!error_state.msg,/error,dialog_parent=self.tlb,/center)
                       dm_print,!error_state.msg
                    endif else dm_print,'Done.'
                    !dm_macsseq_dryrun.execute = 0b
                    if obj_valid(!dm_macsseq_dryrun.message) then obj_destroy,!dm_macsseq_dryrun.message
                    if !dm_macsseq_dryrun.nwait ne 0 then ok = dialog_message('There '+(['is ','are '])[!dm_macsseq_dryrun.nwait gt 1]+dm_to_string(!dm_macsseq_dryrun.nwait)+' wait command'+$
                       (['','s'])[!dm_macsseq_dryrun.nwait gt 1]+' in the sequence.',title='Please note:',/info,dialog_parent=self.tlb,/center)
                    end
         'clear':   begin
                    ok = dialog_message('Are you sure you want to clear the sequence script?',title='Please confirm:',/question,dialog_parent=self.tlb,/center)
                    if ok eq 'No' then return
                    self->getproperty,currenttboxwid=wid
                    widget_control,wid,set_value=''
                    self->insert_command,commands='',help=''
                    end   
         'saveidl': self->save_sequence,/currentidl
         'saveseq': self->save_sequence,/currentice   
         'exit':    begin
                    ok = dialog_message('Are you sure you want to '+(['exit','close'])[strmatch(type,'widget_kill_request',/fold_case)]+' the MACS Sequence Editor?',$
                         title='Please confirm:',/question,/default_no,dialog_parent=self.tlb,/center)
                    if ok eq 'Yes' then widget_control,self.tlb,/destroy
                    end
         'addwait': self->setproperty,addwait=(~self.addwait)
         'autocomplete': self->setproperty,autocomplete = (~self.autocomplete)
         'switchhand': begin
                    self.handedness = ~self.handedness
                    widget_control,event.id,set_value='Switch to '+(['Left','Right'])[self.handedness]+'-Handed Layout' 
                    geom0 = widget_info(widget_info(self.tlb,find_by_uname='textbox'),/geometry)
                    left  = widget_info(self.tlb,find_by_uname='left')
                    right = widget_info(self.tlb,find_by_uname='right')
                    geom1 = widget_info(([right,left])[self.handedness],/geometry)
                    widget_control,self.tlb,update=0
                    self->my_widget_control,['tabbase','shortcutbase'],/destroy
                    tmp   = widget_base(([left,right])[self.handedness],/base_align_bottom,/base_align_left,/row,xpad=1,ypad=2,space=5,uname='shortcutbase')
                    sc_n  = widget_button(tmp,value=self.bmpfiles[3],/bitmap,/flat,uname='newtab')
                    sc_r  = widget_button(tmp,value=self.bmpfiles[2],/bitmap,/flat,uname='renametab')
                    sc_s  = widget_button(tmp,value=self.bmpfiles[1],/bitmap,/flat,uname='saveidl')
                    sc_d  = widget_button(tmp,value=self.bmpfiles[4],/bitmap,/flat,uname='closetab')
                    if self.idl_version ge 5.6 then begin
                       widget_control,sc_s,tooltip='Save current tab.'
                       widget_control,sc_r,tooltip='Rename current tab.'
                       widget_control,sc_n,tooltip='Open a new tab.'
                       widget_control,sc_d,tooltip='Close current tab.'
                    endif
                    tabs = widget_tab(([left,right])[self.handedness],tab_mode=0,uname='tabbase')
                    for i=0,n_elements(*self.sequenceptr[n_elements(self.sequenceptr)-1])-1 do begin
                        name = (*self.sequenceptr[n_elements(self.sequenceptr)-1])[i]
                        tabi = widget_base(tabs,title=name,uname='tab_'+dm_to_string(i+1),xpad=0,ypad=0,uvalue=name)
                        tbox = widget_text(tabi,value=ptr_valid(self.sequenceptr[i])?(*self.sequenceptr[i]):'',/scroll,/editable,/all_events,scr_xsize=geom0.scr_xsize,scr_ysize=geom0.scr_ysize,uname='textbox',/context_events,font=self.font[0])
                    endfor
                    widget_control,([left,right])[self.handedness],scr_xsize=geom1.scr_xsize,scr_ysize=geom1.scr_ysize
                    widget_control,([right,left])[self.handedness],scr_xsize=0,scr_ysize=0 
                    widget_control,self.tlb,/update,/map
                    self->reset_button
                    if widget_info(self.group_leader,/valid_id) ne 0 then begin ;allow mslice to remember the sequencer layout handedness
                       sendevent = {dm_macs_sequencer_size,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,TEXTBOXSIZE:[geom0.scr_xsize,geom0.scr_ysize],HANDEDNESS:self.handedness}
                       widget_control,self.group_leader,send_event=sendevent,/no_copy  
                    endif
                    end
         'kidlimit': dm_macs_kidlimit,event,bgcolor='white'
         'kidoptimize': dm_macs_kidoptimizer,event,bgcolor='white'
         'a3basethetaoffset': dm_macs_a3offset,event,allow_ice=(widget_info(self.tlb,find_by_uname='execute') ne 0)      
         'opensesame': begin
                    ok = dm_dialog_password(dialog_parent=self.tlb)
                    if dm_ip_encrypt(ok,base64=0) eq self.password then begin
                       b_col = widget_info(self.tlb,find_by_uname='b_col')
                       if b_col ne 0 then begin
                          self->getproperty,currentsequence=tmp_value
                          tmp_value = strtrim(tmp_value,2)
                          self->my_widget_control,'clear',b_col,/destroy
                          void = widget_button(b_col,value='Execute Sequence',uname='execute',sensitive=0)
                          void = widget_button(b_col,value='Clear',uname='clear',sensitive=(max(strlen(tmp_value)) gt 0))
                          widget_control,event.id,sensitive=0b
                          tmp = widget_info(b_col,/geometry)
                          self.ymin = tmp.ysize
                       endif
                    endif else ok = dialog_message('Incorrect password.',/error,/center,dialog_parent=self.tlb)
                    end
         'help':    if strlen(self.helpfile) ne 0 then online_help,book=self.helpfile,/full_path
         'macscmdlist': begin
                    cmds = ['getMotorPositionsFromHKL <h> <k> <l> [-et <value>] [-ei <value>] [-ef <value>] ',$
                            string(9b)+'[-l <lattice array>] [-o <orientation array>] [-ub on|off] ',$
                            string(9b)+'[-scatter pos|neg|auto] [-ptaimode a(uto)|m(anual)] [-ptai <index>]',$
                            'gettempstaleness <tdevice> (tdevice cannot be an alias)',$
                            'settempstaleness <tdevice> <time> (time in seconds, tdevice cannot be an alias)',$
                            'setmaxelbowwristmoveattempts <attempts>',$
                            'Setmaxelbowwristmovetime <time> (in seconds)',$
                            'Getmaxelbowwristmoveattempts',$
                            'Getmaxelbowwristmovetime',$
                            'setdetectoroffset <index> <offset>',$
                            'getdetectoroffsets',$
                            'activatedetector <detectorNumber>',$
                            'deactivatedetector <detectorNumber>',$
                            'getactivedetectors',$
                            'autoptai',$
                            'manualptai',$
                            'getptaimode',$
                            'Checkkmlimits <MBTSlide destination> <Kidney destination>',$
                            'Getkidneyrange <Fixed MBTSlide position>',$
                            'Getkidneyrangea2 <A2 position>',$
                            'Getkidneyrangeei <Ei position>',$
                            'Getmbtsliderange <Fixed Kidney position>',$
                            'Getmbtpivot',$
                            'Geta2ref',$
                            'Seta2ref <A2Ref>',$
                            'Setmbtpivot <MBTPivot>',$
                            'Setmaxcenterdist <maxCenteringDistance>',$
                            'Getmaxcenterdist']
                    xdisplayfile,text=cmds,group=self.tlb,title='ICE Commands Unique to MACS',done_button='Exit',/editable,/grow_to_screen,return_id=wid,font=self.font[0]
                    dm_center_kid,wid,self.tlb
                    end
         'idlhelp': dm_idl_help,group=self.tlb,font=self.font[1]
         'about':   begin
                    info = ['--MACS Sequence Editor--','Based on the IDL codes written by Jose A. Rodriguez.','','To communicate with ICE, the location of JICE.jar needs to be added to the IDL java classpath in '+$
                            'IDL resource\bridges\import\java\idljavabrc file.','','Last updated on 11/14/2023.']
                    ok = dialog_message(info,/information,dialog_parent=self.tlb,title='About MACS Sequence Editor',/center)
                    end           
         else:      if strmid(strlowcase(uname),0,8) eq 'ins_com_' then begin   ;all other ins_com_**** buttons      
                    dm_get_help,'dm_'+strmid(strlowcase(uname),8),output=output
                    if n_elements(output) ne 0 then self->insert_command,help=output
                    endif
    endcase
end

pro dm_macs_sequencer::getproperty,tlb=tlb,alltabtitle=alltabtitle,alltabnum=alltabnum,bmpfiles=bmpfiles,currentsequence=currentsequence,currenttabid=currenttabid,$
    currenttabnum=currenttabnum,currenttabtitle=currenttabtitle,currenttboxwid=currenttboxwid,kidoptstr=kidoptstr
    if arg_present(tlb) then tlb = self.tlb
    if arg_present(bmpfiles) then bmpfiles = self.bmpfiles
    if arg_present(kidoptstr) then kidoptstr = self.kidoptstr
    if arg_present(alltabtitle) or arg_present(alltabnum) or arg_present(currentsequence) or arg_present(currenttabid) or arg_present(currenttabnum) or arg_present(currenttabtitle) or arg_present(currenttboxwid) then begin
       wid = widget_info(self.tlb,find_by_uname='tabbase')
       if arg_present(alltabtitle) or arg_present(alltabnum) then begin
          alltabnum = widget_info(wid,/tab_number)
          for i=0,alltabnum-1 do begin ;avoid the same name
              if i eq 0 then tmpwid  = widget_info(wid,/child) else tmpwid = widget_info(tmpwid,/sibling)
              if tmpwid eq 0 then break
              widget_control,tmpwid,get_uvalue=tabtitle
              if (i eq 0) or (n_elements(alltabtitle) eq 0) then alltabtitle=tabtitle else alltabtitle=[alltabtitle,tabtitle]
          endfor
       endif
       currenttabnum = widget_info(wid,/tab_current)
       wid = widget_info(wid,/child)
       for i=0,currenttabnum-1 do wid = widget_info(wid,/sibling)
       currenttboxwid = widget_info(wid,/child) ;locate the textbox widget id
       if arg_present(currenttabid)    then currenttabid = wid
       if arg_present(currenttabtitle) then widget_control,wid,get_uvalue=currenttabtitle
       if arg_present(currentsequence) then widget_control,currenttboxwid,get_value=currentsequence
    endif
end

pro dm_macs_sequencer::insert_command,commands=commands,help=help
    if n_elements(help) ne 0 then begin
       self->my_widget_control,'statusbox',set_value=help
       if n_elements(commands) eq 0 then begin
          ind = where(stregex(help,'syntax\:',/fold_case,/boolean),count)
          if count ne 0 then begin
             commands = strtrim((stregex(help[ind[0]],'syntax\: *(.*)',/fold_case,/extract,/subexpr))[1])
             while (strmid(commands,0,1,/reverse_offset) eq '$') do begin
                   ind[0] = ind[0]+1
                   if ind[0] ge n_elements(help) then break
                   commands = strmid(commands,0,strlen(commands)-1)+strtrim(help[ind[0]],2)
             endwhile
          endif
       endif
    endif
    if n_elements(commands) eq 0 then return
    
    self->getproperty,currenttboxwid=wid,currenttabnum=tcrr
    t_line = widget_info(wid,/text_top_line) 
    n_line = (widget_info(wid,text_offset_to_xy=(widget_info(wid,/text_select))[0]))[1] ;current line number
    widget_control,wid,get_value=txt_str
    s_line = n_elements(txt_str)
    if (n_line eq -1) or (n_line gt s_line-1) then begin
       if total(strlen(txt_str)) eq 0 then txt_str = commands $
       else txt_str = [txt_str,commands] 
       text_select  = total(strlen(txt_str))+n_elements(txt_str)+1
    endif else begin
       if strlen(strtrim(txt_str[n_line],2)) eq 0 then begin
          if n_line eq 0 then begin
             if s_line gt 1 then txt_str = [commands,txt_str[(n_line+1):(s_line-1)]] $
             else txt_str = commands
             text_select = total(strlen(commands))+n_elements(commands)+1
          endif else begin
             if n_line+1 le s_line-1 then txt_str = [txt_str[0L:(n_line-1)],commands,txt_str[(n_line+1):(s_line-1)]] $
             else  txt_str = [txt_str[0L:(n_line-1)],commands]
             text_select = total([strlen(txt_str[0:(n_line)-1]),strlen(commands)])+n_line+n_elements(commands)+1
          endelse
       endif else begin
          if n_line eq 0 then begin
             txt_str = [commands,txt_str[(n_line):(s_line-1)]] 
             text_select = total(strlen(commands))+n_elements(commands)+1
          endif else begin
             txt_str = [txt_str[0L:n_line-1],commands,txt_str[(n_line):(s_line-1)]]
             text_select = total([strlen(txt_str[0:(n_line)-1]),strlen(commands)])+n_line+n_elements(commands)+1
          endelse
       endelse
    endelse
    if (n_elements(txt_str) eq 1) and (txt_str[0] eq '') then text_select=0
    widget_control,wid,set_value=txt_str,set_text_top_line=t_line,set_text_select=text_select,/input_focus,/all_text_events
    ptr_free,self.sequenceptr[tcrr]
    self.sequenceptr[tcrr] = ptr_new(txt_str,/no_copy)
    self->reset_button,/forcedryrun
end

;load sequence file, returns the content as a string array
pro dm_macs_sequencer::load_sequence,file,sequence=sequence,init=init
    if ~file_test(file,/read) then return
    openr,unit,file,/get_lun
    tmp = ''
    while(~ eof(unit)) do begin
        readf,unit,tmp
        if n_elements(sequence) eq 0 then sequence = tmp else sequence = [sequence,tmp]
    endwhile
    free_lun,unit 
    if keyword_set(init) then begin
       ptr_free,self.sequenceptr
       ;if sequence[n_elements(sequence)-1] eq string(9b) then self.handedness = 1b
       ind = where(stregex(sequence,'^[ '+string(9b)+']*'+string('A4'xb),/boolean),count)
       if count eq 0 then begin
          self.sequenceptr[0] = ptr_new(sequence)
          self.sequenceptr[n_elements(self.sequenceptr)-1] = ptr_new('Untitled 1')
       endif else begin
          ii = 0
          for i=0,count-1 do begin
              tmp = sequence[ind[i]+1:(i eq count-1?(n_elements(sequence)-1):(ind[i+1]-1))]
              if max(strlen(strtrim(tmp,2))) gt 0 then begin
                 self.sequenceptr[ii] = ptr_new(tmp)
                 name = strsplit(sequence[ind[i]],string('A4'xb),/extract)
                 name = name[n_elements(name)-1]
                 if ii eq 0 then self.sequenceptr[n_elements(self.sequenceptr)-1] = ptr_new(name) $
                 else *(self.sequenceptr[n_elements(self.sequenceptr)-1]) = [*(self.sequenceptr[n_elements(self.sequenceptr)-1]),name]
                 ii = ii+1
              endif
          endfor
          if ii eq 0 then begin
             self.sequenceptr[0] = ptr_new('')
             self.sequenceptr[n_elements(self.sequenceptr)-1] = ptr_new('Untitled 1')
          endif
       endelse
    endif else begin
       while (strlen(strtrim(sequence[n_elements(sequence)-1],2)) eq 0) and (n_elements(sequence) gt 1) do sequence = sequence[0:(n_elements(sequence)-2)]  ;remove trailing empty strings
    endelse
end

;a widget_control use uname instead of id as identifier
pro dm_macs_sequencer::my_widget_control,uname,base,_ref_extra=extra
    if n_elements(base) eq 0 then base=self.tlb
    if ~widget_info(base,/valid_id) then base=self.tlb
    if ~widget_info(base,/valid_id) then return
    if n_elements(extra) ne 0 then begin
       for i=0,n_elements(uname)-1 do begin
           wid = widget_info(base,find_by_uname=uname[i])
           if wid ne 0 then widget_control,wid,_extra=extra
       endfor
    endif
end

pro dm_macs_sequencer::reset_button,forcedryrun=forcedryrun
    self->getproperty,currentsequence=sequence,alltabnum=ntab
    tmp_value = strtrim(sequence,2)
    self->my_widget_control,['saveidl','dryrun','clear'],sensitive=(max(strlen(tmp_value)) gt 0)
    scbid = widget_info(self.tlb,find_by_uname='shortcutbase')
    self->my_widget_control,'saveidl',scbid,sensitive=(max(strlen(tmp_value)) gt 0)
    self->my_widget_control,'closetab',scbid,sensitive=(ntab gt 1)
    self->my_widget_control,'newtab',scbid,sensitive=(ntab lt n_elements(self.sequenceptr)-1)
    if keyword_set(forcedryrun) then self->my_widget_control,['saveseq','execute'],sensitive=0
end

;save sequence file
;keywords:
;  allintempdir:  if set, save all tabs in temp directory
;  currentice:    if set, save the compiled ice sequence of the current tab
;  currentidl:    if set, save the idl code of the current tab
pro dm_macs_sequencer::save_sequence,allintempdir=allintempdir,currentice=currentice,currentidl=currentidl
    if keyword_set(allintempdir) then begin
       if ~file_test(dm_define_pointer(/gettempdir),/directory,/write) then return
       titls = *(self.sequenceptr[n_elements(self.sequenceptr)-1])
       sequence = ''
       for i=0,n_elements(titls)-1 do begin
           if ptr_valid(self.sequenceptr[i]) then tmp = *(self.sequenceptr[i]) else tmp=''
           while (strlen(strtrim(tmp[n_elements(tmp)-1],2)) eq 0) and (n_elements(tmp) gt 1) do tmp = tmp[0:(n_elements(tmp)-2)]  ;remove trailing empty strings
           sequence = [sequence,string('A4'xb)+titls[i],tmp]
       endfor
       ;sequence = [sequence[1L:(n_elements(sequence)-1)],(['',string(9b)])[self.handedness]]
       sequence = sequence[1L:(n_elements(sequence)-1)]
       file = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'.macs_sequencer'
    endif else if keyword_set(currentice) then begin
       ok = self->dryrun(sequence=sequence,currenttabtitle=name)
       if ~ok then return
       title = 'Save Current Compiled ICE Sequence'
    endif else if keyword_set(currentidl) then begin
       self->getproperty,currentsequence=sequence,currenttabtitle=name
       tmp = strtrim(sequence,2)
       if max(strlen(tmp)) eq 0 then return
       while (strlen(strtrim(sequence[n_elements(sequence)-1],2)) eq 0) and (n_elements(sequence) gt 1) do sequence = sequence[0:(n_elements(sequence)-2)]  ;remove trailing empty strings
       title = 'Save Current IDL Script'
    endif
    if n_elements(sequence) eq 0 then return
    path = self.workdir
    if n_elements(file) eq 0 then begin
       if n_elements(name) ne 0 then file  = name+(['_','_ICE_'])[keyword_set(currentice)]+strmid(dm_to_string(dm_to_number(systime(),/date)),0,8)+'.txt'
       file = dm_choose_file('txt',/write,dialog_parent=self.tlb,title=title,file=file,path=path)
       if strlen(file) eq 0 then return
       self.workdir = path
    endif
    openw,unit,file,/get_lun,error=openerr
    if openerr ne 0 then begin
       if ~keyword_set(allintempdir) then ok = dialog_message("Can't write in "+file+'.',/error,dialog_parent=self.group_leader,/center)
       return
    endif
    for i=0L,n_elements(sequence)-1 do printf,unit,sequence[i]
    free_lun,unit
end

pro dm_macs_sequencer::setproperty,addwait=addwait,autocomplete=autocomplete,kidoptstr=kidoptstr
    if n_elements(addwait) eq 1 then begin
       self.addwait = keyword_set(addwait)
       dm_set_button,widget_info(self.tlb,find_by_uname='addwait'),self.addwait
       if self.addwait then ok = dialog_message('A "wait 6666" command will be added to the beginning of the sequence if it is not already there.',/info,dialog_parent=self.tlb,/center)
    endif
    if n_elements(autocomplete) eq 1 then begin
       self.autocomplete = keyword_set(autocomplete)
       dm_set_button,widget_info(self.tlb,find_by_uname='autocomplete'),self.autocomplete
    endif
    if n_elements(kidoptstr) eq 4 then self.kidoptstr = kidoptstr
end

;cleanup
pro dm_macs_sequencer::Cleanup
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    defsysv,'!dm_macsseq_dryrun',exists=exists
    if exists then begin
       ptr_free,!dm_macsseq_dryrun.sequence,!dm_macsseq_dryrun.scanname
       obj_destroy,!dm_macsseq_dryrun.message
    endif
    self->save_sequence,/allintempdir
    ptr_free,self.sequenceptr
end

;initialization
function dm_macs_sequencer::Init,group_leader=group_leader,lefthanded=lefthanded,textboxsize=textboxsize,workdir=workdir
    registerName = 'dm_macs_sequencer'
    ;self.handedness = keyword_set(lefthanded)
    if xregistered(registerName) then begin ;only allow one copy to be running at one time
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0
       return,0  
    endif
    if n_elements(workdir) ne 0 then begin
       if file_test(workdir[0],/directory,/write) then self.workdir = workdir[0]
    endif
    if n_elements(group_leader) ne 0 then self.group_leader = group_leader
    xysize = [670,500,150,7]
    if n_elements(textboxsize) eq 2 then begin
       if total(finite(textboxsize)) eq 2 then xysize[0:1] = textboxsize
    endif
    self.password = string(byte([164,26,70,157,147,189,218,133,121,42,89,145,3,205,195,250]))
    self.idl_version = dm_to_number(!version.release)
    accel_yn = (self.idl_version ge 6.1)
    allow_ex = dm_macs_sequencer_authorizeip()
    self.sequenceptr[0] = ptr_new('dm_wait, 6666  ;a long wait')
    self.sequenceptr[n_elements(self.sequenceptr)-1] = ptr_new('Untitled 1')
    parmfile = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'.macs_sequencer'
    if file_test(parmfile,/read) then begin
       self->load_sequence,parmfile,/init
       if n_elements(*self.sequenceptr[n_elements(self.sequenceptr)-1]) eq 1 then begin
          if stregex(*self.sequenceptr[n_elements(self.sequenceptr)-1],'^untitled *[0-9]*$',/fold_case,/boolean) then $
             *self.sequenceptr[n_elements(self.sequenceptr)-1] = 'Untitled 1'
       endif
    endif
    defsysv,'!spinstate',exists=exists
    if exists then !spinstate = '' $
    else defsysv,'!spinstate',''
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    self.bmpfiles = ['macs.ico','mslice_'+['save','rename','new','delete','comment','demag','slit']+'.bmp','idl.ico']
    for i=0,n_elements(self.bmpfiles)-1 do begin
        if exists then self.bmpfiles[i] = !DAVE_AUXILIARY_DIR+self.bmpfiles[i] $
        else self.bmpfiles[i] =  file_which(self.bmpfiles[i],/include_current_dir)
    endfor
    if self.idl_version ge 6.4 then icon_extra={bitmap:self.bmpfiles[0]}
    self.tlb = widget_base(title='MACS Sequence Editor',column=1,mbar=mbar,/tlb_size_event,kill_notify='dm_macs_sequencer_Exit',group_leader=group_leader,_extra=icon_extra,map=0,uname='tlb',xpad=0,/tlb_kill_request_events)
    filemenu = widget_button(mbar,value='File',/menu)
    optnmenu = widget_button(mbar,value='Options',/menu)
    toolmenu = widget_button(mbar,value='Tools',/menu)
    helpmenu = widget_button(mbar,value='Help',/menu)
    void = dm_widget_button(filemenu,value='Save IDL Script...',uname='saveidl',sensitive=0,imagefile=self.bmpfiles[1],accelerator='Ctrl+S',/notchecked)
    void = dm_widget_button(filemenu,value='Save Compiled ICE Sequence...',uname='saveseq',sensitive=0,imagefile=self.bmpfiles[1],accelerator='Ctrl+Shift+S',/notchecked)
    void = widget_button(filemenu,value='Exit',uname='exit',/separator)
    self.font[*] = widget_info(void,/fontname) 
    if !version.os_family eq 'Windows' then begin
       tmp = stregex(self.font[0],'\*-?([0-9]+)',/subexpr,len=len)
       if len[1] ne 0 then self.font[0:1] = strmid(self.font[0],0,tmp[1])+dm_to_string(dm_to_number(strmid(self.font[0],tmp[1],len[1]))+4)+strmid(self.font[0],tmp[1]+len[1])
    endif
    self.helpfile = 'ICEUserManual.pdf'
    defsysv,'!DAVE_PDFHELP_DIR',exists=exists
    if exists then self.helpfile = !DAVE_PDFHELP_DIR+dm_define_pointer(/getpathsep)+self.helpfile $
    else self.helpfile = file_which(self.helpfile,/include)
    self.autocomplete = 1b
    void  = dm_widget_button(optnmenu,value='Add Wait',uname='addwait',set_button=self.addwait)
    void  = dm_widget_button(optnmenu,value='Autocomplete',uname='autocomplete',set_button=self.autocomplete)
    ;void  = widget_button(optnmenu,value='Switch to '+(['Left','Right'])[self.handedness]+'-Handed Layout',uname='switchhand',/separator)
    void  = widget_button(toolmenu,value='Kidney Angle Limit',uname='kidlimit')
    void  = widget_button(toolmenu,value='Kidney Step Optimization',uname='kidoptimize')
    void  = widget_button(toolmenu,value='Offset Angle among A3, Basesampletheta, and Kidney',uname='a3basethetaoffset')  
    if strlen(self.helpfile) ne 0 then $
    void  = widget_button(helpmenu,value='ICE User Manual',uname='help',_extra=accel_yn?{accelerator:'F1'}:{no_copy:1})
    void  = widget_button(helpmenu,value='ICE Commands Unique to MACS',uname='macscmdlist')
    void  = dm_widget_button(helpmenu,value='IDL Statement Syntax',uname='idlhelp',imagefile=self.bmpfiles[8],/notchecked)
    if ~allow_ex then $
    void  = widget_button(helpmenu,value='Open Sesame',uname='opensesame',/separator)
    void  = widget_button(helpmenu,value='About MACS Sequence Editor',uname='about',/separator)
    top   = widget_base(self.tlb,/row,uname='top')
    left  = widget_base(top,column=1,uname='left',xpad=0,ypad=0,/base_align_center)
    b_col = widget_base(top,column=1,uname='b_col',space=2)
    right = widget_base(top,column=1,uname='right',xpad=0,ypad=0,/base_align_center)
    b_row = widget_base(([left,right])[self.handedness],/base_align_bottom,/row,xpad=1,ypad=2,space=5,uname='shortcutbase')
    sc_n  = widget_button(b_row,value=self.bmpfiles[3],/bitmap,/flat,uname='newtab')
    sc_r  = widget_button(b_row,value=self.bmpfiles[2],/bitmap,/flat,uname='renametab')
    sc_s  = widget_button(b_row,value=self.bmpfiles[1],/bitmap,/flat,uname='saveidl')
    sc_d  = widget_button(b_row,value=self.bmpfiles[4],/bitmap,/flat,uname='closetab')
    tabs  = widget_tab(([left,right])[self.handedness],tab_mode=0,uname='tabbase')
    for i=0,n_elements(*self.sequenceptr[n_elements(self.sequenceptr)-1])-1 do begin
        name = (*self.sequenceptr[n_elements(self.sequenceptr)-1])[i]
        tabi = widget_base(tabs,title=name,uname='tab_'+dm_to_string(i+1),xpad=0,ypad=0,uvalue=name)
        tbox = widget_text(tabi,value=(*self.sequenceptr[i]),/scroll,/editable,/all_events,scr_xsize=xysize[0],scr_ysize=xysize[1],uname='textbox',/context_events,font=self.font[0])
    endfor
    imenu = widget_button(b_col,value='Insert Commands',/menu)
    tmp   = widget_info(imenu,/geometry) 
    xysize[2] = xysize[2]>(tmp.scr_xsize+40)
    widget_control,imenu,scr_xsize=xysize[2]
    self->commands,imenu,scr_xsize=xysize[2]
    void1 = widget_button(b_col,xsize=xysize[2],value='Dryrun Sequence',uname='dryrun',sensitive=0)
    if allow_ex then $
    void2 = widget_button(b_col,value='Execute Sequence',uname='execute',sensitive=0)
    void3 = widget_button(b_col,value='Clear',uname='clear',sensitive=0)
    if self.idl_version ge 5.6 then begin
       widget_control,imenu,tooltip='Click to insert from a list of available commands and templates.' 
       widget_control,sc_s,tooltip='Save current tab.'
       widget_control,sc_r,tooltip='Rename current tab.'
       widget_control,sc_n,tooltip='Open a new tab.'
       widget_control,sc_d,tooltip='Close current tab.'
       widget_control,void1,tooltip='Dryrun the script. The sequence will not be sent to ICE server.'
       if allow_ex then $
       widget_control,void2,tooltip='Send the sequence to ICE server.'
       widget_control,void3,tooltip='Clear current tab.'
    endif
    sbox  = widget_text(self.tlb,/scroll,ysize=xysize[3],scr_xsize=xysize[0]+xysize[2],uname='statusbox',font=self.font[1])
    dm_center_kid,self.tlb,self.group_leader,/side ;centering the top level base
    widget_control,self.tlb,/realize,/map
    self->reset_button
    ;the following codes are for the offset of the sizes in resize event
    tmp0 = widget_info(self.tlb,/geometry)
    tmp1 = widget_info(([left,right])[self.handedness],/geometry)
    tmp2 = widget_info(b_col,/geometry)
    tmp3 = widget_info(tbox,/geometry)
    tmp4 = widget_info(b_row,/geometry) 
    self.xoffset = tmp0.xsize-tmp1.scr_xsize
    self.yoffset = tmp0.ysize-tmp1.scr_ysize
    self.xmargin = tmp1.scr_xsize-tmp3.scr_xsize
    self.ymargin = tmp1.scr_ysize-tmp3.scr_ysize
    self.xmin    = tmp4.xsize+self.xmargin
    self.ymin    = tmp2.ysize-tmp4.ysize+self.ymargin
    widget_control,self.tlb,set_uvalue=self
    xmanager,registerName,self.tlb,cleanup='dm_macs_sequencer_Exit',/no_block
    return,1
end

;object definition
pro dm_macs_sequencer__define
    void = {dm_macs_sequencer,          $
            group_leader:  0L,          $   ;group_leader id      
            tlb:           0L,          $   ;top level base
            handedness:    0b,          $   ;flag for layout handedness 0-right 1-left
            addwait:       0b,          $   ;flag for adding wait command to the beginning of sequence
            autocomplete:  0b,          $   ;flag for autocomplete in the editor
            workdir:       '',          $   ;derectory for saving files
            font:   strarr(2),          $   ;font
            sequenceptr:   ptrarr(21),  $   ;save the sequence content, maximu 20 tabs, the last saving tab names
            helpfile:      '',          $   ;helpfile location
            password:      '',          $   ;password
            bmpfiles:      strarr(9),   $   ;save bitmap file names ['macs.ico','mslice_'+['save','rename','new','delete','comment','demag','slit']+'.bmp','idl.ico']
            idl_version:   0e,          $   ;save idl version
            xoffset:       0L,          $   ;for resize event
            yoffset:       0L,          $   ;for resize event
            xmargin:       0L,          $   ;for resize event
            ymargin:       0L,          $   ;for resize event
            xmin:          0L,          $   ;minimum xsize for resize event
            ymin:          0L,          $   ;minimum ysize for resize event
            kidoptstr:     strarr(5)    $   ;save the kidney step optimization info
           }
end

pro dm_macs_sequencer,event,lefthanded=lefthanded,textboxsize=textboxsize,workdir=workdir
    if n_elements(event) ne 0 then $
       void = obj_new('dm_macs_sequencer',lefthanded=lefthanded,textboxsize=textboxsize,workdir=workdir,group_leader=event.top) $
    else $
       void = obj_new('dm_macs_sequencer',lefthanded=lefthanded,textboxsize=textboxsize,workdir=workdir)
end
