; $Id: $
;#######################################################################
; 
; NAME:
;  dm_load_macs
;
; PURPOSE:
;  load macs file
;  
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  July, 2025
; 
; HISTORY:
;  02/2013 V1
;  07/2013 Add magnetic field and specified data; take into account the temperture unit.
;  08/2013 Allow dm_load_macs to read a file content string array
;  03/2017 Allow reading histogram data format
;  12/2018 Allow reading event histogram data format
;  04/2020 Add event mode conversion routine
;  08/2022 Add nexus format
;  04/2023 Add rawdata option for event mode data
; 
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

@napi45   ;nexus library

;convert event mode data to histo2d data
;adapted from Tim Reeder's python code
pro dm_conv_macshisto2d


end

;returns the number of trailing channels in histogram mode
function dm_macs_histotail,n_chan
    return,([1,24])[n_chan eq 1024]
end

;read macs 2d histogram file
;input:
;  file:            full file name string
;  rawdata:         if set, data will not be normalized
;output:
;  qty:             data array [n_time,40,n_theta], even channel for spec, odd channel for diff
;  dqty:            error bar, only available if histohigh is the monitor count
;  t_start,t_end:   starting and ending histogram time, in unit of ms
;  t_bin:           bin width for time, in unit of millisecond
;  n_tbin:          number of time channels
;  theta_bin:       bin width for basesampletheta
;  theta_start,theta_end: starting and ending basesampletheta value
;  n_theta:         number of theta bins
;  a3_offset:       a3-kidney-basesampletheta
;  n_chan:          20 or 21
;  nan:             flag for whether nan existed in the 2d data, accumulative
;  histohigh:       store the extra histogram data, if existed  [n_theta,n_tbin, n_high]
;  zero_error:      return error bar for sqrt(intensity=1)
pro dm_load_macshisto2d,file,qty,dqty=dqty,theta_start=theta_start,theta_end=theta_end,theta_bin=theta_bin,t_start=t_start,t_end=t_end,t_bin=t_bin,n_tbin=n_tbin,n_theta=n_theta,$
    n_chan=n_chan,a3_offset=a3_offset,nan=nan,histohigh=histohigh,zero_error=zero_error,headeronly=headeronly,rawdata=rawdata
    fsize        = (file_info(file)).size/4-8
    openr,unit,file,/get_lun
    header       = fltarr(8)
    readu,unit,header
    if header[0] ne 1 then begin          ;header[0] will always be 1 to verify the binary file endianess
       free_lun,unit
       openr,unit,file,/get_lun,/swap_endian
       readu,unit,header
    endif
    t_start      = header[1]*1000.        ;in ms
    t_end        = header[2]*1000.        ;in ms
    n_tbin       = header[3]
    theta_start  = header[4]
    theta_end    = header[5]
    n_theta      = header[6]     
    a3_offset    = header[7]
    n_chan       = fix(fsize/(n_theta*n_tbin))
    ;check format
    if ((n_tbin mod 1) ne 0) or ((n_theta mod 1) ne 0) or (n_chan lt 20) or (n_chan gt 21) then switch_n = 1b
    if keyword_set(switch_n) then begin   ;old format, need to switch n_tbin to t_bin and n_theta to theta_bin
       t_bin     = header[3]*1000.        ;in ms
       theta_bin = header[6]
       n_tbin    = round(abs((header[2]-header[1])/header[3]),/L64)
       n_theta   = round(abs((theta_start-theta_end)/theta_bin),/L64)
       n_chan    = fix(fsize/(n_theta*n_tbin))
    endif else begin
       t_bin     = (t_end-t_start)/n_tbin  ;in ms
       theta_bin = float(abs(theta_end-theta_start))/n_theta
    endelse    
    if keyword_set(headeronly) then begin
       free_lun,unit
       return
    endif
    data1        = fltarr(n_theta,n_tbin,n_chan) ;data written in this order A3bin->next Tbin->next channel Tim Redder email 12/11/2018
    readu,unit,data1
    free_lun,unit
    if (n_chan gt 20) then histohigh = reform(data1[*,*,20],n_theta,n_tbin) ;in case n_tbin = 1
    data1        = data1[*,*,0:19]               ;only keep 20 channels
    ind          = where(data1 eq -999,cnt)
    nan          = keyword_set(nan) or (cnt ne 0)
    if nan then data1[ind] = !values.f_nan
    if n_elements(histohigh) ne 0 then begin    ;data with histohigh have not been normalized, do it here or set the nan values
       d_data1   = sqrt(data1)
       tmp       = histohigh
       ind       = where(histohigh eq 0,cnt,complement=ind1)
       if cnt gt 0 then begin
          tmp[ind] = 1.0
          nan    = 1b
       endif
       if keyword_set(rawdata) then tmp = 1.0
       for i=0,19 do begin
           tmp1  = data1[*,*,i]/tmp
           if cnt gt 0 then tmp1[ind] = !values.f_nan
           data1[*,*,i]   = tmp1
           tmp1  = d_data1[*,*,i]/tmp
           if cnt gt 0 then tmp1[ind] = !values.f_nan
           d_data1[*,*,i] = tmp1
       endfor
       if keyword_set(rawdata) then factor = 1.0 else factor = float((1.0d)/n_tbin/n_theta)
       d_data1   = d_data1*factor
       data1     = data1*factor
       zero_error = factor/max(tmp[ind1])
       histohigh = transpose(histohigh,shift(indgen(size(histohigh,/n_dim)),1))
    endif
    data1        = transpose(data1,[1,2,0]) ;->[n_tbin,20,n_theta]
    qty          = fltarr(n_tbin,40,n_theta)
    qty[*,[2*indgen(20)],*] = temporary(data1)
    if n_elements(d_data1) ne 0 then begin
       d_data1   = transpose(d_data1,[1,2,0]) ;->[n_tbin,20,n_theta]
       dqty      = fltarr(n_tbin,40,n_theta)
       dqty[*,[2*indgen(20)],*] = temporary(d_data1)
    endif
end

;return all the obj names contained in handl.icurrentvg
function dm_NXgetobjname,handle
    n_fields = H5G_GET_NUM_OBJS(handle.icurrentvg)
    if n_fields eq 0 then return,''
    names = strarr(n_fields)
    for i=0,n_fields-1 do names[i]=H5G_GET_OBJ_NAME_BY_IDX(handle.icurrentvg,i)
    return, names
end

;check whether name exist in the current handle.icurrentvg, and name is case insensitive
;keywords: 
;   objnames: input or return the object names of the top most group for the name variable
function dm_NXnameExisted, handle, name, nxclass, objnames=objnames
    if n_elements(nxclass) eq 0 then nxclass = 'NXcollection'
    tmpname = strsplit(name,'/',/extract,count=cnt0)
    for i=0,cnt0-1 do begin
        if i gt 0 then begin
           status = Nxopengroup(handle,tmpname[i-1],nxclass)
           if ~status then begin
              for j=1,i-1 do status = Nxclosegroup(handle)
              return,0b
           endif
        endif
        if (i eq 0) && (n_elements(objnames) ne 0) then namelist = temporary(objnames) else namelist = dm_NXgetobjname(handle)
        if i eq 0 then objnames = namelist
        ind1 = where(strmatch(namelist,tmpname[i],/fold_case),cnt1)
        if cnt1 eq 0 then begin
           for j=1,i do status = Nxclosegroup(handle)
           return,0b
        endif else tmpname[i] = namelist[ind1[0]]
    endfor
    for i=1,cnt0-1 do status = Nxclosegroup(handle)
    name = strjoin(tmpname,'/')
    return, 1b
end

;a clone of DAVEopenGetCloseData, but will check whether name of dataset exists in the current handle.icurrentvg, and name is case insensitive
function dm_NXopenGetCloseData, handle, name, data=data, attrNames=attrNames, attrData=attrData, objnames=objnames
    if dm_NXnameExisted(handle,name,'NXcollection',objnames=objnames) then $
       return, DAVENXopenGetCloseData(handle, name, data=data, attrNames=attrNames, attrData=attrData) $
    else return,0b
end

;a clone of NXopengroup, but will check whether name exists in the current handle.icurrentvg
function dm_NXopengroup, handle, name, nxclass
    if dm_NXnameExisted(handle,name,nxclass) then return,NXopengroup(handle,name,nxclass) else return,0b
end

;read macs nexus data file, called by dm_load_macs, using the same input and output keywords, can also be used to other instrument's nexus files
;for frontpolarization and backpolarization 0-up 1-down 
pro dm_load_macsnexus,file,data,info,data_histo=data_histo,dqty_histo=dqty_histo,str_histo=str_histo,all_neg=all_neg,answerno=answerno,answeryes=answeryes,comment=comment,checkonly=checkonly,$
    deteff=deteff,firstlineonly=firstlineonly,group_leader=group_leader,header=header,histodata=histodata,rawdata=rawdata,t_start=t_start,t_bin=t_bin,n_chan=n_chan,n_tbin=n_tbin,cut_tail=cut_tail,$
    is_histogram=is_histogram,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,maspacing=maspacing,npoints=npoints,pulse_histo=pulse_histo,temperatureunit=temperatureunit,$
    t_mcfx=t_mcfx,nan=nan,weight_histo=weight_histo,zero_error=zero_error,nomessage=nomessage,hfocus=hfocus,vfocus=vfocus,units=units,error=error
    error = 1b
    if n_elements(file) eq 0 then return else file = file[0]  ;only read one file at a time
    if (~file_test(file)) then message,"Can't open "+file+"."
    filename = file_basename(file)
    if (~H5F_IS_HDF5(file)) then message, filename+" is not in HDF5 format."
    
    if arg_present(npoints)     then npoints     = 0
    if arg_present(maspacing)   then maspacing   = fltarr(2)
    if arg_present(latticeparm) then latticeparm = replicate(!values.f_nan,6)
    if arg_present(latticeori)  then latticeori  = replicate(!values.f_nan,6)
    units = ['','','']
    
    idl_version = dm_to_number(!version.release)
    
    ; file handle must be a long integer and must be set before calling Nxopen()
    handle = 0L
    
    ; Open the hdf file
    ; The file handle will subsequently be used just like a file unit
    if (~Nxopen(file,'NXACC_READ',handle)) then message,"Can't open "+file+"."
    
    ; Determine the name of the first main group/entry in the file
    ; Required because groups are opened by name
    nEntry = H5g_get_nmembers(handle.ivid, '/')
    if (nEntry lt 1) then message,filename+" contains no entries."
    
    entryIndex = 0
    group_name = '/'
    entryName = H5g_get_member_name(handle.ivid, group_name, entryIndex)

    ; open the main NXentry group
    if ~Nxopengroup(handle,entryName,'NXentry') then message,"Cannot open " + entryName+" in "+filename+"."
    
    header = '#'
    if idl_version le 8.82 then begin
       status = dm_NXopenGetCloseData(handle,'program_name',data=tmp)
       version = ''
    endif else begin
       status = dm_NXopenGetCloseData(handle,'program_name',data=tmp,attrNames='version',attrData=version)
    endelse
    if status then header = header+tmp+' '+version[0]
    
    if dm_NXopenGetCloseData(handle,'start_time',data=starttime) then header = [header,'#Date        '+starttime]
    
    ; open DAS_logs group
    status = Nxopengroup(handle,'DAS_logs','NXcollection')
    status = Nxopengroup(handle,'experiment','NXcollection')        ; open /DAS_logs/experiment group
    if arg_present(comment) then begin
       if dm_NXopenGetCloseData(handle,'description',data=tmp) then comment = strtrim(tmp,2) else comment = ''
       header = [header,'#Comment     '+((strlen(comment) gt 0)?'"'+comment+'"':'')]
    endif
    if dm_NXopenGetCloseData(handle,'instrument',data=tmp) then begin
       instrname = strlowcase(strtrim(tmp,2))
       header = [header,'#InstrName   '+instrname]
    endif else instrname = ''
    status = Nxclosegroup(handle)
    ;figure out number of data point
    if dm_NXopenGetCloseData(handle,'counter/liveROI',data=tmp_roi,objnames=das_names) then begin
       npoints = n_elements(tmp_roi)
       if dm_NXopenGetCloseData(handle,'counter/stopTime',data=tmp,objnames=das_names) then begin
          if n_elements(tmp) ne npoints then begin
             npoints = n_elements(tmp)
             status = dm_NXopenGetCloseData(handle,'pointDetector/liveROI',data=tmp_roi,objnames=das_names) 
          endif
       endif
    endif else message,"Cannot read "+entryName+"/DAS_logs/counter/liveROI."
    ;figure out number of detectors (applies to bt7 files)
    if dm_NXopenGetCloseData(handle,'diffDetector/detectorEfficiency',data=tmp,objnames=das_names) then ndet = n_elements(tmp) else ndet = 1
    if arg_present(deteff) then deteff = fltarr(2,ndet)
    header = [header,'#Npoints     '+dm_to_string(npoints)]
    tmp = where(stregex(das_names,'^ana(lyzer)*theta$',/boolean,/fold_case),cnt)
    if cnt gt 0 then anath = das_names[tmp[0]] else anath = 'analyzerThetaMotor'
    tmp = where(stregex(das_names,'^ana(lyzer)*twotheta$',/boolean,/fold_case),cnt)
    if cnt gt 0 then anatth = das_names[tmp[0]] else anatth = 'analyzerTwoThetaMotor'
    tmp = where(stregex(das_names,'^(sample){0,1}l[a-z]*tilt$',/boolean,/fold_case),cnt)
    if cnt gt 0 then ltilt = das_names[tmp[0]] else ltilt = 'ltilt'
    tmp = where(stregex(das_names,'^(sample){0,1}u[a-z]*tilt$',/boolean,/fold_case),cnt)
    if cnt gt 0 then utilt = das_names[tmp[0]] else utilt = 'utilt'
    tmp = where(stregex(das_names,'^samplex[a-z]*trans',/boolean,/fold_case),cnt)
    if cnt gt 0 then xtran = [das_names[tmp[0]],'smplx'] else begin
       tmp = where(stregex(das_names,'^samplel[a-z]*trans',/boolean,/fold_case),cnt)
       if cnt gt 0 then xtran = [das_names[tmp[0]],'smplltrn'] else xtran = ['sampleXTranslationMotor','smplx']
    endelse
    tmp = where(stregex(das_names,'^sampley[a-z]*trans',/boolean,/fold_case),cnt)
    if cnt gt 0 then ytran = [das_names[tmp[0]],'smply'] else begin
       tmp = where(stregex(das_names,'^sampleu[a-z]*trans',/boolean,/fold_case),cnt)
       if cnt gt 0 then ytran = [das_names[tmp[0]],'smplutrn'] else ytran = ['sampleYTranslationMotor','smply']
    endelse
    tmp = where(stregex(das_names,'hslit',/boolean,/fold_case),cnt)
    if cnt gt 0 then hslit = das_names[tmp[0]] else begin
       tmp = where(stregex(das_names,'^(presample)*slitwidth',/boolean,/fold_case),cnt)
       if cnt gt 0 then hslit = das_names[tmp[0]] else hslit = 'hSlitMotor'
    endelse
    tmp = where(stregex(das_names,'vslit',/boolean,/fold_case),cnt)
    if cnt gt 0 then vslit = das_names[tmp[0]] else begin
       tmp = where(stregex(das_names,'^(presample)*slitheight',/boolean,/fold_case),cnt)
       if cnt gt 0 then vslit = das_names[tmp[0]] else vslit = 'vSlitMotor'
    endelse
    tmp20      = string(indgen(20)+1,format='(i02)')
    tmp21      = string(indgen(21)+1,format='(i02)')
    tmp13      = string(indgen(13)+1,format='(i02)')
    tmp48      = string(indgen(48),format='(i02)')
    tmp11      = string(indgen(11),format='(i02)')
    info       = ['h','k','l','E','Ef','Ei','a1','a2','a3','a4','a5','a6','time','monitor','detector','AColMon','BColMon','BaseSampleTheta','cfxbe','cfxhopg','cfxmgf','Beta1','Beta2',$
                 'DFMDTS','DIFF','DMBT','Focus','Kidney','MBTSlide','MCFX','MonRot','MonTrans','a2ref','mbtPivot','PTAI','hslit','vslit','SPEC','SmplLTilt','SmplUTilt',xtran[1],ytran[1],$
                 'SmplZ','VBAH','VBAV']
    j          = 0
    ind_h      = (j++)
    ind_k      = (j++)
    ind_l      = (j++)
    ind_en     = (j++)
    ind_ef     = (j++)
    ind_ei     = (j++)
    ind_a1     = (j++)
    ind_a2     = (j++)
    ind_a3     = (j++)
    ind_a4     = (j++)
    ind_a5     = (j++)
    ind_a6     = (j++)
    ind_time   = (j++)
    ind_mon    = (j++)
    ind_det    = (j++)
    ind_acol   = (j++)
    ind_bcol   = (j++)
    ind_bst    = (j++)
    ind_cfxbe  = (j++)
    ind_cfxpg  = (j++)
    ind_cfxmg  = (j++)
    ind_beta1  = (j++)
    ind_beta2  = (j++)
    ind_dfmdts = (j++)
    ind_diff   = (j++)
    ind_dmbt   = (j++)
    ind_focus  = (j++)
    ind_kidney = (j++)
    ind_slide  = (j++)
    ind_mcfx   = (j++)
    ind_monrot = (j++)
    ind_montra = (j++)
    ind_a2ref  = (j++)
    ind_pivot  = (j++)
    ind_ptai   = (j++)
    ind_hslit  = (j++)
    ind_vslit  = (j++)
    ind_spec   = (j++)
    ind_ltilt  = (j++)
    ind_utilt  = (j++)
    ind_smplx  = (j++)
    ind_smply  = (j++)
    ind_smplz  = (j++)
    ind_vbah   = (j++)
    ind_vbav   = (j++)
    if instrname eq 'bt7' then begin
       info = strlowcase([info,'bksltwdth','bkslthght','anablade'+tmp13,'ddc'+['0','1','2'],'tdc'+tmp11,'psdc'+tmp48,'sdc'+['0','1','2']])
       ind_bhslt = (j++)
       ind_bvslt = (j++)
       ind_ana01 = j
       j = j+13
       ind_ddc00 = j
       j = j+3
       ind_tdc00 = j
       j = j+11
       ind_psd00 = j
       j = j+48
       ind_sdc00 = j
       j = j+3
       vba_names = ['preMonoSlitWidth','preMonoSlitHeight']
    endif else begin
       info = strlowcase([info,'AnalyzerTheta'+tmp20,'MonBlade'+tmp21,'DIFF'+tmp20,'SPEC'+tmp20])
       ind_ana01 = j
       j = j+20
       ind_monb01 = j
       j = j+21
       ind_diff01 = j
       j = j+20
       ind_spec01 = j
       j = j+20
       vba_names = ['monoVbaHMotor','monoVbaVMotor']
    endelse       
    if dm_NXnameExisted(handle,'temp',objnames=das_names) then begin
       info = [info,'temp','temperaturecontrolreading', 'temperaturesetpoint']
       ind_temp = (j++)
       ind_tcon = (j++)
       ind_tset = (j++)
    endif
    if dm_NXnameExisted(handle,'mag',objnames=das_names) then begin
       info = [info,'magfield']
       ind_mag = (j++)
    endif
    if dm_NXopenGetCloseData(handle,'frontPolarization/direction',data=tmp_frontpol,objnames=das_names) then begin
       info = [info,'frontpolarization']
       ind_fpol = (j++)
    endif
    if dm_NXopenGetCloseData(handle,'backPolarization/direction',data=tmp_backpol,objnames=das_names) then begin
       info = [info,'backpolarization']
       ind_bpol = (j++)
    endif
    info = [info,'timestamp']
    ind_tstamp = (j++)
    data = dblarr(npoints,n_elements(info))
    data[*,ind_det] = temporary(tmp_roi)
    
    status = Nxopengroup(handle,'counter','NXcollection')           ; open /DAS_logs/counter group
    if dm_NXopenGetCloseData(handle,'countAgainst',data=tmp) then begin
       tmp = strtrim(tmp[0],2)
       if stregex(tmp,'monitor',/fold_case,/boolean) then tmp = 'monitor' 
       ind = where(info eq strlowcase(tmp),cnt)
       header = [header,'#Reference   '+((cnt gt 0)?dm_to_string(ind[0]+1)+' ':'')+tmp]
    endif
    if dm_NXopenGetCloseData(handle,'primaryDetector',data=tmp) then begin
       if stregex(tmp,'diff',/boolean,/fold_case) then tmp = 'DIFF' $
       else if stregex(tmp,'spec',/boolean,/fold_case) then tmp = 'SPEC' $
       else if stregex(tmp,'monitor',/boolean,/fold_case) then tmp = 'MONITOR' $
       else if stregex(tmp,'pointdetector',/boolean,/fold_case) then tmp = 'DETECTOR' 
       ind = where(info eq strlowcase(tmp),cnt)
       header = [header,'#Signal      '+((cnt gt 0)?dm_to_string(ind[0]+1)+' ':'')+tmp]
    endif
    if idl_version le 8.82 then begin  ;get attribute not working, use the starttime from entry
       status = dm_NXopenGetCloseData(handle,'stopTime',data=tmp)
    endif else begin
       status = dm_NXopenGetCloseData(handle,'stopTime',data=tmp,attrNames='start',attrData=starttime)
    endelse
    if status then data[*,ind_tstamp] = dm_to_number(starttime[0],/epoch,/double)+temporary(tmp)
    if dm_NXopenGetCloseData(handle,'liveTime',data=tmp) then data[*,ind_time] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'startTime',data=tmp) then begin
       ind = where(tmp lt 0,cnt)
       if cnt gt 0 then begin   ;liveTime is wrong with negative startTime, usually the first one, or incorrect number of liveTime
          if dm_NXopenGetCloseData(handle,'stopTime',data=tmp1) then data[*,ind_time] = tmp1-(0>tmp)
       endif
    endif
    if dm_NXopenGetCloseData(handle,'liveMonitor',data=tmp) then data[*,ind_mon] = temporary(tmp)
    status = Nxclosegroup(handle)
    if dm_NXopenGetCloseData(handle,'CFXBE/primaryNode',data=tmp,objnames=das_names) then data[*,ind_cfxbe] = stregex(tmp,'down',/boolean,/fold_case)
    if dm_NXopenGetCloseData(handle,'CFXHOPG/primaryNode',data=tmp,objnames=das_names) then data[*,ind_cfxpg] = stregex(tmp,'down',/boolean,/fold_case)
    if dm_NXopenGetCloseData(handle,'mcfxMap/key',data=tmp,objnames=das_names) then data[*,ind_mcfx] = 0>(where(['beo','be','hopg'] eq strlowcase(strtrim(tmp[0],2))))  ;0-BeO 1-Be 2-PG
    if dm_NXopenGetCloseData(handle,'analyzerFilterTemp/primaryNode/value',data=tmp,objnames=das_names) then t_mcfx = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'aColmon/primaryNode',data=tmp,objnames=das_names) then data[*,ind_acol] = stregex(tmp,'in',/boolean,/fold_case)
    if dm_NXopenGetCloseData(handle,'bColmon/primaryNode',data=tmp,objnames=das_names) then data[*,ind_bcol] = stregex(tmp,'in',/boolean,/fold_case)
    if dm_NXopenGetCloseData(handle,vba_names[0]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_vbah] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,vba_names[1]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_vbav] = temporary(tmp)
    if dm_Nxopengroup(handle,'monoTheta','NXcollection') then begin   ; open /DAS_logs/monoTheta group
       if dm_NXopenGetCloseData(handle,'theta',data=tmp) then data[*,ind_a1] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'radiusOfCurvature1',data=tmp) then data[*,ind_focus] = temporary(tmp) else data[*,ind_focus] = 10000  ;default vfocus flat
       if dm_NXopenGetCloseData(handle,'hFocus',data=tmp) then hfocus = strtrim((temporary(tmp))[0],2)
       if dm_NXopenGetCloseData(handle,'vFocus',data=tmp) then vfocus = strtrim((temporary(tmp))[0],2)
       status = Nxclosegroup(handle)
    endif
    if keyword_set(firstlineonly) then begin  ;no need to continue
       status = Nxclosegroup(handle)  ; DAS_logs
       status = Nxclosegroup(handle)  ; NXentry
       status = Nxclose(handle)       ; Nexus/hdf file
       error  = 0
       return
    endif
    if dm_Nxopengroup(handle,'temp','NXcollection') then begin  ; open /DAS_logs/temp group, not always present          
       status = dm_NXopenGetCloseData(handle,'primarySensor',data=id_samp)
       status = dm_NXopenGetCloseData(handle,'primaryControlLoop',data=id_loop)
       status = dm_NXopenGetCloseData(handle,'controlLoopSensor_'+dm_to_string(id_loop[0]),data=id_cont)
       if dm_NXopenGetCloseData(handle,'sensor_'+id_samp+'/average_value',data=tmp) then data[*,ind_temp] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'sensor_'+id_cont+'/average_value',data=tmp) then data[*,ind_tcon] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'desiredPrimaryNode',data=tmp)               then data[*,ind_tset] = temporary(tmp)
       status = Nxclosegroup(handle)
    endif 
    if dm_NXopenGetCloseData(handle,'mag/primaryNode/average_value',data=tmp,objnames=das_names) then data[*,ind_mag] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'PTAI/index',data=tmp,objnames=das_names) then data[*,ind_ptai] = temporary(tmp)
    if dm_Nxopengroup(handle,'Q','NXcollection') then begin ; open /DAS_logs/Q group
       if dm_NXopenGetCloseData(handle,'H',data=tmp) then data[*,ind_h] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'K',data=tmp) then data[*,ind_k] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'L',data=tmp) then data[*,ind_l] = temporary(tmp)
       status = Nxclosegroup(handle)
    endif
    if dm_NXopenGetCloseData(handle,anath+'/softPosition',data=tmp,objnames=das_names)  then data[*,ind_a5] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,anath+'/primaryNode',data=tmp,objnames=das_names)   then data[*,ind_a5] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,anatth+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_a6] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,anatth+'/primaryNode',data=tmp,objnames=das_names)  then data[*,ind_a6] = temporary(tmp)
    if instrname eq 'bt7' then begin
       if idl_version le 8.82 then begin
          if dm_NXopenGetCloseData(handle,'postSampleSlitWidth/softPosition',data=tmp,objnames=das_names) then data[*,ind_bhslt] = temporary(tmp)
       endif else begin
          if dm_NXopenGetCloseData(handle,'postSampleSlitWidth/softPosition',data=tmp,attrNames='units',attrdata=tmpunit2,objnames=das_names) then data[*,ind_bhslt] = temporary(tmp)
          if n_elements(tmpunit2) gt 0 then units[2] = tmpunit2[0]
       endelse
       if dm_NXopenGetCloseData(handle,'postSampleSlitHeight/softPosition',data=tmp,objnames=das_names) then data[*,ind_bvslt] = temporary(tmp)
       for i=0,12 do begin
           if dm_NXopenGetCloseData(handle,'anaBlade'+tmp13[i]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_ana01+i] = temporary(tmp)
       endfor
       if dm_NXopenGetCloseData(handle,'diffDetector/counts',data=tmp,objnames=das_names) && (n_elements(tmp) eq npoints*3.0)   then data[*,ind_ddc00:(ind_ddc00+2)] = transpose(temporary(tmp))
       if dm_NXopenGetCloseData(handle,'doorDetector/counts',data=tmp,objnames=das_names) && (n_elements(tmp) eq npoints*11.0)  then data[*,ind_tdc00:(ind_tdc00+10)] = transpose(temporary(tmp))
       if dm_NXopenGetCloseData(handle,'singleDetector/counts',data=tmp,objnames=das_names) && (n_elements(tmp) eq npoints*3.0) then data[*,ind_sdc00:(ind_sdc00+2)] = transpose(temporary(tmp))
       ;if dm_NXopenGetCloseData(handle,'positionsensitiveDetector/counts',data=tmp) then data[*,ind_psd00:(ind_psd00+47)] = transpose(temporary(tmp))
    endif else if instrname eq 'macs' then begin
       for i=0,20 do begin
           if dm_NXopenGetCloseData(handle,'monoBlade'+tmp21[i]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_monb01+i] = temporary(tmp)
       endfor
       for i=0,19 do begin
           if dm_NXopenGetCloseData(handle,'anaTwoTheta'+tmp20[i]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_ana01+i] = temporary(tmp)/2.0 ;anatheta in ICE column format
       endfor
       if dm_NXopenGetCloseData(handle,'diffDetector/counts',data=tmp,objnames=das_names) && (n_elements(tmp) eq npoints*20.0) then data[*,ind_diff01:(ind_diff01+19)] = transpose(temporary(tmp))
       if dm_NXopenGetCloseData(handle,'specDetector/counts',data=tmp,objnames=das_names) && (n_elements(tmp) eq npoints*20.0) then data[*,ind_spec01:(ind_spec01+19)] = transpose(temporary(tmp))
       if dm_NXopenGetCloseData(handle,'beta1Motor/softPosition',data=tmp,objnames=das_names) then data[*,ind_beta1] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'beta2Motor/softPosition',data=tmp,objnames=das_names) then data[*,ind_beta2] = temporary(tmp)
    endif
    if arg_present(deteff) then begin
       if dm_NXopenGetCloseData(handle,'diffDetector/detectorEfficiency',data=tmp,objnames=das_names) then deteff[1,*] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'specDetector/detectorEfficiency',data=tmp,objnames=das_names) then deteff[0,*] = temporary(tmp)
       header = [header,'#DetectorEfficiencies '+strjoin(transpose([[replicate('SPEC',ndet)+tmp20[0:(ndet-1)]],[replicate('DIFF',ndet)+tmp20[0:(ndet-1)]]])+'='+dm_to_string(deteff,resolution=5),' ',/single)+' Monitor=1.0']
    endif
    if dm_NXopenGetCloseData(handle,'dmbtMotor/softPosition',data=tmp,objnames=das_names) then data[*,ind_dmbt] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'ef/energy',data=tmp,objnames=das_names) then data[*,ind_ef] = temporary(tmp) $
    else if dm_NXopenGetCloseData(handle,'ef/primaryNode',data=tmp,objnames=das_names) then data[*,ind_ef] = temporary(tmp) $
    else if dm_NXopenGetCloseData(handle,'finalEnergy/energy',data=tmp,objnames=das_names) then data[*,ind_ef] = temporary(tmp) $
    else data[*,ind_ef] = !values.f_nan
    if arg_present(maspacing) then begin
       if dm_NXopenGetCloseData(handle,'ef/dSpacing',data=tmp,objnames=das_names) then maspacing[1] = tmp $
       else if dm_NXopenGetCloseData(handle,'finalEnergy/dSpacing',data=tmp,objnames=das_names) then maspacing[1] = tmp
       if dm_NXopenGetCloseData(handle,'ei/dSpacing',data=tmp,objnames=das_names) then maspacing[0] = tmp $
       else if dm_NXopenGetCloseData(handle,'initialEnergy/dSpacing',data=tmp,objnames=das_names) then maspacing[0] = tmp
       header = [header,'#MonoSpacing '+dm_to_string(maspacing[0])]
       header = [header,'#AnaSpacing  '+dm_to_string(maspacing[1])]
    endif
    if dm_NXopenGetCloseData(handle,'ei/energy',data=tmp,objnames=das_names) then data[*,ind_ei] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,'initialEnergy/energy',data=tmp,objnames=das_names) then data[*,ind_ei] = temporary(tmp) $
    else data[*,ind_ei] = !values.f_nan
    if dm_NXopenGetCloseData(handle,'et/deltaE',data=tmp,objnames=das_names) then data[*,ind_en] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,'et/energyTransfer',data=tmp,objnames=das_names) then data[*,ind_en] = temporary(tmp) $
    else data[*,ind_en] = !values.f_nan
    if dm_NXopenGetCloseData(handle,'kidneyMotor/softPosition',data=tmp,objnames=das_names)   then data[*,ind_kidney] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'mbtSlideMotor/softPosition',data=tmp,objnames=das_names) then data[*,ind_slide]  = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'monoDTSMotor/softPosition',data=tmp,objnames=das_names)  then data[*,ind_dfmdts] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'monoRotation/softPosition',data=tmp,objnames=das_names)  then data[*,ind_monrot] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'monoTrans/softPosition',data=tmp,objnames=das_names)     then data[*,ind_montra] = temporary(tmp)
    if dm_Nxopengroup(handle,'monoTwoTheta','NXcollection') then begin    ; open /DAS_logs/monoTwoTheta group
       if dm_NXopenGetCloseData(handle,'a2Ref',data=tmp) then data[*,ind_a2ref] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'mbtPivot',data=tmp) then data[*,ind_pivot] = temporary(tmp)
       if dm_NXopenGetCloseData(handle,'primaryNode',data=tmp) then data[*,ind_a2] = temporary(tmp)
       status = Nxclosegroup(handle)
    endif
    if dm_NXopenGetCloseData(handle,'sampleElev/softPosition',data=tmp,objnames=das_names) then data[*,ind_smplz] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,ltilt+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_ltilt] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,utilt+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_utilt] = temporary(tmp)
    if idl_version le 8.82 then begin
       if dm_NXopenGetCloseData(handle,xtran[0]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_smplx] = temporary(tmp) 
    endif else begin
       if dm_NXopenGetCloseData(handle,xtran[0]+'/softPosition',data=tmp,attrNames='units',attrdata=tmpunit,objnames=das_names) then data[*,ind_smplx] = temporary(tmp)
       if n_elements(tmpunit) gt 0 then units[0] = tmpunit[0]
    endelse
    if dm_NXopenGetCloseData(handle,ytran[0]+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_smply] = temporary(tmp)
    if (arg_present(latticeparm) or arg_present(latticeori)) && dm_Nxopengroup(handle,'sampleState','NXcollection') then begin ; open /DAS_logs/sampleState group     
       if arg_present(latticeparm) then begin
          if dm_NXopenGetCloseData(handle,'A',data=tmp)     then latticeparm[0] = tmp[0]
          if dm_NXopenGetCloseData(handle,'B',data=tmp)     then latticeparm[1] = tmp[0]
          if dm_NXopenGetCloseData(handle,'C',data=tmp)     then latticeparm[2] = tmp[0]
          if dm_NXopenGetCloseData(handle,'Alpha',data=tmp) then latticeparm[3] = tmp[0]
          if dm_NXopenGetCloseData(handle,'Beta',data=tmp)  then latticeparm[4] = tmp[0]
          if dm_NXopenGetCloseData(handle,'Gamma',data=tmp) then latticeparm[5] = tmp[0]
          header = [header,'#Lattice     '+dm_to_string(latticeparm,sep=' ')]
       endif
       if arg_present(latticeori) then begin
          if dm_NXopenGetCloseData(handle,'refPlane1',data=tmp) then latticeori[0:2] = tmp[0:((n_elements(tmp)-1)<2)]
          if dm_NXopenGetCloseData(handle,'refPlane2',data=tmp) then latticeori[3:5] = tmp[0:((n_elements(tmp)-1)<2)]
          header = [header,'#Orient      '+dm_to_string(latticeori,sep=' ')]
       endif
       status = Nxclosegroup(handle)
    endif
    if dm_NXopenGetClosedata(handle,'sampleThetamotor/softPosition',data=tmp,objnames=das_names) then data[*,ind_bst] = temporary(tmp) ;basesampletheta for macs
    if dm_NXopenGetCloseData(handle,'sampleTheta/primaryNode',data=tmp,objnames=das_names) then data[*,ind_a3] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,'sampleTheta/softPosition',data=tmp,objnames=das_names) then data[*,ind_a3] = temporary(tmp) else data[*,ind_a3] = data[*,ind_bst]
    if dm_NXopenGetCloseData(handle,'sampleTwoTheta/position',data=tmp,objnames=das_names) then data[*,ind_a4] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,'sampleTwoTheta/softPosition',data=tmp,objnames=das_names) then data[*,ind_a4] = temporary(tmp) else $
    if dm_NXopenGetCloseData(handle,'sampleTwoThetaMotor/softPosition',data=tmp,objnames=das_names) then data[*,ind_a4] = temporary(tmp)
    if idl_version le 8.82 then begin
       if dm_NXopenGetCloseData(handle,hslit+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_hslit] = temporary(tmp)
    endif else begin
       if dm_NXopenGetCloseData(handle,hslit+'/softPosition',data=tmp,attrNames='units',attrdata=tmpunit1,objnames=das_names) then data[*,ind_hslit] = temporary(tmp)
       if n_elements(tmpunit1) gt 0 then units[1] = tmpunit1[0]
    endelse
    if dm_NXopenGetCloseData(handle,vslit+'/softPosition',data=tmp,objnames=das_names) then data[*,ind_vslit] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'trajectoryData/description',data=tmp,objnames=das_names) then header = [header,'#ScanDescr   '+tmp]
    if dm_NXopenGetCloseData(handle,'trajectoryData/xAxis',data=tmp,objnames=das_names) && (strlen(tmp) gt 0) then begin
       tmp  = (strsplit(tmp,'.',/extract))[stregex(tmp,'^q\.[hkl]$',/boolean,/fold_case)]
       tmp1 = ['qx','qy','qz','monotheta','monotwotheta','sampletheta','sampletwotheta','kidney',anath,anatth,ltilt,utilt,xtran[0],ytran[0],'sampleelev','vbah','vbav']
       tmp2 = ['H','K','L','A1','A2','A3','A4','A4','A5','A6','smplltilt','smplutilt',xtran[1],ytran[1],'smplz','vbah','vbav']
       for i=0,n_elements(tmp1)-1 do begin
           if stregex(tmp,tmp1[i],/boolean,/fold_case) then begin
              tmp = tmp2[i]
              break
           endif
       endfor
       ind  = where(info eq strlowcase(tmp),cnt)
       header = [header,'#Scan        '+((cnt gt 0)?dm_to_string(ind[0]+1)+' ':'')+tmp]
    endif
    if dm_NXopenGetCloseData(handle,'spec/counts',data=tmp,objnames=das_names) then data[*,ind_spec] = temporary(tmp)
    if dm_NXopenGetCloseData(handle,'diff/counts',data=tmp,objnames=das_names) then data[*,ind_diff] = temporary(tmp)
    if n_elements(ind_fpol) ne 0 then data[*,ind_fpol] = ~stregex(tmp_frontpol,'up',/boolean,/fold_case)
    if n_elements(ind_bpol) ne 0 then data[*,ind_bpol] = ~stregex(tmp_backpol,'up',/boolean,/fold_case)

    status = Nxclosegroup(handle)     ; DAS_logs
    status = Nxclosegroup(handle)     ; NXentry
    status = Nxclose(handle)          ; Nexus/hdf file
    error = 0
end

;read macs nice ascii data file, called by dm_load_macs, using the same input and output keywords
;noadjust:  if set, no manipulation of data info
;niceinfo:  save the original column names
;hfocus:    monochromator horizontal focusing state string   
;vfocus:    monochromator vertical focusing state string
;units:     unit of [sample table translation, front slit, back slit] 'cm' or 'mm'
pro dm_load_macsnice,file,data,info,niceinfo=niceinfo,data_histo=data_histo,dqty_histo=dqty_histo,str_histo=str_histo,all_neg=all_neg,answerno=answerno,answeryes=answeryes,comment=comment,$
    checkonly=checkonly,deteff=deteff,firstlineonly=firstlineonly,group_leader=group_leader,header=header,histodata=histodata,rawdata=rawdata,t_start=t_start,t_bin=t_bin,n_chan=n_chan,n_tbin=n_tbin,cut_tail=cut_tail,$
    is_histogram=is_histogram,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,maspacing=maspacing,npoints=npoints,pulse_histo=pulse_histo,string=string,temperatureunit=temperatureunit,$
    trailer=trailer,t_mcfx=t_mcfx,nan=nan,weight_histo=weight_histo,zero_error=zero_error,noadjust=noadjust,nomessage=nomessage,hfocus=hfocus,vfocus=vfocus,units=units,error=error
    if n_elements(data) ne 0    then tmp = temporary(data)
    if n_elements(header) ne 0  then tmp = temporary(header)
    if n_elements(trailer) ne 0 then tmp = temporary(trailer)
    if n_elements(hfocus) ne 0  then tmp = temporary(hfocus)
    if n_elements(vfocus) ne 0  then tmp = temporary(vfocus)
    if arg_present(npoints)     then npoints     = 0
    if arg_present(maspacing)   then maspacing   = fltarr(2)
    if arg_present(latticeparm) then latticeparm = replicate(!values.f_nan,6)
    if arg_present(latticeori)  then latticeori  = replicate(!values.f_nan,6)
    if arg_present(deteff)      then deteff      = fltarr(2,20)
    zerostring = ['beo','nsf','up']        ;strings for 0
    onestring  = ['be','sf','down']        ;strings for 1
    twostring  = ['hopg']                  ;strings for 2
    units      = ['','','']
    
    
    n_line = file_lines(file)
    if n_line eq 0 then return
    lines = strarr(n_line)
    openr,unit,file,/get_lun,error=error
    if keyword_set(error) then return
    readf,unit,lines
    free_lun,unit
    ind = where(strlen(strtrim(lines,2)) gt 0,cnt,ncomplement=cnt1)
    if (cnt1 gt 0) and (cnt gt 0) then begin
       lines = lines[ind]
       n_line = cnt
    endif
    ind = (where(stregex(lines,'^## pt',/boolean,/fold_case),cnt))[0]
    header = lines[0:(ind-(cnt<1))]
    if cnt gt 0 then begin
       info = strsplit(lines[ind],' '+string(9b),/extract,count=n_info)
       info = strlowcase(info[1:*]) & n_info = n_info-1
       ind_ref1 = where(stregex(info,'refplane1',/boolean),cnt_ref1)
       if keyword_set(noadjust) then hklstr = ['','',''] else hklstr = ['h','k','l']
       if cnt_ref1 eq 1 then begin
          info = [info[0:(ind_ref1[0]-1)],info[ind_ref1[0]]+hklstr,info[(ind_ref1[0]+1):*]]
          n_info = n_info+2
       endif
       ind_ref2 = where(stregex(info,'refplane2',/boolean),cnt_ref2)
       if cnt_ref2 eq 1 then begin
          info = [info[0:(ind_ref2[0]-1)],info[ind_ref2[0]]+hklstr,info[(ind_ref2[0]+1):*]]
          n_info = n_info+2
       endif
       if ind+1 lt n_line then begin
          lines = lines[ind+1:*]
          ind1  = where(stregex(lines,'^#',/boolean),cnt1,complement=ind2,ncomplement=cnt2)
          if cnt1 gt 0 then trailer = lines[ind1]
          if cnt2 gt 0 then begin
             data = strarr(cnt2,n_info)
             for ii=0L,cnt2-1 do begin
                 tmp1 = strsplit(lines[ind2[ii]],' '+string(9b),/extract,count=n_tmp1)
                 if n_info ne n_tmp1 then begin
                    if n_elements(inc_ans) eq 0 then begin
                       inc_ans = dialog_message([file+' has inconsistent number of columns.','Do you want to continue?','','If yes, all missing data will be set to 0.'],/default_no,/question,/center,dialog_parent=group_leader,title='Please confirm:')
                       if strmatch(inc_ans,'no',/fold_case) then begin
                          error = 1b & return
                       endif
                    endif
                    if n_tmp1 lt n_info then tmp1 = [tmp1,replicate('',n_info-n_tmp1)]
                 endif
                 if arg_present(latticeori) and (ii eq 0) then begin
                    ind_ref1 = where(stregex(info,'refplane1',/boolean),cnt_ref1) ;might have been changed by cnt_ref2
                    if cnt_ref1 gt 0 then latticeori[0:2] =  tmp1[ind_ref1[0]:(ind_ref1[0]+2)]
                    if cnt_ref2 gt 0 then latticeori[3:5] =  tmp1[ind_ref2[0]:(ind_ref2[0]+2)]
                 endif
                 data[ii,*] = tmp1[0:(n_info-1)]
             endfor
             if ~keyword_set(string) then begin
                for i=0,n_elements(zerostring)-1 do begin
                    ind = where(strmatch(data,zerostring[i],/fold_case),cnt)
                    if cnt gt 0 then data[ind] = '0'
                endfor
                for i=0,n_elements(onestring)-1 do begin
                    ind = where(strmatch(data,onestring[i],/fold_case),cnt)
                    if cnt gt 0 then data[ind] = '1'
                endfor
                for i=0,n_elements(twostring)-1 do begin
                    ind = where(strmatch(data,twostring[i],/fold_case),cnt)
                    if cnt gt 0 then data[ind] = '2'
                endfor
                data = dm_to_number(data,/double)
             endif
          endif
       endif
    endif
    if keyword_set(noadjust) or (n_elements(data) eq 0) then return
    
    ;extrac header info
    ind = (where(stregex(header,'# *experiment\.instrument',/boolean,/fold_case),cnt))[0]
    if cnt eq 1 then instrname = strlowcase(strtrim(strmid(header[ind],strpos(header[ind],'instrument')+10),2)) else instrname = 'macs'
    ind = (where(stregex(header,'# *experiment\.description',/boolean,/fold_case),cnt))[0]
    if cnt eq 1 then begin
       comment = strtrim(strmid(header[ind],strpos(header[ind],'description')+11),2)
       if (strmid(comment,0,1) eq '"') and (strmid(comment,0,1,/reverse_offset) eq '"') then comment = strmid(comment,1,strlen(comment)-2)
    endif
    ind = (where(stregex(header,'# *mcfxMap',/boolean,/fold_case),cnt))[0]
    if cnt eq 1 then mcfx = strtrim(strmid(header[ind],strpos(header[ind],'ap')+2),2)
    ind = (where(stregex(header,'# *a1\.hfocus',/boolean,/fold_case),cnt))[0]
    if cnt eq 1 then hfocus = strtrim(strmid(header[ind],strpos(header[ind],'ocus')+4),2)
    ind = (where(stregex(header,'# *a1\.vfocus',/boolean,/fold_case),cnt))[0]
    if cnt eq 1 then vfocus = strtrim(strmid(header[ind],strpos(header[ind],'ocus')+4),2)
    if arg_present(maspacing) then begin
       ind = (where(stregex(header,'# *ei\.dspacing',/boolean,/fold_case),cnt))[0]
       if cnt eq 1 then maspacing[0] = dm_to_number(strmid(header[ind],strpos(header[ind],'spacing')+7))
       ind = (where(stregex(header,'# *ef\.dspacing',/boolean,/fold_case),cnt))[0]
       if cnt eq 1 then maspacing[1] = dm_to_number(strmid(header[ind],strpos(header[ind],'spacing')+7))
    endif
    if arg_present(deteff) then begin
       ind = (where(stregex(header,'# *SPEC\.detectorEfficiency',/boolean,/fold_case),cnt))[0]
       if cnt gt 0 then deteff[0,*] = dm_to_number(strsplit(strmid(header[ind],strpos(header[ind],'ciency')+6),'[] ',/extract))
       ind = (where(stregex(header,'# *DIFF\.detectorEfficiency',/boolean,/fold_case),cnt))[0]
       if cnt gt 0 then deteff[1,*] = dm_to_number(strsplit(strmid(header[ind],strpos(header[ind],'ciency')+6),'[] ',/extract))
    endif
    if arg_present(latticeparm) then begin
       tmp = 'sampleState\.'+['A','B','C','Alpha','Beta','Gamma']
       for i=0,n_elements(tmp)-1 do begin
           ind = (where(stregex(header,'# *'+tmp[i]+' +',/boolean,/fold_case),cnt))[0]
           if cnt eq 1 then begin
              tmp1 = strsplit(header[ind],'# ',/extract)
              latticeparm[i] = dm_to_number(tmp1[1])
           endif
       endfor
       if total(finite(latticeparm,/nan)) ne 0 then begin
          for i=0,n_elements(tmp)-1 do begin
              ind = (where(stregex(info,tmp[i],/boolean,/fold_case),cnt))[0]
              if cnt ne 0 then latticeparm[i] = dm_to_number(data[0,ind])
          endfor
       endif
    endif
    if arg_present(latticeori) then begin
       if total(finite(latticeori,/nan)) ne 0 then begin  ;refplane data not in the column, try header
          ind = (where(stregex(header,'# *sampleState\.refPlane1',/boolean,/fold_case),cnt))[0]
          if cnt eq 1 then latticeori[0:2] = dm_to_number(strsplit(strmid(header[ind],strpos(header[ind],'lane')+5),', ',/extract))
          ind = (where(stregex(header,'# *sampleState\.refPlane2',/boolean,/fold_case),cnt))[0]
          if cnt eq 1 then latticeori[3:5] = dm_to_number(strsplit(strmid(header[ind],strpos(header[ind],'lane')+5),', ',/extract))
       endif
    endif

    ;rename column names in info
    niceinfo = info
    tmp = ['^sample(x|l)[a-z]*trans','pres[a-z]*SlitWidth','posts[a-z]*SlitWidth']
    for i=0,n_elements(tmp)-1 do begin
        ind = where(stregex(info,tmp[i],/boolean,/fold_case),cnt)
        if cnt gt 0 then begin
           tmp1 = strsplit(info[ind[0]],' ()',/extract,count=cnt)
           if cnt ge 2 then units[i] = tmp1[cnt-1]
        endif
    endfor
    ind = where(stregex(info,'.*\({1}.*\){1}$',/boolean),cnt)  ;remove parenthesis and units
    for i=0,cnt-1 do info[ind[i]] = strmid(info[ind[i]],0,strpos(info[ind[i]],'(',/reverse_search))
    ind = where(strmatch(info,'samplethetamotor'),cnt)
    if (cnt gt 0) and ((instrname eq 'macs') or (instrname eq 'ng0')) then info[ind] = 'basesampletheta'
    ind = where(strmatch(info,'sampletheta'),cnt)
    if cnt gt 0 then info[ind] = 'a3'
    ind = where(stregex(info,'.*(motor){1}$',/boolean),cnt)  ;remove trailing motor
    for i=0,cnt-1 do info[ind[i]] = strmid(info[ind[i]],0,strpos(info[ind[i]],'motor',/reverse_search))
    ind = where(stregex(info,'^(monoblade){1}',/boolean),cnt)
    for i=0,cnt-1 do info[ind[i]] = 'monblade'+strmid(info[ind[i]],9)
    tmp = where(stregex(info,'^diff00',/boolean),diff00)
    if diff00 then begin
       ind = where(stregex(info,'^diff[0-9]{2}$',/boolean),cnt)
       for i=0,cnt-1 do info[ind[i]] = 'diff'+string(dm_to_number(strmid(info[ind[i]],4),/int)+1,format='(i02)')
    endif
    tmp = where(stregex(info,'^spec00',/boolean),spec00)
    if spec00 then begin
       ind = where(stregex(info,'^spec[0-9]{2}$',/boolean),cnt)
       for i=0,cnt-1 do info[ind[i]] = 'spec'+string(dm_to_number(strmid(info[ind[i]],4),/int)+1,format='(i02)')
    endif
    ind = where(stregex(info,'^anatwotheta[0-9]{2}$',/boolean),cnt)
    for i=0,cnt-1 do begin
        info[ind[i]] = 'anatheta'+string(dm_to_number(strmid(info[ind[i]],11),/int),format='(i02)')
    endfor
    if cnt gt 0 then data[*,ind] = data[*,ind]/2.0
    ind = where(strmatch(info,'mcfxelevator'),cnt)
    if cnt gt 0 then begin
       info[ind] = 'mcfx'
       tmp = min(abs([-128,0,128]-data[0,ind[0]]),id)
       if n_elements(mcfx) ne 0 then id = id>(where(['beo','be','hopg'] eq strlowcase(mcfx)))
       data[*,ind] = id
    endif
    ind = where(stregex(info,'^monovba',/boolean),cnt)
    for i=0,cnt-1 do info[ind[i]] = strmid(info[ind[i]],4)
    ind = where(stregex(info,'^sample[xy]{1}trans',/boolean),cnt)
    for i=0,cnt-1 do info[ind[i]] = 'smpl'+strmid(info[ind[i]],6,1)
    ind = where(stregex(info,'^sample[ul]{1}[owerp]*trans',/boolean),cnt)
    for i=0,cnt-1 do info[ind[i]] = 'smpl'+strmid(info[ind[i]],6,1)+'trn'
    ind = where(stregex(info,'^sampleelev',/boolean),cnt)
    if cnt gt 0 then info[ind] = 'smplz'
    ind = where(stregex(info,'^sample[lu]{1}[owerp]*tilt',/boolean),cnt)
    for i=0,cnt-1 do info[ind[i]] = 'smpl'+strmid(info[ind[i]],6,1)+'tilt'
    ind = where(stregex(info,'^et(\.energytransfer)*$',/boolean),cnt)
    if cnt gt 0 then info[ind] = 'e'
    ind = where(strmatch(info,'initialenergy'),cnt)
    if cnt gt 0 then info[ind] = 'ei'
    ind = where(strmatch(info,'finalenergy'),cnt)
    if cnt gt 0 then info[ind] = 'ef'
    ind = where(strmatch(info,'aperthori'),cnt)
    if cnt gt 0 then info[ind] = 'vbah'
    ind = where(strmatch(info,'premonoslitwidth'),cnt)
    if cnt gt 0 then info[ind] = 'vbah'
    ind = where(strmatch(info,'apertvert'),cnt)
    if cnt gt 0 then info[ind] = 'vbav'
    ind = where(strmatch(info,'premonoslitheight'),cnt)
    if cnt gt 0 then info[ind] = 'vbav'
    ind = where(strmatch(info,'mag'),cnt)
    if cnt gt 0 then info[ind] = 'magfield'
    ind = where(strmatch(info,'monodts'),cnt)
    if cnt gt 0 then info[ind] = 'dfmdts'
    ind = where(strmatch(info,'monorotation'),cnt)
    if cnt gt 0 then info[ind] = 'monrot'
    ind = where(strmatch(info,'monotrans'),cnt)
    if cnt gt 0 then info[ind] = 'montrans'
    ind = where(strmatch(info,'timestamp'),cnt)
    if (cnt gt 0) and (data[0,ind[0]] gt 1e12) then data[*,ind] = data[*,ind]/1000.  ;ms->sec

    ;figure out scanning device
    ind = (where(stregex(header,'# *trajectoryData\.xAxis ',/boolean,/fold_case),cnt))[0]
    if cnt gt 0 then begin
       tmp  = strmid(header[ind],strpos(header[ind],'xis')+3)
       tmp1 = strsplit(tmp,' ',/extract)
       ind  = (where(info eq strlowcase(tmp1[0]),cnt))[0]
       header = [header,'#Scan        '+((cnt ge 1)?dm_to_string(ind+1)+' ':'')+tmp1[0]]
    endif
    ind = where(stregex(header,'countAgainst',/boolean,/fold_case),cnt)
    for i=0,cnt-1 do begin
        tmp = strsplit(header[ind[i]],' .,:"'+"'",/extract)
        ind1 = (where(strmatch(tmp,'countAgainst',/fold_case),cnt1))[0] ;avoid countAgainstDetector
        if cnt1 gt 0 then begin
           tmp1 = tmp[(ind1+1)<(n_elements(tmp)-1)]
           if stregex(tmp1,'monitor',/fold_case,/boolean) then tmp1 = 'monitor'
           ind1  = where(info eq strlowcase(tmp1),cnt1)
           header = [header,'#Reference   '+((cnt1 gt 0)?dm_to_string(ind1+1)+' ':'')+tmp1]
           break
        endif
    endfor
    ind = (where(stregex(header,'# *primaryDetector',/boolean,/fold_case),cnt))[0]
    if cnt gt 0 then begin
       tmp  = strmid(header[ind],strpos(header[ind],'tector')+6)
       tmp1 = (strsplit(tmp,' ',/extract))[0]
       if stregex(tmp1,'diff',/boolean,/fold_case)    then tmp1 = 'DIFF'
       if stregex(tmp1,'spec',/boolean,/fold_case)    then tmp1 = 'SPEC'
       if stregex(tmp1,'monitor',/boolean,/fold_case) then tmp1 = 'MONITOR'
       ind1 = where(info eq strlowcase(tmp1),cnt1)
       header = [header,'#Signal   '+((cnt1 gt 0)?dm_to_string(ind1+1)+' ':'')+tmp1]
    endif
    
    ;remove time=-1 data
    ind = where(strmatch(info,'time'),cnt)
    if cnt eq 1 then begin
       ind = where(data[*,ind[0]] lt 0,cnt,complement=ind1,ncomplement=cnt1)
       if cnt1 eq 0 then begin
          tmp = temporary(data)
          all_neg = 1b
       endif else if cnt ne 0 then data = data[ind1,*]
    endif
end

;read macs data file
;input:
;  file:            full file name string(s), or a string array of a full file content
;  group_leader:    parent ID
;  answeryes/no:    suppress the prompt question with default answer yes/no
;  checkonly:       if set, will not promp any question about inconsistent number of columns, and will not check for negative detector counts
;  firstlineonly:   if set, will read only the first line of data
;  histodata:       if set, will look for time channel string, otherwise time channel string will be discarded
;  rawdata:         if set, event mode data will not be normalized
;  string:          if set, output data as string array, exactly the same as the content in the data file, not for nexus file
;  n_chan:          channel number for histogram data, if absent, will be figured out from data file, histodata keyword needs to be set
;  cut_tail:        if set, will throw away the last 24 channels for n_chan=1024, and the last channel for other n_chan
;output:
;  niceinfo:        save the original NICE ascii file column names if the file is NICE format
;  data:            data array [nrow,ncol], if string keyword is set, a string array, otherwise a double array
;  data_histo:      data array [nchan,40,nrow], when histodata keyword is set, and data file has histo information
;  dqty_histo:      error bar  [nchan,40,nrow], only present if weight_histo is read from rg0 files
;  weight_histo:    weight info [nchan,nrow]
;  str_histo:       histogram string array, string keyword needs to be set
;  info:            description of columns, lower case output, if string is set, keep the original letter case
;  is_histogram:    1 if the data has histogram data
;  all_neg:         flag to indicate that all detectors contain -1, no valid data whatsoever
;  comment:         comment
;  deteff:          detector efficiency [2,ndet], [0,*] for spec detector, [1,*] for diff detector
;  error:           error flag
;  filename:        filename
;  flipstring:      flip field string, a three-element string array
;  header:          file header string array  
;  instrname:       instrument name, lower case output
;  latticeori:      [ux,uy,uz,vx,vy,vz]
;  latticeparm:     [a,b,c,alpha,beta,gamma]
;  maspacing:       [monochromator,analyser] spacing
;  npoints:         #npoints value in the header of the first file
;  pulse_histo:     [nrow], save the pulse number for histogram data
;  temperatureunit: the #TemperatureUnits field value
;  t_mcfx:          mcfx temperature, only available in nexus file
;  trailer:         file trailer string array, may be empty
;  hfocus,vfocus:   NICE ascii file only, monochromator focusing string
;  nomessage:       if set, no warning messages
;  noadjust:        keyword for dm_load_macsnice, if set, the column info and data will not be adjusted
;  units:           unit of [sample table translation, front slit, back slit] 'cm' or 'mm'
pro dm_load_macs,file,data,info,niceinfo=niceinfo,data_histo=data_histo,str_histo=str_histo,all_neg=all_neg,answerno=answerno,answeryes=answeryes,comment=comment,checkonly=checkonly,deteff=deteff,$
    filename=filename,firstlineonly=firstlineonly,flipstring=flipstring,group_leader=group_leader,header=header,histodata=histodata,rawdata=rawdata,t_start=t_start,t_bin=t_bin,n_chan=n_chan,n_tbin=n_tbin,$
    cut_tail=cut_tail,is_histogram=is_histogram,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,maspacing=maspacing,npoints=npoints,pulse_histo=pulse_histo,string=string,$
    temperatureunit=temperatureunit,t_mcfx=t_mcfx,trailer=trailer,nan=nan,weight_histo=weight_histo,dqty_histo=dqty_histo,zero_error=zero_error,nomessage=nomessage,noadjust=noadjust,$
    hfocus=hfocus,vfocus=vfocus,units=units,error=error
    if n_elements(file) eq 0            then return 
  	if n_elements(comment) ne 0         then tmp = temporary(comment) 
  	if n_elements(data) ne 0            then tmp = temporary(data)
  	if n_elements(data_histo) ne 0      then tmp = temporary(data_histo)
  	if n_elements(dqty_histo) ne 0      then tmp = temporary(dqty_histo)
  	if n_elements(str_histo) ne 0       then tmp = temporary(str_histo)
  	if n_elements(pulse_histo) ne 0     then tmp = temporary(pulse_histo)
  	if n_elements(info) ne 0            then tmp = temporary(info)   
  	if n_elements(header) ne 0          then tmp = temporary(header)
  	if n_elements(trailer) ne 0         then tmp = temporary(trailer)  
  	if n_elements(instrname) ne 0       then tmp = temporary(instrname)
  	if n_elements(all_neg) ne 0         then tmp = temporary(all_neg)
  	if n_elements(is_histogram) ne 0    then tmp = temporary(is_histogram)
  	if n_elements(temperatureunit) ne 0 then tmp = temporary(temperatureunit)
  	if n_elements(t_mcfx) ne 0          then tmp = temporary(t_mcfx)
  	if n_elements(filename) ne 0        then tmp = temporary(filename)
  	if n_elements(weight_histo) ne 0    then tmp = temporary(weight_histo)
  	if n_elements(zero_error) ne 0      then tmp = temporary(zero_error)
  	if n_elements(niceinfo) ne 0        then tmp = temporary(niceinfo)
  	if n_elements(hfocus) ne 0          then tmp = temporary(hfocus)
  	if n_elements(vfocus) ne 0          then tmp = temporary(vfocus)        
  	if arg_present(npoints)             then npoints     = 0
  	if arg_present(maspacing)           then maspacing   = fltarr(2)
  	if arg_present(latticeparm)         then latticeparm = replicate(!values.f_nan,6)
  	if arg_present(latticeori)          then latticeori  = replicate(!values.f_nan,6)
  	if arg_present(deteff)              then deteff      = fltarr(2,20)
  	if arg_present(flipstring)          then flipstring  = ['','','']
  	zerostring = ['out','beo','a','nsf']      ;strings for 0
  	onestring  = ['in','be','b','sf']         ;strings for 1
  	twostring  = ['hopg']                     ;strings for 2
  	tmp = ' ' & header = ' '
  	error = 0b
  	if total(stregex(file,'#ICERepositoryInfo',/fold_case,/boolean)) gt 0 then begin ;file is actually a file content string array
  	   ind_column = where(stregex(file,'#columns',/boolean,/fold_case),count_col)
  	   if count_col eq 0 then ind_header = n_elements(file)-1 else ind_header = ind_column[count_col-1]-1
  	   for i=0L,ind_header do begin
  	       tmp1 = strlowcase(strsplit(file[i],' '+string(9b),/extract,count=count))
           if arg_present(deteff) then begin
              if tmp1[0] eq '#detectorefficiencies' then begin
                 for j=1,count-1 do begin
                     ind = stregex(tmp1[j],'^spec([0-9]+)=',length=length,/subexpr)
                     if ind[0] ne -1 then deteff[0,fix(strmid(tmp1[j],ind[1],length[1]))-1] = float(strmid(tmp1[j],ind[0]+length[0]))
                     ind = stregex(tmp1[j],'^diff([0-9]+)=',length=length,/subexpr)
                     if ind[0] ne -1 then deteff[1,fix(strmid(tmp1[j],ind[1],length[1]))-1] = float(strmid(tmp1[j],ind[0]+length[0]))
                 endfor
              endif
           endif
           if arg_present(maspacing) and (count gt 1) then begin
              if tmp1[0] eq '#monospacing' then maspacing[0] = float(tmp1[1]) 
              if tmp1[0] eq '#anaspacing'  then maspacing[1] = float(tmp1[1]) 
           endif
           if arg_present(npoints) and (count gt 1) then begin
              if tmp1[0] eq '#npoints' then npoints = long(tmp1[1]) 
           endif 
           if arg_present(latticeparm) and (count gt 6) then begin
              if tmp1[0] eq '#lattice' then latticeparm[*] = float(tmp1[1:6]) 
           endif
           if arg_present(latticeori) and (count gt 6) then begin
              if tmp1[0] eq '#orient' then latticeori[*] = float(tmp1[1:6]) 
           endif
           if arg_present(instrname) and (count gt 1) then begin
              if tmp1[0] eq '#instrname' then instrname = tmp1[1]
           endif
           if arg_present(comment) and (count gt 1) then begin
              marker = '#comment'
              if tmp1[0] eq marker then comment = strtrim(strmid(file[i],strpos(strlowcase(file[i]),marker)+strlen(marker)),2)
           endif
           if arg_present(temperatureunit) and (count gt 1) then begin
              if tmp1[0] eq '#temperatureunits' then temperatureunit = tmp1[1]
           endif
           if (tmp1[0] eq '#filename') and (count gt 1) then filename = tmp1[1]
  	   endfor
  	   header = file[0:ind_header]
  	   if count_col gt 0 then begin
  	      tmp_info = strsplit(file[ind_column[count_col-1]],' '+string(9b),/extract,count=n_tmp1)
  	      tmp1 = strlowcase(tmp_info)
          n_info = n_elements(info)
          if n_info eq 0 then begin
             info   = tmp1[1:(n_tmp1-1)]
             n_info = n_tmp1-1
          endif else if n_info ne (n_tmp1-1) then begin
             ok = dialog_message('The files are of different format',/center,/error,dialog_parent=group_leader)
             error = 1b & return
          endif
          ind_spec = where(stregex(info,'spec[0-2][0-9]',/boolean))
          ind_diff = where(stregex(info,'diff[0-2][0-9]',/boolean))
          if arg_present(flipstring) then ind_flip = (where(info eq 'flip',check_flip))[0]
          for i=ind_column[0]+1,n_elements(file)-1 do begin
              if stregex(file[i],'^[ '+string(9b)+']*#',/boolean) then begin
                 if stregex(file[i],'#ng0:end',/boolean,/fold_case) and n_elements(data) ne 0 then npoints = n_elements(data[0,*]) ;force header npoints to be actual data number since the file is complete 
                 if n_elements(trailer) eq 0 then trailer = file[i] else trailer = [temporary(trailer),file[i]]
                 continue
              endif
              if stregex(file[i],'^[ '+string(9b)+']*$',/boolean) then continue
              tmp2 = strsplit(file[i],' '+string(9b),/extract,count=n_tmp1)
              if keyword_set(string) then begin
                 if n_info ne n_tmp1 then begin
                    if n_tmp1 gt n_info then begin
                       if dm_to_number(tmp2[n_tmp1-1],/double) gt 1243526755.85 then tmp2 = [tmp2[0:(n_info-2)],tmp2[n_tmp1-1]] $  ;last column is epoch, save it
                       else tmp2 = tmp2[0:(n_info-1)]
                    endif else begin
                       if dm_to_number(tmp2[n_tmp1-1],/double) gt 1243526755.85 then tmp2 = [tmp2[0:(n_tmp1-2)],replicate('',n_info-n_tmp1),tmp2[n_tmp1-1]] $  ;last column is epoch, save it
                       else tmp2 = [tmp2,replicate('',n_info-n_tmp1)]
                    endelse
                 endif
                 if n_elements(data) eq 0 then data = tmp2 else data = [[temporary(data)],[tmp2]]
              endif else begin             
                 tmp1 = dm_to_number(tmp2,zerostring=zerostring,onestring=onestring,twostring=twostring,/double)
                 if (~keyword_set(checkonly)) and (total(tmp1[[ind_spec,ind_diff]] lt 0) eq 40) then begin ;-1 in all diff and spec counts, ignore them
                    all_neg = 1b
                    continue  
                 endif
                 if n_info ne n_tmp1 then begin
                    if keyword_set(checkonly) then continue
                    if n_elements(inc_ans) eq 0 then begin
                       if keyword_set(answeryes) or keyword_set(answerno) then inc_ans = (['yes','no'])[keyword_set(answerno)] $
                       else inc_ans = dialog_message([filename+' has inconsistent number of columns.','Do you want to continue?','','If yes, all missing data will be set to 0.'],/default_no,/question,/center,dialog_parent=group_leader,title='Please confirm:')
                       if strmatch(inc_ans,'no',/fold_case) then begin
                          error = 1b & return
                       endif
                    endif
                    if n_tmp1 gt n_info then begin
                       if tmp1[n_tmp1-1] gt 1243526755.85 then tmp1 = [tmp1[0:(n_info-2)],tmp1[n_tmp1-1]] $  ;last column is epoch, save it
                       else tmp1 = tmp1[0:(n_info-1)]
                    endif else begin
                       if tmp1[n_tmp1-1] gt 1243526755.85 then tmp1 = [tmp1[0:(n_tmp1-2)],replicate(0,n_info-n_tmp1),tmp1[n_tmp1-1]] $  ;last column is epoch, save it
                       else tmp1 = [tmp1,replicate(0,n_info-n_tmp1)]
                    endelse
                 endif  
                 if n_elements(data) eq 0 then data = tmp1 else data = [[temporary(data)],[tmp1]]
                 if keyword_set(check_flip) then begin
                    if n_elements(tmp2) gt ind_flip then begin
                       tmp3 = round(tmp1[ind_flip])
                       if (tmp3 ne 0) and (tmp3 ne 1) then tmp3 = 2
                       flipstring[tmp3] = tmp2[ind_flip]
                    endif
                 endif
              endelse
              if keyword_set(firstlineonly) then break
          endfor
  	   endif
  	   if n_elements(comment) ne 0 then begin
          if (strmid(comment,0,1) eq '"') and (strmid(comment,0,1,/reverse_offset) eq '"') then comment = strmid(comment,1,strlen(comment)-2)
       endif
  	   if n_elements(data) ne 0 then data = transpose(data)
       all_neg = keyword_set(all_neg) and (n_elements(data) eq 0)
       if keyword_set(string) then info = tmp_info[1:n_info]
  	   return
  	endif
  	file = file[0]  ;only read one file at a time
    if (~file_test(file))then message,"Can't open "+file+"."
    filename = file_basename(file)
    if stregex(filename,'^ftptmp',/boolean,/fold_case) then filename = strmid(filename,6)
    if H5F_IS_HDF5(file) then begin  ;NICE nexus file
       dm_load_macsnexus,file,data,info,data_histo=data_histo,str_histo=str_histo,all_neg=all_neg,answerno=answerno,answeryes=answeryes,comment=comment,checkonly=checkonly,deteff=deteff,$
          firstlineonly=firstlineonly,group_leader=group_leader,header=header,histodata=histodata,rawdata=rawdata,t_start=t_start,t_bin=t_bin,n_chan=n_chan,n_tbin=n_tbin,cut_tail=cut_tail,$
          is_histogram=is_histogram,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,maspacing=maspacing,npoints=npoints,pulse_histo=pulse_histo,temperatureunit=temperatureunit,$
          t_mcfx=t_mcfx,nan=nan,weight_histo=weight_histo,dqty_histo=dqty_histo,zero_error=zero_error,nomessage=nomessage,units=units,hfocus=hfocus,vfocus=vfocus,error=error
       return
    endif
  	;ICE format ascii data file
  	i_row = -1L
    openr,unit,file,/get_lun
    if eof(unit) then begin
       free_lun,unit & return    ;empty file
    endif
    while(~stregex(tmp,'#columns',/boolean,/fold_case)) do begin
       readf,unit,tmp
       if stregex(tmp,'^ *# *NICE',/boolean,/fold_case) then begin  ;a NICE ascii file
          free_lun,unit
          dm_load_macsnice,file,data,info,data_histo=data_histo,str_histo=str_histo,all_neg=all_neg,answerno=answerno,answeryes=answeryes,comment=comment,checkonly=checkonly,deteff=deteff,$
             firstlineonly=firstlineonly,group_leader=group_leader,header=header,histodata=histodata,rawdata=rawdata,t_start=t_start,t_bin=t_bin,n_chan=n_chan,n_tbin=n_tbin,cut_tail=cut_tail,$
             is_histogram=is_histogram,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,maspacing=maspacing,npoints=npoints,pulse_histo=pulse_histo,temperatureunit=temperatureunit,$
             t_mcfx=t_mcfx,nan=nan,weight_histo=weight_histo,dqty_histo=dqty_histo,zero_error=zero_error,nomessage=nomessage,niceinfo=niceinfo,hfocus=hfocus,vfocus=vfocus,noadjust=noadjust,units=units,error=error
          return
       endif
       if arg_present(maspacing) or arg_present(latticeparm) or arg_present(latticeori) or arg_present(deteff) or arg_present(instrname) or arg_present(comment) or $
          arg_present(npoints) or arg_present(temperatureunit) then begin
          tmp1 = strlowcase(strsplit(tmp,' '+string(9b),/extract,count=count))
          if arg_present(deteff) then begin
             if tmp1[0] eq '#detectorefficiencies' then begin
                for j=1,count-1 do begin
                    ind = stregex(tmp1[j],'^spec([0-9]+)=',length=length,/subexpr)
                    if ind[0] ne -1 then deteff[0,fix(strmid(tmp1[j],ind[1],length[1]))-1] = float(strmid(tmp1[j],ind[0]+length[0]))
                    ind = stregex(tmp1[j],'^diff([0-9]+)=',length=length,/subexpr)
                    if ind[0] ne -1 then deteff[1,fix(strmid(tmp1[j],ind[1],length[1]))-1] = float(strmid(tmp1[j],ind[0]+length[0]))
                endfor
             endif
          endif
          if arg_present(maspacing) and (count gt 1) then begin
             if tmp1[0] eq '#monospacing' then maspacing[0] = float(tmp1[1]) 
             if tmp1[0] eq '#anaspacing'  then maspacing[1] = float(tmp1[1]) 
          endif
          if arg_present(npoints) and (count gt 1) then begin
             if tmp1[0] eq '#npoints' then npoints = long(tmp1[1])
          endif
          if arg_present(latticeparm) and (count gt 6) then begin
             if tmp1[0] eq '#lattice' then latticeparm[*] = float(tmp1[1:6]) 
          endif
          if arg_present(latticeori) and (count gt 6) then begin
             if tmp1[0] eq '#orient' then latticeori[*] = float(tmp1[1:6]) 
          endif
          if arg_present(instrname) and (count gt 1) then begin
             if tmp1[0] eq '#instrname' then instrname = tmp1[1]
          endif
          if arg_present(comment) and (count gt 1) then begin
             marker = '#comment'
             if tmp1[0] eq marker then comment = strtrim(strmid(tmp,strpos(strlowcase(tmp),marker)+strlen(marker)),2)
          endif
          if arg_present(temperatureunit) and (count gt 1) then begin
             if tmp1[0] eq '#temperatureunits' then temperatureunit = tmp1[1]
          endif
          if arg_present(hfocus) and (count gt 1) then begin
             if tmp1[0] eq '#monohorizfocus' then hfocus = tmp1[1]
          endif
          if arg_present(vfocus) and (count gt 1) then begin
             if tmp1[0] eq '#monovertifocus' then vfocus = tmp1[1]
          endif
       endif
       header = [temporary(header),strtrim(tmp,2)]
    endwhile
    header = header[1:n_elements(header)-2]
    ;check next line to see if it starts with #columns again
    if ~eof(unit) then begin
       point_lun,-unit,pos
       tmp1 = ' '
       readf,unit,tmp1
       if ~stregex(tmp1,'#columns',/boolean,/fold_case) then point_lun,unit,pos else tmp = tmp1
    endif
    tmp_info = strsplit(tmp,' '+string(9b),/extract,count=n_tmp1)
    tmp1   = strlowcase(tmp_info)
    info   = tmp1[1:(n_tmp1-1)]
    n_info = n_tmp1-1
    ind_spec = where(stregex(info,'spec[0-2][0-9]',/boolean))
    ind_diff = where(stregex(info,'diff[0-2][0-9]',/boolean))
    ind_time = (where(stregex(info,'time[()sec]*$',/boolean)))[0]
    if arg_present(flipstring) then ind_flip = (where(info eq 'flip',check_flip))[0]
    while (~eof(unit)) do begin
          readf,unit,tmp
          if stregex(tmp,'^[ '+string(9b)+']*#',/boolean) then begin
             if stregex(tmp,'#ng0:end',/boolean,/fold_case) and n_elements(data) ne 0 then npoints = n_elements(data[0,*]) ;force header npoints to be actual data number since the file is complete
             if n_elements(trailer) eq 0 then trailer = tmp else trailer = [temporary(trailer),tmp]
             continue
          endif
          if stregex(tmp,'^[ '+string(9b)+']*$',/boolean) then continue
          tmp2 = strsplit(tmp,' '+string(9b),/extract,count=n_tmp1)
          if keyword_set(string) then begin
             if (n_tmp1 eq 1) and stregex(tmp,'([0-9]+,){255}',/boolean) then begin   ;make sure it is histo string format
                is_histogram = 1b
                if n_elements(str_histo) eq 0 then str_histo = tmp else str_histo = [temporary(str_histo),tmp]
                continue
             endif
             if n_info ne n_tmp1 then begin
                if n_tmp1 gt n_info then begin
                   if dm_to_number(tmp2[n_tmp1-1],/double) gt 1243526755.85 then tmp2 = [tmp2[0:(n_info-2)],tmp2[n_tmp1-1]] $  ;last column is epoch, save it
                   else tmp2 = tmp2[0:(n_info-1)]
                endif else begin
                   if dm_to_number(tmp2[n_tmp1-1],/double) gt 1243526755.85 then tmp2 = [tmp2[0:(n_tmp1-2)],replicate('',n_info-n_tmp1),tmp2[n_tmp1-1]] $  ;last column is epoch, save it
                   else tmp2 = [tmp2,replicate('',n_info-n_tmp1)]
                endelse
             endif
             if n_elements(data) eq 0 then data = tmp2 else data = [[temporary(data)],[tmp2]]
          endif else begin
             if n_tmp1 eq 1 then begin
                is_histogram = 1b
                if keyword_set(histodata) then begin
                   if i_row eq 0 then begin  ;check n_chan
                      tmp3 = strsplit(tmp2,';',/extract)
                      tmp3 = strsplit(tmp3[0],',',/extract,count=n_chan1)
                      if n_elements(n_chan) eq 0 then n_chan=n_chan1 $
                      else if n_chan ne n_chan1 then begin
                         ok = dialog_message(filename+' has '+dm_to_string(n_chan1)+' time channels instead of '+dm_to_string(n_chan)+'. Please change the time channel number.',/error,/center,dialog_parent=group_leader)
                         error = 1b & break
                      endif
                   endif
                   tmp4 = float(strsplit(tmp2,',;',/extract))
                   if n_elements(tmp4) eq n_chan*(41l) then n_col = 41 $
                   else if n_elements(tmp4) eq n_chan*(40l) then n_col = 40 $
                   else continue
                   if n_col eq 40 then warn_missing_pulsenum = 1b
                   tmp5 = reform(tmp4,[n_chan,n_col])
                   if n_elements(data_histo) eq 0 then data_histo = tmp5[*,(1-(41-n_col)):(40-(41-n_col))] else data_histo = [[[temporary(data_histo)]],[[tmp5[*,(1-(41-n_col)):(40-(41-n_col))]]]]
                   if n_elements(pulse_histo) eq 0 then pulse_histo = (n_col eq 41)?(tmp5[0,0]):1 else pulse_histo = [temporary(pulse_histo),(n_col eq 41)?(tmp5[0,0]):1] 
                endif else begin
                   continue ;ignore the histo data
                endelse
             endif else begin
                i_row = i_row+1L
                if keyword_set(histodata) then begin
                   if histodata eq 2 then begin
                      tmp6 = strmid(file,0,strlen(file)-4)+'_'+dm_to_string(i_row)+'.rg0'
                      if ~file_test(tmp6,/read) then break ;file doesn't exit
                      dm_load_macshisto2d,tmp6,data_2d,dqty=dqty_2d,theta_start=theta_start,theta_end=theta_end,theta_bin=theta_bin,t_start=t_start,t_end=t_end,t_bin=t_bin,n_tbin=n_tbin,n_theta=n_theta,$
                         a3_offset=a3_offset,nan=nan,histohigh=histohigh,zero_error=zero_error1,rawdata=rawdata
                      if n_elements(zero_error1) ne 0 then begin
                         if n_elements(zero_error) eq 0 then zero_error = temporary(zero_error1) else zero_error = (zero_error)<(temporary(zero_error1))
                      endif
                   endif
                endif
                tmp1 = dm_to_number(tmp2,zerostring=zerostring,onestring=onestring,twostring=twostring,/double)
                if (~keyword_set(checkonly)) and (~keyword_set(histodata)) and ((total(tmp1[[ind_spec,ind_diff]] lt 0) eq 40) or (tmp1[ind_time] lt 0) ) then begin ;-1 in all diff and spec counts, ignore them
                   all_neg = 1b
                   continue 
                endif
                if n_info ne n_tmp1 then begin
                   if keyword_set(checkonly) then continue
                   if n_elements(inc_ans) eq 0 then begin
                      if keyword_set(answeryes) or keyword_set(answerno) then inc_ans = (['yes','no'])[keyword_set(answerno)] $
                      else inc_ans = dialog_message([filename+' has inconsistent number of columns.','Do you want to continue?','','If yes, all missing data will be set to 0.'],/default_no,/question,/center,dialog_parent=group_leader,title='Please confirm:')
                      if strmatch(inc_ans,'no',/fold_case) then begin
                         error = 1b & break
                      endif
                   endif
                   if n_tmp1 gt n_info then begin
                      if tmp1[n_tmp1-1] gt 1243526755.85 then tmp1 = [tmp1[0:(n_info-2)],tmp1[n_tmp1-1]] $  ;last column is epoch, save it
                      else tmp1 = tmp1[0:(n_info-1)]
                   endif else begin
                      if tmp1[n_tmp1-1] gt 1243526755.85 then tmp1 = [tmp1[0:(n_tmp1-2)],replicate(0,n_info-n_tmp1),tmp1[n_tmp1-1]] $  ;last column is epoch, save it
                      else tmp1 = [tmp1,replicate(0,n_info-n_tmp1)]
                   endelse
                endif
                if keyword_set(histodata) and n_elements(data_2d) ne 0 then begin
                   kidney = tmp1[where(info eq 'kidney')]
                   tmp1 = rebin(tmp1,n_elements(tmp1),n_theta)
                   ind_a3 = where(info eq 'a3')
                   for j=0,n_theta-1 do tmp1[ind_a3,j] = theta_start+theta_bin*(0.5+j)+kidney+a3_offset
                   if n_elements(data_histo) eq 0 then data_histo = temporary(data_2d) else data_histo = [[[temporary(data_histo)]],[[temporary(data_2d)]]]
                   if n_elements(dqty_2d) ne 0 then begin
                      if n_elements(dqty_histo) eq 0 then dqty_histo = temporary(dqty_2d) else dqty_histo = [[[temporary(dqty_histo)]],[[temporary(dqty_2d)]]]
                   endif
                   warn_missing_pulsenum = 1b
                   if n_elements(pulse_histo) eq 0 then pulse_histo = replicate(1.0,n_theta) else pulse_histo = [temporary(pulse_histo),replicate(1.0,n_theta)]
                   if n_elements(histohigh) eq 0 then histohigh = replicate(1./n_tbin/n_theta,n_tbin,n_theta)
                   if n_elements(weight_histo) eq 0 then weight_histo = temporary(histohigh) else weight_histo = [[temporary(weight_histo)],[temporary(histohigh)]]
                   is_histogram = 1b
                endif
                 
                if n_elements(data) eq 0 then data = tmp1 else data = [[temporary(data)],[tmp1]]
                if keyword_set(check_flip) then begin
                   if n_elements(tmp2) gt ind_flip then begin
                      tmp3 = round(tmp1[ind_flip])
                      if (tmp3 ne 0) and (tmp3 ne 1) then tmp3 = 2
                      flipstring[tmp3] = tmp2[ind_flip]
                   endif
                endif
             endelse
          endelse
          if keyword_set(firstlineonly) then begin
             if n_elements(is_histogram) ne 0 then break
             is_histogram = 0b
          endif
    endwhile
    free_lun,unit
    if error then return
	  if n_elements(comment) ne 0 then begin
	     if (strmid(comment,0,1) eq '"') and (strmid(comment,0,1,/reverse_offset) eq '"') then comment = strmid(comment,1,strlen(comment)-2)
	  endif
	  if n_elements(data) ne 0 then data = transpose(data)
	  all_neg = keyword_set(all_neg) and (n_elements(data) eq 0)
	  if n_elements(data_histo) ne 0 then begin
	     if histodata ne 2 then begin
	        if keyword_set(cut_tail) then data_histo = data_histo[0:(n_chan-dm_macs_histotail(n_chan)-1),*,*]	  
	        if ~keyword_set(nomessage) and keyword_set(warn_missing_pulsenum) then $
	          ok = dialog_message('The pulse number is missing in '+filename+' . 1 is used for the pulse number.',/center,dialog_parent=group_leader)
	     endif
	  endif
	  if keyword_set(string) and (n_elements(tmp_info) gt 1) then info = tmp_info[1:n_info]
end

;this procedure reads macs file for dcs_mslice program
;Parameters:
;   open_path:    file path, string
;   open_file:    file name(s), string or string array
;Keywords:
; input:
;   ftpobj:       ftp object for ftp files
;   parent:       message widget parent,  default is self.tlb
;   title:        message title
; output:
;   error:        error flag 1 or 0
;   qty:          [2,ndet,ndat] (0,*,*)-spec (1,*,*) diff  or [40,ntchan,ndat] if histodata keyword is set
;   dqty:         [2,ndet,ndat] or [40,ntchan,ndat] if histodata keyword is set, if histodata=1 histogram mode, histodata=2 event mode
;   a3:           [ndat] 
;   baddet_id:    an array of the bad detector id. for load vanadium file, no warning given here
;   cfx:          [ndat] flag for cfx monitors 0-none 1-Be 2-PG 3-MgF2 
;   comment:      comment string
;   deteff:       [2,ndet] detector efficiency [0,*] for spec detector, [1,*] for diff detector, note: the efficiency is more like normalizaiton factor Int*deteff, instead of Int/deteff
;   ei:           [ndat]
;   ef:           [ndat]
;   hfield:       [ndat], available when self.extravaxis_yn[2]=1
;   info:         {latticeparm:latticeparm,latticeori:latticeori,temperature:setpoint,magfield:magfield}  used for checking file compatibility
;   kidney:       [ndat] kidney angle
;   montype:      0-time 1-monitor, check only the first file
;   ptai:         [ndat], integer array
;   specify:      [ndat], available when self.extravaxis_yn[0]=1
;   temperature:  [ndat,3]  sample, control, and setpoint temperature, in Kelvins
;   weight:       [ndat,3]  [ndat,0]-time [ndat,1]-monitor [ndat,2]-pulse number only available in histogram mode
;   histo_weight: weight info [ntchan,ndata]
;   histo_nchan:  number of histo time channel
;   histo_width:  histo time channel width
;   histo_t0:     starting time for the histogram time channel 
;   zero_error:   return error bar for sqrt(intensity=1)
;   skipextra:    if set, will not ask for temperature or magfield
pro dcs_mslice::dm_load_macs,open_path,open_file,a3=a3,baddet_id=baddet_id,cfx=cfx,comment=comment,deteff=deteff,ei=ei,ef=ef,error=error,ftpobj=ftpobj,hfield=hfield,$
    info=info,kidney=kidney,montype=montype,parent=parent,ptai=ptai,qty=qty,dqty=dqty,specify=specify,temperature=temperature,title=title,weight=weight,histo_nchan=histo_nchan,$
    histo_width=histo_width,histo_t0=histo_t0,histo_weight=histo_weight,zero_error=zero_error,skipextra=skipextra
    cd,current=current
    if n_elements(parent) eq 0 then parent = self.tlb
    if n_elements(title)  eq 0 then title  = 'Loading data ...'
    if n_elements(qty)    ne 0 then tmp    = temporary(qty)               ;destroy qty
    if n_elements(dqty)   ne 0 then tmp    = temporary(dqty)              ;destroy dqty
    histodata   = self.macshistomode
    rawdata     = ((histodata eq 2) and self.macs_rawintn)                ;event mode raw data, no normalization
    checklattuv = self.checklattuv
    n_files = n_elements(open_file)
    cnt_cfx0 = 0L ;number of files with cfx=0
    
    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       if obj_valid(mesg) then obj_destroy,mesg
       ok = dialog_message(!error_state.msg,dialog_parent=self.tlb,/error,/center)
       cd,current
       error = 1b
       return
    end
    
    if obj_valid(ftpobj) then pathsep = '/' else pathsep = self.pathsep
    while(strmid(open_path,0,1,/reverse_offset) eq  pathsep) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    if self.extravaxis_yn[0] then begin
       tmp_spiden = strlowcase(strtrim(self.extravaxis_spec[2],2))
       tmp_spname = self.extravaxis_spec[0]+((strlen(self.extravaxis_spec[1]) eq 0)?'':(' ('+self.extravaxis_spec[1]+')'))
    endif
    for i=0L,n_files-1L do begin
        file = open_path+pathsep+open_file[i]
        if obj_valid(ftpobj) then begin
           ok = ftpobj->GetFileContent(file,localfilename=self.dirs[2]+self.pathsep+'ftptmp'+open_file[i])
           file = self.dirs[2]+self.pathsep+'ftptmp'+open_file[i]
        endif
        if i eq 0 then begin
           mesg =  obj_new('dm_progress',title=title,message='Loading '+file,group_leader=parent) 
           dm_load_macs,file,data,colinfo,data_histo=data_histo,pulse_histo=pulse_histo,header=header,deteff=deteff,maspacing=maspacing,latticeparm=latticeparm,latticeori=latticeori,$
              group_leader=parent,comment=comment,instrname=instrname,temperatureunit=temperatureunit,histodata=histodata,rawdata=rawdata,is_histogram=is_histogram,n_chan=self.macshistonchan,$
              cut_tail=self.macshistocut,t_start=t_start,t_bin=t_bin,n_tbin=n_tbin,nan=nan,dqty_histo=dqty_histo,weight_histo=weight_histo,zero_error=zero_error1,error=error
        endif else begin
           mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)
           dm_load_macs,file,data,colinfo,data_histo=data_histo,pulse_histo=pulse_histo,header=header,deteff=deteff,maspacing=maspacing,latticeparm=latticeparm,latticeori=latticeori,$
              group_leader=parent,instrname=instrname,temperatureunit=temperatureunit,histodata=histodata,rawdata=rawdata,is_histogram=is_histogram,n_chan=self.macshistonchan,$
              cut_tail=self.macshistocut,t_start=t_start,t_bin=t_bin,n_tbin=n_tbin,nan=nan,weight_histo=weight_histo,dqty_histo=dqty_histo,zero_error=zero_error1,error=error
        endelse 
        if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET

        if error ne 0 then begin
           if obj_valid(mesg) then obj_destroy,mesg
           cd,current
           return
        endif

        if (keyword_set(histodata) ne keyword_set(is_histogram)) and (n_elements(histo_ans) eq 0) then begin
           tmp = [open_file[i]+([' does not  contain',' contains'])[keyword_set(is_histogram)]+' histogram data. Do you want to continue?','',$
                  '(Go to Option->View Intensity As->MACS Data Mode to '+(['disable','enable'])[keyword_set(is_histogram)]+' the histogram mode.)']
           histo_ans = dialog_message(tmp,title=(['No ',''])[keyword_set(is_histogram)]+'Histogram Data',/question,/center,dialog_parent=parent)
           if histo_ans eq 'No' then break
        endif
        
        if n_elements(data) eq 0 then continue else data = float(temporary(data))
        if keyword_set(histodata) and (n_elements(data_histo) eq 0) then continue

        if (instrname ne 'ng0') and (instrname ne 'bt9') and (instrname ne 'macs') then begin
           exitinfo = open_file[i]+' is not a MACS data file.'
        endif else if n_elements(info) eq 0 then begin
           info  = {latticeparm:latticeparm,latticeori:latticeori}
           file0 = open_file[i]
        endif else if checklattuv then begin
           if (total(abs(info.latticeparm-latticeparm)) ne 0) or (total(abs(info.latticeori-latticeori)) ne 0) then begin
              exitinfo = 'Selected data files are of different lattice parameters or scattering plane.'
              if n_elements(file0) eq 0 then file0 = 'previous'
              tmp = [file0+':  '+dm_to_string(info.latticeparm,sep=', ')+'  '+dm_to_string(info.latticeori,sep=', '),$
                     open_file[i]+':  '+dm_to_string(latticeparm,sep=', ')+'  '+dm_to_string(latticeori,sep=', '),' ']
              ans = dialog_message([exitinfo,tmp,'Do you want to temporarily ignore the differences?'],/default_no,/question,/center,dialog_parent=parent,title='Please confirm:')
              if strmatch(ans,'Yes',/fold_case) then begin
                 checklattuv = 0b
                 tmp = temporary(exitinfo)
              endif
           endif   
        endif
        if n_elements(exitinfo) ne 0 then begin
           if n_elements(ans) eq 0 then ok = dialog_message(exitinfo,/error,/center,dialog_parent=parent)
           cd,current
           if obj_valid(mesg) then obj_destroy,mesg
           error = 1b
           return
        endif
        ;check index every time, since the file format could be different
        ind_qty = lonarr(2,20)
        ind_err = lonarr(20)  ;for polarization corrected file, only for spec detectors
        ind_ana = lonarr(20)
        for j=1,20 do begin
            tmp = string(j,format='(i02)')
            ind_qty[0,j-1] = (where(colinfo eq 'spec'+tmp))[0]
            ind_qty[1,j-1] = (where(colinfo eq 'diff'+tmp))[0]
            ind_err[j-1]   = (where(colinfo eq 'spec'+tmp+'err'))[0]
            ind_ana[j-1]   = (where(colinfo eq 'analyzertheta'+tmp))[0]
        endfor
        readerr = (total(ind_err ge 0) eq 20)   ;flag for reading intensity error bars
        ind_a3  = (where(colinfo eq 'a3'))[0]
        ind_a2  = (where(colinfo eq 'a2'))[0]
        ind_a5  = (where(colinfo eq 'a5'))[0]
        if finite(data[0,ind_a5],/nan) then begin
           tmp_x = reform(data[0,ind_ana]) & tmp_dy = fltarr(20)+1
           dm_step_bin,0.5,tmp_x,yerr=tmp_dy
           tmp = min(tmp_dy,ind_tmp)
           tmp = min(abs(data[0,ind_ana]-tmp_x[ind_tmp]),ind_tmp)
           ind_a5 = ind_ana[ind_tmp]
        endif
        ind_kn  = (where(colinfo eq 'kidney'))[0]
        ind_wt0 = (where(colinfo eq 'time'))[0]
        ind_wt1 = (where(colinfo eq 'monitor'))[0]
        ind_t   = (where(colinfo eq 'temp',cnt_t))[0]
        ind_tc  = (where(colinfo eq 'temperaturecontrolreading',cnt_tc))[0]
        ind_ts  = (where(colinfo eq 'temperaturesetpoint',cnt_ts))[0]
        ind_pt  = (where(colinfo eq 'ptai',cnt_pt))[0]
        nstr    = 'cfx'+['be','hopg','mgf']
        cnt_cfx = 0
        for j=0,n_elements(nstr)-1 do begin
            ind_cfx = (where(colinfo eq nstr[j],count))[0]
            if count ne 0 then begin
               if finite(data[0,ind_cfx]) then begin
                  if round(data[0,ind_cfx]) then begin
                     cnt_cfx = cnt_cfx+2^j
                  endif
               endif
            endif
        endfor
        ;check if the date is after CFXdate
        tmp = stregex(header,'#Date(.*)',/fold_case,/subexpr,length=len)
        ind = where(len[1,*] gt 0,count)
        if count gt 0 then begin
           date = dm_to_number(strmid(header[ind[0]],tmp[1,ind[0]],len[1,ind[0]]),/date)
           if (date lt self.macs_CFXdate[1]) and (cnt_cfx eq 4) then cnt_cfx = 0
           if (cnt_cfx eq 0) and (date ge self.macs_CFXdate[0]) then cnt_cfx0 = cnt_cfx0+1
        endif
        ;temperature
        if cnt_t ne 0 then begin
           thistemperature = data[*,[ind_t,ind_tc,ind_ts]]
           if self.tempavg_yn then begin
              thistemperature = (fltarr(n_elements(data[*,0]))+1.)#total(thistemperature,1)/n_elements(thistemperature[*,0])  ;average temperature of the file
           endif
           ;convert all temperature units to Kelvin
           if n_elements(temperatureunit) ne 0 then begin
              case temperatureunit of 
                   'celsius' :   thistemperature = thistemperature+273.15
                   'fahrenheit': thistemperature = (thistemperature+459.67)*5./9.
                   else:
              endcase
           endif
        endif else begin
           if self.extravaxis_yn[1] and (~keyword_set(skipextra)) then begin
              repeat begin
                  ans0 = dm_dialog_input('T (K)',/float,default=ans0,dialog_parent=self.tlb,/align_center,info=open_file[i]+" doesn't contain temperature. Please enter it.",title='Missing temperature',cancel=cancel)
                  if keyword_set(cancel) then message,'Missing temperature value. Go to Option->Viewing Axis->Allow Extra Viewing Axis to remove the temperature axis.',level=0,/noprint,/noname
              endrep until finite(ans0)
              thistemperature = ans0+fltarr(n_elements(data[*,0]),3)
           endif else $
              thistemperature = -1.0+fltarr(n_elements(data[*,0]),3)
        endelse      
        
        ;hfield
        ind_hf = (where(colinfo eq 'magfield',cnt_hf))[0]
        if cnt_hf eq 0 then ind_hf = (where(colinfo eq 'hfield',cnt_hf))[0]
        if self.extravaxis_yn[2] then begin
           if (cnt_hf eq 0) and (~keyword_set(skipextra)) then begin
              repeat begin
                  ans1 = dm_dialog_input('H (T)',/float,default=ans1,dialog_parent=self.tlb,/align_center,info=open_file[i]+" doesn't contain magnetic field. Please enter it.",title='Missing magnetic field',cancel=cancel)
                  if keyword_set(cancel) then message,'Missing magnetic field value. Go to Option->Viewing Axis->Allow Extra Viewing Axis to remove the magnetic field axis.',level=0,/noprint,/noname
              endrep until finite(ans1)
              this_hfield = ans1+fltarr(n_elements(data[*,0]))
           endif else this_hfield = data[*,ind_hf]
        endif
        
        ;specified quantity
        if self.extravaxis_yn[0] then begin
           if strlen(tmp_spiden) ne 0 then ind_sp = (where(colinfo eq tmp_spiden,cnt_sp))[0] $
           else cnt_sp = 0
           if (cnt_sp eq 0) and (~keyword_set(skipextra)) then begin
              repeat begin
                  ans2 = dm_dialog_input(tmp_spname,/float,default=ans2,dialog_parent=self.tlb,/align_center,info=open_file[i]+" doesn't contain "+self.extravaxis_spec[0]+'. Please enter it.',title='Missing '+self.extravaxis_spec[0],cancel=cancel)
                  if keyword_set(cancel) then message,'Missing '+self.extravaxis_spec[0]+' value. Go to Option->Viewing Axis->Allow Extra Viewing Axis to remove the extra axis.',level=0,/noprint,/noname
              endrep until finite(ans2)
              this_spec = ans2+fltarr(n_elements(data[*,0]))
           endif else this_spec = data[*,ind_sp]
        endif 
        
        if cnt_ts ne 0 then begin
           tnames = tag_names(info)
           index  = where(tnames eq 'TEMPERATURE',count)
           if count eq 0 then info = create_struct('TEMPERATURE',data[n_elements(data[*,0])-1,ind_ts],info)   ;use the last setpoint temperature   
        endif
        
        if cnt_hf ne 0 then begin
           tnames = tag_names(info)
           index  = where(tnames eq 'MAGFIELD',count)
           if count eq 0 then info = create_struct('MAGFIELD',data[0,ind_hf],info)
        endif
        
        ;some analyzers may be off, needs to check
        for j=0L,n_elements(data[*,0])-1 do begin
            min_da5 = min(abs(data[j,ind_a5]-data[j,ind_ana]))
            bad_id  = where(abs(data[j,ind_a5]-data[j,ind_ana]) gt min_da5+0.5,badcount,complement=good_id)   ;more than 0.5 degree off
            if badcount ne 0 then begin
               data[j,ind_a5] = mean(data[j,ind_ana[good_id]])
               if ~self.ftoolflag[3] then begin
                  if n_elements(baddetid) eq 0 then baddetid = bad_id else begin
                     baddetid = [baddetid,bad_id]
                     baddetid = baddetid[uniq(baddetid,sort(baddetid))] 
                  endelse 
               endif
            endif
        endfor

        if ~keyword_set(histodata) then begin ;check if data masking is used for polarized data (readerr=1)
           tmp_qty = transpose([[[data[*,ind_qty[0,*]]]],[[data[*,ind_qty[1,*]]]]])
           if readerr then begin
              tmp_dqty = transpose([[[data[*,ind_err]]],[[sqrt(data[*,ind_qty[1,*]])]]])
              index = where(tmp_dqty lt 0,count)
              if count gt 0 then begin
                 tmp_qty[index] = !values.f_nan
                 nan = 1b
              endif
           endif
        endif

        if n_elements(qty) eq 0 then begin
           if keyword_set(histodata) then begin
              qty = temporary(data_histo)
              if histodata eq 2 then begin
                 histo_nchan = n_tbin
                 histo_width = t_bin
                 histo_t0    = t_start
                 if n_elements(dqty_histo) ne 0 then dqty = temporary(dqty_histo)
              endif
           endif else begin
              qty = temporary(tmp_qty)
              if readerr then dqty = temporary(tmp_dqty)
           endelse
           a3  = data[*,ind_a3]
           ei  = 81.8042/((2*maspacing[0]*sin(!dtor*data[*,ind_a2]/2))^2)   ;A2->Ei
           ef  = 81.8042/((2*maspacing[1]*sin(!dtor*data[*,ind_a5]))^2)     ;A5->Ef
           ptai = round(data[*,ind_pt])
           kidney = data[*,ind_kn]
           if keyword_set(histodata) then begin
              weight = transpose([[data[*,[ind_wt0,ind_wt1]]],[pulse_histo]])
              histo_weight = weight_histo
           endif else weight = transpose(data[*,[ind_wt0,ind_wt1]])
           temperature = thistemperature         
           cfx = (intarr(n_elements(data[*,0]))+1)*cnt_cfx
           if self.extravaxis_yn[2] then hfield = this_hfield
           if self.extravaxis_yn[0] then specify = this_spec
        endif else begin
           if keyword_set(histodata) then begin
              if histodata eq 2 then begin
                 if ((histo_nchan) ne n_tbin) or (abs(histo_width-t_bin)/histo_width gt 1e-5) then begin
                    ok = dialog_message(['Selected histogram files have different number of time channel or time channel width.',$
                         file0+': n_chan = '+dm_to_string(histo_nchan)+', width = '+dm_to_string(histo_width),$
                         open_file[i]+': n_chan = '+dm_to_string(n_tbin)+', width = '+dm_to_string(t_bin)],/error,/center,dialog_parent=parent)
                    cd,current
                    if obj_valid(mesg) then obj_destroy,mesg
                    error = 1b
                    return
                 endif
                 if histo_t0 ne t_start then dm_shift_histochan,t0_0=histo_t0,t0_1=t_start,t_wid=t_bin,qty=data_histo,dqty=dqty_histo,histo_weight=weight_histo
              endif
              qty = [[[temporary(qty)]],[[temporary(data_histo)]]]
              if n_elements(dqty_histo) ne 0 then dqty = [[[temporary(dqty)]],[[temporary(dqty_histo)]]]
           endif else begin
              if readerr then begin
                 if n_elements(dqty) eq 0 then dqty = sqrt(qty)
                 dqty = [[[temporary(dqty)]],[[temporary(tmp_dqty)]]]
              endif else if n_elements(dqty) ne 0 then begin
                 dqty = [[[temporary(dqty)]],[[sqrt(tmp_qty)]]]
              endif
              qty = [[[temporary(qty)]],[[temporary(tmp_qty)]]]
           endelse
           a3  = [temporary(a3),data[*,ind_a3]]
           ei  = [temporary(ei),81.8042/((2*maspacing[0]*sin(!dtor*data[*,ind_a2]/2))^2)]
           ef  = [temporary(ef),81.8042/((2*maspacing[1]*sin(!dtor*data[*,ind_a5]))^2)]
           ptai = [temporary(ptai),round(data[*,ind_pt])]
           kidney = [temporary(kidney),data[*,ind_kn]]
           if keyword_set(histodata) then begin
              weight = [[temporary(weight)],[transpose([[data[*,[ind_wt0,ind_wt1]]],[pulse_histo]])]]
              histo_weight = [[histo_weight],[temporary(weight_histo)]]
           endif else weight = [[temporary(weight)],[transpose(data[*,[ind_wt0,ind_wt1]])]]    
           temperature = transpose([[transpose(temperature)],[transpose(thistemperature)]])         
           cfx = [cfx,(intarr(n_elements(data[*,0]))+1)*cnt_cfx]
           if self.extravaxis_yn[2] then hfield = [hfield,this_hfield]
           if self.extravaxis_yn[0] then specify = [specify,this_spec]
        endelse
     
        ;check monitor type
        if n_elements(montype) eq 0 then begin
           yn = stregex(header, '#reference',/boolean,/fold_case)
           index = where(yn,count)
           if count ne 0 then begin
              tmp = strlowcase(strsplit(header[index[0]],' '+string(9b),/extract))
              if tmp[n_elements(tmp)-1] eq 'time'    then montype = 0
              if tmp[n_elements(tmp)-1] eq 'monitor' then montype = 1
           endif
           if n_elements(montype) eq 0 then begin ;in case '#reference' info is absent
              tmp = [total(abs(weight[0,*]-weight[0,0])),total(abs(weight[1,*]-weight[1,0]))]
              tmp = min(tmp,montype)
           endif
        endif
        
        if n_elements(zero_error1) ne 0 then begin
           if n_elements(zero_error) eq 0 then zero_error = temporary(zero_error1) else zero_error = (zero_error)<(temporary(zero_error1))
        endif

        if (~ obj_valid(mesg)) then break
    endfor

    if keyword_set(nan) then self.ftypecheckmask[*] = 1b  ;data contain NAN

    ;check for NAN's in Ei and Ef, if found, replace them with the corresponding values in Ef and Ei
    if n_elements(qty) gt 0 then begin
       ind = where(finite(ei,/nan),count)
       if count gt 0 then ei[ind] = ef[ind]
       ind = where(finite(ef,/nan),count)
       if count gt 0 then ef[ind] = ei[ind]
       if keyword_set(histodata) then begin
          if size(qty,/n_dim) eq 2 then begin
             qty = reform(qty,[size(qty,/dim),1])
             if n_elements(dqty) ne 0 then dqty = reform(dqty,[size(dqty,/dim),1])
          endif
          qty = transpose(qty,[1,0,2])  ;[40,nchan,ndat]
          if n_elements(dqty) ne 0 then dqty = transpose(dqty,[1,0,2])
       endif
       if n_elements(dqty) eq 0 then dqty = sqrt(qty)
       weight = transpose(weight) ;[2,ndat]->[ndat,2]
    endif
    
    if n_elements(zero_error) eq 0 then zero_error = 1.0

    cd,current
    if obj_valid(mesg) then obj_destroy,mesg

    ;monitor lambda/2 correction warning
    if cnt_cfx0 ne 0 and ~self.macs_lamb2 then begin
       if n_files eq 1 then mesg = 'No CFX filter is used in the file.' $
       else begin
          if cnt_cfx0 eq n_files then mesg = 'No CFX filter is used in all the files.' $
          else mesg = 'No CFX filter is used in some of the files.'
       endelse
       ok = dialog_message(dialog_parent=self.tlb,mesg+' Do you want to enable the monitor lambda/2 correction in the Option menu?',/question,/center,title='Monitor lambda/2 correction')   
       if ok eq 'Yes' then begin
          self.macs_lamb2 = 1b
          dm_set_button,widget_info(self.tlb,find_by_uname='monlamb2'),self.macs_lamb2,onstring='Apply Monitor Lambda/2 Correction'
       endif
    endif
    
    ;bad detector warning
    nbd = n_elements(baddetid)
    if nbd ne 0 then begin
       if arg_present(baddet_id) then begin ;for load vanadium file, warning will be given there
          baddet_id = baddetid
          return 
       endif
       if ptr_valid(self.mask) then begin
          if self.masktype eq 0 then begin
             if n_elements(*(self.mask)) ge nbd then begin
                tmp = dm_common(baddetid,*(self.mask),count)
                if count eq nbd then nodetwarning = 1b          ;mask already exists, no need to warn
             endif
          endif
       endif
       if ~keyword_set(nodetwarning) then begin
          ok = dialog_message(dialog_parent=self.tlb,['The anaylzer angle A5 in detector '+dm_to_string(baddetid,sep=', ')+([' are',' is'])[nbd eq 1]+' more than 0.5 degree off.',$
               'Do you want to mask '+(['them','it'])[nbd eq 1]+'?','','(Detector number starts from 0.)'],/question,/center,title='mask detector')
          if ok eq 'Yes' then begin
             if ptr_valid(self.mask) and self.masktype eq 0 then begin
                baddetid = [baddetid,*(self.mask)]
                baddetid = baddetid[uniq(baddetid,sort(baddetid))] 
             endif else self.masktype = 0
             self->clear_mask,/keep_ar
             self.mask = ptr_new(baddetid)
             self->my_widget_control,['mksvBut','mkclBut'],/sensitive
             cid = widget_info(self.tlb,find_by_uname='mkstdnum') 
             if cid ne 0 then self->event,{WIDGET_BUTTON,ID:cid,TOP:self.tlb,HANDLER:self.tlb,SELECT:1}
          endif else self.ftoolflag[3] = 1b   ;no warning from here on, clear data will clear this flag
       endif
    endif else self.ftoolflag[3] = 0b         ;reset the warning flag
end