;#######################################################################
;
; NAME:
;  dm_ghoston
;
; PURPOSE:
;  this program search for ghoston for a triple axis instrument
;  reference: Physica B 350 (2004) 11-16
;  
; CATEGORY:
;  TAS tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  March, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

@dcs_cryst_align

pro dm_ghoston_Exit,tlb
    widget_control,tlb,get_uvalue=self
    if obj_valid(self) then obj_destroy,self
end

;widget event handler
pro dm_ghoston_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;form a label string for a [3,n_hkl] array of hkl values
function dm_hkl_label,hkl,lattparm=lattparm
    if (n_elements(hkl) eq 0) or ((n_elements(hkl) mod 3) ne 0) then return,''
    if n_elements(lattparm) ne 6 then lattparm = [1.0,1.0,1.0,90.0,90.0,90.0] ;default cubic lattice
    hkl1  = hkl  ;so that hkl will not be modified
    n_hkl = n_elements(hkl[0,*])
    equ   = intarr(n_hkl)
    lab   = strarr(n_hkl)
    ind   = lindgen(n_hkl)
    while ~keyword_set(done) do begin
          done = 1b
          if n_hkl gt 4 then begin  ;group hkl values together based on most occurence and move them to top
             nexp = intarr(3)
             fact = intarr(3)
             for i=0,2 do begin
                 tmp = hkl1[i,*]
                 ind1 = where((tmp ne 0) and (tmp lt 100),cnt1)
                 if cnt1 gt 0 then begin
                    nexp[i] = max(histogram(tmp[ind1],location=location),max_ind)
                    fact[i] = 1-2*(location[max_ind] gt 0)
                 endif
             endfor
             max_ind = where(nexp eq max(nexp))
             nexp[*] = 0 & nexp[max_ind] = 4
             ind = sort(dcs_cryst_align_qlength1(hkl,lattparm)*1d12+fact[0]*hkl1[0,*]*(10d)^(nexp[0]+4)+fact[1]*hkl1[1,*]*(10d)^(nexp[1]+2)+fact[2]*hkl1[2,*]*(10d)^nexp[2])
          endif
          for i=0,n_hkl-2 do begin ;check for single plus/minus match
              for j=i+1,n_hkl-1 do begin
                  ind1 = where(((hkl1[*,ind[i]]+hkl1[*,ind[j]]) eq 0) and (hkl1[*,ind[i]] ne 0) and (total(abs(abs(hkl1[*,ind[i]])-abs(hkl1[*,ind[j]]))) eq 0),cnt1)
                  if cnt1 eq 1 then begin
                     equ[[ind[i],ind[j]]] = equ[[ind[i],ind[j]]]+2^(ind1[0])
                     hkl1[ind1[0],[ind[i],ind[j]]] = 100*abs(hkl1[ind1[0],ind[i]])
                     if n_hkl gt 3 then done = 0b
                  endif
                  if ~done then break
              endfor
              if ~done then break
          endfor
    endwhile
    equ = (equ<7)
    for i=0,n_hkl-1 do begin
        tmp = [['','',''],['\pm','\pm','\pm']]+dm_to_string([[hkl[*,i]],[abs(hkl[*,i])]])
        lab[i] = '('+(tmp[0,*])[(equ[i] mod 2) eq 1]+','+(tmp[1,*])[(equ[i] mod 4) ge 2]+','+(tmp[2,*])[equ[i] ge 4]+')'
    endfor
    lab = lab[uniq(lab,sort(lab))]
    tmp = fltarr(n_elements(lab))
    for i=0,n_elements(lab)-1 do begin ;sorted output
        tmp0 = dm_to_number(strsplit(lab[i],'\pm,)(',/extract))
        while n_elements(tmp0) lt 3 do tmp0 = [tmp0,0]
        tmp[i] = dcs_cryst_align_qlength1(tmp0[0:2],lattparm)*1e6+tmp0[0]*1e4+tmp0[1]*1e2+tmp0[2]
    endfor
    return,strjoin(lab[sort(tmp)])
end

pro dm_ghoston::adjust_E,eiefchanged=eiefchanged,efeichanged=efeichanged,enchanged=enchanged
    if keyword_set(eiefchanged) then begin
       if finite(self.E[0,0]) then begin
          if self.fixeief[0] then self.Ei[*,0] = !values.f_nan else self.Ef[*,0] = !values.f_nan
       endif
    endif
    if keyword_set(efeichanged) then begin
       if (self.fixeief[0] and finite(self.Ef[0,0])) or ((self.fixeief[0] eq 0) and finite(self.Ei[0,0])) then self.E[*,0] = !values.f_nan
    endif
    if keyword_set(enchanged) then begin
       if self.fixeief[0] and finite(self.Ef[0,0]) then self.Ei[*,0] = !values.f_nan $
       else if (self.fixeief[0] eq 0) and finite(self.Ei[0,0]) then self.Ef[*,0] = !values.f_nan
    endif
    if self.fixeief[0] then self.Ef[*,0] = self.Ef[0,0] else self.Ei[*,0] = self.Ei[0,0]
    for i=0,1 do begin
        ind = where(finite([self.Ei[i,0],self.Ef[i,0],self.E[i,0]],/nan),count)
        if count eq 0 then begin
           if self.E[i,0] ne self.Ei[i,0]-self.Ef[i,0] then begin
              if self.fixeief[0] then self.Ei[i,0] = self.Ef[i,0]+self.E[i,0] else self.Ef[i,0] = self.Ei[i,0]-self.E[i,0]
           endif
        endif else if count eq 1 then begin
           case ind[0] of 
                0:  if self.fixeief[0] then self.Ei[i,0] = self.Ef[i,0]+self.E[i,0]  else self.Ei[*,0] = self.Ef[0,0]+self.E[0,0]
                1:  if self.fixeief[0] then self.Ef[*,0] = self.Ei[0,0]-self.E[0,0]  else self.Ef[i,0] = self.Ei[i,0]-self.E[i,0]  
                2:  self.E[i,0] = self.Ei[i,0]-self.Ef[i,0]
           endcase
        endif
    endfor
    self->my_widget_control,'eief',set_value=dm_to_string(([self.Ei[0,0],self.Ef[0,0]])[self.fixeief[0]])
    if self.fixeief[0] then tmp = self.Ei[*,0] else tmp = self.Ef[*,0]
    self->my_widget_control,'efei',set_value=dm_to_string(tmp[0:finite(tmp[1])],sep=':')
    self->my_widget_control,'en',set_value=dm_to_string(self.E[0:finite(self.E[1,0]),0],sep=':')
end

pro dm_ghoston::calc_ghoston,lefthand=lefthand
    l_r = keyword_set(lefthand)
    self.U[*,1] = self.U[*,0] & self.V[*,1] = self.V[*,0] & self.Q[*,1] = self.Q[*,0] & self.E[*,1] = self.E[*,0] & self.Ei[*,1] = self.Ei[*,0] & self.Ef[*,1] = self.Ef[*,0]
    self.fixeief[1] = self.fixeief[0] & self.lattparm[*,1] = self.lattparm[*,0]  ;saving a copy of the parameters
    self.e_is_range = finite(self.E[1,1])
    max_E = max(self.E[*,1],/nan,min=min_E,subscript_min=indx)
    ptr_free,self.hkl[l_r],self.blast[l_r],self.listind[l_r],self.eran[l_r]
    if self.fixeief[1] then begin
       Ef = self.Ef[0,1]
       if finite(self.dEi) then dE = abs(self.dEi)
    endif else begin
       Ei = self.Ei[0,1]
       if finite(self.dEf) then dE = abs(self.dEf)
    endelse
    if n_elements(dE) ne 0 then begin
       if dE eq 0 then dE = self.minEstep
    endif
    calc_dE = (finite([self.dEf,self.dEi],/nan))[self.fixeief[1]]
    if self.e_is_range then begin   ;En is a range, show list only
       list_hkl = [0,0,0] & list_blt = 0b & list_ei = 0.0 & list_ef = 0.0 & n_old = 0
       E = min_E & Ef = self.Ef[indx,1] & Ei = self.Ei[indx,1]
       while (Ei gt 0) and (Ef gt 0) do begin
           if self.fixeief[1] then begin
              Ei = Ef+E
              if calc_dE then dE = abs(self.eresl*Ei)
           endif else begin
              Ef = Ei-E
              if calc_dE then dE = abs(self.eresl*Ef)
           endelse
           n_hkl = self->find_hkl(Ei,Ef,self.dEi,self.dEf,bragglast=bragglast,hkl=hkl,lefthand=lefthand)
           if n_hkl gt 0 then begin
              new_val = strjoin(transpose([[transpose(hkl)],[bragglast]]))
              n_new   = n_hkl
              if n_old ne 0 then begin
                 comb = [old_val,new_val]
                 vec  = [bytarr(n_old),replicate(1b,n_hkl) ]
                 sub  = sort(comb)
                 comb = comb[sub]
                 vec  = vec[sub]
                 dup  = where((comb eq shift(comb,-1)) and (vec ne shift(vec,-1)),count) ;find duplicates
                 if count gt 0 then begin
                    ind = [lindgen(n_old),lindgen(n_hkl)]
                    ind = ind[sub[[dup,dup+1]]]
                    bragglast[ind[where(vec[[dup,dup+1]])]] = 2b
                    ind = where(bragglast ne 2,n_new)
                    hkl = hkl[*,ind]
                    bragglast = bragglast[ind]
                 endif
              endif
              old_val = new_val
              n_old = n_hkl
              if n_new ne 0 then begin
                 list_hkl = [[temporary(list_hkl)],[temporary(hkl)]]
                 list_blt = [temporary(list_blt),temporary(bragglast)]
                 list_ei  = [temporary(list_ei),replicate(Ei,n_new)]
                 list_ef  = [temporary(list_ef),replicate(Ef,n_new)]
              endif
           endif else n_old = 0
           if E eq max_E then break
           E = (E+dE*2.0)<(max_E)
       endwhile
       if n_elements(list_blt) gt 1 then begin
          list_blt = list_blt[1:*]
          list_hkl = list_hkl[*,1:*]
          list_ei  = list_ei[1:*]
          list_ef  = list_ef[1:*]
       endif else tmp = temporary(list_blt)
    endif else begin
       n_hkl = self->find_hkl(self.Ei[0,1],self.Ef[0,1],self.dEi,self.dEf,bragglast=bragglast,hkl=hkl,a3a4=a3a4,kihkl=kihkl,lefthand=lefthand)
       if n_hkl gt 0 then begin
          list_blt = bragglast
          list_hkl = hkl
          self.a3a4[*,l_r]  = a3a4
          self.kihkl[*,l_r] = kihkl
       endif
    endelse
    n_hkl = n_elements(list_blt)
    list  = 'View all'
    if n_hkl ne 0 then begin
       eran = self->erange(list_hkl,list_blt,lefthand=lefthand,eihkl=list_ei,efhkl=list_ef)
       tmp = list_blt*1d20+eran[0,*]*1d10+dcs_cryst_align_qlength1(list_hkl,self.lattparm[*,1])*1d6+list_hkl[0,*]*1d4+list_hkl[1,*]*1d2+list_hkl[2,*]
       ind = sort(tmp)
       list_hkl = list_hkl[*,ind]
       list_blt = list_blt[ind]
       eran = eran[*,ind]
       self.hkl[l_r] = ptr_new(list_hkl)
       self.blast[l_r] = ptr_new(list_blt)
       self.eran[l_r] = ptr_new(eran)
       listind = -1l
       self.viewsel[l_r] = 0
       if ~self.e_is_range then begin
          tmp = where(list_blt eq 0,count1,ncomplement=count2)
          if ((count1 gt 1) or (count2 gt 1)) and (max([count1,count2]) ne n_elements(list_blt)) then begin
             if count1 ne 0 then begin
                list = [list,'View all Bragg first']
                listind = [listind,-2l]
             endif
             if count2 ne 0 then begin
                list = [list,'View all Bragg last']
                listind = [listind,-3l]
             endif
          endif
       endif
       list = [list,'Bragg '+([' first',' last'])[list_blt]+' :  ('+strjoin(strtrim(string(list_hkl),2),',')+')']
       listind = [listind,lindgen(n_elements(list_blt))]
       if self.e_is_range then begin
          tmp = where(list_blt eq 0,n_bfirst,ncomplement=n_blast)
          if n_bfirst gt 0 then begin
             list = [list,'Bragg first E plot']
             listind = [listind,-2]
          endif
          if n_blast gt 0 then begin
             list = [list,'Bragg last E plot']
             listind = [listind,-1]
          endif
       endif
       self.listind[l_r] = ptr_new(listind[self.e_is_range:*])
    endif else list = ['No ghoston','No ghoston']
    dm_set_droplist,widget_info(self.tlb,find_by_uname='viewsel_'+(['r','l'])[l_r]),select=0,set_value=list[self.e_is_range:*],sensitive=ptr_valid(self.blast[l_r])
    if l_r eq self->currenttabnum() then self->my_widget_control,['saveplot','printplot'],sensitive=ptr_valid(self.blast[l_r])
    self->list,lefthand=lefthand
end

function dm_ghoston::check_parm,nomesg=nomesg
    mesg = ''
    if total(finite(self.lattparm[*,0],/nan)) ne 0 then mesg = [mesg,'Missing lattice parameters.']
    if total([finite(self.U[*,0],/nan),finite(self.V[*,0],/nan)]) ne 0 then mesg = [mesg,'Missing scattering plane vectors.'] $
    else if norm(crossp(self.U[*,0],self.V[*,0])) eq 0 then mesg = [mesg,'Parallel scattering plane axes U and V.']
    if total(finite([self.Ei[0,0],self.Ef[0,0],self.E[0,0]],/nan)) ne 0 then mesg = [mesg,'Missing energy information.'] $
    else begin
       if min([self.Ei[*,0],1.0],/nan) le 0 then mesg = [mesg,'Invalid Ei.']
       if min([self.Ef[*,0],1.0],/nan) le 0 then mesg = [mesg,'Invalid Ef.']
    endelse
    if total(finite(self.Q[*,0],/nan)) ne 0 then mesg = [mesg,'Missing Q vector.']
    if n_elements(mesg) eq 1 then begin ;check if Q is on the scattering plane
       ;direct -> reciprocal basis vectors
       dm_u1u2u3,self.lattparm[*,0],self.U[*,0],self.V[*,0],self.U[*,0],self.V[*,0],U1=U1,U2=U2,U3=U3,abc_r=abc_r,error=error
       if keyword_set(error) then mesg = [mesg,'Error encountered in calculating the resiprocal lattice parameters.'] $
       else begin
          self.abc_r = abc_r
          self.U1 = U1
          self.U2 = U2
          self.U3 = crossp(U1,U2)
          tmp = crossp(self.U[*,0],self.V[*,0])
          if transpose(self.Q[*,0])#tmp ne 0 then mesg = [mesg,'Q is not in the scattering plane.'] $
          else if norm(self.abc_r#self.Q[*,0]) gt 2.*!pi/dm_e2lambda(min(self.Ei[*,0],/nan))+2.*!pi/dm_e2lambda(min(self.Ef[*,0],/nan)) then mesg=[mesg,'Q is too large for the give Ei and Ef.']
       endelse
    endif
    if (self.flags[2] ge 1) and (self.flags[2] le 3) and ((self.lattparm[0,0] ne self.lattparm[1,0]) or (self.lattparm[1,0] ne self.lattparm[2,0]) or (self.lattparm[3,0] ne 90) or $
       (self.lattparm[4,0] ne 90) or (self.lattparm[5,0] ne 90)) then warning = 'The lattice parameters are not for a cubic unit cell.'
    if (self.flags[2] eq 4) and ((self.lattparm[0,0] ne self.lattparm[1,0]) or (self.lattparm[3,0] ne 90) or (self.lattparm[4,0] ne 90) or (self.lattparm[5,0] ne 120)) then $
       warning = 'The lattice parameters are not for a hexagonal unit cell.'
    if ~keyword_set(nomest) and (n_elements(warning) ne 0) then begin
       ok = dialog_message([warning,'Do you want to continue?'],/question,dialog_parent=self.tlb,/center,title='Please confirm:')
       if ok eq 'No' then return,0b
    endif
    if n_elements(mesg) gt 1 then begin ;fatal errors
       if ~keyword_set(nomesg) then ok = dialog_message(mesg[1:*],/error,dialog_parent=self.tlb,/center)
       return,0b
    endif
    if ~self.flags[0] and (total(finite([self.dEi,self.dEf],/nan)) ne 0) and ~keyword_set(nomesg) then begin
       if finite(self.dEi,/nan) then mesg = [mesg,'dEi = '+dm_to_string(self.eresl)+' * Ei is used for the missing Ei resolution.']
       if finite(self.dEf,/nan) then mesg = [mesg,'dEf = '+dm_to_string(self.eresl)+' * Ef is used for the missing Ef resolution.']
       mesg = [mesg,'The factor can be changed in the Options menu.']
       self.flags[0] = 1b
    endif
    if ~keyword_set(nomesg) and (n_elements(mesg) gt 1) then ok = dialog_message(mesg[1:*],dialog_parent=self.tlb,/center)  ;non-fatal error
    return,1b
end

;returns l-1 or r-0
function dm_ghoston::currenttabnum,is_list=is_list
    wid = widget_info(self.tlb,find_by_uname='tabbase')
    r_l = widget_info(wid,/tab_current)
    if arg_present(is_list) then begin
       wid = widget_info(self.tlb,find_by_uname='tabbase_'+(['r','l'])[r_l])
       is_list = widget_info(wid,/tab_current)
    endif 
    return,r_l
end

pro dm_ghoston::draw,lefthand=lefthand
    l_r  = keyword_set(lefthand)
    pobj = self.plots[l_r]
    pobj->erase 
    if ~ptr_valid(self.blast[l_r]) then return
    abc_r = self.abc_r
    hkl   = *(self.hkl[l_r])
    blast = *(self.blast[l_r])
    eran  = *(self.eran[l_r])
    fsize = self.fsize
    self->my_widget_control,['isoscale','circle','showuv','showkikf'],sensitive=((~self.e_is_range) or ((*self.listind[l_r])[self.viewsel[l_r]] ge 0))
    if self.e_is_range then begin
       if (*self.listind[l_r])[self.viewsel[l_r]] lt 0 then begin ;E plot
          ind = where(blast eq (2+(*self.listind[l_r])[self.viewsel[l_r]]),cnt)
          if cnt eq 0 then return
          all_col = pobj->getcolor(/list)
          eplot_eran = dm_to_number(self.eplot_setting[0:1])
          if total(finite(eplot_eran,/nan)) gt 0 then eplot_eran = self.E[*,1]
          default = [dm_to_string(eplot_eran[0:finite(eplot_eran[1])],sep=' : '),self.eplot_setting[2],dm_to_string(0>where(all_col eq self.eplot_setting[3])),$
                     self.eplot_setting[4],dm_to_string(0>where(all_col eq self.eplot_setting[5])),dm_to_string(0>where(['left','center','right'] eq self.eplot_setting[6]))]
          coldefine = self.eplot_coldef
          ans = dm_dialog_input(['energy range:','energy precision:','line color:','show hkl label:','label color:','label position:'],xsize=100,default=default,dialog_parent=self.tlb,$
                is_droplist=[0,1,1,1,1,1],droplist_content=[ptr_new(['1','0.1','0.01','0.001','0.0001']),ptr_new(all_col),ptr_new(['no','yes']),ptr_new(all_col),ptr_new(['left','center','right'])],$
                is_define=[0,0,1,0,1,0],bgcolor=[255,255,255],coldefine=coldefine,info='Use colon, comma, or space to separate the energy range values.')
          tmp = dm_to_number(strsplit(ans[0],' ,:',/extract))
          tmp0 = where(finite(tmp),cnt0)
          if cnt0 ge 2 then eplot_eran = tmp[tmp0[0:1]] else if cnt0 eq 1 then eplot_eran[abs(tmp[tmp0[0]]-eplot_eran[0]) gt abs(tmp[tmp0[0]]-eplot_eran[1])] = tmp[tmp0]
          self.eplot_setting = [dm_to_string(eplot_eran),dm_to_string(0>where(['1','0.1','0.01','0.001','0.0001'] eq ans[1])),ans[2],dm_to_string(0>where(['no','yes'] eq ans[3])),ans[4],ans[5]]
          self.eplot_coldef  = coldefine[*,[2,4]]
          Es   = dm_to_string(eran[2,ind]-eran[3,ind],res=dm_to_number(self.eplot_setting[2])) ;to compare Energy up to specified precision
          xs   = dm_to_number(Es)
          x    = xs[uniq(xs,sort(xs))] ;energy value to plot
          hkl  = hkl[*,ind]
          yran = [0,100]
          pind = where(['left','center','right'] eq self.eplot_setting[6])
          for i=0,n_elements(x)-1 do begin
              pobj->add_plot,[x[i],x[i]],[-1e6,1e6],linestyle='solid',thick=self.thick,color=self.eplot_setting[3],coldefine=self.eplot_coldef[*,0]
              if self.eplot_setting[4] eq '1' then begin
                 pobj->add_text,(['!A',' ','!C!C'])[pind]+dm_hkl_label(hkl[*,where(xs eq x[i])],lattparm=self.lattparm[*,1])+' ',x[i],yran[0]+0.9*(yran[1]-yran[0]),color=self.eplot_setting[5],$
                     coldefine=self.eplot_coldef[*,1],normalize=0,fontsize=fsize,orientation=90,alignment=1,vertical_alignment=([0.3,0.4,0.9])[pind],fill_background=(pind eq 1),fill_color=[255,255,255]
              endif
          endfor
          pobj->setproperty,xtit='E (meV)',ytit='',title='Scattering to '+(['right','left'])[l_r]+', Bragg '+(['first','last'])[2+(*self.listind[l_r])[self.viewsel[l_r]]]+$
              ', '+(['Ei=','Ef='])[self.fixeief[1]]+dm_to_string(([self.Ei[0,1],self.Ef[0,1]])[self.fixeief[1]])+' meV, '+'Q=('+dm_to_string(self.Q[*,1],sep=', ',/nozeroend)+')',$
              axisthick=self.thick,xran=eplot_eran,yran=yran,isotropic=0,/noevent,showxgrid=0,showygrid=0,ytickmajor=0,ytickminor=0 ,xtickminor=-1
          return
       endif
       hkl   = hkl[*,self.viewsel[l_r]]
       blast = blast[self.viewsel[l_r]]
       Ei    = eran[2,self.viewsel[l_r]] 
       Ef    = eran[3,self.viewsel[l_r]]
       eran  = eran[0:1,self.viewsel[l_r]]
       E     = Ei-Ef
       a3a4  = dcs_cryst_align_a3a4(Ei,Ef,self.Q[*,1],U=self.U[*,1],V=self.V[*,1],projU1=self.U1,projU2=self.U2,projU3=self.U3,abc_r=self.abc_r,lefthand=lefthand,kihkl=kihkl,kfhkl=kfhkl,error=error)
       if keyword_set(error) then return
       ki = kihkl
       kf = kfhkl
    endif else begin
       ki = self.kihkl[*,l_r] & kf = ki-self.Q[*,1] & E = self.E[0,1] & a3a4 = self.a3a4[*,l_r]
       if self.viewsel[l_r] gt 0 then begin
          ind   = (*self.listind[l_r])[self.viewsel[l_r]]
          ind_f = where(blast eq 0,complement=ind_l)
          case ind of
            -2:   begin ;all bragg first
                  hkl = hkl[*,ind_f]
                  blast = blast[ind_f]
                  end
            -3:   begin ;all bragg last
                  hkl = hkl[*,ind_l]
                  blast = blast[ind_l]
                  end
            else: begin
                  hkl   = hkl[*,ind]
                  blast = blast[ind]
                  eran  = eran[*,ind]
                  end
          endcase
       endif
    endelse
    if self.flags[5] then cornertxt = 'ki=('+dm_to_string(float(ki),sep=', ',/nozeroend)+'), kf=('+dm_to_string(float(kf),sep=', ',/nozeroend)+')' else cornertxt = ''
    if (total(blast) eq 0) or (total(blast) eq n_elements(blast)) then begin
       p0 = round(([[ki],[kf]])[*,blast[0]])-([[ki],[kf]])[*,blast[0]]  ;move origin close to (0,0,0)
       circle = self.flags[3]
    endif else p0 = [0.,0.,0.]
    U = self.U[*,1]+p0 & V = self.V[*,1]+p0 & ki = ki+p0 & kf = kf+p0
    ps = float(hkl)
    for i=0,n_elements(blast)-1 do if blast[i] then ps[*,i] = kf+hkl[*,i] else ps[*,i] = ki-hkl[*,i]
    col_step = 30
    cols = ['dark green','red','dark red','dark cyan','blue']
    tmp = self.plots[0]->getpalette(33)    ;Blue-Red col table
    tmp->getproperty,blue=col_blue,green=col_green,red=col_red
    obj_destroy,tmp
    xran = [min([p0[0],self.flags[4]?[U[0],V[0]]:0.0,ki[0],kf[0],reform(ps[0,*])]),max([p0[0],self.flags[4]?[U[0],V[0]]:0.0,ki[0],kf[0],reform(ps[0,*])])]
    yran = [min([p0[1],self.flags[4]?[U[1],V[1]]:0.0,ki[1],kf[1],reform(ps[1,*])]),max([p0[1],self.flags[4]?[U[1],V[1]]:0.0,ki[1],kf[1],reform(ps[1,*])])]
    zran = [min([p0[2],self.flags[4]?[U[2],V[2]]:0.0,ki[2],kf[2],reform(ps[2,*])]),max([p0[2],self.flags[4]?[U[2],V[2]]:0.0,ki[2],kf[2],reform(ps[2,*])])]
    if self.flags[4] then cornertxt = 'a3='+dm_to_string(a3a4[0],res=3)+'\deg, a4='+dm_to_string(a3a4[1],res=3)+'\deg'+([', ',''])[strlen(cornertxt) eq 0]+cornertxt else cornertxt = ''+cornertxt
    if n_elements(eran) eq 2 then cornertxt = cornertxt+([', ',''])[strlen(cornertxt) eq 0]+'Eran=['+dm_to_string(eran[0:1],sep=', ',res=self.str_Eresl,/nozeroend)+'] meV'
    if n_elements(blast) eq 1 then title = 'ghoston: q=('+dm_to_string(self.Q[*,1]-hkl,sep=', ',/nozeroend)+'), E='+dm_to_string(E,res=self.str_Eresl)+' meV, \tau=('+dm_to_string(hkl,sep=', ')+')' else title=''
    for i=0,2 do if max(abs([U[i],V[i],reform(hkl[i,*])])) eq 0 then is2d = i+1
    if keyword_set(is2d) then begin
       xid = min([is2d,is2d+1] mod 3,max=yid)
       tmp = [[xran],[yran],[zran]] & xran = tmp[*,xid] & yran = tmp[*,yid]
       pobj->add_plot,p0[xid],p0[yid],psym='filled circle',color=cols[1+blast[0]]  ;origin
       if self.flags[4] then begin
          pobj->add_plot,[p0[xid],U[xid]],[p0[yid],U[yid]],color=cols[0],/addarrow,thick=self.thick
          pobj->add_text,([' !C!C',' !A'])[U[yid] ge p0[yid]]+'U'+([' ',''])[U[xid] ge p0[xid]],U[xid],U[yid],color=cols[0],normalize=0,fontsize=fsize,vertical_alignment=([0.5,0.4])[U[yid] ge p0[yid]],$
                alignment=([1,0.1])[U[xid] ge p0[xid]]
          pobj->add_plot,[p0[xid],V[xid]],[p0[yid],V[yid]],color=cols[0],/addarrow,thick=self.thick
          pobj->add_text,([' !C!C',' !A'])[V[yid] ge p0[yid]]+'V'+([' ',''])[V[xid] ge p0[xid]],V[xid],V[yid],color=cols[0],normalize=0,fontsize=fsize,vertical_alignment=([0.5,0.4])[V[yid] ge p0[yid]],$
                alignment=([1,0.1])[V[xid] ge p0[xid]]
       endif
       pobj->add_plot,[p0[xid],ki[xid]],[p0[yid],ki[yid]],color=cols[1],/addarrow,thick=self.thick
       pobj->add_text,([' !C!C',' !A'])[ki[yid] ge p0[yid]]+'ki'+([' ',''])[ki[xid] ge p0[xid]],ki[xid],ki[yid],color=cols[1],normalize=0,fontsize=fsize,vertical_alignment=([0.5,0.4])[ki[yid] ge p0[yid]],$
             alignment=([1,0.1])[ki[xid] ge p0[xid]]
       pobj->add_plot,[p0[xid],kf[xid]],[p0[yid],kf[yid]],color=cols[2],/addarrow,thick=self.thick
       pobj->add_text,([' !C!C',' !A'])[kf[yid] ge p0[yid]]+'kf'+([' ',''])[kf[xid] ge p0[xid]],kf[xid],kf[yid],color=cols[2],normalize=0,fontsize=fsize,vertical_alignment=([0.5,0.4])[kf[yid] ge p0[yid]],$
             alignment=([1,0.1])[kf[xid] ge p0[xid]]
       pobj->add_plot,[kf[xid],ki[xid]],[kf[yid],ki[yid]],color=cols[3],/addarrow,thick=self.thick
       ang = (atan(kf[yid]-ki[yid],kf[xid]-ki[xid])/!dtor+180) mod 180
       pobj->add_text,([' ',''])[ang le 45]+(['!A',''])[(ang gt 45) and (ang lt 90)]+'Q',mean([ki[xid],kf[xid]]),mean([ki[yid],kf[yid]]),color=cols[3],normalize=0,fontsize=fsize,$
             vertical_alignment=([0.2,0.5])[(ang gt 45) and (ang lt 135)],alignment=([0.5,0.1])[(ang gt 45) and (ang lt 135)]
       for i=0L,n_elements(blast)-1 do begin
           p = ps[*,i]
           ki_kf = ([[ki],[kf]])[*,blast[i]]
           p1 = (p+ki_kf)/2.0
           pobj->add_arc,p[[xid,yid]],ki_kf[[xid,yid]],conv_matrix=abc_r,p_origin=p0[[xid,yid]],circle=circle,arcs=arcs,linestyle='dashed',color=cols[1+blast[i]],thick=self.thick
           orie = atan(hkl[yid,i],hkl[xid,i])/!dtor
           if abs(orie) gt 90 then orie = orie+180*([-1,1])[orie lt 0]
           xran = [(xran[0])<(min(arcs[0,*])),(xran[1])>(max(arcs[0,*]))]
           yran = [(yran[0])<(min(arcs[1,*])),(yran[1])>(max(arcs[1,*]))]
           if n_elements(blast) eq 1 then color = cols[4] else color = [col_red[(1+i*col_step) mod 256],col_green[(1+i*col_step) mod 256],col_blue[(1+i*col_step) mod 256]]
           pobj->add_plot,([[kf],[p],[ki]])[xid,*],([[kf],[p],[ki]])[yid,*],color=color,/addarrow,thick=self.thick
           pobj->add_plot,[p0[xid],p[xid]],[p0[yid],p[yid]],color=color,/addarrow,thick=self.thick
           pobj->add_text,'!A('+dm_to_string(hkl[*,i],sep=',')+')',p1[xid],p1[yid],color=color,normalize=0,fontsize=fsize,orientation=orie,vertical_alignment=0.2
       endfor
       nx = floor(xran[1]+0.25)-ceil(xran[0]-0.25)+1
       ny = floor(yran[1]+0.25)-ceil(yran[0]-0.25)+1
       xs = rebin(ceil(xran[0]-0.25)+indgen(nx),nx,ny)
       ys = transpose(rebin(ceil(yran[0]-0.25)+indgen(ny),ny,nx))
    endif else begin
       pobj->add_plot,p0[0],p0[1],p0[2],psym='sphere',symsize=0.01,color=cols[1+blast[0]]  ;origin
       if self.flags[4] then begin
          pobj->add_plot,[p0[0],U[0]],[p0[1],U[1]],[p0[2],U[2]],color=cols[0],/addarrow,thick=self.thick
          pobj->add_text,'U!C',U[0],U[1],U[2],color=cols[0],normalize=0,fontsize=fsize,vertical_alignment=0.35
          pobj->add_plot,[p0[0],V[0]],[p0[1],V[1]],[p0[2],V[2]],color=cols[0],/addarrow,thick=self.thick
          pobj->add_text,'V!C',V[0],V[1],V[2],color=cols[0],normalize=0,fontsize=fsize,vertical_alignment=0.35
       endif
       pobj->add_plot,[p0[0],ki[0]],[p0[1],ki[1]],[p0[2],ki[2]],color=cols[1],/addarrow,thick=self.thick
       pobj->add_text,'ki!C',ki[0],ki[1],ki[2],color=cols[1],normalize=0,fontsize=fsize,vertical_alignment=0.35
       pobj->add_plot,[p0[0],kf[0]],[p0[1],kf[1]],[p0[2],kf[2]],color=cols[2],/addarrow,thick=self.thick
       pobj->add_text,'kf!C',kf[0],kf[1],kf[2],color=cols[2],normalize=0,fontsize=fsize,vertical_alignment=0.35
       pobj->add_plot,[kf[0],ki[0]],[kf[1],ki[1]],[kf[2],ki[2]],color=cols[3],/addarrow,thick=self.thick
       pobj->add_text,'Q!C',mean([ki[0],kf[0]]),mean([ki[1],kf[1]]),mean([ki[2],kf[2]]),color=cols[3],normalize=0,fontsize=fsize,vertical_alignment=0.3
       pobj->getproperty,ctm=ctm ;for checking text location
       for i=0L,n_elements(blast)-1 do begin
           p = ps[*,i]
           ki_kf = ([[ki],[kf]])[*,blast[i]]
           p1 = (p+ki_kf)/2.0
           pobj->add_arc,p,ki_kf,conv_matrix=abc_r,p_origin=p0,circle=circle,arcs=arcs,linestyle='dashed',color=cols[1+blast[i]],thick=self.thick
           baseline = hkl[*,i]/norm(hkl[*,i])
           if (baseline[0] eq 0) and (baseline[1] eq 0) then updir = [1,0,0] else updir = crossp([-baseline[1],baseline[0],0],baseline)
           if updir[2] lt 0 then updir = -updir
           if (baseline#ctm[0:2,0:2])[0] lt 0 then baseline = -baseline ;to face the viewer
           xran = [(xran[0])<(min(arcs[0,*])),(xran[1])>(max(arcs[0,*]))]
           yran = [(yran[0])<(min(arcs[1,*])),(yran[1])>(max(arcs[1,*]))]
           zran = [(zran[0])<(min(arcs[2,*])),(zran[1])>(max(arcs[2,*]))]
           if n_elements(blast) eq 1 then color = cols[4] else color = [col_red[(1+i*col_step) mod 256],col_green[(1+i*col_step) mod 256],col_blue[(1+i*col_step) mod 256]]
           pobj->add_plot,([[kf],[p],[ki]])[0,*],([[kf],[p],[ki]])[1,*],([[kf],[p],[ki]])[2,*],color=color,/addarrow,thick=self.thick
           pobj->add_plot,[p0[0],p[0]],[p0[1],p[1]],[p0[2],p[2]],color=color,/addarrow,thick=self.thick
           pobj->add_text,'('+dm_to_string(hkl[*,i],sep=',')+')!C',p1[0],p1[1],p1[2],color=color,normalize=0,fontsize=fsize,baseline=baseline,updir=updir,vertical_alignment=0.25
       endfor
       nx = floor(xran[1]+0.25)-ceil(xran[0]-0.25)+1
       ny = floor(yran[1]+0.25)-ceil(yran[0]-0.25)+1
       nz = floor(zran[1]+0.25)-ceil(zran[0]-0.25)+1
       xs = rebin(ceil(xran[0]-0.25)+indgen(nx),nx,ny,nz)
       ys = transpose(rebin(ceil(yran[0]-0.25)+indgen(ny),ny,nx,nz),[1,0,2])
       zs = transpose(rebin(ceil(zran[0]-0.25)+indgen(nz),nz,nx,ny),[1,2,0])
       xid = 0 & yid = 1     
    endelse
    pobj->add_plot,xs,ys,zs,psym=(['sphere','filled circle'])[keyword_set(is2d)],symsize=([0.01,0.008])[keyword_set(is2d)],color='black',linestyle='no line' ;lattice points
    pobj->setproperty,xran=xran+[-0.25,0.25],yran=yran+[-0.25,0.25],zran=zran+[-0.25,0.25],xtickinterval=1,xtickminor=0,ytickinterval=1,ytickminor=0,ztickinterval=1,ztickminor=0,cornertxt=cornertxt,$
       title=title,xtit=(['H','K','L'])[xid],ytit=(['H','K','L'])[yid],ztit='L',/showxgrid,/showygrid,/showzgrid,gridlinestyle='dashed',gridcolor='grey',axisthick=self.thick,isotropic=self.flags[1],/noevent
end

function dm_ghoston::erange,hkl,bragglast,lefthand=lefthand,eihkl=eihkl,efhkl=efhkl
    if (n_elements(hkl) eq 0) or (n_elements(bragglast)*3 ne n_elements(hkl)) then return,[0,0]
    if self.eresl ne 0 then minEstep = abs(self.eresl)<(self.minEstep) else minEstep = self.minEstep
    nhkl = n_elements(bragglast)
    eran = fltarr(2+2*self.e_is_range,nhkl)+!values.f_nan    ;is E is a range, eran[*,i]=[erange, ei, ef]
    if n_elements(eihkl) eq nhkl then Ei0 = eihkl else Ei0 = replicate(self.Ei[0,1],nhkl) 
    if n_elements(efhkl) eq nhkl then Ef0 = efhkl else Ef0 = replicate(self.Ef[0,1],nhkl)
    calc_dEi = finite(self.dEi,/nan)
    calc_dEf = finite(self.dEf,/nan)
    if calc_dEi then dEi0 = abs(self.eresl*Ei0) else dEi0 = replicate(abs(self.dEi),nhkl)
    if calc_dEf then dEf0 = abs(self.eresl*Ef0) else dEf0 = replicate(abs(self.dEf),nhkl)
    dEi = ((dEi0[0] eq 0)?(0.01*Ei0[0]):(dEi0[0]))
    dEf = ((dEf0[0] eq 0)?(0.01*Ef0[0]):(dEf0[0]))
    q_len = norm(self.abc_r#self.Q[*,1])
    if self.fixeief[1] eq 0 then begin   ;fixed Ei
       Ei = Ei0[0] & ki = 2.*!pi/dm_e2lambda(Ei)
       Eimin = ((Ei-dEi) le 0?(minEstep<Ei):(Ei-dEi))
       kiran = 2.*!pi/dm_e2lambda([Eimin,Ei+dEi])
       Efran = dm_lambda2e(2.*!pi/[abs(ki-q_len),ki+q_len])
    endif else begin                  ;fixed Ef
       Ef = Ef0[0]& kf = 2.*!pi/dm_e2lambda(Ef)
       Efmin = ((Ef-dEf) le 0?(minEstep<Ef):(Ef-dEf))
       kfran = 2.*!pi/dm_e2lambda([Efmin,Ef+dEf])
       Eiran = dm_lambda2e(2.*!pi/[abs(kf-q_len),kf+q_len])             
    endelse
    ones = replicate(1,9)
    for i=0L,nhkl-1 do begin
        qq_len = norm(self.abc_r#(self.Q[*,1]-hkl[*,i]))
        hkl1 = hkl[*,i]#ones
        for j=0,1 do begin  ;upper and lower range
            step  = ([dEf0[i],dEi0[i]])[self.fixeief[1]]*([-2.0,2.0])[j]
            if self.fixeief[1] eq 0 then begin  
               Ef = Ef0[i] & oldEf = Ef
            endif else begin
               Ei = Ei0[i] & oldEi = Ei
            endelse
            if self.fixeief[1] then eran[([0,1])[j],i] = Ei-Ef else eran[([1,0])[j],i] = Ei-Ef
            while (abs(step) gt minEstep) do begin  
                  if self.fixeief[1] eq 0 then begin ;fixed Ei
                     if calc_dEf then dEf = abs(self.eresl*Ef)
                     kf    = 2.*!pi/dm_e2lambda(Ef)
                     Eiran = dm_lambda2e(2.*!pi/[abs(kf-q_len),kf+q_len])
                     Efmin = (Efran[0])>((Ef-dEf) le 0?(minEstep<Ef):(Ef-dEf))
                     kfran = 2.*!pi/dm_e2lambda([Efmin,(Ef+dEf)<(Efran[1])])
                     Eis   = [Ei,(Eiran[0])>[Eimin,Eimin,Ei+dEi,Ei+dEi,Ei,Ei,Eimin,Ei+dEi]<(Eiran[1])]
                     Efs   = [Ef,([Efmin,Ef+dEf,Ef+dEf,Efmin,Efmin,Ef+dEf,Ef,Ef])<(Efran[1])]
                  endif else begin                ;fixed Ef
                     if calc_dEi then dEi = abs(self.eresl*Ei)
                     ki    = 2.*!pi/dm_e2lambda(Ei)     
                     Efran = dm_lambda2e(2.*!pi/[abs(ki-q_len),ki+q_len])
                     Eimin = (Eiran[0])>((Ei-dEi) le 0?(minEstep<Ei):(Ei-dEi))
                     kiran = 2.*!pi/dm_e2lambda([Eimin,(Ei+dEi)<(Eiran[1])])
                     Eis   = [Ei,([Eimin,Eimin,Ei+dEi,Ei+dEi,Ei,Ei,Eimin,Ei+dEi])<(Eiran[1])]
                     Efs   = [Ef,(Efran[0])>[Efmin,Ef+dEf,Ef+dEf,Efmin,Efmin,Ef+dEf,Ef,Ef]<(Efran[1])]
                  endelse
                  hklok = 0b
                  if (qq_len lt kiran[1]+kfran[1]) and (kiran[0] lt kfran[1]+qq_len) and (kfran[0] lt kiran[1]+qq_len) then begin
                     tmp = dcs_cryst_align_a3a4(Eis,Efs,self.Q[*,1],U=self.U[*,1],V=self.V[*,1],projU1=self.U1,projU2=self.U2,projU3=self.U3,abc_r=self.abc_r,lefthand=lefthand,kihkl=kihkl,kfhkl=kfhkl,error=errors)
                     if ~errors[0] then begin
                        ind     = where(~errors)
                        kikfs   = ([[[temporary(kihkl)-hkl1]],[[temporary(kfhkl)+hkl1]]])[*,ind,bragglast[i]]
                        kikfran = ([[kiran],[kfran]])[*,bragglast[i]]
                        tmp     = sqrt(total((self.abc_r#kikfs)^2,1))
                        max_tmp = max(tmp,min=min_tmp)
                        hklok   = (max_tmp ge kikfran[0]) and (min_tmp le kikfran[1])
                     endif
                  endif
                  if self.fixeief[1] eq 0 then begin
                     if hklok then eran[([1,0])[j],i] = Ei-Ef $
                     else begin
                        Ef = oldEf
                        step = step/2.0
                     endelse
                     while((Ef+step) lt minEstep) do step=step/2.0
                     oldEf = Ef
                     Ef = Ef+step
                  endif else begin
                     if hklok then eran[([0,1])[j],i] = Ei-Ef $
                     else begin
                        Ei = oldEi
                        step = step/2.0
                     endelse
                     while((Ei+step) lt minEstep) do step=step/2.0
                     oldEi = Ei
                     Ei = Ei+step
                  endelse
            endwhile
        endfor
        if self.e_is_range then begin
           tmp_E = (min(self.E[*,1]))>(mean(eran[0:1,i],/nan))<(max(self.E[*,1]))
           eran[2:3,i] = [([self.Ei[0,1],self.Ef[0,1]+tmp_E])[self.fixeief[1]],([self.Ei[0,1]-tmp_E,self.Ef[0,1]])[self.fixeief[1]]]
        endif
    endfor
    return,reform(eran)
end

function dm_ghoston::find_hkl,Ei,Ef,dEi,dEf,bragglast=bragglast,lefthand=lefthand,hkl=hkl,a3a4=a3a4,kihkl=kihkl
    q_len = norm(self.abc_r#self.Q[*,1])
    kikf  = 2.*!pi/dm_e2lambda([Ei,Ef])
    Eiran = dm_lambda2e(2.*!pi/[abs(kikf[1]-q_len),kikf[1]+q_len])
    Efran = dm_lambda2e(2.*!pi/[abs(kikf[0]-q_len),kikf[0]+q_len])
    if (Ei lt Eiran[0]) or (Ei gt Eiran[1]) or (Ef lt Efran[0]) or (Ef gt Efran[1]) then return,0 
    if finite(dEi,/nan) then dEi = self.eresl*Ei else dEi = abs(dEi)
    if finite(dEf,/nan) then dEf = self.eresl*Ef else dEf = abs(dEf)
    if self.eresl ne 0  then minEstep = abs(self.eresl)<(self.minEstep) else minEstep = self.minEstep
    Eimin = ((Ei-dEi) le 0?(minEstep<Ei):(Ei-dEi))>(Eiran[0])
    Efmin = ((Ef-dEf) le 0?(minEstep<Ef):(Ef-dEf))>(Efran[0])
    Eimax = (Ei+dEi)<(Eiran[1])
    Efmax = (Ef+dEf)<(Efran[1])
    kiran = 2.*!pi/dm_e2lambda([Eimin,Eimax])
    kfran = 2.*!pi/dm_e2lambda([Efmin,Efmax])
    maxk  = (kiran[1]>kfran[1])
    Eis   = [Ei,Ei,Ei,Eimin,Eimin,Eimin,Eimax,Eimax,Eimax]
    Efs   = [Ef,Efmin,Efmax,Ef,Efmin,Efmax,Ef,Efmin,Efmax]
    a3a4  = dcs_cryst_align_a3a4(Eis,Efs,self.Q[*,1],U=self.U[*,1],V=self.V[*,1],projU1=self.U1,projU2=self.U2,projU3=self.U3,abc_r=self.abc_r,lefthand=lefthand,kihkl=kihkl,kfhkl=kfhkl,error=errors)
    if errors[0] then return,0
    ind   = where(~errors,count)
    kihkl = kihkl[*,ind]
    kfhkl = kfhkl[*,ind]
    ones  = replicate(1,count)
    hkl   = [0,0,0] & bragglast = 0b
    max_h = ceil(2*maxk/norm(self.abc_r[*,0]))
    max_k = ceil(2*maxk/norm(self.abc_r[*,1]))
    max_l = ceil(2*maxk/norm(self.abc_r[*,2]))
    for h=-max_h,max_h do begin
        for k=-max_k,max_k do begin
            for l=-max_l,max_l do begin
                if (h eq 0) and (k eq 0) and (l eq 0) then continue             
                skip = 0b
                if self.flags[2] gt 0 then begin
                   case self.flags[2] of
                        1: skip = ((h+k+l) mod 2)                                                                                               ;bcc
                        2: skip = keyword_set(total(abs([h,k,l]) mod 2) mod 3)                                                                  ;fcc
                        3: skip = (keyword_set(total(abs([h,k,l]) mod 2) mod 3) or ((~total(abs([h,k,l]) mod 2)) and (((h+k+l) mod 4) ne 0)))   ;diamond fcc
                        4: skip = ((l mod 2) and (~((h+2*k) mod 3)))                                                                            ;hcp
                        else:
                   endcase
                endif
                if skip then continue
                qq_len = norm(self.abc_r#(self.Q[*,1]-[h,k,l]))
                if (qq_len lt kiran[1]+kfran[1]) and (kiran[0] lt kfran[1]+qq_len) and (kfran[0] lt kiran[1]+qq_len) and (qq_len gt 0) then begin                
                   ;bragg first
                   tmp = sqrt(total((self.abc_r#(kihkl-[h,k,l]#ones))^2,1))                
                   max_tmp = max(tmp,min=min_tmp)
                   if (max_tmp ge kiran[0]) and (min_tmp le kiran[1]) then begin
                      hkl = [[temporary(hkl)],[[h,k,l]]]
                      bragglast = [temporary(bragglast),0b]
                   endif
                   ;bragg last                
                   tmp = sqrt(total((self.abc_r#(kfhkl+[h,k,l]#ones))^2,1))
                   max_tmp = max(tmp,min=min_tmp)
                   if (max_tmp ge kfran[0]) and (min_tmp le kfran[1]) then begin
                      hkl = [[temporary(hkl)],[[h,k,l]]]
                      bragglast = [temporary(bragglast),1b]
                   endif
                endif
            endfor
        endfor
    endfor
    if n_elements(bragglast) eq 1 then return,0
    a3a4  = a3a4[*,0]
    kihkl = kihkl[*,0]
    hkl   = hkl[*,1:*]
    bragglast = bragglast[1:*]
    return,n_elements(bragglast)
end

function dm_ghoston::getsymbolfont,fsize
    fonts=['symbol*',$
           '-adobe-symbol-medium-r-normal--14-140-75-75-p-85-adobe-fontspecific',$
           '-adobe-symbol-medium-r-normal--14-100-100-100-p-85-adobe-fontspecific']
    fonts[0] = fonts[0]+dm_to_string(fsize+2,/int)
    catch, myerror
    if myerror ne 0 then return,fonts[0]
    result = fonts[0]
    switch !version.os_family of
           'Windows':break
           'unix':
           'MacOS':begin
                   spawn,'xlsfonts',result,errresult,exit_status=error
                   if error ne 0 then begin
                      old_win = !d.window
                      device,get_fontnames=result,set_font='*'    ;will pop up a window, reason for using xlsfonts first
                      if !d.window ne old_win then wdelete, !d.window
                   endif
                   nlist = n_elements(result)
                   tmp   = bytarr(nlist)
                   for i=0,nlist-1 do tmp[i] = ((strpos(strlowcase(result[i]),'symbol'))<0)+1
                   index = where(tmp,count)
                   if count ne 0 then f_list=result[index]
                   nlist = n_elements(f_list)
                   if nlist ne 0 then begin
                      tmp = intarr(nlist)
                      for i=0,nlist-1 do tmp[i]=dm_to_number((strsplit(f_list[i],'[A-Za-z-]',/regex,/extract))[0],/int)
                      void = min(abs(tmp-fsize+2),index)
                      fonts[1] = f_list[index[0]]
                   endif
                   result = fonts[1]
                   break
                   end
           else:
    endswitch
    return,result
end

;list
pro dm_ghoston::list,lefthand=lefthand
    l_r = keyword_set(lefthand)
    list = 'No ghoston has been found scattering to the '+(['right','left'])[l_r]+' for E='+dm_to_string(self.E[0:self.e_is_range,1],sep=' to ')+' meV, Q=('+dm_to_string(self.Q[*,1],sep=', ',/nozeroend)+').'
    if ptr_valid(self.blast[l_r]) then begin
       hkl   = *(self.hkl[l_r])
       blast = *(self.blast[l_r])
       eran  = *(self.eran[l_r])
       tab   = string(9b)
       tmp   = min(abs(eran[1,*]-eran[0,*]))
       sEiEf = shift(['Ei','Ef'],self.fixeief[1])
       dEiEf = shift([self.dEi,self.dEf],self.fixeief[1])
       if tmp ne 0 then eresl = ((ceil(-alog10(tmp)))>self.str_Eresl) else eresl = self.str_Eresl
       list  = 'Scattering to the '+(['right','left'])[l_r]+', fixed '+sEiEf[0]+'='+dm_to_string(([self.Ei[0,1],self.Ef[0,1]])[self.fixeief[1]])+' meV, d'+sEiEf[0]+'='+$
               (finite(dEiEf[0])?(dm_to_string(dEiEf[0])+' meV'):(dm_to_string(self.eresl)+'*'+sEiEf[0]))+', d'+sEiEf[1]+'='+(finite(dEiEf[1])?(dm_to_string(dEiEf[1])+' meV'):$
               (dm_to_string(self.eresl)+'*'+sEiEf[1]))+', E='+dm_to_string(self.E[0:self.e_is_range,1],sep=' to ')+' meV, Q=('+dm_to_string(self.Q[*,1],sep=', ',/nozeroend)+')'+$
               ([', a3='+dm_to_string(self.a3a4[0,l_r],res=3)+', a4='+dm_to_string(self.a3a4[1,l_r],res=3),''])[self.e_is_range]
       list  = [list,'lattice parameters=('+dm_to_string(self.lattparm[*,1],sep=', ',/nozeroend)+'), U=('+dm_to_string(self.U[*,1],sep=', ')+'), V=('+dm_to_string(self.V[*,1],sep=', ')+')'+$
               (['',', lattice type: Monatomic '+['BCC','FCC','Diamond FCC','HCP']])[self.flags[2]],'']
       list  = [list,(['Bragg first:','Bragg last:'])[blast[0]]]
       slen  = 0
       for i=0,n_elements(blast)-1 do slen = slen>strlen(dm_to_string(hkl[*,i],sep=', '))
       for i=0,n_elements(blast)-1 do begin
           if self.e_is_range then E = eran[2,i]-eran[3,i] else E = self.E[0,1]
           if blast[i] ne blast[(i-1)>0] then list = [list,'',(['Bragg first:','Bragg last:'])[blast[i]]] ;add a line btw bragg first and bragg last
           fwhm = abs(dm_to_number(dm_to_string(eran[1,i],res=eresl))-dm_to_number(dm_to_string(eran[0,i],res=eresl))) ;to avoid rounding errors
           tmp  = dm_to_string(hkl[*,i],sep=', ')
           list = [list,'('+tmp+'),'+strjoin(' '+strarr(slen-strlen(tmp)+1))+tab+'E='+(dm_to_string([E,0.1],res=eresl))[0]+' meV, '+tab+'fwhm='+$
                  (dm_to_string([fwhm,0.1],res=eresl))[0]+' meV, '+tab+'Eran=['+dm_to_string(eran[0:1,i],sep=', ',res=eresl)+'] meV, '+tab+'ghoston q=('+$
                  dm_to_string(self.Q[*,1]-hkl[*,i],sep=', ',/nozeroend)+')']
       endfor
       list = [list,'','Eran is the energy range in which the ghoston is observable for fixed '+sEiEf[0]+' and given energy resolution.']
    endif
    widget_control,self.tbox[l_r],set_value=list
end

;load parameter file
pro dm_ghoston::load_parm,parmfile=parmfile,init=init,info=info
    if n_elements(parmfile) eq 0 then parmfile = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'.ghoston'
    openr,unit,parmfile,/get_lun,error=openerr
    if openerr ne 0 then return 
    tmp = ''
    while(~ eof(unit)) do begin
        readf,unit,tmp
        tmp  = strtrim(tmp,2)
        head = strmid(tmp,0,7)
        tmp  = strmid(tmp,7,strlen(tmp)-7)
        case strlowcase(head) of
             'lattpm:': self.lattparm[*,0] = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
             'orie_u:': self.U[*,0]        = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
             'orie_v:': self.V[*,0]        = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
             'fixeif:': self.fixeief[0]    = dm_to_number(tmp,/int)
             'e_resl:': self.eresl         = abs(dm_to_number(tmp))
             'ei_val:': self.Ei[*,0]       = ([dm_to_number(strsplit(tmp,'&',/extract,/preserve)),!values.f_nan])[0:1]
             'ei_res:': self.dEi           = dm_to_number(tmp)
             'ef_val:': self.Ef[*,0]       = ([dm_to_number(strsplit(tmp,'&',/extract,/preserve)),!values.f_nan])[0:1]
             'ef_res:': self.dEf           = dm_to_number(tmp)
             'en_val:': self.E[*,0]        = ([dm_to_number(strsplit(tmp,'&',/extract,/preserve)),!values.f_nan])[0:1]
             'q_vect:': self.Q[*,0]        = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
             'isosca:': self.flags[1]      = dm_to_number(tmp)
             'latttp:': if self.flags[2] ne dm_to_number(tmp) then begin
                        self.flags[2]      = 0>(dm_to_number(tmp))<4
                        if self.flags[2] ne 0 then info='The lattice type is now set to monatomic '+(['BCC','FCC','Diamond FCC','HCP'])[self.flags[2]-1]+'. You can change it at the Options menu.'
                        endif
             'tckdir:': self.flags[6]      = dm_to_number(tmp,/int)           
             'wrkdir:': self.workdir       = tmp
             else:
        endcase
    endwhile
    free_lun,unit 
    if keyword_set(init) then return
    uname = ['latt'+['A','B','C','AA','BB','CC'],'deief','defei','orie'+['Ux','Uy','Uz','Vx','Vy','Vz'],'Qx','Qy','Qz']
    value = [self.lattparm[*,0],([self.dEi,self.dEf])[self.fixeief[0]],([self.dEf,self.dEi])[self.fixeief[0]],self.U[*,0],self.V[*,0] ,self.Q[*,0]]
    for i=0,n_elements(uname)-1 do self->my_widget_control,uname[i],set_value=dm_to_string(value[i])
    self->my_widget_control,'eief',set_value=dm_to_string(([self.Ei[0,0],self.Ef[0,0]])[self.fixeief[0]])
    if self.fixeief[0] then tmp = self.Ei[*,0] else tmp = self.Ef[*,0]
    self->my_widget_control,'efei',set_value=dm_to_string(tmp[0:finite(tmp[1])],sep=':')
    self->my_widget_control,'en',set_value=dm_to_string(self.E[0:finite(self.E[1,0]),0],sep=':')
    dm_set_droplist,widget_info(self.tlb,find_by_uname='fixEiEf'),select=self.fixeief[0]
    self->my_widget_control,'lab_efei',set_value=(['Ef','Ei'])[self.fixeief[0]]
    self->my_widget_control,'lab_deief',set_value=(['dEi','dEf'])[self.fixeief[0]]
    self->my_widget_control,'lab_defei',set_value=(['dEf','dEi'])[self.fixeief[0]]
    dm_set_button,widget_info(self.tlb,find_by_uname='isoscale'),self.flags[1]
    self->my_menu_toggle,'tdir'+['inside','outside'],self.flags[6]
    for i=0,1 do self.plots[i]->setproperty,isotropic=self.flags[1],tickdir=self.flags[6],/noevent
    if file_test(self.workdir,/directory,/write) then for i=0,1 do self.plots[i]->setproperty,path=self.workdir
end

;save list
pro dm_ghoston::save_list
    path = self.workdir
    ind  = where(ptr_valid(self.blast),count)
    for i=0,count-1 do begin
        widget_control,self.tbox[ind[i]],get_value=texts
        if total(strlen(texts)) eq 0 then continue
        file = dm_choose_file('txt',dialog_parent=self.tlb,/write,path=path,file='ghoston_'+(['','bcc_','fcc_','dfcc_','hcp_'])[self.flags[2]]+(['r','l'])[ind[i]]+'.txt',$
               title='Please select a file to save the list of scattering to '+(['right','left'])[ind[i]])
        if strlen(file) ne 0 then begin    
           openw,unit,file,/get_lun,error=openerr
           if openerr ne 0 then continue
           for j=0L,n_elements(texts)-1 do printf,unit,texts[j]
           free_lun,unit
        endif else break
    endfor
    if (self.workdir ne path) and file_test(path,/directory,/write) then begin
       self.workdir = path
       for i=0,1 do self.plots[i]->setproperty,path=path
    endif
end

;save parameter file
pro dm_ghoston::save_parm,parmfile=parmfile
    if n_elements(parmfile) eq 0 then parmfile = dm_define_pointer(/gettempdir)+dm_define_pointer(/getpathsep)+'.ghoston'
    openw,unit,parmfile,/get_lun,error=openerr
    if openerr ne 0 then return
    printf,unit,'lattpm:'+dm_to_string(self.lattparm[*,0],sep='&')
    printf,unit,'orie_u:'+dm_to_string(self.U[*,0],sep='&')
    printf,unit,'orie_v:'+dm_to_string(self.V[*,0],sep='&')
    printf,unit,'fixeif:'+dm_to_string(self.fixeief[0])
    printf,unit,'e_resl:'+dm_to_string(self.eresl)
    printf,unit,'ei_val:'+dm_to_string(self.Ei[*,0],sep='&')
    printf,unit,'ei_res:'+dm_to_string(self.dEi)
    printf,unit,'ef_val:'+dm_to_string(self.Ef[*,0],sep='&')
    printf,unit,'ef_res:'+dm_to_string(self.dEf)
    printf,unit,'en_val:'+dm_to_string(self.E[*,0],sep='&')
    printf,unit,'q_vect:'+dm_to_string(self.Q[*,0],sep='&')
    printf,unit,'isosca:'+dm_to_string(self.flags[1])
    printf,unit,'latttp:'+dm_to_string(self.flags[2])
    printf,unit,'tckdir:'+dm_to_string(self.flags[6])
    printf,unit,'wrkdir:'+self.workdir
    free_lun,unit
end

;toggle a group of menu buttons according to uname and a number
pro dm_ghoston::my_menu_toggle,uname,flag
    nuname = n_elements(uname)
    if (nuname eq 0) or (n_elements(flag) eq 0) then return
    ids = lonarr(nuname)
    for i=0,nuname-1 do ids[i] = widget_info(self.tlb,find_by_uname=uname[i])
    ind = where(ids gt 0,n_valid)
    if n_valid eq 0 then return
    check = where(ind eq flag[0],n_check,complement=uncheck,ncomplement=n_uncheck)
    if n_check eq 0 then tmp = temporary(check) else check = ids[ind[check]]
    if n_uncheck eq 0 then tmp = temporary(uncheck) else uncheck = ids[ind[uncheck]]
    dm_toggle_menubut,check=check,uncheck=uncheck
end

;a widget_control use uname instead of id as identifier
pro dm_ghoston::my_widget_control,uname,_ref_extra=extra
    if n_elements(extra) ne 0 then begin
       for i=0,n_elements(uname)-1 do begin
           wid = widget_info(self.tlb,find_by_uname=uname[i])
           if wid ne 0 then widget_control,wid,_extra=extra
       endfor
    endif
end

;widget_tex with tab_mode=1,/editable,/all_events,/kbrd_focus_events
function dm_ghoston::my_widget_text,base,ysize=ysize,_ref_extra=extra
    if n_elements(ysize) eq 0 then ysize=1
    return,widget_text(base,tab_mode=1,/editable,/all_events,/kbrd_focus_events,ysize=ysize,_extra=extra)
end

;object event handler
pro dm_ghoston::event,event
    compile_opt IDL2    ;idl2=defint32,strictarr

    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=self.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       if obj_valid(mesg) then obj_destroy,mesg
       return
    end
    
    widget_type = widget_info(event.id,/type)
    case widget_type of
         1: tmp_value = event.select                                ;button
         3: begin                                                   ;text box
            widget_control,event.id,get_value=tmp_value
            tmp_value = strtrim(tmp_value,2)
            end
         8: tmp_value = widget_info(event.id,/droplist_select)      ;drop list
         12:begin                                                   ;combobox
            tmp_value = event.index
            widget_control,event.id,get_value=tmp_list
            if tmp_list[tmp_value] ne event.str then return         ;to avoid events from set_value in unix operating system
            end
         else:
    endcase
    if n_elements(tmp_value) eq 1 then tmp_value=tmp_value[0]
    uname = widget_info(event.id,/uname)
    type  = tag_names(event,/structure)
    case strlowcase(type) of
         'widget_kbrd_focus':   if widget_info(event.id,/name) eq 'TEXT' then uname = 'text_focus'
         'widget_text_sel':     if self.txtfocusevent then begin
                                   tmp = widget_event(event.id,/nowait)
                                   widget_control,event.id,get_value=tmp
                                   if n_elements(tmp) eq 1 then begin
                                      if strlen(strtrim(tmp,2)) ne 0 then widget_control,event.id,set_text_select=[0,strlen(tmp[0])]
                                   endif   
                                endif
         'dm_plot_changetitle': uname = type
         'dm_plot_isotropic':   uname = type
         else: 
    endcase
    if strlen(uname) eq 0 then return
    if uname ne 'text_focus' then WIDGET_CONTROL,/HOURGLASS
    self.txtfocusevent = 0b
    old_flag2 = self.flags[2]
    case strlowcase(uname) of
         'tlb':       if strlowcase(tag_names(event,/structure)) eq 'widget_base' then begin ;resize event
                      widget_control,self.tlb,update=0
                      if (event.x lt self.xmin) or (event.y lt self.ymin) then begin
                         event.x = (event.x)>(self.xmin)
                         event.y = (event.y)>(self.ymin)
                      endif
                      event1 = event
                      event1.x = event1.x-self.xoffset[0]
                      event1.y = event1.y-self.yoffset[0]
                      self->my_widget_control,'tabbase',scr_xsize=event1.x+self.xmargin,scr_ysize=event1.y+self.ymargin                      
                      for i=0,1 do begin
                          self.plots[i]->resize,event1
                          widget_control,self.tbox[i],scr_xsize=event.x-self.xoffset[1],scr_ysize=event.y-self.yoffset[1]
                      endfor
                      widget_control,self.tlb,/update,/realize
                      endif     
         'tabbase':   self->my_widget_control,['saveplot','printplot'],sensitive=ptr_valid(self.blast[self->currenttabnum()])
         'savelist':  self->save_list
         'saveplot':  begin
                      self->my_widget_control,'tabbase_'+(['r','l'])[self->currenttabnum()],SET_TAB_CURRENT=0
                      self.plots[self->currenttabnum()]->saveas,'png'
                      end
         'printplot': begin
                      self->my_widget_control,'tabbase_'+(['r','l'])[self->currenttabnum()],SET_TAB_CURRENT=0
                      self.plots[self->currenttabnum()]->saveas,'printer'
                      end
         'saveparm':  begin
                      path = self.workdir
                      parmfile = dm_choose_file('prm',dialog_parent=self.tlb,/write,path=path)
                      if strlen(parmfile) ne 0 then begin
                         self->save_parm,parmfile=parmfile
                         if file_test(path,/directory,/write) then begin
                            self.workdir = path
                            for i=0,1 do self.plots[i]->setproperty,path=path
                         endif
                      endif
                      end
         'loadparm':  begin
                      path = self.workdir
                      parmfile = dm_choose_file('prm',dialog_parent=self.tlb,/read,path=path)
                      if strlen(parmfile) ne 0 then begin
                         self->load_parm,parmfile=parmfile,info=info
                         if file_test(path,/directory,/write) then begin
                            self.workdir = path
                            for i=0,1 do self.plots[i]->setproperty,path=path
                         endif
                      endif
                      end                         
         'exit':      widget_control,self.tlb,/destroy
         'isoscale':  begin
                      self.flags[1] = ~self.flags[1]
                      dm_set_button,event.id,self.flags[1]
                      for i=0,1 do if (~self.e_is_range) or ((*self.listind[i])[self.viewsel[i]] ge 0) then self.plots[i]->setproperty,isotropic=self.flags[1],/noevent
                      end 
         'circle':    begin
                      self.flags[3] = ~self.flags[3]
                      dm_set_button,event.id,self.flags[3]
                      for lefthand=0,1 do if (~self.e_is_range) or ((*self.listind[lefthand])[self.viewsel[lefthand]] ge 0) then self->draw,lefthand=lefthand
                      end       
         'showuv':    begin
                      self.flags[4] = ~self.flags[4]
                      dm_set_button,event.id,self.flags[4]
                      for lefthand=0,1 do if (~self.e_is_range) or ((*self.listind[lefthand])[self.viewsel[lefthand]] ge 0) then self->draw,lefthand=lefthand
                      end 
         'showkikf':  begin
                      self.flags[5] = ~self.flags[5]
                      dm_set_button,event.id,self.flags[5]
                      for lefthand=0,1 do if (~self.e_is_range) or ((*self.listind[lefthand])[self.viewsel[lefthand]] ge 0) then self->draw,lefthand=lefthand
                      end                       
         'fsize':     begin
                      new = dm_dialog_input(['font size:'],xsize=100,default=self.fsize,dialog_parent=self.tlb,cancel=cancel,info='The font size affects all texts in the plots.')
                      if (~keyword_set(cancel)) and finite(new) then begin
                         self.fsize = round(abs(new))
                         for i=0,1 do self.plots[i]->change_font,fsize=self.fsize,/all
                      endif
                      end
         'tdirinside':begin
                      self.flags[6] = 0
                      self->my_menu_toggle,'tdir'+['inside','outside'],self.flags[6]
                      for i=0,1 do self.plots[i]->setproperty,tickdir=self.flags[6]
                      end
         'tdiroutside':begin
                      self.flags[6] = 1
                      self->my_menu_toggle,'tdir'+['inside','outside'],self.flags[6]
                      for i=0,1 do self.plots[i]->setproperty,tickdir=self.flags[6]
                      end
         'thick':     begin
                      new = dm_dialog_input(['line thickness:'],xsize=100,default=self.thick,dialog_parent=self.tlb,cancel=cancel,info='The line thickness affects axes and plots.')
                      if (~keyword_set(cancel)) and finite(new) and (new ne self.thick) then begin 
                         self.thick = abs(new)
                         for lefthand=0,1 do self->draw,lefthand=lefthand
                      endif
                      end
         'dm_plot_isotropic': begin
                      self.flags[1] = event.value
                      dm_set_button,widget_info(self.tlb,find_by_uname='isoscale'),self.flags[1]
                      for i=0,1 do if (~self.e_is_range) or ((*self.listind[i])[self.viewsel[i]] ge 0) then self.plots[i]->setproperty,isotropic=self.flags[1],/noevent
                      end
         'eresl':     begin
                      new = dm_dialog_input('factor:',title='E Resolution Factor',dialog_parent=self.tlb,default=self.eresl,info='dEi(f) = factor*Ei(f) will be used if dEi(f) is not specified.',/float,cancel=cancel)
                      if keyword_set(cancel) or finite(new,/nan) then return
                      self.eresl = abs(new)
                      end              
         'text_focus':begin
                      if event.enter then self.txtfocusevent = 1b $
                      else widget_control,event.id,set_text_select=[0,0]
                      end
         'latta':     self.lattparm[0,0] = dm_to_number(tmp_value,/float)   ;a
         'lattb':     self.lattparm[1,0] = dm_to_number(tmp_value,/float)   ;b
         'lattc':     self.lattparm[2,0] = dm_to_number(tmp_value,/float)   ;c
         'lattaa':    self.lattparm[3,0] = dm_to_number(tmp_value,/float)   ;alpha
         'lattbb':    self.lattparm[4,0] = dm_to_number(tmp_value,/float)   ;beta
         'lattcc':    self.lattparm[5,0] = dm_to_number(tmp_value,/float)   ;gama
         'orieux':    self.U[0,0]        = dm_to_number(tmp_value,/float)   ;ux
         'orieuy':    self.U[1,0]        = dm_to_number(tmp_value,/float)   ;uy
         'orieuz':    self.U[2,0]        = dm_to_number(tmp_value,/float)   ;uz
         'orievx':    self.V[0,0]        = dm_to_number(tmp_value,/float)   ;vx
         'orievy':    self.V[1,0]        = dm_to_number(tmp_value,/float)   ;vy
         'orievz':    self.V[2,0]        = dm_to_number(tmp_value,/float)   ;vz  
         'qx':        self.Q[0,0]        = dm_to_number(tmp_value,/float)   ;Qx
         'qy':        self.Q[1,0]        = dm_to_number(tmp_value,/float)   ;Qx
         'qz':        self.Q[2,0]        = dm_to_number(tmp_value,/float)   ;Qx
         'fixeief':   if self.fixeief[0] ne tmp_value then begin
                      self.fixeief[0] = tmp_value
                      self->my_widget_control,'lab_efei',set_value=(['Ef','Ei'])[self.fixeief[0]]
                      self->my_widget_control,'lab_deief',set_value=(['dEi','dEf'])[self.fixeief[0]]
                      self->my_widget_control,'lab_defei',set_value=(['dEf','dEi'])[self.fixeief[0]]
                      tmp = self.Ei[*,0]  & self.Ei[*,0]  = self.Ef[*,0]  & self.Ef[*,0]  = tmp
                      tmp = self.dEi & self.dEi = self.dEf & self.dEf = tmp                                   
                      self->adjust_E
                      end
         'eief':      begin
                      tmp_value = dm_to_number(tmp_value,/float) 
                      if self.fixeief[0] then begin
                         self.Ef[*,0] = tmp_value
                      endif else begin
                         self.Ei[*,0] = tmp_value
                      endelse
                      if event.type eq 0 then if event.ch eq 10b then self->adjust_E,eiefchanged=finite(tmp_value)
                      end
         'deief':     begin
                      if self.fixeief[0] then begin
                         self.dEf = dm_to_number(tmp_value,/float)
                      endif else begin
                         self.dEi = dm_to_number(tmp_value,/float)
                      endelse
                      end              
         'efei':      begin
                      tmp_value = dm_to_number(strsplit(tmp_value,':',/extract,/preserve),/float)
                      if n_elements(tmp_value) eq 1 then tmp_value = [tmp_value,!values.f_nan]
                      if self.fixeief[0] then begin
                         self.Ei[*,0] = tmp_value
                      endif else begin
                         self.Ef[*,0] = tmp_value
                      endelse
                      if event.type eq 0 then if event.ch eq 10b then self->adjust_E,efeichanged=finite(tmp_value[0])
                      end
         'defei':     begin
                      if self.fixeief[0] then begin
                         self.dEi = dm_to_number(tmp_value,/float)
                      endif else begin
                         self.dEf = dm_to_number(tmp_value,/float)
                      endelse
                      end              
         'en':        begin
                      if event.type eq 0 then begin ;check "-" character only when return key is pressed
                         if event.ch eq 10b then begin
                            if stregex(tmp_value,'[0-9\.]+-[0-9\.]+',/boolean) then info = 'Please use ":" for a range of energy.'
                         endif
                      endif
                      tmp_value = dm_to_number(strsplit(tmp_value,':',/extract,/preserve),/float)
                      if n_elements(tmp_value) eq 1 then begin
                         tmp_value = [tmp_value,!values.f_nan]
                         if self.fixeief[0] then self.Ei[1,0] = !values.f_nan else self.Ef[1,0] = !values.f_nan
                      endif
                      self.E[*,0] = tmp_value
                      self.eplot_setting[0:1] = dm_to_string(tmp_value)
                      if event.type eq 0 then if event.ch eq 10b then self->adjust_E,enchanged=finite(tmp_value[0])
                      end  
         'calc':      begin
                      self->adjust_E
                      if self->check_parm() then begin
                         widget_control,/hourglass
                         mesg = obj_new('dm_progress',title='Please wait',message='Calculating scattering to the right...',group_leader=self.tlb,/nostop)
                         for lefthand=0,1 do begin
                             if lefthand eq 1 then mesg->update,message='Calculating scattering to the left...'
                             self->calc_ghoston,lefthand=lefthand
                             self->draw,lefthand=lefthand
                         endfor
                         obj_destroy,mesg
                         self->my_widget_control,'savelist',sensitive=(total(ptr_valid(self.blast)) ne 0)
                         self->my_widget_control,'tabbase_'+['r','l'],SET_TAB_CURRENT=1
                      endif
                      end         
         'viewsel_r': if (self.viewsel[0] ne tmp_value) or (self.e_is_range and ((*self.listind[0])[tmp_value] lt 0)) then begin    
                      self.viewsel[0] = tmp_value
                      if ptr_valid(self.blast[0]) then self->draw
                      endif else self->my_widget_control,['isoscale','circle','showuv','showkikf'],sensitive=((~self.e_is_range) or ((*self.listind[0])[tmp_value] ge 0))
         'viewsel_l': if (self.viewsel[1] ne tmp_value) or (self.e_is_range and ((*self.listind[1])[tmp_value] lt 0)) then begin
                      self.viewsel[1] = tmp_value
                      if ptr_valid(self.blast[1]) then self->draw,/lefthand
                      endif else self->my_widget_control,['isoscale','circle','showuv','showkikf'],sensitive=((~self.e_is_range) or ((*self.listind[1])[tmp_value] ge 0))
         'latt_none': self.flags[2] = 0
         'latt_bcc':  self.flags[2] = 1           
         'latt_fcc':  self.flags[2] = 2
         'latt_dfcc': self.flags[2] = 3
         'latt_hcp':  self.flags[2] = 4
         'ctrlc':     begin
                      r_l = self->currenttabnum(is_list=is_list)
                      if ~keyword_set(is_list) then begin ;save plot
                         self.plots[r_l]->saveas,'clipboard'
                      endif else begin  ;save list
                         text_select = widget_info(self.tbox[r_l],/text_select)
                         widget_control,self.tbox[r_l],get_value=txt,use_text_select=(text_select[1] gt 0)
                         Clipboard.Set,txt
                      endelse
                      end                                                          
         'help':      info = ['Ghoston reference: H.M. Ronnow et al., Physica B 350 (2004) 11-16.','',$
                              'Enter all fields except for dE, and press Calculate Ghoston button. If dE is not given, it will be calculated using the default energy resolution factor. '+$
                              'Use ":" to denote a range of energy. When a range of energy is given, individual ghoston is plotted using the mean energy observed in the given range.','',$
                              'Scattering to the right refers to clockwise rotation from ki to kf, and scattering to the left counterclockwise. Clockwise rotation for A3 is positive. '+$
                              'When A3 is 0, U is perpendicular to ki. For a TOF instrument, psi = 90-A3.','','Bragg first refers to the case when the Bragg scattering occurs first '+$
                              'in the inelastic multiple scattering process, and Bragg last is when the Bragg scattering is the last occurence of the multiple scattering. The condition '+$
                              'for the extinction in the Bragg last scattering still applies, except that the reciprocal space vectors are exactly opposite of the listed ones.','',$
                              'Check the Options menu for more options.','','Right click over the plot to change its properties; right click over the upper left corner of the plot to '+$
                              'access the plot menu.','','Yiming Qiu (yiming.qiu@nist.gov) 9/2019']                                                   
         else:
    endcase
    if n_elements(info) ne 0 then ok = dialog_message(info ,/center,dialog_parent=event.handler,title=(['','About Ghoston Spurion Viewer'])[uname eq 'help'])
    if self.flags[2] ne old_flag2 then begin ;lattice type changed, update the menu
       self->my_menu_toggle,'latt_'+['none','bcc','fcc','dfcc','hcp'],self.flags[2]
       self->my_widget_control,'lattmenu',set_button=(self.flags[2] ne 0),set_value=('Lattice Type'+['',' ('+['BCC','FCC','Diamond FCC','HCP']+')'])[self.flags[2]]
    endif
end

;cleanup
pro dm_ghoston::Cleanup
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    obj_destroy,self.plots
    ptr_free,self.hkl,self.blast,self.listind,self.eran
    self->save_parm
end

;initialization
function dm_ghoston::Init,lattparm=lattparm,u=u,v=v,workdir=workdir,tickdir=tickdir,group_leader=group_leader
    registerName = 'dm_ghoston'
    if xregistered(registerName) then begin ;only allow one copy to be running at one time
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0
       return,0
    endif
    self.idl_version = dm_to_number(!version.release)
    accel_yn = (self.idl_version ge 6.1)
    if self.idl_version ge 5.6 then scr_size = get_screen_size() else device,get_screen_size=scr_size
    if n_elements(group_leader) ne 0 then self.group_leader = group_leader
    self.eplot_setting = ['','','3','green','1','red','center']
    self.flags[3:4]    = 1b ;default show full circle, show U&V
    self.lattparm[*]   = !values.f_nan & self.U[*] = !values.f_nan & self.V[*] = !values.f_nan
    self.Ei[*]         = !values.f_nan & self.dEi = !values.f_nan & self.Ef[*] = !values.f_nan & self.dEf = !values.f_nan & self.E[*] = !values.f_nan & self.Q[*] = !values.f_nan
    self.eresl         = 0.05
    self.fsize         = ([12,13])[scr_size[0] ge 1920]
    self.thick         = 1.5
    self.xysize        = [47L,60L,5L]
    self.minEstep      = 0.001 ;meV
    self.str_Eresl     = 3
    
    self->load_parm,/init,info=info
    if n_elements(lattparm) eq 6 then self.lattparm[*,0] = lattparm
    if n_elements(u) eq 3 then self.U[*,0] = u
    if n_elements(v) eq 3 then self.V[*,0] = v
    if n_elements(workdir) ne 0 then begin
       if file_test(workdir[0],/directory,/write) then self.workdir = workdir[0]
    endif
    if n_elements(tickdir) ne 0 then self.flags[6] = keyword_set(tickdir)
    if self->check_parm(/nomesg) then begin
       if (self.lattparm[0,0] eq self.lattparm[1,0]) and (self.lattparm[0,0] eq self.lattparm[2,0]) and (max(abs(self.lattparm[3:5,0]-90)) eq 0) then self.flags[1] = 1b  ;isotropic scaling for cubic
    endif
    
    bmpfiles = ['mslice_save.bmp','mslice_open.bmp']
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    for i=0,n_elements(bmpfiles)-1 do begin
        if exists then bmpfiles[i] = !DAVE_AUXILIARY_DIR+bmpfiles[i] $
        else bmpfiles[i] =  file_which(bmpfiles[i],/include_current_dir)
    endfor
    
    defsysv,'!ddtor',exists=exists
    if (~exists) then defsysv,'!ddtor',!dpi/(180d) ;double version of !dtor
 
    self.tlb = widget_base(title='Ghoston Spurion Viewer',column=1,mbar=mbar,/tlb_size_event,kill_notify='dm_ghoston_Exit',group_leader=group_leader,map=0,uname='tlb',xpad=0)
    filemenu = widget_button(mbar,value='File',/menu)
    optnmenu = widget_button(mbar,value='Options',/menu)
    helpmenu = widget_button(mbar,value='Help',/menu)
    ;file menu
    void     = dm_widget_button(filemenu,value='Save Lists...',uname='savelist',sensitive=0,imagefile=bmpfiles[0],accelerator='Ctrl+W',/notchecked)
    void     = dm_widget_button(filemenu,value='Save Plot...',uname='saveplot',sensitive=0,imagefile=bmpfiles[0],accelerator='Ctrl+S',/notchecked)
    void     = widget_button(filemenu,value='Print Plot...',uname='printplot',sensitive=0,_extra=(accel_yn?{accelerator:'Ctrl+P'}:{no_copy:1}))
    void     = dm_widget_button(filemenu,value='Save Parameters...',uname='saveparm',imagefile=bmpfiles[0],/notchecked,/separator)
    void     = dm_widget_button(filemenu,value='Load Parameters...',uname='loadparm',imagefile=bmpfiles[1],/notchecked)
    void     = widget_button(filemenu,value='Exit',uname='exit',/separator)
    ;option menu
    lattmenu = dm_widget_button(optnmenu,value='Lattice Type',uname='lattmenu',/menu,set_button=(self.flags[2] gt 0))
    void     = dm_widget_button(lattmenu,value='None',uname='latt_none',set_button=(self.flags[2] eq 0))
    void     = dm_widget_button(lattmenu,value='Monatomic BCC (H+K+L=even)',uname='latt_bcc',set_button=(self.flags[2] eq 1))
    void     = dm_widget_button(lattmenu,value='Monatomic FCC (H,K,L all odd or all even)',uname='latt_fcc',set_button=(self.flags[2] eq 2))
    void     = dm_widget_button(lattmenu,value='Monatomic Diamond FCC (H,K,L all odd, or all even with H+K+L=4n)',uname='latt_dfcc',set_button=(self.flags[2] eq 3))
    void     = dm_widget_button(lattmenu,value='Monatomic HCP (L even, or H+2K#3n for odd L)',uname='latt_hcp',set_button=(self.flags[2] eq 4))
    void     = widget_button(optnmenu,value='Set Default Energy Resolution Factor...',uname='eresl')
    void     = dm_widget_button(optnmenu,value='Isotropic Scaling',uname='isoscale',set_button=self.flags[1],_extra=accel_yn?{accelerator:'Ctrl+I'}:{no_copy:1},/separator)
    void     = dm_widget_button(optnmenu,value='Show Full Ki (Kf) Circle',uname='circle',set_button=self.flags[3])
    void     = dm_widget_button(optnmenu,value='Show U && V',uname='showuv',set_button=self.flags[4])
    void     = dm_widget_button(optnmenu,value='Show Ki && Kf Values',uname='showkikf',set_button=self.flags[5])
    void     = widget_button(optnmenu,value='Plot Font Size...',uname='fsize')
    void     = widget_button(optnmenu,value='Plot Line Thickness...',uname='thick')
    tdirmenu = widget_button(optnmenu,value='Plot Axis Tick Direction',uname='tdirMenu',/menu)
    void     = dm_widget_button(tdirmenu,value='Inside',uname='tdirinside',set_button=~self.flags[6])
    void     = dm_widget_button(tdirmenu,value='Outside',uname='tdiroutside',set_button=self.flags[6])
    void     = widget_button(optnmenu,value='Copy to Clipboard',uname='ctrlc',_extra=accel_yn?{accelerator:'Ctrl+C'}:{no_copy:1},/separator)
    ;help menu
    void     = widget_button(helpmenu,value='Help',uname='help',_extra=accel_yn?{accelerator:'F1'}:{no_copy:1})
  
    toprow   = widget_base(self.tlb,/row)
    left     = widget_base(toprow,/col,ypad=0,xpad=0,tab_mode=1)
    row      = widget_base(left,/row,ypad=0,xpad=0)
    label    = widget_label(row,value='  Lattice Parameters:')
    geom     = widget_info(label,/geometry)
    symfont  = self->getsymbolfont(2*(fix(geom.scr_ysize)/2))
    
    row      = widget_base(left,/row,ypad=0,xpad=0)
    label    = widget_label(row,value='a ('+string('c5'XB)+')',xsize=self.xysize[1],/align_center)
    lattA    = self->my_widget_text(row,value=dm_to_string(self.lattparm[0,0]),scr_xsize=self.xysize[1],uname='lattA')
    label    = widget_label(row,value='b ('+string('c5'XB)+')',xsize=self.xysize[0],/align_center)
    lattB    = self->my_widget_text(row,value=dm_to_string(self.lattparm[1,0]),scr_xsize=self.xysize[1],uname='lattB')
    label    = widget_label(row,value='c ('+string('c5'XB)+')',xsize=self.xysize[0],/align_center)
    lattC    = self->my_widget_text(row,value=dm_to_string(self.lattparm[2,0]),scr_xsize=self.xysize[1],uname='lattC')
    row      = widget_base(left,/row,ypad=0,xpad=0)
    label    = widget_label(row,value='a ('+string('B0'XB)+')',xsize=self.xysize[1],/align_center,font=symfont)
    lattAA   = self->my_widget_text(row,value=dm_to_string(self.lattparm[3,0]),scr_xsize=self.xysize[1],uname='lattAA')
    label    = widget_label(row,value='b ('+string('B0'XB)+')',xsize=self.xysize[0],/align_center,font=symfont)
    lattBB   = self->my_widget_text(row,value=dm_to_string(self.lattparm[4,0]),scr_xsize=self.xysize[1],uname='lattBB')
    label    = widget_label(row,value='g ('+string('B0'XB)+')',xsize=self.xysize[0],/align_center,font=symfont)
    lattCC   = self->my_widget_text(row,value=dm_to_string(self.lattparm[5,0]),scr_xsize=self.xysize[1],uname='lattCC')
    row      = widget_base(left,/row,ypad=0,xpad=0)
    label    = widget_label(row,value=' ') 
    row      = widget_base(left,/row,ypad=0,xpad=0)
    fixeief  = dm_widget_droplist(row,xsize=self.xysize[1],value='fixed E'+['i','f'],select=self.fixeief[0],uname='fixEiEf')
    eief     = self->my_widget_text(row,value=dm_to_string(([self.Ei[0,0],self.Ef[0,0]])[self.fixeief[0]]),scr_xsize=self.xysize[1],uname='eief')
    label    = widget_label(row,value=(['dEi','dEf'])[self.fixeief[0]],xsize=self.xysize[0],/align_center,uname='lab_deief')
    deief    = self->my_widget_text(row,value=dm_to_string(([self.dEi,self.dEf])[self.fixeief[0]]),scr_xsize=self.xysize[1],uname='deief')
    label    = widget_label(row,xsize=self.xysize[0],value='E (meV)',/align_center)
    etran    = self->my_widget_text(row,value=dm_to_string(self.E[0:finite(self.E[1]),0],sep=':'),scr_xsize=self.xysize[1],uname='en')
    row      = widget_base(left,/row,ypad=0,xpad=0)
    label   = widget_label(row,xsize=self.xysize[1],value=(['Ef','Ei'])[self.fixeief[0]],/align_center,uname='lab_efei')
    tmp_val = ([[self.Ef[*,0]],[self.Ei[*,0]]])[*,self.fixeief[0]]
    efei    = self->my_widget_text(row,value=dm_to_string(tmp_val[0:finite(tmp_val[1])],sep=':'),scr_xsize=self.xysize[1],uname='efei')
    label   = widget_label(row,value=(['dEf','dEi'])[self.fixeief[0]],xsize=self.xysize[0],/align_center,uname='lab_defei')
    defei   = self->my_widget_text(row,value=dm_to_string(([self.dEf,self.dEi])[self.fixeief[0]]),scr_xsize=self.xysize[1],uname='defei')
    right   = widget_base(toprow,/col,ypad=0,xpad=0,tab_mode=1)
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value='  Scattering Plane:')
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value='U',xsize=self.xysize[0],/align_center)
    orieUx  = self->my_widget_text(row,value=dm_to_string(self.U[0,0]),scr_xsize=self.xysize[1],uname='orieUx')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieUy  = self->my_widget_text(row,value=dm_to_string(self.U[1,0]),scr_xsize=self.xysize[1],uname='orieUy')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieUz  = self->my_widget_text(row,value=dm_to_string(self.U[2,0]),scr_xsize=self.xysize[1],uname='orieUz')
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value='V',xsize=self.xysize[0],/align_center)
    orieVx  = self->my_widget_text(row,value=dm_to_string(self.V[0,0]),scr_xsize=self.xysize[1],uname='orieVx')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieVy  = self->my_widget_text(row,value=dm_to_string(self.V[1,0]),scr_xsize=self.xysize[1],uname='orieVy')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieVz  = self->my_widget_text(row,value=dm_to_string(self.V[2,0]),scr_xsize=self.xysize[1],uname='orieVz')
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value=' ')
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value='Q',xsize=self.xysize[0],/align_center)
    Qx      = self->my_widget_text(row,value=dm_to_string(self.Q[0,0]),scr_xsize=self.xysize[1],uname='Qx')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieVy  = self->my_widget_text(row,value=dm_to_string(self.Q[1,0]),scr_xsize=self.xysize[1],uname='Qy')
    label   = widget_label(row,value=' ',xsize=self.xysize[2],/align_center)
    orieVz  = self->my_widget_text(row,value=dm_to_string(self.Q[2,0]),scr_xsize=self.xysize[1],uname='Qz')
    row     = widget_base(right,/row,ypad=0,xpad=0)
    label   = widget_label(row,value=' ',scr_xsize=self.xysize[0],/align_center)
    geom    = widget_info(Qx,/geometry)
    calc    = widget_button(row,value='Calculate Ghoston',uname='calc',scr_xsize=self.xysize[1]*3+self.xysize[2]*2+geom.margin*3,frame=1,/align_center)
    if self.idl_version ge 5.6 then widget_control,calc,tooltip='Calculate and plot the ghoston.'
    tabs    = widget_tab(self.tlb,tab_mode=0,uname='tabbase')
    tabr    = widget_base(tabs,title='Scattering to Right',uname='tab_r',xpad=0,ypad=0,/col)
    tabrtab = widget_tab(tabr,tab_mode=0,uname='tabbase_r')
    tabrdrw = widget_base(tabrtab,title='Plot',uname='tab_r_plot',xpad=0,ypad=0,/col)
    tabrlst = widget_base(tabrtab,title='List',uname='tab_r_list',xpad=0,ypad=0)
    void    = dm_widget_droplist(tabrdrw,xsize=self.xysize[0]+2*self.xysize[1],value='View all',select=0,uname='viewsel_r',sensitive=0)
    tabl    = widget_base(tabs,title='Scattering to Left',uname='tab_l',xpad=0,ypad=0,/col)
    tabltab = widget_tab(tabl,tab_mode=0,uname='tabbase_l')
    tabldrw = widget_base(tabltab,title='Plot',uname='tab_l_plot',xpad=0,ypad=0,/col)
    tabllst = widget_base(tabltab,title='List',uname='tab_l_list',xpad=0,ypad=0)
    void    = dm_widget_droplist(tabldrw,xsize=self.xysize[0]+2*self.xysize[1],value='View all',select=0,uname='viewsel_l',sensitive=0)
    geom1   = widget_info(toprow,/geometry)
    xsize   = 930<(fix((scr_size[0]-90)/4*3))>geom1.scr_xsize
    ysize   = fix(2./3*xsize)<((scr_size[1]-100)/2)  
    font    = widget_info(calc,/fontname)
    if !version.os_family eq 'Windows' then begin
       tmp = stregex(font[0],'\*-?([0-9]+)',/subexpr,len=len)
       if len[1] ne 0 then begin
          fsize = dm_to_number(strmid(font,tmp[1],len[1]))
          if (fsize le 12) and (scr_size[0] ge 1920) then fsize = 16
          font = strmid(font,0,tmp[1])+dm_to_string(fsize)+strmid(font,tmp[1]+len[1])
       endif
    endif
    for i=0,1 do begin
        self.plots[i] = obj_new('dm_plot',xsize=xsize,ysize=ysize,widgetbase=([tabrdrw,tabldrw])[i],background='white',group_leader=self.tlb,/compound,/isolatin1,iso3dtype=1,$
                        fsize=self.fsize,ctxtfsize=10,gridontop=0,isotropic=self.flags[1],path=self.workdir,tickdir=self.flags[6])
        self.tbox[i]  = widget_text(([tabrlst,tabllst])[i],/scroll,scr_xsize=xsize,scr_ysize=ysize,font=font,/editable)  
    endfor        
    dm_center_kid,self.tlb,self.group_leader,/side ;centering the top level base
    widget_control,self.tlb,/realize,/map
    tmp  = widget_info(self.tlb,/geometry)
    tmp0 = widget_info(toprow,/geometry)
    tmp1 = widget_info(tabs,/geometry)
    tmp2 = widget_info(void,/geometry)
    tmp3 = widget_info(self.tbox[0],/geometry)
    self.plots[0]->getproperty,xsize=xsize,ysize=ysize
    self.xoffset = tmp.xsize-[xsize,tmp3.scr_xsize]
    self.yoffset = tmp.ysize-[ysize,tmp3.scr_ysize+tmp2.scr_ysize]
    self.xmargin = tmp1.scr_xsize-xsize
    self.ymargin = tmp1.scr_ysize-ysize
    self.xmin = tmp0.xsize
    self.ymin = self.yoffset[0]+fix(self.xmin/3.*2)
    if n_elements(info) ne 0 then ok = dialog_message(info,dialog_parent=self.tlb)
    widget_control,self.tlb,set_uvalue=self
    xmanager,registerName,self.tlb,cleanup='dm_ghoston_Exit',/no_block
    return,1
end

;object definition
pro dm_ghoston__define
    void = {dm_ghoston,                 $
            group_leader:  0L,          $   ;group_leader id
            tlb:           0L,          $   ;top level base
            workdir:       '',          $   ;derectory for saving files
            fsize:         12s,         $   ;default fontsize
            thick:         1.5,         $   ;default line thickness
            idl_version:   0e,          $   ;idl version
            txtfocusevent: 0b,          $   ;flag for text focus event
            flags:         bytarr(7),   $   ;misc flags [dE=0.05*E info,isotropic scaling,lattice type,show full circle,show U&V, show ki&kf in RLU, tickdir]
            e_is_range:    0b,          $   ;flag for a ranged En
            lattparm:      fltarr(6,2), $   ;[*,0] for gui entry, [*,1] values used in ghoston calculation
            U:             fltarr(3,2), $
            V:             fltarr(3,2), $
            Q:             fltarr(3,2), $
            U1:            dblarr(3),   $
            U2:            dblarr(3),   $
            U3:            dblarr(3),   $
            abc_r:         dblarr(3,3), $
            eresl:         0e,          $   ;default 0.05, dE = eresl*Ei
            fixeief:       [0b,0b],     $   ;0-fixed Ei, 1-fixed Ef, [0] gui [1] value used in calculation
            Ei:            fltarr(2,2), $
            dEi:           0e,          $
            Ef:            fltarr(2,2), $
            dEf:           0e,          $
            E:             fltarr(2,2), $
            minEstep:      0.001e,      $   ;minium energy step
            eplot_setting: strarr(7),   $   ;user specified E plot settings [eran[0:1],E precision,linecolor,showlabel,label color,label position]
            eplot_coldef:  bytarr(3,2), $   ;user defined color
            str_Eresl:     3s,          $   ;for string output
            hkl:           ptrarr(2),   $
            eran:          ptrarr(2),   $
            blast:         ptrarr(2),   $
            listind:       ptrarr(2),   $   ;save the index for the viewsel droplist, -1: all, -2: all bragg first  -3:all bragg last 
            kihkl:         fltarr(3,2), $
            a3a4:          fltarr(2,2), $ 
            plots:         objarr(2),   $   ;plot
            tbox:          lonarr(2),   $   ;text box
            xysize:        lonarr(3),   $   ;to record the xsize
            viewsel:       intarr(2),   $   ;
            xoffset:       lonarr(2),   $   ;for resize event
            yoffset:       lonarr(2),   $   ;for resize event
            xmargin:       0L,          $   ;for resize event
            ymargin:       0L,          $   ;for resize event
            xmin:          0L,          $   ;minimum xsize for resize event
            ymin:          0L           $   ;minimum ysize for resize event
           }
end

pro dm_ghoston,event,lattparm=lattparm,u=u,v=v,workdir=workdir,tickdir=tickdir,DAVETool=DAVETool,group_leader=group_leader
    if n_elements(event) ne 0 then begin
       if n_elements(group_leader) eq 0 then group_leader = event.top
       void = obj_new('dm_ghoston',lattparm=lattparm,u=u,v=v,workdir=workdir,tickdir=tickdir,group_leader=group_leader) 
    endif else $
       void = obj_new('dm_ghoston',lattparm=lattparm,u=u,v=v,workdir=workdir,tickdir=tickdir,group_leader=group_leader)
end