; $Id: $
;#######################################################################
;
; NAME:
;  dm_choose_4ddata
;
; PURPOSE:
;  choose 4d rebinned data
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   None
; keywords:
;   x1(2,3,4)dat:      axis data
;   idat,ierr:         intensity and  error bar
;   x1(2,3,4)tit,itit: label of the data
;   weight:            weight for avgsum=-1
pro dcs_mslice::dm_choose_4ddata,x1dat=x1dat,x2dat=x2dat,x3dat=x3dat,x4dat=x4dat,idat=idat,ierr=ierr,x1tit=x1tit,x2tit=x2tit,x3tit=x3tit,x4tit=x4tit,itit=itit,weight=weight,$
    debug=debug,error=error
    if keyword_set(debug) then begin
       print,'&&choose 4d rebinned data...'
       current=systime(/sec)
    endif
    
    ;weight
    if (self.bin_avgsum eq -1) or ((self.instrname eq 'macs') and keyword_set(self.macshistomode) and self.macs_rawintn) then self->dm_choose_trajdata,-4,data=weight
 
    ;intensity & error bar
    if arg_present(ierr) then begin
       ok_ierr = 1b
       self->dm_choose_trajdata,-1,data=idat,error=ierr,title=itit,debug=debug 
    endif else begin
       ok_ierr = 0b
       self->dm_choose_trajdata,-1,data=idat,title=itit,debug=debug 
    endelse
    ndat = n_elements(idat)
    ok_wght = (n_elements(weight) eq ndat)
       
    ;choose x1, x2, x3, and x4 data
    self->dm_choose_trajdata,self.d4sel_x1[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=x1dat,title=x1tit,debug=debug,is_E=is_E_x1
    self->dm_choose_trajdata,self.d4sel_x2[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=x2dat,title=x2tit,debug=debug,is_E=is_E_x2
    self->dm_choose_trajdata,self.d4sel_x3[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=x3dat,title=x3tit,debug=debug,is_E=is_E_x3
    self->dm_choose_trajdata,self.d4sel_x4[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=x4dat,title=x4tit,debug=debug,is_E=is_E_x4
     
    ;ewid
    if self.eint_yn and (self.samp_typ ne 2) then begin
       self->dm_choose_trajdata,-2,data=ewid
       emax = max((*self.projStrPtr).en,min=emin)
       dE   = emax-emin    ;default energy integration width
    endif
    ok_ewid = (n_elements(ewid) eq ndat)
    
    nogrid = [0,0,0,0]   
    if is_E_x1 then begin
       nogrid[0] = (finite(self.d4_x1step,/nan) or (self.d4_x1step eq 0))
       if nogrid[0] then uniq_E_x1 = x1dat[*,0,0]
    endif
    if is_E_x2 then begin
       nogrid[1] = (finite(self.d4_x2step,/nan) or (self.d4_x2step eq 0))
       if nogrid[1] then uniq_E_x2 = x2dat[*,0,0]
    endif
    if is_E_x3 then begin
       nogrid[2] = (finite(self.d4_x3step,/nan) or (self.d4_x3step eq 0))
       if nogrid[2] then uniq_E_x3 = x3dat[*,0,0]
    endif
    if is_E_x4 then begin
       nogrid[3] = (finite(self.d4_x4step,/nan) or (self.d4_x4step eq 0))
       if nogrid[3] then uniq_E_x4 = x4dat[*,0,0]
    endif 
    
    tran_invalid = bytarr(2)
    is_E_t = bytarr(2)
    check_fold = bytarr(10) ;u1,u2,u3,E,T,H,...
    is_fold2 = ((self.samp_typ gt 0) and (self.fold_type eq 1))
    is_u3 = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
    check_fold[0] = (is_fold2 and self.view_u1fold and finite((*(self.view_u1foldcenter))[0]))
    check_fold[1] = (is_fold2 and self.view_u2fold and finite((*(self.view_u2foldcenter))[0]))
    check_fold[2] = (is_fold2 and is_u3 and self.view_u3fold and finite((*(self.view_u3foldcenter))[0]))
    fold_center = [(*(self.view_u1foldcenter))[0],(*(self.view_u2foldcenter))[0],(*(self.view_u3foldcenter))[0],fltarr(7)]
       
    ;thickness
    ;t1
    if self.disp_flag[4] and (total(finite(self.d4sel_tft1)) ne 0) then begin
       self->dm_check_thickness,self.d4sel_t1[0>(self.samp_typ)],range=self.self.d4sel_tft1,ndat=ndat,index=index1,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
             checkfold=check_fold[self.d4sel_t1[0>(self.samp_typ)]],foldcenter=fold_center[self.d4sel_t1[0>(self.samp_typ)]],debug=debug
       is_E_t[0] = is_E
       tran_invalid[0] = traninvalid
    endif
    ;t1
    if self.disp_flag[5] and (total(finite(self.d4sel_tft2)) ne 0) then begin
       self->dm_check_thickness,self.d4sel_t2[0>(self.samp_typ)],range=self.self.d4sel_tft2,ndat=ndat,index=index2,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
             checkfold=check_fold[self.d4sel_t2[0>(self.samp_typ)]],foldcenter=fold_center[self.d4sel_t2[0>(self.samp_typ)]],debug=debug
       is_E_t[1] = is_E
       tran_invalid[1] = traninvalid
    endif   
    if n_elements(index1) ne 0 then index = temporary(index1)
    if n_elements(index2) ne 0 then begin
       if n_elements(index) ne 0 then index = dm_common(index,index2,/no_copy) $
       else index = temporary(index2)
    endif
    if n_elements(index) ne 0 then begin
       if index[0] eq -1 then begin
          emptyitem = 'combined thickness'
          tran_invalid[*] = 1b
       endif else if n_elements(idat) ne n_elements(index) then begin
          x1dat = x1dat[index]
          x2dat = x2dat[index]
          x3dat = x3dat[index]
          x4dat = x4dat[index]
          idat  = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
       endif
       index = -1
    endif
       
    ;fold to both sides
    if check_fold[self.d4sel_x1[0>(self.samp_typ)]] then begin
       x1dat = [x1dat,2.0*fold_center[self.d4sel_x1[self.samp_typ]]-x1dat]
       x2dat = [x2dat,x2dat]
       x3dat = [x3dat,x3dat]
       x4dat = [x4dat,x4dat]
       idat  = [idat,idat]
       if ok_ierr then ierr = [ierr,ierr]
       if ok_ewid then ewid = [ewid,ewid]
       if ok_wght then weight = [weight,weight]
    endif
    if check_fold[self.d4sel_x2[0>(self.samp_typ)]] then begin
       x2dat = [x2dat,2.0*fold_center[self.d4sel_x2[self.samp_typ]]-x2dat]
       x1dat = [x1dat,x1dat]
       x3dat = [x3dat,x3dat]
       x4dat = [x4dat,x4dat]
       idat  = [idat,idat]
       if ok_ierr then ierr = [ierr,ierr]
       if ok_ewid then ewid = [ewid,ewid]
       if ok_wght then weight = [weight,weight]
    endif
    if check_fold[self.d4sel_x3[0>(self.samp_typ)]] then begin
       x3dat = [x3dat,2.0*fold_center[self.d4sel_x3[self.samp_typ]]-x3dat]
       x1dat = [x1dat,x1dat]
       x2dat = [x2dat,x2dat]
       x4dat = [x4dat,x4dat]
       idat  = [idat,idat]
       if ok_ierr then ierr = [ierr,ierr]
       if ok_ewid then ewid = [ewid,ewid]
       if ok_wght then weight = [weight,weight]
    endif
    if check_fold[self.d4sel_x4[0>(self.samp_typ)]] then begin
       x4dat = [x4dat,2.0*fold_center[self.d4sel_x4[self.samp_typ]]-x4dat]
       x1dat = [x1dat,x1dat]
       x2dat = [x2dat,x2dat]
       x3dat = [x3dat,x3dat]
       idat  = [idat,idat]
       if ok_ierr then ierr = [ierr,ierr]
       if ok_ewid then ewid = [ewid,ewid]
       if ok_wght then weight = [weight,weight]
    endif
    
    ;zero error for dcs, macs, and wand data
    if total(finite(self.zero_error,/nan)) eq 0 then zero_error = self.zero_error[0]*self.zero_error[2]
    bin_zeroerror = self.bin_zeroerror*((self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand'))

    if n_elements(emptyitem) ne 0 then $
       ok = dialog_message(['The specified '+emptyitem+' range covers no data.','The entire possible range is used instead.'],dialog_parent=self.tlb,/center)

    ;x1,x2,x3,x4 range
    n_x1ran = total(finite(self.d4_x1ran))
    n_x2ran = total(finite(self.d4_x2ran))
    n_x3ran = total(finite(self.d4_x3ran))
    n_x4ran = total(finite(self.d4_x4ran))
    if n_x1ran eq 2 then begin
       index = where((x1dat le max(self.d4_x1ran)) and (x1dat ge min(self.d4_x1ran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          x1dat = x1dat[index]
          x2dat = x2dat[index]
          x3dat = x3dat[index]
          x4dat = x4dat[index]
          idat  = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          x1start = min(self.d4_x1ran)
       endif
    endif else begin
       if finite(self.d4_x1ran[0]) then begin
          index = where(x1dat ge self.d4_x1ran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             x1start = self.d4_x1ran[0]
          endif
       endif else if finite(self.d4_x1ran[1]) then begin
          index = where(x1dat le self.d4_x1ran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_x2ran eq 2 then begin
       index = where((x2dat le max(self.d4_x2ran)) and (x2dat ge min(self.d4_x2ran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          x1dat = x1dat[index]
          x2dat = x2dat[index]
          x3dat = x3dat[index]
          x4dat = x4dat[index]
          idat  = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          x2start = min(self.d4_x2ran)
       endif
    endif else begin
       if finite(self.d4_x2ran[0]) then begin
          index = where(x2dat ge self.d4_x2ran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             x2start = self.d4_x2ran[0]
          endif
       endif else if finite(self.d4_x2ran[1]) then begin
          index = where(x2dat le self.d4_x2ran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_x3ran eq 2 then begin
       index = where((x3dat le max(self.d4_x3ran)) and (x3dat ge min(self.d4_x3ran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          x1dat = x1dat[index]
          x2dat = x2dat[index]
          x3dat = x3dat[index]
          x4dat = x4dat[index]
          idat  = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          x3start = min(self.d4_x3ran)
       endif
    endif else begin
       if finite(self.d4_x3ran[0]) then begin
          index = where(x3dat ge self.d4_x3ran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             x3start = self.d4_x3ran[0]
          endif
       endif else if finite(self.d4_x3ran[1]) then begin
          index = where(x3dat le self.d4_x3ran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_x4ran eq 2 then begin
       index = where((x4dat le max(self.d4_x4ran)) and (x4dat ge min(self.d4_x4ran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          x1dat = x1dat[index]
          x2dat = x2dat[index]
          x3dat = x3dat[index]
          x4dat = x4dat[index]
          idat  = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          x4start = min(self.d4_x4ran)
       endif
    endif else begin
       if finite(self.d4_x4ran[0]) then begin
          index = where(x4dat ge self.d4_x4ran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             x4start = self.d4_x4ran[0]
          endif
       endif else if finite(self.d4_x4ran[1]) then begin
          index = where(x4dat le self.d4_x4ran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             x1dat = x1dat[index]
             x2dat = x2dat[index]
             x3dat = x3dat[index]
             x4dat = x4dat[index]
             idat  = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse

    ;rebinning according to the steps
    x1max  = max(x1dat,min=x1min)
    x2max  = max(x2dat,min=x2min)
    x3max  = max(x3dat,min=x3min)
    x4max  = max(x4dat,min=x4min)
    x1step = (x1max-x1min)/200.  ;default in case of x1step absent
    x2step = (x2max-x2min)/200.  ;default in case of x2step absent
    x3step = (x3max-x3min)/200.  ;default in case of x3step absent
    x4step = (x4max-x4min)/200.  ;default in case of x4step absent
    if self.instrname eq 'macs' then begin
       if n_elements(uniq_E_x1) ne 0 then begin
          index = where((uniq_E_x1 ge x1min) and (uniq_E_x1 le x1max),count)
          if count ne 0 then uniq_E_x1 = uniq_E_x1[index]
          uniq_E_x1 = round(uniq_E_x1*(1000d),/L64)*0.001
          uniq_E_x1 = uniq_E_x1[uniq(uniq_E_x1,sort(uniq_E_x1))]
       endif
       if n_elements(uniq_E_x2) ne 0 then begin
          index = where((uniq_E_x2 ge x2min) and (uniq_E_x2 le x2max),count)
          if count ne 0 then uniq_E_x2 = uniq_E_x2[index]
          uniq_E_x2 = round(uniq_E_x2*(1000d),/L64)*0.001
          uniq_E_x2 = uniq_E_x2[uniq(uniq_E_x2,sort(uniq_E_x2))]
       endif
       if n_elements(uniq_E_x3) ne 0 then begin
          index = where((uniq_E_x3 ge x3min) and (uniq_E_x3 le x3max),count)
          if count ne 0 then uniq_E_x3 = uniq_E_x3[index]
          uniq_E_x3 = round(uniq_E_x3*(1000d),/L64)*0.001
          uniq_E_x3 = uniq_E_x3[uniq(uniq_E_x3,sort(uniq_E_x3))]
       endif
       if n_elements(uniq_E_x4) ne 0 then begin
          index = where((uniq_E_x4 ge x4min) and (uniq_E_x4 le x4max),count)
          if count ne 0 then uniq_E_x4 = uniq_E_x4[index]
          uniq_E_x4 = round(uniq_E_x4*(1000d),/L64)*0.001
          uniq_E_x4 = uniq_E_x4[uniq(uniq_E_x4,sort(uniq_E_x4))]
       endif
    endif
    nuex1 = n_elements(uniq_E_x1)
    nuex2 = n_elements(uniq_E_x2)
    nuex3 = n_elements(uniq_E_x3)
    nuex4 = n_elements(uniq_E_x4)
    if nogrid[0] and (nuex1 ge 2) then x1step = max([x1step,mean(uniq_E_x1[1:(nuex1-1)]-uniq_E_x1[0:(nuex1-2)])])
    if nogrid[1] and (nuex2 ge 2) then x2step = max([x2step,mean(uniq_E_x2[1:(nuex2-1)]-uniq_E_x2[0:(nuex2-2)])])
    if nogrid[2] and (nuex3 ge 2) then x3step = max([x3step,mean(uniq_E_x3[1:(nuex3-1)]-uniq_E_x3[0:(nuex3-2)])])
    if nogrid[3] and (nuex4 ge 2) then x4step = max([x4step,mean(uniq_E_x4[1:(nuex4-1)]-uniq_E_x4[0:(nuex4-2)])])
    if finite(self.d4_x1step) and (self.d4_x1step ne 0) then x1step = abs(self.d4_x1step)
    if finite(self.d4_x2step) and (self.d4_x2step ne 0) then x2step = abs(self.d4_x2step)
    if finite(self.d4_x3step) and (self.d4_x3step ne 0) then x3step = abs(self.d4_x3step)
    if finite(self.d4_x4step) and (self.d4_x4step ne 0) then x4step = abs(self.d4_x4step)
    
    ;figure out dE
    if ok_ewid then begin 
       if is_E_x1 then begin
          dE = x1step
       endif else if is_E_x2 then begin
          dE = x2step
       endif else if is_E_x3 then begin
          dE = x3step
       endif else if is_E_x4 then begin
          dE = x4step
       endif else if self.disp_flag[4] then begin
          tmp_ran = [!values.f_nan,!values.f_nan]
          if self.disp_flag[4] and is_E_t[0] then tmp_ran = self.d4sel_tft1
          if self.disp_flag[5] and is_E_t[1] then tmp_ran = self.d4sel_tft2
          if finite(tmp_ran[0],/nan) then tmp_ran[0]=-1e30
          if finite(tmp_ran[1],/nan) then tmp_ran[1]=1e30
          tmp_ran = [min(tmp_ran),max(tmp_ran)]
          if tmp_ran[0] lt emin then tmp_ran[0]=emin
          if tmp_ran[1] gt emax then tmp_ran[1]=emax
          if tmp_ran[1] gt tmp_ran[0] then dE=tmp_ran[1]-tmp_ran[0]
       endif
       if dE le 0 then dE=1.0
    endif   
    
    if self.bin_checkstep then error = dm_check_step(x1dat,x1step,label=x1tit,group_leader=self.tlb,nmax=1e3)
    if keyword_set(error) then return else if self.bin_checkstep then error = dm_check_step(x2dat,x2step,label=x2tit,group_leader=self.tlb,nmax=1e3) 
    if keyword_set(error) then return else if self.bin_checkstep then error = dm_check_step(x3dat,x3step,label=x3tit,group_leader=self.tlb,nmax=1e3)
    if keyword_set(error) then return else if self.bin_checkstep then error = dm_check_step(x4dat,x4step,label=x4tit,group_leader=self.tlb,nmax=1e3)
    if keyword_set(error) then return
    
    checkfinite = self.ftypecheckmask[0] or (strlen(self.userf_str) ne 0)
    
    dm_4dgrid_bin,x1step,x2step,x3step,x4step,x1dat,x2dat,x3dat,x4dat,idat,ierr=ierr,avgsum=self.bin_avgsum,x1start=x1start,x2start=x2start,x3start=x3start,x4start=x4start,debug=debug,$
       bintr=self.binthreshold,group_leader=self.tlb,conststep=self.bin_conststep,checkfinite=checkfinite,ewid=ewid,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,$
       zoerrestrange=self.bin_statrange[0]*self.bin_statrange[1],estapplyall=self.bin_statrange[2],foundrange=foundrange
    
    if n_elements(foundrange) ne 0 then self.bin_statrange[3:4] = [foundrange,3]
    
    tmp = size(idat,/dimension)
    if n_elements(tmp) ne 4 then begin
       nx1 = n_elements(x1dat) & nx2 = n_elements(x2dat) & nx3 = n_elements(x3dat) & nx4 = n_elements(x4dat)
       idat = reform(idat,[nx1,nx2,nx3,nx4])
       if ok_ierr then ierr = reform(ierr,[nx1,nx2,nx3,nx4])
       if ok_wght then weight = reform(weight,[nx1,nx2,nx3,nx4])
    endif
    
    if ok_ewid then begin
       idat = idat*dE
       if ok_ierr then ierr = ierr*dE
    endif
    
    if keyword_set(debug) then print,'&&choose 4d rebinned data finished in ',systime(/sec)-current,' sec.'
end